# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import ReloadCli
import Tac
import EosVersion
import os
import Prefdl

sidsWith2GbFlash = [ 'SantaRosa', 'Rosa', 'Cotati', 'Cloverdale', 'Meritage',
                     'Silverado', 'Meadow', 'Eureka' ]

def isHardwareCompatibleWith2GbImage():
   # check if SID in prefdl matches one of the 2GB flash SIDs
   fsRoot = os.environ.get( "FILESYSTEM_ROOT" )
   if fsRoot != None:
      prefdlPath = os.path.join( fsRoot, "etc", "prefdl" )
   else:
      prefdlPath = "/etc/prefdl"

   if os.path.isfile( prefdlPath ):
      with open( prefdlPath, 'r' ) as f:
         prefdlOp = f.read().strip()
   else:
      raise IOError( 'could not find file %s' % prefdlPath )

   prefdlDict = Prefdl.parsePrefdl( prefdlOp )
   sidStr = prefdlDict.get( 'sid' )

   if sidStr == None :
      raise ValueError( 'could not find SID in preFdl' )
   elif any( item in sidStr for item in sidsWith2GbFlash ):
      return True
   else:
      return False

def checkImage2GbCompatible( mode, imagePath, swImage ):
   import zipfile
   try :
      twoGbVersion = EosVersion.swiVersion( imagePath ).is2GBVersion()
   except zipfile.BadZipfile :
      return False, "Not able to read boot image at %s " % swImage

   if not twoGbVersion:
      return True, ""

   warnStr = "Software image %s is not compatible on this hardware" % swImage
   try :
      if not isHardwareCompatibleWith2GbImage():
         return False, warnStr
   except ( ValueError, IOError ):
      return False, "Cant read the required file"

   return True, ""

def checkIfHardwareSupportsImage( mode, power, now ):

   imageFile, swImage = ReloadCli.getImagePath( mode )

   # Check can't be performed on remote URL.
   if not hasattr( imageFile, 'realFilename_' ):
      mode.addError( 'The boot configuration specifies a remote location for the '
                     'SWI image to boot: {}\nThe reload image compatibility check '
                     'is only supported on SWI images present in the local file '
                     'system.\nPlease update the boot configuration to specify a '
                     'SWI image present in the local file system before re-issuing '
                     'the reload command.'.format( str( imageFile ) ) )
      return False 

   # pylint: disable=maybe-no-member
   if imageFile.realFilename_ == None or 'not set' in swImage:
      # Boot image not set, Warning already present in ReloadCli
      return True

   if not imageFile.exists():
      # Boot image is set but it's doesnt exist
      mode.addError( "Boot image doesn't exist at : %s " % swImage )
      return False

   retVal, err = checkImage2GbCompatible( mode, imageFile.realFilename_, swImage )

   if not retVal:
      mode.addError( err )

   return retVal

# Register reload hook as RUN_FIRST
ReloadCli.registerReloadHook( checkIfHardwareSupportsImage, 'Reload2GbImageCheckCli',
                              'RUN_FIRST' )

