# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Toggles.EoamToggleLib
import BasicCli
import CliCommand
import CliMatcher
import CliParser
from CliToken.Monitor import monitorMatcher, monitorMatcherForShow
import ConfigMount
from EthIntfCli import EthPhyIntf
import ShowCommand
import Tac
import Tracing
import LazyMount

from CliMode.Reflector import ReflectorModeBase, ReflectorInterfaceModeBase
from ReflectorTypes import ( stringToDirection, stringToMacAction,
                             directionToString, macActionToString,
                             hwPrgmStatusToString )
from ReflectorModel import ( ReflectorModel, ReflectorInterfaceModel )

__defaultTraceHandle__ = Tracing.Handle( 'ReflectorCli' )
t0 = Tracing.trace0

reflectorConfig = None
reflectorStatus = None
reflectorHwCapabilities = None

class ReflectorConfigMode( ReflectorModeBase, BasicCli.ConfigModeBase ):
   name = "reflector"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ReflectorModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.session_.gotoParentMode()

class ReflectorInterfaceConfigMode( ReflectorInterfaceModeBase,
                                    BasicCli.ConfigModeBase ):
   name = "reflector interface"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      assert isinstance( parent, ReflectorConfigMode )
      self.reflectorInterfaceModeContext = context
      param = ( context.key.toStrep() )
      ReflectorInterfaceModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.reflectorInterfaceModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.reflectorInterfaceModeContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.reflectorInterfaceModeContext
      self.reflectorInterfaceModeContext = None
      context.commit()

class ReflectorInterfaceModeContext( object ):
   def __init__( self, mode, key ):
      self.mode = mode
      self.key_ = key
      interfaceConfig = reflectorConfig.interfaceConfig
      if key in interfaceConfig:
         self.editEntry_ = self.copyEditEntry( interfaceConfig[ key ] )
      else:
         self.editEntry_ = self.newEditEntry()

   @property
   def key( self ):
      return self.key_

   @property
   def editEntry( self ):
      return self.editEntry_

   @staticmethod
   def copyReflectorInterface( newReflectorInterface, oldReflectorInterface ):
      # Copy old to new
      if oldReflectorInterface:
         newReflectorInterface.direction = oldReflectorInterface.direction
         newReflectorInterface.macAction = oldReflectorInterface.macAction
         newReflectorInterface.loopbackIntfId = oldReflectorInterface.loopbackIntfId

   def copyEditEntry( self, oldEntry ):
      newInterfaceConfig = Tac.newInstance( 'Reflector::InterfaceConfig',
                                            self.key_ )
      self.copyReflectorInterface( newInterfaceConfig, oldEntry )
      return newInterfaceConfig

   def newEditEntry( self ):
      newInterfaceConfig = Tac.newInstance( 'Reflector::InterfaceConfig',
                                            self.key_ )
      # We currently fix the loopback interface to match the reflector interface
      # This may change in the future
      newInterfaceConfig.loopbackIntfId = self.key_.intfId
      return newInterfaceConfig

   def commit( self ):
      # Commit current reflector
      interfaceConfig = reflectorConfig.interfaceConfig

      if self.editEntry_:
         if self.key_ not in interfaceConfig:
            interfaceConfig_ = interfaceConfig.newMember( self.key_ )
            bumpVersion = True
         else:
            interfaceConfig_ = interfaceConfig[ self.key_ ]
         bumpVersion = interfaceConfig_.differsExceptVersionFrom( self.editEntry_ )
         self.copyReflectorInterface( interfaceConfig_, self.editEntry_ )
         if bumpVersion:
            interfaceConfig_.version += 1

def gotoReflectorInterfaceMode( mode, args ):
   intfId = Tac.newInstance( 'Arnet::IntfId', args[ 'INTF' ].name )
   key = Tac.newInstance( 'Reflector::InterfaceKey', intfId )
   context = ReflectorInterfaceModeContext( mode, key )

   mode.reflectorInterfaceModeContext = context
   childMode = mode.childMode( ReflectorInterfaceConfigMode,
                               context=context )
   mode.session_.gotoChildMode( childMode )

def deleteReflectorInterface( mode, args ):
   intfId = Tac.newInstance( 'Arnet::IntfId', args[ 'INTF' ].name )
   key = Tac.newInstance( 'Reflector::InterfaceKey', intfId )
   interfaceConfig = reflectorConfig.interfaceConfig
   if key not in interfaceConfig:
      return
   del interfaceConfig[ key ]

def setReflectorInterfaceDirection( mode, args ):
   interfaceConfig = mode.reflectorInterfaceModeContext.editEntry
   if interfaceConfig:
      interfaceConfig.direction = stringToDirection[ args.get( 'DIRECTION' ) ]

def setReflectorInterfaceMacAction( mode, args ):
   interfaceConfig = mode.reflectorInterfaceModeContext.editEntry
   if interfaceConfig:
      interfaceConfig.macAction = stringToMacAction[ args.get( 'ACTION' ) ]

def showReflectorInterface( mode, args ):
   intf = args.get( 'INTF' )
   if intf is None:
      key = None
   else:
      intfId = Tac.newInstance( 'Arnet::IntfId', intf.name )
      key = Tac.newInstance( 'Reflector::InterfaceKey', intfId )

   reflectorInterfaces = {}

   def addReflectorInterfaceModel( key, config, status ):
      if not config:
         return

      if status:
         hwStatus = hwPrgmStatusToString[ status.hwPrgmStatus ]
         hwReason = status.hwPrgmReason
      else:
         # No status
         hwStatus = 'inactive'
         hwReason = 'platform agent not running?'

      reflectorInterfaces[ key ] = ReflectorInterfaceModel(
         _sortOrder=len( reflectorInterfaces ),
         direction=directionToString[ config.direction ],
         macAction=macActionToString[ config.macAction ],
         status=hwStatus,
         statusReason=hwReason )

   interfaceConfig = reflectorConfig.interfaceConfig
   if key is None:
      for interfaceKey, config in interfaceConfig.iteritems():
         status = reflectorStatus.interfaceStatus.get( interfaceKey, None )
         addReflectorInterfaceModel( interfaceKey.toStrep(), config, status )
   elif key in interfaceConfig:
      config = interfaceConfig[ key ]
      status = reflectorStatus.interfaceStatus.get( key, None )
      addReflectorInterfaceModel( key.toStrep(), config, status )
   else:
      addReflectorInterfaceModel( key.toStrep(), None, None )

   return ReflectorModel( reflectorInterfaces=reflectorInterfaces )

def reflectorGuard( mode, token ):
   if not Toggles.EoamToggleLib.toggleEoamReflectorEnabled():
      return CliParser.guardNotThisEosVersion
   if ( not reflectorHwCapabilities.directionSupported or
        not reflectorHwCapabilities.macActionSupported ):
      return CliParser.guardNotThisPlatform
   return None

def directionGuard( mode, token ):
   if ( token not in stringToDirection or
        stringToDirection[ token ] not in
        reflectorHwCapabilities.directionSupported ):
      return CliParser.guardNotThisPlatform
   return None

def macActionGuard( mode, token ):
   if ( token not in stringToMacAction or
        stringToMacAction[ token ] not in
        reflectorHwCapabilities.macActionSupported ):
      return CliParser.guardNotThisPlatform
   return None

ethernetMatcher = CliCommand.singleKeyword( 'ethernet',
      helpdesc='Ethernet protocol configuration' )
reflectorsMatcher = CliCommand.guardedKeyword( 'reflectors',
      helpdesc='Ethernet reflector configuration',
      guard=reflectorGuard )
interfaceMatcher = CliCommand.singleKeyword( 'interface',
      helpdesc='Reflector interface configuration' )

#------------------------------------------------------------------------
# [ no/default ] monitor ethernet reflectors (in GlobalConfig mode)
#------------------------------------------------------------------------

class MonitorEthernetOamCmd( CliCommand.CliCommandClass ):
   syntax = 'monitor ethernet reflectors'
   noOrDefaultSyntax = syntax
   data = {
      'monitor' :  monitorMatcher,
      'ethernet' : ethernetMatcher,
      'reflectors' : reflectorsMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ReflectorConfigMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      interfaceConfig = reflectorConfig.interfaceConfig
      interfaceConfig.clear()

BasicCli.GlobalConfigMode.addCommandClass( MonitorEthernetOamCmd )

#------------------------------------------------------------------------
# [no/default] reflector interface <intf> (in ReflectorConfig mode)
#------------------------------------------------------------------------
class ReflectorInterfaceIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'reflector interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'reflector' : 'Reflector configuration',
      'interface' : interfaceMatcher,
      'INTF' : EthPhyIntf.ethMatcher,
   }

   handler = gotoReflectorInterfaceMode
   noOrDefaultHandler = deleteReflectorInterface

ReflectorConfigMode.addCommandClass( ReflectorInterfaceIntfCmd )

#--------------------------------------------------------------------------------
# [ no | default ] direction ( in | out )
#--------------------------------------------------------------------------------

directionOptions = { 'in': 'Reflect on ingress',
                     'out': 'Reflect on egress' }

class DirectionCmd( CliCommand.CliCommandClass ):
   syntax = 'direction DIRECTION'
   noOrDefaultSyntax = 'direction ...'
   data = {
      'direction' : 'Reflector interface direction configuration',
      'DIRECTION' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( directionOptions ),
         guard=directionGuard )
   }

   handler = setReflectorInterfaceDirection
   noOrDefaultHandler = setReflectorInterfaceDirection

ReflectorInterfaceConfigMode.addCommandClass( DirectionCmd )

#--------------------------------------------------------------------------------
# ( no | default ) mac swap (in ReflectorInterfaceConfig mode)
#--------------------------------------------------------------------------------

macActionOptions = { 'swap': 'Swap source and destination MACs on reflection' }

class MacSwapCmd( CliCommand.CliCommandClass ):
   syntax = 'mac ACTION'
   noOrDefaultSyntax = 'mac ...'
   data = {
      'mac' : 'Reflector interface MAC action configuration',
      'ACTION' : CliCommand.Node(
         matcher=CliMatcher.EnumMatcher( macActionOptions ),
         guard=macActionGuard )
   }

   handler = setReflectorInterfaceMacAction
   noOrDefaultHandler = setReflectorInterfaceMacAction

ReflectorInterfaceConfigMode.addCommandClass( MacSwapCmd )

#--------------------------------------------------------------------------------
# show monitor ethernet reflectors interface INTF
#--------------------------------------------------------------------------------
class MonitorEthernetReflectorsInterfaceIntfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor ethernet reflectors [ interface INTF ]'
   data = {
      'monitor' : monitorMatcherForShow,
      'ethernet' : ethernetMatcher,
      'reflectors' : reflectorsMatcher,
      'interface' : interfaceMatcher,
      'INTF' : EthPhyIntf.ethMatcher,
   }
   cliModel = ReflectorModel

   handler = showReflectorInterface

BasicCli.addShowCommandClass( MonitorEthernetReflectorsInterfaceIntfCmd )

def Plugin( entityManager ):
   global reflectorConfig, reflectorStatus, reflectorHwCapabilities
   reflectorConfig = ConfigMount.mount( entityManager, "reflector/config",
                                        "Reflector::Config", "w" )
   reflectorStatus = LazyMount.mount( entityManager, "reflector/status",
                                      "Reflector::Status", "r" )
   reflectorHwCapabilities = LazyMount.mount( entityManager,
                                              "reflector/hardware/capabilities",
                                              "Reflector::HwCapabilities", "r" )
