# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Tac
import CliModel
import RedSupLib
import Ark

RedundancyMode = Tac.Type( "Election::RedundancyMode" )
RedundancyProtocol = Tac.Type( "Redundancy::RedundancyProtocol" )
PeerElectionState = Tac.Type( "Election::PeerState" )

class RedundancyStates( CliModel.Model ):
   myMode = CliModel.Enum( values=RedundancyMode.attributes,
                           help="Redundancy mode of the supervisor" )
   peerMode = CliModel.Enum( values=RedundancyMode.attributes,
                             help="Redundancy mode of the peer supervisor" )
   configuredProtocol = CliModel.Enum( values=RedundancyProtocol.attributes,
                                       help="Configured redundancy protocol" )
   operationalProtocol = CliModel.Enum( values=RedundancyProtocol.attributes,
                                        help="Operational redundancy protocol" )
   slotId = CliModel.Int( help="Slot id of the supervisor" )
   unitDesc = CliModel.Enum( values=( "Primary", "Secondary" ),
                             help="Description of supervisor" )
   communicationDesc = CliModel.Enum( values=( "Up", "Down" ),
                                      help="Description of communication link" )
   peerState = CliModel.Enum( values=PeerElectionState.attributes,
                              help="Standby supervisor state" )
   globalStageCompletionTimeout = CliModel.Float( 
                            help="switchover completion timeout in seconds" )
   globalStageCompletionTimeoutDefault = CliModel.Float( 
                      help="switchover completion timeout default in seconds" )
   switchoverReady = CliModel.Bool( help="Is switch ready for switchover" )
   agentsNotReady = CliModel.List( valueType=str,
                                   help="Agents not ready for SSO switchover",
                                   optional=True )
   pcieSsoReady = CliModel.Bool( help="Is PCIE switch ready for switchover",
                                 optional=True )
   pcieModulesNotReady = CliModel.List( valueType=str,
                         help="PCIE modules not ready for SSO switchover",
                         optional=True )

   lastRedundancyModeChangeTime = CliModel.Float(
                                 help="Time of last redundancy mode change in UTC",
                                 optional=True )
   lastRedundancyModeChangeReason = CliModel.Str(
                                    help="Reason for last redundancy mode change" )

   def render( self ):
      print "  my state = %s" % RedSupLib.redModeToDisplayFormat( self.myMode,
                                   self.operationalProtocol )
      print "peer state = %s" % RedSupLib.redModeToDisplayFormat( self.peerMode,
                                   self.operationalProtocol )
      print "      Unit = %s" % self.unitDesc
      print "   Unit ID = %d" % self.slotId
      print ""
      msgStr = "Redundancy Protocol (%s) = %s"
      print  msgStr % ( "Operational",
                       RedSupLib.protocolToDisplayFormat( self.operationalProtocol ))
      print  msgStr % ( "Configured",
                       RedSupLib.protocolToDisplayFormat( self.configuredProtocol ))
      print "Communications = %s" % self.communicationDesc
      if self.operationalProtocol == "sso":
         if not self.globalStageCompletionTimeout:
            print "switchover completion timeout disabled"
         elif self.globalStageCompletionTimeout == \
            self.globalStageCompletionTimeoutDefault:
            print "switchover completion timeout = %s seconds (default)" % \
               self.globalStageCompletionTimeout
         else:
            print "switchover completion timeout = %s seconds" % \
               self.globalStageCompletionTimeout
      if self.switchoverReady:
         print "Ready for switchover"
      else:
         switchoverNotReadyDesc = ""
         if self.configuredProtocol == self.operationalProtocol and \
            self.operationalProtocol == "simplex" and \
            self.myMode == "active":
            switchoverNotReadyDesc = "Redundancy protocol is simplex"
         elif self.peerState == "notInserted":
            switchoverNotReadyDesc = "Peer supervisor is not present"
         elif self.peerState == "poweredOff":
            switchoverNotReadyDesc = "Peer supervisor is powered off"
         elif self.peerState == "poweringOn":
            switchoverNotReadyDesc = "Peer supervisor is either "\
                                       "booting up or disabled"
         elif self.peerState == "unknownPeerState":
            switchoverNotReadyDesc = "Peer state is unknown"
         elif self.agentsNotReady:
            switchoverNotReadyDesc = "Agents not ready in standby supervisor"
         elif self.pcieModulesNotReady:
            switchoverNotReadyDesc = "PCIE devices not ready "\
                                      "in standby supervisor"
         print "Not ready for switchover (%s)" % switchoverNotReadyDesc
         if self.agentsNotReady:
            print "Agents not ready ="
            for agent in self.agentsNotReady:
               print "   %s" % agent
         elif self.pcieModulesNotReady:
            print "PCIE modules not ready ="
            for module in self.pcieModulesNotReady:
               print "   %s" % module
      print ""
      if self.myMode == "active":
         if self.lastRedundancyModeChangeTime:
            switchoverTime = Ark.timestampToStr( self.lastRedundancyModeChangeTime -
                                                   Tac.utcNow() + Tac.now() )
         else:
            switchoverTime = Ark.timestampToStr( self.lastRedundancyModeChangeTime )
         print "  Last switchover time = %s" % switchoverTime
         print "Last switchover reason = %s" % self.lastRedundancyModeChangeReason

class SsoSwitchoverLogs( CliModel.Model ):
   switchoverCount = CliModel.Int( help="Number of switchovers since reload" )
   lastSwitchoverTime = CliModel.Float( help="Time of last switchover",
                                        optional=True )
   lastSwitchoverStatus = CliModel.Enum( values=( "Completed", "In Progress" ),
                                         help="Status of last switchover",
                                         optional=True )
   class SwitchoverLog( CliModel.Model ):
      time = CliModel.Float( help="Time of the log relative to onset of switchover" )
      msg = CliModel.Str( help="Log message" )
   logs = CliModel.List( valueType=SwitchoverLog,
                         help="List of switchover logs",
                         optional=True )

   def render( self ):
      print "Total number of Stateful Switchovers completed since reload:",
      print self.switchoverCount

      utcDelta = Tac.utcNow() - Tac.now()
      if self.lastSwitchoverTime:
         lastSwitchoverTime = self.lastSwitchoverTime - utcDelta
         print "Latest Stateful Switchover occured %s @ %s (%s)" % (
                  Ark.timestampToStr( lastSwitchoverTime ),
                  Ark.timestampToStr( lastSwitchoverTime, relative=False ),
                  self.lastSwitchoverStatus.lower() )
      for log in self.logs:
         print "%10.6f: %s" % ( log.time, log.msg )

