# Copyright (c) 2006-2009, 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliParser
import BasicCliModes
import ModuleCli
import ConfigMount
import LazyMount
import CliMode.RedSup
import Cell
import Fru
import Tac
import RedSupModels
import SysdbUtil
import PyClient
import re
import os
from StageMgr import defaultStageInstance
from RedSupPyClient import peerRedSupPyClient
import FileReplicationModels
import TechSupportCli

em = None
fileReplicationCliConfig = None
fileReplicationConfig = None
replicationStatus = None
redundancyConfig = None
switchoverLog = None
sysdbStatus = None
electionDir = None
electionConfig = None
electionStatus = None
redundancySyncStatus = None
powerFuse = None
pcieSsoStatus = None
agentSsoStatus = None
modularInventory = None
ssoSliceStatus = None

FileReplicationStatus = Tac.Type( "FileReplication::FileStatus::Status" )

def doRedForceSwitchover( mode, args ):
   myMode = electionStatus.redundancyMode
   peerMode = electionStatus.peerRedundancyMode

   if myMode not in ( "active", "standby" ):
      print 'Supervisor is disabled; cannot switch over'
      return

   if myMode == 'active':
      if peerMode != 'standby':
         print 'Other supervisor is offline; cannot switch over.'
         return
      print "This supervisor will be restarted."
      # Currently, we rely on the fact that the supervisor's configRedundancyMode
      # is cleared along with its Sysdb state during reboot after the supervisor
      # has been disabled.
   elif myMode == 'standby':
      # Only standby supervisor writes the config redundancy mode
      # as 'configActive'.
      print "Peer supervisor will be restarted."
   # pylint: disable-msg=E1103
   electionConfig.forceSwitchover += 1

def supervisorRedundancyGuard( mode, token ):
   if( Cell.cellType() == 'fixed' or
       ( Cell.cellType() == 'supervisor' and not Cell.electionMgrSupported() and
         'REDSUPDUT_PEER_REDMODE' not in os.environ ) ):
      return CliParser.guardNotThisPlatform
   return None

#------------------------------------------------------
# The "show redundancy states" command, in enable mode.
#
# Legacy "show redundancy states"
#------------------------------------------------------
def doShowRedStatus( mode, args ):
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   redundancyStates = RedSupModels.RedundancyStates()
   redundancyStates.myMode = electionStatus.redundancyMode
   redundancyStates.peerMode = electionStatus.peerRedundancyMode
   if 'disabled' in ( redundancyStates.myMode, redundancyStates.peerMode ):
      redundancyStates.communicationDesc = 'Down'
   else:
      redundancyStates.communicationDesc = 'Up'

   redundancyStates.switchoverReady = redundancyStatus.switchoverCapable
   redundancyStates.agentsNotReady = \
      [ ag for ag in agentSsoStatus.peerAgentSsoReady
        if agentSsoStatus.peerAgentSsoReady.get( ag ) is False ]

   redundancyStates.pcieSsoReady = pcieSsoStatus.pcieSsoReady
   def displayName( module ):
      # module is in the format Slot<id>
      slotId = int( module[ 4: ] )
      if slotId in modularInventory.card:
         return modularInventory.card[ slotId ].sliceId
      else:
         return "unrecognized " + module
   redundancyStates.pcieModulesNotReady = []
   for sliceName in ssoSliceStatus:
      for mod in ssoSliceStatus[ sliceName ].pcieModuleSsoReady:
         if ssoSliceStatus[ sliceName ].pcieModuleSsoReady[ mod ] is False:
            redundancyStates.pcieModulesNotReady.append( displayName( mod ) )

   redundancyStates.peerState = electionStatus.peerState
   redundancyStates.slotId = Fru.slotId()
   redundancyStates.unitDesc = 'Primary' if Fru.slotId() == 1 else 'Secondary'
   redundancyStates.configuredProtocol = redundancyConfig.protocol
   redundancyStates.operationalProtocol = redundancyStatus.protocol
   redundancyStates.globalStageCompletionTimeout = \
      redundancyConfig.globalStageCompletionTimeout
   redundancyStates.globalStageCompletionTimeoutDefault = \
      redundancyConfig.globalStageCompletionTimeoutDefault
   if electionStatus.lastRedundancyModeChangeTime:
      redundancyStates.lastRedundancyModeChangeTime = \
            electionStatus.lastRedundancyModeChangeTime + Tac.utcNow() - Tac.now()
   redundancyStates.lastRedundancyModeChangeReason = \
               electionStatus.lastRedundancyModeChangeReason
   return redundancyStates

def doShowSwitchoverSsoLog( mode, args ):
   ssoSwitchoverLogs = RedSupModels.SsoSwitchoverLogs()
   ssoSwitchoverLogs.switchoverCount = sysdbStatus.switchoverCount
   beginTime = 0
   utcDelta = Tac.utcNow() - Tac.now()
   if defaultStageInstance in switchoverLog.instanceLog:
      logs = switchoverLog.instanceLog[ defaultStageInstance ]
      for t in logs.log.itervalues():
         if beginTime == 0:
            beginTime = t.timestamp
            if mode.session_.entityManager.redundancyStatus().mode == 'active':
               status = 'Completed'
            else:
               status = 'In Progress'
            ssoSwitchoverLogs.lastSwitchoverTime = t.timestamp + utcDelta
            ssoSwitchoverLogs.lastSwitchoverStatus = status
         log = RedSupModels.SsoSwitchoverLogs.SwitchoverLog()
         log.time = t.timestamp - beginTime
         log.msg = t.msg
         ssoSwitchoverLogs.logs.append( log )
   return ssoSwitchoverLogs

#-----------------------------------------------------------
# The redundancy configuration mode commands
#    " protocol (simplex|rpr|sso)"
#-----------------------------------------------------------

class RedundancyMode( CliMode.RedSup.RedundancyMode, BasicCliModes.ConfigModeBase ):
   name = 'Redundant Supervisor Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.RedSup.RedundancyMode.__init__( self, None )
      BasicCliModes.ConfigModeBase.__init__( self, parent, session )

def gotoRedundancyMode( mode, args ):
   childMode = mode.childMode( RedundancyMode )
   mode.session_.gotoChildMode( childMode )

def ssoSupportedGuard( mode, token ):
   if mode.session_.entityManager_.redundancySupportStatus().ssoSupported:
      return None
   return CliParser.guardNotThisPlatform

def setRedundancyProtocol( mode, args ):
   # CliParser guard ensures Ctrl reaches here only on active supervisor
   redundancyProtocol = None
   possibleProtocols = [ 'simplex', 'rpr', 'sso' ]
   for token in possibleProtocols:
      if token in args:
         redundancyProtocol = token
         break
   configProtocol = redundancyProtocol or redundancyConfig.defaultProtocol
   otherSlotId = 2 if Fru.slotId() == 1 else 1
   otherSupervisor = "Supervisor%d" % otherSlotId
   swiVersionMismatch = False
   protoFile = SysdbUtil.REDUNDANCY_PROTOCOL_FILE
   if os.path.exists( protoFile ) and os.access( protoFile, os.R_OK ):
      protoFileContent = file( protoFile ).read()
      swiVersionMismatch = re.search( "SWI_VERSION_MISMATCH=1",
                                      protoFileContent ) is not None

   if configProtocol == 'sso' and \
      not mode.session_.entityManager_.redundancySupportStatus().ssoSupported:
      mode.addWarning( "Redundancy protocol SSO is not supported for one or more " \
                       "of the cards in the chassis.")
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   statusProtocol = redundancyStatus.protocol
   if ( redundancyStatus.peerState != "notInserted" 
        and not mode.session_.startupConfig()
        and configProtocol != redundancyConfig.protocol
        and configProtocol != statusProtocol ):

      resetPeer = True

      if powerFuse.powerOffRequested.get( otherSupervisor ):
         # Cannot reset a peer that is configured to be powered down.
         resetPeer = False

      if ( configProtocol == "sso" and   
           statusProtocol == "rpr" and
           swiVersionMismatch ):
         # Peer negotiated rpr due to incompatible swi, so reset will not help.
         resetPeer = False

      if resetPeer:
         print "Peer supervisor will be restarted."
   redundancyConfig.protocol = configProtocol

#-------------------------------------------------------------------------------
# The redundancy configuration mode commands
#   {no|default} switchover completion timeout [<seconds>]
#-------------------------------------------------------------------------------
def setStageCompletionTimeout( mode, args ):
   timeout = args[ 'TIMEOUT' ]
   redundancyConfig.globalStageCompletionTimeout = timeout

def noStageCompletionTimeout( mode, args ):
   args[ 'TIMEOUT' ] = 0
   setStageCompletionTimeout( mode, args )

def defaultStageCompletionTimeout( mode, args ):
   args[ 'TIMEOUT' ] = redundancyConfig.globalStageCompletionTimeoutDefault
   setStageCompletionTimeout( mode, args )

#-------------------------------------------------------------------------------
# The redundancy file replication configuration mode commands
#-------------------------------------------------------------------------------
class RedundancyFileReplMode( CliMode.RedSup.RedundancyFileReplMode,
                              BasicCliModes.ConfigModeBase ):
   name = 'Redundant Supervisor File Replication Configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.RedSup.RedundancyFileReplMode.__init__( self, None )
      BasicCliModes.ConfigModeBase.__init__( self, parent, session )

def gotoFileReplicationMode( mode, args ):
   childMode = mode.childMode( RedundancyFileReplMode )
   mode.session_.gotoChildMode( childMode )

def noFileReplicationMode( mode, args ):
   fileReplicationCliConfig.path.clear()

def addFileReplPath( mode, args ):
   url = args[ 'URL' ]
   if url.pathname == '/':
      mode.addError( "Cannot add file system root" )
      return
   fileReplicationCliConfig.path[ str( url ) ] = True

def removeFileReplPath( mode, args ):
   del fileReplicationCliConfig.path[ str( args[ 'URL' ] ) ]

#--------------------------------------------------
# show redundancy file-replication
#--------------------------------------------------
def padString( string, length ):
   padding = length - len(string)
   if padding > 0:
      return string + ( padding * ' ' )
   else:
      return string

def doShowRedFileRep( mode, args ):
   config = fileReplicationConfig
   status = replicationStatus

   fileReplication = FileReplicationModels.RequesterStatus()

   # Start with files in config but not status
   configKeys = []
   if 'RedSup' in config.requester:
      configKeys = config.requester[ 'RedSup' ].file.keys()
   configKeys.sort( key=lambda x: x.lower() )
   for key in configKeys:
      if not status.requester.get( 'RedSup' ) or \
             key not in status.requester[ 'RedSup' ].file.keys():
         replicationStsModel = FileReplicationModels.FileStatus()
         replicationStsModel.status = FileReplicationStatus.unsynchronized
         fileReplication.files[ key ] = replicationStsModel

   # Move on to files in status
   statusKeys = []
   if 'RedSup' in status.requester:
      statusKeys = status.requester[ 'RedSup' ].file.keys()
   statusKeys.sort( key=lambda x: x.lower() )
   for key in statusKeys:
      fileObj = status.requester[ 'RedSup' ].file[ key ]
      replicationStsModel = FileReplicationModels.FileStatus()
      replicationStsModel.status = fileObj.status
      if fileObj.lastSynchronized:
         replicationStsModel.lastSyncTime = fileObj.lastSynchronized
      fileReplication.files[ key ] = replicationStsModel
   return fileReplication

#--------------------------------------------------
# show redundancy time-sync
#--------------------------------------------------
def doShowTimeSyncStatus( mode, args ):
   # time-sync command to be applicable only for sso
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   if redundancyStatus.protocol != 'sso':
      print "Information only available for SSO protocol"
      return
      
   import time

   timeSyncStatus = None
   if redundancyStatus.mode == 'active':
      # Only standby supe writes redundancySyncStatus in its own cell path
      # Hence if mode is active, use peer's sync status
      sysname = mode.session_.sysname
      pyClient = None
      if not os.getenv( "A4_CHROOT" ):
         pyClient = peerRedSupPyClient( sysname=sysname )
      else:
         pyClient = PyClient.PyClient( sysname, "Sysdb" )

      if pyClient is None:
         print "Connection to peer supervisor failed, please retry"
         return

      root = pyClient.root()
      peerCell = Cell.peerCellId()
      sysdbRoot = root[ sysname ][ 'Sysdb' ]
      timeSyncStatus = \
         sysdbRoot.entity[ Cell.path( 'redundancy/timeSync/status', cell=peerCell ) ]
   else:
      timeSyncStatus = redundancySyncStatus
      
   print "Last Monotonic Time updated at %s" % time.ctime ( 
      timeSyncStatus.lastMonoTimeUpdateTimestamp )
   if timeSyncStatus.failCount == 0:
      print "Current Monotonic Time Difference = %f" \
         % timeSyncStatus.currentMonoTimeDelta
   else:
      print "Number of times Monotonic Time update failed " \
         "since last successful update = %d" % timeSyncStatus.failCount
      print "Last Error encountered while updating " \
         "monotonic time is : %s" % timeSyncStatus.lastErrorUpdating

#--------------------------------------------------
# Register the redundancy commands to be installed iff
# we're in a modular system
#--------------------------------------------------

def showTechCmds():
   cmds = []

   if Cell.electionMgrSupported():
      cmds += [
         "show redundancy status",
         "bash cat /sys/arista/election_manager/debug_info",
      ]

   cmds += [
      "bash cat /var/log/NorCalInit",
   ]

   if Cell.electionMgrSupported():
      cmds += [
         "show redundancy switchover sso",
         "show redundancy time-sync",
      ]

   return cmds

def showTechSummaryCmds():
   cmds = []

   if Cell.electionMgrSupported():
      cmds += [
         "show redundancy status",
         "bash cat /sys/arista/election_manager/debug_info",
      ]
   return cmds

# Register showTechCmds with 'show tech support'
TechSupportCli.registerShowTechSupportCmdCallback( '2010-06-11 04:00:00',
      showTechCmds,
      summaryCmdCallback=showTechSummaryCmds )

#--------------------------------------------------
# Register a hook for 'show module' to pass on supervisor statuses.
#--------------------------------------------------

def getSupervisorStatus( mode, card ):
   redundancyStatus = mode.session_.entityManager_.redundancyStatus()
   cardName = "%s%s" % ( card.tag, card.label )
   otherSlotId = 2 if Fru.slotId() == 1 else 1
   otherSup = "Supervisor%d" % otherSlotId
   if cardName == otherSup:
      if redundancyStatus.peerState in ( "poweredOff", "poweringOn" ):
         return redundancyStatus.peerState
      else:
         return redundancyStatus.peerMode
   else:
      return redundancyStatus.hwMode

def getSupervisorUptime( mode, card ):
   if card.label == "%d" % Fru.slotId():
      if electionStatus.redundancyMode in [ "active", "standby" ]:
         td = Tac.now() - electionStatus.lastRedundancyModeChangeTime
         uptime = Tac.utcNow() - td
      else:       
         uptime = None
   else:          
      if electionStatus.peerRedundancyMode in [ "active", "standby" ]:
         td = Tac.now() - electionStatus.lastPeerRedundancyModeChangeTime
         uptime = Tac.utcNow() - td
      else:       
         uptime = None
   return uptime

ModuleCli.registerCardStatusHook( "Supervisor", getSupervisorStatus )
ModuleCli.registerCardUptimeHook( "Supervisor", getSupervisorUptime )

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global em, electionDir
   global fileReplicationCliConfig, fileReplicationConfig
   global redundancyConfig, electionConfig
   global replicationStatus, switchoverLog, sysdbStatus
   global redundancySyncStatus, powerFuse, pcieSsoStatus, agentSsoStatus
   global electionStatus
   global modularInventory
   global ssoSliceStatus

   em = entityManager
   redundancyConfig = ConfigMount.mount( entityManager, "redundancy/config",
                                         "Redundancy::RedundancyConfig", "w" )
   fileReplicationCliConfig = ConfigMount.mount(
      entityManager,
      "redundancy/fileReplication/cli/config",
      "Election::FileReplicationCliConfig", "w" )
   fileReplicationConfig = LazyMount.mount( entityManager,
                                            "redundancy/fileReplication/config",
                                            "FileReplication::Config", "r" )
   electionDir = LazyMount.mount( entityManager,
                                  Cell.path( "redundancy/election" ),
                                  "Tac::Dir", "r" )
   electionConfig = LazyMount.mount( entityManager,
                                     Cell.path( "redundancy/election/config" ),
                                     "Election::Config", "wf" )
   electionStatus = LazyMount.mount( entityManager,
                                     Cell.path( 'redundancy/election/status' ),
                                     'Election::Status', 'r' )
   replicationStatus = LazyMount.mount( entityManager,
                                        'redundancy/fileReplication/status',
                                        'FileReplication::Status', 'r' )
   switchoverLog = LazyMount.mount( entityManager,
                                    Cell.path( 'stage/switchover/log' ),
                                    'Stage::Log', 'r' )
   redundancySyncStatus = LazyMount.mount( entityManager,
                                    Cell.path( 'redundancy/timeSync/status' ),
                                    'Election::RedundancyTimeSyncStatus', 'r' ) 
   sysdbStatus = LazyMount.mount( entityManager, 'Sysdb/status', 'Sysdb::Status', 
                                  'r' )
   powerFuse = LazyMount.mount( entityManager, 'power/fuse/status/all',
                                'Power::SoftwareFuse', 'r' )
   pcieSsoStatus = LazyMount.mount(
      entityManager,
      'hardware/pcieSwitch/plx/ssoStatus',
      'Hardware::PcieSwitch::PcieSsoStatusAggregatorState', 'r' )
   modularInventory = LazyMount.mount( entityManager,
                                       'hardware/inventory/modularSystem',
                                       'Inventory::ModularSystem', 'ri' )
   ssoSliceStatus = LazyMount.mount(
      entityManager,
      'hardware/pcieSwitch/plx/ssoSliceStatus',
      'Tac::Dir',
      'ri' )

   agentSsoStatus = LazyMount.mount( entityManager,
                                    'redundancy/agentSsoStatus',
                                    'Redundancy::AgentSsoStatus', 'r' )
