# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliCommand, CliMatcher, CliParser
import IntfCli, LagIntfCli, VirtualIntfRule
import EbraEthIntfCli
import Intf.IntfRange as IntfRange
import LazyMount, ConfigMount
from IntfRangePlugin.RecircIntf import RecircAutoIntfType
from LagCli import recircGuard, isRecirc
from RecircCli import isRecirc
import Tac
import VlanCli
import CliExtensions

lagIntfConfigDir = None
lagIntfStatusDir = None
lagIntfCounterDir = None
recircFeatureStatus = None
recircHwConfig = None
intHwCapability = None

PortChannelNum = Tac.Type( "Lag::PortChannelNum" )
RecircFeature = Tac.Type( "Lag::Recirc::RecircFeature" )
RecircFeatureSet = Tac.Type( "Lag::Recirc::RecircFeatureSet" )

intMirrorSessionHook = CliExtensions.CliHook()

#-------------------------------------------------------------------------------
# A subclass of the EthIntf class for Recirculation interfaces.
# Recirculation interfaces use the existing Lag infrastructure. 
# All configurations and status will exist in the same directories.
#-------------------------------------------------------------------------------
class EthRecircIntf( LagIntfCli.EthLagIntf ):
   
   def createShortname( self ):
      return 'Re%d' % self.lagId

   #----------------------------------------------------------------------------
   # Deletes the Interface::EthLagIntfConfig and Interface::EthRecircIntfConfig 
   # objects for this interface
   #----------------------------------------------------------------------------
   def destroyPhysical( self ):
      lag = self.ethIntfConfigDir.intfConfig.get( self.name )
      lagId = None
      if lag:
         lagId = lag.intfId
         _noSetSwitchportRecircFeatures( lagId )
         del self.ethIntfConfigDir.intfConfig[ self.name ]
      # Make sure no phy intfs point to this port-channel (or any other 
      # obsolete channel)
      self.cleanEthPhyIntfLagConfigs( lagId )

   def setDefault( self ):
      elic = self.ethIntfConfigDir.intfConfig[ self.name ]
      elic.recircFeature.clear() 
      super( EthRecircIntf, self ).setDefault() 

   #----------------------------------------------------------------------------
   # Returns an unsorted list of RecircEthIntf objects representing all the existing
   # Interface::LagEthIntfStatus objects.
   #----------------------------------------------------------------------------
   @staticmethod
   def getAllPhysical( mode ):
      intfs = []
      for name in lagIntfStatusDir.intfStatus.keys():
         if not isRecirc( name ):
            continue
         intf = EthRecircIntf( name, mode )
         if intf.lookupPhysical():
            intfs.append( intf )
      return intfs

   def vrfSupported( self ):
      return False
   
   #----------------------------------------------------------------------------
   # Is this port eligible for "switchport" commands?
   #----------------------------------------------------------------------------
   # def switchportEligible( self ):
   #    return False 

   #----------------------------------------------------------------------------
   # The rule for matching Recirc interface names.  When this pattern matches, it
   # returns an instance of the EthRecircIntf class.
   #
   # This rule gets added to the Intf.rule when this class is registered with
   # the Intf class by calling Intf.addPhysicalIntfType, below.
   #----------------------------------------------------------------------------
   matcher = VirtualIntfRule.VirtualIntfMatcher( 'Recirc-Channel',
                                       PortChannelNum.min,
                                       PortChannelNum.max,
                                       helpdesc="Recirc-Channel Interface",
                                       value=lambda mode, intf:
                                       EthRecircIntf( intf, mode ),
                                       guard=recircGuard )

#-------------------------------------------------------------------------------
# Register the EthRecircIntf class as a type of physical interface.
#-------------------------------------------------------------------------------
IntfCli.Intf.addPhysicalIntfType( EthRecircIntf, 
                                  RecircAutoIntfType,
                                  withIpSupport=False,
                                  withRoutingProtoSupport=False )

EbraEthIntfCli.switchPortMatcher |= EthRecircIntf.matcher

IntfRange.registerIntfTypeGuard( RecircAutoIntfType, recircGuard )

#-------------------------------------------------------------------------------
# Enable Recirc-specific commands to be added to "config-if" mode.
#-------------------------------------------------------------------------------
class RecircIntfConfigModelet( CliParser.Modelet ):
   
   modeletParseTree = CliParser.ModeletParseTree()
   
   def __init__( self, mode ):   # pylint: disable-msg=W0231
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return isinstance( mode.intf, EthRecircIntf )

IntfCli.IntfConfigMode.addModelet( RecircIntfConfigModelet )

# ------------------------------------------------------------------------
# '[no|default] switchport recirculation features [vxlan|( telemetry inband )
#  |openflow]'
# ------------------------------------------------------------------------
recircFeaturesMap = {
      'vxlan' : 1,
      'telemetryinband' : 2,
      'openflow' : 4,
      'cpumirror' : 8,
}

def telemetryRecircGuard( mode, token ):
   if intHwCapability.supportedIntModes:
      return None
   return CliParser.guardNotThisPlatform

def vxlanRecircGuard( mode, token ):
   if recircHwConfig.recircRequired:
      return None
   return CliParser.guardNotThisPlatform

def openflowRecircGuard( mode, token ):
   if recircHwConfig.openFlowMultiTableRecirc:
      return None
   return CliParser.guardNotThisPlatform

def cpumirrorRecircGuard( mode, token ):
   if recircHwConfig.txMirrorToCpuRecircRequired:
      return None
   return CliParser.guardNotThisPlatform

def _noSetSwitchportRecircFeatures( lagName, featuresVal=None ):
   elic = lagIntfConfigDir.intfConfig[ lagName ]
   if featuresVal:
      del elic.recircFeature[ featuresVal ]
   else:
      elic.recircFeature.clear()

def handleIntMirrorSession( mode, intf ):
   if len( intMirrorSessionHook.extensions() ):
      for hook in intMirrorSessionHook.extensions():
         hook( mode, intf )

class RecircSwitchportCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport recirculation features ' \
            '( vxlan | ( telemetry inband ) | openflow | cpu-mirror )'
   noOrDefaultSyntax = 'switchport recirculation features ' \
                       '[ ( vxlan | ( telemetry inband ) | openflow | cpu-mirror ) ]'

   data = {
      'switchport' : 'Set switching mode characteristics',
      'recirculation' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'recirculation', 
            helpdesc='Set recirc mode characteristics of the interface' ),
         guard=recircGuard ),
      'features' : 'Set the features that will use this port for recirculation',
      'vxlan' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'vxlan', 
            helpdesc='VXLAN routing feature' ),
         guard=vxlanRecircGuard ),
      'telemetry' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'telemetry', 
            helpdesc='Inband telemetry feature' ),
         guard=telemetryRecircGuard ),
      'inband' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'inband', 
            helpdesc='Inband telemetry feature' ),
         guard=telemetryRecircGuard ),
      'openflow' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'openflow', 
            helpdesc='OpenFlow multi-table feature' ),
         guard=openflowRecircGuard ),
      'cpu-mirror' : CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher(
            'cpu-mirror',
            helpdesc='Mirror to CPU feature'),
         guard=cpumirrorRecircGuard ),
   }
   @staticmethod
   def handler( mode, args ):
      if 'vxlan' in args:
         featuresVal = recircFeaturesMap[ 'vxlan' ]
      elif 'inband' in args:
         featuresVal = recircFeaturesMap[ 'telemetryinband' ]
      elif 'openflow' in args:
         featuresVal = recircFeaturesMap[ 'openflow' ]
      elif 'cpu-mirror' in args:
         featuresVal = recircFeaturesMap[ 'cpumirror' ]
      else:
         assert 0

      intfRange = mode.intfRange()
      if intfRange is not None and len( intfRange.individualIntfModes_ ) > 0:
         if intfRange.individualIntfModes_[0] is not mode:
            error = ( 'Unable to set feature in interface range mode. Feature can ' 
                      'only be configured on one Recirc-Channel at a time.' )
            mode.addError( error )
         return

      elic = lagIntfConfigDir.intfConfig[ mode.intf.name ]
      # Feature is already set on this lag.
      if featuresVal in elic.recircFeature:
         return

      # Check if this feature is already tied to another recirc-channel
      if featuresVal in recircFeatureStatus.featureToLag:
         lag = recircFeatureStatus.featureToLag[ featuresVal ]
         if lag != mode.intf.name:
            error = 'Feature %s is already part of %s. Unable to apply to %s.' % ( 
                     featuresVal, lag, mode.intf.name )
            mode.addError( error )
            return

      featureBmp = 0 
      for feat in elic.recircFeature.keys():
         featureBmp |= recircFeaturesMap[ feat ] 

      # There is already at least 1 feature specified for the given recirc lag
      if len( elic.recircFeature ) > 0:
         compatibleBmp = recircHwConfig.compatibleFeatures[ featuresVal ]
         if ( featureBmp | compatibleBmp ) != compatibleBmp:
            errmsg = 'Feature %s is not compatible with current set %s' % \
               ( featuresVal, ' '.join( elic.recircFeature ) )
            mode.addError( errmsg )
            return
  
      sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
      sic.enabled = False
      elic.recircFeature[ featuresVal ] = True 
      # Attempt to create an internal inband telemetry mirror
      # session when a recirc channel is associated with the feature.
      # This must be gated through HW capabilities for future
      # implementations that don't require such an operation.
      if ( intHwCapability.supportedIntModes or
           ( not intHwCapability.initialized and
             not mode.session_.guardsEnabled() ) ):
         handleIntMirrorSession( mode, mode.intf.name )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      featuresVal = args.get( 'VAL' )
      _noSetSwitchportRecircFeatures( mode.intf.name, featuresVal )
      sic = VlanCli.getSwitchIntfConfigEvenIfDisabled( mode )
      sic.enabled = True 
      # Delete the internal inband telemetry mirror session
      # when no recirc channel is associated with the feature.
      if ( intHwCapability.supportedIntModes or
           ( not intHwCapability.initialized and
             not mode.session_.guardsEnabled() ) ):
         handleIntMirrorSession( mode, mode.intf.name )

RecircIntfConfigModelet.addCommandClass( RecircSwitchportCmd )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global lagIntfConfigDir, lagIntfStatusDir
   global recircFeatureStatus, recircHwConfig
   global intHwCapability

   lagIntfConfigDir = ConfigMount.mount( entityManager, "interface/config/eth/lag",
                                         "Interface::EthLagIntfConfigDir", "w" )
   lagIntfStatusDir = LazyMount.mount( entityManager, "interface/status/eth/lag",
                                       "Interface::EthLagIntfStatusDir", "r" )
   recircFeatureStatus = LazyMount.mount( entityManager, "lag/recirc/status",
                                            "Lag::Recirc::RecircFeatureStatus", "r" )
   recircHwConfig = LazyMount.mount( entityManager, "lag/recirc/hwconfig", 
                                     "Lag::Recirc::HwConfig", "r" )
   intHwCapability = LazyMount.mount( entityManager, "inbandtelemetry/hwCapability",
                                   "InbandTelemetry::HwCapability", "r" )
