# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import tempfile

import BasicCli
import BasicCliUtil
import Cell
import CliCommand
import CliGlobal
import CliMatcher
import CliParser
import CliPlugin.RcfCliLib as RcfCliLib
from CliPlugin.RouterGeneralCli import (
      RouterGeneralMode,
      routerGeneralCleanupHook,
   )
import CliSession
import CliToken
import ConfigMount
import ShowCommand
import Tac
import Tracing
import Url
from CliMode.Rcf import ControlFunctionsBaseMode
from RcfCliModels import (
      ShowRcfPendingModel,
      ShowRcfErrorsStartupModel,
   )
from RcfCompiler import (
      RcfCompiler,
      RcfCompileRequest,
   )
from RcfConfigReactor import RcfConfigReactor
from RcfLinter import (
      RcfLinter,
      RcfLintRequest,
)
from Toggles.RcfLibToggleLib import toggleRcfAgentEnabled

traceHandle = Tracing.Handle( 'RcfCli' )
t0 = traceHandle.trace0

MAX_U32 = 0xFFFFFFFF

gv = CliGlobal.CliGlobal( dict(
      rcfAllFunctionStatus=None,
      rcfConfig=None,
      # This variable always holds the global Sysdb copy of the rcfConfig,
      # while the ConfigMounted rcfConfig will hold a session copy of the
      # Sysdb config when used within a config session
      rcfConfigSysdb=None,
      rcfCompiler=None,
      rcfLinter=None,
      rcfStatus=None,
      # TODO, when enabling toggleRcfAgentEnabled,
      # rename RcfStatusCi -> RcfStatus, assume that RcfStatus read from Cli.
      rcfStatusCi=None,  # RcfStatus written by Rcf Agent (CI)
      redundancyStatus=None,
      redundancyStatusReactor=None,
   )
)

#

#----------------------------------------------------------------------------------
#                                H E L P E R S
#----------------------------------------------------------------------------------

# There are two cases when parsing the config: manually entering the command
# or reading the startup config. The latter comes with indentation of 3 spaces
# for every sub config mode, 6 spaces in our case. We need to preserve the
# original indentation of the RCF code. If all lines of the input start with at
# least 6 spaces, we prune the first 6 spaces of each line before proceeding.
# BUG416112: Move removeStartupConfigIndentation() to BasicCliUtils.py
def removeStartupConfigIndentation( multiLineRcfInput, indentLen ):
   # First, check if each line of the input starts with indentLen number of spaces.
   # If it doesn't, this multiLineRcfInput did *not* come from startup config and we
   # should not make any changes to it.
   indent = " " * indentLen
   lines = multiLineRcfInput.split( "\n" )
   for line in lines[ : -1 ]: # The last line will be an empty string after split()
      if line[ : indentLen ] != indent and line != '':
         return multiLineRcfInput
   # If we've gotten this far, we know all lines of the input start with inputLen
   # number of spaces. We prune the extraenous spaces, concatenate the lines back
   # into a single string, and return it back to the caller.
   modifiedInputLines = []
   for line in lines:
      if line == '':
         modifiedInputLines.append( line )
      else:
         modifiedInputLines.append( line[ indentLen : ] )
   return "\n".join( modifiedInputLines )

class RcfScratchpad( object ):
   """Temporary scratchpad for holding RCF data before it is published"""
   def __init__( self ):
      self.rcfText = None
      self.rcfCompileResult = None
      self.rcfLintResult = None
      self.editSincePull = False

   def hasVerifiedCode( self ):
      """ Whether the code held in this scratchpad is valid and ready to be
      committed to config.
      """
      return self.result and self.result.success

   @property
   def result( self ):
      if toggleRcfAgentEnabled():
         return self.rcfLintResult
      else:
         return self.rcfCompileResult

   @property
   def errorList( self ):
      assert self.result
      return self.result.errorList

   @property
   def warningList( self ):
      assert self.result
      return self.result.warningList

   def isEmpty( self ):
      return self.rcfText == ""

def getSessionAttribute( mode, attr ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( attr )
      return csAttribute.get( sessionName ) if csAttribute else None
   else:
      return mode.session.sessionData( attr )

def initializeSessionAttribute( mode, attr, initValue ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( attr )
      if csAttribute is None:
         mode.session.sessionDataIs( attr, {} )
         csAttribute = mode.session.sessionData( attr )
      if sessionName not in csAttribute:
         csAttribute[ sessionName ] = initValue
      attribute = csAttribute[ sessionName ]
   else:
      attribute = mode.session.sessionData( attr )
      if attribute is None:
         mode.session.sessionDataIs( attr, initValue )
         attribute = mode.session.sessionData( attr )
   return attribute

def removeSessionAttribute( mode, attr ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( attr )
      if csAttribute and sessionName in csAttribute:
         del csAttribute[ sessionName ]
   else:
      mode.session.sessionDataIs( attr, None )

def getScratchpadAttr( mode ):
   if mode.session.inConfigSession():
      return 'rcfConfigSessionScratchpads'
   else:
      return 'rcfScratchpad'

def getScratchpad( mode ):
   return getSessionAttribute( mode, getScratchpadAttr( mode ) )

def initializeScratchpad( mode ):
   return initializeSessionAttribute( mode, getScratchpadAttr( mode ),
                                      RcfScratchpad() )

def removeScratchpad( mode ):
   removeSessionAttribute( mode, getScratchpadAttr( mode ) )

def getPullUrlAttr( mode ):
   if mode.session.inConfigSession():
      return 'rcfConfigSessionPullUrls'
   else:
      return 'rcfPullUrl'

def getPullUrl( mode ):
   return getSessionAttribute( mode, getPullUrlAttr( mode ) )

def initializePullUrl( mode ):
   return initializeSessionAttribute( mode, getPullUrlAttr( mode ), '' )

def setPullUrl( mode, url ):
   if mode.session.inConfigSession():
      em = mode.session_.entityManager_
      sessionName = CliSession.currentSession( em )
      csAttribute = mode.session.sessionData( getPullUrlAttr( mode ) )
      csAttribute[ sessionName ] = url
   else:
      mode.session.sessionDataIs( getPullUrlAttr( mode ), url )

def removePullUrl( mode ):
   removeSessionAttribute( mode, getPullUrlAttr( mode ) )

def rcfCleanup( mode ):
   if gv.rcfConfig and gv.rcfConfig.rcfText != gv.rcfConfig.rcfTextDefault:
      prompt = 'Remove all routing control functions? [Y/n] '
      if BasicCliUtil.confirm( mode, prompt ):
         gv.rcfConfig.rcfText = gv.rcfConfig.rcfTextDefault
         gv.rcfConfig.enabled = False
         if not toggleRcfAgentEnabled():
            gv.rcfStatus.active = False
            gv.rcfStatus.functionNames.clear()
            gv.rcfAllFunctionStatus.aet.clear()

routerGeneralCleanupHook.addExtension( rcfCleanup )

#----------------------------------------------------------------------------------
#                                K E Y W O R D S
#----------------------------------------------------------------------------------

controlFunctionsKwForConfig = CliMatcher.KeywordMatcher( 'control-functions',
   helpdesc="Routing control functions configuration" )
routerKw = CliToken.Router.routerMatcherForConfig

codeKw = CliMatcher.KeywordMatcher( 'code',
   helpdesc="Routing control function code" )
pullKw = CliMatcher.KeywordMatcher( 'pull',
   helpdesc="Pull routing control function code" )
replaceKw = CliMatcher.KeywordMatcher( 'replace',
   helpdesc="Replace pending routing control function code" )
pushKw = CliMatcher.KeywordMatcher( 'push',
      helpdesc="Push routing control function code" )
pendingKw = CliMatcher.KeywordMatcher( 'pending',
   helpdesc="Pending routing control function code" )
runningConfigKw = CliMatcher.KeywordMatcher( 'running-config',
   helpdesc="Routing control function code from running-config" )
sourceKw = CliMatcher.KeywordMatcher( 'source',
   helpdesc="Routing control function code source" )
pulledFromKw = CliMatcher.KeywordMatcher( 'pulled-from',
   helpdesc="Source file URL of last pull" )
editedKw = CliMatcher.KeywordMatcher( 'edited',
   helpdesc="Routing control function code was edited since the last pull" )
urlMatcherSource = Url.UrlMatcher( lambda fs: fs.supportsRead(),
   'Source file URL' )
urlMatcherPush = Url.UrlMatcher( lambda fs: fs.supportsWrite(),
   'Destination file URL' )
discardKw = CliMatcher.KeywordMatcher( 'discard',
   helpdesc="Discard pending routing control function code" )
compileKw = CliMatcher.KeywordMatcher( 'compile',
   helpdesc="Compile pending routing control function code" )
commitKw = CliMatcher.KeywordMatcher( 'commit',
   helpdesc="Commit pending routing control function code" )
errorsKw = CliMatcher.KeywordMatcher( 'errors',
   helpdesc="Compilation errors" )
startupKw = CliMatcher.KeywordMatcher( 'startup',
   helpdesc="Compilation errors encountered during system startup" )
urlKw = CliMatcher.KeywordMatcher( 'url',
   helpdesc="Source file URL last pulled from" )

#----------------------------------------------------------------------------------
#                                  M O D E S
#----------------------------------------------------------------------------------

class ControlFunctionsMode( ControlFunctionsBaseMode, BasicCli.ConfigModeBase ):
   """CLI mode for interacting with routing control functions."""
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      ControlFunctionsBaseMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      if getScratchpad( self ):
         self.addWarning( 'There are pending routing control function changes' )
         self.addWarning( 'These can be accessed by going back into '
                          'control-functions mode' )

#----------------------------------------------------------------------------------
#                               C O M M A N D S
#----------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# "[ no | default ] control-functions" in router general mode
#--------------------------------------------------------------------------------
class ControlFunctionsCmd( CliCommand.CliCommandClass ):
   syntax = 'control-functions'
   noOrDefaultSyntax = syntax
   data = {
      'control-functions': controlFunctionsKwForConfig,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( ControlFunctionsMode )
      mode.session_.gotoChildMode( childMode )
      if getScratchpad( mode ):
         mode.addWarning( "There are pending routing control function changes" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      rcfCleanup( mode )

RouterGeneralMode.addCommandClass( ControlFunctionsCmd )

def isStartupConfig( mode ):
   return mode.session_.startupConfig()

def compileRcfText( rcfText, strictMode ):
   compileRequest = RcfCompileRequest( rcfText, strictMode=strictMode )
   assert gv.rcfCompiler
   return gv.rcfCompiler.compile( compileRequest )

def lintRcfText( rcfText, strictMode ):
   lintRequest = RcfLintRequest( rcfText, strictMode=strictMode )
   assert gv.rcfLinter
   return gv.rcfLinter.lint( lintRequest )

def commitRcfText( rcfConfigReference, rcfText, rcfResult, rcfUrl,
                   editSincePull ):
   if not rcfResult.success:
      return
   rcfTextToCommit = removeStartupConfigIndentation( rcfText, indentLen=6 )
   # RcfConfigReactor will react to this and publish the AETs if this is run
   # outside a config-session and Sysdb's rcfText is updated
   rcfConfigReference.rcfText = rcfTextToCommit
   if toggleRcfAgentEnabled():
      rcfConfigReference.enabled = True # start Rcf Agent
   if rcfUrl:
      rcfConfigReference.lastPulledUrl = rcfUrl.url
      rcfConfigReference.editSincePull = editSincePull
   else:
      rcfConfigReference.lastPulledUrl = rcfConfigReference.lastPulledUrlDefault
      rcfConfigReference.editSincePull = False

def addCompilationStatusToMode( mode, result ):
   warnings = result.warningList or []
   for warning in warnings:
      mode.addWarning( warning )

   if result.success:
      mode.addMessage( "Compilation successful" )
   else:
      addWarningForErrors = ( not mode.session_.isInteractive() and
                              mode.session.inConfigSession() )
      addToModeFn = mode.addWarning if addWarningForErrors else mode.addError
      for error in result.errorList:
         addToModeFn( error )

def compileRcfHelper( mode ):
   scratchpad = getScratchpad( mode )
   if scratchpad is None:
      mode.addWarning(
         'There are no pending routing control function changes to compile' )
   else:
      scratchpad.rcfCompileResult = compileRcfText( scratchpad.rcfText,
                                                    mode.session_.isInteractive() )
      addCompilationStatusToMode( mode, scratchpad.rcfCompileResult )

def lintRcfHelper( mode ):
   scratchpad = getScratchpad( mode )
   if scratchpad is None:
      mode.addWarning(
         'There are no pending routing control function changes to compile' )
   else:
      scratchpad.rcfLintResult = lintRcfText( scratchpad.rcfText,
                                              mode.session_.isInteractive() )
      addCompilationStatusToMode( mode, scratchpad.rcfLintResult )

def ciLintOrCompileRcfHelper( mode ):
   """ Just a continuous integration wrapper to either call
   lint or compile based on whether RcfAgent toggle is enabled
   or not.

   When removing the toggle, we should replace all calls to
   clLintOrCompileRcfHelper to lintRcfHelper.
   """
   if toggleRcfAgentEnabled():
      lintRcfHelper( mode )
   else:
      compileRcfHelper( mode )

def commitRcfHelper( mode ):
   scratchpad = getScratchpad( mode )
   url = getPullUrl( mode )
   if scratchpad is None:
      mode.addWarning(
         'There are no pending routing control function changes to commit' )
   elif not scratchpad.hasVerifiedCode():
      mode.addWarning(
         'Pending routing control function changes need to be compiled successfully '
         'before committing' )
      if isStartupConfig( mode ):
         assert scratchpad.result # we auto-compile during startup config
         # save error to rcf status
         errorStr = "\n".join( scratchpad.errorList )
         gv.rcfConfig.rcfText = scratchpad.rcfText
         gv.rcfConfig.enabled = True
         if not toggleRcfAgentEnabled():
            gv.rcfStatus.lastStartupCompilationError = errorStr
      elif not mode.session_.isInteractive() and\
           mode.session.inConfigSession() and\
           gv.rcfStatus.active:
         assert scratchpad.result # we auto-compile during config-session
         # preserve Sysdb RCF code if compile fails during config replace
         gv.rcfConfig.rcfText = gv.rcfConfigSysdb.rcfText
         gv.rcfConfig.enabled = True
   elif scratchpad.hasVerifiedCode():
      # save RCF text and functionNames to Sysdb since compile worked
      commitRcfText( gv.rcfConfig, scratchpad.rcfText, scratchpad.result,
                     url, scratchpad.editSincePull )
      scratchpad.rcfText = None
      scratchpad.rcfCompileResult = None
      scratchpad.rcfLintResult = None
      removeScratchpad( mode )
      removePullUrl( mode )

#--------------------------------------------------------------------------------
# "[ no | default ] code" in control-functions mode
#--------------------------------------------------------------------------------
class CodeCmd( CliCommand.CliCommandClass ):
   syntax = 'code'
   noOrDefaultSyntax = syntax
   data = {
      'code': codeKw,
   }

   @staticmethod
   def handler( mode, args ):
      multiLineRcfInput = BasicCliUtil.getMultiLineInput( mode, cmd="code",
                                                          prompt="Enter RCF code" )
      # save RCF text to scratchpad before compilation
      scratchpad = initializeScratchpad( mode )
      scratchpad.rcfText = multiLineRcfInput
      scratchpad.rcfCompileResult = None
      scratchpad.rcfLintResult = None
      scratchpad.editSincePull = True
      if not mode.session_.isInteractive():
         ciLintOrCompileRcfHelper( mode )
         commitRcfHelper( mode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      # Makes the scratchpad empty but present. Can be compiled and committed to
      # clear the AETs and RCF config, unlike 'discard', which deletes the scratchpad
      scratchpad = getScratchpad( mode )
      if scratchpad is None:
         scratchpad = initializeScratchpad( mode )
      scratchpad.rcfText = ''
      scratchpad.rcfCompileResult = None
      scratchpad.rcfLintResult = None
      scratchpad.editSincePull = True

ControlFunctionsMode.addCommandClass( CodeCmd )

#--------------------------------------------------------------------------------
# "pull" in control-functions mode
#--------------------------------------------------------------------------------
class PullCmd( CliCommand.CliCommandClass ):
   syntax = 'pull replace URL'
   data = {
      'pull': pullKw,
      'replace': replaceKw,
      'URL': urlMatcherSource,
   }

   @staticmethod
   def handler( mode, args ):
      url = args[ 'URL' ]
      # populate scratchpad with contents from file
      filename = tempfile.mktemp( prefix="rcfText-" )
      pullError = False
      with open( filename, 'w' ):
         try:
            url.get( filename )
            scratchpad = initializeScratchpad( mode )
            with open( filename, 'r' ) as rcfFile:
               scratchpad.rcfText = rcfFile.read()
            scratchpad.rcfCompileResult = None
            scratchpad.rcfLintResult = None
            scratchpad.editSincePull = False
         except EnvironmentError as e:
            pullError = True
            if e.filename:
               e.filename = Url.filenameToUrl( e.filename )
            mode.addError( e )
      os.remove( filename )
      if not pullError:
         initializePullUrl( mode )
         setPullUrl( mode, url )
         ciLintOrCompileRcfHelper( mode )

ControlFunctionsMode.addCommandClass( PullCmd )

#--------------------------------------------------------------------------------
# "push" in control-functions mode
#--------------------------------------------------------------------------------
class PushCmd( CliCommand.CliCommandClass ):
   syntax = 'push [ pending | running-config ] URL'
   data = {
      'push': pushKw,
      'pending': pendingKw,
      'running-config': runningConfigKw,
      'URL': urlMatcherPush,
   }

   @staticmethod
   def handler( mode, args ):
      rcfText = ''
      scratchpad = getScratchpad( mode )
      if 'running-config' in args or not scratchpad:
         rcfText = gv.rcfConfig.rcfText
      else:
         rcfText = scratchpad.rcfText
      url = args[ 'URL' ]
      filename = None
      # place contents from either scratchpad or config to file
      filename = tempfile.mktemp( prefix="rcfText-" )
      with open( filename, 'w' ) as rcfFile:
         rcfFile.write( rcfText )
      try:
         url.put( filename )
      except EnvironmentError as e:
         if e.filename:
            e.filename = Url.filenameToUrl( e.filename )
         mode.addError( e )
      os.remove( filename )

ControlFunctionsMode.addCommandClass( PushCmd )

#--------------------------------------------------------------------------------
# "discard" in control-functions mode
#--------------------------------------------------------------------------------
class DiscardCmd( CliCommand.CliCommandClass ):
   syntax = 'discard'
   data = {
      'discard': discardKw,
   }

   @staticmethod
   def handler( mode, args ):
      removeScratchpad( mode )
      removePullUrl( mode )

ControlFunctionsMode.addCommandClass( DiscardCmd )

#--------------------------------------------------------------------------------
# "compile" in control-functions mode
#--------------------------------------------------------------------------------
class CompileCmd( CliCommand.CliCommandClass ):
   syntax = 'compile'
   data = {
      'compile': compileKw,
   }

   @staticmethod
   def handler( mode, args ):
      ciLintOrCompileRcfHelper( mode )

ControlFunctionsMode.addCommandClass( CompileCmd )

#--------------------------------------------------------------------------------
# "commit" in control-functions mode
#--------------------------------------------------------------------------------
class CommitCmd( CliCommand.CliCommandClass ):
   syntax = 'commit'
   data = {
      'commit': commitKw,
   }

   @staticmethod
   def handler( mode, args ):
      commitRcfHelper( mode )

ControlFunctionsMode.addCommandClass( CommitCmd )

#--------------------------------------------------------------------------------
# "code source pulled-from URL" in control-functions mode
#--------------------------------------------------------------------------------
class CodeSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'code source pulled-from URL [ edited ]'
   noOrDefaultSyntax = syntax
   data = {
      'code': codeKw,
      'source': sourceKw,
      'pulled-from': pulledFromKw,
      'URL': urlMatcherSource,
      'edited': editedKw,
   }

   @staticmethod
   def handler( mode, args ):
      if mode.session_.isInteractive():
         mode.addError( "This command is not supported in interactive mode" )
      else:
         url = args[ 'URL' ]
         initializePullUrl( mode )
         setPullUrl( mode, url )
      gv.rcfConfig.editSincePull = 'edited' in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      gv.rcfConfig.lastPulledUrl = gv.rcfConfig.lastPulledUrlDefault
      gv.rcfConfig.editSincePull = False

ControlFunctionsMode.addCommandClass( CodeSourceCmd )

#--------------------------------------------------------------------------------
# "show router rcf pending"
#--------------------------------------------------------------------------------
class ShowRcfPendingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router rcf pending'
   data = {
      'router': routerKw,
      'rcf': RcfCliLib.rcfKw,
      'pending': pendingKw,
   }

   cliModel = ShowRcfPendingModel

   @staticmethod
   def handler( mode, args ):
      rcfPendingModel = ShowRcfPendingModel()
      scratchpad = getScratchpad( mode )
      url = getPullUrl( mode )
      if scratchpad:
         rcfPendingModel.pendingChanges = True
         rcfPendingModel.rcfCode = scratchpad.rcfText
      else:
         rcfPendingModel.pendingChanges = False
      if url:
         rcfPendingModel.url = url.url
      return rcfPendingModel

BasicCli.addShowCommandClass( ShowRcfPendingCmd )

#--------------------------------------------------------------------------------
# "show router rcf errors startup"
#--------------------------------------------------------------------------------
class ShowRcfErrorsStartupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show router rcf errors startup'
   data = {
      'router': routerKw,
      'rcf': RcfCliLib.rcfKw,
      'errors': errorsKw,
      'startup': startupKw,
   }

   cliModel = ShowRcfErrorsStartupModel

   @staticmethod
   def handler( mode, args ):
      rcfErrorsModel = ShowRcfErrorsStartupModel()
      rcfErrorsModel.compilationErrors = \
            gv.rcfStatus.lastStartupCompilationError.split( "\n" )
      rcfErrorsModel.active = gv.rcfStatus.active
      return rcfErrorsModel

BasicCli.addShowCommandClass( ShowRcfErrorsStartupCmd )

class RedundancyStatusReactor( Tac.Notifiee ):
   """Reacts to redundancy/status.mode changing. This is needed to
   bump up the agent2AgentmountChangedCounter when the supervisor
   becomes the active supervisor.
   The bump up is done by blindly invoking the callback function.
   """
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redStatusEntity, callback ):
      Tac.Notifiee.__init__( self, redStatusEntity )
      self.callback = callback

   @Tac.handler( "mode" )
   def handleRedundancyMode( self ):
      self.callback()

def PluginCompile( entMan ):
   def initRcf():
      # redundancyStatus got mounted. If a reactor for this is not created yet,
      # create one now.
      if not gv.redundancyStatusReactor:
         gv.redundancyStatusReactor = RedundancyStatusReactor( gv.redundancyStatus,
                                                               initRcf )

      # In a dual-sup situation, only proceed if we are the active supervisor
      if gv.redundancyStatus.mode != "active":
         return

      # If the mount changed counter is non-zero, ConfigAgent has restarted. Since
      # the RCF AETs are not in Sysdb, they do not survive a ConfigAgent restart.
      # Therefore, if there is any RCF config in Sysdb, we have to recompile and
      # recommit the AETs by triggering the python reactor.
      if not hasattr( entMan, "rcfConfigReactor" ):
         entMan.rcfConfigReactor = RcfConfigReactor( gv.rcfConfigSysdb )

      if gv.rcfStatus.agent2AgentMountChangedCounter != 0 and\
          gv.rcfConfigSysdb.rcfText != gv.rcfConfigSysdb.rcfTextDefault:
         entMan.rcfConfigReactor.handleRcfTextChange()

      # At this point, we know /<sysname>/Agent2Agent/RcfAllFunctionStatus has been
      # created in ConfigAgent, so we bump up the counter in rcfStatus to kick ArBgp.
      if gv.rcfStatus.agent2AgentMountChangedCounter == MAX_U32:
         gv.rcfStatus.agent2AgentMountChangedCounter = 1
      else:
         gv.rcfStatus.agent2AgentMountChangedCounter += 1
      t0( "RCF agent2AgentMountChangedCounter",
          gv.rcfStatus.agent2AgentMountChangedCounter )

   gv.rcfConfig = ConfigMount.mount( entMan, 'routing/rcf/config',
                                  'Rcf::Config', 'wiS' )
   sysname = "ar"
   try:
      sysname = Tac.sysname()
   except Exception as _: # pylint: disable=broad-except
      pass
   agent2AgentRcfDir = Tac.root[ sysname ].mkdir( 'Agent2AgentRcf' )
   gv.rcfAllFunctionStatus = agent2AgentRcfDir.newEntity( "Rcf::AllFunctionStatus",
                                                          "RcfAllFunctionStatus" )

   mg = entMan.mountGroup()
   # routing/rcf/config and routing/acl/config must be mounted as 'wi'
   # because they are marked as "CONFIG" in preinit files
   gv.rcfConfigSysdb = mg.mount( 'routing/rcf/config', 'Rcf::Config', 'wi' )
   aclConfig = mg.mount( 'routing/acl/config', 'Acl::AclListConfig', 'wi' )
   gv.rcfStatus = mg.mount( 'routing/rcf/status', 'Rcf::Status', 'w' )
   gv.rcfStatusCi = mg.mount( 'routing/rcf/status_ci', 'Rcf::Status', 'r' )
   gv.redundancyStatus = mg.mount( Cell.path( 'redundancy/status' ),
                                'Redundancy::RedundancyStatus', 'r' )
   gv.rcfCompiler = RcfCompiler( aclConfig, gv.rcfAllFunctionStatus )
   mg.close( initRcf )

def PluginLint( entMan ):
   gv.rcfConfig = ConfigMount.mount( entMan, 'routing/rcf/config',
                                  'Rcf::Config', 'wS' )
   mg = entMan.mountGroup()
   gv.rcfConfigSysdb = mg.mount( 'routing/rcf/config', 'Rcf::Config', 'wS' )
   aclConfig = mg.mount( 'routing/acl/config', 'Acl::AclListConfig', 'w' )
   gv.rcfStatusCi = mg.mount( 'routing/rcf/status_ci', 'Rcf::Status', 'r' )
   gv.rcfStatus = gv.rcfStatusCi
   gv.rcfLinter = RcfLinter( aclConfig )

   mg.close( callback=None, blocking=False )

Plugin = PluginLint if toggleRcfAgentEnabled() else PluginCompile
