# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from CliModel import Dict
from CliModel import Int
from CliModel import Model
from CliModel import Str
from CliModel import List
from CliModel import Float
from CliModel import Submodel
from IntfModel import Interface
import Tac
import Ark

class ServerInfo( Model ):
   hostname = Str( help="IP address or hostname of the RADIUS server" )
   authport = Int( help="Specifies server authorization port number" )
   acctport = Int( help="Specifies server accounting port number", optional=True )
   dynAuthPort = Int( help="Specifies server dynamic authorization port number", 
                      optional=True )
   vrf = Str( help="The VRF over which this server is reachable", optional=True )

class RadiusStats( Model ):
   serverInfo = Submodel( valueType=ServerInfo, help="Configured server "
                                                     "information" )
   vrf = Str( help="Obsolete, do not use this", optional=True )
   messagesSent = Int( help="Number of packets sent to the RADIUS server" )
   messagesReceived = Int( help="Number of packets received from the RADIUS "
                                "server" )
   requestsAccepted = Int( help="Number of requests accepted by the RADIUS server" )
   requestsRejected = Int( help="Number of requests rejected by the RADIUS server" )
   requestsTimeout = Int( help="Number of requests to the RADIUS server timed out" )
   requestsRetransmitted = Int( help="Number of requests retransmitted by the "
                                     "RADIUS server" )
   badResponses = Int( help="Number of bad responses from the RADIUS server" )
   connectionErrors = Int( help="Number of connection errors to the RADIUS server" )
   dnsErrors = Int( help="Number of DNS errors by the RADIUS server" )
   coaRequestsReceived = Int( help="Number of CoA requests received from "
                                   "RADIUS server" )
   dmRequestsReceived = Int( help="Number of DM requests received from "
                                  "RADIUS server" )
   coaAckSent = Int( help="Number of CoA Ack responses sent to RADIUS server" )
   dmAckSent = Int( help="Number of DM Ack responses sent to RADIUS server" )
   coaNakSent = Int( help="Number of CoA Nak responses sent to RADIUS server" )
   dmNakSent = Int( help="Number of DM Nak responses sent to RADIUS server" )

class ServerGroup( Model ):
   serverGroup = Str( help="Configured server-group" )
   members = List( valueType=ServerInfo, help="Mapping of server IP address or "
                                              "hostname to group information" )

class ShowRadius( Model ):
   radiusServers = List( valueType=RadiusStats, help="Mapping of server "
         "IP address or hostname to server information" )
   groups = Dict( keyType=str, valueType=ServerGroup, help="Mapping of group name "
         "to group information", optional=True )
   srcIntf = Dict( valueType=Interface, help="Configured source-interface name" )
   lastCounterClearTime = Float( help="UTC time when the counters were "
                                      "last cleared" )

   def _getVrfString( self, vrf ):
      return " (vrf %s)" % vrf if vrf else ""

   def render( self ):
      template = "%26s:%11d"
      for value in self.radiusServers:
         print "RADIUS server             : %s/%d/%d%s" % (
            value.serverInfo.hostname,
            value.serverInfo.authport,
            value.serverInfo.acctport,
            self._getVrfString( value.serverInfo.vrf ) )
         print "Dynamic authorization UDP port: %s" % (
               value.serverInfo.dynAuthPort )
         print template % ( "Messages sent", value.messagesSent )
         print template % ( "Messages received", value.messagesReceived )
         print template % ( "Requests accepted", value.requestsAccepted )
         print template % ( "Requests rejected", value.requestsRejected )
         print template % ( "Requests timeout", value.requestsTimeout )
         print template % ( "Requests retransmitted", value.requestsRetransmitted )
         print template % ( "Bad responses", value.badResponses )
         print template % ( "Connection errors", value.connectionErrors )
         print template % ( "DNS errors", value.dnsErrors )
         print template % ( "CoA request received", value.coaRequestsReceived )
         print template % ( "DM request received", value.dmRequestsReceived )
         print template % ( "CoA ack sent", value.coaAckSent )
         print template % ( "DM ack sent", value.dmAckSent )
         print template % ( "CoA Nak sent", value.coaNakSent )
         print template % ( "DM Nak sent", value.dmNakSent )
         print

      for serverGroupName, radiusGroup in self.groups.iteritems():
         print "%s server-group: %s" % ( radiusGroup.serverGroup,
                                         serverGroupName )
         for sequence, groupDetails in enumerate( radiusGroup.members ):
            print "    %2d: %s/%d%s" % ( sequence, groupDetails.hostname,
                                         groupDetails.authport,
                                         self._getVrfString( groupDetails.vrf ) )
         if len( radiusGroup.members ) == 0:
            print "    (server-group is empty)"
         print

      print "Last time counters were cleared:", \
      Ark.utcTimeRelativeToNowStr( self.lastCounterClearTime )

      if len( self.srcIntf ) > 0:
         print "\nRADIUS source-interface: Enabled"
         print "  RADIUS outgoing packets will be sourced with an IP address " \
               "associated with interface as follows:\n"
         print "\tVRF \tsource-interface"
         print "\t--- \t----------------"
         for vrf in sorted( self.srcIntf ):
            print "\t%s\t %s" % ( vrf, self.srcIntf[ vrf ] )

