# Copyright (c) 2008-2011, 2013-2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements RADIUS configuration.
#
# In enable mode:
#
#     show radius
#     clear aaa counters radius
#
# In config mode:
#
#     [no] radius-server key <key-text>
#     [no] radius-server timeout <1-1000>
#     [no] radius-server deadtime <1-1000>
#     [no] radius-server retransmit <0-100>
#     [no] radius-server host <ip-addr-or-hostname> [vrf <vrf-name>]
#           [auth-port <0-65535>] [acct-port <0-65535>] [timeout <1-1000>]
#           [retransmit <0-100>] [key <key-text>]
#     [no] radius-server attribute <32> include-in-access-req format <nas-id-string>
#     [no] radius-server qos dscp <0-63>
#     [no] ip radius source-interace <interface-name>
#
# Child mode of config mode:
#
#     aaa group server radius <server-group-name>
#        [no] server <ip-addr-or-hostname> [auth-port <0-65535>] \
#            [acct-port <0-65535>]
#
#-------------------------------------------------------------------------------
import AaaCli
import AaaCliLib
import Ark
import BasicCli
import ConfigMount
import CliCommand
import CliMatcher
import CliPlugin.TechSupportCli
from CliPlugin.VrfCli import DEFAULT_VRF, DEFAULT_VRF_OLD, VrfExprFactory
import CliToken.Clear
import CliToken.Ip
import DscpCliLib
import HostnameCli
import IntfCli
import Intf.Log
import LazyMount
import Radius
import RadiusGroup
import RadiusModel
import ReversibleSecretCli
import ShowCommand
import Tac
import Tracing

t0 = Tracing.trace0

# Need to define this here because I can't access definitions in .tac files
# from Python due to missing support for constAttr.
_timeoutDefault = 5
_timeoutMin = 1
_timeoutMax = 1000

_retriesDefault = 3
_retriesMin = 0
_retriesMax = 100

_deadtimeMin = 1
_deadtimeMax = 1000

aaaConfig = None
radiusConfig = None
radiusStatus = None
radiusCounterConfig = None
radiusInputStatus = None
dscpConfig = None

def radiusHost( mode, hostname, vrf, port, acctPort, create=False ):
   hosts = radiusConfig.host
   assert vrf != ""
   spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                     acctPort=acctPort, vrf=vrf )
   if spec in hosts:
      host = hosts[ spec ]
   elif create:
      t0( "Creating host:", hostname, ":", vrf, ":", port, ":", acctPort )
      host = hosts.newMember( spec )
      if host is None:
         t0( "Unable to create Host:", hostname, ":", vrf, ":", port, ":",
             acctPort )
      else:
         host.index = AaaCliLib.getHostIndex( hosts )
   else:
      host = None
   if host is not None:
      assert host.hostname == hostname
      assert host.vrf == vrf
      assert host.port == port
      assert host.acctPort == acctPort
   return host

#-------------------------------------------------------------------------------
# "show radius" in enable mode
#-------------------------------------------------------------------------------

def showRadiusHost( host, statusCounters ):
   ret1 = RadiusModel.RadiusStats()
   ret1.serverInfo = RadiusModel.ServerInfo( hostname=host.hostname,
                                             authport=host.port,
                                             acctport=host.acctPort,
                                             dynAuthPort=radiusConfig.dynAuthPort )
   if host.vrf != DEFAULT_VRF:
      ret1.serverInfo.vrf = host.vrf
   ret1.messagesSent = statusCounters.authnMessagesSent
   ret1.messagesReceived = statusCounters.authnMessagesReceived
   ret1.requestsAccepted = statusCounters.authnAcceptsReceived
   ret1.requestsRejected = statusCounters.authnRejectsReceived
   ret1.requestsTimeout = statusCounters.authnRequestsTimeout
   ret1.requestsRetransmitted = statusCounters.authnRequestsRetransmitted
   ret1.badResponses = statusCounters.authnBadResponses
   ret1.connectionErrors = statusCounters.authnConnectionErrors
   ret1.dnsErrors = statusCounters.authnHostUnresolvable
   ret1.coaRequestsReceived = statusCounters.coaRequestsReceived
   ret1.dmRequestsReceived = statusCounters.dmRequestsReceived
   ret1.coaAckSent = statusCounters.coaAckResponses
   ret1.dmAckSent = statusCounters.dmAckResponses
   ret1.coaNakSent = statusCounters.coaNakResponses
   ret1.dmNakSent = statusCounters.dmNakResponses
   # the following is really bad - should remove
   vrfString = " (vrf %s)" % host.vrf if host.vrf != DEFAULT_VRF else ""
   if vrfString:
      ret1.vrf = vrfString
   return ret1

def showRadius( mode ):
   ret = RadiusModel.ShowRadius()
   for h in sorted( radiusConfig.host.values(), key=lambda host: host.index ):
      ret.radiusServers.append( showRadiusHost( h,
                                                Radius.getHostCounters( h.spec,
                                                radiusStatus,
                                                radiusInputStatus,
                                                useCheckpoint=True ) ) )
   ret.srcIntf = dict( radiusConfig.srcIntfName )
   for k in sorted( aaaConfig.hostgroup.keys() ):
      g = aaaConfig.hostgroup[ k ]
      if g.groupType == 'radius':
         serverGroupDisplay = AaaCliLib.getCliDisplayFromGroup( g.groupType )
         serverGroupName = g.name
         ret.groups[ serverGroupName ] = RadiusModel.ServerGroup()
         ret.groups[ serverGroupName ].serverGroup = serverGroupDisplay
         for _, m in g.member.iteritems():
            ret2 = RadiusModel.ServerInfo()
            ret2.hostname = m.spec.hostname
            ret2.authport = m.spec.port
            if m.spec.vrf != DEFAULT_VRF :
               ret2.vrf = m.spec.vrf
            ret.groups[ serverGroupName ].members.append( ret2 )
   ret.lastCounterClearTime = Ark.switchTimeToUtc( radiusStatus.lastClearTime )
   return ret

class ShowRadiusCommand( ShowCommand.ShowCliCommandClass ):
   syntax = "show radius"
   data = { "radius" : 'Details of RADIUS operation' }
   cliModel = RadiusModel.ShowRadius

   @staticmethod
   def handler( mode, args ):
      return showRadius( mode )

BasicCli.addShowCommandClass( ShowRadiusCommand )

#-------------------------------------------------------------------------------
# "clear aaa counters radius" in enable mode
#-------------------------------------------------------------------------------
def clearCounters( mode ):
   Intf.Log.logClearCounters( "radius" )
   now = Tac.now()
   radiusCounterConfig.clearCounterRequestTime = now
   try:
      Tac.waitFor(
         lambda: radiusStatus.lastClearTime >= now,
         description='RADIUS clear counter request to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      mode.addWarning( "RADIUS counters may not have been reset yet" )

class ClearRadiusCounterCommand( CliCommand.CliCommandClass ):
   syntax = "clear aaa counters radius"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'aaa' : AaaCli.aaaAfterClearMatcher,
      'counters' : AaaCli.aaaCounterMatcher,
      'radius' : "Clear RADIUS counters"
      }
   @staticmethod
   def handler( mode, args ):
      clearCounters( mode )

BasicCli.EnableMode.addCommandClass( ClearRadiusCounterCommand )

#-------------------------------------------------------------------------------
# config mode commands
#-------------------------------------------------------------------------------
configMode = BasicCli.GlobalConfigMode

radiusServerKwMatcher = CliMatcher.KeywordMatcher(
   'radius-server',
   helpdesc='Modify RADIUS parameters' )

keyExpression = ReversibleSecretCli.reversibleSecretCliExpression( '<KEY>' )

#-------------------------------------------------------------------------------
# "[no] radius-server key <KEY>" in config mode
#-------------------------------------------------------------------------------
def checkKeySize( mode, key ):
   if key and len( key ) / 2 - 1 > Radius.MAX_KEY_SIZE:
      mode.addError( "Maximum key size is %d" % Radius.MAX_KEY_SIZE )
      return False
   return True

class RadiusServerKeyCommand( CliCommand.CliCommandClass ):
   syntax = 'radius-server key <KEY>'
   noOrDefaultSyntax = 'radius-server key ...'
   data = { 'radius-server' : radiusServerKwMatcher,
            'key' : 'Set RADIUS secret key',
            '<KEY>' : keyExpression }

   @staticmethod
   def handler( mode, args ):
      key = args[ '<KEY>' ]
      t0( 'setKey', str( key ) )
      if checkKeySize( mode, key ):
         radiusConfig.key = key

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noKey' )
      radiusConfig.key = ""

configMode.addCommandClass( RadiusServerKeyCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server timeout <1-1000>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerTimeoutCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server timeout <TIMEOUT>"
   noOrDefaultSyntax = "radius-server timeout ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'timeout' : 'Time to wait for a RADIUS server to respond',
      '<TIMEOUT>' : CliMatcher.IntegerMatcher(
         _timeoutMin, _timeoutMax,
         helpdesc='Number of seconds' )
   }
   @staticmethod
   def handler( mode, args ):
      timeout = args[ '<TIMEOUT>' ]
      t0( 'setTimeout', str( timeout ) )
      radiusConfig.timeout = timeout

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noTimeout' )
      radiusConfig.timeout = radiusConfig.defaultTimeout

configMode.addCommandClass( RadiusServerTimeoutCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server deadtime <1-1000>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerDeadtimeCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server deadtime <DEADTIME>"
   noOrDefaultSyntax = "radius-server deadtime ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'deadtime' : 'Time to skip a nonresponsive server',
      '<DEADTIME>' : CliMatcher.IntegerMatcher( _deadtimeMin, _deadtimeMax,
                                                helpdesc='Number of minutes' )
   }
   @staticmethod
   def handler( mode, args ):
      deadtime = args[ '<DEADTIME>' ]
      t0( 'setDeadtime', str( deadtime ) )
      radiusConfig.deadtime = deadtime * 60

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noDeadtime' )
      radiusConfig.deadtime = 0

configMode.addCommandClass( RadiusServerDeadtimeCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server retransmit <0-100>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerRetransmitCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server retransmit <RETRIES>"
   noOrDefaultSyntax = "radius-server retransmit ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'retransmit' : 'Specify the number of retries for the active server',
      '<RETRIES>' : CliMatcher.IntegerMatcher(
         _retriesMin, _retriesMax,
         helpdesc='Number of retries (default %s times)' % ( _retriesDefault ) )
   }
   @staticmethod
   def handler( mode, args ):
      retries = args[ '<RETRIES>' ]
      t0( 'setRetries', str( retries ) )
      radiusConfig.retries = retries

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noRetries' )
      radiusConfig.retries = radiusConfig.defaultRetries

configMode.addCommandClass( RadiusServerRetransmitCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server attribute 32 include-in-access-req format ( ( format NAS )
# | fqdn | hostname | disabled )
# in config mode
#-------------------------------------------------------------------------------
attrKwMatcher = CliMatcher.KeywordMatcher(
   'attribute', helpdesc='Specify the attribute for the active server' )

class RadiusServerNasIdCommand( CliCommand.CliCommandClass ):
   syntax = '''radius-server attribute 32 include-in-access-req
    ( ( format NAS ) | hostname | fqdn )'''
   noOrDefaultSyntax = "radius-server attribute 32 include-in-access-req ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'attribute' : attrKwMatcher,
      '32' : 'NAS-Identifier',
      'include-in-access-req' :
      'Send NAS-Identifier attribute in Access-Request packet',
      'format' : 'Specify the format of the NAS-Idendifier',
      'NAS' : CliMatcher.StringMatcher( helpname='LINE',
         helpdesc='Simple string' ),
      'hostname' : 'Hostname configured for the switch',
      'fqdn' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'fqdn',
         helpdesc='Fully qualified domain name of the switch'), hidden=True ),
   }
   @staticmethod
   def handler( mode, args ):
      nasId = args.get( 'NAS' )
      if nasId:
         t0( 'setNasId', nasId )
         if len( nasId ) > Radius.MAX_NAS_ID_SIZE:
            mode.addError( "Maximum NAS-Identifier length is %d" %
                           Radius.MAX_NAS_ID_SIZE )
            return
         radiusConfig.nasIdType = "custom"
         radiusConfig.nasId = nasId
      elif args.get( 'hostname' ):
         radiusConfig.nasIdType = 'hostname'
         radiusConfig.nasId = ""
      else:
         radiusConfig.nasIdType = 'fqdn'
         radiusConfig.nasId = ""

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'noNasId' )
      radiusConfig.nasIdType = "disabled"
      radiusConfig.nasId = ""

configMode.addCommandClass( RadiusServerNasIdCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server qos dscp <0-63>" in config mode
#-------------------------------------------------------------------------------
def updateDscpRules():
   dscpValue = radiusConfig.dscpValue

   if not dscpValue:
      del dscpConfig.protoConfig[ 'radius' ]
      return

   protoConfig = dscpConfig.newProtoConfig( 'radius' )
   ruleColl = protoConfig.rule
   ruleColl.clear()

   for spec in radiusConfig.host:
      # Traffic connecting to external radius server auth.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'udp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.port, False, spec.vrf,
                              'udp', dscpValue, v6=True )

      # Traffic connecting to internal radius server acct.
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.acctPort, False, spec.vrf,
                              'udp', dscpValue )
      DscpCliLib.addDscpRule( ruleColl, spec.hostname,
                              spec.acctPort, False, spec.vrf,
                              'udp', dscpValue, v6=True )

def setDscp( mode, args ):
   radiusConfig.dscpValue = args[ 'DSCP' ]
   updateDscpRules()

def noDscp( mode, args ):
   radiusConfig.dscpValue = radiusConfig.dscpValueDefault
   updateDscpRules()

DscpCliLib.addQosDscpCommandClass( configMode, setDscp, noDscp,
                                   tokenProto=radiusServerKwMatcher )

#-------------------------------------------------------------------------------
# "[no] radius-server host <ip-addr-or-hostname> [ VRF ]
#  [auth-port <0-65535>] [acct-port <0-65535>] [timeout <1-1000>]
#  [retransmit <1-100>] [key <key-text>]"
# in config mode
#-------------------------------------------------------------------------------
class RadiusServerHostCommand( CliCommand.CliCommandClass ):
   syntax = '''radius-server host <HOSTNAME> [ VRF ]
               [ auth-port <AUTHPORT> ] [ acct-port <ACCTPORT> ]
               [ timeout <TIMEOUT> ] [ retransmit <RETRIES> ]
               [ key <KEY> ]'''
   noOrDefaultSyntax = '''radius-server host
                          [ <HOSTNAME> [ VRF ]
                          [ auth-port <AUTHPORT> ] [ acct-port <ACCTPORT> ] ] ...'''
   data = { 'radius-server' : radiusServerKwMatcher,
            'host' : 'RADIUS server configuration',
            '<HOSTNAME>' : HostnameCli.IpAddrOrHostnameMatcher(
               helpname='WORD',
               helpdesc='Hostname or IP address of RADIUS server',
               ipv6=True ),
            'VRF' : VrfExprFactory( helpdesc='VRF for this Radius server' ),
            'auth-port' : 'RADIUS server authentication port (default is %s)' \
            % RadiusGroup.defaultPort,
            '<AUTHPORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                                             helpdesc="Number of the port to use" ),
            'acct-port' : 'RADIUS server accounting port (default is %s)' \
            % RadiusGroup.defaultAcctPort,
            '<ACCTPORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                                                      helpdesc="Acct port number" ),
            'timeout' : 'Time to wait for this RADIUS server to respond ' \
            '(overrides default)',
            '<TIMEOUT>' : CliMatcher.IntegerMatcher(
               _timeoutMin, _timeoutMax,
               helpdesc='Timeout value in seconds to ' \
               'wait for the server\'s response' ),
            'retransmit' :
            'Specify the number of retries for the server (overrides default)',
            '<RETRIES>' : CliMatcher.IntegerMatcher(
               _retriesMin, _retriesMax,
               helpdesc='Retry limit' ),
            'key' : 'Encryption key for this RADIUS server (overrides default)',
            '<KEY>' : keyExpression
            }
   @staticmethod
   def handler( mode, args ):
      hostname = args[ '<HOSTNAME>' ]
      vrf = args.get( 'VRF', DEFAULT_VRF )
      port = args.get( '<AUTHPORT>', RadiusGroup.defaultPort )
      acctPort = args.get( '<ACCTPORT>', RadiusGroup.defaultAcctPort )
      timeout = args.get( '<TIMEOUT>' )
      retries = args.get( '<RETRIES>' )
      key = args.get( '<KEY>' )
      t0( "setHost hostname:", hostname, "port:", port, "acctPort:", acctPort,
          "timeout:", timeout, "retries:", retries, "key:", str( key ),
          "vrf:", vrf )
      HostnameCli.resolveHostname( mode, hostname, doWarn=True )
      assert vrf != ''
      if timeout is None:
         timeoutVal = _timeoutDefault
      else:
         timeoutVal = timeout
      if retries is None:
         retriesVal = retries or _retriesDefault
      else:
         retriesVal = retries
      if not checkKeySize( mode, key ):
         return
      host = radiusHost( mode, hostname, vrf, port, acctPort, create=True )
      host.useKey = ( key is not None )
      host.key = key or ""
      host.useTimeout = ( timeout is not None )
      host.timeout = timeoutVal
      host.useRetries = ( retries is not None )
      host.retries = retriesVal
      if mode.session_.interactive_:
         vrfString = " in vrf %s" % vrf if vrf != DEFAULT_VRF else ""
         mode.addMessage( "RADIUS host %s with auth-port %s and acct-port %s "
                          "created%s" % ( hostname, port, acctPort, vrfString ) )
      updateDscpRules()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      hostname = args.get( '<HOSTNAME>' )
      hosts = radiusConfig.host

      if hostname:
         vrf = args.get( 'VRF', DEFAULT_VRF )
         port = args.get( '<AUTHPORT>', RadiusGroup.defaultPort )
         acctPort = args.get( '<ACCTPORT>', RadiusGroup.defaultAcctPort )
         t0( "noHost hostname:", hostname, "port:", port )
         spec = Tac.Value( "Aaa::HostSpec", hostname=hostname, port=port,
                           acctPort=acctPort, vrf=vrf )
         if spec in hosts:
            del hosts[ spec ]
         else:
            if mode.session_.interactive_:
               vrfString = " in vrf %s" % vrf if vrf != DEFAULT_VRF else ""
               warningMessage = "RADIUS host %s with auth-port %s and " \
                                "acct-port %s not found%s" % \
                                ( hostname, port, acctPort, vrfString )
               mode.addWarning( warningMessage )
      else:
         # Delete all hosts since no hostname was specified
         hosts.clear()

      updateDscpRules()

configMode.addCommandClass( RadiusServerHostCommand )

#-------------------------------------------------------------------------------
# "[no] ip radius [ VRF ] source-interface <interface-name>"
# in config mode
#-------------------------------------------------------------------------------
class RadiusSourceIntfCommand( CliCommand.CliCommandClass ):
   syntax = "ip radius [ VRF ] source-interface INTF"
   noOrDefaultSyntax = "ip radius [ VRF ] source-interface ..."
   data = {
      "ip" : CliToken.Ip.ipMatcherForConfig,
      "radius" : 'Commands for RADIUS',
      "VRF" : VrfExprFactory( helpdesc='Specify VRF' ),
      "source-interface" : 'Interface providing the IP source '
                           'address of RADIUS packets',
      "INTF" : IntfCli.Intf.matcherWithIpSupport
      }

   @staticmethod
   def handler( mode, args ):
      intf = args[ "INTF" ]
      vrf = args.get( "VRF" )
      if not vrf or vrf == DEFAULT_VRF_OLD:
         vrf = DEFAULT_VRF
      radiusConfig.srcIntfName[ vrf ] = intf.name

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vrf = args.get( "VRF" )
      if not vrf or vrf == DEFAULT_VRF_OLD:
         vrf = DEFAULT_VRF
      del radiusConfig.srcIntfName[ vrf ]

configMode.addCommandClass( RadiusSourceIntfCommand )

#-------------------------------------------------------------------------------
# "[no] radius-server dynamic-authorization port  <0-65535>" in config mode
#-------------------------------------------------------------------------------
class RadiusServerDynamicAuthorizationCommand( CliCommand.CliCommandClass ):
   syntax = "radius-server dynamic-authorization port <DYNAUTHPORT>"
   noOrDefaultSyntax = "radius-server dynamic-authorization ..."
   data = {
      'radius-server' : radiusServerKwMatcher,
      'dynamic-authorization' : 'Dynamic authorization',
      'port' : 'Listen port(default is %s)' \
      % RadiusGroup.defaultDynAuthPort,
      '<DYNAUTHPORT>' : CliMatcher.IntegerMatcher( 0, 65535,
                           helpdesc="Number of the port to use" ),
   }

   @staticmethod
   def handler( mode, args ):
      dynAuthPort = args[ '<DYNAUTHPORT>' ]
      t0( 'set dynAuthPort', str( dynAuthPort ) )
      radiusConfig.dynAuthPort = dynAuthPort

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      t0( 'nodynAuthPort' )
      radiusConfig.dynAuthPort = RadiusGroup.defaultDynAuthPort

configMode.addCommandClass( RadiusServerDynamicAuthorizationCommand )

#-------------------------------------------------------------------------------
# Register "show tech-support" commands
#-------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2010-08-27 04:35:31',
   lambda: [ 'show radius' ],
   summaryCmdCallback=lambda: [ 'show radius' ] )

#-------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   import Cell
   global aaaConfig, radiusConfig, radiusCounterConfig
   global radiusInputStatus, radiusStatus, dscpConfig
   aaaConfig = LazyMount.mount( entityManager, "security/aaa/config",
                                "Aaa::Config", "r" )
   radiusConfig = ConfigMount.mount( entityManager, "security/aaa/radius/config",
                                     "Radius::Config", "w" )
   radiusCounterConfig = LazyMount.mount( entityManager,
                                          "security/aaa/radius/counterConfig",
                                          "AaaPlugin::CounterConfig", "w" )
   radiusStatus = LazyMount.mount( entityManager,
                                   Cell.path( "security/aaa/radius/status" ),
                                   "Radius::Status", "r" )
   radiusInputStatus = LazyMount.mount( entityManager,
                                 Cell.path( "security/aaa/radius/input/status" ),
                                        "Tac::Dir", "ri" )
   dscpConfig = ConfigMount.mount( entityManager, "mgmt/dscp/config",
                                   "Mgmt::Dscp::Config", "w" )
