#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import Arnet
import HumanReadable
import TableOutput
from CliModel import Model, Int, Float, Dict, Submodel, Enum, Bool
from IntfModels import Interface

def counterStr( count ):
   if count is None:
      return 'N/A'
   else:
      return str( count )

class Counters( Model ):
   enqueuedPackets = Int( help="Packets enqueued", optional=True )
   enqueuedBytes = Int( help="Octets (bytes) enqueued", optional=True )
   droppedPackets = Int( help="Packets dropped", optional=True )
   droppedBytes = Int( help="Octets (bytes) dropped", optional=True )
   enqueuedPacketsRate = Float( help="Packets enqueued per second", optional=True )
   enqueuedBitsRate = Float( help="Bits enqueued per second", optional=True )
   droppedPacketsRate = Float( help="Packets dropped per second", optional=True )
   droppedBitsRate = Float( help="Bits dropped per second", optional=True )

class IngressVoqTrafficClassCounters( Model ):
   trafficClasses = Dict( keyType=str, valueType=Counters,
                          help="Mapping between traffic class and counter data" )

class IngressVoqCounters( Model ):
   interfaces = Dict( keyType=Interface, valueType=IngressVoqTrafficClassCounters,
                      help="Mapping between egress interface and traffic class "
                           "counter data" )

class ShapeRateModel( Model ):
   rate = Int( help="Shape rate" )
   unit = Enum( values=( 'kbps', 'pps', 'unspecified', ),
                help="Shape rate unit",
                default='unspecified' )
   shared = Bool( help="Shared LAG shaping configured", optional=True )
   percent = Int( help="Shape rate as percentage of link bandwidth", optional=True )

class EgressQueueDropPrecedenceCounters( Model ):
   # probably doesn't apply to non-Sand, in which case can just call it 'DP0',
   # similar to how the QUEUE-MIB does it
   dropPrecedences = Dict( keyType=str, valueType=Counters,
                           help="Mapping between drop precedence and corresponding "
                                "egress enqueue and drop counter data" )
   schedMode = Enum( values=( "strictPriority", "weightedRoundRobin", ),
                     help="Scheduling mode", optional=True )
   wrrBw = Int( help="Operational Weighted-Round-Robin Bandwidth in percent",
                optional=True )
   shapeRate = Submodel( valueType=ShapeRateModel,
                         help="Operational Shape rate",
                         optional=True )

class EgressQueueTrafficClassCounters( Model ):
   trafficClasses = Dict( keyType=str, valueType=EgressQueueDropPrecedenceCounters,
                          help="Mapping between traffic class and per drop "
                               "precedence counter data. In some cases, some "
                               "classes might be aggregated (ex: \"TC1,4\"), or "
                               "totally aggregated (ex: \"all\")." )

class EgressQueueDestinationTypeCounters( Model ):
   bandwidth = Int( help="Interface speed in kbps", optional=True )
   loadInterval = Float( help="Interval in seconds over which moving average is "
                              "calculated", optional=True )
   ucastQueues = Submodel( valueType=EgressQueueTrafficClassCounters,
                           help="Unicast traffic class counters" )
   mcastQueues = Submodel( valueType=EgressQueueTrafficClassCounters,
                           help="Multicast traffic class counters" )

class EgressQueueCounters( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=EgressQueueDestinationTypeCounters,
                      help="Mapping between interface and destination type counter "
                           "data" )

class IngressQueueCounters( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=Counters,
                      help="Mapping between interface and counter data across all "
                           "traffic classes" )

class QueueCounters( Model ):
   ingressVoqCounters = Submodel( valueType=IngressVoqCounters,
                                  help="VOQ counters on the ingress source",
                                  optional=True )
   ingressQueueCounters = Submodel( valueType=IngressQueueCounters,
                                    help="Queue counters on the ingress chip",
                                    optional=True )
   egressQueueCounters = Submodel( valueType=EgressQueueCounters,
                                   help="Queue counters on the egress chip" )
   _renderType = Enum( values=( "Strata", "Bfn", "Sfe" ),
                                help="Specifies the type of platform" )

   def render( self ):
      if self._renderType == "Strata":
         self.renderStrata()
      elif self._renderType == "Bfn":
         self.renderBfn()
      elif self._renderType == "Sfe":
         self.renderSfe()
      # Don't render anything from this function if not of a known _renderType. We
      # assume that if something needed to be printed to the CLI that these other
      # platforms have done their appropriate print before trying to render()
      # already.

   def renderBfn( self ):
      if not self.egressQueueCounters:
         return
      interfaces = self.egressQueueCounters.interfaces
      header = "%s%8s%20s%20s%20s" % ( "Port", "TxQ", "Pkts",
                                       "Bytes", "DropPkts" )

      for intfId in Arnet.sortIntf( interfaces ):
         print header
         print '-' * len( header )

         tc = interfaces[ intfId ].ucastQueues.trafficClasses
         for tc in [ interfaces[ intfId ].ucastQueues.trafficClasses,
                     interfaces[ intfId ].mcastQueues.trafficClasses ]:
            for tcId in sorted( tc ):
               m = re.match( r'TC(\d+)', tcId )
               queueId = m.group( 1 )
               ctrs = tc[ tcId ].dropPrecedences[ 'DP0' ]
               valueCountersPkts = counterStr( ctrs.enqueuedPackets )
               valueCountersBytes = counterStr( ctrs.enqueuedBytes )
               valueDropPkts = counterStr( ctrs.droppedPackets )
               if "Ethernet" in intfId:
                  portId = intfId.replace( "Ethernet", "Et" )
               print "%-7s%5s%20s%20s%20s" % ( portId, "TxQ" + queueId,
                                               valueCountersPkts,
                                               valueCountersBytes,
                                               valueDropPkts )

         print ""

   def renderStrata( self ):
      if not self.egressQueueCounters:
         return
      interfaces = self.egressQueueCounters.interfaces
      for intfId in Arnet.sortIntf( interfaces ):
         tc = interfaces[ intfId ].ucastQueues.trafficClasses
         tc0 = tc.get( 'TC0' )
         # Skip printing the interfaces if the data is not present
         if tc0 is None:
            continue
         printDCHeader = tc0.dropPrecedences[ 'DP0' ].droppedBytes is None

         print "%s%8s%15s%19s%15s%19s" % ( "Port", "TxQ", "Counter/pkts",
                                           "Counter/bytes", "Drop/pkts",
                                           "Drop/bytes" )
         print "%s%5s%15s%19s%15s%19s" % ( "-" * 7, "-" * 4, "-" * 12,
                                           "-" * 12, "-" * 12, "-" * 12 )

         for tc in [ interfaces[ intfId ].ucastQueues.trafficClasses,
                     interfaces[ intfId ].mcastQueues.trafficClasses ]:
            if tc == interfaces[ intfId ].ucastQueues.trafficClasses:
               destType = 'UC'
            else:
               destType = 'MC'
            for tcId in sorted( tc ):
               m = re.match( r'TC(\d+)', tcId )
               queueId = m.group( 1 )
               ctrs = tc[ tcId ].dropPrecedences[ 'DP0' ]
               hasUCWorkaround = False
               if destType == 'UC' and ( queueId == '0' or queueId == '1' ):
                  hasUCWorkaround = ( ctrs.droppedBytes is None )
               valueCountersPkts = counterStr( ctrs.enqueuedPackets )
               valueCountersBytes = counterStr( ctrs.enqueuedBytes )
               valueDropPkts = counterStr( ctrs.droppedPackets )
               if valueDropPkts != 'N/A' and hasUCWorkaround:
                  valueDropPkts = 'N/A'
               valueDropBytes = counterStr( ctrs.droppedBytes )
               if "UnconnectedEthernet" in intfId:
                  portId = intfId.replace( "UnconnectedEthernet", "Ue" )
               elif "Ethernet" in intfId:
                  portId = intfId.replace( "Ethernet", "Et" )
               print "%-7s%5s%15s%19s%15s%19s" % ( portId, destType + queueId,
                                                   valueCountersPkts,
                                                   valueCountersBytes,
                                                   valueDropPkts,
                                                   valueDropBytes )
         if printDCHeader:
            tc = interfaces[ intfId ].ucastQueues.trafficClasses
            if "UnconnectedEthernet" in intfId:
               portId = intfId.replace( "UnconnectedEthernet", "Ue" )
            elif "Ethernet" in intfId:
               portId = intfId.replace( "Ethernet", "Et" )
            for queueId in xrange( 2 ):
               tcId = "TC%d" % queueId
               ctrs = tc[ tcId ].dropPrecedences[ 'DP0' ]
               print "%-7s%5s%15s%19s%15s%19s" % ( portId, "DC" + str( queueId ),
                                                   "N/A", "N/A",
                                                   str( ctrs.droppedPackets ),
                                                   "N/A" )

         print ""

   def renderSfe( self ):
      if not self.egressQueueCounters:
         return
      interfaces = self.egressQueueCounters.interfaces
      formatStr = "%s%8s%20s%20s%20s"
      header = formatStr % ( "Port", "TxQ", "Pkts", "Bytes", "DropPkts" )
      print header
      print '-' * len( header )

      for intfId in Arnet.sortIntf( interfaces ):
         tc = interfaces[ intfId ].ucastQueues.trafficClasses
         portId = intfId
         if "Ethernet" in intfId:
            portId = intfId.replace( "Ethernet", "Et" )
         for q in range( len( tc ) ):
            qid = "Q%d" % q
            ctrs = tc[ qid ].dropPrecedences[ 'DP0' ]
            valueCountersPkts = counterStr( ctrs.enqueuedPackets )
            valueCountersBytes = counterStr( ctrs.enqueuedBytes )
            valueDropPkts = counterStr( ctrs.droppedPackets )
            print formatStr % ( portId, qid, valueCountersPkts,
                                valueCountersBytes, valueDropPkts )

class QueueCountersRate( Model ):
   egressQueueCounters = Submodel( valueType=EgressQueueCounters,
                                   help="Queue counters on the egress chip" )

   def render( self ):
      def toString( *vals ):
         for val in vals:
            yield "N/A" if val is None else "{:.1f}".format( val )

      def scaleValueSi( valToScale ):
         val, unit = HumanReadable.scaleValueSi( valToScale )
         if unit == "":
            unit = " "
         return val, unit

      def shapeRateFromTc( tcShapeRate ):
         shapeRateUnit = tcShapeRate[ 'unit' ]
         if tcShapeRate[ 'unit' ] == 'kbps':
            shapeRateMult = 1000
         elif tcShapeRate[ 'unit' ] == 'pps':
            shapeRateMult = 1
         else:
            assert False, 'Unexpected shape rate unit {0}'.format( shapeRateUnit )
         return tcShapeRate[ 'rate' ] * shapeRateMult

      if not self.egressQueueCounters:
         return

      interfaces = self.egressQueueCounters.interfaces

      # Get total bps, pps, link percent for offered, dropped, and output rates.
      intfAggregations = {}

      # pylint: disable=too-many-nested-blocks
      for intfId, intf in interfaces.iteritems():
         # Skip the interface if the data is not present.
         if not intf.ucastQueues.trafficClasses:
            continue

         # Create count aggregation dictionary entry.
         if intfId not in intfAggregations:
            intfAggregations[ intfId ] = {
               "offeredPercentLink": 0.0,
               "offeredBps": 0.0,
               "offeredPps": 0.0,
               "offeredRelLink": 0.0,
               "droppedBitsRate": 0.0,
               "droppedPacketsRate": 0.0,
               "outputPercentLink": 0.0,
               "enqueuedBitsRate": 0.0,
               "enqueuedPacketsRate": 0.0,
               "outputRelLink": 0.0,
               "shapeRate": None,
               "wrrBw": None,
               "wrrPercent": None,
               "UC": {},
               "MC": {},
            }

         for pktType, tcs in [ ( "UC", intf.ucastQueues.trafficClasses ),
                               ( "MC", intf.mcastQueues.trafficClasses ) ]:
            for tcId, tc in tcs.iteritems():
               ctrs = tc.dropPrecedences[ "DP0" ]
               # Round these counters while calculating what to render in cases such
               # as 1e-5 will show 0 bps with 0% output instead of 100%, see
               # BUG390672
               rnd = {}
               rnd[ "enqueuedBR" ] = round( ctrs.enqueuedBitsRate, 1 )
               rnd[ "droppedBR" ] = round( ctrs.droppedBitsRate, 1 )
               rnd[ "enqueuedPR" ] = round( ctrs.enqueuedPacketsRate, 1 )
               rnd[ "droppedPR" ] = round( ctrs.droppedPacketsRate, 1 )
               intfAggr = intfAggregations[ intfId ]

               # Create TxQ entry for aggregation.
               if tcId not in intfAggr[ pktType ]:
                  intfAggr[ pktType ][ tcId ] = {
                     "offeredPercentLink": 0.0,
                     "offeredBps": 0.0,
                     "offeredPps": 0.0,
                     "outputPercentLink": 0.0,
                  }
               tcAggr = intfAggr[ pktType ][ tcId ]
               tcAggr[ "offeredBps" ] += rnd[ "enqueuedBR" ] + rnd[ "droppedBR" ]
               tcAggr[ "offeredPps" ] += rnd[ "enqueuedPR" ] + rnd[ "droppedPR" ]
               if intf.bandwidth:
                  tcAggr[ "offeredPercentLink" ] += ( tcAggr[ "offeredBps" ] + 20 *
                     8 * tcAggr[ "offeredPps" ] ) / ( intf.bandwidth * 1000 ) * 100
                  tcAggr[ "outputPercentLink" ] += ( rnd[ "enqueuedBR" ] + 20 * 8 *
                     rnd[ "enqueuedPR" ] ) / ( intf.bandwidth * 1000 ) * 100

               for counterAttr in ( "offeredBps", "offeredPps", "offeredPercentLink",
                                    "outputPercentLink", ):
                  intfAggr[ counterAttr ] += tcAggr[ counterAttr ]

               for counterAttr, roundedAttr in (
                     ( "droppedBitsRate", "droppedBR" ),
                     ( "droppedPacketsRate", "droppedPR" ),
                     ( "enqueuedBitsRate", "enqueuedBR" ),
                     ( "enqueuedPacketsRate", "enqueuedPR" ) ):
                  intfAggr[ counterAttr ] += rnd[ roundedAttr ]

               # UCx and MCx queues share the same traffic class and since QoS
               # shaping and priority is applied per traffic class, only UCx values
               # are needed.
               if pktType == "UC":
                  if tc.shapeRate and tc.shapeRate[ "unit" ] != "unspecified":
                     if intfAggr[ "shapeRate" ]:
                        intfAggr[ "shapeRate" ] += shapeRateFromTc( tc.shapeRate )
                     else:
                        intfAggr[ "shapeRate" ] = shapeRateFromTc( tc.shapeRate )
                  if tc.schedMode == "weightedRoundRobin" and tc.wrrBw:
                     if intfAggr[ "wrrBw" ]:
                        intfAggr[ "wrrBw" ] += tc.wrrBw
                     else:
                        intfAggr[ "wrrBw" ] = tc.wrrBw

      # Calculate relative percentages, and print.
      for intfId in Arnet.sortIntf( interfaces ):
         intf = interfaces[ intfId ]

         # Skip printing the interfaces if the data is not present.
         if not intf.ucastQueues.trafficClasses:
            continue

         shortIntfId = intfId.replace( "UnconnectedEthernet", "Ue" )
         shortIntfId = intfId.replace( "Ethernet", "Et" )

         loadInterval = next( toString( intf.loadInterval ) )
         print "{} - load interval {}s".format( shortIntfId, loadInterval )
         table = TableOutput.createTable( (
            "",
            ( "Offered load (include drop)", "l",
               ( "bps", "pps", "% of\nlink", "% of \ntotal", ) ),
            ( "Config", "l", ( "bps", "SP/WRR", ) ),
            ( "Drop rates", "l", ( "bps", "pps", ) ),
            ( "Output rates", "l", ( "bps", "pps", "% of\nlink",
                                     "% of \ntotal", "WRR %", ) ) ) )

         intfAggr = intfAggregations[ intfId ]
         tcCompiledRe = re.compile( r"TC(?P<trafficClass>\d+(,\d+)?)" )
         for pktType, tcs in [ ( "UC", intf.ucastQueues.trafficClasses ),
                               ( "MC", intf.mcastQueues.trafficClasses ) ]:
            for tcId in sorted( tcs ):
               tc = tcs[ tcId ]
               tcAggr = intfAggr[ pktType ][ tcId ]
               m = tcCompiledRe.match( tcId )
               trafficClass = m.group( 'trafficClass' )
               ctrs = tc.dropPrecedences[ "DP0" ]
               # Round these counters while calculating what to render in cases such
               # as 1e-5 will show 0 bps with 0% output instead of 100%, see
               # BUG390672
               rnd = {}
               rnd[ "enqueuedBR" ] = round( ctrs.enqueuedBitsRate, 1 )
               rnd[ "droppedBR" ] = round( ctrs.droppedBitsRate, 1 )
               rnd[ "enqueuedPR" ] = round( ctrs.enqueuedPacketsRate, 1 )
               rnd[ "droppedPR" ] = round( ctrs.droppedPacketsRate, 1 )

               # offered
               TxQ = pktType + trafficClass
               offeredBps, offeredBpsUnit = scaleValueSi( tcAggr[ "offeredBps" ] )
               offeredPps, offeredPpsUnit = scaleValueSi( tcAggr[ "offeredPps" ] )
               offeredPercentLink = tcAggr[ "offeredPercentLink" ]
               offeredRelLink = tcAggr[ "offeredPercentLink" ] / \
                  intfAggr[ "offeredPercentLink" ] * 100 \
                  if intfAggr[ "offeredPercentLink" ] else 0
               intfAggr[ "offeredRelLink" ] += offeredRelLink
               offeredBps, offeredPps, offeredPercentLink, offeredRelLink = \
                  toString( offeredBps, offeredPps, offeredPercentLink,
                            offeredRelLink )

               # qos
               if tc.shapeRate and tc.shapeRate[ "unit" ] != "unspecified":
                  shapeRate = shapeRateFromTc( tc.shapeRate )
                  shapePps, shapePpsUnit = scaleValueSi( shapeRate )
               else:
                  shapePps, shapePpsUnit = None, " "
               if tc.schedMode == "strictPriority":
                  spwrr = "SP"
               else:
                  spwrr = next( toString( tc.wrrBw ) )
               shapePps = next( toString( shapePps ) )

               # drop
               dropBps, dropBpsUnit = scaleValueSi( rnd[ "droppedBR" ] )
               dropPps, dropPpsUnit = scaleValueSi( rnd[ "droppedPR" ] )
               dropBps, dropPps = toString( dropBps, dropPps )

               # out
               outBps, outBpsUnit = scaleValueSi( rnd[ "enqueuedBR" ] )
               outPps, outPpsUnit = scaleValueSi( rnd[ "enqueuedPR" ] )
               outPercentLink = tcAggr[ "outputPercentLink" ]
               outRelLink  = outPercentLink / intfAggr[ "outputPercentLink" ] * 100 \
                  if intfAggr[ "outputPercentLink" ] else 0
               intfAggr[ "outputRelLink" ] += outRelLink
               if spwrr in ( "SP", "N/A", ):
                  outWRRPer, outWRRPerUnit = None, " "
               else:
                  outWRRPer = ( float( tc.wrrBw ) / intfAggr[ "wrrBw" ] ) * 100 \
                              if intfAggr[ "wrrBw" ] else 0
                  outWRRPerUnit = "%"

                  # UCx and MCx queues share the same traffic class and since QoS
                  # shaping and priority is applied per traffic class, only UCx
                  # values are needed.
                  if pktType == "UC":
                     if intfAggr[ "wrrPercent" ]:
                        intfAggr[ "wrrPercent" ] += outWRRPer
                     else:
                        intfAggr[ "wrrPercent" ] = outWRRPer
               outBps, outPps, outPercentLink, outRelLink, outWRRPer = \
                  toString( outBps, outPps, outPercentLink, outRelLink, outWRRPer )

               # Per-queue values.
               table.newRow( TxQ, offeredBps + offeredBpsUnit, offeredPps +
                             offeredPpsUnit, offeredPercentLink + "%",
                             offeredRelLink + "%", shapePps + shapePpsUnit, spwrr,
                             dropBps + dropBpsUnit, dropPps + dropPpsUnit, outBps +
                             outBpsUnit, outPps + outPpsUnit, outPercentLink + "%",
                             outRelLink + "%", outWRRPer + outWRRPerUnit )

         # Last table row is the aggregated 'Total' queue values.
         offeredBps, offeredBpsUnit = scaleValueSi( intfAggr[ "offeredBps" ] )
         offeredPps, offeredPpsUnit = scaleValueSi( intfAggr[ "offeredPps" ] )
         offeredBps, offeredPps, offeredPercentLink, offeredRelLink = \
            toString( offeredBps, offeredPps, intfAggr[ "offeredPercentLink" ],
                      intfAggr[ "offeredRelLink" ] )
         if intfAggr[ "shapeRate" ]:
            shapePps, shapePpsUnit = scaleValueSi( intfAggr[ "shapeRate" ] )
         else:
            shapePps, shapePpsUnit = None, " "
         shapePps = next( toString( shapePps ) )

         dropBps, dropBpsUnit = scaleValueSi( intfAggr[ "droppedBitsRate" ] )
         dropPps, dropPpsUnit = scaleValueSi( intfAggr[ "droppedPacketsRate" ] )
         dropBps, dropPps = toString( dropBps, dropPps )

         outBps, outBpsUnit = scaleValueSi( intfAggr[ "enqueuedBitsRate" ] )
         outPps, outPpsUnit = scaleValueSi( intfAggr[ "enqueuedPacketsRate" ] )
         outBps, outPps, outPercentLink, outRelLink = toString( outBps, outPps,
            intfAggr[ "outputPercentLink" ], intfAggr[ "outputRelLink" ] )
         if intfAggr[ "wrrPercent" ]:
            outWRRPer, outWRRPerUnit = intfAggr[ "wrrPercent" ], "%"
         else:
            outWRRPer, outWRRPerUnit = None, " "
         outWRRPer = next( toString( outWRRPer ) )

         table.newRow( "Total", offeredBps + offeredBpsUnit, offeredPps +
                       offeredPpsUnit, offeredPercentLink + "%", offeredRelLink +
                       "%", shapePps + shapePpsUnit, "", dropBps + dropBpsUnit,
                       dropPps + dropPpsUnit, outBps + outBpsUnit, outPps +
                       outPpsUnit, outPercentLink + "%", outRelLink + "%",
                       outWRRPer + outWRRPerUnit )

         # Format then print table.
         txQ = TableOutput.Format( justify="left", minWidth=5 )
         numVal = TableOutput.Format( minWidth=5 )
         strVal = TableOutput.Format( justify="center", minWidth=6 )

         txQ.padLimitIs( True )
         numVal.padLimitIs( True )
         strVal.padLimitIs( True )

         table.formatColumns( txQ, numVal, numVal, numVal, numVal, numVal, strVal,
                              numVal, numVal, numVal, numVal, numVal, numVal,
                              numVal )
         print table.output()
      # pylint: enable=too-many-nested-blocks

class UcastDropCounters( Model ):
   droppedPackets = Int( help="Unicast packets dropped", optional=True )
   droppedBytes = Int( help="Unicast octets (bytes) dropped", optional=True )

class EgressUcastQueueTrafficClassCounters( Model ):
   trafficClasses = Dict( keyType=str, valueType=UcastDropCounters,
                          help="Mapping between traffic class and unicast drop "
                               "counter data" )

class EgressUcastQueueDropCounters( Model ):
   ucastQueues = Submodel( valueType=EgressUcastQueueTrafficClassCounters,
                           help="Unicast traffic class counters" )

class EgressQueueDropCounters( Model ):
   interfaces = Dict( keyType=Interface,
                      valueType=EgressUcastQueueDropCounters,
                      help="Mapping between interface and destination type counter "
                           "data" )

class QueueDropCounters( Model ):
   # RFE271183: CAPI Model for the hidden command
   # "show interfaces [<intf>] counters queue drops unicast"
   egressQueueDropCounters = Submodel( valueType=EgressQueueDropCounters,
                                       help="Queue Drop counters on the egress" )

   def render( self ):
      if not self.egressQueueDropCounters:
         return
      tcRegex = re.compile( r'TC(\d+)' )

      def renderIntf( intfId ):
         tc = interfaces[ intfId ].ucastQueues.trafficClasses
         tc0 = tc.get( 'TC0' )
         # Skip printing the interfaces if the data is not present
         if tc0 is None:
            return

         printDCHeader = tc0.droppedBytes is None

         def renderTc( tcId ):
            m = tcRegex.match( tcId )
            queueId = m.group( 1 )
            ctrs = tc[ tcId ]
            hasUCWorkaround = False
            if queueId == '0' or queueId == '1':
               hasUCWorkaround = ( ctrs.droppedBytes is None )
            valueDropPkts = counterStr( ctrs.droppedPackets )
            if valueDropPkts != 'N/A' and hasUCWorkaround:
               valueDropPkts = 'N/A'
            valueDropBytes = counterStr( ctrs.droppedBytes )
            print "%-7s%5s%15s%19s" % ( portId, destType + queueId,
                                                valueDropPkts,
                                                valueDropBytes )

         def renderDCHeader():
            tc = interfaces[ intfId ].ucastQueues.trafficClasses
            for queueId in xrange( 2 ):
               tcId = "TC%d" % queueId
               ctrs = tc[ tcId ]
               print "%-7s%5s%15s%19s%15s%19s" % ( portId, "DC" + str( queueId ),
                                                   "N/A", "N/A",
                                                   str( ctrs.droppedPackets ),
                                                   "N/A" )

         print "%s%8s%15s%19s" % ( "Port", "TxQ", "Drop/pkts", "Drop/bytes" )
         print "%s%5s%15s%19s" % ( "-" * 7, "-" * 4, "-" * 12, "-" * 12 )

         destType = 'UC'
         if "UnconnectedEthernet" in intfId:
            portId = intfId.replace( "UnconnectedEthernet", "Ue" )
         elif "Ethernet" in intfId:
            portId = intfId.replace( "Ethernet", "Et" )
         for tcId in sorted( tc ):
            renderTc( tcId )

         if printDCHeader:
            renderDCHeader()

         print ""

      interfaces = self.egressQueueDropCounters.interfaces
      for intfId in Arnet.sortIntf( interfaces ):
         renderIntf( intfId )

class IngressCounters( Model ):
   # Traffic source (e.g., FAP or core) refers to where packets are ingressing.
   sources = Dict( keyType=str, valueType=IngressVoqCounters,
                   help="Mapping between traffic source and VOQ counters" )

class IngressVoqSourceDropsCounters( Model ):
   countersSum = Submodel( valueType=Counters,
                           help="Counters summed over all sources" )
   sources = Dict( keyType=str, valueType=Counters,
                   help="Mapping between traffic source and drops counter data" )

class EgressQueueDestinationTypeDropsCounters( Model ):
   intfSpeed = Int( help="Interface speed", optional=True )
   loadInterval = Int( help="Load Interval", optional=True )
   ucastQueues = Submodel( valueType=Counters, help="Unicast drops counters" )
   mcastQueues = Submodel( valueType=Counters, help="Multicast drops counters" )

class EgressQueueDropPrecedenceDropsCounters( Model ):
   countersSum = Submodel( valueType=Counters,
                           help="Counters summed over drop precedence and "
                                "destination type",
                                optional=True )
   dropPrecedences = Dict( keyType=str,
                           valueType=EgressQueueDestinationTypeDropsCounters,
                           help="Mapping between drop precedence and corresponding "
                                "egress destination type drops counter data",
                           optional=True )

class IngressEgressQueueDropsCounters( Model ):
   ingressVoqCounters = Submodel( valueType=IngressVoqSourceDropsCounters,
                                  help="VOQ drops counters on the ingress source",
                                  optional=True )
   egressQueueCounters = Submodel( valueType=EgressQueueDropPrecedenceDropsCounters,
                                   help="Queue drops counters on the egress chip" )

class QueueTrafficClassDropsCounters( Model ):
   trafficClasses = Dict( keyType=str, valueType=IngressEgressQueueDropsCounters,
                          help="Mapping between traffic class, and ingress VOQ and "
                               "egress queue drops counter data" )

class QueueDropsCounters( Model ):
   interfaces = Dict( keyType=Interface, valueType=QueueTrafficClassDropsCounters,
                      help="Mapping between egress interface and traffic class "
                           "drops counter data" )
