# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import IntfCli
from QueueCountersModel import QueueCounters, QueueCountersRate, \
   EgressQueueCounters, QueueDropCounters, EgressQueueDropCounters

entityManager = None

#--------------------------------------------------------------------------------
# The "show interfaces [<name>] counters queue [detail]" command, in enable mode.
#--------------------------------------------------------------------------------

# Hook for platform-specific implementations of the
# "show interfaces [ethernet] counters queue [detail]" command and guard.
# Each extension must be a tuple consisting of (func, guard, detailSupported), where
# func is the function that is called for the command, guard is the platform-specific
# CLI guard function that guards the command, and detailSupported indicates if
# "show interfaces counters queue detail" is implemented on the platform.
def countersQueueGuard( mode, token ):
   countersQueueGuardHelper( mode, token )

def countersQueueDetailGuard( mode, token ):
   countersQueueGuardHelper( mode, token, detailGuard=True )

def countersQueueRateGuard( mode, token ):
   countersQueueGuardHelper( mode, token, rateGuard=True )

showInterfacesQueueCountersHook = CliExtensions.CliHook()

def countersQueueGuardHelper( mode, token, detailGuard=False, rateGuard=False ):
   guardCodes = []
   for  _, guard, detailSupported, rateSupported in \
         showInterfacesQueueCountersHook.extensions():
      if ( detailGuard and not detailSupported ) or \
            ( rateGuard and not rateSupported ):
         continue

      guardCode = guard( mode, token )
      if guardCode is None:
         return None
      else:
         guardCodes.append( guardCode )

   if guardCodes:
      # Return the guard of the first guard that complained.
      return guardCodes[ 0 ]
   else:
      return CliParser.guardNotThisPlatform

def showInterfacesQueueCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   detail = 'detail' in args

   queueCounters = QueueCounters()
   queueCounters.egressQueueCounters = EgressQueueCounters()

   numUnguardedHooks = 0
   for func, guard, detailSupported, _ in \
         showInterfacesQueueCountersHook.extensions():
      if detail and not detailSupported:
         continue

      if guard( mode, None ) is None:
         queueCounters = func( mode, intf=intf, mod=mod, detail=detail )
         numUnguardedHooks += 1

   assert numUnguardedHooks <= 1, "Found too many possible results"
   return queueCounters

def showInterfacesQueueCountersRate( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   rates = 'rates' in args

   queueCounters = QueueCountersRate()
   queueCounters.egressQueueCounters = EgressQueueCounters()

   numUnguardedHooks = 0
   # guard for specific platforms, ie. strataguard
   for func, guard, _, rateSupported in showInterfacesQueueCountersHook.extensions():
      if rates and not rateSupported:
         continue

      if guard( mode, None ) is None:
         queueCounters = func( mode, intf=intf, mod=mod, rates=rates )
         numUnguardedHooks += 1

   assert numUnguardedHooks <= 1, "Found too many possible results"
   return queueCounters

def showInterfacesUcastQueueDropCounters( mode, args ):
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   dropsUnicast = 'drops' in args and 'unicast' in args

   queueCounters = QueueDropCounters()
   queueCounters.egressQueueDropCounters = EgressQueueDropCounters()

   numUnguardedHooks = 0
   for func, guard, detailSupported, _ in \
         showInterfacesQueueCountersHook.extensions():
      if dropsUnicast and not detailSupported:
         continue

      if guard( mode, None ) is None:
         queueCounters = func( mode, intf=intf, mod=mod, dropsUnicast=dropsUnicast )
         numUnguardedHooks += 1

   assert numUnguardedHooks <= 1, "Found too many possible results"
   return queueCounters

queueCountersKw = CliCommand.guardedKeyword( 'queue', 'Queue counters',
                                             countersQueueGuard )

class ShowIntfCountersQueue( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters queue [ detail ]'
   data = dict( counters=IntfCli.countersKw,
                queue=queueCountersKw,
                detail=CliCommand.guardedKeyword( 'detail',
                                                  'Queue counters in detail',
                                                  countersQueueDetailGuard ) )
   handler = showInterfacesQueueCounters
   cliModel = QueueCounters

BasicCli.addShowCommandClass( ShowIntfCountersQueue )

class ShowIntfCountersQueueRates( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters queue rates'
   data = dict( counters=IntfCli.countersKw,
                queue=queueCountersKw,
                rates=CliCommand.guardedKeyword( 'rates',
                                                 'Queue counters rates',
                                                 countersQueueRateGuard ) )
   handler = showInterfacesQueueCountersRate
   cliModel = QueueCountersRate

BasicCli.addShowCommandClass( ShowIntfCountersQueueRates )

queueDropCountersKw = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'drops', helpdesc='Queue drop counters' ),
   hidden=True,
   guard=countersQueueDetailGuard )

ucastQueueDropCountersKw = CliCommand.Node(
   CliMatcher.KeywordMatcher( 'unicast', helpdesc='Unicast queue drop counters' ),
   hidden=True,
   guard=countersQueueDetailGuard )

class ShowIntfCountersQueueDropsUnicast( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces counters queue drops unicast'
   data = dict( counters=IntfCli.countersKw,
                queue=queueCountersKw,
                drops=queueDropCountersKw,
                unicast=ucastQueueDropCountersKw )
   handler = showInterfacesUcastQueueDropCounters
   cliModel = QueueDropCounters

BasicCli.addShowCommandClass( ShowIntfCountersQueueDropsUnicast )

def Plugin( em ):
   global entityManager
   entityManager = em

