# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import ShowCommand
import CliParser
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.FabricIntfCli as FabricIntfCli
import CliPlugin.LagIntfCli as LagIntfCli
import CliPlugin.QosCli as QosCli
import CliPlugin.VlanIntfCli as VlanIntfCli
from CliPlugin.QosCliModel import (
      EcnIntfQueueCountersModel,
      WredIntfQueueCountersModel,
      IntfEcnCollectionModel,
      PolicyMapAllModel,
      ClassMapAllModel,
      IntfTrustModeCollectionModel,
      EcnCountersModel,
      IntfAllQosAllModel,
      GlobalEcnModel,
      GlobalQosMapsModel,
      IntfAllEcnCollectionModel,
      DscpRewriteMapAllModel,
      GlobalAllEcnModel,
      FabricAllQosAllModel,
      ClassMapWrapper,
      IntfWredCollectionModel,
      EcnDelayModel,
      QosAllSchedulingGroupModel,
      QosAllSchedulingHierarchyModel,
)
from CliPlugin.QosCli import (
      nodeQosForShow,
      QosProfileMode,
      PolicyMapModeQos,
      ClassMapModeQos
)
from IntfRangePlugin.VxlanIntf import VxlanAutoIntfType
from Intf.IntfRange import IntfRangeMatcher

matcherType = CliMatcher.KeywordMatcher( 'type', 
      helpdesc='Specify type' )
matcherInterfaces = CliMatcher.KeywordMatcher( 'interfaces',
      helpdesc='Show QoS status for a specific interface' )
matcherMls = CliMatcher.KeywordMatcher( 'mls',
      helpdesc='MLS global commands' )
matcherRandomDetect = CliMatcher.KeywordMatcher( 'random-detect',
      helpdesc='Show WRED based congestion control parameters' )
matcherIntf = IntfRangeMatcher( explicitIntfTypes=( 
      EthIntfCli.EthPhyAutoIntfType, LagIntfCli.LagAutoIntfType ) )
matcherPmapName = CliMatcher.DynamicNameMatcher( QosCli.getPMapNameQos,
      'Policy Map Name', 
      pattern='(?!copp|counters$|interface$|summary$)[A-Za-z0-9_:{}\\[\\]-]+' )
nodeMls = CliCommand.Node( matcher=matcherMls, hidden=True )
nodeEcn = CliCommand.guardedKeyword( 'ecn', "Show ECN parameters", 
      guard=QosCli.guardEcn )
nodeWred = CliCommand.guardedKeyword( 'wred', "Show WRED parameters",
      guard=QosCli.guardWred )
nodeFabric = CliCommand.guardedKeyword( 'fabric', "Fabric interface", 
      guard=FabricIntfCli.isFabricSystem )
nodePolicyMap = CliCommand.guardedKeyword( 'policy-map', "Show Policy Map", 
      guard=QosCli.guardPolicyMap )
nodeQosType = CliCommand.guardedKeyword( 'qos', "QoS type", 
      guard=QosCli.guardPMapQos )
nodeScheduling = CliCommand.guardedKeyword( 'scheduling',
      helpdesc='Show QoS scheduling parameters',
      guard=QosCli.guardQosScheduling )

#--------------------------------------------------------------------------------
# show qos interfaces [ INTF ] ecn counters queue
#--------------------------------------------------------------------------------
class QosInterfacesEcnCountersQueueCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos interfaces [ INTF ] ecn counters queue'
   data = {
      'qos': nodeQosForShow, 
      'interfaces': matcherInterfaces,
      'INTF': IntfRangeMatcher( explicitIntfTypes=( 
         EthIntfCli.EthPhyAutoIntfType, ) ),
      'ecn': nodeEcn,
      'counters': CliCommand.guardedKeyword( 'counters', 
         "ECN interface counters", guard=QosCli.guardShowEcnQueueCounter ),
      'queue': 'ECN queue counters', 
   }

   handler = QosCli.showEcnIntfQueueCounters
   cliModel = EcnIntfQueueCountersModel

BasicCli.addShowCommandClass( QosInterfacesEcnCountersQueueCmd )

#--------------------------------------------------------------------------------
# show qos interfaces [ INTF ] wred counters queue
#--------------------------------------------------------------------------------
class QosInterfacesWredCountersQueueCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos interfaces [ INTF ] wred counters queue'
   data = {
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': IntfRangeMatcher( explicitIntfTypes=(
         EthIntfCli.EthPhyAutoIntfType, ) ),
      'wred': nodeWred,
      'counters': CliCommand.guardedKeyword( 'counters',
         "WRED interface counters", guard=QosCli.guardShowWredQueueCounter ),
      'queue': 'WRED queue counters',
   }

   handler = QosCli.showWredIntfQueueCounters
   cliModel = WredIntfQueueCountersModel

BasicCli.addShowCommandClass( QosInterfacesWredCountersQueueCmd )

#-----------------------------------------------------------------
# The show command in 'config-qos-profile' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------
class QosProfileModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ( pending | active | diff )'
   data = {
      'pending': 'Display the new qos profile configuration to be applied',
      'active': 'Display the current qos profile configuration',
      'diff': ( 'Display the diff between the current qos profile config and '
          'to be applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         QosCli.showQosProfileDiff( mode )
      elif 'active' in args:
         QosCli.showQosProfileActive( mode )
      else:
         QosCli.showQosProfilePending( mode )

QosProfileMode.addShowCommandClass( QosProfileModeShowCmd )

#--------------------------------------------------------------------------------
# show qos map traffic-class to dscp [ MAP ]
#--------------------------------------------------------------------------------
class QosMapTrafficClassToDscpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos map traffic-class to dscp [ MAP ]'
   data = {
      'qos': nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class', 
         "Show traffic-class Map Configuration", 
         guard=QosCli.guardEgressDscpRewrite ),
      'to': 'Configure a named map', 
      'dscp': CliCommand.guardedKeyword( 'dscp', 
         "Set Differentiated Services Code Point (DSCP) value", 
         guard=QosCli.guardTcToDscpMap ),
      'MAP': CliMatcher.DynamicNameMatcher( 
         lambda mode : QosCli.qosInputConfig.dscpRewriteMap, 'Map name', 
         pattern='(?!default-map$)[A-Za-z0-9_:{}\\[\\]-]+' ),
   }

   handler = QosCli.showDscpRewriteMaps
   cliModel = DscpRewriteMapAllModel

BasicCli.addShowCommandClass( QosMapTrafficClassToDscpCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF ]
#--------------------------------------------------------------------------------
class QosInterfacesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF ]'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
   }

   handler = QosCli.showInterfacesQos
   cliModel = IntfAllQosAllModel

BasicCli.addShowCommandClass( QosInterfacesCmd )

#--------------------------------------------------------------------------------
# show qos interfaces fabric [ detail ]
#--------------------------------------------------------------------------------
class QosInterfacesFabricCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos interfaces fabric [ detail ]'
   data = {
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'fabric': nodeFabric,
      'detail': 'More comprehensive output',
   }

   handler = QosCli.showInterfacesQos
   cliModel = FabricAllQosAllModel

BasicCli.addShowCommandClass( QosInterfacesFabricCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF ] trust
#--------------------------------------------------------------------------------
class QosInterfacesTrustCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF ] trust'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'trust': 'Show trust mode for the interface',
   }

   handler = QosCli.showInterfacesTrust
   cliModel = IntfTrustModeCollectionModel

BasicCli.addShowCommandClass( QosInterfacesTrustCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos random-detect ecn
#--------------------------------------------------------------------------------
class QosRandomDetectEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos random-detect ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'random-detect': CliCommand.guardedKeyword( 'random-detect', 
         "Show WRED based congestion control parameters", 
         guard=QosCli.guardGlobalEcn ),
      'ecn': nodeEcn,
   }
   hidden = True

   handler = QosCli.showGlobalEcn
   cliModel = GlobalEcnModel

BasicCli.addShowCommandClass( QosRandomDetectEcnCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos ecn
#--------------------------------------------------------------------------------
class QosEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'ecn': nodeEcn,
   }

   handler = QosCli.showGlobalEcn
   cliModel = GlobalAllEcnModel

BasicCli.addShowCommandClass( QosEcnCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF | fabric ] random-detect ecn
#--------------------------------------------------------------------------------
class QosInterfacesRandomDetectEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF | fabric ] random-detect ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'fabric': nodeFabric,
      'random-detect': matcherRandomDetect,
      'ecn': nodeEcn,
   }
   hidden = True

   handler = QosCli.showInterfacesEcn
   cliModel = IntfEcnCollectionModel

BasicCli.addShowCommandClass( QosInterfacesRandomDetectEcnCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF | fabric ] ecn
#--------------------------------------------------------------------------------
class QosInterfacesEcnCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF | fabric ] ecn'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf, 
      'fabric': nodeFabric,
      'ecn': nodeEcn,
   }

   handler = QosCli.showInterfacesEcn
   cliModel = IntfAllEcnCollectionModel

BasicCli.addShowCommandClass( QosInterfacesEcnCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos interfaces [ INTF ] random-detect drop
#--------------------------------------------------------------------------------
class QosInterfacesIntfRandomDetectDropCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos interfaces [ INTF ] random-detect drop'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'interfaces': matcherInterfaces,
      'INTF': matcherIntf,
      'random-detect': matcherRandomDetect,
      'drop': CliCommand.guardedKeyword( 'drop', "Show WRED parameters", 
         guard=QosCli.guardWred ),
   }

   handler = QosCli.showInterfacesWred
   cliModel = IntfWredCollectionModel

BasicCli.addShowCommandClass( QosInterfacesIntfRandomDetectDropCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos maps
#--------------------------------------------------------------------------------
class QosMapsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos maps'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'maps': 'Show various QoS mappings',
   }

   handler = QosCli.showQosMaps
   cliModel = GlobalQosMapsModel

BasicCli.addShowCommandClass( QosMapsCmd )

#--------------------------------------------------------------------------------
# show [ mls ] qos ecn delay
#--------------------------------------------------------------------------------
class QosEcnDelayCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ mls ] qos ecn delay'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForShow,
      'ecn': CliCommand.guardedKeyword( 'ecn', "Show general ECN parameters", 
         guard=QosCli.guardEcnShow ),
      'delay': CliCommand.guardedKeyword( 'delay', 
         "Show parameters for delay-based ECN", guard=QosCli.guardEcnDelay ),
   }
   hidden = True

   handler = QosCli.showEcnDelay
   cliModel = EcnDelayModel

BasicCli.addShowCommandClass( QosEcnDelayCmd )

#--------------------------------------------------------------------------------
# show qos ecn counters
#--------------------------------------------------------------------------------
class QosEcnCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos ecn counters'
   data = {
      'qos': nodeQosForShow,
      'ecn': CliCommand.guardedKeyword( 'ecn', "Show general ECN parameters", 
         guard=QosCli.guardEcnShow ),
      'counters': CliCommand.guardedKeyword( 'counters', 
         "Show per chip ECN counters", guard=QosCli.guardEcnCounters ),
   }

   handler = QosCli.showEcnCounters
   cliModel = EcnCountersModel

BasicCli.addShowCommandClass( QosEcnCountersCmd )

#--------------------------------------------------------------------------------
# show qos scheduling group [ GROUP ] [ INTFS ]
#--------------------------------------------------------------------------------
class QosScheduleingGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos scheduling group [ GROUP ] [ INTFS ]'
   data = {
      'qos': nodeQosForShow,
      'scheduling': nodeScheduling,
      'group': 'Show QoS scheduling group parameters',
      'GROUP': CliMatcher.DynamicNameMatcher(
         QosCli.getAllSchedulingGroups,
         pattern=r'(?!scheduling-group$)[A-Za-z0-9_:{}\[\]-]+',
         helpname='group-name',
         helpdesc='Scheduling group name',
         priority=CliParser.PRIO_LOW ),
      'INTFS': IntfRangeMatcher( explicitIntfTypes=(
         EthIntfCli.EthPhyAutoIntfType, ) ),
   }

   handler = QosCli.showQosSchedulingGroup
   cliModel = QosAllSchedulingGroupModel

BasicCli.addShowCommandClass( QosScheduleingGroupCmd )

#--------------------------------------------------------------------------------
# show qos scheduling hierarchy [ INTFS ]
#--------------------------------------------------------------------------------
class QosSchedulingHierarchyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos scheduling hierarchy [ INTFS ]'
   data = {
      'qos': nodeQosForShow,
      'scheduling': nodeScheduling,
      'hierarchy': 'Show QoS scheduling hierarchy parameters',
      'INTFS': IntfRangeMatcher( explicitIntfTypes=(
         EthIntfCli.EthPhyAutoIntfType, ) ),
   }
   handler = QosCli.showQosSchedulingHierarchy
   cliModel = QosAllSchedulingHierarchyModel

BasicCli.addShowCommandClass( QosSchedulingHierarchyCmd )
#-----------------------------------------------------------------
# The show command in 'config-cmap' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------
class CmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new class-map configuration to be applied',
      'active': 'Display the class-map configuration in the current running-config',
      'current': ( 'Display the class-map configuration '
         'in the current running-config' ),
      'diff': ( 'Display the diff between class-map configuration '
          'current running-config and to be applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         QosCli.showCMapDiff( mode )
      elif 'active' in args or 'current' in args:
         QosCli.showCMapActive( mode )
      else:
         QosCli.showCMapPending( mode )

ClassMapModeQos.addShowCommandClass( CmapModeShowCmd )

#-----------------------------------------------------------------
# The show command in 'config-pmap' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------
class PmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new policy-map configuration to be applied',
      'active': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'current': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'diff': ( 'Display the diff between policy-map configuration'
          'current running-config and to be applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         QosCli.showPMapDiff( mode )
      elif 'active' in args or 'current' in args:
         QosCli.showPMapActive( mode )
      else:
         QosCli.showPMapCurrent( mode )

PolicyMapModeQos.addShowCommandClass( PmapModeShowCmd )

#--------------------------------------------------------------------------------
# show class-map [ type qos ] [ CMAP ]
#--------------------------------------------------------------------------------
class ClassMapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show class-map [ type qos ] [ CMAP ]'
   data = {
      'class-map': CliCommand.guardedKeyword( 'class-map', 
         "Class map configuration", guard=QosCli.guardClassMap ),
      'type': matcherType,
      'qos': nodeQosType,
      'CMAP': CliMatcher.DynamicNameMatcher( QosCli.getCMapNameQos, 
         'Class Map Name' ),
   }

   handler = QosCli.ClassMapModeQos.showClassMap
   cliModel = ClassMapAllModel

BasicCli.addShowCommandClass( ClassMapCmd )

#--------------------------------------------------------------------------------
# show policy-map [ type qos ] [ PMAP ] [ summary ]
#--------------------------------------------------------------------------------
class PolicyMapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] [ PMAP ] [ summary ]'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'PMAP': matcherPmapName,
      'summary': 'Policy Map summary',
   }

   handler = QosCli.PolicyMapModeQos.showPolicyMap
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapCmd )

#--------------------------------------------------------------------------------
# show policy-map [ type qos ] PMAP class CMAP
#--------------------------------------------------------------------------------
class PolicyMapPmapClassCmapCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] PMAP class CMAP'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'PMAP': matcherPmapName, 
      'class': 'Policy criteria',
      'CMAP': CliMatcher.DynamicNameMatcher( QosCli.getClassNameRuleQos, 
         'Class Map Name' ),
   }

   handler = QosCli.PolicyMapClassMode.showPolicyMapClass
   cliModel = ClassMapWrapper

BasicCli.addShowCommandClass( PolicyMapPmapClassCmapCmd )

#--------------------------------------------------------------------------------
# show policy-map [ type qos ] PMAP [ input ] counters [ detail ]
#--------------------------------------------------------------------------------
class PolicyMapPmapCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] PMAP [ input ] counters [ detail ]'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'PMAP': matcherPmapName, 
      'input': 'Policy-map at input of an interface',
      'counters': 'Policy Map counters',
      'detail': 'More comprehensive output',
   }

   handler = QosCli.showPolicyMapCounters
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapPmapCountersCmd )

#--------------------------------------------------------------------------------
# show policy-map [ type qos ] [ interface ] counters 
#--------------------------------------------------------------------------------
class PolicyMapPmapInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map [ type qos ] [ interface ] counters'
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'qos': nodeQosType,
      'interface': 'Service Policy on interface',
      'counters': 'Policy Map counters',
   }

   handler = QosCli.showPolicyMapCounters
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapPmapInterfaceCmd )

#--------------------------------------------------------------------------------
# show policy-map interface ( INTF | SVI_INTF | VTI_INTF ) 
#       [ type qos ] [ input ] [ counters ]
#--------------------------------------------------------------------------------
class PolicyMapInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map interface ( INTF | SVI_INTF | VTI_INTF ) ' 
              '[ type qos ] [ input ] [ counters ]' )
   data = {
      'policy-map': nodePolicyMap,
      'interface': 'Service Policy on interface',
      'INTF': matcherIntf,
      'SVI_INTF': CliCommand.Node( matcher=IntfRangeMatcher( explicitIntfTypes=( 
         VlanIntfCli.VlanAutoIntfType, ) ), 
         guard=QosCli.guardSviPolicyQosSupported ),
      'VTI_INTF': CliCommand.Node( matcher=IntfRangeMatcher( explicitIntfTypes=[ 
         VxlanAutoIntfType ] ), guard=QosCli.vxlanPolicyQosSupported ),
      'type': matcherType,
      'qos': nodeQosType,
      'input': 'Policy-map at input of an interface',
      'counters': 'Policy Map counters',
   }

   handler = QosCli.showPMapInterface
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapInterfaceCmd )
