#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.QosCli as QosCli
from CliPlugin.QosCli import (
      guardNonEctParams, guardNonEctWeight, guardEcnWeight, guardEcn,
      guardWred, guardEcnMarkProb, guardWredWeight,
      RandomDetectCmdsThdWithLatencyFactory, RandomDetectCmdsThdFactory,
      nodeRandomDetect
)
import MultiRangeRule

matcherDroprate = CliMatcher.IntegerMatcher( 1, 100, 
      helpdesc='Specify maximum drop rate' )
matcherMinimumThresholdEcn = CliMatcher.KeywordMatcher( 'minimum-threshold',
      helpdesc='Set minimum threshold for ECN' )
thresholdMatcher = RandomDetectCmdsThdFactory()
thresholdLatencyMatcher = RandomDetectCmdsThdWithLatencyFactory()
matcherSubIntfTxQueueRange = MultiRangeRule.MultiRangeMatcher( 
      rangeFn=QosCli.subIntfTxQueueRangeFn, noSingletons=False, 
      helpdesc="Tx-Queue ID or range(s) of Tx-Queue IDs" )

nodeSubIntfTxQueue = CliCommand.guardedKeyword( 'tx-queue', 
      helpdesc='Configure transmit queue parameters',
      guard=QosCli.guardSubIntfTxQueue )
nodeTxQueue = CliCommand.guardedKeyword( 'tx-queue', 
      helpdesc='Configure transmit queue parameters', guard=QosCli.guardTxQueue )
nodeUcTxQueue = CliCommand.guardedKeyword( 'uc-tx-queue', 
      helpdesc='Configure unicast transmit queue parameters', 
      guard=QosCli.guardUcMcTxQueue )
nodeMcTxQueue = CliCommand.guardedKeyword( 'mc-tx-queue', 
      helpdesc='Configure multicast transmit queue parameters', 
      guard=QosCli.guardUcMcTxQueue )

#--------------------------------------------------------------------------------
# [ no | default ] random-detect drop minimum-threshold <MIN_THD> <unit> 
# maximum-threshold <MAX_THD> <unit> drop-probability MAX_DROPRATE [ weight WEIGHT ]
# unit can be one of bytes, kbytes, mbytes, segments, milliseconds, or nanoseconds
#--------------------------------------------------------------------------------
class RandomDetectDropCmds( CliCommand.CliCommandClass ):
   syntax = 'random-detect drop minimum-threshold THD_EXPR '\
            'drop-probability MAX_DROPRATE [ weight WEIGHT ]'
   noOrDefaultSyntax = 'random-detect drop ...'
   data = {
      'random-detect': nodeRandomDetect,
      'drop': CliCommand.guardedKeyword( 'drop',
                                         helpdesc='Set WRED parameters', 
                                         guard=guardWred ),
      'minimum-threshold': 'Set minimum threshold for WRED',
      'THD_EXPR': thresholdLatencyMatcher,
      'drop-probability': 'Set maximum drop probability rate',
      'MAX_DROPRATE': matcherDroprate,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set WRED weight',
                                           guard=guardWredWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           QosCli.weightRangeFn,
                                           helpdesc='Value of WRED weight' ),
   }

   adapter = QosCli.randomDetectCmdsAdapter
   handler = QosCli.setWred
   noOrDefaultHandler = QosCli.setNoOrDefaultWred

QosCli.IntfUcTxQueueModelet.addCommandClass( RandomDetectDropCmds )

#--------------------------------------------------------------------------------
# [ no | default ] random-detect ecn minimum-threshold <MIN_THD> <unit> 
# maximum-threshold <MAX_THD> <unit> [ max-mark-probability MAX_DROPRATE ]
# [ weight WEIGHT ]
# unit can be one of bytes, kbytes, mbytes, segments, milliseconds, or nanoseconds
#--------------------------------------------------------------------------------
class RandomDetectEcnCmds( CliCommand.CliCommandClass ):
   syntax = 'random-detect ecn minimum-threshold THD_EXPR '\
            '[ max-mark-probability MAX_DROPRATE ] [ weight WEIGHT ]'
   noOrDefaultSyntax = 'random-detect ecn minimum-threshold...'
   data = {
      'random-detect': nodeRandomDetect,
      'ecn':  CliCommand.guardedKeyword( 'ecn', 
                                         helpdesc='Set ECN parameters',
                                         guard=guardEcn ),
      'minimum-threshold': matcherMinimumThresholdEcn,
      'THD_EXPR': thresholdLatencyMatcher,
      'max-mark-probability': CliCommand.guardedKeyword( 'max-mark-probability',
                                    helpdesc='Set maximum marking probability rate',
                                    guard=guardEcnMarkProb ),
      'MAX_DROPRATE': matcherDroprate,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set ECN weight',
                                           guard=guardEcnWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher(
                                           QosCli.weightRangeFn,
                                           helpdesc='Value of ECN weight' ),
   }

   adapter = QosCli.randomDetectCmdsAdapter
   handler = QosCli.setEcn
   noOrDefaultHandler = QosCli.setNoOrDefaultEcn

QosCli.IntfUcTxQueueModelet.addCommandClass( RandomDetectEcnCmds )

#--------------------------------------------------------------------------------
# [ no | default ] random-detect non-ect minimum-threshold <MIN_THD> <unit> 
# maximum-threshold <MAX_THD> <unit> [ weight WEIGHT ]
# unit can be either bytes, kbytes, mbytes, or segments
#--------------------------------------------------------------------------------
class RandomDetectNonEctCmd( CliCommand.CliCommandClass ):
   syntax = 'random-detect non-ect minimum-threshold THD_EXPR [ weight WEIGHT ]'
   noOrDefaultSyntax = 'random-detect non-ect ...'
   data = {
      'random-detect': nodeRandomDetect,
      'non-ect': CliCommand.guardedKeyword( 'non-ect',
                                            helpdesc='Set NON-ECT parameters',
                                            guard=guardNonEctParams ),
      'minimum-threshold': matcherMinimumThresholdEcn,
      'THD_EXPR': thresholdMatcher,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set NON-ECT weight',
                                           guard=guardNonEctWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher( 
                                           QosCli.weightRangeFn, 
                                           helpdesc='Value of NON-ECT weight' ),
   }

   adapter = QosCli.randomDetectCmdsAdapter
   handler = QosCli.setNonEctParams
   noOrDefaultHandler = QosCli.setNoOrDefaultNonEctParams

QosCli.IntfUcTxQueueModelet.addCommandClass( RandomDetectNonEctCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ( ( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) |
#                                          ( mc-tx-queue MCTXQSET ) )
#--------------------------------------------------------------------------------
class QueueSetIntfRangeModeletCmd( CliCommand.CliCommandClass ):
   syntax = ( '( ( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) | '
              '( mc-tx-queue MCTXQSET ) )' )
   noOrDefaultSyntax = syntax
   data = {
      'tx-queue': nodeTxQueue,
      'uc-tx-queue': nodeUcTxQueue,
      'mc-tx-queue': nodeMcTxQueue,
      'TXQSET': QosCli.matcherTxQueueRange,
      'UCTXQSET': QosCli.matcherUcTxQueueRange,
      'MCTXQSET': QosCli.matcherMcTxQueueRange,
   }

   @staticmethod
   def handler( mode, args ):
      if 'tx-queue' in args:
         QosCli.goToTxQueueRangeMode( mode, 'tx-queue',
               args.get( 'TXQSET' ).values() )
      elif 'uc-tx-queue' in args:
         QosCli.goToTxQueueRangeMode( mode, 'uc-tx-queue',
               args.get( 'UCTXQSET' ).values() )
      elif 'mc-tx-queue' in args:
         QosCli.goToTxQueueRangeMode( mode, 'mc-tx-queue',
               args.get( 'MCTXQSET' ).values() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      if 'tx-queue' in args:
         QosCli.setDefaultTxQRangeConfig( mode, True, 'tx-queue',
               args.get( 'TXQSET' ).values() )
      elif 'uc-tx-queue' in args:
         QosCli.setDefaultTxQRangeConfig( mode, True, 'uc-tx-queue',
               args.get( 'UCTXQSET' ).values() )
      elif 'mc-tx-queue' in args:
         QosCli.setDefaultTxQRangeConfig( mode, True, 'mc-tx-queue',
               args.get( 'MCTXQSET' ).values() )

QosCli.QosModelet.addCommandClass( QueueSetIntfRangeModeletCmd )
QosCli.QosProfileMode.addCommandClass( QueueSetIntfRangeModeletCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ( tx-queue TXQSET )
#--------------------------------------------------------------------------------
class QueueSetSubIntfRangeModeletCmd( CliCommand.CliCommandClass ):
   syntax = ( 'tx-queue TXQSET' )
   noOrDefaultSyntax = syntax
   data = {
      'tx-queue': nodeSubIntfTxQueue,
      'TXQSET': matcherSubIntfTxQueueRange,
   }

   @staticmethod
   def handler( mode, args ):
      QosCli.goToTxQueueRangeMode( mode, 'tx-queue',
                                   args.get( 'TXQSET' ).values() )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      QosCli.setDefaultTxQRangeConfig( mode, True, 'tx-queue',
                                       args.get( 'TXQSET' ).values() )

QosCli.QosSubIntfModelet.addCommandClass( QueueSetSubIntfRangeModeletCmd )
