#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from BasicCliModes import GlobalConfigMode
import CliCommand
import CliMatcher
import MultiRangeRule
import CliParser
import CliPlugin.QosCli as QosCli
import CliPlugin.PolicingCli as PolicingCli
from CliPlugin.QosCli import ( 
      guardGlobalEcn, guardEcn, guardEcnConfigureNonEct,
      RandomDetectCmdsThdFactory,
)
from CliPlugin.AclCli import inKwMatcher
from CliToken.Hardware import hardwareMatcherForConfig

nodeRandomDetect = CliCommand.guardedKeyword( 'random-detect',
      helpdesc='Set WRED based congestion control parameters (including ECN)',
      guard=guardGlobalEcn )
thresholdMatcher = RandomDetectCmdsThdFactory()

matcherMlsGlobal = CliMatcher.KeywordMatcher( 'mls',
      helpdesc='MLS global commands' )
matcherMapTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Mapping traffic-class values' )
matcherMultipleCosvalues = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( QosCli.tacCos.min, QosCli.tacCos.max ), noSingletons=False,
      helpdesc='Class of Service (COS) value(s) or range(s) of COS values' )
matcherExpValue = CliMatcher.IntegerMatcher( 0, 7,
      helpdesc='Exp value', priority=CliParser.PRIO_HIGH )
matcherMultipleExpValue = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( QosCli.tacExp.min, QosCli.tacExp.max ), noSingletons=False,
      helpdesc='MPLS Exp value(s) or range(s) of Exp values' )
matcherPoliceCirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify lower rate' )
matchPoliceRateUnit = CliMatcher.EnumMatcher( {
   'bps': 'Rate in bps (default)',
   'kbps': 'The rate expressed in units of kilobits per second',
   'mbps': 'Rate in Mbps',
   'pps': 'Rate in pps',
} )
matcherPoliceCommittedBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst size' )
matcherMatchAny = CliMatcher.KeywordMatcher( 'match-any', 
      helpdesc='Logical-OR all match statements under this class-map' )
matcherSrcPMapTypeQos = CliMatcher.DynamicNameMatcher( QosCli.getPMapNameQos,
      helpdesc="Source policy map name" )
matcherCopy = CliMatcher.KeywordMatcher( 'copy', 
      helpdesc="Copy configuration from an existing policy map" )
matcherResource = CliMatcher.KeywordMatcher( 'resource',
      helpdesc="Hardware resources" )
matcherSharing = CliMatcher.KeywordMatcher( 'sharing', 
      helpdesc="Sharing behaviour of hardware resources" )
matcherAccessList = CliMatcher.KeywordMatcher( 'access-list',
      helpdesc="Hardware parameter related to access-list" )
nodeMlsGlobal = CliCommand.Node( matcher=matcherMlsGlobal, hidden=True )
nodeDropPrecedence = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc="Map drop-precedence value",
      guard=QosCli.guardDscpToDropPrecedenceMap )
nodeMapExp = CliCommand.guardedKeyword( 'exp', helpdesc="Map MPLS Exp value",
      guard=QosCli.guardExpToAndFromTcMap )
nodeToCos = CliCommand.guardedKeyword( 'cos',
      helpdesc="Set Class of Service (CoS) value", guard=QosCli.guardSetTctoCos )
nodeTo = CliCommand.guardedKeyword( 'to', helpdesc="Configure a named Map",
      guard=QosCli.guardEgressDscpRewrite )
nodeToExp = CliCommand.guardedKeyword( 'exp', helpdesc="Set MPLS Exp value",
      guard=QosCli.guardExpToAndFromTcMap )
nodeToTxQueue = CliCommand.guardedKeyword( 'tx-queue', helpdesc="Set TX queue",
      guard=QosCli.guardTxQueue )
nodeToUcTxQueue = CliCommand.guardedKeyword( 'uc-tx-queue', 
      helpdesc="Set unicast TX queue", guard=QosCli.guardUcMcTxQueue )
nodeMcTxQueue = CliCommand.guardedKeyword( 'mc-tx-queue', 
      helpdesc="Set multicast TX queue", guard=QosCli.guardUcMcTxQueue )
nodePriorityGroup = CliCommand.guardedKeyword( 'priority-group',
      helpdesc="Set priority group", guard=QosCli.guardTcToPriorityGroupMap )
nodeCosToTcProfileTo = CliCommand.guardedKeyword( 'to',
      helpdesc="Configure a named Cos To Tc Map",
      guard=QosCli.guardCosToTcPerInterface )
nodeRewriteCos = CliCommand.guardedKeyword( 'cos', 
      helpdesc="Set CoS rewrite", guard=QosCli.guardCosRewrite )
nodeRewriteDscp = CliCommand.guardedKeyword( 'dscp', 
      helpdesc="Set DSCP rewrite", guard=QosCli.guardDscpRewrite )
nodeEcnDelay = CliCommand.guardedKeyword( 'ecn', 
      helpdesc="General ECN configuration", guard=QosCli.guardEcnDelay )
nodeDelay = CliCommand.guardedKeyword( 'delay', 
      helpdesc="Configure parameters for delay-based ECN", 
      guard=QosCli.guardEcnDelay )
nodePolicer = CliCommand.guardedKeyword( 'policer', 
      helpdesc="Configure shared QoS policer", 
      guard=QosCli.guardNamedSharedPolicerSupported )
nodePoliceLRate = CliCommand.guardedKeyword( 'rate', helpdesc="Set lower Rate", 
      guard=QosCli.guardIngressPolicing )
nodePoliceLBs = CliCommand.guardedKeyword( 'burst-size',
      helpdesc="Set burst-size for lower rate", guard=QosCli.guardIngressPolicing )
nodeClassMapConfig = CliCommand.guardedKeyword( 'class-map', 
      helpdesc='Configure Class Map', guard=QosCli.guardClassMap )
nodePMapQos = CliCommand.guardedKeyword( 'qos', helpdesc="Qos type", 
      guard=QosCli.guardPMapQos )
nodePolicyMap = CliCommand.guardedKeyword( 'policy-map', 
      helpdesc="Configure Policy Map", guard=QosCli.guardPolicyMap )
nodeQualityOfService = CliCommand.guardedKeyword( 'quality-of-service', 
      helpdesc="Qos type",guard=QosCli.guardPMapQos )
nodeRoutedport = CliCommand.guardedKeyword( 'routedport', helpdesc="routed port",
      guard=QosCli.guardRoutedPortSubIntfAclSharing )
nodeSubinterface = CliCommand.guardedKeyword( 'subinterface', 
      helpdesc="sub-interface", guard=QosCli.guardRoutedPortSubIntfAclSharing )
nodeMapTypeQosDeprecated = CliCommand.guardedKeyword( 'qos', 
      helpdesc='Qos type', guard=QosCli.guardPMapQos,
      deprecatedByCmd='[no] policy-map type quality-of-service <pmap-name>' )
nodeDrop = CliCommand.guardedKeyword( 'drop', helpdesc="Drop packets",
                                      guard=QosCli.guardCmapDropCounter )
nodeCounter = CliMatcher.KeywordMatcher( 'counter', helpdesc='Count packets' )

#--------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos random-detect ecn allow non-ect
#--------------------------------------------------------------------------------
class QosRandomDetectEcnAllowNonEctCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos random-detect ecn allow non-ect'
   noOrDefaultSyntax = 'qos random-detect ecn allow non-ect'
   data = {
      'mls': 'MLS global commands',
      'qos': QosCli.nodeQosForConfig,
      'random-detect': nodeRandomDetect,
      'ecn': CliCommand.guardedKeyword( 'ecn',
                                        helpdesc='Set ECN parameters',
                                        guard=guardEcn ),
      'allow': CliCommand.guardedKeyword( 'allow',
                                          helpdesc='Allow forwarding of packets '
                                                   'regardless of WRED parameters',
                                          guard=guardEcnConfigureNonEct ),
      'non-ect': 'Non ECN-Capable Transport',
   }
   handler = QosCli.configureAllowNonEct
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosRandomDetectEcnAllowNonEctCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos random-detect ecn global-buffer minimum-threshold MINTHD segments 
# maximum-threshold MAXTHD segments [ weight GLOBALECNWEIGHTVALUE ]
#--------------------------------------------------------------------------------
class QosRandomDetectEcnGlobalBufferCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos random-detect ecn global-buffer minimum-threshold '\
            'THD_EXPR [ weight WEIGHT ]'
   noOrDefaultSyntax = '[ mls ] qos random-detect ecn global-buffer ...'
   data = {
      'mls': 'MLS global commands',
      'qos': QosCli.nodeQosForConfig,
      'random-detect': nodeRandomDetect,
      'ecn': CliCommand.guardedKeyword( 'ecn',
                                        helpdesc='Set ECN parameters',
                                        guard=guardEcn ),
      'global-buffer': 'Set global shared memory thresholds',
      'minimum-threshold': 'Set minimum threshold for ECN',
      'THD_EXPR': thresholdMatcher,
      'weight': CliCommand.guardedKeyword( 'weight',
                                           helpdesc='Set global ECN weight',
                                           guard=QosCli.guardGlobalEcnWeight ),
      'WEIGHT': CliMatcher.DynamicIntegerMatcher( 
                                           QosCli.weightRangeFn, 
                                           helpdesc='Value of global ECN weight' ),
   }

   adapter = QosCli.randomDetectCmdsAdapter
   handler = QosCli.setGlobalEcn
   noOrDefaultHandler = QosCli.setNoOrDefaultGlobalEcn

GlobalConfigMode.addCommandClass( QosRandomDetectEcnGlobalBufferCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos map cos to traffic-class name PROFILENAME
#--------------------------------------------------------------------------------
class QosMapCosToTrafficClassProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cos to traffic-class name PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'cos': QosCli.matcherMapCos,
      'to': nodeCosToTcProfileTo,
      'traffic-class': QosCli.matcherTrafficClass,
      'name': 'Name of the Cos-to-Tc Map',
      'PROFILENAME': QosCli.matcherCosToTcProfileName,
   }

   handler = QosCli.configureCosToTcProfile
   noOrDefaultHandler = QosCli.noCosToTcProfile

GlobalConfigMode.addCommandClass( QosMapCosToTrafficClassProfilenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos map dscp to traffic-class name MAP_NAME
#--------------------------------------------------------------------------------
class QosMapDscpToTrafficClassMapNameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map dscp to traffic-class name MAP_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'dscp': QosCli.nodeDscp,
      'to': QosCli.nodeDscpToTcMapTo,
      'traffic-class': QosCli.matcherTrafficClass,
      'name': 'Name of the Map',
      'MAP_NAME': QosCli.matcherDscpToTcMapName,
   }

   handler = QosCli.createDscpToTcMap
   noOrDefaultHandler = QosCli.removeDscpToTcMap

GlobalConfigMode.addCommandClass( QosMapDscpToTrafficClassMapNameCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos map dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class
#               TC_VALUE [ drop-precedence DROP_PRECEDENCE ]
# ( no | default ) [ mls ] qos map dscp
#               ( { DSCP_VALUE } | DSCP_VALUES ) ...
#--------------------------------------------------------------------------------
class QosMapDscpToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map dscp ( { DSCP_VALUE } | DSCP_VALUES ) to '
              'traffic-class TC_VALUE [ drop-precedence DROP_PRECEDENCE ] ' )
   noOrDefaultSyntax = ( '[ mls ] qos map dscp '
                         '( { DSCP_VALUE } | DSCP_VALUES ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'dscp': QosCli.matcherMapDscp,
      'DSCP_VALUE': QosCli.matcherDscpValue,
      'DSCP_VALUES': QosCli.matcherMultipleDscpValues,
      'to': 'QoS parameter to map DSCP to',
      'traffic-class': QosCli.matcherTrafficClass,
      'TC_VALUE': QosCli.matcherTrafficClassValue,
      'drop-precedence': nodeDropPrecedence,
      'DROP_PRECEDENCE': CliMatcher.IntegerMatcher(
         QosCli.validDropPrecedenceValues[ 0 ],
         QosCli.validDropPrecedenceValues[ -1 ], helpdesc='Drop precedence value' ),
   }

   handler = QosCli.setDscpToTcDp
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapDscpToTrafficClassCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos map ( ( cos ( { COS_VALUE } | COS_VALUES ) ) |
#                   ( exp ( { EXP_VALUE } | EXP_VALUES ) ) )
#                        to traffic-class TC_VALUE
# ( no | default ) [ mls ] qos map (
#                   ( cos ( { COS_VALUE } | COS_VALUES ) ) |
#                   ( exp ( { EXP_VALUE } | EXP_VALUES ) ) ) ...
#--------------------------------------------------------------------------------
class QosMapToTrafficClassToCosOrExpCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map ( '
              '( cos ( { COS_VALUE } | COS_VALUES ) ) | '
              '( exp ( { EXP_VALUE } | EXP_VALUES ) ) ) '
              'to traffic-class TC_VALUE' )
   noOrDefaultSyntax = ( '[ mls ] qos map ( '
                         '( cos ( { COS_VALUE } | COS_VALUES ) ) | '
                         '( exp ( { EXP_VALUE } | EXP_VALUES ) ) ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'cos': QosCli.matcherMapCos,
      'exp': nodeMapExp,
      'COS_VALUE': QosCli.matcherCosValue,
      'COS_VALUES': matcherMultipleCosvalues,
      'EXP_VALUE': matcherExpValue,
      'EXP_VALUES': matcherMultipleExpValue,
      'to': 'QoS parameter to map CoS to',
      'traffic-class': QosCli.matcherTrafficClass,
      'TC_VALUE': QosCli.matcherTrafficClassValue,
   }

   @staticmethod
   def handler( mode, args ):
      tc = args.get( 'TC_VALUE' )
      noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
      if 'cos' in args:
         cosList = args.get( 'COS_VALUE' ) or \
               args.get( 'COS_VALUES' ).values()
         QosCli.setCosToTc( mode, cosList, noOrDefaultKw, tc )
      elif 'exp' in args:
         expList = args.get( 'EXP_VALUE' ) or \
               args.get( 'EXP_VALUES' ).values()
         QosCli.setExpToTc( mode, expList, noOrDefaultKw, tc )

   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapToTrafficClassToCosOrExpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos map traffic-class to dscp MAPNAME
#--------------------------------------------------------------------------------
class QosMapTrafficClassToDscpMapnameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map traffic-class to dscp MAPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherMapTrafficClass,
      'to': nodeTo,
      'dscp': QosCli.nodeToDscp,
      'MAPNAME': QosCli.matcherDscpRewriteMapName,
   }

   handler = QosCli.configureDscpRewriteMap
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapTrafficClassToDscpMapnameCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC to
#                 ( ( cos COS) | ( dscp DSCP ) | ( exp EXP ) )
# ( no | default ) [ mls ] qos map traffic-class
#                 ( { TC_LIST } | TC_VALUE ) to ( cos | dscp | exp ) ...
#--------------------------------------------------------------------------------
class QosMapTrafficClassToCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC to '
              '( ( cos COS ) | ( dscp DSCP ) | ( exp EXP ) )' )
   noOrDefaultSyntax = ( '[mls] qos map traffic-class TC '
                         'to ( cos | dscp | exp ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': matcherMapTrafficClass,
      'TC': QosCli.TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'cos': nodeToCos,
      'COS': QosCli.matcherCosValue,
      'dscp': QosCli.nodeToDscp,
      'DSCP': QosCli.matcherDscpValue,
      'exp': nodeToExp,
      'EXP': matcherExpValue,
   }

   @staticmethod
   def handler( mode, args ):
      tcList = args.get( 'TC_VALUE' ) or args.get( 'TC_LIST' ).values()
      noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
      if 'cos' in args:
         QosCli.setTrafficClassToCos( mode, tcList, noOrDefaultKw,
               args.get( 'COS' ) )
      elif 'dscp' in args:
         QosCli.setTrafficClassToDscp( mode, tcList, noOrDefaultKw,
               args.get( 'DSCP' ) )
      elif 'exp' in args:
         QosCli.setTrafficClassToExp( mode, tcList, noOrDefaultKw,
               args.get( 'EXP' ) )

   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapTrafficClassToCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC to
#                 ( ( tx-queue TXQ_VALUE ) | ( uc-tx-queue UCTXQ_VALUE ) )
# ( no | default ) [ mls ] qos map traffic-class TC 
#                  to ( tx-queue | uc-tx-queue ) ...
#--------------------------------------------------------------------------------
class QosMapTrafficClassToTxQueueCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC to '
              '( ( tx-queue TXQ_VALUE ) | ( uc-tx-queue UCTXQ_VALUE ) )' )
   noOrDefaultSyntax = ( '[ mls ] qos map traffic-class TC '
                         'to ( tx-queue | uc-tx-queue ) ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': QosCli.matcherTrafficClass,
      'TC': QosCli.TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'tx-queue': nodeToTxQueue,
      'TXQ_VALUE': CliMatcher.DynamicIntegerMatcher( QosCli.txQueueRangeFn,
         helpdesc='Transmit queue id' ),
      'uc-tx-queue': nodeToUcTxQueue,
      'UCTXQ_VALUE': CliMatcher.DynamicIntegerMatcher( QosCli.ucTxQueueRangeFn,
         helpdesc='Unicast transmit queue ID' ),
   }

   handler = QosCli.setTrafficClassToTxQueueMap
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapTrafficClassToTxQueueCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC 
#         to mc-tx-queue MCTXQ_VALUE
# ( no | default ) [ mls ] qos map traffic-class TC 
#                          to mc-tx-queue ...
#--------------------------------------------------------------------------------
class QosMapTrafficClassToMcTxQueueCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC '
              'to mc-tx-queue MCTXQ_VALUE' )
   noOrDefaultSyntax = ( '[ mls ] qos map traffic-class TC '
                         'to mc-tx-queue ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': QosCli.matcherTrafficClass,
      'TC': QosCli.TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'mc-tx-queue': nodeMcTxQueue,
      'MCTXQ_VALUE': CliMatcher.DynamicIntegerMatcher( QosCli.mcTxQueueRangeFn,
         helpdesc='Multicast transmit queue ID' ),
   }

   handler = QosCli.setTrafficClassToMcTxQueueMap
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapTrafficClassToMcTxQueueCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos map traffic-class TC to
#                  priority-group PRIORITYGROUPVALUE
# ( no | default ) [ mls ] qos map traffic-class TC to
#                  priority-group ...
#--------------------------------------------------------------------------------
class QosMapTrafficClassToPriorityGroupCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos map traffic-class TC to '
              'priority-group PRIORITYGROUPVALUE' )
   noOrDefaultSyntax = ( '[ mls ] qos map traffic-class TC '
                         'to priority-group ...' )
   data = {
      'mls': nodeMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'map': QosCli.matcherMap,
      'traffic-class': QosCli.matcherTrafficClass,
      'TC': QosCli.TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'priority-group': nodePriorityGroup,
      'PRIORITYGROUPVALUE': CliMatcher.DynamicKeywordMatcher(
         QosCli.priorityGroupVal ),
   }

   handler = QosCli.setTrafficClassToPriorityGroupMap
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosMapTrafficClassToPriorityGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos profile PROFILENAME
#--------------------------------------------------------------------------------
class QosProfileProfilenameCmd( CliCommand.CliCommandClass ):
   syntax = 'qos profile PROFILENAME'
   noOrDefaultSyntax = syntax
   data = {
      'qos': QosCli.nodeQosForConfig,
      'profile': 'QoS profile',
      'PROFILENAME': QosCli.matcherQosProfileName,
   }

   handler = QosCli.gotoQosProfileMode
   noOrDefaultHandler = QosCli.deleteQosProfile

GlobalConfigMode.addCommandClass( QosProfileProfilenameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos rewrite cos
#--------------------------------------------------------------------------------
class QosRewriteCosCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos rewrite cos'
   noOrDefaultSyntax = syntax
   data = {
      'mls': matcherMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'rewrite': QosCli.matcherRewrite,
      'cos': nodeRewriteCos,
   }

   handler = QosCli.setCosRewrite
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosRewriteCosCmd )

#--------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos rewrite dscp
#--------------------------------------------------------------------------------
class QosRewriteDscpCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos rewrite dscp'
   noOrDefaultSyntax = syntax
   data = {
      'mls': matcherMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'rewrite': QosCli.matcherRewrite,
      'dscp': nodeRewriteDscp,
   }

   handler = QosCli.setDscpRewrite
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosRewriteDscpCmd )

#--------------------------------------------------------------------------------
# [ mls ] qos ecn delay threshold ECNDELAYTHRESHOLDUSVALUE [ microseconds ]
# [ mls ] qos ecn delay threshold ...
#--------------------------------------------------------------------------------
class QosEcnDelayThresholdEcndelayCmd( CliCommand.CliCommandClass ):
   syntax = ( '[ mls ] qos ecn delay threshold '
              'ECNDELAYTHRESHOLDUSVALUE [ microseconds ]' )
   noOrDefaultSyntax = '[ mls ] qos ecn delay threshold ...'
   data = {
      'mls': matcherMlsGlobal,
      'qos': QosCli.nodeQosForConfig,
      'ecn': nodeEcnDelay,
      'delay': nodeDelay,
      'threshold': 'Configure delay threshold',
      'ECNDELAYTHRESHOLDUSVALUE': QosCli.matcherEcnDelayThresholdUsValue,
      'microseconds': 'Microseconds',
   }

   handler = QosCli.setGlobalEcnDelayThresholdUs
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosEcnDelayThresholdEcndelayCmd )

#--------------------------------------------------------------------------------
# qos policer POLICER_NAME rate CIR_VALUE [ RATE_UNIT ] 
#                    burst-size BC_VALUE [ BURST_UNIT ]
# ( no | default ) qos policer POLICER_NAME
#--------------------------------------------------------------------------------
class QosPolicerNameCmd( CliCommand.CliCommandClass ):
   syntax = ( 'qos policer POLICER_NAME rate CIR_VALUE [ RATE_UNIT ] '
              'burst-size BC_VALUE [ BURST_UNIT ] ' )
   noOrDefaultSyntax = 'qos policer POLICER_NAME'
   data = {
      'qos': QosCli.nodeQosForConfig,
      'policer': nodePolicer,
      'POLICER_NAME': CliMatcher.DynamicNameMatcher( QosCli.getPolicerNameRule,
         helpdesc="Policer name" ),
      'rate': nodePoliceLRate,
      'CIR_VALUE': matcherPoliceCirValue,
      'RATE_UNIT': matchPoliceRateUnit,
      'burst-size': nodePoliceLBs,
      'BC_VALUE': matcherPoliceCommittedBurstValue,
      'BURST_UNIT': QosCli.matcherBurstUnit,
   }

   adapter = QosCli.rateBurstUnitAdapter

   @staticmethod
   def handler( mode, args ):
      QosCli.configureNamedPolicer( mode, no=False,
            policerName=args[ 'POLICER_NAME' ], cir=int( args[ 'CIR_VALUE' ] ),
            cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args[ 'BC_VALUE' ] ),
            bcUnit=args.get( 'BURST_BC_UNIT' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      QosCli.configureNamedPolicer( mode, no=no, 
            policerName=args[ 'POLICER_NAME' ] )

GlobalConfigMode.addCommandClass( QosPolicerNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] class-map [ type qos ] [ match-any ] CMAP
#--------------------------------------------------------------------------------
class ClassMapMatchAnyCmapnameCmd( CliCommand.CliCommandClass ):
   syntax = 'class-map [ type qos ] match-any CMAP'
   noOrDefaultSyntax = 'class-map [ type qos ] [ match-any ] CMAP'
   data = {
      'class-map': nodeClassMapConfig,
      'type': QosCli.matcherType,
      'qos': nodePMapQos,
      'match-any': matcherMatchAny,
      'CMAP': CliMatcher.DynamicNameMatcher( QosCli.getCMapNameQos,
                                       helpdesc='Class Map Name' ),
   }

   handler = QosCli.gotoQosClassMapMode

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      QosCli.deleteClassMap( mode, args[ 'CMAP' ], 'qos', 'match-any' )

GlobalConfigMode.addCommandClass( ClassMapMatchAnyCmapnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] policy-map type quality-of-service PMAP
#--------------------------------------------------------------------------------
class PolicyMapTypeQosPmapnameCmd( CliCommand.CliCommandClass ):
   syntax = 'policy-map type ( quality-of-service | qos ) PMAP'
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMap,
      'type': QosCli.matcherType,
      'quality-of-service': nodeQualityOfService,
      'qos': nodeMapTypeQosDeprecated,
      'PMAP': QosCli.matcherPMapNameTypeQos,
   }

   handler = QosCli.gotoQosPolicyMapMode
   noOrDefaultHandler = QosCli.deleteQosPolicyMap

GlobalConfigMode.addCommandClass( PolicyMapTypeQosPmapnameCmd )

#--------------------------------------------------------------------------------
# policy-map [ type quality-of-service ] DST_PMAP copy SRC_PMAP
#--------------------------------------------------------------------------------
class CopyQosPolicyMapCmd( CliCommand.CliCommandClass ):
   syntax = ( 'policy-map [ type ( quality-of-service | qos ) ] '
            'DST_PMAP copy SRC_PMAP' )
   data = {
      'policy-map': nodePolicyMap,
      'type': QosCli.matcherType,
      'quality-of-service': nodeQualityOfService,
      'qos': nodeMapTypeQosDeprecated,
      'DST_PMAP': QosCli.matcherPMapNameTypeQos,
      'copy': matcherCopy,
      'SRC_PMAP': matcherSrcPMapTypeQos,
   }

   handler = QosCli.copyQosPolicyMap

GlobalConfigMode.addCommandClass( CopyQosPolicyMapCmd )

#--------------------------------------------------------------------------------
# [ no | default ] policing
#--------------------------------------------------------------------------------
class PolicingModeCmd( CliCommand.CliCommandClass ):
   syntax = 'policing'
   data = {
      'policing': PolicingCli.nodePolicingMode,
   }

   handler = PolicingCli.gotoPolicingMode

GlobalConfigMode.addCommandClass( PolicingModeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list qos resource sharing vlan in
#--------------------------------------------------------------------------------
class HwAccessListQosResourceSharingVlanInCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list qos resource sharing vlan in'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'qos': QosCli.nodeQosForConfig,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'vlan': CliCommand.guardedKeyword( 'vlan',
         helpdesc='Share hardware resources when qos access-list is '
                  'attached to VLAN interfaces', guard=QosCli.guardSharingVlan ),
      'in': inKwMatcher,
   }

   handler = QosCli.enableQosAclSharing
   noOrDefaultHandler = QosCli.disableQosAclSharing

GlobalConfigMode.addCommandClass( HwAccessListQosResourceSharingVlanInCmd )

#--------------------------------------------------------------------------------
# [ no | default ] hardware access-list [ qos ] resource sharing 
#                                    routedport subinterface
#--------------------------------------------------------------------------------
class RoutedPortSubintfCmd( CliCommand.CliCommandClass ):
   syntax = 'hardware access-list [ qos ] resource sharing routedport subinterface'
   noOrDefaultSyntax = syntax
   data = {
      'hardware': hardwareMatcherForConfig,
      'access-list': matcherAccessList,
      'qos': QosCli.nodeQosForConfig,
      'resource': matcherResource,
      'sharing': matcherSharing,
      'routedport': nodeRoutedport,
      'subinterface': nodeSubinterface,
   }
   
   handler = QosCli.gotoRoutedPortSubIntfAclSharing
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( RoutedPortSubintfCmd )

#--------------------------------------------------------------------------------
# qos scheduling
#--------------------------------------------------------------------------------
class QosSchedulingConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'qos scheduling'
   data = {
      'qos': QosCli.nodeQosForConfig,
      'scheduling': QosCli.nodeScheduling,
   }
   handler = QosCli.goToQosSchedulingMode

GlobalConfigMode.addCommandClass( QosSchedulingConfigCmd )

#--------------------------------------------------------------------------------
# [ no | default ] policy-map type quality-of-service policer drop counter
#--------------------------------------------------------------------------------
class ClassMapDropCounterCmd( CliCommand.CliCommandClass ):
   syntax = 'policy-map type quality-of-service policer drop counter'
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMap,
      'type': QosCli.matcherType,
      'quality-of-service': nodeQualityOfService,
      'policer' : CliMatcher.KeywordMatcher( 'policer',
                                             helpdesc="policy-map policer" ),
      'drop' : nodeDrop,
      'counter' : nodeCounter,
   }

   handler = QosCli.enableQosAclCmapDropCounter
   noOrDefaultHandler = QosCli.disableQosAclCmapDropCounter

GlobalConfigMode.addCommandClass( ClassMapDropCounterCmd )

#--------------------------------------------------------------------------------
# [ no | default ] qos subinterface tx-queue count <4|8>
#--------------------------------------------------------------------------------
class QosNumTxqsPerSubIntfCmd( CliCommand.CliCommandClass ):
   syntax = 'qos subinterface tx-queue count NUMTXQS'
   noOrDefaultSyntax = 'qos subinterface tx-queue count ...'
   data = {
      'qos': QosCli.nodeQosForConfig,
      'subinterface' : 'Configure subinterface parameters',
      'tx-queue': QosCli.nodeTxQueue,
      'count' : CliCommand.guardedKeyword( 'count',
                   helpdesc='Configure number of transmit queues per subinterface',
                   guard=QosCli.guardNumTxqsPerSubIntf ),
      'NUMTXQS' : CliMatcher.EnumMatcher( {
         '4' : 'Transmit queues',
         '8' : 'Transmit queues',
       } ),
   }
   handler = QosCli.handleNumTxqsPerSubIntf
   noOrDefaultHandler = handler

GlobalConfigMode.addCommandClass( QosNumTxqsPerSubIntfCmd )

