#!/usr/bin/env python
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import Cell
import CliCommand
import CliMatcher
import CliParser
import CliToken
import ConfigMount
import IntfCli
import LazyMount
import Plugins
import QosLib
import ShowCommand
import Tac

from FabricIntfCli import fabricIntfKw
from PfcTypes import tacPfcWatchdogAction, tacPfcStatus
from CliPlugin import TechSupportCli
from QosCli import ( QosProfileMode, setQosProfileConfig, counterSupportedEthIntfs,
                     counterSupportedEthPhyIntfs )
from QosCliModel import ( PfcStatusModel, PfcWatchdogTimerModel,
                          PfcWatchdogNDPrioritiesModel, PfcWatchdogStatusModel,
                          PfcIntfStatusModel, PfcFrameCountersModel,
                          PfcIntfDetailedCountersModel,
                          PfcWatchdogTxQueueCountersModel,
                          PfcWatchdogEgressDropCountersModel,
                          PfcWatchdogIngressDropCountersModel,
                          PfcWatchdogIntfCountersModel, PfcFabricIntfStatusModel,
                          PfcIntfRangeStatusModel, PfcIntfRangeCountersModel,
                          PfcWatchdogIntfEgressDropCountersModel,
                          PfcWatchdogIntfRangeDropCountersModel,
                          PfcIntfRangeDetailedCountersModel,
                          PfcWatchdogIntfRangeCountersModel,
                          PfcIntfRangeModel )
from QosCliModel import ( numPriority, fabricIntfName, fabricShortName )

#------------------------------------------------------------------------------------
# The Pfc globals
#------------------------------------------------------------------------------------
qosInputConfig = None
qosProfileConfigDir=None
pfcStatus = None
qosHwStatus = None
qosSliceHwStatus = None
qosConfig = None
dcbxConfig = None
pfcConfig = None

#------------------------------------------------------------------------------------
# Some CONSTANTS
#------------------------------------------------------------------------------------
PFC_WATCHDOG_INTERMEDIATE_BOUNDARY = 0.2

def isDefaultPfcPortTimerConfig( portTimerConfig ):
   return ( ( not portTimerConfig ) or
            ( portTimerConfig.usePerPortTimerValues == False ) )

def isDefaultPfcPortConfig( pfcPortConfig ):
   if pfcPortConfig is None:
      return True
   return ( ( pfcPortConfig.enabled == False ) and
            ( pfcPortConfig.priorities == 0 ) and
            ( pfcPortConfig.watchdogEnabled == True ) and
            ( pfcPortConfig.watchdogPortAction == \
              tacPfcWatchdogAction.invalid ) and
            ( isDefaultPfcPortTimerConfig( pfcPortConfig.portTimerConfig )
            ) )
#------------------------------------------------------------------------------------
# Pfc module to be registered with Qos
#------------------------------------------------------------------------------------
class QosPfcModuleBase( QosLib.QosModuleBase ):
   def isDefaultIntfConfig( self, intfConfig ):
      cfgPfcPortConfig = intfConfig.pfcPortConfig
      return isDefaultPfcPortConfig( cfgPfcPortConfig )

QosLib.registerQosModule( 'pfc', QosPfcModuleBase( qosHwStatus ) )

#------------------------------------------------------------------------------------
# Pfc
#------------------------------------------------------------------------------------
class pfcModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      # This is a bit hacky, but there's no better way to recognize physical Ethernet
      # interfaces than to look at their name.
      # CliParser.Modelet.__init__( self ) # This is a no-op
      return ( mode.intf.name.startswith( 'Ethernet' ) and
               not mode.intf.isSubIntf() )

class PfcIntf( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intf = self.intf_.name
      intfConfig = qosInputConfig.intfConfig.get( intf )
      if intfConfig and intfConfig.pfcPortConfig:
         intfConfig.pfcPortConfig = None

IntfCli.IntfConfigMode.addModelet( pfcModelet )

def prioritiesString( priorities, separator="" ):
   return separator.join( map( str, ( i for i in range( 8 )
                                        if ( 1 << i ) & priorities ) ) )

def hardwareSupports( mode, priorities ):
   if mode.session.disableGuards_:
      return True

   supported = pfcStatus.prioritiesSupported
   if supported == 0:
      mode.addError( "The hardware does not support priority flow control, "
                     "or forwarding agent is still initializing." )
      return False
   elif ( supported & priorities ) == 0:
      mode.session_.addError( "The hardware only supports PFC on priorities",
                              prioritiesString( supported, " " ) )
      return False
   return True

def maybeDeletePortConfig( intfName ):
   intfConfig = qosInputConfig.intfConfig.get( intfName )
   if intfConfig:
      if isDefaultPfcPortConfig( intfConfig.pfcPortConfig ):
         intfConfig.pfcPortConfig = None
      if QosLib.isDefaultIntfConfig( qosHwStatus, intfConfig ):
         del qosInputConfig.intfConfig[ intfName ]

# priority-flow-control mode on
# no priority-flow-control mode
# no priority-flow-control

def pfcOff( mode, args ):
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig:
      pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      if pfcPortConfig:
         qosInputConfig.intfConfig[ intf ].pfcPortConfig.enabled = False
      maybeDeletePortConfig( intf )

def pfcOn( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   intf = mode.intf.name
   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]
   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, True, )

   intfConfig.pfcPortConfig.enabled = True

def noPfc( mode, args ):
   """Turn off PFC completely, including priorities."""
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig and \
          qosInputConfig.intfConfig[ intf ].pfcPortConfig:
      qosInputConfig.intfConfig[ intf ].pfcPortConfig = None
   maybeDeletePortConfig( intf )

# enable pfc watchdog on per interface basis
def pfcWatchdogOnOff( mode, args ):
   # by default pfc watchdog is on
   intf = mode.intf.name
   no = CliCommand.isNoCmd( args )
   if intf in qosInputConfig.intfConfig and \
         qosInputConfig.intfConfig[ intf ].pfcPortConfig:
      qosInputConfig.intfConfig[ intf ].pfcPortConfig.watchdogEnabled = not no
   elif no:
      config = qosInputConfig.intfConfig.newMember( intf )
      config.pfcPortConfig = ( intf, False )
      config.pfcPortConfig.watchdogEnabled = False
   maybeDeletePortConfig( intf )

def configWatchdogPerPortAction( mode, args ):

   intf = mode.intf.name

   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]

   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, False, )

   if 'errdisable' in args:
      intfConfig.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.errdisable
   elif 'drop' in args:
      intfConfig.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.drop
   elif 'notify-only' in args:
      intfConfig.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.notifyOnly

def disableConfigWatchdogPerPortAction( mode, args ):
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig:
      pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      if pfcPortConfig:
         pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.invalid
      maybeDeletePortConfig( intf )

# priority-flow-control pause watchdog default timeout < 0.01 - 60 > s
# priority-flow-control pause watchdog default recovery-time < 0.01 -60 > s
# no priority-flow-control pause watchdog

def pfcWatchdogGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   return None

def pfcWatchdogPortConfigGuard( mode, token ):
   if not pfcStatus.watchdogPortConfigSupported:
      return CliParser.guardNotThisPlatform
   return None

def pfcWatchdogErrdisableGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   else:
      if pfcStatus.watchdogErrdisableSupported:
         return None
      else:
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if sliceHwStatus.watchdogErrdisableSupported:
               return None
            else:
               return CliParser.guardNotThisPlatform

def pfcWatchdogNotifyOnlyGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   else:
      if pfcStatus.watchdogNotifyOnlySupported:
         return None
      else:
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if sliceHwStatus.watchdogNotifyOnlySupported:
               return None
            else:
               return CliParser.guardNotThisPlatform

def pfcWatchdogDropGuard( mode, token ):
   if not pfcStatus.watchdogSupported:
      return CliParser.guardNotThisPlatform
   else:
      if pfcStatus.watchdogDropSupported:
         return None
      else:
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if sliceHwStatus.watchdogDropSupported:
               return None
            else:
               return CliParser.guardNotThisPlatform

# This function takes the value of the watchdog attribute that conflicts
# with polling-interval and corrects the polling-interval as follows:
# here timeoutVal can either be timeout/recoveryTime
def autoCorrectPfcWatchdogPollingInterval( timeoutVal ):
   # default value is 100 milli seconds
   operPollingInterval = 0.1
   # But polling-interval for watchdog should be at max half of timeoutVal
   # If not, set operPollingInterval to half of timeoutVal for watchdog to work
   if( timeoutVal < PFC_WATCHDOG_INTERMEDIATE_BOUNDARY ):
      operPollingInterval = ( 0.5 * timeoutVal )
      operPollingInterval = round( operPollingInterval, 3 )
   return operPollingInterval

def addPfcWatchdogWarning( mode, conflictCfgType,
                           timeout, recoveryTime, pollingInterval ):
   if( conflictCfgType == "timeout" ):
      conflictCfgTimeInterval = timeout
      operPollingInterval = autoCorrectPfcWatchdogPollingInterval( timeout )
   elif( conflictCfgType == "recovery-time" ):
      conflictCfgTimeInterval = recoveryTime
      operPollingInterval = autoCorrectPfcWatchdogPollingInterval( recoveryTime )
   warningMessage = "User configured polling-interval %.5g second(s) is " \
                    "greater than half of %s %.4g second(s). Setting " \
                    "polling-interval to %.5g second(s)" % \
                    ( pollingInterval, conflictCfgType,
                      conflictCfgTimeInterval, operPollingInterval )
   mode.addWarning( warningMessage )

# Rules for verifying if the config values are in coherence/conflict
# 1. Coherency check when pollingInterval is configured:
#    pollingInterval <= 0.5 * timeout or pollingInterval <= 0.5 * recoveryTime
# 2. No conflict when timeout is 0 even if configured values are incoherent
# 3. If recovery-mode is forced, no conflict even if there is incoherence
#    between recoveryTime and pollingInterval values
# 4. When both timeout and recoveryTime are incoherent with pollingInterval,
#    the we will return config type which has minimum value
def getPfcWatchdogCfgConflictDetails( perPort=False, intf=None, profile=None ):
   if( not perPort ):
      timeout = qosInputConfig.watchdogTimeout
      recoveryCfg = qosInputConfig.watchdogRecoveryCfg
      recoveryTime = qosInputConfig.watchdogRecoveryCfg.recoveryTime
      forcedRecovery = qosInputConfig.watchdogRecoveryCfg.forcedRecovery
      pollingInterval = qosInputConfig.watchdogPollingInterval
   else:
      # per port timer values
      if intf:
         pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      else:
         pfcPortConfig = profile.pfcPortConfig
      portTimerConfig = pfcPortConfig.portTimerConfig
      timeout = portTimerConfig.portWatchdogTimeout
      recoveryCfg = portTimerConfig.portWatchdogRecoveryCfg
      recoveryTime = recoveryCfg.recoveryTime
      forcedRecovery = recoveryCfg.forcedRecovery
      pollingInterval = portTimerConfig.portWatchdogPollingInterval

   if( timeout and pollingInterval ):
      if( recoveryTime and not forcedRecovery and
          ( recoveryTime < 2 * pollingInterval ) and ( recoveryTime < timeout ) ):
         return ( "recovery-time", recoveryCfg, pollingInterval )
      elif( timeout < 2 * pollingInterval ):
         return ( "timeout", timeout, pollingInterval )

   return None

# Compare the warning details, both before and after a config command,
# and give warning only if there is a change in config values and
def verifyPfcWatchdogConfig( mode, prevConflictDetails, perPort=False, intf=None,
                             profile=None ):
   if not perPort:
      timeout = qosInputConfig.watchdogTimeout
      recoveryTime = qosInputConfig.watchdogRecoveryCfg.recoveryTime
      pollingInterval = qosInputConfig.watchdogPollingInterval
   else :
      if intf:
         pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      else :
         pfcPortConfig = profile.pfcPortConfig
      portTimerConfig = pfcPortConfig.portTimerConfig
      timeout = portTimerConfig.portWatchdogTimeout
      recoveryTime = portTimerConfig.portWatchdogRecoveryCfg.recoveryTime
      pollingInterval = portTimerConfig.portWatchdogPollingInterval

   currConflictDetails = getPfcWatchdogCfgConflictDetails( perPort=perPort,
                                                         intf=intf, profile=profile )
   if( currConflictDetails and ( prevConflictDetails != currConflictDetails ) ):
      conflictCfgType = currConflictDetails[ 0 ]
      addPfcWatchdogWarning( mode, conflictCfgType,
                             timeout, recoveryTime, pollingInterval )

def configWatchdogTimeout( mode, args ):
   timeout = args.get( 'TIMEOUT' )
   prevConflictDetails = getPfcWatchdogCfgConflictDetails()
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.watchdogTimeout = 0
   elif timeout:
      # verify whether hardware supports all the pfc priorities
      # which is a necessary condition for watchdog to work
      if not hardwareSupports( mode, 0xff ):
         return
      # round the value to a float with 2 digits after the decimal point
      qosInputConfig.watchdogTimeout = round( timeout, 2 )
   verifyPfcWatchdogConfig( mode, prevConflictDetails )

def configWatchdogRecovery( mode, args ):
   recoveryTime = args.get( 'RECOVERYTIME' )
   prevConflictDetails = getPfcWatchdogCfgConflictDetails()
   recoveryConfig = Tac.Value( "Pfc::WatchdogRecoveryConfig" )
   if CliCommand.isNoOrDefaultCmd( args ):
      recoveryConfig.recoveryTime = 0
      recoveryConfig.forcedRecovery = False
      qosInputConfig.watchdogRecoveryCfg = recoveryConfig
   elif recoveryTime:
      if not hardwareSupports( mode, 0xff ):
         return
      # round the value to a float with 2 digits after the decimal point
      recoveryConfig.recoveryTime = round( recoveryTime, 2 )
      recoveryConfig.forcedRecovery = 'forced' in args
      qosInputConfig.watchdogRecoveryCfg = recoveryConfig
   verifyPfcWatchdogConfig( mode, prevConflictDetails )

def configWatchdogAction( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )

   if noOrDefaultKw or 'errdisable' in args:
      qosInputConfig.watchdogAction = tacPfcWatchdogAction.errdisable
   elif 'drop' in args:
      qosInputConfig.watchdogAction = tacPfcWatchdogAction.drop
   elif 'notify-only' in args:
      qosInputConfig.watchdogAction = tacPfcWatchdogAction.notifyOnly

def configWatchdogPollingInterval( mode, args ):
   interval = args.get( 'INTERVAL' )
   prevConflictDetails = getPfcWatchdogCfgConflictDetails()
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.watchdogPollingInterval = 0
   elif interval:
      if not hardwareSupports( mode, 0xff ):
         return
      # round the value to a float with 3 digits after the decimal point
      qosInputConfig.watchdogPollingInterval = round( interval, 3 )
   verifyPfcWatchdogConfig( mode, prevConflictDetails )

def configWatchdogNonDisruptivePriority( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if noOrDefaultKw:
      qosInputConfig.watchdogNonDisruptivePriorities = 0
      return
   priorityList = args.get( 'PRIORITY' )
   if ( pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported and 
      len( priorityList ) > pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported ):
      mode.addError( 'Maximum number of priorities supported is ' +
            str( pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported ) )
      return
   nonDisruptivePriorities = 0
   for priority in priorityList:
      priorityBit = 1 << priority
      if not hardwareSupports( mode, priorityBit ):
         return
      nonDisruptivePriorities |= priorityBit
   if not nonDisruptivePriorities:
      return
   qosInputConfig.watchdogNonDisruptivePriorities = nonDisruptivePriorities

def enableWatchdogNonDisruptiveForced( mode, args ):
   enabled = not CliCommand.isNoOrDefaultCmd( args )
   qosInputConfig.forcedNonDisruptiveBehaviour = enabled 

def enableWatchdogOverrideAction( mode, args ):
   enabled = not CliCommand.isNoOrDefaultCmd( args )
   qosInputConfig.overrideAction = enabled

def configWatchdogNonDisruptiveAction( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.watchdogNonDisruptiveAction = tacPfcWatchdogAction.invalid
   else:
      qosInputConfig.watchdogNonDisruptiveAction = tacPfcWatchdogAction.ignorePfc

def configWatchdogPerPortTimers( mode,args ):
   timeout = args[ 'TIMEOUT' ]
   intf = mode.intf.name
   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]

   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, False, )
   pfcPortConfig = intfConfig.pfcPortConfig

   if not hardwareSupports( mode, 0xff ):
      return

   portTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )

   portTimerConfig.portWatchdogTimeout = round( timeout, 2 )

   interval = args.get( 'INTERVAL', 0 )
   portTimerConfig.portWatchdogPollingInterval = round( interval, 3 )

   recoveryCfg = Tac.Value( "Pfc::WatchdogRecoveryConfig" )
   recoveryTime = args.get( 'RECOVERYTIME', 0 )
   recoveryCfg.recoveryTime = round( recoveryTime, 2 )

   recoveryCfg.forcedRecovery = 'forced' in args
   portTimerConfig.portWatchdogRecoveryCfg = recoveryCfg

   portTimerConfig.usePerPortTimerValues = True

   pfcPortConfig.portTimerConfig = portTimerConfig
   verifyPfcWatchdogConfig( mode, None, perPort=True, intf=intf )

def noConfigWatchdogPerPortTimers( mode, args ):
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig:
      pfcPortConfig = qosInputConfig.intfConfig[ intf ].pfcPortConfig
      if pfcPortConfig:
         portTimerConfig = pfcPortConfig.portTimerConfig
         if portTimerConfig:
            defPortTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
            pfcPortConfig.portTimerConfig = defPortTimerConfig
      maybeDeletePortConfig( intf )

def pfcWatchdogForcedNDGuard( mode, token ):
   if pfcStatus.watchdogSupported and \
      pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported and \
      pfcStatus.watchdogForcedNDSupported:
      return None
   return CliParser.guardNotThisPlatform

def pfcWatchdogNonDisruptivePriorityGuard( mode, token ):
   if pfcStatus.watchdogSupported and \
         pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      return None
   return CliParser.guardNotThisPlatform

def maxPfcWdNonDisruptPriorities():
   if pfcStatus.watchdogSupported:
      return pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported
   return 0

def pfcSelectivePrioritiesGuard( mode, token ):
   if pfcStatus.pfcRxSelectivePriorities:
      return None
   return CliParser.guardNotThisPlatform

def pfcWdPollingIntervalRangeFn( mode ):
   # This code can be called when the startup-config is being parsed
   # and before Sysdb is populated. The range is made the minimum and
   # maximum possible in this case.
   if pfcStatus is None:
      return( tacPfcStatus.minPollingInterval, tacPfcStatus.maxPollingInterval )
   else:
      return( round( pfcStatus.minPollingInterval, 3 ),
              round( pfcStatus.maxPollingInterval, 3 ) )

def pfcWdTimeoutRangeFn( mode ):
   # This code can be called when the startup-config is being parsed
   # and before Sysdb is populated. The range is made the minimum and
   # maximum possible in this case.
   if pfcStatus is None:
      return( tacPfcStatus.minTimeout, tacPfcStatus.maxTimeout )
   else:
      return( round( pfcStatus.minTimeout, 3 ),
              round( pfcStatus.maxTimeout, 3 ) )

def pfcWdRecoverTimeRangeFn( mode ):
   # This code can be called when the startup-config is being parsed
   # and before Sysdb is populated. The range is made the minimum and
   # maximum possible in this case.
   if pfcStatus is None:
      return( tacPfcStatus.minRecoveryTime, tacPfcStatus.maxRecoveryTime )
   else:
      return( round( pfcStatus.minRecoveryTime, 3 ),
              round( pfcStatus.maxRecoveryTime, 3 ) )

def configPfcPriorityList( mode, args ):
   priorityList = args.get( 'PRIORITIES' )
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.rxPfcPrioritiesHonored = 0xFF
      return
   rxPfcPrioritiesHonored = 0
   for priority in priorityList:
      priorityBit = 1 << priority
      rxPfcPrioritiesHonored |= priorityBit
   if rxPfcPrioritiesHonored:
      qosInputConfig.rxPfcPrioritiesHonored = rxPfcPrioritiesHonored

def hwMonitorGuard( mode, token ):
   if mode.name in ( 'Configure', 'config' ):
      if not pfcStatus.hardwareMonitorSupported:
         return CliParser.guardNotThisPlatform
   return None

def guardDlb( mode, token ):
   if qosHwStatus.dlbSupported:
      return None
   return CliParser.guardNotThisPlatform

matcherPfc = CliMatcher.KeywordMatcher( 'priority-flow-control', 
      helpdesc='Configure PFC' )
matcherMode = CliMatcher.KeywordMatcher( 'mode', 
      helpdesc='Configure PFC mode' )
nodePause = CliCommand.guardedKeyword( 'pause', "Pause configuration",
      guard=pfcWatchdogGuard )
nodePortConfigPause = CliCommand.guardedKeyword( 'pause', "Pause configuration",
      guard=pfcWatchdogPortConfigGuard )
matcherWatchdog = CliMatcher.KeywordMatcher( 'watchdog', 
      helpdesc='Enable watchdog on transmit queues' )
matcherPriority = CliMatcher.IntegerMatcher( 0, 7, helpdesc='Priority number' )
matcherDefault = CliMatcher.KeywordMatcher( 'default', 
      helpdesc='Change default values' )
matcherTimeout = CliMatcher.KeywordMatcher( 'timeout', 
      helpdesc= 'Configure timeout after which port should be errdisabled '
      'or should start dropping on congested priorities' )
matcherTimeoutValue = CliMatcher.DynamicFloatMatcher( 
      rangeFn=pfcWdTimeoutRangeFn, helpdesc='Timeout value in seconds', 
      precisionString='%.5g' )
matcherRecoveryTime = CliMatcher.KeywordMatcher( 'recovery-time', 
      helpdesc='Configure recovery-time after which stuck queue should '
      'recover and start forwarding' )
matcherRecoveryTimeValue = CliMatcher.DynamicFloatMatcher( 
      rangeFn=pfcWdRecoverTimeRangeFn, helpdesc='Recovery time in seconds', 
      precisionString='%.5g' )
matcherForced = CliMatcher.KeywordMatcher( 'forced', 
      helpdesc='Force recover any stuck queue(s) after the duration, '
      'irrespective of whether PFC frames are being received or not' )
matcherZeroValue = CliMatcher.KeywordMatcher( '0', 
      helpdesc='No recovery' ) 
matcherPollingInterval = CliMatcher.KeywordMatcher( 'polling-interval', 
      helpdesc='Configure the interval at which the watchdog should poll '
      'the queues' )
matcherPollingIntervalValue = CliMatcher.DynamicFloatMatcher( 
      rangeFn=pfcWdPollingIntervalRangeFn, helpdesc='Polling interval in seconds',
      precisionString='%.5g' )
nodeErrdisable = CliCommand.guardedKeyword( 'errdisable',
      "Watchodg action for stuck transmit queues",
      guard=pfcWatchdogErrdisableGuard )
nodeNotifyOnly = CliCommand.guardedKeyword( 'notify-only',
      "No action on the stuck queue",
      guard=pfcWatchdogNotifyOnlyGuard )
nodeDrop = CliCommand.guardedKeyword( 'drop', "Drop traffic on the stuck queue",
      guard=pfcWatchdogDropGuard )
nodeNonDisruptive = CliCommand.guardedKeyword( 'non-disruptive',
      "Watchdog non-disruptive configuration", 
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeWatchdogPriorityList = CliCommand.guardedKeyword( 'priority',
      "Configure watchdog non-disruptive priorities",
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeHardware = CliCommand.guardedKeyword( 'hardware', 
      "Configure PFC priority through hardware",
      guard=hwMonitorGuard )
nodeMonitor = CliCommand.guardedKeyword( 'monitor',
      "Configure PFC priority for hardware monitoring",
      guard=hwMonitorGuard )
nodeWatchdogHardware = CliCommand.guardedKeyword( 'hardware',
      "Configure PFC priority through hardware",
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeWatchdogNDPort = CliCommand.guardedKeyword( 'port', 
      "Select behaviour for all ports", guard=pfcWatchdogForcedNDGuard )
nodeWatchogNDOnly = CliCommand.guardedKeyword( 'non-disruptive-only',
      "Set strict non-disruptive behaviour", guard=pfcWatchdogForcedNDGuard )
nodePfcPriorityList = CliCommand.guardedKeyword( 'priority',
      "Configure PFC priorities to be honored", guard=pfcSelectivePrioritiesGuard )
nodeWatchdogNDAction = CliCommand.guardedKeyword( 'action', 
      "Configure watchdog action for stuck non-disruptive transmit queues",
      guard=pfcWatchdogNonDisruptivePriorityGuard )
nodeDlb = CliCommand.guardedKeyword( 'dlb', "Configure dynamic load balancing",
      guard=guardDlb )

#--------------------------------------------------------------------------------
# priority-flow-control on
# legacy:
#     priority-flow-control mode on
#--------------------------------------------------------------------------------
class PriorityFlowControlOnCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control [ modeDeprecated ] on'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'modeDeprecated': CliCommand.Node( matcher=matcherMode,
         deprecatedByCmd='priority-flow-control on' ),
      'on': 'Enable PFC',
   }

   handler = pfcOn
   noOrDefaultHandler = pfcOff

pfcModelet.addCommandClass( PriorityFlowControlOnCmd )

#--------------------------------------------------------------------------------
# ( no | default ) priority-flow-control
#--------------------------------------------------------------------------------
class PriorityFlowControlCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'priority-flow-control'
   data = {
      'priority-flow-control': matcherPfc,
   }
   
   noOrDefaultHandler = noPfc

pfcModelet.addCommandClass( PriorityFlowControlCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog
#--------------------------------------------------------------------------------
class PriorityFlowControlPauseWatchdogCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
   }
   
   handler = pfcWatchdogOnOff
   noOrDefaultHandler = handler

pfcModelet.addCommandClass( PriorityFlowControlPauseWatchdogCmd )

def pfcGlobalEnabled( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   qosInputConfig.pfcGlobalEnabled = True

def pfcGlobalDisabled( mode, args ):
   if not hardwareSupports( mode, 0xff ):
      return
   qosInputConfig.pfcGlobalEnabled = False

#--------------------------------------------------------------------------------
# # [no|default] priority-flow-control all off
# legacy:
#     [no|default] priority-flow-control all mode off
#--------------------------------------------------------------------------------
class PriorityFlowControlAllOffCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control all [ modeDeprecated ] off'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'all': 'Configure PFC for all interfaces',
      'modeDeprecated': CliCommand.Node(
         matcher=matcherMode,
         deprecatedByCmd='priority-flow-control all off' ),
      'off': 'Disable PFC',
   }
   
   handler = pfcGlobalDisabled
   noOrDefaultHandler = pfcGlobalEnabled

BasicCliModes.GlobalConfigMode.addCommandClass( PriorityFlowControlAllOffCmd )

# priority-flow-control priority <0-7> <no-drop|drop>
# no priority-flow-control priority <0-7>

def pfcPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   intf = mode.intf.name
   if intf not in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig.newMember( intf )
   else:
      intfConfig = qosInputConfig.intfConfig[ intf ]

   if not intfConfig.pfcPortConfig:
      intfConfig.pfcPortConfig = ( intf, False, )

   pfcPortConfig = intfConfig.pfcPortConfig
   priorities = pfcPortConfig.priorities

   bit = 1 << priority
   if 'drop' in args:
      priorities &= ~bit
   else:
      if not hardwareSupports( mode, bit ):
         return
      priorities |= bit
   intfConfig.pfcPortConfig.priorities = priorities
   maybeDeletePortConfig( intf )

def noPfcPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   intf = mode.intf.name
   if intf in qosInputConfig.intfConfig and \
          qosInputConfig.intfConfig[ intf ].pfcPortConfig:
      bit = 1 << priority
      qosInputConfig.intfConfig[ intf ].pfcPortConfig.priorities &= ~bit
      maybeDeletePortConfig( intf )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control priority PRIORITY ( drop | no-drop )
#--------------------------------------------------------------------------------
class PriorityFlowControlPriorityDropCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control priority PRIORITY ( drop | no-drop )'
   noOrDefaultSyntax = 'priority-flow-control priority PRIORITY ...'
   data = {
      'priority-flow-control': matcherPfc,
      'priority': 'Configure PFC priority',
      'PRIORITY': matcherPriority,
      'drop': 'Disable PFC for a priority',
      'no-drop': 'Enable PFC for a priority',
   }
   
   handler = pfcPriority
   noOrDefaultHandler = noPfcPriority

pfcModelet.addCommandClass( PriorityFlowControlPriorityDropCmd )

#--------------------------------------------------------------------------------
# priority-flow-control pause watchdog port timer timeout TIMEOUT polling-interval
# ( auto | INTERVAL ) recovery-time ( 0 | RECOVERYTIME ) [ forced ]
# [no | default ] priority-flow-control pause watchdog port timer
#--------------------------------------------------------------------------------
class PriorityFlowControlWatchdogPollingCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port timer timeout TIMEOUT ' 
              'polling-interval ( auto | INTERVAL ) recovery-time '
              '( 0 | RECOVERYTIME ) [ forced ]' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port timer ...'
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'timer': 'Per port timer values',
      'timeout': matcherTimeout,
      'TIMEOUT': matcherTimeoutValue, 
      'polling-interval': matcherPollingInterval,
      'auto': 'Auto polling interval value based on recovery-time and timeout',
      'INTERVAL': matcherPollingIntervalValue,
      'recovery-time': matcherRecoveryTime,
      '0': matcherZeroValue, 
      'RECOVERYTIME': matcherRecoveryTimeValue,
      'forced': matcherForced,
   }

   handler = configWatchdogPerPortTimers
   noOrDefaultHandler = noConfigWatchdogPerPortTimers

pfcModelet.addCommandClass( PriorityFlowControlWatchdogPollingCmd )

# priority-flow-control pause watchdog hardware monitor priority <priority>

def hwMonitor( mode, args ):
   priorityList = args.get( 'PRIORITIES' )
   if len( qosInputConfig.hwMonitoredPri ) >= 2:
      return
   for priority in priorityList:
      qosInputConfig.hwMonitoredPri[ priority ] = True

def noHwMonitor( mode, args ):
   priorities = args.get( 'PRIORITIES', qosInputConfig.hwMonitoredPri )
   for priority in priorities:
      del qosInputConfig.hwMonitoredPri[ priority ]

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware monitor
# priority { PRIORITIES }
#--------------------------------------------------------------------------------
class PfCPauseWatchdogHwMonitorPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware monitor '
              'priority { PRIORITIES }' )
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'hardware': nodeHardware,
      'monitor': nodeMonitor,
      'priority': 'Configure PFC priority',
      'PRIORITIES': CliCommand.Node( matcher=matcherPriority, maxMatches=2 ),
   }
   
   handler = hwMonitor
   noOrDefaultHandler = noHwMonitor

BasicCliModes.GlobalConfigMode.addCommandClass( 
      PfCPauseWatchdogHwMonitorPriorityCmd )

#--------------------------------------------------------------------------------
# priority-flow-control pause watchdog default timeout TIMEOUT
# ( no | default ) priority-flow-control pause watchdog default timeout ...
#--------------------------------------------------------------------------------
class PfCPauseWatchdogDefaultTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog default timeout TIMEOUT'
   noOrDefaultSyntax = 'priority-flow-control pause watchdog default timeout ...'
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'default': matcherDefault,
      'timeout': matcherTimeout,
      'TIMEOUT': matcherTimeoutValue,
   }
   
   handler = configWatchdogTimeout
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PfCPauseWatchdogDefaultTimeoutCmd )

#--------------------------------------------------------------------------------
# priority-flow-control pause watchdog default recovery-time 
# RECOVERYTIME [ forced ]
# ( no | default ) priority-flow-control pause watchdog default recovery-time ...
#--------------------------------------------------------------------------------
class PfCPauseWatchdogDefaultRecoveryTimeCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog default recovery-time '
              'RECOVERYTIME [ forced ]' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog default '
                         'recovery-time ...' )
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'default': matcherDefault,
      'recovery-time': matcherRecoveryTime,
      'RECOVERYTIME': matcherRecoveryTimeValue,
      'forced': matcherForced,
   }
   
   handler = configWatchdogRecovery
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( 
      PfCPauseWatchdogDefaultRecoveryTimeCmd )

#--------------------------------------------------------------------------------
# priority-flow-control pause watchdog default polling-interval INTERVAL
# ( no | default ) priority-flow-control pause watchdog 
# default polling-interval ...'
#--------------------------------------------------------------------------------
class PfCPauseWatchdogDefaultPollingIntervalCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog default '
              'polling-interval INTERVAL' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog '
                         'default polling-interval ...' )
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'default': matcherDefault,
      'polling-interval': matcherPollingInterval,
      'INTERVAL': matcherPollingIntervalValue,
   }
   
   handler = configWatchdogPollingInterval
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( 
      PfCPauseWatchdogDefaultPollingIntervalCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog action 
# ( errdisable | drop | notify-only )
#--------------------------------------------------------------------------------
class PfCPauseWatchdogActionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog action '
              ' ( errdisable | drop | notify-only ) ' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog action ...'
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause, 
      'watchdog': matcherWatchdog,
      'action': 'Watchdog action for stuck transmit queues',
      'errdisable': nodeErrdisable,
      'notify-only': nodeNotifyOnly,
      'drop': nodeDrop,
   }

   handler = configWatchdogAction
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PfCPauseWatchdogActionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog port action 
# ( errdisable | drop | notify-only )
#--------------------------------------------------------------------------------
class PfCPauseWatchdogPortActionCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port action '
              '( errdisable | drop | notify-only )' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port action ...'
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'action': 'Watchdog action for stuck transmit queues',
      'errdisable': 'Error disable port which has the stuck transmit queue',
      'drop': nodeDrop,
      'notify-only': 'No action on the stuck queue',
   }

   handler = configWatchdogPerPortAction
   noOrDefaultHandler = disableConfigWatchdogPerPortAction

pfcModelet.addCommandClass( PfCPauseWatchdogPortActionCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware 
# non-disruptive priority { PRIORITY }
#--------------------------------------------------------------------------------
class PfcPauseWatchdogHwNDPriorityCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware '
              'non-disruptive priority { PRIORITY }' )
   noOrDefaultSyntax = ( 'priority-flow-control pause watchdog hardware '
                         'non-disruptive priority ...' )
   data = {
      'priority-flow-control': matcherPfc,
      'priority': nodeWatchdogPriorityList,
      'pause': nodePortConfigPause,
      'watchdog': matcherWatchdog,
      'hardware': nodeWatchdogHardware,
      'non-disruptive': nodeNonDisruptive,
      'PRIORITY': matcherPriority,
   }

   handler = configWatchdogNonDisruptivePriority
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PfcPauseWatchdogHwNDPriorityCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware 
# non-disruptive port non-disruptive-only
#--------------------------------------------------------------------------------
class PfCPauseWatchdogHwNonDisruptivePortOnlyCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware '
              'non-disruptive port non-disruptive-only' )
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'hardware': nodeWatchdogHardware,
      'non-disruptive': nodeNonDisruptive,
      'port': nodeWatchdogNDPort,
      'non-disruptive-only': nodeWatchogNDOnly,
   }

   handler = enableWatchdogNonDisruptiveForced
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( 
      PfCPauseWatchdogHwNonDisruptivePortOnlyCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control priority default { PRIORITIES }
#--------------------------------------------------------------------------------
class PfCPriorityDefaultCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control priority default { PRIORITIES }'
   noOrDefaultSyntax = 'priority-flow-control priority default ...'
   data = {
      'priority-flow-control': matcherPfc,
      'priority': nodePfcPriorityList,
      'default': matcherDefault,
      'PRIORITIES': CliCommand.Node( matcher=matcherPriority, maxMatches=8 ),
   }

   handler = configPfcPriorityList
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PfCPriorityDefaultCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog hardware 
# non-disruptive action forward
#--------------------------------------------------------------------------------
class PfCPauseWdHwNonDisruptiveActionForwardCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog hardware non-disruptive '
              'action forward' )
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'hardware': nodeWatchdogHardware,
      'non-disruptive': nodeNonDisruptive,
      'action': nodeWatchdogNDAction,
      'forward': 'Forward traffic on the stuck queue',
   }
   
   handler = configWatchdogNonDisruptiveAction
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( 
      PfCPauseWdHwNonDisruptiveActionForwardCmd )

#------------------------------------------------------------------------
# QosProfile Mode
#------------------------------------------------------------------------

# PFC cli commands in qos profile mode
def qosProfilePfcOn( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig.enabled = True
   else:
      profile.pfcPortConfig = ( "QosProfile", True, )

def qosProfilePfcOff( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig.enabled = False
      setQosProfileConfig( profile )

def qosProfileNoPfc( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig = None

def qosProfilePfcWatchdogOnOff( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if CliCommand.isNoCmd( args ):
      if profile.pfcPortConfig:
         profile.pfcPortConfig.watchdogEnabled = False
      else:
         profile.pfcPortConfig = ( "QosProfile", False, )
         profile.pfcPortConfig.watchdogEnabled = False
   else:
      if profile.pfcPortConfig:
         profile.pfcPortConfig.watchdogEnabled = True
         setQosProfileConfig( profile )

#------------------------------------------------------------------------
# PFC commands in QosProfile mode
#------------------------------------------------------------------------
# priority-flow-control on
# legacy:
#     priority-flow-control mode on

class PfCOnProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control [ modeDeprecated ] on'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'modeDeprecated': CliCommand.Node(
         matcher=matcherMode,
         deprecatedByCmd='priority-flow-control on' ),
      'on': 'Enable PFC',
   }

   handler = qosProfilePfcOn
   noOrDefaultHandler = qosProfilePfcOff

QosProfileMode.addCommandClass( PfCOnProfileCmd )

#--------------------------------------------------------------------------------
# ( no | default ) priority-flow-control in QosProfile Mode
#--------------------------------------------------------------------------------
class PfCProfileCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'priority-flow-control'
   data = {
      'priority-flow-control': matcherPfc,
      }

   noOrDefaultHandler = qosProfileNoPfc

QosProfileMode.addCommandClass( PfCProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog in QosProfile Mode
#--------------------------------------------------------------------------------
class PfCPauseWatchdogProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog'
   noOrDefaultSyntax = syntax
   data = {
         'priority-flow-control': matcherPfc,
         'pause': nodePause,
         'watchdog': matcherWatchdog,
      }

   handler = qosProfilePfcWatchdogOnOff
   noOrDefaultHandler = handler

QosProfileMode.addCommandClass( PfCPauseWatchdogProfileCmd )

def qosProfilePfcPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   profile = mode.qosProfileModeContext.currentEntry_
   if not profile.pfcPortConfig:
      profile.pfcPortConfig = ( "QosProfile", False, )
   bit = 1 << priority
   if 'drop' in args:
      profile.pfcPortConfig.priorities &= ~bit
      profile.fabricPfcDlb &= ~bit
      setQosProfileConfig( profile )
   else:
      if 'dlb' in args:
         profile.fabricPfcDlb |= bit
      elif bit & profile.fabricPfcDlb:
         profile.fabricPfcDlb &= ~bit
      profile.pfcPortConfig.priorities |= bit

def qosProfileNoPfcPriority( mode, args ):
   priority = args[ 'PRIORITY' ]
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      bit = 1 << priority
      profile.pfcPortConfig.priorities &= ~bit
      if bit & profile.fabricPfcDlb:
         profile.fabricPfcDlb &= ~bit
      setQosProfileConfig( profile )

def qosProfilePortWatchdogTimer( mode, args ):
   timeout = args[ 'TIMEOUT' ]
   profile = mode.qosProfileModeContext.currentEntry_
   if not profile.pfcPortConfig:
      profile.pfcPortConfig = ( "QosProfile", False, )

   portTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
   portTimerConfig.portWatchdogTimeout = round( timeout, 2 )
   interval = args.get( 'INTERVAL', 0 )
   portTimerConfig.portWatchdogPollingInterval = round( interval, 3 )
   
   recoveryCfg = Tac.Value( "Pfc::WatchdogRecoveryConfig" )
   recoveryTimeValue = args.get( 'RECOVERYTIME', 0 )
   recoveryCfg.recoveryTime = round( recoveryTimeValue, 2 )

   recoveryCfg.forcedRecovery = 'forced' in args
   portTimerConfig.portWatchdogRecoveryCfg = recoveryCfg
   portTimerConfig.usePerPortTimerValues = True

   profile.pfcPortConfig.portTimerConfig = portTimerConfig
   verifyPfcWatchdogConfig( mode, None, perPort=True, profile=profile )

def qosProfileNoPortWatchdogTimer( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      portTimerConfig = profile.pfcPortConfig.portTimerConfig
      if portTimerConfig:
         defPortTimerConfig = Tac.Value( "Pfc::PortTimerConfig" )
         profile.pfcPortConfig.portTimerConfig = defPortTimerConfig
      setQosProfileConfig( profile )

def qosProfilePortWatchdogAction( mode, args ):

   profile = mode.qosProfileModeContext.currentEntry_
   if not profile.pfcPortConfig:
      profile.pfcPortConfig = ( "QosProfile", False, )

   if 'errdisable' in args:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.errdisable
   elif 'drop' in args:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.drop
   elif 'notify-only' in args:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.notifyOnly

def qosProfileNoPortWatchdogAction( mode, args ):
   profile = mode.qosProfileModeContext.currentEntry_
   if profile.pfcPortConfig:
      profile.pfcPortConfig.watchdogPortAction = tacPfcWatchdogAction.invalid
      setQosProfileConfig( profile )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control priority PRIORITY 
# ( drop | no-drop [ dlb ] ) in QosProfile Mode
#--------------------------------------------------------------------------------
class PfCPriorityDropProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control priority PRIORITY ( drop | no-drop [ dlb ] )'
   noOrDefaultSyntax = 'priority-flow-control priority PRIORITY ...'
   data = {
      'priority-flow-control': matcherPfc,
      'priority': 'Configure PFC priority',
      'PRIORITY': matcherPriority,
      'drop': 'Disable PFC for a priority',
      'no-drop': 'Enable PFC for a priority',
      'dlb': nodeDlb,
   }

   handler = qosProfilePfcPriority
   noOrDefaultHandler = qosProfileNoPfcPriority

QosProfileMode.addCommandClass( PfCPriorityDropProfileCmd )

#--------------------------------------------------------------------------------
# priority-flow-control pause watchdog port timer timeout TIMEOUT polling-interval
# ( auto | INTERVAL ) recovery-time ( 0 | RECOVERYTIME ) [ forced ]
# [no | default ] priority-flow-control pause watchdog port timer - QosProfile Mode
#--------------------------------------------------------------------------------
class PfCWatchdogPollingProfileCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port timer timeout TIMEOUT '
              'polling-interval ( auto | INTERVAL ) recovery-time '
              '( 0 | RECOVERYTIME ) [ forced ]' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port timer ...'
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'timer': 'Per port timer values',
      'timeout': matcherTimeout,
      'TIMEOUT': matcherTimeoutValue,
      'polling-interval': matcherPollingInterval,
      'auto': 'Auto polling interval value based on recovery-time and timeout',
      'INTERVAL': matcherPollingIntervalValue,
      'recovery-time': matcherRecoveryTime,
      '0': matcherZeroValue,
      'RECOVERYTIME': matcherRecoveryTimeValue,
      'forced': matcherForced,
   }

   handler = qosProfilePortWatchdogTimer
   noOrDefaultHandler = qosProfileNoPortWatchdogTimer

QosProfileMode.addCommandClass( PfCWatchdogPollingProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog port action
# ( errdisable | drop | notify-only ) in QosProfile Mode
#--------------------------------------------------------------------------------
class PfCPauseWdPortActionProfileCmd( CliCommand.CliCommandClass ):
   syntax = ( 'priority-flow-control pause watchdog port action '
              '( errdisable | drop | notify-only )' )
   noOrDefaultSyntax = 'priority-flow-control pause watchdog port action ...'
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'port': 'Per port watchdog configuration',
      'action': 'Watchdog action for stuck transmit queues',
      'errdisable': 'Error disable port which has the stuck transmit queue',
      'drop': nodeDrop,
      'notify-only': 'No action on the stuck queue',
   }

   handler = qosProfilePortWatchdogAction
   noOrDefaultHandler = qosProfileNoPortWatchdogAction

QosProfileMode.addCommandClass( PfCPauseWdPortActionProfileCmd )

#--------------------------------------------------------------------------------
# [ no | default ] priority-flow-control pause watchdog override action drop
# in QosProfile Mode
#--------------------------------------------------------------------------------
class PfCPauseWdOverrideActionDropCmd( CliCommand.CliCommandClass ):
   syntax = 'priority-flow-control pause watchdog override action drop'
   noOrDefaultSyntax = syntax
   data = {
      'priority-flow-control': matcherPfc,
      'pause': nodePause,
      'watchdog': matcherWatchdog,
      'override': 'Override configured watchdog actions',
      'action': 'Watchdog action for stuck transmit queues',
      'drop': nodeDrop,
   }

   handler = enableWatchdogOverrideAction
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( PfCPauseWdOverrideActionDropCmd )

#
# -----------------------------------------------------------------------
# SECTION: Priority Flow Control (PFC) Show Command Value Functions
# -----------------------------------------------------------------------
# This section contains definitions of the value functions used for the
# priority flow control show commands.
#
# The value functions make use of helper and model-get routines to
# instantiate and populate the appropriate CLI model objects.
#
# Helper routines:
# . prioritiesToList
# . priArrayToList
# . watchdogActionToEnum
# . getAttrRecursive
# . statistics
# . portPfcConfig
# . portDcbxNote
# . portHwActivationStatus
# . portWatchdogHwStatus
# . txQueueWatchdogSkippable
# . counterSupportedEthIntfs
# . counterSupportedEthPhyIntfs
#
# Sub-Model Get routines:
# . pfcStatusModelGet
# . pfcWatchdogTimerModelGet
# . pfcWatchdogIntfTimerModelGet
# . pfcWatchdogNDPrioritiesModelGet
# . pfcWatchdogNDActionModelGet
# . pfcWatchdogForcedNDModelGet
# . pfcWatchdogOverrideActionModelGet
# . pfcWatchdogStatusModelGet
# . pfcIntfStatusModelGet
# . pfcIntfCountersModelGet
# . pfcPriorityCountersModelGet
# . pfcIntfDetailedCountersModelGet
# . pfcWatchdogTxQueueCountersModelGet
# . pfcWatchdogIntfCountersModelGet
#
# Model Get routines:
# . pfcFabricIntfStatusModelGet
# . pfcIntfRangeStatusModelGet
# . pfcIntfRangeCountersModelGet
# . pfcIntfRangeDetailedCountersModelGet
# . pfcWatchdogIntfRangeCountersModelGet
# . pfcIntfRangeModelGet
#
# Value functions:
# . showPfcFabricStatus
# . showPfcInterfaceStatus
# . showPfcInterfaceCounters
# . showPfcInterfaceDetailedCounters
# . showPfcWatchdogInterfaceCounters
# . showPfcInterface
# -----------------------------------------------------------------------
#

#
# Converts an 8-bit unsigned integer encoding of priorities into a list.
#
# Priority p becomes a member of the list if bit p is set in the 8-bit
# unsigned integer.
#
def prioritiesToList( priorities ):
   return [ p for p in range( numPriority ) if ( ( 1 << p ) & priorities ) ]

#
# Converts a boolean array into a list containing array indices
# as members for which the value (array[i]) is True,
# otherwise returns empty list if priority array is empty
#
def priArrayToList( priorities ):
   return [ idx for idx in priorities if priorities[ idx ] ]

#
# Converts a watchdog action into an enum-string for use in a watchdog model.
#
def watchdogActionToEnum( action ):

   if action == tacPfcWatchdogAction.notifyOnly:
      return 'notify-only'

   if action == tacPfcWatchdogAction.errdisable:
      return 'errdisable'

   if action == tacPfcWatchdogAction.drop:
      return 'drop'

   if action == tacPfcWatchdogAction.ignorePfc:
      return 'forward'

   return None

#
# Recursively traverses an attribute hierarchy and obtains the value of the
# leaf attribute.
#
def getAttrRecursive( value, attrs, index=None, allowNone=False ):
   for attr in attrs:
      try:
         value = getattr( value, attr, None )
      except AttributeError:
         return None if allowNone else 0

   if value and index is not None:
      value = value[ index ]

   return 0 if ( not allowNone and value is None ) else value

#
# Obtains the value of an interface counter present in the ethStatistics
# object hierarchy.
#
def statistics( counter, checkpoint, attrs, index=None, allowNone=False ):
   attrs.insert( 0, 'ethStatistics' )

   counterValue = getAttrRecursive( counter, attrs, index, allowNone=allowNone )
   checkpointValue = getAttrRecursive( checkpoint, attrs, index )

   if counterValue is None:
      return None
   else:
      return counterValue - checkpointValue

#
# Obtains the PFC port configuration for an interface.
#
def portPfcConfig( intfName ):
   intfConfig = qosConfig.intfConfig.get( intfName )

   if not intfConfig:
      return None

   portConfig = intfConfig.pfcPortConfig

   return portConfig

#
# Conditionally prepends the DCBX configuration state to the informational
# note for an interface.
#
def portDcbxNote( intfName, pfcEnabled, note ):

   if not pfcEnabled:
      return note

   dcbxMode = dcbxConfig.portEnabled.get( intfName )

   if not ( ( dcbxMode is None ) or ( dcbxMode == 'modeDisabled' ) ):
      return note

   dcbxNote = "DCBX disabled"

   if note is None:
      return dcbxNote

   return dcbxNote + "; " + note

#
# Obtains the status of PFC activation on the interface hardware.
#
def portHwActivationStatus( intfName ):
   portStatus = None

   for sliceHwStatus in qosSliceHwStatus.itervalues():
      portStatus = sliceHwStatus.portStatus.get( intfName )

      if portStatus:
         break

   if portStatus:
      active = portStatus.active
      note = portStatus.note if portStatus.note else None
   else:
      active = False
      note = None

   return ( active, note )

#
# Obtains the status of PFC Watchdog on the interface hardware.
#
def portWatchdogHwStatus( intfName ):
   for sliceHwStatus in qosSliceHwStatus.itervalues():
      portWatchdogStatus = sliceHwStatus.portWatchdogStatus.get( intfName )

      if portWatchdogStatus:
         return portWatchdogStatus

   return None

#
# Obtains the status of PFC Watchdog drop counters on the interface hardware.
#
def portWatchdogDropCountersHwStatus( intfName ):
   for sliceHwStatus in qosSliceHwStatus.itervalues():
      portWatchdogStatus = sliceHwStatus.portWatchdogDropCountersStatus.get(
         intfName )
      if portWatchdogStatus:
         return portWatchdogStatus
   return None

#
# Skips multicast queues as they are not fully supported by the PFC Watchdog.
# Refer bug/133643 for details.
#
def txQueueWatchdogSkippable( txQueue ):
   return True if txQueue.type == 'mcqueue' else False


#
# Creates and populates an instance of PfcStatusModel.
#
def pfcStatusModelGet():
   """Return the model for PFC global status."""

   model = PfcStatusModel()

   model.enabled = qosConfig.pfcGlobalEnabled

   if not pfcStatus.prioritiesSupported:
      model.supported = False
      model.priorities = None
      return model

   model.supported = True
   model.priorities = prioritiesToList( pfcStatus.prioritiesSupported )
   
   model.rxPfcPrioritiesHonored = prioritiesToList(
         qosConfig.rxPfcPrioritiesHonored )
   
   return model

#
# Creates and populates an instance of PfcWatchdogTimerModel.
#
def pfcWatchdogTimerModelGet():
   """Return the model for PFC Watchdog global timer configuration."""

   model = PfcWatchdogTimerModel()

   model.timeout = qosConfig.watchdogTimeout
   model.pollInterval = qosConfig.watchdogPollingInterval
   model.operPollInterval = pfcConfig.watchdogOperPollingInterval
   model.recoveryTime = qosConfig.watchdogRecoveryCfg.recoveryTime
   model.forcedRecovery = qosConfig.watchdogRecoveryCfg.forcedRecovery

   return model

#
# Creates and populates an instance of PfcWatchdogTimerModel for an interface.
#
def pfcWatchdogIntfTimerModelGet( intfName, timerConfig ):
   """Return the model for PFC Watchdog interface timer configuration."""

   #
   # Nothing to do if interface-level timer values are not in use.
   #
   if not timerConfig.usePerPortTimerValues:
      return None

   model = PfcWatchdogTimerModel()

   model.timeout = timerConfig.portWatchdogTimeout

   model.pollInterval = timerConfig.portWatchdogPollingInterval

   model.recoveryTime = timerConfig.portWatchdogRecoveryCfg.recoveryTime
   model.forcedRecovery = timerConfig.portWatchdogRecoveryCfg.forcedRecovery

   if intfName not in pfcConfig.portConfig:
      model.operPollInterval = None
      return model

   pfcPortConfig = pfcConfig.portConfig[ intfName ]

   model.operPollInterval = \
      pfcPortConfig.portTimerConfig.portWatchdogOperPollingInterval

   return model

#
# Creates and populates an instance of PfcWatchdogNDPrioritiesModel.
#
def pfcWatchdogNDPrioritiesModelGet():
   """Return the model for PFC Watchdog non-disruptive priorities global status."""

   model = PfcWatchdogNDPrioritiesModel()

   if not pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      model.supported = False
      model.priorities = None
      return model

   model.supported = True
   model.priorities = prioritiesToList(
      qosConfig.watchdogNonDisruptivePriorities )

   return model

#
# Creates and populates an instance of PfcWatchdogNDActionModel.
#
def pfcWatchdogNDActionModelGet():
   """Return the model for PFC Watchdog non-disruptive action global status."""

   nDAction = watchdogActionToEnum( qosConfig.watchdogNonDisruptiveAction )

   return nDAction if nDAction is not None else 'invalid'

#
# Creates and populates an instance of PfcWatchdogNDPrioritiesModel.
#
def pfcWatchdogForcedNDModelGet():
   """Return the model for PFC Watchdog non-disruptive priorities global status."""

   if not pfcStatus.watchdogMaxNonDisruptivePrioritiesSupported:
      return None

   return qosConfig.forcedNonDisruptiveBehaviour

#
# Creates and populates an instance of PfcWatchdogOverrideActionModel.
#
def pfcWatchdogOverrideActionModelGet():
   """Return the model for PFC Watchdog override action drop status."""

   return qosConfig.overrideAction

#
# Creates and populates an instance of PfcWatchdogStatusModel.
#
def pfcWatchdogStatusModelGet():
   """Return the model for PFC Watchdog global status."""

   #
   # Nothing to do if PFC itself is not supported.
   #
   if not pfcStatus.prioritiesSupported:
      return None

   model = PfcWatchdogStatusModel()

   if not pfcStatus.watchdogSupported:
      model.supported = False
      model.action = None
      model.timer = None
      model.nDPriorities = None
      model.nDAction = None
      model.forcedNd = None
      model.overrideAction = None
      model.hwMonitoredPri = None
      return model

   model.supported = True
   model.action = watchdogActionToEnum( qosConfig.watchdogAction )
   model.timer = pfcWatchdogTimerModelGet()
   model.nDPriorities = pfcWatchdogNDPrioritiesModelGet()
   model.nDAction = pfcWatchdogNDActionModelGet()
   model.forcedNd = pfcWatchdogForcedNDModelGet()
   model.overrideAction = pfcWatchdogOverrideActionModelGet()
   if not pfcStatus.hardwareMonitorSupported:
      model.hwMonitoredPri = None
   else:
      model.hwMonitoredPri = priArrayToList( qosConfig.hwMonitoredPri )
   return model

#
# Creates and populates an instance of PfcIntfStatusModel.
#
def pfcIntfStatusModelGet( intf ):
   """Return the model for PFC status on a given interface."""

   model = PfcIntfStatusModel()

   ( model.active, model.note ) = portHwActivationStatus( intf.name )

   #
   # Configuration specific to the interface.
   #
   portConfig = portPfcConfig( intf.name )

   if not portConfig:
      model.enabled = False
      model.priorities = []
      model.watchdogEnabled = False
      model.watchdogAction = None
      model.watchdogTimer = None
      return model

   model.enabled = portConfig.enabled
   model.priorities = prioritiesToList( portConfig.priorities )

   model.watchdogEnabled = \
      portConfig.watchdogEnabled if pfcStatus.watchdogSupported else False

   model.watchdogAction = watchdogActionToEnum( portConfig.watchdogPortAction )

   model.watchdogTimer = \
      pfcWatchdogIntfTimerModelGet( intf.name, portConfig.portTimerConfig )

   model.note = portDcbxNote( intf.name, model.enabled, model.note )

   return model

#
# Creates and populates an instance of PfcFrameCountersModel for an interface.
#
def pfcIntfCountersModelGet( intf ):
   """Return the model for PFC counters on a given interface."""

   model = PfcFrameCountersModel()

   counter = intf.counter()
   checkpoint = intf.getLatestCounterCheckpoint()

   model.rxFrames = statistics( counter, checkpoint, [ 'inPfcFrames' ] )
   model.txFrames = statistics( counter, checkpoint, [ 'outPfcFrames' ] )

   return model

#
# Creates and populates an instance of PfcFrameCountersModel for a priority.
#
def pfcPriorityCountersModelGet( priority, counter, checkpoint ):
   """Return the model for PFC interface counters for a given priority."""

   model = PfcFrameCountersModel()

   model.rxFrames = statistics( counter, checkpoint,
                                [ 'inPfcClassFrames', 'count' ],
                                priority,
                                allowNone=True )

   model.txFrames = statistics( counter, checkpoint,
                                [ 'outPfcClassFrames', 'count' ],
                                priority,
                                allowNone=True )

   return model

#
# Creates and populates an instance of PfcIntfDetailedCountersModel.
#
def pfcIntfDetailedCountersModelGet( intf ):
   """Return the model for PFC detailed counters on a given interface."""

   model = PfcIntfDetailedCountersModel()

   model.priorities = {}

   counter = intf.counter()
   checkpoint = intf.getLatestCounterCheckpoint()

   for p in range( numPriority ):
      model.priorities[ p ] = pfcPriorityCountersModelGet( p, counter, checkpoint )

   return model

#
# Creates and populates an instance of PfcWatchdogTxQueueCountersModel.
#
def pfcWatchdogTxQueueCountersModelGet( txQueue, queueStatus ):
   """Return the model for PFC Watchdog interface counters on a given
      transmit queue."""

   model = PfcWatchdogTxQueueCountersModel()

   model.queueType = 'unicast' if txQueue.type == 'ucqueue' else 'multicast'
   model.stuckCount = ( queueStatus.stuckInfo.count -
                      queueStatus.stuckInfoOnClear.count )
   model.recoveryCount = ( queueStatus.recoveryInfo.count -
                         queueStatus.recoveryInfoOnClear.count )

   return model

#
# Creates and populates an instance of PfcWatchdogEgressDropCountersModel.
#
def pfcWatchdogEgressDropCountersModelGet( txQueue, pfcWatchdogIntfDropCounters ):
   """Return the model for PFC Watchdog interface drop counters on a given
      transmit queue."""

   model = PfcWatchdogEgressDropCountersModel()

   egressDropCounters = pfcWatchdogIntfDropCounters.egressDropCounters
   if pfcStatus.watchdogDropCountersSupported:
      model.queueType = 'unicast' if txQueue.type == 'ucqueue' else 'multicast'
      model.outCurrentDrops = egressDropCounters.currentPktsDroppedSnapshot - \
                           egressDropCounters.pktsDroppedSnapshotAtStuck - \
                           egressDropCounters.currentPktsDroppedSnapshotOnClear
      model.outCurrentByteDrops = egressDropCounters.currentBytesDroppedSnapshot - \
                              egressDropCounters.bytesDroppedSnapshotAtStuck - \
                              egressDropCounters.currentBytesDroppedSnapshotOnClear
      model.outAggregateDrops = egressDropCounters.aggregatePktsDropped - \
                             egressDropCounters.aggregatePktsDroppedOnClear + \
                             model.outCurrentDrops
      model.outAggregateByteDrops = egressDropCounters.aggregateBytesDropped - \
                                 egressDropCounters.aggregateBytesDroppedOnClear + \
                                 model.outCurrentByteDrops
   return model

#
# Creates and populates an instance of PfcWatchdogIntfIngressDropCountersModel.
#
def pfcWatchdogIntfIngressDropCountersModelGet( tc, pfcWatchdogIntfDropCounters ):
   """Return the model for PFC Watchdog interface drop counters on a given
      transmit queue."""

   model = PfcWatchdogIngressDropCountersModel()

   ingressDropCounters = pfcWatchdogIntfDropCounters.ingressDropCounters
   if pfcStatus.watchdogDropCountersSupported:
      model.currentDrops = ingressDropCounters.currentPktsDroppedSnapshot - \
                          ingressDropCounters.pktsDroppedSnapshotAtStuck - \
                          ingressDropCounters.currentPktsDroppedSnapshotOnClear
      model.currentByteDrops = ingressDropCounters.currentBytesDroppedSnapshot - \
                               ingressDropCounters.bytesDroppedSnapshotAtStuck - \
                               ingressDropCounters.currentBytesDroppedSnapshotOnClear
      model.aggregateDrops = ingressDropCounters.aggregatePktsDropped - \
                             ingressDropCounters.aggregatePktsDroppedOnClear + \
                             model.currentDrops
      model.aggregateByteDrops = ingressDropCounters.aggregateBytesDropped - \
                                 ingressDropCounters.aggregateBytesDroppedOnClear + \
                                 model.currentByteDrops
   return model

#
# Creates and populates an instance of PfcWatchdogIntfCountersModel.
#
def pfcWatchdogIntfCountersModelGet( intf, portWatchdogStatus ):
   """Return the model for PFC Watchdog counters on a given interface."""

   model = PfcWatchdogIntfCountersModel()

   model.txQueues = {}

   for txQueue, queueStatus in portWatchdogStatus.watchdogQueueStatus.iteritems():
      if txQueueWatchdogSkippable( txQueue ):
         continue

      model.txQueues[ txQueue.id ] = \
         pfcWatchdogTxQueueCountersModelGet( txQueue, queueStatus )

   return model

#
# Creates and populates an instance of PfcWatchdogIntfEgressDropCountersModel.
#
def pfcWatchdogIntfEgressDropCountersModelGet( intf, portWatchdogStatus ):
   """Return the model for PFC Watchdog drop counters on a given interface."""

   model = PfcWatchdogIntfEgressDropCountersModel()

   model.egressCounters = {}

   for txQueue, queueStatus in portWatchdogStatus.watchdogEgressDropCounterStatus. \
                                                 iteritems():
      if txQueueWatchdogSkippable( txQueue ):
         continue

      model.egressCounters[ txQueue.id ] = \
         pfcWatchdogEgressDropCountersModelGet( txQueue, queueStatus )

   for tc, watchdogDropCounterStatus in \
       portWatchdogStatus.watchdogIngressDropCounterStatus.iteritems():
      model.ingressCounters[ tc ] = pfcWatchdogIntfIngressDropCountersModelGet(
            tc, watchdogDropCounterStatus )

   return model

#
# Creates and populates an instance of PfcFabricIntfStatusModel.
#
def pfcFabricIntfStatusModelGet():
   """Return the model for PFC status on the Fabric interface."""

   model = PfcFabricIntfStatusModel()

   model.intfName = fabricShortName

   model.status = pfcStatusModelGet()
   model.watchdogStatus = pfcWatchdogStatusModelGet()

   #
   # Configuration specific to the Fabric interface.
   #
   portConfig = portPfcConfig( fabricIntfName )

   if not portConfig:
      model.enabled = False
      model.priorities = []
      return model

   model.enabled = portConfig.enabled
   model.priorities = prioritiesToList( portConfig.priorities )

   return model

#
# Creates and populates an instance of PfcIntfRangeStatusModel.
#
def pfcIntfRangeStatusModelGet( intfs ):
   """Return the model for PFC status on a given range of interfaces."""

   model = PfcIntfRangeStatusModel()

   model.status = pfcStatusModelGet()
   model.watchdogStatus = pfcWatchdogStatusModelGet()

   model.interfaceStatuses = {}

   for intf in intfs:
      model.interfaceStatuses[ intf.status().intfId ] = \
                                                      pfcIntfStatusModelGet( intf )

   return model

#
# Creates and populates an instance of PfcIntfRangeCountersModel.
#
def pfcIntfRangeCountersModelGet( intfs ):
   """Return the model for PFC counters on a given range of interfaces."""

   model = PfcIntfRangeCountersModel()

   model.interfaceCounters = {}

   for intf in intfs:
      model.interfaceCounters[ intf.status().intfId ] = \
                                                     pfcIntfCountersModelGet( intf )

   return model

#
# Creates and populates an instance of PfcIntfRangeDetailedCountersModel.
#
def pfcIntfRangeDetailedCountersModelGet( intfs ):
   """Return the model for PFC detailed counters on a given range of interfaces."""
   __revision__ = 3
   model = PfcIntfRangeDetailedCountersModel()

   model.interfaces = {}

   for intf in intfs:
      model.interfaces[ intf.status().intfId ] = \
                                             pfcIntfDetailedCountersModelGet( intf )

   return model

#
# Creates and populates an instance of PfcWatchdogIntfRangeCountersModel.
#
def pfcWatchdogIntfRangeCountersModelGet( intfs ):
   """Return the model for PFC Watchdog counters on a given range of interfaces."""

   model = PfcWatchdogIntfRangeCountersModel()

   model.interfaces = {}
   model.cliOutputIs( bool( intfs ) )

   for intf in intfs:
      portWatchdogStatus = portWatchdogHwStatus( intf.name )

      if not portWatchdogStatus:
         continue

      model.interfaces[ intf.status().intfId ] = \
         pfcWatchdogIntfCountersModelGet( intf, portWatchdogStatus )

   return model

#
# Creates and populates an instance of PfcWatchdogIntfRangeDropCountersModel.
#
def pfcWatchdogIntfRangeDropCountersModelGet( intfs ):
   """Return the model for PFC Watchdog d counters on a given range of interfaces."""

   model = PfcWatchdogIntfRangeDropCountersModel()
   model.interfaces = {}
   model.cliOutputIs( bool( intfs ) )

   model.cliOutputIs( False )

   for intf in intfs:
      portWatchdogStatus = portWatchdogDropCountersHwStatus( intf.name )
      if not portWatchdogStatus:
         continue
      model.cliOutputIs( True )
      model.interfaces[ intf.status().intfId ] = \
         pfcWatchdogIntfEgressDropCountersModelGet( intf, portWatchdogStatus )

   return model

#
# Creates and populates an instance of PfcIntfRangeModel.
#
def pfcIntfRangeModelGet( intfs ):
   """Return the model for PFC on a given range of interfaces."""

   model = PfcIntfRangeModel()

   model.status = pfcStatusModelGet()
   model.watchdogStatus = pfcWatchdogStatusModelGet()

   model.interfaceStatuses = {}

   for intf in intfs:
      model.interfaceStatuses[ intf.status().intfId ] = pfcIntfStatusModelGet( intf )

   model.interfaceCounters = {}

   for intf in intfs:
      model.interfaceCounters[ intf.status().intfId ] = \
                                                    pfcIntfCountersModelGet( intf )

   return model

#
# Value function for the PFC fabric status show command.
#
def showPfcFabricStatus( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces Fabric priority-flow-control
      - show priority-flow-control interfaces Fabric status

      Model := PfcFabricIntfStatusModel"""

   return pfcFabricIntfStatusModelGet()

#
# Value function for the PFC interface status show command.
#
def showPfcInterfaceStatus( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control status
      - show priority-flow-control [ interfaces <range> ] status

      Model := PfcIntfRangeStatusModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeStatusModelGet( intfs )

#
# Value function for the PFC interface counters show command.
#
def showPfcInterfaceCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters
      - show priority-flow-control [ interfaces <range> ] counters

      Model := PfcIntfRangeCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeCountersModelGet( intfs )

#
# Value function for the PFC interface detailed counters show command.
#
def showPfcInterfaceDetailedCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters detail
      - show priority-flow-control [ interfaces <range> ] counters detail

      Model := PfcIntfRangeDetailedCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeDetailedCountersModelGet( intfs )

#
# Value function for the PFC Watchdog interface counters show command.
#
def showPfcWatchdogInterfaceCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters watchdog
      - show priority-flow-control [ interfaces <range> ] counters watchdog

      Model := PfcWatchdogIntfRangeCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )

   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcWatchdogIntfRangeCountersModelGet( intfs )

#
# Value function for the PFC Watchdog interface drop counters show command.
#
def showPfcWatchdogInterfaceDropCounters( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control counters watchdog drop
      - show priority-flow-control [ interfaces <range> ] counters watchdog drop

      Model := PfcWatchdogIntfRangeDropCountersModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcWatchdogIntfRangeDropCountersModelGet( intfs )

#
# Value function for the PFC Watchdog interface drop counters show command.
#
def pfcWatchdogInterfaceDropCountersClear( mode, intf=None, mod=None ):
   """clear watchdog status counters on:
      - clear priority-flow-control counters watchdog drop"""

   intfs = counterSupportedEthIntfs( mode, intf, mod )
   clearCounters = any( portWatchdogDropCountersHwStatus( intf.name )
                        for intf in intfs )
   if clearCounters:
      qosInputConfig.clearWatchdogDropCounters = Tac.now()
#
# Value function for the PFC Watchdog interface stuck/recovery counters show command.
#
def pfcWatchdogInterfaceCountersClear( mode, intf=None, mod=None ):
   """clear watchdog status counters on:
      - clear priority-flow-control counters watchdog"""

   qosInputConfig.clearWatchdogCounters = Tac.now()
#
# Value function for the PFC interface show command.
#
def showPfcInterface( mode, args ):
   """Return the CLI model for either of the following commands:

      - show interfaces [ <range> ] priority-flow-control
      - show priority-flow-control [ interfaces <range> ]

      Model := PfcIntfRangeModel"""
   intf = args.get( 'INTF' )
   mod = args.get( 'MOD' )
   intfs = counterSupportedEthPhyIntfs( mode, intf, mod )

   return pfcIntfRangeModelGet( intfs )

#
# -----------------------------------------------------------------------
# SECTION: Priority Flow Control (PFC) Show Commands
# -----------------------------------------------------------------------
# This section contains registrations of the priority flow control show
# commands with the CLI infrastructure.
#
# These commands come in synonymous pairs. Commands within a pair share
# the same CLI value function and model, thus producing the same output.
# The pairs have been delineated accordingly in the list below.
#
# Commands:
#
# 1.  show interfaces Fabric PFC
# 2.  show PFC interfaces Fabric status
#
# 3.  show interfaces [ <range> ] PFC status
# 4.  show PFC [ interfaces <range> ] status
#
# 5.  show interfaces [ <range> ] PFC counters
# 6.  show PFC [ interfaces <range> ] counters
#
# 7.  show interfaces [ <range> ] PFC counters detail
# 8.  show PFC [ interfaces <range> ] counters detail
#
# 9.  show interfaces [ <range> ] PFC counters watchdog
# 10. show PFC [ interfaces <range> ] counters watchdog
#
# 11. show interfaces [ <range> ] PFC
# 12. show PFC [ interfaces <range> ]
#
# where PFC = 'priority-flow-control'
#
# NOTE: Registration of any priority flow control show commands with the
#       CLI Tech-Support infrastructure is captured at the end of this
#       section.
# -----------------------------------------------------------------------
#

#
# CLI guards.
#
def pfcShowDetailGuard( mode, token ):
   """Guard function for the CLI token: 'detail'"""

   if pfcStatus.pfcClassFramesCounterSupported:
      return None

   return CliParser.guardNotThisPlatform

def pfcShowWatchdogGuard( mode, token ):
   """Guard function for the CLI token: 'watchdog'"""

   if pfcStatus.watchdogSupported:
      return None

   return CliParser.guardNotThisPlatform

def pfcClearWatchdogGuard( mode, token ):
   """Guard function for the CLI token: 'watchdog'"""

   if pfcStatus.watchdogSupported and pfcStatus.watchdogClearSupported:
      return None

   return CliParser.guardNotThisPlatform

# See BUG412221
def pfcWatchdogDropCountersGuard( mode, token ):
   """Guard function for the CLI token: 'drop'"""
   if pfcStatus.watchdogSupported:
      return None

   return CliParser.guardNotThisPlatform


#
# CLI tokens used in the priority flow control show commands.
#

pfcShowKw = CliMatcher.KeywordMatcher( 'priority-flow-control',
                                       helpdesc="Show interface PFC information" )

statusPfcKw = CliMatcher.KeywordMatcher( 'status',
                                         helpdesc="Show interface PFC status" )

countersPfcKw = CliMatcher.KeywordMatcher( 'counters',
                                           helpdesc="Show interface PFC counters" )

detailPfcKw = CliCommand.guardedKeyword( 'detail',
                                         "Detailed view",
                                         pfcShowDetailGuard )

watchdogCountersKw = CliCommand.guardedKeyword( 'watchdog',
                                                "PFC watchdog counters",
                                                pfcShowWatchdogGuard )

watchdogDropCountersKw = CliCommand.guardedKeyword( 'drop',
                                                    "PFC watchdog drop counters",
                                                    pfcWatchdogDropCountersGuard )

interfacePfcKw = CliMatcher.KeywordMatcher( 'interfaces',
                                        helpdesc="Show PFC information for a "
                                        "specific interface" )


def registerPfcIntfCmd( extraSyntax, extraData, cmdHandler, pfcCliModel ):
   # This registers the following two commands:
   # 1. show interface [ INTF ] priority-flow-control XXX
   # 2. show priority-flow-control [ interface INTF ] XXX
   pfcData = { 'priority-flow-control' : pfcShowKw }
   pfcData.update( extraData )

   # register the 'show interface' command
   class ShowIntfPfcCmd( IntfCli.ShowIntfCommand ):
      syntax = 'show interfaces priority-flow-control ' + extraSyntax
      data = pfcData
      handler = cmdHandler
      moduleAtEnd = True
      cliModel = pfcCliModel

   BasicCli.addShowCommandClass( ShowIntfPfcCmd )

   # register the 'show priority-flow-control' command
   class ShowPfcCmd( ShowCommand.ShowCliCommandClass ):
      syntax = "show priority-flow-control [ interfaces INTF ] " + extraSyntax
      data = {
         "priority-flow-control" : pfcShowKw,
         "interfaces" : interfacePfcKw,
         "INTF" : IntfCli.Intf.rangeMatcher,
      }
      data.update( extraData )
      cliModel = pfcCliModel
      handler = cmdHandler

   BasicCli.addShowCommandClass( ShowPfcCmd )

#
# Register command #1:
# show interfaces Fabric priority-flow-control
#
class ShowFabricPfc( ShowCommand.ShowCliCommandClass ):
   syntax = "show interfaces fabric priority-flow-control"
   data = {
      'interfaces' : IntfCli.interfacesShowKw,
      'fabric' : fabricIntfKw,
      'priority-flow-control' : pfcShowKw
      }
   cliModel = PfcFabricIntfStatusModel
   handler = showPfcFabricStatus

BasicCli.addShowCommandClass( ShowFabricPfc )

#
# Register command #2:
# show priority-flow-control interfaces Fabric status
#
class ShowPfcFabricStatus( ShowCommand.ShowCliCommandClass ):
   syntax = "show priority-flow-control interfaces fabric status"
   data = {
      'priority-flow-control' : pfcShowKw,
      'interfaces' : interfacePfcKw,
      'fabric' : fabricIntfKw,
      'status' : statusPfcKw
      }
   cliModel = PfcFabricIntfStatusModel
   handler = showPfcFabricStatus

BasicCli.addShowCommandClass( ShowPfcFabricStatus )

#
# Register command #3:
# show interfaces [ <range> ] priority-flow-control status
# show priority-flow-control [ interfaces <range> ] status
registerPfcIntfCmd( "status",
                    dict( status=statusPfcKw ),
                    showPfcInterfaceStatus,
                    PfcIntfRangeStatusModel )

#
# Register command #4:
# show interfaces [ <range> ] priority-flow-control counters
# show priority-flow-control [ interfaces <range> ] counters
#
registerPfcIntfCmd( 'counters',
                    dict( counters=countersPfcKw ),
                    showPfcInterfaceCounters,
                    PfcIntfRangeCountersModel )

#
# Register command #5:
# show interfaces [ <range> ] priority-flow-control counters detail
# show priority-flow-control [ interfaces <range> ] counters detail
#
registerPfcIntfCmd( 'counters detail',
                    dict( counters=countersPfcKw,
                          detail=detailPfcKw ),
                    showPfcInterfaceDetailedCounters,
                    PfcIntfRangeDetailedCountersModel )

#
# Register command #6:
# show interfaces [ <range> ] priority-flow-control counters watchdog
# show priority-flow-control [ interfaces <range> ] counters watchdog
#
registerPfcIntfCmd( 'counters watchdog',
                    dict( counters=IntfCli.countersKw,
                          watchdog=watchdogCountersKw ),
                    showPfcWatchdogInterfaceCounters,
                    PfcWatchdogIntfRangeCountersModel )

#
# Register command #7:
# show interfaces [ <range> ] priority-flow-control
# show priority-flow-control [ interfaces <range> ]
#
registerPfcIntfCmd( '', {}, showPfcInterface, PfcIntfRangeModel )

#
# Register command #8:
# show interfaces [ <range> ] priority-flow-control counters watchdog drop
# show priority-flow-control [ interfaces <range> ] counters watchdog drop
#
registerPfcIntfCmd( 'counters watchdog drop',
                    dict( counters=countersPfcKw,
                          watchdog=watchdogCountersKw,
                          drop=watchdogDropCountersKw ),
                    showPfcWatchdogInterfaceDropCounters,
                    PfcWatchdogIntfRangeDropCountersModel )

#
# Register command #9:
# clear priority-flow-control counters watchdog [ drop ]
#
clearWatchdogKw = CliMatcher.KeywordMatcher( 'watchdog',
                                             helpdesc="PFC watchdog counters" )

class ClearPfcCountersWatchdog( CliCommand.CliCommandClass ):
   syntax = "clear priority-flow-control counters watchdog_G | ( watchdog drop )"
   data = {
      'clear' : CliToken.Clear.clearKwNode,
      'priority-flow-control' : "Clear PFC information",
      'counters' : "Clear PFC counters",
      'watchdog_G' : CliCommand.Node( clearWatchdogKw,
                                      guard=pfcClearWatchdogGuard ),
      'watchdog' : clearWatchdogKw,
      'drop' : watchdogDropCountersKw
      }
   @staticmethod
   def handler( mode, args ):
      if 'drop' in args:
         pfcWatchdogInterfaceDropCountersClear( mode )
      else:
         pfcWatchdogInterfaceCountersClear( mode )

BasicCli.EnableMode.addCommandClass( ClearPfcCountersWatchdog )

#
# Registrations under Tech-Support.
#
TechSupportCli.registerShowTechSupportCmdCallback(
   '2010-01-01 00:07:00',
   lambda : [
               'show priority-flow-control'
            ] )

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
@Plugins.plugin()
def Plugin( entityManager ):
   global qosInputConfig, pfcStatus, qosHwStatus, qosConfig, dcbxConfig, \
      pfcConfig, qosSliceHwStatus, qosProfileConfigDir

   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosProfileConfigDir = ConfigMount.mount( entityManager, "qos/profile", 
                                            "Qos::QosProfileConfigDir", "w" )
   pfcStatus = LazyMount.mount( entityManager, "dcb/pfc/status", "Pfc::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   dcbxConfig = LazyMount.mount(
      entityManager, "dcb/dcbx/config", "Dcbx::Config", "r" )
   pfcConfig = LazyMount.mount( entityManager, "dcb/pfc/config", "Pfc::Config", "r" )
   IntfCli.Intf.registerDependentClass( PfcIntf )
