#!/usr/bin/env python
# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac           
from QosTypes import tacDirection, tacCMapNm, tacClassMapCpStaticType, \
      tacActionRateType, tacRateUnit, tacActionType, tacClassMapCpType, \
      tacEcnDelayThreshold, tacPMapNm, tacPercent, tacShapeRate, tacBurstUnit, \
      tacPolicerType
import Tracing
import QosLib
import MultiRangeRule
import AclCliLib
from QosLib import coppMapType, pdpMapType, qosMapType, rateUnitFromEnum, \
      isDefaultClass, coppStaticClassesByPrio, classNameFromCpStaticType, \
      classMapCpStaticType, pmapQosActionTypes, burstUnitFromEnum, \
      coppDynamicClassPresent, policyQosClassPresent, coppPerPortClassPresent, \
      coppStaticClassFromHwStatus, bsToBytes, isLagPort
import Cell
from QosCliModel import PolicyMapModel, BandwidthModel, \
      ClassMapAllModel, PoliceModel, QosProfileModel, TxQueueProfileModel, \
      GuaranteedBwModel, ShapeRateModel, EcnParametersModel, PfcProfileModel, \
      EcnDelayConfigurationModel, WredParametersModel, PolicyQosModel, \
      PfcWatchdogPortTimerModel, DropPrecedenceThresholdsModel, \
      LatencyThresholdModel, MatchL2Params, BurstSizeModel, ProfileModel, \
      PolicerInstanceModel, InterfacePolicerModel, HwProgrammingEnumModel

__defaultTraceHandle__ = Tracing.Handle( 'QosCli' )
t0 = Tracing.trace0

def populateL2ParamsModel( match, matchRule, vlanValue=None, innerVlanValue=None,
                           cosValue=None ):
   match.matchRule = matchRule
   vlanString = ''
   innerVlanString = ''
   cosString = ''
   valueString = ''
   modelL2Params = MatchL2Params()

   if vlanValue:
      modelL2Params.vlanMaskValid = vlanValue.maskValid
      if not vlanValue.maskValid:
         ids = set()
         for key in vlanValue.vlanColl:
            ids.update( range( key.min, key.max + 1 ) )
            vlanString = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
      else:
         modelL2Params.vlanMask = vlanValue.vlanMask
         vlanString = str( vlanValue.vlan )
      valueString = vlanString
      modelL2Params.vlanValue = vlanString

   if innerVlanValue:
      modelL2Params.innerVlanMaskValid = innerVlanValue.maskValid
      if not innerVlanValue.maskValid:
         ids = set()
         for key in innerVlanValue.innerVlanColl:
            ids.update( range( key.min, key.max + 1 ) )
            innerVlanString = MultiRangeRule.multiRangeToCanonicalString(
               list( ids ) )
      else:
         modelL2Params.innerVlanMask = innerVlanValue.innerVlanMask
         innerVlanString = str( innerVlanValue.innerVlan )
      valueString = innerVlanString if valueString == '' else valueString
      modelL2Params.innerVlanValue = innerVlanString

   if cosValue:
      ids = set()
      for key in cosValue.cosColl:
         ids.update( range( key.min, key.max + 1 ) )
      cosString = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
      modelL2Params.cosValue = cosString

   match.l2ParamsValue = modelL2Params
   valueString = cosString if valueString == '' else valueString
   return ( valueString, match )

def populateDscpEcnModel( match, matchRule, dscpValue ):
   match.matchRule = matchRule
   ecnName = AclCliLib.ecnNameFromValue( dscpValue.ecn )
   match.matchEcnValue = ecnName
   if dscpValue.dscp == Tac.Value( "Qos::DscpVal" ).invalid:
      if not len( dscpValue.dscpColl ):
         # 'match ecn <ecnName>'
         value = ecnName
      else:
         # 'match dscp <comma-separated range>'
         match.matchDscpNameValid = dscpValue.dscpNameValid
         ids = set()
         for key in dscpValue.dscpColl:
            ids.update( range( key.min, key.max + 1 ) )
         value = \
                 MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
   else:
      match.matchDscpNameValid = dscpValue.dscpNameValid
      if dscpValue.dscpNameValid:
         # 'match dscp <dscpName>'
         value = AclCliLib.dscpNameFromValue( dscpValue.dscp )
         match.dscpValue = dscpValue.dscp
      else:
         # 'match dscp <dscpId>'
         value = str( dscpValue.dscp )
   return( value, match )
   
def populateShapeRateModel( shapeRate, statusShapeRate=None, lagSize=None ):
   if shapeRate is None or shapeRate == tacShapeRate():
      return None
   modelShapeRate = ShapeRateModel()
   modelShapeRate.rate = shapeRate.rate
   modelShapeRate.unit = QosLib.shapeRateUnitFromEnum( shapeRate.unit )
   modelShapeRate.shared = shapeRate.shared
   if shapeRate.percent != tacPercent.invalid:
      if statusShapeRate:
         modelShapeRate.rate = statusShapeRate.rate
      if lagSize and modelShapeRate.shared:
         modelShapeRate.percent = shapeRate.percent / lagSize
      else:   
         modelShapeRate.percent = shapeRate.percent
   return modelShapeRate   

def populateBurstSizeModel( burstSize ):
   modelBurstSize = BurstSizeModel()
   # We always want to store in bytes.
   modelBurstSize.value = bsToBytes( burstSize.value, burstSize.unit )
   modelBurstSize.unit = tacBurstUnit.burstUnitBytes
   return modelBurstSize

def deleteInterfacePolicingConfig( qosInputConfig, intf ):
   intfPair = ( Tac.Value( "Qos::IntfPairKey", "", intf ) if isLagPort( intf ) else
                Tac.Value( "Qos::IntfPairKey", intf, "" ) )
   if qosInputConfig.policingConfig:
      t0( 'Deleting ', intfPair, ' from intf-Policer collection' )
      del qosInputConfig.policingConfig.intfPolicer[ intfPair ]

class PMapModelContainer( object ):
   def __init__( self, qosConfig, qosAclConfig, status, hwStatus, sliceHwStatus,
                 aclHwStatus, aclSliceHwStatus, mapType, direction, hwEpochStatus,
                 defaultPdpPmapCfg, policyMapAllModel=None ):
      self.qosConfig_ = qosConfig
      self.qosAclConfig_ = qosAclConfig
      self.status_ = status
      self.hwStatus_ = hwStatus
      self.sliceHwStatus_ = sliceHwStatus
      self.aclHwStatus_ = aclHwStatus
      self.aclSliceHwStatus_ = aclSliceHwStatus
      self.mapType_ = mapType
      self.direction_ = direction
      self.hwEpochStatus_ = hwEpochStatus
      if self.mapType_ == coppMapType:
         self.direction_ = tacDirection.input
      self.defaultPdpPmapCfg_ = defaultPdpPmapCfg
      self.policyMapAllModel = policyMapAllModel
      self.counterDetail = None
      self.classMaps = dir()
      if self.policyMapAllModel:
         self.policyMapAllModel.maxCmapIs( aclHwStatus.policyQosNumCMapSupported )
         self.policyMapAllModel.mapTypeIs( self.mapType_ )
   
   def populateAll( self, summary=None ):
      pmapList = sorted( self.qosAclConfig_.pmapType[ self.mapType_ ].pmap )
      if self.mapType_ == pdpMapType and self.defaultPdpPmapCfg_:
         pmapList.append( self.defaultPdpPmapCfg_.name )
      self.policyMapAllModel.summaryIs( summary )
      for name in pmapList:
         if self.counterDetail is None:
            self.populatePolicyMap( name, summary )
         else:
            self.populatePolicyMapCounters( name )
      
   def populatePolicyMap( self, pMap, summary=None ):
      policyMap = self._populatePolicyMap( pMap, interfaceOption=False,
                                           summary=summary )
      if policyMap:
         policyMap.mapType = self.mapType_ 
         self.policyMapAllModel.append( pMap, policyMap )

   def populatePolicyMapClassMap( self, pMap, cMap ):
      policyMap = self._populatePolicyMap( pMap, clMap=cMap )
      if policyMap:
         policyMap.mapType = self.mapType_ 
         self.policyMapAllModel.append( pMap, policyMap )

   def populatePolicyMapInterface( self, pMap ):
      policyMap = self._populatePolicyMap( pMap, interfaceOption=True,
                                           counterOption=self.counterDetail )
      if policyMap:
         policyMap.mapType = self.mapType_ 
         self.policyMapAllModel.append( pMap, policyMap )

   def populatePolicyMapCounters( self, pMap ):
      policyMap = self._populatePolicyMap( pMap, interfaceOption=True,
                                           counterOption=True )
      if policyMap:
         policyMap.mapType = self.mapType_ 
         self.policyMapAllModel.append( pMap, policyMap )
   
   def pmapHwStatusPresent( self, pMap, clMap=None ):
      if not self.direction_:
         return None
      key = Tac.newInstance( "Qos::PolicyMapHwStatusKey",
                             self.direction_, pMap.name )
      hwStatus = None
      for sliceHwStatus_ in self.sliceHwStatus_.itervalues():
         if self.mapType_ in sliceHwStatus_.pmapType:
            hwStatus = sliceHwStatus_
            break

      for aclSliceHwStatus_ in self.aclSliceHwStatus_.itervalues():
         if self.mapType_ in aclSliceHwStatus_.pmapType:
            if( coppDynamicClassPresent( self.aclHwStatus_, aclSliceHwStatus_, key,
                                         self.mapType_, clMap=clMap ) or \
               policyQosClassPresent( aclSliceHwStatus_, key, 
                                      self.mapType_, clMap=clMap ) or \
               coppPerPortClassPresent( self.aclHwStatus_, aclSliceHwStatus_, key,
                                        self.mapType_, pMap=pMap, clMap=clMap ) ):
               hwStatus = aclSliceHwStatus_
               break

      if hwStatus is None:
         return None

      if self.mapType_ not in hwStatus.pmapType:
         return None

      if key in hwStatus.pmapType[ self.mapType_ ].pmap:
         pmapHwStatus = hwStatus.pmapType[ self.mapType_ ].pmap[ key ]
         return pmapHwStatus
      return None
       
   def isPolicyMapApplied( self, direction, pMap ):
      # Returns whether policy map is actually applied
      # along with the hwProgrammingStatus
      spKey = Tac.newInstance( "Qos::ServicePolicyKey", pMap.type,
                               direction, pMap.name )
      programmingStatus = ""
      applied = False
      if spKey in self.status_.servicePolicyStatus:
         applied = True
         programmingStatus = QosLib.programmedPMapHwStatus( self.hwStatus_,
                                                            self.aclHwStatus_,
                                                            self.sliceHwStatus_,
                                                            self.aclSliceHwStatus_,
                                                            self.mapType_,
                                                            pMap.name, direction )

      return applied, programmingStatus

   def populatePolicyMapQosInterface( self, pMap, intfServicePolicyQosModel=None ):
      if isinstance( pMap, str ):
         if pMap in self.qosAclConfig_.pmapType[ self.mapType_ ].pmap:
            pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap[ pMap ]
         else:
            return
      pmapHwStatus = self.pmapHwStatusPresent( pMap )
      if pmapHwStatus == None :
         return

      if self.direction_ != None:
         directions = [ self.direction_ ]
      else:
         directions = [ tacDirection.input, tacDirection.output ]
      spApplied = {}
      status = {}
      for direction in directions:
         applied, prgmHwStatus = self.isPolicyMapApplied( direction, pMap )
         spApplied[ direction ] = applied
         status[ direction ] = prgmHwStatus

      if True in spApplied.values():
         # If service policy is applied in either direction, populate the policy
         # with direction and programming status.
         for direction, isApplied in spApplied.iteritems():
            if isApplied:
               policyQosModel = PolicyQosModel()
               policyQosModel.name = pMap.name
               policyQosModel.directionString = direction
               policyQosModel.status = status[ direction ]
               intfServicePolicyQosModel.intfPolicyMaps[ direction ] = policyQosModel

   def _populatePolicyMap( self, pMap, interfaceOption=False,
                           counterOption=False, clMap=None, summary=None ):
      policyMap = PolicyMapModel()
      if isinstance( pMap, str ):
         if pMap in self.qosAclConfig_.pmapType[ self.mapType_ ].pmap:
            pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap[ pMap ]
         elif self.mapType_ == pdpMapType and \
              self.defaultPdpPmapCfg_ and \
              pMap == self.defaultPdpPmapCfg_.name:
            pMap = self.defaultPdpPmapCfg_
         else:
            return None

      if pMap.shared:
         policyMap.shared = True
      pmapHwStatus = self.pmapHwStatusPresent( pMap, clMap=clMap )
      if( ( interfaceOption and self.direction_ ) and pmapHwStatus == None and \
         self.mapType_ != pdpMapType ):
         return None
      if self.direction_ != None:
         directions = [ self.direction_ ]
      else:
         directions = [ tacDirection.input, tacDirection.output ]
      spApplied = {}
      status = {}
      for direction in directions:
         applied, prgmHwStatus = self.isPolicyMapApplied( direction, pMap )
         spApplied[ direction ] = applied
         status[ direction ] = prgmHwStatus
      policyMap.spApplied = spApplied
      if True in spApplied.values():
         # If service policy is applied in either direction, populate the policy
         # with direction and programming status.
         for direction, isApplied in spApplied.iteritems():
            if isApplied:
               hardwareStatus = PolicyMapModel.HwStatus() 
               if self.mapType_ == coppMapType: 
                  hardwareStatus.unitsProgrammed = \
                     sum( 1 for phySlice in self.sliceHwStatus_.itervalues()
                          if phySlice.physicalSlice )
               hardwareStatus.status =  status[ direction ]
               policyMap.hwStatus[ direction ] = hardwareStatus

      if self.mapType_ == coppMapType and pMap.name == tacPMapNm.coppName and \
            self.hwStatus_.intfCoppSupported:
         policyMap.name = pMap.name + \
               " ( default applied per input port if not overridden )"
      else:
         policyMap.name = pMap.name

      # pylint: disable-msg=too-many-nested-blocks
      if summary:
         policyMap.summary = True
         for direction, _ in spApplied.iteritems():
            key = Tac.newInstance( "Qos::ServicePolicyKey", self.mapType_,
                                   direction, pMap.name )
            policyMap.configuredIntfs[ direction ] = \
                                                     PolicyMapModel.InterfaceList()
            if key in self.qosConfig_.servicePolicyConfig:
               configuredIntfs = self.qosConfig_.servicePolicyConfig[ key ].intfIds
               interfaceList = PolicyMapModel.InterfaceList()
               interfaceList.interfaces = []
               if configuredIntfs:
                  interfaceList.interfaces = [
                     intfName for intfName in configuredIntfs if \
                     configuredIntfs[ intfName ] ]
               policyMap.configuredIntfs[ direction ] = interfaceList
            policyMap.activeIntfs[ direction ] = \
                                                 PolicyMapModel.InterfaceList()
            if key in self.status_.servicePolicyStatus:
               activeIntfs = self.status_.servicePolicyStatus[ key ].intfIds
               interfaceList = PolicyMapModel.InterfaceList()
               interfaceList.interfaces = []
               # activeIntfs is a list of IntfPairKeys
               for intf in activeIntfs:
                  if intf.lagIntf:
                     if intf.lagIntf not in interfaceList.interfaces:
                        interfaceList.interfaces.append( intf.lagIntf )
                  else:
                     interfaceList.interfaces.append( intf.intfId )
               policyMap.activeIntfs[ direction ] = interfaceList
            # list of interfaces on which the service policy is not active
            interfaceList = PolicyMapModel.InterfaceList()
            interfaceList.interfaces = list(
               set( policyMap.configuredIntfs[ direction ].interfaces ) - \
               set( policyMap.activeIntfs[ direction ].interfaces ) )
            policyMap.inactiveIntfs[ direction ] = interfaceList
         # We continue generating the list of class maps even for
         # 'summary' case as we need the count of class maps anyways

      classMaps = {}
      clMapObj = None
      if clMap:
         if isinstance( clMap, str ):
            clMapName = clMap
            if clMapName in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
               clMapObj = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ \
                  clMapName ]
            elif isDefaultClass( self.mapType_, clMapName ) == True:
               clMapObj = pMap.classDefault
         else:
            # Store class map name and object seperately as we may have a case where
            # class map is not registered.
            clMapName = clMap.name
            clMapObj = clMap 
         if self.mapType_ == coppMapType and clMapObj == tacCMapNm.coppDrop:
            return
         if  isDefaultClass( self.mapType_, clMapName ) == True or \
                clMapName in pMap.classAction:
            classMap = PolicyMapModel.ClassMap()
            if clMapObj is not None:
               self._populateClassMap( clMapObj, pMap, interfaceOption, \
                                       counterOption, classMap, 1, self.mapType_ )
               self.populateAction( pMap, clMapObj, interfaceOption, counterOption, \
                                    classMap )
            else:
               self._populateClassMap( clMapName, pMap, interfaceOption, \
                                       counterOption, classMap, 1, self.mapType_ )
               self.populateAction( pMap, clMapName, interfaceOption, \
                                    counterOption, classMap )
            if classMap.matchCondition:
               policyMap.classMaps[ clMapName ] = classMap
               return policyMap

      self.populateClassMap( pMap, interfaceOption, counterOption,
                             classMaps=classMaps )
      for cMapName, cMap in classMaps.iteritems():
         if cMapName != None:
            policyMap.classMaps[ cMapName ] = cMap
      return policyMap

   def populateClassMap( self, pMap, interfaceOption, counterOption, clMap=None,
                         classMaps=None ):
      if isinstance( pMap, str ):
         if pMap in self.qosAclConfig_.pmapType[ self.mapType_ ].pmap:
            pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap[ pMap ]
         else:
            return

      if clMap is None:
         # populate dynamic cmaps
         self.populateClassMapDynamic( pMap, interfaceOption, counterOption,
                                       classMaps )
         if self.mapType_ == coppMapType:
            # populate static cmaps
            self.populateClassMapStatic( pMap, interfaceOption, counterOption,
                  classMaps )
      else:
         if isinstance( clMap, str ):
            if clMap in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
               clMap = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMap ]
            elif isDefaultClass( self.mapType_, clMap ) == True:
               clMap = pMap.classDefault
            else:
               return
         if self.mapType_ == coppMapType and clMap == tacCMapNm.coppDrop:
            return
         if  isDefaultClass( self.mapType_, clMap ) == True or \
                clMap.name in pMap.classAction:
            classMap = PolicyMapModel.ClassMap()
            self._populateClassMap( clMap, pMap, interfaceOption, counterOption,
                                    classMap, 1, self.mapType_ )
            self.populateAction( pMap, clMap, interfaceOption, counterOption,
                                 classMap )
   
   def populateClassMapDynamic( self, pMap, interfaceOption, counterOption,
                                classMaps=None ):
      for _, clPrio in sorted( pMap.classPrio.iteritems() ):
         classMap = PolicyMapModel.ClassMap()
         clName = clPrio.cmapName
         # ClassPrios' is a sequence of consecutive integers starting from 1
         # Not using the index as classPrio because it might lead to 
         # nonconsecutive seq
         classPrio = len( classMaps ) + 1
         self._populateClassMap( clName, pMap, interfaceOption, counterOption,
                                 classMap, classPrio, self.mapType_ )
         self.populateAction( pMap, clName, interfaceOption, counterOption,
                              classMap )
         if classMap.matchCondition:
            classMaps[ clName ] = classMap
      # For qosMapType, display the default class
      if self.mapType_ != coppMapType:
         classMap = PolicyMapModel.ClassMap() 
         clName = tacCMapNm.classDefault
         self._populateClassMap( clName, pMap, interfaceOption, counterOption,
                                 classMap, len( classMaps ) + 1, self.mapType_ )
         self.populateAction( pMap, clName, interfaceOption, counterOption,
                              classMap )
         if classMap.matchCondition:
            classMaps[ clName ] = classMap

   def populateClassMapStatic( self, pMap, interfaceOption, counterOption,
                               classMaps ):
      if not self.mapType_ == coppMapType:
         return
      numOfDynamicClasses = len( classMaps )
      idx = 1
      prioAndTypes = coppStaticClassesByPrio( self.sliceHwStatus_ )
      if prioAndTypes is None:
         # Ideally should never fall here
         return
      if pMap.name == tacPMapNm.coppName:
         for classPrioAndType in prioAndTypes:
            cmapCpStatic = classPrioAndType[ 1 ]
            if cmapCpStatic == tacClassMapCpStaticType.cmapCpStaticDrop:
               # BUG470644: Remove this exception for cmapCpStatiDrop once Drop VOQ
               # Mirroring is enabled in eos-trunk.
               continue
            cmapName = classNameFromCpStaticType( cmapCpStatic )
            classMap = PolicyMapModel.ClassMap()
            classPrio = numOfDynamicClasses + idx
            self._populateClassMap( cmapName, pMap, interfaceOption, counterOption,
                                    classMap, classPrio, self.mapType_ )
            self.populateAction( pMap, cmapName, interfaceOption, counterOption,
                                 classMap )
            if classMap.matchCondition:
               classMaps[ cmapName ] = classMap
               idx += 1
      else:
         for classPrio in pMap.coppStaticClassPrio:
            cmapCpStatic = classPrio
            cmapName = pMap.coppStaticClassPrio[ classPrio ].cmapName
            classMap = PolicyMapModel.ClassMap()
            classPrio = numOfDynamicClasses + idx
            self._populateClassMap( cmapName, pMap, interfaceOption, counterOption,
                                    classMap, classPrio, self.mapType_ )
            self.populateAction( pMap, cmapName, interfaceOption, counterOption,
                                 classMap )
            # Do not show empty converted class maps on Sand platforms
            if pMap.name == tacPMapNm.convertedPmapName and \
               not self.hwStatus_.coppActionPolicerSupported and \
               classMap.shape is None and classMap.bandwidth is None:
               continue
            if classMap.matchCondition:
               classMaps[ cmapName ] = classMap
               idx += 1

         # default class
         if pMap.classActionDefault.policer != None or \
            pMap.classActionDefault.policyAction:
            classMap = PolicyMapModel.ClassMap()
            cmapName = pMap.classDefault.name
            self._populateClassMap( cmapName, pMap, interfaceOption, counterOption,
                                    classMap, mapType=self.mapType_ )
            self.populateAction( pMap, cmapName, interfaceOption, counterOption,
                                 classMap )
            if classMap.matchCondition:
               classMaps[ cmapName ] = classMap


   def _populateClassMap( self, clMap, pMap, interfaceOption, counterOption,
                          classMap=None, classPrio=1, mapType='qos' ):
      if isinstance( clMap, str ):
         if clMap in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
            clMap = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMap ]
         elif isDefaultClass( self.mapType_, clMap ) == True:
            clMap = pMap.classDefault
         else:
            # This class-map is not present in qosAclConfig, so for printing this 
            # class-map, we need to populate it's attributes with dummy values.
            # Also match rule is not required for this class-map.
            classMap.matchCondition = 'matchConditionAny'
            classMap.name = clMap
            classMap.classPrio = classPrio
            classMap.mapType = mapType
            return
      
      clMapHwStatus = None
      pmapHwStatus = self.pmapHwStatusPresent( pMap, clMap=clMap )
      if pmapHwStatus != None:
         if clMap.name in pmapHwStatus.cmap:
            clMapHwStatus = pmapHwStatus.cmap[ clMap.name ]
      if( ( interfaceOption and self.direction_ ) and clMapHwStatus == None and \
          self.mapType_ != pdpMapType ):
         return
      assert clMap.matchCondition == 'matchConditionAny'
      classMap.name = clMap.name
      classMap.matchCondition = clMap.matchCondition
      classMap.classPrio = classPrio
      classMap.mapType = mapType
      if not isDefaultClass( self.mapType_, clMap.name ):
         for matchRule in clMap.match:
            match = PolicyMapModel.ClassMap.Match()
            if matchRule in [ 'matchIpAccessGroup', 'matchIpv6AccessGroup', 
                              'matchMacAccessGroup' ]:
               match.matchRule = matchRule
               classMap.match[ clMap.match[ matchRule ].strValue ] = match
            elif matchRule in [ 'matchL2Params' ]:
               vlanValue = clMap.match[ matchRule ].vlanValue
               innerVlanValue = clMap.match[ matchRule ].innerVlanValue
               cosValue = clMap.match[ matchRule ].cosValue
               ( l2ParamsString, match ) = populateL2ParamsModel(
                  match, matchRule, vlanValue, innerVlanValue, cosValue )
               classMap.match[ l2ParamsString ] = match
            elif matchRule in [ 'matchDscpEcn' ]:
               dscpValue = clMap.match[ matchRule ].dscpEcnValue
               ( value, match ) = populateDscpEcnModel( match, matchRule, dscpValue )
               classMap.match[ value ] = match
            elif matchRule in [ 'matchMplsTrafficClass' ]:
               match.matchRule = matchRule
               mplsTrafficClassVal = clMap.match[ matchRule ].mplsTrafficClassVal
               ids = set()
               for key in mplsTrafficClassVal.mplsTrafficClassColl:
                  ids.update( range( key.min, key.max + 1 ) )
               mplsTrafficClassString = \
                  MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
               classMap.match[ mplsTrafficClassString ] = match

   def populateAction( self, pMap, clMap,  interfaceOption, counterOption,
                       classMap=None ):
      if isinstance( pMap, str ):
         if pMap in self.qosAclConfig_.pmapType[ self.mapType_ ].pmap:
            pMap = self.qosAclConfig_.pmapType[ self.mapType_ ].pmap[ pMap ]
         else:
            return
      clMapObj = None
      if isinstance( clMap, str ):
         clMapName = clMap
         if clMapName in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
            clMapObj = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMapName ]
         elif isDefaultClass( self.mapType_, clMapName ) == True:
            clMapObj = pMap.classDefault
      else:
         clMapName = clMap.name
         clMapObj = clMap

      def counter( cmapName, out=True ):
         count = 0
         # Make sure class map is defined before accessing it attributes.
         if clMapObj and ( self.mapType_ in self.status_.pmapType and
              pMap.name in self.status_.pmapType[ self.mapType_ ].pmap ):
            pmapStatus = self.status_.pmapType[ self.mapType_ ].pmap[ pMap.name ]
            clMapStatus = pmapStatus.cmapCounters[ cmapName ]
            if out:
               count = clMapStatus.outCount
            else:
               count = clMapStatus.dropCount
         return count
      
      clMapHwStatus = None
      pmapHwStatus = self.pmapHwStatusPresent( pMap, clMap=clMapObj )
      if pmapHwStatus != None:
         if clMapName in pmapHwStatus.cmap:
            clMapHwStatus = pmapHwStatus.cmap[ clMapName ]
      # Interface option make sense only if the class-map is defined.
      if clMapObj and ( ( interfaceOption and self.direction_ ) and \
         clMapHwStatus == None and self.mapType_ != pdpMapType ):
         return
     
      if isDefaultClass( self.mapType_, clMapName ) == True:
         classAction = pMap.classActionDefault
      elif clMapName in pMap.classAction:
         classAction = pMap.classAction[ clMapName ]
      else:
         return

      def _populateActionCopp( _clMapObj ):
         maxPps = tacActionRateType.noValue
         minPps = tacActionRateType.noValue
         maxRu = minRu = None
         for policyAction in classAction.policyAction.values():
            if policyAction.actionType == tacActionType.actionSetShape:
               if policyAction.rate:
                  if clMapHwStatus:
                     maxPps = clMapHwStatus.shapeVal
                     maxRu = rateUnitFromEnum( policyAction.rate.rateUnit )
               if maxRu == None:
                  if self.hwStatus_.coppPpsSupported:
                     maxRu = rateUnitFromEnum( tacRateUnit.rateUnitPps )
                  elif self.hwStatus_.coppKbpsSupported:
                     maxRu = rateUnitFromEnum( tacRateUnit.rateUnitKbps )
            elif policyAction.actionType == tacActionType.actionSetBandwidth:
               if policyAction.rate:
                  if clMapHwStatus:
                     minPps = clMapHwStatus.bandwidthVal
                     minRu = rateUnitFromEnum( policyAction.rate.rateUnit )
               if minRu == None:
                  if self.hwStatus_.coppPpsSupported:
                     minRu = rateUnitFromEnum( tacRateUnit.rateUnitPps )
                  elif self.hwStatus_.coppKbpsSupported:
                     minRu = rateUnitFromEnum( tacRateUnit.rateUnitKbps )
         shape = None
         bandwidth = None
         outPackets = None
         dropPackets = None
         if maxPps != tacActionRateType.noValue:
            shape = ShapeRateModel( rate=maxPps, unit=maxRu )
         if minPps != tacActionRateType.noValue:
            bandwidth = BandwidthModel(rate=minPps, unit=minRu)
         if interfaceOption:
            if clMapHwStatus:
               # Check whether class map is defined before accessing it's
               # attributes as we may have undefined class as well.
               if _clMapObj and self.hwStatus_.coppOutCounterSupported:
                  outPackets = counter( clMapName, out=True )
               if _clMapObj and self.hwStatus_.coppDropCounterSupported:
                  dropPackets = counter( clMapName, out=False )
         return ( shape, bandwidth, outPackets, dropPackets )

      def _populateActionQosPolicer( cmapName, _clMapObj ):
         actPolice = None
         if pMap.name == tacPMapNm.coppName and \
               self.mapType_ == coppMapType:
            if clMapHwStatus:
               policerRateVal = pmapHwStatus.cmap[ cmapName ].policerRateVal
               policerBurstVal = pmapHwStatus.cmap[ cmapName ].policerBurstVal
               policerRateUnit = pmapHwStatus.cmap[ cmapName ].policerRateUnit
               policerBurstUnit = pmapHwStatus.cmap[ cmapName ].policerBurstUnit
               actPolice = PoliceModel( cir=policerRateVal,
                                        cirUnit=rateUnitFromEnum( policerRateUnit ),
                                        bc=policerBurstVal,
                                        bcUnit=burstUnitFromEnum( policerBurstUnit ),
                                        pir=0, be=0)
         else:
            policer = classAction.policer
            actPolice = PoliceModel( 
               cir=policer.cir, cirUnit=rateUnitFromEnum( policer.cirUnit ),
               bc=policer.bc, bcUnit=burstUnitFromEnum( policer.bcUnit ),
               pir=policer.pir, pirUnit=rateUnitFromEnum( policer.pirUnit ),
               be=policer.be, beUnit=burstUnitFromEnum( policer.beUnit ) )

            actPolice.policerName = policer.name if policer.named else ''
            actPolice.yellowActions = PoliceModel.PolicerActions()
            yellowActions = policer.yellowActions
            if tacActionType.actionSetDropPrecedence in yellowActions:
               actPolice.yellowActions.dropPrecedence = True
            action = yellowActions.get( tacActionType.actionSetDscp )
            if action is not None:
               actPolice.yellowActions.dscp = action.value
               actPolice.yellowActions.dscpConfiguredAsName = \
                                             classAction.dscpConfiguredAsNameYellow
         if _clMapObj and ( self.mapType_ in self.status_.pmapType and
              pMap.name in self.status_.pmapType[ self.mapType_ ].pmap ):
            pmapStatus = self.status_.pmapType[ self.mapType_ ].pmap[ pMap.name ]
            clMapStatus = pmapStatus.cmapCounters[ cmapName ]

         # Do not populate policer related counters in the model for PDP policy
         # classes.
         # We only publish hit counters for PDP policy classes when they have
         # 'count' action set.
         if _clMapObj and counterOption and clMapStatus and not \
            self.mapType_ == pdpMapType:
            # pktConform gets count of green packets
            pktConform = byteConform = pktExceed = byteExceed = 0
            pktYellow = byteYellow = 0
            actPolice.redCounter = self.aclHwStatus_.redCounterSupported
            # accumulated, per-interface specific police counters
            for k, v in clMapStatus.policerStatus.iteritems():
               # conformed pkts = pktInCount (all hits) - pktDropCount (drops)
               pktConform += ( v.pktInCount - v.pktDropCount )
               byteConform += ( v.byteInCount - v.byteDropCount )
               pktYellow += v.yellowPktInCount
               byteYellow += v.yellowByteInCount
               pktExceed += v.pktDropCount
               byteExceed += v.byteDropCount
               actPolice.greenCounter = self.aclHwStatus_.greenCounterSupported
               if self.counterDetail:
                  # per interface counters
                  policeCounters = PoliceModel.IntfPoliceCounters(
                       conformedPackets=( v.pktInCount - v.pktDropCount ),
                       conformedBytes=( v.byteInCount - v.byteDropCount ),
                       yellowPackets=( v.yellowPktInCount ),
                       yellowBytes=( v.yellowByteInCount ),
                       exceededPackets=v.pktDropCount,
                       exceededBytes=v.byteDropCount )
                  actPolice.counters[ k ] = policeCounters
            if self.aclHwStatus_.sharedPolicerSupported and \
               self.mapType_ == qosMapType:
               actPolice.redCounter = self.aclHwStatus_.redCounterSupported
               actPolice.greenCounter = self.aclHwStatus_.greenCounterSupported
               actPolice.perSviRedCounterWithSharedPolicer = \
                       self.aclHwStatus_.perSviRedCounterWithSharedPolicerSupported
               # In case of shared policer for Qos Policy, clMapStatus.policerStatus
               # is not populated by Feature Agent.
               # Feature agent populates clMapStatus.dropCount and
               # clMapStatus.outCount which are the aggregate counters across the
               # interfaces.
               if not self.aclHwStatus_.perSviRedCounterWithSharedPolicerSupported:
                  assert not clMapStatus.policerStatus
               exceededPackets = exceededBytes = 0
               if self.aclHwStatus_.redCounterSupported:
                  exceededPackets = clMapStatus.dropCount
                  exceededBytes = clMapStatus.dropBytes
               policeCounters = PoliceModel.IntfPoliceCounters(
                  conformedPackets=clMapStatus.outCount,
                  conformedBytes=clMapStatus.outBytes,
                  exceededPackets=exceededPackets,
                  exceededBytes=exceededBytes,
                  yellowPackets=clMapStatus.yellowCount,
                  yellowBytes=clMapStatus.yellowBytes )
               actPolice.counters[ 'all' ] = policeCounters
            elif not self.counterDetail:
               # accumulated police counters for all interfaces
               policeCounters = PoliceModel.IntfPoliceCounters(
                  conformedBytes=byteConform, exceededPackets=pktExceed,
                  conformedPackets=pktConform, exceededBytes=byteExceed,
                  yellowPackets=pktYellow,
                  yellowBytes=byteYellow )
               actPolice.counters[ 'all' ] = policeCounters 
         return actPolice

      def _populateActionQos( _clMapObj ):
         def addAction( actionProgrammed ):
            if interfaceOption:
               if self.direction_:
                  if actionProgrammed:
                     return True
               else:
                  return True
            else:
               return True
            return False

         actPolice = None
         if classAction.policer:
            actPolice = _populateActionQosPolicer( clMapName, _clMapObj )
         actions = classAction.policyAction
         # Maintain the order during display
         actStrSetCos = None
         actStrSetDscp = None
         actStrSetDrop = None
         actStrSetTc = None
         actStrSetDropPrecedence = None
         matchedPackets = None
         dropActionRestricted = False
         dscpConfiguredAsName = None
         if self.hwEpochStatus_.isAclTcamRestricted() and \
               Cell.cellType() != 'supervisor':
            dropActionRestricted = True
         for actionType in pmapQosActionTypes:
            if actionType not in actions:
               continue
            if actionType == tacActionType.actionSetDscp:
               if addAction( clMapHwStatus and clMapHwStatus.dscpPrgmd ):
                  action = actions[ actionType ]
                  actStrSetDscp = action.value
                  dscpConfiguredAsName = classAction.dscpConfiguredAsName
            if actionType == tacActionType.actionSetDrop:
               if addAction( clMapHwStatus and clMapHwStatus.dropPrgmd ):
                  action = actions[ actionType ]
                  actStrSetDrop = True
            elif actionType == tacActionType.actionSetCos:
               if addAction( clMapHwStatus and clMapHwStatus.cosPrgmd ):
                  action = actions[ actionType ]
                  actStrSetCos = action.value
            elif actionType == tacActionType.actionSetTc:
               if addAction( clMapHwStatus and clMapHwStatus.tcPrgmd ):
                  action = actions[ actionType ]
                  actStrSetTc = action.value
            elif actionType == tacActionType.actionSetDropPrecedence:
               if addAction( clMapHwStatus and clMapHwStatus.dropPrecedencePrgmd ):
                  action = actions[ actionType ]
                  actStrSetDropPrecedence = action.value
         if counterOption:
            matchedPackets = counter( clMapName, out=True )
         return actPolice, actStrSetCos, actStrSetDscp, actStrSetDrop, \
                actStrSetTc, actStrSetDropPrecedence, \
                dropActionRestricted, dscpConfiguredAsName, \
                matchedPackets

      if self.mapType_ == coppMapType:
         classMap.intfPacketCounters = \
               PolicyMapModel.ClassMap.IntfPacketCounters()
         classMap.coppActionPoliceSupported = \
                                 self.hwStatus_.coppActionPolicerSupported
         if self.hwStatus_.coppActionPolicerSupported:
            if ( classAction.policer or pMap.name == tacPMapNm.coppName ):
               classMap.police = _populateActionQosPolicer( clMapName, clMapObj )
            else:
               classMap.police = None
         else:
            classMap.shape, classMap.bandwidth, \
                classMap.intfPacketCounters.outPackets, \
                classMap.intfPacketCounters.dropPackets = \
                _populateActionCopp(clMapObj) 
      else:
         classMap.qosParams = PolicyMapModel.ClassMap.QosParams()
         classMap.police, classMap.qosParams.cos, classMap.qosParams.dscp, \
               classMap.qosParams.drop, \
               classMap.qosParams.trafficClass, \
               classMap.qosParams.dropPrecedence, \
               classMap.qosParams.dropActionRestricted, \
               classMap.qosParams.dscpConfiguredAsName, \
               matchedPackets = _populateActionQos( clMapObj )
         # Populate matchedPackets attr in the model if it is not a PDP policy
         if self.mapType_ != pdpMapType:
            classMap.matchedPackets = matchedPackets
         # Populate count attr in the model if 'count' action is set.
         # Populate matchedPackets attr in the model if the PDP policy class has
         # 'count' action set.
         if classAction.count:
            classMap.count = True
            classMap.matchedPackets = matchedPackets

class CMapModelContainer( object ):
   def __init__( self, aclConfig, hwStatus, aclHwStatus, sliceHwStatus,
                 cmapType,
                 classMapAllModel ):
      self.qosAclConfig_ = aclConfig
      self.hwStatus_ = hwStatus
      self.aclHwStatus_ = aclHwStatus
      self.sliceHwStatus_ = sliceHwStatus
      self.mapType_ = cmapType
      self.classMapAllModel_ = classMapAllModel

   def populateAll( self ):
      classPrio = 1
      for name in sorted( self.qosAclConfig_.cmapType[ self.mapType_ ].cmap ):
         if self.populateClassMap( name, classPrio ):
            classPrio += 1

   def populateClassMap( self, clMap, classPrio ):
      if isinstance( clMap, str ):
         if clMap in self.qosAclConfig_.cmapType[ self.mapType_ ].cmap:
            clMap = self.qosAclConfig_.cmapType[ self.mapType_ ].cmap[ clMap ]
         else:
            return False

      classMap = ClassMapAllModel.ClassMapModel()
      if clMap.type == coppMapType:
         if clMap.cpType == tacClassMapCpType.cmapCpStatic:
            cmapCpStaticType = classMapCpStaticType( self.qosAclConfig_, clMap.name )
            coppStaticClass = coppStaticClassFromHwStatus( self.sliceHwStatus_ )
            if cmapCpStaticType not in coppStaticClass:
               return False
            if cmapCpStaticType == tacClassMapCpStaticType.cmapCpStaticDrop:
               return False

      assert clMap.matchCondition == 'matchConditionAny'
      classMap.matchCondition = clMap.matchCondition
      classMap.classPrio = classPrio
      if not isDefaultClass( self.mapType_, clMap.name ):
         for matchRule in clMap.match:
            match = ClassMapAllModel.ClassMapModel.Match()
            if matchRule in [ 'matchIpAccessGroup', 'matchIpv6AccessGroup',
                              'matchMacAccessGroup' ]:
               match.matchRule = matchRule
               classMap.match[ clMap.match[ matchRule ].strValue ] = match
            elif matchRule in [ "matchL2Params" ]:
               vlanValue = clMap.match[ matchRule ].vlanValue
               innerVlanValue = clMap.match[ matchRule ].innerVlanValue
               cosValue = clMap.match[ matchRule ].cosValue
               ( l2ParamsString, match ) = populateL2ParamsModel(
                  match, matchRule, vlanValue, innerVlanValue, cosValue )
               classMap.match[ l2ParamsString ] = match
            elif matchRule in [ 'matchDscpEcn' ]:
               dscpValue = clMap.match[ matchRule ].dscpEcnValue
               ( value, match ) = populateDscpEcnModel( match, matchRule, dscpValue )
               classMap.match[ value ] = match
            elif matchRule in [ 'matchMplsTrafficClass' ]:
               match.matchRule = matchRule
               mplsTrafficClassVal = clMap.match[ matchRule ].mplsTrafficClassVal
               ids = set()
               for key in mplsTrafficClassVal.mplsTrafficClassColl:
                  ids.update( range( key.min, key.max + 1 ) )
               mplsTrafficClassString = \
                  MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
               classMap.match[ mplsTrafficClassString ] = match

      self.classMapAllModel_.append( clMap.name, classMap )
      return True

class QosProfileModelContainer( object ):
   def __init__( self, config, hwStatus, qosProfileAllModel=None ):
      self.config_ = config
      self.hwStatus_ = hwStatus
      self.qosProfileAllModel = qosProfileAllModel

   def populateAll( self ):
      for qosProfileName in sorted( self.config_ ):
         self.populateQosProfile( qosProfileName )
   
   def populateQosProfile( self, qosProfileName ):
      qosProfile = self._populateQosProfile( qosProfileName )
      if qosProfile:
         self.qosProfileAllModel.append( qosProfileName, qosProfile )

   def _populateQosProfile( self, qosProfileName ):
      qosProfile = QosProfileModel()
      profileName = qosProfileName
      if isinstance( qosProfileName, str ):
         if qosProfileName in self.config_:
            qosProfileName = self.config_[ qosProfileName ]
         else:
            return None

      qosProfileInConfig = self.config_[ profileName ].qosProfile

      qosProfile.trustMode = str( qosProfileInConfig.trustMode )
      
      if self.hwStatus_.defaultCosSupported:
         qosProfile.defaultCosSupported = True
         qosProfile.defaultCos = int( qosProfileInConfig.defaultCos )

      if self.hwStatus_.defaultDscpSupported:
         qosProfile.defaultDscpSupported = True
         qosProfile.defaultDscp = int( qosProfileInConfig.defaultDscp )
         
      qosProfile.configuredPortShapeRate = \
         populateShapeRateModel( qosProfileInConfig.shapeRate )

      # Pfc
      pfcPortConfig = qosProfileInConfig.pfcPortConfig
      if pfcPortConfig:
         priorities = pfcPortConfig.priorities
         qosProfile.pfcWatchdogEnabled = pfcPortConfig.watchdogEnabled
         qosProfile.pfcWatchdogPortAction = pfcPortConfig.watchdogPortAction

         if pfcPortConfig.enabled:
            qosProfile.pfcEnabled = True
         for priority in range( 8 ):
            bit = 1 << priority
            if bit & priorities:
               pfcModel = PfcProfileModel() 
               if bit & qosProfileInConfig.fabricPfcDlb:
                  pfcModel.dynamicLoadBalancing = True
               qosProfile.pfcPriorities[ priority ] = pfcModel
         if pfcPortConfig.portTimerConfig.usePerPortTimerValues:
            qosProfile.pfcWatchdogPortTimerValues = PfcWatchdogPortTimerModel()
            portTimerConfig = pfcPortConfig.portTimerConfig
            portTimerModel = qosProfile.pfcWatchdogPortTimerValues
            portTimerModel.timeout = portTimerConfig.portWatchdogTimeout
            portTimerModel.pollingInterval = \
                                       portTimerConfig.portWatchdogPollingInterval
            portTimerModel.recoveryTime = \
                           portTimerConfig.portWatchdogRecoveryCfg.recoveryTime
            portTimerModel.forcedRecovery = \
                           portTimerConfig.portWatchdogRecoveryCfg.forcedRecovery

      # Qos Service Policies
      servicePolicyConfig = qosProfileInConfig.servicePolicyConfig
      if servicePolicyConfig:
         key = servicePolicyConfig.key
         qosProfile.configuredServicePolicy = PolicyQosModel()
         qosProfile.configuredServicePolicy.name = key.pmapName
         qosProfile.configuredServicePolicy.directionString = key.direction
         qosProfile.configuredServicePolicy.status = \
             QosLib.tacPMapHwPrgmStatus.hwPrgmStatusSuccess

      qosProfileTxQueueConfig = self.config_[ profileName ].\
            qosProfile.txQueueConfig
      qosProfile.numTxQueueSupported = self.hwStatus_.numTxQueueSupported
      for queueConfig in qosProfileTxQueueConfig:
         txQueueConfig = qosProfileTxQueueConfig[ queueConfig ]
         txQueueProfileModel = TxQueueProfileModel()
         txQueueModel = TxQueueProfileModel.TxQueue()
         txQueueModel.ecnWeightSupported = self.hwStatus_.ecnWeightSupported
         txQueueModel.wredWeightSupported = self.hwStatus_.wredWeightSupported
         txQueueModel.nonEctWeightSupported = self.hwStatus_.nonEctWeightSupported
         txQueueModel.ecnMarkProbSupported = self.hwStatus_.ecnMarkProbSupported
         # Priority
         txQueuePriority = Tac.Type( 'Qos::TxQueuePriority' )
         if txQueueConfig.priority != txQueuePriority.priorityInvalid:
            txQueueModel.priority = True

         # Wrr
         bandwidthPercent = Tac.Type( 'Qos::Percent' )
         if txQueueConfig.bandwidth != bandwidthPercent.invalid:
            txQueueModel.wrrBandwidthPercent = txQueueConfig.bandwidth

         # Shape
         txQueueModel.shapeRate = populateShapeRateModel( txQueueConfig.shapeRate )

         # Guaranteed Bw
         invalidGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
         if txQueueConfig.guaranteedBw and \
               txQueueConfig.guaranteedBw != invalidGuaranteedBw:
            txQueueModel.guaranteedBw = GuaranteedBwModel()
            txQueueModel.guaranteedBw.bandwidth = txQueueConfig.guaranteedBw.bw
            txQueueModel.guaranteedBw.unit = QosLib.guaranteedBwUnitFromEnum(
                  txQueueConfig.guaranteedBw.unit )

         # Ecn
         if txQueueConfig.ecnConfig:
            txQueueModel.ecn = EcnParametersModel()
            txQueueModel.ecn.minThreshold = txQueueConfig.ecnConfig.minThd
            txQueueModel.ecn.maxThreshold = txQueueConfig.ecnConfig.maxThd
            txQueueModel.ecn.unit = txQueueConfig.ecnConfig.unit
            txQueueModel.ecn.maxDroprate = txQueueConfig.ecnConfig.maxDroprate
            txQueueModel.ecn.weight = txQueueConfig.ecnConfig.weight

         if self.hwStatus_.ecnDelaySupported and txQueueConfig.delayEcnEnabled:
            txQueueModel.ecnDelay = EcnDelayConfigurationModel()
            if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold:
               txQueueModel.ecnDelay.source = "local"
               txQueueModel.ecnDelay.configuredThreshold = \
                     txQueueConfig.ecnDelayThreshold.threshold / 1000.0
            else:
               txQueueModel.ecnDelay.source = "global"

         # Wred
         if txQueueConfig.wredConfig:
            txQueueModel.wred = WredParametersModel()
            txQueueModel.wred.minThreshold = txQueueConfig.wredConfig.minThd
            txQueueModel.wred.maxThreshold = txQueueConfig.wredConfig.maxThd
            txQueueModel.wred.unit = txQueueConfig.wredConfig.unit
            txQueueModel.wred.maxDroprate = txQueueConfig.wredConfig.maxDroprate
            txQueueModel.wred.weight = txQueueConfig.wredConfig.weight

         # DP Thresholds
         for dp in txQueueConfig.dropThresholds:
            dpThreshold = DropPrecedenceThresholdsModel()
            dpThreshold.percent = txQueueConfig.dropThresholds[ dp ]
            txQueueModel.dpThresholds[ dp ] = dpThreshold

         txQueueId = txQueueConfig.txQueue.id 
         txQueueType = txQueueConfig.txQueue.type
         qosProfile.append( txQueueId, txQueueType, txQueueModel, 
                            txQueueProfileModel )
         # NonEct
         if txQueueConfig.nonEctConfig:
            txQueueModel.nonEct = EcnParametersModel()
            txQueueModel.nonEct.minThreshold = txQueueConfig.nonEctConfig.minThd
            txQueueModel.nonEct.maxThreshold = txQueueConfig.nonEctConfig.maxThd
            txQueueModel.nonEct.unit = txQueueConfig.nonEctConfig.unit
            txQueueModel.nonEct.weight = txQueueConfig.nonEctConfig.weight
         
         txQueueModel.latencyThreshold = LatencyThresholdModel()
         txQueueModel.latencyThreshold.threshold = \
               txQueueConfig.latencyThreshold.configThreshold()
         txQueueModel.latencyThreshold.unit = \
               txQueueConfig.latencyThreshold.configUnit

         txQueueId = txQueueConfig.txQueue.id 
         txQueueType = txQueueConfig.txQueue.type
         qosProfile.append( txQueueId, txQueueType, txQueueModel, 
                            txQueueProfileModel )
      return qosProfile

class PolicingModelContainer( object ):
   def __init__( self, qosInputConfig, qosConfig, qosStatus,
                 intfPolicingHwStatus, modePolicingModel=None ):
      self.qosConfig = qosConfig
      self.qosStatus = qosStatus
      self.intfPolicingHwStatus = intfPolicingHwStatus
      self.modePolicingModel = modePolicingModel

   def populateAllPolicerProfiles( self, profileOutput, policerOutput ):
      if not self.qosStatus.policingStatus:
         return self.modePolicingModel
      if profileOutput and policerOutput:
         self.modePolicingModel.populateAllOutput( True )
      if profileOutput:
         profileList = sorted( self.qosStatus.policingStatus.profile )
         for profile in profileList:
            self.populateProfile( profile )
      if policerOutput:
         policerList = sorted( self.qosStatus.policingStatus.policerInstance )
         for policerName in policerList:
            self.populatePolicerInstance( policerName )

   def populatePolicerProfileIntfAssociations( self, intfPair, policerProfile ):
      for policingHwStatus in self.intfPolicingHwStatus.itervalues():
         if intfPair in policingHwStatus.intfPolicerHwStatus:
            intfPairStatus = policingHwStatus.intfPolicerHwStatus[ intfPair ].status
            if intfPair.lagIntf:
               if intfPair.lagIntf in policerProfile.configuredIngressIntfs:
                  confIntfStatus = policerProfile.configuredIngressIntfs[
                     intfPair.lagIntf ]
                  if confIntfStatus is False:
                     return
                  elif confIntfStatus == intfPairStatus:
                     return
               policerProfile.configuredIngressIntfs[
                  intfPair.lagIntf ] = intfPairStatus
               return
            else:
               policerProfile.configuredIngressIntfs[
                  intfPair.intfId ] = intfPairStatus
               return
      if intfPair.lagIntf:
         policerProfile.configuredIngressIntfs[ intfPair.lagIntf ] = False
      else:
         policerProfile.configuredIngressIntfs[ intfPair.intfId ] = False

   def populateProfile( self, profileName ):
      profile = ProfileModel()
      profile.profileName = profileName
      if profileName in self.qosStatus.policingStatus.profile:
         profileConfig = self.qosStatus.policingStatus.profile[
            profileName ].profileConfig
         profile.police = PoliceModel( cir=profileConfig.cir,
                                 cirUnit=rateUnitFromEnum( profileConfig.cirUnit ),
                                 bc=profileConfig.bc,
                                 bcUnit=burstUnitFromEnum( profileConfig.bcUnit ),
                                 pir=profileConfig.pir,
                                 pirUnit=rateUnitFromEnum( profileConfig.pirUnit ),
                                 be=profileConfig.be,
                                 beUnit=burstUnitFromEnum( profileConfig.beUnit ) )
         if profileName in self.qosStatus.qosPolicingMap.profileToPolicerInstanceMap:
            profile.policerAssociations = \
                           self.qosStatus.qosPolicingMap.profileToPolicerInstanceMap[
                              profileName ].policerInstanceMap.keys()
         intfPairList = []
         if profileName in self.qosStatus.qosPolicingMap.profileToIntfMap:
            intfPairList = self.qosStatus.qosPolicingMap.profileToIntfMap[
               profileName ].intfMap.keys()
         profile.configuredIngressIntfs = {}
         for intfPair in intfPairList:
            self.populatePolicerProfileIntfAssociations( intfPair, profile )
      self.modePolicingModel.populateProfile( profileName, profile )

   def populatePolicerInstance( self, policerName ):
      policer = PolicerInstanceModel()
      policer.policerName = policerName
      if policerName in self.qosStatus.policingStatus.policerInstance:
         policer.profileName = self.qosStatus.policingStatus.policerInstance[
            policerName ].profileName
         intfPairList = []
         if policerName in self.qosStatus.qosPolicingMap.policerInstanceToIntfMap:
            intfPairList = self.qosStatus.qosPolicingMap.policerInstanceToIntfMap[
               policerName ].intfMap.keys()
         policer.configuredIngressIntfs = {}
         for intfPair in intfPairList:
            self.populatePolicerProfileIntfAssociations( intfPair, policer )
      self.modePolicingModel.populatePolicerInstance( policerName, policer )

class IntfPolicerModelContainer( object ):
   def __init__( self, qosInputConfig, qosConfig, qosStatus,
                 intfPolicingHwStatus, intfPolicerAllModel ):
      self.qosConfig = qosConfig
      self.qosStatus = qosStatus
      self.intfPolicingHwStatus = intfPolicingHwStatus
      self.intfPolicerAllModel = intfPolicerAllModel

   def populateAllIntfPolicers( self, detail ):
      if not self.qosStatus.policingStatus:
         return self.intfPolicerAllModel
      intfList = sorted( self.qosStatus.policingStatus.intfPolicer.keys() )
      self.intfPolicerAllModel.detailIs( detail )
      for intfPair in intfList:
         intf = intfPair.lagIntf if intfPair.lagIntf else intfPair.intfId
         self.populateIntfPolicers( intf )

   def populateIntfRangePolicers( self, intfs, detail ):
      if not self.qosStatus.policingStatus:
         return self.intfPolicerAllModel
      self.intfPolicerAllModel.detailIs( detail )
      for intf in intfs:
         self.populateIntfPolicers( intf.name )

   def populatePortChannelStatus( self, intfPairStatus, intfPolicerInfo ):
      if intfPolicerInfo.isConfigured is None:
         hwState_ = 'active' if intfPairStatus else 'inactive'
         intfPolicerInfo.isConfigured = HwProgrammingEnumModel( hwState=hwState_ )
      elif intfPolicerInfo.isConfigured.hwState == 'inactive':
         if intfPairStatus:
            intfPolicerInfo.isConfigured.hwState = 'partial'
         else:
            return
      elif intfPolicerInfo.isConfigured.hwState == 'active':
         if intfPairStatus:
            return
         else:
            intfPolicerInfo.isConfigured.hwState = 'partial'
      else:
         return

   def populateMemberAndStatus( self, intfPairKey, policingHwStatus,
                                intfPolicerInfo ):
      intfPairStatus = policingHwStatus.intfPolicerHwStatus[ intfPairKey ].status
      if intfPairStatus:
         intfPolicerInfo.portChannelMembers[
            intfPairKey.intfId ] = HwProgrammingEnumModel( hwState='active' )
      else:
         intfPolicerInfo.portChannelMembers[
            intfPairKey.intfId ] = HwProgrammingEnumModel( hwState='inactive' )
      self.populatePortChannelStatus( intfPairStatus, intfPolicerInfo )

   def populatePortChannelMembers( self, intf, intfPolicerInfo ):
      for intfPairKey in self.qosStatus.policingStatus.intfPolicer:
         if intf == intfPairKey.lagIntf:
            for policingHwStatus in self.intfPolicingHwStatus.itervalues():
               if intfPairKey in policingHwStatus.intfPolicerHwStatus:
                  self.populateMemberAndStatus( intfPairKey, policingHwStatus,
                                                intfPolicerInfo )
                  break
            else:
               intfPolicerInfo.portChannelMembers[
                  intfPairKey.intfId ] = HwProgrammingEnumModel(
                     hwState='inactive' )
               self.populatePortChannelStatus( intfPairStatus=False,
                                               intfPolicerInfo=intfPolicerInfo )

   def populateInterfaceStatus( self, intfPair, intfPolicerInfo ):
      for policingHwStatus in self.intfPolicingHwStatus.itervalues():
         if intfPair in policingHwStatus.intfPolicerHwStatus:
            intfPairStatus = policingHwStatus.intfPolicerHwStatus[ intfPair ].status
            if intfPairStatus:
               intfPolicerInfo.isConfigured = HwProgrammingEnumModel(
                  hwState='active' )
            else:
               intfPolicerInfo.isConfigured = HwProgrammingEnumModel(
                  hwState='inactive' )
            break
      else:
         intfPolicerInfo.isConfigured = HwProgrammingEnumModel( hwState='inactive' )

   def populateIntfPolicers( self, intf ):
      intfPolicerInfo = InterfacePolicerModel()
      for interface in self.qosStatus.policingStatus.intfPolicer:
         intfStatusInfo = self.qosStatus.policingStatus.intfPolicer[
            interface ]
         intfStatusPairKey = intfStatusInfo.intfPairKey
         if intf == intfStatusPairKey.intfId or intf == intfStatusPairKey.lagIntf:
            if intfStatusInfo.policerName.has_key( tacPolicerType.dedicatedPolicer ):
               intfPolicerInfo.profileName = intfStatusInfo.policerName[
                  tacPolicerType.dedicatedPolicer ]
            if intfStatusInfo.policerName.has_key( tacPolicerType.policerInstance ):
               intfPolicerInfo.policerName = intfStatusInfo.policerName[
                  tacPolicerType.policerInstance ]
            if not isLagPort( intf ):
               self.populateInterfaceStatus( intfStatusPairKey, intfPolicerInfo )
            else:
               intfPolicerInfo.portChannelMembers = {}
               self.populatePortChannelMembers( intf, intfPolicerInfo )
            self.intfPolicerAllModel.populateIntfPolicerInfo( intf, intfPolicerInfo )
            break
