# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import QosCli
from QosCli import * # pylint: disable-msg=wildcard-import

import Plugins
import BasicCliModes

__defaultTraceHandle__ = Tracing.Handle( 'QosCliCopp' )

needToConvertOldCoppSystemPolicy = False

def guardPMapCopp( mode, token ):
   if qosHwStatus.coppSupported or qosHwStatus.intfCoppSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCMapCopp( mode, token ):
   if qosAclHwStatus.coppDynamicClassSupported or qosHwStatus.intfCoppSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCMapDynamicCopp( mode, token ):
   if qosAclHwStatus.coppDynamicClassSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapCoppEnableDisable( mode, token ):
   if qosHwStatus.coppEnableDisable:
      return None
   return CliParser.guardNotThisEosVersion

def guardPMapCoppRateUnitPps( mode, token ):
   if ( qosHwStatus.coppSupported or qosHwStatus.intfCoppSupported ) \
         and qosHwStatus.coppPpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCoppSystemPolicy( mode, token ):
   if not qosHwStatus.intfCoppSupported and \
      qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfCoppSupported( mode, token ):
   if qosHwStatus.intfCoppSupported and \
      qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapCoppRateUnitKbps( mode, token ):
   if ( qosHwStatus.coppSupported or qosHwStatus.intfCoppSupported ) \
         and qosHwStatus.coppKbpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def coppActionShaperSupportedGuard( mode, token ):
   if qosHwStatus.coppActionShaperSupported and \
      qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def coppActionPolicerSupportedGuard( mode, token ):
   if qosHwStatus.coppActionPolicerSupported and \
      qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

def guardCoppPoliceRateInBytes( mode, token ):
   if qosHwStatus.coppActionPolicerSupported and \
      qosHwStatus.coppKbpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCoppPoliceRateInPps( mode, token ):
   if qosHwStatus.coppActionPolicerSupported and \
      qosHwStatus.coppPpsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyCopp( mode, token ):
   if qosHwStatus.intfCoppSupported:
      if type( mode ) == QosProfileMode:
         return None
      elif mode.intf.isSubIntf():
         if qosAclHwStatus.subIntfPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.sviPolicyQosSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def guardServicePolicyCopp( mode, token ):
   if qosHwStatus.coppSupported:
      return None
   return CliParser.guardNotThisPlatform

def coppActionRateShapeAndBw( mode ):
   return QosLib.coppRateShapeAndBandwidth( qosHwStatus )

def getPMapNameRuleCopp( mode ):
   return getPMapNameRule( mode, coppMapType )

def getClassNameRuleCopp( mode ):
   return getClassNameRule( mode, coppMapType )

def getCMapNameRuleCopp( mode ):
   return getCMapNameRule( mode, coppMapType )

matcherPoliceCirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify lower rate' )
matcherPoliceCommittedBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst size' )
matcherType = CliMatcher.KeywordMatcher( 'type', 
      helpdesc='Specify type' )
matcherClassName = CliMatcher.DynamicNameMatcher( getClassNameRuleCopp,
      "Class Map Name" )
matcherCmapName = CliMatcher.DynamicNameMatcher( getCMapNameRuleCopp, 
      "Class Map Name" )
matcherClass = CliMatcher.KeywordMatcher( 'class', 
      helpdesc='Policy criteria' )
matcherCoppSystemPolicy = CliMatcher.KeywordMatcher( tacPMapNm.coppName,
      helpdesc='Policy-map name' )
matcherMatchAny = CliMatcher.KeywordMatcher( 'match-any',
      helpdesc='Logical-OR all match statements under this class-map' )
matcherPmapName = CliMatcher.DynamicNameMatcher( getPMapNameRuleCopp, 
      'Policy Map Name', 
      pattern='(?!counters$|interface$|summary$)[A-Za-z0-9_:{}\\[\\]-]+' )
nodeCoppPolice = CliCommand.guardedKeyword( 'police',
      "Configure policer parameters", guard=coppActionPolicerSupportedGuard )
nodePolicyMap = CliCommand.guardedKeyword( 'policy-map', 
      "Configure Policy Map", guard=QosCli.guardPolicyMap )
nodePolicyMapShow = CliCommand.guardedKeyword( 'policy-map',
      "Show Policy Map", guard=QosCli.guardPolicyMap )
nodeCopp = CliCommand.guardedKeyword( 'copp',
      "Control-plane type", guard=guardPMapCopp )
nodeCoppDeprecated = CliCommand.guardedKeyword( 'control-plane', 
      "Control-plane type", guard=guardPMapCopp, 
      deprecatedByCmd='[no] policy-map type copp <pmap-name>' )
nodeCoppIntf = CliCommand.guardedKeyword( 'copp',
      "Control-plane type", guard=guardIntfCoppSupported )
nodeControlPlaneDeprecated = CliCommand.guardedKeyword( 'control-plane',
      "Control-plane type", guard=guardPMapCopp,
      deprecatedByCmd='show policy-map type copp [ Policy Map Name ]' )
nodeCoppSystemPolicy = CliCommand.guardedKeyword( tacPMapNm.coppName,
      "Copp Policy-map name", guard=guardCoppSystemPolicy )
nodeCoppPps = CliCommand.guardedKeyword( 'pps',
      "Specify Rate in packets per second", guard=guardPMapCoppRateUnitPps )
nodeCoppKbps = CliCommand.guardedKeyword( 'kbps',
      "Specify Rate in Kilo-bits per second", guard=guardPMapCoppRateUnitKbps )

class ClassMapModeCopp( ClassMapMode ):
   name = "Class Map Copp Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      ClassMapMode.__init__( self, parent, session, context )

class RateExpression( CliCommand.CliExpression ):
   expression = 'bps | kbps | mbps | pps'
   data = {
         'bps': CliCommand.guardedKeyword( 'bps',
            "rate in bps (default unit)", guard=guardCoppPoliceRateInBytes ),
         'kbps': CliCommand.guardedKeyword( 'kbps',
            "The rate expressed in units of kilobits per second",
            guard=guardCoppPoliceRateInBytes ),
         'mbps': CliCommand.guardedKeyword( 'mbps',
            "rate in Mbps", guard=guardCoppPoliceRateInBytes ),
         'pps': CliCommand.guardedKeyword( 'pps',
            "rate in pps", guard=guardCoppPoliceRateInPps ),
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      rateUnitsToTypes = {
         'bps': tacRateUnit.rateUnitbps,
         'kbps': tacRateUnit.rateUnitKbps,
         'mbps': tacRateUnit.rateUnitMbps,
         'pps': tacRateUnit.rateUnitPps,
      }

      for rate in ( 'bps', 'kbps', 'mbps', 'pps' ):
         if rate in args:
            args[ 'RATE_CIR_UNIT' ] = rateUnitsToTypes[ args[ rate ] ]
            break

class BurstExpression( CliCommand.CliExpression ):
   expression = 'bytes | kbytes | mbytes | packets'
   data = {
         'bytes': CliCommand.guardedKeyword( 'bytes',
            "burst size in bytes (default unit)", guard=guardCoppPoliceRateInBytes ),
         'kbytes': CliCommand.guardedKeyword( 'kbytes',
            "burst size in kbytes", guard=guardCoppPoliceRateInBytes ),
         'mbytes': CliCommand.guardedKeyword( 'mbytes',
            "burst size in mbytes", guard=guardCoppPoliceRateInBytes ),
         'packets': CliCommand.guardedKeyword( 'packets',
            "burst size in packets", guard=guardCoppPoliceRateInPps ),
         }

   @staticmethod
   def adapter( mode, args, argsList ):
      burstUnitsToTypes = {
         'bytes': tacBurstUnit.burstUnitBytes,
         'kbytes': tacBurstUnit.burstUnitKBytes,
         'mbytes': tacBurstUnit.burstUnitMBytes,
         'packets': tacBurstUnit.burstUnitPackets
      }

      for rate in ( 'bytes', 'kbytes', 'mbytes', 'packets' ):
         if rate in args:
            args[ 'BURST_BC_UNIT' ] =  burstUnitsToTypes[ args[ rate ] ]
            break

def rateUnitAdapter( mode, args, argsList ):
   rateUnitsToTypes = {
      'pps': tacRateUnit.rateUnitPps,
      'kbps': tacRateUnit.rateUnitKbps,
   }
   args[ 'rateUnit' ] = rateUnitsToTypes[ args.get( 'kbps', 'pps' ) ]

#-----------------------------------------------------------------
# The show command in 'config-cmap' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------
class CmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new class-map configuration to be applied',
      'active': 'Display the class-map configuration in the current running-config',
      'current': ( 'Display the class-map configuration '
         'in the current running-config' ),
      'diff': ( 'Display the diff between class-map configuration '
          'current running-config and to be applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         QosCli.showCMapDiff( mode )
      elif 'active' in args or 'current' in args:
         QosCli.showCMapActive( mode )
      else:
         QosCli.showCMapPending( mode )

ClassMapModeCopp.addShowCommandClass( CmapModeShowCmd )

#------------------------------------------------------------------------
# Register top-level CLI commands ( class-map )
#------------------------------------------------------------------------
# Goto 'class-map' config mode. Create a class map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.
def gotoCoppClassMapMode( mode, args ):
   mapType = 'control-plane'
   cmapName = args.get( 'CMAP' )
   matchType = 'match-any'
   t0( 'gotoClassMapMode %s' % ( cmapName ) )
   emapType = mapTypeToEnum( mapType )
   if emapType == coppMapType:
      cpType = classMapCpType( cliQosAclConfig, cmapName )
      if cpType == tacClassMapCpType.cmapCpStatic:
         mode.addError( CliError[ 'cannotConfigureCMap' ] )
         return
   context = ClassMapContext( mode, emapType, cmapName, matchType )

   if cmapName in cliQosAclConfig.cmapType[ emapType ].cmap:
      entry = cliQosAclConfig.cmapType[ emapType ].cmap[ cmapName ]
      context.copyEditEntry( entry )
      if entry.dynamic:
         mode.addWarning( DYNAMIC_CMAP_EDIT_WARN )
   else:
      context.newEditEntry()

   mode.cmapContext = context
   childMode = mode.childMode( ClassMapModeCopp, context=context )
   mode.session_.gotoChildMode( childMode )

def deleteCoppClassMap( mode, args ):
   mapType = 'control-plane'
   cmapName = args.get( 'CMAP' )
   matchType = args.get( 'match-any' )
   cpType = classMapCpType( cliQosAclConfig, cmapName )
   if cpType == tacClassMapCpType.cmapCpStatic:
      mode.addError( CliError[ 'cannotDeleteCMap' ] )
      return
   emapType = mapTypeToEnum( mapType )
   if cmapName in cliQosAclConfig.cmapType[ emapType ].cmap:
      deleteClassMap( mode, cmapName, mapType, matchType )

#--------------------------------------------------------------------------------
# class-map type ( copp | control-plane ) match-any CMAP
#--------------------------------------------------------------------------------
class QosCoppClassMapModeCmd( CliCommand.CliCommandClass ):
   syntax = 'class-map type ( copp | control-plane ) match-any CMAP'
   data = {
      'class-map' : CliCommand.guardedKeyword( 'class-map', 
         "Configure Class Map", guard=guardClassMap ),
      'type' : matcherType,
      'copp' : CliCommand.guardedKeyword( 'copp', 
         "Control-plane type", guard=guardCMapDynamicCopp ),
      'control-plane' : CliCommand.guardedKeyword( 'control-plane', 
         "Control-plane type", guard=guardCMapDynamicCopp, 
         deprecatedByCmd='class-map type copp match-any <class-name>' ),
      'match-any' : matcherMatchAny,
      'CMAP' : matcherCmapName, 
   }

   handler = gotoCoppClassMapMode
   autoConfigSessionAllowed=False

BasicCliModes.GlobalConfigMode.addCommandClass( QosCoppClassMapModeCmd )

#--------------------------------------------------------------------------------
# ( no | default ) class-map type ( copp | control-plane ) [ match-any ] CMAP
#--------------------------------------------------------------------------------
class QosCoppClassMapModeNoOrDefaultCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = ( 'class-map type ( copp | control-plane ) '
                         '[ match-any ] CMAP' )
   data = {
      'class-map' : CliCommand.guardedKeyword( 'class-map', 
         "Configure Class Map", guard=guardClassMap ),
      'type' : matcherType,
      'copp' : nodeCopp, 
      'control-plane' : CliCommand.guardedKeyword( 'control-plane', 
         "Control-plane type", guard=guardPMapCopp, 
         deprecatedByCmd='no class-map type copp [ match-any ] <class-name>' ),
      'match-any' : matcherMatchAny,
      'CMAP' : matcherCmapName,
   }

   noOrDefaultHandler = deleteCoppClassMap

BasicCliModes.GlobalConfigMode.addCommandClass( QosCoppClassMapModeNoOrDefaultCmd )

#--------------------------------------------------------------------------------
# show class-map type control-plane [ CMAP ]
#--------------------------------------------------------------------------------
class ClassMapTypeControlPlaneCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show class-map type control-plane [ CMAP ]'
   data = {
      'class-map': CliCommand.guardedKeyword( 'class-map',
         "Show Class Map", guard=QosCli.guardClassMap ),
      'type': matcherType,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         "Control-plane type", guard=guardPMapCopp ),
      'CMAP': CliMatcher.DynamicNameMatcher( getCMapNameRuleCopp,
         'Class Map Name' ),
   }

   handler = ClassMapModeCopp.showClassMap
   cliModel = ClassMapAllModel

BasicCli.addShowCommandClass( ClassMapTypeControlPlaneCmd )

#------------------------------------------------------------------------
# Match and set value binding rules ( class-map )
#------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] match ip access-group GROUP
#--------------------------------------------------------------------------------
class MatchIpAccessGroupCoppCmd( CliCommand.CliCommandClass ):
   syntax = 'match ip access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Match the access rule specified',
      'ip': CliCommand.guardedKeyword( 'ip', "Specify Ip Access-groups",
         guard=QosCli.guardIpAcls ),
      'access-group': 'Match with given access group',
      'GROUP': AclCli.standardIpAclNameMatcher, 
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( noOrDefault, 'ip', args[ 'GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModeCopp.addCommandClass( MatchIpAccessGroupCoppCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match ipv6 access-group GROUP
#--------------------------------------------------------------------------------
class MatchIpv6AccessGroupCoppCmd( CliCommand.CliCommandClass ):
   syntax = 'match ipv6 access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': 'Match the access rule specified',
      'ipv6': CliCommand.guardedKeyword( 'ipv6', "Specify Ipv6 Access-groups",
         guard=QosCli.guardIp6Acls ),
      'access-group': 'Match with given access group',
      'GROUP': AclCli.standardIp6AclNameMatcher, 
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( noOrDefault, 'ipv6', args[ 'GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModeCopp.addCommandClass( MatchIpv6AccessGroupCoppCmd )

#-------------------------------------------------------------------------------
# policy-map class mode
#-------------------------------------------------------------------------------
class PolicyMapClassContextCopp( PolicyMapClassContext ):
   def newEditEntry( self, cmapName, # pylint: disable-msg=arguments-differ
                     bypassCheck=False ):
      newClassAction = Tac.newInstance( 'Qos::ClassAction', cmapName )
      if bypassCheck or qosHwStatus.coppActionShaperSupported:
         newClassAction.policyAction.newMember( tacActionType.actionSetShape )
         newClassAction.policyAction[ tacActionType.actionSetShape ].rate = ()
         newClassAction.policyAction[ tacActionType.actionSetShape ].rate.val = \
             tacActionRateType.noValue
         newClassAction.policyAction.newMember( tacActionType.actionSetBandwidth )
         newClassAction.policyAction[ tacActionType.actionSetBandwidth ].rate = ()
         newClassAction.policyAction[ tacActionType.actionSetBandwidth ].rate.val = \
             tacActionRateType.noValue
      self.currentEntry_ = newClassAction

   def commit( self ):
      assert self.mapType_ == coppMapType
      cmapName = self.cmapName_

      def commitStaticORDefaultClass( cmapName ):
         clAction = None
         cpType = classMapCpType( cliQosAclConfig, cmapName )
         if cpType == tacClassMapCpType.cmapCpStatic:
            # since statiClass is being configured, no reordering.
            # Update classAction and return.
            if QosLib.isDefaultClass( self.mapType_, cmapName ):
               clAction = self.map_.classActionDefault
            else:
               if cmapName in self.map_.classAction:
                  clAction = self.map_.classAction[ cmapName ]
               else:
                  coppPMapStaticClassIs( self.map_, cmapName, coppMapType )
                  clAction = self.map_.classAction[ cmapName ]

         if clAction is not None:
            copyClassAction( self.currentEntry_, clAction )
            return True

         return False

      # For copp static / default class no re-ordering
      if not commitStaticORDefaultClass( cmapName ):
         self._commit()

class PolicyMapClassModeCopp( PolicyMapClassMode ):
   name = "Policy Map Copp Class Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def obtainShapeOrBandwidthValue( self, no, rateUnit, value ):
      cpType = classMapCpType( cliQosAclConfig, self.cmapName )
      if cpType == tacClassMapCpType.cmapCpStatic:
         if no == 'default':
            value = tacActionRateType.invalid
            rateUnit = tacRateUnit.rateUnitInvalid
         elif no:
            value = tacActionRateType.noValue
            rateUnit = tacRateUnit.rateUnitInvalid
      else:
         if no:
            value = tacActionRateType.noValue
            rateUnit = tacRateUnit.rateUnitInvalid
      return ( value, rateUnit )

   def configureShape( self, no, rateUnit, value ):
      context = self.pmapClassContext
      classAction = context.currentEntry()

      if tacActionType.actionSetShape not in classAction.policyAction:
         context.newEditEntry( self.cmapName, bypassCheck=True )
         classAction = context.currentEntry()

      def currentShape():
         rate = classAction.policyAction[ tacActionType.actionSetShape ].rate
         return rate.val

      def defaultShape():
         cpType = classMapCpType( cliQosAclConfig, self.cmapName )
         if cpType == tacClassMapCpType.cmapCpStatic:
            cpStaticType = classMapCpStaticType( cliQosAclConfig, self.cmapName )
            hwCoppStaticClass = coppStaticClassFromHwStatus( qosSliceHwStatus )
            return hwCoppStaticClass[ cpStaticType ].defaultMax
         return None

      ( value, rateUnit ) = self.obtainShapeOrBandwidthValue( no, rateUnit, value )
      rate = classAction.policyAction[ tacActionType.actionSetShape ].rate
      if value != None:
         rate.val = value
      if rateUnit != None:
         rate.rateUnit = rateUnit

      # For a static class, if parameters have been restored to default,
      # then we should write invalid values in config. But if it is startup
      # we don't have default min and max and we skip this check.
      if not self.session_.startupConfig():
         if not qosHwStatus.intfCoppSupported:
            value = defaultShape()
            if value != None and value == currentShape():
               rate = classAction.policyAction[ tacActionType.actionSetShape ].rate
               rate.val = tacActionRateType.invalid
               rate.rateUnit = tacRateUnit.rateUnitInvalid

   def configureBandwidth( self, no, rateUnit, value ):
      context = self.pmapClassContext
      classAction = context.currentEntry()

      if tacActionType.actionSetBandwidth not in classAction.policyAction:
         context.newEditEntry( self.cmapName, bypassCheck=True )
         classAction = context.currentEntry()

      def currentBandwidth():
         rate = classAction.policyAction[ tacActionType.actionSetBandwidth ].rate
         return rate.val

      def defaultBandwidth():
         cpType = classMapCpType( cliQosAclConfig, self.cmapName )
         if cpType == tacClassMapCpType.cmapCpStatic:
            cpStaticType = classMapCpStaticType( cliQosAclConfig, self.cmapName )
            hwCoppStaticClass = coppStaticClassFromHwStatus( qosSliceHwStatus )
            return hwCoppStaticClass[ cpStaticType ].defaultMin
         return None

      ( value, rateUnit ) = self.obtainShapeOrBandwidthValue( no, rateUnit, value )
      rate = classAction.policyAction[ tacActionType.actionSetBandwidth ].rate
      if value != None:
         rate.val = value
      if rateUnit != None:
         rate.rateUnit = rateUnit

      if not self.session_.startupConfig():
         if not qosHwStatus.intfCoppSupported:
            value = defaultBandwidth()
            if value != None and value == currentBandwidth():
               rate = \
                   classAction.policyAction[ tacActionType.actionSetBandwidth ].rate
               rate.val = tacActionRateType.invalid
               rate.rateUnit = tacRateUnit.rateUnitInvalid

class PolicyMapModeCopp( PolicyMapMode ):
   name = "Copp Policy Map Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def getChildMode( self, mapType, context ):
      childMode = self.childMode( PolicyMapClassModeCopp,
                                  context=context )
      return childMode

   def onExit( self ):
      t0( 'PolicyMapModeCopp onExit...' )
      self.commitContext()
      if self.session_.startupConfig() and self.pmapName == tacPMapNm.coppName \
            and needToConvertOldCoppSystemPolicy:
         convertCoppSystemPolicyForIntfCopp( self )
      BasicCli.ConfigModeBase.onExit( self )

# Goto 'policy-map' config mode. Create a policy map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.
def gotoCoppPolicyMapMode( mode, args ):
   pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' ) 
   shared = args.get( 'shared' ) 
   t0( 'gotoCoppPolicyMapMode of policy', pmapName )
   mapType = 'control-plane'
   # Creating copp pmap apart from copp-system-policy is only
   # supported on platforms where qosHwStatus.intfCoppSupported is True
   if pmapName != tacPMapNm.coppName and \
         not qosHwStatus.intfCoppSupported and \
         not mode.session_.startupConfig():
      mode.addError( "Not Supported on this hardware platform" )
      return
   cmapName = tacCMapNm.coppDefault
   context = gotoPolicyMapModeCommon( mode, pmapName, mapType, cmapName, shared )
   if context:
      mode.pmapContext = context
      childMode = mode.childMode( PolicyMapModeCopp, context=context )
      mode.session_.gotoChildMode( childMode )

def deleteCoppPolicyMap( mode, args ):
   pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' ) 
   shared = args.get( 'shared' )
   mapType = 'control-plane'
   deletePolicyMap( mode, pmapName, mapType, shared )

#------------------------------------------------------------------------
# Register top-level CLI commands ( policy-map )
#------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] policy-map type ( copp | control-plane ) 
#       ( copp-system-policy | PMAP )
#--------------------------------------------------------------------------------
class PolicyMapTypeCoppPmapCmd( CliCommand.CliCommandClass ):
   syntax = ( 'policy-map type ( copp | control-plane ) ' 
              '( copp-system-policy | PMAP )' )
   noOrDefaultSyntax = syntax
   data = {
      'policy-map': nodePolicyMap,
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': nodeCoppDeprecated,
      'copp-system-policy': nodeCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   autoConfigSessionAllowed = False
   handler = gotoCoppPolicyMapMode
   noOrDefaultHandler = deleteCoppPolicyMap

BasicCliModes.GlobalConfigMode.addCommandClass( PolicyMapTypeCoppPmapCmd )

#------------------------------------------------------------------------
# show policy-map type copp [ ( copp-system-policy | PMAP ) ]
#
# legacy:
#     show policy-map type control-plane [ ( copp-system-policy | PMAP ) ]
#------------------------------------------------------------------------
class PolicyMapTypeCoppCmd( ShowCommand.ShowCliCommandClass ):
   syntax =  ( 'show policy-map type ( copp | control-plane )'
               ' [ ( copp-system-policy | PMAP ) ]' )
   data = {
      'policy-map': nodePolicyMapShow,
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': nodeControlPlaneDeprecated,
      'copp-system-policy': matcherCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   handler =  PolicyMapModeCopp.showPolicyMap
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapTypeCoppCmd )

#------------------------------------------------------------------------
# show policy-map type copp ( copp-system-policy | PMAP ) class CMAP
#
# legacy:
#     show policy-map type control-plane 
#       ( copp-system-policy | PMAP ) class CMAP
#------------------------------------------------------------------------
class PolicyMapTypeCoppClassCmapnameCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map type ( copp | control-plane )'
              ' ( copp-system-policy | PMAP ) class CMAP' )
   data = {
      'policy-map': nodePolicyMapShow,
      'type': matcherType,
      'copp': nodeCopp,
      'control-plane': nodeControlPlaneDeprecated,
      'copp-system-policy': matcherCoppSystemPolicy,
      'PMAP': matcherPmapName,
      'class': matcherClass,
      'CMAP': matcherClassName,
   }

   handler = PolicyMapClassModeCopp.showPolicyMapClass
   cliModel = ClassMapWrapper

BasicCli.addShowCommandClass( PolicyMapTypeCoppClassCmapnameCmd )

#-----------------------------------------------------------------
# The show command in 'config-pmap' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------
class PmapModeShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show [ pending | active | current | diff ]'
   data = {
      'pending': 'Display the new policy-map configuration to be applied',
      'active': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'current': ( 'Display the policy-map configuration '
          'in the current running-config' ),
      'diff': ( 'Display the diff between policy-map configuration'
          'current running-config and to be applied' ),
   }

   @staticmethod
   def handler( mode, args ):
      if 'diff' in args:
         QosCli.showPMapDiff( mode )
      elif 'active' in args or 'current' in args:
         QosCli.showPMapActive( mode )
      else:
         QosCli.showPMapCurrent( mode )

PolicyMapModeCopp.addShowCommandClass( PmapModeShowCmd )

#------------------------------------------------------------------------
# Match and set value binding rules ( policy-map )
#------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] class CMAP [ insert-before CLASS ]
#--------------------------------------------------------------------------------
class InsertBeforeClassCmd( CliCommand.CliCommandClass ):
   syntax = 'class CMAP [ insert-before CLASS ]'
   noOrDefaultSyntax = syntax
   data = {
      'insert-before': CliCommand.guardedKeyword( 'insert-before',
         "insert the class with a higher priority than a given class",
         guard=guardClassMap ),
      'class': matcherClass,
      'CMAP': matcherClassName,
      'CLASS': matcherClassName,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setClass( False, args[ 'CMAP' ], args.get( 'CLASS' ) )

   @staticmethod
   def noHandler( mode, args ):
      mode.setClass( True, args[ 'CMAP' ], args.get( 'CLASS' ) )

   @staticmethod
   def defaultHandler( mode, args ):
      mode.setClass( False, args[ 'CMAP' ],
            args.get( 'CLASS' ), default=True )

PolicyMapModeCopp.addCommandClass( InsertBeforeClassCmd )

#--------------------------------------------------------------------------------
# abort commands ( class-map )
# abort commands ( policy-map )
# abort commands ( policy-map-class )
# abort commands ( PolicyMapClassModeQos )
# abort commands ( QosProfileMode )
# abort commands ( ClassMapModeQos )
# abort commands ( PolicyMapModeQos )
#--------------------------------------------------------------------------------
class AbortCmd( CliCommand.CliCommandClass ):
   syntax = 'abort'
   data = {
      'abort': 'Abandon all changes',
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

ClassMapModeCopp.addCommandClass( AbortCmd )
PolicyMapModeCopp.addCommandClass( AbortCmd )
PolicyMapClassModeCopp.addCommandClass( AbortCmd )
QosCli.PolicyMapClassModeQos.addCommandClass( AbortCmd )
QosCli.QosProfileMode.addCommandClass( AbortCmd )
QosCli.ClassMapModeQos.addCommandClass( AbortCmd )
QosCli.PolicyMapModeQos.addCommandClass( AbortCmd )

#------------------------------------------------------------------------
# Set commands ( policy-map )
#------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] shape ( kbps | pps ) RATE
#--------------------------------------------------------------------------------
class ShapeCmd( CliCommand.CliCommandClass ):
   syntax = 'shape ( kbps | pps ) RATE'
   noOrDefaultSyntax = 'shape ...'
   data = {
      'shape': CliCommand.guardedKeyword( 'shape',
         "Specify maximum rate limit", guard=coppActionShaperSupportedGuard ),
      'kbps': nodeCoppKbps,
      'pps': nodeCoppPps,
      'RATE': CliMatcher.DynamicIntegerMatcher( coppActionRateShapeAndBw,
         helpdesc='Shape value in the rate unit specified' ),
   }

   adapter = rateUnitAdapter

   @staticmethod
   def handler( mode, args ):
      PolicyMapClassModeCopp.configureShape( mode, False, args[ 'rateUnit' ],
            args.get( 'RATE' ) )

   @staticmethod
   def noHandler( mode, args ):
      PolicyMapClassModeCopp.configureShape( mode, True, None, None )

   @staticmethod
   def defaultHandler( mode, args ):
      PolicyMapClassModeCopp.configureShape( mode, 'default', None, None )

PolicyMapClassModeCopp.addCommandClass( ShapeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] bandwidth ( kbps | pps ) BANDWIDTH
#--------------------------------------------------------------------------------
class BandwidthCmd( CliCommand.CliCommandClass ):
   syntax = 'bandwidth ( kbps | pps ) BANDWIDTH'
   noOrDefaultSyntax = 'bandwidth ...'
   data = {
      'bandwidth': CliCommand.guardedKeyword( 'bandwidth',
         "Specify minimum bandwidth", guard=coppActionShaperSupportedGuard ),
      'kbps': nodeCoppKbps,
      'pps': nodeCoppPps,
      'BANDWIDTH': CliMatcher.DynamicIntegerMatcher( coppActionRateShapeAndBw,
         helpdesc='Bandwidth value in the rate unit specified' ),
   }

   adapter = rateUnitAdapter

   @staticmethod
   def handler( mode, args ):
      PolicyMapClassModeCopp.configureBandwidth( mode, False, args[ 'rateUnit' ],
            args.get( 'BANDWIDTH' ) )

   @staticmethod
   def noHandler( mode, args ):
      PolicyMapClassModeCopp.configureBandwidth( mode, True, None, None )

   @staticmethod
   def defaultHandler( mode, args ):
      PolicyMapClassModeCopp.configureBandwidth( mode, 'default', None, None )

PolicyMapClassModeCopp.addCommandClass( BandwidthCmd )

#------------------------------------------------------------------------------
# [ no | default ] police rate RATE_VALUE [ RATE_UNIT ]
#       burst-size BURST_VALUE [ BURST_UNIT ]
#------------------------------------------------------------------------------
class CoppPoliceRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'police rate RATE_VALUE [ RATE_UNIT ]'
              'burst-size BURST_VALUE [ BURST_UNIT ]' )
   noOrDefaultSyntax = 'police ...'
   data = {
      'police': nodeCoppPolice,
      'rate': CliCommand.guardedKeyword( 'rate', "Set lower rate",
         guard=guardIngressPolicing ),
      'RATE_VALUE': matcherPoliceCirValue,
      'RATE_UNIT': RateExpression,
      'burst-size': CliCommand.guardedKeyword( 'burst-size',
         "Set burst-size for lower rate", guard=guardIngressPolicing ),
      'BURST_VALUE': matcherPoliceCommittedBurstValue,
      'BURST_UNIT': BurstExpression,
   }

   @staticmethod
   def handler( mode, args ):
      mode.configurePolicer( mode, no=False, cir=int( args[ 'RATE_VALUE' ] ),
            cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args[ 'BURST_VALUE' ] ),
            bcUnit=args.get( 'BURST_BC_UNIT' ), cmdVersion=2 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.configurePolicer( mode, no=True )

PolicyMapClassModeCopp.addCommandClass( CoppPoliceRateCmd )

#--------------------------------------------------------------------------------
# police cir CIR_VALUE [ RATE_UNIT ] bc BC_VALUE [ BURST_UNIT ]
#--------------------------------------------------------------------------------
class CoppPoliceCirCmd( CliCommand.CliCommandClass ):
   syntax = 'police cir CIR_VALUE [ RATE_UNIT ] bc BC_VALUE [ BURST_UNIT ]'
   data = {
      'police': nodeCoppPolice,
      'cir': CliCommand.guardedKeyword( 'cir', "Set committed information rate",
         guard=guardIngressPolicing ),
      'CIR_VALUE': matcherPoliceCirValue,
      'RATE_UNIT': RateExpression,
      'bc': CliCommand.guardedKeyword( 'bc', "Set committed burst rate",
         guard=guardIngressPolicing ),
      'BC_VALUE': matcherPoliceCommittedBurstValue,
      'BURST_UNIT': BurstExpression,

   }

   @staticmethod
   def handler( mode, args ):
      mode.configurePolicer( mode, no=False, cir=int( args.get( 'CIR_VALUE' ) ),
            cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args.get( 'BC_VALUE' ) ),
            bcUnit=args.get( 'BURST_BC_UNIT' ), cmdVersion=1 )

PolicyMapClassModeCopp.addCommandClass( CoppPoliceCirCmd )

def showPMapInterfaceIntfCopp( mode, args ):
   mapType = args.get( 'control-plane', 'copp' )
   pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
   mapType = 'control-plane' if mapType == 'copp' else mapType
   emapType = mapTypeToEnum( mapType )
   for sliceHwStatus in qosSliceHwStatus.itervalues():
      if emapType not in sliceHwStatus.pmapType:
         return

   intfIdModel = InterfaceIdModel()
   intfs = getIntfsForServicePolicy( emapType, pmapName, tacDirection.input )
   for intf in intfs:
      intfIdModel.append( intf )
   return intfIdModel

def showPMapInterfaceControlPlane( mode, args ):
   mapType = args.get( 'control-plane', 'copp' )
   pmapName = args[ 'copp-system-policy' ]
   policyMapAllModel = PolicyMapAllModel()
   mapType = 'control-plane' if mapType == 'copp' else mapType
   emapType = mapTypeToEnum( mapType )
   for sliceHwStatus in qosSliceHwStatus.itervalues():
      if emapType not in sliceHwStatus.pmapType:
         return

   direction = tacDirection.input
   policyMapsContainer = PMapModelContainer( qosConfig, cliQosAclConfig, qosStatus,
                                qosHwStatus, qosSliceHwStatus, qosAclHwStatus,
                                qosAclSliceHwStatus, emapType, direction,
                                hwEpochStatus, None,
                                policyMapAllModel )
   policyMapsContainer.populatePolicyMapInterface( pmapName )
   return policyMapAllModel


def showPMapInterfaceTypeIntfCopp( mode, args ):
   mapType = args.get( 'control-plane', 'copp' )
   mapType = 'control-plane' if mapType == 'copp' else mapType
   return showPMapInterface( mode, args ) 

#--------------------------------------------------------------------------------
# show policy-map ( copp | ( interface control-plane ) ) copp-system-policy
#--------------------------------------------------------------------------------
class PolicyMapCoppCoppSystemPolicyShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map ( copp | ( interface control-plane ) ) ' 
              'copp-system-policy' )
   data = {
      'policy-map': nodePolicyMapShow,
      'copp': nodeCopp,
      'interface': 'Service Policy on interface',
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         helpdesc='Control-plane type',  guard=guardPMapCopp,
         deprecatedByCmd="show policy-map copp [ Policy Map Name ]" ),
      'copp-system-policy': nodeCoppSystemPolicy,
   }

   handler = showPMapInterfaceControlPlane
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapCoppCoppSystemPolicyShowCmd )

#--------------------------------------------------------------------------------
# show policy-map ( copp | ( interface control-plane ) )
#       ( copp-system-policy | PMAP )
#--------------------------------------------------------------------------------
class PolicyMapCoppShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map ( copp | ( interface control-plane ) ) '
              '( copp-system-policy | PMAP )' )
   data = {
      'policy-map': nodePolicyMapShow,
      'copp': nodeCoppIntf,
      'interface': 'Service Policy on interface',
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         helpdesc='Control-plane type', guard=guardIntfCoppSupported,
         deprecatedByCmd='show policy-map copp [ Policy Map Name ]' ),
      'copp-system-policy': nodeCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   handler = showPMapInterfaceIntfCopp
   cliModel = InterfaceIdModel

BasicCli.addShowCommandClass( PolicyMapCoppShowCmd )

#--------------------------------------------------------------------------------
# show policy-map interface INTF type control-plane
#--------------------------------------------------------------------------------
class PolicyMapInterfaceIntfTypeControlPlaneCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policy-map interface INTF type control-plane'
   data = {
      'policy-map': nodePolicyMapShow,
      'interface': 'Service Policy on interface',
      'INTF': IntfRangeMatcher( explicitIntfTypes=(
         EthIntfCli.EthPhyAutoIntfType, LagIntfCli.LagAutoIntfType ) ),
      'type': matcherType,
      'control-plane': CliCommand.guardedKeyword( 'control-plane',
         helpdesc='Control-plane type', guard=guardIntfCoppSupported ),
   }

   handler = showPMapInterfaceTypeIntfCopp
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapInterfaceIntfTypeControlPlaneCmd )

#--------------------------------------------------------------------------------
# show policy-map type ( copp | control-plane ) 
#       [ ( copp-system-policy | PMAP ) ] counters
#--------------------------------------------------------------------------------
class PolicyMapTypeCoppCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show policy-map type ( copp | control-plane )'
              ' [ ( copp-system-policy | PMAP ) ] counters' )
   data = {
      'policy-map': nodePolicyMapShow,
      'type': matcherType,
      'copp': nodeCoppIntf,
      'control-plane': CliCommand.guardedKeyword( 'control-plane', 
         "Control-plane type", guard=guardIntfCoppSupported, 
         deprecatedByCmd='show policy-map type copp [ Policy Map Name ]' ),
      'copp-system-policy': matcherCoppSystemPolicy,
      'PMAP': matcherPmapName,
      'counters': 'Policy Map counters',
   }

   handler = QosCli.showPolicyMapCounters
   cliModel = PolicyMapAllModel

BasicCli.addShowCommandClass( PolicyMapTypeCoppCountersCmd )

#--------------------------------------------------------------------------------
# [ no | default ] service-policy input copp-system-policy
#--------------------------------------------------------------------------------
def _configureCpServicePolicy( mode, args ):
   coppServicePolicyIs( qosInputConfig )

def _configureNoCpServicePolicy( mode, args ):
   coppServicePolicyIs( qosInputConfig, no=True )
   # This command is not allowed in per port copp. If it is saved in the startup
   # config, it means copp-system-policy was detached in the config before
   # per port copp, so no need to convert the old copp-system-policy
   if mode.session.startupConfig():
      global needToConvertOldCoppSystemPolicy
      needToConvertOldCoppSystemPolicy = False

def _configureDefaultCpServicePolicy( mode, args ):
   if hwEpochStatus.globalProtectionModeEnabled:
      _configureCpServicePolicy( mode, args )
   else:
      _configureNoCpServicePolicy( mode, args )

class ServicePolicyInputCoppSystemPolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy input copp-system-policy'
   noOrDefaultSyntax = syntax
   data = {
      'service-policy': CliCommand.guardedKeyword( 'service-policy', 
         "Configure QoS Control-plane Service Policy", 
         guard=guardServicePolicyCopp ),
      'input': 'Apply the policy map to ingress packets',
      'copp-system-policy': nodeCoppSystemPolicy,
   }

   handler = _configureCpServicePolicy
   noHandler = _configureNoCpServicePolicy
   defaultHandler = _configureDefaultCpServicePolicy

AclCli.CpConfigMode.addCommandClass( ServicePolicyInputCoppSystemPolicyCmd )

def setCoppServicePolicy( mode, noOrDefaultKw=None, pmapName=None ):
   if noOrDefaultKw is None:
      intfName = mode.intf.name
      if isLagPort( intfName ):
         mode.addError( CliError[ 'lagPolicingNotSupported' ] )
         return
   setServicePolicy( mode, noOrDefaultKw=noOrDefaultKw, mapType='control-plane',
         pmapName=pmapName, direction='input' )

#--------------------------------------------------------------------------------
# [ no | default ] service-policy type control-plane ( copp-system-policy | PMAP )
#--------------------------------------------------------------------------------
class ServicePolicyTypeControlPlaneCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy type control-plane ( copp-system-policy | PMAP )'
   noOrDefaultSyntax = ( 'service-policy type control-plane ' 
                         '[ copp-system-policy | PMAP ]' )
   data = {
      'service-policy': CliCommand.guardedKeyword( 'service-policy', 
         "Service policy configuration", guard=QosCli.guardIntfModePolicy ),
      'type': matcherType,
      'control-plane': CliCommand.guardedKeyword( 'control-plane', 
         "Control-plane type", guard=guardIntfCoppSupported ),
      'copp-system-policy': nodeCoppSystemPolicy,
      'PMAP': matcherPmapName,
   }

   @staticmethod
   def handler( mode, args ):
      pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
      setCoppServicePolicy( mode, None, pmapName )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
      setCoppServicePolicy( mode, True, pmapName )

QosModelet.addCommandClass( ServicePolicyTypeControlPlaneCmd )

#------------------------------------------------------------------------------
# configure convert copp-system-policy
#-----------------------------------------------------------------------------
# This will read copp-system-policy from a non-per-port-copp config
# It then creates a new policy map with the same class maps and attaches
# the new policy map to all ports.

def assignClassActionCoppPolicer( classAction, className, cir, cirUnit, bc, bcUnit ):
   # pir and be are 0 because they are not used for copp policers
   classAction.policer = ( className, cir, bc, 0, 0 )
   classAction.policer.cirUnit = cirUnit
   classAction.policer.bcUnit = bcUnit
   action = tacActionType.actionSetDrop
   classAction.policer.redActions.newMember( action )
   classAction.policer.redActions[ action ].value = 1

def convertCoppSystemPolicyForIntfCopp( mode ):
   '''
   The global copp-system-policy is no longer configurable in per-port-copp.
   If the startup config contains configuration for copp-system-policy from a
   pre-per-port-copp image, we will copy the config to new a new per port policy
   map and apply the new policy map on all interfaces listed in the startup config
   '''
   t0('beginning copp-system-policy config convert')
   newPmapName = tacPMapNm.convertedPmapName
   coppSystemPolicy = \
      cliQosAclConfig.pmapType[ 'mapControlPlane' ].pmap[ 'copp-system-policy' ]
   newPmap = cliQosAclConfig.pmapType[ 'mapControlPlane' ].pmap.newMember(
      newPmapName )
   copier = Tac.newInstance( 'Cli::Session::EntityCopy' )
   copier.handler = \
      Tac.newInstance( "Cli::Session::EntityCopyGlobalSessionHandler" )
   copier.handler.sessionStatus = mode.session.cliSessionStatus
   copier.doCopyEntityAttrs( newPmap, coppSystemPolicy, '', None )
   t8( 'entities copied, about to apply to all intfs' )
   classesToDelete = []
   # Delete classes without a user configured rate. These will use the default
   # rates, and we do not want the cli to show "None" for the values.
   for classActionName, classAction in newPmap.classAction.iteritems():
      shapeRateVal = \
         classAction.policyAction[ tacActionType.actionSetShape ].rate.val
      bandwidthRateVal = \
         classAction.policyAction[ tacActionType.actionSetBandwidth ].rate.val
      if ( ( shapeRateVal == tacActionRateType.noValue or
             shapeRateVal == tacActionRateType.invalid ) and
           ( bandwidthRateVal == tacActionRateType.noValue or
             bandwidthRateVal == tacActionRateType.invalid ) ):
         classesToDelete.append( classActionName )
   for cmapName in classesToDelete:
      t8( "Deleting class ", cmapName )
      del newPmap.classAction[ cmapName ]
      if classMapCpType( cliQosAclConfig, cmapName ) == \
         tacClassMapCpType.cmapCpDynamic:
         deleteClassPrio( newPmap.classPrio, cmapName )
      else:
         deleteClassPrio( newPmap.coppStaticClassPrio, cmapName )

   # Delete shape and bandwidth from copp-system-default if they're empty. Same
   # idea as above, but copp-system-default is not in the classAction collection
   defaultActionSetShape = \
      newPmap.classActionDefault.policyAction[ 'actionSetShape' ]
   defaultActionSetBandwidth = \
      newPmap.classActionDefault.policyAction[ 'actionSetBandwidth' ]
   if ( ( defaultActionSetShape.rate.val == tacActionRateType.noValue or
          defaultActionSetShape.rate.val == tacActionRateType.invalid ) and
        ( defaultActionSetBandwidth.rate.val == tacActionRateType.noValue or
          defaultActionSetBandwidth.rate.val == tacActionRateType.invalid ) ):
      del newPmap.classActionDefault.policyAction[ 'actionSetShape' ]
      del newPmap.classActionDefault.policyAction[ 'actionSetBandwidth' ]
   else:
      # popluate policer for copp-system-default
      if newPmap.classActionDefault.policer is None:
         cir, cirUnit, bc, bcUnit = convertShapeValueToPoliceValue(
            defaultActionSetShape )
         t0( 'convert shape: %d to policer ( %d, %d ) for %s' %
             ( defaultActionSetShape.rate.val, cir, bc, tacCMapNm.coppDefault ) )
         assignClassActionCoppPolicer( newPmap.classActionDefault,
                                       tacCMapNm.coppDefault,
                                       cir, cirUnit, bc, bcUnit )

   # For each classAction in the newPmap, popluate policer
   # action. Forwarding agent need to choose either shaper or policer
   # as class action. At the time of parsing startup_config, some
   # flags (like coppActionPolicerSupported ) has not yet been setup.
   t0( 'populate policer actions' )
   for classActionName, classAction in newPmap.classAction.iteritems():
      t8( 'populate policer actions for %s' % classActionName )
      if tacActionType.actionSetShape not in classAction.policyAction:
         continue
      if classAction.policer is not None:
         continue
      shape = classAction.policyAction[ tacActionType.actionSetShape ]
      cir, cirUnit, bc, bcUnit = convertShapeValueToPoliceValue( shape )
      t0( 'convert shape: %d to policer ( %d, %d ) for %s' %
          ( shape.rate.val, cir, bc, classActionName ) )
      assignClassActionCoppPolicer( classAction, classActionName,
                                    cir, cirUnit, bc, bcUnit )

   intfs = []
   # Apply the new policy map to all interfaces. Because this is being run in
   # startup config, IntfCli.Intf.getAll is not yet populated, so we look at
   # intf config in ethPhySliceDir which is created earlier in startup config
   for sliceNum, sliceDir in intfConfigEthPhySliceDir.iteritems():
      t8( 'sliceNum is ', sliceNum)
      for intfName in sliceDir.intfConfig:
         t8( 'intf is ', intfName )
         if intfName.startswith( 'Ethernet' ):
            newIntf = EthIntfCli.EthPhyIntf( intfName, mode )
            intfs.append( newIntf )
   t8( 'intfs is ', intfs )
   for intf in intfs:
      intfConfigMode = mode.childMode( IntfCli.IntfConfigMode, intf=intf )
      setServicePolicy( intfConfigMode, mapType='control-plane',
                        pmapName=newPmapName, direction='input' )

# Setup hooks in QosCli.py
# pylint: disable-msg=protected-access
QosCli._guardPolicyMapCallback[ coppMapType ] = guardPMapCopp
QosCli._guardClassMapCallback[ coppMapType ] = guardCMapCopp
QosCli._guardIntfModePolicyCallback[ coppMapType ] = guardIntfModePolicyCopp
QosCli._policyMapClassContextType[ coppMapType ] = PolicyMapClassContextCopp
# pylint: enable-msg=protected-access

pluginGlobals = {}
for _name in QosCli.Plugin.__code__.co_names:
   if _name in globals() and globals()[ _name ] is None:
      pluginGlobals[ _name ] = globals()[ _name ]

@Plugins.plugin( requires=( "QosCli", ) )
def Plugin( entityManager ): # pylint: disable-msg=function-redefined
   # Update this module's copy of global variables which got updated in QosCli
   # module after its Plugin function was run
   for name in pluginGlobals:
      globals()[ name ] = getattr( QosCli, name )
