# Copyright (c) 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import sys

import AclCliLib
import AclLib
import Arnet
import BasicCli
import BasicCliUtil
import Cell
import CliCommand
import CliExtensions
import CliMatcher
import CliParser
import CliParserCommon
import CliPlugin.ConfigConvert
import CliPlugin.TechSupportCli
import CliRangeExpansion
import CliSession
import ConfigMount
import FileUrl
import LazyMount
import MultiRangeRule
import Plugins
import QosLib
import SharedMem
import ShowCommand
import Smash
import Tac
import Tracing
import Url
from CliMode.Fabric import FabricMode
from CliMode.Qos import ( ClassMapModeBase, CosToTcProfileModeBase,
                         DscpRewriteModeBase, IntfTxQueueMode,
                         PolicyMapClassModeBase, PolicyMapModeBase,
                         QosProfileModeBase, QosSchedulingIntfGroupModeBase,
                         QosSchedulingIntfModeBase, QosSchedulingModeBase,
                         QosSchedulingPolicyModeBase, TcToCosModeBase,
                         DscpToTcModeBase )
from CliPlugin.VirtualIntfRule import IntfMatcher
from Intf.IntfRange import IntfRangeMatcher
from PfcTypes import tacPfcWatchdogAction
from QosLib import *  # pylint: disable-msg=W0401
from QosTypes import ( tacActionRateType, tacActionType, tacBurstUnit,
                      tacClassMapCpType, tacCos, tacCosToTcProfileName,
                      tacDropPrecedence, tacDscp, tacDscpRewriteMapName,
                      tacEcnDelayThreshold, tacEcnDelayThresholdUnit, tacExp,
                      tacGuaranteedBwUnit, tacGuaranteedBwVal, tacMatchOption,
                      tacPercent, tacPriorityGroup, tacQueueThresholdUnit,
                      tacQueueType, tacRateUnit, tacRewriteEnableMode,
                      tacShapeRateUnit, tacShapeRateVal, tacTcDpPair,
                      tacTcToCosMapName, tacTrafficClass, tacTrustMode,
                      tacTxQueueId, tacTxQueuePriority )
from Toggles.QosToggleLib import toggleEcnCountersSmashEnabled

import AclCli
import EthIntfCli
import FabricIntfCli
import FileCli
import IntfCli
import LagIntfCli
import SubIntfCli
import VlanIntfCli
import VxlanCli
from IntfCli import IntfConfigMode, counterSupportedIntfs
from IntfRangeCli import IntfRangeConfigMode
from QosCliLib import ( CMapModelContainer, PMapModelContainer,
                        QosProfileModelContainer, populateBurstSizeModel,
                        populateShapeRateModel, deleteInterfacePolicingConfig )
from QosCliModel import *  # pylint: disable-msg=W0401

__defaultTraceHandle__ = Tracing.Handle( 'QosCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t8 = Tracing.trace8

DYNAMIC_PMAP_EDIT_WARN = 'Editing dynamically created ' \
                           '(by agents and not CLI) policy map'
DYNAMIC_CMAP_EDIT_WARN = 'Editing dynamically created ' \
                           '(by agents and not CLI) class map'
CLI_TIMEOUT = 'Operation timed out'

#------------------------------------------------------------------------------------
# The Qos globals
#------------------------------------------------------------------------------------
aclConfig = None # standard acl list
aclStatus = None
cliCounterConfig = None
cliQosAclConfig = None
cliQosAclConfigReadOnly = None
defaultPdpPmapCfgReadOnly = None
entMan = None
ecnCounterTable = None
ecnSnapshotTable = None
fabricIntfConfigDir = None
hwEpochStatus = None
intfConfigEthPhySliceDir = None
interfaceHwCapability = None
lagInputConfig = None
pfcStatus = None
profileConfigDir = None
qosAclConfig = None
qosAclConfigDir = None
qosAclConfigMergeSm = None
qosAclHwStatus = None
qosAclSliceHwStatus = None
qosConfig = None
qosHwConfig = None
qosHwCounter = None
qosHwStatus = None
qosTmMapping = None
qosInputConfig = None
qosInputProfileConfig = None
qosSliceHwStatus = None
qosStatus = None
subIntfHwStatus = None
qosSchedulerConfig = None
qosGlobalConfig = None

# Using BITMAPs to state which feature configuration has changed
MODIFY_DEF_COS = 1
MODIFY_DEF_DSCP = 2
MODIFY_TRUST_MODE = 4
MODIFY_SHAPE_RATE = 8
MODIFY_PFC = 16
MODIFY_TXQCONFIG = 32
MODIFY_ECN_TXQCONFIG = 64
MODIFY_ECN_DELAY_TXQCONFIG = 128
MODIFY_WRED_TXQCONFIG = 256
MODIFY_NONECT_TXQCONFIG = 512
MODIFY_DROP_THRESHOLDS = 1024
MODIFY_LATENCY_THRESHOLD = 2048
MODIFY_GUARANTEED_BW = 4096
MODIFY_ALL = 8191

ROLLBACK_PROFILE = 1

tempShapeRate = Tac.Value( 'Qos::ShapeRate' )
invalidShapeRate = Tac.Value( 'Qos::ShapeRate' )

tempGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
invalidGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )

invalidDscp = Tac.Value( "Qos::DscpVal" ).invalid
ecnDontCare = Tac.Type( "Acl::Ecn" ).dontCare

FapId = Tac.Type( 'FlexCounters::FapId' )
EcnCounterKey = Tac.Type( 'Qos::EcnCounterKey' )

# Ecn drop probability
ecnMaxDroprate = 100

# Ecn default weight
defaultWeight = 0

showWredIntfQueueCountersHook = CliExtensions.CliHook()
#
# Filters counter-supporting ethernet interfaces from a given interface range.
#
def counterSupportedEthIntfs( mode, intf, mod ):
   intfs = counterSupportedIntfs( mode, intf, mod, EthIntfCli.EthIntf )

   return intfs if intfs else []

#
# Filters counter-supporting physical ethernet interfaces from a given interface 
# range.
#
def counterSupportedEthPhyIntfs( mode, intf, mod ):
   intfs = counterSupportedIntfs( mode, intf, mod, EthIntfCli.EthPhyIntf )

   return intfs if intfs else []

def isSubIntfConfigMode( mode ):
   if isinstance( mode.parent_, IntfCli.IntfConfigMode ):
      return mode.parent_.intf.isSubIntf()
   elif isinstance( mode.parent_, IntfRangeConfigMode ):
      for intfMode in mode.parent_.individualIntfModes_:
         if intfMode.intf.isSubIntf():
            return True
   return False

class QosGuaranteedBwModuleBase( QosLib.QosModuleBase ):
   def isDefaultIntfConfig( self, intfConfig ):
      cfgGuaranteedBw = intfConfig.guaranteedBw
      return ( ( cfgGuaranteedBw.bw == tacGuaranteedBwVal.invalid ) and
               ( cfgGuaranteedBw.unit == tacGuaranteedBwUnit.guaranteedBwKbps ) )

QosLib.registerQosModule( 'guaranteedBw', QosGuaranteedBwModuleBase( qosHwStatus ) )

#------------------------------------------------------------------------------------
# Lag Helpers
#------------------------------------------------------------------------------------
def getConfiguredLagMembers( lagIntf ):
   if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( lagIntf ):
      lagIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( lagIntf )

   lagMem = []
   for phyIntf, phyIntfVal in lagInputConfig.phyIntf.iteritems():
      if phyIntfVal.lag and phyIntfVal.lag.intfId == lagIntf:
         lagMem.append( phyIntf )
   return lagMem

def getIntfStatus( intf ):
   intfStatus = None
   if isinstance( intf, str ):
      intfName = intf
   else:
      if intf.name.startswith( fabricShortName ):
         intfName = fabricIntfName
      else:
         intfName = intf.name

   if isLagPort( intfName ) :
      lagMem = getConfiguredLagMembers( intfName )
      if lagMem:
         firstMember = lagMem[ 0 ]
         if firstMember and firstMember in qosStatus.intfStatus:
            intfStatus = qosStatus.intfStatus[ firstMember ]
   else:
      if intfName in qosStatus.intfStatus:
         intfStatus = qosStatus.intfStatus[ intfName ]
   return intfStatus

#------------------------------------------------------------------------------------
# Blocking Cli utility functions and vars
#------------------------------------------------------------------------------------

def hwConfigVerificationSupported():
   return ( qosInputConfig.hwConfigVerificationEnabled and 
            qosHwStatus.hwStatusReportingSupported )

def hwConfigAclVerificationSupported():
   return ( qosInputConfig.hwConfigVerificationEnabled and 
            qosAclHwStatus.hwStatusReportingSupported )

# Waits for config to be applied at hardware, polls for ack sent
# by forwarding agents
def cliBlockingToApplyConfigChange( mode, configVersionUpdateTime, description,
                                    policyMap=False ):
   def waitForHwStatusUpdate():
      statusUpdated  = True
      for sliceHwStatus in hwStatusPerSlice.itervalues():
         # Break and return when hwStatusPerSlice.itervalues() is [ None ]
         if not sliceHwStatus:
            statusUpdated = False
            break
         delta = sliceHwStatus.configVersionUpdateTime - \
             qosHwConfig.configVersionUpdateTime
         if abs( delta ) > 0.001:
            statusUpdated = False
            break
      return statusUpdated

   if policyMap:
      hwStatusPerSlice = qosAclSliceHwStatus
      if not hwConfigAclVerificationSupported() or \
             ( mode and mode.session_.startupConfig() ):
         return True, ""
   else:
      hwStatusPerSlice = qosSliceHwStatus
      if not hwConfigVerificationSupported() or \
             ( mode and mode.session_.startupConfig() ):
         return True, ""

   # configVersionUpdateTime is a token that is sent by the CLI and which 
   # is acknowledged by forwarding agent which implies that all prior config
   # updates sent to forwarding were processed and their status are updated
   # appropriately
   qosHwConfig.configVersionUpdateTime = configVersionUpdateTime
   t0( 'Waiting for Version: ', qosHwConfig.configVersionUpdateTime ) 
   result = False
   errMsg = ""
   timeout = int( os.getenv( 'QOS_TEST_CLI_TIMEOUT', QOS_CLI_TIMEOUT ) )
   try:
      Tac.waitFor( waitForHwStatusUpdate,
                   warnAfter=3.0, sleep=True,
                   maxDelay=0.1, timeout=timeout,
                   description=description )
      result = True
   except Tac.Timeout:
      for sliceName, sliceHwStatus in hwStatusPerSlice.iteritems():
         t0( 'Version:', sliceHwStatus.configVersionUpdateTime,
             ' seen at slice: ', sliceName )
      errMsg = "Operation timed out"
      result = False
   except KeyboardInterrupt:
      errMsg = "Keyboard Interrupt"
      result = False

   return result, errMsg

def handleSessionCommit( mode ):

   hwAck, errMsg = cliBlockingToApplyConfigChange( 
      mode, Tac.now(), "Cli Session" )

   if hwAck:      
      result = True
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         if sliceHwStatus.status != tacHwProgrammingStatus.hwPrgmSuccess:
            result = False
            errMsg = "Hardware Programming Failed"
            break
      for aclSliceHwStatus in qosAclSliceHwStatus.itervalues():
         if aclSliceHwStatus.status != tacHwProgrammingStatus.hwPrgmSuccess:
            result = False
            errMsg = "Hardware Programming Failed"
            break
   else:
      result = False

   if not result:
      if errMsg == CLI_TIMEOUT:
         mode.addWarning( QosLib.qosTimeoutWarning() )
      else:
         warning = "Error in QOS commit, configuration may differ " \
             "from hardware (%s)" % errMsg
         mode.addWarning( warning )

def cliBlockingFail( mode, timestamp, feature, description, intf=None ):

   if ( not hwConfigVerificationSupported() or mode.session_.startupConfig() ):
      return False

   if mode.session.inConfigSession():
      CliSession.registerSessionOnCommitHandler( 
         mode.session_.entityManager, "QOS", 
         lambda m, onSessionCommit: handleSessionCommit( m ) )
      qosHwConfig.clearHwStatusErrorRequestTime = Tac.now()
      return False

   errMessage = None
   result, errMsg = cliBlockingToApplyConfigChange( mode, timestamp, description )
   if result:
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         status = sliceHwStatus.hwProgrammingVersionAndStatus.get( feature )
         if( status and
             status.hwProgrammingStatus != tacHwProgrammingStatus.hwPrgmSuccess ):
            errMsg = status.errMessage
            result = False
      for aclSliceHwStatus in qosAclSliceHwStatus.itervalues():
         status = aclSliceHwStatus.hwProgrammingVersionAndStatus.get( feature )
         if( status and
             status.hwProgrammingStatus != tacHwProgrammingStatus.hwPrgmSuccess ):
            errMsg = status.errMessage
            result = False

   if not result:
      if not errMsg:
         errMsg = "Unknown reason"
      elif errMsg == CLI_TIMEOUT:
         mode.addWarning( QosLib.qosTimeoutWarning() )
         return False
      description = ( description + " on %s" % intf ) if intf else description
      errMessage = "Error: Programming " + description + " (%s)" % errMsg
      mode.addError( errMessage )

   return errMessage
 
#-------- Blocking CLI helpers for Policy Qos ----------------------
def getPMapHwStatus( hwStatus, pmapName, mapType, direction ):
   pmapHwStatusKey = Tac.newInstance( "Qos::PolicyMapHwStatusKey", 
                                      direction, pmapName )
   pmapHwStatusTypePtr = hwStatus.pmapType.get( mapType )
   if pmapHwStatusTypePtr is None:
      return None 
   return pmapHwStatusTypePtr.pmap.get( pmapHwStatusKey )

def waitForPMapHwPrgmStatus( mode, pmapName, mapType, direction, 
                             updateRequestTime, updateStr=None,
                             profile=None, intf=None ):
   if not hwConfigAclVerificationSupported():
      return True, ""

   if mode and mode.session.inConfigSession():
      CliSession.registerSessionOnCommitHandler( 
         mode.session_.entityManager, "QOS", 
         lambda m, onSessionCommit: handleSessionCommit( m ) )
      qosHwConfig.clearHwStatusErrorRequestTime = Tac.now()
      return True, ""

   spConfigKey = Tac.newInstance( "Qos::ServicePolicyKey", mapType, 
                                  direction, pmapName )   

   if profile == None:
      if spConfigKey not in qosConfig.servicePolicyConfig:
         return True, ""
      spConfig = qosConfig.servicePolicyConfig[ spConfigKey ]
   else:
      if profile:
         inputConfig = qosInputProfileConfig
      else:
         inputConfig = qosInputConfig
      spConfig = inputConfig.servicePolicyConfig.get( spConfigKey )
      if not spConfig:
         return True, ""


   def getHwPrgmStatus( pmapHwStatusPtr ):
      errMsg = ""
      if pmapHwStatusPtr and \
            pmapHwStatusPtr.prgmStatus == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
         t0( "updateRequestTime : %s " % updateRequestTime, 
               " for : ", spConfigKey ) 
         result = True
      else:
         result = False
         if pmapHwStatusPtr:
            errMsg = pmapHwStatusPtr.prgmStatusErrMessage
         else:
            # May have to fill up appropriately, can happen when the
            # port-channel flaps or any other non-CLI trigger during
            # config update
            errMsg = "Unknown error"
      return result, errMsg

   if mapType != coppMapType:
      # For Qos Service-Policy, there should be atleast one 
      # valid interface for an update to be sent to hw
      # By valid interface here it means it should be either be a:
      # -> physical interfase not part of port-channel
      # -> port-channel which has non zero member interfaces
      sendsHwUpdateForSp = False
      for intfName in spConfig.intfIds:
         if isLagPort( intfName ):
            # for port-channels additionally need to check if 
            # it has any members.
            if getConfiguredLagMembers( intfName ):
               sendsHwUpdateForSp =  True
         else:
            # if interface is member of a port-channel,
            # then don't send hw update for that case. In all other
            # cases send hw update for that service-policy
            sendsHwUpdateForSp = True
            if intfName in lagInputConfig.phyIntf and \
                   lagInputConfig.phyIntf[ intfName ].lag:
               sendsHwUpdateForSp = False

         if sendsHwUpdateForSp:
            # Found atleast one valid interface
            break

      if not sendsHwUpdateForSp:
         t0( "Does not send hw update for ServicePolicy : ", spConfigKey )
         return True, ""

   t0( "Sends hw update for ServicePolicy : ", spConfigKey )

   # wait for hw status and verify if it was successful
   result = False
   errMsg = ""
   description = updateStr + " to be applied"

   result, errMsg = cliBlockingToApplyConfigChange( mode, updateRequestTime, 
                                                    description, policyMap=True )
   bestKey = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
   if result:
      if not qosConfig.servicePolicyConfig.get( bestKey ):
         # If key is not the best policy chosen by QosInputSelectorSm
         # it would not exist in pmapHwStatusTypePtr
         result = True
         bestKey = None
      elif intf:
         if intf not in qosConfig.servicePolicyConfig.get( bestKey ).intfIds:
            result = True
            bestKey = None
            for key in qosConfig.servicePolicyConfig:
               if intf in qosConfig.servicePolicyConfig[ key ].intfIds:
                  bestKey = key
                  break
      if bestKey:
         if mapType == 'mapQos' or mapType == 'mapPdp' or \
            qosAclHwStatus.coppDynamicClassSupported or \
            ( qosHwStatus.intfCoppSupported and
              qosHwStatus.coppActionPolicerSupported ):
            for aclSliceHwStatus in qosAclSliceHwStatus.itervalues():
               pmapHwStatus = getPMapHwStatus( aclSliceHwStatus, bestKey.pmapName,
                                               bestKey.type, bestKey.direction )
               result, errMsg = getHwPrgmStatus( pmapHwStatus )
               if not result:
                  return result, errMsg
         else:
            # On Sand platforms we have a multiple writer issue currently ( SandAcl,
            # SandFap ) which write to prgmStatus, until that is reserved we are
            # adding a stopgap flag qosAclHwStatus.coppHwStatusReportingSupported
            # which is False only on Sand. Note that we still check the
            # configVersionUpdateTime in cliBlockingToApplyConfigChange, which is
            # guarded by qosAclHwStatus.hwStatusReportingSupported. We cannot set
            # hwStatusReportingSupported to False, since that is also used for
            # verifying Acl programming on Sand chips ( which can possibly fail ).
            # XXX_Copp XXX_seerpini
            for sliceHwStatus in qosSliceHwStatus.itervalues():
               if qosAclHwStatus.coppHwStatusReportingSupported:
                  pmapHwStatus = getPMapHwStatus( sliceHwStatus, bestKey.pmapName,
                                                  bestKey.type, bestKey.direction )
                  result, errMsg = getHwPrgmStatus( pmapHwStatus )
                  if not result:
                     return result, errMsg
   return result, errMsg

def spWaitForHwStatusForPmapChange( mode, pmapName, pmapType, updateTime,
                                    updateStr="policy-map change" ):
   if mode.session_.startupConfig():
      # no agent running, don't wait
      return True, ""

   if not hwConfigAclVerificationSupported():
      return True, ""

   if pmapType == coppMapType and not \
      qosAclHwStatus.coppDynamicClassSupported:
      return True,""

   for direction in [ tacDirection.input, tacDirection.output ]:

      # wait for all of the slices to return program status
      rc, errMsg = waitForPMapHwPrgmStatus( mode, pmapName, pmapType, 
                                            direction, updateTime,
                                            updateStr )
      if not rc:
         return rc, errMsg

   return True, ""

def spWaitForHwStatusForCmapChange( mode, cmapName, cmapType, updateTime,
                                    updateStr="class-map change" ):
   if not hwConfigAclVerificationSupported():
      return True, ""

   pmapType = cliQosAclConfig.pmapType.get( cmapType )
   if not pmapType:
      return True, ""
   if cmapType == coppMapType and not \
      qosAclHwStatus.coppDynamicClassSupported:
      return True,""


   for pmap in pmapType.pmap.itervalues():
      pmapAffected = False
      for classPrio in pmap.classPrio.itervalues():
         if classPrio.cmapName == cmapName:
            pmapAffected = True
            break

      if pmapAffected:
         rc, errMsg = spWaitForHwStatusForPmapChange( mode, pmap.name, pmap.type, 
                                                      updateTime, updateStr )
         if not rc:
            return rc, errMsg

   return True, ""

def spWaitForHwStatusForAclChange( mode, aclName, aclType, updateTime,
                                   updateStr="ACL change" ):
   if aclType == 'ip':
      matchOption = tacMatchOption.matchIpAccessGroup
   elif aclType == 'ipv6':
      matchOption = tacMatchOption.matchIpv6AccessGroup
   elif aclType == 'l2Params':
      matchOption = tacMatchOption.matchL2Params
   elif aclType == 'dscpEcn' or aclType == 'dscp' or \
        aclType == 'ecn':
      matchOption = tacMatchOption.matchDscpEcn
   elif aclType == 'mpls-traffic-class':
      matchOption = tacMatchOption.matchMplsTrafficClass
   elif aclType == 'mac':
      matchOption = tacMatchOption.matchMacAccessGroup
   else:
      return True, ""

   for cmapType in cliQosAclConfig.cmapType.itervalues():
      for cmap in cmapType.cmap.itervalues():
         cmapMatch = cmap.match.get( matchOption )
         if cmapMatch and cmapMatch.strValue == aclName:
            rc, errMsg = spWaitForHwStatusForCmapChange( mode, cmap.name, cmap.type,
                                                         updateTime, updateStr )
            if not rc:
               return rc, errMsg

   return True, ""

def isQosPmapHwStatusOk( mode, aclName, aclType, aclUpdateType ):
   err = {}
   err[ 'error' ] = ''
   if not hwConfigAclVerificationSupported():
      t0( 'Not waiting for hardware. All good.' )
      return True, err

   # Bump up the version of the affected pmaps and wait
   # for the hardware status
   if aclUpdateType == AclCli.aclUpdateTypeCreate or  \
         aclUpdateType == AclCli.aclUpdateTypeModify:

      currTime = getCurrentTimeForConfigUpdate()
      rc, errMsg = spWaitForHwStatusForAclChange( mode, aclName, aclType, currTime )
      if not rc: 
         err[ 'error' ] = errMsg
         return rc, err

   return True, err

AclCli.registerAclConfigValidCallback( isQosPmapHwStatusOk )

#------------------------------------------------------------------------------------
# The Qos tokens
#------------------------------------------------------------------------------------
def guardQos( mode, token ):
   if qosHwStatus.hwInitialized:
      return None
   return CliParser.guardNotThisPlatform

_guardPolicyMapCallback = {}
def guardPolicyMap( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      return None
   for cb in _guardPolicyMapCallback.itervalues():
      if cb( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

def guardPMapQos( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapShared( mode, token ):
   if qosAclHwStatus.sharedPolicyMapSupported:
      return None
   return CliParser.guardNotThisPlatform

_guardClassMapCallback = {}
def guardClassMap( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      return None
   for cb in _guardClassMapCallback.itervalues():
      if cb( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

def guardPMapQosAction( mode, token ):
   action = actionToEnum( token.lower() )
   if qosAclHwStatus.policyQosSupported and \
      action in qosAclHwStatus.policyQosActionsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPMapQosActionDropPrecedence( mode, token ):
   if qosAclHwStatus.dropPrecedenceSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpToDropPrecedenceMap( mode, token ):
   if qosHwStatus.dscpToDpMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDropPrecedenceThreshold( mode, token ):
   if qosHwStatus.dropPrecedenceThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardTxQDropPrecedenceThreshold( mode, token ):
   if isSubIntfConfigMode( mode ):
      return CliParser.guardNotThisPlatform
   if qosHwStatus.dropPrecedenceThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPoliceRateInBytes( mode, token ):
   if qosAclHwStatus.policeByteModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPoliceRateInPps( mode, token ):
   if qosAclHwStatus.policePacketModeSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIngressTrTcmPolicing( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressPolicingSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDropPrecedenceYellowAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported and \
      qosAclHwStatus.dropPrecedenceYellowActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpYellowAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported and \
      qosAclHwStatus.dscpYellowActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardPoliceAction( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
      qosAclHwStatus.ingressTrTcmPolicingSupported:
      if qosAclHwStatus.dropPrecedenceYellowActionSupported or \
         qosAclHwStatus.dscpYellowActionSupported:
         return None
   return CliParser.guardNotThisPlatform

def guardIngressPolicing( mode, token ):
   if qosAclHwStatus.policyQosSupported and qosAclHwStatus.ingressPolicingSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDropAction( mode, token ):
   if qosAclHwStatus.dropActionSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIpAcls( mode, token ):
   if qosAclHwStatus.ipAclsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIp6Acls( mode, token ):
   if qosAclHwStatus.ip6AclsSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardGlobalEcn( mode, token ):
   if qosHwStatus.globalEcnSupported and \
      qosHwStatus.ecnSupported:
      return None
   return CliParser.guardNotThisPlatform 

def guardSegment( mode, token ):
   if qosHwStatus.segmentUnitForQueueThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardLatency( mode, token ):
   if qosHwStatus.latencyUnitsForQueueThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardVlanMatch( mode, token ):
   if qosAclHwStatus.vlanMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardInnerVlanMatch( mode, token ):
   if qosAclHwStatus.innerVlanMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardCosMatch( mode, token ):
   if qosAclHwStatus.cosMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpEcnMatch( mode, token ):
   if qosAclHwStatus.dscpEcnMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardMplsTrafficClassMatch( mode, token ):
   if qosAclHwStatus.mplsTrafficClassMatchInClassMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardMacAccessGroupV6Match( mode, token ):
   if qosAclHwStatus.matchMacInClassMapV6Supported:
      return None  
   return CliParser.guardNotThisPlatform

def guardMacAccessGroupMatch( mode, token ):
   if qosAclHwStatus.matchMacInClassMapSupported:
      return None 
   return CliParser.guardNotThisPlatform

def guardIntfModePolicy( mode, token ):
   for cb in _guardIntfModePolicyCallback.itervalues():
      if cb( mode, token ) is None:
         return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyQos( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      if type( mode ) == QosProfileMode:
         return None
      elif mode.intf.isSubIntf():
         if ( qosAclHwStatus.egressSubintfRateLimitSupported
           or qosAclHwStatus.subIntfPolicyQosSupported ):
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.sviPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VxlanCli.VxlanIntf ):
         if qosAclHwStatus.vxlanPolicyQosSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyQosIn( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      if type( mode ) == QosProfileMode:
         return None
      elif mode.intf.isSubIntf():
         if qosAclHwStatus.egressSubintfRateLimitSupported:
            return CliParser.guardNotThisPlatform
         if qosAclHwStatus.subIntfPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VlanIntfCli.VlanIntf ):
         if qosAclHwStatus.sviPolicyQosSupported:
            return None
      elif isinstance( mode.intf, VxlanCli.VxlanIntf ):
         if qosAclHwStatus.vxlanPolicyQosSupported:
            return None
      else:
         return None
   return CliParser.guardNotThisPlatform

def guardIntfModePolicyQosOut( mode, token ):
   if qosAclHwStatus.policyQosSupported:
      if type( mode ) == QosProfileMode:
         return CliParser.guardNotThisPlatform

      if isinstance( mode.intf, VxlanCli.VxlanIntf ):
         if qosAclHwStatus.vxlanPolicyQosSupported:
            return None
      elif mode.intf.name.startswith( 'Ethernet' ) and mode.intf.isSubIntf():
         if qosAclHwStatus.egressSubintfRateLimitSupported:
            return None
   return CliParser.guardNotThisPlatform

def guardPwDecapDscpQosMap( mode, token ):
   if qosHwStatus.pwDecapDscpQosMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpRewritePerInterface( mode, token ):
   if not qosHwStatus.dscpRewritePerInterfaceSupported:
      return CliParser.guardNotThisPlatform

   if mode.intf.isSubIntf() and \
         not qosHwStatus.dscpRewritePerSubInterfaceSupported:
      return CliParser.guardNotThisPlatform

   return None

def guardEgressDscpRewrite( mode, token ):
   if not qosHwStatus.dscpRewritePerInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardCosToTcPerInterface( mode, token ):
   if not qosHwStatus.cosToTcPerInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardTcToCosPerInterface( mode, token ):
   if not qosHwStatus.tcToCosPerInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardCpuTcToCosPerInterface( mode, token ):
   if not qosHwStatus.cpuTcToCosPerInterfaceSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardAnyTcToCosPerInterface( mode, token ):
   if guardTcToCosPerInterface( mode, token ) and \
      guardCpuTcToCosPerInterface( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

_guardIntfModePolicyCallback = {}
_guardIntfModePolicyCallback[ QosLib.qosMapType ] = guardIntfModePolicyQos

def macClassCounts( pmap, cmapExclusionList=None ):
   if cmapExclusionList is None:
      cmapExclusionList = []
   cmaps = cliQosAclConfig.cmapType[ QosLib.qosMapType ].cmap
   macClassMapCount = 0
   for classPriority in pmap.classPrio:
      cmapName = pmap.classPrio[ classPriority ].cmapName
      if cmapName in cmapExclusionList or cmapName not in cmaps:
         continue
      if 'matchMacAccessGroup' in cmaps[ cmapName ].match:
         macClassMapCount += 1
   return macClassMapCount
         
def ipClassCounts( pmap, cmapExclusionList=None ):
   if cmapExclusionList is None:
      cmapExclusionList = []
   cmaps = cliQosAclConfig.cmapType[ QosLib.qosMapType ].cmap
   ipv4ClassMapCount = ipv6ClassMapCount = 0
   for classPriority in pmap.classPrio:
      cmapName = pmap.classPrio[ classPriority ].cmapName
      if cmapName in cmapExclusionList or cmapName not in cmaps:
         continue
      if 'matchIpAccessGroup' in cmaps[ cmapName ].match:
         ipv4ClassMapCount += 1
      elif 'matchIpv6AccessGroup' in cmaps[ cmapName ].match:
         ipv6ClassMapCount += 1
   return ipv4ClassMapCount, ipv6ClassMapCount
   
def policerInDefaultClass( pmap ):
   classActionDefault =  pmap.classActionDefault
   if classActionDefault and classActionDefault.policer:
      return True
   return False

def cmapPresentInPolicyMap( pmapName, cmap ):
   pmaps = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap
   if not pmapName in pmaps:
      return False
   
   for classPrio in pmaps[ pmapName ].classPrio:
      if cmap.name == pmaps[ pmapName ].classPrio[ classPrio ].cmapName:
         return True
   return False

def isOtherPolicyActionSet( pmaps, pmapName, cmapName ):
   for action in pmaps[ pmapName ].classAction[ cmapName ].policyAction:
      if pmaps[ pmapName ].classAction[ cmapName ].policyAction[ action ]:
         return True
   return False        

def allowMatchOptionInClassMap( cmap, matchType, vlan=False, cos=False,
                                innerVlan=False ):
   pmaps = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap
   allow = True
   pmapNames = []
   macAclClassCounts = []
   ipv6ClassCounts = []
   vxlanIntfPresent = False
   for pmapName in pmaps:
      cmapPresentInPmap = cmapPresentInPolicyMap( pmapName, cmap )
      if cmapPresentInPmap and \
            not qosAclHwStatus.policerInDefaultClassV4V6Supported \
            and policerInDefaultClass( pmaps[ pmapName ] ):
            # If the class-map is present in a policy map that has a policer in
            # class-default and the match type being configured in the class-map
            # is different from the match type of the existing classe-maps in the
            # policy-map, return False. True Otherwise.
         ipv4ClassCount, ipv6ClassCount = ipClassCounts( pmaps[ pmapName ], 
                                              cmapExclusionList=[ cmap.name ] )
         if matchType == 'ip' and ipv6ClassCount or  \
                matchType == 'ipv6' and ipv4ClassCount:
            allow = False
            pmapNames.append( pmapName )
      if cmapPresentInPmap and qosAclHwStatus.vlanMatchInClassMapSupported \
            and isSviInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                      pmapName ):
         # If the class-map is present in the policy-map which is configured on
         # an SVI and the match type configured in the class-map is vlan/vlan+cos
         # return False. True otherwise.
         if matchType == 'l2Params' and vlan:
            allow = False
            pmapNames.append( pmapName )

      if cmapPresentInPmap and qosAclHwStatus.vxlanPolicyQosSupported \
         and isVxlanIntfInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                         pmapName ):
         # If the class-map is present in the policy-map which is configured on
         # a VTI and the match type configured in the class-map is other than vlan
         # return False. True otherwise.
         vxlanIntfPresent = True
         if ( matchType != 'l2Params' or
              ( matchType == 'l2Params' and ( not vlan or cos or innerVlan ) ) ):
            allow = False
            pmapNames.append( pmapName )

      if cmapPresentInPmap and qosAclHwStatus.matchMacInClassMapSupported:
         if( ( matchType == 'mac' or matchType == 'ipv6' ) and not \
               qosAclHwStatus.matchMacInClassMapV6Supported ):
            # If mac acl on v6 group is not supported then we can't program
            # class map with ipv6 or mac if other class map in the same policy
            # group has vice versa config
            macClassCount = macClassCounts( pmaps[ pmapName ],
                                                   cmapExclusionList=[ cmap.name ] ) 
            ipv4ClassCount, ipv6ClassCount = ipClassCounts( pmaps[ pmapName ],
                                                   cmapExclusionList=[ cmap.name ] )
            if matchType == 'mac' and ipv6ClassCount or \
               matchType == 'ipv6' and macClassCount:
               allow = False
               pmapNames.append( pmapName )
               macAclClassCounts.append( macClassCount )
               ipv6ClassCounts.append( ipv6ClassCount ) 

         if matchType == 'mac' and \
            isOtherPolicyActionSet( pmaps, pmapName, cmap.name ):
            # If a class-map inside a policy-map has only 'police' action and the 
            # match type is 'mac' return True, False otherwise
            allow = False
            pmapNames.append( pmapName )

   return allow, pmapNames, macAclClassCounts, ipv6ClassCounts, vxlanIntfPresent

def allowClassInPolicyMap( self, pmap, cmapName ):
   allow = True
   cmaps = cliQosAclConfig.cmapType[ QosLib.qosMapType ].cmap
   if cmapName in cmaps:
      if not qosAclHwStatus.policerInDefaultClassV4V6Supported and \
            policerInDefaultClass( pmap ):
         ipv4ClassCount, ipv6ClassCount = ipClassCounts( pmap )
         cmapIpType = None
         # If the class-map is already configured with a match type and its 
         # match type is different from the match type of the existing
         # class-maps in the policy-map, return False. True otherwise.
         if 'matchIpAccessGroup' in cmaps[ cmapName ].match:
            cmapIpType = 'ipv4'
         if 'matchIpv6AccessGroup' in cmaps[ cmapName ].match:
            cmapIpType = 'ipv6'
         if cmapIpType == 'ipv4' and ipv6ClassCount or \
            cmapIpType == 'ipv6' and ipv4ClassCount:
            allow = False
            errorMsg = 'When policer is in class-default, both ipv4 '
            errorMsg += 'class-maps and ipv6 class-maps are not supported '
            errorMsg += 'in the policy-map'
            self.addError( errorMsg )

      if ( qosAclHwStatus.vlanMatchInClassMapSupported and
           isSviInServicePolicy( qosInputConfig, qosInputProfileConfig, pmap.name )
           and 'matchL2Params' in cmaps[ cmapName ].match ):
         option = 'l2Params'
         vlanMatchPresent = False
         l2ParamsMatch = cmaps[ cmapName ].match[ matchOptionToEnum( 'l2Params' ) ]

         matchString = 'VLAN'
         # Check for presence of vlan match in the class-map
         if l2ParamsMatch.vlanValue:
            vlanMatchPresent = True

         if not vlanMatchPresent:
            # It can be allowed on SVI
            allow = True
         elif not cmapPresentInPolicyMap( pmap.name, cmaps[ cmapName ] ):
            allow = False
         else:
            # If the class-map is already configured with match type vlan
            # and if a policy-map which is configured on an SVI tries to 
            # instantiate the class-map, return False. True otherwise.
            allow = allowMatchOptionInClassMap( cmaps[ cmapName ],
                                                option, vlanMatchPresent )
            allow = allow[ 0 ]
         if not allow:
            errorMsg = "%s match in class-maps is not supported in " % matchString
            errorMsg += "QoS policy-maps applied to VLAN interfaces."
            self.addError( errorMsg )

      if qosAclHwStatus.vxlanPolicyQosSupported and \
            isVxlanIntfInServicePolicy( qosInputConfig, qosInputProfileConfig,
                                        pmap.name ):
         cmap = cmaps[ cmapName ]
         vlanMatchPresent = False
         cosMatchPresent = False
         innerVlanMatchPresent = False

         option = cmap.match
         l2ParamsMatch = \
            cmap.match.get( matchOptionToEnum( 'l2Params' ), None )
         if l2ParamsMatch:
            if l2ParamsMatch.vlanValue:
               vlanMatchPresent = True
            if l2ParamsMatch.cosValue:
               cosMatchPresent = True
            if l2ParamsMatch.innerVlanValue:
               innerVlanMatchPresent = True

         if not cmapPresentInPolicyMap( pmap.name, cmap ):
            for matchType in cmap.match:
               if ( matchType != 'matchL2Params' or
                    ( matchType == 'matchL2Params' and
                      ( not vlanMatchPresent or cosMatchPresent or
                        innerVlanMatchPresent ) ) ):
                  allow = False
         else:
            # If the class-map is already configured with match type other than vlan
            # and if a policy-map which is configured on a VTI tries to
            # instantiate the class-map, return False. True otherwise.
            allow = allowMatchOptionInClassMap( cmap, option,
                                                vlanMatchPresent,
                                                cosMatchPresent )
            allow = allow[ 0 ]
         if not allow:
            errorMsg = "Only VLAN match is supported in QoS policy-map(s) applied"
            errorMsg += " to a VxLAN interface"
            self.addError( errorMsg )
   return allow

def getL2ParamsMatchCombo( match ):
   matchCombo = list()
   if match.vlanValue:
      matchCombo.append( 'vlan' )
   if match.cosValue:
      matchCombo.append( 'cos' )
   if match.innerVlanValue:
      matchCombo.append( 'vlan inner' )
   return matchCombo

class QosSubIntfModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( ( 'Ethernet',
                                            'Port-Channel' ) ) and
               mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( QosSubIntfModelet )

# From LldpConfigCli
class QosModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( ( mode.intf.name.startswith( 'Ethernet' ) or
                 mode.intf.name.startswith( 'Port-Channel' ) ) and 
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( QosModelet )

class DscpRewriteModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):

      return ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Port-Channel' ) )

IntfCli.IntfConfigMode.addModelet( DscpRewriteModelet )

class CosToTcProfileModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( ( mode.intf.name.startswith( 'Ethernet' ) or
                 mode.intf.name.startswith( 'Port-Channel' ) ) and
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( CosToTcProfileModelet )

class TcToCosModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Port-Channel' ) )

IntfCli.IntfConfigMode.addModelet( TcToCosModelet )

class CpuTcToCosModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) and
               not mode.intf.isSubIntf() )

IntfCli.IntfConfigMode.addModelet( CpuTcToCosModelet )

class QosModeletServicePolicy( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( mode.intf.name.startswith( 'Ethernet' ) or
               mode.intf.name.startswith( 'Port-Channel' ) or
               isinstance( mode.intf, ( VlanIntfCli.VlanIntf, VxlanCli.VxlanIntf )))

IntfCli.IntfConfigMode.addModelet( QosModeletServicePolicy )

class QosProfileMode( QosProfileModeBase, BasicCli.ConfigModeBase ):
   name = "Qos Profile Configuration"
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   def __init__( self, parent, session, context ):
      self.qosProfileModeContext = context
      self.profileName_ = context.profileName()
      param = ( self.profileName_ )
      QosProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.qosProfileModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.qosProfileModeContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.qosProfileModeContext
      self.qosProfileModeContext = None
      context.commit()

   # print one of more profiles
   @staticmethod
   def showQosProfile( mode, args ):
      qosProfileName = args.get( 'PROFILENAME' )
      qosProfileConfig = profileConfigDir.config
      qosProfileAllModel = QosProfileAllModel()
      qosProfilesContainer = QosProfileModelContainer( qosProfileConfig,
                                                       qosHwStatus,
                                                       qosProfileAllModel )
      if qosProfileName == None:
         qosProfilesContainer.populateAll()
      else:
         qosProfilesContainer.populateQosProfile( qosProfileName )
      return qosProfileAllModel

def guardSharingVlan( mode, token ):
   if ( qosAclHwStatus.sviPolicyQosSupported and
        qosAclHwStatus.sviPolicyQosSharingSupported and
        not qosAclHwStatus.sviPolicyQosDefaultSharing ):
      return None
   return CliParser.guardNotThisPlatform

validDropPrecedenceValues = sorted( QosLib.validDpConfigValues() )
nodeQosForShow = CliCommand.guardedKeyword( 'qos',
      helpdesc='Show QoS status', guard=guardQos )

def guardTxQPrioritySupport( mode, token ):
   # By the time this function is called, hwStatus should have been initialized.
   if qosHwStatus.wrrSupported is False:
      return CliParser.guardNotThisPlatform
   # check if the txqueue is in configurable scheduling group if we are @
   # txqueue config mode
   if hasattr( mode, 'txQueue' ) and qosHwStatus.isSchGroupConfigurable[ \
         qosHwStatus.cliTxQueueToHwTxQueue[ mode.txQueue ].schGroupId ] is False:
      return CliParser.guardNotThisPlatform
   if ( isSubIntfConfigMode( mode ) and
        not subIntfHwStatus.subIntfTxQueueSchSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardGuaranteedBw( mode, token ):
   if ( isinstance( mode, ( IntfTxQueueConfigMode, IntfTxQueueRangeConfigMode ) ) and
        not qosHwStatus.guaranteedBwSupported ):
      return CliParser.guardNotThisPlatform
   if ( isinstance( mode, IntfConfigMode ) and
        mode.intf.isSubIntf() and
        not subIntfHwStatus.subIntfBandwidthSupported ):
      return CliParser.guardNotThisPlatform
   if ( isinstance( mode, QosProfileMode ) and
        not subIntfHwStatus.subIntfBandwidthSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardTxQPriorityOrGuaranteedBwSupport( mode, token ):
   if guardTxQPrioritySupport( mode, token ) and \
      guardGuaranteedBw( mode, token ):
      return CliParser.guardNotThisPlatform
   if ( isSubIntfConfigMode( mode ) and
        not subIntfHwStatus.subIntfTxQueueSchSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardGuaranteedBwPps( mode, token ):
   if qosHwStatus.guaranteedBwUnitPpsSupported == False:
      return CliParser.guardNotThisPlatform
   return None

def guaranteedBwKbpsRangeFn( mode ):
   return ( qosHwStatus.minGuaranteedBwKbps, qosHwStatus.maxGuaranteedBwKbps )

def guaranteedBwPpsRangeFn( mode ):
   return ( qosHwStatus.minGuaranteedBwPps, qosHwStatus.maxGuaranteedBwPps )

def ucTxQueueRangeFn( mode=None ):
   return ( 0, qosHwStatus.numUnicastQueueSupported - 1 )

def mcTxQueueRangeFn( mode=None ):
   return ( 0, qosHwStatus.numMulticastQueueSupported - 1 )

def subIntfTxQueueRangeFn( mode=None ):
   if qosGlobalConfig.numTxqsPerSubIntf == 8 and \
      qosHwStatus.tc7ToAnyTxQueueRestricted:
      return ( 0, qosGlobalConfig.numTxqsPerSubIntf - 2 )
   return ( 0, qosGlobalConfig.numTxqsPerSubIntf - 1 )

def txQueueRangeFn( mode=None ):
   if qosHwStatus.tc7ToAnyTxQueueRestricted:
      return ( 0, qosHwStatus.numTxQueueSupported - 2 )
   return ( 0, qosHwStatus.numTxQueueSupported - 1 )

def trafficClassRangeFn( mode=None ):
   return ( 0, qosHwStatus.numTcSupported - 1 )

def getIntfIdentity( intf ):
   if hasattr( intf, 'intfId' ):
      intfId = intf.intfId
   elif hasattr( intf, 'name' ):
      intfId = intf.name
   else:
      intfId = intf
   return intfId 

def setQosProfileConfig( profile ):
   for txQueue, txQueueConfig in profile.txQueueConfig.iteritems():
      if isDefaultTxQueueConfig( txQueueConfig ):
         del profile.txQueueConfig[ txQueue ]
   if ( ( profile.pfcPortConfig ) and 
        ( not profile.pfcPortConfig.enabled ) and
        ( not profile.pfcPortConfig.priorities ) and
        ( profile.pfcPortConfig.watchdogEnabled ) and
        ( not profile.pfcPortConfig.portTimerConfig or
          profile.pfcPortConfig.portTimerConfig ==
          Tac.Value( "Pfc::PortTimerConfig" ) ) and
        ( profile.pfcPortConfig.watchdogPortAction ==
          tacPfcWatchdogAction.invalid ) ):
      profile.pfcPortConfig = None

def setIntfConfig( intf, cfgTrustMode=None, cfgDefaultCos=None,
                   cfgDefaultDscp=None, cfgShapeRate=None, cfgShapeRateUnit=None,
                   profile=False, cfgShapeRateShared=None,
                   cfgShapeRatePercent=None,
                   cfgCosToTcProfileName=None, cfgTcToCosMapName=None,
                   cfgCpuTcToCosMapName=None, burstSizeVal=None,
                   burstSizeUnit=None,
                   cfgGuaranteedBw=None, cfgGuaranteedBwUnit=None,
                   cfgSchedulerGroupName=None ):

   intfConfig = None
   if profile:
      cfg = qosInputProfileConfig
   else:
      cfg = qosInputConfig
   intfCfgd = intf in cfg.intfConfig
   if intfCfgd:
      intfConfig = cfg.intfConfig[ intf ]

   if cfgTrustMode is None:
      if not intfCfgd:
         cfgTrustMode = tacTrustMode.invalid
      else:
         cfgTrustMode = intfConfig.trustMode

   if cfgDefaultCos is None:
      if not intfCfgd:
         cfgDefaultCos = tacCos.invalid
      else:
         cfgDefaultCos = intfConfig.defaultCos

   if cfgDefaultDscp is None:
      if not intfCfgd:
         cfgDefaultDscp = tacDscp.invalid
      else:
         cfgDefaultDscp = intfConfig.defaultDscp

   if cfgShapeRate is None:
      if not intfCfgd:
         cfgShapeRate = tacShapeRateVal.invalid
         cfgShapeRateUnit = tacShapeRateUnit.shapeRateKbps
         cfgShapeRateShared = False
      else:
         cfgShapeRate = intfConfig.shapeRate.rate
         cfgShapeRateUnit = intfConfig.shapeRate.unit
         cfgShapeRateShared = intfConfig.shapeRate.shared

   if cfgShapeRateUnit is None:
      cfgShapeRateUnit = tacShapeRateUnit.shapeRateKbps

   if cfgShapeRatePercent is None:
      if intfCfgd:
         cfgShapeRatePercent = intfConfig.shapeRate.percent
      else:
         cfgShapeRatePercent = Tac.Value( 'Qos::Percent' )

   tempShapeRate = Tac.Value( 'Qos::ShapeRate' )
   burstSizeCfg = Tac.Value( 'Qos::BurstSize' )
   if burstSizeVal:
      burstSizeCfg.value = burstSizeVal
   else:
      if intfCfgd:
         burstSizeCfg.value = intfConfig.shapeRate.burstSize.value
   if burstSizeUnit:
      burstSizeCfg.unit = burstSizeUnit
   else:
      if intfCfgd:
         burstSizeCfg.unit = intfConfig.shapeRate.burstSize.unit

   tempShapeRate.burstSize = burstSizeCfg
   tempShapeRate.rate = cfgShapeRate
   tempShapeRate.unit = cfgShapeRateUnit
   tempShapeRate.shared = bool( cfgShapeRateShared )
   tempShapeRate.percent = cfgShapeRatePercent

   if cfgGuaranteedBw is None:
      if not intfCfgd:
         cfgGuaranteedBw = tacGuaranteedBwVal.invalid
         cfgGuaranteedBwUnit = tacGuaranteedBwUnit.guaranteedBwKbps
      else:
         cfgGuaranteedBw = intfConfig.guaranteedBw.bw
         cfgGuaranteedBwUnit = intfConfig.guaranteedBw.unit

   if cfgGuaranteedBwUnit is None:
      cfgGuaranteedBwUnit = tacGuaranteedBwUnit.guaranteedBwKbps

   tempGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
   tempGuaranteedBw.bw = cfgGuaranteedBw
   tempGuaranteedBw.unit = cfgGuaranteedBwUnit

   if cfgCosToTcProfileName is None:
      if not intfCfgd:
         cfgCosToTcProfileName = tacCosToTcProfileName.defaultProfileName
      else:
         cfgCosToTcProfileName = intfConfig.cosToTcProfileName

   if cfgTcToCosMapName is None:
      if not intfCfgd:
         cfgTcToCosMapName = tacTcToCosMapName.defaultMapName
      else:
         cfgTcToCosMapName = intfConfig.tcToCosMapName

   if cfgCpuTcToCosMapName is None:
      if not intfCfgd:
         cfgCpuTcToCosMapName = tacTcToCosMapName.defaultMapName
      else:
         cfgCpuTcToCosMapName = intfConfig.cpuTcToCosMapName

   if cfgSchedulerGroupName is None:
      # Source this field from the qos scheduling config
      cfgSchedulerGroupName = getSubIntfGroupMembership( intf )

   if intfCfgd: # Interface config exist ( update the current config )
      intfConfig.trustMode = cfgTrustMode
      intfConfig.defaultCos = cfgDefaultCos
      intfConfig.defaultDscp = cfgDefaultDscp
      intfConfig.shapeRate = tempShapeRate
      intfConfig.guaranteedBw = tempGuaranteedBw
      intfConfig.cosToTcProfileName = cfgCosToTcProfileName
      intfConfig.tcToCosMapName = cfgTcToCosMapName
      intfConfig.cpuTcToCosMapName = cfgCpuTcToCosMapName
      intfConfig.schedulerGroupName = cfgSchedulerGroupName
      # Will delete if the values are invalid or hw defaults
      for txQueue, txQueueConfig in intfConfig.txQueueConfig.iteritems():
         if isDefaultTxQueueConfig( txQueueConfig ):
            del intfConfig.txQueueConfig[ txQueue ]

      if isDefaultIntfConfig( qosHwStatus, intfConfig ):
         del cfg.intfConfig[ intf ]
   else:  # interface config doesn't exist
      if ( isDefaultTrustMode( cfgTrustMode ) and
           isDefaultCos( cfgDefaultCos, qosHwStatus )  and
           isDefaultDscp( cfgDefaultDscp, qosHwStatus ) and
           isDefaultShapeRate( tempShapeRate ) and
           isDefaultGuaranteedBw( tempGuaranteedBw ) and
           isDefaultCosToTcProfile( cfgCosToTcProfileName ) and
           isDefaultTcToCosMap( cfgTcToCosMapName ) and
           isDefaultTcToCosMap( cfgCpuTcToCosMapName ) and
           isDefaultSchedulerGroupName( cfgSchedulerGroupName ) ):
         # If there is no interface configuration and we are doing
         # default 'qos' command OR passing HwDefaults, then lets not create the
         # interface config
         return
      else:
         # Add a new config only if there is non-default qos config to be configured
         intfConfig = cfg.intfConfig.newMember( intf )
         intfConfig.trustMode = cfgTrustMode
         intfConfig.defaultCos = cfgDefaultCos
         intfConfig.defaultDscp = cfgDefaultDscp
         intfConfig.shapeRate = tempShapeRate
         intfConfig.guaranteedBw = tempGuaranteedBw
         intfConfig.cosToTcProfileName = cfgCosToTcProfileName
         intfConfig.tcToCosMapName = cfgTcToCosMapName
         intfConfig.cpuTcToCosMapName = cfgCpuTcToCosMapName
         intfConfig.schedulerGroupName = cfgSchedulerGroupName

#------------------------------------------------------------------------------------
# Commands in "interface config" mode:
# "[ mls ] qos trust [ cos | dscp ]" 
# "[ no|default ] [ mls ] qos trust [ cos | dscp ]"
#------------------------------------------------------------------------------------

def setQosTrustMode( mode, args ):
   timestamp = Tac.now()
   trustMode = args.get( 'MODE', 'None' )
   if type( mode ) == QosProfileMode:
      profile = mode.qosProfileModeContext.currentEntry_
      prevTrustMode = profile.trustMode
      if CliCommand.isDefaultCmd( args ):
         trustMode = tacTrustMode.invalid
      elif CliCommand.isNoCmd( args ):
         if ( trustMode == tacTrustMode.dscp or trustMode == tacTrustMode.cos ):
            if ( ( trustMode == tacTrustMode.dscp and 
                  prevTrustMode == tacTrustMode.dscp ) or
                  ( trustMode == tacTrustMode.cos and 
                  prevTrustMode == tacTrustMode.cos ) ):
               trustMode = tacTrustMode.invalid
            else:
               return
         else:
            trustMode = tacTrustMode.untrusted
      if prevTrustMode == trustMode:
         return
      profile.trustMode = trustMode
   else:
      # Get the current qos config for the intf
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if CliCommand.isDefaultCmd( args ):
         trustMode = tacTrustMode.invalid
      elif CliCommand.isNoCmd( args ):
         if (trustMode == tacTrustMode.dscp or trustMode == tacTrustMode.cos):
            if ( ( trustMode == tacTrustMode.dscp and intfConfig and 
                  intfConfig.trustMode == tacTrustMode.dscp) or
                  ( trustMode == tacTrustMode.cos and intfConfig and 
                  intfConfig.trustMode == tacTrustMode.cos) ):
               trustMode = tacTrustMode.invalid
            else:
               return             
         else:
            trustMode = tacTrustMode.untrusted
               
      if ( trustMode == tacTrustMode.untrusted or 
         ( trustMode == tacTrustMode.dscp and 
           not pfcStatus.dscpBasedPfcSupported ) ):
         portStatus = None
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            portStatus = sliceHwStatus.portStatus
            if mode.intf.name in portStatus:
               if portStatus[ mode.intf.name ].active:
                  mode.addError( "Cannot change trust mode on "
                                 "priority-flow-control enabled interface" )
                  return

      if not intfConfig:
         prevTrustMode = tacTrustMode.invalid
      else:
         prevTrustMode = intfConfig.trustMode
      if prevTrustMode == trustMode:
         return

      setIntfConfig( mode.intf.name, cfgTrustMode=trustMode )
      if cliBlockingFail( mode, timestamp, tacFeatureName.trustMode, "Trust Mode",
                          mode.intf.name ):
         setIntfConfig( mode.intf.name, cfgTrustMode=prevTrustMode )

matcherMls = CliMatcher.KeywordMatcher( 'mls', helpdesc='MLS interface commands' )
nodeMls = CliCommand.Node( matcher=matcherMls, hidden=True )
nodeQosForConfig = CliCommand.guardedKeyword( 'qos', 
      helpdesc='Configure QoS parameters', guard=guardQos )

#--------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos trust MODE 
#--------------------------------------------------------------------------------
class QosTrustCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos trust MODE'
   noOrDefaultSyntax = '[ mls ] qos trust [ MODE ]'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForConfig,
      'trust': 'Set trust mode for the interface',
      'MODE': CliMatcher.EnumMatcher( {
        'cos': 'Set trust mode to CoS',
        'dscp': 'Set trust mode to DSCP',
      } )
   }

   handler = setQosTrustMode
   noOrDefaultHandler = handler

QosModelet.addCommandClass( QosTrustCmd )
QosProfileMode.addCommandClass( QosTrustCmd )

#------------------------------------------------------------------------------------
# For Bali and Trident
# The "[no | default] [mls] qos cos [<0-7>]" command, in "interface config" mode.

# For Petra 
# platform petraA <chip-name> traffic-class <tc-value>
#------------------------------------------------------------------------------------
def guardDefaultCos( mode, token ):
   if qosHwStatus.defaultCosSupported:
      return None
   return CliParser.guardNotThisPlatform

def setQosCosDefault( mode, args ):
   timestamp = Tac.now()
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   defaultCos = args.get( 'COS' )
    # Get the current default Cos value in the intfConfig
   if type( mode ) == QosProfileMode:
      profile = mode.qosProfileModeContext.currentEntry_
      prevDefaultCos = profile.defaultCos
      if noOrDefault:
         defaultCos = tacCos.invalid
      if prevDefaultCos == defaultCos:
         return
      profile.defaultCos = defaultCos
   else:
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if intfConfig:
         prevDefaultCos = intfConfig.defaultCos
      else:
         prevDefaultCos = tacCos.invalid

      if noOrDefault:
         defaultCos = tacCos.invalid
      if prevDefaultCos == defaultCos:
         return
      setIntfConfig( mode.intf.name, cfgDefaultCos=defaultCos )
      if cliBlockingFail( mode, timestamp, tacFeatureName.defaultCos,
                          "Default Cos", mode.intf.name ):
         setIntfConfig( mode.intf.name, cfgDefaultCos=prevDefaultCos )

nodeCos = CliCommand.guardedKeyword( 'cos', 
      helpdesc='Set default CoS value for the interface', guard=guardDefaultCos )

#--------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos cos COS
#--------------------------------------------------------------------------------
class QosCosCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos cos COS'
   noOrDefaultSyntax = '[ mls ] qos cos ...'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForConfig,
      'cos': nodeCos,
      'COS': CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
                helpdesc='Class of Service (CoS) value' ),
   }

   handler = setQosCosDefault
   noOrDefaultHandler = handler

QosModelet.addCommandClass( QosCosCmd )
QosProfileMode.addCommandClass( QosCosCmd )

#------------------------------------------------------------------------------------
# For Bali and Trident
# The "[no | default] [mls] qos dscp [<0-63>]" command, in "interface config" mode.
#------------------------------------------------------------------------------------
def guardDefaultDscp( mode, token ):
   if qosHwStatus.defaultDscpSupported:
      return None
   return CliParser.guardNotThisPlatform

def setQosDscpDefault( mode, args ):
   timestamp = Tac.now()
   noOrDefault = CliCommand.isNoOrDefaultCmd( args )
   defaultDscp = args.get( 'DSCP' )
    # Get the current default Dscp value in the intfConfig
   if type( mode ) == QosProfileMode:
      profile = mode.qosProfileModeContext.currentEntry_
      prevDefaultDscp = profile.defaultDscp
      if noOrDefault:
         defaultDscp = tacDscp.invalid
      if prevDefaultDscp == defaultDscp:
         return
      profile.defaultDscp = defaultDscp
   else:
      intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
      if intfConfig:
         prevDefaultDscp = intfConfig.defaultDscp
      else:
         prevDefaultDscp = tacDscp.invalid

      if noOrDefault:
         defaultDscp = tacDscp.invalid
      if prevDefaultDscp == defaultDscp:
         return
      setIntfConfig( mode.intf.name, cfgDefaultDscp=defaultDscp )
      if cliBlockingFail( mode, timestamp, tacFeatureName.defaultDscp,
                          "Default Dscp", mode.intf.name ):
         setIntfConfig( mode.intf.name, cfgDefaultDscp=prevDefaultDscp )

nodeDscp = CliCommand.guardedKeyword( 'dscp', 
      helpdesc='Set default DSCP value for the interface', guard=guardDefaultDscp )

#--------------------------------------------------------------------------------
# [ no | default ] [ mls ] qos dscp DSCP
#--------------------------------------------------------------------------------
class QosDscpCmd( CliCommand.CliCommandClass ):
   syntax = '[ mls ] qos dscp DSCP'
   noOrDefaultSyntax = '[ mls ] qos dscp ...'
   data = {
      'mls': nodeMls,
      'qos': nodeQosForConfig,
      'dscp': nodeDscp,
      'DSCP': CliMatcher.IntegerMatcher( tacDscp.min, tacDscp.max, 
         helpdesc='DSCP value' ),
   }

   handler = setQosDscpDefault
   noOrDefaultHandler = handler

QosModelet.addCommandClass( QosDscpCmd )
QosProfileMode.addCommandClass( QosDscpCmd )

#-------------------------------------------------------------------------------
# tx-queue config mode. A new instance of this mode is created when the
# user enters "tx-queue <queue-id>" inside config-if mode.
#-------------------------------------------------------------------------------
class IntfTxQueueConfigMode( IntfTxQueueMode, BasicCli.ConfigModeBase ):
   #----------------------------------------------------------------------------
   # Attributes required of every Mode class.
   #----------------------------------------------------------------------------
   name = 'Tx-queue configuration'
   modeParseTree = CliParser.ModeParseTree()

   #----------------------------------------------------------------------------
   # Constructs a new TxQueueMode instance.
   #----------------------------------------------------------------------------
   def __init__( self, parent, session, tokenQueueType=None, txQueueId=None ):
      assert txQueueId != None
      self.txQueue = Tac.Value( 'Qos::TxQueue' )
      self.txQueue.id = txQueueId
      if 'tx-queue' == tokenQueueType:
         self.queueToken = 'txq'
         self.txQueue.type = tacQueueType.unknown
      elif 'uc-tx-queue' == tokenQueueType:
         self.queueToken = 'uc-txq'
         self.txQueue.type = tacQueueType.ucq
      elif 'mc-tx-queue' == tokenQueueType:
         self.queueToken = 'mc-txq'
         self.txQueue.type = tacQueueType.mcq
      else:
         assert 0
      assert type( parent ) == IntfCli.IntfConfigMode or \
          type( parent ) == IntfRangeConfigMode or \
          type( parent ) == QosProfileMode 
      # To support tx-queue mode in intf Range mode, we first create individual modes
      # (per intf queue pair) in IntfTxQueueMode and create a multimode in case of 
      # interface range or a single mode in case of single interface. The new mode
      # is entered into, as a child mode in the caller of this function.
      if type( parent ) == IntfCli.IntfConfigMode:
         iName = 'if-%s' % ( parent.intf.shortname )
         IntfTxQueueMode.__init__( self, ( iName,
                                           self.queueToken,
                                           txQueueId, parent ) )
         BasicCli.ConfigModeBase.__init__( self, parent, session )
      elif type( parent ) == QosProfileMode:
         IntfTxQueueMode.__init__( self, ( parent.longModeKey, 
                                           self.queueToken, 
                                           txQueueId, parent ) )
         BasicCli.ConfigModeBase.__init__( self, parent, session )
      else:
         IntfTxQueueMode.__init__( self, ( parent.longModeKey,
                                           self.queueToken,
                                           txQueueId, parent ) )
         self.individualIntfChildModes_ = []
         for intfMode in parent.individualIntfModes_:
            mode = IntfTxQueueConfigMode( intfMode, session,
                                          tokenQueueType, txQueueId )
            self.individualIntfChildModes_.append( mode )
         self.modeKey = self.queueToken
         self.longModeKey = '%s-%s-%d' % ( parent.longModeKey,
                                           self.queueToken, txQueueId )
         BasicCli.ConfigModeBase.__init__( 
            self, parent, session, multiInstance=True, 
            multiModes=self.individualIntfChildModes_ )

   def showActive( self ):
      parent = self.parent_
      if type( parent ) != IntfRangeConfigMode:
         BasicCli.ConfigModeBase.showActive( self )
         return
      # If parent mode is IntfRange then run show active for each intf mode
      url = FileUrl.localRunningConfig( *Url.urlArgsFromMode( self ) )
      content = url.open().readlines()
      for intfMode in parent.individualIntfModes_:
         BasicCliUtil.showRunningConfigWithFilter( content, self.filterExp(),
                                                   [ intfMode.filterExp() ] )

   def showActiveAll( self, showDetail ):
      parent = self.parent_
      if type( parent ) != IntfRangeConfigMode:
         BasicCli.ConfigModeBase.showActiveAll( self, showDetail )
         return
      # If parent mode is IntfRange then run show active for each intf mode
      url = FileUrl.localRunningConfigAll( *Url.urlArgsFromMode( self ),
                                           showDetail=showDetail )
      content = url.open().readlines()
      for intfMode in parent.individualIntfModes_:
         BasicCliUtil.showRunningConfigWithFilter( content, self.filterExp(),
                                                   [ intfMode.filterExp() ] )

# Tx-Queue Range config mode for tx-queue range feature.
class IntfTxQueueRangeConfigMode( BasicCli.ConfigModeBase ):
   # Necessary attributes for every Mode class
   name = "Tx-Queue Range Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, tokenQueueType=None, txQueueId=None ):
      assert txQueueId is not None
      self.individualTxQueueChildModes_ = []
      for txQ in txQueueId:
         mode = IntfTxQueueConfigMode( parent, session, tokenQueueType, txQ )
         self.individualTxQueueChildModes_.append( mode )
      if tokenQueueType == 'tx-queue' :
         self.queueToken = 'txq'
      elif 'uc-tx-queue' == tokenQueueType:
         self.queueToken = 'uc-txq'
      elif 'mc-tx-queue' == tokenQueueType:
         self.queueToken = 'mc-txq'
      else:
         assert 0, 'Incorrect Tx-Queue type'
      self.longModeKey = '%s-%s-%s' % ( parent.longModeKey, self.queueToken,
                                        MultiRangeRule.multiRangeToCanonicalString(
                                           txQueueId ) )
      self.modeKey = self.queueToken
      # Calling base class constructor
      BasicCli.ConfigModeBase.__init__(
         self,parent,session,multiInstance=True,
         multiModes=self.individualTxQueueChildModes_ )

   def getCompletions( self, tokens, partialToken, startWithPartialToken=False ):
      # Overriding getCompletions. 
      # Any of my member modes is as good as any other for completions.
      return self.individualModes_[0].getCompletions( tokens,
                                                      partialToken,
                                                      startWithPartialToken )

   def parse( self, tokens, autoComplete=True, authz=True, acct=True ):

      # If there is a range expression, eg. {1,3,1,1} then this function will handle
      # that.
      if CliRangeExpansion.tryRangeParse( self, tokens, autoComplete=autoComplete,
                                          authz=authz, acct=acct ):
         return
      try:
         # This is for handling commands like 'exit'.
         return BasicCli.ConfigModeBase.parse( self, tokens,
                                               autoComplete=autoComplete,
                                               authz=authz, acct=acct )
      except CliParser.ParseError:
         # If the command was not parsed then, we try to parse ourselves.
         pass

      # If the command is not parsed by any of the above parsers successfully
      # then we try to parse it with each individual child tx-queue modes.

      results = {}
      for m in self.individualModes_:
         results[ m ] = m.parse( tokens, autoComplete=autoComplete, authz=authz,
                                acct=acct )

      return { "valueFunc" : self._invokeIndividualChildValueFunc,
               "autoConfigSessionAllowed" : True,
               "allowCache": False,
               "kargs" : { "results" : results, "authz" : authz, "acct" : acct },
               "aaa" : None,
               "cmdDeprecatedBy": None }

   def _invokeIndividualChildValueFunc( self, mode, results, authz, acct ):
      try:
         # Try invoking the valueFunc of each child
         for m in self.individualModes_:
            # pylint: disable-msg=W0212
            m.session._invokeValueFunc( m, results[ m ], authz, acct )

            # Suppressing accounting for further calls
            acct = False

      finally:
         pass
      return None
     
   def showActive( self ):
      for m in self.individualModes_:
         m.showActive()

   def showActiveAll( self, showDetail ):
      for m in self.individualModes_:
         m.showActiveAll( showDetail )

# This modelet is used for ECN only so we test gurad here too.
class IntfUcTxQueueModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()
   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

IntfTxQueueConfigMode.addModelet( IntfUcTxQueueModelet )

def guardSubIntfTxQueue( mode, token ):
   if not qosHwStatus.hwInitialized or 0 == qosHwStatus.numTxQueueSupported:
      return CliParser.guardNotThisPlatform
   if qosHwStatus.numMulticastQueueSupported != 0:
      return CliParser.guardNotThisPlatform
   if ( not subIntfHwStatus.subIntfTxQueueShapeRateSupported and
        not subIntfHwStatus.subIntfTxQueueSchSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardTxQueue( mode, token ):
   if not qosHwStatus.hwInitialized or 0 == qosHwStatus.numTxQueueSupported:
      return CliParser.guardNotThisPlatform
   if qosHwStatus.numMulticastQueueSupported != 0:
      return CliParser.guardNotThisPlatform
   return None

def guardUcMcTxQueue( mode, token ):
   if not qosHwStatus.hwInitialized or 0 == qosHwStatus.numTxQueueSupported:
      return CliParser.guardNotThisPlatform
   if 0 == qosHwStatus.numMulticastQueueSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardTxQueueRandomDetect( mode, token ):
   if isSubIntfConfigMode( mode ) or mode.queueToken == 'mc-txq':
      return CliParser.guardNotThisPlatform
   return None

def guardEcnDelay( mode, token ):
   if isSubIntfConfigMode( mode ):
      return CliParser.guardNotThisPlatform
   if qosHwStatus.ecnDelaySupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnCounters( mode, token ):
   if qosHwStatus.ecnCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnShow( mode, token ):
   if qosHwStatus.ecnDelaySupported or qosHwStatus.ecnCountersSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcn( mode, token ):
   if qosHwStatus.ecnSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnWeight( mode, token ):
   if qosHwStatus.ecnWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardWredWeight( mode, token ):
   if qosHwStatus.wredWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNonEctWeight( mode, token ):
   if qosHwStatus.nonEctWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardGlobalEcnWeight( mode, token ):
   if qosHwStatus.globalEcnWeightSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardNonEctParams( mode, token ):
   if qosHwStatus.nonEctThdSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardWred( mode, token ):
   if qosHwStatus.wredSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnConfigureNonEct( mode, token ):
   if qosHwStatus.ecnConfigureNonEctSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardEcnQueueCounter( mode, token ):
   if not qosHwStatus.ecnQueueCounterSupported or \
      isinstance( mode.parent_, QosProfileMode ):
      return CliParser.guardNotThisPlatform
   intfList = getIntfListFromMode( mode.parent_ )
   for intf in intfList:
      if not intf.startswith( 'Ethernet' ):
         return CliParser.guardNotThisPlatform
   return None

def guardShowEcnQueueCounter( mode, token ):
   if not qosHwStatus.ecnQueueCounterSupported:
      return CliParser.guardNotThisPlatform
   return None

def guardShowWredQueueCounter( mode, token ):
   if not qosHwStatus.wredQueueCounterSupported:
      return CliParser.guardNotThisPlatform
   return None

def ucTxQueueEcnThresholdRangeSegmentsFn( mode ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes / 
            qosHwStatus.ecnSegmentSizeInBytes )

def ucTxQueueEcnThresholdRangeBytesFn( mode ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes )

def ucTxQueueEcnThresholdRangeKBytesFn( mode ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes / 1000 )

def ucTxQueueEcnThresholdRangeMBytesFn( mode ):
   return ( 1, qosHwStatus.ecnMaxQueueThresholdInBytes / 1000000 )

def maxThresholdInSeconds( mult ):
   # formula for converting max bytes to max milliseconds is
   # 1 sec = ( max_bytes * mult * 8 bits ) / ( bandwith_val * bits/sec )
   # max bps is 400 Gbps = 4 * 1e+9bps
   maxBitrate = tacShapeRateVal.maxKbps * 1000
   # multiplier can be added to convert secs to millis/micros/etc
   return max( 1, ( qosHwStatus.ecnMaxQueueThresholdInBytes * mult * 8 ) /
                  ( maxBitrate ) )

def ucTxQueueEcnThresholdRangeMillisecondsFn( mode ):
   # 1000ms = 1s
   maxThresholdInMilliseconds = maxThresholdInSeconds( 1000 )
   return ( 1, maxThresholdInMilliseconds )

def ucTxQueueEcnThresholdRangeMicrosecondsFn( mode ):
   # 1e+6us = 1s
   maxThresholdInMicroseconds = maxThresholdInSeconds( 10**6 )
   return ( 1, maxThresholdInMicroseconds )

def weightRangeFn( mode=None ):
   return ( qosHwStatus.minWeightValue, qosHwStatus.maxWeightValue )

def guardEcnMarkProb( mode, token ):
   if not qosHwStatus.ecnMarkProbSupported:
      return CliParser.guardNotThisPlatform
   return None

def goToTxQueueMode( mode, tokenQueueType, txQueueId ):
   childMode = mode.childMode( IntfTxQueueConfigMode, tokenQueueType=tokenQueueType,
                               txQueueId=txQueueId )
   mode.session_.gotoChildMode( childMode )

# Function that handles tx-queue range
def goToTxQueueRangeMode( mode, tokenQueueType, txQueueIds ):
   ( txQueueId, txQueueMode ) = ( txQueueIds[ 0 ], IntfTxQueueConfigMode )\
                                if len( txQueueIds ) == 1 else ( txQueueIds,
                                                         IntfTxQueueRangeConfigMode )
   childMode = mode.childMode( txQueueMode, tokenQueueType=tokenQueueType,
                               txQueueId=txQueueId )
   mode.session_.gotoChildMode( childMode )
   
def setDefaultTxQConfig( mode, noOrDefaultKw, tokenQueueType, txQueueId ):
   txQueueConfig = None
   txQueue = Tac.Value( 'Qos::TxQueue' )
   txQueue.id = txQueueId
   if 'tx-queue' == tokenQueueType:
      txQueue.type = tacQueueType.unknown
   elif 'uc-tx-queue' == tokenQueueType:
      txQueue.type = tacQueueType.ucq
   elif 'mc-tx-queue' == tokenQueueType:
      txQueue.type = tacQueueType.mcq
   else:
      assert 0, 'Unknown queue type supplied'

   if type( mode ) == QosProfileMode:
      profile = mode.qosProfileModeContext.currentEntry()
      txQueueConfig = profile.txQueueConfig.get( txQueue )
   else:
      intf = mode.intf.name
      intfConfig = qosInputConfig.intfConfig.get( intf )
      if intfConfig:
         txQueueConfig = intfConfig.txQueueConfig.get( txQueue )
   if txQueueConfig:
      txQueueConfig.priority = tacTxQueuePriority.priorityInvalid
      txQueueConfig.shapeRate = invalidShapeRate
      txQueueConfig.guaranteedBw = invalidGuaranteedBw
      txQueueConfig.bandwidth = tacPercent.invalid
      txQueueConfig.ecnConfig = None
      txQueueConfig.nonEctConfig = None
      txQueueConfig.delayEcnEnabled = False
      txQueueConfig.ecnDelayThreshold = tacEcnDelayThreshold
      txQueueConfig.wredConfig = None
      txQueueConfig.dropThresholds.clear()
      txQueueConfig.latencyThreshold = tacLatencyThreshold()
      if type( mode ) == QosProfileMode:
         setQosProfileConfig( profile )
      else:
         setIntfConfig( intf )

   # Remove queue counter config
   if ( type( mode ) != QosProfileMode ) and \
      ( txQueue.type == tacQueueType.unknown or
        txQueue.type == tacQueueType.ucq ) and \
      mode.intf.name.startswith( 'Ethernet' ):
      intf = mode.intf.name
      setEcnQueueCounterConfig( intf, mode, txQueueId, False )

# Default function for tx-queue range feature
def setDefaultTxQRangeConfig( mode, noOrDefaultKw, tokenQueueType, txQueueIds ):
   for txQueueId in txQueueIds:
      # Iterate through all the tx-queues and set default for each.
      setDefaultTxQConfig( mode, noOrDefaultKw, tokenQueueType, txQueueId )

def guardQosScheduling( mode, token ):
   if not subIntfHwStatus.subIntfSchedulingGroupSupported:
      return CliParser.guardNotThisPlatform
   return None

matcherTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=txQueueRangeFn, noSingletons=False,
      helpdesc="Tx-Queue ID or range(s) of Tx-Queue IDs" )
matcherUcTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=ucTxQueueRangeFn, noSingletons=False,
      helpdesc="Unicast Tx-Queue ID or range(s) of Unicast Tx-Queue ID" )
matcherMcTxQueueRange = MultiRangeRule.MultiRangeMatcher(
      rangeFn=mcTxQueueRangeFn, noSingletons=False,
      helpdesc="Multicast Tx-Queue ID or range(s) of Multicast Tx-Queue ID" )
matcherRewrite = CliMatcher.KeywordMatcher( 'rewrite',
      helpdesc='Set QoS rewrite setting' )
matcherDscpRewriteMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.dscpRewriteMap,
      'Map name', pattern='(?!default-map$)[A-Za-z0-9_:{}\\[\\]-]+' )
matcherMapDscp = CliMatcher.KeywordMatcher( 'dscp',
      helpdesc='Map Differentiated Services Code Point (DSCP) values' )
matcherDscpValue = CliMatcher.IntegerMatcher( tacDscp.min, tacDscp.max,
      helpdesc='DSCP value', priority=CliParser.PRIO_HIGH )
matcherMultipleDscpValues = MultiRangeRule.MultiRangeMatcher(
      rangeFn=lambda: ( tacDscp.min, tacDscp.max ), noSingletons=False,
      helpdesc='Differentiated Services Code Point (DSCP) value(s) or '
      'range(s) of DSCP values' )
matcherMapCos = CliMatcher.KeywordMatcher( 'cos',
      helpdesc='Map Class of Service (CoS) values' )
matcherCosValue = CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
      helpdesc='Class of Service (CoS) value',
      priority=CliParser.PRIO_HIGH )
matcherTrafficClass = CliMatcher.KeywordMatcher( 'traffic-class',
      helpdesc='Set traffic-class value' )
matcherMap = CliMatcher.KeywordMatcher( 'map',
      helpdesc='Mapping configuration of different QoS parameters' )
matcherCosToTcProfileName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.cosToTcProfile, helpdesc='Map Name',
      pattern=r'(?!global-profile$)[A-Za-z0-9_:{}\[\]-]+' )
matcherDscpToTcMapName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosInputConfig.dscpToTcNamedMap, helpdesc='Dscp to Tc Map Name',
      pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )
matcherTrafficClassValue = CliMatcher.DynamicIntegerMatcher(
      trafficClassRangeFn,
      helpdesc='Traffic class value', priority=CliParser.PRIO_HIGH )
matcherTrafficClassList = MultiRangeRule.MultiRangeMatcher(
      rangeFn=trafficClassRangeFn, noSingletons=False,
      helpdesc='Traffic Class (TC) value(s) or range(s) of TC values' )
matcherSchedulingGroupName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosSchedulerConfig.qosSchedulerIntfConfig[
         mode.intf ].name,
      helpdesc='Scheduling group name',
      pattern=r'(?!scheduling-group$)[A-Za-z0-9_:{}\[\]-]+' )
matcherSchedulingPolicyName = CliMatcher.DynamicNameMatcher(
      lambda mode: qosSchedulerConfig.policy, helpdesc='Scheduling policy name',
      pattern=r'(?!scheduling-policy$)[A-Za-z0-9_]+' )
matcherPercent = CliMatcher.IntegerMatcher( tacPercent.min, tacPercent.max,
      helpdesc='Percent value' )
matcherAccessGroup = CliMatcher.KeywordMatcher( 'access-group',
      helpdesc='Match with given access group' )
matcherCe = CliMatcher.KeywordMatcher( 'ce',
      helpdesc='ECN codepoint CE' )
matcherDscp = CliMatcher.KeywordMatcher( 'dscp',
      helpdesc='Specify DSCP match' )
matcherEcn = CliMatcher.KeywordMatcher( 'ecn',
      helpdesc='Specify ECN parameter' )
matcherEct = CliMatcher.KeywordMatcher( 'ect',
      helpdesc='ECN codepoint ECT(0) or ECT(1)' )
matcherEctCe = CliMatcher.KeywordMatcher( 'ect-ce',
      helpdesc='ECN codepoint ECT(0), ECT(1) or CE' )
matcherMatch = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Match the access rule specified' )
matcherMpls = CliMatcher.KeywordMatcher( 'mpls',
      helpdesc='Specify MPLS match' )
matcherNonEct = CliMatcher.KeywordMatcher( 'non-ect',
      helpdesc='ECN codepoint Not-ECT' )
nodeDscpToTcMapTo = CliCommand.guardedKeyword( 'to',
      helpdesc="Map to Traffic-Class",
      guard=guardPwDecapDscpQosMap )
nodeDscpMatch = CliCommand.guardedKeyword( 'dscp',
      helpdesc='Specify DSCP match', guard=guardDscpEcnMatch )
nodeDscpEcn = CliCommand.guardedKeyword( 'ecn',
      helpdesc='Specify ECN parameter', guard=guardDscpEcnMatch )
nodeMac = CliCommand.guardedKeyword( 'mac',
      helpdesc='Specify MAC Access-groups', guard=guardMacAccessGroupMatch )
nodeMpls = CliCommand.guardedKeyword( 'mpls',
      helpdesc='Specify MPLS match', guard=guardMplsTrafficClassMatch )
nodeTrafficClass = CliCommand.guardedKeyword( 'traffic-class',
      helpdesc='Specify MPLS Traffic Class match', guard=guardMplsTrafficClassMatch )
nodeTxQueue = CliCommand.guardedKeyword( 'tx-queue',
      helpdesc='Configure transmit queue parameters', guard=guardTxQueue )
nodeUcTxQueue = CliCommand.guardedKeyword( 'uc-tx-queue',
      helpdesc='Configure unicast transmit queue parameters',
      guard=guardUcMcTxQueue )
nodeMcTxQueue = CliCommand.guardedKeyword( 'mc-tx-queue',
      helpdesc='Configure multicast transmit queue parameters',
      guard=guardUcMcTxQueue )
nodePriority = CliCommand.guardedKeyword( 'priority',
      helpdesc='Configure priority of this queue', guard=guardTxQPrioritySupport )
nodeBandwidth = CliCommand.guardedKeyword( 'bandwidth',
      helpdesc='Configure guaranteed/round-robin minimum bw for this queue',
      guard=guardTxQPriorityOrGuaranteedBwSupport )
nodeGuaranteed = CliCommand.guardedKeyword( 'guaranteed',
      helpdesc='Set guaranteed bandwidth', guard=guardGuaranteedBw )
nodeScheduling = CliCommand.guardedKeyword( 'scheduling',
      helpdesc='Configure qos scheduling parameters',
      guard=guardQosScheduling )

#--------------------------------------------------------------------------------
# ( ( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) | ( mc-tx-queue MCTXQSET ) )
#--------------------------------------------------------------------------------
class QueueSetIntfRangeConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''( tx-queue TXQSET ) | ( uc-tx-queue UCTXQSET ) 
             | ( mc-tx-queue MCTXQSET )''' 
   data = {
      'tx-queue': nodeTxQueue,
      'uc-tx-queue': nodeUcTxQueue,
      'mc-tx-queue': nodeMcTxQueue,
      'TXQSET': matcherTxQueueRange,
      'UCTXQSET': matcherUcTxQueueRange,
      'MCTXQSET': matcherMcTxQueueRange,
   }

   @staticmethod
   def handler( mode, args ):
      queue = { 'tx-queue' : 'TXQSET',
                'uc-tx-queue' : 'UCTXQSET',
                'mc-tx-queue' : 'MCTXQSET' }
      for key, val in queue.items():
         if key in args:
            goToTxQueueRangeMode( mode, key, 
                  args.get( val ).values() )
            return

IntfRangeConfigMode.addCommandClass( QueueSetIntfRangeConfigCmd )

def getIntfListFromMode( intfMode ):
   if type( intfMode ) == IntfCli.IntfConfigMode:
      return  [ intfMode.intf.name ]
   else:
      return intfMode.intfList.intfNames()

def setPriority( mode, args ):
   if CliCommand.isNoCmd( args ):
      priority = tacTxQueuePriority.priorityRoundRobin
   elif not CliCommand.isDefaultCmd( args ) and isSubIntfConfigMode( mode ) :
      # For SubIntfs, publish priorityStrict in case of 'priority strict' cmd.
      priority = tacTxQueuePriority.priorityStrict
   else:
      priority = tacTxQueuePriority.priorityInvalid

   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         if priority == tacTxQueuePriority.priorityInvalid:
            return 
         txQueueConfig = profile.txQueueConfig.newMember( 
            mode.txQueue, priority, tacPercent.invalid, 
            invalidShapeRate, invalidGuaranteedBw )
      else:
         txQueueConfig.priority = priority
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if priority == tacTxQueuePriority.priorityInvalid:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember( 
               mode.txQueue, priority, tacPercent.invalid, 
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if priority == tacTxQueuePriority.priorityInvalid:
                  continue
               txQueueConfig = intfConfig.txQueueConfig.newMember( 
                  mode.txQueue, priority, tacPercent.invalid, 
                  invalidShapeRate, invalidGuaranteedBw )
            else:
               txQueueConfig.priority = priority
         setIntfConfig( intf )
         
#--------------------------------------------------------------------------------
# priority strict
# ( no | default ) priority ...
#--------------------------------------------------------------------------------
class PriorityStrictCmd( CliCommand.CliCommandClass ):
   syntax = 'priority strict'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': nodePriority,
      'strict': 'Set priority to be strict',
   }

   handler = setPriority
   noOrDefaultHandler = handler

IntfTxQueueConfigMode.addCommandClass( PriorityStrictCmd )

def setBandwidth( mode, args ):
   bandwidth = args[ 'PERCENT' ]
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, bandwidth,
               invalidShapeRate, invalidGuaranteedBw )
      else:
         txQueueConfig.bandwidth = bandwidth
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, bandwidth,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid, bandwidth,
                  invalidShapeRate, invalidGuaranteedBw )
            else:
               txQueueConfig.bandwidth = bandwidth
         setIntfConfig( intf )

def noBandwidth( mode, args ):
   bandwidth = tacPercent.invalid
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return
      else:
         txQueueConfig.bandwidth = bandwidth
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is not None:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is not None:
               txQueueConfig.bandwidth = bandwidth
            setIntfConfig( intf )

#--------------------------------------------------------------------------------
# bandwidth percent PERCENT
# ( no | default ) bandwidth percent ...
#--------------------------------------------------------------------------------
class BandwidthPercentCmd( CliCommand.CliCommandClass ):
   syntax = 'bandwidth percent PERCENT'
   noOrDefaultSyntax = 'bandwidth percent ...'
   data = {
      'bandwidth': nodeBandwidth,
      'percent': CliCommand.guardedKeyword( 'percent',
         helpdesc=( 'Set proportion of the total bandwidth '
            'when scheduling is round-robin' ), guard=guardTxQPrioritySupport ),
      'PERCENT': matcherPercent,
   }

   handler = setBandwidth
   noOrDefaultHandler = noBandwidth 

IntfTxQueueConfigMode.addCommandClass( BandwidthPercentCmd )

def setGuaranteedBw( mode, args ):
   guaranteedBw = args.get( 'PERCENT' ) or args.get( 'PPS' ) or \
                  args.get( 'KBPS' )
   guaranteedBwUnit = args.get( 'percent' ) or args.get( 'pps', 'kbps' )
   cfgGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
   cfgGuaranteedBw.bw = guaranteedBw
   if guaranteedBwUnit == 'pps':
      cfgGuaranteedBw.unit = tacGuaranteedBwUnit.guaranteedBwPps
   elif guaranteedBwUnit == 'percent':
      cfgGuaranteedBw.unit = tacGuaranteedBwUnit.guaranteedBwPercent
   else:
      cfgGuaranteedBw.unit = tacGuaranteedBwUnit.guaranteedBwKbps
   if isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      profile.guaranteedBw = cfgGuaranteedBw
   elif isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid,
               tacPercent.invalid, invalidShapeRate, cfgGuaranteedBw )
         return
      txQueueConfig.guaranteedBw = cfgGuaranteedBw
      setQosProfileConfig( profile )
   elif isinstance( mode, QosSchedulingPolicyMode ):
      policy = mode.qosSchedulingPolicyModeContext.currentEntry_
      policy.guaranteedBw = cfgGuaranteedBw
   elif isinstance( mode, IntfCli.IntfConfigMode ):
      intfList = getIntfListFromMode( mode )
      for intf in intfList:
         setIntfConfig( intf, cfgGuaranteedBw=cfgGuaranteedBw.bw,
                        cfgGuaranteedBwUnit=cfgGuaranteedBw.unit )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, cfgGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, cfgGuaranteedBw )
            else:
               txQueueConfig.guaranteedBw = cfgGuaranteedBw
         setIntfConfig( intf )

def noGuaranteedBw( mode, args ):
   guaranteedBw = tacGuaranteedBwVal.invalid
   cfgGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
   cfgGuaranteedBw.bw = guaranteedBw
   cfgGuaranteedBw.unit = tacGuaranteedBwUnit.guaranteedBwKbps
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return
      txQueueConfig.guaranteedBw = cfgGuaranteedBw
      setQosProfileConfig( profile )
   elif isinstance( mode, QosProfileMode ):
      profile = mode.qosProfileModeContext.currentEntry_
      profile.guaranteedBw = cfgGuaranteedBw
      setQosProfileConfig( profile )
   elif isinstance( mode, QosSchedulingPolicyMode ):
      policy = mode.qosSchedulingPolicyModeContext.currentEntry_
      policy.guaranteedBw = cfgGuaranteedBw
   elif isinstance( mode, IntfCli.IntfConfigMode ):
      intfList = getIntfListFromMode( mode )
      for intf in intfList:
         if not mode.intf.isSubIntf():
            continue
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if not intfConfig:
            return
         intfConfig.guaranteedBw = cfgGuaranteedBw
         setIntfConfig( intf )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is not None:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is not None:
               txQueueConfig.guaranteedBw = cfgGuaranteedBw
            setIntfConfig( intf )

#--------------------------------------------------------------------------------
# bandwidth guaranteed ( ( percent PERCENT ) | ( PPS pps ) | ( KBPS [ kbps ] ) )
# ( no | default ) bandwidth guaranteed ...
#--------------------------------------------------------------------------------
class BandwidthGuaranteedCmd( CliCommand.CliCommandClass ):
   syntax = ( 'bandwidth guaranteed ( ( percent PERCENT ) | '
              '( PPS pps ) | ( KBPS [ kbps ] ) )' )
   noOrDefaultSyntax = 'bandwidth guaranteed ...'
   data = {
      'bandwidth': nodeBandwidth,
      'guaranteed': nodeGuaranteed,
      'percent': 'Bandwidth Guaranteed in Percent',
      'PERCENT': matcherPercent,
      'PPS' : CliMatcher.DynamicIntegerMatcher(
         rangeFn=guaranteedBwPpsRangeFn, helpdesc='Bandwidth Guaranteed in Pps' ),
      'pps': CliCommand.guardedKeyword( 'pps',
         helpdesc='Bandwidth Guaranteed in Pps', guard=guardGuaranteedBwPps ),
      'KBPS': CliMatcher.DynamicIntegerMatcher(
         rangeFn=guaranteedBwKbpsRangeFn, helpdesc='Bandwidth Guaranteed in Kbps' ),
      'kbps': 'Bandwidth Guaranteed in Kbps (default unit)',
   }

   handler = setGuaranteedBw
   noOrDefaultHandler = noGuaranteedBw 

IntfTxQueueConfigMode.addCommandClass( BandwidthGuaranteedCmd )
SubIntfCli.SubIntfModelet.addCommandClass( BandwidthGuaranteedCmd )
QosProfileMode.addCommandClass( BandwidthGuaranteedCmd )

def randomDetectCmdsAdapter( mode, args, argsList ):
   if CliCommand.isNoOrDefaultCmd( args ):
      return
   minThd, maxThd, _ = args[ 'THD_RANGE' ]
   if minThd > maxThd:
      mode.addError( 'Minimum-threshold cannot be more than Maximum-threshold' )
      raise CliParserCommon.AlreadyHandledError()
  
bytesKwMatcher = CliMatcher.KeywordMatcher( 'bytes',
                                 helpdesc='Set threshold in bytes' )
kbytesKwMatcher = CliMatcher.KeywordMatcher( 'kbytes',
                                 helpdesc='Set threshold in kbytes' )
mbytesKwMatcher = CliMatcher.KeywordMatcher( 'mbytes',
                                 helpdesc='Set threshold in mbytes' )
bytesThdMatcher = CliMatcher.DynamicIntegerMatcher( 
                                 ucTxQueueEcnThresholdRangeBytesFn, 
                                 helpdesc='Threshold value (in bytes)' )
kbytesThdMatcher = CliMatcher.DynamicIntegerMatcher( 
                                 ucTxQueueEcnThresholdRangeKBytesFn, 
                                 helpdesc='Threshold value (in kbytes)' )
mbytesThdMatcher = CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeMBytesFn,
                                 helpdesc='Threshold value (in mbytes)' )
segmentsThdMatcher = CliMatcher.DynamicIntegerMatcher(
                                 ucTxQueueEcnThresholdRangeSegmentsFn,
                                 helpdesc='Threshold value (in segments)' )
millisecondsThdMatcher = CliMatcher.DynamicIntegerMatcher( 
                                 ucTxQueueEcnThresholdRangeMillisecondsFn, 
                                 helpdesc='Threshold value (in milliseconds)' )
microsecondsThdMatcher = CliMatcher.DynamicIntegerMatcher( 
                                 ucTxQueueEcnThresholdRangeMicrosecondsFn, 
                                 helpdesc='Threshold value (in microseconds)' )
minThdBytes = "MIN_VAL_BYTES"
maxThdBytes = "MAX_VAL_BYTES"
minThdKbytes = "MIN_VAL_KBYTES"
maxThdKbytes = "MAX_VAL_KBYTES"
minThdMbytes = "MIN_VAL_MBYTES"
maxThdMbytes = "MAX_VAL_MBYTES"
minThdSegments = "MIN_VAL_SEGMENTS"
maxThdSegments = "MAX_VAL_SEGMENTS"
minThdMilliseconds = "MIN_VAL_MILLISECONDS"
maxThdMilliseconds = "MAX_VAL_MILLISECONDS"
minThdMicroseconds = "MIN_VAL_MICROSECONDS"
maxThdMicroseconds = "MAX_VAL_MICROSECONDS"
thdsUnitsEnum = {
      'BYTES1' : [ minThdBytes, maxThdBytes ],
      'KBYTES1' : [ minThdKbytes, maxThdKbytes ],
      'MBYTES1' : [ minThdMbytes, maxThdMbytes ],
      'segments' : [ minThdSegments, maxThdSegments ], 
      'milliseconds' : [ minThdMilliseconds, maxThdMilliseconds ], 
      'microseconds' : [ minThdMicroseconds, maxThdMicroseconds ], 
}
 
class RandomDetectCmdsThdFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      class ThdExpr( CliCommand.CliExpression ):
         expression = '( %s BYTES1 maximum-threshold %s BYTES2 ) | '\
                      '( %s KBYTES1 maximum-threshold %s KBYTES2 ) | '\
                      '( %s MBYTES1  maximum-threshold %s MBYTES2 ) | '\
                      '( %s SEGMENTS_KW maximum-threshold %s segments ) |' % (
                      minThdBytes, maxThdBytes, minThdKbytes, maxThdKbytes, 
                      minThdMbytes, maxThdMbytes, minThdSegments, maxThdSegments )
         data = {
            'maximum-threshold': 'Set maximum threshold for WRED',
            'BYTES1': bytesKwMatcher,
            'BYTES2': bytesKwMatcher,
            minThdBytes: bytesThdMatcher,
            maxThdBytes: bytesThdMatcher,
            'KBYTES1': kbytesKwMatcher,
            'KBYTES2': kbytesKwMatcher,
            minThdKbytes: kbytesThdMatcher,
            maxThdKbytes: kbytesThdMatcher,
            'MBYTES1': mbytesKwMatcher,
            'MBYTES2': mbytesKwMatcher,
            minThdMbytes: mbytesThdMatcher,
            maxThdMbytes: mbytesThdMatcher,
            'SEGMENTS_KW': CliCommand.guardedKeyword( 'segments',
                                       helpdesc='Set threshold in segments', 
                                       guard=guardSegment ),
            'segments': 'Set threshold in segments', 
            minThdSegments: segmentsThdMatcher,
            maxThdSegments: segmentsThdMatcher,
         }
 
         @staticmethod
         def adapter( mode, args, argsList ):
            for unit, thds in thdsUnitsEnum.items():
               if unit not in args:
                  continue
               minThd = args.pop( thds[ 0 ] )
               maxThd = args.pop( thds[ 1 ] )
               unit = args.pop( unit )
               args[ 'THD_RANGE' ] = ( minThd, maxThd, unit )
               break
      return ThdExpr

class RandomDetectCmdsThdWithLatencyFactory( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      baseCmdInfo = RandomDetectCmdsThdFactory().generate( name )
      class ThdExpr( CliCommand.CliExpression ):
         expression = baseCmdInfo.expression + \
                      '( %s MILLISECONDS_KW maximum-threshold %s milliseconds ) | '\
                      '( %s MICROSECONDS_KW maximum-threshold %s microseconds ) ' % (
                      minThdMilliseconds, maxThdMilliseconds,
                      minThdMicroseconds, maxThdMicroseconds )
         data = baseCmdInfo.data
         data.update( { 
            'MILLISECONDS_KW': CliCommand.guardedKeyword( 'milliseconds',
                                       helpdesc='Set threshold in milliseconds',
                                       guard=guardLatency ),
            'milliseconds': 'Set threshold in milliseconds',
            minThdMilliseconds: millisecondsThdMatcher,
            maxThdMilliseconds: millisecondsThdMatcher,
            'MICROSECONDS_KW': CliCommand.guardedKeyword( 'microseconds',
                                       helpdesc='Set threshold in microseconds',
                                       guard=guardLatency ),
            'microseconds': 'Set threshold in microseconds',
            minThdMicroseconds: microsecondsThdMatcher,
            maxThdMicroseconds: microsecondsThdMatcher,
         } )

         adapter = baseCmdInfo.adapter
      return ThdExpr

#------------------------------------------------------------------------------------
# The "[ mls ] qos random-detect ecn  minimum-threshold <min_threashold>
# maximum-threshold <max_threshold>" command, in "uc-tx-queue" mode.
#------------------------------------------------------------------------------------
 
def setNoOrDefaultEcn( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return

      setTxQueueEcn( mode, txQueueConfig )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList: 
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            continue   
         txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
         if txQueueConfig is None:
            continue

         prevEcnConfig = txQueueConfig.ecnConfig

         setTxQueueEcn( mode, txQueueConfig )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevEcnConfig:
               txQueueConfig.ecnConfig = ( prevEcnConfig.minThd,
                                           prevEcnConfig.maxThd,
                                           prevEcnConfig.unit,
                                           prevEcnConfig.maxDroprate,
                                           prevEcnConfig.weight )
            else:
               txQueueConfig.ecnConfig = None
            setIntfConfig( intf )

def setEcn( mode, args ):                                                            
   maxMarkRate = args.get( 'MAX_DROPRATE', tacPercent.max )
   weight = args.get( 'WEIGHT', defaultWeight )
   minThd, maxThd, unit = args[ 'THD_RANGE' ]

   msg = "Cannot apply ECN configuration. WRED configuration is already present."
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      # ECN and WRED configs are mutual exclusive
      prevWredConfig = txQueueConfig.wredConfig
      if prevWredConfig:
         mode.addError( msg )
         return

      setTxQueueEcn( mode, txQueueConfig, False, minThd,
                     maxThd, unit, maxMarkRate, weight )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList: 
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw ) 

         prevEcnConfig = txQueueConfig.ecnConfig

         # ECN and WRED configs are mutual exclusive
         prevWredConfig = txQueueConfig.wredConfig
         if prevWredConfig:
            mode.addError( msg )
            continue
         setTxQueueEcn( mode, txQueueConfig, False, minThd,
                        maxThd, unit, maxMarkRate, weight )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevEcnConfig:
               txQueueConfig.ecnConfig = ( prevEcnConfig.minThd,
                                           prevEcnConfig.maxThd,
                                           prevEcnConfig.unit,
                                           prevEcnConfig.maxDroprate,
                                           prevEcnConfig.weight )
            else:
               txQueueConfig.ecnConfig = None
            setIntfConfig( intf )

def setTxQueueEcn( mode, txQueueConfig, noOrDefaultKw=True, minThd=None, maxThd=None,
                   ecnUnit=None, maxMarkRate=tacPercent.max, weight=defaultWeight ):
   prevEcnConfig = txQueueConfig.ecnConfig
   if noOrDefaultKw:
      if prevEcnConfig:
         txQueueConfig.ecnConfig = None
   else:
      if not ( prevEcnConfig and
               prevEcnConfig.minThd == minThd and
               prevEcnConfig.maxThd == maxThd and
               prevEcnConfig.unit == ecnUnit and
               prevEcnConfig.maxDroprate == maxMarkRate and
               prevEcnConfig.weight == weight ):
         txQueueConfig.ecnConfig = ( minThd, maxThd, ecnUnit,
                                     maxMarkRate, weight )
   return txQueueConfig

#------------------------------------------------------------------------------------
# The "[ mls ] qos random-detect non-ect minimum-threshold <min_threashold>
# maximum-threshold <max_threshold>" command, in "uc-tx-queue" mode.
#------------------------------------------------------------------------------------

def setNoOrDefaultNonEctParams( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return

      setTxQueueNonEct( mode, txQueueConfig )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            continue
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               continue

         prevNonEctConfig = txQueueConfig.nonEctConfig

         setTxQueueNonEct( mode, txQueueConfig )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn,
                             "NON-ECT parameters", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevNonEctConfig:
               txQueueConfig.nonEctConfig = ( prevNonEctConfig.minThd,
                                              prevNonEctConfig.maxThd,
                                              prevNonEctConfig.unit,
                                              ecnMaxDroprate,
                                              prevNonEctConfig.weight )
            else:
               txQueueConfig.nonEctConfig = None
            setIntfConfig( intf )

def setNonEctParams( mode, args ): 
   weight = args.get( 'WEIGHT', defaultWeight )
   minThd, maxThd, unit = args[ 'THD_RANGE' ]

   msg = "Cannot apply Non ECT configuration. WRED configuration is already present."
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember( mode.txQueue,
                         tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
                         invalidShapeRate, invalidGuaranteedBw )

      # NON-ECT and WRED configs are mutually exclusive
      prevWredConfig = txQueueConfig.wredConfig
      if prevWredConfig:
         mode.addError( msg )
         return
      setTxQueueNonEct( mode, txQueueConfig, False, minThd,
                        maxThd, unit, weight )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         # we need a timestamp to pass as an argument to cliBlocking fail
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )

         prevNonEctConfig = txQueueConfig.nonEctConfig

         # NONECT and WRED configs are mutual exclusive
         prevWredConfig = txQueueConfig.wredConfig
         if prevWredConfig:
            mode.addError( msg )
            continue
         setTxQueueNonEct( mode, txQueueConfig, False, minThd,
                           maxThd, unit, weight )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn,
                             "NON-ECT parameters", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevNonEctConfig:
               txQueueConfig.nonEctConfig = ( prevNonEctConfig.minThd,
                                              prevNonEctConfig.maxThd,
                                              prevNonEctConfig.unit,
                                              ecnMaxDroprate,
                                              prevNonEctConfig.weight )
            else:
               txQueueConfig.nonEctConfig = None
            setIntfConfig( intf )

def setTxQueueNonEct( mode, txQueueConfig, noOrDefaultKw=True,
                      minThd=None, maxThd=None,
                      unit=None, weight=defaultWeight ):
   prevNonEctConfig = txQueueConfig.nonEctConfig
   if noOrDefaultKw:
      if prevNonEctConfig:
         txQueueConfig.nonEctConfig = None
   else:
      warningMsg = "NON-ECT thresholds will not take effect " \
      "as ECN is not enabled on this tx-queue"
      if not ( prevNonEctConfig and
               prevNonEctConfig.minThd == minThd and
               prevNonEctConfig.maxThd == maxThd and
               prevNonEctConfig.unit == unit and
               prevNonEctConfig.weight == weight ):
         if txQueueConfig.ecnConfig is None:
            mode.addWarning( warningMsg )
         txQueueConfig.nonEctConfig = ( minThd, maxThd, unit,
                                        ecnMaxDroprate, weight )
   return txQueueConfig

#------------------------------------------------------------------------------------
# The "[ mls ] qos random-detect drop  minimum-threshold <min_threashold>
# maximum-threshold <max_threshold> drop-probability <maxDroprate> " 
# command, in "uc-tx-queue" mode.
#------------------------------------------------------------------------------------

def setNoOrDefaultWred( mode, args ):
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         return
      setTxQueueWred( mode, txQueueConfig )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList: 
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            continue
         txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
         if txQueueConfig is None:
            continue

         prevWredConfig = txQueueConfig.wredConfig

         setTxQueueWred( mode, txQueueConfig )
         setIntfConfig( intf )
      
         if cliBlockingFail( mode, timestamp, tacFeatureName.wred, "WRED", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember( 
                  mode.txQueue, tacTxQueuePriority.priorityInvalid, 
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevWredConfig:
               txQueueConfig.wredConfig = ( prevWredConfig.minThd,
                     prevWredConfig.maxThd, prevWredConfig.unit,
                     prevWredConfig.maxDroprate, prevWredConfig.defaultWeight )
            else:
               txQueueConfig.wredConfig = None
            setIntfConfig( intf )

def setWred( mode, args ):
   msg = "Cannot apply WRED configuration. ECN configuration is already present."
   weight = args.get( 'WEIGHT', defaultWeight )
   maxDroprate = args[ 'MAX_DROPRATE' ]
   minThd, maxThd, wredUnit = args[ 'THD_RANGE' ]

   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      # ECN/NON-ECT and WRED configs are mutual exclusive
      prevEcnConfig = txQueueConfig.ecnConfig
      prevGlobalEcnConfig = qosInputConfig.globalEcnConfig
      nonEctConfig = txQueueConfig.nonEctConfig
      if prevEcnConfig or prevGlobalEcnConfig or nonEctConfig:
         mode.addError( msg )
         return

      setTxQueueWred( mode, txQueueConfig, False , minThd,
                      maxThd, wredUnit, maxDroprate, weight )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList: 
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember( 
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               txQueueConfig = intfConfig.txQueueConfig.newMember( 
                  mode.txQueue, tacTxQueuePriority.priorityInvalid, 
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )  

         prevWredConfig = txQueueConfig.wredConfig

         # ECN/NON-ECT and WRED configs are mutual exclusive
         prevEcnConfig = txQueueConfig.ecnConfig
         prevGlobalEcnConfig = qosInputConfig.globalEcnConfig
         nonEctConfig = txQueueConfig.nonEctConfig
         if prevEcnConfig or prevGlobalEcnConfig or nonEctConfig:
            mode.addError( msg )
            continue

         setTxQueueWred( mode, txQueueConfig, False, minThd, 
                         maxThd, wredUnit, maxDroprate, weight )
         setIntfConfig( intf )
      
         if cliBlockingFail( mode, timestamp, tacFeatureName.wred, "WRED", intf ):
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember( 
                  mode.txQueue, tacTxQueuePriority.priorityInvalid, 
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            if prevWredConfig:
               txQueueConfig.wredConfig = ( prevWredConfig.minThd,
                     prevWredConfig.maxThd, prevWredConfig.unit,
                     prevWredConfig.maxDroprate, prevWredConfig.weight )
            else:
               txQueueConfig.wredConfig = None
            setIntfConfig( intf )

def setTxQueueWred( mode, txQueueConfig, noOrDefaultKw=True, minThd=None,
                    maxThd=None, wredUnit=None,
                    maxDroprate=100, weight=defaultWeight ):
   prevWredConfig = txQueueConfig.wredConfig
   if noOrDefaultKw:
      if prevWredConfig:
         txQueueConfig.wredConfig = None
   else:
      if not ( prevWredConfig and
               prevWredConfig.minThd == minThd and
               prevWredConfig.maxThd == maxThd and
               prevWredConfig.unit == wredUnit and
               prevWredConfig.maxDroprate == maxDroprate and
               prevWredConfig.weight == weight ):
         txQueueConfig.wredConfig = ( minThd, maxThd,
                                      wredUnit, maxDroprate, weight )
   return txQueueConfig

#------------------------------------------------------------------------------------
# "random-detect ecn count" command under tx-queue
#------------------------------------------------------------------------------------
def setEcnQueueCounterConfig( intf, mode, txQueueId, enable ):
   intfCounterCfg = qosInputConfig.ecnIntfCounterConfig.get( intf )
   if intfCounterCfg is None:
      if not enable:
         return
      intfCounterCfg = qosInputConfig.ecnIntfCounterConfig.newMember( intf )
   queueCounterCfg = intfCounterCfg.ecnTxQueueCounterConfig.get( txQueueId )
   if queueCounterCfg is None:
      if not enable:
         return
      queueCounterCfg = intfCounterCfg.ecnTxQueueCounterConfig.newMember( txQueueId )
   if enable == queueCounterCfg.counterEnable:
      return

   queueCounterCfg.counterEnable = enable
   
   timestamp = Tac.now()
   if enable:
      # Disable ecn count should always success, no need to block cli.
      if cliBlockingFail( mode, timestamp, tacFeatureName.ecnCount,
                          "ECN Count", intf ):
         queueCounterCfg.counterEnable = False
         enable = False
   if not enable:
      del intfCounterCfg.ecnTxQueueCounterConfig[ txQueueId ]
      if len( intfCounterCfg.ecnTxQueueCounterConfig ) == 0:
         del qosInputConfig.ecnIntfCounterConfig[ intf ]
      qosHwConfig.configVersionUpdateTime = timestamp

def setEcnQueueCount( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   intfList = getIntfListFromMode( mode.parent_ )
   for intf in intfList:
      setEcnQueueCounterConfig( intf, mode, mode.txQueue.id, not no )

nodeEcn = CliCommand.guardedKeyword( 'ecn',
      helpdesc='Set ECN parameters', guard=guardEcn )
nodeEcnQueueCount = CliCommand.guardedKeyword( 'count',
      helpdesc='Count ECN marked packets', guard=guardEcnQueueCounter )
nodeRandomDetect = CliCommand.guardedKeyword( 'random-detect',
      helpdesc='Set WRED based congestion control parameters (including ECN)',
      guard=guardTxQueueRandomDetect )

#------------------------------------------------------------------------------------
# "random-detect ecn count" command under tx-queue
#------------------------------------------------------------------------------------
class UcTxQueueModeRandDetectEcnCmd( CliCommand.CliCommandClass ):
   syntax = 'random-detect ecn count'
   noOrDefaultSyntax = syntax
   data = {
      'random-detect': nodeRandomDetect, 
      'ecn': nodeEcn,
      'count': nodeEcnQueueCount,
   }
   
   handler = setEcnQueueCount
   noOrDefaultHandler = handler

IntfUcTxQueueModelet.addCommandClass( UcTxQueueModeRandDetectEcnCmd )

#------------------------------------------------------------------------------------
# The "drop-precedence <1-2> drop-threshold percent <0-100>" command under tx-queue
#------------------------------------------------------------------------------------
def setTxQueueDropThresholds( mode, txQueueConfig, noOrDefaultKw=None, dp=None,
                              percent=None ):
   if noOrDefaultKw:
      del txQueueConfig.dropThresholds[ dp ]
   else:
      txQueueConfig.dropThresholds[ dp ] = percent

def setDropThresholds( mode, args ):
   dp = args[ 'DP' ]
   percent = args.get( 'PERCENT' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         if noOrDefaultKw:
            return
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      setTxQueueDropThresholds( mode, txQueueConfig, noOrDefaultKw, dp, percent )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if noOrDefaultKw:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if noOrDefaultKw:
                  continue
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )

         prevDropThresholds = txQueueConfig.dropThresholds
         setTxQueueDropThresholds( mode, txQueueConfig, noOrDefaultKw, dp, percent )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.ecn,
                             "TxQueue Drop Thresholds", intf ):
            intfConfig = qosInputConfig.intfConfig.get( intf )
            if intfConfig is None:
               intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            for dp in prevDropThresholds:
               txQueueConfig.dropThresholds[ dp ] = prevDropThresholds[ dp ]
            dpList = txQueueConfig.dropThresholds.keys()
            for dp in dpList:
               if dp not in prevDropThresholds:
                  del txQueueConfig.dropThresholds[ dp ]
            setIntfConfig( intf )

nodeDropPrecedence = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc='Set drop-precedence value', guard=guardTxQDropPrecedenceThreshold )
matcherDropPrecedenceValue = CliMatcher.IntegerMatcher(
      validDropPrecedenceValues[ 0 ], validDropPrecedenceValues[ -1 ],
      helpdesc='Drop precedence value' )
#------------------------------------------------------------------------------------
# "drop-precedence DP drop-threshold percent PERCENT" command under tx-queue
#------------------------------------------------------------------------------------
class IntfUcTxQueueDpCmd( CliCommand.CliCommandClass ):
   syntax = 'drop-precedence DP drop-threshold percent PERCENT'
   noOrDefaultSyntax = 'drop-precedence DP drop-threshold percent ...'
   data = {
      'drop-precedence':  nodeDropPrecedence,
      'DP': matcherDropPrecedenceValue,
      'drop-threshold': 'Set tail drop-threshold value',
      'percent': 'Set drop-threshold value in percent',
      'PERCENT': CliMatcher.IntegerMatcher( 0, tacPercent.max,
         helpdesc='Percent value between 0 and 100' )
   }

   handler = setDropThresholds
   noOrDefaultHandler = handler

IntfUcTxQueueModelet.addCommandClass( IntfUcTxQueueDpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] latency maximum LATENCY microseconds | milliseconds
#--------------------------------------------------------------------------------
def guardLatencyThreshold( mode, token ):
   if isSubIntfConfigMode( mode ):
      return CliParser.guardNotThisPlatform
   if qosHwStatus.latencyThresholdSupported:
      return None
   return CliParser.guardNotThisPlatform

nodeLatency = CliCommand.guardedKeyword( 'latency',
      helpdesc='Set latency parameters', guard=guardLatencyThreshold )

def handleTxQueueLatencyThreshold( mode, args ):   
   latency = args.get( 'LATENCY' )

   latencyThreshold = tacLatencyThreshold()

   # Convert to base unit of nanoseconds
   if args.get( 'microseconds' ):
      latencyThreshold.threshold = latency * 1000
      latencyThreshold.configUnit = 'microseconds'
   elif args.get( 'milliseconds' ):
      latencyThreshold.threshold = latency * 1000000
      latencyThreshold.configUnit = 'milliseconds'

   noOrDefault = latencyThreshold == tacLatencyThreshold()

   def initTxQueueConfig( parent ):
      return parent.txQueueConfig.newMember(
               mode.txQueue,
               tacTxQueuePriority.priorityInvalid,
               tacPercent.invalid,
               invalidShapeRate,
               invalidGuaranteedBw )

   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if noOrDefault and not txQueueConfig:
         return

      if not txQueueConfig:
         txQueueConfig = initTxQueueConfig( profile )

      txQueueConfig.latencyThreshold = latencyThreshold
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList:
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if noOrDefault:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = initTxQueueConfig( intfConfig )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if noOrDefault:
                  continue
               txQueueConfig = initTxQueueConfig( intfConfig )

         prevLatencyThreshold = txQueueConfig.latencyThreshold
         txQueueConfig.latencyThreshold = latencyThreshold
         setIntfConfig( intf )

         if cliBlockingFail( mode, Tac.now(), tacFeatureName.latencyThreshold,
                             "Latency threshold", intf ):
            txQueueConfig.latencyThreshold = prevLatencyThreshold
            setIntfConfig( intf )

class TxQueueLatencyThresholdMicrosecondsCmd( CliCommand.CliCommandClass ):
   syntax = 'latency maximum LATENCY microseconds'
   noOrDefaultSyntax = 'latency maximum ...'
   data = {
      'latency': nodeLatency, 
      'maximum': 'Set maximum latency',
      'LATENCY': CliMatcher.IntegerMatcher( 
                     tacLatencyThreshold().min, 
                     tacLatencyThreshold().maxMicroseconds(),
                     helpdesc='Latency value in microseconds' ),
      'microseconds': 'Latency value in microseconds',
   }
   handler = handleTxQueueLatencyThreshold
   noOrDefaultHandler = handleTxQueueLatencyThreshold

IntfUcTxQueueModelet.addCommandClass( TxQueueLatencyThresholdMicrosecondsCmd )
   
class TxQueueLatencyThresholdMillisecondsCmd( CliCommand.CliCommandClass ):
   syntax = 'latency maximum LATENCY milliseconds'
   data = {
      'latency': nodeLatency, 
      'maximum': 'Set maximum latency',
      'LATENCY': CliMatcher.IntegerMatcher( 
                     tacLatencyThreshold().min, 
                     tacLatencyThreshold().maxMilliseconds(),
                     helpdesc='Latency value in milliseconds' ),
      'milliseconds': 'Latency value in milliseconds',
   }
   handler = handleTxQueueLatencyThreshold

IntfUcTxQueueModelet.addCommandClass( TxQueueLatencyThresholdMillisecondsCmd )
   
#--------------------------------------------------------------------------------
# show qos interfaces INTERFACE latency maximum
#--------------------------------------------------------------------------------
def txQueueLatencyThresholdShowHandler( mode, args ):
   intfLatencyThresholdCollectionModel = IntfLatencyThresholdCollectionModel()

   intfs = IntfCli.Intf.getAll( mode, args.get( 'INTERFACE' ), None,
                                EthIntfCli.EthIntf )

   def genTxqLatencyThresholdModel( intf, txqId ):
      txqLatencyThresholdModel = TxQueueLatencyThresholdModel()
      txq = qosHwStatus.hwTxQueue[ txqId ].txQueue

      prefix = ''
      if qosHwStatus.numMulticastQueueSupported:
         prefix = txq.type[ :2 ].upper()
         if prefix != 'UC':
            return txqLatencyThresholdModel

      latencyThreshold = tacLatencyThreshold()
      txqLatencyThresholdModel.txQueue = prefix + str( txqId )
      if isLagPort( intf.name ):
         intfConfig = qosInputConfig.intfConfig.get( intf.name )
         if intfConfig:
            txQueueConfig = intfConfig.txQueueConfig.get( txq )
            if txQueueConfig:
               latencyThreshold = txQueueConfig.latencyThreshold
      else:
         intfStatus = qosStatus.intfStatus.get( intf.name )
         if intfStatus:
            txQueueStatus = intfStatus.txQueueStatus.get( txq )
            if txQueueStatus:
               latencyThreshold = txQueueStatus.latencyThreshold

      txqLatencyThresholdModel.threshold = latencyThreshold.configThreshold()
      txqLatencyThresholdModel.unit = latencyThreshold.configUnitShortName()

      return txqLatencyThresholdModel   

   interfaces = intfLatencyThresholdCollectionModel.interfaces
   for intf in intfs:
      if not intf.name.startswith( ( 'Ethernet', 'Port-Channel' ) ):
         continue

      intfLatencyThresholdModel = IntfLatencyThresholdModel()
      txqLatencyThresholdList = intfLatencyThresholdModel.txQueues
      for txqId in xrange( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
         txqLatencyThresholdList.append( genTxqLatencyThresholdModel( intf,
                                                                      txqId ) )
      interfaces[ intf.name ] = intfLatencyThresholdModel

   return intfLatencyThresholdCollectionModel

class TxQueueLatencyThresholdShowCmd( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos interfaces [ INTERFACE ] latency maximum"

   data = {
      'qos': nodeQosForShow,
      'interfaces' : 'Show QoS status for a specific interface',
      'INTERFACE': IntfRangeMatcher( explicitIntfTypes=(
                                        EthIntfCli.EthPhyAutoIntfType, 
                                        LagIntfCli.LagAutoIntfType ) ),
      'latency': CliCommand.guardedKeyword( 'latency', 
         helpdesc='Show latency parameters', guard=guardLatencyThreshold ),
      'maximum' : 'Maximum latency',
   }
   handler = txQueueLatencyThresholdShowHandler
   cliModel = IntfLatencyThresholdCollectionModel

BasicCli.addShowCommandClass( TxQueueLatencyThresholdShowCmd )

#------------------------------------------------------------------------------------
# The "show qos interfaces [ INTF ] ecn counters queue" command
#------------------------------------------------------------------------------------
def ecnCounterFromSmash( intfId, queueId ):
   key = EcnCounterKey( intfId, queueId )
   counter = ecnCounterTable.ecnCounter.get( key )
   snapshot = ecnSnapshotTable.ecnCounter.get( key )
   if counter is None:
      packetCount = 0
   elif snapshot is None:
      packetCount = counter.pkts
   else:
      packetCount = counter.pkts - snapshot.pkts
   return packetCount

def showEcnIntfQueueCounters( mode, args ):
   intf = args.get( 'INTF' )
   ecnIntfQueueCounterModel = EcnIntfQueueCountersModel()
   intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )
   intfNames = [ intf.name for intf in intfs ]
   for sliceHwStatus in qosSliceHwStatus.itervalues():
      for intfId, ecnCounterStatus in \
          sliceHwStatus.ecnIntfCounterHwStatus.iteritems():
         if intf and ( intfId not in intfNames ):
            continue
         queueCounterModel = EcnQueueCounterModel()
         numTxQueueIds = qosHwStatus.numTxQueueSupported
         if qosHwStatus.numMulticastQueueSupported > 0:
            numTxQueueIds = qosHwStatus.numUnicastQueueSupported
         for queueId in xrange( numTxQueueIds ):
            if ecnCounterStatus.counterEnabled.get( queueId ):
               if qosHwStatus.useEcnCountersSmash:
                  packetCount = ecnCounterFromSmash( intfId, queueId )
               else:
                  txCount = ecnCounterStatus.txQueuePktCount.get( queueId, 0 )
                  txCountClear = ecnCounterStatus.txQueuePktCountLastClear.get(
                     queueId, 0 )
                  packetCount = txCount - txCountClear
               queueCounterModel.insert( queueId, str( packetCount ) )
            else:
               queueCounterModel.insert( queueId, '-' )
         ecnIntfQueueCounterModel.insert( intfId, queueCounterModel )
   return ecnIntfQueueCounterModel

#------------------------------------------------------------------------------------
# The "show qos interfaces [ INTF ] wred counters queue" command
#------------------------------------------------------------------------------------
def showWredIntfQueueCounters( mode, args ):
   intf = args.get( 'INTF' )

   wredCounters = WredIntfQueueCountersModel()

   numUnguardedHooks = 0
   for func, guard in \
         showWredIntfQueueCountersHook.extensions():

      if guard( mode, None ) is None:
         wredCounters = func( mode, intf )
         numUnguardedHooks += 1

   assert numUnguardedHooks <= 1, "Found too many possible results"
   return wredCounters

#------------------------------------------------------------------------------------
#------------------------------------------------------------------------------------
# The "[ default|no ] [ mls ] qos map cos <0-7> to traffic-class <tc>" command,
# in "global config" mode.
#------------------------------------------------------------------------------------
def setCosToTc( mode, cosList, noOrDefaultKw=None, tc=None ):
    # Get the current cosToTcMap config
   prevCosToTcMap = dict( qosInputConfig.cosToTcMap )
   for cos in cosList:
      timestamp = Tac.now()
      if noOrDefaultKw or tc == qosHwStatus.defaultCosToTcMap[ cos ]:
         cosToTcVal = tacTrafficClass.invalid
      else:
         cosToTcVal = tc
      if prevCosToTcMap[ cos ] == cosToTcVal:
         continue
      qosInputConfig.cosToTcMap[ cos ] = cosToTcVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Cos to Tc Map" ):
         # If any of the cosToTc setting fails, revert the entire command
         for cos in prevCosToTcMap:
            qosInputConfig.cosToTcMap[ cos ] =  prevCosToTcMap[ cos ]
         break

#------------------------------------------------------------------------------------
# "qos map cos to traffic-class name <profilename>" to enter into
# CosToTcProfileMode
#------------------------------------------------------------------------------------

def getCurrentGlobalCosToTcMapping( cos ):
   # While creating a profile we copy from the current global mapping
   # if global mapping does not exist for the cos then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say cos 5 to tc 0 and
   # then goes and does a no cos 5 to tc 0 in the profile.
   tc = qosStatus.cosToTcMap[ cos ]
   if tc == tacTrafficClass.invalid:
      tc = qosHwStatus.defaultCosToTcMap[ cos ]
   return tc

def noCosToTcProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   del qosInputConfig.cosToTcProfile[ profileName ]

def configureCosToTcProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   if profileName not in qosInputConfig.cosToTcProfile:
      qosInputConfig.cosToTcProfile.newMember( profileName )
      for cos in range( 0, 8 ):
         # Copy from global/platform default mapping
         qosInputConfig.cosToTcProfile[ profileName ].cosToTcMap[
               cos ] = getCurrentGlobalCosToTcMapping( cos )
   childMode = mode.childMode( CosToTcProfileMode, param=profileName )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------------------
# "qos map dscp to traffic-class name <map_name>" to enter into
# DscpToTcMapMode
#------------------------------------------------------------------------------------

def getCurrentGlobalDscpToTcMapping( dscp ):
   # While creating a map we copy from the current global mapping
   # if global mapping does not exist for the dscp then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say dscp 5 to tc 0 and
   # then goes and does a no dscp 5 to tc 0 in the map.
   tc = qosStatus.dscpToTcMap[ dscp ]
   if tc == tacTrafficClass.invalid:
      tc = qosHwStatus.defaultDscpToTcMap[ dscp ]
   return tc

def removeDscpToTcMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   del qosInputConfig.dscpToTcNamedMap[ mapName ]

def createDscpToTcMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   if mapName not in qosInputConfig.dscpToTcNamedMap:
      qosInputConfig.dscpToTcNamedMap.newMember( mapName )
      for dscp in range( tacDscp.min, tacDscp.max + 1 ):
         # Copy from global/platform default mapping
         qosInputConfig.dscpToTcNamedMap[ mapName ].dscpToTcNamedMap[
               dscp ] = getCurrentGlobalDscpToTcMapping( dscp )
   childMode = mode.childMode( DscpToTcMode, param=mapName )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------------------
# The "[ default|no ] [ mls ] qos map dscp <0-7> to traffic-class <tc>
# [ drop-precedence <dp> ]" command,
# in "global config" mode.
#------------------------------------------------------------------------------------
def setDscpToTcDp( mode, args ):
   dscpList = args.get( 'DSCP_VALUE' ) or args.get( 'DSCP_VALUES' ).values()
   tc = args.get( 'TC_VALUE' )
   dp = args.get( 'DROP_PRECEDENCE' )
   no = CliCommand.isNoOrDefaultCmd( args )

   # Get the current dscpToTcMap config
   prevDscpToTcMap = dict( qosInputConfig.dscpToTcMap )
   prevDscpToDpMap = dict( qosInputConfig.dscpToDpMap )
   for dscp in dscpList:
      timestamp = Tac.now()
      if no or tc == qosHwStatus.defaultDscpToTcMap[ dscp ]:
         dscpToTcVal = tacTrafficClass.invalid
      else:
         dscpToTcVal = tc
      if not dp or no or dp == qosHwStatus.defaultDropPrecedence:
         dscpToDpVal = tacDropPrecedence.invalid
      else:
         dscpToDpVal = dp
      if prevDscpToTcMap[ dscp ] == dscpToTcVal and \
         prevDscpToDpMap[ dscp ] == dscpToDpVal:
         continue
      qosInputConfig.dscpToTcMap[ dscp ] = dscpToTcVal
      qosInputConfig.dscpToDpMap[ dscp ] = dscpToDpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Dscp to Tc/Dp Map" ):
         # If any of the dscpToTc setting fails, revert the entire command
         for dscp in prevDscpToTcMap:
            qosInputConfig.dscpToTcMap[ dscp ] = prevDscpToTcMap[ dscp ]
         for dscp in prevDscpToDpMap:
            qosInputConfig.dscpToDpMap[ dscp ] = prevDscpToDpMap[ dscp ]
         break

#------------------------------------------------------------------------------------
# The "[ default|no ] [ mls ] qos map exp <0-7> to traffic-class <tc>" command,
# in "global config" mode.
#------------------------------------------------------------------------------------
def guardExpToAndFromTcMap( mode, token ):
   if qosHwStatus.dynamicExpMappingSupported:
      return None
   return CliParser.guardNotThisPlatform

def setExpToTc( mode, expList, noOrDefaultKw=None, tc=None ):
   # Get the current expToTcMap config
   prevExpToTcMap = dict( qosInputConfig.expToTcMap )
   for exp in expList:
      timestamp = Tac.now()
      if noOrDefaultKw or tc == qosHwStatus.defaultExpToTcMap[ exp ]:
         expToTcVal = tacTrafficClass.invalid
      else:
         expToTcVal = tc
      if prevExpToTcMap[ exp ] == expToTcVal:
         continue
      qosInputConfig.expToTcMap[ exp ] = expToTcVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Exp to Tc Map" ):
         # If any of the expToTc settings fail, revert the entire command
         for exp in prevExpToTcMap:
            qosInputConfig.expToTcMap[ exp ] = prevExpToTcMap[ exp ]
         break

#------------------------------------------------------------------------------------
# The "qos map traffic-class <tc> to cos <0-7>" command, in "global config" mode.
#------------------------------------------------------------------------------------
def guardSetTctoCos( mode, token ):
   if qosHwStatus.cosRewriteSupported:
      return None
   return CliParser.guardNotThisPlatform

def setTrafficClassToCos( mode, tcList, noOrDefaultKw=None, cos=None ):
    # Get the current tcToCosMap config
   prevTcToCosMap = dict( qosInputConfig.tcToCosMap )
   for tc in tcList:
      timestamp = Tac.now()
      if noOrDefaultKw or cos == qosHwStatus.defaultTcToCosMap[ tc ]:
         tcToCosVal = tacCos.invalid
      else:
         tcToCosVal = cos
      if prevTcToCosMap[ tc ] == tcToCosVal:
         continue
      qosInputConfig.tcToCosMap[ tc ] = tcToCosVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "Tc to Cos Map" ):
         # If any of the tcToCos setting fails, revert the entire command
         for tc in prevTcToCosMap:
            qosInputConfig.tcToCosMap[ tc ] =  prevTcToCosMap[ tc ]
         break

# "qos map traffic-class to cos name <mapname>" to enter into
# TcToCosMapMode
#------------------------------------------------------------------------------------

def getCurrentGlobalTcDpToCosMapping( tc ):
   # While creating a profile we copy from the current global mapping
   # if global mapping does not exist for the cos then we fallback on the
   # platform default.
   # We also do this when user had configured a mapping say tc 5 to cos 0 and
   # then goes and does a no tc 5 to cos 0 in the profile.
   cos = qosStatus.tcToCosMap[ tc ]
   if cos == tacCos.invalid:
      cos = qosHwStatus.defaultTcToCosMap[ tc ]
   return cos

tokenTrafficClassHelpdesc = 'Mapping traffic-class values'
tcToCosMapNameRule = CliMatcher.DynamicNameMatcher(
         lambda mode:qosInputConfig.tcToCosNamedMap, 'Map Name',
         pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )
tokenTcToCosMapTo = CliCommand.guardedKeyword( 'to', 
      helpdesc="Configure a named Map", guard=guardAnyTcToCosPerInterface )
MapToCosHelpdesc = 'Map to Cos'

def createTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   if mapName not in qosInputConfig.tcToCosNamedMap: 
      qosInputConfig.tcToCosNamedMap.newMember( mapName )
      for tc in range( tacTrafficClass.min, tacTrafficClass.min +
                       qosHwStatus.numTcSupported ):
         for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
            tcDpPair = tacTcDpPair( tc, dp )
            qosInputConfig.tcToCosNamedMap[ mapName ].tcToCosNamedMap[
               tcDpPair ] = getCurrentGlobalTcDpToCosMapping( tc )

   childMode = mode.childMode( TcToCosMode, param=mapName )
   mode.session_.gotoChildMode( childMode )

def noOrDefaultCreateTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   del qosInputConfig.tcToCosNamedMap[ mapName ]

class TcToCosMapNameCmd( CliCommand.CliCommandClass ):
   syntax = "qos map traffic-class to cos name MAP_NAME"
   noOrDefaultSyntax = "qos map traffic-class to cos name MAP_NAME"
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'traffic-class': tokenTrafficClassHelpdesc,
      'to': tokenTcToCosMapTo,
      'cos': CliCommand.guardedKeyword( 'cos', 
         helpdesc=MapToCosHelpdesc, guard=guardAnyTcToCosPerInterface ),
      'name': 'Name of the map',
      'MAP_NAME': tcToCosMapNameRule
   }
   handler = createTcToCosMap
   noOrDefaultHandler = noOrDefaultCreateTcToCosMap

BasicCli.GlobalConfigMode.addCommandClass( TcToCosMapNameCmd )

#------------------------------------------------------------------------------------
# The "qos map traffic-class <tc> to dscp <0-63>" command, in "global config" mode.
#------------------------------------------------------------------------------------
def guardTcToDscpMap( mode, token ):
   if qosHwStatus.tcToDscpMapSupported:
      return None
   return CliParser.guardNotThisPlatform

nodeToDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc="Set Packet DSCP value",
      guard=guardTcToDscpMap )

def setTrafficClassToDscp( mode, tcList, noOrDefaultKw=None, dscp=None ):
   # Get the current tcToDscpMap config
   prevTcToDscpMap = dict( qosInputConfig.tcToDscpMap )
   for tc in tcList:
      timestamp = Tac.now()
      if noOrDefaultKw or dscp == qosHwStatus.defaultTcToDscpMap[ tc ]:
         tcToDscpVal = tacDscp.invalid
      else:
         tcToDscpVal = dscp
      if prevTcToDscpMap[ tc ] == tcToDscpVal:
         continue
      qosInputConfig.tcToDscpMap[ tc ] = tcToDscpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "TC to DSCP Map" ):
         # If any of the tcToDscp setting fails, revert the entire command
         for tc in prevTcToDscpMap:
            qosInputConfig.tcToDscpMap[ tc ] =  prevTcToDscpMap[ tc ]
         break

#-----------------------------------------------------------------------------------
# "qos map traffic-class to dscp <mapname>" to enter into DscpRewriteConfigMode
#-----------------------------------------------------------------------------------
def configureDscpRewriteMap( mode, args ):
   mapName = args.get( 'MAPNAME' )
   if CliCommand.isNoOrDefaultCmd( args ):
      del qosInputConfig.dscpRewriteMap[ mapName ]
   else:
      if ( len( qosInputConfig.dscpRewriteMap ) >=
           qosHwStatus.numDscpRewriteMapsSupported and
           not mode.session_.startupConfig() ):
         if mapName not in qosInputConfig.dscpRewriteMap:
            mode.addError( "This platform supports only %d unique DscpRewriteMaps to"
                           " be created. Please delete an existing map before "
                           "creating a new one " %
                           qosHwStatus.numDscpRewriteMapsSupported )
            return
      qosInputConfig.dscpRewriteMap.newMember( mapName )
      childMode = mode.childMode( DscpRewriteMode, param=mapName )
      mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------
# Qos Profile Infrastructure
#------------------------------------------------------------------------
def identicalEcnOrWredConfig( config1, config2 ):
   if ( not config1 and config2 ) or \
          ( not config2 and config1 ):
      return False
   if config1 and ( config1.minThd != config2.minThd or
                    config1.maxThd != config2.maxThd or
                    config1.unit != config2.unit or
                    config1.maxDroprate != config2.maxDroprate or
                    config1.weight != config2.weight ):
      return False
   return True

def identicalDropThresholds( config1, config2 ):
   if config1.keys() != config2.keys():
      return False
   for dp in config1:
      if config1[ dp ] != config2[ dp ]:
         return False
   return True

def identicalQosProfile( profile, currentEntry ):
   if not profile or not currentEntry:
      return False
   if profile.defaultCos != currentEntry.defaultCos or \
          profile.defaultDscp != currentEntry.defaultDscp or \
          profile.trustMode != currentEntry.trustMode or \
          profile.shapeRate != currentEntry.shapeRate or \
          profile.guaranteedBw != currentEntry.guaranteedBw:
      return False
   if sorted( currentEntry.txQueueConfig ) != \
      sorted( profile.txQueueConfig ):
      return False
   for txQueue in currentEntry.txQueueConfig:
      txQConfig = currentEntry.txQueueConfig[ txQueue ]
      txQueueConfig = profile.txQueueConfig.get( txQueue )
      ecnConfig = txQConfig.ecnConfig
      wredConfig = txQConfig.wredConfig
      ecnConfig2 = txQueueConfig.ecnConfig
      wredConfig2 = txQueueConfig.wredConfig
      nonEctConfig = txQConfig.nonEctConfig
      nonEctConfig2 = txQueueConfig.nonEctConfig
      dropThresholds = txQConfig.dropThresholds
      dropThresholds2 = txQueueConfig.dropThresholds
      if txQueueConfig.guaranteedBw != txQConfig.guaranteedBw or \
             txQueueConfig.shapeRate != txQConfig.shapeRate or \
             txQueueConfig.bandwidth != txQConfig.bandwidth or \
             txQueueConfig.priority != txQConfig.priority or \
             not identicalDropThresholds( dropThresholds, dropThresholds2 ) or \
             not identicalEcnOrWredConfig( ecnConfig, ecnConfig2 ) or \
             not identicalEcnOrWredConfig( wredConfig, wredConfig2 ) or \
             not identicalEcnOrWredConfig( nonEctConfig, nonEctConfig2 ):
         return False
   if currentEntry.pfcPortConfig and not profile.pfcPortConfig:
      return False
   if not currentEntry.pfcPortConfig and profile.pfcPortConfig:
      return False
   if currentEntry.pfcPortConfig and \
          ( currentEntry.pfcPortConfig.enabled != profile.pfcPortConfig.enabled or
            currentEntry.pfcPortConfig.priorities !=
            profile.pfcPortConfig.priorities or
            currentEntry.pfcPortConfig.watchdogEnabled !=
            profile.pfcPortConfig.watchdogEnabled or
            currentEntry.pfcPortConfig.portTimerConfig !=
            profile.pfcPortConfig.portTimerConfig or
            currentEntry.pfcPortConfig.watchdogPortAction !=
            profile.pfcPortConfig.watchdogPortAction ):
      return False
   if currentEntry.fabricPfcDlb != profile.fabricPfcDlb:
      return False
   if currentEntry.servicePolicyConfig and not profile.servicePolicyConfig:
      return False
   if not currentEntry.servicePolicyConfig and profile.servicePolicyConfig:
      return False
   if currentEntry.servicePolicyConfig:
      key1 = currentEntry.servicePolicyConfig.key
      key2 = profile.servicePolicyConfig.key
      if key1.pmapName != key2.pmapName or key1.direction != key2.direction or \
             key1.type != key2.type:
         return False
   return True

class QosProfileModeContext( object ):
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.editedEntries_ = {} 
      self.currentEntry_ = None
      self.previousEntry_ = None # used for config rollback
      qosProfileConfig = profileConfigDir.config
      if profileName in qosProfileConfig:
         self.profile_ = qosProfileConfig[ profileName ].qosProfile
         self.profileName_ = profileName
         prevQosProfile =  Tac.newInstance( 'Qos::QosProfile', 
                                             self.profileName_ )
         copyQosProfile( prevQosProfile, self.profile_ )
         self.previousEntry_ = prevQosProfile

   def copyEditEntry( self ):
      newQosProfile =  Tac.newInstance( 'Qos::QosProfile', self.profileName_ )
      copyQosProfile( newQosProfile, self.profile_ )
      self.currentEntry_ = newQosProfile
      return newQosProfile

   def newEditEntry( self ):
      newQosProfile = Tac.newInstance( 'Qos::QosProfile', self.profileName_ )
      self.currentEntry_ = newQosProfile

   def profileName( self ):
      return self.profileName_

   def currentEntry( self ):
      return self.currentEntry_

   def qosProfile( self ):
      return self.profile_

   def commit( self ):
      # Commit current profile
      qosProfileConfig = profileConfigDir.config
      intfToProfileMap = profileConfigDir.intfToProfileMap
      # ModifyBitString contains the information about which parts of the qos/config
      # the qos profile modifies. This is depicted by the different bits being set.
      modifyBitString = 0
      if identicalQosProfile( self.profile_, self.currentEntry_ ):
         return
      if self.profile_ == None:
         profileConfig = qosProfileConfig.newMember( self.profileName_ )
         self.profile_ = profileConfig.qosProfile
      profileConfig = qosProfileConfig.get( self.profileName_ )
      profileConfig.qosProfile = ( self.profileName_, )
      self.profile_ = profileConfig.qosProfile
      for txQueue in self.currentEntry_.txQueueConfig:
         txQConfig = self.currentEntry_.txQueueConfig[ txQueue ]
         txQueueConfig = self.profile_.txQueueConfig.get( txQueue )
         ecnConfig = txQConfig.ecnConfig
         wredConfig = txQConfig.wredConfig
         nonEctConfig = txQConfig.nonEctConfig
         dropThresholds = txQConfig.dropThresholds
         if not txQueueConfig:
            txQueueConfig = self.profile_.txQueueConfig.newMember( 
               txQueue, txQConfig.priority, txQConfig.bandwidth,
               txQConfig.shapeRate, txQConfig.guaranteedBw )
            modifyBitString |= MODIFY_TXQCONFIG
         else:
            if txQueueConfig.guaranteedBw != txQConfig.guaranteedBw or \
                   txQueueConfig.shapeRate != txQConfig.shapeRate or \
                   txQueueConfig.bandwidth != txQConfig.bandwidth or \
                   txQueueConfig.priority != txQConfig.priority:
               modifyBitString |= MODIFY_TXQCONFIG
            txQueueConfig.guaranteedBw = txQConfig.guaranteedBw
            txQueueConfig.shapeRate = txQConfig.shapeRate
            txQueueConfig.bandwidth = txQConfig.bandwidth
            txQueueConfig.priority = txQConfig.priority

         if not identicalEcnOrWredConfig( txQueueConfig.ecnConfig, ecnConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_ECN_TXQCONFIG
         if ecnConfig:
            txQueueConfig.ecnConfig = ( ecnConfig.minThd, ecnConfig.maxThd,
                                        ecnConfig.unit, ecnConfig.maxDroprate,
                                        ecnConfig.weight )
         else:
            txQueueConfig.ecnConfig = None

         if txQueueConfig.delayEcnEnabled != txQConfig.delayEcnEnabled or \
            txQueueConfig.ecnDelayThreshold != txQConfig.ecnDelayThreshold:
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_ECN_DELAY_TXQCONFIG
            txQueueConfig.delayEcnEnabled = txQConfig.delayEcnEnabled
            txQueueConfig.ecnDelayThreshold = txQConfig.ecnDelayThreshold

         if not identicalEcnOrWredConfig( txQueueConfig.wredConfig, wredConfig ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_WRED_TXQCONFIG
            modifyBitString |= MODIFY_NONECT_TXQCONFIG
         if wredConfig:
            txQueueConfig.wredConfig = ( wredConfig.minThd, wredConfig.maxThd,
                                         wredConfig.unit, wredConfig.maxDroprate,
                                         wredConfig.weight )
         else:
            txQueueConfig.wredConfig = None

         if not identicalEcnOrWredConfig( txQueueConfig.nonEctConfig, nonEctConfig ):
            if nonEctConfig:
               txQueueConfig.nonEctConfig = ( nonEctConfig.minThd,
                                              nonEctConfig.maxThd,
                                              nonEctConfig.unit,
                                              nonEctConfig.maxDroprate,
                                              nonEctConfig.weight )
            else:
               txQueueConfig.nonEctConfig = None

         if not identicalDropThresholds( txQueueConfig.dropThresholds,
                                         dropThresholds ):
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_DROP_THRESHOLDS
         for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
            if dp in dropThresholds:
               txQueueConfig.dropThresholds[ dp ] = dropThresholds[ dp ]
            elif dp in txQueueConfig.dropThresholds:
               del txQueueConfig.dropThresholds[ dp ]

         if txQueueConfig.latencyThreshold != txQConfig.latencyThreshold:
            modifyBitString |= MODIFY_TXQCONFIG
            modifyBitString |= MODIFY_LATENCY_THRESHOLD
            txQueueConfig.latencyThreshold = txQConfig.latencyThreshold

      for txQueue in self.profile_.txQueueConfig:
         if txQueue not in self.currentEntry_.txQueueConfig:
            del self.profile_.txQueueConfig[ txQueue ]
            modifyBitString |= MODIFY_TXQCONFIG
      if self.currentEntry_.pfcPortConfig:
         if not self.profile_.pfcPortConfig:
            self.profile_.pfcPortConfig = ( "QosProfile",
               self.currentEntry_.pfcPortConfig.enabled, )
            modifyBitString |= MODIFY_PFC
         if self.profile_.pfcPortConfig.enabled != \
                self.currentEntry_.pfcPortConfig.enabled or \
                self.profile_.pfcPortConfig.priorities != \
                self.currentEntry_.pfcPortConfig.priorities or \
                self.profile_.pfcPortConfig.watchdogEnabled != \
                self.currentEntry_.pfcPortConfig.watchdogEnabled or \
                self.profile_.pfcPortConfig.portTimerConfig != \
                self.currentEntry_.pfcPortConfig.portTimerConfig or \
                self.profile_.pfcPortConfig.watchdogPortAction != \
                self.currentEntry_.pfcPortConfig.watchdogPortAction :
            modifyBitString |= MODIFY_PFC
         self.profile_.pfcPortConfig.enabled = \
               self.currentEntry_.pfcPortConfig.enabled
         self.profile_.pfcPortConfig.priorities = \
               self.currentEntry_.pfcPortConfig.priorities
         self.profile_.pfcPortConfig.watchdogEnabled = \
               self.currentEntry_.pfcPortConfig.watchdogEnabled
         self.profile_.pfcPortConfig.portTimerConfig = \
               self.currentEntry_.pfcPortConfig.portTimerConfig
         self.profile_.pfcPortConfig.watchdogPortAction = \
               self.currentEntry_.pfcPortConfig.watchdogPortAction
         self.profile_.fabricPfcDlb = \
               self.currentEntry_.fabricPfcDlb         
      elif self.profile_.pfcPortConfig and not self.currentEntry_.pfcPortConfig:
         self.profile_.pfcPortConfig = None
         self.profile_.fabricPfcDlb = 0
         modifyBitString |= MODIFY_PFC
      if self.profile_.defaultCos != self.currentEntry_.defaultCos:
         modifyBitString |= MODIFY_DEF_COS
         self.profile_.defaultCos = self.currentEntry_.defaultCos
      if self.profile_.defaultDscp != self.currentEntry_.defaultDscp:
         modifyBitString |= MODIFY_DEF_DSCP
         self.profile_.defaultDscp = self.currentEntry_.defaultDscp
      if self.profile_.trustMode != self.currentEntry_.trustMode:
         modifyBitString |= MODIFY_TRUST_MODE
         self.profile_.trustMode = self.currentEntry_.trustMode
      if self.profile_.shapeRate != self.currentEntry_.shapeRate:
         modifyBitString |= MODIFY_SHAPE_RATE
         self.profile_.shapeRate = self.currentEntry_.shapeRate
      if self.profile_.guaranteedBw != self.currentEntry_.guaranteedBw:
         modifyBitString |= MODIFY_GUARANTEED_BW
         self.profile_.guaranteedBw = self.currentEntry_.guaranteedBw
      if self.currentEntry_.servicePolicyConfig:
         key = self.currentEntry_.servicePolicyConfig.key
         self.profile_.servicePolicyConfig = ( key, )
      elif self.profile_.servicePolicyConfig:
         self.profile_.servicePolicyConfig = None
      if os.environ.get( "TEST_QOS_PROFILE_VERSION" ):
         if os.environ.get( "QOS_PROFILE_VERSION" ):
            os.environ[ "QOS_PROFILE_VERSION" ] = \
                str( int( os.environ[ "QOS_PROFILE_VERSION" ] ) + 1 )
         else:
            os.environ[ "QOS_PROFILE_VERSION" ] = '1'
      for intf in intfToProfileMap:
         if intfToProfileMap[ intf ] == self.profileName_:
            k = applyProfile( mode=self.mode, profileName=self.profileName_, 
                              interfaceName=intf,
                              checkCliBlocking=True,
                              modifyBitString=modifyBitString )
            if k == ROLLBACK_PROFILE:
               self.currentEntry_ = self.previousEntry_
               self.commit()

def copyQosProfile( newProfile, oldProfile ):
   for txQueue in oldProfile.txQueueConfig:
      oldTxQConfig = oldProfile.txQueueConfig[ txQueue ]
      newTxQConfig = newProfile.txQueueConfig.newMember(
            txQueue, oldTxQConfig.priority, oldTxQConfig.bandwidth,
            oldTxQConfig.shapeRate, oldTxQConfig.guaranteedBw )
      ecnConfig = oldTxQConfig.ecnConfig
      if ecnConfig:
         newTxQConfig.ecnConfig = ( ecnConfig.minThd, ecnConfig.maxThd,
                                    ecnConfig.unit, ecnConfig.maxDroprate,
                                    ecnConfig.weight )
      newTxQConfig.delayEcnEnabled = oldTxQConfig.delayEcnEnabled
      newTxQConfig.ecnDelayThreshold = oldTxQConfig.ecnDelayThreshold
      wredConfig = oldTxQConfig.wredConfig
      if wredConfig:
         newTxQConfig.wredConfig = ( wredConfig.minThd, wredConfig.maxThd,
                                     wredConfig.unit, wredConfig.maxDroprate,
                                     wredConfig.weight )
      nonEctConfig = oldTxQConfig.nonEctConfig
      if nonEctConfig:
         newTxQConfig.nonEctConfig = ( nonEctConfig.minThd, nonEctConfig.maxThd,
                                       nonEctConfig.unit, nonEctConfig.maxDroprate,
                                       nonEctConfig.weight )
      dropThresholds = oldTxQConfig.dropThresholds
      for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
         if dp in dropThresholds:
            newTxQConfig.dropThresholds[ dp ] = dropThresholds[ dp ]
         elif dp in newTxQConfig.dropThresholds:
            del newTxQConfig.dropThresholds[ dp ]
      newTxQConfig.latencyThreshold = oldTxQConfig.latencyThreshold      

   if oldProfile.pfcPortConfig:
      newProfile.pfcPortConfig = ( "QosProfile",
                                    oldProfile.pfcPortConfig.enabled, )
      newProfile.pfcPortConfig.priorities = oldProfile.pfcPortConfig.priorities
      newProfile.pfcPortConfig.watchdogEnabled = oldProfile.pfcPortConfig.\
          watchdogEnabled
      newProfile.pfcPortConfig.portTimerConfig = oldProfile.pfcPortConfig.\
                                                 portTimerConfig
      newProfile.pfcPortConfig.watchdogPortAction = oldProfile.pfcPortConfig.\
         watchdogPortAction
      newProfile.fabricPfcDlb = oldProfile.fabricPfcDlb
   newProfile.defaultCos = oldProfile.defaultCos
   newProfile.defaultDscp = oldProfile.defaultDscp
   newProfile.trustMode = oldProfile.trustMode   
   newProfile.shapeRate = oldProfile.shapeRate
   newProfile.guaranteedBw = oldProfile.guaranteedBw
   if oldProfile.servicePolicyConfig:
      key = oldProfile.servicePolicyConfig.key
      newProfile.servicePolicyConfig = ( key, )

class DscpRewriteMode( DscpRewriteModeBase, BasicCli.ConfigModeBase ):
   name = "Dscp Rewrite Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      DscpRewriteModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class DscpToTcMode( DscpToTcModeBase, BasicCli.ConfigModeBase ):
   name = "Dscp To Traffic-Class Mapping Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      DscpToTcModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class CosToTcProfileMode( CosToTcProfileModeBase, BasicCli.ConfigModeBase ):
   name = "Cos To Traffic-Class Profile Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      CosToTcProfileModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class TcToCosMode( TcToCosModeBase, BasicCli.ConfigModeBase ):
   name = "Traffic-Class to Cos Mapping Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      TcToCosModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def gotoQosProfileMode( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   qosProfileConfig = profileConfigDir.config
   context = QosProfileModeContext( mode, profileName )
   if profileName in qosProfileConfig:
      context.copyEditEntry()
   else:
      context.newEditEntry()

   mode.qosProfileModeContext = context
   childMode = mode.childMode( QosProfileMode, context=context )
   mode.session_.gotoChildMode( childMode )

def qosProfiles( mode ):
   qosProfileConfig = profileConfigDir.config
   return qosProfileConfig

#-------------------------------------------------------------------------------
# Tokens for Qos Profile Infrastructure.
#-------------------------------------------------------------------------------
def showQosProfileActive( mode ):
   mode.showActive()

def _packetDropOrEcnConfigToPrintable( configurationType, dropOrEcnConfig ):
   unit = dropOrEcnConfig.unit
   dropOrEcnConfigPrintable = 'minimum-threshold %d %s maximum-threshold %d %s' % (
                              dropOrEcnConfig.minThd, unit,
                              dropOrEcnConfig.maxThd, unit )
   if qosHwStatus.ecnMarkProbSupported:
      if configurationType == "ecn":
         dropOrEcnConfigPrintable += ( " max-mark-probability %d" %
                       ( dropOrEcnConfig.maxDroprate ) )
   if configurationType == "wred":
      dropOrEcnConfigPrintable += ( " drop-probability %d" %
                       ( dropOrEcnConfig.maxDroprate ) )
   if( ( configurationType == "ecn" and  qosHwStatus.ecnWeightSupported ) or
       ( configurationType == "nonEct" and  qosHwStatus.nonEctWeightSupported ) or
       ( configurationType == "wred" and  qosHwStatus.wredWeightSupported ) ):
      dropOrEcnConfigPrintable += ' weight ' + str( dropOrEcnConfig.weight )
   return dropOrEcnConfigPrintable

def _txQueueRender( txQueueConfig, output=None ):
   if not output:
      output = sys.stdout
   if txQueueConfig == None:
      return
   if txQueueConfig.priority == tacTxQueuePriority.priorityRoundRobin:
      output.write( "%6sno priority\n" % ''  )
   if txQueueConfig.bandwidth != tacPercent.invalid:
      output.write( "%6s%s %d\n" % ( '', "bandwidth percent",
         txQueueConfig.bandwidth ) )
   if ( txQueueConfig.guaranteedBw and
        txQueueConfig.guaranteedBw.bw != tacGuaranteedBwVal.invalid ):
      output.write( "%6sbandwidth guaranteed " % '' )
      guaranteedBwUnit = QosLib.guaranteedBwUnitFromEnum(
         txQueueConfig.guaranteedBw.unit )
      if guaranteedBwUnit.lower() == 'percent':
         output.write(  "%s %d\n" % (
            guaranteedBwUnit,
            txQueueConfig.guaranteedBw.bw ) )
      else:
         if guaranteedBwUnit != 'kbps':
            output.write( "%d %s\n" % (
               txQueueConfig.guaranteedBw.bw,
               guaranteedBwUnit ) )
         else:
            output.write( "%d\n" % txQueueConfig.guaranteedBw.bw )
   if ( txQueueConfig.shapeRate and
        txQueueConfig.shapeRate.rate != tacShapeRateVal.invalid ):
      if txQueueConfig.shapeRate.percent != tacPercent.invalid:
         shapeRateUnit = 'percent'
      else:   
         shapeRateUnit = QosLib.shapeRateUnitFromEnum( txQueueConfig.shapeRate.unit )

      if shapeRateUnit == 'kbps':
         shapeRateUnit = ''
      else:
         shapeRateUnit = ' ' + shapeRateUnit
      output.write( "%6sshape rate %d%s\n" %
                    ( '', txQueueConfig.shapeRate.rate,
                      shapeRateUnit ) )

   if qosHwStatus.ecnSupported and txQueueConfig.ecnConfig:
      output.write( "%6srandom-detect ecn %s\n" %
                    ( '', _packetDropOrEcnConfigToPrintable( "ecn",
                       txQueueConfig.ecnConfig ) ) )

   if qosHwStatus.nonEctThdSupported and txQueueConfig.nonEctConfig:
      output.write( "%6srandom-detect non-ect %s\n"
                    % ( '', _packetDropOrEcnConfigToPrintable( "nonEct",
                       txQueueConfig.nonEctConfig ) ) )

   if ( txQueueConfig.delayEcnEnabled and
        txQueueConfig.ecnDelayThreshold is not None ):
      thresholdConfig = txQueueConfig.ecnDelayThreshold
      minThresh = thresholdConfig.minEcnDelayThreshold().threshold
      if thresholdConfig.unit == 'ecnDelayThresholdInvalidUnit':
         output.write( '%6secn delay\n' % '' )
      elif thresholdConfig.threshold >= minThresh:
         maxThresh = thresholdConfig.maxEcnDelayThreshold().threshold
         if thresholdConfig.threshold <= maxThresh:
            #changing the value to microseconds
            val = thresholdConfig.threshold / 1000
            output.write( "%6secn delay threshold %d microseconds\n" % (
               '', val ) )

   if qosHwStatus.wredSupported and txQueueConfig.wredConfig:
      output.write( "%6srandom-detect drop %s\n"
                    % ( '', _packetDropOrEcnConfigToPrintable( "wred",
                       txQueueConfig.wredConfig ) ) )

   if qosHwStatus.dropPrecedenceThresholdSupported and \
      len( txQueueConfig.dropThresholds ) != 0 :
      for dp in txQueueConfig.dropThresholds:
         output.write( "%6sdrop-precedence %s drop-threshold percent %s\n"
                       % ( '', dp, txQueueConfig.dropThresholds[ dp ] ) )

   if qosHwStatus.latencyThresholdSupported and \
      txQueueConfig.latencyThreshold != tacLatencyThreshold():
      output.write( "%6slatency maximum %d %s\n" % \
                    ( '', txQueueConfig.latencyThreshold.configThreshold(),
                    txQueueConfig.latencyThreshold.configUnit) )

def  _pfcPortConfigPrioritiesPrintable( pfcPortConfig, fabricPfcDlb, output=None ):
   if not output:
      output = sys.stdout
   noDropList = pfcPortConfig.priorities
   dlbEnabledList = fabricPfcDlb
   for noDropId in range ( 16 ):
      bit = 1 << noDropId
      if bit & dlbEnabledList:
         output.write( "%3spriority-flow-control priority %d no-drop dlb\n" % (
           '', noDropId ) )
      elif bit & noDropList:
         output.write( "%3spriority-flow-control priority %d no-drop\n" % (
            '', noDropId ) )

def _convertPortTimerConfigPrintable( pfcPortConfig, output=None ):
   if not output:
      output = sys.stdout
   portTimerConfig = pfcPortConfig.portTimerConfig
   cmdString = 'priority-flow-control pause watchdog port timer'
   timeoutString = ' timeout %2.2f' % ( portTimerConfig.portWatchdogTimeout )
   cmdString = cmdString + timeoutString
   pollingIntervalValue = "auto" if not \
                          portTimerConfig.portWatchdogPollingInterval else \
                          "%.3f" % ( portTimerConfig.portWatchdogPollingInterval )
   pollingIntervalString = " polling-interval %s" % ( pollingIntervalValue )
   cmdString = cmdString + pollingIntervalString
   recoveryCfg = portTimerConfig.portWatchdogRecoveryCfg
   recoveryTimeValue = "0" if not recoveryCfg.recoveryTime else \
                       "%2.2f" % ( recoveryCfg.recoveryTime )
   if recoveryCfg.forcedRecovery:
      recoveryTimeValue = recoveryTimeValue + " forced"
   recoveryTimeString = " recovery-time %s" % ( recoveryTimeValue )
   cmdString = cmdString + recoveryTimeString
   output.write( "%3s%s\n" % ( '', cmdString ) )
   
def _convertQosProfileToPrintable( currentQosProfile, output=None ):
   import cStringIO
   if output == None:
      output = sys.stdout
   if currentQosProfile == None:
      return
   output.write( "qos profile %s \n" % ( currentQosProfile.name ) )
   if  currentQosProfile.trustMode != tacTrustMode.invalid:
      if currentQosProfile.trustMode != tacTrustMode.untrusted :
         output.write( "%3sqos trust %s\n" % ( '', currentQosProfile.trustMode ) )
      else:
         output.write( "%3sno qos trust\n" % '' )
   if ( currentQosProfile.defaultCos != qosHwStatus.defaultCos and
        currentQosProfile.defaultCos != tacCos.invalid ):
      output.write( "%3sqos cos %d\n" % ( '', currentQosProfile.defaultCos ) )
   if ( currentQosProfile.defaultDscp != qosHwStatus.defaultDscp and
        currentQosProfile.defaultDscp != tacDscp.invalid ):
      output.write( "%3sqos dscp %d\n"
                    % ( '', currentQosProfile.defaultDscp ) )
   shape = currentQosProfile.shapeRate
   if shape and shape.rate != tacShapeRateVal.invalid:
      output.write( "%3sshape rate %d" % ( '', shape.rate ) )
      if shape.percent != tacPercent.invalid:
         output.write( ' percent' )
      elif QosLib.shapeRateUnitFromEnum( shape.unit ) != 'kbps':
         output.write( ' %s' % QosLib.shapeRateUnitFromEnum( shape.unit ) )
      if shape.shared:
         output.write( ' shared' )
      output.write( '\n' )
   if subIntfHwStatus.subIntfBandwidthSupported:
      gbw = currentQosProfile.guaranteedBw
      if gbw and gbw.bw != tacGuaranteedBwVal.invalid:
         if gbw.unit == tacGuaranteedBwUnit.guaranteedBwPercent:
            output.write( '%3sbandwidth guaranteed percent %d\n' % ( '', gbw.bw ) )
         elif gbw.unit == tacGuaranteedBwUnit.guaranteedBwPps:
            output.write( '%3sbandwidth guaranteed %d pps\n' % ( '', gbw.bw ) )
         else:
            output.write( '%3sbandwidth guaranteed %d\n' % ( '', gbw.bw ) )
   pfcPortConfig = currentQosProfile.pfcPortConfig
   if pfcPortConfig:
      if pfcPortConfig.enabled:
         output.write( "%3spriority-flow-control on\n" % '' )
      if not pfcPortConfig.watchdogEnabled:
         output.write( "%3sno priority-flow-control pause watchdog\n" % '' )
      if pfcPortConfig.priorities:
         _pfcPortConfigPrioritiesPrintable( pfcPortConfig,
            currentQosProfile.fabricPfcDlb, output )
      if pfcPortConfig.portTimerConfig.usePerPortTimerValues:
         _convertPortTimerConfigPrintable( pfcPortConfig, output )
      if pfcPortConfig.watchdogPortAction != tacPfcWatchdogAction.invalid:
         output.write( "%3spriority-flow-control pause watchdog port action %s\n" \
                       % ( pfcPortConfig.watchdogPortAction ) )
   servicePolicyConfig = currentQosProfile.servicePolicyConfig
   if servicePolicyConfig:
      output.write( "%3sservice-policy type qos %s %s\n" % ( '',
          servicePolicyConfig.key.direction,
          servicePolicyConfig.key.pmapName ) )
   if qosHwStatus.numTxQueueSupported:
      for txQueue in sorted( currentQosProfile.txQueueConfig ):
         txQueueRenderedOutput = cStringIO.StringIO()
         txQueueConfig = currentQosProfile.txQueueConfig[ txQueue ]
         _txQueueRender( txQueueConfig, txQueueRenderedOutput )
         if len( txQueueRenderedOutput.getvalue() ):
            output.write( "%3s" % '' )
            if txQueue.type == 'mcq':
               output.write( "mc-tx-queue " )
            elif txQueue.type == 'ucq':
               output.write( "uc-tx-queue " )
            else:
               output.write( "tx-queue " )
            output.write( "%d\n" % txQueue.id )
            output.write( txQueueRenderedOutput.getvalue() )

def showQosProfilePending( mode ):
   profile = mode.qosProfileModeContext.currentEntry_
   _convertQosProfileToPrintable( profile )

def showQosProfileDiff( mode ):
   import difflib, cStringIO
   activeOutput1 = cStringIO.StringIO()
   currentQosProfileConfig = mode.qosProfileModeContext.profile_
   pendingQosProfileConfig = mode.qosProfileModeContext.currentEntry_
   _convertQosProfileToPrintable(
      currentQosProfileConfig,
      output=activeOutput1 )
   activeOutput2 = cStringIO.StringIO()
   _convertQosProfileToPrintable(
      pendingQosProfileConfig,
      output=activeOutput2 )
   diff = difflib.unified_diff( activeOutput1.getvalue().splitlines( ),
                                activeOutput2.getvalue().splitlines( ),
                                lineterm='' )
   sys.stdout.write( '\n'.join( list( diff ) ) )
   sys.stdout.write( '\n' )

def deleteQosProfile( mode, args ):
   profileName = args[ 'PROFILENAME' ]
   qosProfileConfig = profileConfigDir.config
   intfToProfileMap = profileConfigDir.intfToProfileMap
   if not profileName in qosProfileConfig:
      return
   if qosInputProfileConfig.servicePolicyConfig:
      key = qosInputProfileConfig.servicePolicyConfig.keys()[ 0 ]
   for intf in intfToProfileMap:
      if intfToProfileMap[ intf ] != profileName:
         continue
      if qosInputProfileConfig.servicePolicyConfig:
         del qosInputProfileConfig.servicePolicyConfig[ key ].intfIds[ intf ]
         if len( qosInputProfileConfig.servicePolicyConfig[ key ].intfIds ) == 0:
            del qosInputProfileConfig.servicePolicyConfig[ key ]
      intfConfig = qosInputProfileConfig.intfConfig.get( intf )
      if not intfConfig:
         continue
      del qosInputProfileConfig.intfConfig[ intf ]
   del qosProfileConfig[ profileName ]

def removeProfile( mode, profileName, intfName ):
   intfToProfileMap = profileConfigDir.intfToProfileMap
   # return when non-existent profile is removed
   if intfName not in intfToProfileMap:
      return
   if mode and profileName and intfToProfileMap[ intfName ] != profileName:
      mode.addWarning( '%s not attached to interface %s' % ( profileName,
                                                             intfName ) )
      return
   if not profileName:
      profileName = intfToProfileMap[ intfName ]
   if intfName in qosInputProfileConfig.intfConfig:
      intfConfig = qosInputProfileConfig.intfConfig.get( intfName )
      if intfConfig.txQueueConfig:
         intfConfig.txQueueConfig.clear()
      intfConfig.pfcPortConfig = None
      intfConfig.defaultDscp = tacDscp.invalid
      intfConfig.defaultCos = tacCos.invalid
      intfConfig.trustMode = tacTrustMode.invalid
      intfConfig.shapeRate = Tac.Value( 'Qos::ShapeRate' )
      del qosInputProfileConfig.intfConfig[ intfName ]
   if qosInputProfileConfig.servicePolicyConfig:
      for key in qosInputProfileConfig.servicePolicyConfig:
         spConfig = qosInputProfileConfig.servicePolicyConfig[ key ]
         if intfName in spConfig.intfIds:
            delServicePolicy( mode, key.type, key.pmapName,
                              key.direction, intfName, profile=True )
   del intfToProfileMap[ intfName ]

def applyPrevProfileOrRollBack( profileName, prevProfile, intfName ):
   if prevProfile != profileName:
      applyProfile( profileName=prevProfile,
                    interfaceName=intfName,
                    checkCliBlocking=False )
   else:
      return ROLLBACK_PROFILE
      
def applyProfile( mode=None, profileName=None, noOrDefaultKw=None, 
                  interfaceName=None, checkCliBlocking=True,
                  modifyBitString=MODIFY_ALL ):
   intfList = []
   intfName = None
   prevProfile = None
   qosProfileConfig = profileConfigDir.config
   intfToProfileMap = profileConfigDir.intfToProfileMap
   if os.environ.get( 'SKIP_CLI_BLOCK_CHECK' ):
      checkCliBlocking = False
   # This function is called through the cli in IntfConfigMode
   # and Fabric Mode. It can alos be called from the commit()
   # function in QosProfileModeContext. In IntfConfigMode, we 
   # can get the applied intfs as mode.intf, while in Fabric mode,
   # we iterate over fabricIntfConfigDir to get the applied intfs.
   # When called from commit(), the intfName is passed as an argument
   applicableModes = [ IntfCli.IntfConfigMode, FabricMode ]
   if mode:
      if type( mode ) == IntfCli.IntfConfigMode: 
         intfList = [ mode.intf ] 
      elif type( mode ) == FabricMode:          
         for intf in fabricIntfConfigDir:
            intfList.append( fabricIntfConfigDir[ intf ] ) 
      elif interfaceName:
         intfList = [ interfaceName ]         
   else:
      intfList = [ interfaceName ]
   if noOrDefaultKw:
      for intf in intfList:
         intfName = getIntfIdentity( intf )
         removeProfile( mode, profileName, intfName )
      return
   if profileName in qosProfileConfig:
      for intf in intfList:
         intfName = getIntfIdentity( intf )
         if mode and type( mode ) in applicableModes and \
                 intfName in intfToProfileMap and \
                 intfToProfileMap[ intfName ] == profileName :
            mode.addWarning( "%s already applied on %s" % ( profileName,
                                                            intfName ) )            
            continue
         for qosProfileName in qosProfileConfig:
            if intfName in intfToProfileMap and \
                  intfToProfileMap[ intfName ] == qosProfileName:
               prevProfile = qosProfileName
         if prevProfile != profileName:
            removeProfile( mode, None, intfName )
         intfConfig = qosInputProfileConfig.intfConfig.get( intfName )
         shapeWarning = True
         timestamp = Tac.now()
         profile = qosProfileConfig[ profileName ].qosProfile
         if profile.defaultCos != tacCos.invalid or \
                profile.defaultDscp != tacDscp.invalid or \
                profile.trustMode != tacTrustMode.invalid or \
                profile.shapeRate.rate != tacShapeRateVal.invalid or \
                profile.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
            if not intfConfig:
               intfConfig = qosInputProfileConfig.intfConfig.newMember( intfName )
               modifyBitString = MODIFY_ALL
            if modifyBitString & MODIFY_DEF_COS:
               intfConfig.defaultCos = profile.defaultCos
               if mode and checkCliBlocking and intfName != fabricIntfName and \
                      cliBlockingFail( mode, timestamp, tacFeatureName.defaultCos,
                                       "Default Cos", intfName ):
                  if intfName in qosInputProfileConfig.intfConfig:
                     del qosInputProfileConfig.intfConfig[ intfName ]
                  if prevProfile:
                     return applyPrevProfileOrRollBack( profileName, prevProfile,
                                                        intfName )
                  return
            if modifyBitString & MODIFY_DEF_DSCP:
               intfConfig.defaultDscp = profile.defaultDscp
               if mode and checkCliBlocking and intfName != fabricIntfName and \
                      cliBlockingFail( mode, timestamp, tacFeatureName.defaultDscp,
                                       "Default Dscp", intfName ):
                  if intfName in qosInputProfileConfig.intfConfig:
                     del qosInputProfileConfig.intfConfig[ intfName ]
                  if prevProfile:
                     return applyPrevProfileOrRollBack( profileName, prevProfile,
                                                        intfName )
                  return
            if modifyBitString & MODIFY_TRUST_MODE:
               intfConfig.trustMode = profile.trustMode
               if mode and checkCliBlocking and intfName != fabricIntfName and \
                      cliBlockingFail( mode, timestamp, tacFeatureName.trustMode,
                                       "Trust Mode", intfName ):
                  if intfName in qosInputProfileConfig.intfConfig:
                     del qosInputProfileConfig.intfConfig[ intfName ]
                  if prevProfile:
                     return applyPrevProfileOrRollBack( profileName, prevProfile,
                                                        intfName )
                  return
            if modifyBitString & MODIFY_SHAPE_RATE:
               intfConfig.shapeRate = profile.shapeRate
            if modifyBitString & MODIFY_GUARANTEED_BW:
               intfConfig.guaranteedBw = profile.guaranteedBw
         elif intfConfig:
            if profile.defaultCos == tacCos.invalid and \
                   modifyBitString & MODIFY_DEF_COS:
               intfConfig.defaultCos = profile.defaultCos
            if profile.defaultDscp == tacDscp.invalid and \
                   modifyBitString & MODIFY_DEF_DSCP:
               intfConfig.defaultDscp = profile.defaultDscp
            if profile.trustMode == tacTrustMode.invalid and \
                   modifyBitString & MODIFY_TRUST_MODE:
               intfConfig.trustMode = profile.trustMode
            if profile.shapeRate.rate == tacShapeRateVal.invalid and \
                   modifyBitString & MODIFY_SHAPE_RATE:
               intfConfig.shapeRate = profile.shapeRate
            if subIntfHwStatus.subIntfBandwidthSupported and \
                   profile.guaranteedBw.bw == tacGuaranteedBwVal.invalid and \
                   modifyBitString & MODIFY_GUARANTEED_BW:
               intfConfig.guaranteedBw = profile.guaranteedBw
         if modifyBitString & MODIFY_TXQCONFIG:
            if intfConfig:
               for txQueue in intfConfig.txQueueConfig:
                  if txQueue not in profile.txQueueConfig:
                     del intfConfig.txQueueConfig[ txQueue ]

            for txQueue in profile.txQueueConfig:
               timestamp = Tac.now()
               profileTxQConfig = profile.txQueueConfig[ txQueue ]
               if intfName == fabricIntfName:
                  if mode and type( mode ) in applicableModes and \
                         profileTxQConfig.shapeRate != invalidShapeRate and \
                         shapeWarning:
                     mode.addWarning( "Tx-Queue shape " + 
                                      "not supported on fabric interface" )
                     shapeWarning = False
                  if profileTxQConfig.guaranteedBw == invalidGuaranteedBw and \
                         not profileTxQConfig.ecnConfig and \
                         not profileTxQConfig.delayEcnEnabled and \
                         not profileTxQConfig.wredConfig and \
                         not profileTxQConfig.nonEctConfig and \
                         len( profileTxQConfig.dropThresholds ) == 0 and \
                         profileTxQConfig.priority == \
                         tacTxQueuePriority.priorityInvalid and \
                         profileTxQConfig.bandwidth == tacPercent.invalid:
                     if intfConfig and txQueue in intfConfig.txQueueConfig:
                        del intfConfig.txQueueConfig[ txQueue ]
               if not intfConfig:
                  intfConfig = qosInputProfileConfig.intfConfig.newMember( intfName )

               # prepare for setting qos/config for the following
               # - tx-q priority / bw percent
               # - tx-q guaranteed bandwidth
               # - tx-q ecn configuration
               # - tx-q wred configuration
               # - tx-q nonect configuration

               # priority / bw percent
               priority = profileTxQConfig.priority
               bwPercent = profileTxQConfig.bandwidth

               # guaranteed bandwidth
               guaranteedBw = profileTxQConfig.guaranteedBw
               if intfName != fabricIntfName:
                  shapeRate = profileTxQConfig.shapeRate
               else:
                  shapeRate = invalidShapeRate

               intfTxQConfig = intfConfig.txQueueConfig.get( txQueue )
               if not intfTxQConfig:
                  intfTxQConfig = intfConfig.txQueueConfig.newMember(
                     txQueue, priority, bwPercent,
                     shapeRate, guaranteedBw )
               else:
                  intfTxQConfig.priority = priority
                  intfTxQConfig.bandwidth = bwPercent
                  intfTxQConfig.shapeRate = shapeRate
                  intfTxQConfig.guaranteedBw = guaranteedBw

               # ecn configuration
               ecnConfig = profileTxQConfig.ecnConfig
               if modifyBitString & MODIFY_ECN_TXQCONFIG:
                  if ecnConfig:
                     intfTxQConfig.ecnConfig = ( ecnConfig.minThd, ecnConfig.maxThd,
                                                 ecnConfig.unit, ecnMaxDroprate,
                                                 ecnConfig.weight )
                     if mode and checkCliBlocking and cliBlockingFail( mode,
                                     timestamp, tacFeatureName.ecn, "Ecn" ):
                        if intfName in qosInputProfileConfig.intfConfig:
                           del qosInputProfileConfig.intfConfig[ intfName ]
                        if prevProfile:
                           return applyPrevProfileOrRollBack( profileName,
                                                              prevProfile, intfName )
                        return
                  elif intfTxQConfig and intfTxQConfig.ecnConfig:
                     intfTxQConfig.ecnConfig = None

               if modifyBitString & MODIFY_ECN_DELAY_TXQCONFIG:
                  intfTxQConfig.delayEcnEnabled = profileTxQConfig.delayEcnEnabled
                  intfTxQConfig.ecnDelayThreshold = \
                        profileTxQConfig.ecnDelayThreshold

               # wred configuration
               wredConfig = profileTxQConfig.wredConfig
               if modifyBitString & MODIFY_WRED_TXQCONFIG:
                  if wredConfig:
                     intfTxQConfig.wredConfig = ( wredConfig.minThd,
                                                  wredConfig.maxThd,
                                                  wredConfig.unit,
                                                  wredConfig.maxDroprate,
                                                  wredConfig.weight )
                     if mode and checkCliBlocking and cliBlockingFail( mode,
                                    timestamp, tacFeatureName.wred, "Wred" ):
                        if intfName in qosInputProfileConfig.intfConfig:
                           del qosInputProfileConfig.intfConfig[ intfName ]
                        if prevProfile:
                           return applyPrevProfileOrRollBack( profileName,
                                                              prevProfile, intfName )
                        return
                  elif intfTxQConfig and intfTxQConfig.wredConfig:
                     intfTxQConfig.wredConfig = None

               # nonect configuration
               nonEctConfig = profileTxQConfig.nonEctConfig
               if modifyBitString & MODIFY_NONECT_TXQCONFIG:
                  if nonEctConfig:
                     intfTxQConfig.nonEctConfig = ( nonEctConfig.minThd,
                                                    nonEctConfig.maxThd,
                                                    nonEctConfig.unit,
                                                    nonEctConfig.maxDroprate,
                                                    nonEctConfig.weight )
                     if mode and checkCliBlocking and cliBlockingFail( mode,
                                    timestamp, tacFeatureName.ecn, "Non-Ect" ):
                        if intfName in qosInputProfileConfig.intfConfig:
                           del qosInputProfileConfig.intfConfig[ intfName ]
                        if prevProfile:
                           return applyPrevProfileOrRollBack( profileName,
                                                              prevProfile, intfName )
                        return
                  elif intfTxQConfig and intfTxQConfig.nonEctConfig:
                     intfTxQConfig.nonEctConfig = None

               # dropThreshold configuration
               dropThresholds = profileTxQConfig.dropThresholds
               if modifyBitString & MODIFY_DROP_THRESHOLDS:
                  if len( dropThresholds ) != 0:
                     for dp in range( tacDropPrecedence.min, tacDropPrecedence.max ):
                        if dp in dropThresholds:
                           intfTxQConfig.dropThresholds[ dp ] = dropThresholds[ dp ]
                        elif dp in intfTxQConfig.dropThresholds:
                           del intfTxQConfig.dropThresholds[ dp ]
                     if mode and checkCliBlocking and cliBlockingFail(
                           mode, timestamp, tacFeatureName.dropThresholds,
                           "Drop-Thresholds" ):
                        if intfName in qosInputProfileConfig.intfConfig:
                           del qosInputProfileConfig.intfConfig[ intfName ]
                        if prevProfile:
                           return applyPrevProfileOrRollBack( profileName,
                                                              prevProfile, intfName )
                        return
                  elif intfTxQConfig:
                     intfTxQConfig.dropThresholds.clear()

               # Latency threshold configuration
               if modifyBitString & MODIFY_LATENCY_THRESHOLD:
                  latencyThreshold = profileTxQConfig.latencyThreshold
                  intfTxQConfig.latencyThreshold = latencyThreshold
                  if latencyThreshold != tacLatencyThreshold():
                     failed = False
                     if mode and checkCliBlocking:
                        failed = cliBlockingFail( mode, 
                                                  timestamp,
                                                  tacFeatureName.latencyThreshold,
                                                  "Latency Threshold" )
                     if failed:                             
                        del qosInputProfileConfig.intfConfig[ intfName ]
                        if prevProfile:
                           return applyPrevProfileOrRollBack( profileName,
                                                              prevProfile, 
                                                              intfName )
                        return

         if modifyBitString & MODIFY_PFC:
            if profile.pfcPortConfig:
               if isLagPort( intfName ):
                  mode.addWarning( "PFC configuration is not supported on " \
                                   "Port-Channel" )
               else:
                  if not intfConfig:
                     intfConfig = qosInputProfileConfig.intfConfig.newMember( 
                        intfName )
                  else:
                     intfConfig = qosInputProfileConfig.intfConfig[ intfName ]
                  if not intfConfig.pfcPortConfig:
                     intfConfig.pfcPortConfig = \
                         ( intfName, profile.pfcPortConfig.enabled, )
                  pfcPortConfig = intfConfig.pfcPortConfig
                  pfcPortConfig.enabled = profile.pfcPortConfig.enabled
                  pfcPortConfig.priorities = profile.pfcPortConfig.priorities
                  pfcPortConfig.watchdogEnabled = \
                     profile.pfcPortConfig.watchdogEnabled
                  pfcPortConfig.portTimerConfig = \
                     profile.pfcPortConfig.portTimerConfig
                  pfcPortConfig.watchdogPortAction = \
                     profile.pfcPortConfig.watchdogPortAction
                  intfConfig.fabricPfcDlb = profile.fabricPfcDlb
            elif intfConfig and intfConfig.pfcPortConfig:
               intfConfig.pfcPortConfig = None
               intfConfig.fabricPfcDlb = 0
         if profile.servicePolicyConfig:
            if intfName == fabricIntfName:
               mode.addWarning( "Service Policy is not supported on Fabric" )
            else:
               key = profile.servicePolicyConfig.key
               mapType = mapTypeFromEnum( key.type )
               direction = directionFromEnum( key.direction )
               pmapName = key.pmapName
               rollBackProfile = setServicePolicy( mode, None, mapType, pmapName,
                                                   direction, intfName, profile=True)
               if mode and checkCliBlocking and rollBackProfile == ROLLBACK_PROFILE:
                  if intfName in qosInputProfileConfig.intfConfig:
                     del qosInputProfileConfig.intfConfig[ intfName ]
                  if key in qosInputProfileConfig.servicePolicyConfig and \
                         intfName in qosInputProfileConfig.servicePolicyConfig[ 
                     key ].intfIds:
                     del qosInputProfileConfig.servicePolicyConfig[ key ].\
                         intfIds[ intfName ]
                     if len( qosInputProfileConfig.servicePolicyConfig[ key ].\
                                intfIds ) == 0:
                        del qosInputProfileConfig.servicePolicyConfig[ key ]
                  if prevProfile:
                     return applyPrevProfileOrRollBack( profileName,
                                                        prevProfile, intfName )
                  return
         else:
            for spKey in qosInputProfileConfig.servicePolicyConfig:
               spConfig = qosInputProfileConfig.servicePolicyConfig[ spKey ]
               if intfName in spConfig.intfIds:
                  delServicePolicy( None, spKey.type, spKey.pmapName,
                                    spKey.direction, intfName, profile=True )
         intfToProfileMap[ intfName ] = profileName
         setIntfConfig( intfName, profile=True )
   else:
      for intf in intfList:
         intfName = getIntfIdentity( intf )
         removeProfile( None, None, intfName )
         intfToProfileMap[ intfName ] = profileName

matcherServiceProfile = CliMatcher.KeywordMatcher( 'service-profile',
      helpdesc='Configure QoS profile' )
matcherQosProfileName = CliMatcher.DynamicNameMatcher( qosProfiles,
      helpdesc='QoS profile name', pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

#------------------------------------------------------------------------
# service-profile PROFILE
#------------------------------------------------------------------------
class ServiceProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'service-profile PROFILE'
   data = {
      'service-profile': matcherServiceProfile,
      'PROFILE': matcherQosProfileName
   }

   @staticmethod
   def handler( mode, args ):
      applyProfile( mode, profileName=args[ 'PROFILE' ] )

FabricMode.addCommandClass( ServiceProfileCmd )
QosModelet.addCommandClass( ServiceProfileCmd )
QosSubIntfModelet.addCommandClass( ServiceProfileCmd )

#------------------------------------------------------------------------
# ( no | default ) service-profile [ PROFILE ]
#------------------------------------------------------------------------
class ServiceProfileNoDefCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'service-profile [ PROFILE ]'
   data = {
      'service-profile': matcherServiceProfile,
      'PROFILE': matcherQosProfileName
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      applyProfile( mode, profileName=args.get( 'PROFILE' ), noOrDefaultKw=True )

FabricMode.addCommandClass( ServiceProfileNoDefCmd )
IntfCli.IntfConfigMode.addCommandClass ( ServiceProfileNoDefCmd )

def showQosProfileSummary( mode, args ):
   qosProfileName = args.get( 'PROFILENAME' )
   qosProfileConfig = profileConfigDir.config
   intfToProfileMap = profileConfigDir.intfToProfileMap
   qosProfileSummaryModel = QosProfileSummaryModel()   
   profileNames = [ ]
   if qosProfileName:
      if qosProfileName in qosProfileConfig:
         profileNames = [ qosProfileName ]
   else:
      profileNames = qosProfileConfig.keys()
   for key in profileNames:
      intfList = [ ]
      for intf in intfToProfileMap:
         intfToProfileMapKey = intfToProfileMap.get( intf )
         if intfToProfileMapKey == key:
            if intf == fabricIntfName:
               intf = fabricShortName
            intfList.append( intf )
      qosProfileSummaryModel.qosProfileApplication[ key ] = \
          QosProfileConfiguredIntfs( intfs=intfList )

   return qosProfileSummaryModel

matcherProfile = CliMatcher.KeywordMatcher( 'profile', helpdesc='Show QoS profile' )
profileNameMatcher = CliMatcher.DynamicNameMatcher(
      qosProfiles, "QoS profile name",
      pattern=r'(?!summary$)[A-Za-z0-9_:{}\[\]-]+' )

#--------------------------------------------------------------------------------
# show qos profile [ PROFILENAME ]
#--------------------------------------------------------------------------------
class QosProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos profile [ PROFILENAME ]'
   data = {
      'qos' : nodeQosForShow,
      'profile' : matcherProfile,
      'PROFILENAME': profileNameMatcher,
      }

   handler = QosProfileMode.showQosProfile
   cliModel = QosProfileAllModel

BasicCli.addShowCommandClass( QosProfileCmd )

#--------------------------------------------------------------------------------
# show qos profile [ PROFILENAME ] summary
#--------------------------------------------------------------------------------
class QosProfileSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show qos profile [ PROFILENAME ] summary'
   data = {
      'qos': nodeQosForShow,
      'profile': matcherProfile,
      'PROFILENAME': profileNameMatcher,
      'summary': 'Qos profile summary',
   }

   handler = showQosProfileSummary
   cliModel = QosProfileSummaryModel

BasicCli.addShowCommandClass( QosProfileSummaryCmd )

#------------------------------------------------------------------------
# show qos map traffic-class to dscp [ MAP ] 
#------------------------------------------------------------------------
def showDscpRewriteMaps( mode, args ):
   mapName = args.get( 'MAP' )
   dscpRewriteMapAllModel = DscpRewriteMapAllModel()
   if mapName:
      if mapName in qosConfig.dscpRewriteMap:
         mapNames = [ mapName ]
      else:
         mapNames = []
   else:
      mapNames = qosConfig.dscpRewriteMap.keys()
   for mapName in mapNames:
      dscpRewriteMapModel = DscpRewriteMapModel()
      tcToDscpMap = qosConfig.dscpRewriteMap[ mapName ].tcToDscpMap
      for tc in tcToDscpMap:
         if tcToDscpMap[ tc ] != tacDscp.invalid:
            dscpRewriteMapModel.tcToDscpMap[ tc ] = tcToDscpMap[ tc ]
      dscpRewriteMapAllModel.dscpRewriteMaps[ mapName ] = dscpRewriteMapModel
   return dscpRewriteMapAllModel

#------------------------------------------------------------------------
# show qos map cos to traffic-class <name>
#------------------------------------------------------------------------
def showCosToTcProfile( mode, args ):
   profileName = args.get( 'NAME', None )
   cosToTcProfileAllModel = CosToTcProfileAllModel()
   if profileName:
      if profileName in qosStatus.cosToTcProfile:
         profileNames = [ profileName ]
      else:
         profileNames = []
   else:
      profileNames = qosStatus.cosToTcProfile
   for profileName in profileNames:
      cosToTcProfileModel = CosToTcProfileModel()
      cosToTcMap = qosStatus.cosToTcProfile[ profileName ].cosToTcMap
      for cos in cosToTcMap:
         cosToTcProfileModel.cosToTcMap[ cos ] = cosToTcMap[ cos ]
         cosToTcProfileAllModel.cosToTcProfiles[ profileName ] = cosToTcProfileModel
   return cosToTcProfileAllModel

class ShowCosToTcProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map cos to traffic-class [ NAME ]"

   data = {
      'qos': nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'cos': CliCommand.guardedKeyword( 'cos', 
         helpdesc='Show CoS To Traffic-Class map configuration', 
         guard=guardCosToTcPerInterface ),
      'to': 'Map to Traffic-Class',
      'traffic-class': 'Map to Traffic-Class',
      'NAME': CliMatcher.DynamicNameMatcher(
         lambda mode:qosInputConfig.cosToTcProfile, 'Map Name',
         pattern=r'(?!global-profile$)[A-Za-z0-9_:{}\[\]-]+' )
   }
   handler = showCosToTcProfile
   cliModel = CosToTcProfileAllModel

BasicCli.addShowCommandClass( ShowCosToTcProfile )

#-----------------------------------------------------------------------------------
# show qos map dscp to traffic-class <name>
#-----------------------------------------------------------------------------------

def ShowDscpToTcNamedMap( mode, args ):
   mapName = args.get( 'NAME' )
   dscpToTcMapAllModel = DscpToTcMapAllModel()
   if mapName:
      mapNames = [ mapName ] if mapName in qosStatus.dscpToTcNamedMap else []
   else:
      mapNames = qosStatus.dscpToTcNamedMap
   for mapName in mapNames:
      dscpToTcMapModel = DscpToTcMapModel()
      dscpToTcNamedMap = qosStatus.dscpToTcNamedMap[ mapName ].dscpToTcNamedMap
      for dscp in dscpToTcNamedMap:
         dscpToTcMapModel.dscpToTcNamedMap[ dscp ] = dscpToTcNamedMap[ dscp ]
                                                          
      dscpToTcMapAllModel.dscpToTcNamedMaps[ mapName ] = dscpToTcMapModel
   return dscpToTcMapAllModel

class ShowDscpToTcProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map dscp to traffic-class [ NAME ]"

   data = {
      'qos': nodeQosForShow,
      'map': "Mapping Configuration of different QoS parameters",
      'dscp': matcherMapDscp,
      'to': nodeDscpToTcMapTo,
      'traffic-class': matcherTrafficClass,
      'NAME': matcherDscpToTcMapName
      }
   handler = ShowDscpToTcNamedMap
   cliModel = DscpToTcMapAllModel

BasicCli.addShowCommandClass( ShowDscpToTcProfile )

#-----------------------------------------------------------------------------------
# show qos map traffic-class to cos <name>
#-----------------------------------------------------------------------------------

def showTcToCosProfile( mode, args ):
   mapName = args.get( 'NAME' )
   tcToCosMapAllModel = TcToCosMapAllModel()
   if mapName:
      mapNames = [ mapName ] if mapName in qosStatus.tcToCosNamedMap else []
   else:
      mapNames = qosStatus.tcToCosNamedMap
   for mapName in mapNames:
      tcToCosMapModel = TcToCosMapModel()
      tcToCosNamedMap = qosStatus.tcToCosNamedMap[ mapName ].tcToCosNamedMap
      for tcDpPair in tcToCosNamedMap:
         tcDpIndex = 'tc-%d dp-%d' % ( tcDpPair.tc, tcDpPair.dp )
         tcToCosMapModel.tcToCosNamedMap[ tcDpIndex ] = tcToCosNamedMap[ tcDpPair ]
                                                          
      tcToCosMapAllModel.tcToCosNamedMaps[ mapName ] = tcToCosMapModel
   return tcToCosMapAllModel

class ShowTcToCosProfile( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos map traffic-class to cos [ NAME ]"

   data = {
      'qos': nodeQosForShow,
      'map': 'Mapping Configuration of different QoS parameters',
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class', 
         helpdesc='Show traffic-class Map Configuration', 
         guard=guardAnyTcToCosPerInterface ),
      'to': 'Map to Traffic-Class',
      'cos': 'cos',
      'NAME': CliMatcher.DynamicNameMatcher(
         lambda mode : qosInputConfig.tcToCosNamedMap, 'Map Name',
         pattern=r'(?!default-map$)[A-Za-z0-9_:{}\[\]-]+' )
   }
   handler = showTcToCosProfile
   cliModel = TcToCosMapAllModel

BasicCli.addShowCommandClass( ShowTcToCosProfile )

#-----------------------------------------------------------------------------------
# "traffic-class <tc> to dscp <d> in DscpRewrite mode
#-----------------------------------------------------------------------------------
def getDscpRewriteMap( mode ):
   return qosInputConfig.dscpRewriteMap.get( mode.mapName )

def setTcToDscpMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or args.get( 'TC_LIST' ).values()
   dscp = args [ 'DSCP' ]
   dscpRewriteMap = getDscpRewriteMap( mode )
   assert dscpRewriteMap
   for tc in tcList:
      dscpRewriteMap.tcToDscpMap[ tc ] = dscp

def delTcToDscpMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or args.get( 'TC_LIST' ).values()
   mapName = getDscpRewriteMap( mode )
   for tc in tcList:
      del mapName.tcToDscpMap[ tc ]

class TrafficClassExpression( CliCommand.CliExpression ):
   expression = 'TC_LIST | { TC_VALUE }'
   data = {
      'TC_LIST': matcherTrafficClassList,
      'TC_VALUE': matcherTrafficClassValue,
   }

#--------------------------------------------------------------------------------
# [ no | default ] traffic-class TC to dscp DSCP
#--------------------------------------------------------------------------------
class TrafficClassToDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'traffic-class TC to dscp DSCP'
   noOrDefaultSyntax = 'traffic-class TC to dscp [ DSCP ]'
   data = {
      'traffic-class': 'Map traffic-class values',
      'TC': TrafficClassExpression,
      'to': 'QoS parameter to map traffic-class to',
      'dscp': nodeToDscp,
      'DSCP': matcherDscpValue,
   }

   handler = setTcToDscpMap
   noOrDefaultHandler = delTcToDscpMap

DscpRewriteMode.addCommandClass( TrafficClassToDscpCmd )

#-----------------------------------------------------------------------------------
# "[ no | default ] dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class TC_VALUE
#-----------------------------------------------------------------------------------
def getDscpToTcMap( mode ):
   return qosInputConfig.dscpToTcNamedMap.get( mode.mapName )

def setDscpToTcMap( mode, args ):
   dscpToTcNamedMap = getDscpToTcMap( mode )
   assert dscpToTcNamedMap
   dscpList = args.get( 'DSCP_VALUE' ) or args.get( 'DSCP_VALUES' ).values()
   tc = args[ 'TC_VALUE' ]
   for dscp in dscpList:
      dscpToTcNamedMap.dscpToTcNamedMap[ dscp ] = tc

def deleteDscpToTcMap( mode, args ):
   dscpToTcNamedMap = getDscpToTcMap( mode )
   assert dscpToTcNamedMap
   dscpList = args.get( 'DSCP_VALUE' ) or args.get( 'DSCP_VALUES' ).values()
   for dscp in dscpList:
      dscpToTcNamedMap.dscpToTcNamedMap[ dscp ] = \
            getCurrentGlobalDscpToTcMapping( dscp )

class DscpToTcMapCmd( CliCommand.CliCommandClass ):
   syntax = "dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class TC_VALUE"
   noOrDefaultSyntax = "dscp ( { DSCP_VALUE } | DSCP_VALUES ) to traffic-class ..."
   data = {
      'dscp': matcherMapDscp,
      'DSCP_VALUE': matcherDscpValue,
      'DSCP_VALUES' : matcherMultipleDscpValues,
      'to': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'to', helpdesc='Map to Traffic-Class' ), ),
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue
   }
   handler = setDscpToTcMap
   noOrDefaultHandler = deleteDscpToTcMap

DscpToTcMode.addCommandClass( DscpToTcMapCmd )

#-----------------------------------------------------------------------------------
# "traffic-class <tc> to cos <cos> in TcToCos mode
#-----------------------------------------------------------------------------------
def getTcToCosMap( mode ):
   return qosInputConfig.tcToCosNamedMap.newMember( mode.mapName )

def setTcToCosMap( mode, args ):
   cos = args[ 'COS' ]
   dp = args.get( 'DP' )
   tc = args[ 'TC' ]
   tcToCosNamedMap = getTcToCosMap( mode )
   assert tcToCosNamedMap
   dpList = sorted( QosLib.tcDpToCosSupportedDpValues() ) if dp is None else [ dp ]
   for dp in dpList:
      tcDpPair = tacTcDpPair( tc, dp )
      tcToCosNamedMap.tcToCosNamedMap[ tcDpPair ] = cos

def delTcToCosMap( mode, args ):
   tcToCosNamedMap = getTcToCosMap( mode )
   dp = args.get( 'DP' )
   tc = args[ 'TC' ]
   dpList = sorted( QosLib.tcDpToCosSupportedDpValues() ) if dp is None else [ dp ]
   for dp in dpList:
      tcDpPair = tacTcDpPair( tc, dp )
      tcToCosNamedMap.tcToCosNamedMap[
         tcDpPair ] = getCurrentGlobalTcDpToCosMapping( tc )

class TcToCosMapCmd( CliCommand.CliCommandClass ):
   syntax = "traffic-class TC [drop-precedence DP] to cos COS "
   noOrDefaultSyntax = "traffic-class TC [drop-precedence DP] to cos ..."
   data = {
      'traffic-class': tokenTrafficClassHelpdesc,
      'TC': CliMatcher.IntegerMatcher( 0, 7, helpdesc='Value of traffic-class' ),
      'drop-precedence': 'Map drop-precedence values',
      'DP': CliMatcher.IntegerMatcher( 0, 2, helpdesc='Value of drop-precedence' ),
      'to': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'to', helpdesc=MapToCosHelpdesc ), ),
      'cos': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
         'cos', helpdesc=MapToCosHelpdesc ) ),
      'COS': CliMatcher.IntegerMatcher( 0, 7, helpdesc='Value of CoS' ),
   }
   handler = setTcToCosMap
   noOrDefaultHandler = delTcToCosMap

TcToCosMode.addCommandClass( TcToCosMapCmd )

def intfTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   setIntfConfig( mode.intf.name, cfgTcToCosMapName=mapName )

def intfRemoveTcToCosMap( mode, args ):
   setIntfConfig( mode.intf.name,
         cfgTcToCosMapName=tacTcToCosMapName.defaultMapName )

class IntfTcToCosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map traffic-class to cos MAP_NAME'
   noOrDefaultSyntax = 'qos map traffic-class to cos'
   data = {
         'qos': nodeQosForConfig,
         'map': matcherMap,
         'traffic-class': tokenTrafficClassHelpdesc,
         'to': tokenTcToCosMapTo,
         'cos': CliCommand.guardedKeyword( 'cos', helpdesc=MapToCosHelpdesc, 
            guard=guardTcToCosPerInterface ),
         'MAP_NAME': tcToCosMapNameRule,
   }
   handler = intfTcToCosMap
   noOrDefaultHandler = intfRemoveTcToCosMap

TcToCosModelet.addCommandClass( IntfTcToCosMapCmd )

def intfCpuTcToCosMap( mode, args ):
   mapName = args[ 'MAP_NAME' ]
   setIntfConfig( mode.intf.name, cfgCpuTcToCosMapName=mapName )

def intfCpuRemoveTcToCosMap( mode, args ):
   setIntfConfig( mode.intf.name,
                  cfgCpuTcToCosMapName=tacTcToCosMapName.defaultMapName )

class IntfCpuTcToCosMapCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cpu traffic-class to cos MAP_NAME'
   noOrDefaultSyntax = 'qos map cpu traffic-class to cos'
   data = {
         'qos': nodeQosForConfig,
         'map': matcherMap,
         'cpu': CliCommand.guardedKeyword( 'cpu', helpdesc="CPU generated traffic", 
            guard=guardCpuTcToCosPerInterface ),
         'traffic-class': tokenTrafficClassHelpdesc,
         'to': tokenTcToCosMapTo,
         'cos': CliCommand.Node( matcher=CliMatcher.KeywordMatcher(
            'cos', helpdesc=MapToCosHelpdesc ) ),
         'MAP_NAME': tcToCosMapNameRule,
   }
   handler = intfCpuTcToCosMap
   noOrDefaultHandler = intfCpuRemoveTcToCosMap

CpuTcToCosModelet.addCommandClass( IntfCpuTcToCosMapCmd )

#-----------------------------------------------------------------------------------
# "cos <cos> to traffic-class <tc> in CosToTcProfile mode
#-----------------------------------------------------------------------------------
def getCosToTcProfile( mode ):
   return qosInputConfig.cosToTcProfile.get( mode.profileName )

def setCosToTcProfileMap( mode, args ):
   cosValues = args.get( 'COSVALUES' ) or args.get( 'COS' ).values()
   cosToTcProfile = getCosToTcProfile( mode )
   assert cosToTcProfile
   for cos in cosValues:
      cosToTcProfile.cosToTcMap[ cos ] = args[ 'TC_VALUE' ]

def delCosToTcProfileMap( mode, args ):
   cosValues = args.get( 'COSVALUES' ) or args.get( 'COS' ).values()
   cosToTcProfile = getCosToTcProfile( mode )
   for cos in cosValues:
      # Revert to global/platform default cos-to-tc mapping
      cosToTcProfile.cosToTcMap[ cos ] = getCurrentGlobalCosToTcMapping( cos )

#--------------------------------------------------------------------------------
# cos ( COS | { COSVALUES } ) to traffic-class TC_VALUE 
# ( no | default ) cos ( COS | { COSVALUES } ) to traffic-class ...
#--------------------------------------------------------------------------------
class CosToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'cos ( COS | { COSVALUES } ) to traffic-class TC_VALUE'
   noOrDefaultSyntax = 'cos ( COS | { COSVALUES } ) to traffic-class ...'
   data = {
      'cos': matcherMapCos,
      'COSVALUES': matcherCosValue,
      'COS': MultiRangeRule.MultiRangeMatcher( lambda: ( tacCos.min, tacCos.max ),
         False, 'Class of Service (COS) value(s) or range(s) of COS values' ),
      'to': 'QoS parameter to map CoS to',
      'traffic-class': matcherTrafficClass,
      'TC_VALUE': matcherTrafficClassValue,
   }

   handler = setCosToTcProfileMap
   noOrDefaultHandler = delCosToTcProfileMap

CosToTcProfileMode.addCommandClass( CosToTrafficClassCmd )

#-------------------------------------------------------------------------
# qos map cos to traffic-class <profilename>
#-------------------------------------------------------------------------
def applyCosToTcProfile( mode, args ):
   setIntfConfig( mode.intf.name, cfgCosToTcProfileName=args[ 'MAP' ] )

def removeCosToTcProfile( mode, args ):
   profileName = args.get( 'MAP' )
   intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      shouldDelete = True
      if profileName and intfConfig.cosToTcProfileName != profileName:
         shouldDelete = False

      if shouldDelete:
         setIntfConfig( mode.intf.name,
                     cfgCosToTcProfileName=tacCosToTcProfileName.defaultProfileName )

#--------------------------------------------------------------------------------
# qos map cos to traffic-class MAP
# ( no | default ) qos map cos to traffic-class [ MAP ]
#--------------------------------------------------------------------------------
class QosCosToTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'qos map cos to traffic-class MAP'
   noOrDefaultSyntax = 'qos map cos to traffic-class [ MAP ]'
   data = {
      'qos': nodeQosForConfig,
      'map': matcherMap,
      'cos': CliCommand.guardedKeyword( 'cos',
         helpdesc='Configure a Cos To Tc Map', guard=guardCosToTcPerInterface ),
      'to': 'QoS parameter to map CoS to',
      'traffic-class': matcherTrafficClass,
      'MAP': matcherCosToTcProfileName,
   }

   handler = applyCosToTcProfile
   noOrDefaultHandler = removeCosToTcProfile

CosToTcProfileModelet.addCommandClass( QosCosToTrafficClassCmd )

#------------------------------------------------------------------------------------
# The "qos map traffic-class <tc> to exp <0-7>" command, in "global config" mode.
#------------------------------------------------------------------------------------

def setTrafficClassToExp( mode, tcList, noOrDefaultKw=None, exp=None ):
   # Get the current tcToDscpMap config
   prevTcToExpMap = dict( qosInputConfig.tcToExpMap )
   for tc in tcList:
      timestamp = Tac.now()
      if noOrDefaultKw or exp == qosHwStatus.defaultTcToExpMap[ tc ]:
         tcToExpVal = tacExp.invalid
      else:
         tcToExpVal = exp
      if prevTcToExpMap[ tc ] == tcToExpVal:
         continue
      qosInputConfig.tcToExpMap[ tc ] = tcToExpVal
      if cliBlockingFail( mode, timestamp, tacFeatureName.map, "TC to Exp Map" ):
         # If any of the tcToDscp setting fails, revert the entire command
         for tc in prevTcToExpMap:
            qosInputConfig.tcToExpMap[ tc ] = prevTcToExpMap[ tc ]
         break

#------------------------------------------------------------------------------------
# Rewrite enable/disable commands in "global config" mode:
# "[ mls ] qos rewrite [ cos | dscp ]" 
# "[ no|default ] [ mls ] qos rewrite"
#------------------------------------------------------------------------------------
def guardCosRewrite( mode, token ):
   if qosHwStatus.cosRewriteSupported and qosHwStatus.cosRewriteDisableSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardDscpRewrite( mode, token ):
   if qosHwStatus.dscpRewriteSupported:
      return None
   return CliParser.guardNotThisPlatform

def setCosRewrite( mode, args ):
   timestamp = Tac.now()
   prevRewriteConfig = qosInputConfig.cosRewriteEnabled
   if CliCommand.isNoOrDefaultCmd( args ):
      if CliCommand.isDefaultCmd( args ):
         if prevRewriteConfig is tacRewriteEnableMode.rewriteInvalid:
            return
         qosInputConfig.cosRewriteEnabled = tacRewriteEnableMode.rewriteInvalid
      else: # means 'no'
         if prevRewriteConfig is tacRewriteEnableMode.rewriteDisabled:
            return
         qosInputConfig.cosRewriteEnabled = tacRewriteEnableMode.rewriteDisabled
   else:
      if prevRewriteConfig == tacRewriteEnableMode.rewriteEnabled:
         return
      qosInputConfig.cosRewriteEnabled = tacRewriteEnableMode.rewriteEnabled
   if cliBlockingFail( mode, timestamp, tacFeatureName.cosRewrite, "COS Rewrite" ):
      qosInputConfig.cosRewriteEnabled = prevRewriteConfig

def setDscpRewrite( mode, args ):
   timestamp = Tac.now()
   prevRewriteConfig = qosInputConfig.dscpRewriteEnabled
   if CliCommand.isNoOrDefaultCmd( args ):
      if not prevRewriteConfig:
         return
      qosInputConfig.dscpRewriteEnabled = False
   else:
      if prevRewriteConfig:
         return
      qosInputConfig.dscpRewriteEnabled = True

   if cliBlockingFail( mode, timestamp, tacFeatureName.dscpRewrite, 
                       "DSCP rewrite" ):
      qosInputConfig.dscpRewriteEnabled = prevRewriteConfig

#-------------------------------------------------------------------------
# qos rewrite traffic-class to dscp <mapname>
#-------------------------------------------------------------------------
def applyRewriteMap( mode, args ):
   intfConfig = qosInputConfig.intfConfig.newMember( mode.intf.name )
   intfConfig.dscpRewriteMapName = args[ 'MAP' ]

def deleteRewriteMap( mode, args ):
   intfConfig = qosInputConfig.intfConfig.get( mode.intf.name )
   if intfConfig:
      intfConfig.dscpRewriteMapName = tacDscpRewriteMapName.defaultMapName

#--------------------------------------------------------------------------------
# qos rewrite traffic-class to dscp MAP
# ( no | default ) qos rewrite traffic-class to dscp [ MAP ]
#--------------------------------------------------------------------------------
class QosTrafficClassToDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'qos rewrite traffic-class to dscp MAP'
   noOrDefaultSyntax = 'qos rewrite traffic-class to dscp [ MAP ]'
   data = {
      'qos': nodeQosForConfig,
      'rewrite': matcherRewrite,
      'traffic-class': CliCommand.guardedKeyword( 'traffic-class',
         helpdesc='Configure a Rewrite Map', guard=guardDscpRewritePerInterface ),
      'to': 'QoS parameter to map traffic-class to',
      'dscp': nodeToDscp,
      'MAP': matcherDscpRewriteMapName,
   }

   handler = applyRewriteMap
   noOrDefaultHandler = deleteRewriteMap

DscpRewriteModelet.addCommandClass( QosTrafficClassToDscpCmd )

def showQosIntfTxQueues( intf, sliceName=None, mode=None ):
   txQueueQosModel = InterfaceQosModel.TxQueueQosModel()
   if intf.name.startswith( fabricShortName ):
      intfName = fabricIntfName
   else:
      intfName = intf.name
   intfConfig = None
   intfStatus = None

   intfConfig = qosConfig.intfConfig.get( intfName )
   intfStatus = getIntfStatus( intf )

   # SubIntf related parameters.
   isSubIntf = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfName )
   # Set to True if according to qosConfig the default SubIntf Schedulng scheme
   # is in effect.
   shapedSubIntfDefaultSchConfigPresent = False
   # Checks the above in qosStatus.
   shapedSubIntfDefaultSchStatusPresent = False
   # In case there is no scheduling / shaping config on a SubIntf, it uses its
   # parent's VOQs. This should be set to True in that case.
   subIntfUseParentQueues = False

   if isSubIntf:
      subIntfTxQueueSchPresent = False
      subIntfOrTxQueueShpPresent = False
      if intfStatus:
         for txQStatus in intfStatus.txQueueStatus.values():
            if txQStatus.shapeRate.rate != tacShapeRateVal.invalid:
               subIntfOrTxQueueShpPresent = True
            if txQStatus.priority != tacTxQueuePriority.priorityInvalid:
               subIntfTxQueueSchPresent = True
            if subIntfOrTxQueueShpPresent and subIntfTxQueueSchPresent:
               break
         if intfStatus.shapeRate.rate != tacShapeRateVal.invalid:
            subIntfOrTxQueueShpPresent = True
      shapedSubIntfDefaultSchStatusPresent = ( subIntfOrTxQueueShpPresent and
                                               not subIntfTxQueueSchPresent )
      subIntfUseParentQueues = not ( subIntfTxQueueSchPresent or
                                     subIntfOrTxQueueShpPresent )

      subIntfTxQueueSchPresent = False
      subIntfOrTxQueueShpPresent = False
      if intfConfig:
         for txQConfig in intfConfig.txQueueConfig.values():
            if txQConfig.shapeRate.rate != tacShapeRateVal.invalid:
               subIntfOrTxQueueShpPresent = True
            if txQConfig.priority != tacTxQueuePriority.priorityInvalid:
               subIntfTxQueueSchPresent = True
            if subIntfOrTxQueueShpPresent and subIntfTxQueueSchPresent:
               break
         if intfConfig.shapeRate.rate != tacShapeRateVal.invalid:
            subIntfOrTxQueueShpPresent = True
      shapedSubIntfDefaultSchConfigPresent = ( subIntfOrTxQueueShpPresent and
                                               not subIntfTxQueueSchPresent )
      if isLagPort( intfName ):
         # For lag subIntfs, this value is decided from intfConfig.
         subIntfUseParentQueues = not ( subIntfTxQueueSchPresent or
                                        subIntfOrTxQueueShpPresent )

   if subIntfUseParentQueues:
      parentIntf = IntfCli.Intf( Tac.Type( "Arnet::SubIntfId" ).parentIntfId(
         intfName ), mode )
      return showQosIntfTxQueues( parentIntf, sliceName=sliceName, mode=mode )

   txQueueQosModel.wrrSupported = qosHwStatus.wrrSupported
   txQueueQosModel.guaranteedBwSupported = qosHwStatus.guaranteedBwSupported
   txQueueQosModel.ecnSupported = qosHwStatus.ecnSupported
   txQueueQosModel.delayEcnSupported = qosHwStatus.ecnDelaySupported
   txQueueQosModel.wredSupported = qosHwStatus.wredSupported
   txQueueQosModel.nonEctSupported = qosHwStatus.nonEctThdSupported
   isMulticastQueueSupported = qosHwStatus.numMulticastQueueSupported != 0
   txQueueQosModel.multipleSchedGroupsSupported = isMulticastQueueSupported
   portTmMap = qosTmMapping.portTmMap
   if intfName in portTmMap:
      txQueueQosModel.numTxQueuePrint = portTmMap[ intfName ].numberOfQueues
   if not isSubIntf:
      txQueueQosModel.numTxQueue = qosHwStatus.numTxQueueSupported
   else:
      txQueueQosModel.numTxQueue = qosGlobalConfig.numTxqsPerSubIntf

   txQueueQosModel.burstSizeConfigSupported = \
                                       qosHwStatus.shapeRateBurstSizeConfigSupported
   if txQueueQosModel.numTxQueue == 0:
      return txQueueQosModel

   for hwtxqid in xrange( txQueueQosModel.numTxQueue - 1, -1, -1 ):
      schGroupId = qosHwStatus.hwTxQueue[ hwtxqid ].schGroupId
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         txQName = clitxq.type[ :2 ].upper() + str( clitxq.id )
      else:
         txQName = str( clitxq.id )

      txQueueModel = InterfaceQosModel.TxQueueQosModel.TxQueue()
      txQueueModel.txQueue = txQName
      txQueueModel.txQueuePriority = hwtxqid
      txQueueModel.schedGroupId = schGroupId
      txQueueModel.operationalBurstSize = None
      txQueueModel.configuredBurstSize = None

      subIntfTxQueueDefaultWrrBw = None
      subIntfTxQueueDefaultWrrPriority = None
      # Default scheduling for SubIntfs with port / tx-queue shaping may not be all
      # SP. If different, the PD agent is supposed to pusblish it in qosHwStatus.
      if clitxq in qosHwStatus.subIntfDefaultSchConfig:
         subIntfTxQueueDefaultWrrBw = qosHwStatus.subIntfDefaultSchConfig[
            clitxq ].bandwidth
         subIntfTxQueueDefaultWrrPriority = (
            qosHwStatus.subIntfDefaultSchConfig[ clitxq ].priority )

      txQueueQosModel.txQueueList.append( txQueueModel )

      txQueueConfig = None
      txQueueStatus = None

      if intfConfig:
         txQueueConfig = intfConfig.txQueueConfig.get( clitxq )

      if intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )

      if shapedSubIntfDefaultSchConfigPresent:
         if subIntfTxQueueDefaultWrrBw != tacPercent.invalid:
            # This value will get overwritten if txQueueConfig is not None
            # and txQueueConfig.bandwidth != tacPercent.invalid
            txQueueModel.configuredWrrBw = subIntfTxQueueDefaultWrrBw
         txQueueModel.configuredSchedMode = SchedulingModeModel()
         txQueueModel.configuredSchedMode.schedulingMode = schedModeEnumToStr(
            subIntfTxQueueDefaultWrrPriority )

      # Configured parameters 
      if txQueueConfig:

         # WRR Bandwidth
         if txQueueConfig.bandwidth != tacPercent.invalid:
            txQueueModel.configuredWrrBw = txQueueConfig.bandwidth

         # Guaranteed bandwidth
         if txQueueConfig.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
            txQueueModel.configuredGuaranteedBw = GuaranteedBwModel()
            cfgGuaranteedBw = txQueueModel.configuredGuaranteedBw
            cfgGuaranteedBw.bandwidth = txQueueConfig.guaranteedBw.bw
            cfgGuaranteedBw.unit = \
               guaranteedBwUnitFromEnum( txQueueConfig.guaranteedBw.unit )
      
         # Shape rate
         txQueueModel.configuredShapeRate = \
            populateShapeRateModel( txQueueConfig.shapeRate,
                                    statusShapeRate=txQueueStatus.shapeRate \
                                    if txQueueStatus else None)
         # Burst-size config
         if qosHwStatus.shapeRateBurstSizeConfigSupported:
            txQueueModel.configuredBurstSize = populateBurstSizeModel(
               txQueueConfig.shapeRate.burstSize )

         # Scheduling priority/mode
         if not shapedSubIntfDefaultSchConfigPresent:
            txQueueModel.configuredSchedMode = SchedulingModeModel()
            cfgSchedMode = txQueueModel.configuredSchedMode
            cfgSchedMode.schedulingMode = schedModeEnumToStr(
               txQueueConfig.priority )

      # ECN Status
      ecnStatus = None
      delayEcnEnabled = \
            qosStatus.ecnDelayThreshold != tacEcnDelayThreshold and \
            qosStatus.ecnDelayThreshold.unit != tacEcnDelayThreshold.unit
      if not intfStatus:
         if txQueueConfig:
            ecnStatus = txQueueConfig.ecnConfig
            delayEcnEnabled &= txQueueConfig.delayEcnEnabled
         else:
            delayEcnEnabled = False
      elif clitxq in intfStatus.txQueueStatus:
         ecnStatus = intfStatus.txQueueStatus[ clitxq ].ecnStatus
         txQueueStatus = intfStatus.txQueueStatus[ clitxq ]
         delayEcnEnabled &= intfStatus.txQueueStatus[ clitxq ].delayEcnEnabled
      else:
         delayEcnEnabled = False

      if ecnStatus and delayEcnEnabled:
         txQueueModel.ecnStatus = 'lengthAndDelayEnabled'
      elif ecnStatus:
         txQueueModel.ecnStatus = 'enabled'
      elif delayEcnEnabled:
         txQueueModel.ecnStatus = 'delayEnabled'
      else:
         if txQName.startswith( 'MC' ):
            txQueueModel.ecnStatus = 'not applicable'
         else:
            txQueueModel.ecnStatus = 'disabled'

      # WRED status
      wredStatus = None
      if not intfStatus:
         if txQueueConfig:
            wredStatus = txQueueConfig.wredConfig
      elif clitxq in intfStatus.txQueueStatus:
         wredStatus = intfStatus.txQueueStatus[ clitxq ].wredStatus
         txQueueStatus = intfStatus.txQueueStatus[ clitxq ]

      if wredStatus:
         txQueueModel.wredStatus = 'enabled'
      else:
         if txQName.startswith( 'MC' ):
            txQueueModel.wredStatus = 'not applicable'
         else:
            txQueueModel.wredStatus = 'disabled'

      # NON-ECT status
      nonEctStatus = None
      if not intfStatus:
         if txQueueConfig:
            nonEctStatus = txQueueConfig.nonEctConfig
      elif clitxq in intfStatus.txQueueStatus:
         txQueueStatus = intfStatus.txQueueStatus[ clitxq ]
         nonEctStatus = txQueueStatus.nonEctStatus

      if nonEctStatus:
         txQueueModel.nonEctStatus = 'enabled'
      else:
         if txQName.startswith( 'MC' ):
            txQueueModel.nonEctStatus = 'not applicable'
         else:
            txQueueModel.nonEctStatus = 'disabled'

      # Operational parameters
      if txQueueStatus:
         # WRR Bandwidth operational status
         if txQueueStatus.bandwidth != tacPercent.invalid:
            txQueueModel.operationalWrrBw = txQueueStatus.bandwidth
         
         # Check if detailed interface status is supported 
         #( i.e., Would hwStatus be populated by this platform )
         if qosHwStatus.interfaceDetailsSupported:

            # Guaranteed bandwidth operation status 
            if txQueueStatus.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
               intfHwGuaranteedBw = None
               for sliceHwStatus in qosSliceHwStatus.itervalues():
                  intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( intfName )
                  if intfHwGuaranteedBw != None:
                     break
               if intfName == fabricIntfName:
                  for sliceHwStatus in qosSliceHwStatus.itervalues():
                     if sliceHwStatus.name == sliceName:
                        hwGuaranteedBw = sliceHwStatus.hwGuaranteedBw
                        intfHwGuaranteedBw = hwGuaranteedBw.get( intfName )
                        break
               if intfHwGuaranteedBw and \
                     clitxq in intfHwGuaranteedBw.txQueueGuaranteedBw:

                  txQueueHwGuaranteedBw = \
                     intfHwGuaranteedBw.txQueueGuaranteedBw[ clitxq ]

                  if txQueueHwGuaranteedBw.guaranteedBw.bw != \
                        tacGuaranteedBwVal.invalid:
                     txQueueModel.operationalGuaranteedBw = GuaranteedBwModel()
                     operGuaranteedBw = txQueueModel.operationalGuaranteedBw
                     bw = txQueueHwGuaranteedBw.guaranteedBw.bw
                     unit = txQueueHwGuaranteedBw.guaranteedBw.unit
                     operGuaranteedBw.bandwidth = bw
                     operGuaranteedBw.unit = guaranteedBwUnitFromEnum( unit )

            # Shape rate operation status
            if txQueueStatus.shapeRate.rate != tacShapeRateVal.invalid:
               intfHwShapeRate = None
               for sliceHwStatus in qosSliceHwStatus.itervalues():
                  intfHwShapeRate = sliceHwStatus.hwShapeRates.get( intfName )
                  if intfHwShapeRate != None:
                     break
               if intfHwShapeRate and \
                     clitxq in intfHwShapeRate.txQueueShapeRate:

                  txQueueHwShapeRate = intfHwShapeRate.txQueueShapeRate[ clitxq ]
                  txQueueModel.operationalShapeRate = \
                     populateShapeRateModel( txQueueHwShapeRate.shapeRate )
                  # Burst-size operational Value
                  if qosHwStatus.shapeRateBurstSizeConfigSupported:
                     txQueueModel.operationalBurstSize = \
                              populateBurstSizeModel(
                                 txQueueHwShapeRate.shapeRate.burstSize )

         else: # Interface details not supported/no hw status available

            # Guaranteed Bandwidth
            if txQueueStatus.guaranteedBw.bw != tacGuaranteedBwVal.invalid:
               txQueueModel.operationalGuaranteedBw = GuaranteedBwModel()
               operGuaranteedBw = txQueueModel.operationalGuaranteedBw
               operGuaranteedBw.bandwidth = txQueueStatus.guaranteedBw.bw
               operGuaranteedBw.unit = \
                  guaranteedBwUnitFromEnum( txQueueStatus.guaranteedBw.unit )

            # Shape Rate
            txQueueModel.operationalShapeRate = \
               populateShapeRateModel( txQueueStatus.shapeRate )

            # Burst Size
            if txQueueStatus.shapeRate:
               txQueueModel.operationalBurstSize = populateBurstSizeModel(
                  txQueueStatus.shapeRate.burstSize )

      # Scheduling priority/mode
      if not shapedSubIntfDefaultSchStatusPresent:
         txQueueModel.operationalSchedMode = SchedulingModeModel()
         if txQueueStatus:
            txQueueModel.operationalSchedMode.schedulingMode = (
               schedModeEnumToStr( txQueueStatus.priority ) )
      else:
         if subIntfTxQueueDefaultWrrBw != tacPercent.invalid:
            txQueueModel.operationalWrrBw = subIntfTxQueueDefaultWrrBw
         txQueueModel.operationalSchedMode = SchedulingModeModel()
         txQueueModel.operationalSchedMode.schedulingMode = schedModeEnumToStr(
            subIntfTxQueueDefaultWrrPriority )


   return txQueueQosModel

def populateConfiguredBw( configBw, statusBw ):
   guaranteedBw = GuaranteedBwModel()
   if configBw.unit == tacGuaranteedBwUnit.guaranteedBwPercent:
      guaranteedBw.percent = configBw.bw
      guaranteedBw.bandwidth = tacGuaranteedBwVal.invalid
   if statusBw:
      guaranteedBw.bandwidth = statusBw.bw
      guaranteedBw.unit = guaranteedBwUnitFromEnum( statusBw.unit )
   elif configBw.unit != tacGuaranteedBwUnit.guaranteedBwPercent:
      guaranteedBw.bandwidth = configBw.bw
      guaranteedBw.unit = guaranteedBwUnitFromEnum( configBw.unit )
   return guaranteedBw

def populateConfiguredSr( configSr, statusSr ):
   shapeRate = ShapeRateModel()
   if configSr.percent != tacPercent.invalid:
      shapeRate.percent = configSr.percent
      shapeRate.rate = tacShapeRateVal.invalid
   if statusSr is not None:
      shapeRate.rate = statusSr.rate
      shapeRate.unit = QosLib.shapeRateUnitFromEnum( statusSr.unit )
   elif configSr.percent == tacPercent.invalid:
      shapeRate.rate = configSr.rate
      shapeRate.unit = QosLib.shapeRateUnitFromEnum( configSr.unit )
   return shapeRate

def populateBwModel( qosGuaranteedBw ):
   operBw = GuaranteedBwModel()
   operBw.bandwidth = qosGuaranteedBw.bw
   operBw.unit = guaranteedBwUnitFromEnum( qosGuaranteedBw.unit )
   return operBw

def isValidGuaranteedBw( guaranteedBw ):
   return guaranteedBw.bw != tacGuaranteedBwVal.invalid

def isValidShapeRate( shapeRate ):
   return shapeRate.rate != tacShapeRateVal.invalid

def getAllSchedulingGroups( mode ):
   groups = []
   for intfConfig in qosStatus.intfStatus.values():
      for group in intfConfig.schedulerGroupStatus.keys():
         if group not in groups:
            groups.append( group )
   return groups

def populateSchedulingHierarchy( subIntf, mode ):

   def populateIntfParameters( intf ):
      params = QosSchedulingHierarchyModel.HierarchyModel()
      params.name = intf
      intfConfig = qosConfig.intfConfig.get( intf )
      intfStatus = getIntfStatus( intf )
      if intfConfig and intfStatus:
         statusBw = intfStatus.guaranteedBw
         statusSr = intfStatus.shapeRate

         # configured value
         if isValidGuaranteedBw( intfConfig.guaranteedBw ):
            params.configuredGuaranteedBw = \
                        populateConfiguredBw( intfConfig.guaranteedBw, statusBw )

         if isValidShapeRate( intfConfig.shapeRate ):
            params.configuredShapeRate = \
                           populateConfiguredSr( intfConfig.shapeRate, statusSr )

         # populate operational value
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            hwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( intf )
            hwShapeRate = sliceHwStatus.hwShapeRates.get( intf )
            if hwGuaranteedBw != None or hwShapeRate != None:
               break

         if hwGuaranteedBw and \
            isValidGuaranteedBw( hwGuaranteedBw.intfGuaranteedBw ):
            params.operationalGuaranteedBw = \
                                 populateBwModel( hwGuaranteedBw.intfGuaranteedBw )

         if hwShapeRate and isValidShapeRate( hwShapeRate.intfShapeRate ):
            params.operationalShapeRate = \
                                 populateShapeRateModel( hwShapeRate.intfShapeRate )
      return params

   subIntfName = subIntf.name
   if not subIntf.name.startswith( 'Et' ) or \
      not Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( subIntfName ):
      return None
   
   intfHierarchyModel = QosSchedulingHierarchyModel()

   # subintf
   subIntfConfig = qosConfig.intfConfig.get( subIntfName )
   if not subIntfConfig:
      return None
   subIntfStatus = getIntfStatus( subIntfName )
   intfHierarchyModel.subinterface = populateIntfParameters( subIntfName )

   # txqueue
   txQueueList = []
   for key, qConfig in subIntfConfig.txQueueConfig.items():
      qHierarchyModel = QosSchedulingHierarchyModel.HierarchyModel()
      qHierarchyModel.name = str( key.id )
      qStatus = subIntfStatus.txQueueStatus.get( key )
      statusBw = None
      statusSr = None
      if qStatus:
         statusBw = qStatus.guaranteedBw
         statusSr = qStatus.shapeRate

      # Configured guaranteed bandwidth
      if isValidGuaranteedBw( qConfig.guaranteedBw ):
         qHierarchyModel.configuredGuaranteedBw = \
                           populateConfiguredBw( qConfig.guaranteedBw, statusBw )

      # Configured shape rate
      if isValidShapeRate( qConfig.shapeRate ):
         qHierarchyModel.configuredShapeRate = \
                              populateConfiguredSr( qConfig.shapeRate, statusSr )

      # Operational values
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( subIntfName )
         intfHwShapeRate = sliceHwStatus.hwShapeRates.get( subIntfName )
         if intfHwGuaranteedBw != None or intfHwShapeRate != None:
            break

      # Operational guaranteed bandwidth
      if intfHwGuaranteedBw and key in intfHwGuaranteedBw.txQueueGuaranteedBw:
         op = intfHwGuaranteedBw.txQueueGuaranteedBw[ key ].guaranteedBw
         if isValidGuaranteedBw( op ):
            qHierarchyModel.operationalGuaranteedBw = populateBwModel( op )

      # Operational shape rate
      if intfHwShapeRate and key in intfHwShapeRate.txQueueShapeRate:
         op = intfHwShapeRate.txQueueShapeRate[ key ].shapeRate
         if isValidShapeRate( op ):
            qHierarchyModel.operationalShapeRate = populateShapeRateModel( op )

      txQueueList.append( qHierarchyModel )
   if txQueueList:
      intfHierarchyModel.txQueues = txQueueList

   # parent intf
   parentIntfName = subIntfName[ :subIntfName.find( '.' ) ]
   intfConfig = qosConfig.intfConfig.get( parentIntfName )
   intfStatus = getIntfStatus( parentIntfName )
   intfHierarchyModel.parentInterface = populateIntfParameters( parentIntfName )
   parentIntf = EthIntfCli.EthPhyIntf( parentIntfName, mode )
   intfHierarchyModel.parentInterface.interfaceSpeed = parentIntf.bandwidth() * 1000

   # scheduling group
   schedulingGroup = QosSchedulingHierarchyModel.HierarchyModel()

   group = subIntfConfig.schedulerGroupName
   if group and intfConfig and group in intfConfig.schedulerGroupConfig:
      schedulingGroup.name = group
      groupConfig = intfConfig.schedulerGroupConfig[ group ]
      statusBw = None
      statusSr = None
      if intfStatus and group in intfStatus.schedulerGroupStatus:
         statusBw = intfStatus.schedulerGroupStatus[ group ].guaranteedBw
         statusSr = intfStatus.schedulerGroupStatus[ group ].shapeRate

      # Configured guaranteed bandwidth
      if isValidGuaranteedBw( groupConfig.guaranteedBw ):
         schedulingGroup.configuredGuaranteedBw = \
                           populateConfiguredBw( groupConfig.guaranteedBw, statusBw )

      # Configured shape rate
      if isValidShapeRate( groupConfig.shapeRate ):
         schedulingGroup.configuredShapeRate = \
                              populateConfiguredSr( groupConfig.shapeRate, statusSr )

      # Operational values
      intfHwShapeRate = None
      intfHwGuaranteedBw = None
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( parentIntfName )
         intfHwShapeRate = sliceHwStatus.hwShapeRates.get( parentIntfName )
         if intfHwGuaranteedBw != None or intfHwShapeRate != None:
            break

      # Operational guaranteed bandwidth
      if intfHwGuaranteedBw and \
         group in intfHwGuaranteedBw.schedulerGroupGuaranteedBw:
         op = intfHwGuaranteedBw.schedulerGroupGuaranteedBw[ group ].guaranteedBw
         if isValidGuaranteedBw( op ):
            schedulingGroup.operationalGuaranteedBw = populateBwModel( op )

      # Operational shape rate
      if intfHwShapeRate and group in intfHwShapeRate.schedulerGroupShapeRate:
         op = intfHwShapeRate.schedulerGroupShapeRate[ group ].shapeRate
         if isValidShapeRate( op ):
            schedulingGroup.operationalShapeRate = populateShapeRateModel( op )

      intfHierarchyModel.schedulingGroup = schedulingGroup

   return intfHierarchyModel

def populateQosSchedulingGroupsForIntf( intf, group=None, members=None ):
   if not intf.name.startswith( 'Et' ):
      return None

   if members is None:
      schedulingGroupModel = IntfSchedulingGroupModel()
      groupModel = SchedulingElementModel
   else:
      schedulingGroupModel = IntfSchedulingGroupWithMembersModel()
      groupModel = SchedulingGroupWithMembersModel

   intfName = intf.name
   intfConfig = qosConfig.intfConfig.get( intfName )
   intfStatus = getIntfStatus( intf )
   if intfConfig:
      for name, groupConfig in intfConfig.schedulerGroupConfig.iteritems():
         if group and name != group:
            continue
         schedulingGroup = groupModel()

         statusBw = None
         statusSr = None
         if intfStatus and name in intfStatus.schedulerGroupStatus:
            statusBw = intfStatus.schedulerGroupStatus[ name ].guaranteedBw
            statusSr = intfStatus.schedulerGroupStatus[ name ].shapeRate
         # Configured guaranteed bandwidth
         if isValidGuaranteedBw( groupConfig.guaranteedBw ):
            schedulingGroup.configuredGuaranteedBw = \
                           populateConfiguredBw( groupConfig.guaranteedBw, statusBw )

         # Configured shape rate
         if isValidShapeRate( groupConfig.shapeRate ):
            schedulingGroup.configuredShapeRate = \
                              populateConfiguredSr( groupConfig.shapeRate, statusSr )

         # Operational values
         intfHwShapeRate = None
         intfHwGuaranteedBw = None
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            intfHwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( intfName )
            intfHwShapeRate = sliceHwStatus.hwShapeRates.get( intfName )
            if intfHwGuaranteedBw != None or intfHwShapeRate != None:
               break

         # Operational guaranteed bandwidth
         if intfHwGuaranteedBw and \
            name in intfHwGuaranteedBw.schedulerGroupGuaranteedBw:
            op = intfHwGuaranteedBw.schedulerGroupGuaranteedBw[ name ].guaranteedBw
            if isValidGuaranteedBw( op ):
               schedulingGroup.operationalGuaranteedBw = populateBwModel( op )

         # Operational shape rate
         if intfHwShapeRate and name in intfHwShapeRate.schedulerGroupShapeRate:
            op = intfHwShapeRate.schedulerGroupShapeRate[ name ].shapeRate
            if isValidShapeRate( op ):
               schedulingGroup.operationalShapeRate = populateShapeRateModel( op )

         if members:
            for subIntf in members:
               intfConfig = qosConfig.intfConfig.get( subIntf )
               if intfConfig and intfConfig.schedulerGroupName == name:
                  params = SchedulingElementModel()
                  subIntfStatus = getIntfStatus( subIntf )
                  if subIntfStatus:
                     statusBw = subIntfStatus.guaranteedBw
                     statusSr = subIntfStatus.shapeRate

                  # configured value
                  if isValidGuaranteedBw( intfConfig.guaranteedBw ):
                     params.configuredGuaranteedBw = \
                           populateConfiguredBw( intfConfig.guaranteedBw, statusBw )

                  if isValidShapeRate( intfConfig.shapeRate ):
                     params.configuredShapeRate = \
                              populateConfiguredSr( intfConfig.shapeRate, statusSr )

                  # populate operational value
                  for sliceHwStatus in qosSliceHwStatus.itervalues():
                     hwGuaranteedBw = sliceHwStatus.hwGuaranteedBw.get( subIntf )
                     hwShapeRate = sliceHwStatus.hwShapeRates.get( subIntf )
                     if hwGuaranteedBw != None or hwShapeRate != None:
                        break

                  if hwGuaranteedBw and \
                     isValidGuaranteedBw( hwGuaranteedBw.intfGuaranteedBw ):
                     params.operationalGuaranteedBw = \
                                 populateBwModel( hwGuaranteedBw.intfGuaranteedBw )

                  if hwShapeRate and isValidShapeRate( hwShapeRate.intfShapeRate ):
                     params.operationalShapeRate = \
                                 populateShapeRateModel( hwShapeRate.intfShapeRate )

                  schedulingGroup.interfaces[ subIntf ] = params

         schedulingGroupModel.schedulingGroups[ name ] = schedulingGroup
   return schedulingGroupModel

def showQosInterface( intf, sliceName=None, detail=None, mode=None ):
   # detail specifies whether to display the detailed output for 
   # fabric interface corresponding to the sliceName
   intfQosModel = InterfaceQosModel()
   intfName = intf.name
   if intf.name.startswith( fabricShortName ):
      intfName = fabricIntfName
      if detail:
         intfQosModel.sliceName = sliceName

   isLagIntf = isLagPort( intfName )
   isSubIntf = Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfName )
   isLagSubIntf = isSubIntf and isLagIntf

   intfQosModel.interface = intfName
   intfQosModel.intfIsLag = isLagIntf

   trustMode = tacTrustMode.invalid 
   defaultCos = tacCos.invalid 
   defaultDscp = tacDscp.invalid

   lagMem = None
   if isLagIntf:
      lagMem = getConfiguredLagMembers( intfName )

   if isLagIntf and not lagMem:
      # For a lag having no members, show intfConfig
      if intfName in qosConfig.intfConfig:
         intfConfig = qosConfig.intfConfig[ intfName ]
         trustMode = intfConfig.trustMode
         defaultCos = intfConfig.defaultCos
         defaultDscp = intfConfig.defaultDscp
   else:
      # For phyIntf or lag with atleast one member, show intfStatus
      intfStatus = getIntfStatus( intf )
      if intfStatus:
         trustMode = intfStatus.trustMode
         defaultCos = intfStatus.defaultCos
         defaultDscp = intfStatus.defaultDscp

   if isSubIntf:
      if not isLagSubIntf:
         # For Ethernet subInterface, get the trustMode, defaultCos and
         # defaultDscp from parent
         parentIntf = Tac.Type( "Arnet::SubIntfId" ).parentIntfId( intfName )
         parentQosIntfStatus = getIntfStatus( parentIntf )
         if parentQosIntfStatus:
            trustMode = parentQosIntfStatus.trustMode
            defaultCos = parentQosIntfStatus.defaultCos
            defaultDscp = parentQosIntfStatus.defaultDscp
      else:
         # xxx_rajat LAG SUBINTF HACK
         # Po1.1 -> Po1 -> Et1 -> Et1.1
         if lagMem:
            # show the intfstatus from any one member
            chosenLagMember = lagMem[ 0 ] # Et1
            intfStatus = getIntfStatus( chosenLagMember )
            if intfStatus:
               trustMode = intfStatus.trustMode
               defaultCos = intfStatus.defaultCos
               defaultDscp = intfStatus.defaultDscp

   # Now we populate the intfQosModel( trustMode, defaultCos, defaultDscp )
   intfQosModel.intfQosInfoAvailable = True
   trustModeStr = compareAttributeToDefault( qosHwStatus, "defaultTrustMode", 
                                          trustMode, tacTrustMode.invalid ) 
   intfQosModel.trustMode = convertTrustModeModelFromStr( trustModeStr )
   intfQosModel.intfQosDetailedInfoAvailable = qosHwStatus.interfaceDetailsSupported
   if qosHwStatus.defaultCosSupported == True: 
      intfQosModel.defaultCosSupported = True
      intfQosModel.defaultCos = int( compareAttributeToDefault(
                                              qosHwStatus, "defaultCos",
                                              defaultCos, tacCos.invalid ) )

   if qosHwStatus.defaultDscpSupported == True: 
      intfQosModel.defaultDscpSupported = True
      intfQosModel.defaultDscp = int( compareAttributeToDefault(
                                             qosHwStatus, "defaultDscp",
                                             defaultDscp, tacDscp.invalid ) )

   if qosHwStatus.cosToTcPerInterfaceSupported:
      if not isLagIntf:
         # For Front Panel Ports, we populate operational value
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if intfName in sliceHwStatus.intfToCosToTcProfile:
               intfQosModel.cosToTcProfile = sliceHwStatus.intfToCosToTcProfile[
                  intfName ]
      else:
         # For LAG, we populate configured value
         if intfName in qosConfig.intfConfig:
            intfQosModel.cosToTcProfile = qosConfig.intfConfig[
               intfName ].cosToTcProfileName
   
   if qosHwStatus.tcToCosPerInterfaceSupported:
      if not isLagIntf:
         # For Front Panel Ports, we populate operational value
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if intfName in sliceHwStatus.intfToTcToCosNamedMap:
               intfQosModel.tcToCosNamedMap = sliceHwStatus.intfToTcToCosNamedMap[
                  intfName ]
               break
      else:
         # For LAG, we populate configured value
         if intfName in qosConfig.intfConfig:
            intfQosModel.tcToCosNamedMap = qosConfig.intfConfig[
               intfName ].tcToCosMapName

   if qosHwStatus.cpuTcToCosPerInterfaceSupported:
      if intfName in qosConfig.intfConfig:
         intfQosModel.cpuTcToCosNamedMap = qosConfig.intfConfig[
            intfName ].cpuTcToCosMapName

   if qosHwStatus.dscpRewritePerInterfaceSupported:
      if not isLagIntf:
         # For Front Panel Ports, we populate operational value
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if intfName in sliceHwStatus.intfToTcToDscpRewriteMap:
               intfQosModel.dscpRewriteMap = sliceHwStatus.intfToTcToDscpRewriteMap[
                  intfName ]
               break
      else:
         # For LAG, we populate configured value
         if intfName in qosConfig.intfConfig:
            intfQosModel.dscpRewriteMap = qosConfig.intfConfig[
               intfName ].dscpRewriteMapName

   if trustMode == tacTrustMode.dscp:
      # Incase of DSCP trust mode and COS rewrite is enabled but not active,
      # then indicate to the user.
      hwProgrammingSuccess = True
      # Flag sliceHwStatus.dscpToCosRewriteEnabled is set to False when 
      # hardware programming fails.
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         if not sliceHwStatus.dscpToCosRewriteEnabled:
            hwProgrammingSuccess = False
            break
      if qosStatus.cosRewriteEnabled  and not hwProgrammingSuccess:
         intfQosModel.dscpToCosRewriteActive = False
   else:
      # For other trust modes if DSCP rewrite is enabled but not active,
      # then indicate to the user.
      hwProgrammingSuccess = True
      for sliceHwStatus in qosSliceHwStatus.itervalues():
         if not sliceHwStatus.cosToDscpRewriteEnabled:
            hwProgrammingSuccess = False
            break
      if qosStatus.dscpRewriteEnabled  and not hwProgrammingSuccess:
         intfQosModel.dscpRewriteActive = False

   # Fetch the correct intfStatus and intfConfig for the interface
   intfStatus = getIntfStatus( intf )
   intfConfig = qosConfig.intfConfig.get( intfName )

   # configured port shape rate
   if intfConfig:
      if isSubIntf and lagMem and '.' not in lagMem[ 0 ]:
         memSubIntf = lagMem[ 0 ] + intfName[ intfName.find( '.' ) : ]
         memSubIntfStatus = getIntfStatus( memSubIntf )
         statusShapeRate = memSubIntfStatus.shapeRate if memSubIntfStatus else None
      else:
         statusShapeRate = intfStatus.shapeRate if intfStatus else None
      intfQosModel.configuredPortShapeRate = \
         populateShapeRateModel( intfConfig.shapeRate,
                                 statusShapeRate=statusShapeRate,
                                 lagSize=len( lagMem ) if lagMem else None )

   # operational port shape rate
   if not isLagIntf or lagMem:
      # only for physical interfaces or lag having atleast one member
      # operational shape rate is populated
      if qosHwStatus.interfaceDetailsSupported:
         intfHwShapeRate = None
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if intfName in sliceHwStatus.hwShapeRates and  \
               sliceHwStatus.hwShapeRates[ intfName ].intfShapeRate.rate != \
                  tacShapeRateVal.invalid:
               intfHwShapeRate = sliceHwStatus.hwShapeRates[ intfName ].\
                                 intfShapeRate
               intfQosModel.operationalPortShapeRate = \
                  populateShapeRateModel( intfHwShapeRate )
               break
      else:
         if intfStatus and intfStatus.shapeRate.rate != tacShapeRateVal.invalid:
            intfQosModel.operationalPortShapeRate = \
               populateShapeRateModel( intfStatus.shapeRate )

   if not isSubIntf or subIntfHwStatus.subIntfShapeRateSupported :
      intfQosModel.txQueueQosModel = showQosIntfTxQueues( intf, sliceName=sliceName,
                                                          mode=mode )

   if getIntfStatus( intf ) and subIntfHwStatus.subIntfSchedulingGroupSupported \
      and not isSubIntf:
      intfQosModel.schedulingGroupModel = populateQosSchedulingGroupsForIntf( intf )

   # configured burst-size
   if qosHwStatus.shapeRateBurstSizeConfigSupported:
      intfQosModel.burstSizeConfigSupported = True
      intfQosModel.configuredBurstSize = None
      intfQosModel.operationalBurstSize = None
      if intfConfig and intfConfig.shapeRate:
         intfQosModel.configuredBurstSize = populateBurstSizeModel(
            intfConfig.shapeRate.burstSize )
      if qosHwStatus.interfaceDetailsSupported:
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if intfName in sliceHwStatus.hwShapeRates:
               intfQosModel.operationalBurstSize = populateBurstSizeModel(
                  sliceHwStatus.hwShapeRates[ intfName ].intfShapeRate.burstSize )
               break
      else:
         if intfStatus and intfStatus.shapeRate:
            intfQosModel.operationalBurstSize = populateBurstSizeModel(
               intfStatus.shapeRate.burstSize ) 

   return intfQosModel

def findServicePolicyForIntf( intf, emapType, directions=None ):
   if directions is None:
      directions = [ tacDirection.input, tacDirection.output ]
   pmapsFound = []
   if isLagPort( intf.name ):
      spConfigOrStatus = qosConfig.servicePolicyConfig
      for direction in directions:
         for key, spObj in spConfigOrStatus.items():
            if intf.name in spObj.intfIds:
               if key.direction == direction and key.type == emapType:
                  pmapsFound.append( ( key.pmapName, key.direction ) )
   else:
      spConfigOrStatus = qosStatus.servicePolicyStatus
      for direction in directions:
         for key, spObj in spConfigOrStatus.items():
            for intfPair in spObj.intfIds:
               if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intfPair.lagIntf ):
                  # Members of port-channels will show policy information
                  # only from parent port-channel and not from port-channel
                  # subinterfaces.
                  continue
               if intfPair.intfId == intf.name:
                  if key.direction == direction and key.type == emapType:
                     pmapsFound.append( ( key.pmapName, key.direction ) )
   return pmapsFound

def ShowQosIntfServicePolicy( intf, emapType ):
   intfServicePolicyQosModel = IntfServicePolicyQosModel()
   pmapsFound = findServicePolicyForIntf( intf, emapType )
   for pmapName, direction in pmapsFound:
      policyMapsContainer = PMapModelContainer(
         qosConfig, qosAclConfig, qosStatus, qosHwStatus, qosSliceHwStatus,
         qosAclHwStatus, qosAclSliceHwStatus, QosLib.qosMapType, direction,
         hwEpochStatus, None )
      policyMapsContainer.populatePolicyMapQosInterface( \
            pmapName, intfServicePolicyQosModel )
   return intfServicePolicyQosModel


#------------------------------------------------------------------------------------
# The "show [mls] qos interface [ INTF ] [ detail ]" command, in "enable" mode.
# For now this command is in global config mode showing for all interfaces
# The detail token is available only when the interface-name is fabric.
#------------------------------------------------------------------------------------

def showInterfacesQos( mode, args ):
   intf = args.get( 'INTF' )
   if 'fabric' in args:
      intf = 'Fabric'
   detail = 'detail' in args 
   intfAllQosAll = None
   if intf == fabricShortName:
      intfAllQosAll = FabricAllQosAllModel()
      intf = FabricIntfCli.FabricIntf( intf, mode )
      intfs = [ intf ]
   else:
      intfAllQosAll = IntfAllQosAllModel()
      intfs = IntfCli.Intf.getAll( mode, intf, None, [ EthIntfCli.EthIntf,
                                                      SubIntfCli.SubIntf ] )
   if not intfs:
      return intfAllQosAll
   for intf in intfs:
      if intf.name.startswith( fabricShortName ):
         for sliceHwStatus in sorted( qosSliceHwStatus.itervalues() ):
            intfQosAll = IntfQosAllModel()
            intfQosModel = showQosInterface( intf, sliceHwStatus.name, detail )
            intfServicePolicyQosModel = ShowQosIntfServicePolicy( intf, qosMapType )
            
            intfQosAll.intfQosModel =  intfQosModel
            intfQosAll.intfServicePolicyQosModel = intfServicePolicyQosModel 
            if detail:
               intfAllQosAll.insert( sliceHwStatus.name, intfQosAll )
            else:
               intfAllQosAll.insert( fabricIntfName, intfQosAll )
               break
      else:
         intfQosAll = IntfQosAllModel()
         if not intf.name.startswith( 'Ethernet' ) and \
                not intf.name.startswith( 'Port-Channel' ):
            continue
         intfQosModel = showQosInterface( intf, mode=mode )
         intfServicePolicyQosModel = ShowQosIntfServicePolicy( intf, qosMapType )

         intfQosAll.intfQosModel =  intfQosModel
         intfQosAll.intfServicePolicyQosModel = intfServicePolicyQosModel 
         intfAllQosAll.insert( intf.name, intfQosAll )

   return intfAllQosAll


#------------------------------------------------------------------------------------
# The "show qos scheduling group [ GROUP ] [ INTF ]" command, in "enable" mode.
#------------------------------------------------------------------------------------
def showQosSchedulingGroup( mode, args ):
   if not subIntfHwStatus.subIntfSchedulingGroupSupported:
      return QosAllSchedulingGroupModel()

   group = args.get( 'GROUP' )
   intf = args.get( 'INTFS' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, [ EthIntfCli.EthIntf ] )
   intfQosScheduling = QosAllSchedulingGroupModel()
   subIntfs = [ intf for intf in qosStatus.intfStatus
                if Tac.Type( "Arnet::SubIntfId" ).isSubIntfId( intf ) ]

   for intf in intfs:
      if not getIntfStatus( intf ):
         continue
      members = [ subi for subi in subIntfs if subi.startswith( intf.name ) ]
      intfSchedulingGroup = populateQosSchedulingGroupsForIntf( intf, group=group,
                                                                members=members )
      if intfSchedulingGroup is not None and \
         not intfSchedulingGroup == IntfSchedulingGroupWithMembersModel():
         intfQosScheduling.insert( intf.name, intfSchedulingGroup )

   return intfQosScheduling

#------------------------------------------------------------------------------------
# The "show qos scheduling hierarchy [ INTF ]" command, in "enable" mode.
#------------------------------------------------------------------------------------
def showQosSchedulingHierarchy( mode, args ):
   if not subIntfHwStatus.subIntfSchedulingGroupSupported:
      return QosAllSchedulingHierarchyModel()

   allIntfHierarchy = QosAllSchedulingHierarchyModel()
   intf = args.get( 'INTFS' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, SubIntfCli.SubIntf )
   for intf in intfs:
      if not getIntfStatus( intf ):
         continue
      intfHierarchy = populateSchedulingHierarchy( intf, mode )
      allIntfHierarchy.interfaces[ intf.name ] = intfHierarchy
   return allIntfHierarchy

#------------------------------------------------------------------------------------
# The "show [ mls ] qos interfaces [ INTF ] trust" command, in "enable" mode.
#------------------------------------------------------------------------------------

def convertInvalidStrToDefault( value, invalid ):
   if value == invalid:
      return "DEFAULT"
   return "%s" % str( value ).upper() 

def showInterfacesTrust( mode, args ):
   intf = args.get( 'INTF' )
   intfTrustModeCollection = IntfTrustModeCollectionModel()

   intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )
   if not intfs:
      return intfTrustModeCollection
   for intf in intfs:
      if not intf.name.startswith( 'Ethernet' ) and \
             not intf.name.startswith( 'Port-Channel' ):
         continue
      intfTrustMode = showTrustInterface( intf )
      intfTrustModeCollection.intfTrustModeCollection[ intf.name ] = intfTrustMode

   return intfTrustModeCollection

def showTrustInterface( intf ):

   intfTrustModeModel = IntfTrustModeModel()

   operTrustMode = tacTrustMode.invalid 
   if isLagPort( intf.name ):
      lagMem = getConfiguredLagMembers( intf.name )
      # For Lag, pick up operTrustMode from intfConfig if configured
      if intf.name in qosConfig.intfConfig:
         intfConfig = qosConfig.intfConfig[ intf.name ]
         operTrustMode = intfConfig.trustMode
      else:
         # Pick up the operTrustMode from its members if present.
         if lagMem:
            # choosing the first configured member.
            firstMember = lagMem[ 0 ]
            if firstMember in qosStatus.intfStatus:
               operTrustMode = qosStatus.intfStatus[ firstMember ].trustMode
   else: # For phyIntf, pick up operTrustMode from intfStatus
      intfStatus = qosStatus.intfStatus.get( intf.name )
      if intfStatus:
         intfStatus = qosStatus.intfStatus[ intf.name ]
         operTrustMode = intfStatus.trustMode

   # Pick up configured trust mode for interface
   cfgTrustMode = tacTrustMode.invalid 
   if intf.name in qosConfig.intfConfig:
      intfConfig = qosConfig.intfConfig[ intf.name ]
      cfgTrustMode = intfConfig.trustMode

   intfTrustModeModel.intfTrustModeParameters = IntfTrustModeParametersModel()
   intfTrustModeModel.intfTrustModeParameters.operationalTrustMode = \
       compareAttributeToDefault( qosHwStatus, "defaultTrustMode", operTrustMode,
                                  tacTrustMode.invalid )
   intfTrustModeModel.intfTrustModeParameters.configuredTrustMode = \
       convertInvalidStrToDefault( cfgTrustMode, tacTrustMode.invalid )
   return intfTrustModeModel

#------------------------------------------------------------------------------------
# The "show [ mls ] qos random-detect ecn" command, in "enable" mode.
# The "show [ mls ] qos ecn" command, in "enable" mode.
#------------------------------------------------------------------------------------

def showGlobalEcn( mode, args ):
   allEcn = 'random-detect' not in args
   globalEcn = GlobalAllEcnModel() if allEcn else GlobalEcnModel()
   globalEcn.globalEcnSupported = qosHwStatus.globalEcnSupported
   globalEcn.globalEcnWeightSupported = qosHwStatus.globalEcnWeightSupported
   globalEcn.delayEcnSupported = qosHwStatus.ecnDelaySupported
   globalEcnStatus = qosStatus.globalEcnStatus
   if globalEcnStatus:
      globalEcn.globalEcnParameters = EcnParametersModel()
      globalEcn.globalEcnParameters.minThreshold = globalEcnStatus.minThd
      globalEcn.globalEcnParameters.maxThreshold = globalEcnStatus.maxThd
      globalEcn.globalEcnParameters.unit = globalEcnStatus.unit
      globalEcn.globalEcnParameters.weight = globalEcnStatus.weight
      if globalEcnStatus.unit == tacQueueThresholdUnit.segments:
         globalEcn.globalEcnParameters.segmentSizeInBytes = \
             qosHwStatus.ecnSegmentSizeInBytes
   if allEcn and \
      qosStatus.ecnDelayThreshold != tacEcnDelayThreshold:
      globalEcn.globalEcnDelayParameters = EcnDelayParametersModel()
      globalEcn.globalEcnDelayParameters.configuredThreshold = \
            qosStatus.ecnDelayThreshold.threshold / 1000.0
      actualThresh = \
            ( qosStatus.ecnDelayThreshold.threshold /
              qosHwStatus.ecnDelayGranularity.threshold ) * \
            qosHwStatus.ecnDelayGranularity.threshold
      globalEcn.globalEcnDelayParameters.threshold = actualThresh / 1000.0
   return globalEcn
   
#------------------------------------------------------------------------------------
# The "show [ mls ] qos interfaces [ INTF | fabric ] random-detect ecn" command, in 
# "enable" mode.
# The "show [ mls ] qos interfaces [ INTF | fabric ] ecn" command, in "enable" mode.
#------------------------------------------------------------------------------------

def showInterfacesEcn( mode, args ):
   intf = args.get( 'INTF' )
   if 'fabric' in args:
      intf = 'Fabric'
   allEcn = 'random-detect' not in args 
   intfEcnCollection = IntfAllEcnCollectionModel() if allEcn else \
                       IntfEcnCollectionModel()
   if intf == fabricShortName:
      intf = FabricIntfCli.FabricIntf( intf, mode )
      intfs = [ intf ]
   else:
      intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )
   if not intfs:
      return intfEcnCollection
   for intf in intfs:
      if intf.name.startswith( fabricShortName ):
         intfName = fabricIntfName
      else:
         intfName = intf.name
      if not intfName.startswith( 'Ethernet' ) and \
             not intfName.startswith( 'Port-Channel' ) and \
             not intfName.startswith( fabricShortName ):
         continue
      intfEcn = showEcnInterface( intf, allEcn )
      intfEcnCollection.intfEcnCollection[ intfName ] = intfEcn

   return intfEcnCollection

def showEcnInterface( intf, allEcn=True ):
   if intf.name.startswith( fabricShortName ):
      intfName = fabricIntfName
   else:
      intfName = intf.name
   intfStatus = qosStatus.intfStatus.get( intfName )
   if intfStatus:
      intfStatus = qosStatus.intfStatus[ intfName ]   

   intfEcnModel = IntfEcnModel()

   if 0 == qosHwStatus.numTxQueueSupported:
      return intfEcnModel

   for hwtxqid in xrange( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         prefix = clitxq.type[ :2 ].upper()
         txQName = prefix + str( clitxq.id )
         if prefix != 'UC':
            continue
      else:
         txQName = str( clitxq.id )

      txQueueEcn = TxQueueEcnModel()
      txQueueEcn.txQueue = txQName
      txQueueEcn.ecnMarkProbSupported = qosHwStatus.ecnMarkProbSupported
      txQueueEcn.ecnWeightSupported = qosHwStatus.ecnWeightSupported
      ecnSetting = None
      delayThresh = tacEcnDelayThreshold
      delayThreshSrc = 'none'
      if isLagPort( intfName ): # For Lag read from input/config
         if intfName in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intfName ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
            if txQueueConfig:
               if txQueueConfig.ecnConfig:
                  ecnSetting = txQueueConfig.ecnConfig
               if txQueueConfig.delayEcnEnabled and \
                  qosStatus.ecnDelayThreshold != tacEcnDelayThreshold:
                  delayThresh = txQueueConfig.ecnDelayThreshold \
                     if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold \
                     else qosStatus.ecnDelayThreshold
                  delayThreshSrc = 'local' \
                        if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold \
                        else 'global'
      elif intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )
         txQueueConfig = None
         if intf.name in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intf.name ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
         if txQueueStatus:
            if txQueueStatus.ecnStatus:
               ecnSetting = txQueueStatus.ecnStatus
            if txQueueStatus.delayEcnEnabled:
               delayThresh = txQueueStatus.ecnDelayThreshold
               if txQueueConfig:
                  delayThreshSrc = 'local' \
                     if txQueueConfig.ecnDelayThreshold != tacEcnDelayThreshold \
                     else 'global'

      if ecnSetting:
         txQueueEcn.txQueueEcnParameters = EcnParametersModel()
         txQueueEcn.txQueueEcnParameters.minThreshold = ecnSetting.minThd
         txQueueEcn.txQueueEcnParameters.maxThreshold = ecnSetting.maxThd
         txQueueEcn.txQueueEcnParameters.unit = ecnSetting.unit
         txQueueEcn.txQueueEcnParameters.maxDroprate = ecnSetting.maxDroprate
         txQueueEcn.txQueueEcnParameters.weight = ecnSetting.weight
         if ecnSetting.unit == tacQueueThresholdUnit.segments:
            txQueueEcn.txQueueEcnParameters.segmentSizeInBytes = \
                qosHwStatus.ecnSegmentSizeInBytes

      txQueueEcn.txQueueEcnDelaySupported = qosHwStatus.ecnDelaySupported
      if allEcn and delayThresh != tacEcnDelayThreshold:
         txQueueEcn.txQueueEcnDelayParameters = EcnDelayParametersModel()
         txQueueEcn.txQueueEcnDelayParameters.threshold = \
               ( delayThresh.threshold /
                 qosHwStatus.ecnDelayGranularity.threshold ) * \
               qosHwStatus.ecnDelayGranularity.threshold / 1000.0
         txQueueEcn.txQueueEcnDelayParameters.configuredThreshold = \
               delayThresh.threshold / 1000.0
         txQueueEcn.txQueueEcnDelayParameters.source = delayThreshSrc

      intfEcnModel.txQueueEcnList.append( txQueueEcn )

   return intfEcnModel

#------------------------------------------------------------------------------------
# The "show [ mls ] qos interfaces [ INTF ] random-detect drop" command, in 
# "enable" mode.
#------------------------------------------------------------------------------------
def showInterfacesWred( mode, args ):  
   intf = args.get( 'INTF' )
   intfWredCollection = IntfWredCollectionModel()

   intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )
   if not intfs:
      return intfWredCollection
   for intf in intfs:
      if not intf.name.startswith( 'Ethernet' ) and \
             not intf.name.startswith( 'Port-Channel' ):
         continue
      intfWred = showWredInterface( intf )
      intfWredCollection.intfWredCollection[ intf.name ] = intfWred

   return intfWredCollection

def showWredInterface( intf ):

   intfStatus = qosStatus.intfStatus.get( intf.name )
   if intfStatus:
      intfStatus = qosStatus.intfStatus[ intf.name ]  

   intfWredModel = IntfWredModel()

   if 0 == qosHwStatus.numTxQueueSupported:
      return intfWredModel

   for hwtxqid in xrange( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         prefix = clitxq.type[ :2 ].upper()
         txQName = prefix + str( clitxq.id )
         if prefix != 'UC':
            continue
      else:
         txQName = str( clitxq.id )

      txQueueWred = TxQueueWredModel()
      txQueueWred.txQueue = txQName
      txQueueWred.wredWeightSupported = qosHwStatus.wredWeightSupported

      wredSetting = None
      if isLagPort( intf.name ): # For Lag read from config
         if intf.name in qosInputConfig.intfConfig:
            intfConfig = qosInputConfig.intfConfig[ intf.name ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
            if txQueueConfig and txQueueConfig.wredConfig:
               wredSetting = txQueueConfig.wredConfig
      elif intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )
         if txQueueStatus and txQueueStatus.wredStatus:
            wredSetting = txQueueStatus.wredStatus

      if wredSetting:
         txQueueWred.txQueueWredParameters = WredParametersModel()
         txQueueWred.txQueueWredParameters.minThreshold = wredSetting.minThd
         txQueueWred.txQueueWredParameters.maxThreshold = wredSetting.maxThd
         txQueueWred.txQueueWredParameters.unit = wredSetting.unit
         txQueueWred.txQueueWredParameters.maxDroprate = wredSetting.maxDroprate
         txQueueWred.txQueueWredParameters.weight = wredSetting.weight
         if wredSetting.unit == tacQueueThresholdUnit.segments:
            txQueueWred.txQueueWredParameters.segmentSizeInBytes = \
                qosHwStatus.ecnSegmentSizeInBytes
      
      intfWredModel.txQueueWredList.append( txQueueWred )

   return intfWredModel

#------------------------------------------------------------------------------------
# The "show qos [mls] interfaces <interface-name> drop-thresholds" command,
# in "enable" mode.
#------------------------------------------------------------------------------------

def showInterfacesDropThresholds( mode, args ):
   intfDropThresholds = IntfDropThresholdsCollectionModel()
   intfDropThresholds.dropThresholdsSupported = qosHwStatus.\
                                                dropPrecedenceThresholdSupported
   intf = args.get( 'INTF' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, EthIntfCli.EthIntf )
   if not intfs:
      return intfDropThresholds
   for intf in intfs:
      intfName = intf.name
      if not intfName.startswith( 'Ethernet' ) and \
         not intfName.startswith( 'Port-Channel' ):
         continue
      intfDropThreshold = showDropThresholdInterface( intfName )
      if intfDropThreshold.txQueueDropThresholdsList:
         intfDropThresholds.intfDropThresholdsCollection[
            intfName ] = intfDropThreshold
   return intfDropThresholds

def showDropThresholdInterface( intfName ):
   intfStatus = qosStatus.intfStatus.get( intfName )

   intfDropThresholdsModel = IntfDropThresholdsModel()
   if not qosHwStatus.numTxQueueSupported:
      return intfDropThresholdsModel

   for hwtxqid in xrange( qosHwStatus.numTxQueueSupported - 1, -1, -1 ):
      clitxq = qosHwStatus.hwTxQueue[ hwtxqid ].txQueue
      if qosHwStatus.numMulticastQueueSupported != 0:
         prefix = clitxq.type[ :2 ].upper()
         txQName = prefix + str( clitxq.id )
         if prefix != 'UC':
            continue
      else:
         txQName = str( clitxq.id )

      txQueueDpThresholds = TxQueueDropThresholdsModel()
      dpThresholdSetting = None
      txQueueDpThresholds.txQueue = txQName
      if isLagPort( intfName ): # For Lag read from input/config
         if intfName in qosConfig.intfConfig:
            intfConfig = qosConfig.intfConfig[ intfName ]
            txQueueConfig = intfConfig.txQueueConfig.get( clitxq )
            if txQueueConfig:
               if txQueueConfig.dropThresholds:
                  dpThresholdSetting = txQueueConfig.dropThresholds
      elif intfStatus:
         txQueueStatus = intfStatus.txQueueStatus.get( clitxq )
         if txQueueStatus and txQueueStatus.dropThresholds:
            dpThresholdSetting = txQueueStatus.dropThresholds

      if dpThresholdSetting:
         txQueueDpThresholds.dropThresholds.update( dpThresholdSetting )
         intfDropThresholdsModel.txQueueDropThresholdsList.append(
            txQueueDpThresholds )

   return intfDropThresholdsModel

class ShowInterfacesDropThresholds( ShowCommand.ShowCliCommandClass ):
   syntax = "show qos interfaces [ INTF ] drop-thresholds"

   data = {
      'qos': nodeQosForShow,
      'interfaces': 'Show QoS status for a specific interface',
      'INTF': IntfRangeMatcher( explicitIntfTypes=(
         EthIntfCli.EthPhyAutoIntfType, LagIntfCli.LagAutoIntfType ) ),
      'drop-thresholds': CliCommand.guardedKeyword( 'drop-thresholds', 
         helpdesc='Show tail drop-threshold values', 
         guard=guardDropPrecedenceThreshold )
   }
   handler = showInterfacesDropThresholds
   cliModel = IntfDropThresholdsCollectionModel

BasicCli.addShowCommandClass( ShowInterfacesDropThresholds )

#------------------------------------------------------------------------------------
# The "show [ mls ] qos maps" command, in "enable" mode.
#------------------------------------------------------------------------------------
def showQosMaps( mode, args ):
   globalQosMaps = GlobalQosMapsModel()
   
   globalQosMaps.numTcSupported = qosHwStatus.numTcSupported
   globalQosMaps.numTxQueueSupported = qosHwStatus.numTxQueueSupported
   globalQosMaps.numMulticastQueueSupported = qosHwStatus.numMulticastQueueSupported

   globalQosMaps.cosRewriteSupported = qosHwStatus.cosRewriteSupported
   globalQosMaps.cosRewriteDisableSupported = qosHwStatus.cosRewriteDisableSupported
   globalQosMaps.dscpRewriteSupported = qosHwStatus.dscpRewriteSupported
   globalQosMaps.tcToDscpMapSupported = qosHwStatus.tcToDscpMapSupported
   globalQosMaps.dscpToDpMapSupported = qosHwStatus.dscpToDpMapSupported
   globalQosMaps.dynamicExpMappingSupported = qosHwStatus.dynamicExpMappingSupported

   globalQosMaps.cosRewriteEnabled = qosStatus.cosRewriteEnabled
   globalQosMaps.dscpRewriteEnabled = qosStatus.dscpRewriteEnabled
   globalQosMaps.tcToPriorityGroupMapSupported = \
      qosHwStatus.tcToPriorityGroupMapSupported
   globalQosMaps.cosToTcProfilePerInterfaceSupported = \
      qosHwStatus.cosToTcPerInterfaceSupported
   globalQosMaps.tcToCosMapPerInterfaceSupported = \
      ( qosHwStatus.tcToCosPerInterfaceSupported or
        qosHwStatus.cpuTcToCosPerInterfaceSupported )
   globalQosMaps.dscpRewritePerInterfaceSupported = \
      qosHwStatus.dscpRewritePerInterfaceSupported
   globalQosMaps.pwDecapDscpQosMapSupported = qosHwStatus.pwDecapDscpQosMapSupported

   def getArrayValues( low, high, array, invalid, defaultArray, defaultVal=None ):
      values = {}
      for index in xrange( low, high + 1 ):
         if array[ index ] == invalid:
            value = defaultVal if defaultVal != None else defaultArray[ index ]
         else:
            value = array[ index ]
         if value == invalid:
            value = INVALID
         values[ index ] = value
      return values
      
   globalQosMaps.cosToTcMap = getArrayValues( 0, 7, qosStatus.cosToTcMap, 
           tacTrafficClass.invalid, qosHwStatus.defaultCosToTcMap ) 

   globalQosMaps.dscpToTcMap = getArrayValues( 0, 63, qosStatus.dscpToTcMap, 
           tacTrafficClass.invalid, qosHwStatus.defaultDscpToTcMap )

   if qosHwStatus.dscpToDpMapSupported:
      globalQosMaps.dscpToDpMap = getArrayValues( 0, 63, qosStatus.dscpToDpMap,
        tacDropPrecedence.invalid, [], defaultVal=qosHwStatus.defaultDropPrecedence )
   if qosHwStatus.dynamicExpMappingSupported:
      globalQosMaps.expToTcMap = getArrayValues( 0, 7, qosStatus.expToTcMap,
            tacTrafficClass.invalid, qosHwStatus.defaultExpToTcMap )

   if qosHwStatus.cosRewriteSupported:
      globalQosMaps.tcToCosMap = getArrayValues( 0, qosHwStatus.numTcSupported - 1, 
        qosStatus.tcToCosMap, tacCos.invalid, qosHwStatus.defaultTcToCosMap )

   if qosHwStatus.tcToDscpMapSupported:
      globalQosMaps.tcToDscpMap = getArrayValues( 0, qosHwStatus.numTcSupported - 1, 
        qosStatus.tcToDscpMap, tacDscp.invalid, qosHwStatus.defaultTcToDscpMap )

   if qosHwStatus.dynamicExpMappingSupported:
      globalQosMaps.tcToExpMap = getArrayValues( 0, qosHwStatus.numTcSupported - 1, 
         qosStatus.tcToExpMap, tacExp.invalid, qosHwStatus.defaultTcToExpMap )

   if qosHwStatus.numTxQueueSupported:
      globalQosMaps.tcToTxQueueMap = getArrayValues( 0,
        qosHwStatus.numTcSupported - 1, qosStatus.tcToTxQueueMap,
        tacTxQueueId.invalid, qosHwStatus.defaultTcToTxQueueMap )

      if qosHwStatus.numMulticastQueueSupported:
         globalQosMaps.tcToMcTxQueueMap = getArrayValues( 0, 
           qosHwStatus.numTcSupported - 1, qosStatus.tcToMcTxQueueMap,
           tacTxQueueId.invalid, qosHwStatus.defaultTcToMcTxQueueMap )

   if qosHwStatus.tcToPriorityGroupMapSupported:
      globalQosMaps.tcToPriorityGroupMap = getArrayValues( 0,
         qosHwStatus.numTcSupported - 1, qosStatus.tcToPriorityGroupMap,
         tacPriorityGroup.invalid, qosHwStatus.defaultTcToPriorityGroupMap )

   if qosHwStatus.cosToTcPerInterfaceSupported:
      cosToTcProfileAllModel = CosToTcProfileAllModel()
      for profileName in qosStatus.cosToTcProfile:
         cosToTcProfileModel = CosToTcProfileModel()
         cosToTcMap = qosStatus.cosToTcProfile[ profileName ].cosToTcMap
         for cos in cosToTcMap:
            cosToTcProfileModel.cosToTcMap[ cos ] = cosToTcMap[ cos ]
         cosToTcProfileAllModel.cosToTcProfiles[ profileName ] = \
               cosToTcProfileModel
      globalQosMaps.cosToTcProfileAllModel = cosToTcProfileAllModel
   
   if ( qosHwStatus.tcToCosPerInterfaceSupported or
        qosHwStatus.cpuTcToCosPerInterfaceSupported ):
      tcToCosMapAllModel = TcToCosMapAllModel()
      for mapName in qosStatus.tcToCosNamedMap:
         tcToCosMapModel = TcToCosMapModel()
         tcToCosNamedMap = qosStatus.tcToCosNamedMap[ mapName ].tcToCosNamedMap
         for tcDpPair in tcToCosNamedMap:
            tcDpIndex = 'tc-%d dp-%d' % ( tcDpPair.tc, tcDpPair.dp )
            tcToCosMapModel.tcToCosNamedMap[ tcDpIndex ] = \
                                                          tcToCosNamedMap[ tcDpPair ]
         tcToCosMapAllModel.tcToCosNamedMaps[ mapName ] = tcToCosMapModel
      globalQosMaps.tcToCosMapAllModel = tcToCosMapAllModel

   if qosHwStatus.pwDecapDscpQosMapSupported:
      dscpToTcMapAllModel = DscpToTcMapAllModel()
      for mapName in qosStatus.dscpToTcNamedMap:
         dscpToTcMapModel = DscpToTcMapModel()
         dscpToTcNamedMap = qosStatus.dscpToTcNamedMap[ mapName ].dscpToTcNamedMap
         for dscp in dscpToTcNamedMap:
            dscpToTcMapModel.dscpToTcNamedMap[ dscp ] = dscpToTcNamedMap[ dscp ]
                                                          
         dscpToTcMapAllModel.dscpToTcNamedMaps[ mapName ] = dscpToTcMapModel
      globalQosMaps.dscpToTcMapAllModel = dscpToTcMapAllModel

   if qosHwStatus.dscpRewritePerInterfaceSupported:
      dscpRewriteMapAllModel = DscpRewriteMapAllModel()
      for mapName in qosStatus.dscpRewriteMap:
         dscpRewriteMapModel = DscpRewriteMapModel()
         tcToDscpMap = qosStatus.dscpRewriteMap[ mapName ].tcToDscpMap
         for tc in range( qosHwStatus.numTcSupported ):
            dscpRewriteMapModel.tcToDscpMap[ tc ] = tcToDscpMap[ tc ]
         dscpRewriteMapAllModel.dscpRewriteMaps[ mapName ] = \
               dscpRewriteMapModel
      globalQosMaps.dscpRewriteMapAllModel = dscpRewriteMapAllModel

   return globalQosMaps

#------------------------------------------------------------------------------------
# The "qos map traffic-class <tc's> to tx-queue <0-7>" command, in "global config" 
# mode.
#------------------------------------------------------------------------------------
def setTrafficClassToTxQueueMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or args.get( 'TC_LIST' ).values()
   txQueueId = ( args.get( 'TXQ_VALUE' ) if 'tx-queue' in args
                                         else args.get( 'UCTXQ_VALUE' ) )
   no = CliCommand.isNoOrDefaultCmd( args )
   if 7 in tcList and qosHwStatus.tc7ToAnyTxQueueRestricted:
      if no:
         # Avoid removing from the list passed to this function
         tcList = [ tc for tc in tcList if tc != 7 ]
      else:
         mode.addError( 'Traffic class 7 to tx-queue mapping cannot be changed' )
         return
   for tc in tcList:
      if no or txQueueId == qosHwStatus.defaultTcToTxQueueMap[ tc ]:
         qosInputConfig.tcToTxQueueMap[ tc ] = tacTxQueueId.invalid
      else:
         qosInputConfig.tcToTxQueueMap[ tc ] = txQueueId

#------------------------------------------------------------------------------------
# The "qos map traffic-class <tc's> to mc-tx-queue <0-3>" command, in "global config"
# mode.
#------------------------------------------------------------------------------------
def setTrafficClassToMcTxQueueMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or args.get( 'TC_LIST' ).values()
   txQueueId = args.get( 'MCTXQ_VALUE' )
   for tc in tcList:
      if CliCommand.isNoOrDefaultCmd( args ) or \
            txQueueId == qosHwStatus.defaultTcToMcTxQueueMap[ tc ]:
         qosInputConfig.tcToMcTxQueueMap[ tc ] = tacTxQueueId.invalid
      else:
         qosInputConfig.tcToMcTxQueueMap[ tc ] = txQueueId

#------------------------------------------------------------------------------------
# The "qos map traffic-class <tc's> to priority-group <0-7>" command, in
# "global config" mode.
#------------------------------------------------------------------------------------
def guardTcToPriorityGroupMap( mode, token ):
   if qosHwStatus.tcToPriorityGroupMapSupported:
      return None
   return CliParser.guardNotThisPlatform

def priorityGroupVal( mode=None ):
   pgsSupported = {}
   pgList = []
   if qosHwStatus.hwInitialized:
      for pg in list( qosHwStatus.supportedPgs ):
         pgList.append( pg )
   else:
      for pg in range( 0, qosHwStatus.controlPg ) + [ qosHwStatus.controlPg + 1 ]:
         pgList.append( pg )
   for pg in pgList:
      pgsSupported.update( { str( pg ) : 'Priority Group id %s' % ( str( pg ) ) } )
   return pgsSupported

def setTrafficClassToPriorityGroupMap( mode, args ):
   tcList = args.get( 'TC_VALUE' ) or args.get( 'TC_LIST' ).values()
   pgId = args.get( 'PRIORITYGROUPVALUE' )
   no = CliCommand.isNoOrDefaultCmd( args )
   for tc in tcList:
      if no or int( pgId ) == qosHwStatus.defaultTcToPriorityGroupMap[ tc ]:
         qosInputConfig.tcToPriorityGroupMap[ tc ] = tacPriorityGroup.invalid
      else:
         qosInputConfig.tcToPriorityGroupMap[ tc ] = int( pgId )

def ecnDelayThresholdUsRangeFn( mode ):
   return ( qosHwStatus.minEcnDelayThreshold.threshold / 1000,
            qosHwStatus.maxEcnDelayThreshold.threshold / 1000 )

def setEcnDelayThresholdUs( mode, config, noOrDefaultKw=None, thresholdUs=None ):
   if noOrDefaultKw or not thresholdUs:
      config.ecnDelayThreshold = tacEcnDelayThreshold
   else:
      # Currently we store as ns internally while displaying microseconds in CLI.
      config.ecnDelayThreshold = \
          Tac.Value( 'Qos::EcnDelayThreshold',
                     tacEcnDelayThresholdUnit.ecnDelayThresholdNs,
                     thresholdUs * 1000 )

def setTxQueueEcnDelayThresholdUs( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   threshold = args.get( 'THRESHOLD' )
   if qosInputConfig.ecnDelayThreshold == tacEcnDelayThreshold:
      print "WARNING: Delay based ECN is disabled globally." 
   if isinstance( mode.parent_, QosProfileMode ):
      profile = mode.parent_.qosProfileModeContext.currentEntry_
      txQueueConfig = profile.txQueueConfig.get( mode.txQueue )
      if not txQueueConfig:
         if noOrDefaultKw:
            return
         txQueueConfig = profile.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )

      txQueueConfig.delayEcnEnabled = False if noOrDefaultKw else True
      setEcnDelayThresholdUs( mode, txQueueConfig, noOrDefaultKw, threshold )
      setQosProfileConfig( profile )
   else:
      intfList = getIntfListFromMode( mode.parent_ )
      for intf in intfList: 
         timestamp = Tac.now()
         intfConfig = qosInputConfig.intfConfig.get( intf )
         if intfConfig is None:
            if noOrDefaultKw:
               continue
            intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.newMember(
               mode.txQueue, tacTxQueuePriority.priorityInvalid, tacPercent.invalid,
               invalidShapeRate, invalidGuaranteedBw )
         else:
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if txQueueConfig is None:
               if noOrDefaultKw:
                  continue
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )

         prevEcnDelayEnabled = txQueueConfig.delayEcnEnabled
         prevEcnDelayThreshold = txQueueConfig.ecnDelayThreshold
         txQueueConfig.delayEcnEnabled = False if noOrDefaultKw else True
         setEcnDelayThresholdUs( mode, txQueueConfig, noOrDefaultKw, threshold )
         setIntfConfig( intf )

         if cliBlockingFail( mode, timestamp, tacFeatureName.delayEcn, "Delay ECN",
                             intf ):
            intfConfig = qosInputConfig.intfConfig.get( intf )
            if intfConfig is None:
               intfConfig = qosInputConfig.intfConfig.newMember( intf )
            txQueueConfig = intfConfig.txQueueConfig.get( mode.txQueue )
            if not txQueueConfig:
               txQueueConfig = intfConfig.txQueueConfig.newMember(
                  mode.txQueue, tacTxQueuePriority.priorityInvalid,
                  tacPercent.invalid, invalidShapeRate, invalidGuaranteedBw )
            txQueueConfig.delayEcnEnabled = prevEcnDelayEnabled
            txQueueConfig.ecnDelayThreshold = prevEcnDelayThreshold
            setIntfConfig( intf )

def setGlobalEcnDelayThresholdUs( mode, args ):
   threshold = args.get( 'ECNDELAYTHRESHOLDUSVALUE' )
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   timestamp = Tac.now()
   prevEcnDelayThreshold = qosInputConfig.ecnDelayThreshold
   setEcnDelayThresholdUs( mode, qosInputConfig, noOrDefaultKw, threshold )
   if cliBlockingFail( mode, timestamp, tacFeatureName.delayEcn, "Delay ECN" ):
      qosInputConfig.ecnDelayThreshold = prevEcnDelayThreshold

nodeEcnDelay = CliCommand.guardedKeyword( 'ecn',
      helpdesc='General ECN configuration', guard=guardEcnDelay )
matcherEcnDelayThresholdUsValue = CliMatcher.DynamicIntegerMatcher( 
      rangeFn=ecnDelayThresholdUsRangeFn, 
      helpdesc="Threshold value in microseconds" )
#-----------------------------------------------------------------------------------
# [ mls ] qos ecn delay threshold <n> [microseconds], in "global config" mode
# ecn delay, in "uc-tx-queue" mode
# ecn delay threshold <n> [microseconds], in "uc-tx-queue" mode
#----------------------------------------------------------------------------------- 
class IntfUcTxQueueEcnDelayThCmd( CliCommand.CliCommandClass ):
   syntax = 'ecn delay [ threshold THRESHOLD [ microseconds ] ]'
   noOrDefaultSyntax = 'ecn delay [ threshold ... ]'
   data = {
      'ecn': nodeEcnDelay,
      'delay': 'Configure parameters for delay-based ECN',
      'threshold': 'Configure delay threshold',
      'THRESHOLD': matcherEcnDelayThresholdUsValue,
      'microseconds': 'Microseconds',
   }

   handler = setTxQueueEcnDelayThresholdUs
   noOrDefaultHandler = handler

IntfUcTxQueueModelet.addCommandClass( IntfUcTxQueueEcnDelayThCmd )

#------------------------------------------------------------------------------------
# The "show [ mls ] qos ecn [delay|counters] " command, in "enable" mode.
#------------------------------------------------------------------------------------

def showEcnDelay( mode, args ):
   ecn = EcnDelayModel()
   if qosStatus.ecnDelayThreshold != tacEcnDelayThreshold:
      ecn.ecnDelayParameters = EcnDelayParametersModel()
      ecn.ecnDelayParameters.configuredThreshold = \
            qosStatus.ecnDelayThreshold.threshold / 1000.0
      actualThresh = \
            ( qosStatus.ecnDelayThreshold.threshold /
              qosHwStatus.ecnDelayGranularity.threshold ) * \
            qosHwStatus.ecnDelayGranularity.threshold
      ecn.ecnDelayParameters.threshold = actualThresh / 1000.0
      if qosStatus.ecnDelayThreshold.unit != \
         tacEcnDelayThresholdUnit.ecnDelayThresholdNs:
         assert False, "Unknown unit"
   return ecn

def showEcnCounters( mode, args ):
   ecnCounter = EcnCountersModel()
   for sliceCounter in qosHwCounter.sliceHwCounter.itervalues():
      for chipName, chipCount in sliceCounter.perChipCount.iteritems():
         ecnCounter.insert( chipName, chipCount.ecnCount )
   return ecnCounter

#------------------------------------------------------------------------------------
# The "[ mls ] qos random-detect ecn global-buffer minimum-threshold <min_threashold>
# maximum-threshold <max_threshold>" command, in "global config" mode.
#------------------------------------------------------------------------------------

def wredApplied( ):
   for intf in qosConfig.intfConfig:
      intfConfig = qosConfig.intfConfig[ intf ]
      for txQueue in intfConfig.txQueueConfig:
         txQueueConfig = intfConfig.txQueueConfig[ txQueue ]
         if txQueueConfig.wredConfig:
            return True
   return False

def setNoOrDefaultGlobalEcn( mode, args ):
   timestamp = Tac.now()
   prevEcnConfig = qosInputConfig.globalEcnConfig
   if not prevEcnConfig:
      return
   qosInputConfig.globalEcnConfig = None
   if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN" ):
      if prevEcnConfig:
         qosInputConfig.globalEcnConfig = ( prevEcnConfig.minThd,
                                            prevEcnConfig.maxThd, prevEcnConfig.unit,
                                            ecnMaxDroprate, prevEcnConfig.weight )
      else:
         qosInputConfig.globalEcnConfig = None

def setGlobalEcn( mode, args ):
   weight = args.get( 'WEIGHT', defaultWeight )
   minThd, maxThd, unit = args[ 'THD_RANGE' ]
   timestamp = Tac.now()
   errMsg = "Cannot apply ECN configuration. WRED configuration is already present."
   if wredApplied():
      mode.addError( errMsg )
      return
   prevEcnConfig = qosInputConfig.globalEcnConfig
   if ( prevEcnConfig and prevEcnConfig.minThd == minThd and 
        prevEcnConfig.maxThd == maxThd and prevEcnConfig.unit == unit and
        prevEcnConfig.weight == weight ):
      return
   qosInputConfig.globalEcnConfig = ( minThd, maxThd, unit,
                                      ecnMaxDroprate, weight )
   if cliBlockingFail( mode, timestamp, tacFeatureName.ecn, "ECN" ):
      if prevEcnConfig:
         qosInputConfig.globalEcnConfig = ( prevEcnConfig.minThd,
                                            prevEcnConfig.maxThd, prevEcnConfig.unit,
                                            ecnMaxDroprate, prevEcnConfig.weight )
      else:
         qosInputConfig.globalEcnConfig = None

#------------------------------------------------------------------------------------
# The "[ mls ] qos random-detect ecn allow non-ect" command, in "global config" mode.
#------------------------------------------------------------------------------------

def configureAllowNonEct( mode, args ):
   if CliCommand.isNoOrDefaultCmd( args ):
      qosInputConfig.ecnAllowNonEct = False
   elif qosHwStatus.allowNonEctWithDropPrecedence:
      qosInputConfig.ecnAllowNonEct = True
   else:
      spConfigs = [ qosInputConfig.servicePolicyConfig,
                    qosInputProfileConfig.servicePolicyConfig ]
      for spConfig in spConfigs:
         for key, _ in spConfig.iteritems():
            if pmapHasDropPrecedenceAction( key.pmapName ):
               mode.addError( "Policy-map action drop-precedence and allow "
                              "non-ect cannot be configured together." )
               return
      qosInputConfig.ecnAllowNonEct = True

#-------------------------------------------------------------------------------
# 1) To delete Qos Lag intfConfig object when the corresponding interface gets 
#    deleted.
# 2) Support for default interface command.
#-------------------------------------------------------------------------------
class QosIntfJanitor( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      t0( "QosIntfJanitor: interface", self.intf_.name, "going away" )
      intfConfig = qosInputConfig.intfConfig.get( self.intf_.name )
      profileIntfConfig = qosInputProfileConfig.intfConfig.get( self.intf_.name )
      if intfConfig is not None:
         del qosInputConfig.intfConfig[ self.intf_.name ]
      if profileIntfConfig is not None: 
         removeProfile( None, None, self.intf_.name )
         del qosInputProfileConfig.intfConfig[ self.intf_.name ]
      for spKey in qosInputConfig.servicePolicyConfig:
         spConfig = qosInputConfig.servicePolicyConfig[ spKey ]
         if self.intf_.name in spConfig.intfIds:
            delServicePolicy( None, spKey.type, spKey.pmapName, spKey.direction, 
                              self.intf_.name )
      for spKey in qosInputProfileConfig.servicePolicyConfig:
         spConfig = qosInputProfileConfig.servicePolicyConfig[ spKey ]
         if self.intf_.name in spConfig.intfIds:
            delServicePolicy( None, spKey.type, spKey.pmapName, spKey.direction, 
                              self.intf_.name, profile=True )
      if self.intf_.name in profileConfigDir.intfToProfileMap:
         del profileConfigDir.intfToProfileMap[ self.intf_.name ]
      if self.intf_.name in qosInputConfig.ecnIntfCounterConfig:
         del qosInputConfig.ecnIntfCounterConfig[ self.intf_.name ]
      deleteInterfacePolicingConfig( qosInputConfig, self.intf_.name )

#-------------------------------------------------------------------------------
# Tokens for Policy maps and class maps.
#-------------------------------------------------------------------------------
#------------------------------------------------------------------------
# qos policer <name> rate <value> {bps, kbps, mbps} burst-size <value> \
#       {bytes, kbytes, mbytes}
#------------------------------------------------------------------------
def guardNamedSharedPolicerSupported( mode, token ):
   if qosAclHwStatus.namedSharedPolicerSupported:
      return None
   return CliParser.guardNotThisPlatform

def getPolicerNameRule( mode ):
   suggestedCompletions = cliQosAclConfig.namedPolicer.members()
   return sorted( suggestedCompletions )

def copyNamedPolicer( src, dst ):
   dst.named = src.named
   dst.cir = src.cir
   dst.bc = src.bc
   dst.cirUnit = src.cirUnit
   dst.bcUnit = src.bcUnit

def identicalNamedPolicer( p1, p2 ):
   if ( p1.cir != p2.cir or p1.cirUnit != p2.cirUnit or
        p1.bc != p2.bc or p1.bcUnit != p2.bcUnit or
        p1.pir != p2.pir or p1.pirUnit != p2.pirUnit or
        p1.be != p2.be or p1.beUnit != p2.beUnit ):
      return False
   return True

def policerOutOfRange( mode, cir=None, cirUnit=None, bc=None, bcUnit=None,
                       pir=None, pirUnit=None, be=None, beUnit=None,
                       policerMode='committed' ):
   outOfRange = False
   if cirUnit == tacRateUnit.rateUnitPps:
      if bcUnit != None and bcUnit != tacBurstUnit.burstUnitPackets:
         mode.addError( 'Invalid burst size unit' )
         mode.addError( ' Burst size unit not in packets when rate unit is'
                       ' in PPS not supported' )
         outOfRange = True
      if cir < qosAclHwStatus.policePacketModeMinRatePps or \
         cir > qosAclHwStatus.policePacketModeMaxRatePps:
         mode.addError( 'Invalid lower rate value' )
         mode.addError( ' Lower rate range <%d-%d> pps' % (
            qosAclHwStatus.policePacketModeMinRatePps,
            qosAclHwStatus.policePacketModeMaxRatePps ) )
         outOfRange = True
   else:
      tcirKbps = irToKbps( cir, cirUnit )
      if tcirKbps < qosAclHwStatus.policeByteModeMinRateKbps or \
         tcirKbps > qosAclHwStatus.policeByteModeMaxRateKbps:
         mode.addError( 'Invalid lower rate value' )
         mode.addError( ' Lower rate range <%d-%d> kbps' % (
               qosAclHwStatus.policeByteModeMinRateKbps,
               qosAclHwStatus.policeByteModeMaxRateKbps ) )
         outOfRange = True
   if bcUnit == tacBurstUnit.burstUnitPackets:
      if cirUnit != None and cirUnit != tacRateUnit.rateUnitPps:
         mode.addError( 'Invalid burst size unit' )
         mode.addError( ' Burst size unit in packets when rate unit is'
                       ' not in PPS not supported' )
         outOfRange = True
      if bc < qosAclHwStatus.policePacketModeMinBurstPackets or \
         bc > qosAclHwStatus.policePacketModeMaxBurstPackets:
         mode.addError( 'Invalid burst size for lower rate' )
         mode.addError( ' Burst range <%d-%d> packets' % (
            qosAclHwStatus.policePacketModeMinBurstPackets,
            qosAclHwStatus.policePacketModeMaxBurstPackets ) )
         outOfRange = True
   else:
      tbcBytes = bsToBytes( bc, bcUnit )
      if tbcBytes < qosAclHwStatus.policeByteModeMinBurstBytes or \
         tbcBytes > qosAclHwStatus.policeByteModeMaxBurstBytes:
         mode.addError( 'Invalid burst size for lower rate' )
         mode.addError( ' Burst range <%d-%d> bytes' % (
               qosAclHwStatus.policeByteModeMinBurstBytes,
               qosAclHwStatus.policeByteModeMaxBurstBytes ) )
         outOfRange = True

   if policerMode == 'trTCM':
      # If cirUnit is in pps, make sure pirUnit is in pps and beUnit is in packets
      if cirUnit == tacRateUnit.rateUnitPps:
         if pirUnit != None and  pirUnit != tacRateUnit.rateUnitPps:
            mode.addError( 'Invalid higher rate unit' )
            mode.addError( ' Higher rate unit not in PPS when lower rate unit is in'
                            ' PPS not supported' )
            outOfRange = True
         if beUnit != None and beUnit != tacBurstUnit.burstUnitPackets:
            mode.addError( 'Invalid burst size unit for higher rate' )
            mode.addError( ' Burst size unit not in packets when rate unit'
                           '(lower or higher) is in PPS not supported' )
            outOfRange = True
      else:
         if pirUnit == tacRateUnit.rateUnitPps:
            mode.addError( 'Invalid higher rate unit' )
            mode.addError( ' Higher rate unit in PPS when lower rate unit is '
                           'not in PPS not supported' )
            outOfRange = True
         if beUnit == tacBurstUnit.burstUnitPackets:
            mode.addError( 'Invalid burst size unit for higher rate' )
            mode.addError( ' Burst size unit in packets when rate(lower or higher)'
                           ' unit is not in PPS not supported' )
            outOfRange = True

      # Range check for Pir (pps)
      if cirUnit == tacRateUnit.rateUnitPps and ( pirUnit == tacRateUnit.rateUnitPps
                                                  or pirUnit == None ):
         teirPps = pir - cir
         if teirPps < qosAclHwStatus.policePacketModeMinRatePps or \
            teirPps > qosAclHwStatus.policePacketModeMaxRatePps:
            mode.addError( 'Invalid higher rate value' )
            mode.addError( ' Higher rate range for specified lower rate is '
                           '<%d-%d> pps' % (
                           qosAclHwStatus.policePacketModeMinRatePps + cir,
                           qosAclHwStatus.policePacketModeMaxRatePps + cir ) )
            outOfRange = True

      #Range check for be (packets)
      if cirUnit == tacRateUnit.rateUnitPps and \
         ( beUnit == tacBurstUnit.burstUnitPackets or beUnit == None ):
         if be < qosAclHwStatus.policePacketModeMinBurstPackets or \
            be > qosAclHwStatus.policePacketModeMaxBurstPackets:
            mode.addError( 'Invalid burst size for higher rate' )
            mode.addError( ' Burst range <%d-%d> packets ' % (
                           qosAclHwStatus.policePacketModeMinBurstPackets,
                           qosAclHwStatus.policePacketModeMaxBurstPackets ) )
            outOfRange = True

      #Range check for cir (bps, kbps, mbps)
      if cirUnit != tacRateUnit.rateUnitPps and pirUnit != tacRateUnit.rateUnitPps:
         tpirKbps = irToKbps( pir, pirUnit )
         teirKbps = tpirKbps - tcirKbps
         if teirKbps < qosAclHwStatus.policeByteModeMinRateKbps or \
            teirKbps > qosAclHwStatus.policeByteModeMaxRateKbps:
            mode.addError( 'Invalid higher rate' )
            mode.addError( ( ' Higher rate range for specified lower rate is '
                             '<%d-%d> kbps' % (
                             qosAclHwStatus.policeByteModeMinRateKbps + tcirKbps,
                             qosAclHwStatus.policeByteModeMaxRateKbps + tcirKbps ) )
                         )
            outOfRange = True

      #Range check for be (bytes, kbytes, mbytes)
      if cirUnit != tacRateUnit.rateUnitPps and \
         beUnit != tacBurstUnit.burstUnitPackets:
         tbeBytes = bsToBytes( be, beUnit )
         if tbeBytes < qosAclHwStatus.policeByteModeMinBurstBytes or \
            tbeBytes > qosAclHwStatus.policeByteModeMaxBurstBytes:
            mode.addError( 'Invalid burst size for higher rate' )
            mode.addError( ' Burst range <%d-%d> bytes' % (
                           qosAclHwStatus.policeByteModeMinBurstBytes,
                           qosAclHwStatus.policeByteModeMaxBurstBytes ) )
            outOfRange = True
   return outOfRange

def configureNamedPolicer( mode, no, policerName, cir=None, cirUnit=None, bc=None,
                           bcUnit=None, shared=True ):
   if no:
      del cliQosAclConfig.namedPolicer[ policerName ]
   else:
      if policerOutOfRange( mode, cir, cirUnit, bc, bcUnit,
                            policerMode='committed' ):
         return
      pir = be = 0
      cirUnit = cirUnit if cirUnit else tacRateUnit.rateUnitbps
      bcUnit = bcUnit if bcUnit else tacBurstUnit.burstUnitBytes

      oldPolicer = None
      policer = None
      bumpVersion = False
      if policerName not in cliQosAclConfig.namedPolicer:
         policer = cliQosAclConfig.namedPolicer.newMember(
                   policerName, cir, bc, pir, be )
         bumpVersion = True
      else:
         oldPolicer = Tac.newInstance( "Qos::PolicerConfig", policerName,
                                       cir, bc, pir, be )
         copyNamedPolicer( cliQosAclConfig.namedPolicer[ policerName ],
                           oldPolicer )

      policer = cliQosAclConfig.namedPolicer[ policerName ]
      policer.named = True
      policer.cir = cir
      policer.bc = bc
      policer.cirUnit = cirUnit
      policer.bcUnit = bcUnit
      # We only support named shared policers using this command.
      policer.shared = True
      if bumpVersion:
         policer.uniqueId = Tac.Value( 'Qos::UniqueId' )
         policer.version += 1
      else:
         if not identicalNamedPolicer( policer, oldPolicer ):
            # We support only named 'shared' policers, and we can only
            # modify rates/burst sizes of this named shared policer.
            # Update the 'paramChangeVersion' for this policer to trigger updating
            # the policer directly in the hw.
            policer.uniqueId = Tac.Value( 'Qos::UniqueId' )
            policer.paramChangeVersion += 1
      # BUG1947473: Should we block waiting for policy maps using this policer
      # to be programmed?
      # In case of rate/burst size changes, oldPolicerConfig has the previous
      # policer configuration.

# Functions for policy map dynamic name rules
def getPMapNameRule( mode, mapType ):
   suggestedCompletions = cliQosAclConfig.pmapType[ mapType ].pmap.members()
   return sorted( suggestedCompletions )

def getPMapNameQos( mode ):
   return getPMapNameRule( mode, QosLib.qosMapType )

# Functions for class dynamic name rules
def getClassNameRule( mode, mapType ):
   suggestedCompletions = cliQosAclConfig.cmapType[ mapType ].cmap.members()
   if mapType == coppMapType:
      suggestedCompletions.remove( tacCMapNm.coppDrop )
      suggestedCompletions.append( tacCMapNm.coppDefault )
   else:
      suggestedCompletions.append( tacCMapNm.classDefault )
   return sorted( suggestedCompletions )

def getClassNameRuleQos( mode ):
   return getClassNameRule( mode, QosLib.qosMapType )

# Functions for class map  dynamic name rules
def getCMapNameRule( mode, mapType ):
   suggestedCompletions = cliQosAclConfig.cmapType[ mapType ].cmap.members()
   if mapType == coppMapType:
      # static class maps cannot be edited, remove them.
      suggestedCompletions = filter( 
         lambda x: classMapCpType( cliQosAclConfig, x ) != \
            tacClassMapCpType.cmapCpStatic, suggestedCompletions )
   return sorted( suggestedCompletions )

def getCMapNameQos( mode ):
   return getCMapNameRule( mode, QosLib.qosMapType )

# Functions for dynamic name rules   
def getStandardAccessListNames( ver ):
   names = []   
   config = aclConfig.config[ ver ].acl
   for name in config:
      if config[ name ].standard:
         names.append( name )
   return names

# Copy contents of one map entry to another
def copyClassMap( dstEntry, srcEntry ):
   def copyClassMapL2Params( srcMatch, dstMatch ):
      dstMatch.vlanValue = None
      dstMatch.cosValue = None
      dstMatch.innerVlanValue = None

      if srcMatch.vlanValue:
         dstMatch.vlanValue = ( srcMatch.vlanValue.vlan,
                                srcMatch.vlanValue.vlanMask )
         dstMatch.vlanValue.vlanColl.clear()
         dstMatch.vlanValue.maskValid = srcMatch.vlanValue.maskValid
         for key, value in srcMatch.vlanValue.vlanColl.iteritems():
            dstMatch.vlanValue.vlanColl[ key ] = value
      if srcMatch.cosValue:
         dstMatch.cosValue = ( '', )
         dstMatch.cosValue.cosColl.clear()
         for key, value in srcMatch.cosValue.cosColl.iteritems():
            dstMatch.cosValue.cosColl[ key ] = value
      if srcMatch.innerVlanValue:
         dstMatch.innerVlanValue = ( srcMatch.innerVlanValue.innerVlan,
                                     srcMatch.innerVlanValue.innerVlanMask )
         dstMatch.innerVlanValue.innerVlanColl.clear()
         dstMatch.innerVlanValue.maskValid = srcMatch.innerVlanValue.maskValid
         for key, value in srcMatch.innerVlanValue.innerVlanColl.iteritems():
            dstMatch.innerVlanValue.innerVlanColl[ key ] = value

   def copyClassMapDscpEcn( srcMatch, dstMatch ):
      dstMatch.dscpEcnValue = ( srcMatch.dscpEcnValue.dscp,
                                srcMatch.dscpEcnValue.ecn )
      dstMatch.dscpEcnValue.dscpColl.clear()
      dstMatch.dscpEcnValue.dscpNameValid = srcMatch.dscpEcnValue.dscpNameValid
      for key, value in srcMatch.dscpEcnValue.dscpColl.iteritems():
         dstMatch.dscpEcnValue.dscpColl[ key ] = value

   def copyClassMapMplsTrafficClass( srcMatch, dstMatch ):
      dstMatch.mplsTrafficClassVal = ( '', )
      dstMatch.mplsTrafficClassVal.mplsTrafficClassColl.clear()
      dstMatch.mplsTrafficClassVal.isValid = srcMatch.mplsTrafficClassVal.isValid
      for key, value in \
          srcMatch.mplsTrafficClassVal.mplsTrafficClassColl.iteritems():
         dstMatch.mplsTrafficClassVal.mplsTrafficClassColl[ key ] = value

   dstEntry.cpType = srcEntry.cpType
   dstEntry.dynamic = srcEntry.dynamic
   # Copy Match Rules
   for op in srcEntry.match:
      srcMatch = srcEntry.match[ op ]
      if op in dstEntry.match:
         dstMatch = dstEntry.match[ op ]
      else:
         #deleting the existing member before attaching new
         del dstEntry.match[ tacMatchOption.matchIpAccessGroup ]
         del dstEntry.match[ tacMatchOption.matchIpv6AccessGroup ]
         del dstEntry.match[ tacMatchOption.matchL2Params ]
         del dstEntry.match[ tacMatchOption.matchDscpEcn ]
         del dstEntry.match[ tacMatchOption.matchMplsTrafficClass ]
         del dstEntry.match[ tacMatchOption.matchMacAccessGroup ]
         dstMatch = dstEntry.match.newMember( op )
      if op in [ tacMatchOption.matchIpAccessGroup,
                 tacMatchOption.matchIpv6AccessGroup,
                 tacMatchOption.matchMacAccessGroup ]:
         dstMatch.strValue = srcMatch.strValue
      elif op in [ tacMatchOption.matchL2Params ]:
         dstMatch.strValue = srcMatch.strValue
         copyClassMapL2Params( srcMatch, dstMatch )
      elif op in [ tacMatchOption.matchDscpEcn ]:
         dstMatch.strValue = srcMatch.strValue
         copyClassMapDscpEcn( srcMatch, dstMatch )
      elif op in [ tacMatchOption.matchMplsTrafficClass ]:
         dstMatch.strValue = srcMatch.strValue
         copyClassMapMplsTrafficClass( srcMatch, dstMatch )

   for op in dstEntry.match:
      dstMatch = dstEntry.match[ op ]
      if op in [ tacMatchOption.matchIpAccessGroup,
                 tacMatchOption.matchIpv6AccessGroup, 
                 tacMatchOption.matchL2Params,
                 tacMatchOption.matchDscpEcn,
                 tacMatchOption.matchMplsTrafficClass,
                 tacMatchOption.matchMacAccessGroup ] \
            and op not in srcEntry.match:
         del dstEntry.match[ op ]

def identicalClassMap( src, dst ):

   def identicalClassMapL2Params( srcMatch, dstMatch ):
      # Returns true only if all the values match
      srcMatchCombo = getL2ParamsMatchCombo( srcMatch )
      dstMatchCombo = getL2ParamsMatchCombo( dstMatch )
      if set( srcMatchCombo ) != set( dstMatchCombo ):
         return False
      if srcMatch.vlanValue and dstMatch.vlanValue:
         if srcMatch.vlanValue.maskValid != dstMatch.vlanValue.maskValid:
            return False

         if srcMatch.vlanValue.maskValid:
            if ( srcMatch.vlanValue.vlan != dstMatch.vlanValue.vlan or
                 srcMatch.vlanValue.vlanMask != dstMatch.vlanValue.vlanMask ):
               return False
         else:
            if ( set( srcMatch.vlanValue.vlanColl.items() ) !=
                 set( dstMatch.vlanValue.vlanColl.items() ) ):
               return False

      if srcMatch.cosValue and dstMatch.cosValue:
         if ( set( srcMatch.cosValue.cosColl.items() ) !=
              set( dstMatch.cosValue.cosColl.items() ) ):
            return False

      if srcMatch.innerVlanValue and dstMatch.innerVlanValue:
         if srcMatch.innerVlanValue.maskValid != dstMatch.innerVlanValue.maskValid:
            return False

         if srcMatch.innerVlanValue.maskValid:
            if ( srcMatch.innerVlanValue.innerVlan !=
                 dstMatch.innerVlanValue.innerVlan or
                 srcMatch.innerVlanValue.innerVlanMask !=
                 dstMatch.innerVlanValue.innerVlanMask ):
               return False
         else:
            if ( set( srcMatch.innerVlanValue.innerVlanColl.items() ) !=
                 set( dstMatch.innerVlanValue.innerVlanColl.items() ) ):
               return False
      return True

   def identicalClassMapDscpEcn( srcMatch, dstMatch ):
      if srcMatch.dscpEcnValue.dscp != dstMatch.dscpEcnValue.dscp:
         return False
      if srcMatch.dscpEcnValue.dscpNameValid != \
         dstMatch.dscpEcnValue.dscpNameValid:
         return False
      if srcMatch.dscpEcnValue.ecn != dstMatch.dscpEcnValue.ecn:
         return False
      return set( srcMatch.dscpEcnValue.dscpColl.items() ) == \
         set( dstMatch.dscpEcnValue.dscpColl.items() )

   def identicalClassMapMplsTrafficClass( srcMatch, dstMatch ):
      if srcMatch.mplsTrafficClassVal.isValid != \
         dstMatch.mplsTrafficClassVal.isValid:
         return False
      return set( srcMatch.mplsTrafficClassVal.mplsTrafficClassColl.items() ) == \
         set( dstMatch.mplsTrafficClassVal.mplsTrafficClassColl.items() )

   # Basic checks to verify that cmaps are identical
   if src.cpType != dst.cpType:
      return False
   srcOp = set( src.match )
   dstOp = set( dst.match )

   if srcOp != dstOp:
      return False

   for key in srcOp:
      srcMatch = src.match[ key ]
      dstMatch = dst.match[ key ]
      if srcMatch.strValue != dstMatch.strValue:
         return False
      if ( key == 'matchL2Params' and
           not identicalClassMapL2Params( srcMatch, dstMatch ) ):
         return False
      if key == 'matchDscpEcn' and not \
           identicalClassMapDscpEcn( srcMatch, dstMatch ):
         return False
      if key == 'matchMplsTrafficClass' and not \
           identicalClassMapMplsTrafficClass( srcMatch, dstMatch ):
         return False
   return True

#-------------------------------------------------------------------------------
# class-map mode
#-------------------------------------------------------------------------------
#------------------------------------------------------------------------
# Editing context ( applicable for all modes )
#
# A context is created for each child mode. The context either copies
# an existing sequence entry or creates new entry. 
#
# The editedEntries_ hold all sequences that are either new or edited.
# If a "no" command was issued, the "match" or "set" is removed from
# the entry. At commit the old entry is completely replaced with the
# entry in context.
#
# Note that a "no class-map" to remove a sequence is outside of the
# class-map config mode and handled in deleteClassMapMode.
#------------------------------------------------------------------------
class ClassMapContext( object ):
   def __init__( self, mode, mapType, cmapName, matchType ):
      self.mode = mode 
      self.map_ = None      
      self.mapType_ = mapType
      self.cmapName_ = cmapName
      self.matchType_ = matchType
      self.editedEntries_ = [] 
      self.currentEntry_ = None
      self.previousEntry_ = None # for config rollback

      cmaps = cliQosAclConfig.cmapType[ mapType ].cmap
      if cmapName in cmaps:
         self.map_ = cmaps[ cmapName ]
         
         # Previous instance for config rollback 
         prevEntry = Tac.newInstance( 'Qos::ClassMapConfig', self.cmapName_,
                                      self.mapType_ )
         copyClassMap( prevEntry, self.map_ )
         self.previousEntry_ = prevEntry

   def copyEditEntry( self, entry ):
      # Create a new cmap of type 0; i.e ControlPlane Type
      newEntry = Tac.newInstance( 'Qos::ClassMapConfig', self.cmapName_,
                                  self.mapType_ )
      copyClassMap( newEntry, entry )
      self.editedEntries_.append( newEntry )
      self.currentEntry_ = newEntry
      return newEntry

   def newEditEntry( self ):
      # Add an edit by creating a new entry
      # Create a new cmap of type 0 i.e ControlPlane Type
      entry = Tac.newInstance( 'Qos::ClassMapConfig', self.cmapName_, 
                               coppMapType )
      self.editedEntries_.append( entry )
      self.currentEntry_ = entry
      if self.mapType_ == coppMapType:
         entry.cpType = 'cmapCpDynamic'
      else:
         entry.cpType = 'cmapCpNone'      
      return entry

   def cmapName( self ):
      return self.cmapName_

   def mapType( self ):
      return self.mapType_

   def matchType( self ):
      return self.matchType_
   
   def cmap( self ):
      return self.map_ # may be None

   def currentEntry( self ):
      return self.currentEntry_
   
   def commit( self ):

      # Commit current map. Create a new map if not exist
      cmType = cliQosAclConfig.cmapType[ self.mapType_ ]
      cmaps = cmType.cmap
      newCmap = False
      if self.map_ is None:
         if self.cmapName_ in cmaps:
            self.map_ = cmaps[ self.cmapName_ ]
         else:
            self.map_ = cmaps.newMember( self.cmapName_, cmType.type )
            newCmap = True
      entry = self.map_
      
      # We bump the version of cmap only if
      # we have actually changed it.
      bumpVersion = False
      for en in self.editedEntries_:
         bumpVersion = not identicalClassMap( entry, en )
         copyClassMap( entry, en )

      # Updating version field
      if bumpVersion:
         # Update all the service policy config
         # that will get affected because of this change
         currTime = getCurrentTimeForConfigUpdate()
         
         self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
         self.map_.version += 1

         rc, errMsg = spWaitForHwStatusForCmapChange( self.mode, self.cmapName_, 
                                                      self.mapType_, currTime )
         if not rc:
            if errMsg == CLI_TIMEOUT:
               self.mode.addWarning( QosLib.qosTimeoutWarning() )
            else:
               # rollback
               self.mode.addError( "Error: Cannot commit class-map " +
                                   "%s, %s (%s)" % \
                                    ( self.cmapName_, self.mapType_, errMsg ) )

               if newCmap:
                  t0( "Deleting new class-map %s" % self.cmapName_ )
                  del cmaps[ self.cmapName_ ]
               else:
                  t0( "Reverting class-map update for %s" % self.cmapName_ )
                  copyClassMap( entry, self.previousEntry_ )
                  self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
                  self.map_.version += 1

class ClassMapMode( ClassMapModeBase, BasicCli.ConfigModeBase ):
   name = "Class Map Configuration"
   modeParseTree = CliParser.ModeParseTree()
   showActiveCmdRegistered_ = True

   # Each mode object has a session object. We associate the classmap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      self.cmapContext = context
      self.cmapName_ = context.cmapName()
      self.mapType_ = context.mapType()
      self.matchType_ = context.matchType()
      param = ( self.mapType_, self.cmapName_ )
      ClassMapModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'ClassMapMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.subConfig = None
      self.cmapContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.cmapContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.cmapContext
      self.cmapContext = None
      context.commit()

   @staticmethod
   def matchOptionExists( entry, matchOption ):
      return matchOption in entry.match

   @staticmethod
   def isCollSame( rangeColl, values, ecnCheck=True ):
      ids = set()
      for key in rangeColl:
         ids.update( range( key.min, key.max + 1 ) )
      return ids == set( values ) and ecnCheck

   @staticmethod
   def setMatchVlanValue( entry, value, mask, matchOption, vlanRangeString ):
      vlanValue = value
      isMaskValid = True
      if vlanRangeString:
         vlanValue = Tac.Value( "Bridging::VlanId" ).invalid
         mask = 0
         isMaskValid = False
      entry.match[ matchOption ].vlanValue = ( vlanValue, mask )
      for key in entry.match[ matchOption ].vlanValue.vlanColl:
         del entry.match[ matchOption ].vlanValue.vlanColl[ key ]
      if vlanRangeString:
         valueString = MultiRangeRule.multiRangeToCanonicalString(
            vlanRangeString )
         valueString = valueString.split( ',' )
         for rangeStr in valueString:
            minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
            minVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 0 ] )
            maxVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 1 ] )
            key = Tac.Value( "Qos::VlanRange" )
            key.min = minVal
            key.max = maxVal
            entry.match[ matchOption ].vlanValue.vlanColl[ key ] = True
      entry.match[ matchOption ].vlanValue.maskValid = isMaskValid

   @staticmethod
   def setMatchInnerVlanValue( entry, value, mask, innerVlanRangeString ):
      matchOption = matchOptionToEnum( 'l2Params' )
      innerVlanValue = value
      isMaskValid = True
      if innerVlanRangeString:
         innerVlanValue = Tac.Value( "Bridging::VlanId" ).invalid
         mask = 0
         isMaskValid = False
      entry.match[ matchOption ].innerVlanValue = ( innerVlanValue, mask )
      entry.match[ matchOption ].innerVlanValue.innerVlanColl.clear()
      if innerVlanRangeString:
         valueString = MultiRangeRule.multiRangeToCanonicalString(
            innerVlanRangeString )
         valueString = valueString.split( ',' )
         for rangeStr in valueString:
            minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
            minVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 0 ] )
            maxVal = Tac.Value( "Bridging::VlanId", minMaxValue[ 1 ] )
            key = Tac.Value( "Qos::VlanRange" )
            key.min = minVal
            key.max = maxVal
            entry.match[ matchOption ].innerVlanValue.innerVlanColl[ key ] = True
      entry.match[ matchOption ].innerVlanValue.maskValid = isMaskValid

   @staticmethod
   def setMatchCosValue( entry, value, matchOption ):
      entry.match[ matchOption ].cosValue = ( '', )
      for key in entry.match[ matchOption ].cosValue.cosColl:
         del entry.match[ matchOption ].cosValue.cosColl[ key ]

      valueString = None
      rangeStr = str( value ) if isinstance( value, int ) else value
      valueString = MultiRangeRule.multiRangeToCanonicalString( rangeStr )
      cosString = valueString.split( ',' )
      for rangeStr in cosString:
         minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
         minVal = Tac.Value( "Qos::Cos", minMaxValue[ 0 ] )
         maxVal = Tac.Value( "Qos::Cos", minMaxValue[ 1 ] )
         key = Tac.Value( "Qos::CosRange" )
         key.min = minVal
         key.max = maxVal
         entry.match[ matchOption ].cosValue.cosColl[ key ] = True

   def setMatchDscpEcnValue( self, entry, value, ecnValue, matchRange, matchOption,
                                valueString, dscpNameValid ):
      dscpValue = value
      if not ecnValue:
         ecnValue = ecnDontCare
      else:
         ecnValue = AclCliLib.ecnValueFromCli( self, ecnValue )
      # dscp = 0 is a valid value, hence 'not dscpValue' will be wrong here
      if matchRange or dscpValue == None:
         # 'match ecn' case
         dscpValue = invalidDscp
      entry.match[ matchOption ].dscpEcnValue = ( dscpValue, ecnValue )
      for key in entry.match[ matchOption ].dscpEcnValue.dscpColl:
         del entry.match[ matchOption ].dscpEcnValue.dscpColl[ key ]
      entry.match[ matchOption ].dscpEcnValue.dscpNameValid = dscpNameValid
      if matchRange:
         # dscp range configured
         dscpString = valueString.split( ',' )
         for rangeStr in dscpString:
            minMaxValue = rangeToMinMaxValue( rangeStr )
            minVal = Tac.Value( "Qos::DscpVal", minMaxValue[ 0 ] )
            maxVal = Tac.Value( "Qos::DscpVal", minMaxValue[ 1 ] )
            key = Tac.Value( "Qos::DscpRange" )
            key.min = minVal
            key.max = maxVal
            entry.match[ matchOption ].dscpEcnValue.dscpColl[ key ] = True

   @staticmethod
   def setMatchMplsTrafficClassValue( entry, value, matchOption ):
      entry.match[ matchOption ].mplsTrafficClassVal = ( '', )
      for key in entry.match[ matchOption ].mplsTrafficClassVal.\
          mplsTrafficClassColl:
         del entry.match[ matchOption ].mplsTrafficClassVal.mplsTrafficClassColl[
            key ]

      rangeStr = str( value ) if isinstance( value, int ) else value
      mplsTrafficClassString = MultiRangeRule.multiRangeToCanonicalString( rangeStr )

      mplsTrafficClassString = mplsTrafficClassString.split( ',' )
      for rangeStr in mplsTrafficClassString:
         minMaxValue = QosLib.rangeToMinMaxValue( rangeStr )
         minVal = Tac.Value( 'Qos::MplsTrafficClassVal', minMaxValue[ 0 ] )
         maxVal = Tac.Value( 'Qos::MplsTrafficClassVal', minMaxValue[ 1 ] )
         key = Tac.Value( "Qos::MplsTrafficClassRange" )
         key.min = minVal
         key.max = maxVal
         entry.match[ matchOption ].mplsTrafficClassVal.mplsTrafficClassColl[
            key ] = True
         entry.match[ matchOption ].mplsTrafficClassVal.isValid = True

   def setL2ParamsMatchValue( self, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      matchCombo = args.get( 'matchCombo' )

      vlanValue = args.get( 'VLAN_VALUE' )
      vlanMask = args.get( 'VLAN_MASK' )
      vlanRangeString = args.get( 'VLAN_SET' )

      innerVlanValue = args.get( 'INNER_VLAN_VALUE' )
      innerVlanMask = args.get( 'INNER_VLAN_MASK' )
      innerVlanRangeString = args.get( 'INNER_VLAN_SET' )

      cosValue = args.get( 'COS_RANGE' )

      context = self.cmapContext
      entry = context.currentEntry()
      matchType = 'l2Params'
      matchOption = matchOptionToEnum( matchType )

      removeUnconditionally = False
      vlanPresent = 'vlan' in matchCombo
      cosPresent = 'cos' in matchCombo
      innerVlanPresent = 'vlan inner' in matchCombo

      if self.matchOptionExists( entry, matchOption ):
         entryMatch = entry.match[ matchOption ]
         # Check matchCombo of the existing ones
         existingMatchCombo = getL2ParamsMatchCombo( entryMatch )
      else:
         existingMatchCombo = list()

      if noOrDefault:
         # If there is no match l2Params configured and no command is executed
         if not existingMatchCombo:
            return
         # Removing unconditionally is only applicable if none of the values are
         # entered and the matchCombo matches with the existing matchCombo
         if ( vlanValue is None and cosValue is None and vlanRangeString is None and
              innerVlanValue is None and innerVlanRangeString is None ):
            if set( matchCombo ).issubset( set( existingMatchCombo ) ):
               removeUnconditionally = True


      if vlanMask == None:
         vlanMask = 0xFFF
      if innerVlanMask == None:
         innerVlanMask = 0xFFF

      if not noOrDefault:
         allow, pmapNames, _macAclClassCounts, _ipv6ClassCounts, vxlanIntfPresent = \
                     allowMatchOptionInClassMap( entry, matchType, vlanPresent,
                                                 cosPresent, innerVlanPresent )
         if not allow:
            errorMsg = 'The match type %s is not allowed in the class-map. ' % \
                ( matchType )
            errorMsg += \
                '\nThe class-map is present in the following policy-maps: %s ' % \
                ( pmapNames )
            if qosAclHwStatus.vlanMatchInClassMapSupported and vlanPresent:
               errorMsg += 'which are applied on VLAN interfaces'
            elif qosAclHwStatus.vxlanPolicyQosSupported and vxlanIntfPresent and \
                 not vlanPresent and cosPresent and innerVlanPresent:
               errorMsg += 'which are applied on the VTI'
            else:
               errorMsg += ( 'which have a policer configured in the class-default. '
                             'When class-default has policer, the policy-map is not '
                             'allowed to have both ipv4 class-maps and ipv6 '
                             'class-maps' )
            self.addError( errorMsg )
            return

         if not self.matchOptionExists( entry, matchOption ):
            entry.match.newMember( matchOption )
         else:
            # Option exists, now check if the matchCombos are the same
            if set( existingMatchCombo ) != set( matchCombo ):
               del entry.match[ matchOption ]
               entry.match.newMember( matchOption )

         if vlanPresent:
            self.setMatchVlanValue( entry, vlanValue, vlanMask, matchOption,
                                    vlanRangeString )
         if innerVlanPresent:
            self.setMatchInnerVlanValue( entry, innerVlanValue, innerVlanMask,
                                         innerVlanRangeString )
         if cosPresent:
            self.setMatchCosValue( entry, cosValue, matchOption )
         entry.match[ matchOption ].strValue = matchType


         # Delete other matchOption if any
         matchOptionsOtherToEnum = matchOptionOtherToEnum( matchOption )
         for matchOption_ in matchOptionsOtherToEnum:
            if self.matchOptionExists( entry, matchOption_ ):
               del entry.match[ matchOption_ ]
      else:
         if removeUnconditionally:
            # To do this because we want 'no match vlan' to remove
            # 'match vlan 5 0xfff cos 3'
            # When we do 'no match vlan'
            # CLI passes matchVlan as the option type
            # So explicitly checking if CLI passed matchVlan and only if matchVlanCos
            # is configured we delete the entry
            del entry.match[ matchOption ]
         elif self.matchOptionExists( entry, matchOption ):
            # If incoming commands have more parameters than the existing one
            # reject it
            if len( matchCombo ) != len( existingMatchCombo ):
               return

            if vlanValue != None or vlanRangeString != None:
               if not entryMatch.vlanValue:
                  return
               if vlanRangeString is None:
                  if entryMatch.vlanValue.vlan != vlanValue or \
                     entryMatch.vlanValue.vlanMask != vlanMask:
                     return
               else:
                  rangeColl = entryMatch.vlanValue.vlanColl
                  if not self.isCollSame( rangeColl, vlanRangeString ):
                     return

            if innerVlanValue != None or innerVlanRangeString != None:
               if not entryMatch.innerVlanValue:
                  return
               if innerVlanRangeString is None:
                  if ( entryMatch.innerVlanValue.innerVlan != innerVlanValue or
                       entryMatch.innerVlanValue.innerVlanMask != innerVlanMask ):
                     return
               else:
                  rangeColl = entryMatch.innerVlanValue.innerVlanColl
                  if not self.isCollSame( rangeColl, innerVlanRangeString ):
                     return

            if cosValue != None :
               if entryMatch.cosValue is None:
                  return
               if isinstance( cosValue, int ):
                  cosValue = [ cosValue ]
               if not self.isCollSame( entryMatch.cosValue.cosColl, cosValue ):
                  return

            del entry.match[ matchOption ]

   def setMatchValue( self, no, matchType, value, mask=None,
                      ecnValue=None ):

      context = self.cmapContext
      entry = context.currentEntry()

      removeUnconditionally = False
      if no and value == None and not mask and not ecnValue:
         removeUnconditionally = True

      matchRange = False
      matchOption = matchOptionToEnum( matchType )
      if matchOption == tacMatchOption.matchDscpEcn and \
          value and isinstance( value, list ):
         matchRange = True

      dscpNameValid = False
      if matchOption == tacMatchOption.matchDscpEcn and value and \
         isinstance( value, str ):
         value = AclCliLib.dscpValueFromCli( self, value )
         value = value[ 0 ]
         dscpNameValid = True

      valueString = None
      if matchRange:
         valueString = MultiRangeRule.multiRangeToCanonicalString( value )

      if not no:
         allow, pmapNames, macAclClassCounts, ipv6ClassCounts, vxlanIntfPresent = \
                     allowMatchOptionInClassMap( entry, matchType )
         if not allow:
            errorMsg = 'The match type %s is not allowed in the class-map. ' % \
                ( matchType )
            errorMsg += \
                '\nThe class-map is present in the following policy-maps: %s ' % \
                ( pmapNames )
            if qosAclHwStatus.matchMacInClassMapSupported:
               if( ( matchType == 'mac' or matchType == 'ipv6' ) and not \
                  qosAclHwStatus.matchMacInClassMapV6Supported ):
                  if matchType == 'mac' and ipv6ClassCounts or \
                        matchType == 'ipv6' and macAclClassCounts:
                     errorMsg = 'Invalid class map match types are configured in '
                     errorMsg += 'the following policy-maps %s: '  % ( pmapNames )
                     errorMsg += 'The policy-map is not allowed to have both ipv6 '
                     errorMsg += 'class-maps and mac class-maps' 
                  elif matchType == 'mac':
                     errorMsg = 'Invalid action configured in the following '
                     errorMsg += 'policy-maps %s: ' % ( pmapNames )
                     errorMsg += 'Reconfigure the policy-map to use police action'
               elif matchType == 'mac':  
                  errorMsg = 'Invalid action configured in the following policy-maps'
                  errorMsg += ' %s: ' % ( pmapNames )
                  errorMsg += 'Reconfigure the policy-map to use police action'
            elif qosAclHwStatus.vxlanPolicyQosSupported and vxlanIntfPresent and \
                 matchType != 'l2Params':
               errorMsg += 'which are applied on the VTI'
            else:
               errorMsg += 'which have a policer configured in the class-default. '
               errorMsg += ' When class-default has policer, the policy-map is not ' 
               errorMsg += 'allowed to have both ipv4 class-maps and ipv6 class-maps'
            self.addError( errorMsg ) 
            return 
         if not self.matchOptionExists( entry, matchOption ):
            entry.match.newMember( matchOption )
         # access-group ip/ipv6/mac
         if matchOption != tacMatchOption.matchDscpEcn and \
            matchOption != tacMatchOption.matchMplsTrafficClass:
            entry.match[ matchOption ].strValue = value
         elif matchOption == tacMatchOption.matchDscpEcn:
            self.setMatchDscpEcnValue( entry, value, ecnValue, matchRange,
                                       matchOption, valueString, dscpNameValid )
            entry.match[ matchOption ].strValue = 'dscpEcn'  #'dscpEcn'
         elif matchOption == tacMatchOption.matchMplsTrafficClass:
            self.setMatchMplsTrafficClassValue( entry, value, matchOption )
            entry.match[ matchOption ].strValue = matchType
         # Delete other matchOption if any
         matchOptionsOtherToEnum = matchOptionOtherToEnum( matchOption )
         for matchOption_ in matchOptionsOtherToEnum:
            if self.matchOptionExists( entry, matchOption_ ):
               del entry.match[ matchOption_ ]
      else:
         if removeUnconditionally:
            del entry.match[ matchOption ]
         elif self.matchOptionExists( entry, matchOption ):
            entryMatch = entry.match[ matchOption ]
            if entryMatch.option == matchOption :
               if ( matchOption != tacMatchOption.matchDscpEcn and \
                    value == entryMatch.strValue ) or ( matchOption \
                    == tacMatchOption.matchMacAccessGroup and \
                    ( value == entryMatch.strValue or removeUnconditionally ) ):
                  del entry.match[ matchOption ]
               elif not matchRange and matchOption == tacMatchOption.matchDscpEcn:
                  # match dscp when no range is configured
                  onlyEcnCfg = entryMatch.dscpEcnValue.dscp == invalidDscp and \
                               len( entryMatch.dscpEcnValue.dscpColl ) == 0 and \
                               entryMatch.dscpEcnValue.ecn != ecnDontCare
                  if removeUnconditionally:
                     if matchType == 'ecn':
                        if onlyEcnCfg:
                           del entry.match[ matchOption ]
                     else:
                        if not onlyEcnCfg:
                           del entry.match[ matchOption ]
                     return
                  if not ecnValue:
                     ecnValue = ecnDontCare
                  else:
                     ecnValue = AclCliLib.ecnValueFromCli( self, ecnValue )
                  if value == None:
                     value = invalidDscp
                  if entryMatch.dscpEcnValue.dscpNameValid == dscpNameValid and \
                     value == entryMatch.dscpEcnValue.dscp and \
                     ecnValue == entryMatch.dscpEcnValue.ecn:
                     del entry.match[ matchOption ]
               elif matchOption == tacMatchOption.matchMplsTrafficClass:
                  if value == None:
                     del entry.match[ matchOption ]
                     return
                  if isinstance( value, int ):
                     value = [ value ]
                  if self.isCollSame( entryMatch.mplsTrafficClassVal.
                                      mplsTrafficClassColl, value ):
                     del entry.match[ matchOption ]
               elif matchRange:
                  # match vlan/dscp when range is configured
                  if removeUnconditionally:
                     del entry.match[ matchOption ]
                     return
                  ecnCheck = True
                  if matchOption == tacMatchOption.matchDscpEcn:
                     rangeColl = entryMatch.dscpEcnValue.dscpColl
                     if not ecnValue:
                        ecnValue = ecnDontCare
                     else:
                        ecnValue = AclCliLib.ecnValueFromCli( self, ecnValue )
                     if ecnValue != entryMatch.dscpEcnValue.ecn:
                        ecnCheck = False
                  if self.isCollSame( rangeColl, value, ecnCheck ):
                     del entry.match[ matchOption ]

   # print one or all class maps
   @staticmethod
   def showClassMap( mode, args ):
      mapType = args.get( 'control-plane', 'qos' )
      cmapName = args.get( 'CMAP' )
      classMapsAllModel = ClassMapAllModel()
      classMapsContainer = CMapModelContainer( qosAclConfig, qosHwStatus,
                                               qosAclHwStatus, 
                                               qosSliceHwStatus,
                                               mapTypeToEnum( mapType ),
                                               classMapsAllModel )
      if cmapName == None:
         classMapsContainer.populateAll()
      else:
         classMapsContainer.populateClassMap( cmapName, classPrio=1 )
      return classMapsAllModel

class ClassMapModeQos( ClassMapMode ):
   name = "Class Map Qos Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      ClassMapMode.__init__( self, parent, session, context )

#-----------------------------------------------------------------

#-----------------------------------------------------------------
# The show command in 'config-cmap' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------

def _showCMapList( clMapType, clMapName, clMap, output=None ):
   if clMap is None:
      return
   if output is None:
      output = sys.stdout

   assert clMap.matchCondition == 'matchConditionAny'
   matchCondition = 'match-any'
   mapTypeString = mapTypeFromEnum( clMapType )
   if clMapType == coppMapType:
      mapTypeString = 'copp'
   output.write( "class-map type %s %s %s\n" %
         ( mapTypeString, matchCondition, clMap.name ) )
   for matchRule in clMap.match:
      ipStr = matchOptionFromEnum( matchRule )
      clMapMatch = clMap.match[ matchRule ]
      if matchRule == tacMatchOption.matchL2Params:
         outputString = "   match"
         vlanValue = clMapMatch.vlanValue
         cosValue = clMapMatch.cosValue
         innerVlanValue = clMapMatch.innerVlanValue

         if vlanValue is not None:
            if vlanValue.maskValid:
               outputString += " vlan %d %#5.3x" % \
                               ( clMapMatch.vlanValue.vlan,
                                 clMapMatch.vlanValue.vlanMask )
            else:
               ids = set()
               for key in vlanValue.vlanColl:
                  ids.update( range( key.min, key.max + 1 ) )
               vlanStr = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
               outputString += " vlan %s" % vlanStr

         if innerVlanValue is not None:
            if innerVlanValue.maskValid:
               outputString += " vlan inner %d %#5.3x" % (
                  clMapMatch.innerVlanValue.innerVlan,
                  clMapMatch.innerVlanValue.innerVlanMask )
            else:
               ids = set()
               for key in innerVlanValue.innerVlanColl:
                  ids.update( range( key.min, key.max + 1 ) )
               innerVlanStr = MultiRangeRule.multiRangeToCanonicalString(
                  list( ids ) )
               outputString += " vlan inner %s" % innerVlanStr

         if cosValue is not None:
            ids = set()
            for key in cosValue.cosColl:
               ids.update( range( key.min, key.max + 1 ) )
            cosStr = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
            outputString += " cos %s" % cosStr

         outputString += "\n"
         output.write( outputString )

      elif matchRule in [ tacMatchOption.matchDscpEcn ]:
         dscpValue = clMapMatch.dscpEcnValue
         ecnStr = ""
         if dscpValue.ecn != ecnDontCare:
            # if ecn is configured with/without dscp
            ecnName = AclCliLib.ecnNameFromValue( dscpValue.ecn )
            ecnStr = " %s %s" % ( ipStr[ 1 ], ecnName )
         if len( dscpValue.dscpColl ) == 0:
            if not dscpValue.dscpNameValid:
               # 'match dscp <dscpId>'
               if dscpValue.dscp != Tac.Type( "Qos::DscpVal" ).invalid:
                  output.write( "   match %s %d%s\n" % \
                                ( ipStr[ 0 ], dscpValue.dscp, ecnStr ) )
               else:
                  # 'match ecn <ecnName>'
                  output.write( "   match%s\n" % ( ecnStr ) )
            else:
               # 'match dscp <dscpName>'
               val = dscpValue.dscp
               name = AclCliLib.dscpNameFromValue( val )
               output.write( "   match %s %s%s\n" % \
                                ( ipStr[ 0 ], name, ecnStr ) )
         else:
            # 'match dscp <comma-separated range>'
            ids = set()
            for key in dscpValue.dscpColl:
               ids.update( range( key.min, key.max + 1 ) )
            dscpStr = MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
            output.write( "   match %s %s%s\n" % ( ipStr[ 0 ], dscpStr, ecnStr ) )
      elif matchRule in [ tacMatchOption.matchMplsTrafficClass ]:
         ipStr = 'traffic-class'
         mplsTrafficClassVal = clMapMatch.mplsTrafficClassVal
         ids = set()
         if len( mplsTrafficClassVal.mplsTrafficClassColl ) == 1:
            for key in mplsTrafficClassVal.mplsTrafficClassColl:
               if key.min == key.max:
                  output.write( "   match mpls %s %d\n" % \
                                ( ipStr, key.min ) )
               else:
                  ids.update( range( key.min, key.max + 1 ) )
                  mplsTrafficClassStr = \
                     MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
                  output.write( "   match mpls %s %s\n" % \
                                ( ipStr, mplsTrafficClassStr ) )
         else:
            for key in mplsTrafficClassVal.mplsTrafficClassColl:
               ids.update( range( key.min, key.max + 1 ) )
            mplsTrafficClassStr = \
                  MultiRangeRule.multiRangeToCanonicalString( list( ids ) )
            output.write( "   match mpls %s %s\n" % \
                             ( ipStr, mplsTrafficClassStr ) )
      else:
         output.write( "   match %s access-group %s\n" % \
                ( ipStr, clMapMatch.strValue ) )
   output.write( '\n' )

def showCMapDiff( mode, output=None ):
   if output is None:
      output = sys.stdout
   # generate diff between active and pending
   import difflib, cStringIO
   activeOutput = cStringIO.StringIO( )
   _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.map_,
                  output=activeOutput )
   pendingOutput = cStringIO.StringIO( )
   _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.currentEntry_,
                  output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue( ).splitlines( ),
                                pendingOutput.getvalue( ).splitlines( ),
                                lineterm='' )
   output.write( '\n'.join( list( diff ) ) )
   output.write( '\n' )

def showCMapPending( mode ):
   import cStringIO
   changeOutput = cStringIO.StringIO( )
   showCMapDiff( mode, changeOutput )
   if changeOutput.getvalue() != '\n':
      #show pending only if there is a change
      _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.currentEntry_ )

def showCMapActive( mode ):
   _showCMapList( mode.mapType_, mode.cmapName, mode.cmapContext.map_ )

#--------------------------------------------------------------------------------
# class-map type qos match-any <name>
#--------------------------------------------------------------------------------
# Goto 'class-map' config mode. Create a class map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.
def gotoQosClassMapMode( mode, args ):
   cmapName = args[ 'CMAP' ]
   mapType = 'qos'
   t0( 'gotoQosClassMapMode %s' % ( cmapName ) )
   emapType = mapTypeToEnum( mapType )
   context = ClassMapContext( mode, emapType, cmapName, 'match-any' )
   
   if cmapName in cliQosAclConfig.cmapType[ emapType ].cmap:
      entry = cliQosAclConfig.cmapType[ emapType ].cmap[ cmapName ]
      context.copyEditEntry( entry )
      if entry.dynamic:
         mode.addWarning( DYNAMIC_CMAP_EDIT_WARN )
   else:
      context.newEditEntry()

   mode.cmapContext = context
   childMode = mode.childMode( ClassMapModeQos, context=context )
   mode.session_.gotoChildMode( childMode )   

def deleteClassMap( mode, cmapName, mapType='qos', matchType='match-any' ):
   emapType = mapTypeToEnum( mapType )
   if not cmapName in cliQosAclConfig.cmapType[ emapType ].cmap:
      return
   deleteComments( mode, mapType, cmapName=cmapName )
   # delete the entire map
   del cliQosAclConfig.cmapType[ emapType ].cmap[ cmapName ]
   t0( 'Deleted map %s' % cmapName )
   if not mode.session.inConfigSession():
      currTime = getCurrentTimeForConfigUpdate()
      description = "class-map to be deleted"
      cliBlockingToApplyConfigChange( mode, currTime, description, 
                                      policyMap=True )

#------------------------------------------------------------------------
# Register top-level CLI commands ( class-map )
#------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] match ip access-group GROUP
#--------------------------------------------------------------------------------
class MatchIpAccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'match ip access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'ip': CliCommand.guardedKeyword( 'ip',
         helpdesc='Specify Ip Access-groups', guard=guardIpAcls ),
      'access-group': matcherAccessGroup,
      'GROUP': AclCli.standardIpAclNameMatcher, 
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( noOrDefault, 'ip', args[ 'GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchIpAccessGroupCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match ipv6 access-group GROUP
#--------------------------------------------------------------------------------
class MatchIpv6AccessGroupCmd( CliCommand.CliCommandClass ):
   syntax = 'match ipv6 access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'ipv6': CliCommand.guardedKeyword( 'ipv6',
         helpdesc='Specify Ipv6 Access-groups', guard=guardIp6Acls ),
      'access-group': matcherAccessGroup,
      'GROUP': AclCli.standardIp6AclNameMatcher, 
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( noOrDefault, 'ipv6', args[ 'GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchIpv6AccessGroupCmd )

# [ no ] { vlan <id/range> | cos <id/range> | vlan inner <id/range> }
vlanKeywordMatcher = CliMatcher.KeywordMatcher( "vlan",
                                                helpdesc="Specify VLAN match" )

cosKeywordMatcher = CliMatcher.KeywordMatcher( "cos", helpdesc="Specify COS match" )

vlanInnerKeywordMatcher = CliMatcher.KeywordMatcher( "vlan",
                                                helpdesc="Specify VLAN match" )

innerVlanKeywordMatcher = CliMatcher.KeywordMatcher( "inner",
                                                helpdesc="Specify inner VLAN match" )

cosSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 0, 7 ),
   noSingletons=False,
   helpdesc='CoS value(s) or range(s) of CoS values',
   value=lambda mode, grList: grList.values(),
   priority=CliParser.PRIO_HIGH
)

vlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 4094 ),
   noSingletons=True,
   helpdesc='VLAN value(s) or range(s) of VLAN values',
   value=lambda mode, grList: grList.values(),
   priority=CliParser.PRIO_HIGH
)

innerVlanSetMatcher = MultiRangeRule.MultiRangeMatcher(
   rangeFn=lambda: ( 1, 4094 ),
   noSingletons=True,
   helpdesc="Inner VLAN value(s) or range(s) of inner VLAN values",
   value=lambda mode, grList: grList.values(),
   priority=CliParser.PRIO_HIGH
)

class CosMatchExpression( CliCommand.CliExpression ):
   expression = "cos COS_RANGE"
   data = {
      "cos": CliCommand.Node( matcher=cosKeywordMatcher,
                              maxMatches=1, guard=guardCosMatch ),
      "COS_RANGE": CliCommand.Node( matcher=cosSetMatcher, maxMatches=1 )
   }

class VlanMatchExpression( CliCommand.CliExpression ):
   expression = "vlan ( VLAN_SET | ( VLAN_VALUE [ VLAN_MASK ] ) )"
   data = {
      "vlan"  : CliCommand.Node( matcher=vlanKeywordMatcher, maxMatches=1,
                                 guard=guardVlanMatch ),
      "VLAN_SET": CliCommand.Node( matcher=vlanSetMatcher,
                                   maxMatches=1 ),
      "VLAN_VALUE": CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 4094, helpdesc="VLAN value" ),
         maxMatches=1 ),
      "VLAN_MASK": CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0x0, 0xfff, helpdesc="VLAN mask" ),
         maxMatches=1 )
   }

class InnerVlanMatchExpression( CliCommand.CliExpression ):
   expression = ( "vlan_ inner ( INNER_VLAN_SET | ( INNER_VLAN_VALUE "
                  "[ INNER_VLAN_MASK ] ) )" )
   data = {
      "vlan_" : CliCommand.Node( matcher=vlanInnerKeywordMatcher, maxMatches=1,
                                 guard=guardInnerVlanMatch ),
      "inner" : CliCommand.Node( matcher=innerVlanKeywordMatcher, maxMatches=1,
                                 guard=guardInnerVlanMatch ),
      "INNER_VLAN_SET" : CliCommand.Node( matcher=innerVlanSetMatcher,
                                          maxMatches=1 ),
      "INNER_VLAN_VALUE" : CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 1, 4094, helpdesc="Inner VLAN value" ),
         maxMatches=1 ),
      "INNER_VLAN_MASK" : CliCommand.Node(
         matcher=CliMatcher.IntegerMatcher( 0x0, 0xfff, helpdesc="Inner VLAN mask" ),
         maxMatches=1 )
   }

class L2ParamsMatchCmd( CliCommand.CliCommandClass ):
   syntax = "match { VLAN | INNER_VLAN | COS }"
   noOrDefaultSyntax = syntax
   data = {
      "match" : "Match the access rule specified",
      "VLAN"  : VlanMatchExpression,
      "INNER_VLAN" : InnerVlanMatchExpression,
      "COS"   : CosMatchExpression
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      matchCombo = list()
      if 'vlan' in args:
         matchCombo.append( 'vlan' )
         args.pop( 'vlan' )
      if 'inner' in args:
         matchCombo.append( 'vlan inner' )
         args.pop( 'vlan_' )
         args.pop( 'inner' )
      if 'cos' in args:
         matchCombo.append( 'cos' )
         args.pop( 'cos' )
      args[ 'matchCombo' ] = matchCombo

   handler = ClassMapModeQos.setL2ParamsMatchValue
   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( L2ParamsMatchCmd )

# This command is to remove matches on L2Params when the input
# command is 'no match vlan' to remove 'match vlan 5 cos 2'
# The command works when match options in the command exists in
# class-map match.
class NoL2ParamsMatchCmd( CliCommand.CliCommandClass ):
   syntax = None
   noOrDefaultSyntax = "match ( vlan | ( vlan_ inner ) | cos )"

   data = {
      "match" : "Match the access rule specified",
      "vlan" : CliCommand.Node( matcher=vlanKeywordMatcher,
                                guard=guardVlanMatch ),
      "vlan_" : CliCommand.Node( matcher=vlanKeywordMatcher,
                                 guard=guardInnerVlanMatch ),
      "inner" : CliCommand.Node( matcher=innerVlanKeywordMatcher,
                                 guard=guardInnerVlanMatch ),
      "cos" : CliCommand.Node( matcher=cosKeywordMatcher,
                               guard=guardCosMatch )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      matchCombo = list()
      if 'vlan' in args:
         matchCombo.append( 'vlan' )
         args.pop( 'vlan' )
      if 'inner' in args:
         matchCombo.append( 'vlan inner' )
         args.pop( 'vlan_' )
         args.pop( 'inner' )
      if 'cos' in args:
         matchCombo.append( 'cos' )
         args.pop( 'cos' )
      args[ 'matchCombo' ] = matchCombo

   handler = None
   noOrDefaultHandler = ClassMapModeQos.setL2ParamsMatchValue

ClassMapModeQos.addCommandClass( NoL2ParamsMatchCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match dscp ( DSCPVALUE | DSCP_NAME | DSCPRANGE )
#       [ ecn ( ect-ce | non-ect | ce | ect ) ]
#--------------------------------------------------------------------------------
class MatchDscpCmd( CliCommand.CliCommandClass ):
   syntax = ( 'match dscp ( DSCPVALUE | DSCP_NAME | DSCPRANGE ) '
              '[ ecn ( ect-ce | non-ect | ce | ect ) ]' )
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'dscp': nodeDscpMatch,
      'DSCPVALUE': CliMatcher.IntegerMatcher( 0, AclLib.MAX_DSCP,
         helpdesc='DSCP Value' ),
      'DSCP_NAME': CliMatcher.EnumMatcher(
         { k: v[ 1 ] for k, v in AclCliLib.dscpAclNames.iteritems() } ),
      'DSCPRANGE': MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 0, AclLib.MAX_DSCP ), noSingletons=True,
         helpdesc='DSCP value(s) or range(s) of DSCP values',
         priority=CliParser.PRIO_HIGH ),
      'ecn': nodeDscpEcn,
      'ect': matcherEct,
      'ect-ce': matcherEctCe,
      'ce': matcherCe,
      'non-ect': matcherNonEct,
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      dscpValue = args.get( 'DSCP_NAME' ) or args.get( 'DSCPVALUE' )
      if dscpValue is None:
         dscpValue = args.get( 'DSCPRANGE' ).values()
      ecnValue = args.get( 'ect-ce' ) or args.get( 'non-ect' ) or \
            args.get( 'ce' ) or args.get( 'ect' )
      mode.setMatchValue( noOrDefault, 'dscp', dscpValue, None, ecnValue )

   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchDscpCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match ecn ( ect-ce | non-ect | ce | ect )
#--------------------------------------------------------------------------------
class MatchEcnCmd( CliCommand.CliCommandClass ):
   syntax = 'match ecn ( ect-ce | non-ect | ce | ect )'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'ecn': nodeDscpEcn,
      'ect': matcherEct,
      'ect-ce': matcherEctCe,
      'ce': matcherCe,
      'non-ect': matcherNonEct,
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      ecnValue = args.get( 'ect-ce' ) or args.get( 'non-ect' ) or \
            args.get( 'ce' ) or args.get( 'ect' )
      mode.setMatchValue( noOrDefault, 'ecn', None, None, ecnValue )

   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchEcnCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match mpls traffic-class ( { TC } | TC_RANGE )
#--------------------------------------------------------------------------------
class MatchMplsTrafficClassCmd( CliCommand.CliCommandClass ):
   syntax = 'match mpls traffic-class ( { TC } | TC_RANGE )'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'mpls': nodeMpls,
      'traffic-class': nodeTrafficClass,
      'TC': CliMatcher.IntegerMatcher( 0, 7,
         helpdesc='Match packets by Mpls Traffic Class value' ),
      'TC_RANGE': MultiRangeRule.MultiRangeMatcher(
         rangeFn=lambda: ( 0, 7 ), noSingletons=True,
         helpdesc=( 'Mpls traffic-class value(s) or range(s) '
            'of Mpls traffic-class values' ), priority=CliParser.PRIO_HIGH ),
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( noOrDefault, 'mpls-traffic-class', 
            args.get( 'TC' ) or args.get( 'TC_RANGE' ).values() )

   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchMplsTrafficClassCmd )

#--------------------------------------------------------------------------------
# [ no | default ] match mac access-group GROUP
#--------------------------------------------------------------------------------
class MatchMacAccessGroupAccessCmd( CliCommand.CliCommandClass ):
   syntax = 'match mac access-group GROUP'
   noOrDefaultSyntax = syntax
   data = {
      'match': matcherMatch,
      'mac': nodeMac,
      'access-group': matcherAccessGroup,
      'GROUP': AclCli.userMacAclNameMatcher, 
   }

   @staticmethod
   def handler( mode, args ):
      noOrDefault = CliCommand.isNoOrDefaultCmd( args )
      mode.setMatchValue( noOrDefault, 'mac', args[ 'GROUP' ] )

   noOrDefaultHandler = handler

ClassMapModeQos.addCommandClass( MatchMacAccessGroupAccessCmd )

#--------------------------------------------------------------------------------
# ( no | default ) match ( dscp | ecn | ( mpls traffic-class ) |
#       ( mac access-group ) )
#--------------------------------------------------------------------------------
class NoOrDefaultClassMapModeMatchCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = ( 'match ( dscp | ecn | ( mpls traffic-class ) | '
                         '( mac access-group ) )' )
   data = {
      'match': matcherMatch,
      'dscp': nodeDscpMatch,
      'ecn': nodeDscpEcn,
      'mpls': nodeMpls,
      'traffic-class': nodeTrafficClass,
      'mac': nodeMac,
      'access-group': matcherAccessGroup,
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      matchType = args.get( 'dscp' ) or args.get( 'ecn' ) or \
                  args.get( 'mpls-traffic-class', 'mac' )
      mode.setMatchValue( True, matchType, None )

ClassMapModeQos.addCommandClass( NoOrDefaultClassMapModeMatchCmd )

#-------------------------------------------------------------------------------
# policy-map class mode
#-------------------------------------------------------------------------------
def getDefaultPdpPmapCfg():
   defaultPdpPmapCfg = None
   if defaultPdpPmapCfgReadOnly.pmapType.get( pdpMapType, None ):
      defaultPdpPmapCfg = defaultPdpPmapCfgReadOnly.pmapType[
         pdpMapType ].pmap.get( tacPMapNm.defaultPdpPmapName, None )
   return defaultPdpPmapCfg

class PolicyMapClassContext( object ):
   def __init__( self, session, pmap, cmapName, insertBeforeClass, mapType ):
      self.session_ = session
      self.map_ = pmap
      self.pmapName_ = pmap.name
      self.cmapName_ = cmapName
      self.mapType_ = mapType
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.insertBeforeClass = insertBeforeClass

   def copyEditEntry( self, cmapName ):
      newClassAction = Tac.newInstance( 'Qos::ClassAction', cmapName )

      oldClassAction = None
      if QosLib.isDefaultClass( self.mapType_, cmapName ) == True:
         oldClassAction = self.map_.classActionDefault
      else:
         oldClassAction = self.map_.classAction[ cmapName ]
      assert oldClassAction != None

      copyClassAction( oldClassAction, newClassAction )
      self.currentEntry_ = newClassAction

   def newEditEntry( self, cmapName ):
      newClassAction = Tac.newInstance( 'Qos::ClassAction', cmapName )
      self.currentEntry_ = newClassAction

   def pmapName( self ):
      return self.pmapName_

   def policyClassName( self ):
      return self.cmapName_
   
   def mapType( self ):
      return self.mapType_

   def pmap( self ):
      return self.map_ # may be None

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      cmapName = self.cmapName_

      def commitStaticORDefaultClass( cmapName ):
         clAction = None
         if QosLib.isDefaultClass( self.mapType_, cmapName ):
            clAction = self.map_.classActionDefault

         if clAction is not None:
            copyClassAction( self.currentEntry_, clAction )
            return True

         return False

      # For copp static / default class no re-ordering
      if not commitStaticORDefaultClass( cmapName ):
         self._commit()

   def _commit( self ):
      cmapName = self.cmapName_

      # Find the location of the class
      parentClassPrio = self.map_.classPrio
      numberOfClasses = len( parentClassPrio )
      if cmapName in self.map_.classAction:
         parentClassAction = self.map_.classAction[ cmapName ]
         for classPrio in parentClassPrio.values():
            if cmapName == classPrio.cmapName:
               currentPrio = classPrio.index
               break
         # Unless the class is being reordered, we want it to be in the same place
         expectedPrio = currentPrio
      else:
         parentClassAction  = self.map_.classAction.newMember( cmapName )
         i = numberOfClasses + 1
         parentClassPrio.newMember( i )
         parentClassPrio[ i ].cmapName = cmapName
         currentPrio = i
         expectedPrio = i

      if self.insertBeforeClass and self.insertBeforeClass in self.map_.classAction:
         for classPrio in parentClassPrio.values():
            if self.insertBeforeClass == classPrio.cmapName:
               expectedPrio = classPrio.index
               break
      if expectedPrio > currentPrio:
         # We dont want the place occupied by insert before class,
         # we want a place just above it.
         expectedPrio -= 1

      # This loops shuffles the priority of classes.
      # Shifts all classes in classPrio  down or up till required class
      # has expected Priority.
      while currentPrio != expectedPrio:
         if currentPrio > expectedPrio:
            parentClassPrio[ currentPrio ].cmapName = \
                parentClassPrio[ currentPrio - 1 ].cmapName
            currentPrio -= 1

         if expectedPrio > currentPrio:
            parentClassPrio[ currentPrio ].cmapName = \
                parentClassPrio[ currentPrio + 1 ].cmapName
            currentPrio += 1
        
      parentClassPrio[ expectedPrio ].cmapName = cmapName
      copyClassAction( self.currentEntry_, parentClassAction )

def copyClassAction( src, dst ):
   # This function is invoked
   #  
   #  Entering policy-map mode: copy from sysdb -> policy map editing copy
   #  Entering policy-class mode: copy from policy map editing copy -> 
   #    policy-class mode editing copy
   #
   #  Committing from policy-class mode: copy from policy-class mode editing copy ->
   #    policy map editing copy
   #  Committing from policy-map mode: copy from policy map editing copy -> sysdb
   #
   #  Entering a new user-defined pdp policy map mode: copy from default-pdp-policy
   #  to policy map editing copy
   #  Reverting PDP class action to the action present for the class in
   #  default-pdp-policy

   # If we have removed the action from the source, then it means
   # 'no' command was issued from the action, therefore remove from 
   # destination
   for action in dst.policyAction:
      if action not in src.policyAction:
         del dst.policyAction[ action ]

   for action in src.policyAction:
      dstPolicyAction = dst.policyAction.newMember( action )
      srcPolicyAction = src.policyAction[ action ]
      if action in [ tacActionType.actionSetShape, 
                     tacActionType.actionSetBandwidth ]:
         dstPolicyAction.rate = ()
         dstPolicyAction.rate.val = srcPolicyAction.rate.val
         dstPolicyAction.rate.rateUnit = srcPolicyAction.rate.rateUnit
      elif action in [ tacActionType.actionSetDscp, 
                       tacActionType.actionSetCos, 
                       tacActionType.actionSetDrop, 
                       tacActionType.actionSetDropPrecedence, 
                       tacActionType.actionSetTc ]:
         dstPolicyAction.value = srcPolicyAction.value

   dst.dscpConfiguredAsName = src.dscpConfiguredAsName
   dst.dscpConfiguredAsNameYellow = src.dscpConfiguredAsNameYellow

   # If we have removed policer from source, then it means 'no' command was 
   # issued for the policer, remove from destination
   if src.policer:
      dst.policer = ( src.policer.name, src.policer.cir, src.policer.bc,
                      src.policer.pir, src.policer.be )
      dst.policer.named = src.policer.named
      dst.policer.cir = src.policer.cir
      dst.policer.cirUnit = src.policer.cirUnit
      dst.policer.bc = src.policer.bc
      dst.policer.bcUnit = src.policer.bcUnit
      dst.policer.pir = src.policer.pir
      dst.policer.pirUnit = src.policer.pirUnit
      dst.policer.be = src.policer.be
      dst.policer.beUnit = src.policer.beUnit
      dst.policer.cmdVersion = src.policer.cmdVersion

      # Deleting the yellow-actions that were already present in the policer
      # useful when a trtcm-mode policer is changed to a committed one
      for action in dst.policer.yellowActions:
         if action not in src.policer.yellowActions:
            del dst.policer.yellowActions[ action ]

      # Though not now but when hardware allows for separate actions on
      # policed traffic like 'set tc', 'set dscp' etc, then the existing action for
      # yellow-packets will have to be modified. Since drop-precedence doesn't have a
      # value, no value is updated.
      for action in src.policer.yellowActions:
         dstPolicerYellowAction = dst.policer.yellowActions.newMember( action )
         srcPolicerYellowAction = src.policer.yellowActions[ action ]
         if action in [ tacActionType.actionSetDscp,
                        tacActionType.actionSetCos,
                        tacActionType.actionSetDropPrecedence, 
                        tacActionType.actionSetTc ]:
            dstPolicerYellowAction.value = srcPolicerYellowAction.value

      action = tacActionType.actionSetDrop
      dst.policer.redActions.newMember( action )
      dst.policer.redActions[ action ].value = 5
   else:
      dst.policer = None
   # Copy 'count' action - used only for PDP classes.
   dst.count = src.count

def irToKbps( ir, irUnit ):
   if irUnit == tacRateUnit.rateUnitbps:
      irKbps = ir / 1000
   elif irUnit == tacRateUnit.rateUnitKbps:
      irKbps = ir
   elif irUnit == tacRateUnit.rateUnitMbps:
      irKbps = ir * 1000
   return irKbps

def pdpPolicerYellowActionNeeded( defPdpPolicy ):
   # Return True if default-pdp-policy contains classActions with policers
   # having yellow actions (e.g Strata)
   if not defPdpPolicy:
      return False
   for _, classAction in defPdpPolicy.classAction.items():
      if classAction.policer and classAction.policer.yellowActions:
         return True
   return False

class PolicyMapClassMode( PolicyMapClassModeBase, BasicCli.ConfigModeBase ):
   name = "Policy Map Class Configuration"
   modeParseTree = CliParser.ModeParseTree()

   # Each mode object has a session object. We associate the policymap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      self.pmapClassContext = context
      self.pmapName = context.pmapName()
      self.cmapName = context.policyClassName()
      self.mapType_ = context.mapType()
      param = ( self.mapType_, self.pmapName, self.cmapName )
      PolicyMapClassModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'PolicyMapClassMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.subConfig = None
      self.pmapClassContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.pmapClassContext is None:
         t0( 'commitContext has no context' )
         return
      context = self.pmapClassContext
      self.pmapClassContext = None

      if self.mapType_ == pdpMapType and \
         self.pmapName == tacPMapNm.defaultPdpPmapName:
         # Do not commit any configuration of default-pdp-policy classes.
         # This should happen only when replaying a config saved from the output
         # of 'show running-config all', because configuring default-pdp-policy
         # is guarded.
         return
      context.commit()

   # print one or all policy maps
   @staticmethod
   def showPolicyMapClass( mode, args ):
      pmapName = args.get( 'PMAP', 'copp-system-policy' ) 
      cmapName = args[ 'CMAP' ]
      mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
                args.get( 'pdp', 'qos' )
      mapType = 'control-plane' if mapType == 'copp' else mapType
      emapType = mapTypeToEnum( mapType )
      policyMapAllModel =  PolicyMapAllModel()
      classMapWrapper = ClassMapWrapper()
      policyMapsContainer = PMapModelContainer( qosConfig, qosAclConfig, qosStatus,
                                   qosHwStatus,
                                   qosSliceHwStatus, qosAclHwStatus, 
                                   qosAclSliceHwStatus, emapType, None,
                                   hwEpochStatus, None,
                                   policyMapAllModel )
      if pmapName is None:
         pmapName = mode.pmapName
      if cmapName is None:
         cmapName = mode.cmapName
      policyMapsContainer.populatePolicyMapClassMap( pmapName, cmapName )
      if pmapName in policyMapAllModel.policyMaps:
         pMap = policyMapAllModel.policyMaps[ pmapName ]
         if cmapName in pMap.classMaps:
            classMapWrapper.classMap = pMap.classMaps[ cmapName ]
            return classMapWrapper
         else:
            return classMapWrapper
      else:
         return classMapWrapper

   def configurePolicer( self, mode, no, policerMode='committed', name=None,
                         cir=None, cirUnit=None, bc=None, bcUnit=None,
                         policerYellowAction=None, policerYellowValue=None, pir=None,
                         pirUnit=None, be=None, beUnit=None, policerRedAction=None,
                         cmdVersion=1, default=False ):
      context = self.pmapClassContext
      classAction = context.currentEntry()
      if not no and not default:
         if QosLib.isDefaultClass( self.mapType_, classAction.name ):
            if not qosAclHwStatus.policerInDefaultClassV4V6Supported:
               ipv4ClassCount, ipv6ClassCount = ipClassCounts( context.map_ )
               if ipv4ClassCount > 0 and ipv6ClassCount > 0:
                  self.addError( ( 'Policer in default-class is not supported when '
                                   'both ipv4 class-maps and ipv6 class-maps are '
                                   'present in policy map. To configure policer in '
                                   'default-class, remove either all ipv4 class-maps'
                                   ' or all ipv6 class-maps from the policy-map' ) )
                  return
         if tacActionType.actionSetDrop in classAction.policyAction:
            self.addError( 'Drop action not supported when policer enabled' )
            return
         if cirUnit == None:
            if bcUnit == tacBurstUnit.burstUnitPackets:
               cirUnit = tacRateUnit.rateUnitPps
            else:
               cirUnit = tacRateUnit.rateUnitbps
         if bcUnit == None:
            if cirUnit == tacRateUnit.rateUnitPps:
               bcUnit = tacBurstUnit.burstUnitPackets
            else:
               bcUnit = tacBurstUnit.burstUnitBytes
         if pirUnit == None:
            if cirUnit != tacRateUnit.rateUnitPps:
               pirUnit = tacRateUnit.rateUnitbps
            else:
               pirUnit = tacRateUnit.rateUnitPps
         if beUnit == None:
            if pirUnit != tacRateUnit.rateUnitPps:
               beUnit = tacBurstUnit.burstUnitBytes
            else:
               beUnit = tacBurstUnit.burstUnitPackets

         if policerMode == 'committed':
            pir = be = 0
         if( not name and not mode.session_.startupConfig() and
             not( self.mapType_ == pdpMapType and
                  self.pmapName == tacPMapNm.defaultPdpPmapName ) and
             policerOutOfRange( self, cir, cirUnit, bc, bcUnit,
                                pir, pirUnit, be, beUnit, policerMode ) ):
            # When in startup-config, or for default-pdp-policy,
            # ignore checks for policer rates and burst-sizes.
            return
         if isLagInServicePolicy( qosInputConfig, qosInputProfileConfig, 
                                  self.pmapName ):
            if not( qosAclHwStatus.policerOnLagSupported or not \
                  ( qosAclHwStatus.hwInitialized or \
                  mode.session_.guardsEnabled() ) ):
               mode.addError( CliError[ 'lagPolicingNotSupported' ] )
               return
            # Per port CoPP not supported on Port-Channel
            if qosHwStatus.intfCoppSupported and self.mapType == coppMapType:
               mode.addError( CliError[ 'lagPolicingNotSupported' ] )
               return

         policerName = name if name else classAction.name
         if name:
            cir = bc = 0
         classAction.policer = ( policerName, cir, bc, pir, be )
         classAction.policer.named = True if name else False
         classAction.policer.cir = cir
         classAction.policer.cirUnit = cirUnit
         classAction.policer.bc = bc
         classAction.policer.bcUnit = bcUnit
         classAction.policer.pir = pir
         classAction.policer.pirUnit = pirUnit
         classAction.policer.be = be
         classAction.policer.beUnit = beUnit
         classAction.policer.cmdVersion = cmdVersion
         if policerYellowAction:
            if policerYellowAction == "drop-precedence":
               action = tacActionType.actionSetDropPrecedence
               classAction.policer.yellowActions.newMember( action )
            if policerYellowAction == "dscp":
               action = tacActionType.actionSetDscp
               classAction.policer.yellowActions.newMember( action )
               if isinstance( policerYellowValue, str ):
                  classAction.dscpConfiguredAsNameYellow = True
                  policerYellowValue, isValid = AclCliLib.dscpValueFromCli( self,
                                                               policerYellowValue )
                  assert isValid
               classAction.policer.yellowActions[ action ].value = \
                                                                   policerYellowValue
               
         if self.mapType_ == pdpMapType and \
            pdpPolicerYellowActionNeeded( getDefaultPdpPmapCfg() ):
            # If policer in PDP policy class should have yellow action (drop),
            # do not forget to add it when configuring the police action.
            # Currently, adding action drop in yellowActions is enough for
            # the required platforms (e.g Strata).
            action = tacActionType.actionSetDrop
            if action not in classAction.policer.yellowActions:
               classAction.policer.yellowActions.newMember( action )

         # set default red actions to Drop
         action = tacActionType.actionSetDrop
         if action not in classAction.policer.redActions:
            classAction.policer.redActions.newMember( action )
            classAction.policer.redActions[ action ].value = 1
         classAction.count = False
      else:
         if self.mapType_ == pdpMapType:
            # 'no/default police' in a PDP policy class will revert to the action for
            # the class specified in the default-pdp-policy.
            src = None
            _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
            if QosLib.isDefaultClass( self.mapType_, classAction.name ):
               src = _defaultPdpPmapCfg.classActionDefault
            else:
               src = _defaultPdpPmapCfg.classAction[ classAction.name ]
            copyClassAction( src, classAction )
         else:
            # no police
            if classAction.policer:
               classAction.policer = None

class PolicyMapClassModeQos( PolicyMapClassMode ):
   name = "Policy Map Qos Class Configuration"
   modeParseTree = CliParser.ModeParseTree()
   outputActionUnsupportedString = "Action set %s is not supported in " + \
                                   "output service policies"

   def isPmapApplied( self ):
      pmapName = self.pmapClassContext.pmapName_
      spConfigs = [ qosInputConfig.servicePolicyConfig,
                    qosInputProfileConfig.servicePolicyConfig ]
      for spConfig in spConfigs:
         for key, _ in spConfig.iteritems():
            if key.pmapName == pmapName:
               return True
         return False


   def isPmapAppliedInOutDirection( self ):
      pmapName = self.pmapClassContext.pmapName_
      spConfigs = [ qosInputConfig.servicePolicyConfig,
                    qosInputProfileConfig.servicePolicyConfig ]
      for spConfig in spConfigs:
         for key, _ in spConfig.iteritems():
            if key.pmapName == pmapName and key.direction == tacDirection.output:
               return True
         return False

   def configureSetU32( self, no, value, action ):
      context = self.pmapClassContext
      classAction = context.currentEntry()
      # If a class-map inside a policy-map has 'police' action and the match
      # type is 'mac' return True, False otherwise.
      cmaps = qosAclConfig.cmapType[ QosLib.qosMapType ].cmap
      if qosAclHwStatus.matchMacInClassMapSupported and \
         context.cmapName_ in cmaps and \
         'matchMacAccessGroup' in cmaps[ context.cmapName_ ].match:
         errorMsg = ('Policy-map: %s configured has invalid action: %s, '\
         'reconfigure the policy-map to use police action' %
                     ( context.pmapName_, action ) )
         self.addError( errorMsg )
         return False

      if not no:
         if action not in classAction.policyAction:
            classAction.policyAction.newMember( action )
         classAction.policyAction[ action ].value = value
      else:
         if action in classAction.policyAction:
            del classAction.policyAction[ action ]

   def configureSetDscp( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'DSCP_NAME' ) or args.get( 'DSCP' )
      if ( not qosHwStatus.dscpActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString % "dscp" )
         return True

      classAction = self.pmapClassContext.currentEntry()
      if no:
         classAction.dscpConfiguredAsName = False
         return self.configureSetU32( no, value, tacActionType.actionSetDscp )

      if isinstance( value, str ):
         # DSCP value provided as a name-string.
         classAction.dscpConfiguredAsName = True
         value, isValid = AclCliLib.dscpValueFromCli( self, value )
         assert isValid
         self.configureSetU32( no, value, tacActionType.actionSetDscp )
      else:
         # DSCP value provided as a number.
         classAction.dscpConfiguredAsName = False
         self.configureSetU32( no, value, tacActionType.actionSetDscp )

   def configureSetCos( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'COS' )
      if ( not qosHwStatus.cosActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString % "cos" )
         return True
      self.configureSetU32( no, value, tacActionType.actionSetCos )

   def configureSetTc( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'TRAFFIC_CLASS' )
      if ( not qosHwStatus.tcActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString % "traffic-class" )
         return True
      self.configureSetU32( no, value, tacActionType.actionSetTc )

   def configureSetDrop( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      context = self.pmapClassContext
      classAction = context.currentEntry()
      if classAction.policer is not None:
         self.addError( 'Drop action not supported when policer enabled' )
         return True
      self.configureSetU32( no, 1, tacActionType.actionSetDrop )

   def configureSetDropPrecedence( self, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      value = args.get( 'DROP_PRECEDENCE' )
      if ( not qosHwStatus.dpActionInOutSpSupported ) and \
         self.isPmapAppliedInOutDirection():
         self.addError( self.outputActionUnsupportedString % "drop-precedence" )
         return True

      yellowAction = QosLib.validDpConfigValues()[ 0 ]
      if not qosHwStatus.allowNonEctWithDropPrecedence:
         if qosInputConfig.ecnAllowNonEct and value == yellowAction and \
            self.isPmapApplied() :
            self.addError( "Policy-map action drop-precedence and allow"
                           " non-ect cannot be configured together." )
            return True
      self.configureSetU32( no, value, tacActionType.actionSetDropPrecedence )

#-------------------------------------------------------------------------------
# policy-map mode
#-------------------------------------------------------------------------------
class PolicyMapContext( object ):
   def __init__( self, mode, pmapName, pmapType, shared=False ):
      self.mode = mode
      self.map_ = None
      self.pmapName_ = pmapName
      self.mapType_ = pmapType
      self.shared = shared
      self.editedEntries_ = {} 
      self.currentEntry_ = None
      self.previousEntry_ = None # used for config rollback

      if pmapName in cliQosAclConfig.pmapType[ self.mapType_ ].pmap:
         self.map_ = cliQosAclConfig.pmapType[ self.mapType_ ].pmap[ pmapName ]

         # this instance will be used when config needs to be rolled back
         prevPolicyMap = Tac.newInstance( 'Qos::PolicyMapConfig', self.pmapName_, 
                                           self.mapType_ )
         copyPolicyMap( self.map_, prevPolicyMap, self.mapType_, self.mode )
         self.previousEntry_ = prevPolicyMap

   def copyEditEntry( self, cmapName ):
      newPolicyMap = Tac.newInstance( 'Qos::PolicyMapConfig', self.pmapName_, 
                                      self.mapType_ )
      copyPolicyMap( self.map_, newPolicyMap, self.mapType_, self.mode )
      self.currentEntry_ = newPolicyMap

   def newEditEntry( self, cmapName ):
      newPolicyMap = Tac.newInstance( 'Qos::PolicyMapConfig', self.pmapName_,
                                      self.mapType_ )
      newPolicyMap.shared = self.shared
      newPolicyMap.classDefault = ( cmapName, self.mapType_ )
      # Revisit! alway create default class-map for ipv4 type vs ipv6
      match = newPolicyMap.classDefault.match.newMember( 'matchIpAccessGroup' )
      match.strValue = 'default'
      newPolicyMap.classActionDefault = ( cmapName, )
      _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
      if self.mapType_ == pdpMapType and _defaultPdpPmapCfg:
         # Case when mapType is pdpMapType and platforms have populated
         # defaultPdpPmapCfg. If defaultPdpPmapCfg has not been populated yet,
         # the new user-defined policy would be empty( no classes or actions ).

         # Copy classAction
         for cmap in _defaultPdpPmapCfg.classAction:
            src = _defaultPdpPmapCfg.classAction[ cmap ]
            newPolicyMap.classAction.newMember( cmap )
            dst = newPolicyMap.classAction[ cmap ]
            copyClassAction( src, dst )
            if dst.policer:
               # Set cmdVersion of PDP policers to 2.
               dst.policer.cmdVersion = 2
         # Copy classPrio
         for index in _defaultPdpPmapCfg.classPrio:
            newPolicyMap.classPrio.newMember( index )
            newPolicyMap.classPrio[ index ].cmapName = \
               _defaultPdpPmapCfg.classPrio[ index ].cmapName
         src = _defaultPdpPmapCfg.classActionDefault
         dst = newPolicyMap.classActionDefault
         if src:
            # Copy class-default action from default-pdp-policy.
            copyClassAction( src, dst )
            if dst.policer:
               # Currently, all platforms have class-default action as 'count' for
               # PDP policy-maps in default-pdp-policy.
               # If any platform publishes class-default action as a policer action,
               # set cmdVersion to 2 as done for other built-in classes.
               dst.policer.cmdVersion = 2

      self.currentEntry_ = newPolicyMap

   def pmapName( self ):
      return self.pmapName_

   def pmap( self ):
      return self.map_ # may be None

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      # Commit current map. Create a new map if not exist
      # self.map_ is none when user creates pmap for the first time.
      # otherwise, if pmap already exists, self.map_ is assigned
      # appropriate config.
      pmapTypePtr = cliQosAclConfig.pmapType[ self.mapType_ ]
      if self.map_ == None:
         self.map_ = pmapTypePtr.pmap.newMember(
            self.pmapName_, pmapTypePtr.type )

      # we need to bump up the version if we change the pmap or
      # we failed to program it properly or not present in hwstatus
      bumpVersion = True
      prgmdPMapHwStatus = programmedPMapHwStatus( qosHwStatus, qosAclHwStatus,
                                                  qosSliceHwStatus,
                                                  qosAclSliceHwStatus,
                                                  self.mapType_, 
                                                  self.pmapName_ )
      if prgmdPMapHwStatus == tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
         if identicalPolicyMap( self.currentEntry_, self.map_ ):
            bumpVersion = False
      if bumpVersion:
         copyPolicyMap( self.currentEntry_, self.map_, self.mapType_, self.mode )
         currTime = getCurrentTimeForConfigUpdate()
        
         self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
         self.map_.version += 1

         rc, errMsg = spWaitForHwStatusForPmapChange( self.mode, self.pmapName_,
                                                      self.mapType_, currTime )
         if not rc:
            if errMsg == CLI_TIMEOUT:
               self.mode.addWarning( QosLib.qosTimeoutWarning() )
            else:
               # rollback
               self.mode.addError( "Error: Cannot commit policy-map " +
                                   "%s, %s (%s)" % \
                                      ( self.pmapName_, self.mapType_, errMsg ) )

               if self.previousEntry_:
                  t0( "Reverting policy-map update for %s " % self.pmapName_  )
                  copyPolicyMap( self.previousEntry_ , self.map_, self.mapType_,
                                 self.mode )
                  self.map_.uniqueId = Tac.Value( 'Qos::UniqueId' )
                  self.map_.version += 1
               else:
                  t0( "Deleting new policy-map: %s" % self.pmapName_ )
                  del pmapTypePtr.pmap[ self.pmapName_ ]

# PolicyMapCopyContext behaves like the PolicyMapContext except for the
# newEditEntry routine. In case of the copy context, the new edit entry
# should not be empty, instead, it should contain the configuration
# of the source policy it is to be copied from
class PolicyMapCopyContext( PolicyMapContext ):
   def __init__( self, mode, pmapName, srcPmap ):
      assert srcPmap
      super( PolicyMapCopyContext, self ).__init__( mode, pmapName,
            srcPmap.type, srcPmap.shared )
      self.srcPmap_ = srcPmap

   def newEditEntry( self, cmapName ):
      t0( 'PolicyMapCopyContext newEditEntry called for %s' % self.pmapName_ )
      # We will first call the parent method to create a new policy-map
      super( PolicyMapCopyContext, self ).newEditEntry( cmapName )
      # Once the policy-map is created, we copy the srcPmap contents into the
      # empty policy-map
      copyPolicyMap( self.srcPmap_, self.currentEntry_, self.mapType_, self.mode )

def copyPolicyMap( src, dst, mapType, mode ):
   # This function is invoked
   #
   # Entering policy-map mode: copy from sysdb -> policy map editing copy
   # Committing from policy-map mode: copy from policy map editing copy -> sysdb

   # If we have removed the class action from the source, then it means
   # 'no' command was issued from the class action, therefore remove from 
   # destination

   dst.dynamic = src.dynamic
   dst.shared = src.shared

   for cmap in dst.classAction:
      if cmap not in src.classAction:
         deleteComments( mode, mapTypeFromEnum( dst.type ), pmapName=dst.name,
               cmapName=cmap )
         del dst.classAction[ cmap ]

   allClasses = src.classAction.keys()
   if mapType == coppMapType:
      allClasses += [ tacCMapNm.coppDefault ]
   else:
      allClasses += [ tacCMapNm.classDefault ]

   # Make a editing copy for the classes
   for cmap in allClasses:
      if QosLib.isDefaultClass( mapType, cmap ) == True:
         if dst.classDefault == None:
            dst.classDefault = ( cmap, mapType )
            # Do not set match criteria for per port copp
            # we wont be creating any acl in platform
            if not ( mapType == coppMapType and src.name != tacPMapNm.coppName ):
               #Revisit! default is ipv4 vs ipv6
               match = dst.classDefault.match.newMember( 'matchIpAccessGroup' )
               match.strValue = 'default'
         dst.classDefault.cpType = src.classDefault.cpType
         dst.classDefault.cpStaticType = src.classDefault.cpStaticType

         if dst.classActionDefault == None:
            dst.classActionDefault = ( cmap, )
         dstAction = dst.classActionDefault
         srcAction = src.classActionDefault
      else:
         srcAction = src.classAction[ cmap ]
         dstAction = dst.classAction.newMember( cmap )
      copyClassAction( srcAction, dstAction )

   # This is to avoid duplicate valued( by cmapName ) priority in src.classPrio.
   # Refer to Escalation97805 where a duplicated valued cmapPrio
   # resulted in Ale crash.
   cmapPrios = {}  
   cmapNames = set()  
   for cmapPrio, cmap in src.classPrio.items():  
      if cmap.cmapName not in cmapNames:  
         cmapPrios[ cmapPrio ] = cmap  
         cmapNames.add( cmap.cmapName )  
   for cmapPrio in src.classPrio:
      # Delete duplicate key
      if cmapPrio not in cmapPrios:
         del src.classPrio[ cmapPrio ]  

   srcNumClasses = len( src.classPrio )
   dstNumClasses = len( dst.classPrio )

   # Shrink or expand classPrio to make room for new classes.
   while srcNumClasses != dstNumClasses:
      if srcNumClasses > dstNumClasses:
         dst.classPrio.newMember( srcNumClasses )
         dst.classPrio[ srcNumClasses ].cmapName = \
             dst.classPrio[ srcNumClasses ].cmapName
         srcNumClasses -= 1
      else:
         del dst.classPrio[ dstNumClasses ]
         dstNumClasses -= 1
   # Copy classPrio
   for index, classPrio in src.classPrio.iteritems():
      dst.classPrio[ index ].cmapName = classPrio.cmapName

   # Shrink or expand coppStaticClassPrio to make room for new classes.
   srcStaticNumClasses = len( src.coppStaticClassPrio )
   dstStaticNumClasses = len( dst.coppStaticClassPrio )
   while srcStaticNumClasses != dstStaticNumClasses:
      if srcStaticNumClasses > dstStaticNumClasses:
         dst.coppStaticClassPrio.newMember( srcStaticNumClasses )
         dst.coppStaticClassPrio[ srcStaticNumClasses ].cmapName = \
             dst.coppStaticClassPrio[ srcStaticNumClasses ].cmapName
         srcStaticNumClasses -= 1
      else:
         del dst.coppStaticClassPrio[ dstStaticNumClasses ]
         dstStaticNumClasses -= 1
 
   # Copy coppStaticClassPrio
   for index, classPrio in src.coppStaticClassPrio.iteritems():
      if index not in dst.coppStaticClassPrio:
         dst.coppStaticClassPrio.newMember( index )
      dst.coppStaticClassPrio[ index ].cmapName = classPrio.cmapName

_policyMapClassContextType = {}
#-------------------------------------------------------------------------------
# policy-map mode
#-------------------------------------------------------------------------------
class PolicyMapMode( PolicyMapModeBase, BasicCli.ConfigModeBase ):
   name = "Policy Map Configuration"
   modeParseTree = CliParser.ModeParseTree()
   # We already have a previous implementation for 'show active'.
   showActiveCmdRegistered_ = True

   # Each mode object has a session object. We associate the policymap
   # context with the mode object.
   def __init__( self, parent, session, context ):
      self.pmapContext = context
      self.pmapClassContext = None
      self.pmapName = context.pmapName()
      self.mapType_ = context.mapType_
      self.shared = context.shared
      param = ( self.mapType_, self.pmapName, self.shared )
      PolicyMapModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'PolicyMapMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.subConfig = None
      self.pmapContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.pmapContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.pmapContext
      self.pmapContext = None

      if self.mapType_ == pdpMapType and \
         self.pmapName == tacPMapNm.defaultPdpPmapName:
         # Do not commit any configuration of default-pdp-policy.
         # This should happen only when replaying a config saved from the output
         # of 'show running-config all', because configuring default-pdp-policy
         # is guarded.
         return
      context.commit()

   def getChildMode( self, mapType, context ):
      assert 0, 'Derived class should implement it'

   def gotoPolicyMapClassMode( self, pmapName, cmapName, insertBeforeClass ):
      def maxNumClassSupported():
         if self.mapType_ == coppMapType:
            return qosHwStatus.coppNumCMapSupported
         else:
            return qosAclHwStatus.policyQosNumCMapSupported

      def checkPMapClassCopp( cmapName, insertBeforeClass ):
         # Skip this check at startup
         if not self.session_.startupConfig():
            cpType = classMapCpType( cliQosAclConfig, cmapName )
            if cpType == tacClassMapCpType.cmapCpStatic:
               if not isCoppStaticClassSupported( cliQosAclConfig, qosSliceHwStatus, 
                                                  cmapName ):
                  self.addError( CliError[ 'cMapNotSupported' ] % cmapName ) 
                  return False
               if cmapName == tacCMapNm.coppDrop:
                  self.addError( CliError[ 'cannotConfigureDropClass' ] % cmapName )
                  return False
            if insertBeforeClass:
               cpTypeInsertBefore = classMapCpType( cliQosAclConfig,
                                                    insertBeforeClass )
               if cpType == tacClassMapCpType.cmapCpStatic or \
                      cpTypeInsertBefore == tacClassMapCpType.cmapCpStatic:
                  self.addError( CliError[ 'cannotReorder' ] )
                  return False
         return True
         
      context = None
      t0( 'gotoPolicyMapClassMode %s, %s' % ( pmapName, cmapName ) )
      
      # Checks for Copp
      if self.mapType_ == coppMapType:
         if checkPMapClassCopp( cmapName, insertBeforeClass ) == False:
            return

      # Create the context
      if context == None:
         policyMapClassContextType = \
               _policyMapClassContextType.get( self.mapType_,
                                               PolicyMapClassContext )
         context = policyMapClassContextType( self.session_,
                                              self.pmapContext.currentEntry(), 
                                              cmapName, insertBeforeClass, 
                                              self.mapType_ )
      currentEntry = self.pmapContext.currentEntry()

      def pmapClassPresent( cmapName ):
         if cmapName in currentEntry.classAction:
            return True
         elif QosLib.isDefaultClass( self.mapType_, cmapName ) == True:
            return True
         return False

      def numOfClasses( thisCmapName ):
         curCmapNames = set()
         if qosHwStatus.intfCoppSupported and self.mapType_ == coppMapType:
            # Find out all different supported static classes and
            # configured dynamic classes that does not match this pmap
            pmapCol = cliQosAclConfig.pmapType[ self.mapType_ ].pmap
            for pmapName, pmap in pmapCol.iteritems():
               if pmapName == self.pmapName:
                  continue
               for cmapName in pmap.classAction:
                  cpType = classMapCpType( cliQosAclConfig, cmapName )
                  if cpType == tacClassMapCpType.cmapCpStatic and \
                     not isCoppStaticClassSupported( cliQosAclConfig,
                                                     qosSliceHwStatus, cmapName ):
                     # Not all platforms supports every static classes
                     continue
                  curCmapNames.add( cmapName )

         for cmapName in currentEntry.classAction:
            if self.mapType_ == coppMapType:
               cpType = classMapCpType( cliQosAclConfig, cmapName )
               if cpType == tacClassMapCpType.cmapCpStatic and \
                  not isCoppStaticClassSupported( cliQosAclConfig, qosSliceHwStatus, 
                                                  cmapName ):
                  # Not all platforms supports every static classes
                  continue
            curCmapNames.add( cmapName )
         # Remove thisCmapName from curCmapNames as the caller expect
         # the num excluding thisCmapName
         curCmapNames.discard( thisCmapName )
         # +1 to account for default class
         return len( curCmapNames ) + 1

      def allowedClass( cmapName ):
         if self.mapType_ == pdpMapType:
            _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
            if _defaultPdpPmapCfg is None:
               return False
            if cmapName not in _defaultPdpPmapCfg.classAction:
               return False
         return True

      def checkNewPMapClass( cmapName ):
         # At startup, we skip this check.
         if not self.session_.startupConfig():
            # numClasses is  static classes + dynamic classes + default class
            numClasses = numOfClasses( cmapName )
            if self.mapType_ == coppMapType:
               cmapCpType = classMapCpType( cliQosAclConfig, cmapName )
               if tacClassMapCpType.cmapCpStatic != cmapCpType and \
                     not qosAclHwStatus.coppDynamicClassSupported:
                  self.addError( CliError[ 'dynamicClassError' ] )
                  return False

            if numClasses >= maxNumClassSupported():
               self.addError( CliError[ 'maxClassError' ] % maxNumClassSupported() )
               return False

            # At startup, platform may not be up. So, the defaultPdpPmapCfg
            # might not be populated. So we need to skip this check at startup time.
            if not allowedClass( cmapName ):
               self.addError( CliError[ 'cannotAddBuiltInCmap' ] )
               return False

         if self.mapType_ == qosMapType and not allowClassInPolicyMap( self, 
                                                                       currentEntry,
                                                                       cmapName ):
            return False

      if pmapClassPresent( cmapName ) == True:
         context.copyEditEntry( cmapName )
      else:
         if checkNewPMapClass( cmapName ) == False:
            return
         context.newEditEntry( cmapName )

      self.pmapClassContext = context
      childMode = self.getChildMode( self.mapType_, context )
      self.session_.gotoChildMode( childMode )

   def noClassPrio( self, cmapName ):
      if cmapName not in self.pmapContext.currentEntry_.classAction:
         # cmap not a part of pmap. Return quietly.
         return
      # we are in PolicyMapMode. Update currentEntry of its Context.
      t0( "Deleting class ", cmapName )
      del self.pmapContext.currentEntry_.classAction[ cmapName ]
      deleteClassPrio( self.pmapContext.currentEntry_.classPrio, cmapName )

   def noCoppStaticClassPrio( self, cmapName ):
      if cmapName not in self.pmapContext.currentEntry_.classAction:
         # cmap not a part of pmap. Return quietly.
         return
      # we are in PolicyMapMode. Update currentEntry of its Context.
      t0( "Deleting class ", cmapName )
      del self.pmapContext.currentEntry_.classAction[ cmapName ]
      deleteClassPrio( self.pmapContext.currentEntry_.coppStaticClassPrio, cmapName )
      
   def noClassCopp( self, cmapName ):

      if classMapCpType( cliQosAclConfig, cmapName ) == \
            tacClassMapCpType.cmapCpDynamic:
         self.noClassPrio( cmapName )
      else:
         # class must be Static
         # restore the shaping parameters to default.
         if cmapName == tacCMapNm.coppDrop:
            self.addError( CliError[ 'cannotConfigureDropClass' ] % cmapName )
            return
         if QosLib.isDefaultClass( self.mapType_, cmapName ) == True:
            clAction = self.pmapContext.currentEntry_.classActionDefault
         else:
            clAction = self.pmapContext.currentEntry_.classAction.get( cmapName )

         # class-map is not present in policy-map configuration, nothing to do
         if not clAction:
            return
            
         if qosHwStatus.coppActionShaperSupported:
            shape = clAction.policyAction[ tacActionType.actionSetShape ].rate
            shape.val = tacActionRateType.invalid
            shape.rateUnit = tacRateUnit.rateUnitInvalid
            bw = clAction.policyAction[ tacActionType.actionSetBandwidth ].rate
            bw.val = tacActionRateType.invalid
            bw.rateUnit = tacRateUnit.rateUnitInvalid
         elif qosHwStatus.coppActionPolicerSupported:
            if clAction.policer:
               clAction.policer = None

         if self.pmapName != tacPMapNm.coppName and \
               qosHwStatus.intfCoppSupported:
            self.noCoppStaticClassPrio( cmapName )

   def noClassPdp( self, cmapName ):
      if cmapName == tacCMapNm.classDefault:
         self.addError( CliError[ 'cannotDeleteClassDefault' ] )
      else:
         self.addError( CliError[ 'cannotDeleteBuiltInCmap' ] )

   def noClass( self, cmapName ):
      lastMode = self.session_.modeOfLastPrompt()
      if isinstance( lastMode, PolicyMapClassMode ):
         # we are in PolicyMapClassMode. Set it's context to None.
         if cmapName == lastMode.cmapName:
            lastMode.pmapClassContext = None
      if self.mapType_ == coppMapType:
         self.noClassCopp( cmapName )
      elif self.mapType_ == pdpMapType:
         self.noClassPdp( cmapName )
      else:
         self.noClassPrio( cmapName )

   def setClass( self, no, cmapName, insertBeforeClass=None, default=False ):
      if no:
         self.noClass( cmapName )
      else:
         if self.mapType_ == pdpMapType and default:
            # 'default class <PDP policy class>' will revert to the action for
            # the class specified in the default-pdp-policy.
            src, dst = None, None
            _defaultPdpPmapCfg = getDefaultPdpPmapCfg()
            if QosLib.isDefaultClass( self.mapType_, cmapName ):
               src = _defaultPdpPmapCfg.classActionDefault
               dst = self.pmapContext.currentEntry_.classActionDefault
            else:
               src = _defaultPdpPmapCfg.classAction[ cmapName ]
               self.pmapContext.currentEntry_.classAction.newMember( cmapName )
               dst = self.pmapContext.currentEntry_.classAction[ cmapName ]
            copyClassAction( src, dst )
         else:
            pmapName = self.pmapContext.pmapName()
            self.gotoPolicyMapClassMode( pmapName, cmapName, insertBeforeClass )

   # print one or all policy maps
   @staticmethod
   def showPolicyMap( mode, args ):
      pmapName = args.get( 'copp-system-policy' ) or args.get( 'PMAP' )
      mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
                args.get( 'pdp', 'qos' )
      summary = 'summary' in args 
      policyMapAllModel = PolicyMapAllModel()
      mapType = 'control-plane' if mapType == 'copp' else mapType
      emapType = mapTypeToEnum( mapType )
      policyMapsContainer = PMapModelContainer( qosConfig, qosAclConfig, qosStatus,
                                   qosHwStatus,
                                   qosSliceHwStatus, qosAclHwStatus, 
                                   qosAclSliceHwStatus, emapType, None,
                                   hwEpochStatus, None,
                                   policyMapAllModel )

      if 'counters' in args:
         policyMapsContainer.counterDetail = True
         policyMapsContainer.direction_ = tacDirection.input
         waitForCountersUpdate( mode, mapType )
         if pmapName is None:
            pmapList = sorted( qosAclConfig.pmapType[ emapType ].pmap )
            for policyName in pmapList:
               policyMapsContainer.populatePolicyMapCounters( policyName )
         else:
            policyMapsContainer.populatePolicyMapCounters( pmapName )
         return policyMapAllModel 

      if pmapName == None:
         policyMapsContainer.populateAll( summary )
      else:
         policyMapsContainer.populatePolicyMap( pmapName, summary )
      return policyMapAllModel 

class PolicyMapModeQos( PolicyMapMode ):
   name = "Qos Policy Map Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def getChildMode( self, mapType, context ):
      childMode = self.childMode( PolicyMapClassModeQos,
                                  context=context )
      return childMode

def gotoPolicyMapModeCommon( mode, pmapName, mapType, cmapName, shared ):
   emapType = mapTypeToEnum( mapType )
   shared = bool( shared )
   context = PolicyMapContext( mode, pmapName, emapType, shared=shared )

   if pmapName in cliQosAclConfig.pmapType[ emapType ].pmap:
      pmapShared = cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ].shared
      sharedErrStr = ""
      currentSharedStr = "shared" if pmapShared else "unshared"
      if pmapShared and not shared:
         sharedErrStr = "unshared"
      elif not pmapShared and shared:
         sharedErrStr = "shared"
      if sharedErrStr != "":
         mode.addError( "%s is %s and cannot be configured as %s policy-map" % (
                        pmapName, currentSharedStr, sharedErrStr ) )
         return None
      context.copyEditEntry( cmapName )
      if cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ].dynamic:
         mode.addWarning( DYNAMIC_PMAP_EDIT_WARN )
   else:
      context.newEditEntry( cmapName )
   return context

# Goto 'policy-map' config mode. Create a policy map context.
# The context holds all current editing values. The context
# is associated with the session of the mode.
#------------------------------------------------------------------------
# [no] policy-map type qos <name>
#------------------------------------------------------------------------
def gotoQosPolicyMapMode( mode, args ): 
   pmapName = args[ 'PMAP' ]
   mapType = 'qos'
   t0( 'gotoQosPolicyMapMode of policy', pmapName )
   emapType = mapTypeToEnum( mapType )
   if emapType != coppMapType and pmapName == tacPMapNm.coppName:
      # Don't allow copp-system-policy to be configured as QoS map type
      mode.addError( "%s cannot be configured as QoS map type" % pmapName )
      return
   cmapName = tacCMapNm.classDefault
   context = gotoPolicyMapModeCommon( mode, pmapName, mapType, cmapName, None )
   if context:
      mode.pmapContext = context
      childMode = mode.childMode( PolicyMapModeQos, context=context )
      mode.session_.gotoChildMode( childMode )

def deleteQosPolicyMap( mode, args ):
   pmapName = args[ 'PMAP' ]
   mapType = 'qos'
   deletePolicyMap( mode, pmapName, mapType, None )

def deleteComments( mode, mapType, pmapName=None, cmapName=None ):
   if not cliQosAclConfig or not mode or not mapType:
      return
   emapType_ = mapTypeToEnum( mapType )
   mapTypeString = mapType
   if mapType == 'control-plane':
      mapTypeString = 'copp'
   commentKeys_ = []
   if pmapName:
      if mapType == 'qos':
         mapTypeString = 'quality-of-service'
      if emapType_ in cliQosAclConfig.pmapType:
         if pmapName not in cliQosAclConfig.pmapType[ emapType_ ].pmap:
            return
         cmapNames_ = []
         pmap_ = cliQosAclConfig.pmapType[ emapType_ ].pmap[ pmapName ]
         if cmapName:
            if type( cmapName ) == str:
               cmapNames_ = [ cmapName ]
         else:
            cmapNames_ = [ c.name for c in
                        pmap_.classAction.values() + [ pmap_.classActionDefault ] ]
         for cmapName_ in cmapNames_:
            commentKeys_.append( 'pmap-c-%s-%s-%s' % ( mapType, pmapName,
               cmapName_ ) )
            if mapTypeString != mapType:
               commentKeys_.append( 'pmap-c-%s-%s-%s' % ( mapTypeString, pmapName,
                  cmapName_ ) )
         if not cmapName:
            commentKeys_.append( 'pmap-%s-%s' % ( mapType, pmapName ) )
            if mapTypeString != mapType:
               commentKeys_.append( 'pmap-%s-%s' % ( mapTypeString, pmapName ) )
   elif cmapName:
      if emapType_ in cliQosAclConfig.cmapType:
         if cmapName not in cliQosAclConfig.cmapType[ emapType_ ].cmap:
            return
         commentKeys_.append( 'cmap-%s-%s' % ( mapType, cmapName ) )
         if mapTypeString != mapType:
            commentKeys_.append( 'cmap-%s-%s' % ( mapTypeString, cmapName ) )
   for commentKey_ in commentKeys_:
      mode.removeCommentWithKey( commentKey_ )

def deletePolicyMap( mode, pmapName, mapType='qos', shared=None ):
   emapType = mapTypeToEnum( mapType )
   if mapType == 'pdp' and pmapName == tacPMapNm.defaultPdpPmapName:
      # Cannot delete default-pdp-policy
      mode.addError( "%s cannot be deleted" % pmapName )
      return False

   if pmapName in qosAclConfig.pmapType[ emapType ].pmap:
      pmapShared = qosAclConfig.pmapType[ emapType ].pmap[ pmapName ].shared
      if not pmapShared and shared:
         mode.addError( "%s is unshared and cannot be deleted as shared"
                        " policy-map" % ( pmapName ) )
         return False

   lastMode = mode.session_.modeOfLastPrompt()

   if isinstance( lastMode, PolicyMapClassMode ) :
      lastMode.pmapClassContext = None
      lastMode = lastMode.parent_
   
   if isinstance( lastMode, PolicyMapMode ) and lastMode.pmapName == pmapName:
      lastMode.pmapContext = None
   
   if pmapName not in cliQosAclConfig.pmapType[ emapType ].pmap:
      return False

   deleteComments( mode, mapType, pmapName=pmapName )

   if mapType == 'control-plane' and pmapName == tacPMapNm.coppName:
      # Revert back the pmap to default state
      coppPmap = cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ]
      allClassActions = coppPmap.classAction.values() + \
                        [ coppPmap.classActionDefault ]
      for clAction in allClassActions:
         cmapCpType = classMapCpType( cliQosAclConfig, clAction.name )
         if cmapCpType == tacClassMapCpType.cmapCpStatic:
            shape = clAction.policyAction[ tacActionType.actionSetShape ].rate
            bw = clAction.policyAction[ tacActionType.actionSetBandwidth ].rate
            shape.val = tacActionRateType.invalid
            shape.rateUnit = tacRateUnit.rateUnitInvalid
            bw.val = tacActionRateType.invalid
            bw.rateUnit = tacRateUnit.rateUnitInvalid
         else:
            del coppPmap.classAction[ clAction.name ]

      for prio in coppPmap.classPrio:
         del coppPmap.classPrio[ prio ]
         
      # Update the config version
      coppPmap.uniqueId = Tac.Value( 'Qos::UniqueId' )
      coppPmap.version += 1
   else:
      del cliQosAclConfig.pmapType[ emapType ].pmap[ pmapName ]
      if not mode.session.inConfigSession():
         currTime = getCurrentTimeForConfigUpdate()
         description = "policy-map to be deleted"
         cliBlockingToApplyConfigChange( mode, currTime, description,
                                         policyMap=True )
   return True

# Action for qos policy-map copy command. It takes a Source policy map
# as argument and creates PolicyMapCopyContext.
# TODO: Currently only Qos type policy maps are copyable. In future
# when we support more types, this method should be renamed appropriately
#------------------------------------------------------------------------
# policy-map [type qos] <name> copy <name>
#------------------------------------------------------------------------
def copyQosPolicyMap( mode, args ):
   dstPmapName = args[ 'DST_PMAP' ]
   srcPmapName = args[ 'SRC_PMAP' ]
   mapType = 'qos'
   t0( 'copyQosPolicyMap: copying parameters from %s to %s' %
       ( srcPmapName, dstPmapName ) )
   emapType = mapTypeToEnum( mapType )
   assert emapType in copyablePMapTypes()

   if srcPmapName not in cliQosAclConfig.pmapType[ emapType ].pmap:
      mode.addError(
            'Cannot create %s. Policy %s does not exist or is not copyable.' %
            ( dstPmapName, srcPmapName ) )
      return None

   if dstPmapName in cliQosAclConfig.pmapType[ emapType ].pmap:
      mode.addError( 'Policy map %s already exists.' %
                     dstPmapName )
      return None

   # Find the source policy-map
   srcPmap = cliQosAclConfig.pmapType[ emapType ].pmap[ srcPmapName ]
   ctx = PolicyMapCopyContext( mode, dstPmapName, srcPmap )

   # Create a new empty policy-map with class-default and copy the
   # src policy-map
   ctx.newEditEntry( cmapName=tacCMapNm.classDefault )
   mode.pmapContext = ctx
   childMode = mode.childMode( PolicyMapModeQos, context=ctx )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------
# Register top-level CLI commands ( policy-map )
#------------------------------------------------------------------------

#-----------------------------------------------------------------
# The show command in 'config-pmap' mode
#
#              show active|pending|diff
#-----------------------------------------------------------------
def _showPMapList( pmapType, pmapName, pmap, output=None ):
   if pmap is None:
      return
   if output is None:
      output = sys.stdout

   def _printClassActionQos( classAction ):
      actions = classAction.policyAction
      for actionType in pmapQosActionTypes:
         if actionType in actions:
            if actionType in [ tacActionType.actionSetDscp, 
                               tacActionType.actionSetCos,
                               tacActionType.actionSetDropPrecedence,
                               tacActionType.actionSetTc ]:
               actStr = actionFromEnum( actionType )
               action = actions[ actionType ]

               # For DSCP, the name-string should be printed, if configured.
               if ( actionType == tacActionType.actionSetDscp ) and \
                  classAction.dscpConfiguredAsName:
                  name = AclCliLib.dscpNameFromValue( action.value )
                  output.write( "    set %s %s\n" % ( actStr, name ) )
               else:
                  output.write( "    set %s %d\n" % ( actStr, action.value ) )

            elif actionType in [ tacActionType.actionSetDrop ]:
               output.write( "    drop\n" )

      policer = classAction.policer
      if policer:
         # BUG197856: Currently a named policer can only be a shared policer.
         # When we support policer profiles for unshared policers, we need to
         # update this check to look at shared and named attributes.
         if policer.named:
            output.write( "    police shared %s" % ( policer.name ) )
         else:
            output.write( "    police rate %d %s burst-size %d %s" % (
               policer.cir, rateUnitFromEnum( policer.cirUnit ),
               policer.bc, burstUnitFromEnum( policer.bcUnit ) ) )
         if policer.pir:
            yellowActions = policer.yellowActions
            for actionType in yellowActions:
               actStr = actionFromEnum( actionType )
               action = yellowActions[ actionType ]
               if actionType in [ tacActionType.actionSetDscp,
                                  tacActionType.actionSetCos,
                                  tacActionType.actionSetDropPrecedence, 
                                  tacActionType.actionSetTc ]:
                  output.write( " action set %s %d" % ( actStr, action.value ) )
               else:
                  output.write( " action set %s" % actStr )
            output.write( " rate %d %s burst-size %d %s\n"
                          % ( policer.pir, rateUnitFromEnum( policer.pirUnit ),
                              policer.be, burstUnitFromEnum( policer.beUnit ) ) )
         else:
            output.write( "\n" )
      output.write( "\n" )

   def _printClassActionCopp( classAction ):
      actions = classAction.policyAction

      def _getStaticClass( classAction ):
         cpStaticType =  classMapCpStaticType ( cliQosAclConfig, classAction.name )
         hwCoppStaticClass = coppStaticClassFromHwStatus( qosSliceHwStatus ) 
         return hwCoppStaticClass[ cpStaticType ]

      if tacActionType.actionSetShape in actions:
         shape = actions[ tacActionType.actionSetShape ].rate
         if shape.val == tacActionRateType.noValue:
            output.write( "    no shape\n" )
         else: 
            if shape.val != tacActionRateType.invalid:
               output.write( "    shape %s %d\n" % ( 
                     rateUnitFromEnum( shape.rateUnit ), 
                     shape.val ) )
            else:
               staticClass = _getStaticClass ( classAction )
               output.write( "    shape %s %d\n" % (
                     rateUnitFromEnum( staticClass.defaultRateUnit ),
                     staticClass.defaultMax ) )

      if tacActionType.actionSetBandwidth in actions:
         bandwidth = actions[ tacActionType.actionSetBandwidth ].rate
         if bandwidth.val == tacActionRateType.noValue:
            output.write( "    no bandwidth\n" )
         else: 
            if bandwidth.val != tacActionRateType.invalid:
               output.write( "    bandwidth %s %d\n" % ( 
                     rateUnitFromEnum( bandwidth.rateUnit ), 
                     bandwidth.val ) )
            else:
               staticClass = _getStaticClass ( classAction )
               output.write( "    bandwidth %s %d\n" % (
                     rateUnitFromEnum( staticClass.defaultRateUnit ),
                     staticClass.defaultMin ) )

   def _printClassAction( classAction ):
      if classAction is None:
         return
      if classAction.policyAction is None:
         return
      if pmapType == coppMapType:
         _printClassActionCopp( classAction )
      else:
         _printClassActionQos( classAction )
         
   mapTypeString = mapTypeFromEnum( pmapType )
   if pmapType == coppMapType:
      mapTypeString = 'copp'
   elif pmapType == qosMapType:
      mapTypeString = 'quality-of-service'
   output.write( "policy-map type %s %s%s\n" % ( 
         mapTypeString, "shared " if pmap.shared else "", pmapName ) )

   for _, classPrio in sorted( pmap.classPrio.iteritems() ):
      cmapName = classPrio.cmapName
      output.write( "  class %s%s\n" % ( "built-in " if pmapType == pdpMapType and
                                         cmapName in builtInClassMapNames else "",
                                         cmapName ) )
      classAction = pmap.classAction[ cmapName ]
      _printClassAction( classAction )

   for _, coppStaticClassPrio in sorted( pmap.coppStaticClassPrio.iteritems() ):
      cmapName = coppStaticClassPrio.cmapName
      if isCoppStaticClassSupported( cliQosAclConfig, qosSliceHwStatus, cmapName ):
         if cmapName != tacCMapNm.coppDrop:
            output.write( "  class %s\n" % ( cmapName ) )
            classAction = pmap.classAction[ cmapName ]
            _printClassAction( classAction )

   cmapName = defaultClassName( pmapType )
   output.write( "  class %s\n" % ( cmapName ) )
   classAction = pmap.classActionDefault
   _printClassAction( classAction )

def showPMapCurrent( mode ):
   _showPMapList( mode.mapType_, mode.pmapName, mode.pmapContext.currentEntry_ )

def showPMapActive( mode ):
   mode.showActive() 

def showPMapDiff( mode ):
   # Generate diff between active and pending
   import difflib, cStringIO
   activeOutput = cStringIO.StringIO( )
   _showPMapList( mode.mapType_, mode.pmapName, mode.pmapContext.map_, 
                  output=activeOutput )
   pendingOutput = cStringIO.StringIO( )
   _showPMapList( mode.mapType_, mode.pmapName, mode.pmapContext.currentEntry_, 
                  output=pendingOutput )
   diff = difflib.unified_diff( activeOutput.getvalue( ).splitlines( ),
                                pendingOutput.getvalue( ).splitlines( ),
                                lineterm='' )
   print '\n'.join( list( diff ) )

#------------------------------------------------------------------------
# Match and set value binding rules ( policy-map )
#------------------------------------------------------------------------
#--------------------------------------------------------------------------------
# [ no | default ] class CMAP [ insert-before CLASS ]
#--------------------------------------------------------------------------------
class InsertBeforeClassPmapModeCmd( CliCommand.CliCommandClass ):
   syntax = 'class CMAP [ insert-before CLASS ]'
   noOrDefaultSyntax = syntax
   data = {
      'insert-before': CliCommand.guardedKeyword( 'insert-before',
         helpdesc='insert the class with a higher priority than a given class',
         guard=guardClassMap ),
      'class': 'Policy criteria',
      'CMAP': CliMatcher.DynamicNameMatcher( getClassNameRuleQos, 
         helpdesc='Class Map Name' ),
      'CLASS': CliMatcher.DynamicNameMatcher( getClassNameRuleQos, 
         helpdesc='Class Map Name' ),
   }

   @staticmethod
   def handler( mode, args ):
      no = CliCommand.isNoOrDefaultCmd( args )
      mode.setClass( no, args[ 'CMAP' ], args.get( 'CLASS' ) )

   noHandler = handler
   defaultHandler = handler

PolicyMapModeQos.addCommandClass( InsertBeforeClassPmapModeCmd )

matcherSet = CliMatcher.KeywordMatcher( 'set', 
      helpdesc="Set QoS values" )
matcherPoliceBurstValue = CliMatcher.PatternMatcher( r'\d+', 
      helpname='<integer value>', helpdesc='Specify burst value' )
matcherPolicerName = CliMatcher.DynamicNameMatcher( getPolicerNameRule,
      helpdesc="Policer name" )
matcherPoliceCirValue = CliMatcher.PatternMatcher( r'\d+', 
      helpname='<integer value>', helpdesc="Specify lower rate" )
matcherRateUnit = CliMatcher.EnumMatcher( {
   'bps': 'Rate in bps (default)',
   'kbps': 'The rate expressed in units of kilobits per second',
   'mbps': 'Rate in Mbps',
} )
matcherBurstUnit = CliMatcher.EnumMatcher( {
   'bytes': 'Burst size in bytes (default unit)',
   'kbytes': 'Burst size in kbytes',
   'mbytes': 'Burst size in mbytes',
} )
nodePMapQosDropPrecedence = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc="Set drop precedence", guard=guardPMapQosActionDropPrecedence )
nodeActionDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc="DSCP values", guard=guardPMapQosAction )
nodeActionCos = CliCommand.guardedKeyword( 'cos', 
      helpdesc="Set COS", guard=guardPMapQosAction )
nodeActionTc = CliCommand.guardedKeyword( 'traffic-class', 
      helpdesc="Set Traffic class", guard=guardPMapQosAction )
# Drop action supported support on trident+ 
nodeActionDrop = CliCommand.guardedKeyword( 'drop', 
      helpdesc="Drop Packets", guard=guardDropAction )
nodePolice = CliCommand.guardedKeyword( 'police', 
      helpdesc="Configure policer parameters", guard=guardIngressPolicing )
nodePoliceShared = CliCommand.guardedKeyword( 'shared', 
      helpdesc="Use a shared policer", guard=guardNamedSharedPolicerSupported )
nodePoliceCir = CliCommand.guardedKeyword( 'cir', 
      helpdesc='Set committed information rate', guard=guardIngressPolicing )
nodePoliceBc = CliCommand.guardedKeyword( 'bc', 
      helpdesc='Set committed burst rate', guard=guardIngressPolicing )
nodePoliceLRate = CliCommand.guardedKeyword( 'rate', helpdesc="Set lower Rate",
      guard=guardIngressPolicing )
nodePoliceLBs = CliCommand.guardedKeyword( 'burst-size',
      helpdesc="Set burst-size for lower rate", guard=guardIngressPolicing )

#------------------------------------------------------------------------
# set drop-precedence DROP_PRECEDENCE
# ( no | default ) set drop-precedence ...
#------------------------------------------------------------------------
class SetDropPrecedenceCmd( CliCommand.CliCommandClass ):
   syntax = 'set drop-precedence DROP_PRECEDENCE'
   noOrDefaultSyntax = 'set drop-precedence ...'
   data = {
      'set': matcherSet,
      'drop-precedence': nodePMapQosDropPrecedence,
      'DROP_PRECEDENCE': CliMatcher.IntegerMatcher(
         validDropPrecedenceValues[ 0 ],
         validDropPrecedenceValues[ -1 ], helpdesc='Drop precedence value' ),
   }

   handler = PolicyMapClassModeQos.configureSetDropPrecedence
   noOrDefaultHandler = handler

PolicyMapClassModeQos.addCommandClass( SetDropPrecedenceCmd )

#------------------------------------------------------------------------
# set dscp ( DSCP_NAME | DSCP )
# ( no | default ) set dscp ...
#------------------------------------------------------------------------
class PolicyMapClassModeSetDscpCmd( CliCommand.CliCommandClass ):
   syntax = 'set dscp ( DSCP_NAME | DSCP )'
   noOrDefaultSyntax = 'set dscp ...'
   data = {
      'set': matcherSet,
      'dscp': nodeActionDscp,
      'DSCP': CliMatcher.IntegerMatcher( 0, AclLib.MAX_DSCP,
         helpdesc="DSCP Value" ),
      'DSCP_NAME': CliMatcher.EnumMatcher(
         { k : v[1] for k, v in AclCliLib.dscpAclNames.iteritems() } ),
   }

   handler = PolicyMapClassModeQos.configureSetDscp
   noOrDefaultHandler = handler

PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeSetDscpCmd )

#------------------------------------------------------------------------
# set cos COS
# ( no | default ) set cos ...
#------------------------------------------------------------------------
class PolicyMapClassModeSetCosCmd( CliCommand.CliCommandClass ):
   syntax = 'set cos COS'
   noOrDefaultSyntax = 'set cos ...'
   data = {
      'set': matcherSet,
      'cos': nodeActionCos,
      'COS': CliMatcher.IntegerMatcher( tacCos.min, tacCos.max,
         helpdesc="Class of Service (CoS) value" ),
   }

   handler = PolicyMapClassModeQos.configureSetCos
   noOrDefaultHandler = handler

PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeSetCosCmd )

#------------------------------------------------------------------------
# set traffic-class TRAFFIC_CLASS
# ( no | default ) set traffic-class ...
#------------------------------------------------------------------------
class PolicyMapClassModeSetTcCmd( CliCommand.CliCommandClass ):
   syntax = 'set traffic-class TRAFFIC_CLASS'
   noOrDefaultSyntax = 'set traffic-class ...'
   data = {
      'set': matcherSet,
      'traffic-class': nodeActionTc,
      'TRAFFIC_CLASS': CliMatcher.DynamicIntegerMatcher( trafficClassRangeFn,
         helpdesc="Traffic class value" ),
   }

   handler = PolicyMapClassModeQos.configureSetTc
   noOrDefaultHandler = handler

PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeSetTcCmd )

#------------------------------------------------------------------------
# drop
# ( no | default ) drop ...
#------------------------------------------------------------------------
class PolicyMapClassModeDropCmd( CliCommand.CliCommandClass ):
   syntax = 'drop'
   noOrDefaultSyntax = 'drop ...'
   data = {
      'drop': nodeActionDrop,
   }

   handler = PolicyMapClassModeQos.configureSetDrop
   noOrDefaultHandler = handler

PolicyMapClassModeQos.addCommandClass( PolicyMapClassModeDropCmd )

#------------------------------------------------------------------------
# police rate <value> {bps, kbps, mbps, gbps} burst-size <value>
# {bytes, kbytes, mbytes} [ [ action set {drop-precedence, dscp <value>} ]
# rate <value> {bps, kbps, mbps, gbps} burst-size <value> {bytes, kbytes, mbytes} ]
#------------------------------------------------------------------------
class CirExpression( CliCommand.CliExpression ):
   expression = 'bps | kbps | mbps | pps'
   data = {
      'bps': 'rate in bps (default unit)',
      'kbps': 'The rate expressed in units of kilobits per second',
      'mbps': 'rate in Mbps',
      'pps': CliCommand.guardedKeyword( 'pps', helpdesc='rate in pps',
         guard=guardPoliceRateInPps ),
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      rateUnitsToTypes = {
            'bps': tacRateUnit.rateUnitbps,
            'kbps': tacRateUnit.rateUnitKbps,
            'mbps': tacRateUnit.rateUnitMbps,
            'pps': tacRateUnit.rateUnitPps,
            }

      for rate in ( 'bps', 'kbps', 'mbps', 'pps' ):
         if rate in args:
            args[ 'RATE_CIR_UNIT' ] = rateUnitsToTypes[ args[ rate ] ]
            break

class PolicerBurstExpression( CliCommand.CliExpression ):
   expression = 'bytes | kbytes | mbytes | packets'
   data = {
      'bytes': 'burst size in bytes (default unit)',
      'kbytes': 'burst size in kbytes',
      'mbytes': 'burst size in mbytes',
      'packets': CliCommand.guardedKeyword( 'packets',
         helpdesc='burst size in packets', guard=guardPoliceRateInPps ),
      }

   @staticmethod
   def adapter( mode, args, argsList ):
      burstUnitsToTypes = {
            'bytes': tacBurstUnit.burstUnitBytes,
            'kbytes': tacBurstUnit.burstUnitKBytes,
            'mbytes': tacBurstUnit.burstUnitMBytes,
            'packets': tacBurstUnit.burstUnitPackets,
            }

      for rate in ( 'bytes', 'kbytes', 'mbytes', 'packets' ):
         if rate in args:
            args[ 'BURST_BC_UNIT' ] =  burstUnitsToTypes[ args[ rate ] ]
            break

class PoliceRateBurstVerTwoCmd( CliCommand.CliCommandClass ):
   syntax = 'police rate CIR [ RATE_UNIT ] burst-size BURST [ BURST_UNIT ]'
   data = {
      'police': nodePolice,
      'rate': nodePoliceLRate,
      'CIR': matcherPoliceCirValue,
      'RATE_UNIT': CirExpression,
      'burst-size': nodePoliceLBs,
      'BURST': matcherPoliceBurstValue,
      'BURST_UNIT': PolicerBurstExpression,
   }

   @staticmethod
   def handler( mode, args ):
      mode.configurePolicer( mode, no=False, cir=int( args[ 'CIR' ] ),
            cirUnit=args.get( 'RATE_CIR_UNIT' ), bc=int( args[ 'BURST' ] ),
            bcUnit=args.get( 'BURST_BC_UNIT' ), cmdVersion=2 )

PolicyMapClassModeQos.addCommandClass( PoliceRateBurstVerTwoCmd )

matcherPolicePirValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify higher rate' )
matcherPoliceExcessBurstValue = CliMatcher.PatternMatcher( r'\d+',
      helpname='<integer value>', helpdesc='Specify burst size' )
nodePoliceAction = CliCommand.guardedKeyword( 'action',
      helpdesc='Specify action for lower rate', guard=guardPoliceAction )
nodePoliceSetAction = CliCommand.guardedKeyword( 'set',
      helpdesc='Set action for policed traffic', guard=guardPoliceAction )
nodePoliceActionDP = CliCommand.guardedKeyword( 'drop-precedence',
      helpdesc='Mark yellow packets with drop-precedence',
      guard=guardDropPrecedenceYellowAction )
nodePoliceHRate = CliCommand.guardedKeyword( 'rate',
      helpdesc='Set higher rate', guard=guardIngressTrTcmPolicing )
nodePoliceHBs = CliCommand.guardedKeyword( 'burst-size',
      helpdesc='Set burst-size for higher rate', guard=guardIngressTrTcmPolicing )
nodePoliceActionDscp = CliCommand.guardedKeyword( 'dscp',
      helpdesc='Mark yellow packets with DSCP value',
      guard=guardDscpYellowAction )

class CirPirUnit( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      nodeCirPirUnit = CliCommand.Node( matcher=matcherRateUnit, alias=name )
      nodepps = CliCommand.guardedKeyword( 'pps', helpdesc='Rate in pps',
            guard=guardPoliceRateInPps, alias=name )
      
      class CirPirExpression( CliCommand.CliExpression ):
         expression = '%s_PIR_CIR_UNIT | %s_pps' % ( name, name )
         data = { 
            '%s_PIR_CIR_UNIT' %name : nodeCirPirUnit, 
            '%s_pps' %name: nodepps,
            }
         
         @staticmethod
         def adapter( mode, args, argsList ):
            rateUnitsToTypes = {
                  'bps': tacRateUnit.rateUnitbps,
                  'kbps': tacRateUnit.rateUnitKbps,
                  'mbps': tacRateUnit.rateUnitMbps,
                  'pps': tacRateUnit.rateUnitPps,
               }

            args[ 'CIR_SPEED_UNIT' ] = \
                  rateUnitsToTypes[ args.get( 'CIRUNIT', 'kbps' ) ]
            args[ 'PIR_SPEED_UNIT' ] = \
                  rateUnitsToTypes[ args.get( 'PIRUNIT', 'kbps' ) ]
      return CirPirExpression

class BurstHigherUnit( CliCommand.CliExpressionFactory ):
   def generate( self, name ):
      nodeHBurstUnit = CliCommand.Node( matcher=matcherBurstUnit, alias=name )
      nodePackets = CliCommand.guardedKeyword( 'packets', 
            helpdesc='Burst size in packets', 
            guard=guardPoliceRateInPps, alias=name )
      
      class BurstHigherRateExpression( CliCommand.CliExpression ):
         expression = '%s_H_BURST_UNIT | %s_packets' % ( name, name )
         data = {
               '%s_H_BURST_UNIT' %name : nodeHBurstUnit,
               '%s_packets' %name : nodePackets,
            }
         
         @staticmethod
         def adapter( mode, args, argsList ):
            burstUnitsToTypes = {
                  'bytes': tacBurstUnit.burstUnitBytes,
                  'kbytes': tacBurstUnit.burstUnitKBytes,
                  'mbytes': tacBurstUnit.burstUnitMBytes,
                  'packets': tacBurstUnit.burstUnitPackets,
               }

            args[ 'BURST_RATE_UNIT' ] = \
                  burstUnitsToTypes[ args.get( 'BURSTUNIT', 'bytes' ) ]
            args[ 'HBURST_RATE_UNIT' ] = \
                  burstUnitsToTypes[ args.get( 'HBUNIT', 'bytes' ) ]
      return BurstHigherRateExpression

#------------------------------------------------------------------------
# police rate CIR [ CIRUNIT ] burst-size BURSTSIZE [ BUNIT ] 
#    [ action set ( drop-precedence | (  dscp ( DSCP | DSCP_NAME ) ) ) ]
#      highrate PIRVALUE [ PIRUNIT ] highbsize HBURSTSIZE [ HBUNIT ]
#------------------------------------------------------------------------
class PoliceRateHigherRateBurstVerTwoCmd( CliCommand.CliCommandClass ):
   syntax = ( 'police rate CIR [ CIRUNIT ] burst-size BURSTSIZE '
              '[ BURSTUNIT ] [ action set ( drop-precedence | ( dscp '
              '( DSCP | DSCP_NAME ) ) ) ] highrate PIRVALUE [ PIRUNIT ] '
              'highbsize HBURSTSIZE [ HBUNIT ]' )
   data = {
      'police': nodePolice,
      'rate': nodePoliceLRate,
      'CIR': matcherPoliceCirValue,
      'CIRUNIT': CirPirUnit(),
      'burst-size': nodePoliceLBs,
      'BURSTSIZE': matcherPoliceBurstValue,
      'BURSTUNIT': BurstHigherUnit(),
      'action': nodePoliceAction,
      'set': nodePoliceSetAction,
      'drop-precedence':nodePoliceActionDP,
      'dscp': nodePoliceActionDscp,
      'DSCP': CliMatcher.IntegerMatcher( 0, AclLib.MAX_DSCP,
         helpdesc="DSCP Value" ),
      'DSCP_NAME': CliMatcher.DynamicKeywordMatcher(
         lambda mode: dict( ( key, val[1] ) for key, val in
            AclCliLib.dscpAclNames.iteritems() ) ),
      'highrate': nodePoliceHRate,
      'PIRVALUE': matcherPolicePirValue,
      'PIRUNIT' : CirPirUnit(),
      'highbsize': nodePoliceHBs,
      'HBURSTSIZE': matcherPoliceExcessBurstValue,
      'HBUNIT' : BurstHigherUnit(),
      }
   
   @staticmethod
   def handler( mode, args ):
      if 'drop-precedence' in args:
         mode.configurePolicer( mode, no=False, policerMode='trTCM',
               cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
               bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
               policerYellowAction=args[ 'drop-precedence' ],
               pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
               be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
               cmdVersion=2 )
      elif 'dscp' in args:
         dscpValue = args.get( 'DSCP_NAME' ) or args.get( 'DSCP' )
         mode.configurePolicer( mode, no=False, policerMode='trTCM',
               cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
               bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
               policerYellowAction=args[ 'dscp' ], policerYellowValue=dscpValue,
               pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
               be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
               cmdVersion=2 )
      else:
         mode.configurePolicer( mode, no=False, policerMode='trTCM',
               cir = int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
               bc=int( args[ 'BURSTSIZE' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ],
               pir=int( args[ 'PIRVALUE' ] ), pirUnit=args[ 'PIR_SPEED_UNIT' ],
               be=int( args[ 'HBURSTSIZE' ] ), beUnit=args[ 'HBURST_RATE_UNIT' ],
               cmdVersion=2 )

PolicyMapClassModeQos.addCommandClass( PoliceRateHigherRateBurstVerTwoCmd )

#------------------------------------------------------------------------
# police cir <value> {bps, kbps, mbps} bc <value> {bytes, kbytes, mbytes}
# ( no | default ) police ...
#------------------------------------------------------------------------
def rateBurstUnitAdapter( mode, args, argsList ):
   rateUnitsToTypes = {
      'bps': tacRateUnit.rateUnitbps,
      'kbps': tacRateUnit.rateUnitKbps,
      'mbps': tacRateUnit.rateUnitMbps,
      'pps': tacRateUnit.rateUnitPps,
   }

   burstUnitsToTypes = {
      'bytes': tacBurstUnit.burstUnitBytes,
      'kbytes': tacBurstUnit.burstUnitKBytes,
      'mbytes': tacBurstUnit.burstUnitMBytes,
      'packets': tacBurstUnit.burstUnitPackets,
   }

   args[ 'RATE_CIR_UNIT' ] = rateUnitsToTypes[ args.get( 'RATE_UNIT', 'bps' ) ]
   args[ 'BURST_BC_UNIT' ] = burstUnitsToTypes[ args.get( 'BURST_UNIT', 'bytes' ) ]

class PoliceRateBurstVerOneCmd( CliCommand.CliCommandClass ):
   syntax = 'police cir CIR [ RATE_UNIT ] bc BC [ BURST_UNIT ]'
   noOrDefaultSyntax = 'police ...'
   data = {
      'police': nodePolice,
      'cir': nodePoliceCir,
      'CIR': matcherPoliceCirValue,
      'RATE_UNIT': matcherRateUnit,
      'bc': nodePoliceBc,
      'BC': matcherPoliceBurstValue,
      'BURST_UNIT': matcherBurstUnit,
   }

   adapter = rateBurstUnitAdapter

   @staticmethod
   def handler( mode, args ):
      mode.configurePolicer( mode, no=False, cir=int( args[ 'CIR' ] ),
            cirUnit=args[ 'RATE_CIR_UNIT' ], bc=int( args[ 'BC' ] ),
            bcUnit=args[ 'BURST_BC_UNIT' ], cmdVersion=1 )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.configurePolicer( mode, no=True )

PolicyMapClassModeQos.addCommandClass( PoliceRateBurstVerOneCmd )

#----------------------------
# police shared <policerName>
#----------------------------
class PolicyMapClassSharedPolicerCmd( CliCommand.CliCommandClass ):
   syntax = 'police shared POLICER_NAME'
   data = {
      'police': nodePolice,
      'shared': nodePoliceShared,
      'POLICER_NAME': matcherPolicerName,
   }

   @staticmethod
   def handler( mode, args ):
      mode.configurePolicer( mode, no=False, name=args[ 'POLICER_NAME' ] )

PolicyMapClassModeQos.addCommandClass( PolicyMapClassSharedPolicerCmd )

def convertPolicerConfigCmdVersion( mode ):
   for mapType in cliQosAclConfig.pmapType:
      qosPMapConfig = cliQosAclConfig.pmapType[ mapType ]
      for pmapName in qosPMapConfig.pmap:
         pmapConfig = qosPMapConfig.pmap.get( pmapName )
         for classMapName in pmapConfig.classAction:
            classMapConfig = pmapConfig.classAction.get( classMapName )
            if classMapConfig.policer:
               classMapConfig.policer.cmdVersion = 2

#------------------------------------------------------------------------
# Register convertPolicerConfigCmdVersion via "config convert new-syntax"
#------------------------------------------------------------------------
CliPlugin.ConfigConvert.registerConfigConvertCallback(
   convertPolicerConfigCmdVersion )

# Clear counters for qos and pdp mapType
def clearPMapIntfCounters( mode, args ):
   mapTypeOptions = [ 'qos', 'pdp' ]
   mapType = None
   for opt in mapTypeOptions:
      if opt in args:
         mapType = args[ opt ]
         break
   pmapName = args.get( 'PMAPNAME' )
   # Signal QosAgent to clear counters from "qos/status"
   if pmapName is None and mapType in [ 'qos', 'pdp' ]:
      # Clear counters for all policy-maps of mapType
      if mapType == 'qos':
         cliCounterConfig.clearPolicyQosCountersRequestTime = Tac.now()
      elif mapType == 'pdp':
         cliCounterConfig.clearPdpCountersRequestTime = Tac.now()
   elif mapType in [ 'qos', 'pdp' ]:
      # Clear counters for a single policy-map pmapName of type mapType.
      direction = tacDirection.input
      emapType = mapTypeToEnum( mapType )
      spConfigKey = Tac.newInstance( "Qos::ServicePolicyKey", emapType,
                                     direction, pmapName )
      clearCountersTime = Tac.now()
      if spConfigKey in qosInputConfig.servicePolicyConfig:
         spConfig = qosInputConfig.servicePolicyConfig[ spConfigKey ]
         spConfig.clearCountersTime = clearCountersTime
      if spConfigKey in qosInputProfileConfig.servicePolicyConfig:
         spConfig = qosInputProfileConfig.servicePolicyConfig[ spConfigKey ]
         spConfig.clearCountersTime = clearCountersTime
   else:
      cliCounterConfig.clearCoppCountersRequestTime = Tac.now()

def getIntfsForServicePolicy( mapType, pmapName, direction ):
   servicePolicyKey = None
   for servicePolicy in qosStatus.servicePolicyStatus:
      if servicePolicy.pmapName == pmapName and \
             servicePolicy.type == mapType and \
             servicePolicy.direction == direction:
         servicePolicyKey = servicePolicy
   if servicePolicyKey is None:
      return []
   intfsConfigured = []
   for intf, enabled in \
          qosStatus.servicePolicyStatus[ servicePolicyKey ].intfIds.iteritems():
      if enabled:
         intfsConfigured.append( intf.intfId )
   intfsConfigured = Arnet.sortIntf( intfsConfigured )
   return intfsConfigured

def showPMapInterface( mode, args ):
   mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
             args.get( 'pdp', 'qos' )
   intf = args.get( 'INTF' ) or args.get( 'SVI_INTF' ) or args.get( 'VTI_INTF' )  
   direction = args.get( 'input' )
   intfs = IntfCli.Intf.getAll( mode, intf, None )
   if not intfs:
      return
   emapType = mapTypeToEnum( mapType )
   if mapType != 'qos' and mapType != 'pdp':
      for hwStatus_ in qosSliceHwStatus.itervalues(): 
         if emapType not in hwStatus_.pmapType:
            return
   else:
      for aclHwStatus_ in qosAclSliceHwStatus.itervalues():
         if emapType not in aclHwStatus_.pmapType:
            return
   if direction == None:
      directions = [ tacDirection.input ] if mapType == 'pdp' else \
                   [ tacDirection.input, tacDirection.output ]
   else:
      directions = [ directionToEnum( direction ) ]
   policyMapAllModel = PolicyMapAllModel()
   if 'counters' in args:
      waitForCountersUpdate( mode, mapType )
   for intf in intfs:
      pmapsFound = findServicePolicyForIntf( intf, emapType, directions )
      pmapDict = {}
      # If a pmap is applied in both the direction we need to set the direction as
      # None
      for pmapName, direction in pmapsFound:
         if pmapName in pmapDict:
            if pmapDict[ pmapName ] != direction:
               pmapDict[ pmapName ] = None
         else:
            pmapDict[ pmapName ] = direction

      for pmapName in pmapDict:
         direction = pmapDict[ pmapName ]
         policyMapsContainer = PMapModelContainer( qosConfig, qosAclConfig,
                                                   qosStatus,
                                                   qosHwStatus,
                                                   qosSliceHwStatus,
                                                   qosAclHwStatus,
                                                   qosAclSliceHwStatus,
                                                   emapType, direction,
                                                   hwEpochStatus,
                                                   None,
                                                   policyMapAllModel )
         if 'counters' in args:
            policyMapsContainer.counterDetail = True
         policyMapsContainer.populatePolicyMapInterface( pmapName )
   return policyMapAllModel

#------------------------------------------------------------------------
# show policy-map interface <interface> type qos
#------------------------------------------------------------------------
def guardSviPolicyQosSupported( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
          qosAclHwStatus.sviPolicyQosSupported:
      return None
   return CliParser.guardNotThisPlatform

def vxlanPolicyQosSupported( mode, token ):
   if qosAclHwStatus.policyQosSupported and \
          qosAclHwStatus.vxlanPolicyQosSupported:
      return None
   return CliParser.guardNotThisPlatform

def showPolicyMapCountersHelper( mode, emapType, pmapName, direction,
                                 detail, policyMapAllModel, hwStatus ):
   pmapConfigTypePtr = qosAclConfig.pmapType.get( emapType )
   if pmapConfigTypePtr is None:
      return policyMapAllModel
   pmapConfigPtr = pmapConfigTypePtr.pmap.get( pmapName )
   if pmapConfigPtr is None:
      return policyMapAllModel
   for hwStatus_ in hwStatus.itervalues():
      # Static CoPP is not divergent so can break the loop 
      # if pmapType is present in any of the slices.
      pmapHwStatusTypePtr = hwStatus_.pmapType.get( emapType )
      if pmapHwStatusTypePtr is not None:
         break
      else:
         return policyMapAllModel
   key = Tac.newInstance( "Qos::PolicyMapHwStatusKey", direction, pmapName )
   pmapHwStatusPtr = pmapHwStatusTypePtr.pmap.get( key )
   if pmapHwStatusPtr is None:
      return policyMapAllModel

   if pmapHwStatusPtr.prgmStatus != tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
      return policyMapAllModel
   policyMapsContainer = PMapModelContainer( qosAclConfig, qosStatus,
                                             qosHwStatus, qosSliceHwStatus,
                                             qosAclHwStatus,
                                             qosAclSliceHwStatus,
                                             emapType, direction,
                                             hwEpochStatus, None,
                                             policyMapAllModel )
   policyMapsContainer.counterDetail = True
   policyMapsContainer.populatePolicyMapCounters( pmapName )
   return policyMapAllModel

def waitForCountersUpdate( mode, mapType ):
   cliCounterConfig.pmapCounterUpdateRequestTime = Tac.now() # monotonic time
   if mapType == 'control-plane' and \
      not qosAclHwStatus.coppDynamicClassSupported and \
      not qosAclHwStatus.intfCoppSupported:
      hwStatus = qosSliceHwStatus
   else:
      hwStatus = qosAclSliceHwStatus
   def countersUpdated():
      status = True
      if hwStatus == qosSliceHwStatus:
         for sliceHwStatus in qosSliceHwStatus.itervalues():
            if sliceHwStatus.pmapCounterUpdateTime < \
                  cliCounterConfig.pmapCounterUpdateRequestTime:
               status = False
               break
      else:
         for aclSliceHwStatus in qosAclSliceHwStatus.itervalues():
            if aclSliceHwStatus.pmapCounterUpdateTime < \
                  cliCounterConfig.pmapCounterUpdateRequestTime:
               status = False
               break
      return status

   # Check if we have any policy maps of type mapType configured. Prevents
   # "show policy-map counters" from timing out when there are no qos policy-maps
   # configured.
   emapType = mapTypeToEnum( mapType )
   mapTypePolicyPresent = False
   if hwStatus == qosAclSliceHwStatus:
      for aclSliceHwStatus in hwStatus.itervalues():
         pmapType = aclSliceHwStatus.pmapType.get( emapType, None )
         if pmapType is not None and len( pmapType.pmap ) > 0:
            mapTypePolicyPresent = True
            break
   else:
      for sliceHwStatus in hwStatus.itervalues():
         pmapType = sliceHwStatus.pmapType.get( emapType, None )
         if pmapType is not None and len( pmapType.pmap ) > 0:
            # Static CoPP is not divergent so can break the loop 
            # if pmapType is present in any of the slices.
            mapTypePolicyPresent = True
            break
   if not mapTypePolicyPresent:
      mode.addWarning( "Warning: No %s policy maps configured" % mapType )
      return

   try:
      Tac.waitFor( countersUpdated, description="counter update", maxDelay=0.1,
                   sleep=True, timeout=30.0 )
   except Tac.Timeout:
      mode.addWarning( "Warning: displaying stale counters" )

def showPolicyMapCounters( mode, args ):
   pmapName = args.get( 'PMAP', 'copp-system-policy' ) 
   mapType = args.get( 'copp' ) or args.get( 'control-plane' ) or \
             args.get( 'pdp', 'qos' )
   direction = args.get( 'input' )
   detail = 'detail' in args
   mapType = 'control-plane' if mapType == 'copp' else mapType
   if 'counters' in args or 'interface' in args:
      return PolicyMapMode.showPolicyMap( mode, args )
   policyMapAllModel = PolicyMapAllModel()
   if qosHwStatus is None:
      return policyMapAllModel
   emapType = mapTypeToEnum( mapType )
   
   hwStatus = None
   if mapType == 'control-plane' and \
      not qosAclHwStatus.coppDynamicClassSupported :
      hwStatus = qosSliceHwStatus
   else:
      hwStatus = qosAclSliceHwStatus
   pmapConfigTypePtr = qosAclConfig.pmapType.get( emapType )
   if pmapConfigTypePtr is None:
      return policyMapAllModel
   pmapConfigPtr = pmapConfigTypePtr.pmap.get( pmapName )
   if pmapConfigPtr is None:
      return policyMapAllModel

   for hwStatus_ in hwStatus.itervalues():
      # Static CoPP is not divergent so can break the loop 
      # if pmapType is present in any of the slices.
      pmapHwStatusTypePtr = hwStatus_.pmapType.get( emapType )
      if pmapHwStatusTypePtr != None:
         break
      else:
         return policyMapAllModel

   if direction == None:
      directions = [ tacDirection.input, tacDirection.output ]
   else:
      directions = [ direction ]

   if mapType == 'control-plane':
      directions = [ tacDirection.input ]

   waitForCountersUpdate( mode, mapType )

   for direction in directions:
      return showPolicyMapCountersHelper( mode, emapType, pmapName, direction,
                                          detail, policyMapAllModel, hwStatus )

def delServicePolicyNoLock( mapType, pmapName, direction, intfName,
                            profile=False ):
   if profile:
      inputConfig = qosInputProfileConfig
   else:
      inputConfig = qosInputConfig

   if pmapName is None:
      # if pmapName is not specified, traverse and find attached pmap
      for key, spCfg in inputConfig.servicePolicyConfig.iteritems():
         if intfName in spCfg.intfIds and \
               key.direction == direction and  \
               key.type == mapType:
            pmapName = key.pmapName
            break
      if pmapName is None:
         # do not return any warning/error, otherwise tests need to
         # maintain states for cleanup
         return None
   else:
      # if pmapName specified, match attached pmap with given pmapName
      key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
      if key not in inputConfig.servicePolicyConfig:
         errMsg = pmapName + " not attached to interface " + intfName
         return errMsg
      else:
         spCfg = inputConfig.servicePolicyConfig[ key ]
         if intfName not in spCfg.intfIds:
            errMsg = pmapName + " not attached to interface " + intfName
            return errMsg
   # all validations done, now delete
   if len( spCfg.intfIds ) > 1:
      del spCfg.intfIds[ intfName ]
   else:
      del spCfg.intfIds[ intfName ]
      del inputConfig.servicePolicyConfig[ key ]
   return None

def delServicePolicy( mode, mapType, pmapName, direction, intfName, profile=False ):
   errMsg = delServicePolicyNoLock( mapType, pmapName, direction, intfName,
                                    profile=profile )
   if errMsg is not None and mode is not None:
      mode.addError( errMsg )

def addServicePolicy( mode, mapType, pmapName, direction, intfName, profile=False ):
   # Check if this interface belongs to any service-policy, if so we need
   # delete the interface from the old service policy, before we add to the
   # new service-policy
   oldKey = None
   waitForHwStatus = False
   if profile:
      inputConfig = qosInputProfileConfig
   else:
      inputConfig = qosInputConfig
   for key, spCfg in inputConfig.servicePolicyConfig.items():
      if intfName in spCfg.intfIds:
         if key.direction == direction and key.type == mapType:
            oldKey = key
            # Do not delete the service policy if its the same AND
            # hwPrgmStatus == Success
            if key.pmapName == pmapName:
               for aclSliceHwStatus in qosAclSliceHwStatus.itervalues():
                  pmapHwStatus = getPMapHwStatus( aclSliceHwStatus, pmapName, 
                                                  mapType, direction )
                  if pmapHwStatus and \
                     pmapHwStatus.prgmStatus == \
                     tacPMapHwPrgmStatus.hwPrgmStatusSuccess:
                     finalSpCfg = qosConfig.servicePolicyConfig.get( key )
                     if finalSpCfg and intfName in finalSpCfg.intfIds:
                        return

            if key in inputConfig.servicePolicyConfig:
               delServicePolicy( mode, key.type, key.pmapName, key.direction,
                                 intfName, profile=profile )
            waitForHwStatus = True
            break # interface can have only one service policy

   # Add this interface to new service policy
   key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
   if key in inputConfig.servicePolicyConfig:
      spCfg = inputConfig.servicePolicyConfig[ key ]
   else:
      spCfg = inputConfig.servicePolicyConfig.newMember( key )
   if intfName not in spCfg.intfIds:
      currTime = getCurrentTimeForConfigUpdate()
      spCfg.intfIds[ intfName ] = True

      # check if pmap exists in config
      pmapTypeConfigPtr = cliQosAclConfig.pmapType.get( key.type )
      if ( ( pmapTypeConfigPtr and pmapTypeConfigPtr.pmap.get( key.pmapName ) ) or
           ( mapType == pdpMapType and pmapName == tacPMapNm.defaultPdpPmapName and
             getDefaultPdpPmapCfg() ) ):
         # wait for all slices to return success status
         rc, errMsg = waitForPMapHwPrgmStatus( mode, key.pmapName, key.type,
                                               key.direction, currTime,
                                               updateStr="service-policy",
                                               profile=profile, intf=intfName )
         waitForHwStatus = False
         if not rc:
            if errMsg == CLI_TIMEOUT:
               if mode:
                  mode.addWarning( QosLib.qosTimeoutWarning() )
            else:
               # rollback
               t0( "Reverting service-policy" )
               delServicePolicyNoLock( key.type, key.pmapName, key.direction, 
                                       intfName, profile=profile )
               if oldKey:
                  t0( "Applying back old service-policy to interface" )
                  if oldKey in inputConfig.servicePolicyConfig:
                     oldSpCfg = inputConfig.servicePolicyConfig[ oldKey ]
                  else:
                     oldSpCfg = inputConfig.servicePolicyConfig.newMember( oldKey )

                  oldSpCfg.intfIds[ intfName ] = True
               if mode:
                  mode.addError( "Error: Cannot apply service-policy to " \
                                 "%s (%s)" % ( intfName, errMsg ) )
               if profile:
                  return ROLLBACK_PROFILE

   if waitForHwStatus and not mode.session.inConfigSession():
      currTime = getCurrentTimeForConfigUpdate()
      description = "existing service-policy to be detached"
      cliBlockingToApplyConfigChange( mode, currTime, description, policyMap=True )

def pmapHasDropPrecedenceAction( pmapName ):
   pmap = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap.get( pmapName )
   action = tacActionType.actionSetDropPrecedence
   yellowAction = QosLib.validDpConfigValues()[ 0 ]
   if pmap:
      policyActionList = []
      for cmapName in pmap.classAction.keys():
         policyActionList.append( pmap.classAction[ cmapName ].policyAction )
      policyActionList.append( pmap.classActionDefault.policyAction )
      for policyAction in policyActionList:
         if ( action in policyAction ) and (
               policyAction[ action ].value == yellowAction ):
            # Yellow action is not supported along with ecnAllowNonect
            return True
   return False

def appliedUnsupportedOutSPAction( pmapName ):
   actionsAndFlag = [
      [ tacActionType.actionSetCos, qosHwStatus.cosActionInOutSpSupported,
        "set cos" ],
      [ tacActionType.actionSetDscp, qosHwStatus.dscpActionInOutSpSupported,
        "set dscp" ],
      [ tacActionType.actionSetTc, qosHwStatus.tcActionInOutSpSupported,
        "set traffic-class" ],
      [ tacActionType.actionSetDropPrecedence, qosHwStatus.dpActionInOutSpSupported,
        "set drop-precedence" ],
   ]
   unsupportedAction = []
   pmap = cliQosAclConfig.pmapType[ QosLib.qosMapType ].pmap.get( pmapName )
   if pmap:
      policyActionList = []
      for cmapName in pmap.classAction.keys():
         policyActionList += pmap.classAction[ cmapName ].policyAction
      policyActionList += pmap.classActionDefault.policyAction
      for action, flag, name in actionsAndFlag:
         if not flag and ( action in policyActionList ):
            unsupportedAction.append( name )
   return unsupportedAction


def setServicePolicy( mode, noOrDefaultKw=None, mapType='qos', pmapName=None,
                      direction=None, intfName=None, profile=False ):
   if not intfName:
      intfName = mode.intf.name
   mapType = mapTypeToEnum( mapType )
   direction = directionToEnum( direction )
   if noOrDefaultKw:
      delServicePolicy( mode, mapType, pmapName, direction, intfName,
                        profile=profile )
      if not mode.session.inConfigSession():
         currTime = getCurrentTimeForConfigUpdate()
         description = "service-policy to be detached"
         cliBlockingToApplyConfigChange( mode, currTime, description,  
                                         policyMap=True )
   else:
      if direction == tacDirection.output:
         unSupportedActions = appliedUnsupportedOutSPAction( pmapName )
         error = "Following action(s) are not supported on output service policies:"
         if unSupportedActions != []:
            for actions in unSupportedActions:
               error += "\n" + actions
            mode.addError( error )
            if profile:
               return ROLLBACK_PROFILE
            return

      if isLagPort( intfName ) and isPolicerInPmap( cliQosAclConfig, pmapName ) and \
            not qosAclHwStatus.policerOnLagSupported:
         mode.addError( "Policing on Lag not supported on this platform" )
         if profile:
            return ROLLBACK_PROFILE
         return
      elif qosAclHwStatus.vlanMatchInClassMapSupported and \
            isSvi( intfName ) and isL2ParamsMatchInPmap( cliQosAclConfig, pmapName,
                                                         checkFor=[ 'vlan' ] ):
         errorMsg = "VLAN match in class-maps is not supported in QoS policy-maps"
         errorMsg += " applied to VLAN interfaces."
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return
      elif isinstance( mode, IntfCli.IntfConfigMode ) and \
           isinstance( mode.intf, VxlanCli.VxlanIntf ) and \
           qosAclHwStatus.vxlanPolicyQosSupported and \
           not isPmapSupportedForVxlan( cliQosAclConfig, pmapName ):
         errorMsg = "Only VLAN match is supported in QoS policy-map(s) applied to"
         errorMsg += " a VxLAN interface"
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return
      elif qosInputConfig.ecnAllowNonEct and pmapHasDropPrecedenceAction(
            pmapName ) and not qosHwStatus.allowNonEctWithDropPrecedence:
         errorMsg = "Policy-map action drop-precedence and allow "
         errorMsg += "non-ect cannot be configured together."
         mode.addError( errorMsg )
         if profile:
            return ROLLBACK_PROFILE
         return

      rollBackProfile = addServicePolicy( mode, mapType, pmapName, direction,
                                          intfName, profile=profile )
      if profile:
         return rollBackProfile

def qosProfileSetServicePolicy( mode, args ):
   noOrDefaultKw = CliCommand.isNoOrDefaultCmd( args )
   mapType = 'qos'
   pmapName = args.get( 'POLICY' )
   direction = args.get( 'input' ) or args.get( 'output' )
   mapType = mapTypeToEnum( mapType )
   direction = directionToEnum( direction )
   profile = mode.qosProfileModeContext.currentEntry_
   if noOrDefaultKw:
      if profile.servicePolicyConfig:
         key = profile.servicePolicyConfig.key
         if pmapName is None:
            if key.direction == direction and key.type == mapType:
               profile.servicePolicyConfig = None
         elif pmapName == key.pmapName:
            profile.servicePolicyConfig = None
   else:
      key = Tac.newInstance( "Qos::ServicePolicyKey", mapType, direction, pmapName )
      profile.servicePolicyConfig = ( key, )
 
nodeServicePolicy = CliCommand.guardedKeyword( 'service-policy', 
      helpdesc='Service policy configuration', guard=guardIntfModePolicy )
matcherType = CliMatcher.KeywordMatcher( 'type',
      helpdesc='Specify type' )
nodeMapTypeQos = CliCommand.guardedKeyword( 'qos', 
      helpdesc='Qos type', guard=guardPMapQos )
nodeInput = CliCommand.guardedKeyword( 'input',
      helpdesc='Apply the policy map to ingress packets',
      guard=guardIntfModePolicyQosIn )
nodeOutput = CliCommand.guardedKeyword( 'output',
      helpdesc='Apply the policy map to egress packets',
      guard=guardIntfModePolicyQosOut )
matcherPMapNameTypeQos = CliMatcher.DynamicNameMatcher( getPMapNameQos, 
      "Policy Map Name", 
      pattern=r'(?!counters$|interface$|summary$)[A-Za-z0-9_:{}\[\]-]+' )
#------------------------------------------------------------------------
# [no] service-policy [ type qos ] { input | output } [<policy-name>]
#------------------------------------------------------------------------
class ModeletServicePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy [ type qos ] ( input | output ) POLICY'
   noOrDefaultSyntax = 'service-policy [ type qos ] ( input | output ) [ POLICY ]'
   data = {
      'service-policy': nodeServicePolicy,
      'type': matcherType,
      'qos': nodeMapTypeQos,
      'input': nodeInput,
      'output': nodeOutput,
      'POLICY': matcherPMapNameTypeQos,
   }

   @staticmethod
   def handler( mode, args ):
      direction = args.get( 'input' ) or args.get( 'output' )
      setServicePolicy( mode, mapType='qos', 
            pmapName=args[ 'POLICY' ], direction=direction )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      direction = args.get( 'input' ) or args.get( 'output' )
      setServicePolicy( mode, noOrDefaultKw=True, mapType='qos',
            pmapName=args.get( 'POLICY' ), direction=direction )

QosModeletServicePolicy.addCommandClass( ModeletServicePolicyCmd )

#------------------------------------------------------------------------
# Service Policy commands in QosProfile mode
#------------------------------------------------------------------------
class QosProfileServicePolicyCmd( CliCommand.CliCommandClass ):
   syntax = 'service-policy [ type qos ] ( input | output ) POLICY'
   noOrDefaultSyntax = 'service-policy [ type qos ] ( input | output ) [ POLICY ]'
   data = {
      'service-policy': nodeServicePolicy,
      'type': matcherType,
      'qos': nodeMapTypeQos,
      'input': nodeInput,
      'output': nodeOutput,
      'POLICY': matcherPMapNameTypeQos,
   }
   
   handler = qosProfileSetServicePolicy
   noOrDefaultHandler = handler
   
QosProfileMode.addCommandClass( QosProfileServicePolicyCmd )

#------------------------------------------------------------------------------------
# qos scheduling mode
#------------------------------------------------------------------------------------

class QosSchedulingMode( QosSchedulingModeBase, BasicCli.ConfigModeBase ):
   name = 'QoS scheduling'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session ):
      QosSchedulingModeBase.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

def goToQosSchedulingMode( mode, args ):
   childMode = mode.childMode( QosSchedulingMode )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------------------
# qos scheduling interface mode
#------------------------------------------------------------------------------------

class QosSchedulingIntfMode( QosSchedulingIntfModeBase, BasicCli.ConfigModeBase ):
   name = 'QoS interface scheduling'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session, intfId ):
      self.intfId_ = intfId
      self.intfName_ = intfId.stringValue
      param = self.intfName_
      QosSchedulingIntfModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def instanceRuleKey( self ):
      return self.intfName_

   def abort( self ):
      self.session_.gotoParentMode()

   def intfId( self ):
      return self.intfId_

def goToQosSchedulingIntfMode( mode, args ):
   cliIntf = args[ 'INTFNAME' ]
   if cliIntf.config():
      intfId = cliIntf.config().getRawAttribute( 'intfId' )
   else:
      intfId = Tac.Value( 'Arnet::IntfId', cliIntf.name )
   qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if intfId not in qosSchedulerIntfConfig:
      qosSchedulerIntfConfig.newMember( intfId )

   childMode = mode.childMode( QosSchedulingIntfMode,
                               intfId=intfId )
   mode.session_.gotoChildMode( childMode )

def deleteQosSchedulingIntf( mode, args ):
   cliIntf = args[ 'INTFNAME' ]
   qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if cliIntf.config():
      intfId = cliIntf.config().intfId
   else:
      intfId = Tac.Value( 'Arnet::IntfId', cliIntf.name )
   if intfId in qosSchedulerIntfConfig:
      for groupName, groupConfig in qosSchedulerIntfConfig[
            intfId ].group.iteritems():
         cleanQosSchedulerGroupConfig( intfId, groupName,
                                       set( groupConfig.members ) )
      del qosSchedulerIntfConfig[ intfId ]

ethOrLagMatcher = IntfMatcher()
ethOrLagMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
ethOrLagMatcher |= LagIntfCli.EthLagIntf.matcher

class QosSchedulingIntfConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'interface INTFNAME'
   noOrDefaultSyntax = syntax
   data = {
      'interface': 'Configure qos scheduling parameters for interface',
      'INTFNAME': ethOrLagMatcher,
   }
   handler = goToQosSchedulingIntfMode
   noOrDefaultHandler = deleteQosSchedulingIntf

QosSchedulingMode.addCommandClass( QosSchedulingIntfConfigCmd )

#------------------------------------------------------------------------------------
# qos scheduling group mode
#------------------------------------------------------------------------------------

class QosSchedulingIntfGroupMode( QosSchedulingIntfGroupModeBase,
                                  BasicCli.ConfigModeBase ):
   name = 'QoS scheduling group'
   modeParseTree = CliParser.ModeParseTree()
   def __init__( self, parent, session, context ):
      self.qosSchedulingIntfGroupModeContext = context
      self.intfId_ = context.intfId()
      self.intfName_ = context.intfName()
      self.groupName_ = context.groupName()
      param = ( self.intfName_, self.groupName_ )
      QosSchedulingIntfGroupModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.qosSchedulingIntfGroupModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.qosSchedulingIntfGroupModeContext is None:
         t0( 'QosSchedulingIntfGroupMode.commitContext has no context' )
         return

      context = self.qosSchedulingIntfGroupModeContext
      self.qosSchedulingIntfGroupModeContext = None
      context.commit()

def identicalQosSchedulerGroupMemberLine( newLine, currentLine ):
   if ( not newLine or not currentLine or
        len( newLine.members ) != len( currentLine.members ) ):
      return False
   for m in newLine.members:
      if newLine.members[ m ] != currentLine.members[ m ]:
         return False
   return True

def identicalQosSchedulingGroup( group, currentEntry ):
   if not group or not currentEntry or group.policyName != currentEntry.policyName:
      return False
   if len( group.members ) != len( currentEntry.members ):
      return False
   for intfId in currentEntry.members:
      if group.members[ intfId ] != currentEntry.members[ intfId ]:
         return False
   return True

class QosSchedulingIntfGroupModeContext( object ):
   def __init__( self, mode, intfId, groupName ):
      self.mode = mode
      self.intf_ = None
      self.intfId_ = intfId
      self.intfName_ = intfId.stringValue
      self.group_ = None
      self.groupName_ = groupName
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.previousEntry_ = None
      qosSchedulingIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
      if intfId not in qosSchedulingIntfConfig:
         return
      qosSchedulingGroupConfig = qosSchedulingIntfConfig[ intfId ].group
      if groupName in qosSchedulingGroupConfig:
         self.group_ = qosSchedulingGroupConfig[ groupName ]
         prevIntfGroup = Tac.newInstance( 'Qos::QosSchedulerGroup',
                                       self.groupName_ )
         copyQosSchedulerGroup( prevIntfGroup, self.group_ )
         self.previousEntry_ = prevIntfGroup

   def copyEditEntry( self ):
      newIntfGroup = Tac.newInstance( 'Qos::QosSchedulerGroup', self.groupName_ )
      copyQosSchedulerGroup( newIntfGroup, self.group_ )
      self.currentEntry_ = newIntfGroup
      return newIntfGroup

   def newEditEntry( self ):
      newIntfGroup = Tac.newInstance( 'Qos::QosSchedulerGroup', self.groupName_ )
      self.currentEntry_ = newIntfGroup

   def groupName( self ):
      return self.groupName_

   def intfName( self ):
      return self.intfName_

   def intfId( self ):
      return self.intfId_

   def currentEntry( self ):
      return self.currentEntry_

   def qosSchedulingIntfGroup( self ):
      return self.group_

   def commit( self ):
      if identicalQosSchedulingGroup( self.group_, self.currentEntry_ ):
         return

      intfConfig = qosInputConfig.intfConfig
      qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
      if self.intfId_ not in qosSchedulerIntfConfig:
         qosSchedulerIntfConfig.newMember( self.intfId_ )
      qosSchedulingGroupConfig = qosSchedulerIntfConfig[ self.intfId_ ].group
      if self.group_ is None:
         groupConfig = qosSchedulingGroupConfig.newMember( self.groupName_ )
      else:
         groupConfig = qosSchedulingGroupConfig.get( self.groupName_ )
      oldMembers = set( groupConfig.members.iterkeys() )
      groupConfig.members.clear()
      groupConfig.lines.clear()
      policyChanged = False
      if groupConfig.policyName != self.currentEntry_.policyName:
         groupConfig.policyName = self.currentEntry_.policyName
         policyChanged = True
      commitLineIndex = 0

      # Copy all lines from currentEntry_ to groupConfig, collapsing any empty
      # lines to avoid wrapping around the index space.
      for pendingLineIndex in self.currentEntry_.lines:
         currentLine = self.currentEntry_.lines[ pendingLineIndex ]
         if not currentLine:
            break
         if currentLine.count == 0:
            continue
         newLine = groupConfig.lines.newMember( commitLineIndex )
         newLine.count = currentLine.count
         for memberIndex in currentLine.members:
            subIntfId = currentLine.members.getRawEntry( memberIndex )
            t8( 'commit: groupConfig[', commitLineIndex, '] = currentEntry_[',
                pendingLineIndex, '] memberIndex:', memberIndex,
                'subIntfId:', subIntfId )
            if not subIntfId:
               break
            newLine.members[ memberIndex ] = subIntfId
            groupConfig.members[ subIntfId ] = commitLineIndex
            oldMembers.discard( subIntfId )
         commitLineIndex += 1
      groupConfig.nextLineIndex = commitLineIndex
      self.group_ = groupConfig

      for subIntfId in oldMembers:
         t8( 'Deleting', subIntfId, 'from group config for', self.groupName_ )
         if subIntfId in intfConfig:
            setIntfConfig( subIntfId, cfgSchedulerGroupName="" )
      for subIntfId in groupConfig.members:
         t0( 'create intf config for', subIntfId )
         setIntfConfig( subIntfId, cfgSchedulerGroupName=self.groupName_ )
      if self.intfId_ not in intfConfig:
         intfConfig.newMember( self.intfId_ )
      parentIntfConfig = intfConfig.get( self.intfId_ )
      if self.groupName_ not in parentIntfConfig.schedulerGroupConfig:
         parentIntfConfig.schedulerGroupConfig.newMember(
            self.groupName_, invalidShapeRate, invalidGuaranteedBw )
      if policyChanged:
         policyName = groupConfig.policyName
         policyConfig = ( qosSchedulerConfig.policy.get( policyName )
                          if policyName else None )
         if policyConfig:
            cfgShapeRate = policyConfig.shapeRate
            cfgGuaranteedBw = policyConfig.guaranteedBw
         else:
            # If this policy is deleted or does not exist, apply defaults
            cfgShapeRate = Tac.Value( 'Qos::ShapeRate' )
            cfgGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
         groupIntfConfig = parentIntfConfig.schedulerGroupConfig.get(
            self.groupName_ )
         if groupIntfConfig.shapeRate != cfgShapeRate:
            groupIntfConfig.shapeRate = cfgShapeRate
         if groupIntfConfig.guaranteedBw != cfgGuaranteedBw:
            groupIntfConfig.guaranteedBw = cfgGuaranteedBw

def copyQosSchedulerGroup( newGroup, oldGroup ):
   newGroup.policyName = oldGroup.policyName
   for m in oldGroup.lines:
      oldLine = oldGroup.lines[ m ]
      newLine = newGroup.lines.newMember( m )
      newLine.count = oldLine.count
      for m2 in range( oldLine.count ):
         intfId = oldLine.members[ m2 ]
         assert oldGroup.members[ intfId ] == m
         newGroup.members[ intfId ] = oldGroup.members[ intfId ]
         t8( 'copyQosSchedulerGroup: copy %s at %d,%d' % ( intfId, m, m2 ) )
         newLine.members[ m2 ] = intfId
   newGroup.nextLineIndex = oldGroup.nextLineIndex

def cleanQosSchedulerGroupConfig( intfName, groupName, members ):
   if intfName in qosInputConfig.intfConfig:
      intfConfig = qosInputConfig.intfConfig[ intfName ]
      del intfConfig.schedulerGroupConfig[ groupName ]
   for m in members:
      if m in qosInputConfig.intfConfig:
         qosInputConfig.intfConfig[ m ].schedulerGroupName = ''

class QosSchedulingIntfGroupConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'scheduling group GROUPNAME'
   noOrDefaultSyntax = syntax
   data = {
      'scheduling': nodeScheduling,
      'group': 'Configure qos scheduling group for interface',
      'GROUPNAME': matcherSchedulingGroupName,
   }

   @staticmethod
   def handler( mode, args ):
      groupName = args[ 'GROUPNAME' ]
      intfId = mode.intfId()
      qosSchedulingIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
      if intfId in qosSchedulingIntfConfig:
         qosSchedulingGroupConfig = qosSchedulingIntfConfig[ intfId ].group
      else:
         qosSchedulingGroupConfig = None
      context = QosSchedulingIntfGroupModeContext( mode, intfId, groupName )
      if qosSchedulingGroupConfig and groupName in qosSchedulingGroupConfig:
         context.copyEditEntry()
      else:
         context.newEditEntry()

      mode.qosSchedulingIntfGroupContext = context
      childMode = mode.childMode( QosSchedulingIntfGroupMode, context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      groupName = args[ 'GROUPNAME' ]
      intfId = mode.intfId()
      qosSchedulingIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
      if intfId in qosSchedulingIntfConfig:
         qosSchedulingGroupConfig = qosSchedulingIntfConfig[ intfId ].group
         if groupName in qosSchedulingGroupConfig:
            members = set( qosSchedulingGroupConfig[ groupName ].members )
            del qosSchedulingGroupConfig[ groupName ]
            cleanQosSchedulerGroupConfig( intfId, groupName, members )

QosSchedulingIntfMode.addCommandClass( QosSchedulingIntfGroupConfigCmd )

class QosSchedulingPolicyNameCmd( CliCommand.CliCommandClass ):
   syntax = 'policy POLICYNAME'
   noOrDefaultSyntax = syntax
   data = {
      'policy': 'Configure qos scheduling policy',
      'POLICYNAME': matcherSchedulingPolicyName,
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICYNAME' ]
      context = mode.qosSchedulingIntfGroupModeContext
      group = context.currentEntry_
      group.policyName = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = mode.qosSchedulingIntfGroupModeContext
      group = context.currentEntry_
      group.policyName = ""

QosSchedulingIntfGroupMode.addCommandClass( QosSchedulingPolicyNameCmd )

class SubIntfWrapperMatcher( CliMatcher.WrapperMatcher ):
   def __init__( self, matcher ):
      CliMatcher.WrapperMatcher.__init__( self, matcher )

   def valueFunction( self, context ):
      f = self.matcher_.valueFunction( context )
      def g( mode, name ):
         intf = f( mode, name )
         if not intf.isSubIntf():
            t0( 'not subinterface' )
            raise CliParser.InvalidInputError()
         parentStatus = intf.parentStatus()
         if parentStatus:
            parentIntfId = intf.parentStatus().getRawAttribute( 'intfId' )
         else:
            parentIntfId = Tac.Value( 'Arnet::IntfId', intf.name.split( '.' )[ 0 ] )
         if parentIntfId != mode.intfId_:
            t0( 'parent name not match', parentIntfId, '!=', mode.intfId_ )
            raise CliParser.InvalidInputError()
         return intf
      return g

ethOrLagSubIntfMatcher = IntfMatcher()
ethOrLagSubIntfMatcher |= SubIntfCli.subMatcher
ethOrLagSubIntfMatcher |= LagIntfCli.subMatcher
matcherSubIntfName = SubIntfWrapperMatcher( ethOrLagSubIntfMatcher )

class QosSchedulingGroupMemberCmd( CliCommand.CliCommandClass ):
   syntax = 'members { INTFNAME }'
   noOrDefaultSyntax = 'members [ { INTFNAME } ]'
   data = {
      'members': 'Configure qos scheduling group members',
      'INTFNAME': matcherSubIntfName,
   }

   @staticmethod
   def handler( mode, args ):
      context = mode.qosSchedulingIntfGroupModeContext
      group = context.currentEntry_
      intfs = args[ 'INTFNAME' ]
      lineIndex = group.nextLineIndex
      group.nextLineIndex += 1
      memberLine = group.lines.newMember( lineIndex )
      count = 0
      for k, intf in enumerate( intfs ):
         if intf.config():
            intfId = intf.config().getRawAttribute( 'intfId' )
         else:
            intfId = Tac.Value( 'Arnet::IntfId', intf.name )
         if intfId not in group.members:
            prevGroup = getSubIntfGroupMembership( intfId.stringValue )
            if prevGroup:
               mode.addErrorAndStop(
                  'Subinterface %s is already assigned to group %s' % (
                     intfId, prevGroup ) )
            group.members[ intfId ] = lineIndex
            memberLine.members[ k ] = intfId
            count += 1
      memberLine.count = count
      if count == 0:
         del group.lines[ lineIndex ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      context = mode.qosSchedulingIntfGroupModeContext
      group = context.currentEntry_
      if 'INTFNAME' not in args:
         group.members.clear()
         group.lines.clear()
         return
      intfs = args[ 'INTFNAME' ]
      for intf in intfs:
         if intf.config():
            intfId = intf.config().getRawAttribute( 'intfId' )
         else:
            intfId = Tac.Value( 'Arnet::IntfId', intf.name )
         if intfId not in context.currentEntry_.members:
            continue
         m = context.currentEntry_.members[ intfId ]
         del context.currentEntry_.members[ intfId ]
         t8( 'Deleting', intfId, 'which should be on line', m,
             'from group', context.groupName() )
         line = context.currentEntry_.lines[ m ]
         found = False
         for m2 in xrange( line.count ):
            if found:
               line.members[ m2 - 1 ] = line.members[ m2 ]
               line.members[ m2 ] = Tac.Value( 'Arnet::IntfId' )
               continue
            entry = line.members.getRawEntry( m2 )
            t0( 'Entry', m2, 'is', entry )
            if entry == intfId:
               line.members[ m2 ] = Tac.Value( 'Arnet::IntfId' )
               found = True
               line.count -= 1
         assert found

QosSchedulingIntfGroupMode.addCommandClass( QosSchedulingGroupMemberCmd )


#------------------------------------------------------------------------------------
# qos scheduling policy mode
#------------------------------------------------------------------------------------

class QosSchedulingPolicyMode( QosSchedulingPolicyModeBase,
                               BasicCli.ConfigModeBase ):
   name = 'QoS scheduling policy'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, context ):
      self.qosSchedulingPolicyModeContext = context
      self.policyName_ = context.policyName()
      param = ( self.policyName_ )
      QosSchedulingPolicyModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      #pylint: disable-msg=W0201
      self.currentEntry_ = None
      self.qosSchedulingPolicyModeContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.qosSchedulingPolicyModeContext is None:
         t0( 'QosSchedulingPolicyMode.commitContext has no context' )
         return

      context = self.qosSchedulingPolicyModeContext
      self.qosSchedulingPolicyModeContext = None
      context.commit()

def identicalQosSchedulingPolicy( policy, currentEntry ):
   if not policy or not currentEntry:
      return False
   return ( policy.guaranteedBw == currentEntry.guaranteedBw and
            policy.shapeRate == currentEntry.shapeRate )

class QosSchedulingPolicyModeContext( object ):
   def __init__( self, mode, policyName ):
      self.mode = mode
      self.policy_ = None
      self.policyName_ = policyName
      self.editedEntries_ = {}
      self.currentEntry_ = None
      self.previousEntry_ = None
      qosSchedulingPolicyConfig = qosSchedulerConfig.policy
      if policyName in qosSchedulingPolicyConfig:
         self.policy_ = qosSchedulingPolicyConfig[ policyName ]
         self.policyName_ = policyName
         prevPolicy = Tac.newInstance( 'Qos::QosSchedulerPolicy',
                                       self.policyName_ )
         copyQosSchedulerPolicy( prevPolicy, self.policy_ )
         self.previousEntry_ = prevPolicy

   def copyEditEntry( self ):
      newPolicy = Tac.newInstance( 'Qos::QosSchedulerPolicy', self.policyName_ )
      copyQosSchedulerPolicy( newPolicy, self.policy_ )
      self.currentEntry_ = newPolicy
      return newPolicy

   def newEditEntry( self ):
      newPolicy = Tac.newInstance( 'Qos::QosSchedulerPolicy', self.policyName_ )
      self.currentEntry_ = newPolicy

   def policyName( self ):
      return self.policyName_

   def currentEntry( self ):
      return self.currentEntry_

   def qosSchedulingPolicy( self ):
      return self.policy_

   def commit( self ):
      qosSchedulingPolicyConfig = qosSchedulerConfig.policy
      if identicalQosSchedulingPolicy( self.policy_, self.currentEntry_ ):
         return
      if self.policy_ is None:
         policyConfig = qosSchedulingPolicyConfig.newMember( self.policyName_ )
      else:
         policyConfig = qosSchedulingPolicyConfig.get( self.policyName_ )
      policyConfig.guaranteedBw = self.currentEntry_.guaranteedBw
      policyConfig.shapeRate = self.currentEntry_.shapeRate
      self.policy_ = policyConfig

      setSchedulingPolicy( self.policyName_ )

def copyQosSchedulerPolicy( newPolicy, oldPolicy ):
   newPolicy.guaranteedBw = oldPolicy.guaranteedBw
   newPolicy.shapeRate = oldPolicy.shapeRate

def setSchedulingPolicy( policyName ):
   policyConfig = qosSchedulerConfig.policy.get( policyName )
   if policyConfig:
      cfgShapeRate = policyConfig.shapeRate
      cfgGuaranteedBw = policyConfig.guaranteedBw
   else:
      # If this policy is deleted or does not exist, apply defaults
      cfgShapeRate = Tac.Value( 'Qos::ShapeRate' )
      cfgGuaranteedBw = Tac.Value( 'Qos::GuaranteedBw' )
   for intfId in qosSchedulerConfig.qosSchedulerIntfConfig:
      schIntf = qosSchedulerConfig.qosSchedulerIntfConfig[ intfId ]
      for schGroupName in schIntf.group:
         schGroupIn = schIntf.group[ schGroupName ]
         if schGroupIn.policyName == policyName:
            if intfId not in qosInputConfig.intfConfig:
               qosInputConfig.intfConfig.newMember( intfId )
            if schGroupIn.name not in qosInputConfig.intfConfig[
                  intfId ].schedulerGroupConfig:
               schGroupConfig = qosInputConfig.intfConfig[
                  intfId ].schedulerGroupConfig.newMember( schGroupName,
                                                           cfgShapeRate,
                                                           cfgGuaranteedBw )
            else:
               schGroupConfig = qosInputConfig.intfConfig[
                  intfId ].schedulerGroupConfig.get( schGroupName )
               if schGroupConfig.shapeRate != cfgShapeRate:
                  schGroupConfig.shapeRate = cfgShapeRate
               if schGroupConfig.guaranteedBw != cfgGuaranteedBw:
                  schGroupConfig.guaranteedBw = cfgGuaranteedBw

def gotoQosSchedulingPolicyMode( mode, args ):
   policyName = args[ 'POLICYNAME' ]
   qosSchedulingPolicyConfig = qosSchedulerConfig.policy
   context = QosSchedulingPolicyModeContext( mode, policyName )
   if policyName in qosSchedulingPolicyConfig:
      context.copyEditEntry()
   else:
      context.newEditEntry()

   mode.qosSchedulingPolicyModeContext = context
   childMode = mode.childMode( QosSchedulingPolicyMode, context=context )
   mode.session_.gotoChildMode( childMode )

def deleteQosSchedulingPolicy( mode, args ):
   policyName = args[ 'POLICYNAME' ]
   qosSchedulingPolicyConfig = qosSchedulerConfig.policy
   if policyName in qosSchedulingPolicyConfig:
      del qosSchedulingPolicyConfig[ policyName ]
      setSchedulingPolicy( policyName )

def getSubIntfGroupMembership( subIntf ):
   if '.' not in subIntf:
      return ''

   parentIntf = subIntf.split( '.' )[ 0 ]
   qosSchedulerIntfConfig = qosSchedulerConfig.qosSchedulerIntfConfig
   if parentIntf not in qosSchedulerIntfConfig:
      return ''
   parentIntfConfig = qosSchedulerIntfConfig[ parentIntf ]
   for groupName in parentIntfConfig.group:
      groupConfig = parentIntfConfig.group[ groupName ]
      if subIntf in groupConfig.members:
         return groupName
   return ''

class QosSchedulingPolicyConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'scheduling policy POLICYNAME'
   noOrDefaultSyntax = syntax
   data = {
      'scheduling': nodeScheduling,
      'policy': 'Configure qos scheduling policy',
      'POLICYNAME': matcherSchedulingPolicyName,
   }
   handler = gotoQosSchedulingPolicyMode
   noOrDefaultHandler = deleteQosSchedulingPolicy

QosSchedulingMode.addCommandClass( QosSchedulingPolicyConfigCmd )

QosSchedulingPolicyMode.addCommandClass( BandwidthGuaranteedCmd )

#------------------------------------------------------------------------------------
# The "[ default|no ] service config verification qos" command,
# in "global config" mode.
#------------------------------------------------------------------------------------

def guardHwConfigVerification( mode, token ):
   if qosHwStatus.hwStatusReportingSupported:
      return None
   return CliParser.guardNotThisPlatform

class ServiceVerifyQos( CliCommand.CliCommandClass ):
   syntax = "service configuration verification qos"
   noSyntax = syntax
   defaultSyntax = syntax
   data = {
      "service" : FileCli.serviceKw,
      "configuration" : FileCli.configKwAfterService,
      "verification" : FileCli.verificationKwAfterService,
      "qos" : CliCommand.guardedKeyword( "qos", 'Verify QoS config',
                                         guardHwConfigVerification )
      }
   @staticmethod
   def handler( mode, args ):
      qosInputConfig.hwConfigVerificationEnabled = True

   @staticmethod
   def noHandler( mode, args ):
      qosInputConfig.hwConfigVerificationEnabled = False

   defaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( ServiceVerifyQos )

#---------------------------------------------------------------------
# hardware access-list qos resource sharing vlan in
# hardware access-list qos resource sharing routedport subinterface
#---------------------------------------------------------------------

def guardRoutedPortSubIntfAclSharing( mode, token ):
   if qosAclHwStatus.routedPortSubIntfQosAclSharingSupported:
      return None
   return CliParser.guardNotThisPlatform

def qosAclSharingKnobSupported( mode, token ):
   if not guardRoutedPortSubIntfAclSharing( mode, token ) or \
      not guardSharingVlan( mode, token ):
      return None
   return CliParser.guardNotThisPlatform

def enableQosAclSharing( mode, args ):
   cliQosAclConfig.sviPolicyQosSharing = True

def disableQosAclSharing( mode, args ):
   cliQosAclConfig.sviPolicyQosSharing = False

routedPortSubIntfQosAclSharingCanPromptForAbortHook = CliExtensions.CliHook()

def convertRoutedPortSubIntfAclSharing( mode ):
   cliQosAclConfig.usingQosRoutedPortSubIntfAclSharingCli = True 

def gotoRoutedPortSubIntfAclSharing( mode, args ):
   cliQosAclConfig.usingQosRoutedPortSubIntfAclSharingCli = 'qos' in args
   shareWithSubIntf = not CliCommand.isNoOrDefaultCmd( args )
   aclType = 'qos'

   if aclType == 'qos':
      if cliQosAclConfig.routedPortSubIntfQosAclSharing == shareWithSubIntf:
         return
      if mode.session.commandConfirmation():
         for hook in \
             routedPortSubIntfQosAclSharingCanPromptForAbortHook.extensions():
            if hook( mode ):
               return

      cliQosAclConfig.routedPortSubIntfQosAclSharing = shareWithSubIntf

#-------------------------------------------------------------------------
# [ no | default ] policy-map type quality-of-service policer drop counter
#-------------------------------------------------------------------------
def guardCmapDropCounter( mode, token ):
   if qosAclHwStatus.redCounterSupported:
      return None
   return CliParser.guardNotThisPlatform

def enableQosAclCmapDropCounter( mode, args ):
   cliQosAclConfig.enablePolicerDropCounter = True

def disableQosAclCmapDropCounter( mode, args ):
   cliQosAclConfig.enablePolicerDropCounter = False

#--------------------------------------------------------------------------------
# [ no | default ] qos subinterface tx-queue count <4|8>
#--------------------------------------------------------------------------------
def guardNumTxqsPerSubIntf( mode, token ):
   if not subIntfHwStatus.subIntfNumTxqsConfigSupported:
      return CliParser.guardNotThisPlatform
   return None

def handleNumTxqsPerSubIntf( mode, args ):   
   numTxqs = int( args.get( 'NUMTXQS', '4' ) )
   if qosGlobalConfig.numTxqsPerSubIntf != numTxqs:
      qosGlobalConfig.numTxqsPerSubIntf = numTxqs
      mode.addWarning( "Change will take effect only after switch reboot." )

#-------------------------------------------------------------------------------
#Register convertRoutedPortSubIntfAclSharing via "config convert new-syntax"

CliPlugin.ConfigConvert.registerConfigConvertCallback(
                                             convertRoutedPortSubIntfAclSharing )
#-------------------------------------------------------------------------------

def startQosAclConfigMergeSm():
   global qosAclConfig, qosAclConfigMergeSm
   qosAclConfig = Tac.newInstance( "Qos::QosAclConfig" )
   qosAclConfigMergeSm = Tac.newInstance( "Qos::Agent::QosAclConfigMergeSm",
                                          qosAclConfigDir, cliQosAclConfigReadOnly,
                                          defaultPdpPmapCfgReadOnly, qosAclConfig )


#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
@Plugins.plugin( provides=( "QosCli", ) )
def Plugin( entityManager ):
   global qosInputConfig, qosInputProfileConfig, cliQosAclConfig, qosConfig, \
          qosHwConfig, cliCounterConfig, qosStatus, qosHwStatus, qosTmMapping, \
          pfcStatus, aclConfig, lagInputConfig, entMan, qosSliceHwStatus, \
          qosAclHwStatus, qosAclSliceHwStatus, fabricIntfConfigDir, \
          profileConfigDir, qosHwCounter, intfConfigEthPhySliceDir, \
          cliQosAclConfigReadOnly, defaultPdpPmapCfgReadOnly, qosAclConfigDir, \
          aclStatus, hwEpochStatus, subIntfHwStatus, qosSchedulerConfig, \
          qosGlobalConfig, ecnCounterTable, ecnSnapshotTable

   entMan = entityManager
   profileConfigDir = ConfigMount.mount( entityManager, "qos/profile", 
                                         "Qos::QosProfileConfigDir", "w" )
   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosSchedulerConfig = ConfigMount.mount( entityManager, "qos/scheduler/config",
                                           "Qos::QosSchedulerConfig", "w" )
   qosGlobalConfig = ConfigMount.mount( entityManager, "qos/global/config",
                                        "Qos::GlobalConfig", "w" )
   qosInputProfileConfig = ConfigMount.mount( 
         entityManager, "qos/input/config/qosProfile",
         "Qos::Input::Config", "w" )
   cliQosAclConfig = ConfigMount.mount( entityManager, "qos/acl/input/cli",
                                        "Qos::Input::AclConfig", "w" )
   aclConfig = LazyMount.mount( entityManager, "acl/config/cli",
                                "Acl::Input::Config", "r" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosHwConfig = LazyMount.mount( entityManager, "qos/hardware/config", 
                                  "Qos::HwConfig", "w" )
   cliCounterConfig = LazyMount.mount( entityManager, "qos/cliCounterConfig", 
                                       "Qos::CounterConfig", "w" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosHwStatus = LazyMount.mount( entityManager, "qos/hardware/status/global",
                                  "Qos::HwStatus", "r" )
   qosTmMapping= LazyMount.mount( entityManager, "hardware/tm/mapping",
                                  "Qos::TmMapping", "r" )
   qosSliceHwStatusDirPath = \
         "cell/" + str( Cell.cellId() ) + "/qos/hardware/status/slice"
   qosSliceHwStatus = LazyMount.mount( entityManager, qosSliceHwStatusDirPath,
                                       "Tac::Dir", "ri" )
   qosAclHwStatus = LazyMount.mount( entityManager, 
         "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   qosAclSliceHwStatus = LazyMount.mount( entityManager, 
         "qos/hardware/acl/status/slice", "Tac::Dir", "ri" )
   qosHwCounter = LazyMount.mount( entityManager,
                                   "qos/hardware/counter", 
                                   "Qos::HwCounter", "ri" )
   pfcStatus = LazyMount.mount( entityManager, "dcb/pfc/status", "Pfc::Status", "r" )
   lagInputConfig = LazyMount.mount( entityManager, "lag/input/config/cli",
                                     "Lag::Input::Config", "r" )

   fabricIntfConfigDir = ConfigMount.mount( entityManager,
                                            "qos/fabric/config",
                                            "Interface::FabricIntfConfigDir",
                                            "w" )
   intfConfigEthPhySliceDir = ConfigMount.mount( entityManager,
         "interface/config/eth/phy/slice/", "Tac::Dir", "w" )
   IntfCli.Intf.registerDependentClass( QosIntfJanitor )
   aclStatus = LazyMount.mount( entityManager, "acl/status/all",
                                "Acl::Status", "r" )
   hwEpochStatus = LazyMount.mount( entityManager, "hwEpoch/status",
                                    "HwEpoch::Status", "r" )
   subIntfHwStatus = LazyMount.mount( entityManager, "interface/hardware/capability",
                                      "Interface::Hardware::Capability", "r" )

   # Set up a single instance of QosAclConfigMergeSm. It needs to mount the Cli
   # config and the input dirs for config from EosSdk. The aggregated view of
   # Qos::QosAclConfig is instantiated in startQosAclConfigMergeSm().
   # Note: cliQosAclConfigReadOnly is used to directly access the cli config entity
   # for use by QosAclConfigMergeSm. This can only be used as an input to that merge
   # SM. cliQosAclConfig, on the other hand is a ConfigMount Proxy that the plugin
   # can write to.
   mg = entityManager.mountGroup()

   cliQosAclConfigReadOnly = mg.mount( "qos/acl/input/cli",
                                       "Qos::Input::AclConfig", "wi" )
   defaultPdpPmapCfgReadOnly = mg.mount( "qos/acl/input/defaultPdpPmapCfg",
                                         "Qos::Input::AclConfig", "ri" )
   qosAclConfigDir = mg.mount( "qos/acl/input/eosSdkConfig", "Tac::Dir", "ri" )
   mg.close( startQosAclConfigMergeSm )

   if toggleEcnCountersSmashEnabled():
      # Mount ECN Queue counter current and snapshot smash tables.
      shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
      readerInfo = Smash.mountInfo( 'reader' )
      mountPath = 'flexCounters/counterTable/Ecn/%u' % ( FapId.allFapsId )
      ecnCounterTable = shmemEm.doMount( mountPath, "Qos::EcnCounterTable",
                                         readerInfo )
      mountPath = 'flexCounters/snapshotTable/Ecn/%u' % ( FapId.allFapsId )
      ecnSnapshotTable = shmemEm.doMount( mountPath, "Qos::EcnCounterTable",
                                          readerInfo )

#---------------------------------------------------------------------------------
# Register commands to show tech-support
#---------------------------------------------------------------------------------
def _showTechCmds():
   cmds = []
   if qosHwStatus.hwInitialized:
      cmds.extend( [
         'show qos maps'
      ] )
   return cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
   '2019-12-04 16:19:43', _showTechCmds )
