# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Arnet import IpGenAddr
from ArnetModel import IpGenericAddress
from CliModel import Bool
from CliModel import Dict
from CliModel import Enum
from CliModel import Int
from CliModel import List
from CliModel import Model
from CliModel import Str
from CliModel import Submodel
from collections import namedtuple
from IntfModels import Interface
from TableOutput import Format, createTable
from TunnelTypeLib import tunnelTypeStrDict
import CliPlugin.IntfCli
import Tac
import Toggles.ArBgpToggleLib
from PseudowireLib import (
   flBoth,
   flDisabled,
   flReceive,
   flTransmit,
   flValCapabilityMsg,
   flValUsedMsg,
   FlowLabelMode,
   pwPingCcValToEnum,
   pwPingCvValToEnum,
   pwPingCcMsg,
   pwPingCvMsg,
   undefStr,
   strVccv,
)

#---------------------------------------
# Sample Output
#---------------------------------------
#
# show patch panel
#
# Patch      Connector                                Status
# ---------- ---------------------------------------- ------------
# foo1       1: Ethernet1 802.1Q VLAN 1000            Up
#            2: Ethernet2 802.1Q VLAN 1500    
# foo2       1: Ethernet3                             Up
#            2: Ethernet4    
# bar        Lo: Ethernet5 802.1Q VLAN 2000           Unprogrammed
#            Re: LDP neighbor 255.255.255.0 PW ID 300
#
# show patch panel detail
#
# PW Fault Legend:
#    ET-IN - Ethernet receive fault
#    ET-OUT - Ethernet transmit fault
#    TUN-IN - Tunnel receive fault
#    TUN-OUT - Tunnel transmit fault
#    NF - Pseudowire not forwarding (other reason)
#
# Patch: ldpPatch, Status: Up
#    Connector Lo: Ethernet1 802.1Q VLAN 100
#       Status: up
#    Connector Re: LDP neighbor 2.2.2.2 PW ID 1020
#       Status: up
#       Local MPLS label: 100096, Group ID: 0
#          MTU: 9200, 802.1Q VLAN request sent: -
#          Flow label capability: none
#       Neighbor MPLS label: 200000, Group ID: 0
#          MTU: 9213, 802.1Q VLAN request received: 1213
#          Interface description: Hear my voice
#          Flow label capability: none
#          PW fault: ET-IN NF
#       PW type: 4 (tagged), Control word: -
#       Flow label used: no
#       Tunnel type: LDP, Tunnel index: 2
# Patch: loPatch, Status: Up
#    Connector Lo1: Ethernet2
#       Status: up
#    Connector Lo2: Port-Channel2
#       Status: up
#
# show patch panel forwarding
#
# Legend:
#    * - Not forwarding
#    Type - Pseudowire type: 4 (tagged)
#                            5 (raw)
#    CW - Control word used
#    FL - Flow label used
#
#  In/Out                                 Source   Type   Flags    Patch
# -------------------------------------- -------- ------ -------- ---------
#  Et1 802.1Q VLAN 10                                              lo-lo2
#    Po1 802.1Q VLAN 200
#  Et2                                                             lo-lo1
#    Po2
#  Et3.1                                                           lo-Rmt3
#    3.3.3.3, Label 899999, LDP Tun 197   EVPN        5   *
#  Po1 802.1Q VLAN 200                                             lo-lo2
#    Et1 802.1Q VLAN 10
#  Po1999 802.1Q VLAN 300                                          lo-Ldp2
#    2.2.2.2, Label 900000, LDP Tun 198   LDP         4   CW, FL
#  Po2                                                             lo-lo1
#    Et2
#  Po2000                                                          lo-Ldp1
#    1.1.1.1, Label 900000, LDP Tun 198   LDP         5
#
#  PW Label   Out                      Source   Type   Flags    Patch
# ---------- ------------------------ -------- ------ -------- ---------
#  100001     Po2000                   LDP         5            lo-Ldp1
#  100002     Po1999 802.1Q VLAN 300   LDP         4   CW, FL   lo-Ldp2
#  100003     Et3.1                    EVPN        5            lo-Rmt3

#---------------------------------------
# Useful constants and helper methods
#---------------------------------------
ConnectorIndexType = Tac.Type( "Pseudowire::ConnectorIndex" )
ConnectorStatusType = Tac.Type( "Pseudowire::PseudowireConnectorStatus" )
PwType = Tac.Type( "Pseudowire::PseudowireType" )
MplsLabel = Tac.Type( "Arnet::MplsLabel" )

supportedConnectorTypes = [ "port",
                            "tagged",
                            "ldp",
                            "bgp" ]

pwSources = ( "EVPN", "LDP" )

EnumItem = namedtuple( "EnumItem", [ "enum", "help" ] )

getShortName = CliPlugin.IntfCli.Intf.getShortname

CST = ConnectorStatusType
# The values in this dictionary are returned over JSON/shown on Cli.
# Case and formatting matters!
connStatusMsg = {
   CST.unknown: "Unknown",
   CST.adminDown: "Admin down",
   CST.cliConflict: "CLI conflict",
   CST.intfMode: "Interface mode",
   CST.cliIncomplete: "CLI incomplete",
   CST.waitingForVlanId: "Awaiting internal VLAN",
   CST.cannotAllocateVlanId: "No internal VLAN",
   CST.hasTaggedIntf: "Conflicting tagged interface",
   CST.noLocalMapping: "No label",
   CST.noLocalEnd: "Interface unavailable",
   CST.noRemoteEnd: "No remote",
   CST.noPeerLabel: "No peer label",
   CST.pwTypeMismatch: "Type mismatch",
   CST.controlWordMismatch: "Control word mismatch",
   CST.mtuMismatch: "MTU mismatch",
   CST.pwNotForwardingRemote: "Remote fail",
   CST.noTunnel: "No tunnel",
   CST.noTunnelRib: "No tunnel RIB",
   CST.down: "Down",
   CST.encapConnInfoMissing: "Invalid local connector encapsulation info",
   CST.encapInvalidConnConfig: "Invalid local connector encapsulation info",
   CST.encapHwFecUnprogrammed: "Unprogrammed Hardware FEC",
   CST.encapHwProgrammingFailed: "Encapsulation programming failed",
   CST.connectorUnprogrammed: "Unprogrammed local connector",
   CST.decapConnInfoMissing: "Invalid local connector decapsulation info",
   CST.decapInvalidConnConfig: "Invalid local connector decapsulation info",
   CST.decapHwProgrammingFailed: "Decapsulation programming failed",
   CST.unprogrammed: "Unprogrammed",
   CST.up: "Up",
   CST.disabledByPw: "Disabled by Pseudowire",
}

# The enum description here is returned over CAPI/shown on Cli.
# Case and formatting matters!
ldpErrFlags = {
   "notForwarding": EnumItem( "NF", "Pseudowire not forwarding (other reason)" ),
   "etInFault": EnumItem( "ET-IN", "Ethernet receive fault" ),
   "etOutFault": EnumItem( "ET-OUT", "Ethernet transmit fault" ),
   "tunInFault": EnumItem( "TUN-IN", "Tunnel receive fault" ),
   "tunOutFault": EnumItem( "TUN-OUT", "Tunnel transmit fault" )
   }

# First line will have "   Type - " as prefix
typeStr = \
       """Pseudowire type: 4 (tagged)
                           5 (raw)"""

fibOptionEnumItems = (
   EnumItem( "*", "Not forwarding" ),
   EnumItem( "Type", typeStr ),
   EnumItem( "CW", "Control word used" ),
   EnumItem( "FL", "Flow label used" ),
)

fibOptionWoFLEnumItems = (
   EnumItem( "Type", typeStr ),
   EnumItem( "CW", "Control word used" ),
)

def enumHelpStr( enumItems ):
   '''Generate a portion of the help string containing enum name and 
   help string from enumItems. The returned string is indented by
   3 white spaces. Here is a sample output.
   "   enum1 - help1\n
       enum2 - help2\n
       enum3 - help3"
   '''
   return "\n".join( ( "   %s - %s" % ( ei.enum, ei.help ) for ei in enumItems ) )

#---------------------------------------
# Render helper methods
#---------------------------------------

def pwTypeToInt( pwType ):
   pwTypeInt = 0
   if pwType == PwType.pwType4:
      pwTypeInt = 4
   elif pwType == PwType.pwType5:
      pwTypeInt = 5
   return pwTypeInt

def flowLabelModeToType( flowLabelMode ):
   if flowLabelMode == FlowLabelMode.transmit:
      return flTransmit
   elif flowLabelMode == FlowLabelMode.receive:
      return flReceive
   elif flowLabelMode == FlowLabelMode.both:
      return flBoth
   else:
      return flDisabled

def printt( msg, indent=0 ):
   '''Print the given string 'msg' with given indentation'''
   print ' ' * 3 * indent + msg

def strBool( b ):
   return 'Y' if b else 'N'

ldpErrKeys = [ "etInFault", "etOutFault", "tunInFault", "tunOutFault",
               "notForwarding" ]

def ldpStatusLegendList():
   '''Returns the legend of Ldp status flags.'''
   legend = [ "%s - %s" % ( ldpErrFlags[k].enum, ldpErrFlags[k].help )
              for k in ldpErrKeys ]  
   return legend

#---------------------------------------
# Cli models
#---------------------------------------
class LdpPseudowireErrorFlags( Model ):
   '''Model for Ldp pseudowire error flags/status codes. One or more of these 
   error codes may be present at the same time.'''
   etInFault = Bool( optional=True, help=ldpErrFlags[ "etInFault" ].help )
   etOutFault = Bool( optional=True, help=ldpErrFlags[ "etOutFault" ].help )
   tunInFault = Bool( optional=True, help=ldpErrFlags[ "tunInFault" ].help )
   tunOutFault = Bool( optional=True, help=ldpErrFlags[ "tunOutFault" ].help )
   notForwarding = Bool( optional=True, help=ldpErrFlags[ "notForwarding" ].help )

   def toStr( self ):
      '''Returns a string of error codes based on the flags enabled in this
      model'''
      flagList = [ ldpErrFlags[ k ].enum for k in ldpErrKeys if getattr( self, k ) ]

      return " ".join( flagList )

class LdpPseudowirePeerConfig( Model ):
   '''Model encapsulating high level information for an Ldp pseudowire peer,
   either remote or local.'''
   label = Int( optional=True, help="Pseudowire Label" )
   groupId = Int( optional=True, help="Group ID" )
   mtu = Int( optional=True, help="Maximum transmission unit" )
   dot1qTagRequested = Int( optional=True, help="802.1Q VLAN requested" )
   intfDescription = Str( optional=True, help="Interface description" )
   _isLocal = Bool( help="Local or peer's LDP config" )
   vccvCvTypes = Int( optional=True, help="Supported VCCV CV types" )
   vccvCcTypes = Int( optional=True, help="Supported VCCV CC types" )
   flowLabelCapability = Enum( optional=True, values=FlowLabelMode.attributes,
                               help="Flow label config" )

   # pylint: disable=arguments-differ
   def render( self, indent=0 ):
      label = ( "%d" % self.label ) if self.label is not None and \
                                       self.label != MplsLabel.null else undefStr
      groupId = ( "0x%x" % self.groupId ) if self.groupId is not None else undefStr
      mtu = ( "%d" % self.mtu ) if self.mtu is not None else undefStr

      tagReq = "%d" % self.dot1qTagRequested if self.dot1qTagRequested is not None \
               else undefStr

      if self._isLocal:
         printt( "Local MPLS label: %s, Group ID: %s" % ( label, groupId ),
                 indent=indent )
         if ( mtu, tagReq ) != ( undefStr, undefStr ):
            printt( "MTU: %s, 802.1Q VLAN request sent: %s" % ( mtu, tagReq ),
                    indent=indent+1 )
            if self.intfDescription:
               printt( "Interface description: %s" % self.intfDescription, 
                       indent=indent+1 )
      else:
         printt( "Neighbor MPLS label: %s, Group ID: %s" % ( label, groupId ),
                 indent=indent )
         if ( mtu, tagReq ) != ( undefStr, undefStr ):
            printt( "MTU: %s, 802.1Q VLAN request received: %s" % ( mtu, tagReq ),
                    indent=indent+1 )
            if self.intfDescription:
               printt( "Interface description: %s" % self.intfDescription,
                       indent=indent+1 )

      if self.flowLabelCapability is not None:
         flowLabelValue = flowLabelModeToType( self.flowLabelCapability )
         printt( "Flow label capability: %s" %
                  flValCapabilityMsg[ flowLabelValue.value ], indent=indent + 1 )

      vccvCvTypeInt = int( self.vccvCvTypes ) if self.vccvCvTypes else 0
      vccvCcTypeInt = int( self.vccvCcTypes ) if self.vccvCcTypes else 0
      vccvCvTypeStr = strVccv( vccvCvTypeInt, pwPingCvValToEnum, pwPingCvMsg )
      vccvCcTypeStr = strVccv( vccvCcTypeInt, pwPingCcValToEnum, pwPingCcMsg )
      if self.vccvCvTypes is not None or self.vccvCcTypes is not None:
         printt( "Supported VCCV CV types: %s" % vccvCvTypeStr,
                 indent=indent + 1 )
         printt( "Supported VCCV CC types: %s" % vccvCcTypeStr,
                 indent=indent + 1 )

class LdpPseudowireConnector( Model ):
   '''Model for an LDP pseudowire connector.'''
   name = Str( optional=True, help="LDP pseudowire name" )
   neighbor = IpGenericAddress( optional=True, help="LDP neighbor" )
   pseudowireId = Int( optional=True, help="Pseudowire ID" )
   pseudowireType = Enum( values=PwType.attributes, help="Pseudowire type" )
   controlWord = Bool( help="Control word used" )
   flowLabelUsed = Enum( optional=True, values=FlowLabelMode.attributes,
                         help="Flow label used" )
   localConfig = Submodel( valueType=LdpPseudowirePeerConfig, optional=True,
                           help="Local LDP pseudowire config" )
   neighborConfig = Submodel( valueType=LdpPseudowirePeerConfig, optional=True,
                              help="LDP neighbor's pseudowire config" )
   neighborForwarding = Bool( help="Pseudowire forwarding traffic" )
   neighborFaultFlags = Submodel( valueType=LdpPseudowireErrorFlags,
                                  optional=True,
                                  help="Neighbor pseudowire error flags" )
   tunnelType = Enum( values=tunnelTypeStrDict.values(), optional=True,
                      help="Tunnel type" )
   tunnelIndexes = List( valueType=int, optional=True,
                      help="List of tunnel indexes ending at this tunnel endpoint" )

   def getRenderString( self ):
      nbr = self.neighbor or undefStr
      if self.pseudowireId is None:
         pwId = undefStr
      else:
         pwId = "%d" % self.pseudowireId
      return 'LDP neighbor %s PW ID %s' % ( nbr, pwId )
   
   # pylint: disable=arguments-differ
   def render( self, connName, connectorStatus, indent=0 ):
      assert connName is not None
      assert connectorStatus is not None
      nbr = self.neighbor or undefStr
      if self.pseudowireId is None:
         pwId = undefStr
      else:
         pwId = "%d" % self.pseudowireId

      pwType = undefStr
      if self.pseudowireType == PwType.pwType4:
         pwType = "4 (tagged)"
      elif self.pseudowireType == PwType.pwType5:
         pwType = "5 (raw)"

      cw = strBool( self.controlWord )
      
      printt( "Connector %s: LDP neighbor %s PW ID %s" %
              ( connName or undefStr, nbr, pwId ), indent=indent )
      
      printt( "Status: %s" % connStatusMsg[ connectorStatus ], indent=indent+1 )

      if self.localConfig:
         self.localConfig.render( indent=indent+1 )

      if self.neighborConfig:
         self.neighborConfig.render( indent=indent+1 )

         if not self.neighborForwarding and self.neighborFaultFlags:
            # show error flags
            errorFlags = self.neighborFaultFlags.toStr()
            printt( "PW fault: " + errorFlags, indent=indent+2 )

      printt( "PW type: %s, Control word: %s" % ( pwType, cw ), indent=indent+1 )
      flowLabelValue = flowLabelModeToType( self.flowLabelUsed )
      printt( "Flow label used: %s" % flValUsedMsg[ flowLabelValue.value ],
              indent=indent + 1 )

      if self.tunnelType:
         idxStr = ", ".join( [ "%d" % idx for idx in self.tunnelIndexes ] )
         printt( "Tunnel type: %s, Tunnel index: %s" % ( self.tunnelType, idxStr ),
                 indent=indent+1 )

class BgpPseudowireLocalConfig( Model ):
   """"Model encapsulating high level information for a BGP pseudowire local
   endpoint."""
   label = Int( optional=True, help="Pseudowire Label" )
   mtu = Int( optional=True, help="Maximum transmission unit" )
   controlWord = Bool( optional=True, help="Control word used" )

   def render( self, indent=0 ): # pylint: disable=arguments-differ
      label = ( ( "%d" % self.label ) if self.label is not None and
                                         self.label != MplsLabel.null else undefStr )
      printt( "Local MPLS label: %s" % label, indent=indent )

      if ( self.mtu, self.controlWord ) != ( None, None ):
         printt( "MTU: %s, Control word: %s" %
                 ( self.mtu, strBool( self.controlWord ) ), indent=indent+1 )

class BgpPseudowireNeighborConfig( Model ):
   """Model encapsulating high level information for a BGP pseudowire remote
   endpoint."""
   mtu = Int( optional=True, help="Maximum transmission unit" )
   controlWord = Bool( optional=True, help="Control word used" )
   label = Int( optional=True, help="Pseudowire label" )
   tunnelType = Enum( values=tunnelTypeStrDict.values(), optional=True,
                      help="Tunnel type" )
   tunnelIndex = Int( optional=True, help="Tunnel ending at this tunnel endpoint" )

   def render( self, neighbor, indent=0 ): # pylint: disable=arguments-differ
      label = ( ( "%d" % self.label ) if self.label is not None and
                                         self.label != MplsLabel.null else undefStr )
      printt( "Neighbor %s, MPLS label: %s" % ( neighbor, label ), indent=indent )

      if self.tunnelType:
         printt( "Tunnel type: %s, Tunnel index: %s" %
                 ( self.tunnelType, self.tunnelIndex ), indent=indent+1 )

      if ( self.mtu, self.controlWord ) != ( None, None ):
         printt( "MTU: %s, Control word: %s" %
                 ( self.mtu, strBool( self.controlWord ) ), indent=indent+1 )

vpwsTypeEnumItems = (
      EnumItem( "port", "port-based" ),
      EnumItem( "vlan", "VLAN-based" ) )

class BgpPseudowireConnector( Model ):
   """Model for a BGP pseudowire connector."""
   vpwsName = Str( optional=True, help="BGP VPWS instance name" )
   pwName = Str( optional=True, help="BGP VPWS pseudowire name" )
   localConfig = Submodel( valueType=BgpPseudowireLocalConfig,
                           optional=True,
                           help="Local BGP pseudowire config" )
   neighborConfigs = Dict( keyType=IpGenericAddress,
                           valueType=BgpPseudowireNeighborConfig,
                           optional=True,
                           help="BGP neighbors' pseudowire configs" )
   vpwsType = Enum( values=( item.enum for item in vpwsTypeEnumItems ),
                    optional=True,
                    help="EVPN VPWS type." + enumHelpStr( vpwsTypeEnumItems ) )
   flowLabelUsed = Enum( optional=True, values=FlowLabelMode.attributes,
                         help="Flow label used" )

   def getRenderString( self ):
      vpwsName = self.vpwsName or undefStr
      pwName = self.pwName or undefStr
      return "BGP VPWS %s Pseudowire %s" % ( vpwsName, pwName )

   # pylint: disable=arguments-differ
   def render( self, connName, connectorStatus, indent=0 ):
      printt( "Connector %s: %s" % ( connName or undefStr, self.getRenderString() ),
              indent=indent )

      printt( "Status: %s" % connStatusMsg[ connectorStatus ], indent=indent+1 )

      if self.localConfig:
         self.localConfig.render( indent=indent+1 )

      if self.neighborConfigs:
         for neighborAddr in sorted( self.neighborConfigs.keys() ):
            self.neighborConfigs[ neighborAddr ].render( neighborAddr,
                                                         indent=indent+1 )

      vpwsType = undefStr
      if self.vpwsType == "port":
         vpwsType = "port-based"
      elif self.vpwsType == "vlan":
         vpwsType = "VLAN-based"

      printt( "EVPN VPWS type: %s" % vpwsType, indent=indent+1 )

      if Toggles.ArBgpToggleLib.toggleVpwsFlowLabelToggleEnabled():
         flowLabelValue = flowLabelModeToType( self.flowLabelUsed )
         printt( "Flow label used: %s" % flValUsedMsg[ flowLabelValue.value ],
                 indent=indent + 1 )

class TaggedConnector( Model ):
   '''Model for a local, tagged connector.'''
   interface = Interface( help="Local interface" )
   dot1qTag = Int( help="802.1Q Tag" )

   def getRenderString( self ):
      intf = str( self.interface ).strip( "'" )
      return '%s 802.1Q VLAN %d' % ( intf, self.dot1qTag )

   # pylint: disable-msg=arguments-differ
   def render( self, connName=None, connectorStatus=None, indent=0 ):
      assert connName is not None
      assert connectorStatus is not None
      # TODO self.interface returns 'Ethernet1' with the quotes.
      # (Pdb) p Tac.Value( "Arnet::IntfId", "Ethernet1" ).stringValue
      # 'Ethernet1'
      intf = str( self.interface ).strip( "'" )
      printt( "Connector %s: %s 802.1Q VLAN %d" %
              ( connName or undefStr, intf, self.dot1qTag ), indent=indent )
      printt( "Status: %s" % connStatusMsg[ connectorStatus ], indent=indent+1 )

class PortConnector( Model ):
   '''Model for a local, port connector.'''
   interface = Interface( help="Local interface" )

   def getRenderString( self ):
      return str( self.interface ).strip( "'" )

   # pylint: disable-msg=arguments-differ
   def render( self, connName=None, connectorStatus=None, indent=0 ):
      assert connName is not None
      assert connectorStatus is not None
      # TODO self.interface returns 'Ethernet1' with the quotes.
      # (Pdb) p Tac.Value( "Arnet::IntfId", "Ethernet1" ).stringValue
      # 'Ethernet1'
      intf = str( self.interface ).strip( "'" )
      printt( "Connector %s: %s" %
              ( connName or undefStr, intf ), indent=indent )
      printt( "Status: %s" % connStatusMsg[ connectorStatus ], indent=indent+1 )

connectorTypeHelp = '''Connector type.
   tagged - Local interface delimited by an 802.1Q tag value
   port - Local interface not delimited by 802.1Q tag values
   ldp - LDP pseudowire
   bgp - BGP pseudowire'''

class Connector( Model ):
   '''Model a generic connector for a patch. The connector itself may be of type
   "tagged", "port", "ldp" or "bgp". Accordingly, taggedConnectorInfo,
   portConnectorInfo, ldpPseudowireConnectorInfo or bgpPseudowireConnectorInfo
   is populated.'''
   connType = Enum( values=supportedConnectorTypes, help=connectorTypeHelp )
   status = Enum( values=ConnectorStatusType.attributes,
                  help="Connector status" )

   taggedConnectorInfo = Submodel( valueType=TaggedConnector, optional=True,
                                   help="Tagged connector information" ) 
   portConnectorInfo = Submodel( valueType=PortConnector, optional=True,
                                 help="Port connector information" ) 
   ldpPseudowireConnectorInfo = Submodel( valueType=LdpPseudowireConnector,
                                   optional=True,
                                   help="LDP pseudowire connector information" )
   bgpPseudowireConnectorInfo = Submodel( valueType=BgpPseudowireConnector,
                                   optional=True,
                                   help="BGP pseudowire connector information" )

   def getRenderString( self, connName ):
      pfx = "%s: " % ( connName or undefStr )
      if self.connType == "tagged":
         return pfx + self.taggedConnectorInfo.getRenderString()
      elif self.connType == "port":
         return pfx + self.portConnectorInfo.getRenderString()
      elif self.connType == "ldp":
         return pfx + self.ldpPseudowireConnectorInfo.getRenderString()
      elif self.connType == "bgp":
         return pfx + self.bgpPseudowireConnectorInfo.getRenderString()
      else:
         assert False, "Unhandled connType {}".format( self.connType )
         return None

   def render( self, connName, indent=0 ): # pylint: disable-msg=arguments-differ
      if self.connType == "tagged":
         self.taggedConnectorInfo.render( connName=connName,
                                          connectorStatus=self.status,
                                          indent=indent )
      elif self.connType == "port":
         self.portConnectorInfo.render( connName=connName,
                                        connectorStatus=self.status,
                                        indent=indent )
      elif self.connType == "ldp":
         self.ldpPseudowireConnectorInfo.render( connName=connName,
                                                 connectorStatus=self.status,
                                                 indent=indent )
      elif self.connType == "bgp":
         self.bgpPseudowireConnectorInfo.render( connName=connName,
                                                 connectorStatus=self.status,
                                                 indent=indent )
      else:
         assert False, "Unhandled connType {}".format( self.connType )

patchStatusEnums = [ CST.adminDown, CST.down, CST.cliConflict, 
                     CST.unprogrammed, CST.up ]
patchStatusEnumItems = (
      EnumItem( "Admin down", "Patch disabled" ),
      EnumItem( "Down", "Patch connectors not ready" ),
      EnumItem( "Cli conflict", "Patch configuration is invalid" ),
      EnumItem( "Unprogrammed", "Patch configured but not programmed in hardware" ),
      EnumItem( "Up", "Patch configured and installed" ) )
patchStatusHelp = "Patch status.\n" + enumHelpStr( patchStatusEnumItems )

class PatchSummary( Model ):
   """Model for the output of show patch summary"""
   patchCount = Int( help="Patch count" )
   bgpVpwsCount = Dict(
         keyType=str,
         valueType=int,
         help="BGP EVPN VPWS patch count by status" )
   ldpCount = Dict(
         keyType=str,
         valueType=int,
         help="LDP patch count by status" )
   localCount = Dict(
         keyType=str,
         valueType=int,
         help="Local patch count by status" )

   def yieldRenderLine( self ):
      yield "Patch Summary"

      for heading, protoCounts in [
            ( "EVPN VPWS", self.bgpVpwsCount ),
            ( "LDP", self.ldpCount ),
            ( "Local", self.localCount ),
            ]:

         yield ""
         yield heading

         for status in reversed( ConnectorStatusType.attributes ):
            if status not in protoCounts:
               continue
            yield "{}: {}".format( connStatusMsg[ status ],
                                   protoCounts[ status ] )

         yield "Total: {}".format( sum( protoCounts.itervalues() ) )

   def render( self ):
      for line in self.yieldRenderLine():
         print line

class Patch( Model ):
   '''Model a patch. A patch has a status and two connectors.
   The connectors can be local or remote.'''
   status = Enum( values=patchStatusEnums, help=patchStatusHelp )
   connectors = Dict( keyType=str,
                      valueType=Connector,
                      help="Maps connector names to their respective "
                           "configuration" )

class Patches( Model ):
   '''Model for show patch panel and show patch panel detail'''
   patches = Dict( keyType=str,
                   valueType=Patch,
                   help="Maps patch names to their respective configuration "
                        "and operational state" )

   _summary = Bool( help='Whether to show patch panel information in detail or ' 
                         'as a summary' )

   # This method is used for the 'show patch panel' command. Example output of the
   # command is given at the top of the file.
   def renderSummary( self ):
      formats = {}
      formats[ 'Patch' ] = Format( justify='left' )
      formats[ 'Connector' ] = Format( justify='left' )
      formats[ 'Status' ] = Format( justify='left' )
      for columnFormat in formats.values():
         columnFormat.noPadLeftIs( True )
         columnFormat.padLimitIs( True )

      patchTable = createTable( ( 'Patch', 'Connector', 'Status' ), tableWidth=132 )
      patchTable.formatColumns( formats[ 'Patch' ], formats[ 'Connector' ],
                                formats[ 'Status' ] )

      for name in sorted( self.patches ):
         patch = self.patches[ name ]
         connectors = [ ( connName, conn ) for ( connName, conn )
                        in patch.connectors.items() ]
         connectors = sorted( connectors, key=lambda x: x[0] )

         if not connectors:
            patchTable.newRow( name, '', connStatusMsg[ patch.status ] )
            continue

         firstConn = True
         for connName, conn in connectors:
            connString = conn.getRenderString( connName )
            if firstConn:
               patchTable.newRow( name, connString, connStatusMsg[ patch.status ] )
               firstConn = False
            else:
               patchTable.newRow( '', connString, '' )

      print patchTable.output()

   def renderLegend( self ):
      printt( "PW Fault Legend:" )
      for line in ldpStatusLegendList():
         printt( line, indent=1 )

      printt( "" )

   # This method is used for the 'show patch panel detail' command. Example output 
   # of the command is given at the top of the file.
   def renderDetail( self ):
      self.renderLegend()
      patchNames = sorted( self.patches.keys() )

      for name in patchNames:
         patch = self.patches.get( name )
         printt( "Patch: %s, Status: %s" % ( name, connStatusMsg[ patch.status ] ) )

         connectors = [ ( connName, conn ) for ( connName, conn )
                        in patch.connectors.items() ]
         connectors = sorted( connectors, key=lambda c: c[0] )

         for connName, conn in connectors:
            conn.render( connName, indent=1 )
      
   def render( self ):
      if self._summary:
         self.renderSummary()
      else:
         self.renderDetail()

class TaggedConnectorFwdingInfo( Model ):
   '''Model for a tagged connector for show patch panel forwarding'''
   interface = Interface( help="Local interface" )
   dot1qVlan = Int( help="802.1Q VLAN" )
   decapControlWord = Bool(
      optional=True,
      help="Control word used to decapsulate packets toward this AC" )

   def toStr( self ):
      intf = str( self.interface ).strip( "'" )
      return '%s 802.1Q VLAN %d' % \
             ( getShortName( intf ), self.dot1qVlan )

class PortConnectorFwdingInfo( Model ):
   '''Model for a port connector for show patch panel forwarding'''
   interface = Interface( help="Local interface" )
   decapControlWord = Bool(
      optional=True,
      help="Control word used to decapsulate packets toward this AC" )

   def toStr( self ):
      intf = str( self.interface ).strip( "'" )
      return getShortName( intf )

class LabelConnectorFwdingPeerInfo( Model ):
   '''Model for a single peer of a label connector'''
   tunnelType = Enum( values=tunnelTypeStrDict.values(), help="Tunnel type" )
   peerLabel = Int( help="Label received from the peer" )
   tunnelIndex = Int( help="Tunnel index corresponding to this tunnel endpoint" )
   encapControlWord = Bool(
      optional=True,
      help="Control word used to encapsulate packets toward this peer" )

class LabelConnectorFwdingInfo( Model ):
   '''Model for a label connector for show patch panel forwarding'''
   __revision__ = 2
   peers = Dict( keyType=IpGenericAddress, valueType=LabelConnectorFwdingPeerInfo,
                 help="List of peers for this connector" )

   def sortedPeers( self ):
      return sorted( self.peers.items(), key=lambda item: IpGenAddr( item[ 0 ] ) )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         if self.peers:
            _, peerInfo = self.sortedPeers()[ 0 ]
            dictRepr[ "tunnelType" ] = peerInfo.tunnelType
            dictRepr[ "tunnelIndexes" ] = [ peerInfo.tunnelIndex ]
            dictRepr[ "label" ] = peerInfo.peerLabel
         else:
            dictRepr[ "tunnelType" ] = "invalidTunnel"
            dictRepr[ "tunnelIndexes" ] = []
            dictRepr[ "label" ] = MplsLabel.null

      return dictRepr

   def toStr( self ):
      '''Return a list of tuples (tunStr, cw), one for each remote peer'''
      res = []
      if self.peers:
         fmt = "{}, Label {}, {} Tun {}"
         for peerAddr, peerInfo in self.sortedPeers():
            peerStr = fmt.format( peerAddr, peerInfo.peerLabel,
                                  peerInfo.tunnelType, peerInfo.tunnelIndex )
            res.append( ( peerStr, peerInfo.encapControlWord ) )
      return res

class ForwardingConnectorInfo( Model ):
   '''Model for the forwarding connector information, used for the in/out connector
   fields in the forwardingInfo models. Can either be a local(tagged/raw) connector,
   or a remote(MPLS label) connector'''
   _infoType = Enum( values=supportedConnectorTypes,
                     help='The connector type of the forwarding information entry' )
   taggedConnFwdingInfo = Submodel( valueType=TaggedConnectorFwdingInfo,
                                    help='Tagged connector forwarding information',
                                    optional=True )
   portConnFwdingInfo = Submodel( valueType=PortConnectorFwdingInfo,
                                  help='Port connector forwarding information',
                                  optional=True )
   labelConnFwdingInfo = Submodel( valueType=LabelConnectorFwdingInfo,
                                   help='Label connector forwarding information',
                                   optional=True )

   def controlWord( self ):
      '''Return control word (encap or decap) to be used when degrading model'''
      if self._infoType == 'tagged':
         return ( self.taggedConnFwdingInfo.decapControlWord
                  if self.taggedConnFwdingInfo else None )

      if self._infoType == 'port':
         return ( self.portConnFwdingInfo.decapControlWord
                  if self.portConnFwdingInfo else None )

      if ( self._infoType in ( 'bgp', 'ldp' ) and
           self.labelConnFwdingInfo and
           self.labelConnFwdingInfo.peers ):
         _, peerInfo = self.labelConnFwdingInfo.sortedPeers()[ 0 ]
         return peerInfo.encapControlWord

      return None

   def toStr( self ):
      if self._infoType == 'tagged':
         return self.taggedConnFwdingInfo.toStr()
      if self._infoType == 'port':
         return self.portConnFwdingInfo.toStr()
      if self._infoType in ( 'bgp', 'ldp' ):
         return self.labelConnFwdingInfo.toStr()
      return None

def flagsStr( status, cw, fl ):
   flags = [ "*" ] if status != ConnectorStatusType.up else []
   flags += [ 'CW' ] if cw else []
   flags += [ 'FL' ] if fl else []
   return ", ".join( flags )

def pad( s ):
   return " {} ".format( s )

def renderOutConnInTable( fwdInfoModel, table ):
   outConnStrs = fwdInfoModel.outConnInfo.toStr()
   if not isinstance( outConnStrs, list ):
      outConnStrs = [ outConnStrs ]
   for outConnStr in outConnStrs:
      if isinstance( outConnStr, tuple ):
         outConnStr, encapControlWord = outConnStr
      else:
         encapControlWord = None

      source = ( pad( fwdInfoModel.source ) if fwdInfoModel.source else "" )
      pwType = ( pad( pwTypeToInt( fwdInfoModel.pseudowireType ) )
                 if fwdInfoModel.pseudowireType is not None else "" )
      flags = flagsStr( fwdInfoModel.status, encapControlWord,
                        fwdInfoModel.flowLabel )
      # Patch name is printed on the "In" line
      table.newRow( pad( "  " + outConnStr ), source, pwType, pad( flags ), "" )

class TaggedForwardingInfo( Model ):
   ''' Model for 1 row of tagged forwarding information,
   shows the incoming/outgoing connectors, the pseudowire type, 802.1Q VLAN, and
   control word used flag for ldp connections, patch the connection belongs is for,
   and the status of the connection'''
   __revision__ = 2
   inConnInfo = Submodel( valueType=ForwardingConnectorInfo, help="In connector" )
   outConnInfo = Submodel( valueType=ForwardingConnectorInfo, help="Out connector" )
   source = Enum( optional=True, values=pwSources, help='Pseudowire source' )
   pseudowireType = Enum( optional=True,
                          values=PwType.attributes,
                          help="Pseudowire type" )
   flowLabel = Str( optional=True, help="Flow label used" )
   patch = Str( help="Patch name" )
   dot1qVlan = Int( optional=True, help="802.1Q VLAN" )
   status = Enum( values=ConnectorStatusType.attributes,
                  help="Connector status" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         controlWord = self.outConnInfo.controlWord()
         if controlWord is not None:
            dictRepr[ "controlWord" ] = controlWord

      return dictRepr

   def renderInTable( self, table ):
      inConnStr = self.inConnInfo.toStr()
      # Render "In" connector: source, type, flags are all empty on this line
      table.newRow( pad( inConnStr ), "", "", "", pad( self.patch ) )
      # Render all "Out" connectors
      renderOutConnInTable( self, table )

class PortForwardingInfo( Model ):
   ''' Model for 1 row of port forwarding information,
   shows the incoming/outgoing connectors, the pseudowire type, 802.1Q VLAN, and
   control word used flag for ldp connections, patch the connection belongs is for,
   and the status of the connection'''
   __revision__ = 2
   inConnInfo = Submodel( valueType=ForwardingConnectorInfo, help="In connector" )
   outConnInfo = Submodel( valueType=ForwardingConnectorInfo, help="Out connector" )
   source = Enum( optional=True, values=pwSources, help='Pseudowire source' )
   pseudowireType = Enum( optional=True,
                          values=PwType.attributes,
                          help="Pseudowire type" )
   flowLabel = Str( optional=True, help="Flow label used" )
   patch = Str( help="Patch name" )
   status = Enum( values=ConnectorStatusType.attributes,
                  help="Connector status" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         controlWord = self.outConnInfo.controlWord()
         if controlWord is not None:
            dictRepr[ "controlWord" ] = controlWord

      return dictRepr

   def renderInTable( self, table ):
      inConnStr = self.inConnInfo.toStr()
      # Render "In" connector: source, type, flags are all empty on this line
      table.newRow( pad( inConnStr ), "", "", "", pad( self.patch ) )
      # Render all "Out" connectors
      renderOutConnInTable( self, table )

class LabelForwardingInfo( Model ):
   '''Model for 1 row of Label forwarding information, shows the decap label,
      pseudowire type( 4 or 5 ), pseudowire source( EVPN or LDP ),
      outgoing connector, the patch the connector belongs to, 
      and the patch the connector belongs to '''   
   __revision__ = 2
   label = Int( help="Pseudowire decap label" )
   outConnInfo = Submodel( valueType=ForwardingConnectorInfo,
                           help="Out connector" )
   source = Enum( values=pwSources, help='Pseudowire source' )
   pseudowireType = Enum( values=PwType.attributes, help="Pseudowire type" )
   flowLabel = Str( optional=True, help="Flow label used" )
   patch = Str( help="Patch name" )
   status = Enum( values=ConnectorStatusType.attributes,
                  help="Connector status" )

   def degrade( self, dictRepr, revision ):
      if revision < 2:
         controlWord = self.outConnInfo.controlWord()
         # controlWord was not optional
         dictRepr[ "controlWord" ] = controlWord or False

      return dictRepr

   def renderInTable( self, table ):
      pwTypeStr = ( pad( pwTypeToInt( self.pseudowireType ) )
                    if self.pseudowireType is not None else "" )
      flags = flagsStr( self.status, self.outConnInfo.controlWord(), self.flowLabel )

      # outConnInfo is either port or tagged
      table.newRow( pad( self.label ),
                    pad( self.outConnInfo.toStr() ),
                    pad( self.source ),
                    pwTypeStr,
                    pad( flags ),
                    pad( self.patch ) )

fwdingOptionHelp = "Legend:\n" + enumHelpStr( fibOptionEnumItems )
fwdingOptionWoFlHelp = "Legend:\n" + enumHelpStr( fibOptionWoFLEnumItems )

class TaggedForwardingInfoTable( Model ):
   ''' Model for the tagged forwarding information table'''
   __revision__ = 2
   taggedForwardingInfos = Dict( keyType=str, valueType=TaggedForwardingInfo,
                                 help='A mapping of tagged connectors to '
                                      'connector forwarding information entries' )

class PortForwardingInfoTable( Model ):
   ''' Model for the port forwarding information table'''
   __revision__ = 2
   portForwardingInfos = Dict( keyType=str, valueType=PortForwardingInfo,
                               help='A mapping of port connectors to '
                                    'connector forwarding information entries' )

class LabelForwardingInfoTable( Model ):
   ''' Model for the label forwarding information table'''
   __revision__ = 2
   labelForwardingInfos = Dict( keyType=int, valueType=LabelForwardingInfo,
                                help='a mapping of decap labels to '
                                     'Label forwarding information entries' )

class PatchForwardingTable( Model ):
   '''Model for patch panel forwarding tables,
   contains forwarding tables for both tagged/raw and label connections '''
   __revision__ = 2
   taggedForwardingInfoTable = Submodel( valueType=TaggedForwardingInfoTable,
                                         help='Tagged forwarding information table' )
   portForwardingInfoTable = Submodel( valueType=PortForwardingInfoTable,
                                       help='Port forwarding information table' )
   labelForwardingInfoTable = Submodel( valueType=LabelForwardingInfoTable,
                                        help='Label forwarding information table' )

   def renderLocalFwdInfoTable( self, portFwdInfos, taggedFwdInfos, tableWidth ):
      leftJustify = Format( justify='left' )
      rightJustify = Format( justify='right' )
      for fmt in ( leftJustify, rightJustify ):
         fmt.padLimitIs( True )

      headings = ( ' In/Out ', ' Source ', ' Type ', ' Flags ', ' Patch ' )
      fmts = ( leftJustify, leftJustify, rightJustify, leftJustify, leftJustify )

      localFwdingInfoTable = createTable( headings, tableWidth=tableWidth )
      localFwdingInfoTable.formatColumns( *fmts )

      localFwdInfos = taggedFwdInfos.items() + portFwdInfos.items()
      for _, entry in sorted( localFwdInfos ):
         entry.renderInTable( localFwdingInfoTable )

      print localFwdingInfoTable.output()

   def renderLabelFwdInfoTable( self, labelFwdInfos, tableWidth ):
      leftJustify = Format( justify='left' )
      rightJustify = Format( justify='right' )
      for fmt in ( leftJustify, rightJustify ):
         fmt.padLimitIs( True )

      headings = ( ' PW Label ', ' Out ', ' Source ',
                   ' Type ', ' Flags ', ' Patch ' )
      labelFwdingInfoTable = createTable( headings, tableWidth=tableWidth )
      labelFwdingInfoTable.formatColumns( leftJustify, leftJustify, leftJustify,
                                          rightJustify, leftJustify, leftJustify )

      for _, entry in sorted( labelFwdInfos.items() ):
         entry.renderInTable( labelFwdingInfoTable )

      print labelFwdingInfoTable.output()

   def render( self ):
      print fwdingOptionHelp + '\n'

      # Set the width of the output tables to the wider of the two tables
      tableWidth = 132
      self.renderLocalFwdInfoTable(
         taggedFwdInfos=self.taggedForwardingInfoTable.taggedForwardingInfos,
         portFwdInfos=self.portForwardingInfoTable.portForwardingInfos,
         tableWidth=tableWidth )
      self.renderLabelFwdInfoTable(
         labelFwdInfos=self.labelForwardingInfoTable.labelForwardingInfos,
         tableWidth=tableWidth )
