# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from collections import defaultdict

import BasicCli, CliParser
import CliCommand
import CliMatcher
import ShowCommand
import LazyMount
import SharedMem
import Smash
from TunnelModels import tunnelTypeStrDict as tunTypeToStr
from TunnelCli import getTunnelTypeFromTunnelId as tunIdToType
from PseudowireLib import (
      flEnumValDisabled,
      flEnumValTransmit,
      flValToEnum,
      FlowLabelMode,
)
import PwaModel
from PwaModel import ConnectorStatusType as CST
from CliPlugin.Pseudowire import anyCaseRegex, validNameRegex
import TechSupportCli
import Tac
import Toggles.ArBgpToggleLib
from Toggles.EbraToggleLib import toggleMplsEvpnVpwsP25Enabled
from TypeFuture import TacLazyType

pwConfig = None
pwStatus = None
pwLfib = None
pwaPatchPanel = None
pwaLocalConnectorStatusColl = None
pwaRemoteConnectorStatusColl = None
pwHwCapability = None
pwHwDir = None
pwaLabelBinding = None
pwaLdpRequest = None
pwaLdpResponse = None
pwaEbraResponse = None
pwaBgpRequest = None
pwaBgpResponse = None

Dot1qEncap = TacLazyType( 'Bridging::Dot1qEncap' )
TunnelId = Tac.Type( 'Tunnel::TunnelTable::TunnelId' )
PwLdpReqKey = Tac.Type( "Pseudowire::PseudowireLdpRequestKey" )
PwLdpRespKey = Tac.Type( "Pseudowire::PseudowireLdpResponseKey" )
PwBgpReqKey = TacLazyType( "Pseudowire::PseudowireBgpRequestKey" )
PwBgpRespKey = TacLazyType( "Pseudowire::PseudowireBgpResponseKey" )
PwConnectorKey = TacLazyType( "Pseudowire::ConnectorKey" )
PwConnectorType = TacLazyType( "Pseudowire::ConnectorType" )
PwIntfId = Tac.Type( 'Arnet::PseudowireIntfId' )
PwType = Tac.Type( "Pseudowire::PseudowireType" )
IntfId = Tac.Type( "Arnet::IntfId" )
SubIntfId = Tac.Type( "Arnet::SubIntfId" )
MplsLabel = TacLazyType( "Arnet::MplsLabel" )
VlanId = Tac.Type( "Bridging::VlanId" )
VlanIdOrNone = Tac.Type( "Bridging::VlanIdOrNone" )
CSH = Tac.Value( 'Pseudowire::PseudowireConnStatusHelper' )
UnprogrammedReason = Tac.Type( "AlePw::UnprogrammedReason" )
IntfEncapHwStatus = TacLazyType( "Ebra::IntfEncapHwStatus" )

connStatusFromIntfEncapHwStatus = { # Revisit this mapping with BUG498650
   IntfEncapHwStatus.failed: CST.connectorUnprogrammed,
   IntfEncapHwStatus.notReady: CST.connectorUnprogrammed,
   IntfEncapHwStatus.success: CST.up
}

encapStatusFromUnprogrammedReason = {
   UnprogrammedReason.connInfoMissing: CST.encapConnInfoMissing,
   UnprogrammedReason.invalidEndPointConfig: CST.encapInvalidConnConfig,
   UnprogrammedReason.hwFecNotProgrammed: CST.encapHwFecUnprogrammed,
   UnprogrammedReason.hwProgrammingFailed: CST.encapHwProgrammingFailed,
   }

decapStatusFromUnprogrammedReason = {
   UnprogrammedReason.connInfoMissing: CST.decapConnInfoMissing,
   UnprogrammedReason.invalidEndPointConfig: CST.decapInvalidConnConfig,
   UnprogrammedReason.hwProgrammingFailed: CST.decapHwProgrammingFailed,
   }

class PatchTypes( object ):
   loLo = 'local-local'
   loRmt = 'local-remote'

def pwSupportedGuard( mode, token ):
   if pwHwCapability.mplsPseudowireSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def isPatchUp( status ):
   '''Returns if patch->status is up.
   @status is patchPanel->patch->status which is of
         type PseudowireConnectorStatus (and not the human readable
         versions of them). (The PwaModel.Patch.status is of enum with
         "Up", "Down", "Unprogrammed", "Admin down")'''
   return status == CST.up

#---------------------------------------------
# Useful tokens for show commands
#---------------------------------------------
matcherPanel = CliMatcher.KeywordMatcher( 'panel',
      helpdesc='Show patch panel information' )
nodePatch = CliCommand.guardedKeyword( 'patch',
      helpdesc='Show patch panel information',
      guard=pwSupportedGuard )

disallowRegex = anyCaseRegex( "(detail|forwarding)" )
showConnNameRegex = "^(?!%s$)%s$" % ( disallowRegex, validNameRegex )
patchNameMatcher = CliMatcher.DynamicNameMatcher(
                        lambda mode: sorted( pwConfig.patch.keys() ),
                        pattern=showConnNameRegex,
                        helpdesc='Patch name' )

#---------------------------------------------
# Local connector helper methods
# Local connector = tagged + port connectors
#---------------------------------------------
def getLocalConnectorProgrammedStatusLegacyApi( pwIntfId, patchType ):
   '''Check if the local connector has been programmed in ale (legacy API)'''

   for name, pwHwStatus in pwHwDir.items():
      if not name.startswith( 'status' ):
         continue

      if patchType == PatchTypes.loLo:
         reason = pwHwStatus.unprogrammedEncapAdj.get( pwIntfId )
         if reason is None:
            continue
         elif reason == "hwProgrammingFailed":
            if pwIntfId in pwHwStatus.unprogrammedLocalConnector:
               return CST.connectorUnprogrammed
         return encapStatusFromUnprogrammedReason[ reason ]

      elif patchType == PatchTypes.loRmt:
         if pwIntfId in pwHwStatus.unprogrammedLocalConnector:
            return CST.connectorUnprogrammed
         continue

      assert False, "Unknown patchType %s" % str( patchType )

   # Finally, if no hardware/ale/pw/status... entry said that it's down,
   # then it must be up!
   return CST.up

def getLocalConnectorProgrammedStatus( serviceIntfId ):
   """Check if the local connector has been programmed in ale (new API).

   If there is no entry saying that it's down, then it must be up!
   This is in line with the legacy API because PwA does not currently have
   a status that says 'hw programming pending'. Improve this with BUG498650
   """
   reason = pwHwDir.hwStatus.get( serviceIntfId, IntfEncapHwStatus.success )
   return connStatusFromIntfEncapHwStatus.get( reason, CST.connectorUnprogrammed )

def getLocalConnectorStatus( connKey, patchType, patchName ):
   '''Returns the status of a local connector identified
   by connKey. It has the logic to check the status of the
   local connector in
   1. PatchPanel
   2. LocalConnectorStatus
   3. Ale
   
   For bug 144948,  we'll also check the status of the patch it belongs to,
   if applicable, in order to determine if the connector status is relevant,
   ( if the patch is admin down, the connectors should also be admin down )
   '''
   connStatus = pwaPatchPanel.connStatus.get( connKey )
   if not connStatus:
      return CST.down

   # If the patch is available, then check if patch is enabled
   patch = pwaPatchPanel.patch.get( patchName, None )
   # If patch is disabled, then always return 'adminDown' as status of the 
   # constituent connectors
   if not patch or not patch.enabled:
      return CST.adminDown

   status = connStatus.status
   if CSH.isDown( status ):
      return status

   # status is Up. Check LCS get status from there.
   connStatus = pwaLocalConnectorStatusColl.connectorStatus.get( connKey, None )
   if not connStatus:
      return CST.noLocalEnd

   status = connStatus.status
   if CSH.isDown( status ):
      return status
   elif status == CST.disabledByPw:
      # The interface is error disabled. Let's return this
      # immediately
      return status

   # Note that we explicitly are not checking for whether encap adjacency is
   # written by PWA before we look at what Ale is reporting. If we checked
   # for encap adjacency, we would see output like:
   #   conn1 status = Cli conflict
   #   conn2 status = Encap not present (instead of Up)
   # The only way to evaluate conn2's status would be that it knows the
   # reason for its non-installation is conn1's status
   #
   # Similarly conn1's status would depend on conn2's status and thus
   # evaluating either of them indenpendently would be unnecassry
   # logic.
   #
   # Relying on "show patch panel forwarding" is more appropriate
   # to look at published state as that only walks over all
   # encap and decap entries.

   if toggleMplsEvpnVpwsP25Enabled():
      return getLocalConnectorProgrammedStatus( connKey.serviceIntfId() )
   else:
      pwIntfId = connKey.localPwIntfId( pwaPatchPanel.getDot1qEncap( connKey ) )
      return getLocalConnectorProgrammedStatusLegacyApi( pwIntfId, patchType )

def taggedConnector( connKey, connStatus, patchType, conn, patchName ):
   '''Populates the tagged connector specific details in
   conn (PwaModel.Connector) and returns the modified
   connector and status enum.'''
   conn.taggedConnectorInfo = PwaModel.TaggedConnector(
            interface=connStatus.connector.connectorKey.localIntfId(),
            dot1qTag=connStatus.connector.dot1qEncap.outerTag )
   status = getLocalConnectorStatus( connKey, patchType, patchName )
   conn.status = status
   return status, conn

def portConnector( connKey, connStatus, patchType, conn, patchName ):
   '''Populates the port connector specific details in
   conn (PwaModel.Connector) and returns the modified
   connector and status enum.'''
   conn.portConnectorInfo = PwaModel.PortConnector(
      interface=connStatus.connector.connectorKey.localIntfId() )
   status = getLocalConnectorStatus( connKey, patchType, patchName )
   conn.status = status
   return status, conn

def getConnectorType( pwIntfId ):
   # VlanId::invalid() indicates that it is a port type
   lcInfo = pwStatus.localConnectorInfo.get( pwIntfId )
   if lcInfo and SubIntfId.isSubIntfId( lcInfo.parentIntfId ):
      return "port" # treat subinterface connector as port to avoid dot1q output
   elif PwIntfId.vlanId( pwIntfId ) == VlanId.invalid:
      return "port"
   else:
      return "tagged"

#---------------------------------------------
# Remote pseudowire connector helper methods
#---------------------------------------------

def getRemotePseudowireProgrammedStatusLegacyApi( localLabel, pwIntfId ):
   '''Checks if the remote pseudowire has been programmed in Ale (legacy). Assumes
   that the encap/decap information is already written into PseudowireStatus.'''

   # Check encap programming state if loConnKey is provided
   for name, pwHwStatus in pwHwDir.items():
      if not name.startswith( 'status' ):
         continue

      if pwIntfId:
         unprogReason = pwHwStatus.unprogrammedEncapAdj.get( pwIntfId )
         if ( unprogReason and encapStatusFromUnprogrammedReason[ unprogReason ] ==
              CST.encapHwProgrammingFailed ):
            # Check if the reason for unprogramed encap is connector being
            # unprogrammed
            connUnprogReason = pwHwStatus.unprogrammedLocalConnector.get( pwIntfId )
            if connUnprogReason:
               return CST.connectorUnprogrammed

         if unprogReason:
            return encapStatusFromUnprogrammedReason[ unprogReason ]

      # Encap adjacency is programmed, check if decap adjacency is programmed
      unprogReason = pwHwStatus.unprogrammedDecapAdj.get( localLabel )
      if unprogReason:
         return decapStatusFromUnprogrammedReason[ unprogReason ]

   return CST.up

def getRemotePseudowireProgrammedStatus( serviceIntfId ):
   """"Checks if the remote pseudowire has been programmed in Ale (new API).

   The new Ale API does not publish separate programming states for enap and decap,
   as a result there is no dedicated programming status for a the remote end.
   Simply mirror the local connector programmed status.

   Once we add FXC, this function may want to return 'connectorUnprogrammed' or some
   other dedicated status if any of the local connectors is unprogrammed.
   """
   return getLocalConnectorProgrammedStatus( serviceIntfId )

def getRemotePseudowireConnectorStatus( connKey, patchName, loIntfId=None ):
   '''Returns the status of a remote pseudowire connector identified
   by connKey. It has the logic to check the status of the remote 
   connector in
   1. PatchPanel
   2. LabelBinding (if one exists)
   3. PseudowireLdpRequest or PseudowireBgpRequest (if a request is made)
   4. PseudowireLdpResponse or PseudowireBgpResponse (if a response exists)
   5. RemoteConnectorStatus
   6. Ale: if loIntfId is provided, check both encap/local connector status
      otherwise only report if the *localLabel* (decap side) is programmed

   For bug 144948,  we'll also check the status of the patch it belongs to,
   if applicable, in order to determine if the connector status is relevant
   '''
   connStatus = pwaPatchPanel.connStatus.get( connKey, None )

   if connStatus is None:
      return CST.down

   # connKey exists in patchpanel. Get status from here first
   status = connStatus.status

   if CSH.isDown( status ):
      return status
  
   # If the patch is available, then check if patch is enabled
   patch = pwaPatchPanel.patch.get( patchName, None )
   # If patch is disabled, then always return 'adminDown' as status of the 
   # constituent connectors
   if not patch or not patch.enabled:
      return CST.adminDown

   # Status is patchpanel is up, lets check if label binding succeeded
   localLabel = pwaLabelBinding.localLabel.get( connKey, None )

   if localLabel is None:
      # localBinding failed
      return CST.noLocalMapping

   # localLabel is assigned. Let's check remote connector request and response
   if connKey.connectorType == PwConnectorType.ldp:
      status = getLdpRequestResponseStatus( connStatus )
   elif connKey.connectorType == PwConnectorType.bgp:
      status = getBgpRequestResponseStatus( connKey )
   else:
      assert False, "Unhandled connectorType {}".format( connKey.connectorType )
   if status != CST.up:
      return status

   # Check RemoteConnectorStatus
   rcsStatus = pwaRemoteConnectorStatusColl.connectorStatus.get( connKey )
   
   if rcsStatus is None:
      if not pwaRemoteConnectorStatusColl.tunnelRibStatus:
         return CST.noTunnelRib
      return CST.noTunnel

   if CSH.isDown( rcsStatus.status ):
      return rcsStatus.status

   # Check if the corresponding encap and decap entries exist. If they don't,
   # ale can't even report that they are unprogrammed
   if not loIntfId:
      return CST.unprogrammed

   # Note that we explicitly are not checking for whether encap adjacency is
   # written by PWA before we look at what Ale is reporting. If we checked
   # for encap adjacency, we would see output like:
   #   conn1 status = Cli conflict
   #   conn2 status = Encap not present (instead of Up)
   # The only way to evaluate conn2's status would be that it knows the
   # reason for its non-installation is conn1's status
   #
   # Similarly conn1's status would depend on conn2's status and thus
   # evaluating either of them indenpendently would be unnecassry
   # logic.
   #
   # Relying on "show patch panel forwarding" is more appropriate
   # to look at published state as that only walks over all
   # encap and decap entries.

   if toggleMplsEvpnVpwsP25Enabled():
      return getRemotePseudowireProgrammedStatus( loIntfId )
   else:
      localLabel = pwaLabelBinding.localLabel.get( connKey, None )
      return getRemotePseudowireProgrammedStatusLegacyApi( localLabel, loIntfId )

#---------------------------------------------
# Ldp pseudowire connector helper methods
#---------------------------------------------

def getLdpRequestResponseStatus( connStatus ):
   """Returns a status corresponding to the presence of LDP request and response"""
   neighbor = connStatus.connector.neighborAddr
   pwId = connStatus.connector.pwId

   # Check LDP request
   reqKey = PwLdpReqKey( neighbor, pwId )

   if reqKey not in pwaLdpRequest.ldpRequest:
      return CST.noLocalEnd

   # Check LDP response
   respKey = PwLdpRespKey( neighbor, pwId )

   if respKey not in pwaLdpResponse.response:
      # For the transient instance when LDP request is created
      # but LDP response doesn't exist, return noRemoteEnd
      return CST.noRemoteEnd

   return CST.up

def getLdpPwFlowLabelType( neighbor, pwId ):
   '''Returns the flowLabelType from pseudowire LDP response'''
   if neighbor and pwId:
      respKey = PwLdpRespKey( neighbor, pwId )
      response = pwaLdpResponse.response.get( respKey )
      if response is not None:
         return flValToEnum[ response.flowLabelType ]
   return flValToEnum[ flEnumValDisabled ]

def getLdpPwForwardingState( connKey, neighbor, pwId ):
   '''Returns the ( forwarding, errorFlags = PwaModel.LdpPseudowireErrorFlags )
   tuple for the provided remote connector identified by (connKey, neighbor, pwId).
   forwarding - indicates if the peer is forwarding pseudowire bidirectional
                traffic
   errorFlags - if the peer is not forwarding traffic, the appropriate error flags
                are returned.
   '''
   respKey = PwLdpRespKey( neighbor, pwId )
   response = pwaLdpResponse.response.get( respKey )
   if response is not None:
      statusVal = response.pwStatusFlags
      flags = Tac.Value( "Pseudowire::PseudowireLdpStatusFlags", statusVal )

      forwarding = flags.pwForwarding
      errorFlags = None
      if not forwarding:
         errorFlags = PwaModel.LdpPseudowireErrorFlags(
                  notForwarding = flags.pwNotForwarding or None,
                  etInFault = flags.localACIngrRecvFault or None,
                  etOutFault = flags.localACEgrTransFault or None,
                  tunInFault = flags.localPsnPwIngrRecvFault or None,
                  tunOutFault = flags.localPsnPwEgrTransFault or None
               )

      return forwarding, errorFlags

   return False, None

def getLdpPwNeighborConfig( connKey, neighbor, pwId ):
   '''Returns LdpPseudowirePeerConfig model for an LDP remote
   connector.'''
   reqKey = PwLdpReqKey( neighbor, pwId )
   respKey = PwLdpRespKey( neighbor, pwId )
   req = pwaLdpRequest.ldpRequest.get( reqKey, None )
   resp = pwaLdpResponse.response.get( respKey, None )
   if not req:
      return None, PwType.pwTypeNone

   nbrState = PwaModel.LdpPseudowirePeerConfig()

   if resp is None:
      return nbrState, req.pwType

   nbrState.mtu = resp.mtu
   nbrState.intfDescription = resp.intfDescription

   nbrState.label = resp.peerLabel
   nbrState.groupId = resp.groupId

   dot1qTagRequested = ( resp.requestedDot1qTag != 0 )
   if dot1qTagRequested:
      nbrState.dot1qTagRequested = resp.requestedDot1qTag

   nbrState._isLocal = False # pylint:disable=W0212

   nbrState.flowLabelCapability = resp.peerFlowLabelMode

   nbrState.vccvCcTypes = resp.peerVccvCcTypes
   nbrState.vccvCvTypes = resp.peerVccvCvTypes

   return nbrState, req.pwType

def getLdpPwLocalConfig( connKey, neighbor, pwId ):
   '''Returns LdpPseudowirePeerConfig model for a local connector.'''
   localState = PwaModel.LdpPseudowirePeerConfig()
   localLabel = pwaLabelBinding.localLabel.get( connKey )
   if localLabel:
      localState.label = localLabel

   connStatus = pwaPatchPanel.connStatus.get( connKey, None )
   if not connStatus:
      return None

   localState.groupId = connStatus.connector.groupId
   localState.mtu = connStatus.mtu

   localState._isLocal = True # pylint:disable=W0212
   reqKey = PwLdpReqKey( neighbor, pwId )
   req = pwaLdpRequest.ldpRequest.get( reqKey )
   if req:
      localState.vccvCcTypes = req.localVccvCcTypes
      localState.vccvCvTypes = req.localVccvCvTypes
      localState.flowLabelCapability = connStatus.connector.flowLabelMode

   return localState

def ldpPseudowireConnector( connKey, connStatus, conn, patchName ):
   '''Populates an LDP pseudowire connector specific details in
   conn (PwaModel.Connector) and returns the modified connector
   and status enum.
   @connStatus - connectorStatus in cliPatch/patchPanel
   '''
   connInfo = PwaModel.LdpPseudowireConnector( name=connKey.name )
   connInfo.tunnelIndexes = None

   if connStatus.connector.neighborAddrPresent:
      connInfo.neighbor = connStatus.connector.neighborAddr

   if connStatus.connector.pwIdPresent:
      connInfo.pseudowireId = connStatus.connector.pwId

   connInfo.pseudowireType = PwType.pwTypeNone
   connInfo.controlWord = connStatus.connector.controlWord

   # Assume it to be not up. We get exact state later
   connInfo.neighborForwarding = False

   connInfo.flowLabelUsed = getLdpPwFlowLabelType( connInfo.neighbor,
                                                   connInfo.pseudowireId )

   # If either of the mandatory pieces of information are missing,
   # there is nothing else we can add to the show commands
   if ( not connStatus.connector.neighborAddrPresent or
        not connStatus.connector.pwIdPresent or
        connStatus.mtu == 0 ):
      conn.ldpPseudowireConnectorInfo = connInfo
      status = getRemotePseudowireConnectorStatus( connKey, patchName )
      conn.status = status
      return status, conn
 
   forwarding, faultFlags = getLdpPwForwardingState( connKey, connInfo.neighbor,
                                                     connInfo.pseudowireId )

   connInfo.neighborForwarding = forwarding
   connInfo.neighborFaultFlags = faultFlags

   connInfo.localConfig = getLdpPwLocalConfig( connKey, connInfo.neighbor,
                                               connInfo.pseudowireId )
   connInfo.neighborConfig, connInfo.pseudowireType = \
         getLdpPwNeighborConfig( connKey, connInfo.neighbor, connInfo.pseudowireId )

   conn.ldpPseudowireConnectorInfo = connInfo
   
   # Tunnel information
   # To get the tunnel information in use, we need to retrieve this from
   # RemoteConnectorStatus which is keyed by the connector key.
   # Verify patch status is up before trying to access partner key.
   connToPatch = pwaPatchPanel.connToPatch.get( connKey )
   if not connToPatch or len( connToPatch.activePatch ) != 1:
      status = getRemotePseudowireConnectorStatus( connKey, patchName )
      conn.status = status
      return status, conn

   patch = pwaPatchPanel.patch.get( patchName, None )
   if not patch or not isPatchUp( patch.status ):
      status = getRemotePseudowireConnectorStatus( connKey, patchName )
      conn.status = status
      return status, conn

   loConnKey = connStatus.peerConnectorKey
   loIntfId = ( loConnKey.serviceIntfId() if toggleMplsEvpnVpwsP25Enabled()
                else loConnKey.localPwIntfId( pwaPatchPanel.getDot1qEncap(
                                                loConnKey ) ) )

   status = getRemotePseudowireConnectorStatus( connKey, patchName,
                                                loIntfId=loIntfId )
   conn.status = status

   rcStatus = pwaRemoteConnectorStatusColl.connectorStatus.get( connKey )

   if not rcStatus or not rcStatus.peerInfo or CSH.isDown( rcStatus.status ):
      return status, conn

   tunnelId = rcStatus.peerInfo.values()[ 0 ].tunnelId # There will be only one
   connInfo.tunnelType = tunTypeToStr[ tunIdToType( tunnelId ) ]
   connInfo.tunnelIndexes = [ TunnelId( tunnelId ).tunnelIndex() ]

   return status, conn

#---------------------------------------------
# BGP pseudowire connector helper methods
#---------------------------------------------

def getBgpRequestResponseStatus( connKey ):
   """Returns a status corresponding to the presence of BGP request and response"""

   # If remote connector status is present then don't check BGP request and
   # response, because they might not be there if for example the local interface
   # is down. This is different from LDP which creates LDP request/response in
   # this these cases.
   if connKey not in pwaRemoteConnectorStatusColl.connectorStatus:
      # Check BGP request
      reqKey = PwBgpReqKey( connKey.name, connKey.vpwsName )

      if reqKey not in pwaBgpRequest.bgpRequest:
         return CST.noLocalEnd

      # Check BGP response
      respKey = PwBgpRespKey( connKey.name, connKey.vpwsName )

      if respKey not in pwaBgpResponse.bgpResponse:
         # For the transient instance when BGP request is created
         # but BGP response doesn't exist, return noRemoteEnd
         return CST.noRemoteEnd

   return CST.up

def getBgpPwLocalConfig( connKey, loConnKey ):
   """Returns BgpPseudowireLocalConfig model for a BGP remote connector."""
   localState = PwaModel.BgpPseudowireLocalConfig()

   localLabel = pwaLabelBinding.localLabel.get( connKey )
   if localLabel:
      localState.label = localLabel

   respKey = PwBgpRespKey( connKey.name, connKey.vpwsName )
   resp = pwaBgpResponse.bgpResponse.get( respKey )
   flowLabel = None
   if resp:
      localState.controlWord = resp.controlWord
      localState.mtu = resp.mtu
      if Toggles.ArBgpToggleLib.toggleVpwsFlowLabelToggleEnabled():
         if resp.flowLabel:
            flowLabel = FlowLabelMode.both
         else:
            flowLabel = FlowLabelMode.disabled

   vpwsType = "port" if loConnKey.isPort() else "vlan"

   return localState, vpwsType, flowLabel

def getBgpPwNeighborConfig( connKey ):
   """Returns BgpPseudowireNeighborConfig model for a BGP remote connector."""
   respKey = PwBgpRespKey( connKey.name, connKey.vpwsName )
   resp = pwaBgpResponse.bgpResponse.get( respKey )
   if not resp:
      return None

   nbrState = {}

   for peerInfo in resp.peerInfo.itervalues():
      nbrConfig = PwaModel.BgpPseudowireNeighborConfig()
      nbrConfig.label = peerInfo.label
      tunnelId = TunnelId( peerInfo.tunnelId )
      nbrConfig.tunnelType = tunTypeToStr[ tunIdToType( peerInfo.tunnelId ) ]
      nbrConfig.tunnelIndex = tunnelId.tunnelIndex()
      nbrConfig.controlWord = peerInfo.controlWord
      nbrConfig.mtu = peerInfo.mtu
      nbrState[ peerInfo.addr ] = nbrConfig

   return nbrState

def bgpPseudowireConnector( connKey, connStatus, conn, patchName ):
   """Populates a BGP pseudowire connector specific details in conn
   (PwaModel.Connector) and returns the modified connector and status enum.
   """
   connInfo = PwaModel.BgpPseudowireConnector( vpwsName=connKey.name,
                                               pwName=connKey.vpwsName )
   loConnKey = connStatus.peerConnectorKey
   if loConnKey.isValid():
      connInfo.localConfig, connInfo.vpwsType, connInfo.flowLabelUsed = \
            getBgpPwLocalConfig( connKey, loConnKey )
   connInfo.neighborConfigs = getBgpPwNeighborConfig( connKey )

   conn.bgpPseudowireConnectorInfo = connInfo
   if loConnKey.isValid():
      loIntfId = ( loConnKey.serviceIntfId() if toggleMplsEvpnVpwsP25Enabled()
                   else loConnKey.localPwIntfId( pwaPatchPanel.getDot1qEncap(
                                                   loConnKey ) ) )
   else:
      # patch with only a remote connector for example
      # can not look at the local connector to figure out hardware programming
      loIntfId = None
   status = getRemotePseudowireConnectorStatus( connKey, patchName,
                                                loIntfId=loIntfId )
   conn.status = status
   return status, conn

#---------------------------------------------
# "show patch panel" and "show patch panel detail" helper
# methods
#---------------------------------------------

def getConnectorName( patchName, connKey ):
   if not patchName or not connKey:
      return ""

   patch = pwConfig.patch.get( patchName )
   if not patch:
      return ""

   # Find cliConnectors that have connKey
   cliConns = [ cc for cc in patch.connector.values()
                if cc.connectorKey == connKey ]
   if cliConns:
      return cliConns[ 0 ].name

   return ""

def getConnector( patchName, connKey, patchType ):
   '''Creates and populates a PwaModel.Connector() corresponding
   for a connector identified by connKey.'''
   if not connKey:
      return None

   connStatus = pwaPatchPanel.connStatus.get( connKey, None )
   if not connStatus:
      return None

   if connKey.isLocal():
      connType = "tagged" if connKey.isLegacyTagged() else "port"
   else:
      connType = connKey.connectorType
   assert connType in PwaModel.supportedConnectorTypes

   connector = PwaModel.Connector( connType=connType )

   if connType == "tagged":
      return taggedConnector( connKey, connStatus, patchType, connector, patchName )
   if connType == "port":
      return portConnector( connKey, connStatus, patchType, connector, patchName )
   if connType == "ldp":
      return ldpPseudowireConnector( connKey, connStatus, connector, patchName )
   if connType == "bgp":
      return bgpPseudowireConnector( connKey, connStatus, connector, patchName )
   return None

def getPatchStatus( patchName, status1=None, status2=None ):
   '''If either of the connector is not present or has status as down,
   we return "down". If both connectors are up, we return "up". For
   all other cases (up, unprogrammed) or (unprogrammed, unprogrammed),
   we return "unprogrammed" as the status of the patch.
   @conn1, @conn2 are of type PwaModel.Connector'''
   patch = pwaPatchPanel.patch.get( patchName, None )

   if not patch:
      return CST.down

   if not patch.enabled:
      return CST.adminDown

   if status1 is None or CSH.isDown( status1 ):
      return CST.down
   
   if status2 is None or CSH.isDown( status2 ):
      return CST.down
   
   if patch.status == CST.cliConflict:
      return CST.cliConflict
   
   if CSH.isUp( status1 ) and CSH.isUp( status2 ):
      return CST.up

   return CST.unprogrammed

def getPatch( patchName ):
   '''Returns a PwaModel.Patch object corresponding to patchName'''
   patchStatus = pwaPatchPanel.patch.get( patchName )

   if patchStatus is None:
      return None

   connKeys = patchStatus.patchConnectorKey
   connectors = {}
   status1, status2 = None, None

   # Determine the patch type. Current supported ones are
   # lo-lo and lo-ldp. If any of the connectors is of type
   connTypes = { ck.connectorType for ck in connKeys }
   patchType = ( PatchTypes.loRmt if connTypes.intersection( { "bgp", "ldp" } )
                 else PatchTypes.loLo )

   for connKey in connKeys:
      ret = getConnector( patchName, connKey, patchType )
      if ret is not None:
         if not status1:
            status1, connector = ret
         elif not status2:
            status2, connector = ret
         else:
            assert False, "more than two connectors"
         connName = getConnectorName( patchName, connKey )
         connectors[ connName ] = connector

   patchStatus = getPatchStatus( patchName, status1=status1, status2=status2 )

   return PwaModel.Patch( status=patchStatus, connectors=connectors )

#--------------------------------------------------------------------------------
# show patch panel summary
#--------------------------------------------------------------------------------
class PatchPanelSummaryCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show patch panel summary'
   data = {
      'patch': nodePatch,
      'panel': matcherPanel,
      'summary': 'Show patch panel summary',
   }
   cliModel = PwaModel.PatchSummary

   @staticmethod
   def handler( mode, args ):
      patchSummary = PwaModel.PatchSummary()
      patchSummary.patchCount = len( pwaPatchPanel.patch )

      bgpVpwsCount = defaultdict( int )
      ldpCount = defaultdict( int )
      localCount = defaultdict( int )

      for patchName in pwaPatchPanel.patch:
         patch = getPatch( patchName )
         connTypes = { conn.connType for conn in patch.connectors.itervalues() }
         if "bgp" in connTypes:
            bgpVpwsCount[ patch.status ] += 1
         elif "ldp" in connTypes:
            ldpCount[ patch.status ] += 1
         else:
            # local-local connector
            localCount[ patch.status ] += 1

      patchSummary.bgpVpwsCount = dict( bgpVpwsCount )
      patchSummary.ldpCount = dict( ldpCount )
      patchSummary.localCount = dict( localCount )
      return patchSummary

BasicCli.addShowCommandClass( PatchPanelSummaryCmd )

#--------------------------------------------------------------------------------
# show patch panel [ PATCH_NAME ] [ detail ]
#--------------------------------------------------------------------------------
class PatchPanelDetailCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show patch panel [ PATCH_NAME ] [ detail ]'
   data = {
      'patch': nodePatch,
      'panel': matcherPanel,
      'PATCH_NAME': patchNameMatcher,
      'detail': 'Show patch panel details',
   }
   cliModel = PwaModel.Patches

   @staticmethod
   def handler( mode, args ):
      patchName = args.get( 'PATCH_NAME' )
      patches = {}

      if patchName:
         # patchName is specified, only return this if it is valid
         patch = getPatch( patchName )
         if patch:
            patches[ patchName ] = patch
      else:
         # patchName wasn't specified. Return all patches in
         # patchPanel.
         for patchName in pwaPatchPanel.patch:
            patch = getPatch( patchName )
            if patch:
               patches[ patchName ] = patch

      detail = 'detail' in args
      return PwaModel.Patches( patches=patches, _summary=not detail )

BasicCli.addShowCommandClass( PatchPanelDetailCmd )

#-------------------------------------------------------------------------------
# Helpers for the 'show patch panel forwarding' command
#-------------------------------------------------------------------------------

def getConnectorPatchInfo( connKey ):
   connStatus = pwaPatchPanel.connStatus.get( connKey )
   if connStatus:
      return connStatus.patchName, connStatus.peerConnectorKey
   else:
      return None, None

def getLocalConnectorFwdingInfo( connKey, decapCw=None ):
   localIntfId = connKey.localIntfId()
   if connKey.isLegacyTagged():
      taggedConnFwdingInfo = PwaModel.TaggedConnectorFwdingInfo(
         interface=localIntfId,
         dot1qVlan=connKey.legacyDot1qTag(),
         decapControlWord=decapCw )
      connFwdingInfo = PwaModel.ForwardingConnectorInfo(
                                          _infoType="tagged",
                                          taggedConnFwdingInfo=taggedConnFwdingInfo )
   else:
      portConnFwdingInfo = PwaModel.PortConnectorFwdingInfo(
         interface=localIntfId, decapControlWord=decapCw )
      connFwdingInfo = PwaModel.ForwardingConnectorInfo(
                                          _infoType="port",
                                          portConnFwdingInfo=portConnFwdingInfo )
   return connFwdingInfo

def getLocalEntryLegacyApi( pwIntfId ):
   # Convert pwIntfId to connKey
   lcInfo = pwStatus.localConnectorInfo.get( pwIntfId )
   if not lcInfo:
      return None
   if SubIntfId.isSubIntfId( lcInfo.parentIntfId ):
      connKey = PwConnectorKey()
      connKey.localConnectorKeyIs( lcInfo.parentIntfId )
   else:
      connKey = PwConnectorKey.fromServiceIntf( pwIntfId )

   return getLocalEntry( connKey, pwIntfId=pwIntfId )

def getLocalEntry( connKey, pwIntfId=None ):
   patchName, peerConnKey = getConnectorPatchInfo( connKey )
   if not patchName or not peerConnKey or not peerConnKey.isValid():
      return None

   connectorFwdingInfo = getLocalConnectorFwdingInfo( connKey )
   entry = ( PwaModel.TaggedForwardingInfo( inConnInfo=connectorFwdingInfo )
             if connKey.isLegacyTagged()
             else PwaModel.PortForwardingInfo( inConnInfo=connectorFwdingInfo ) )

   entry.patch = patchName
   if peerConnKey.isLocal():
      entry.outConnInfo = getLocalConnectorFwdingInfo( peerConnKey )
   else:
      rcs = pwaRemoteConnectorStatusColl.connectorStatus.get( peerConnKey )
      if not rcs:
         return None
      entry.source = labelSources.get( peerConnKey.connectorType )
      entry.pseudowireType = rcs.remotePwType
      if rcs.flowLabelType & flEnumValTransmit:
         entry.flowLabel = 'Out'
      if rcs.requestedDot1qTag:
         entry.dot1qVlan = rcs.requestedDot1qTag

      labelConnFwdingInfo = PwaModel.LabelConnectorFwdingInfo()
      for tep, peerInfo in rcs.peerInfo.items():
         tunId = TunnelId( peerInfo.tunnelId )
         labelConnFwdingPeerInfo = PwaModel.LabelConnectorFwdingPeerInfo(
            tunnelType=tunTypeToStr[ tunIdToType( peerInfo.tunnelId ) ],
            peerLabel=peerInfo.peerLabel,
            tunnelIndex=tunId.tunnelIndex(),
            encapControlWord=peerInfo.encapControlWord )
         labelConnFwdingInfo.peers[ tep ] = labelConnFwdingPeerInfo

      entry.outConnInfo = PwaModel.ForwardingConnectorInfo(
         _infoType=peerConnKey.connectorType,
         labelConnFwdingInfo=labelConnFwdingInfo )

   if pwIntfId is not None:
      # Use legacy API
      entry.status = getLocalConnectorProgrammedStatusLegacyApi(
         pwIntfId, PatchTypes.loLo if peerConnKey.isLocal() else PatchTypes.loRmt )
   else:
      entry.status = getLocalConnectorProgrammedStatus( connKey.serviceIntfId() )

   return entry

labelSources = { PwConnectorType.bgp: "EVPN",
                 PwConnectorType.ldp: "LDP" }

def getLabelEntry( routeKey ):
   # Gets the label entry related to the connectorkey from the status
   label = routeKey.topLabel
   labelEntry = PwaModel.LabelForwardingInfo( label=label )

   pwRoute = pwLfib.lfibRoute.get( routeKey )
   if not pwRoute:
      return None

   pwViaSet = pwLfib.viaSet.get( pwRoute.viaSetKey )
   if not pwViaSet:
      return None

   pwVia = pwLfib.pwVia.get( pwViaSet.viaKey[ 0 ] )
   if not pwVia:
      return None

   connKey = pwaLabelBinding.connectorKey.get( label )
   if not connKey:
      return None

   labelEntry.patch, localConnKey = getConnectorPatchInfo( connKey )
   if not labelEntry.patch or not localConnKey or not localConnKey.isValid():
      return None

   labelEntry.outConnInfo = getLocalConnectorFwdingInfo(
      localConnKey, decapCw=pwVia.controlWordPresent )

   if toggleMplsEvpnVpwsP25Enabled():
      # new API
      labelEntry.status = getRemotePseudowireProgrammedStatus(
                              localConnKey.serviceIntfId() )
   else:
      # legacy API
      labelEntry.status = getRemotePseudowireProgrammedStatusLegacyApi(
                              label, pwVia.legacyPwIntfId )

   labelEntry.source = labelSources.get( connKey.connectorType )
   if pwVia.flowLabelPresent:
      labelEntry.flowLabel = 'In'
   labelEntry.pseudowireType = pwVia.pwType

   return labelEntry

def getTaggedForwardingTable():
   # Creates a table with all the entries for tagged forwarding 
   # that are up from the status
   taggedEntries = {}

   if pwStatus.acToDestIntfEncap:
      # use new API
      for connKey, connStatus in pwaLocalConnectorStatusColl.connectorStatus.items():
         if not connKey.isLegacyTagged():
            continue
         if connStatus.status not in ( 'up', 'errdisabledUp' ):
            continue
         taggedEntry = getLocalEntry( connKey )
         if taggedEntry:
            taggedEntries[ taggedEntry.inConnInfo.toStr() ] = taggedEntry
   else:
      for pwIntfId in pwStatus.encapAdjacency.keys():
         if getConnectorType( pwIntfId ) != "tagged":
            continue
         taggedEntry = getLocalEntryLegacyApi( pwIntfId )
         if taggedEntry:
            taggedEntries[ taggedEntry.inConnInfo.toStr() ] = taggedEntry

   return PwaModel.TaggedForwardingInfoTable( taggedForwardingInfos=taggedEntries )

def getPortForwardingTable():
   # Creates a table with all the entries for port forwarding 
   # that are up from the status
   portEntries = {}

   if pwStatus.acToDestIntfEncap:
      # use new API
      for connKey, connStatus in pwaLocalConnectorStatusColl.connectorStatus.items():
         if connKey.isLegacyTagged():
            continue
         if connStatus.status not in ( 'up', 'errdisabledUp' ):
            continue
         portEntry = getLocalEntry( connKey )
         if portEntry:
            portEntries[ portEntry.inConnInfo.toStr() ] = portEntry
   else:
      # use legacy API
      for pwIntfId in pwStatus.encapAdjacency.keys():
         if getConnectorType( pwIntfId ) != "port":
            continue
         portEntry = getLocalEntryLegacyApi( pwIntfId )
         if portEntry:
            portEntries[ portEntry.inConnInfo.toStr() ] = portEntry

   return PwaModel.PortForwardingInfoTable( portForwardingInfos=portEntries )

def getLabelForwardingTable():
   # Creates a table with all the entries for label forwarding
   # that are up from the status
   labelEntries = {}

   if pwLfib:
      # Only look for routes if pwStatus has been instantiated.
      for routeKey in pwLfib.lfibRoute:
         labelEntry = getLabelEntry( routeKey )
         if labelEntry is not None:
            labelEntries[ labelEntry.label ] = labelEntry

   return PwaModel.LabelForwardingInfoTable( labelForwardingInfos=labelEntries )

#--------------------------------------------------------------------------------
# show patch panel forwarding
#--------------------------------------------------------------------------------
class PatchPanelForwardingCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show patch panel forwarding'
   data = {
      'patch': nodePatch,
      'panel': matcherPanel,
      'forwarding': 'Show patch forwarding information',
   }
   cliModel = PwaModel.PatchForwardingTable

   @staticmethod
   def handler( mode, args ):
      taggedTable = getTaggedForwardingTable()
      portTable = getPortForwardingTable()
      labelTable = getLabelForwardingTable()

      return PwaModel.PatchForwardingTable( taggedForwardingInfoTable=taggedTable,
                                            portForwardingInfoTable=portTable,
                                            labelForwardingInfoTable=labelTable )

BasicCli.addShowCommandClass( PatchPanelForwardingCmd )

#-------------------------------------------------------------------------------
# Commands to be run for 'show tech-support'
#-------------------------------------------------------------------------------

def _showTechCmds():
   cmds = []
   if pwHwCapability.mplsPseudowireSupported:
      cmds += [ 'show patch panel detail' ]

   return cmds

def _showTechSummaryCmds():
   cmds = []
   if pwHwCapability.mplsPseudowireSupported:
      cmds += [ 'show patch panel summary' ]

   return cmds

# Timestamps are made up to maintain historical order within show tech-support
timeStamp = '2016-12-06 10:00:00'
TechSupportCli.registerShowTechSupportCmdCallback( timeStamp, _showTechCmds,
      summaryCmdCallback=_showTechSummaryCmds )

#-------------------------------------------------------------------------------
# Mount necessary mountpoints for show commands
#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global pwConfig
   global pwStatus
   global pwLfib
   global pwaPatchPanel
   global pwaLocalConnectorStatusColl, pwaRemoteConnectorStatusColl
   global pwHwCapability, pwHwDir
   global pwaLdpRequest
   global pwaLdpResponse
   global pwaBgpRequest
   global pwaBgpResponse
   global pwaLabelBinding
   global pwaEbraResponse

   # Pseudowire Cli config in Sysdb
   pwConfig = LazyMount.mount( entityManager, 'pseudowire/config',
                                'Pseudowire::Config', 'r' )
   # Pseudowire agent's published status to ale/hw
   pwStatus = LazyMount.mount( entityManager, 'pseudowire/status',
                               'Pseudowire::Status', 'r' )

   shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
   pwLfib = shmemEm.doMount( 'mpls/protoLfibInputDir/pseudowire', 'Mpls::LfibStatus',
                             Smash.mountInfo( 'reader' ) )

   # Pseudowire's internal entities for relaying state between state machines
   pwaPatchPanel = LazyMount.mount( entityManager, 'pseudowire/agent/patch',
                                     'Pseudowire::PatchPanel', 'r' )

   pwaLocalConnectorStatusColl = LazyMount.mount( entityManager, 
                                       'pseudowire/agent/localConnectorStatusColl',
                                       'PseudowireAgent::LocalConnectorStatusColl',
                                       'r' )
   pwaRemoteConnectorStatusColl = LazyMount.mount( entityManager, 
                                       'pseudowire/agent/remoteConnectorStatusColl',
                                       'Pseudowire::RemoteConnectorStatusColl',
                                       'r' )

   pwaLabelBinding = LazyMount.mount( entityManager,
                                      'pseudowire/agent/labelBinding',
                                      'Pseudowire::LabelBindingColl',
                                      'r' )

   # Pseudowire agent's requests to and responses from Ldp agent to setup
   # targeted sessions
   pwaLdpRequest = LazyMount.mount( entityManager,
                           "pseudowire/ldp/config",
                           "Pseudowire::PseudowireLdpConfig",
                           "r" )
   pwaLdpResponse = LazyMount.mount( entityManager,
                           "pseudowire/ldp/status",
                           "Pseudowire::PseudowireLdpStatusPtr",
                           "r" )

   # Pseudowire agent's requests to and responses from BGP agent
   pwaBgpRequest = LazyMount.mount( entityManager,
                           "pseudowire/bgp/config",
                           "Pseudowire::PseudowireBgpConfig",
                           "r" )
   pwaBgpResponse = LazyMount.mount( entityManager,
                           "pseudowire/bgp/status",
                           "Pseudowire::PseudowireBgpStatus",
                           "r" )

   # Internal vlanId allocation from Ebra for local connectors
   pwaEbraResponse = LazyMount.mount( entityManager,
                            "bridging/external/status",
                            "EbraExt::ResponseReservedVlanId",
                            "r" )

   # Indicates if pseudowire is supported on this platform
   pwHwCapability = LazyMount.mount( entityManager,
                                     "routing/hardware/pseudowire/capability",
                                     "Pseudowire::Hardware::Capability", "r" )
   # Lists unprogrammed encap and decap adjacencies
   if toggleMplsEvpnVpwsP25Enabled():
      pwHwDir = LazyMount.mount( entityManager,
                                 "interface/encap/hw/serviceIntf/status",
                                 "Ebra::ServiceIntfHwStatusColl", "r" )
   else:
      pwHwDir = LazyMount.mount( entityManager,
                                 "hardware/ale/pw",
                                 "Tac::Dir", "ri" )
