# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements PVLAN configuration.  In particular, it provides:
# -  the "[no] private-vlan { isolated|community } primary vlan" command
# -  the "show vlan private-vlan" command
# -  the "show pvlan mapping interfaces" command
# -  the "[no] switchport pvlan mapping" command
# -  the "[no] switchport trunk private-vlan secondary" command
# -  the "[no] pvlan mapping" command
#-------------------------------------------------------------------------------
import Tac, Arnet, CliParser, BasicCli, IntfCli
from Vlan import  vlanSetToCanonicalString
import LazyMount
import CliCommand
import CliMatcher
import ConfigMount
import Intf
import re
import MultiRangeRule
import ShowCommand
from .PvlanModel import VlanPrivateVlan
from EbraLib import privateVlanAllowed
from VlanCli import VlanConfigMode, Vlan, VlanSet
from VlanCli import SwitchportModelet, vlanMappingSupportedGuard
from VlanCli import getSwitchIntfConfigEvenIfDisabled, vlanIdMatcher
from VlanIntfCli import VlanIntfModelet

# Module globals set up by the Plugin function below
cliConfig = None
bridgingConfig = None
bridgingHwCapabilities = None
vlanIntfConfigDir = None
epochStatus = None

def bridgingSupportedGuard( mode, token ):
   if bridgingHwCapabilities.bridgingSupported:
      return None
   return CliParser.guardNotThisPlatform

#-------------------------------------------------------------------------------
# Rule to match the token class <vlan_set>, returning a corresponding VlanSet
# object.
#-------------------------------------------------------------------------------
def vlanIdListFunc( mode, grList ):
   return VlanSet( mode, vlanIds=grList.values(), 
                   vlanSetString=str( grList ) )

#-------------------------------------------------------------------------------
# The "[no] private-vlan { isolated|community } primary vlan", in "config-vlan" mode
#
# The full syntax of this command is:
#
#     private-vlan { isolated|community } primary vlan <vlan_id>
#     no private-vlan [ { isolated|community } primary vlan <vlan_id> ]
#     default private-vlan [ { isolated|community } primary vlan <vlan_id> ]
#-------------------------------------------------------------------------------
def setPrivateVlan( mode, args ):
   vlanType = args[ 'PVLAN_TYPE' ]
   vlanId = args[ 'VLAN_ID' ]
   mode.vlan.setPrivateVlan( mode, vlanType, vlanId )

def noPrivateVlan( mode, args ):
   mode.vlan.noPrivateVlan( mode )

def privateVlansSupportedGuard( mode, token ):
   if bridgingHwCapabilities.privateVlansSupported:
      if privateVlanAllowed( epochStatus ):
         return None
   return CliParser.guardNotThisPlatform

nodePrivateVlan = CliCommand.guardedKeyword( 'private-vlan',
   helpdesc='Configure a private VLAN',
   guard=privateVlansSupportedGuard )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
   helpdesc='Identifier for a Virtual LAN' )

class PrivateVlanPrimaryVlanVlanidCmd( CliCommand.CliCommandClass ):
   syntax = 'private-vlan PVLAN_TYPE primary vlan VLAN_ID'
   noOrDefaultSyntax = 'private-vlan ...'
   data = {
      'private-vlan': nodePrivateVlan,
      'vlan': matcherVlan,
      'PVLAN_TYPE': CliMatcher.EnumMatcher( {
            'isolated': 'Configure the VLAN as an isolated private VLAN',
            'community': 'Configure the VLAN as a community private VLAN',
      } ),
      'primary': 'Configure the associated primary VLAN',
      'VLAN_ID': vlanIdMatcher,
   }
   handler = setPrivateVlan
   noOrDefaultHandler = noPrivateVlan

VlanConfigMode.addCommandClass( PrivateVlanPrimaryVlanVlanidCmd )

#-------------------------------------------------------------------------------
# The "show vlan private-vlan" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show vlan private-vlan
#-------------------------------------------------------------------------------
def doShowPrivateVlan( mode, args ):
   vlans = Vlan.getAll( mode )
   ret = VlanPrivateVlan()

   for v in vlans:
      ports = []
      vc = bridgingConfig.vlanConfig.get( v.id )
      primaryVlan = vc.primaryVlan
      if vc and primaryVlan:
         # Get active ports from primary/secondary vlans
         ports = v.activePorts( mode, promoted=False )
         ports += Vlan( primaryVlan ).activePorts( mode, promoted=False )

         interfaces = set()
         for port in ports:
            interfaces.add( port.name )

         interfaces = Arnet.sortIntf( interfaces )

         # Build privateVlans dictionary with primary->secondary vlan lists
         # For example the output will be as below:
         #    3 - (4,isolated,'Et1,Et2'), (5,community,'Et2,Et3')
         #    6 - (7,isolated,'Et4,Et5')

         privateVlan = VlanPrivateVlan.PrivateVlans.PrivateVlan()
         privateVlan.secondaryVlanId = v.id
         privateVlan.vlanType = vc.vlanType
         privateVlan.interfaces = interfaces

         if primaryVlan not in ret.privateVlans:
            ret.privateVlans[ primaryVlan ] = VlanPrivateVlan.PrivateVlans()
         ret.privateVlans[ primaryVlan ].privateVlan.append( privateVlan )

   return ret

nodePrivateVlanForShow = CliCommand.guardedKeyword( 'private-vlan',
   helpdesc='Private VLAN information',
   guard=privateVlansSupportedGuard )

class VlanPrivateVlanCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show vlan private-vlan'
   data = {
      'vlan': 'Details on VLAN operation',
      'private-vlan': nodePrivateVlanForShow,
   }
   handler = doShowPrivateVlan
   cliModel = VlanPrivateVlan

BasicCli.addShowCommandClass( VlanPrivateVlanCmd )

#-------------------------------------------------------------------------------
# The "show interfaces private-vlan mapping" command, in "enable" mode.
#
# The full syntax of this command is:
#
#   show interfaces private-vlan mapping
#-------------------------------------------------------------------------------
pvlanShowDeprecated = CliCommand.Node(
   CliMatcher.KeywordMatcher(
      'private-vlan',
      helpdesc='Show interface private VLAN information' ),
   guard=privateVlansSupportedGuard,
   deprecatedByCmd='show pvlan mapping interfaces' )

pvlanShowKw = CliCommand.guardedKeyword( 'pvlan',
                                         'Show interface private VLAN information',
                                         privateVlansSupportedGuard )

mappingShowKw = CliMatcher.KeywordMatcher(
   'mapping',
   helpdesc='Private VLAN mapping information' )

def doShowPrivateVlanMapping( mode, args ):
   def convertToPrintForm( mappedVlans ):
      if not mappedVlans:
         return 'NONE'
      elif mappedVlans == '1-4094':
         return 'ALL'
      else:
         return mappedVlans

   fmt = '%-9s    %-15s'
   print 'Interface    Secondary Vlans'
   print '---------    ---------------'
   vlans = []
   for name in vlanIntfConfigDir.intfConfig:
      m = re.match( 'Vlan(\d+)$', name )
      vlans.append( int( m.group( 1 ) ) )

   vlans.sort()
   for vlanId in vlans:
      if cliConfig.sviMapping.get( vlanId ):
         mappedPrivateVlans = vlanSetToCanonicalString(
            cliConfig.sviMapping[ vlanId ].secondaryVlan )
      else:
         mappedPrivateVlans = 'ALL'
      print fmt % ( ( "Vlan" + str( vlanId ) ),
                    convertToPrintForm( mappedPrivateVlans ) )

class ShowIntfPVlan( IntfCli.ShowIntfCommand ):
   syntax = 'show interfaces private-vlan mapping'
   data = { 'private-vlan' : pvlanShowDeprecated,
            'mapping' : mappingShowKw }
   handler = doShowPrivateVlanMapping

BasicCli.addShowCommandClass( ShowIntfPVlan )

class ShowPVlanMapping( ShowCommand.ShowCliCommandClass ):
   syntax = "show pvlan mapping interfaces [ INTF ]"
   data = {
      'pvlan' : pvlanShowKw,
      'mapping' : mappingShowKw,
      'interfaces' : 'Private VLAN mapping interface information',
      'INTF' : Intf.IntfRange.IntfRangeMatcher()
   }

   handler = doShowPrivateVlanMapping

BasicCli.addShowCommandClass( ShowPVlanMapping )

#-------------------------------------------------------------------------------
# The "[no] switchport pvlan mapping" command, in "config-if" mode.
#
# legacy:
# The "[no] switchport private-vlan mapping" command, in "config-if" mode.
#
#
# The full syntax of this command is:
#
#   switchport pvlan mapping <vlanSet>
#   switchport pvlan mapping add <vlanSet>
#   switchport pvlan mapping remove <vlanSet>
#   {no|default} pvlan mapping [ [add|remove] <vlanSet> ]
#-------------------------------------------------------------------------------
privateVlanSet = MultiRangeRule.MultiRangeMatcher(
   lambda: ( 1, 4094 ),
   False,
   'Secondary VLAN IDs of the private VLAN mapping',
   value=vlanIdListFunc )
matcherSwitchport = CliMatcher.KeywordMatcher( 'switchport',
   helpdesc='Set switching mode characteristics' )
nodePvlan = CliCommand.guardedKeyword( 'pvlan',
   helpdesc='Configure a private VLAN',
   guard=privateVlansSupportedGuard )
matcherPrivateVlan = CliMatcher.KeywordMatcher( 'private-vlan',
   helpdesc='Set the private VLAN configuration' )
matcherMapping = CliMatcher.KeywordMatcher( 'mapping',
   helpdesc='Set the private VLAN mapping' )

def setPrivateVlanMapping( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   vlanSet = args[ 'VLAN_SET' ]
   vlanOp = args.get( 'OP' )
   if not vlanOp:
      vlanOp = 'set'
   if vlanOp == 'set':
      sic.privateVlanMapping.clear()
      for i in vlanSet.ids:
         sic.privateVlanMapping[ i ] = True
   elif vlanOp == 'add':
      if sic.defaultPrivateVlanMapping:
         for i in range( 1, 4095 ):
            sic.privateVlanMapping[ i ] = True
      for i in vlanSet.ids:
         sic.privateVlanMapping[ i ] = True
   elif vlanOp == 'remove':
      if sic.defaultPrivateVlanMapping:
         for i in range( 1, 4095 ):
            sic.privateVlanMapping[ i ] = True
      for i in vlanSet.ids:
         del sic.privateVlanMapping[ i ]
   sic.defaultPrivateVlanMapping = False

def noPrivateVlanMapping( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.defaultPrivateVlanMapping = True

class SwitchportPrivatevlanMappingCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport ( pvlan | private-vlan ) mapping [ OP ] VLAN_SET'
   noOrDefaultSyntax = 'switchport ( pvlan | private-vlan ) mapping ...'
   data = {
      'switchport': matcherSwitchport,
      'pvlan': nodePvlan,
      'private-vlan': CliCommand.Node( matcher=matcherPrivateVlan,
         deprecatedByCmd='switchport pvlan mapping',
         guard=privateVlansSupportedGuard ),
      'mapping': matcherMapping,
      'OP': CliMatcher.EnumMatcher( {
        'add': 'Add VLANs to the current list',
        'remove': 'Remove VLANs from the current list',
       } ),
      'VLAN_SET': privateVlanSet,
   }

   handler = setPrivateVlanMapping
   noOrDefaultHandler = noPrivateVlanMapping

SwitchportModelet.addCommandClass( SwitchportPrivatevlanMappingCmd )


#-------------------------------------------------------------------------------
# "[no|default] switchport trunk private-vlan secondary" command in "config-if" mode.
#-------------------------------------------------------------------------------
nodeSecondary = CliCommand.guardedKeyword( 'secondary',
   helpdesc='Enable secondary VLAN mapping',
   guard=vlanMappingSupportedGuard )
nodeTrunk = CliCommand.guardedKeyword( 'trunk',
   helpdesc='Set trunking characteristics of the interface',
   guard=bridgingSupportedGuard )

def setTrunkPrivateVlanMapping( mode, args ):
   sic = getSwitchIntfConfigEvenIfDisabled( mode )
   if not sic:
      return
   sic.trunkPrivateVlanMapping = not CliCommand.isNoOrDefaultCmd( args )

class SwitchportTrunkPrivateVlanSecondaryCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport trunk private-vlan secondary'
   noOrDefaultSyntax = 'switchport trunk private-vlan secondary ...'
   data = {
      'switchport': matcherSwitchport,
      'trunk': nodeTrunk,
      'private-vlan': nodePrivateVlan,
      'secondary': nodeSecondary,
   }
   handler = setTrunkPrivateVlanMapping
   noOrDefaultHandler = setTrunkPrivateVlanMapping

SwitchportModelet.addCommandClass( SwitchportTrunkPrivateVlanSecondaryCmd )

#-------------------------------------------------------------------------------
# new syntax:
# The "[no] pvlan mapping" command, in "config-vlan-if" mode.
#
# legacy:
# The "[no] private-vlan mapping" command, in "config-vlan-if" mode.
#
# The full syntax of this command is:
#
#   pvlan mapping <vlanSet>
#   pvlan mapping add <vlanSet>
#   pvlan mapping remove <vlanSet>
#   {no|default} pvlan mapping [ [add|remove] <vlanSet> ]
#-------------------------------------------------------------------------------
def setSviPrivateVlanMapping( mode, args ):
   sviMappingState = cliConfig.sviMapping.get( mode.intf.vlanId )
   vlanSet = args[ 'VLAN_SET' ]
   vlanOp = args.get( 'OP' )
   if not vlanOp:
      vlanOp = 'set'
   if not sviMappingState:
      if ( vlanOp == 'add' or
           ( vlanOp == 'set' and len( vlanSet.ids ) == 4094 ) ):
         # nothing to do
         return

      sviMappingState = cliConfig.sviMapping.newMember( mode.intf.vlanId )
      for i in range( 1, 4095 ):
         sviMappingState.secondaryVlan[ i ] = True

   if vlanOp == 'set':
      sviMappingState.secondaryVlan.clear()
      for i in vlanSet.ids:
         sviMappingState.secondaryVlan[ i ] = True
   elif vlanOp == 'add':
      for i in vlanSet.ids:
         sviMappingState.secondaryVlan[ i ] = True
   elif vlanOp == 'remove':
      for i in vlanSet.ids:
         del sviMappingState.secondaryVlan[ i ]
   sviMappingState.defaultSviMapping = False

def noSviPrivateVlanMapping( mode, args ):
   sviMappingState = cliConfig.sviMapping.get( mode.intf.vlanId )
   if sviMappingState:
      sviMappingState.defaultSviMapping = True
   del cliConfig.sviMapping[ mode.intf.vlanId ]

class PvlanMappingCmd( CliCommand.CliCommandClass ):
   syntax = '( pvlan | private-vlan ) mapping [ OP ] VLAN_SET'
   noOrDefaultSyntax = '( pvlan | private-vlan ) mapping ...'
   data = {
      'pvlan': nodePvlan,
      'private-vlan': CliCommand.Node( matcher=matcherPrivateVlan,
         deprecatedByCmd='pvlan mapping',
         guard=privateVlansSupportedGuard ),
      'mapping': matcherMapping,
      'OP': CliMatcher.EnumMatcher( {
         'add': 'Add VLANs to the current list',
         'remove': 'Remove VLANs from the current list',
       } ),
      'VLAN_SET': privateVlanSet,
   }

   handler = setSviPrivateVlanMapping
   noOrDefaultHandler = noSviPrivateVlanMapping

VlanIntfModelet.addCommandClass( PvlanMappingCmd )

def Plugin( entityManager ):
   global cliConfig, bridgingConfig
   global bridgingHwCapabilities
   global vlanIntfConfigDir
   global epochStatus

   cliConfig = ConfigMount.mount( entityManager, "bridging/input/config/cli", 
                                  "Bridging::Input::CliConfig", "w" )
   mount = LazyMount.mount
   bridgingHwCapabilities = mount( entityManager, "bridging/hwcapabilities",
                           "Bridging::HwCapabilities", "r" )
   bridgingConfig = mount( entityManager, "bridging/config",
                           "Bridging::Config", "r" )
   vlanIntfConfigDir = mount( entityManager, "interface/config/eth/vlan",
                              "Interface::VlanIntfConfigDir", "r" )
   epochStatus = mount( entityManager, "hwEpoch/status", "HwEpoch::Status", "r" )

