# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import IpGenAddrMatcher
import Tac
from VlanCli import SwitchportModelet

from CliMode.PtpUnicastNegotiation import PtpUnicastNegotiationMode
import CliPlugin.PtpCli as PtpCli

# Globals
ptpConfig = None

# Constants
UnicastNegotiationConstants = Tac.Type( 'Ptp::Constants::UnicastNegotiation' )
defaultProfileName = 'default'

#-------------------------------------------------------------------------------
# Functions to create, delete and modify unicast negotiation related configs
#-------------------------------------------------------------------------------

def setMsgTypeAttr( profile, msgType, attr, value=None, useDefaultValue=False ):
   """Set the attribute value to the corresponding UcastNegMsgProfile

   Args:
     profile: profile name
     msgType: Expecting one of the following:
                 * announce
                 * sync
                 * delayResp
     attr: The UcastNegMsgProfile Attribute to modify.
           Expecting one of the following:
              * logInterval
              * duration
     value: New value of the attribute. This is ignored if useDefaultValue is True
     useDefaultValue: Whether we use the attribute default value or not

   Returns:
      Nothing
   """

   def newMsgProfile( oldMsgProfile ):
      logInterval = oldMsgProfile.logInterval
      duration = oldMsgProfile.duration
      if attr == 'logInterval':
         logInterval = value
      elif attr == 'duration':
         duration = value
      return Tac.Value( "Ptp::UcastNegMsgProfile", logInterval, duration )

   if useDefaultValue:
      value = getattr( UnicastNegotiationConstants, attr + "Default" )

   oldProfile = ptpConfig.ucastNegProfile[ profile ]
   announceProfile = oldProfile.announceProfile
   syncProfile = oldProfile.syncProfile
   delayRespProfile = oldProfile.delayRespProfile
   if msgType == 'announce':
      announceProfile = newMsgProfile( announceProfile )
   elif msgType == 'sync':
      syncProfile = newMsgProfile( syncProfile )
   elif msgType == 'delayResp':
      delayRespProfile = newMsgProfile( delayRespProfile )
   newProfile = Tac.Value( "Ptp::UcastNegProfile",
                           announceProfile,
                           syncProfile,
                           delayRespProfile )
   ptpConfig.ucastNegProfile[ profile ] = newProfile

#-------------------------------------------------------------------------------
# The "config-unicast-negotiation-profile" mode.
#-------------------------------------------------------------------------------
class UnicastNegotiationConfigMode( PtpUnicastNegotiationMode,
                                    BasicCli.ConfigModeBase ):

   # Attributes required of every Mode class.
   name = 'PTP Unicast Negotiation profile configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, profile ):
      self.profile = profile
      PtpUnicastNegotiationMode.__init__( self, profile )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-------------------------------------------------------------------------------
# In "config" mode
#
# [no|default] ptp unicast-negotiation profile <name>
#-------------------------------------------------------------------------------

def gotoProfileMode( mode, args ):
   """ Create the unicast negotiation profile and enter in its session """
   profile = args[ 'PROFILE' ]
   if profile == defaultProfileName:
      errMsg = "Profile name " + profile + " is reserved."
      mode.addError( errMsg )
      return

   childMode = mode.childMode( UnicastNegotiationConfigMode, profile=profile )
   if profile not in ptpConfig.ucastNegProfile:
      ptpConfig.ucastNegProfile[ profile ] = Tac.Value( "Ptp::UcastNegProfile" )
   mode.session_.gotoChildMode( childMode )

def deleteProfile( mode, args ):
   """ Delete the unicast negotiation profile """
   profile = args[ 'PROFILE' ]
   if profile == defaultProfileName:
      errMsg = "Profile name " + profile + " cannot be removed."
      mode.addError( errMsg )
      return

   del ptpConfig.ucastNegProfile[ profile ]

unicastNegotiationMatcher = CliMatcher.KeywordMatcher(
      'unicast-negotiation',
      helpdesc='Configure PTP Unicast Negotiation' )
profileMatcher = CliMatcher.KeywordMatcher(
      'profile',
      helpdesc='Configure a Unicast Negotiation profile' )
profileNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ptpConfig.ucastNegProfile, 'Unicast Negotiation profile name' )

#--------------------------------------------------------------------------------
# [ no | default ] ptp unicast-negotiation profile PROFILE ...
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationProfileCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp unicast-negotiation profile PROFILE ...'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'unicast-negotiation': unicastNegotiationMatcher,
      'profile': profileMatcher,
      'PROFILE': profileNameMatcher,
   }
   handler = gotoProfileMode
   noOrDefaultHandler = deleteProfile

BasicCliModes.GlobalConfigMode.addCommandClass( PtpUnicastNegotiationProfileCmd )

#-------------------------------------------------------------------------------
# In "unicast-negotiation-profile" mode
#
# [no|default] announce|delay-resp|sync interval|duration VALUE
#-------------------------------------------------------------------------------
durationMatcher = CliMatcher.IntegerMatcher(
      UnicastNegotiationConstants.minDuration,
      UnicastNegotiationConstants.maxDuration,
      helpdesc='duration in secs' )

announceIntervalMatcher = CliMatcher.IntegerMatcher(
      UnicastNegotiationConstants.minlogAnnounceInterval,
      UnicastNegotiationConstants.maxlogAnnounceInterval,
      helpdesc='Log of announce message interval (secs)' )

syncIntervalMatcher = CliMatcher.IntegerMatcher(
      UnicastNegotiationConstants.minlogSyncInterval,
      UnicastNegotiationConstants.maxlogSyncInterval,
      helpdesc='Log of sync message interval (secs)' )

delayRespIntervalMatcher = CliMatcher.IntegerMatcher(
      UnicastNegotiationConstants.minlogDelayRespInterval,
      UnicastNegotiationConstants.maxlogDelayRespInterval,
      helpdesc='Log of delay response message interval (secs)' )

msgTypeHelpDesc = {
   'announce': 'Configure announce message parameters',
   'delay-resp': 'Configure delay response message parameters',
   'sync': 'Configure sync message parameters'
}

attributeHelpDesc = {
   'interval': 'Log base 2 of the message interval',
   'duration': 'Duration for how long the message will be sent',
}

def addUnicastNegoCmd( msgName, msgType, attrName, attrType, valueMatcher ):
   class Cmd( CliCommand.CliCommandClass ):
      syntax = '%s %s VALUE ...' % ( msgName, attrName )
      noOrDefaultSyntax = '%s %s...' % ( msgName, attrName )
      data = {
         msgName: msgTypeHelpDesc[ msgName ],
         attrName: attributeHelpDesc[ attrName ],
         'VALUE': valueMatcher
      }

      @staticmethod
      def handler( mode, args ):
         setMsgTypeAttr( mode.profile, msgType, attrType, value=args[ 'VALUE' ] )

      @staticmethod
      def noOrDefaultHandler( mode, args ):
         setMsgTypeAttr( mode.profile, msgType, attrType, useDefaultValue=True )

   UnicastNegotiationConfigMode.addCommandClass( Cmd )

for i in ( ( 'announce', 'announce', announceIntervalMatcher ),
           ( 'delay-resp', 'delayResp', delayRespIntervalMatcher ),
           ( 'sync', 'sync', syncIntervalMatcher ) ):
   addUnicastNegoCmd( i[ 0 ], i[ 1 ], 'interval', 'logInterval', i[ 2 ] )
   addUnicastNegoCmd( i[ 0 ], i[ 1 ], 'duration', 'duration', durationMatcher )

#-------------------------------------------------------------------------------
# In "interface" mode
#
# [no] ptp unicast-negotiation candidate-grantor <IP address> [profile <name>]
#-------------------------------------------------------------------------------

def setIpGrantorGrantee( mode, grantor, ipGenAddr, profile=None, no=False ):
   # Get the IpGenAddr if it's a IpGenAddrWirhMask to check if it's a
   # unicast address
   ip = ipGenAddr.ipGenAddr if hasattr( ipGenAddr, 'ipGenAddr' ) else ipGenAddr
   if not ip.isUnicast:
      mode.addError( "\"{}\" is not a Unicast IP address".format( ip ) )
      return

   # Check transport mode and ip address
   ( intfConfig, _ ) = PtpCli.setIntfDefaults( mode.intf.name )
   if ipGenAddr.af == 'ipv4' and intfConfig.transportMode != 'ipv4':
      mode.addError( "Transport Mode is not IPv4" )
      return
   elif ipGenAddr.af == 'ipv6' and intfConfig.transportMode  != 'ipv6':
      mode.addError( "Transport Mode is not IPv6" )
      return

   if grantor:
      profileMap = intfConfig.ucastNegGrantorIpToProfileMap
   else:
      profileMap = intfConfig.ucastNegGranteeIpToProfileMap

   if no:
      if profile is None:
         # no ptp unicast-negotiation candidate-grantor <ip>
         del profileMap[ ipGenAddr ]
      else:
         # no ptp unicast-negotiation candidate-grantor <ip> profile
         profileMap[ ipGenAddr ] = defaultProfileName
   else:
      if profile is None:
         # ptp unicast-negotiation candidate-grantor <ip>
         profileMap[ ipGenAddr ] = defaultProfileName
      else:
         # ptp unicast-negotiation candidate-grantor <ip> profile <name>
         if profile not in ptpConfig.ucastNegProfile:
            msg = """"{}" profile doesn't exist. Default values will be used
                  until the profile is added""".format( profile )
            mode.addWarning( msg )
         profileMap[ ipGenAddr ] = profile

def setIpCandidateGrantor( mode, ipGenAddr, profile=None, no=False ):
   setIpGrantorGrantee( mode, True, ipGenAddr, profile, no )

class IpCandidateGrantorCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp unicast-negotiation candidate-grantor IP [ [ profile ] PROFILE ]'
   noOrDefaultSyntax = 'ptp unicast-negotiation candidate-grantor IP ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'unicast-negotiation': unicastNegotiationMatcher,
      'candidate-grantor': 'Configure a grantor profile',
      'IP': IpGenAddrMatcher.ipGenAddrMatcher,
      'profile': profileMatcher,
      'PROFILE': profileNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      setIpGrantorGrantee( mode, True, args[ 'IP' ], args.get( 'PROFILE' ), False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setIpGrantorGrantee( mode, True, args[ 'IP' ], None, True )

SwitchportModelet.addCommandClass( IpCandidateGrantorCmd )

#-------------------------------------------------------------------------------
# In "interface" mode
#
# [no] ptp unicast-negotiation remote-grantee <IP address with mask> [profile <name>]
#-------------------------------------------------------------------------------
def setIpRemoteGrantee( mode, ipGenWithMask, profile=None, no=False ):
   setIpGrantorGrantee( mode, False, ipGenWithMask, profile, no )

class IpRemoteGranteeCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp unicast-negotiation remote-grantee IP [ [ profile ] PROFILE ]'
   noOrDefaultSyntax = 'ptp unicast-negotiation remote-grantee IP ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'unicast-negotiation': unicastNegotiationMatcher,
      'remote-grantee': 'Configure a profile to a grantee IP address range',
      'IP': IpGenAddrMatcher.IpGenPrefixMatcher( 'IP address with mask',
                                                 addrWithMask=True ),
      'profile': profileMatcher,
      'PROFILE': profileNameMatcher
   }
   @staticmethod
   def handler( mode, args ):
      setIpGrantorGrantee( mode, False, args[ 'IP' ], args.get( 'PROFILE' ), False )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      setIpGrantorGrantee( mode, False, args[ 'IP' ], None, True )

SwitchportModelet.addCommandClass( IpRemoteGranteeCmd )

def Plugin( entityManager ):
   global ptpConfig
   ptpConfig = ConfigMount.mount( entityManager, "ptp/config", "Ptp::Config", "w" )
