#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import Tac
import CliPlugin.PtpCli as PtpCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin.PtpGlobalConfigMode import ( matcherMessageType, matcherDscp,
                                            matcherDscpValue )

matcherAnnounce = CliMatcher.KeywordMatcher( 'announce',
      helpdesc='Configure announce message parameters' )
matcherInterval = CliMatcher.KeywordMatcher( 'interval',
      helpdesc='Interval between messages, base 2 log' )
matcherSync = CliMatcher.KeywordMatcher( 'sync',
      helpdesc='Configure sync message parameters' )
matcherTimeout = CliMatcher.KeywordMatcher( 'timeout',
      helpdesc='Announce message interval timeout multiplier' )

#--------------------------------------------------------------------------------
# [ no | default ] ptp enable
#--------------------------------------------------------------------------------
class PtpEnableCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp enable'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'enable': 'Enable PTP on this port',
   }
   handler = lambda mode, args: PtpCli.enablePtp( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.enablePtp( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpEnableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp enable
#--------------------------------------------------------------------------------
class PtpEnableSyncTestCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp enable synctest'
   noOrDefaultSyntax = syntax
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'enable': 'Enable PTP on this port',
      'synctest': 'Puts interface in synctest mode'
   }
   hidden = True
   handler = lambda mode, args: PtpCli.enableSyncTest( mode )
   noOrDefaultHandler = lambda mode, args: PtpCli.enableSyncTest( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpEnableSyncTestCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp delay-mechanism VALUE
#--------------------------------------------------------------------------------
class PtpDelayMechanismCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp delay-mechanism VALUE'
   noOrDefaultSyntax = 'ptp delay-mechanism'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'delay-mechanism': 'Delay mechanism in boundary clock mode',
      'VALUE': CliMatcher.EnumMatcher( {
         'e2e': 'Delay req/resp mechanism',
         'p2p': 'Peer-to-peer mechanism',
      } )
   }
   handler = lambda mode, args: PtpCli.setDelayMechanism( mode,
         'delayMechanism', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setDelayMechanism( mode,
         'delayMechanism', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpDelayMechanismCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp announce interval VALUE
#--------------------------------------------------------------------------------
class PtpAnnounceIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp announce interval VALUE'
   noOrDefaultSyntax = 'ptp announce interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'announce': matcherAnnounce,
      'interval': matcherInterval,
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.announceIntervalRange,
         helpdesc='Log of announce message interval (secs)' ),
   }

   @staticmethod
   def handler( mode, args ):
      if PtpCli.ptpStatus.g82751Enabled:
         mode.addWarning( "Announce message rate is fixed to 8/s "
                          "in PTP profile g8275.1" ) 
      PtpCli.setIntfAttr( mode, 'logAnnounceInterval', args[ 'VALUE' ] )

   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logAnnounceInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpAnnounceIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp delay-req interval VALUE
#--------------------------------------------------------------------------------
class PtpDelayReqIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp delay-req interval VALUE'
   noOrDefaultSyntax = 'ptp delay-req interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'delay-req': 'Configure delay request message parameters',
      'interval': matcherInterval,
      'VALUE': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.minlogMinDelayReqInterval,
         PtpCli.CliConstants.maxlogMinDelayReqInterval,
         helpdesc='Log of delay req interval (secs)' ),
   }

   @staticmethod
   def handler( mode, args ):
      if PtpCli.ptpStatus.g82751Enabled:
         mode.addWarning( "Delay-req message rate is fixed to 16/s "
                          "in PTP profile g8275.1" ) 
      PtpCli.setIntfAttr( mode, 'logMinDelayReqInterval', args[ 'VALUE' ] )

   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logMinDelayReqInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpDelayReqIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp local-priority VALUE
#--------------------------------------------------------------------------------
class PtpLocalPriorityValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp local-priority VALUE'
   noOrDefaultSyntax = 'ptp local-priority ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'local-priority': ( 'Set the value of interface local priority '
                          '( G8275 profile BMCA )' ),
      'VALUE': CliMatcher.IntegerMatcher( 1, 255, helpdesc='Local Priority' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'localPriority', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'localPriority', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpLocalPriorityValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp message-type event dscp VALUE
#--------------------------------------------------------------------------------
class PtpMessageTypeEventDscpValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp message-type event dscp VALUE'
   noOrDefaultSyntax = 'ptp message-type event dscp ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'message-type': matcherMessageType,
      'event': 'Configure ptp event message attributes',
      'dscp': matcherDscp,
      'VALUE': matcherDscpValue
   }
   handler = lambda mode, args: PtpCli.setIntfDscpEvent( mode, args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfDscpEvent( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpMessageTypeEventDscpValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp message-type general dscp VALUE
#--------------------------------------------------------------------------------
class PtpMessageTypeGeneralDscpValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp message-type general dscp VALUE'
   noOrDefaultSyntax = 'ptp message-type general dscp ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'message-type': matcherMessageType,
      'general': 'Configure ptp general message attributes',
      'dscp': matcherDscp,
      'VALUE': matcherDscpValue
   }
   handler = lambda mode, args: PtpCli.setIntfDscpGeneral( mode, args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfDscpGeneral( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpMessageTypeGeneralDscpValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp pdelay-neighbor-threshold VALUE
#--------------------------------------------------------------------------------
class PtpPdelayNeighborThresholdValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp pdelay-neighbor-threshold VALUE'
   noOrDefaultSyntax = 'ptp pdelay-neighbor-threshold ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'pdelay-neighbor-threshold': 'Configure link propagation threshold',
      'VALUE': CliMatcher.IntegerMatcher( 0, 10**10,
         helpdesc='Threshold in nanoseconds' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'linkDelayThreshold', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'linkDelayThreshold', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpPdelayNeighborThresholdValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp pdelay-req interval VALUE
#--------------------------------------------------------------------------------
class PtpPdelayReqIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp pdelay-req interval VALUE'
   noOrDefaultSyntax = 'ptp pdelay-req interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'pdelay-req': 'Configure peer delay request message parameters',
      'interval': matcherInterval,
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.pDelayIntervalRange,
         helpdesc='Log of peer delay req interval (secs)' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logMinPdelayReqInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'logMinPdelayReqInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpPdelayReqIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp role ROLE
#--------------------------------------------------------------------------------
class PtpRoleCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp role ROLE'
   noOrDefaultSyntax = 'ptp role ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'role': 'Role in boundary clock mode',
      'ROLE': CliMatcher.EnumMatcher( {
         'dynamic': 'PTP protocol determines role',
         'master': 'PTP master port',
      } )
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'role', args[ 'ROLE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'role', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpRoleCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp sync interval VALUE
#--------------------------------------------------------------------------------
class PtpSyncIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync interval VALUE'
   noOrDefaultSyntax = 'ptp sync interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync': matcherSync,
      'interval': CliCommand.Node( matcher=matcherInterval,
         deprecatedByCmd='[no] ptp sync-message interval' ),
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.syncIntervalRange,
         helpdesc='Log of sync interval (secs)' ),
   }
   handler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', args[ 'VALUE' ], deprecatedCmd=True )
   noOrDefaultHandler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', no=True, deprecatedCmd=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSyncIntervalValueCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp sync-message interval VALUE
#--------------------------------------------------------------------------------
class PtpSyncMessageIntervalValueCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync-message interval VALUE'
   noOrDefaultSyntax = 'ptp sync-message interval ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync-message': 'Configure sync message parameters',
      'interval': matcherInterval,
      'VALUE': CliMatcher.DynamicIntegerMatcher( PtpCli.syncIntervalRange,
         helpdesc='Log of sync interval (secs)' ),
   }
   handler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', args[ 'VALUE' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setSyncInterval( mode,
         'logSyncInterval', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSyncMessageIntervalValueCmd )


#--------------------------------------------------------------------------------
# [ no | default ] ptp sync timeout TIMEOUT
#--------------------------------------------------------------------------------
class PtpSyncTimeoutTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp sync timeout TIMEOUT'
   noOrDefaultSyntax = 'ptp sync timeout ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'sync': matcherSync,
      'timeout': 'Sync message interval timeout multiplier',
      'TIMEOUT': CliMatcher.IntegerMatcher( 2, 255,
         helpdesc='Sync interval timeout multiplier' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'syncReceiptTimeout', args[ 'TIMEOUT' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'syncReceiptTimeout', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpSyncTimeoutTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp announce timeout TIMEOUT
#--------------------------------------------------------------------------------
class PtpAnnounceTimeoutTimeoutCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp announce timeout TIMEOUT'
   noOrDefaultSyntax = 'ptp announce timeout ...'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'announce': matcherAnnounce,
      'timeout': matcherTimeout,
      'TIMEOUT': CliMatcher.IntegerMatcher(
         PtpCli.CliConstants.minannounceReceiptTimeout,
         PtpCli.CliConstants.maxannounceReceiptTimeout,
         helpdesc='Announce interval timeout multiplier' ),
   }
   handler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'announceReceiptTimeout', args[ 'TIMEOUT' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setIntfAttr( mode,
         'announceReceiptTimeout', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpAnnounceTimeoutTimeoutCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp transport TRANSPORT
#--------------------------------------------------------------------------------
class PtpTransportCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp transport TRANSPORT'
   noOrDefaultSyntax = 'ptp transport'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'transport': 'Transport mode in boundary clock mode',
      'TRANSPORT': CliMatcher.EnumMatcher( {
         'layer2': 'Layer 2 messaging',
         'ipv4': 'Ipv4 messaging',
         'ipv6': 'Ipv6 messaging',
      } )
   }
   handler = lambda mode, args: PtpCli.setTransportMode( mode,
         'transportMode', args[ 'TRANSPORT' ] )
   noOrDefaultHandler = lambda mode, args: PtpCli.setTransportMode( mode,
         'transportMode', no=True )

VlanCli.SwitchportModelet.addCommandClass( PtpTransportCmd )

#-------------------------------------------------------------------------------
# ptp profile g8275.1 destination mac-address ( forwardable | non-forwardable )
#
# [no|default] no ptp profile g8275.1 destination mac-address
#-------------------------------------------------------------------------------
class MacAddressCmd( CliCommand.CliCommandClass ):
   syntax = "ptp profile g8275.1 destination mac-address \
             ( forwardable | non-forwardable )"
   noOrDefaultSyntax = "ptp profile g8275.1 destination mac-address ..."
   data = {
      'ptp' : PtpCli.ptpMatcherForConfig,
      'profile' : "PTP profile",
      'g8275.1' : "G8275.1 profile",
      'destination': "Set destination Mac address",
      'mac-address': "Set destination Mac address",
      'forwardable': CliMatcher.KeywordMatcher( 'forwardable', helpdesc='set \
            multicast address to 01:1B:19:00:00:00' ),
      'non-forwardable': CliMatcher.KeywordMatcher(
         'non-forwardable', helpdesc='set multicast address \
               to 01:80:C2:00:00:0E' )
   }

   @staticmethod
   def handler( mode, args ):
      EthAddr = Tac.Type( "Arnet::EthAddr" )
      if not PtpCli.ptpStatus.g82751Enabled:
         mode.addWarning( "Destination multicast address is only used "
                          "in PTP profile g8275.1" ) 
      if 'forwardable' in args:
         PtpCli.setIntfAttr( mode, 'pktTxMacAddress', EthAddr.ethAddrPtp )
      elif 'non-forwardable' in args:
         PtpCli.setIntfAttr( mode, 'pktTxMacAddress', EthAddr.ethAddrPtpPeerDelay )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      EthAddr = Tac.Type( "Arnet::EthAddr" )
      if not PtpCli.ptpStatus.g82751Enabled:
         mode.addWarning( "Destination multicast address is only used "
                          "in PTP profile g8275.1" ) 
      PtpCli.setIntfAttr( mode, 'pktTxMacAddress', EthAddr.ethAddrPtpPeerDelay )

VlanCli.SwitchportModelet.addCommandClass( MacAddressCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ptp vlan ( all | VLANSET )
#--------------------------------------------------------------------------------
class PtpVlanCmd( CliCommand.CliCommandClass ):
   syntax = 'ptp vlan ( all | VLAN_SET )'
   noOrDefaultSyntax = 'ptp vlan [ all | VLAN_SET ]'
   data = {
      'ptp': PtpCli.ptpMatcherForConfig,
      'vlan': 'Configure PTP to run on VLANs in trunk mode',
      'all': 'All allowed vlans on trunk',
      'VLAN_SET': VlanCli.vlanSetMatcher
   }
   handler = PtpCli.setPtpVlan
   noOrDefaultHandler = PtpCli.setPtpVlan

VlanCli.SwitchportModelet.addCommandClass( PtpVlanCmd )
