#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCli
import CliCommand
import CliMatcher
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliModel as AclCliModel
import CliPlugin.IntfCli as IntfCli
import CliPlugin.PtpCli as PtpCli
import CliPlugin.PtpCliModel as PtpCliModel
import CliPlugin.PtpDebugCounterModel as PtpDebugCounterModel
import CliPlugin.VlanCli as VlanCli
import CliToken.Ip
import ShowCommand

matcherInterface = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Details on PTP' )
matcherLocalClock = CliMatcher.KeywordMatcher( 'local-clock',
      helpdesc='Show PTP local clock information' )
matcherSource = CliMatcher.KeywordMatcher( 'source',
      helpdesc='Show the source IP for PTP' )
matcherUnicastNegotiation = CliMatcher.KeywordMatcher( 'unicast-negotiation',
      helpdesc='Show Unicast Negotiation information' )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Show PTP VLAN information' )

#--------------------------------------------------------------------------------
# show ptp
#--------------------------------------------------------------------------------
class PtpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
   }
   handler = PtpCli.showPtpSummary
   cliModel = PtpCliModel.PtpSummary

BasicCli.addShowCommandClass( PtpCmd )

#--------------------------------------------------------------------------------
# show ptp foreign-master-record
#--------------------------------------------------------------------------------
class PtpForeignMasterRecordCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp foreign-master-record'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'foreign-master-record': 'Show PTP foreign master information',
   }
   handler = PtpCli.showFMR
   cliModel = PtpCliModel.PtpForeignMasterRecords

BasicCli.addShowCommandClass( PtpForeignMasterRecordCmd )

#--------------------------------------------------------------------------------
# show ptp hold-ptp-time
#--------------------------------------------------------------------------------
class PtpHoldPtpTimeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp hold-ptp-time'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'hold-ptp-time': 'Show PTP hold ptp time interval information',
   }
   handler = PtpCli.showHoldPtpTime
   cliModel = PtpCliModel.PtpHoldPtpTimeForShow

BasicCli.addShowCommandClass( PtpHoldPtpTimeCmd )

#--------------------------------------------------------------------------------
# show ptp interface [ INTF ] [ enabled ]
#--------------------------------------------------------------------------------
class PtpInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTF ] [ enabled ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTF': IntfCli.Intf.rangeMatcher,
      'enabled': 'Displays only ptp enabled interfaces',
   }
   handler = PtpCli.showIntfList
   cliModel = PtpCliModel.PtpInterfaces

BasicCli.addShowCommandClass( PtpInterfaceCmd )

#--------------------------------------------------------------------------------
# show ptp interface INTF vlan VLAN_ID
#--------------------------------------------------------------------------------
class PtpInterfaceIntfVlanVlanidCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface INTF vlan VLAN_ID'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTF': IntfCli.Intf.rangeMatcher,
      'vlan': matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
   }
   handler = PtpCli.showIntfListVlan
   cliModel = PtpCliModel.PtpInterfaces

BasicCli.addShowCommandClass( PtpInterfaceIntfVlanVlanidCmd )

#--------------------------------------------------------------------------------
# show ptp interface [ INTF ] [ vlan VLAN_ID ] counters
#--------------------------------------------------------------------------------
class PtpInterfaceVlanVlanidCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTF ] [ vlan VLAN_ID ] counters'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTF': IntfCli.Intf.rangeMatcher,
      'vlan': matcherVlan,
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'counters': 'Show PTP counter information',
   }
   handler = PtpCli.showIntfListCounters
   cliModel = PtpCliModel.PtpCounters

BasicCli.addShowCommandClass( PtpInterfaceVlanVlanidCountersCmd )

#--------------------------------------------------------------------------------
# show ptp interface [ INTF ] counters drop
#--------------------------------------------------------------------------------
class PtpShowIntfCounterDrop( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTF ] counters drop'
   data = { 'ptp' : PtpCli.ptpMatcherForShow,
            'interface' : matcherInterface,
            'INTF' : IntfCli.Intf.rangeMatcher,
            'counters' : 'Show PTP counter information',
            'drop' : 'Show PTP drop counter information'
         }
   handler = PtpCli.showIntfDropCounters
   cliModel = PtpDebugCounterModel.PtpDebugCounters

BasicCli.addShowCommandClass( PtpShowIntfCounterDrop )

#--------------------------------------------------------------------------------
# show ptp interface [ INTF ] vlans
#--------------------------------------------------------------------------------
class PtpInterfaceIntfVlansCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp interface [ INTF ] vlans'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'interface': matcherInterface,
      'INTF': IntfCli.Intf.rangeMatcher,
      'vlans': 'Show PTP VLANs information',
   }
   handler = PtpCli.showIntfListVlans
   cliModel = PtpCliModel.PtpVlans

BasicCli.addShowCommandClass( PtpInterfaceIntfVlansCmd )

#--------------------------------------------------------------------------------
# show ptp ( clock | local-clock )
#--------------------------------------------------------------------------------
class PtpClockCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ( clock | local-clock )'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'clock': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'clock',
            helpdesc='Show PTP local clock information' ),
         deprecatedByCmd='show ptp local-clock' ),
      'local-clock': matcherLocalClock,
   }
   handler = lambda mode, args: PtpCli.showClock( mode )
   cliModel = PtpCliModel.PtpClock

BasicCli.addShowCommandClass( PtpClockCmd )

#--------------------------------------------------------------------------------
# show ptp ( masters | parent )
#--------------------------------------------------------------------------------
class PtpMastersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ( masters | parent )'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'masters': 'Show PTP master properties',
      'parent': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'parent',
            helpdesc='Show PTP parent properties' ),
         deprecatedByCmd='show ptp masters' ),
   }
   handler = PtpCli.showParent
   cliModel = PtpCliModel.PtpParentWrapper

BasicCli.addShowCommandClass( PtpMastersCmd )

#--------------------------------------------------------------------------------
# show ptp monitor
#--------------------------------------------------------------------------------
class PtpMonitorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp monitor'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'monitor': 'Show recent PTP data',
   }
   handler = PtpCli.showPtpMonitor
   cliModel = PtpCliModel.PtpMonitorData

BasicCli.addShowCommandClass( PtpMonitorCmd )

#--------------------------------------------------------------------------------
# show ptp source
#--------------------------------------------------------------------------------
class PtpSourceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp source'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'source': matcherSource,
   }
   handler = PtpCli.showSrcIp
   cliModel = PtpCliModel.PtpSourceIp

BasicCli.addShowCommandClass( PtpSourceCmd )

#--------------------------------------------------------------------------------
# show ptp source ip
#--------------------------------------------------------------------------------
class PtpSourceIpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp source ip'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'source': matcherSource,
      'ip': CliToken.Ip.ipMatcherForShow,
   }
   handler = PtpCli.showSrcIp4
   cliModel = PtpCliModel.PtpSourceIp

BasicCli.addShowCommandClass( PtpSourceIpCmd )

#--------------------------------------------------------------------------------
# show ptp local-clock time-properties
#--------------------------------------------------------------------------------
class PtpLocalClockTimePropertiesCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ( ( local-clock time-properties ) | time-property )'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'local-clock': matcherLocalClock,
      'time-properties': 'Show PTP clock time properties',
      'time-property': CliCommand.Node(
         matcher=CliMatcher.KeywordMatcher( 'time-property',
            helpdesc='Show PTP clock time property' ),
         deprecatedByCmd='show ptp local-clock time-properties' ),
   }
   handler = PtpCli.showTimeProperty
   cliModel = PtpCliModel.PtpTimePropertyWrapper

BasicCli.addShowCommandClass( PtpLocalClockTimePropertiesCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation candidate-grantor
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationCandidateGrantorCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation candidate-grantor'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'candidate-grantor': 'Show candidate masters of an interface',
   }
   handler = PtpCli.showCandidateGrantors
   cliModel = PtpCliModel.UcastNegCandidateGrantor

BasicCli.addShowCommandClass( PtpUnicastNegotiationCandidateGrantorCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation granted
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationGrantedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation granted'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'granted': 'Show unicast negotiation contracts granted',
   }
   handler = PtpCli.showGrantedSlaves
   cliModel = PtpCliModel.UcastNegStatus

BasicCli.addShowCommandClass( PtpUnicastNegotiationGrantedCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation profile
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationProfileCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation profile [ PROFILE ]'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'profile': 'Show Unicast Negotiation profiles',
      'PROFILE': CliMatcher.DynamicNameMatcher(
         lambda mode: PtpCli.ptpConfig.ucastNegProfile,
         'Unicast Negotiation profile name' ),
   }
   handler = PtpCli.showUcastNegProfiles
   cliModel = PtpCliModel.UcastNegProfiles

BasicCli.addShowCommandClass( PtpUnicastNegotiationProfileCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation remote-grantee
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationRemoteGranteeCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation remote-grantee'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'remote-grantee': 'Show remote grantees of an interface',
   }
   handler = PtpCli.showRemoteGrantees
   cliModel = PtpCliModel.UcastNegRemoteGrantee

BasicCli.addShowCommandClass( PtpUnicastNegotiationRemoteGranteeCmd )

#--------------------------------------------------------------------------------
# show ptp unicast-negotiation requested
#--------------------------------------------------------------------------------
class PtpUnicastNegotiationRequestedCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp unicast-negotiation requested'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'unicast-negotiation': matcherUnicastNegotiation,
      'requested': 'Show unicast negotiation contracts requested',
   }
   handler = PtpCli.showRequestedMasters
   cliModel = PtpCliModel.UcastNegStatus

BasicCli.addShowCommandClass( PtpUnicastNegotiationRequestedCmd )

#--------------------------------------------------------------------------------
# show ptp ip access-list
#--------------------------------------------------------------------------------
class PtpIpAccessListCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ptp ip access-list'
   data = {
      'ptp': PtpCli.ptpMatcherForShow,
      'ip': AclCli.ipKwForShowServiceAcl,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
   }
   handler = PtpCli.showPtpIpAcl
   cliModel = AclCliModel.AllAclList

BasicCli.addShowCommandClass( PtpIpAccessListCmd )
