# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from PtpCliModel import messageTypeToName

import CliModel
import IntfModel
import ArnetModel
import TableOutput
import Tac
from time import gmtime, strftime 

# only the reasons for implemented counter are listed below
# the map will grow as more counter get implemented
# please see counter AID/6454 for the complete list 
debugCounterReasonMap = {
   'alreadySent' : ' master',
   'alternateMasterFlag' : 'alternate master flag is set',
   'arpNonExist' : 'no arp entry exists for destination Ip',
   'badDestinedPort' : 'packet has invalid destination port',
   'badDomainNum' : 'packet has invalid ptp domain number',
   'badMaster' : 'sender is not the correct master',
   'badReqPort' : 'packet has invalid requesting port',
   'badSender' : 'packet has invalid source port',
   'blacklistedMasterIp' : 'ip is blacklisted',
   'duplicateFollowUp' : 'receiving duplicate followup message',
   'invalidDelayMsgType' : 'invalid delay message type',
   'invalidDelayReq' : 'invalid delay request',
   'invalidDmac' : 'packet has invalid destination mac address',
   'invalidEthernetType' : 'packet has invalid ethertype',
   'invalidGrantee' : 'grantee is invalid, unconfigured',
   'invalidMgmtId' : 'packet has invalid management id',
   'invalidMsgLen' : 'message has invalid length',
   'invalidPdelayMsgType' : 'invalid peer delay message type',
   'invalidSeqId' : 'invalid sequence id',
   'invalidUdp' : 'packet has invalid udp header',
   'invalidUdpPort' : 'packet has invalid udp port number',
   'invalidUcastTlv' : 'invalid unicast negotiation TLV format',
   'invalidIpv4' : 'packet has invalid ip4 header or invalid ipv4 address',
   'invalidIpv6' : 'packet has invalid ipv6 header or invalid ipv6 address',
   'mismatchLastReceivedSync' : 'sequence id mismatches last received sync',
   'missingIntf' : 'missing interface',
   'missingUdpHdr' : 'packet has no udp header',
   'missingIpHdr' : 'packet has no ip header',
   'missingPtpHdr' : 'packet has no ptp header',
   'managementLoop' : 'management message was looped',
   'maxReqWithoutDelayResp' : 'max number of request exceeds',
   'msgTxRxOnSamePort' : 'receive message sent from this port',
   'msgTxRxOnSameClk' : 'receive message sent from this clock',
   'negativeTimestamp' : 'timestamp earlier than the previous timestamp',
   'noEgressSyncTimestamp' : 'egress sync timestamp is missing',
   'noEventMsgEgressTimestamp' : 'event message egress timestamp is missing',
   'noIngressSyncTimestamp' : 'ingress sync timestamp is missing',
   'noPendingToSend' : 'no pending message to respond',
   'noPamForTxIntf' : 'no pam for the interface',
   'notAristaEthernetType' : 'missing arista specific ethertype encapsulation',
   'notBoundary' : 'not in boundary mode',
   'notFoundInitialPkt' : 'no matching initial packet',
   'notInUCast' : 'ptp is not in unicast negotiation mode',
   'noReqSent' : 'no request has been sent',
   'noUCastGrant' : 'unicast negotiation has not been granted',
   'pamFailsTx' : 'pan fails to transmit packet',
   'portIsDown' : 'ptp port is down',
   'portNotSlave' : 'received port is not slave',
   'received' : 'message has been received',
   'sent' : 'message has been sent', 
   'serviceAcl' : 'packet dropped by service acl',
   'staledFollowUp' : 'staled follow up is ignored',
   'staledPdelayReq' : 'staled peer delay request',
   'staledPdelayResp' : 'staled peer delay response',
   'stpNotForwarding' : 'port is STP blocked',
   'truncatedDot1QTag' : 'packet has truncated dot1q tag',
   'uCastGrantExpired' : 'unicast negotiation grant expired',
   'hwNotReadyForPacket' : 'hardware is not yet ready for packet',
   'udpNotOnRoutedPort' : 'udp packet needs to be send on routed port',
   'unconfRequestor' : 'the requestor is not configured',
   'undersizedPkt' : 'packet is undersized',
   'unexpectedFollowUp' : 'unexpected followup message',
   'unexpectedGrant' : 'unexpected grant is received',
   'unsupportedPtpVersion' : 'packet has unsupported ptp version',
   'unsupportedMsgTypeForUcast' : 'message type is not supported for unicast' +\
                                  ' negotiation',
   'vlanNotConfOnIntf' : 'packet is tagged but ptp vlan is not configured.',
   'waitForFollowUp' : 'waiting to receive follow up message',
   'unhealthyPortState' : 'port is in disabled/initializing/faulty state',
   'stepRemovedLarger255' : 'stepRemoved has value larger than 255',
   'recvMyOwnClockMsg' : 'sender is my own clock identity',
   'invalidTlv' : 'message TLV is invalid',
   'ordinaryMaster' : 'Ptp is in ordinary master clock mode',
   'syncMaster' : 'port is in syncMaster role',
   'lessBoundaryHops' : 'boundaryhops has less initial value',
   'lessForeignMasterThreshold' : 'less than foreign master threshold value',
   'roleMaster' : 'port is configured as a master',
   'vlanOnG8275_1' : 'packet is tagged but ptp is in profile G8275.1',
}

debugCounterDirection = \
   list( Tac.Type( "Ptp::DebugCounter::Direction::Constant").attributes )

invalidPktFmt = 'invalidPktFmt'
debugCounterTypeMap = messageTypeToName.copy()
debugCounterTypeMap[ invalidPktFmt ] = "Invalid packet"
debugCounterTypeMap.pop( 'none', None )

class PtpEndpoint( CliModel.Model ):
   intf = IntfModel.Interface( 
             help="Interface of the PTP endpoint" )
   vlanId = CliModel.Int( help="VLAN of the PTP endpoint" )
   domain = CliModel.Int( help="Domain number of the PTP endpoint")
   ip = ArnetModel.IpGenericAddress( 
           help="IP address of the PTP endpoint (only for PTP unicast negotiation)" )
   
   def str( self ):
      endpoint = self.intf.shortName
      # only show vlan if vlan is valid
      if self.vlanId > Tac.Value("Ptp::Constants").defaultPortDSVlanId:
         endpoint += ( ", VLAN %s" % self.vlanId )
      # only show ip if IPV4 or IPV6
      if self.ip.af != Tac.Type( "Arnet::AddressFamily" ).ipunknown:
         endpoint += ( ", %s" % self.ip )

      return endpoint

class PtpDebugCounterDropReason( CliModel.Model ):
   debugType = CliModel.Enum( help="Type of the PTP packet drop reason",
                              values=debugCounterTypeMap.keys() )
   debugReason = CliModel.Enum( 
                    help="Reason of the PTP packet drop",
                    values=debugCounterReasonMap.keys() )

   def str( self ):
      msgDisplay = ""
      if self.debugType != invalidPktFmt:
         msgDisplay = "message"

      return '%s %s with %s.' % ( debugCounterTypeMap[ self.debugType ],
                                  msgDisplay,
                                  debugCounterReasonMap[ self.debugReason ] )

class PtpDebugCounter( CliModel.Model ):
   endpoint = CliModel.Submodel( 
      help="PTP endpoint where the packets are being dropped",
      valueType=PtpEndpoint ) 
   direction = CliModel.Enum( 
      help="Flow direction of the PTP packets dropped (Rx/Tx)",
      values=debugCounterDirection )
   dropReason = CliModel.Submodel(
      help="Reason why the PTP packets are being dropped",
      valueType=PtpDebugCounterDropReason )
   count = CliModel.Int( help="Number of the PTP packets dropped" )
   lastSeen = CliModel.Float( help="Last time a packet drop is seen, in seconds" )

class PtpDebugIntfCounters( CliModel.Model ):
   intfCounters = CliModel.List( help="List of PTP drop counters on the interface",
                                 valueType=PtpDebugCounter )  
   
   def secToUtcTimestamp( self, sec ):
      return strftime( '%Y-%m-%d UTC %H:%M:%S',
                        gmtime( sec ) )

   def renderTable( self, table ):
      # achieve order endpoint > direction( tx > rx ) > reason ( alphabetical )
      sortedByReason = sorted( self.intfCounters,
                               key=lambda x : x.dropReason.str() )
      sortedByDirection = sorted( sortedByReason,
                                  key=lambda x: x.direction, reverse=True )

      for counter in sorted( sortedByDirection,
                             key=lambda x: x.endpoint.str() ):
         endpoint = counter.endpoint.str()
         direction = counter.direction
         reason = counter.dropReason.str()
         count = counter.count
         lastSeenStr = self.secToUtcTimestamp( counter.lastSeen )
         table.newRow( endpoint, direction, reason, count, lastSeenStr )

class PtpDebugCounters( CliModel.Model ):
   disabled = CliModel.Bool( help="PTP is disabled", default=False )
   counters = CliModel.Dict( help="PTP drop counters for all of the PTP interfaces",
                             valueType=PtpDebugIntfCounters,
                             keyType=IntfModel.Interface )
   def render( self ):
      fEndpoint = TableOutput.Format( justify="left", maxWidth=40, wrap=True)
      fTxRx = TableOutput.Format( justify="left", maxWidth=3, minWidth=3 )
      fReason = TableOutput.Format( justify="left", maxWidth=50, wrap=True )
      fCount = TableOutput.Format( justify="right", maxWidth=7 )
      fLastSeen = TableOutput.Format( justify="right", maxWidth=23, minWidth=23 )

      tableHeadings = ( "Endpoint", "Dir", "Reason", "Count", "Last Seen" )
      table = TableOutput.createTable( tableHeadings, tableWidth=123 )
      table.formatColumns( fEndpoint, fTxRx, fReason, fCount, fLastSeen )

      for intf in sorted( self.counters.keys() ):
         self.counters[ intf ].renderTable( table )
      print table.output()
