# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements PTP configuration.  In particular, it provides:
#
# Global Configuration
# [no|default] ptp mode disabled|boundary|e2etransparent|p2ptransparent|gptp
# [no|default] ptp forward-v1
# [no|default] ptp priority1
# [no|default] ptp priority2
# [no|default] ptp clock-identity <eui-64 number>|<mac address>
# [no|default] ptp clock-accuracy < value >
# [no|default] ptp hold-ptp-time <interval>
# [no|default] ptp monitor
# [no|default] ptp monitor threshold offset-from-master <threshold>
# [no|default] ptp monitor threshold mean-path-delay <threshold>
# [no|default] ptp monitor threshold skew <threshold>
#
# Ethernet/Port-Channel Interface Configuration
# [no|default] ptp announce interval <interval>
# [no|default] ptp announce timeout <timeout>
# [no|default] ptp sync-message interval <interval>
# The command sync timeout is used only in gPTP mode
# [no|default] ptp sync timeout <timeout>
# [no|default] ptp sync limit <limit>
# [no|default] ptp delay-req interval <interval>
# [no|default] ptp pdelay-req interval <interval>
# [no|default] ptp pdelay-neighbor-threshold <threshold in nanoseconds>
# [no|default] ptp delay-mechanism p2p|e2e|disabled
# [no|default] ptp enable
#-------------------------------------------------------------------------------

import datetime
import math
from cStringIO import StringIO
from operator import attrgetter

import AgentCommandRequest
import BasicCliModes
import CliCommand
import CliMatcher
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.IntfCli as IntfCli
from CliPlugin.PtpCliModel import PtpHoldPtpTimeForShow, PtpClockQuality, PtpClock, \
   PtpParent, PtpParentWrapper, PtpTimeProperty, PtpTimePropertyWrapper, \
   PtpIntfCounters, PtpIntfCountersWrapper, PtpCounters, PtpIntfVlansCounters, \
   PtpIntf, PtpIntfWrapper, PtpIntfVlans, PtpInterfaces, PtpSummary, \
   PtpIntfSummary, PtpIntfVlanSummary, PtpPortIdentity, \
   PtpForeignMasterDS, PtpIntfForeignMasterRecords, PtpForeignMasterRecords, \
   PtpIntfVlanForeignMasterRecords, PtpSourceIp, \
   PtpMonitorDataEntry, PtpMonitorData, \
   PtpVlans, PtpVlansExtendedWrapper, \
   UcastNegProfiles, UcastNegProfile, \
   UcastNegCandidateGrantor, UcastNegCandidateGrantorEntry, \
   UcastNegRemoteGrantee, UcastNegRemoteGranteeEntry, \
   UcastNegMessage, UcastNegMessages, UcastNegStatus
import CliPlugin.TechSupportCli
import CliPlugin.VlanCli as VlanCli
import CliToken.Clear
import CliToken.Ip
import CliToken.Ipv6
import ConfigMount
import Ethernet
import IpUtils
from IpLibConsts import DEFAULT_VRF
import Ptp
import LazyMount
import SharedMem
import Smash
import Tac
from TypeFuture import TacLazyType
from Vlan import computeVlanRangeSet
from PtpDebugCounterModel import PtpDebugCounters, PtpDebugIntfCounters, \
   PtpDebugCounter, PtpEndpoint, PtpDebugCounterDropReason

modeSpecificIntervalAttrs = [ 'logSyncInterval',
                              'logAnnounceInterval',
                              'logMinPdelayReqInterval' ]
PtpMode = TacLazyType( 'Ptp::PtpMode' )
PtpProfile = TacLazyType( 'Ptp::PtpProfile' )
CliConstants = TacLazyType( 'Ptp::Constants' )
UcastNegGrantorState = TacLazyType( 'Ptp::UcastNegGrantorState' )
CounterDir = TacLazyType( "Ptp::DebugCounter::Direction::Constant" )
CounterType = TacLazyType( "Ptp::DebugCounter::Type::Constant" )
CounterReason = TacLazyType( "Ptp::DebugCounter::Reason::Constant" )
CounterPortKeyConstant = TacLazyType( "Ptp::DebugCounter::CounterPortKeyConstant" )
MTN = TacLazyType( "Ptp::MessageTypeNumber" )

ptpConfig = None
ptpStatus = None
ptpHwStatusDir = None
brConfig = None
bridgingSic = None
serviceName = 'ptp'
aclCheckpoint = None
ptpUcastNegProfiles = None
ucastNegStatus = None
debugCounter = None
pktCounter = None

def ptpSupported():
   if not ptpHwStatusDir:
      return False
   for name in ptpHwStatusDir:
      if ptpHwStatusDir[ name ].ptpSupported:
         return True
   return False

def ptpSupportedGuard( mode, token ):
   if ptpSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpForwardV1Supported():
   if not ptpHwStatusDir:
      return False
   for ptpHwStatus in ptpHwStatusDir.itervalues():
      if ptpHwStatus.ptpForwardV1Supported:
         return True
   return False

def ptpForwardV1SupportedGuard( mode, token):
   if ptpForwardV1Supported():
      return None
   return CliParser.guardNotThisPlatform

def ptpOneStepClockSupported():
   if not ptpHwStatusDir:
      return False
   return all( ptpHwStatus.ptpOneStepClockSupported for ptpHwStatus in
               ptpHwStatusDir.itervalues() )

def ptpOneStepClockSupportedGuard( mode, token ):
   if ptpOneStepClockSupported():
      return None
   return CliParser.guardNotThisPlatform

def ptpG82751Guard( mode, token ):
   if not ptpHwStatusDir:
      return False
   for name in ptpHwStatusDir:
      if ptpHwStatusDir[ name ].ptpG82751Supported:
         return None
   return CliParser.guardNotThisPlatform

def ptpForwardUnicastSupported():
   if not ptpHwStatusDir:
      return False
   for ptpHwStatus in ptpHwStatusDir.itervalues():
      if ptpHwStatus.ptpForwardUnicastSupported:
         return True
   return False

def ptpForwardUnicastGuard( mode, token ):
   if ptpForwardUnicastSupported():
      return None
   return CliParser.guardNotThisPlatform

def portDSIntfList( intfs=None, vlanId=None ):
   if intfs:
      intfList = list( intfs.intfNames() )
   return [ portDS for portDS in ptpStatus.portDS
            if ( intfs is None or portDS.intf in intfList ) and
               ( vlanId is None or portDS.vlanId == vlanId ) ]

def createDefaultCliIntfConfig( intf, createEnabled=False ):
   intfConfig = ptpConfig.newIntfConfig( intf, createEnabled )
   # Add default portDS vlan to newly created interface, to only send VLAN untagged
   # packet by default
   intfConfig.vlanConfig.newMember( CliConstants.defaultPortDSVlanId, True, True )
   return intfConfig

def intfConfigIntfList( intfs=None ):
   if intfs:
      return intfs.intfNames()
   return ptpConfig.intfConfig.keys()

def setIntfDefaults( intf, createEnabled=False ):
   ptpMode = ''
   if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
      ptpMode = '802Dot1AS'
   else:
      ptpMode = '1588V2'

   intfConfig = ptpConfig.intfConfig.get( intf )
   if not intfConfig:
      intfConfig = createDefaultCliIntfConfig( intf, createEnabled )
      for intervalAttr in modeSpecificIntervalAttrs:
         intervalValue = getattr( CliConstants, intervalAttr + 'Default' + ptpMode )
         setattr( intfConfig, intervalAttr, intervalValue )
   return ( intfConfig, ptpMode )

attrsUsingDeprecatedCmds = {
   'logSyncInterval' : 'syncIntervalUseDeprecatedCmd'
}

def setIntfAttr( mode, attr, value=None, no=None, deprecatedCmd=False ):
   ( intfConfig, ptpMode ) = setIntfDefaults( mode.intf.name )
   assert hasattr( intfConfig, attr )
   deprecatedAttr = attrsUsingDeprecatedCmds.get( attr )
   if deprecatedAttr:
      setattr( intfConfig, deprecatedAttr, deprecatedCmd )
   if value is None or no is not None:
      if attr in modeSpecificIntervalAttrs:
         value = getattr( CliConstants, attr + 'Default' + ptpMode )
      else:
         value = getattr( CliConstants, attr + 'Default' )
   setattr( intfConfig, attr, value )

ptpMatcherForConfig = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Precision Time Protocol',
   guard=ptpSupportedGuard )

#-------------------------------------------------------------------------------
# [no|default] ptp mode disabled|boundary|e2etransparent|p2ptransparent|gptp
#-------------------------------------------------------------------------------
def setGlobalAttr( mode, attr, value=None, no=None ):
   assert hasattr( ptpConfig, attr )
   if value is None or no is not None:
      value = getattr( CliConstants, attr + 'Default' )
   setattr( ptpConfig, attr, value )

def setGlobalMapAttr( mode, attr, key, value=None, no=None ):
   assert hasattr( ptpConfig, attr )
   attrMap = getattr( ptpConfig, attr )
   attrMap[ key ] = value

ptpModeMapping = {
   'disabled': 'ptpDisabled',
   'boundary': 'ptpBoundaryClock',
   'e2etransparent': 'ptpEndToEndTransparentClock',
   'p2ptransparent': 'ptpPeerToPeerTransparentClock',
   'gptp': 'ptpGeneralized',
   'ordinarymaster': 'ptpOrdinaryMasterClock',
   'ptpOneStepEndToEndTransparentClock': 'ptpOneStepEndToEndTransparentClock'
}

def setPtpMode( mode, ptpMode ):
   ptpMode = ptpModeMapping[ ptpMode ] #translate the mode
   oldPtpMode = ptpConfig.ptpMode
   if ptpMode == PtpMode.ptpGeneralized:
      lagIntfs = []
      for intf in ptpConfig.intfConfig:
         if Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( intf ) \
            and ptpConfig.intfConfig[ intf ].enabled:
            lagIntfs.append( intf )
      if lagIntfs:
         mode.addError( "Gptp is not supported on port-channels." )
         mode.addError( "Please disable ptp on the following port-channels "
                        "before attempting to enable gptp:" )
         for lagIntf in lagIntfs:
            mode.addError( lagIntf )
         return

   ptpConfig.ptpMode = ptpMode
   ptpConfig.enabled = ( ptpMode != 'ptpDisabled' )
   if PtpMode.ptpGeneralized in ( oldPtpMode, ptpMode ):
      if ptpConfig.ptpMode == PtpMode.ptpGeneralized:
         ptpType = '802Dot1AS'
         oldPtpType = '1588V2'
      else:
         ptpType = '1588V2'
         oldPtpType = '802Dot1AS'
      outOfIntervalAttrWarning = { }
      attrOutputMap = { 'logSyncInterval':'sync-interval',
                        'logAnnounceInterval':'announce-interval',
                        'logMinPdelayReqInterval':'peer-delay-req-interval' }
      for intfConfig in ptpConfig.intfConfig.values():
         for attr in modeSpecificIntervalAttrs:
            oldDefaultValue = getattr( CliConstants, attr + 'Default' + oldPtpType )
            if getattr( intfConfig, attr ) == oldDefaultValue:
               newDefaultValue = getattr( CliConstants, attr + 'Default' + ptpType )
               setattr( intfConfig, attr, newDefaultValue )
               continue
            lowerLimit = getattr( CliConstants, 'min' + attr + ptpType )
            upperLimit = getattr( CliConstants, 'max' + attr + ptpType )
            oldValue = getattr( intfConfig, attr )
            # Reset to default only if oldValue does not lie in the new mode's
            # allowed interval
            if oldValue < lowerLimit or oldValue > upperLimit:
               outOfIntervalAttrWarning[ attr ] = True
               defaultValue = getattr( CliConstants, attr + 'Default' + ptpType )
               setattr( intfConfig, attr, defaultValue )
      for attr in outOfIntervalAttrWarning:
         mode.addWarning( "Unsupported Value of %s on some interfaces "
                          "in %s mode" % ( attrOutputMap[ attr ] , ptpMode ) )

#-------------------------------------------------------------------------------
# ptp profile ( g8275.2 | g8275.1 )
#
# [no|default] ptp profile
#-------------------------------------------------------------------------------
ucastNegRequiredProfiles = { PtpProfile.ptpG8275_2 }

def setPtpProfile( mode, attr, value=None, no=None ):
   if value == 'g8275.2':
      value = PtpProfile.ptpG8275_2
   elif value == 'g8275.1':
      value = PtpProfile.ptpG8275_1
      mode.addWarning( "Announce message rate will be fixed to 8/s" )
      mode.addWarning( "Sync/FollowUp message rate will be fixed to 16/s" )
      mode.addWarning( "Delay-req message rate will be fixed to 16/s" )

   # Warn the user but set it so that if the user switch to boundary mode
   # the wanted profile will already be selected
   if ptpConfig.ptpMode != "ptpBoundaryClock":
      mode.addWarning( "PTP profile configuration will be ignored while not in"
                       " Boundary mode" )
   setGlobalAttr( mode, attr, value, no )
   if value in ucastNegRequiredProfiles and no is None:
      # ptp source ip is not supported with unicast negotiation
      # Reset to default values
      setGlobalAttr( mode, 'srcIp4', value=None, no=True )
      setGlobalAttr( mode, 'srcIp6', value=None, no=True )
      ptpConfig.unicastNegotiation = True
   else:
      ptpConfig.unicastNegotiation = False

#-------------------------------------------------------------------------------
# [no|default] ptp management all
#-------------------------------------------------------------------------------
def setManagementEnabled( mode, no=False ):
   if ptpConfig.ptpMode != "ptpBoundaryClock":
      mode.addWarning( "PTP Management Messages configuration will be ignored while "
                       "not in Boundary mode" )
   ptpConfig.managementEnabled = not no

#-------------------------------------------------------------------------------
# [no|default] ptp netsync-monitor delay-request
#-------------------------------------------------------------------------------
def setNetSyncMonitor( mode, no=False ):
   if ptpConfig.ptpMode != "ptpBoundaryClock":
      mode.addWarning( "PTP NetSync Monitor configuration will be ignored while not "
                       "in Boundary mode" )
   ptpConfig.netSyncMonitor = not no

#-------------------------------------------------------------------------------
# [no|default] ptp forward-v1
#-------------------------------------------------------------------------------
def forwardPtpV1( mode, no=False ):
   ptpConfig.forwardPtpV1 = not no

def forwardPtpUnicast( mode, no=False ):
   ptpConfig.forwardUnicast = not no

#-------------------------------------------------------------------------------
# [no|default] ptp domain <domain>
#-------------------------------------------------------------------------------
def domainNumberRange( mode ):
   if ptpConfig.ptpMode == 'ptpBoundaryClock' and \
      ptpConfig.ptpProfile == 'ptpG8275_2':
      ptpProfile = 'G8275_2'
   elif ptpConfig.ptpMode == 'ptpBoundaryClock' and \
        ptpConfig.ptpProfile == 'ptpG8275_1':
      ptpProfile = 'G8275_1'
   else:
      ptpProfile = '1588V2'
   lowerLimit = getattr( CliConstants, 'min' + 'DomainNumber' + ptpProfile )
   upperLimit = getattr( CliConstants, 'max' + 'DomainNumber' + ptpProfile )
   return ( lowerLimit, upperLimit )

#-------------------------------------------------------------------------------
# [no|default] ptp priority1 <priority1>
#-------------------------------------------------------------------------------
def setPriority1( mode, attr, value=None, no=None ):
   if ptpConfig.unicastNegotiation:
      mode.addWarning( "Priority1 is not used in PTP G8275.2 profile and has a "
                       "static value of 128" )
   setGlobalAttr( mode, 'priority1', value, no )

#-------------------------------------------------------------------------------
# [no|default] ptp clock-identity <eui-64 number>|<mac address>
#-------------------------------------------------------------------------------
colonPattern = ':'.join( [ Ethernet.pair for x in range( 8 ) ] ) + r'$'
dashPattern = '\\-'.join( [ Ethernet.pair for x in range( 8 ) ] ) + r'$'
dotPattern = '\\.'.join( [ Ethernet.quad for x in range( 4 ) ] ) + r'$'
eui64Pattern = '(%s|%s|%s)' % ( colonPattern, dashPattern, dotPattern )

def setPtpClockIdentity( mode, value=None, no=None ):
   # Parse the given EUI string, and create an instance of ClockIdentity.
   def createClockIdentity( euiStr ):
      # Split raw eui string by its delimeters and place leading 0's in each octet
      if '.' in euiStr:
         euiStr = [ '%04x' % int( n, 16 ) for n in euiStr.split('.') ]
      elif ':' in euiStr:
         euiStr = [ '%02x' % int( n, 16 ) for n in euiStr.split(':') ]
      elif '-' in euiStr:
         euiStr = [ '%02x' % int( n, 16 ) for n in euiStr.split('-') ]
      else:
         # We should never hit here, but just in case
         mode.addError( "Invalid character found in EUI number" )
         return None

      # Join the all octet to get a hexadecimal string of the eui number
      euiStr = ''.join( euiStr )

      isMacAddress = ( len( euiStr ) == 12 )
      isEui64Number = ( len( euiStr ) == 16 )
      if not ( isMacAddress or isEui64Number ):
         # We should never hit here, but just in case
         mode.addError( "input was in neither MAC address nor EUI-64 bit format: " +
                        euiStr )
         return None

      # If it's in MAC address format, convert it to EUI-64
      if isMacAddress:
         # from IEEE 1588 - 7.5.2.2.2
         # When using an EUI-48, the first 3 octets, ... of the IEEE EUI-48 are
         # assigned in order to the first 3 octets of the clockIdentity with the
         # most significant octet of the IEEE EUI-64. ... Octets with index 3 and
         # 4 have hex values FF and FE, respectively. The remaining 3 octets of
         # the IEEE EUI-48 are assigned in order to the last 3 octets of the
         # clockIdentity
         # But we are using FFFF to follow the current codebase practice. Once
         # our codebase changes back to follow the spec, we should chnage this to
         # FFFE as well.
         euiStr = euiStr[ : 6 ] + 'ffff' + euiStr[ 6 : ]

      # according to IEEE 1588 7.5.2.2 the least significant two bits of the most
      # significant octet of the OUI shall both be 0. However, we omit this check
      # since the Ptp agent itself never checks for this restriction. And upon
      # inspecting the spec, not following it will not result in unexpected
      # errors.

      # Convert to integer
      eui64 = int( euiStr, 16 )

      return Tac.newInstance( "Ptp::ClockIdentity",
                              Tac.newInstance( "Arnet::NetU64", eui64 ) )

   no = not value or bool( no )
   if no:
      ptpConfig.clockIdentityConfigured = False
   else:
      ptpConfig.clockIdentityConfigured = True
      ptpConfig.clockIdentity = createClockIdentity( value )

#-------------------------------------------------------------------------------
# [no|default] ptp clock-accuracy VALUE
#-------------------------------------------------------------------------------
def newClockQuality( clkQual ):
   return Tac.Value( "Ptp::ClockQuality", clkQual.clockClass,
                     clkQual.clockAccuracy, clkQual.offsetScaledLogVariance )

def setPtpClockAccuracy( mode, args ):
   value = args.get( 'VALUE' )
   ptpConfig.clockQualityConfigured = value is not None


   clkQual = newClockQuality( ptpConfig.clockQuality )

   if ptpConfig.clockQualityConfigured:
      clkQual.clockAccuracy = value
   else:
      clkQual.clockAccuracy = ptpStatus.localClockQuality.clockAccuracy

   ptpConfig.clockQuality = clkQual

#-------------------------------------------------------------------------------
# [no|default] ptp source ip address <address>
#-------------------------------------------------------------------------------
def setPtpSrcIp4( mode, ipAddr=None, no=None ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return

   if ipAddr is not None:
      ip = Tac.Value( "Arnet::IpGenAddr", ipAddr )
      if not ip.isUnicast:
         mode.addError( "Source IP must be an unicast address" )
         return
   setGlobalAttr( mode, 'srcIp4', value=ipAddr, no=no )

#-------------------------------------------------------------------------------
# [no|default] ptp source ipv6 address <address>
#-------------------------------------------------------------------------------
def setPtpSrcIp6( mode, ip6Addr=None, no=None ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return
   setGlobalAttr( mode, 'srcIp6', value=ip6Addr, no=no )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor
#-------------------------------------------------------------------------------
def setPtpMonitor( mode, no=None ):
   setGlobalAttr( mode, 'monitorEnabled', value=not no )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor sequence-id
#-------------------------------------------------------------------------------
def setLogMissingMsgs( mode, no=None ):
   # Enable/disable all missing message type
   seqIdMap = 'logMissingMessageSeqId'
   thresholdMap = 'missingSequenceThresholds'
   setGlobalAttr( mode, 'logMissingMessages', value=not no)
   setGlobalMapAttr( mode, seqIdMap, MTN.messageSync, value=not no )
   setGlobalMapAttr( mode, seqIdMap, MTN.messageFollowUp, value=not no )
   setGlobalMapAttr( mode, seqIdMap, MTN.messageDelayResp, value=not no )
   setGlobalMapAttr( mode, seqIdMap, MTN.messageAnnounce, value=not no )

   # Default threshold 1
   setGlobalMapAttr( mode, thresholdMap, MTN.messageSync, value=1 )
   setGlobalMapAttr( mode, thresholdMap, MTN.messageFollowUp, value=1 )
   setGlobalMapAttr( mode, thresholdMap, MTN.messageDelayResp, value=1 )
   setGlobalMapAttr( mode, thresholdMap, MTN.messageAnnounce, value=1 )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold missing-message TYPE INTERVAL intervals
#-------------------------------------------------------------------------------
def setMissingMsgTimer( mode, messageType, threshold=2, no=None ):
   timerMap = 'logMissingMessageTimer'
   thresholdMap = 'missingTimerThresholds'
   if messageType == 'sync':
      setGlobalMapAttr( mode, timerMap, MTN.messageSync, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageSync, threshold )
   if messageType == 'follow-up':
      setGlobalMapAttr( mode, timerMap, MTN.messageFollowUp, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageFollowUp, threshold )
   if messageType == 'announce':
      setGlobalMapAttr( mode, timerMap, MTN.messageAnnounce, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageAnnounce, threshold )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold missing-message TYPE VALUE sequence-ids
#-------------------------------------------------------------------------------
def checkIfMsgsFalse( mode ):
   if not ptpConfig.logMissingMessageSeqId.get( MTN.messageSync, False ) and \
      not ptpConfig.logMissingMessageSeqId.get( MTN.messageAnnounce, False ) and \
      not ptpConfig.logMissingMessageSeqId.get( MTN.messageDelayResp, False ) and \
      not ptpConfig.logMissingMessageSeqId.get( MTN.messageFollowUp, False ):
      setGlobalAttr( mode, 'logMissingMessages', value=False )

def setMissingMsgSeqId( mode, messageType, threshold=2, no=None ):
   setGlobalAttr( mode, 'logMissingMessages', value=True )
   seqIdMap = 'logMissingMessageSeqId'
   thresholdMap = 'missingSequenceThresholds'
   if messageType == 'sync':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageSync, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageSync, threshold )
   if messageType == 'follow-up':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageFollowUp, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageFollowUp, threshold )
   if messageType == 'delay-resp':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageDelayResp, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageDelayResp, threshold )
   if messageType == 'announce':
      setGlobalMapAttr( mode, seqIdMap, MTN.messageAnnounce, value=not no )
      setGlobalMapAttr( mode, thresholdMap, MTN.messageAnnounce, threshold )
   checkIfMsgsFalse( mode )

def setMissingMsgTimerAndSeqId(
      mode, messageType, timerThreshold=2, seqIdThreshold=2, no=None ):
   setMissingMsgTimer( mode, messageType, timerThreshold, no )
   setMissingMsgSeqId( mode, messageType, seqIdThreshold, no )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold offset-from-master <threshold>
# [no|default] ptp monitor threshold mean-path-delay <threshold>
# [no|default] ptp monitor threshold skew <threshold>
#-------------------------------------------------------------------------------
def setPtpMonitorOffsetThreshold( mode, args ):
   constant = CliConstants.invalidOffsetFromMasterThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'offsetFromMasterThreshold', value=value )

def setPtpMonitorDelayThreshold( mode, args ):
   constant = CliConstants.invalidMeanPathDelayThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'meanPathDelayThreshold', value=value )

def setPtpMonitorSkewThreshold( mode, args ):
   constant = CliConstants.invalidSkewThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'skewThreshold', value=value )

#-------------------------------------------------------------------------------
# [no|default] ptp monitor threshold offset-from-master <threshold> nanoseconds drop
# [no|default] ptp monitor threshold mean-path-delay <threshold> nanoseconds drop
# [no|default] ptp monitor threshold skew <threshold> drop
#-------------------------------------------------------------------------------
def setPtpMonitorOffsetDropThreshold( mode, args ):
   constant = CliConstants.invalidOffsetFromMasterDropThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'offsetFromMasterDropThreshold', value=value )

def setPtpMonitorDelayDropThreshold( mode, args ):
   constant = CliConstants.invalidMeanPathDelayDropThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'meanPathDelayDropThreshold', value=value )

def setPtpMonitorSkewDropThreshold( mode, args ):
   constant = CliConstants.invalidSkewDropThreshold
   value = args.get( 'VALUE', constant )
   setGlobalAttr( mode, 'skewDropThreshold', value=value )

#-------------------------------------------------------------------------------
# [no|default] ptp ip access-group <acl-name> in
#-------------------------------------------------------------------------------
def aclNameIs( mode, aclType, aclName ):
   if ptpConfig.serviceAclTypeVrfMap is None:
      ptpConfig.serviceAclTypeVrfMap = ( serviceName, )
   aclKey = Tac.Value( "Acl::AclTypeAndVrfName", aclType, DEFAULT_VRF )
   # pylint: disable-msg=E1103,maybe-no-member
   ptpConfig.serviceAclTypeVrfMap.aclName[ aclKey ] = aclName

def aclNameDel( mode, aclType ):
   if ptpConfig.serviceAclTypeVrfMap is None:
      return
   aclKey = Tac.Value( "Acl::AclTypeAndVrfName", aclType, DEFAULT_VRF )
   # pylint: disable-msg=E1103,maybe-no-member
   del ptpConfig.serviceAclTypeVrfMap.aclName[ aclKey ]

def setIntfDscpGeneral( mode, value=None, no=None ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if value is None or no is not None:
      value = getattr( CliConstants, 'globalDscpGeneralDefault' )
   setattr( intfConfig, 'dscpGeneralConfigured', not no )
   setattr( intfConfig, 'dscpGeneral', value )

def setIntfDscpEvent( mode, value=None, no=None ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if value is None or no is not None:
      value = getattr( CliConstants, 'globalDscpEventDefault' )
   setattr( intfConfig, 'dscpEventConfigured', not no )
   setattr( intfConfig, 'dscpEvent', value )

#-------------------------------------------------------------------------------
# [no|default] ptp announce interval <interval>
#-------------------------------------------------------------------------------
def announceIntervalRange( mode ):
   ptpMode = '802Dot1AS' if ptpConfig.ptpMode == 'ptpGeneralized' else '1588V2'
   lowerLimit = getattr( CliConstants, 'min' + 'logAnnounceInterval' + ptpMode )
   upperLimit = getattr( CliConstants, 'max' + 'logAnnounceInterval' + ptpMode )
   return ( lowerLimit, upperLimit )

#-------------------------------------------------------------------------------
# [no|default] ptp sync-message interval <interval>
#-------------------------------------------------------------------------------
def syncIntervalRange( mode ):
   ptpMode = '802Dot1AS' if ptpConfig.ptpMode == 'ptpGeneralized' else '1588V2'
   lowerLimit = getattr( CliConstants, 'min' + 'logSyncInterval' + ptpMode )
   upperLimit = getattr( CliConstants, 'max' + 'logSyncInterval' + ptpMode )
   return ( lowerLimit, upperLimit )

def setSyncInterval( mode, attr, value=None, no=None, deprecatedCmd=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   # We are limited in the number of timestamped packet we can send (200 pkts/sec)
   # If we have configured PTP to run per VLAN on a trunk interface, we need to
   # prevent that the Sync Interval * Configured VLANs exceeds this limit.
   if value is None:
      setIntfAttr( mode, attr, value, no, deprecatedCmd )
      return
   configuredVlans = len( intfConfig.vlanConfig )
   rate = ( 1 / ( 2**value ) ) * configuredVlans
   maxRate = 1 / CliConstants.minEgressTsFifoPollInterval # = 200
   if rate > maxRate:
      mode.addError( "Number of PTP configured VLANs on a trunk is too high."
                     " Please consider increasing the sync messages interval or"
                     " configuring less VLANs" )
      return
   if ptpStatus.g82751Enabled:
      mode.addWarning( "Sync message rate is fixed to 16/s in PTP profile g8275.1" )

   setIntfAttr( mode, attr, value, no, deprecatedCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp pdelay-req interval <interval>
#-------------------------------------------------------------------------------
def pDelayIntervalRange( mode ):
   ptpMode = '802Dot1AS' if ptpConfig.ptpMode == 'ptpGeneralized' else '1588V2'
   lowerLimit = getattr( CliConstants, 'min' + 'logMinPdelayReqInterval' + ptpMode )
   upperLimit = getattr( CliConstants, 'max' + 'logMinPdelayReqInterval' + ptpMode )
   return ( lowerLimit, upperLimit )

#-------------------------------------------------------------------------------
# [no|default] ptp delay-mechanism p2p|e2e|disabled
#-------------------------------------------------------------------------------
def setDelayMechanism( mode, attr, value=None, no=None, deprecatedCmd=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   def _isTrunkWithPtpVlan():
      switchIntfConfig = bridgingSic.switchIntfConfig.get( mode.intf.name )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      return isIntfTrunkMode and \
             CliConstants.defaultPortDSVlanId not in intfConfig.vlanConfig

   if not no and value == 'p2p' and _isTrunkWithPtpVlan():
      mode.addWarning( "P2P delay mechanism is not supported with PTP configured to"
                       " run per VLANs on a trunk interface" )

   if ptpStatus.unicastNegotiation and value == 'p2p':
      mode.addWarning( "P2P delay mechanism is not supported in PTP G8275.2 "
                       "profile" )

   setIntfAttr( mode, attr, value, no, deprecatedCmd )

#-------------------------------------------------------------------------------
# [no|default] ptp transport layer2|ipv4|ipv6
#-------------------------------------------------------------------------------
def setTransportMode( mode, attr, value=None, no=None ):
   # Do nothing if we're already in the transport mode
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if intfConfig.transportMode == value:
      return

   setIntfAttr( mode, attr, value, no )
   # Removing all unicast-negotiation profiles as they are related to the
   # transport mode
   intfConfig.ucastNegGrantorIpToProfileMap.clear()
   intfConfig.ucastNegGranteeIpToProfileMap.clear()

#-------------------------------------------------------------------------------
# [no|default] ptp enable
#-------------------------------------------------------------------------------
def enablePtp( mode, no=False ):
   if ptpConfig.ptpMode == PtpMode.ptpGeneralized and not no and \
      Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( mode.intf.name ):
      mode.addError( "gPTP is not supported on port-channels" )
      return

   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name, createEnabled=not no )
   intfConfig.enabled = not no

#-------------------------------------------------------------------------------
# [no|default] ptp enable synctest (hidden)
#-------------------------------------------------------------------------------
def enableSyncTest( mode, no=False ):
   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )
   if not no:
      synctestPorts = 0
      for portDS in ptpStatus.portDS.values():
         # TODO BUG344577
         if portDS.vlanId != 0:
            continue
         if not portDS.syncTestRole == "syncTestDisabled":
            synctestPorts += 1
      portsAvailable = synctestPorts < 2
      intfConfig.syncTestEnabled = portsAvailable
      if not portsAvailable:
         print "Reached limit of interfaces that can be configured in synctest mode"
         return
   else:
      intfConfig.syncTestEnabled = False

   intfConfig.enabled = not no

def getClockIdentityString( clockIdentity ):
   return "0x%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x" % \
       ( clockIdentity.v0, clockIdentity.v1,
         clockIdentity.v2, clockIdentity.v3, clockIdentity.v4, clockIdentity.v5,
         clockIdentity.v6, clockIdentity.v7 )

#-------------------------------------------------------------------------------
# [no|default] ptp vlan [ ( all | <vlan-range> ) ]
#-------------------------------------------------------------------------------
def allAllowedVlanIds( mode ):
   switchIntfConfig = bridgingSic.switchIntfConfig.get( mode.intf.name )
   if switchIntfConfig is None:
      return set()
   return computeVlanRangeSet( switchIntfConfig.trunkAllowedVlans )

def setPtpVlan( mode, args ):
   if 'all' in args:
      vlanSet = allAllowedVlanIds( mode )
   elif 'VLAN_SET' in args:
      vlanSet = args[ 'VLAN_SET' ].ids_
   else:
      vlanSet = None
   no = CliCommand.isNoOrDefaultCmd( args )

   ( intfConfig, _ ) = setIntfDefaults( mode.intf.name )

   # By default, when `ptp vlan` is not configured, only one PortDS with vlanId=0 as
   # key is created. This is the defaultVlanId.
   defaultVlanId = CliConstants.defaultPortDSVlanId

   if no and vlanSet is None:
      # `no ptp vlan` revert to the initial state, i.e. only vlanConfig[ 0 ] exists
      intfConfig.vlanConfig.clear()
      intfConfig.vlanConfig.newMember( defaultVlanId, True, True )
      return
   elif no and defaultVlanId in intfConfig.vlanConfig:
      # `no ptp vlan <vlanId>` without previous `ptp vlan` configuration, skipping
      return
   elif defaultVlanId in intfConfig.vlanConfig:
      # `ptp vlan <vlan-range>`

      # We are limited in the number of timestamped packet we can send (200 pkts/sec)
      # We need to prevent that the Sync Interval * Configured VLANs exceeds this
      # limit.
      if defaultVlanId not in intfConfig.vlanConfig:
         configuredVlans = len( set( intfConfig.vlanConfig ) | vlanSet )
      else:
         # We will delete the defaultVlanId entry so just need to count the number
         # of vlans in vlanSet
         configuredVlans = len( vlanSet )

      rate = ( 1 / ( 2**intfConfig.logSyncInterval ) ) * configuredVlans
      maxRate = 1 / CliConstants.minEgressTsFifoPollInterval # = 200
      if rate > maxRate:
         mode.addError( "Number of PTP configured VLANs on a trunk is too high."
                        " Please consider increasing the sync messages interval or"
                        " configuring less VLANs" )
         return

      # Once the user configure with the `ptp vlan` CLI cmd, we are removing the old
      # default behaviour to only send untagged packets. If the user wants to send
      # untagged packets, he'll have to configure `ptp vlan <nativeVlan>` manually
      del intfConfig.vlanConfig[ defaultVlanId ]

   for vlanId in vlanSet:
      vlan = VlanCli.Vlan( vlanId )
      if no and vlan.id in intfConfig.vlanConfig: # Deleting
         del intfConfig.vlanConfig[ vlan.id ]
      elif not no and not vlan.id in intfConfig.vlanConfig: # Adding
         switchIntfConfig = bridgingSic.switchIntfConfig.get( mode.intf.name )
         trunkAllowed = switchIntfConfig is not None and \
            ( vlan.id in computeVlanRangeSet( switchIntfConfig.trunkAllowedVlans ) )
         vlanCreated = vlanId in brConfig.vlanConfig

         intfConfig.vlanConfig.newMember( vlan.id, trunkAllowed, vlanCreated )

#-------------------------------------------------------------------------------
# ptp boundary ( unicast-negotiation | multicast )
#
# BUG284794 - This is not something needed for now, just comment this out.
#-------------------------------------------------------------------------------
def boundaryUnicastNegotiation( mode, ptpBoundaryModeType ):
   if ptpConfig.ptpMode == "ptpBoundaryClock":
      ptpConfig.unicastNegotiation = ptpBoundaryModeType == "unicast-negotiation"
   else:
      mode.addWarning( "In order to configure to unicast-negotiation or multicast, "
                       "you need to be in boundary mode first." )

#===============================================================================
#   Show Commands
#===============================================================================
ptpMatcherForShow = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Show Precision Time Protocol information',
   guard=ptpSupportedGuard )

# This is for timesync CLI which do not need to depend on PTP supported guard.
# They will be guarded at the next token by timeSync guard
ptpMatcherForPlatformShow = CliMatcher.KeywordMatcher( 'ptp',
      helpdesc='Show platform specific Precision Time Protocol information' )

#-------------------------------------------------------------------------------
# show ptp hold ptp time interval
#-------------------------------------------------------------------------------
def showHoldPtpTime( mode, args ):
   model = PtpHoldPtpTimeForShow()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return model
   model.ptpMode = ptpStatus.defaultDS.ptpMode
   model.holdPtpTimeInterval = None
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return model
   model.holdPtpTimeInterval = ptpStatus.defaultDS.holdPtpTimeInterval
   model.holdPtpTimeState = ptpStatus.holdoverState
   model.holdPtpTimeSinceEpoch = ptpStatus.holdoverSinceEpoch
   model.holdPtpTimeExpiry = ptpStatus.holdoverExpiry
   return model

#-------------------------------------------------------------------------------
# show ptp local-clock
#-------------------------------------------------------------------------------
def showClock( mode ):
   ptpClockModel = PtpClock()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return ptpClockModel
   ptpClockModel.ptpMode = ptpStatus.defaultDS.ptpMode
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return ptpClockModel
   ptpOrdinaryOrBC = None
   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock',
                                       'ptpOrdinaryMasterClock',
                                       'ptpGeneralized' ]:
      ptpOrdinaryOrBC = PtpClock.PtpOrdinaryOrBoundaryClock()
      clockIdentity = ptpStatus.defaultDS.clockIdentity
      clockIdentityStr = "0x%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x" % \
         ( clockIdentity.v0, clockIdentity.v1,
         clockIdentity.v2, clockIdentity.v3, clockIdentity.v4, clockIdentity.v5,
         clockIdentity.v6, clockIdentity.v7 )
      ptpOrdinaryOrBC.clockIdentity = clockIdentityStr
      ptpOrdinaryOrBC.domainNumber = ptpStatus.defaultDS.domainNumber
      ptpOrdinaryOrBC.numberPorts = ptpStatus.defaultDS.numberPorts
      ptpOrdinaryOrBC.priority1 = ptpStatus.defaultDS.priority1
      ptpOrdinaryOrBC.priority2 = ptpStatus.defaultDS.priority2
      if ptpStatus.defaultDS.ptpMode == 'ptpBoundaryClock' and \
            ptpConfig.ptpProfile in [ PtpProfile.ptpG8275_1,
                  PtpProfile.ptpG8275_2 ]:
         ptpOrdinaryOrBC.localPriority = ptpStatus.defaultDS.localPriority
      clockClass = ptpStatus.defaultDS.clockQuality.clockClass
      accuracy = ptpStatus.defaultDS.clockQuality.clockAccuracy
      offsetScaledLogVariance = \
          ptpStatus.defaultDS.clockQuality.offsetScaledLogVariance.value
      clockQuality = PtpClockQuality()
      clockQuality.clockClass = clockClass
      clockQuality.accuracy = accuracy
      clockQuality.offsetScaledLogVariance = offsetScaledLogVariance
      ptpOrdinaryOrBC.clockQuality = clockQuality
      if ptpStatus.defaultDS.ptpMode == 'ptpOrdinaryMasterClock':
         ptpClockModel.ptpOrdinaryOrBoundaryClock = ptpOrdinaryOrBC
         return ptpClockModel
      bcProperties = PtpClock.PtpOrdinaryOrBoundaryClock.PtpBoundaryClockProperties()
      bcProperties.offsetFromMaster = int( ptpStatus.currentDS.offsetFromMaster )
      bcProperties.meanPathDelay = int( ptpStatus.currentDS.meanPathDelay )
      bcProperties.stepsRemoved = ptpStatus.currentDS.stepsRemoved.value
      if ptpStatus.defaultDS.ptpMode != 'ptpGeneralized':
         bcProperties.skew = ptpStatus.currentDS.skew
      else:
         bcProperties.neighborRateRatio = ptpStatus.currentDS.neighborRateRatio
         bcProperties.gptpRateRatio = ptpStatus.currentDS.gptpRateRatio
         bcProperties.gptpCumulativeRateOffset = \
             ptpStatus.currentDS.gptpCumulativeRateOffset
      if ptpStatus.currentDS.lastSyncLocalTime:
         masterTime = int( ptpStatus.currentDS.lastSyncLocalTime -
                           ptpStatus.currentDS.offsetFromHardware )
         seconds = masterTime / 1000000000
         if ptpStatus.timePropertiesDS.ptpTimescale and \
                ptpStatus.timePropertiesDS.currentUtcOffsetValid:
            seconds = seconds - ptpStatus.timePropertiesDS.currentUtcOffset.value
         bcProperties.lastSyncTime = int( seconds )
         estimatedTime = seconds + Tac.now() - ptpStatus.currentDS.lastSyncSystemTime
         bcProperties.currentPtpSystemTime = int( estimatedTime )
      ptpOrdinaryOrBC.boundaryClockProperties = bcProperties
   ptpClockModel.ptpOrdinaryOrBoundaryClock = ptpOrdinaryOrBC
   return ptpClockModel

#-------------------------------------------------------------------------------
# show ptp ip access-list
#-------------------------------------------------------------------------------
def showPtpIpAcl( mode, args ):
   return AclCli.showServiceAcl( mode, ptpConfig.serviceAclTypeVrfMap,
                                 ptpStatus.aclStatusService, aclCheckpoint, 'ip',
                                 ( None, None ), supressVrf=True,
                                 vrfUnaware=True )

#-------------------------------------------------------------------------------
# show ptp masters
#-------------------------------------------------------------------------------
def showParent( mode, args ):
   # if the PTP agent never ran, defaultDs is None
   parentWrapper = PtpParentWrapper()
   if not ptpStatus.defaultDS:
      parentWrapper.message = 'PTP is not configured'
      return parentWrapper

   if ptpStatus.defaultDS.ptpMode not in ['ptpBoundaryClock', 'ptpGeneralized']:
      parentWrapper.message = 'Not a boundary clock, no parent clock information.'
      return parentWrapper

   parent = PtpParent()
   clockIdentity = \
         ptpStatus.parentDS.parentSourceAddrs.parentPortIdentity.clockIdentity
   clockIdentityStr = "0x%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x" % \
      ( clockIdentity.v0, clockIdentity.v1,
      clockIdentity.v2, clockIdentity.v3, clockIdentity.v4, clockIdentity.v5,
      clockIdentity.v6, clockIdentity.v7 )
   parent.parentClockIdentity = clockIdentityStr
   parent.parentPortNumber = \
         ptpStatus.parentDS.parentSourceAddrs.parentPortIdentity.portNumber
   if ptpStatus.parentDS.parentSourceAddrs.parentIpAddr != \
         CliConstants.parentIpAddrDefault:
      if ptpStatus.parentDS.parentSourceAddrs.parentIpAddr.af == 'ipv4':
         parent.parentIpAddr = \
               ptpStatus.parentDS.parentSourceAddrs.parentIpAddr.v4Addr
      else:
         parent.parentIp6Addr = \
               ptpStatus.parentDS.parentSourceAddrs.parentIpAddr.v6Addr
   clockIdentity = ptpStatus.parentDS.grandmasterIdentity
   clockIdentityStr = "0x%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x" % \
      ( clockIdentity.v0, clockIdentity.v1,
      clockIdentity.v2, clockIdentity.v3, clockIdentity.v4, clockIdentity.v5,
      clockIdentity.v6, clockIdentity.v7 )
   parent.gmClockIdentity = clockIdentityStr
   gmClockQuality = PtpClockQuality()
   gmClockQuality.clockClass = ptpStatus.parentDS.grandmasterClockQuality.clockClass
   gmClockQuality.accuracy = ptpStatus.parentDS.grandmasterClockQuality.clockAccuracy
   gmClockQuality.offsetScaledLogVariance = \
       ptpStatus.parentDS.grandmasterClockQuality.offsetScaledLogVariance.value
   parent.gmClockQuality = gmClockQuality
   parent.gmPriority1 = ptpStatus.parentDS.grandmasterPriority1
   parent.gmPriority2 = ptpStatus.parentDS.grandmasterPriority2
   parentWrapper.parent = parent
   return parentWrapper

#-------------------------------------------------------------------------------
# show ptp unicast-negotiation candidate-grantor|remote-grantee
#-------------------------------------------------------------------------------
def showCandidateGrantors( mode, args ):
   ucastNegCandidateGrantors = UcastNegCandidateGrantor()
   for intf in sorted( intfConfigIntfList() ):
      intfConfig = ptpConfig.intfConfig.get( intf )
      key = Tac.newInstance( "Ptp::PortDSKey", intf,
                             CliConstants.defaultPortDSVlanId )
      portDS = ptpStatus.portDS.get( key )
      if not intfConfig or not portDS:
         continue

      profileMap = intfConfig.ucastNegGrantorIpToProfileMap
      for ip, profile in sorted( profileMap.iteritems() ):
         entry = UcastNegCandidateGrantorEntry()
         entry.intf = intf
         entry.ipAddr = ip
         entry.profileName = profile
         if ip in portDS.g8275_2.signalFail:
            entry.grantorState = UcastNegGrantorState.ucastNegBlacklisted
         elif ( portDS.portState == 'psSlave' and
                ptpStatus.parentDS.parentSourceAddrs.parentIpAddr == ip ):
            entry.grantorState = UcastNegGrantorState.ucastNegMaster
         else:
            entry.grantorState = UcastNegGrantorState.ucastNegCandidateMaster
         ucastNegCandidateGrantors.candidateGrantors.append( entry )
   return ucastNegCandidateGrantors

def showRemoteGrantees( mode, args ):
   ucastNegRemoteGrantees = UcastNegRemoteGrantee()
   for intf in sorted( intfConfigIntfList() ):
      intfConfig = ptpConfig.intfConfig.get( intf )
      if not intfConfig:
         continue

      profileMap = intfConfig.ucastNegGranteeIpToProfileMap
      for ipAndMask, profile in sorted( profileMap.iteritems() ):
         entry = UcastNegRemoteGranteeEntry()
         entry.intf = intf
         entry.ipAddrAndMask = ipAndMask
         entry.profileName = profile
         ucastNegRemoteGrantees.remoteGrantees.append( entry )
   return ucastNegRemoteGrantees

#-------------------------------------------------------------------------------
# show ptp unicast-negotiation profile [<name>]
#-------------------------------------------------------------------------------
def showUcastNegProfiles( mode, args ):
   profileName = args.get( 'PROFILE' )
   profileKeys = None
   profileList = {}
   if profileName is None:
      profileKeys = ptpConfig.ucastNegProfile.keys()
   else:
      profileKeys = profileName.split()

   profiles = ptpConfig.ucastNegProfile
   for profileKey in profileKeys:
      tmp = UcastNegProfile()
      if profileKey in profiles:
         profile = profiles.get( profileKey )
         tmp.announceInterval = profile.announceProfile.logInterval
         tmp.announceDuration = profile.announceProfile.duration
         tmp.syncInterval = profile.syncProfile.logInterval
         tmp.syncDuration = profile.syncProfile.duration
         tmp.delayResponseInterval = profile.delayRespProfile.logInterval
         tmp.delayResponseDuration = profile.delayRespProfile.duration
         profileList[ profileKey ] = tmp

   return UcastNegProfiles( profiles=profileList )

#-------------------------------------------------------------------------------
# show ptp unicast-negotiation granted|requested
#-------------------------------------------------------------------------------
def getUcastGrantsAndRequests( granted ):
   interfaces = {}
   for intf in sorted( intfConfigIntfList() ):
      key = Tac.newInstance( "Ptp::PortDSKey", intf,
                             CliConstants.defaultPortDSVlanId )
      portDS = ptpStatus.portDS.get( key )
      if not portDS:
         continue

      ucastNegMessages = UcastNegMessages()
      messagesIpv4 = []
      messagesIpv6 = []
      if granted:
         messageStatus = { ucastKey:ucastStatus for ucastKey, ucastStatus in
                           ucastNegStatus.grant.iteritems()
                           if ucastKey.intf == intf }
      else:
         messageStatus = { ucastKey:ucastStatus for ucastKey, ucastStatus in
                           ucastNegStatus.request.iteritems()
                           if ucastKey.intf == intf }
      sigFail = portDS.g8275_2.signalFail
      for ucastKey, ucastStatus in messageStatus.iteritems():
         if ucastKey.ipAddr in sigFail:
            continue
         tmp = UcastNegMessage()
         tmp.ipAddr = ucastKey.ipAddr
         tmp.messageType = ucastKey.messageType
         tmp.logInterval = ucastStatus.logInterval
         tmp.duration = ucastStatus.duration
         tmp.expirationTime = ucastStatus.expirationTime - Tac.now()
         if tmp.ipAddr.af == "ipv4":
            messagesIpv4.append( tmp )
         else:
            messagesIpv6.append( tmp )

      key = attrgetter( "ipAddr" )
      messagesIpv4 = sorted( messagesIpv4, cmp=IpUtils.compareIpAddress, key=key )
      messagesIpv6 = sorted( messagesIpv6, cmp=IpUtils.compareIp6Address, key=key )
      messages = messagesIpv4 + messagesIpv6

      ucastNegMessages.messages = messages
      interfaces[ intf ] = ucastNegMessages
   return UcastNegStatus( interfaces=interfaces )

def showGrantedSlaves( mode, args ):
   return getUcastGrantsAndRequests( True )

def showRequestedMasters( mode, args ):
   return getUcastGrantsAndRequests( False )

#-------------------------------------------------------------------------------
# show ptp time-property
#-------------------------------------------------------------------------------
def showTimeProperty( mode, args ):
   # if the PTP agent never ran, defaultDs is None
   timePropertyWrapper = PtpTimePropertyWrapper()
   if not ptpStatus.defaultDS:
      timePropertyWrapper.message = 'PTP is not configured'
      return timePropertyWrapper

   if ptpStatus.defaultDS.ptpMode not in \
          [ 'ptpBoundaryClock', 'ordinaryMaster', 'ptpGeneralized' ]:
      timePropertyWrapper.message = \
          "Not a boundary clock, no time-property information."
      return timePropertyWrapper

   timePropertiesDS = ptpStatus.timePropertiesDS
   timeProperty = PtpTimeProperty()
   timeProperty.currentUtcOffsetValid =  timePropertiesDS.currentUtcOffsetValid
   timeProperty.currentUtcOffset = timePropertiesDS.currentUtcOffset.value
   timeProperty.leap59 = timePropertiesDS.leap59
   timeProperty.leap61 = timePropertiesDS.leap61
   timeProperty.timeTraceable = timePropertiesDS.timeTraceable
   timeProperty.frequencyTraceable = timePropertiesDS.frequencyTraceable
   timeProperty.ptpTimescale = timePropertiesDS.ptpTimescale
   timeProperty.timeSource = timePropertiesDS.timeSource
   timePropertyWrapper.timeProperty = timeProperty
   return timePropertyWrapper


#---------------------------------------------------------------------------------
# show ptp interface [ INTF ] counters drop
#---------------------------------------------------------------------------------
def showIntfDropCounters( mode, args ):

   disabled = not ptpStatus.defaultDS or ptpStatus.defaultDS.ptpMode == 'ptpDisabled'

   if disabled:
      return PtpDebugCounters( disabled=disabled )

   intfs = args.get( 'INTF', [] )
   if intfs:
      intfs = [ name for name in intfs.intfNames() ]
   intfsAll = not intfs

   counters = { intf : PtpDebugIntfCounters() for intf in intfs }
   for key, value in debugCounter.counter.items():
      # retrive the PTP port from the portKeyMap
      portHashKey = Tac.Value( "Ptp::DebugCounter::CounterPortHashKey",
                               key.counterPortKeyHash )
      port = debugCounter.counterPortKeyMap[ portHashKey ].counterPortKey
      # only accept this counter if interface is specified
      # or when no interface is provided ( default to all interfaces)
      # no need to show sent/recv counter
      if port.intfId in intfs or intfsAll:
         endpoint = PtpEndpoint()
         endpoint.intf = port.intfId
         endpoint.vlanId = port.vlanId
         endpoint.domain = port.domain
         endpoint.ip = port.ip
         dropReason = PtpDebugCounterDropReason()
         dropReason.debugType = key.type
         dropReason.debugReason = key.reason
         direction = key.direction
         lastSeen = value.lastSeen
         counter = PtpDebugCounter( endpoint=endpoint, dropReason=dropReason,
                                    direction=direction, lastSeen=lastSeen,
                                    count=value.count )
         if intfsAll:
            value = counters.setdefault( port.intfId, PtpDebugIntfCounters() )
         else:
            value = counters[ port.intfId ]
         value.intfCounters.append( counter )

   return PtpDebugCounters( counters=counters )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>] counters
#-------------------------------------------------------------------------------
def getPktCounter( portDS, messageType, sentOrRecv ):
   # pkt counters should be saved with counterPortKeyConstant domain and Ip,
   # modify to support showing different domain/ip
   intf = Tac.Value( "Arnet::IntfId", portDS.intf )
   vlanId = Tac.Value( "Bridging::VlanIdOrNone", portDS.vlanId )
   ip = CounterPortKeyConstant.ip()
   domain = CounterPortKeyConstant.domain
   portKeyHash = Tac.Value( "Ptp::DebugCounter::CounterPortKey",
                            intf, vlanId, domain, ip ).hash()

   direction = CounterDir.rx
   reason = CounterReason.received
   if sentOrRecv == 'Sent':
      direction = CounterDir.tx
      reason = CounterReason.sent

   key = Tac.Value( 'Ptp::DebugCounter::CounterReasonKey', portKeyHash,
                    direction, messageType, reason )
   counterValue = pktCounter.counter.get( key )
   return counterValue.count if counterValue else 0

def showIntfCounters( mode, portDS ):
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return PtpIntfCountersWrapper()

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpIntfCountersWrapper()

   if not portDS.enabled:
      if ptpStatus.defaultDS.ptpMode != PtpMode.ptpGeneralized:
         return PtpIntfCountersWrapper()
      # In gptp, port is disabled if asCapable is false, but we
      # still want to show the counters
      for disabledReason in portDS.disabledReason.keys():
         if( disabledReason != 'asCapableFalse' and
             portDS.disabledReason[disabledReason] ):
            return PtpIntfCountersWrapper()


   intfCounters = PtpIntfCounters()
   if ptpStatus.defaultDS.ptpMode in ['ptpBoundaryClock', 'ptpGeneralized']:
      intfCounters.announceTx = \
         getPktCounter( portDS, CounterType.messageAnnounce, 'Sent' )
      intfCounters.announceRx = \
         getPktCounter( portDS, CounterType.messageAnnounce, 'Received' )
   intfCounters.syncTx = getPktCounter( portDS, CounterType.messageSync, 'Sent' )
   intfCounters.syncRx = getPktCounter( portDS, CounterType.messageSync, 'Received' )
   intfCounters.followUpTx = \
      getPktCounter( portDS, CounterType.messageFollowUp, 'Sent' )
   intfCounters.followUpRx = \
      getPktCounter( portDS, CounterType.messageFollowUp, 'Received' )
   intfCounters.delayReqTx = \
      getPktCounter( portDS, CounterType.messageDelayReq, 'Sent' )
   intfCounters.delayReqRx = \
      getPktCounter( portDS, CounterType.messageDelayReq, 'Received' )
   intfCounters.delayRespTx = \
      getPktCounter( portDS, CounterType.messageDelayResp, 'Sent' )
   intfCounters.delayRespRx = \
      getPktCounter( portDS, CounterType.messageDelayResp, 'Received' )
   intfCounters.pDelayReqTx = \
      getPktCounter( portDS, CounterType.messagePDelayReq, 'Sent' )
   intfCounters.pDelayReqRx = \
      getPktCounter( portDS, CounterType.messagePDelayReq, 'Received' )
   intfCounters.pDelayRespTx = \
      getPktCounter( portDS, CounterType.messagePDelayResp, 'Sent' )
   intfCounters.pDelayRespRx = \
      getPktCounter( portDS, CounterType.messagePDelayResp, 'Received' )
   intfCounters.pDelayRespFollowUpTx = \
       getPktCounter( portDS, CounterType.messagePDelayRespFollowup, 'Sent' )
   intfCounters.pDelayRespFollowUpRx = \
       getPktCounter( portDS, CounterType.messagePDelayRespFollowup, 'Received' )
   if ptpStatus.defaultDS.ptpMode == 'ptpBoundaryClock':
      intfCounters.managementTx = \
         getPktCounter( portDS, CounterType.messageManagement, 'Sent' )
      intfCounters.managementRx = \
         getPktCounter( portDS, CounterType.messageManagement, 'Received' )
      intfCounters.signalingTx = \
         getPktCounter( portDS, CounterType.messageSignaling, 'Sent' )
      intfCounters.signalingRx = \
         getPktCounter( portDS, CounterType.messageSignaling, 'Received' )


   if ptpStatus.defaultDS.ptpMode == 'ptpGeneralized':
      intfCounters.gptpFollowUpMessagesTxMissed = \
         portDS.gptpFollowUpMessagesSentMissed

   vlanId = None
   if ptpStatus.defaultDS.ptpMode == 'ptpBoundaryClock' and portDS.vlanId != 0:
      switchIntfConfig = bridgingSic.switchIntfConfig.get( portDS.intf )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      if isIntfTrunkMode:
         vlanId = portDS.vlanId

   return PtpIntfCountersWrapper( interfaceCounters=intfCounters,
                                  vlanId=vlanId )

def showIntfListCounters( mode, args ):
   intfList = args.get( 'INTF' )
   vlanId = args.get( 'VLAN_ID' )
   if not ptpStatus.defaultDS:
      return PtpCounters( interfaceCounters={} )
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpCounters( interfaceCounters={} )

   vlan = vlanId.id if vlanId is not None else None
   keyList = sorted( portDSIntfList( intfList, vlan ) )
   intfCounters = {}
   for key in keyList:
      portDS = ptpStatus.portDS.get( key )
      if not portDS:
         continue
      intf = portDS.intf
      intfCounter = showIntfCounters( mode, portDS )
      if intfCounter.interfaceCounters is not None:
         if intf not in intfCounters:
            intfCounters[ intf ] = PtpIntfVlansCounters( interface=intf )
         intfCounters[ intf ].ptpInterfaceVlansCounters.append( intfCounter )
   return PtpCounters( interfaceCounters=intfCounters )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>] vlans
#-------------------------------------------------------------------------------

def _getConfiguredVlans( intf ):
   ''' This function returns 3 lists and 1 boolean
   vlans: Actual PTP configured vlans, where ptp packets will be sent or received
   notCreatedVlans: Vlans that are not created ( received pkts will be dropped )
   notAllowedVlans: Vlans not allowed in trunk port
   '''
   ( vlans, notCreatedVlans, notAllowedVlans )  = ( [], [], [] )
   for portDSKey in ptpStatus.vlanStatus:
      if portDSKey.intf != intf:
         continue

      # If the intf is not in trunk mode, just don't display anything
      switchIntfConfig = bridgingSic.switchIntfConfig.get( portDSKey.intf )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      if not isIntfTrunkMode:
         continue

      vs = ptpStatus.vlanStatus.get( portDSKey )
      vlanId = portDSKey.vlanId
      # If vlanId == 0, we have not configured any PTP VLAN to run and no other
      # PortDSKey of this interface exist.
      if vlanId == 0:
         continue
      elif vs.trunkAllowed and vs.vlanCreated:
         vlans.append( vlanId )
      else:
         if not vs.vlanCreated:
            notCreatedVlans.append( vlanId )
         if not vs.trunkAllowed:
            notAllowedVlans.append( vlanId )
   return ( vlans, notCreatedVlans, notAllowedVlans )

def showIntfListVlans( mode, args ):
   intfList = args.get( 'INTF' )
   if not ptpStatus.defaultDS:
      return PtpVlans( interfaces={}, message="PTP is not configured" )
   if ptpStatus.defaultDS.ptpMode != 'ptpBoundaryClock':
      return PtpVlans( interfaces={},
                       message="Not a boundary clock, no PTP vlan information" )
   if intfList:
      intfList = intfList.intfNames()
   else:
      intfList = set( [ key.intf for key in ptpStatus.vlanStatus ] )

   ptpIntfVlans = {}
   for intf in intfList:
      ( vlans, notCreatedVlans, notAllowedVlans ) = _getConfiguredVlans( intf )
      if not vlans and not notCreatedVlans and not notAllowedVlans:
         continue

      ptpIntfVlans[ intf ] = PtpVlansExtendedWrapper(
                                vlans=vlans,
                                notCreatedVlans=notCreatedVlans,
                                notAllowedVlans=notAllowedVlans )
   return PtpVlans( interfaces=ptpIntfVlans )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>] vlan <vlanId>
#-------------------------------------------------------------------------------
def showIntfListVlan( mode, args ):
   intfList = args[ 'INTF' ]
   vlanId = args[ 'VLAN_ID' ]
   if not ptpStatus.defaultDS:
      return PtpInterfaces( interfaces={} )

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpInterfaces( interfaces={} )

   keyList = sorted( portDSIntfList( intfList, vlanId.id ) )
   ptpIntfs = {}
   for key in keyList:
      portDS = ptpStatus.portDS.get( key )
      if not portDS:
         continue
      intf = portDS.intf
      ptpIntf = showIntf( mode, portDS, None )
      if ptpIntf.interface is not None:
         if intf not in ptpIntfs:
            ptpIntfs[ intf ] = PtpIntfVlans( interface=intf )
         ptpIntfs[ intf ].ptpInterfaceVlansStates.append( ptpIntf )
   return PtpInterfaces( interfaces=ptpIntfs )

#-------------------------------------------------------------------------------
# show ptp interface [<interface list>]
#-------------------------------------------------------------------------------
def getAsCapableLastChanged( lastUpdateTime ):
   if lastUpdateTime == 0:
      return 'Never'
   lastMoveDelta = int( Tac.utcNow() - lastUpdateTime )
   return str( datetime.timedelta( seconds=lastMoveDelta ) )

def showIntf( mode, portDS, enabled ):
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return PtpIntfWrapper()

   if portDS.adminDisabled and enabled:
      return PtpIntfWrapper()

   ptpIntf = PtpIntf()
   ptpIntf.adminDisabled = portDS.adminDisabled
   ptpIntf.syncTestRole = portDS.syncTestRole
   ptpIntf.ptpMode = ptpStatus.defaultDS.ptpMode
   ptpIntf.counters = showIntfCounters( mode, portDS )
   if ptpStatus.defaultDS.ptpMode == 'ptpEndToEndTransparentClock':
      return PtpIntfWrapper( interface=ptpIntf )

   # The 'ptp vlan' cmd only works in boundary mode
   # According to gPTP spec ( 11.3.3 ) IEEE 802.1AS message shall not have a VLAN tag
   if ptpStatus.defaultDS.ptpMode == 'ptpBoundaryClock' and portDS.vlanId != 0:
      switchIntfConfig = bridgingSic.switchIntfConfig.get( portDS.intf )
      isIntfTrunkMode = switchIntfConfig is not None and \
                        switchIntfConfig.switchportMode == 'trunk'
      if isIntfTrunkMode:
         ptpIntf.vlanId = portDS.vlanId

   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock', 'ptpGeneralized' ]:
      ptpIntf.portState = portDS.portState
      if ptpStatus.defaultDS.ptpMode == PtpMode.ptpGeneralized:

         ptpIntf.asCapable = portDS.asCapable
         ptpIntf.timeSinceAsCapableLastChanged = \
             getAsCapableLastChanged( portDS.lastUpdateTimeAsCapable )
         if ptpIntf.timeSinceAsCapableLastChanged != 'Never':
            ptpIntf.timeSinceAsCapableLastChanged += ' ago'

         if portDS.portState == 'psMaster':
            ptpIntf.lastGptpResidenceTime = portDS.lastGptpResidenceTime
            ptpIntf.minGptpResidenceTime = portDS.minGptpResidenceTime
            ptpIntf.maxGptpResidenceTime = portDS.maxGptpResidenceTime
      if not ptpStatus.unicastNegotiation:
         ptpIntf.syncInterval =  math.pow( 2, portDS.logSyncInterval )
         ptpIntf.announceInterval = math.pow( 2, portDS.logAnnounceInterval )
         ptpIntf.announceReceiptTimeout = portDS.announceReceiptTimeout
      if ptpStatus.defaultDS.ptpMode == 'ptpGeneralized':
         ptpIntf.syncReceiptTimeout = portDS.syncReceiptTimeout
   ptpIntf.delayMechanism = portDS.delayMechanism
   if portDS.delayMechanism == 'e2e' and \
      ptpStatus.defaultDS.ptpMode == 'ptpBoundaryClock' and \
      not ptpStatus.unicastNegotiation:
      ptpIntf.delayReqInterval = math.pow(
         2, portDS.logMinDelayReqInterval )
   elif portDS.delayMechanism == 'p2p':
      ptpIntf.peerDelayReqInterval = math.pow(
         2, portDS.logMinPdelayReqInterval )
      ptpIntf.peerPathDelay = portDS.peerMeanPathDelay

   if ptpConfig.ptpProfile in [ PtpProfile.ptpG8275_1, PtpProfile.ptpG8275_2 ]:
      ptpIntf.localPriority = portDS.localPriority
   if ptpStatus.g82751Enabled:
      intfConfig = ptpConfig.intfConfig.get( portDS.intf )
      if intfConfig is not None:
         ptpIntf.mcastTxAddr = intfConfig.pktTxMacAddress

   if ptpStatus.defaultDS.ptpMode == 'ptpBoundaryClock' or \
          ptpStatus.defaultDS.ptpMode == 'ptpGeneralized':
      ptpIntf.transportMode = portDS.transportMode

   if Tac.Type( 'Arnet::PortChannelIntfId' ).isPortChannelIntfId( portDS.intf ) and \
         portDS.txLagMember is not None:
      ptpIntf.txLagMember = portDS.txLagMember

   return PtpIntfWrapper( interface=ptpIntf )

def showIntfList( mode, args ):
   intfList = args.get( 'INTF' )
   enabled = 'enabled' in args
   if not ptpStatus.defaultDS:
      return PtpInterfaces( interfaces={} )

   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return PtpInterfaces( interfaces={} )

   keyList = sorted( portDSIntfList( intfList ) )
   ptpIntfs = {}
   for key in keyList:
      portDS = ptpStatus.portDS.get( key )
      if not portDS:
         continue
      intf = portDS.intf
      ptpIntf = showIntf( mode, portDS, enabled )
      if ptpIntf.interface is not None:
         if intf not in ptpIntfs:
            ptpIntfs[ intf ] = PtpIntfVlans( interface=intf )
         ptpIntfs[ intf ].ptpInterfaceVlansStates.append( ptpIntf )
   return PtpInterfaces( interfaces=ptpIntfs )

#-------------------------------------------------------------------------------
# show ptp
#-------------------------------------------------------------------------------
def showIntfSummary( mode, portDS ):
   if not ptpStatus.defaultDS:
      return None
   if portDS.adminDisabled:
      return None
   intfSummary = PtpIntfSummary()
   intfSummary.transportMode = portDS.transportMode
   if mode in [ 'ptpBoundaryClock', 'ptpGeneralized' ]:
      intfSummary.portState = portDS.portState
      intfSummary.delayMechanism = portDS.delayMechanism
      if mode == 'ptpGeneralized':
         intfSummary.asCapable = portDS.asCapable
         intfSummary.timeSinceAsCapableLastChanged = \
             getAsCapableLastChanged( portDS.lastUpdateTimeAsCapable )
         intfSummary.lastGptpResidenceTime = portDS.lastGptpResidenceTime
         intfSummary.peerPathDelay = portDS.peerMeanPathDelay
         intfSummary.neighborRateRatio = portDS.neighborRateRatio
      elif portDS.vlanId != 0:
         intfSummary.vlanId = portDS.vlanId

   return intfSummary

def showPtpSummary( mode, args ):
   ptpSummary = PtpSummary()
   # if the PTP agent never ran, defaultDs is None
   if not ptpStatus.defaultDS:
      return ptpSummary
   ptpSummary.ptpMode = ptpStatus.defaultDS.ptpMode
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return ptpSummary
   ptpClockSummary = None
   if ptpStatus.defaultDS.ptpMode in [ 'ptpBoundaryClock',
                                       'ptpGeneralized' ]:
      ptpClockSummary = ptpSummary.PtpClockSummary()
      ptpClockSummary.numberOfSlavePorts = 0
      ptpClockSummary.numberOfMasterPorts = 0
      clock = showClock( ptpStatus.defaultDS.ptpMode ).ptpOrdinaryOrBoundaryClock
      ptpClockSummary.clockIdentity = clock.clockIdentity
      ptpClockSummary.gmClockIdentity = getClockIdentityString(
         ptpStatus.parentDS.grandmasterIdentity )
      ptpClockSummary.stepsRemoved = clock.boundaryClockProperties.stepsRemoved
      ptpClockSummary.meanPathDelay = clock.boundaryClockProperties.meanPathDelay
      if ptpSummary.ptpMode == 'ptpBoundaryClock':
         ptpSummary.ptpProfile = ptpConfig.ptpProfile
         ptpClockSummary.skew = clock.boundaryClockProperties.skew
         ptpClockSummary.lastSyncTime = clock.boundaryClockProperties.lastSyncTime
         ptpClockSummary.currentPtpSystemTime = \
             clock.boundaryClockProperties.currentPtpSystemTime
         ptpClockSummary.offsetFromMaster = \
             clock.boundaryClockProperties.offsetFromMaster
      else:
         ptpClockSummary.gptpRateRatio = clock.boundaryClockProperties.gptpRateRatio
         ptpClockSummary.neighborRateRatio = \
             clock.boundaryClockProperties.neighborRateRatio
   ptpSummary.ptpClockSummary = ptpClockSummary

   ptpSummary.ptpIntfSummaries = {}
   for portDS in ptpStatus.portDS.itervalues():
      intf = portDS.intf
      if ptpClockSummary:
         if portDS.portState == 'psMaster':
            ptpClockSummary.numberOfMasterPorts += 1
         elif portDS.portState == 'psSlave':
            ptpClockSummary.numberOfSlavePorts += 1
            ptpClockSummary.slavePort = intf
            ptpClockSummary.slaveVlanId = portDS.vlanId
      ptpIntfSummary = showIntfSummary( ptpSummary.ptpMode, portDS )
      if ptpIntfSummary:
         if intf not in ptpSummary.ptpIntfSummaries:
            ptpSummary.ptpIntfSummaries[ intf ] = PtpIntfVlanSummary(
                  interface=intf )
         ptpSummary.ptpIntfSummaries[ intf ].ptpIntfVlanSummaries.append(
               ptpIntfSummary )
   return ptpSummary

ptpMatcherForExec = CliCommand.guardedKeyword(
   'ptp',
   helpdesc='Precision Time Protocol',
   guard=ptpSupportedGuard )

# This is for timesync CLI which do not need to depend on PTP supported guard.
# They will be guarded at the next token by timeSync guard
ptpMatcherForReset = CliMatcher.KeywordMatcher(
   'ptp',
   helpdesc='Precision Time Protocol' )

#-------------------------------------------------------------------------------
# "clear ptp interface [<interface list>] counters" in enable mode
#-------------------------------------------------------------------------------
def clearCounters( mode, intfList=None, vlanId=None ):
   # if intfList is None, clean all intf
   # if vlanId is None clean all vlanId
   if intfList:
      intfList = set( intfList.intfNames() )

   if vlanId:
      vlanId = vlanId.id
   # fill clearLists with the pkt counter and drop counter to clear
   counters = [ debugCounter, pktCounter ]
   counterClearLists = [ ptpConfig.dropCounterClearList,
                         ptpConfig.pktCounterClearList ]

   for ( counter, counterClearList ) in zip( counters, counterClearLists ):
      for key in counter.counter:
         portHashKey = Tac.Value( "Ptp::DebugCounter::CounterPortHashKey",
                                  key.counterPortKeyHash )
         portKey = counter.counterPortKeyMap[ portHashKey ].counterPortKey
         if ( intfList is None or portKey.intfId in intfList ) and \
            ( vlanId is None or portKey.vlanId == vlanId ):
            counterClearList.addKey( key )

   # clear the counters
   ptpConfig.clearCountersRequest = True

   try:
      Tac.waitFor(
         lambda: ptpStatus.clearCountersRequestCompleted,
         description='PTP clear interface counter request to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      print 'PTP counters may not have been reset yet'

   for counterClearList in counterClearLists:
      counterClearList.clearList()
   ptpConfig.clearCountersRequest = False

def clearIntfCounters( mode, portDS ):
   intfConfig = ptpConfig.intfConfig.get( portDS.intf )
   if not intfConfig:
      return
   assert hasattr( intfConfig, 'clearCountersRequestTime' )
   intfConfig.clearCountersRequestTime = Tac.now()
   try:
      Tac.waitFor(
         lambda:
            portDS.lastClearTime >= intfConfig.clearCountersRequestTime,
         description='PTP clear interface counter request to complete',
         warnAfter=None, sleep=True, maxDelay=0.5, timeout=5 )
   except Tac.Timeout:
      print 'PTP interface', portDS.intf, 'counters may not have been reset yet'

def clearIntfListCounters( mode, args ):
   intfList = args.get( 'INTF' )
   vlanId = args.get( 'VLAN_ID' )
   if not ptpStatus.defaultDS:
      return
   if ptpStatus.defaultDS.ptpMode == 'ptpDisabled':
      return

   vlan = vlanId.id if vlanId is not None else None
   keyList = portDSIntfList( intfList, vlanId=vlan )
   for key in keyList:
      portDS = ptpStatus.portDS.get( key )
      if not portDS:
         continue
      clearIntfCounters( mode, portDS )
   # clear the drop counter and pkt counter
   clearCounters( mode, intfList=intfList, vlanId=vlanId )

class ClearIntfListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ptp interface [ INTF ] [ vlan VLAN_ID ] counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ptp': ptpMatcherForExec,
      'interface': 'Details on PTP',
      'INTF': IntfCli.Intf.rangeMatcher,
      'vlan': 'Show PTP VLAN information',
      'VLAN_ID': VlanCli.vlanIdMatcher,
      'counters': 'Show PTP counter information',
   }
   handler = clearIntfListCounters

BasicCliModes.EnableMode.addCommandClass( ClearIntfListCountersCmd )

#-------------------------------------------------------------------------------
# clear ptp ip access-list counters
#-------------------------------------------------------------------------------
def clearPtpIpAclCounters( mode, args ):
   AclCli.clearServiceAclCounters( mode, ptpStatus.aclStatusService,
                                   aclCheckpoint, 'ip' )

class ClearPtpIpAccessListCountersCmd( CliCommand.CliCommandClass ):
   syntax = 'clear ptp ip access-list counters'
   data = {
      'clear': CliToken.Clear.clearKwNode,
      'ptp': ptpMatcherForExec,
      'ip': AclCli.ipKwForClearServiceAclMatcher,
      'access-list': AclCli.accessListKwMatcherForServiceAcl,
      'counters': AclCli.countersKwMatcher,
   }
   handler = clearPtpIpAclCounters

BasicCliModes.EnableMode.addCommandClass( ClearPtpIpAccessListCountersCmd )

#-------------------------------------------------------------------------------
# show ptp foreign-master-record
#-------------------------------------------------------------------------------
def showFMR( mode, args ):
   # if the PTP agent never ran, defaultDS is None
   if not ptpStatus.defaultDS:
      return PtpForeignMasterRecords( message='PTP is not configured',
                                      intfForeignMasterRecords={} )

   if ptpStatus.defaultDS.ptpMode != 'ptpBoundaryClock':
      return PtpForeignMasterRecords(
         message='Not a boundary clock, no foreign master information.',
         intfForeignMasterRecords={} )

   keyList = []
   for key in ptpStatus.intfStatus:
      intfStatus = ptpStatus.intfStatus.get( key )
      if intfStatus.foreignMasterDS:
         keyList.append( key )

   if keyList == []:
      return PtpForeignMasterRecords( intfForeignMasterRecords={} )

   keyList = sorted( keyList )

   ptpForeignMasterRecords = {}
   for key in keyList:
      intfStatus = ptpStatus.intfStatus.get( key )
      ptpIntfForeignMasterRecords = PtpIntfForeignMasterRecords()
      ptpIntfForeignMasterRecords.maxForeignRecords = intfStatus.maxForeignRecords
      for record in intfStatus.foreignMasterDS.values():
         portIdentity = PtpPortIdentity()
         clockIdentity = record.foreignMasterPortIdentity.clockIdentity
         clockIdentityStr = "0x%02x:%02x:%02x:%02x:%02x:%02x:%02x:%02x" % \
             ( clockIdentity.v0, clockIdentity.v1,
               clockIdentity.v2, clockIdentity.v3,
               clockIdentity.v4, clockIdentity.v5,
               clockIdentity.v6, clockIdentity.v7 )
         portIdentity.clockIdentity = clockIdentityStr
         portIdentity.portNumber = record.foreignMasterPortIdentity.portNumber
         foreignMasterRecord = PtpForeignMasterDS()
         foreignMasterRecord.foreignMasterPortIdentity = portIdentity
         foreignMasterRecord.foreignMasterIpAddr = None
         foreignMasterRecord.foreignMasterIp6Addr = None
         if record.foreignMasterIpAddr != CliConstants.foreignMasterIpAddrDefault:
            if record.foreignMasterIpAddr.af == 'ipv4':
               foreignMasterRecord.foreignMasterIpAddr = \
                  record.foreignMasterIpAddr.v4Addr
            else:
               foreignMasterRecord.foreignMasterIp6Addr = \
                  record.foreignMasterIpAddr.v6Addr
         msgs = record.foreignMasterAnnounceMessages
         foreignMasterRecord.foreignMasterAnnounceMessages = msgs
         foreignMasterRecord.timeAgo = record.updateTime
         ptpIntfForeignMasterRecords.foreignMasterRecords.append(
            foreignMasterRecord )
         if key.vlanId != 0:
            ptpIntfForeignMasterRecords.vlanId = key.vlanId
         if key.intf not in ptpForeignMasterRecords:
            ptpForeignMasterRecords[ key.intf ] = PtpIntfVlanForeignMasterRecords(
                  interface=key.intf )
      ptpForeignMasterRecords[ key.intf ].intfForeignMasterRecords \
                                         .append( ptpIntfForeignMasterRecords )
   return PtpForeignMasterRecords( intfForeignMasterRecords=ptpForeignMasterRecords )

#-------------------------------------------------------------------------------
# show ptp monitor
#-------------------------------------------------------------------------------
def showPtpMonitor( mode, args ):
   ptpMonitorData = PtpMonitorData()

   if not ptpStatus.defaultDS:
      return ptpMonitorData

   ptpMonitorData.monitorEnabled = ptpConfig.monitorEnabled
   ptpMonitorData.ptpMode = ptpStatus.defaultDS.ptpMode
   if ( ptpConfig.offsetFromMasterThreshold !=
        CliConstants.invalidOffsetFromMasterThreshold ):
      ptpMonitorData.offsetFromMasterThreshold = ptpConfig.offsetFromMasterThreshold
   if ptpConfig.meanPathDelayThreshold != CliConstants.invalidMeanPathDelayThreshold:
      ptpMonitorData.meanPathDelayThreshold = ptpConfig.meanPathDelayThreshold
   if ptpConfig.skewThreshold != CliConstants.invalidSkewThreshold:
      ptpMonitorData.skewThreshold = ptpConfig.skewThreshold

   if ptpMonitorData.ptpMode in [ 'ptpBoundaryClock', 'ptpGeneralized' ]:
      output = StringIO()
      command = "BC_SHOW_MONITOR"
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            Ptp.agentName(), "ShowPtpCallback",
                                            command=command, timeout=30,
                                            stringBuff=output )

      # Parse the output from the agent
      # First line contains single integer repesenting the number of data, N
      # Each of the next N lines contains interface, real last sync time, offset
      # fro master, mean path delay and skew in order separated by whitespace.
      output.seek( 0, 0 )
      while True:
         line = output.readline()
         if not line or 'Cli connection' in line:
            break

         numberOfRecords = int( line.strip() )
         records = []
         for _ in range( numberOfRecords ):
            line = output.readline().strip()
            slave, timestamp, offset, delay, skew, seqId = line.split( ' ' )

            entry = PtpMonitorDataEntry()
            entry.intf = Tac.Value( 'Arnet::IntfId', slave )
            entry.realLastSyncTime = int( timestamp )
            entry.offsetFromMaster = int( offset )
            entry.meanPathDelay = int( delay )
            entry.skew = float( skew )
            entry.lastSyncSeqId = int( seqId )
            records.append( entry )
         records.sort( key=lambda entry: -entry.realLastSyncTime )
         ptpMonitorData.ptpMonitorData = records

   return ptpMonitorData

#-------------------------------------------------------------------------------
# show ptp source
# show ptp source ip
#-------------------------------------------------------------------------------
def showSrcIp( mode, args ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return None
   if not ptpStatus.defaultDS:
      return PtpSourceIp()
   return PtpSourceIp( sourceIp=ptpStatus.defaultDS.srcIp4,
                       sourceIp6=ptpStatus.defaultDS.srcIp6 )

# We are keeping the existing command `show ptp source ip` which only displays IPv4
# address
def showSrcIp4( mode, args ):
   if ptpConfig.unicastNegotiation:
      mode.addError( "PTP source is not supported with unicast negotiation" )
      return None
   if not ptpStatus.defaultDS:
      return PtpSourceIp()
   return PtpSourceIp( sourceIp=ptpStatus.defaultDS.srcIp4 )

#---------------------------------------------------------------------------------
# Register to show tech-support
#---------------------------------------------------------------------------------

def _showTechCmds( cmds ):
   def validCmds():
      if ptpSupported():
         return cmds
      return []
   return validCmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2014-02-11 12:53:04', _showTechCmds( [ 'show ptp interface enabled' ] ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2014-01-25 10:34:23', _showTechCmds( [ 'show ptp' ] ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      "2016-12-19 09:21:00", _showTechCmds( [ "show ptp local-clock",
                                              "show ptp masters" ] ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2018-02-20 10:38:54', _showTechCmds( [ 'show ptp monitor' ] ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2019-04-15 14:08:40', _showTechCmds( [ 'show ptp unicast-negotiation profile',
                                    'show ptp unicast-negotiation candidate-grantor',
                                    'show ptp unicast-negotiation remote-grantee'
                                            ] ) )

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
      '2020-05-26 22:15:47', _showTechCmds( [ 'show ptp interface counter drop' ] ) )

def Plugin( entityManager ):
   global ptpConfig, ptpStatus, ptpHwStatusDir
   global brConfig, bridgingSic, aclCheckpoint, ucastNegStatus
   global debugCounter, pktCounter
   sem = SharedMem.entityManager( sysdbEm=entityManager )
   ucastNegStatusInfo = Smash.mountInfo( 'reader' )
   ucastNegStatus = sem.doMount( "ptp/status/unicastnegotiation/",
                                 "Ptp::UcastNegStatusSmash",
                                 ucastNegStatusInfo )
   ptpConfig = ConfigMount.mount( entityManager, "ptp/config", "Ptp::Config", "w" )
   ptpStatus = LazyMount.mount( entityManager, "ptp/status", "Ptp::Status", "rO" )
   ptpHwStatusDir = LazyMount.mount( entityManager,
                                     "ptp/hwStatus", "Tac::Dir", "ri" )
   aclCheckpoint = LazyMount.mount( entityManager, "ptp/aclCheckpoint",
                                    "Acl::CheckpointStatus", "w" )
   brConfig = LazyMount.mount( entityManager, "bridging/config",
                               "Bridging::Config", "r" )
   bridgingSic = LazyMount.mount( entityManager, "bridging/switchIntfConfig",
                                  "Bridging::SwitchIntfConfigDir", "r" )
   debugCounterInfo = Smash.mountInfo( 'reader' )
   debugCounter = sem.doMount( "ptp/debug/counter",
                               "Ptp::DebugCounter::DebugCounter",
                               debugCounterInfo )
   pktCounterInfo = Smash.mountInfo( 'reader' )
   pktCounter = sem.doMount( "ptp/pkt/counter",
                             "Ptp::DebugCounter::DebugCounter",
                             pktCounterInfo )
