# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import exceptions

import Arnet.MplsLib
import BasicCli
from BasicCliUtil import anyCaseRegex
from BgpLib import getVpwsName, getVpwsEviName
import CliCommand
import CliMatcher
from CliMatcher import DynamicNameMatcher
from CliMode.Pseudowire import (
   FxcMode,
   LdpPseudowireMode,
   MplsLdpPseudowiresMode,
   PatchMode,
   PatchPanelMode,
)
from CliPlugin import (
   EthIntfCli,
   LagIntfCli,
   SubIntfCli,
)
from CliPlugin.BgpMacVrfConfigCli import BgpMacVrfVpwsMode
import IpAddrMatcher
from PseudowireLib import (
   ConnectorType,
   connIndex0,
   connIndex1,
   FlowLabelMode,
   getBgpConnector,
   getBgpConnectorKey,
   getInterfaceConnector,
   getInterfaceConnectorKey,
   getLdpConnectorKey,
   getMplsLdpConnector,
   pwPingCcEnumToVal,
   PwPingCcType,
   pwPingCvEnumToVal,
   PwPingCvType,
   vlanIdAny as portConnVlanId,
)
from CliPlugin.LdpCli import LdpConfigMode, getLdpConfigPseudowireCleanupHook
from CliPlugin.TunnelRibCli import TunnelRibNameMatcher, systemTunnelRibName
import CliParser
import ConfigMount
import LazyMount
import Tac
from Toggles.PseudowireToggleLib import toggleEvpnVpwsFxc1Enabled
import Tracing
import Arnet
from ArPyUtils.Decorators import (
   traced,
   tracedFuncName,
)
from VirtualIntfRule import IntfMatcher
from TypeFuture import TacLazyType

th = Tracing.Handle( 'PseudowireCli' )
t0 = th.t0
t1 = th.t1

AddressFamily = TacLazyType( 'Arnet::AddressFamily' )
CliConnector = TacLazyType( 'Pseudowire::CliConnector' )
CliFxcHelper = TacLazyType( 'PseudowireAgent::CliFxcHelper' )
DynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )
FecId = TacLazyType( 'Smash::Fib::FecId' )
PatchType = TacLazyType( 'Pseudowire::Patch' )
PseudowireBgpResponseKey = TacLazyType( 'Pseudowire::PseudowireBgpResponseKey' )

# Depending on ArBgp-cli is needed for BGP VPWS (forced dependency, see AID10)
# pkgdeps: import CliPlugin.BgpMacVrfConfigCli

pwConfig = None
pwRunnability = None
bridgingHwCapabilities = None
pwHwCapability = None
pwaMockLdpStatus = None
pwaMockBgpStatus = None
bgpMacVrf = None

# Common tokens
controlWordMatcher = CliMatcher.KeywordMatcher( 'control-word',
      helpdesc='Enable control word' )
matcherConnector = CliMatcher.KeywordMatcher( 'connector',
      helpdesc='Configure a patch connector' )
ldpPwNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( [ connKey.name for connKey in pwConfig.connector
                               if connKey.connectorType == ConnectorType.ldp ] ),
      'MPLS LDP pseudowire name' )
labelMatcher = CliMatcher.IntegerMatcher( Arnet.MplsLib.labelMin,
                                          Arnet.MplsLib.labelMax,
                                          helpdesc='MPLS label' )
mtuMatcher = CliMatcher.KeywordMatcher( 'mtu', helpdesc='MTU' )

validNameRegex = r"[A-Za-z0-9_:{}\[\]-]+"
# Allow any connector name but pseudowire and interface
disallowRegex = anyCaseRegex( "(pseudowire|interface)" )
connNameRegex = "^(?!%s$)%s$" % ( disallowRegex, validNameRegex )
connNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: [ conn.name for conn in
                     pwConfig.patch[ mode.patchName ].connector.values() ]
      if mode.patchName in pwConfig.patch else [], 'Connector name',
      pattern=connNameRegex )

def pwSupportedGuard( mode, token ):
   if pwHwCapability.mplsPseudowireSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def flowLabelSupportedGuard( mode, token ):
   if pwHwCapability.flowLabelSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform
#-------------------------------------------------------------------------------
# Patch panel config mode and related helpers
#-------------------------------------------------------------------------------
class PatchPanelConfigMode( PatchPanelMode, BasicCli.ConfigModeBase ):
   name = "Patch panel configuration"
   modeParseTree = CliParser.ModeParseTree()

   _nameErrorFmt = 'Name "{}" is already in use as a {}'

   def __init__( self, parent, session ):
      PatchPanelMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   @tracedFuncName( trace=t1 )
   def gotoPatchMode( self, args, funcName=None ):
      patchName = args[ 'PATCH_NAME' ]
      existingPatch = pwConfig.patch.get( patchName )
      if existingPatch and existingPatch.fxc:
         error = self._nameErrorFmt.format( patchName, "flexible cross connect" )
         t0( funcName, 'ERROR:', error )
         self.addError( error )
         return
      childMode = self.childMode( PatchConfigMode, patchName=patchName )
      self.session_.gotoChildMode( childMode )

   def noPatch( self, args ):
      patchName = args[ 'PATCH_NAME' ]
      del pwConfig.patch[ patchName ]
      pwRunnability.config = bool( pwConfig.patch )

   @tracedFuncName( trace=t1 )
   def gotoFxcMode( self, args, funcName=None ):
      patchName = args[ 'NAME' ]
      existingPatch = pwConfig.patch.get( patchName )
      if existingPatch and not existingPatch.fxc:
         error = self._nameErrorFmt.format( patchName, "patch" )
         t0( funcName, 'ERROR:', error )
         self.addError( error )
         return
      childMode = self.childMode( FlexibleCrossConnectConfigMode,
                                  patchName=patchName )
      self.session_.gotoChildMode( childMode )

   def noFxc( self, args ):
      t0( 'noFxc', repr( args ) )
      patchName = args[ 'NAME' ]
      del pwConfig.patch[ patchName ]
      pwRunnability.config = bool( pwConfig.patch )

def gotoPatchPanelMode( mode, args ):
   childMode = mode.childMode( PatchPanelConfigMode )
   mode.session_.gotoChildMode( childMode )

def noPatchPanel( mode, args ):
   pwConfig.patch.clear()
   pwConfig.intfReviewDelayRange = pwConfig.intfReviewDelayRangeDefault
   pwRunnability.config = False

def updateMtu( value ):
   if not value:
      value = pwConfig.getMtuDefault()
   pwConfig.mtu = value

class PatchConfigModeHelper( object ):
   def __init__( self, mode, patchName, patchClassDescription ):
      """
      Create a helper object for the patch / FXC config mode for `patchName`

      patchClassDescription is used in CLI error messages and describes the context
      in which the helper is used, e.g. "Patch":
       --> "Patch" patchName "is not present"
      """
      self.mode = mode
      self.patchName = patchName
      self.patchClassDescription = patchClassDescription

   def patchPresentOrError( self, patch ):
      # Ground has shifted underneath our feet. All subsequent commands
      # should fail.
      if not patch:
         self.mode.addError( "{} {} not present".format( self.patchClassDescription,
                                                         self.patchName ) )
         return False
      return True

class PatchConfigMode( PatchMode, BasicCli.ConfigModeBase ):
   name = "Patch configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, patchName ):
      PatchMode.__init__( self, patchName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.modeHelper = PatchConfigModeHelper( self, patchName, "Patch" )

      if patchName not in pwConfig.patch:
         patch = pwConfig.newPatch( patchName )
         patch.fxc = False
         pwRunnability.config = True
      else:
         if pwConfig.patch[ patchName ].fxc:
            assert not pwConfig.patch[ patchName ].fxc, "name {} was reused".format(
               patchName )

   def nextConnectorIndex( self, patch ):
      '''Assumes number of connectors in the patch are at most 2'''
      if not connIndex0 in patch.connector:
         return connIndex0
      if not connIndex1 in patch.connector:
         return connIndex1

      assert False, "Both indexes can't be occupied"
      return None

   def findConnector( self, cliConnector, patch, ignoreConnName=False ):
      for connIndex in patch.connector:
         if ( cliConnector == patch.connector[ connIndex ] or 
              ( ignoreConnName and cliConnector.connectorKey == 
                patch.connector[ connIndex ].connectorKey ) ):
            return connIndex

      return None

   def patchPresentOrError( self, patch ):
      return self.modeHelper.patchPresentOrError( patch )

   #-------------------------------------------------------------------
   # Common connector related methods
   #-------------------------------------------------------------------
   def getConnectors( self, getIntfConnectors, connName=None, dot1qTag=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      connectorList = []
      if getIntfConnectors:
         # If getting interface connectors, filter out any LDP pseudowire connectors
         connectorList = dict(
            ( connIndex, connector )
            for connIndex, connector in patch.connector.iteritems()
            if connector.interface )
         # If dot1qTag/connName is specified,
         # then filter out remaining entries based on connName/dot1qTag
         if connName:
            connectorList = dict(
               ( connIndex, connector )
               for connIndex, connector in connectorList.iteritems()
               if str( connector.interface ) == str( connName ) )
         if dot1qTag:
            connectorList = dict(
               ( connIndex, connector )
               for connIndex, connector in connectorList.iteritems()
               if connector.dot1qTag == dot1qTag )
      else:
         # If getting LDP pseudowire connectors, filter out any interface connectors
         connectorList = dict(
            ( connIndex, connector )
            for connIndex, connector in patch.connector.iteritems()
            if not connector.interface )
         # If connName is specified,
         # then filter out remaining entries based on connName
         if connName:
            connectorList = dict(
               ( connIndex, connector )
               for connIndex, connector in connectorList.iteritems()
               if connector.connectorKey.name == connName )

      return connectorList

   def connMatch( self, cliConn1, cliConn2, ignoreConnName=False ):
      if cliConn1 == cliConn2:
         return True

      if not ignoreConnName:
         return False

      return cliConn1.connectorKey == cliConn2.connectorKey

   def noConnectorName( self, connName ):
      '''Delete all connectors in the patch with the name connName.'''
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      for key in patch.connector:
         if patch.connector[ key ].name == connName:
            del patch.connector[ key ]

   def pickDefaultConnectorName( self, patch ):
      '''
      Return default connector names of "1" and "2" if none are provided.
      If "1" is already in use, return "2". Returns "" if 2 connectors
      are already defined.
      '''
      if len( patch.connector ) == 2:
         return ""

      found1 = any( ( c.name == "1" for c in patch.connector.values() ) )
      return "2" if found1 else "1"

   def hasConnectorInPatch( self, patch, connName, connectorKey ):
      '''Check if there is already a connector in the patch with the given
         connName (if passed) and connectorKey. Passing a valid connName is optional,
         if it passed as None or empty string, then only the connectorKey is checked.
      '''
      for connector in patch.connector.values():
         if connName:
            if connector.name == connName:
               return connector.connectorKey == connectorKey
         else:
            if connector.connectorKey == connectorKey:
               return True
      return False
   #-------------------------------------------------------------------
   # Connector interface/local connector related methods
   #-------------------------------------------------------------------

   def canAddInterfaceConnector( self, patch, newConnName, newConnKey ):
      numConnectorsInPatch = 0

      # Check if an incompatible connector is configured in this patch
      for cliConnector in patch.connector.itervalues():
         if cliConnector.name == newConnName:
            # this connector is going to be replaced, therefore ignore its presence
            continue
         if cliConnector.connectorKey == newConnKey:
            # this connector is going to be renamed, therefore ignore its presence
            continue
         numConnectorsInPatch += 1
         if cliConnector.connectorKey.isLocal():
            if newConnKey.isSubIntf() != cliConnector.connectorKey.isSubIntf():
               self.addError( "Local to local patch with only one subinterface "
                              "connector not supported" )
               return False
         elif ( newConnKey.isLegacyTagged() and
                cliConnector.connectorKey.connectorType == ConnectorType.bgp ):
            self.addError( "Patch with BGP VPWS and tagged connectors "
                           "not supported" )
            return False

      if numConnectorsInPatch == 2:
         self.addError( "Patch has two connectors already" )
         return False

      return True

   def setInterfaceConnector( self, intf, dot1qTag=None, connName=None ):
      '''
      The caller is trying to add a new CliConnector (with or without
      the connector name specified) or modify in-place an existing 
      CliConnector (connector name required).

      Note that giving the name of an existing connName necessarily
      overwrites it and doesn't add another connector with the same name.
      '''
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      # If not dot1qTag is provided, assume it's a port connector
      if dot1qTag is None:
         dot1qTag = portConnVlanId

      connKey = getInterfaceConnectorKey( intf, dot1qTag=dot1qTag )
      if self.hasConnectorInPatch( patch, connName, connKey ):
         # If the new connector is identical to the original one, then don't
         # do anything to avoid churn
         return

      if not self.canAddInterfaceConnector( patch, connName, connKey ):
         return

      if connName:
         # Delete existing connector with same name, as it is going to be replaced
         self.noConnectorName( connName )
         # Delete existing connector with same key, as it is going to be renamed
         self.noInterfaceConnector( intf, dot1qTag )
      else:
         connName = self.pickDefaultConnectorName( patch )
         assert connName

      # connector can be added successfully
      cliConnector = getInterfaceConnector( intf, dot1qTag, connName )
      connIndex = self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector

   def noInterfaceConnector( self, intf, dot1qTag=None, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      # If not dot1qTag is provided, assume it's a port connector
      if dot1qTag is None:
         dot1qTag = portConnVlanId

      delCliConn = getInterfaceConnector( intf, dot1qTag, connName or "" )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if self.connMatch( cliConn, delCliConn, ignoreConnName=not connName ):
            del patch.connector[ connIndex ]

   #-------------------------------------------------------------------
   # Remote pseudowire connector related methods
   #-------------------------------------------------------------------
   def canAddRemoteConnector( self, patch, newConnName, newConnKey ):
      numConnectorsInPatch = 0

      # Check if a BGP or LDP connector is configured in this patch
      for cliConnector in patch.connector.values():
         if cliConnector.name == newConnName:
            # this connector is going to be replaced, therefore ignore its presence
            continue
         if cliConnector.connectorKey == newConnKey:
            # this connector is going to be renamed, therefore ignore its presence
            continue
         numConnectorsInPatch += 1
         if cliConnector.connectorKey.connectorType in ( ConnectorType.ldp,
                                                         ConnectorType.bgp ):
            self.addError( "Patch with two remote connectors not supported" )
            return False

      if numConnectorsInPatch == 2:
         self.addError( "Patch has two connectors already" )
         return False

      return True

   #-------------------------------------------------------------------
   # MPLS LDP pseudowire connector related methods
   #-------------------------------------------------------------------
   def tryAddLdpPwConnector( self, patch, ldpPwName, connName=None ):
      connName = connName or self.pickDefaultConnectorName( patch )
      if not connName:
         # We have two connectors already
         self.addError( "Patch has two connectors already" )
         return False

      cliConnector = getMplsLdpConnector( ldpPwName, connName )

      # Delete duplicate connectors, if any, so the new connName gets priority.
      self.noConnectorPwLdp( True, ldpPwName )

      if not self.canAddRemoteConnector( patch, connName,
                                         cliConnector.connectorKey ):
         return False

      connIndex = self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector
      return True

   def setConnectorPwLdp( self, args ):
      '''
      The caller is trying to add a new CliConnector (with or without
      the new connector name specified) or modify in-place an existing
      CliConnector (connector name required).

      Note that giving the name of an existing connName necessarily
      overwrites it and doesn't add another connector with the same name.
      '''
      ldpPwName = args[ 'LDP_PW_NAME' ]
      connName = args.get( 'CONN_NAME' )
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      # If connector name is not specified, this is necessarily a connector
      # add attempt
      if not connName:
         return self.tryAddLdpPwConnector( patch, ldpPwName )

      # Otherwise (connector name is specified) this may be an add or edit
      # attempt. If a CliConnector exists by connName already, delete it.
      # Then add the new one.
      newConnectorKey = getLdpConnectorKey( ldpPwName )
      if self.hasConnectorInPatch( patch, connName, newConnectorKey ):
         # If the new connector is identical to the original one, then don't
         # add the new one, as it causes churn
         return
      self.noConnectorName( connName )
      return self.tryAddLdpPwConnector( patch, ldpPwName, connName )

   def noConnectorPwLdp( self, no, ldpPwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      delCliConn = getMplsLdpConnector( ldpPwName, connName or "" )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if self.connMatch( cliConn, delCliConn, ignoreConnName=not connName ):
            del patch.connector[ connIndex ]

   #-------------------------------------------------------------------
   # BGP EVPN pseudowire connector related methods
   #-------------------------------------------------------------------
   def canAddBgpRemoteConnector( self, patch, connName, connKey ):
      if not self.canAddRemoteConnector( patch, connName, connKey ):
         return False

      # Check if a legacy tagged connector is configured in this patch
      for cliConnector in patch.connector.values():
         if cliConnector.name == connName:
            # this connector is going to be replaced, therefore ignore its presence
            continue
         if cliConnector.connectorKey.isLegacyTagged():
            self.addError( "Patch with BGP VPWS and tagged connectors "
                           "not supported" )
            return False

      return True

   def setConnectorPwBgp( self, vpwsName, pwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      connKey = getBgpConnectorKey( vpwsName, pwName )
      if self.hasConnectorInPatch( patch, connName, connKey ):
         # The new connector is identical to the original one.
         # Don't do anything, as it would cause churn.
         return

      if not self.canAddBgpRemoteConnector( patch, connName, connKey ):
         return

      if connName:
         # Delete existing connector with same name, as it is going to be replaced
         self.noConnectorName( connName )
         # Delete existing connector with same key, as it is going to be renamed
         self.noConnectorPwBgp( vpwsName, pwName )
      else:
         connName = self.pickDefaultConnectorName( patch )
         assert connName

      cliConnector = getBgpConnector( vpwsName, pwName, connName )
      connIndex = self.nextConnectorIndex( patch )
      patch.connector[ connIndex ] = cliConnector

   def noConnectorPwBgp( self, vpwsName, pwName, connName=None ):
      patch = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( patch ):
         return

      delCliConn = getBgpConnector( vpwsName, pwName, connName or "" )

      for connIndex, cliConn in patch.connector.items():
         # If connName is provided, it must also match
         if self.connMatch( cliConn, delCliConn, ignoreConnName=not connName ):
            del patch.connector[ connIndex ]

   def shutdown( self ):
      patch = pwConfig.patch.get( self.patchName )
      if not patch:
         return

      patch.enabled = False

   def noShutdown( self ):
      patch = pwConfig.patch.get( self.patchName )
      if not patch:
         return

      patch.enabled = True

class FlexibleCrossConnectConfigMode( FxcMode, BasicCli.ConfigModeBase ):
   _fxc = "Flexible cross connect"
   name = "{fxc} configuration".format( fxc=_fxc )
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, patchName ):
      t0( "FlexibleCrossConnectConfigMode init", patchName )
      FxcMode.__init__( self, patchName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
      self.modeHelper = PatchConfigModeHelper( self, patchName, self._fxc )
      t0( "FlexibleCrossConnectConfigMode init", repr( patchName ) )
      if patchName not in pwConfig.patch:
         patch = pwConfig.patch.newMember( patchName )
         patch.fxc = True
         pwRunnability.config = True
      else:
         assert pwConfig.patch[ patchName ].fxc, "name {} was reused".format(
            patchName )

   @staticmethod
   def nextIndex( fxc ):
      return CliFxcHelper.nextIndex( fxc )

   @staticmethod
   def connectorCount( fxc ):
      return len( fxc.fxcConnector )

   def patchPresentOrError( self, patch ):
      return self.modeHelper.patchPresentOrError( patch )

   def _noRemote( self, fxc ):
      del fxc.fxcConnector[ fxc.fxcRemoteConnectorName ]
      fxc.fxcRemoteConnectorName = ""

   @traced( trace=t1 )
   def _noConnectorName( self, fxc, connName ):
      del fxc.fxcConnector[ connName ]
      if fxc.fxcRemoteConnectorName == connName:
         self._noRemote( fxc )

   @tracedFuncName( trace=t1 )
   def noConnectorName( self, connName, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
      else:
         self._noConnectorName( fxc, connName )

   def addErrorNoRoom( self, fxcName ):
      self.addError(
         "{fxc} {fxcName} cannot have more than {limit} connectors".format(
            fxc=self._fxc, fxcName=fxcName, limit=PatchType.fxcMaxConnector ) )

   @tracedFuncName( trace=t1 )
   def _setConnectorCommon( self, connKey, connName=None, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( fxc ):
         t0( funcName, "patch not found", self.patchName )
         return

      if connKey.isLocal():
         existingName = CliFxcHelper.findConnector( fxc, connKey, "" )
         exists = bool( existingName )
      else:
         existingName = fxc.fxcRemoteConnectorName
         exists = fxc.fxcRemoteConnector.connectorKey == connKey

      # Clean up an existing connector, if it exists with the same name, or else
      # check if this is a no-op and return early.
      if exists:
         if connName == existingName:
            # Nothing to do - same name and same connector
            t1( funcName, "nothing changed", connKey.stringValue, repr( connName ) )
            return

         if connName:
            # Re-adding the same connector, different name -> rename.
            # This seems like user error, but this is what we're doing for patches,
            # so maintain this behavior.

            # NOTE: This will invalidate fxc.fxcRemoteConnector.isValid for the
            # remote case so that we don't hit the "only 1 remote connector" error.
            self._noConnectorName( fxc, existingName )
         else:
            t1( funcName, "Re-use the existing name", repr( existingName ),
                "(no-op)" )
            return

      # Special case for remote: only 1 remote connector allowed
      isRemote = not connKey.isLocal()
      if ( isRemote and fxc.fxcRemoteConnector.isValid()
           and connName != fxc.fxcRemoteConnectorName ):
         self.addError(
            "{fxc} {fxcName} can only have a single remote connector".format(
               fxc=self._fxc, fxcName=self.patchName ) )
         return

      if self.connectorCount( fxc ) >= PatchType.fxcMaxConnector:
         self.addErrorNoRoom( self.patchName )
         return

      if not connName:
         nextIndex = self.nextIndex( fxc )
         # Handle race condition between multiple threads (otherwise it's handled by
         # the fxcMaxConnector check above).
         if nextIndex == PatchType.fxcInvalidIndex: # pragma: no cover
            self.addErrorNoRoom( self.patchName )
            return
         t1( funcName, "set connName from nextIndex", nextIndex )
         connName = str( nextIndex )

      if connName in fxc.fxcConnector:
         t1( funcName, "remove connector", connName )
         self._noConnectorName( fxc, connName )

      cliConnector = CliConnector( connKey )
      cliConnector.name = connName
      t1( funcName, "fxcConnector", connName, "->", connKey.stringValue )
      if not connKey.isLocal():
         fxc.fxcRemoteConnectorName = connName
      fxc.fxcConnector[ connName ] = cliConnector

   @tracedFuncName( trace=t1 )
   def setInterfaceConnector( self, intf, connName=None, funcName=None ):
      connKey = getInterfaceConnectorKey( intf )
      return self._setConnectorCommon( connKey, connName=connName )

   @tracedFuncName( trace=t1 )
   def setConnectorPwBgp( self, vpwsName, pwName, connName=None, funcName=None ):
      connKey = getBgpConnectorKey( vpwsName, pwName )
      return self._setConnectorCommon( connKey, connName=connName )

   @tracedFuncName( trace=t1 )
   def noInterfaceConnector( self, intf, connName=None, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
         return

      connKey = getInterfaceConnectorKey( intf )
      removeName = CliFxcHelper.findConnector( fxc, connKey, connName or "" )
      if removeName:
         t1( funcName, "remove", connKey.stringValue, repr( connName ),
             repr( removeName ) )
         del fxc.fxcConnector[ removeName ]
         return

      t1( funcName, "no match for", connKey.stringValue, repr( connName ) )

   @tracedFuncName( trace=t1 )
   def noConnectorPwBgp( self, vpwsName, pwName, connName=None, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
         return

      connKey = getBgpConnectorKey( vpwsName, pwName )
      if fxc.fxcRemoteConnector.connectorKey == connKey:
         if connName == fxc.fxcRemoteConnectorName or connName is None:
            t1( funcName, "remove", connKey.stringValue, repr( connName ),
                fxc.fxcRemoteConnectorName )
            self._noRemote( fxc )
            return
      t1( funcName, "no match for", connKey.stringValue, repr( connName ) )

   @tracedFuncName( trace=t1 )
   def _setEnabledCommon( self, enabled, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not fxc:
         t0( funcName, "patch not found", self.patchName )
         return

      fxc.enabled = enabled

   @traced( trace=t1 )
   def shutdown( self ):
      self._setEnabledCommon( False )

   @traced( trace=t1 )
   def noShutdown( self ):
      self._setEnabledCommon( True )

   @tracedFuncName( trace=t1 )
   def setVidNormalization( self, vidNormalization, funcName=None ):
      fxc = pwConfig.patch.get( self.patchName )
      if not self.patchPresentOrError( fxc ):
         t0( funcName, "patch not found", self.patchName )
         return

      value = {
         None: 'noNormalization',
         'single': 'singleVid',
         'double': 'doubleVid',
      }[ vidNormalization ]

      t0( funcName, "set vidNormalization", value )
      fxc.vidNormalization = value

#--------------------------------------------------------------------------------
# [ no | default ] patch panel
#--------------------------------------------------------------------------------
class PatchPanelCmd( CliCommand.CliCommandClass ):
   syntax = 'patch panel'
   noOrDefaultSyntax = syntax
   data = {
      'patch': CliCommand.guardedKeyword( 'patch',
         helpdesc='Configure pseudowire patch',
         guard=pwSupportedGuard ),
      'panel': 'Configure pseudowire patches',
   }

   handler = gotoPatchPanelMode
   noOrDefaultHandler = noPatchPanel

BasicCli.GlobalConfigMode.addCommandClass( PatchPanelCmd )

def _patches( _mode ):
   return pwConfig.patch.members()

#--------------------------------------------------------------------------------
# [ no | default ] patch PATCH_NAME
# under "patch panel" config mode
#--------------------------------------------------------------------------------
class PatchPatchnameCmd( CliCommand.CliCommandClass ):
   syntax = 'patch PATCH_NAME'
   noOrDefaultSyntax = syntax
   # yapf: disable
   data = {
      'patch': 'Configure a patch',
      'PATCH_NAME': DynamicNameMatcher( _patches, helpdesc='Patch name' ),
   }
   # yapf: enable

   handler = PatchPanelConfigMode.gotoPatchMode
   noOrDefaultHandler = PatchPanelConfigMode.noPatch

PatchPanelConfigMode.addCommandClass( PatchPatchnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] flexible-cross-connect NAME
# under "patch panel" config mode
#--------------------------------------------------------------------------------
class PatchPanelFxcCmd( CliCommand.CliCommandClass ):
   syntax = 'flexible-cross-connect NAME'
   noOrDefaultSyntax = syntax
   data = {
      'flexible-cross-connect': 'Configure a flexible cross connect',
      'NAME': DynamicNameMatcher( _patches, helpdesc='Flexible cross connect name' ),
   }

   handler = PatchPanelConfigMode.gotoFxcMode
   noOrDefaultHandler = PatchPanelConfigMode.noFxc

if toggleEvpnVpwsFxc1Enabled():
   PatchPanelConfigMode.addCommandClass( PatchPanelFxcCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connector [ CONN_NAME ] interface INTF [ dot1q vlan DOT1Q_TAG ]
# commands under the "patch [name]" config mode
# Note: For the no versions of the commands, only exact matching connectors will
# be removed
#--------------------------------------------------------------------------------
interfaceMatcher = CliMatcher.KeywordMatcher( 'interface',
      helpdesc='Local connector configuration' )

# We only allow ethernet and port-channel interfaces in config
intfMatcher = IntfMatcher()
intfMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfMatcher |= LagIntfCli.EthLagIntf.matcher

# and ethernet sub-interfaces
intfOrSubIntfNameMatcher = IntfMatcher()
intfOrSubIntfNameMatcher |= EthIntfCli.EthPhyIntf.ethMatcher
intfOrSubIntfNameMatcher |= LagIntfCli.EthLagIntf.matcher
intfOrSubIntfNameMatcher |= SubIntfCli.subMatcher

class InterfaceConnectorCmd( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] interface INTF dot1q vlan DOT1Q_TAG'
   noOrDefaultSyntax = syntax
   data = {
      'connector': matcherConnector,
      'CONN_NAME': connNameMatcher,
      'interface': interfaceMatcher,
      'INTF': intfMatcher,
      'dot1q': '802.1Q VLAN tag',
      'vlan': '802.1Q VLAN tag',
      'DOT1Q_TAG': CliMatcher.IntegerMatcher( 1, 4094, helpdesc='802.1Q VLAN tag' ),
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setInterfaceConnector( args[ 'INTF' ],
         dot1qTag=args.get( 'DOT1Q_TAG' ),
         connName=args.get( 'CONN_NAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      connName = args.get( 'CONN_NAME' )
      dot1qTag = args.get( 'DOT1Q_TAG' )
      return PatchConfigMode.noInterfaceConnector( mode, intf, dot1qTag, connName )

PatchConfigMode.addCommandClass( InterfaceConnectorCmd )

class SubInterfaceConnectorCmd( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] interface INTF'
   noOrDefaultSyntax = syntax
   data = {
      'connector': matcherConnector,
      'CONN_NAME': connNameMatcher,
      'interface': interfaceMatcher,
      'INTF': intfOrSubIntfNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setInterfaceConnector( args[ 'INTF' ],
                                         connName=args.get( 'CONN_NAME' ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      intf = args[ 'INTF' ]
      connName = args.get( 'CONN_NAME' )
      return mode.noInterfaceConnector( intf, connName=connName )

PatchConfigMode.addCommandClass( SubInterfaceConnectorCmd )
FlexibleCrossConnectConfigMode.addCommandClass( SubInterfaceConnectorCmd )

#--------------------------------------------------------------------------------
# [ no | default ] connector [ CONN_NAME ] pseudowire ldp LDP_PW_NAME
# command under "patch [name]" config mode
#--------------------------------------------------------------------------------
class ConnectorPwLdpCmd( CliCommand.CliCommandClass ):
   syntax = 'connector [ CONN_NAME ] pseudowire ldp LDP_PW_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'connector': matcherConnector,
      'CONN_NAME': connNameMatcher,
      'pseudowire': 'LDP or BGP pseudowire',
      'ldp': 'LDP pseudowire',
      'LDP_PW_NAME': ldpPwNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      return mode.setConnectorPwLdp( args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      ldpPwName = args[ 'LDP_PW_NAME' ]
      connName = args.get( 'CONN_NAME' )
      return mode.noConnectorPwLdp( True, ldpPwName, connName )

PatchConfigMode.addCommandClass( ConnectorPwLdpCmd )

#-------------------------------------------------------------------------------
# The "[no|default] connector [connName] pseudowire bgp vpws [name]
#      pseudowire [pw-name]" command under "patch [name]" config mode
#-------------------------------------------------------------------------------
bgpVpwsNameMatcher = CliMatcher.DynamicNameMatcher(
      lambda mode: ( sorted( [ getVpwsEviName( macVrf.name )
                               for macVrf in bgpMacVrf.config.itervalues()
                               if macVrf.isVpwsMacVrf() ] ) ),
      "EVPN VPWS instance name" )

def getPseudowireNames( mode, ctx ):
   vpwsName = getVpwsName( ctx.sharedResult[ "VPWS_NAME" ] )
   bgpMacVrfConfig = bgpMacVrf.config.get( vpwsName )
   return sorted( bgpMacVrfConfig.pseudowireConfig ) if bgpMacVrfConfig else []

bgpPseudowireNameMatcher = CliMatcher.DynamicNameMatcher(
      getPseudowireNames,
      "EVPN VPWS pseudowire name", passContext=True )

class PseudowireBgpConnectorCmd( CliCommand.CliCommandClass ):
   syntax = "connector [CONN_NAME] pseudowire bgp vpws VPWS_NAME pwName PW_NAME"
   noOrDefaultSyntax = syntax
   data = {
      "connector": "Configure a patch connector",
      "CONN_NAME": connNameMatcher,
      "pseudowire": "LDP or BGP pseudowire",
      "bgp": "BGP pseudowire",
      "vpws": "EVPN VPWS pseudowire",
      "VPWS_NAME": CliCommand.Node( bgpVpwsNameMatcher, storeSharedResult=True ),
      "pwName": CliMatcher.KeywordMatcher( "pseudowire",
                                           "EVPN VPWS pseudowire name" ),
      "PW_NAME": bgpPseudowireNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.setConnectorPwBgp( args[ "VPWS_NAME" ], args[ "PW_NAME" ],
                              args.get( "CONN_NAME" ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.noConnectorPwBgp( args[ "VPWS_NAME" ], args[ "PW_NAME" ],
                             args.get( "CONN_NAME" ) )

PatchConfigMode.addCommandClass( PseudowireBgpConnectorCmd )
FlexibleCrossConnectConfigMode.addCommandClass( PseudowireBgpConnectorCmd )

#--------------------------------------------------------------------------------
# ( no | default ) connector CONN_NAME
# command under "patch [name]" config mode
#--------------------------------------------------------------------------------
class ConnectorConnnameCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'connector CONN_NAME'
   data = {
      'connector': matcherConnector,
      'CONN_NAME': connNameMatcher
   }

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noConnectorName( args[ 'CONN_NAME' ] )

PatchConfigMode.addCommandClass( ConnectorConnnameCmd )
FlexibleCrossConnectConfigMode.addCommandClass( ConnectorConnnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
# command under "patch [name]" config mode
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Disable patch',
   }

   @staticmethod
   def handler( mode, args ):
      return mode.shutdown()

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.noShutdown()

PatchConfigMode.addCommandClass( ShutdownCmd )
FlexibleCrossConnectConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [no|default] vlan tag normalization (single|double)
# command under "patch [name]" config mode
#--------------------------------------------------------------------------------
class VlanTagNormalizationCommand( CliCommand.CliCommandClass ):
   syntax = 'vlan tag normalization VALUE'
   noOrDefaultSyntax = 'vlan tag normalization ...'
   data = dict(
      vlan='FXC VLAN configuration',
      tag='FXC VLAN tag configuration',
      normalization='FXC VLAN tag normalization configuration',
      VALUE=CliMatcher.EnumMatcher(
         dict(
            single='Use a single VLAN tag for normalization',
            double='Use double VLAN tags for normalization',
         ) ),
   )

   @staticmethod
   def handler( mode, args ):
      return mode.setVidNormalization( args[ 'VALUE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return mode.setVidNormalization( None )

FlexibleCrossConnectConfigMode.addCommandClass( VlanTagNormalizationCommand )

#-------------------------------------------------------------------------------
# MPLS LDP pseudowire config mode related helpers
#-------------------------------------------------------------------------------
class MplsLdpPseudowiresConfigMode( MplsLdpPseudowiresMode,
                                    BasicCli.ConfigModeBase ):
   name = "MPLS LDP pseudowire configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      MplsLdpPseudowiresMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )
   
   def setOrNoMtu( self, mtu ):
      '''If mtu is None, it implies we are disabling mtu'''
      oldValue = pwConfig.mtu
      if oldValue != mtu:
         updateMtu( mtu )

   def setMtu( self, args ):
      self.setOrNoMtu( args[ 'MTU' ] )

   def noMtu( self, args ):
      self.setOrNoMtu( None )

   def setMtuIgnore( self, args ):
      pwConfig.mtuIgnore = True 

   def setLdpPseudowireTunnelRib( self, args ):
      pwConfig.tunnelRibName = args[ 'TUNNEL_NAME' ]

   def noLdpPseudowireTunnelRib( self, args ):
      pwConfig.tunnelRibName = systemTunnelRibName

   def gotoLdpPseudowireMode( self, args ):
      ldpPwName = args[ 'LDP_PW_NAME' ]
      childMode = self.childMode( LdpPseudowireConfigMode, ldpPwName=ldpPwName )
      self.session_.gotoChildMode( childMode )

   def noLdpPseudowireMode( self, args ):
      ldpPwName = args[ 'LDP_PW_NAME' ]
      connKey = getLdpConnectorKey( ldpPwName )
      if connKey in pwConfig.connector:
         del pwConfig.connector[ connKey ]

def gotoMplsLdpPseudowiresMode( mode, args ):
   childMode = mode.childMode( MplsLdpPseudowiresConfigMode )
   mode.session_.gotoChildMode( childMode )

def noMplsLdpPseudowires( mode=None, args=None ):
   pwConfig.connector.clear()
   updateMtu( None )

class LdpPseudowireConfigMode( LdpPseudowireMode, BasicCli.ConfigModeBase ):
   name = "LDP pseudowire configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, ldpPwName ):
      LdpPseudowireMode.__init__( self, ldpPwName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

      self.maybeAddPseudowire( ldpPwName )

   def addPseudowireAttribute( self, nbrAddr=None, keepNbrAddr=None,
                               pwId=1, keepPwId=None, 
                               mtu=0, keepMtu=None,
                               controlWord=False, keepControlWord=None,
                               flowLabelMode=FlowLabelMode.disabled,
                               keepFlowLabelMode=None ):
      connKey = getLdpConnectorKey( self.ldpPwName )
      oldConnector = pwConfig.connector[ connKey ]

      def newAttr( oldVal, defaultVal, newVal, keep ):
         attr = oldVal
         if keep is True:
            attr = newVal
         elif keep is False:
            attr = defaultVal
         return attr

      newNbrAddr = newAttr( oldConnector.neighborAddr,
                            Tac.Value( "Arnet::IpGenAddr" ),
                            nbrAddr, keepNbrAddr )
      newNbrAddrPresent = newAttr( oldConnector.neighborAddrPresent,
                                   False,
                                   keepNbrAddr, keepNbrAddr )

      newPwId = newAttr( oldConnector.pwId, 0, pwId, keepPwId )
      newPwIdPresent = newAttr( oldConnector.pwIdPresent,
                                False,
                                keepPwId, keepPwId )

      newMtu = newAttr( oldConnector.mtu, 0, mtu, keepMtu )
      newMtuPresent = newAttr( oldConnector.mtuPresent,
                               False,
                               keepMtu, keepMtu )

      newControlWord = newAttr( oldConnector.controlWord,
                                False,
                                controlWord, keepControlWord )

      newFlowLabelMode = newAttr( oldConnector.flowLabelMode,
                                  FlowLabelMode.disabled,
                                  flowLabelMode, keepFlowLabelMode )

      pwConfig.connector[ connKey ] = Tac.Value( "Pseudowire::Connector",
                                connectorKey=connKey,
                                neighborAddr=newNbrAddr,
                                neighborAddrPresent=newNbrAddrPresent,
                                pwId=newPwId,
                                pwIdPresent=newPwIdPresent,
                                mtu=newMtu,
                                mtuPresent=newMtuPresent,
                                controlWord=newControlWord,
                                flowLabelMode=newFlowLabelMode )

   def maybeAddPseudowire( self, ldpPwName ):
      connKey = getLdpConnectorKey( ldpPwName )
      if connKey not in pwConfig.connector:
         connector = Tac.Value( "Pseudowire::Connector", connectorKey=connKey )
         pwConfig.connector[ connKey ] = connector

   def setNeighbor( self, args ):
      ipGenAddr = Tac.Value( "Arnet::IpGenAddr", args[ 'IPADDR' ] )
      self.addPseudowireAttribute( nbrAddr=ipGenAddr, keepNbrAddr=True )

   def noNeighbor( self, args ):
      self.addPseudowireAttribute( keepNbrAddr=False )

   def setPwId( self, args ):
      self.addPseudowireAttribute( pwId=args[ 'PWID' ], keepPwId=True )

   def noPwId( self, args ):
      self.addPseudowireAttribute( keepPwId=False )

   def setMtu( self, args ):
      self.addPseudowireAttribute( mtu=args[ 'MTU' ], keepMtu=True )

   def noMtu( self, args ):
      self.addPseudowireAttribute( keepMtu=False )

   def setControlWord( self, args ):
      self.addPseudowireAttribute( controlWord=True, keepControlWord=True )

   def noControlWord( self, args ):
      self.addPseudowireAttribute( keepControlWord=False )

   def setFlowLabel( self, args ):
      if 'transmit' in args:
         flowLabelMode = FlowLabelMode.transmit
      elif 'receive' in args:
         flowLabelMode = FlowLabelMode.receive
      else:
         flowLabelMode = FlowLabelMode.both
      self.addPseudowireAttribute( flowLabelMode=flowLabelMode,
                                   keepFlowLabelMode=True )

   def noFlowLabel( self, args ):
      self.addPseudowireAttribute( keepFlowLabelMode=False )

#--------------------------------------------------------------------------------
# [ no | default ] mpls ldp pseudowires
#--------------------------------------------------------------------------------
mplsLdpPseudowiresNode = CliCommand.guardedKeyword( 'pseudowires',
      helpdesc='Configure MPLS LDP pseudowires', guard=pwSupportedGuard )

class MplsLdpPseudowiresCmd( CliCommand.CliCommandClass ):
   syntax = 'mpls ldp pseudowires'
   noOrDefaultSyntax = syntax
   data = {
      'mpls': CliCommand.guardedKeyword( 'mpls',
         helpdesc='Global MPLS configuration commands', guard=pwSupportedGuard ),
      'ldp': 'Configure MPLS LDP pseudowires',
      'pseudowires': mplsLdpPseudowiresNode
   }
   hidden = True

   handler = gotoMplsLdpPseudowiresMode
   noOrDefaultHandler = noMplsLdpPseudowires

BasicCli.GlobalConfigMode.addCommandClass( MplsLdpPseudowiresCmd )

#--------------------------------------------------------------------------------
# [ no | default ] pseudowires
#--------------------------------------------------------------------------------
class PseudowiresCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowires'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowires': mplsLdpPseudowiresNode
   }

   handler = gotoMplsLdpPseudowiresMode
   noOrDefaultHandler = noMplsLdpPseudowires

LdpConfigMode.addCommandClass( PseudowiresCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mtu MTU
# command under "mpls ldp pseudowires" config mode
#--------------------------------------------------------------------------------
# Our l2MtuMin and l2MtuMax size include the ethernet header and 802.1Q tag 
# (i.e., 18 bytes). However, RFC4447 requires the MTU used in LDP to exclude
# the encapsulation overhead.
def mtuRangeFn( mode ):
   encapOctets = 18
   return ( bridgingHwCapabilities.l2MtuMin - encapOctets,
            bridgingHwCapabilities.l2MtuMax - encapOctets )

mtuMatcher = CliMatcher.KeywordMatcher( 'mtu', helpdesc='MTU for pseudowire' )
mtuNumMatcher = CliMatcher.DynamicIntegerMatcher(
                                             rangeFn=mtuRangeFn,
                                             helpdesc='MTU for pseudowire' )

class MtuMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu MTU'
   noOrDefaultSyntax = 'mtu ...'
   data = {
      'mtu': mtuMatcher,
      'MTU': mtuNumMatcher
   }

   handler = MplsLdpPseudowiresConfigMode.setMtu
   noOrDefaultHandler = MplsLdpPseudowiresConfigMode.noMtu

MplsLdpPseudowiresConfigMode.addCommandClass( MtuMtuCmd )

#--------------------------------------------------------------------------------
# mtu ignore
# command under "mpls ldp pseudowires" config mode
#--------------------------------------------------------------------------------
class MtuIgnoreCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu ignore'
   data = {
      'mtu': mtuMatcher,
      'ignore': 'Ignore MTU mismatch',
   }

   handler = MplsLdpPseudowiresConfigMode.setMtuIgnore

MplsLdpPseudowiresConfigMode.addCommandClass( MtuIgnoreCmd )

#--------------------------------------------------------------------------------
# [ no | default ] pseudowire LDP_PW_NAME
# config mode under "mpls ldp pseudowires" config mode
#--------------------------------------------------------------------------------
class PseudowireLdppwnameCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowire LDP_PW_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'pseudowire': 'Configure a named pseudowire',
      'LDP_PW_NAME': ldpPwNameMatcher,
   }

   handler = MplsLdpPseudowiresConfigMode.gotoLdpPseudowireMode
   noOrDefaultHandler = MplsLdpPseudowiresConfigMode.noLdpPseudowireMode

MplsLdpPseudowiresConfigMode.addCommandClass( PseudowireLdppwnameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor resolution tunnel-rib ( system-tunnel-rib | TUNNEL_NAME )
# under "mpls ldp pseudowires" config mode
#--------------------------------------------------------------------------------
class LdpPseudowireTunnelRibCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor resolution tunnel-rib ( system-tunnel-rib | TUNNEL_NAME )'
   noOrDefaultSyntax = 'neighbor resolution tunnel-rib ...'
   data = {
      'neighbor': 'LDP neighbor',
      'resolution': 'Resolution of all LDP pseudowires',
      'tunnel-rib': 'Configure a tunnel RIB',
      'system-tunnel-rib': 'The system tunnel RIB',
      'TUNNEL_NAME': TunnelRibNameMatcher( helpdesc='Name of the tunnel RIB' )
   }

   @staticmethod
   def adapter( mode, args, argsList ):
      if systemTunnelRibName in args:
         args[ 'TUNNEL_NAME' ] = systemTunnelRibName

   handler = MplsLdpPseudowiresConfigMode.setLdpPseudowireTunnelRib
   noOrDefaultHandler = MplsLdpPseudowiresConfigMode.noLdpPseudowireTunnelRib

MplsLdpPseudowiresConfigMode.addCommandClass( LdpPseudowireTunnelRibCmd )

#--------------------------------------------------------------------------------
# [ no | default ] neighbor IPADDR
# command under "pseudowire [name]" config mode
#--------------------------------------------------------------------------------
class NeighborIpaddrCmd( CliCommand.CliCommandClass ):
   syntax = 'neighbor IPADDR'
   noOrDefaultSyntax = 'neighbor ...'
   data = {
      'neighbor': 'LDP neighbor',
      'IPADDR': IpAddrMatcher.ipAddrMatcher,
   }

   handler = LdpPseudowireConfigMode.setNeighbor
   noOrDefaultHandler = LdpPseudowireConfigMode.noNeighbor

LdpPseudowireConfigMode.addCommandClass( NeighborIpaddrCmd )

#--------------------------------------------------------------------------------
# [ no | default ] pseudowire-id PWID
# command under "pseudowire [name]" config mode
#--------------------------------------------------------------------------------
class PseudowireIdPwidCmd( CliCommand.CliCommandClass ):
   syntax = 'pseudowire-id PWID'
   noOrDefaultSyntax = 'pseudowire-id ...'
   data = {
      'pseudowire-id': 'Pseudowire ID',
      'PWID': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF, helpdesc='Pseudowire ID' ),
   }

   handler = LdpPseudowireConfigMode.setPwId
   noOrDefaultHandler = LdpPseudowireConfigMode.noPwId

LdpPseudowireConfigMode.addCommandClass( PseudowireIdPwidCmd )

#-------------------------------------------------------------------------------
# The "[no|default] mtu [bytes]" command under "pseudowire [name]" config mode
#-------------------------------------------------------------------------------
class LdpPseudowireMtuMtuCmd( CliCommand.CliCommandClass ):
   syntax = 'mtu MTU'
   noOrDefaultSyntax = 'mtu ...'
   data = {
      'mtu': mtuMatcher,
      'MTU': mtuNumMatcher
   }

   handler = LdpPseudowireConfigMode.setMtu
   noOrDefaultHandler = LdpPseudowireConfigMode.noMtu

LdpPseudowireConfigMode.addCommandClass( LdpPseudowireMtuMtuCmd )

#--------------------------------------------------------------------------------
# [ no | default ] control-word
# command under "pseudowire [name]" config mode
#--------------------------------------------------------------------------------
class ControlWordCmd( CliCommand.CliCommandClass ):
   syntax = 'control-word'
   noOrDefaultSyntax = syntax
   data = {
      'control-word': 'Enable control word',
   }

   handler = LdpPseudowireConfigMode.setControlWord
   noOrDefaultHandler = LdpPseudowireConfigMode.noControlWord

LdpPseudowireConfigMode.addCommandClass( ControlWordCmd )

#-------------------------------------------------------------------------------
# The "[no|default] label flow [ transmit | receive ]" command under
# "pseudowire [name]" config mode
#-------------------------------------------------------------------------------
class FlowLabelCmd( CliCommand.CliCommandClass ):
   syntax = 'label flow [ transmit | receive ]'
   noOrDefaultSyntax = 'label flow ...'
   data = {
      'label': CliCommand.guardedKeyword( 'label', helpdesc='Label operation',
                                          guard=flowLabelSupportedGuard ),
      'flow': 'Enable flow label',
      'transmit': 'Enable flow label for transmit only',
      'receive': 'Enable flow label for receive only'
   }
   handler = LdpPseudowireConfigMode.setFlowLabel
   noOrDefaultHandler = LdpPseudowireConfigMode.noFlowLabel

LdpPseudowireConfigMode.addCommandClass( FlowLabelCmd )

#-------------------------------------------------------------------------------
# The "[no|default] debug pseudowire-id PWID neighbor PWPEER peer-label LABEL
# type PWTYPE [mtu MTU] [control-word] [flow-label FLMODE] [dot1q vlan DOT1QTAG]
# [group-id GROUPID]" hidden command under "mpls ldp pseudowires" config mode
#-------------------------------------------------------------------------------
class DebugPwLdpCmd( CliCommand.CliCommandClass ):
   syntax = '''debug pseudowire-id PWID neighbor PWPEER peer-label LABEL
               type PWTYPE [mtu MTU] [control-word] [flow-label FLMODE]
               [dot1q vlan DOT1QTAG] [group-id GROUPID]
            '''
   noOrDefaultSyntax = '''debug [pseudowire-id PWID neighbor PWPEER] ...'''
   data = {
      'debug': 'LDP pseudowire debug',
      'pseudowire-id': 'Pseudowire ID',
      'PWID': CliMatcher.IntegerMatcher( 1, 0xFFFFFFFF,
                                           helpdesc='Pseudowire ID' ),
      'neighbor': 'Pseudowire neighbor',
      'PWPEER': IpAddrMatcher.IpAddrMatcher( helpdesc='Pseudowire neighbor' ),
      'peer-label': 'Peer label',
      'LABEL': labelMatcher,
      'type': 'Pseudowire type',
      'PWTYPE': CliMatcher.IntegerMatcher( 4, 5,
                                          helpdesc='Pseudowire type (4 or 5)' ),
      'mtu': mtuMatcher,
      'MTU': mtuNumMatcher,
      'control-word': 'Enable control word',
      'flow-label': 'Configure flow label mode',
      'FLMODE': CliMatcher.EnumMatcher( {
         'both': 'Configure flow label for transmit and receive',
         'transmit': 'Configure flow label for transmit',
         'receive': 'Configure flow label for receive',
      } ),
      'dot1q': '802.1Q VLAN tag',
      'vlan': '802.1Q VLAN tag',
      'DOT1QTAG': CliMatcher.IntegerMatcher( 1, 4094,
                                               helpdesc='802.1Q VLAN tag' ),
      'group-id': 'Pseudowire group ID',
      'GROUPID': CliMatcher.IntegerMatcher( 0, 0xFFFFFFFF,
                                              helpdesc='Pseudowire group ID' ),
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      pwId = args[ 'PWID' ]
      pwPeer = args[ 'PWPEER' ]
      peerLabel = args[ 'LABEL' ]
      pwType = args[ 'PWTYPE' ]
      if pwType == 4:
         pwType = 'pwType4'
      elif pwType == 5:
         pwType = 'pwType5'
      mtu = args.get( 'MTU' )
      controlWord = 'control-word' in args
      flowLabelMode = args.get( 'FLMODE', 'disabled' )
      flowLabelMode = getattr( FlowLabelMode, flowLabelMode )
      requestedDot1qTag = args.get( 'DOT1QTAG' )
      groupId = args.get( 'GROUPID' )

      respKey = Tac.Value( 'Pseudowire::PseudowireLdpResponseKey',
                           Arnet.IpGenAddr( pwPeer ), pwId )
      mockResp = pwaMockLdpStatus.newResponse( respKey )

      mockResp.peerLabel = peerLabel
      mockResp.remotePwType = pwType
      mockResp.status = 'up'
      mockResp.controlWord = controlWord
      mockResp.peerFlowLabelMode = flowLabelMode
      mockResp.flowLabelType = Tac.enumValue( FlowLabelMode, flowLabelMode )
      mockResp.intfDescription = ''
      mockResp.requestedDot1qTag = requestedDot1qTag or 0
      mockResp.mtu = mtu or 0
      mockResp.groupId = groupId or 0
      mockResp.pwStatusFlags = 0
      mockResp.peerVccvCcTypes = pwPingCcEnumToVal[ PwPingCcType.ra ]
      mockResp.peerVccvCvTypes = pwPingCvEnumToVal[ PwPingCvType.lspPing ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwId = args.get( 'PWID' )
      pwPeer = args.get( 'PWPEER' )

      if pwId and pwPeer:
         respKey = Tac.Value( 'Pseudowire::PseudowireLdpResponseKey',
                              Arnet.IpGenAddr( pwPeer ), pwId )
         if respKey in pwaMockLdpStatus.response:
            del pwaMockLdpStatus.response[ respKey ]
      else:
         for key in pwaMockLdpStatus.response:
            del pwaMockLdpStatus.response[ key ]

MplsLdpPseudowiresConfigMode.addCommandClass( DebugPwLdpCmd )

#-------------------------------------------------------------------------------
# The "[no|default] debug pseudowire PWNAME neighbor PWPEER peer-label LABEL
# [mtu MTU] [control-word]" hidden command under "vpws EVINAME" config mode
#-------------------------------------------------------------------------------
class DebugPwBgpCmd( CliCommand.CliCommandClass ):
   syntax = '''debug pseudowire PWNAME neighbor PWPEER peer-label LABEL
               [ local [ mtu-l LOCALMTU ] [ control-word-l ] ]
               [ peer [ mtu-p PEERMTU ] [ control-word-p ] ]
            '''
   noOrDefaultSyntax = '''debug pseudowire [ PWNAME ] ...'''
   data = {
      'debug': 'BGP pseudowire debug',
      'pseudowire': 'EVPN VPWS pseudowire name',
      'PWNAME': bgpPseudowireNameMatcher,
      'neighbor': 'Pseudowire neighbor',
      'PWPEER': IpAddrMatcher.IpAddrMatcher( helpdesc='Pseudowire neighbor' ),
      'peer-label': 'Peer label',
      'LABEL': labelMatcher,
      'local': 'Local options',
      'peer': 'Peer options',
      'mtu-l': mtuMatcher,
      'LOCALMTU': mtuNumMatcher,
      'control-word-l': controlWordMatcher,
      'mtu-p': mtuMatcher,
      'PEERMTU': mtuNumMatcher,
      'control-word-p': controlWordMatcher,
   }
   hidden = True

   @staticmethod
   def handler( mode, args ):
      vpwsName = getVpwsEviName( mode.name )
      pwName = args[ 'PWNAME' ]
      pwPeer = args[ 'PWPEER' ]
      peerLabel = args[ 'LABEL' ]
      localMtu = args.get( 'LOCALMTU' )
      localControlWord = 'control-word-l' in args
      peerMtu = args.get( 'PEERMTU' )
      peerControlWord = 'control-word-p' in args

      respKey = PseudowireBgpResponseKey( vpwsName, pwName )
      mockResp = pwaMockBgpStatus.newBgpResponse( respKey )

      mockResp.controlWord = localControlWord
      if localMtu is not None:
         mockResp.mtu = localMtu
      mockResp.status = "noTunnel"

      peerAddr = Arnet.IpGenAddr( pwPeer )
      peer = mockResp.peerInfo.newMember( peerAddr )
      peer.controlWord = peerControlWord
      if peerMtu is not None:
         peer.mtu = peerMtu
      peer.label = peerLabel
      peer.status = "noTunnel"

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      vpwsName = getVpwsEviName( mode.name )
      pwName = args.get( 'PWNAME' )

      if pwName:
         respKey = PseudowireBgpResponseKey( vpwsName, pwName )
         if respKey in pwaMockBgpStatus.bgpResponse:
            del pwaMockBgpStatus.bgpResponse[ respKey ]
      else:
         for key in pwaMockBgpStatus.bgpResponse:
            del pwaMockBgpStatus.bgpResponse[ key ]

BgpMacVrfVpwsMode.addCommandClass( DebugPwBgpCmd )

#---------------------------------------------
# Local connector interface review on error-disable removal
#---------------------------------------------
class ConnectorInterfaceRecoveryCmd( CliCommand.CliCommandClass ):
   syntax = """connector interface recovery review delay MIN MAX"""
   noOrDefaultSyntax = """connector interface recovery review delay ..."""
   data = {
         "connector": "Connector",
         "interface": "Local connector",
         "recovery": "Interface recovery from error disabled state",
         "review": "Interface recovery review",
         "delay": "Range of maximum seconds to wait for interface to recover "
                  "before declaring interface state",
         "MIN": CliMatcher.IntegerMatcher( 10, 600, helpdesc="Minimum delay" ),
         "MAX": CliMatcher.IntegerMatcher( 15, 900, helpdesc="Maximum delay" ),
      }

   @staticmethod
   def handler( mode, args ):
      try:
         pwConfig.intfReviewDelayRange = Tac.Value( "Pseudowire::TimeRange",
               args[ 'MIN' ], args[ 'MAX' ] )
      except exceptions.IndexError: # Tac::RangeException
         mode.addError( "Interface recovery review delay range invalid" )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      pwConfig.intfReviewDelayRange = pwConfig.intfReviewDelayRangeDefault

PatchPanelConfigMode.addCommandClass( ConnectorInterfaceRecoveryCmd )

#-------------------------------------------------------------------------------
# Cleanup
#-------------------------------------------------------------------------------

getLdpConfigPseudowireCleanupHook().addExtension( noMplsLdpPseudowires )

def Plugin( entityManager ):
   global pwConfig, bridgingHwCapabilities, pwHwCapability
   global pwRunnability
   global pwaMockLdpStatus, pwaMockBgpStatus
   global bgpMacVrf

   pwConfig = ConfigMount.mount( entityManager, "pseudowire/config",
                                 "Pseudowire::Config", "w" )
   pwRunnability = ConfigMount.mount( entityManager, "pseudowire/runnability/config",
                                      "Pseudowire::Runnability", "w" )
   bridgingHwCapabilities = LazyMount.mount( entityManager,
                                             "bridging/hwcapabilities",
                                             "Bridging::HwCapabilities", "r" )
   pwHwCapability = LazyMount.mount( entityManager,
                                     "routing/hardware/pseudowire/capability",
                                     "Pseudowire::Hardware::Capability", "r" )
   pwaMockLdpStatus = ConfigMount.mount( entityManager,
                                         "pseudowire/ldp/mockStatus",
                                         "Pseudowire::PseudowireLdpStatus",
                                         "w" )
   pwaMockBgpStatus = ConfigMount.mount( entityManager,
                                         "pseudowire/bgp/mockStatus",
                                         "Pseudowire::PseudowireBgpStatus",
                                         "w" )
   bgpMacVrf = LazyMount.mount( entityManager, "routing/bgp/macvrf",
                                "Routing::Bgp::MacVrfConfigDir", "r" )
