#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliUtil

from CliParserStructs import ParserCtx
from CliParserCommon import MatchResult, noMatch, partialMatch

class FieldException( Exception ):
   # Thrown when a field has invalid input
   pass

class PromptTreeField( object ):
   def __init__( self, field, matcher, promptText, required=True,
                 promptLambda=None ):
      self.field = field
      self.matcher = matcher
      self.promptText = promptText
      self.required = required
      self.promptLambda = promptLambda

   def prompt( self, mode, reprompt=False, currentValue=None ):
      promptText = '{}: '.format( self.promptText )
      if reprompt and currentValue:
         promptText = '{} ({}): '.format(
               self.promptText, currentValue )
      value = BasicCliUtil.getSingleLineInput( mode, promptText )
      if self.matcher:
         finalValue = self.matcher.match( mode=mode, token=value,
                                          context=ParserCtx( None, None, None ) )
         # Matchers can return either a regular value, MatchResult, noMatch or
         # partialMatch
         if not finalValue or finalValue == noMatch or finalValue == partialMatch:
            if value == '' and reprompt and currentValue:
               # We're re-prompting for an existing value. If the user inputs no
               # new value then we'll reuse the previous value.
               return currentValue
            elif value == '' and not self.required:
               return value
            else:
               raise FieldException(
                     'Invalid value specified for {}'.format( self.promptText ) )
         if isinstance( finalValue, MatchResult ):
            return finalValue.result
         return finalValue
      else:
         if not value and self.required:
            raise FieldException(
                  'Invalid value specified for {}'.format( self.promptText ) )
         return value

class PromptTreeChoice( PromptTreeField ):
   def __init__( self, field, promptText, choices, promptLambda=None ):
      PromptTreeField.__init__( self, field, None, promptText,
                                promptLambda=promptLambda )
      self.choices = choices

   def prompt( self, mode, reprompt=False, currentValue=None ):
      promptText = '{} ({}): '.format( self.promptText, ', '.join( self.choices ) )
      if reprompt and currentValue is not None:
         promptText = '{} ({}) ({}): '.format(
               self.promptText, ', '.join( self.choices ), currentValue )
      value = BasicCliUtil.getSingleLineInput( mode, promptText )
      if value == '' and reprompt and currentValue:
         return currentValue
      if value not in self.choices:
         raise FieldException( 'Invalid option specified' )
      return value

class PromptTreeBase( object ):
   def __init__( self ):
      self.fields = []
      # Each entry in furtherPromptTrees is of the following format:
      # Tuple( PromptTree, lambda function )
      # Where the lambda function defines if this prompt tree is required. Can be
      # None.
      self.furtherPromptTrees = []

   def prompt( self, mode, treeDict=None, reprompt=False ):
      if not treeDict:
         treeDict = {}

      for field in self.fields:
         # Check that we should prompt for this field
         if field.promptLambda and not field.promptLambda( treeDict ):
            continue

         # Check that there isn't an existing value for this field
         currentValue = treeDict.get( field.field )
         if reprompt or currentValue is None:
            value = field.prompt( mode, reprompt, currentValue )
            treeDict[ field.field ] = value

      for promptTree, requiredLambda in self.furtherPromptTrees:
         if not requiredLambda or requiredLambda( treeDict ):
            promptTree.prompt( mode, treeDict, reprompt )

      return treeDict
