# Copyright (c) 2007-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import ArPyUtils
import BasicCli
import BasicCliModes
import CliCommand
import CliMode.PowerManager
import CliParser
import CliPlugin.ModuleCli as ModuleCli
from CliPlugin.FruCli import SlotExpressionFactory
from CliMatcher import IntegerMatcher, KeywordMatcher
from CliModel import Dict, Float, Model
from CliToken.System import systemMatcherForShow, systemMatcherForConfig
import ConfigMount
import EntityMib
import LazyMount
import ShowCommand
import Tac
import Toggles.PowerManagerToggleLib as PowerManagerToggles
# pkgdeps: rpmwith %{_libdir}/preinit/PowerManager

MAX_POWER_BUDGET = 100000

entityMib = None
powerManager = None
powerManagerCliConfig = None
powerCliConfig = None
managementMode = Tac.Type( "PowerManager::PowerManagerCliConfig::ManagementMode" )

def _systemInitialized():
   return entityMib.root and entityMib.root.initStatus == 'ok'

def _isStandbyMode( mode ):
   return ( mode.session_ and
            mode.session_.entityManager_.redundancyStatus().mode == 'standby' )

def _powerMngString( name, consumedPower ):
   return "{0:<23} {1:<23}".format( name, consumedPower )

class PowerConsumer( Model ):
   amount = Float( help="Amount of power the device is consuming in watts" )

class PowerManager( Model ):
   powerConsumers = Dict( keyType=str, valueType=PowerConsumer,
                          help="Devices consuming power" )
   totalAvailablePower = Float( help="Total available power in watts" )
   powerBudget = Float( help="Budgeted power in watts" )
   consumedPower = Float( help="Total consumed power in watts" )

   def render( self ):
      if self.powerBudget > MAX_POWER_BUDGET:
         defaultBudgetStr = "Budget is not set, using all of {0:.1f}W of total power"
         print( defaultBudgetStr.format( self.totalAvailablePower ) )
      else:
         budgetStr = "Budget is {0:.1f}W out of {1:.1f}W of total power"
         print( budgetStr.format( self.powerBudget, self.totalAvailablePower ) )
      print( "Active devices using up to {0:.1f}W\n".format( self.consumedPower ) )
      print( _powerMngString( "Device", "Consumed" ) )
      print( _powerMngString( "Name", "Power" ) )
      print( _powerMngString( "-" * 23, "-" * 23 ) )
      for name in ArPyUtils.naturalsorted( self.powerConsumers ):
         consumer = self.powerConsumers[ name ]
         print( _powerMngString( name, str( round( consumer.amount, 1 ) ) + "W" ) )
      print( _powerMngString( "Total",
                              str( round( self.consumedPower, 1 ) ) + "W" ) )

#--------------------------------------------------
# '[ no | default ] power budget [n] watts'
#--------------------------------------------------
def powerSupplyBudget( mode, args ):
   powerBudgetAmount = float( args[ "WATTS" ] )
   powerManagerCliConfig.powerBudget = powerBudgetAmount
   if powerManager.totalPower < powerBudgetAmount <= MAX_POWER_BUDGET:
      warning = "Budget is set to {0:.1f}W, which is greater than {1:.1f}W of " \
                "total power"
      mode.addWarning( warning.format( powerBudgetAmount,
                                       powerManager.totalPower ) )

class PowerBudgetCmd( CliCommand.CliCommandClass ):
   syntax = 'power budget WATTS watts'
   noOrDefaultSyntax = 'power budget ...'
   data = {
      'power': 'Configure power supplies',
      'budget': 'Configure power budget (excluding chassis power)',
      'WATTS': IntegerMatcher( 0, MAX_POWER_BUDGET,
         helpdesc='Amount of power budgeted to the system' ),
      'watts': 'Unit for power budget',
   }

   @staticmethod
   def handler( mode, args ):
      powerSupplyBudget( mode, args )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      args[ "WATTS" ] = "inf"
      powerSupplyBudget( mode, args )

BasicCliModes.GlobalConfigMode.addCommandClass( PowerBudgetCmd )

#--------------------------------------------------
# '[ no | default ] power threshold [n] watts'
#--------------------------------------------------
def powerSupplyThreshold( mode, args ):
   powerThresholdAmount = float( args.get( "WATTS", "inf" ) )
   powerCliConfig.powerThreshold = powerThresholdAmount
   if powerManager.totalPower < powerThresholdAmount <= MAX_POWER_BUDGET:
      warning = "Threshold is set to {0:.1f}W, which is greater than {1:.1f}W of " \
                "total power"
      mode.addWarning( warning.format( powerThresholdAmount,
                                       powerManager.totalPower ) )

class PowerThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'power threshold WATTS watts'
   noOrDefaultSyntax = 'power threshold ...'
   data = {
      'power': 'Configure power supplies',
      'threshold': 'Configure power threshold',
      'WATTS': IntegerMatcher( 0, MAX_POWER_BUDGET,
         helpdesc='Amount of power consumed that will trigger a warning' ),
      'watts': 'Unit for power threshold',
   }
   handler = powerSupplyThreshold
   noOrDefaultHandler = handler

powerToggle = PowerManagerToggles.togglePowerManagementModularEnabled()
if powerToggle:
   BasicCliModes.GlobalConfigMode.addCommandClass( PowerThresholdCmd )

#-----------------------------------------------------------------------------
# '[ no | default ] power budget exceeded action [ warning | hold-down ]
#-----------------------------------------------------------------------------
matcherWarning = KeywordMatcher(
   'warning', helpdesc='Raise a log message but allow the device to power on' )
matcherHoldDown = KeywordMatcher(
   'hold-down', helpdesc='Prevent the device from powering on' )

class PowerBudgetActionCmd( CliCommand.CliCommandClass ):
   syntax = 'power budget exceeded action [ warning | hold-down ]'
   noOrDefaultSyntax = 'power budget exceeded action ...'
   data = {
      'power': 'Configure power supplies',
      'budget': 'Configure power budget (excluding chassis power)',
      'exceeded': 'Actions when devices exceed budget',
      'action': 'Actions when devices exceed budget',
      'warning': matcherWarning,
      'hold-down': matcherHoldDown,
   }

   @staticmethod
   def handler( mode, args ):
      if "warning" in args:
         powerManagerCliConfig.mode = managementMode.warn
      elif "hold-down" in args:
         powerManagerCliConfig.mode = managementMode.strict

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      powerManagerCliConfig.mode = managementMode.strict

BasicCliModes.GlobalConfigMode.addCommandClass( PowerBudgetActionCmd )

#-----------------------------------------------------------------------------
# '[ no | default ] power threshold exceeded action [ warning ]
#-----------------------------------------------------------------------------
class PowerThresholdActionCmd( CliCommand.CliCommandClass ):
   syntax = 'power threshold exceeded action [ warning ]'
   noOrDefaultSyntax = 'power threshold exceeded action ...'
   data = {
      'power': 'Configure power supplies',
      'threshold': 'Configure power threshold',
      'exceeded': 'Actions when power exceeds threshold',
      'action': 'Actions when power exceeds threshold',
      'warning': matcherWarning,
   }

   # Establish basis for future cli actions with regards to threshold commands
   # Exists now to indicate to the user what is being configured.
   # Currently only has the one option of warning so handlers do nothing
   @staticmethod
   def handler( mode, args ):
      return

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      return

if powerToggle:
   BasicCliModes.GlobalConfigMode.addCommandClass( PowerThresholdActionCmd )

#--------------------------------------------------------------------------------
# show system environment power budget
#--------------------------------------------------------------------------------
matcherEnvironment = KeywordMatcher( 'environment',
                                     helpdesc='Show system environment status' )
matcherPower = KeywordMatcher( 'power',
                               helpdesc='Show system power status' )
matcherBudget = KeywordMatcher( 'budget',
                                helpdesc='Show usage levels for the power budget'
                                ' ( excluding chassis power )' )

class SystemEnvironmentPowerBudgetCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show system environment power budget'
   data = {
      'system': systemMatcherForShow,
      'environment': matcherEnvironment,
      'power': matcherPower,
      'budget': matcherBudget,
   }

   @staticmethod
   def handler( mode, args ):
      if not _systemInitialized():
         if _isStandbyMode( mode ):
            mode.addError( 'This command only works on the active supervisor.' )
         else:
            mode.addError( 'System is not yet initialized.' )
         return None

      model = PowerManager()

      consumedPower = float( 0 )
      for consumer in powerManager.powerConsumers:
         device = powerManager.powerConsumers[ consumer ]
         model.powerConsumers[ consumer ] = PowerConsumer( amount=device.amount )
         consumedPower += device.amount

      model.totalAvailablePower = powerManager.totalPower
      model.powerBudget = powerManagerCliConfig.powerBudget
      model.consumedPower = consumedPower
      return model

   cliModel = PowerManager

BasicCli.addShowCommandClass( SystemEnvironmentPowerBudgetCmd )

#------------------------------------------------------------------------------------
# power management config mode
#------------------------------------------------------------------------------------
class PowerManagementMode( CliMode.PowerManager.PowerManagementMode,
                           BasicCli.ConfigModeBase ):
   name = 'Power Management configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.PowerManager.PowerManagementMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#------------------------------------------------------------------------------------
# power management card config mode
#------------------------------------------------------------------------------------
class PowerManagementCardMode( CliMode.PowerManager.PowerManagementCardMode,
                                   BasicCli.ConfigModeBase ):
   name = 'Power Management card configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      CliMode.PowerManager.PowerManagementCardMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------------------------------
# system power
# in global config mode
#-----------------------------------------------------------------------------------
matcherPowerConfig = KeywordMatcher( 'power',
                                     helpdesc='configure system power settings' )

class PowerManagement( CliCommand.CliCommandClass ):
   syntax = 'system power'
   data = {
      'system': systemMatcherForConfig,
      'power': matcherPowerConfig
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PowerManagementMode )
      mode.session_.gotoChildMode( childMode )

modeToggle = PowerManagerToggles.togglePowerManagementModeEnabled()
if modeToggle:
   BasicCli.GlobalConfigMode.addCommandClass( PowerManagement )

#------------------------------------------------------------------------------------
# module MODNAME
# in power management config mode
#------------------------------------------------------------------------------------
class EnterModuleCardCmd( CliCommand.CliCommandClass ):
   syntax = 'module MODNAME'
   data = {
      'module': ModuleCli.moduleNode,
      'MODNAME': SlotExpressionFactory(),
   }

   @staticmethod
   def handler( mode, args ):
      card = args[ 'MODNAME' ]
      childMode = mode.childMode( PowerManagementCardMode,
                                  param=str( card.tag ) + str( card.label ) )
      mode.session_.gotoChildMode( childMode )

PowerManagementMode.addCommandClass( EnterModuleCardCmd )

#--------------------------------------------------
# Plugin method - Mount the objects we need from Sysdb
#--------------------------------------------------
def Plugin( entityManager ):
   global entityMib
   global powerManager, powerManagerCliConfig, powerCliConfig

   entityMib = LazyMount.mount( entityManager, 'hardware/entmib',
                                'EntityMib::Status', 'r' )
   powerManager = LazyMount.mount( entityManager,
                                   'environment/archer/power/status/powerManager',
                                   'PowerManager::PowerManagerStatus', 'r' )
   powerManagerCliConfig = ConfigMount.mount( entityManager,
                           'environment/archer/power/config/powerManagerCliConfig',
                           'PowerManager::PowerManagerCliConfig', 'w' )
   powerCliConfig = ConfigMount.mount( entityManager,
                                       "hardware/powerSupply/config/cli",
                                       "Hardware::PowerSupply::CliConfig", "w" )
