#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
from CliModel import Dict, List, Model, Str, Submodel, Int, Bool
import Arnet
import ArnetModel
from IntfModels import Interface
from Intf.IntfRange import intfListToCanonical
from CliPlugin.TrafficPolicyCliModel import Rule

class ProfileModel( Model ):
   interfaces = List( valueType=Interface, help='Interface list' )
   appliedInterfaces = List( valueType=Interface, help='Applied interface list' )
   samplePolicy = Str( help="Sample Policy", optional=True )

   def render( self ):
      print( 'Sample policy:', self.samplePolicy )
      print( 'Configured on:',
             ','.join( intfListToCanonical( Arnet.sortIntf( self.interfaces ) ) ) )
      print( 'Active on:',
             ','.join( intfListToCanonical( Arnet.sortIntf(
                self.appliedInterfaces ) ) ) )
      print( "" )

class ModelPostcardProfile( Model ):
   profiles = Dict( keyType=str, valueType=ProfileModel,
                    help="Maps core profiles to their corresponding configuration." )

   def render( self ):
      if self.profiles:
         print( 'Profiles' )
         for name in sorted( self.profiles.keys() ):
            print( 'Name:', name )
            profile = self.profiles[ name ]
            profile.render()

class SamplePolicyModel( Model ):
   rules = List( valueType=Rule, help="Detailed information of match rules" )

   def render( self ):
      print( "Total number of rules configured:", len( self.rules ) )
      for rule in self.rules:
         print( "match {} {}:".format( rule.matchOption, rule.ruleString ) )
         if rule.matches.srcPrefixes:
            srcPrefixes = [ "%s" % src for src in rule.matches.srcPrefixes ]
            srcStr = ' '.join( srcPrefixes ).rstrip( ',' )
            print( "\tSource:", srcStr )
         if rule.matches.destPrefixes:
            destPrefixes = [ "%s" % dst for dst in rule.matches.destPrefixes ]
            dstStr = ' '.join( destPrefixes ).rstrip( ',' )
            print( "\tDestination:", dstStr )
         if rule.matches.protocols:
            for proto in sorted( rule.matches.protocols,
                                 key=lambda proto: proto.protocolRange.low ):
               protoStr = ""
               low = proto.protocolRange.low
               high = proto.protocolRange.high
               if low == high == 6:
                  protoStr += "tcp"
               elif low == high == 17:
                  protoStr += "udp"
               if proto.ports:
                  protoPort = proto.ports[ 0 ]
                  if protoPort.srcPorts:
                     srcPorts = protoPort.srcPorts
                     protoStr += '\n\t\tSource port: '
                     protoStr += ', '.join(
                        str( port.low ) if port.low == port.high else
                        str( port.low ) + '-' + str( port.high )
                        for port in sorted( srcPorts, key=lambda port: port.low ) )
                     protoStr = protoStr.rstrip( " " )
                     protoStr = protoStr.rstrip( "," )
                  if protoPort.destPorts:
                     destPorts = protoPort.destPorts
                     protoStr += '\n\t\tDestination port: '
                     protoStr += ', '.join(
                        str( port.low ) if port.low == port.high else
                        str( port.low ) + '-' + str( port.high )
                        for port in sorted( destPorts, key=lambda port: port.low ) )
                     protoStr = protoStr.rstrip( " " )
                     protoStr = protoStr.rstrip( "," )
               print( "\tProtocol:", protoStr )
         actions = rule.actions
         if actions.sample or actions.sampleAll:
            action = "sample" if actions.sample else "sample all"
            print( "\tActions:", action )
      print( "" )

class SamplePoliciesModel( Model ):
   policies = Dict( keyType=str, valueType=SamplePolicyModel,
                    help="Maps sample policy name to its configuration" )

   def render( self ):
      if self.policies:
         for name in sorted( self.policies.keys() ):
            policy = self.policies[ name ]
            print( "Sample policy", name )
            policy.render()

class Collection( Model ):
   srcIp = ArnetModel.Ip4Address( help="Source IP address" )
   dstIp = ArnetModel.Ip4Address( help="Destination IP address" )
   def render( self ):
      print( "Ingress collection type: GRE" )
      print( "Ingress collection source:", str( self.srcIp ) )
      print( "Ingress collection destination:", str( self.dstIp ) )

class PostcardTelemetry( Model ):
   enabled = Bool( help="Postcard telemetry is enabled" )
   sampleRate = Int( help="Sampling rate of 1 in 16k, 32k or 64k packets" )
   collection = Submodel( valueType=Collection, help="Collection attributes",
                          optional=True )

   def render( self ):
      print( "Enabled:", str( self.enabled ).lower() )
      print( "Ingress collection sample rate:", self.sampleRate )
      if self.collection:
         self.collection.render()
