#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=ungrouped-imports
from __future__ import absolute_import, division, print_function
import Cell
import CliParser
import ConfigMount
import LazyMount
import Plugins
import CliCommand
import ShowCommand
import CliMatcher
import BasicCli
import CliPlugin.IntfCli
from CliPlugin.IpGenAddrMatcher import IpGenAddrMatcher
import Tac
import Tracing
from CliPlugin.ClassificationCliLib import ( CommitAbortModelet,
                                             tcpUdpProtoExpr,
                                             portKwMatcher,
                                             ProtocolMixin, PrefixIpv4SingletonCmd )
from CliMode.PostcardTelemetry import ( PostcardPolicyConfigMode,
                                        PostcardSamplePolicyConfigMode,
                                        PostcardSamplePolicyMatchRuleIpv4ConfigMode,
                                        PostcardSamplePolicyActionRuleConfigMode,
                                        ProfileConfigMode, FEATURE )
from CliPlugin.TrafficPolicyCli import ( matchRuleName, portExpression,
                                         MatchRuleConfigCmdBase,
                                         ActionsConfigCmdBase )
from CliPlugin.PostcardTelemetryPolicyCliLib import (
   PostcardSamplePolicyMatchRuleContext,
   PostcardSamplePolicyContext )
from CliPlugin.TrafficPolicyCliModel import (
   NumericalRange,
   Protocol,
   Actions,
   Rule,
   Matches,
   Port,
   )
from CliPlugin.PostcardTelemetryPolicyCliModel import (
   PostcardTelemetry,
   ProfileModel,
   ModelPostcardProfile,
   SamplePolicyModel,
   SamplePoliciesModel,
   Collection,
   )
from Arnet import IpGenAddr
import CliToken
from CliToken.Monitor import monitorMatcher
from CliPlugin.SamplePolicyCliLib import ( SamplePolicyConfigCmd,
                                           SampleActionCmd )
import Toggles.PostcardTelemetryCommonToggleLib

__defaultTraceHandle__ = Tracing.Handle( 'PostcardTelemetryCli' )
t0 = Tracing.trace0

#------------------------------------------------------------------------------------
# Global variables
#------------------------------------------------------------------------------------
postcardTelemetryConfig = None
postcardTelemetryStatus = None
allVrfStatus = None
hwCapability = None
policiesCliConfig = None
policiesStatusRequestDir = None
policiesStatus = None
tacSampleRate = Tac.Type( 'PostcardTelemetry::SampleRate' )
postcardDefaults = Tac.Type( 'PostcardTelemetry::Defaults' )
tacInstalledStatus = Tac.Type( 'PolicyMap::InstalledStatus' )

#------------------------------------------------------------------------------------
# Utility functions
#------------------------------------------------------------------------------------
# pylint: disable-msg=R1703

def identicalProfile( profile, currentEntry ):
   if not profile or not currentEntry:
      return False
   if profile.samplePolicy != currentEntry.samplePolicy:
      return False
   return True

def identicalTelemetryConfig( oldConfig, newConfig ):
   if oldConfig.enable != newConfig.enable:
      return False
   oldPortProfile = oldConfig.portProfile.keys()
   newPortProfile = newConfig.portProfile.keys()
   oldIntfToPortProfile = oldConfig.intfToPortProfile.values()
   newIntfToPortProfile = newConfig.intfToPortProfile.values()
   if set( oldPortProfile ) != set( newPortProfile ):
      oldSet = set( oldPortProfile )
      newSet = set( newPortProfile )
      if list( oldSet - newSet ) != [ 'default' ] or \
         list( newSet - oldSet ) != []:
         return False
   for profile in oldPortProfile:
      if profile == 'default':
         continue
      oldProfile = oldConfig.portProfile[ profile ]
      newProfile = newConfig.portProfile[ profile ]
      if not identicalProfile( oldProfile, newProfile ):
         return False
   if oldIntfToPortProfile != newIntfToPortProfile:
      return False
   if oldConfig.destinationConfig != newConfig.destinationConfig:
      return False
   if oldConfig.sampleRate != newConfig.sampleRate:
      return False
   return True

def isDefaultDestination( dstConfig ):
   return dstConfig.srcIp.v4Addr == '0.0.0.0' and \
      dstConfig.dstIp.v4Addr == '0.0.0.0'

def copyProfile( newProfile, oldProfile ):
   newProfile.samplePolicy = oldProfile.samplePolicy

def getPortProfileNames( mode, context ):
   return postcardTelemetryConfig.portProfile.keys()

def defaultProfileGuard( mode, token ):
   if token == 'default':
      return CliParser.guardNotThisPlatform
   return None

profileNameMatcher = CliCommand.Node( matcher=CliMatcher.DynamicNameMatcher(
      getPortProfileNames, "Profile name",
      passContext=True ), storeSharedResult=True )

def postcardTelemetryGuard( mode, token ):
   if hwCapability.postcardTelemetrySupported:
      return None
   return CliParser.guardNotThisPlatform

#------------------------------------------------------------------------------------
# Modelet under Intf mode for postcard telemetry command
#------------------------------------------------------------------------------------

class IntfPostcardTelemetryModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )

   @staticmethod
   def shouldAddModeletRule( mode ):
      return mode.intf.name.startswith( 'Ethernet' ) and not mode.intf.isSubIntf()

CliPlugin.IntfCli.IntfConfigMode.addModelet( IntfPostcardTelemetryModelet )

#------------------------------------------------------------------------------------
# The "monitor telemetry postcard" mode context
#------------------------------------------------------------------------------------

class PostcardTelemetryContext( object ):
   def __init__( self, mode ):
      self.mode = mode
      self.config = postcardTelemetryConfig
      self.currentEntry_ = None
      self.previousEntry_ = Tac.newInstance( 'PostcardTelemetry::PolicyConfig',
                                             "intConfig" )
      self.previousEntry_.sampleRate = postcardTelemetryConfig.sampleRate
      self.previousEntry_.enable = postcardTelemetryConfig.enable

   def copyEditEntry( self ):
      newEntry = Tac.newInstance( 'PostcardTelemetry::PolicyConfig',
                                  "intConfig" )
      for profile in self.config.portProfile:
         newEntry.portProfile.newMember( profile )
         newProfile = newEntry.portProfile[ profile ]
         newProfile.samplePolicy = self.config.portProfile[
            profile ].samplePolicy
      for intf, profile in self.config.intfToPortProfile.items():
         newEntry.intfToPortProfile[ intf ] = profile
      newEntry.destinationConfig = self.config.destinationConfig
      newEntry.sampleRate = self.config.sampleRate
      newEntry.enable = self.config.enable
      self.currentEntry_ = newEntry
      return newEntry

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      if self.currentEntry_:
         if identicalTelemetryConfig( self.config, self.currentEntry_ ):
            self.config.enable = self.currentEntry_.enable
            return

         # Set enable upon successfully committing from
         # postcard telemetry mode's context.
         self.config.sampleRate = self.currentEntry_.sampleRate
         self.config.destinationConfig = self.currentEntry_.destinationConfig
         self.config.enable = self.currentEntry_.enable

#------------------------------------------------------------------------------------
# The "monitor telemetry MODE" mode command
#------------------------------------------------------------------------------------

class MonitorTelemetryPostcard( CliCommand.CliCommandClass ):
   syntax = '''monitor telemetry postcard policy'''
   noOrDefaultSyntax = syntax
   data = {
      'monitor': monitorMatcher,
      'telemetry': 'Telemetry configuration',
      'postcard': CliCommand.guardedKeyword( 'postcard',
                                             helpdesc='Postcard telemetry',
                                             guard=postcardTelemetryGuard ),
      'policy': 'Postcard telemetry of type match policy',
   }

   @staticmethod
   def handler( mode, args ):
      context = PostcardTelemetryContext( mode )
      context.copyEditEntry()
      childMode = mode.childMode( PostcardPolicyConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      for name in policiesCliConfig.pmapType.pmap:
         # pylint: disable=protected-access
         if name != 'default':
            PostcardSamplePolicyConfigCmd._removePolicy( mode, name )
      profiles = postcardTelemetryConfig.portProfile.keys()
      for profile in profiles:
         if profile != 'default':
            del postcardTelemetryConfig.portProfile[ profile ]
      postcardTelemetryConfig.sampleRate = tacSampleRate.sampleRate1in32k
      postcardTelemetryConfig.destinationConfig = Tac.Value(
         'PostcardTelemetry::DestinationConfig',
         IpGenAddr( '0.0.0.0' ), IpGenAddr( '0.0.0.0' ),
         postcardDefaults.ttl, postcardDefaults.dscp,
         postcardDefaults.vrf, postcardDefaults.greProtocolType )
      postcardTelemetryConfig.enable = False

#[ no ]  match  <match-rule-name> ipv4
class PostcardSamplePolicyMatchRuleIpv4ConfigCmd( MatchRuleConfigCmdBase ):
   syntax = 'match RULE_NAME ipv4'
   noOrDefaultSyntax = 'match RULE_NAME ipv4'
   data = {
      'match': 'Configure match rule',
      'RULE_NAME': matchRuleName,
      'ipv4': 'IPv4 match rule',
   }

   @staticmethod
   def _feature():
      return FEATURE

   @staticmethod
   def _context( trafficPolicyContext, ruleName, matchOption ):
      return PostcardSamplePolicyMatchRuleContext( trafficPolicyContext, ruleName,
                                                   matchOption )

class PostcardSamplePolicyActionsConfigCmd( ActionsConfigCmdBase ):
   @staticmethod
   def _feature():
      return FEATURE

#------------------------------------------------------------------------------------
# The profile  mode context
#------------------------------------------------------------------------------------

class ProfileModeContext( object ):
   def __init__( self, mode, profileName ):
      self.mode = mode
      self.profile_ = None
      self.profileName_ = profileName
      self.currentEntry_ = None
      self.previousEntry_ = None
      portProfileConfig = postcardTelemetryConfig.portProfile
      if profileName in portProfileConfig:
         self.profile_ = portProfileConfig[ profileName ]
         prevProfile = Tac.newInstance( 'PostcardTelemetry::PolicyPortProfileConfig',
                                        profileName )
         copyProfile( prevProfile, self.profile_ )
         self.previousEntry_ = prevProfile

   def copyEditEntry( self ):
      newProfile = Tac.newInstance( 'PostcardTelemetry::PolicyPortProfileConfig',
                                    self.profileName_ )
      copyProfile( newProfile, self.profile_ )
      self.currentEntry_ = newProfile
      return newProfile

   def newEditEntry( self ):
      newProfile = Tac.newInstance( 'PostcardTelemetry::PolicyPortProfileConfig',
                                    self.profileName_ )
      self.currentEntry_ = newProfile

   def profileName( self ):
      return self.profileName_

   def currentEntry( self ):
      return self.currentEntry_

   def commit( self ):
      if identicalProfile( self.profile_, self.currentEntry_ ):
         return
      if self.profile_ is None:
         self.profile_ = postcardTelemetryConfig.portProfile.newMember(
            self.profileName_ )
      for intf in postcardTelemetryConfig.intfToPortProfile:
         if postcardTelemetryConfig.intfToPortProfile[ intf ] != \
            self.profileName_:
            continue
      self.profile_.samplePolicy = self.currentEntry_.samplePolicy

def getSamplePolicyNames( mode ):
   if not policiesCliConfig:
      return []
   return policiesCliConfig.pmapType.pmapInput

samplePolicyNameMatcher = CliCommand.Node( CliMatcher.DynamicNameMatcher(
   getSamplePolicyNames, "Sample Policy Name",
   pattern=r'[A-Za-z0-9_:{}\[\]-]+' ), guard=defaultProfileGuard )

class PostcardSamplePolicyConfigCmd( SamplePolicyConfigCmd ):
   data = {
      'POLICY_NAME': samplePolicyNameMatcher,
   }
   data.update( SamplePolicyConfigCmd._baseData )

   @staticmethod
   def _feature():
      return FEATURE

   @classmethod
   def _context( cls, name ):
      return PostcardSamplePolicyContext( policiesCliConfig,
                                          policiesStatusRequestDir,
                                          policiesStatus, name )

ingressMatcher = CliMatcher.KeywordMatcher( 'ingress',
                                            helpdesc='Configure ingress paramters' )

class GreDestinationCmd( CliCommand.CliCommandClass ):
   syntax = """ingress collection gre source GRESRCADDR destination GREDSTADDR"""
   # For phase 1, optional parameters are not supported in this CLI command
   #
   #            [ { ( ttl TTLVAL )
   #              | ( dscp DSCPVAL )
   #              | ( protocol PROTOVAL )
   #              | ( VRF ) } ]"""
   noOrDefaultSyntax = "ingress collection gre ..."
   data = {
      'ingress': ingressMatcher,
      'collection': 'Collector configuration',
      'gre': 'GRE conifguration',
      'source': 'GRE source configuration',
      'GRESRCADDR': IpGenAddrMatcher(
         helpdesc='Source IP address of GRE tunnel' ),
      'destination': 'GRE destination configuration',
      'GREDSTADDR': IpGenAddrMatcher(
         helpdesc='Destination IP address of GRE tunnel' )

      #'ttl': CliCommand.Node( CliMatcher.KeywordMatcher(
      #   'ttl', 'TTL of the GRE tunnel' ), maxMatches=1 ),
      #'TTLVAL': CliCommand.Node(
      #   CliMatcher.IntegerMatcher( 1, 255, helpdesc='TTL range' ), maxMatches=1 ),
      #'dscp': CliCommand.Node(
      #   CliMatcher.KeywordMatcher( 'dscp', 'DSCP of the GRE tunnel' ),
      #   maxMatches=1 ),
      #'DSCPVAL': CliCommand.Node(
      #   CliMatcher.IntegerMatcher( 0, 63, helpdesc='DSCP range' ), maxMatches=1 ),
      #'protocol': CliCommand.Node(
      #   CliMatcher.KeywordMatcher( 'protocol', 'Protocol type in GRE header' ),
      #    maxMatches=1 ),
      #'PROTOVAL': CliCommand.Node( CliMatcher.IntegerMatcher( 0, 65535,
      #   helpdesc='Protocol range', helpname='0x0000-0xFFFF' ), maxMatches=1 ),
      #'VRF': VrfExprFactory( helpdesc='VRF name of the GRE tunnel', maxMatches=1 )
   }

   @staticmethod
   def handler( mode, args ):
      greSrcAddr = args[ 'GRESRCADDR' ].stringValue
      greDstAddr = args[ 'GREDSTADDR' ].stringValue
      ttlVal = args[ 'TTLVAL' ] if 'ttl' in args else postcardDefaults.ttl
      dscpVal = args[ 'DSCPVAL' ] if 'dscp' in args else postcardDefaults.dscp
      protocol = args[ 'PROTOVAL' ] \
                 if 'protocol' in args else postcardDefaults.greProtocolType
      vrf = args.get( 'VRF', postcardDefaults.vrf )
      mode.telemetryContext.currentEntry_.destinationConfig = Tac.Value(
         'PostcardTelemetry::DestinationConfig',
         IpGenAddr( greSrcAddr ), IpGenAddr( greDstAddr ),
         ttlVal, dscpVal, vrf, protocol )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      mode.telemetryContext.currentEntry_.destinationConfig = Tac.Value(
         'PostcardTelemetry::DestinationConfig',
         IpGenAddr( '0.0.0.0' ), IpGenAddr( '0.0.0.0' ),
         postcardDefaults.ttl, postcardDefaults.dscp,
         postcardDefaults.vrf, postcardDefaults.greProtocolType )

#------------------------------------------------------------------------------------
# The "profile <profile-type> <profile-name>" mode command
#------------------------------------------------------------------------------------
class ProfileConfigCmd( CliCommand.CliCommandClass ):
   syntax = '''profile PROFILE'''
   noOrDefaultSyntax = syntax
   data = {
      'profile': 'Configure postcard telemetry profile',
      'PROFILE': profileNameMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      profileName = args[ 'PROFILE' ]
      context = ProfileModeContext( mode, profileName )

      if profileName in postcardTelemetryConfig.portProfile:
         context.copyEditEntry()
      else:
         context.newEditEntry()
      childMode = mode.childMode( ProfileConfigMode,
                                  context=context )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE' ]
      del postcardTelemetryConfig.portProfile[ profileName ]

rateMatcher = CliMatcher.EnumMatcher( {
   '16384': 'Set sample rate to 1 in 16k packets',
   '32768': 'Set sample rate to 1 in 32k packets',
   '65536': 'Set sample rate to 1 in 64k packets',
} )

class SampleRateCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample rate RATE_VAL'''
   noOrDefaultSyntax = '''ingress sample rate ...'''
   data = {
      'ingress': ingressMatcher,
      'sample': 'Configure sampling parameters',
      'rate': 'Configure sampling rate',
      'RATE_VAL': rateMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      sampleRate = args[ 'RATE_VAL' ]
      currEntry = mode.telemetryContext.currentEntry_
      if sampleRate == '16384':
         currEntry.sampleRate = tacSampleRate.sampleRate1in16k
      elif sampleRate == '32768':
         currEntry.sampleRate = tacSampleRate.sampleRate1in32k
      elif sampleRate == '65536':
         currEntry.sampleRate = tacSampleRate.sampleRate1in64k

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      currEntry = mode.telemetryContext.currentEntry_
      currEntry.sampleRate = tacSampleRate.sampleRate1in32k

class SamplePolicyCmd( CliCommand.CliCommandClass ):
   syntax = '''ingress sample policy POLICY_NAME'''
   noOrDefaultSyntax = '''ingress sample policy [ POLICY_NAME ]'''
   data = {
      'ingress': ingressMatcher,
      'sample': 'Configure sampling parameters',
      'policy': 'Configure sample policy',
      'POLICY_NAME': samplePolicyNameMatcher
   }

   @staticmethod
   def handler( mode, args ):
      policyName = args[ 'POLICY_NAME' ]
      profile = mode.profileContext.currentEntry_
      profile.samplePolicy = policyName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profile = mode.profileContext.currentEntry_
      policyName = args.get( 'POLICY_NAME' )
      if not policyName or policyName == profile.samplePolicy:
         profile.samplePolicy = ""

class DisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'disabled'
   noOrDefaultSyntax = syntax
   data = {
      'disabled': 'Enable or disable the postcard telemetry feature'
   }

   @staticmethod
   def handler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = False

   defaultHandler = handler

   @staticmethod
   def noHandler( mode, args ):
      mode.telemetryContext.currentEntry_.enable = True

class PostcardTelemetryInterface( CliCommand.CliCommandClass ):
   syntax = '''telemetry postcard policy [ profile PROFILE ]'''
   noOrDefaultSyntax = syntax
   data = {
      'postcard': 'Apply postcard variant',
      'telemetry': 'Apply telemetry feature',
      'policy' : 'Apply postcard telemetry profile of type match policy',
      'profile': 'Apply postcard telemetry profile',
      'PROFILE': profileNameMatcher
      }

   @staticmethod
   def handler( mode, args ):
      profileName = args.get( 'PROFILE', 'default' )
      intfName = mode.intf.name
      intfToPortProfile = postcardTelemetryConfig.intfToPortProfile
      if intfToPortProfile.get( intfName ) == profileName:
         return
      del postcardTelemetryConfig.intfToPortProfile[ intfName ]
      intfToPortProfile[ intfName ] = profileName

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args.get( 'PROFILE' )
      intfName = mode.intf.name
      intfToPortProfile = postcardTelemetryConfig.intfToPortProfile
      if intfName in intfToPortProfile:
         if not profileName or ( profileName and
                                 intfToPortProfile.get( intfName ) == profileName ):
            del intfToPortProfile[ intfName ]

class AbortCmd( CliCommand.CliCommandClass ):
   syntax = '''abort'''
   data = {
      'abort': CliToken.Cli.abortMatcher,
   }

   @staticmethod
   def handler( mode, args ):
      mode.abort()

class PostcardProtocolIpv4ConfigCmd( ProtocolMixin ):
   syntax = ( 'protocol TCP_UDP ( source port1 [ SPORT ] ) | '
              '( destination port2 [ DPORT ] ) |'
              '( source port1 SPORT destination port2 DPORT ) )' )
   noOrDefaultSyntax = ( 'protocol TCP_UDP '
                         '( source port1 [ SPORT ] ) | '
                         '( destination port2 [ DPORT ] ) | '
                         '( source port1 SPORT destination port2 DPORT ) ) ]' )

   data = {
      'protocol': 'Protocol',
      'TCP_UDP': tcpUdpProtoExpr,
      'port1': portKwMatcher,
      'port2': portKwMatcher,
      'source': 'Source',
      'SPORT': portExpression( 'SPORT' ),
      'destination': 'Destination',
      'DPORT': portExpression( 'DPORT' )
   }

   @classmethod
   def handler( cls, mode, args ):
      proto = ( args.get( cls._tcpUdpArgsListName ) )
      assert proto, args
      source = 'source' in args
      destination = 'destination' in args
      cls._updateProtoAndPort( mode, args, proto,
                               source, destination, add=True )
      cls._maybeHandleErrors( mode, args, proto, source, destination )

   @classmethod
   def noOrDefaultHandler( cls, mode, args ):
      proto = args.get( cls._tcpUdpArgsListName )
      source = 'source' in args
      destination = 'destination' in args
      if not proto and not source and not destination:
         # 'no protocol' removes all protocols
         proto = True
      cls._updateProtoAndPort( mode, args, proto,
                               source, destination, add=False )

#------------------------------------------------------------------------------------
# "show monitor telemetry inband profile [ NAME ]"
#------------------------------------------------------------------------------------

def isAppliedOnProfile( profileName, intf ):
   profile = postcardTelemetryConfig.portProfile[ profileName ]
   samplePolicy = profile.samplePolicy
   if samplePolicy:
      if 'strataPostcard' in policiesStatus:
         if samplePolicy in policiesStatus[ 'strataPostcard' ].status:
            policyStatus = policiesStatus[ 'strataPostcard' ].status[
               samplePolicy ]
            if intf in policyStatus.intfStatus:
               intfStatus = policyStatus.intfStatus[ intf ]
               matchOption = 'matchIpAccessGroup'
               if matchOption in intfStatus.installed and \
                  intfStatus.installed[ matchOption ] == \
                  tacInstalledStatus( 'success', '' ):
                  return True
   return False

def nameInterfaceLinker( profileName ):
   profileNameToIntf = []
   profileNameToAppliedIntf = []
   for intf, profName in\
     sorted( postcardTelemetryConfig.intfToPortProfile.iteritems() ):
      if profName == profileName:
         profileNameToIntf.append( intf )
         if isAppliedOnProfile( profileName, intf ):
            profileNameToAppliedIntf.append( intf )

   return profileNameToIntf, profileNameToAppliedIntf

def getProfiles( profileName=None ):
   profiles = {}
   profileNameToIntf = {}
   if not profileName:
      profileList = postcardTelemetryConfig.portProfile
   else:
      profileList = [ profileName ]

   for name in profileList:
      profileNameToIntf, profileNameToAppliedIntf = nameInterfaceLinker( name )
      samplePolicy = None
      if name in postcardTelemetryConfig.portProfile:
         profileConfig = postcardTelemetryConfig.portProfile[ name ]
         samplePolicy = profileConfig.samplePolicy
      profiles[ name ] = ProfileModel( interfaces=profileNameToIntf,
                                       appliedInterfaces=profileNameToAppliedIntf,
                                       samplePolicy=samplePolicy )
   return profiles

def showMonitorTelemetryPostcardPolicyProfile( mode, args ):
   profileName = args.get( 'NAME' )
   profiles = getProfiles( profileName )
   return ModelPostcardProfile( profiles=profiles )

class ShowMonitorTelemetryPostcardPolicyProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard policy profile [ NAME ]'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher(
            'telemetry', helpdesc='Show monitor telemetry information' ),
            guard=postcardTelemetryGuard ),
         'postcard': 'Show monitor postcard telemetry',
         'policy' : 'Show monitor postcard telemetry of type policy',
         'profile': 'Show monitor postcard telemetry profile',
         'NAME': profileNameMatcher,
         }
   cliModel = ModelPostcardProfile
   handler = showMonitorTelemetryPostcardPolicyProfile
   privileged = True

#------------------------------------------------------------------------------------
# The "show monitor telemetry postcard sample policy" command
#------------------------------------------------------------------------------------
def getPolicyModel( policyName ):
   rules = []
   pmapConfig = policiesCliConfig.pmapType.pmapInput[ policyName ]
   ruleNames = pmapConfig.currCfg.classPrio.values()
   for ruleName in ruleNames:
      protocols = []
      rule = pmapConfig.currCfg.rawClassMap[ ruleName ]
      match = rule.match.items()[ 0 ][ 1 ]
      matchOpt = "ipv4" if match.option == 'matchIpAccessGroup' else "ipv6"
      for protoField in match.structuredFilter.proto.itervalues():
         protocolRange = NumericalRange( low=protoField.protocolRange.rangeStart,
                                         high=protoField.protocolRange.rangeEnd )
         assert len( protoField.port ) <= 1
         srcPorts = []
         destPorts = []
         if protoField.port:
            portField = protoField.port.values().pop()
            srcPorts = [ NumericalRange( low=sport.rangeStart,
                                         high=sport.rangeEnd )
                         for sport in portField.sport ]
            destPorts = [ NumericalRange( low=dport.rangeStart,
                                          high=dport.rangeEnd )
                          for dport in portField.dport ]
         ports = [ Port( srcPorts=srcPorts, destPorts=destPorts ) ]
         protocols.append( Protocol( protocolRange=protocolRange, ports=ports ) )

      policyActions = pmapConfig.currCfg.classAction[ ruleName ].policyAction
      actions = Actions( sample='sample' in policyActions,
                         sampleAll='sampleAll' in policyActions )
      rules.append( Rule( ruleString=ruleName, matchOption=matchOpt, matches=Matches(
         srcPrefixes=list( match.structuredFilter.source ),
         destPrefixes=list( match.structuredFilter.destination ),
         protocols=protocols ), actions=actions ) )
   return SamplePolicyModel( rules=rules )

def getPolicies( policyName=None ):
   config = {}
   if policyName:
      config[ policyName ] = getPolicyModel( policyName )
   else:
      for name in policiesCliConfig.pmapType.pmapInput:
         config[ name ] = getPolicyModel( name )
   return config

def showSamplePolicyPostcardHandler( mode, args ):
   policyName = args.get( 'POLICY_NAME' )
   policies = getPolicies( policyName )
   return SamplePoliciesModel( policies=policies )

class ShowSamplePostcardPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard sample policy [ POLICY_NAME ]'
   data = { 'monitor': CliToken.Monitor.monitorMatcherForShow,
            'telemetry': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'telemetry', helpdesc='Show monitor telemetry information' ),
               guard=postcardTelemetryGuard ),
            'postcard': 'Show monitor postcard telemetry',
            'sample': CliCommand.Node(
               matcher=CliMatcher.KeywordMatcher(
                  'sample', helpdesc='Show sampling information' ),
               guard=postcardTelemetryGuard ),
            'policy': 'Show sample policy information',
            'POLICY_NAME': samplePolicyNameMatcher,
   }
   cliModel = SamplePoliciesModel
   handler = showSamplePolicyPostcardHandler
   privileged = True

#------------------------------------------------------------------------------------
# The "show monitor telemetry postcard" command
#------------------------------------------------------------------------------------

def showPostcardTelemetryHandler( mode, args ):
   enabled = postcardTelemetryConfig.enable
   sampleRate = postcardTelemetryConfig.sampleRate
   if sampleRate == tacSampleRate.sampleRate1in16k:
      sampleRate = 16384
   elif sampleRate == tacSampleRate.sampleRate1in32k:
      sampleRate = 32768
   elif sampleRate == tacSampleRate.sampleRate1in64k:
      sampleRate = 65536
   destinationConfig = postcardTelemetryConfig.destinationConfig
   collection = None
   if not isDefaultDestination( destinationConfig ):
      collection = Collection( srcIp=destinationConfig.srcIp.v4Addr,
                               dstIp=destinationConfig.dstIp.v4Addr )
   return PostcardTelemetry( enabled=enabled, sampleRate=sampleRate,
                             collection=collection )

class ShowPostcardTelemetryPolicy( ShowCommand.ShowCliCommandClass ):
   syntax = 'show monitor telemetry postcard policy'
   data = {
         'monitor': CliToken.Monitor.monitorMatcherForShow,
         'telemetry': CliCommand.Node(
            matcher=CliMatcher.KeywordMatcher(
            'telemetry', helpdesc='Show monitor telemetry information' ),
            guard=postcardTelemetryGuard ),
         'postcard': 'Show monitor postcard telemetry',
         'policy' : 'Show monitor postcard telemetry of type policy',
         }
   cliModel = PostcardTelemetry
   handler = showPostcardTelemetryHandler
   privileged = True

# Commands guarded by Toggle. Also protecting exclusively
if Toggles.PostcardTelemetryCommonToggleLib.\
   toggleFeaturePostcardTelemetryEnabled():
   BasicCli.addShowCommandClass( ShowMonitorTelemetryPostcardPolicyProfile )
   BasicCli.addShowCommandClass( ShowSamplePostcardPolicy )
   BasicCli.addShowCommandClass( ShowPostcardTelemetryPolicy )
   BasicCli.GlobalConfigMode.addCommandClass( MonitorTelemetryPostcard )
   PostcardPolicyConfigMode.addCommandClass( DisabledCmd )
   PostcardPolicyConfigMode.addCommandClass( GreDestinationCmd )
   PostcardPolicyConfigMode.addCommandClass( ProfileConfigCmd )
   PostcardPolicyConfigMode.addCommandClass( SampleRateCmd )
   PostcardPolicyConfigMode.addCommandClass( PostcardSamplePolicyConfigCmd )
   PostcardSamplePolicyConfigMode.addCommandClass(
      PostcardSamplePolicyMatchRuleIpv4ConfigCmd )
   PostcardSamplePolicyConfigMode.addModelet( CommitAbortModelet )
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
      PostcardSamplePolicyActionsConfigCmd )
   PostcardSamplePolicyActionRuleConfigMode.addCommandClass( SampleActionCmd )
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addModelet( CommitAbortModelet )
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
      PostcardProtocolIpv4ConfigCmd )
   PostcardSamplePolicyMatchRuleIpv4ConfigMode.addCommandClass(
      PrefixIpv4SingletonCmd )
   PostcardSamplePolicyActionRuleConfigMode.addModelet( CommitAbortModelet )
   IntfPostcardTelemetryModelet.addCommandClass( PostcardTelemetryInterface )
   ProfileConfigMode.addCommandClass( SamplePolicyCmd )

#-------------------------------------------------------------------------------
# Support for default interface command.
# Clean up all the configuration set by cmds in interface config mode.
#-------------------------------------------------------------------------------
class PostcardTelemetryIntfJanitor( CliPlugin.IntfCli.IntfDependentBase ):
   def setDefault( self ):
      intfName = self.intf_.name
      if intfName in postcardTelemetryConfig.intfToPortProfile:
         del postcardTelemetryConfig.intfToPortProfile[ intfName ]

#--------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#--------------------------------------------------------------------------------
@Plugins.plugin( provides=( "PostcardTelemetryPolicyCli", ) )
def Plugin( entityManager ):
   global postcardTelemetryConfig, postcardTelemetryStatus, allVrfStatus
   global policiesCliConfig, policiesStatus, policiesStatusRequestDir
   global hwCapability
   postcardTelemetryConfig = ConfigMount.mount( entityManager,
                                                "postcardtelemetry/policyconfig",
                                                "PostcardTelemetry::PolicyConfig",
                                                "w" )
   postcardTelemetryStatus = LazyMount.mount( entityManager,
                                              "postcardtelemetry/status",
                                              "PostcardTelemetry::Status", "r" )
   allVrfStatus = LazyMount.mount( entityManager, "ip/vrf/status/global",
                                   "Ip::AllVrfStatusGlobal", "r" )
   hwCapability = LazyMount.mount( entityManager, "postcardtelemetry/hwCapability",
                                   "PostcardTelemetry::HwCapability", "r" )
   samplePoliciesConfigPath = 'postcardtelemetry/samplePolicies/input/cli'
   samplePoliciesCliConfigType = 'TrafficPolicy::TrafficPolicyConfig'
   policiesStatusPath = 'cell/%d/postcardtelemetry/samplePolicies/status' % \
                        Cell.cellId()
   policiesStatusType = 'Tac::Dir'
   policiesStatusRequestDirPath = 'postcardtelemetry/samplePolicies/statusRequest'
   policiesStatusRequestDirType = 'PolicyMap::PolicyMapStatusRequestDir'

   mountGroup = entityManager.mountGroup()
   policiesStatus = mountGroup.mount( policiesStatusPath, policiesStatusType, 'ri' )
   mountGroup.close( callback=None, blocking=False )

   policiesCliConfig = ConfigMount.mount( entityManager, samplePoliciesConfigPath,
                                          samplePoliciesCliConfigType, 'wi' )
   policiesStatusRequestDir = LazyMount.mount( entityManager,
                                               policiesStatusRequestDirPath,
                                               policiesStatusRequestDirType, 'w' )

   CliPlugin.IntfCli.Intf.registerDependentClass( PostcardTelemetryIntfJanitor )
