#!/usr/bin/env python
# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.PortSecCliHelpers as PortSecCliHelpers
import CliPlugin.SubIntfCli as SubIntfCli
import CliPlugin.VlanCli as VlanCli
from CliPlugin.IntfCli import Intf
from CliToken.Clear import clearKwNode
import CliToken.Mac

subIntfLimitGuard = SubIntfCli.theL2SubintfMacLearningDisableSupported
globalMacAddressConfigGuard = PortSecCliHelpers.guardGlobalMacAddressConfig
matcherMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Maximum number of MAC addresses allowed on the interface' )
matcherMaximum = CliMatcher.KeywordMatcher( 'maximum',
      helpdesc='Maximum number of MAC addresses allowed on the interface' )
matcherMaximumDisabled = CliMatcher.KeywordMatcher( 'disabled',
      helpdesc='Disable port level check for port security (only in shutdown mode)' )
matcherPortSecurity = CliMatcher.KeywordMatcher( 'port-security',
      helpdesc='Configure MAC-address-based port security' )
matcherProtect = CliMatcher.KeywordMatcher( 'protect',
      helpdesc='Configure port security in protect mode' )
matcherSwitchport = CliMatcher.KeywordMatcher( 'switchport',
      helpdesc='Set switching mode characteristics' )
matcherViolation = CliMatcher.KeywordMatcher( 'violation',
      helpdesc='Configure violation mode (shutdown or protect)' )
matcherLimit = CliMatcher.KeywordMatcher( 'limit',
      helpdesc='Configure MAC-address-based port security on this sub-interface' )
nodeAddress = CliCommand.Node( matcher=SubIntfCli.matcherAddressKw,
      guard=subIntfLimitGuard )
matcherGlobalMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Configure global settings for Port Security MAC addresses' )
matcherAging = CliMatcher.KeywordMatcher( 'aging',
      helpdesc='Enable aging for Port Security MACs' )
matcherMoveable = CliMatcher.KeywordMatcher( 'moveable',
      helpdesc='Enable MAC moves for Port Security MACs' )
nodeAging = CliCommand.Node( matcher=matcherAging,
      guard=globalMacAddressConfigGuard )
nodeMoveable = CliCommand.Node( matcher=matcherMoveable,
      guard=globalMacAddressConfigGuard )
matcherPersistence = CliMatcher.KeywordMatcher( 'persistence',
      helpdesc="Configure Persistent Port Security feature" )
matcherPersistDisabled = CliMatcher.KeywordMatcher( 'disabled',
      helpdesc="Disable Persistent Port Security feature" )
nodeData = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'data',
      helpdesc='Allow MAC move for the data traffic'), hidden=True )
nodePhone = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'phone',
      helpdesc='Allow MAC move for the voice traffic'), hidden=True )
matcherVlan = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Configure VLAN-Based port security' )
matcherDefaultVlan = CliMatcher.KeywordMatcher( 'default',
      helpdesc='Select all VLAN on this interface' )
matcherVlanMacAddress = CliMatcher.KeywordMatcher( 'mac-address',
      helpdesc='Maximum number of MAC addresses allowed on the VLAN' )
matcherVlanMaximum = CliMatcher.KeywordMatcher( 'maximum',
      helpdesc='Maximum number of MAC addresses allowed on the VLAN' )
matcherVlanBased = CliMatcher.KeywordMatcher( 'vlan',
      helpdesc='Configure port security as only VLAN based' )

#--------------------------------------------------------------------------------
# [ no | default ] switchport port-security
#--------------------------------------------------------------------------------
class SwitchportPortSecurityCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': matcherSwitchport,
      'port-security': matcherPortSecurity,
   }

   @staticmethod
   def handler( mode, args ):
      PortSecCliHelpers.doEnDisPortSecurity( mode )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      PortSecCliHelpers.doEnDisPortSecurity( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( SwitchportPortSecurityCmd )

#--------------------------------------------------------------------------------
# switchport port-security mac-address maximum ( MAXIMUM | disabled )
#--------------------------------------------------------------------------------
baseMaximumCmdSyntax = 'switchport port-security mac-address maximum'

class MacAddressMaximumCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security ( mac-address maximum ) | ' \
            'deprecated-maximum ( MAXIMUM | disabled )'
   noOrDefaultSyntax = 'switchport port-security mac-address maximum ...'

   data = {
      'switchport': matcherSwitchport,
      'port-security': matcherPortSecurity,
      'mac-address': matcherMacAddress,
      'maximum': matcherMaximum,
      'deprecated-maximum': CliCommand.Node(
         matcher=matcherMaximum,
         deprecatedByCmd='switchport port-security mac-address maximum' ),
      'MAXIMUM': CliMatcher.DynamicIntegerMatcher( 
         PortSecCliHelpers.maximumRange,
         helpdesc='MAC address limit' ),
      'disabled': matcherMaximumDisabled,
   }

   handler = PortSecCliHelpers.doSetMaximum
   noOrDefaultHandler = handler

VlanCli.SwitchportModelet.addCommandClass( MacAddressMaximumCmd )

#--------------------------------------------------------------------------------
# switchport port-security violation protect
#--------------------------------------------------------------------------------
class ViolationProtectCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security violation protect'
   data = {
      'switchport': matcherSwitchport,
      'port-security': matcherPortSecurity,
      'violation': matcherViolation,
      'protect': matcherProtect,
   }

   @staticmethod
   def handler( mode, args ):
      PortSecCliHelpers.doEnDisPortSecurity( mode, portSecMode='protect' )

VlanCli.SwitchportModelet.addCommandClass( ViolationProtectCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport port-security violation protect log
#--------------------------------------------------------------------------------
class ViolationProtectLogCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security violation protect log'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': matcherSwitchport,
      'port-security': matcherPortSecurity,
      'violation': matcherViolation,
      'protect': matcherProtect,
      'log': 'Log new addresses seen after limit is reached in protect mode',
   }

   @staticmethod
   def handler( mode, args ):
      PortSecCliHelpers.doEnDisPortSecurityLogging( mode )

   @staticmethod
   def noOrDefaultHandler ( mode, args ):
      PortSecCliHelpers.doEnDisPortSecurityLogging( mode, no=True )

VlanCli.SwitchportModelet.addCommandClass( ViolationProtectLogCmd )

#--------------------------------------------------------------------------------
# switchport port-security violation shutdown
#--------------------------------------------------------------------------------
class ViolationShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security violation shutdown'
   data = {
      'switchport': matcherSwitchport,
      'port-security': matcherPortSecurity,
      'violation': matcherViolation,
      'shutdown': 'Configure port security in shutdown mode',
   }
   @staticmethod
   def handler( mode, args ):
      PortSecCliHelpers.doEnDisPortSecurity( mode )

VlanCli.SwitchportModelet.addCommandClass( ViolationShutdownCmd )

#--------------------------------------------------------------------------------
# clear port-security interface [ INTF ]
#--------------------------------------------------------------------------------
class ClearPortSecurityInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'clear port-security interface [ INTF ]'
   data = {
      'clear': clearKwNode,
      'port-security': 'Clear MAC-address-based port security',
      'interface': 'Name of the interface',
      'INTF': Intf.rangeMatcher,
   }
   handler = PortSecCliHelpers.clearPortSecurity

BasicCliModes.EnableMode.addCommandClass( ClearPortSecurityInterfaceCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac address limit
#--------------------------------------------------------------------------------
subIntfBaseLimitCmdSyntax = 'mac address limit'
class SubIntfLimitCmd( CliCommand.CliCommandClass ):
   syntax = subIntfBaseLimitCmdSyntax
   noOrDefaultSyntax = syntax
   data = {
      'mac': CliToken.Mac.macMatcherForConfigIf,
      'address': nodeAddress,
      'limit': matcherLimit,
   }
   handler = PortSecCliHelpers.doEnDisMacAddressLimitMaximum
   noOrDefaultHandler = handler

SubIntfCli.SubIntfVlanModelet.addCommandClass( SubIntfLimitCmd )

#--------------------------------------------------------------------------------
# mac address limit MAXIMUM
#--------------------------------------------------------------------------------
class SubIntfMacAddressMaximumCmd( CliCommand.CliCommandClass ):
   syntax = subIntfBaseLimitCmdSyntax + ' MAXIMUM'

   data = {
      'mac': CliToken.Mac.macMatcherForConfigIf,
      'address': nodeAddress,
      'limit': matcherLimit,
      'MAXIMUM': CliMatcher.IntegerMatcher( 1, 1000,
                                            helpdesc='MAC address limit' ),
   }
   handler = PortSecCliHelpers.doEnDisMacAddressLimitMaximum

SubIntfCli.SubIntfVlanModelet.addCommandClass( SubIntfMacAddressMaximumCmd )

#--------------------------------------------------------------------------------
# mac address limit violation protect
#--------------------------------------------------------------------------------
class SubIntfViolationProtectCmd( CliCommand.CliCommandClass ):
   syntax = subIntfBaseLimitCmdSyntax + ' violation protect'
   data = {
      'mac': CliToken.Mac.macMatcherForConfigIf,
      'address': nodeAddress,
      'limit': matcherLimit,
      'violation': matcherViolation,
      'protect': matcherProtect,
   }
   handler = PortSecCliHelpers.doSetMacAddressLimitViolationMode

SubIntfCli.SubIntfVlanModelet.addCommandClass( SubIntfViolationProtectCmd )

#--------------------------------------------------------------------------------
# [ no | default ] mac address limit violation protect log
# is not implemented as we do not support for logging on L2 subinterfaces
# refer to BUG420319 for details
#--------------------------------------------------------------------------------


#--------------------------------------------------------------------------------
# mac address limit violation shutdown
#--------------------------------------------------------------------------------
class SubIntfViolationShutdownCmd( CliCommand.CliCommandClass ):
   syntax = subIntfBaseLimitCmdSyntax + ' violation shutdown'
   data = {
      'mac': CliToken.Mac.macMatcherForConfigIf,
      'address': nodeAddress,
      'limit':matcherLimit,
      'violation': matcherViolation,
      'shutdown': 'Configure port security in shutdown mode',
   }
   handler = PortSecCliHelpers.doSetMacAddressLimitViolationMode

SubIntfCli.SubIntfVlanModelet.addCommandClass( SubIntfViolationShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport port-security mac-address aging
#--------------------------------------------------------------------------------
class PortSecurityMacAddressAgingCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security mac-address aging'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': VlanCli.switchportMatcher,
      'port-security': matcherPortSecurity,
      'mac-address': matcherGlobalMacAddress,
      'aging': nodeAging,
   }
   handler = PortSecCliHelpers.doEnAging
   noOrDefaultHandler = PortSecCliHelpers.doDisAging

BasicCliModes.GlobalConfigMode.addCommandClass( PortSecurityMacAddressAgingCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport port-security mac-address moveable [ data | phone ]
#--------------------------------------------------------------------------------
class PortSecurityMacAddressMoveableCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security mac-address moveable [ data | phone ]'
   noOrDefaultSyntax = 'switchport port-security mac-address moveable ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'port-security': matcherPortSecurity,
      'mac-address': matcherGlobalMacAddress,
      'moveable': nodeMoveable,
      'data' : nodeData,
      'phone' : nodePhone,
   }
   handler = PortSecCliHelpers.doEnMoves
   noOrDefaultHandler = PortSecCliHelpers.doDisMoves

BasicCliModes.GlobalConfigMode.addCommandClass( PortSecurityMacAddressMoveableCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport port-security persistence disabled
#--------------------------------------------------------------------------------
class PortSecurityPersistenceDisabledCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security persistence disabled'
   noOrDefaultSyntax = syntax
   data = {
      'switchport': VlanCli.switchportMatcher,
      'port-security': matcherPortSecurity,
      'persistence': matcherPersistence,
      'disabled': matcherPersistDisabled,
   }
   handler = PortSecCliHelpers.doDisPersistence
   noOrDefaultHandler = PortSecCliHelpers.doEnPersistence

BasicCliModes.GlobalConfigMode.addCommandClass( PortSecurityPersistenceDisabledCmd )

#--------------------------------------------------------------------------------
# [ no | default ] switchport port-security vlan ( default | VLAN_SET )
# mac-address maximum MAXIMUM
#--------------------------------------------------------------------------------
class VlanBasedPortSecurityConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'switchport port-security vlan ( default | VLAN_SET )' \
            'mac-address maximum MAXIMUM'
   noOrDefaultSyntax = 'switchport port-security vlan ( default | VLAN_SET )' \
                       'mac-address maximum ...'
   data = {
      'switchport': VlanCli.switchportMatcher,
      'port-security': matcherPortSecurity,
      'vlan': matcherVlan,
      'default': matcherDefaultVlan,
      'VLAN_SET': VlanCli.vlanSetMatcher,
      'mac-address': matcherVlanMacAddress,
      'maximum': matcherVlanMaximum,
      'MAXIMUM': CliMatcher.IntegerMatcher( 0, 1000,
                                            helpdesc='MAC address limit' ),
   }

   handler = PortSecCliHelpers.doEnDisVlanBasedPortSec
   noOrDefaultHandler = handler

VlanCli.SwitchportModelet.addCommandClass( VlanBasedPortSecurityConfigCmd )
