# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function
import Ark
import Arnet
import CliModel, IntfModels, ArnetModel
from CliPlugin import IntfCli
import Ethernet
import Tac
import TableOutput

# Methods of handling security violations
securityViolationModes = ( "shutdown", "protect", "none" )
ethStr = Ethernet.convertMacAddrCanonicalToDisplay
timeStr = Ark.timestampToStr

# NOTE The original submit history for GeneralPortSecurityStatistics,
# PortSecurityAddresses, and PortSecurityInterfaces classes can be found
# in PortSecCliHelpers.py

class GeneralPortSecurityStatistics( CliModel.Model ):
   class PortStatistic( CliModel.Model ):
      maxSecureAddr = CliModel.Int( help="Maximum number of learned addresses" )
      currentAddr = CliModel.Int( help="Current number of address learned" )
      numberOfViolations = CliModel.Int(
            help="Number of security violations that have occurred" )
      securityAction = CliModel.Enum( values=securityViolationModes,
            help="Action to take to protect the ports security" )

   portStatistics = CliModel.Dict( keyType=IntfModels.Interface,
                                   valueType=PortStatistic,
                                   help="Port Security Statistics" )
   totalAddresses = CliModel.Int( help="Total number of addresses in the system" )
   secureAddressMoves = CliModel.Bool( help="A secured address can move" )
   secureAddressAging = CliModel.Bool( help="A secured address can age out" )
   persistence = CliModel.Bool( help="Port security is persistent" )

   def render( self ):
      separater = 76 * '-'
      enDis = ( 'disabled', 'enabled' )
      moves = enDis[ int( self.secureAddressMoves ) ]
      aging = enDis[ int( self.secureAddressAging ) ]
      persist = enDis[ int( self.persistence ) ]

      print( "Secure address moves:", moves )
      print( "Secure address aging:", aging )
      print( "Secure address reboot persistence:", persist )
      print( "Secure address link down persistence:", persist )
      print( "Secure Port  MaxSecureAddr  CurrentAddr  "
             "SecurityViolation  Security Action")
      print( "                (Count)       (Count)          (Count)" )
      print( separater )
      fmt = " %-10s        %7d   %10d          %9d          %s"
      for iname in Arnet.sortIntf( self.portStatistics ):
         portStatistic = self.portStatistics[ iname ]
         print( fmt % ( IntfCli.Intf.getShortname( iname ),
                        portStatistic.maxSecureAddr,
                        portStatistic.currentAddr,
                        portStatistic.numberOfViolations,
                        portStatistic.securityAction.capitalize() ) )
      print( separater )
      print( "Total Addresses in System:", self.totalAddresses )

class PortSecurityAddresses( CliModel.Model ):
   class Address( CliModel.Model ):
      interface = IntfModels.Interface(
                           help="Interface that the address is learned on" )
      macAddress = ArnetModel.MacAddress( help="MAC address of host" )
      vlan = CliModel.Int( help="VLAN host is learned in" )
      entryType = CliModel.Enum( values=( "secureConfigured", "secureDynamic" ),
                            help="Type of secured host" )
      remainingAge = CliModel.Int( optional=True,
                            help="Remaining minutes before interface ages out" )
   addresses = CliModel.List( valueType=Address,
      help="Statistics for MAC addresses under port-security for this criteria" )
   totalAddresses = CliModel.Int(
      help="Total number of MAC address under port-security for this criteria" )

   def render( self ):
      def intfVlanMacKey( host ):
         return ( Arnet.intfNameKey( host.interface.stringValue ),
                  host.vlan, host.macAddress )
      fmt = '%4d    %-14s    %-21s    %-5s   %s'
      print( '          Secure Mac Address Table' )
      print( '---------------------------------------------------------------' )
      print( 'Vlan    Mac Address       Type' + ' ' * 21 + 'Ports   Remaining Age' )
      print( ' ' * 62 + '(mins)' )
      print( '----    -----------       ----' + ' ' * 21 + '-----   -------------' )
      for host in sorted( self.addresses, key=intfVlanMacKey ):
         shortIntf = IntfCli.Intf.getShortname( host.interface )
         entryType = host.entryType[0].capitalize() + host.entryType[1:]
         remainingAge = host.remainingAge or "N/A"
         print( fmt % ( host.vlan,
                        ethStr( str( host.macAddress ) ),
                        entryType,
                        shortIntf,
                        remainingAge ) )
      print( '-' * 72 )
      print( 'Total Mac Addresses for this criterion:', self.totalAddresses )

class PortSecurityInterfaces( CliModel.Model ):
   class Interface( CliModel.Model ):
      class InterestingAddress( CliModel.Model ):
         macAddress = ArnetModel.MacAddress( help="MAC Address" )
         vlan = CliModel.Int( help="VLAN the MAC address is in" )
         time = CliModel.Float( help="UTC representation of when the "
                                     "event occurred" )

      portSecurityEnabled = CliModel.Bool( help="Is port security enabled" )
      portStatus = CliModel.Enum( values=( "secure-down",
                                           "secure-shutdown",
                                           "secure-protected",
                                           "secure-up" ),
                                  help="The status of the interface "
                                       "under port security" )
      violationMode = CliModel.Enum( values=( securityViolationModes ),
                                     help="Action port security will take to "
                                           "stop a violation" )
      portMaxEnabled = CliModel.Bool( help="Port level limit is enabled",
                                      optional=True )
      maxMacAddresses = CliModel.Int( help="Maximum MAC addresses allowed "
                                           "on the interface" )
      agingTime = CliModel.Int( help="Time before addresses age out (in minutes)" )
      agingType = CliModel.Enum( values=( "inactivity", ),
                                 help="Criteria for an address aging" )
      secureStaticAddressAging = CliModel.Enum(
                                    values=( "disabled", ),
                                    help="Are Secure Static addresses "
                                         "allowed to age (currently unused)" )
      secureAddressMoves = CliModel.Bool( help="A secured address can move" )
      secureAddressAging = CliModel.Bool( help="A secured address can age out" )
      persistence = CliModel.Bool( help="Port security is persistent" )
      totalMacAddresses = CliModel.Int( help="Number of addresses learned "
                                             "on the interface",
                                        optional=True )
      configuredMacAddresses = CliModel.Int( help="Number of configured MAC "
                                              "addresses on the interface",
                                         optional=True )
      addressChanges = CliModel.Int( help="Number of learn/move/age events on "
                                          "the interface",
                                     optional=True )
      lastChangeDetails = CliModel.Submodel( valueType=InterestingAddress,
                                      help="Last changed source address information",
                                      optional=True )
      lastViolation = CliModel.Submodel( valueType=InterestingAddress,
                                         help="Last violating address information",
                                         optional=True )
      securityViolationCount = CliModel.Int( help="Number of security "
                                                  "violations on the interface",
                                             optional=True )
      logAddrsAfterLimit = CliModel.Enum( values=( "enabled", "disabled" ),
                                          help="Log new addresses seen past limit "
                                               "in protect mode" )
      allowedAddresses = CliModel.List( help="Which Address:VlanId are allowed "
                                                   "in protect mode",
                                        valueType=str,
                                        optional=True )
   interfaces = CliModel.Dict( keyType=IntfModels.Interface, valueType=Interface,
         help="Per interface breakdown of port-security information" )

   def render( self ):
      sortedIntfs = sorted( self.interfaces, key=Arnet.intfNameKey )
      for intfName in sortedIntfs:
         interface = self.interfaces[ intfName ]
         attributes = []
         enDis = ( 'Disabled', 'Enabled' )
         portSecurityEnabled = enDis[ int( interface.portSecurityEnabled ) ]
         moves = enDis[ int( interface.secureAddressMoves ) ]
         aging = enDis[ int( interface.secureAddressAging ) ]
         persist = enDis[ int( interface.persistence ) ]
         attributes.append( ( "Interface", intfName ) )
         attributes.append( ( "Port Security", portSecurityEnabled ) )
         attributes.append( ( "Port Status", interface.portStatus.capitalize() ) )
         attributes.append( ( "Violation Mode",
                              interface.violationMode.capitalize() ) )
         if interface.violationMode == 'shutdown':
            attributes.append( ( "Port Level Maximum Enabled",
                                 interface.portMaxEnabled ) )
         attributes.append( ( "Maximum MAC Addresses",
                              str( interface.maxMacAddresses ) ) )
         attributes.append( ( "Aging Time", "%d mins" % interface.agingTime ) )
         attributes.append( ( "Aging Type", interface.agingType.capitalize() ) )
         attributes.append( ( "SecureStatic Address Aging",
                              interface.secureStaticAddressAging.capitalize() ) )
         attributes.append( ( "Secure Address Moves", moves ) )
         attributes.append( ( "Secure Address Aging", aging ) ) 
         attributes.append( ( "Secure Address Reboot Persistence", persist ) )
         attributes.append( ( "Secure Address Link Down Persistence", persist ) )
         if interface.totalMacAddresses is not None:
            attributes.append( ( "Total MAC Addresses",
                                 str( interface.totalMacAddresses ) ) )
            attributes.append( ( "Configured MAC Addresses",
                                 str( interface.configuredMacAddresses ) ) )
            attributes.append( ( "Learn/Move/Age Events",
                                 str( interface.addressChanges ) ) )
            if interface.lastChangeDetails is not None:
               tacTime = interface.lastChangeDetails.time - Tac.utcNow() + Tac.now()
               lastSourceAddr = "%s:%s" % \
                     ( ethStr( str( interface.lastChangeDetails.macAddress ) ),
                       interface.lastChangeDetails.vlan )
               attributes.append( ( "Last Source Address:Vlan", lastSourceAddr ) )
               attributes.append( ( "Last Address Change Time",
                                    timeStr( tacTime ) ) )
            attributes.append( ( "Security Violation Count",
                                 str( interface.securityViolationCount ) ) )
            if interface.lastViolation is not None:
               tacTime = interface.lastViolation.time - Tac.utcNow() + Tac.now()
               lastSourceAddr = "%s:%s" % \
                     ( ethStr( str( interface.lastViolation.macAddress ) ),
                       interface.lastViolation.vlan )
               attributes.append( ( "Last Violation Time",
                                    timeStr( tacTime ) ) )
               attributes.append( ( "Last Violating Address:Vlan", lastSourceAddr ) )

         width = max( len( x[ 0 ] ) for x in attributes )

         if interface.violationMode == 'protect':
            logAddrsAfterLim = "Log addresses after limit"
            allowedAddrsAttribute = "Allowed Addresses:Vlan"
            width = max( length for length in [ width, len( logAddrsAfterLim ),
                                                len( allowedAddrsAttribute ) ] )
            attributes.append( ( logAddrsAfterLim,
                                 interface.logAddrsAfterLimit.capitalize() ) )
            if interface.allowedAddresses:
               allowedAddrs = '%s' % interface.allowedAddresses.pop( 0 )
               for addr in interface.allowedAddresses:
                  allowedAddrs += '\n%-*s   %s' % ( width, ' ', addr )

               attributes.append( ( allowedAddrsAttribute,
                                    allowedAddrs ) )

         for field in attributes:
            print( "%-*s : %s" % ( width, field[0], field[1] ) )
         print()

class GeneralPortSecurityVlanStatistics( CliModel.Model ):
   class VlanDict( CliModel.Model ):
      class VlanStatistic( CliModel.Model ):
         maxAddrs = CliModel.Int( help="Maximum number of learned addresses" )
         numAddrs = CliModel.Int( help="Current number of learned addresses" )
         numViolations = CliModel.Int(
               help="Number of security violations that have occurred on the VLAN" )
         action = CliModel.Enum( values=securityViolationModes,
               help="Action to take to protect the ports security" )
      vlans = CliModel.Dict( keyType=int, valueType=VlanStatistic,
            help="Per vlan breakdown of vlan based port-security information" )

   interfaces = CliModel.Dict( keyType=IntfModels.Interface, valueType=VlanDict,
         help="Per interface breakdown of vlan based port-security information" )
   totalAddresses = CliModel.Int( help="Total number of addresses in the system" )

   def render( self ):
      tableHeadings = ( ( "Secure Port", ( "", ) ), ( "Secure VLAN", ( "", ) ),
                        ( "MaxSecureAddr", ( "(Count)", ) ),
                        ( "CurrentAddr", ( "(Count)", ) ),
                        ( "SecurityViolation", ( "(Count)", ) ),
                        ( "Security Action", ( "", ) ), )
      table = TableOutput.createTable( tableHeadings, tableWidth=113 )
      left = TableOutput.Format( justify="left", minWidth=10 )
      center = TableOutput.Format( justify="center", minWidth=10 )
      table.formatColumns( left, center, center, center, center, center )
      for intfName in sorted( self.interfaces, key=Arnet.intfNameKey ):
         for vlanId in sorted( self.interfaces[ intfName ].vlans ):
            statistic = self.interfaces[ intfName ].vlans[ vlanId ]
            table.newRow( IntfCli.Intf.getShortname( intfName ),
                          vlanId,
                          statistic.maxAddrs,
                          statistic.numAddrs,
                          statistic.numViolations,
                          statistic.action )
      print( table.output() )
      print( "-" * 113 )
      print( "Total Addresses in System:", self.totalAddresses )
