#!/usr/bin/env python

# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import weakref

import AclLib
import AclCliLib
import BasicCli
import CliCommand
import CliParser
import CliPlugin.AclCli as AclCli
import CliPlugin.AclCliRules as AclCliRules
import CliPlugin.IraIpCli as IraIpCli
import CliMatcher
from CliMode.PolicyMap import PolicyMapModeBase, ClassMapModeBase
from CliMode.PolicyMap import PolicyMapClassModeBase
import CliSession
from PolicyMap import matchOptionToEnum
import Tracing
import Tac

__defaultTraceHandle__ = Tracing.Handle( 'PolicyMapCli' )
t0 = Tracing.trace0
t1 = Tracing.trace1
t7 = Tracing.trace8
t8 = Tracing.trace8
t9 = Tracing.trace9

# max rule sequence number (32-bit)
MAX_SEQ = 0xFFFFFFFF # 32-bit integer
defaultSeqStart = 10
defaultSeqInc = 10

matcherClass = CliMatcher.KeywordMatcher( 'class', helpdesc='policy criteria' )
matcherMatch = CliMatcher.KeywordMatcher( 'match',
      helpdesc='Match the access rule specified' )
matcherType = CliMatcher.KeywordMatcher( 'type', helpdesc='Specify type' )
matcherAbort = CliMatcher.KeywordMatcher( 'abort', helpdesc='Abandon all changes' )
matcherSeqNum = CliMatcher.IntegerMatcher( 1, MAX_SEQ,
      helpdesc='Index in the sequence' )
matcherActionSet = CliMatcher.KeywordMatcher( 'set', helpdesc='Set values' )
matcherAccessGroup = CliMatcher.KeywordMatcher( 'access-group',
      helpdesc='Match with given access group' )
matcherIpAccessGroupType = CliMatcher.KeywordMatcher( 'ip',
      helpdesc='Specify Ipv4 Access-groups' )
matcherDrop = CliMatcher.KeywordMatcher( 'drop', helpdesc='Drop packets' )
matcherResequence = CliMatcher.KeywordMatcher( 'resequence',
      helpdesc='Resequence the list' )
matcherStartSequence = CliMatcher.IntegerMatcher( 1, AclLib.MAX_SEQ,
      helpdesc='Starting sequence number (default {})'.format(
         defaultSeqStart ) )
matcherIncSequence = CliMatcher.IntegerMatcher( 1, AclLib.MAX_SEQ,
      helpdesc='Step to increment the sequence number (default {})'.format(
         defaultSeqInc ) )

UniqueId = Tac.Type( "Ark::UniqueId" )
tacMatchOption = Tac.Type( 'PolicyMap::ClassMapMatchOption' )
matchIpAccessGroup = tacMatchOption.matchIpAccessGroup
matchIpv6AccessGroup = tacMatchOption.matchIpv6AccessGroup
matchMacAccessGroup = tacMatchOption.matchMacAccessGroup
matchMplsAccessGroup = tacMatchOption.matchMplsAccessGroup

tacAclIpFilterType = Tac.Type( 'Acl::IpFilter' )
tacAclIp6FilterType = Tac.Type( 'Acl::Ip6Filter' )
tacAclMacFilterType = Tac.Type( 'Acl::MacFilter' )
tacAclIpRuleConfigType = Tac.Type( 'Acl::IpRuleConfig' )
tacAclIp6RuleConfigType = Tac.Type( 'Acl::Ip6RuleConfig' )
tacAclMacRuleConfigType = Tac.Type( 'Acl::MacRuleConfig' )

CHANGED = 0
IDENTICAL = 1
RESEQUENCED = 2

def matchOptionOtherToEnum( matchOption ):
   otherOptions = [ tacMatchOption.matchIpAccessGroup,
                    tacMatchOption.matchIpv6AccessGroup,
                    tacMatchOption.matchMacAccessGroup ]
   if matchOption in otherOptions:
      otherOptions.remove( matchOption )
      return otherOptions
   else:
      return None

class PolicyOpStatus( object ):
   def __init__( self, statusRequestDir, status ):
      self.policyTypeStatus = status
      self.statusRequestDir = statusRequestDir

   @staticmethod
   def policyUpdateType( updateType, name ):
      assert updateType in ( 'acl', 'policy', 'class', 'intf' )
      reasonType = 'updateReason' + updateType.capitalize() + 'Change'
      updateType = Tac.Value( "PolicyMap::PolicyUpdateType",
                              reasonType, name )
      return updateType

   def getResult( self, updateType, name ):
      # Add our status request
      statusId = UniqueId()
      updateType = self.policyUpdateType( updateType, name )
      self.statusRequestDir.statusRequest[ statusId ] = updateType

      try:
         # Wait for the reply to appear.
         Tac.waitFor( lambda: statusId in self.policyTypeStatus.result,
                      maxDelay=0.1, sleep='PBR_COHAB_TEST' not in os.environ,
                      timeout=30.0,
                      description='status to appear for (%s)' % \
                              self.policyTypeStatus.name )
         result = self.policyTypeStatus.result[ statusId ]
      except ( Tac.Timeout, KeyboardInterrupt ):
         result = None

      # Remove the status request.
      del self.statusRequestDir.statusRequest[ statusId ]
      return result

   def checkResult( self, updateType, name, expected='operationSucceeded' ):
      ''' returns  ( err, result ) '''
      result = self.getResult( updateType, name )
      if not result:
         return None
      return ( result.progress == expected or \
                  result.progress == 'operationIgnored' ), result

class PolicyOpChkr( object ):
   waitStates = [ 'operationNotHandled', 'operationInProgress' ]
   failStates =  [ 'operationFailed' ]
   passStates = [ 'operationIgnored', 'operationSucceeded' ]

   def __init__( self, statusRequestDir, statusDir ):
      self.statusRequestDir = statusRequestDir
      self.statusDir = statusDir

   def verify( self, updateType, name ):
      ''' returns  err, result '''
      if 'SKIP_POLICYMAP_STATUS_CHECK' in os.environ:
         return True, None
      opResult = { 'success' : True, 'result' : None }
      for statkey in self.statusDir.keys():
         pmapTypeStatus = self.statusDir.get( statkey )
         if not pmapTypeStatus or not pmapTypeStatus.initialized:
            continue
         opStatus = PolicyOpStatus( self.statusRequestDir, pmapTypeStatus )

         def chkOpResult( opStatus, updateType, name, opRes ):
            rc = False
            result = opStatus.getResult( updateType, name )
            t0( 'update %s name %s result %s' % \
                   ( updateType, name, result ))
            if result:
               opRes[ 'result' ] = result
               opRes[ 'success' ] = result.progress in self.passStates
               rc = result.progress not in self.waitStates
            return rc

         try:
            Tac.waitFor( lambda: chkOpResult( opStatus, updateType,
                                              name, opResult ),
                         timeout=120.0, sleep='PBR_COHAB_TEST' not in os.environ,
                         description='policy status for (%s)' % \
                                 pmapTypeStatus.name )
         except ( Tac.Timeout, KeyboardInterrupt ):
            opResult[ 'success' ] = False

         if not opResult[ 'success' ]:
            break

      return opResult[ 'success' ], opResult[ 'result' ]

class PolicyMapClassContext( object ):
   def __init__( self, pmap, cmapName, mapType, pmapContext ):
      self.pmapContext = pmapContext
      self.map_ = pmap
      self.pmapName_ = pmap.policyName
      self.cmapName_ = cmapName
      self.mapType_ = mapType
      self.policyRuleAction = None
      self.seqnum = 0

   def copyEditPCmap( self, cmapName, seqnum ):
      t0( "copyEditPCmap" )
      self.seqnum = seqnum
      self.policyRuleAction = PolicyClassRuleAction( self.pmapContext, 
                                                     cmapName, seqnum )
      classAction = self.map_.classAction[ cmapName ]
      for actType, action in classAction.policyAction.iteritems():
         self.policyRuleAction.originalActions.add( action )
         self.policyRuleAction.addAction( actType, action )

   def newEditPCmap( self, cmapName, seqnum ):
      t0( "newEditPCmap" )
      self.seqnum = seqnum
      self.policyRuleAction = PolicyClassRuleAction( self.pmapContext, 
                                                     cmapName, seqnum )

   def pmapName( self ):
      return self.pmapName_

   def policyClassName( self ):
      return self.cmapName_

   def mapType( self ):
      return self.mapType_

   def pmap( self ):
      return self.map_

   def commit( self ):
      self.pmapContext.addRuleCommon( self.seqnum, self.policyRuleAction )

class PolicyMapClassMode( PolicyMapClassModeBase, BasicCli.ConfigModeBase ):
   name = 'Policy Map Class Configuration'
   modeParseTree = CliParser.ModeParseTree()
   autoConfigSessionAllowed = False

   def __init__( self, parent, session, context, mapTypeStr ):
      self.pmapClassContext = context
      self.pmapName = context.pmapName()
      self.cmapName = context.policyClassName()
      self.mapType_ = context.mapType()
      self.mapStr_ = mapTypeStr
      param = ( self.mapType_, self.mapStr_, self.pmapName, self.cmapName )
      PolicyMapClassModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'PolicyMapClassMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.pmapClassContext.policyRuleAction.abort( )
      self.pmapClassContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      t0( "PolicyMapClassMode commitContext" )
      if self.pmapClassContext is None:
         t0( 'commitContext is empty' )
         return
      context = self.pmapClassContext
      self.pmapClassContext = None
      context.commit()

   @staticmethod
   def showPolicyMapClass( mode, pmapName, cmapName, mapType ):
      return


class PolicyRuleActionBase( object ):
   keyTag_ = 'PolicyRuleActionBaseKeyTag_'

   def __init__( self, pmapContext, cmapName, seqnum=0, rule=None, actions=None  ):
      self.pmapContext = pmapContext
      self.cmapName = cmapName
      self.seqnum = seqnum
      self.rule = rule
      self.policyActions = actions if actions else {}
      self.removedActions = set()
      self.addedActions = set()
      self.originalActions = set()

   @classmethod
   def keyTag( cls ):
      return cls.keyTag_

   def actions( self ):
      return self.policyActions

   def addAction( self, actionType, action ):
      self.policyActions[ actionType ] = action
      self.addedActions.add( action )

   def delAction( self, actionType ):
      action = self.policyActions.pop( actionType, None )
      if action:
         self.removedActions.add( action )
      return action

   def abort( self ):
      # When aborting, delete all actions that were created, except for the original
      # actions which are still applied to the policy-class.
      toRemove = self.addedActions - self.originalActions
      for action in toRemove:
         self.pmapContext.removeAction( action )
         
   def addToPmap( self, pmap, seqnum ):
      classAction = pmap.classAction.get( self.cmapName, None )
      if classAction is not None:
         del pmap.classAction[ self.cmapName ]
      # Delete all the actions that are no longer present.
      for oldAction in self.removedActions:
         self.pmapContext.removeAction( oldAction )

      pmap.classPrio[ seqnum ] = self.cmapName
      classAction = pmap.newClassAction( self.cmapName )
      for action in self.actions().itervalues():
         classAction.policyAction.addMember( action )

class PolicyClassRuleAction( PolicyRuleActionBase ):
   keyTag_ = 'classMap'

   def __init__( self, pmapContext, cmapName, seqnum=0 ):
      PolicyRuleActionBase.__init__( self, pmapContext, cmapName, seqnum, cmapName )

   def key( self ):
      return self.rule

class PolicyRawRuleActionBase( PolicyRuleActionBase ):
   def __init__( self, pmapContext, cmapName, ipRuleConfig, matchOption ):
      PolicyRuleActionBase.__init__( self, pmapContext, cmapName, 0, ipRuleConfig )
      self.matchOption = matchOption
      self.cmapMatch = None

   def key( self ):
      return self.rule.filter if self.rule else None

   def addToPmap( self, pmap, seqnum ):
      super( PolicyRawRuleActionBase, self ).addToPmap( pmap, seqnum )
      cmap = pmap.rawClassMap.newMember( self.cmapName,
                                         UniqueId() )
      cmap.match.newMember( self.matchOption )
      self.cmapMatch = cmap.match[ self.matchOption ]

class PolicyMplsRuleAction( PolicyRawRuleActionBase ):
   keyTag_ = 'mplsFilter'
   cmapName_ = '__allMplsRule'

   def __init__( self, pmapContext ):
      PolicyRawRuleActionBase.__init__( self, pmapContext, self.__class__.cmapName_, 
                     None, matchMplsAccessGroup )

   def key( self ):
      return self.__class__.cmapName_

class PolicyIpRuleAction( PolicyRawRuleActionBase ):
   keyTag_ = 'ipFilter'

   def __init__( self, pmapContext, cmapName, ipRuleConfig ):
      PolicyRawRuleActionBase.__init__( self, pmapContext, cmapName, ipRuleConfig,
                                       matchIpAccessGroup )

   def key( self ):
      return self.rule.filter

   def addToPmap( self, pmap, seqnum ):
      super( PolicyIpRuleAction, self ).addToPmap( pmap, seqnum )
      self.cmapMatch.ipRule[ 0 ] = self.rule

class PolicyIp6RuleAction( PolicyRawRuleActionBase ):
   keyTag_ = 'ip6Filter'

   def __init__( self, pmapContext, cmapName, ip6RuleConfig ):
      PolicyRawRuleActionBase.__init__( self, pmapContext, cmapName, ip6RuleConfig,
                                     matchIpv6AccessGroup )

   def key( self ):
      return self.rule.filter

   def addToPmap( self, pmap, seqnum ):
      super( PolicyIp6RuleAction, self ).addToPmap( pmap, seqnum )
      self.cmapMatch.ip6Rule[ 0 ] = self.rule

class PolicyRuleToSeqDict:
   def __init__ ( self ):
      self.map = {}
      self.dict = {}

   def __getitem__( self, key ):
      return self.dict.__getitem__( repr( key ) )

   def __setitem__( self, key, value ):
      self.dict.__setitem__( repr( key ), value)
      self.map[ repr( key ) ] = key

   def keys( self ):
      return self.map.values()

   def values ( self ):
      return self.dict.values()

   def pop ( self, key, default ):
      return self.dict.pop ( repr( key ), default )

   def get ( self, key, default ):
      return self.dict.get ( repr( key ), default )

class PolicyMacRuleAction( PolicyRawRuleActionBase ):
   keyTag_ = 'macFilter'

   def __init__( self, cmapName, seqnum, macRuleConfig ):
      PolicyRawRuleActionBase.__init__( self, cmapName, seqnum, macRuleConfig,
                                        matchMacAccessGroup )
   def key( self ):
      return self.rule.filter

   def addToPmap( self, pmap, seqnum ):
      super( PolicyMacRuleAction, self ).addToPmap( pmap, seqnum )
      self.cmapMatch.macRule[ 0 ] = self.rule

class PolicyMapContext( object ):
   ROLLBACK_ENABLED_TEST_OVERRIDE = None

   def __init__( self, config, statusReqDir, status, pmapName, pmapType,
                 rollbackEnabled=True ):
      self.config_ = config
      self.statusReqDir_ = statusReqDir
      self.status_ = status
      self.pmapName_ = pmapName
      self.mapType_ = pmapType
      self.rollbackEnabled = rollbackEnabled

      setting = PolicyMapContext.rollbackEnabledOrNone()
      if setting is not None:
         self.rollbackEnabled = setting
      t9( "PolicyMapContext.rollbackEnabled={}".format( self.rollbackEnabled ) )

      self.mode = None   # This should be a weakref

      # original policy map config, subconfig, and version
      self.opmapCfg = getPmapConfig( self.config(), pmapName )
      self.opmap = getPmapSubConfig( self.config(), pmapName )
      self.opmapVers = self.opmap.pmapVersion if self.opmap else None

      self.editedEntries_ = {}

      self.initializeRuleToSeq()
      self.lastSequenceIs( 0 )

      # the policy map subconfig that may be modified in this session
      self.npmap = None
      self.npmapVers = None

      # This is set in derived classes (TrafficPolicyContext) when a rule is being
      # move to a new sequence number but otherwise unchanged. In this case, actions
      # should not be cleared as they will not be re-added.
      self.moving = False

      # This is set by the derived class if the feature always install the
      # default deny rule. In this case, we need to block the class name as
      # well as the default deny priority from the CLI.
      self.installDefaultDeny = False

   def initializeRuleToSeq( self ):
      # Cache matching-rule to seq number mappings
      self.ruleToSeq = {}
      self.ruleToSeq[ PolicyClassRuleAction.keyTag() ] = PolicyRuleToSeqDict()
      self.ruleToSeq[ PolicyIpRuleAction.keyTag() ] = PolicyRuleToSeqDict()
      self.ruleToSeq[ PolicyIp6RuleAction.keyTag() ] = PolicyRuleToSeqDict()
      self.ruleToSeq[ PolicyMacRuleAction.keyTag() ] = PolicyRuleToSeqDict()
      self.ruleToSeq[ PolicyMplsRuleAction.keyTag() ] = PolicyRuleToSeqDict()

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def copyEditPmap( self ):
      pmaps = self.config().pmapType.pmap
      pmapCfg = pmaps.get( self.pmapName_, None )
      if not pmapCfg:
         pmapCfg = self.config().pmapType.newPmapInput( self.pmapName_,
               self.config().pmapType.type )
         pmaps.addMember( pmapCfg )

      if self.opmap:
         self.lastSequenceIs( self.setRuleToSeq( self.opmap ) )

      if self.rollbackEnabled:
         t0( 'Copy existing policy map for %s' % self.pmapName_ )
         self.npmap = pmapCfg.subconfig.newMember( self.pmapName_,
                                                   UniqueId() )
         self.npmapVers = self.npmap.pmapVersion
         if self.opmap:
            self.copyPolicyMap( self.opmap, self.npmap, self.mapType_ )
      else:
         t0( 'Modifying existing policy map for %s in-place' % self.pmapName_ )
         self.npmap = self.opmap
         self.npmapVers = self.opmapVers

   def newEditPmap( self ):
      t0( 'Create new policy map for %s' % self.pmapName_ )
      pmaps = self.config().pmapType.pmap
      pmapCfg = pmaps.get( self.pmapName_, None )
      if not pmapCfg:
         pmapCfg = self.config().pmapType.newPmapInput( self.pmapName_,
               self.config().pmapType.type )
         pmaps.addMember( pmapCfg )
      self.npmap = pmapCfg.subconfig.newMember( self.pmapName_,
                                                UniqueId() )
      self.npmapVers = self.npmap.pmapVersion

   def delPolicyResources( self ):
      # don't delete the config we're editing unless we created it
      if not self.rollbackEnabled and self.npmap == self.opmap:
         return

      self.delPmapSubConfig( self.npmap )
      pmapCfg = getPmapConfig( self.config(), self.pmapName_ )
      if pmapCfg:
         if len( pmapCfg.subconfig.keys() ) == 0:
            pmapCfg.currCfg = None
            self.delPmapCfg( self.mode(), self.mode().commentKey(), self.pmapName_ )

   def config( self ):
      return self.config_

   def statusReqDir( self ):
      return self.statusReqDir_

   def status( self ):
      return self.status_

   def pmapName( self ):
      return self.pmapName_

   def currentPmap( self ):
      return self.npmap

   def lastSequence( self ):
      return self.lastseq

   def lastSequenceIs( self, seqnum ):
      self.lastseq = seqnum

   def defaultSequenceInc( self ):
      return 10

   def maxRules( self ):
      raise NotImplementedError

   def allocSeqnum( self, lastSeqnum=None ):
      if lastSeqnum is None:
         lastSeqnum = self.lastSequence()
      step = self.defaultSequenceInc()
      return lastSeqnum + step

   def maxSeq( self ):
      return MAX_SEQ

   def mapTypeStr( self ):
      return 'policy map'

   def getRawTagAndFilter( self, cmapName ):
      """
      Given a ClassMapMatch instance, select which filter is used by 'rawClassMaps'
      for this policy type. For ACLs, this is ip{6}Rules, whereas traffic policies
      use structuredFilters (see TrafficPolicyContext.getRawFilter).
      """
      rawCmap = self.npmap.rawClassMap.get( cmapName, None )
      match = rawCmap.match
      if matchIpAccessGroup in match:
         cmapMatch = match[ matchIpAccessGroup ]
         return PolicyIpRuleAction.keyTag(), cmapMatch.ipRule.values()[ 0 ].filter
      elif matchIpv6AccessGroup in match:
         cmapMatch = match[ matchIpv6AccessGroup ]
         return PolicyIp6RuleAction.keyTag(), cmapMatch.ip6Rule.values()[ 0 ].filter
      elif matchMacAccessGroup in match:
         cmapMatch = match[ matchMacAccessGroup ]
         return PolicyMacRuleAction.keyTag(), cmapMatch.macRule.values()[ 0 ].filter
      elif matchMplsAccessGroup in match:
         # There should be only one mpls rawmatch per entire policymap
         return PolicyMplsRuleAction.keyTag(), PolicyMplsRuleAction.cmapName_
      else:
         assert False, 'Unsupported ClassMap matchTypes %s' % match
         return None, None

   def resequence( self, start, inc ):
      # seqNum overflow check
      oclassPrio = sorted( self.npmap.classPrio.items() )
      if start + ( len( oclassPrio ) - 1 ) * inc > self.maxSeq():
         return 'errSequenceOutOfRange'

      self.npmap.classPrio.clear()
      self.initializeRuleToSeq()

      currseq = start
      for _, cmapName in oclassPrio:
         self.npmap.classPrio[ currseq ] = cmapName
        
         tag = PolicyClassRuleAction.keyTag()
         key = cmapName

         rawCmap = self.npmap.rawClassMap.get( cmapName, None )
         if rawCmap:
            tag, key = self.getRawTagAndFilter( cmapName )
         
         self.ruleToSeq.setdefault( tag, PolicyRuleToSeqDict() )[ key ] = currseq
         currseq += inc
         
      self.lastSequenceIs( currseq )
      return 'success'

   def addRuleAtSeqnum( self, seqnum, policyRuleAction ):
      policyRuleAction.addToPmap( self.npmap, seqnum )

   def delRuleAtSeqnum( self, seqnum, clearActions=False ):
      cmapName = self.npmap.classPrio.get( seqnum, None )
      if cmapName:
         if self.npmap.rawClassMap.get( cmapName, None ):
            del self.npmap.rawClassMap[ cmapName ]
         if clearActions:
            for action in self.npmap.classAction[ cmapName 
                  ].policyAction.itervalues():
               self.removeAction( action )
            del self.npmap.classAction[ cmapName ]
         del self.npmap.classPrio[ seqnum ]

   def lookupRuleToSeq( self, policyRuleAction ):
      rule = policyRuleAction.key()
      keyTag = policyRuleAction.keyTag()
      return self.ruleToSeq[ keyTag ].get( rule, None )

   def addRuleToSeq( self, policyRuleAction, seqnum ):
      rule = policyRuleAction.key()
      keyTag = policyRuleAction.keyTag()
      self.ruleToSeq[ keyTag ][ rule ] = seqnum

   def delRuleToSeq( self, policyRuleAction ):
      rule = policyRuleAction.key()
      keyTag = policyRuleAction.keyTag()
      self.ruleToSeq[ keyTag ].pop( rule, None )

   def updateLastSequence( self ):
      maxSeq = 0
      for keyTag in self.ruleToSeq.keys():
         seqnumList = self.ruleToSeq[ keyTag ].values()
         if seqnumList != []:
            maxSeq = max( maxSeq, max( seqnumList ) )
      self.lastSequenceIs( maxSeq )

   def getRuleAtSeqnum( self, seqnum ):
      '''
        Return only the matching rule. If necessary, we can return
        the actions later on.
      '''
      policyRule = None
      cmapName = self.npmap.classPrio.get( seqnum, None )
      if cmapName:
         rawCmap = self.npmap.rawClassMap.get( cmapName, None )
         if rawCmap:
            if matchIpAccessGroup in rawCmap.match:
               cmapMatch = rawCmap.match.get( matchIpAccessGroup )
               if cmapMatch:
                  ipRule = cmapMatch.ipRule.values()[0]
                  policyRule = PolicyIpRuleAction( self, cmapName, ipRule )

            elif matchIpv6AccessGroup in rawCmap.match:
               cmapMatch = rawCmap.match.get( matchIpv6AccessGroup )
               if cmapMatch:
                  ip6Rule = cmapMatch.ip6Rule.values()[0]
                  policyRule = PolicyIp6RuleAction( self, cmapName, ip6Rule )

            elif matchMacAccessGroup in rawCmap.match:
               cmapMatch = rawCmap.match.get( matchMacAccessGroup )
               if cmapMatch:
                  macRule = cmapMatch.macRule.values()[0]
                  policyRule = PolicyMacRuleAction( self, cmapName, macRule )

            elif matchMplsAccessGroup in rawCmap.match:
               policyRule = PolicyMplsRuleAction( self )

            else:
               assert False, 'No match types recognized ' + \
                     str( rawCmap.match.keys() )
         else:
            policyRule = PolicyClassRuleAction( self, cmapName, seqnum )
      return policyRule

   def addRuleCommon( self, seqnum, policyRuleAction ):
      oseqnum = self.lookupRuleToSeq( policyRuleAction )
      if oseqnum:   # rule exists?
         if not seqnum:
            seqnum = oseqnum
         else:
            if oseqnum != seqnum:
               # If we are simply moving the rule (via before / after CLI), do not
               # clear the actions.
               self.delRuleAtSeqnum( oseqnum, not self.moving )
      else:
         if not seqnum:
            seqnum = self.allocSeqnum()

      if seqnum > self.maxSeq():
         reason = 'sequence number %d exceeded maximum sequence supported' % seqnum
         self.mode().addError( 'Failed to add rule: %s' % reason )
         return False

      # If replacing a rule at seqnum, then make sure it's gone from the
      # rule lookup cache too.
      existingRule = self.getRuleAtSeqnum( seqnum )
      if existingRule:
         clearActions = existingRule.cmapName != policyRuleAction.cmapName
         self.delRuleAtSeqnum( seqnum, clearActions )
         self.delRuleToSeq( existingRule )

      self.addRuleAtSeqnum( seqnum, policyRuleAction )
      self.addRuleToSeq( policyRuleAction, seqnum )
      self.lastSequenceIs( seqnum )
      return True

   def delRuleCommon( self, seqnum, policyRuleAction ):
      oseqnum = self.lookupRuleToSeq( policyRuleAction )
      if not oseqnum:
         return  # silently
      if seqnum and oseqnum != seqnum:
         reason = 'Sequence number and rule mismatch'
         self.mode().addWarning( 'Failed to delete class rule: %s' % reason )
         return
      seqnum = oseqnum
      self.delRuleAtSeqnum( seqnum, True )
      self.delRuleToSeq( policyRuleAction )
      if seqnum == self.lastSequence():
         self.updateLastSequence()

   def rollback( self, mode, commentKey, npmap=None, opmap=None ):
      ''' Roll back from npmap to opmap'''
      if not npmap:
         npmap = self.npmap
      if not opmap:
         opmap = self.opmap
      if npmap:    # delete new policy subCfg
         self.delPmapSubConfig( npmap )
      if opmap:    # restore old policy subCfg
         self.opmapCfg.currCfg = opmap
      else:        # delete policy cfg if none existed.
         self.delPmapCfg( mode, commentKey, self.pmapName_ )

   def _checkStatus( self, mode, commentKey ):
      chkr = PolicyOpChkr( self.statusReqDir(), self.status() )
      commitStatus, result = chkr.verify( 'policy', self.pmapName_ )

      # Rollback config
      if not commitStatus:
         reason = result.error if result and result.error else 'unknown'
         mode.addError( 'Failed to commit %s : %s' % ( self.mapTypeStr(), reason ) )
         self.rollback( mode, commentKey )
         return reason
      else:
         if self.opmap:
            self.delPmapSubConfig( self.opmap )
         if result and result.warning:
            mode.addWarning( "Warning: policy map %s %s" %
                                    ( self.pmapName_, result.warning ) )
         return None

   def commit( self ):
      pmaps = self.config().pmapType.pmap
      pmapCfg = pmaps.get( self.pmapName_, None )

      if not pmapCfg:
         # pmap has been deleted by another session
         errmsg = 'Aborting this change. Policy map deleted by another CLI session.'
         self.mode().addError( errmsg )
         self.delPolicyResources()
         return

      # policy changed by another session?
      if pmapCfg and pmapCfg.currCfg and self.opmapCfg and self.opmapCfg.currCfg \
             and self.identicalPolicyMap( pmapCfg.currCfg, 
                                     self.opmapCfg.currCfg ) != IDENTICAL:
         errmsg = 'Aborting this change. Policy map changed by another CLI session.'
         self.mode().addError( errmsg )
         self.delPolicyResources()
         return

      # When in a config-session we edit the npmap directly to avoid
      # excessive copying of policies in config-session. Because of this we will not
      # not see a CHANGED state.
      if not self.rollbackEnabled:
         t0( 'Setting currCfg to policy map %s' % self.npmap.policyName )
         pmapCfg.currCfg = self.npmap
         mode = self.mode()
         commentKey = mode.commentKey()
         if mode.session_.inConfigSession():
            handler = lambda mode, onSessionCommit=True, key=commentKey: \
               self._checkStatus( mode, key )
            CliSession.registerSessionOnCommitHandler(
                  mode.session_.entityManager,
                  "policy-map-%s" % self.npmap.policyName, handler )
         return

      # policy changed by "this" session?
      cfgChanged = CHANGED
      if self.opmap:
         cfgChanged = self.identicalPolicyMap( self.opmap, self.npmap )

      if cfgChanged == CHANGED:
         t0( 'Setting currCfg to policy map %s' % self.npmap.policyName )
         pmapCfg.currCfg = self.npmap
         t0( 'currCfg policy map %s version %s' % \
                       ( self.npmap.policyName, self.npmap.pmapVersion) )

         mode = self.mode()
         commentKey = mode.commentKey()
         if mode.session_.inConfigSession():
            handler = lambda mode, onSessionCommit=True, key=commentKey: \
               self._checkStatus( mode, key )
            CliSession.registerSessionOnCommitHandler(
                  mode.session_.entityManager,
                  "policy-map-%s" % self.npmap.policyName, handler )
            return

         if mode and not ( mode.session_.startupConfig() or
                           mode.session_.isStandalone() ):
            self._checkStatus( mode, commentKey )
      elif cfgChanged == RESEQUENCED:
         # Only the sequence numbers have changed, so just copy those over.
         self.opmap.classPrio.clear()
         for key, val in self.npmap.classPrio.iteritems():
            self.opmap.classPrio[ key ] = val
         self.delPolicyResources()
      else:
         # Delete the new policy, since we're staying with the old one.
         self.delPolicyResources()
   
   def identicalClassActions( self, c1Action, c2Action ):
      raise NotImplementedError

   def identicalPolicyMap( self, p1, p2 ):
      if p1 is None or p2 is None:
         return CHANGED
      
      if p1 == p2:
         return IDENTICAL

      # compare named counters
      p1NamedCounters = p1.namedCounter
      p2NamedCounters = p2.namedCounter
      if set( p1NamedCounters ) != set( p2NamedCounters ):
         return CHANGED
      
      # compare class Actions
      p1Cacts = p1.classAction.keys()
      p2Cacts = p2.classAction.keys()
      if p1Cacts != p2Cacts:
         return CHANGED
      for cname in p1Cacts:
         c1Action = p1.classAction[ cname ]
         c2Action = p2.classAction[ cname ] 
            
         ret = self.identicalClassActions( c1Action, c2Action )
         if ret != IDENTICAL:
            return ret
               
      # compare class priorities
      p1Prios = p1.classPrio.keys()
      p2Prios = p2.classPrio.keys()
      if p1Prios != p2Prios:
         if p1.classPrio.values() == p2.classPrio.values():
            # The keys were changed, but the values are still in the same order
            return RESEQUENCED
         else:
            return CHANGED
      if p1.classPrio.values() != p2.classPrio.values():
         return CHANGED

      return IDENTICAL

   def copyPolicyMap( self, src, dst, mapType ):
      for cmap in dst.rawClassMap:
         if cmap not in src.rawClassMap:
            del dst.rawClassMap[ cmap ]

      for c in dst.namedCounter:
         if c not in src.namedCounter:
            del dst.namedCounter[ c ]
   
      for cmap in dst.classAction:
         if cmap not in src.classAction:
            del dst.classAction[ cmap ]
   
      for prio in dst.classPrio:
         if prio not in src.classPrio:
            del dst.classPrio[ prio ]

      # Copy class maps corresponding to raw match statements
      for cmap in src.rawClassMap.itervalues():
         dstRawCmap = dst.rawClassMap.newMember( cmap.className,
                                                 UniqueId() )
         self.copyRawClassMap( cmap, dstRawCmap, mapType )

      for c in src.namedCounter:
         dst.namedCounter.add( c )
   
      allClasses = src.classAction.keys()
   
      for cmap in allClasses:
         srcAction = src.classAction[ cmap ]
         dstAction = dst.classAction.newMember( cmap )
         self.copyClassAction( srcAction, dstAction )

      for prio in src.classPrio:
         dst.classPrio[ prio ] = src.classPrio[ prio ]

   def copyRawClassMap( self, src, dst, mapType ):
      ''' src and dst are of type ClassMapSubConfig
      '''
      dst.matchCondition = src.matchCondition
      for option in src.match.keys():
         srcMatch = src.match[ option ]
         dst.match.newMember( option )
         dstMatch = dst.match[ option ]
         if option == matchIpAccessGroup:
            for _, srcIpRule in srcMatch.ipRule.iteritems():
               dstMatch.ipRule[ 0 ] = srcIpRule
         elif option == matchIpv6AccessGroup:
            for _, srcIp6Rule in srcMatch.ip6Rule.iteritems():
               dstMatch.ip6Rule[ 0 ] = srcIp6Rule
         elif option == matchMacAccessGroup:
            for _, srcMacRule in srcMatch.macRule.iteritems():
               dstMatch.macRule[ 0 ] = srcMacRule
         else:
            pass
   
      for option in dst.match.keys():
         if option not in src.match:
            del src.match[ option ]

   def copyClassAction( self, src, dst ):
      for action in dst.policyAction.keys():
         actionObj = dst.policyAction.pop( action )
         self.removeAction( actionObj )
    
      for action in src.policyAction.values():
         newAction = self.copyAction( action )
         if newAction:
            dst.policyAction.addMember( newAction )

   def delPolicySubConfigActions( self, pmapSubCfg ):
      '''
         Takes a PolicyMapSubConfig object as input
      '''
      for classAction in pmapSubCfg.classAction.itervalues():
         for action in classAction.policyAction.itervalues():
            self.removeAction( action )

   def delAllPolicyActions( self, pmapCfg ):
      '''
          Takes a PolicyMapConfig object as input
      '''
      if pmapCfg:
         for pmapSubCfg in pmapCfg.subconfig.itervalues():
            self.delPolicySubConfigActions( pmapSubCfg )

   def delPmapSubConfig( self, pmap ):
      if not self.rollbackEnabled and self.npmap == self.opmap:
         return
      self.delPolicySubConfigActions( pmap )
      pmapCfg = getPmapConfig( self.config(), pmap.policyName )
      if pmapCfg and pmap.pmapVersion in pmapCfg.subconfig:
         t0( 'deleting subConfig for policy map %s' % pmap.policyName )
         del pmapCfg.subconfig[ pmap.pmapVersion ]

   def delPmapCfg( self, mode, commentKey, pmapname ):
      t9( "delPmapCfg({})".format( pmapname ) )
      pmapCfg = getPmapConfig( self.config(), pmapname )
      if pmapCfg:
         for pmapSubCfg in pmapCfg.subconfig.values():
            self.delPmapSubConfig( pmapSubCfg )
         if pmapname in self.config().pmapType.pmap:
            del self.config().pmapType.pmap[ pmapname ]
            del self.config().pmapType.pmapInput[ pmapname ]
      mode.removeCommentWithKey( commentKey )
      
   def delPmap( self, mode, pmapname ):
      t9( "delPmap({})".format( pmapname ) )
      # Delete all internal class maps for this policymap and then the policy map
      # unlink all actions from pbrConfig.pbrActions
      self.delPmapCfg( mode, mode.commentKey(), pmapname )

      chkr = PolicyOpChkr( self.statusReqDir(), self.status() )
      delStatus, result = chkr.verify( 'policy', pmapname )
      if delStatus is False:
         reason = result.error if result else 'unknown'
         mode.addWarning( 'Failed to delete policy map: %s' % reason )

   def removeAction( self, action ):
      raise NotImplementedError

   def copyAction( self, src ):
      raise NotImplementedError

   def setRuleToSeq( self, policyMapSubConfig ):
      maxSeq = 0
      classMapToSeq = self.ruleToSeq[ 'classMap' ]
      ipFilterToSeq = self.ruleToSeq[ 'ipFilter' ]
      ip6FilterToSeq = self.ruleToSeq[ 'ip6Filter' ]
      macFilterToSeq = self.ruleToSeq[ PolicyMacRuleAction.keyTag() ]
      mplsFilterToSeq = self.ruleToSeq[ PolicyMplsRuleAction.keyTag() ]
      for prio in policyMapSubConfig.classPrio:
         maxSeq = max( maxSeq, prio )
         cmapName = policyMapSubConfig.classPrio[ prio ]
         rawCmap = policyMapSubConfig.rawClassMap.get( cmapName, None )
         if rawCmap:
            if matchIpAccessGroup in rawCmap.match:
               ipRule = rawCmap.match[ matchIpAccessGroup ].ipRule.values()[0]
               ipFilterToSeq[ ipRule.filter ] = prio
            elif matchIpv6AccessGroup in rawCmap.match:
               ip6Rule = rawCmap.match[ matchIpv6AccessGroup ].ip6Rule.values()[0]
               ip6FilterToSeq[ ip6Rule.filter ] = prio
            elif matchMacAccessGroup in rawCmap.match:
               macRule = rawCmap.match[ matchMacAccessGroup ].macRule.values()[0]
               macFilterToSeq[ macRule.filter ] = prio
            elif matchMplsAccessGroup in rawCmap.match:
               mplsFilterToSeq[ PolicyMplsRuleAction.cmapName_ ] = prio
            else:
               assert False, \
                  'Unsupported ClassMap matchTypes %s' % rawCmap.match
         else:
            # It's a "normal" class map reference
            classMapToSeq[ cmapName ] = prio
      return maxSeq

   @staticmethod
   def setRollbackEnabled( rollbackEnabled ):
      PolicyMapContext.ROLLBACK_ENABLED_TEST_OVERRIDE = \
         not rollbackEnabled == False

   @staticmethod
   def rollbackEnabledOrNone():
      return PolicyMapContext.ROLLBACK_ENABLED_TEST_OVERRIDE

def getPmapConfig( config, pmapName ):
   return config.pmapType.pmap.get( pmapName, None )

def getPmapSubConfig( config, pmapName, version=None ):
   pmap = getPmapConfig( config, pmapName )
   subconfig = None
   if pmap:
      if version:
         subconfig = pmap.subconfig.get( version, None )
      else:
         subconfig = pmap.currCfg
   return subconfig

class ClassMapContext( object ):
   def __init__( self, config, statusReqDir, status, mapType, cmapName, matchType ):
      self.config_ = config
      self.statusReqDir_ = statusReqDir
      self.status_ = status
      self.mapType_ = mapType
      self.cmapName_ = cmapName
      self.matchType_ = matchType
      self.currentSeqNum = 10

      self.mode = None   # This should be a weakref

      # original class map config, subconfig, and version
      self.ocmapCfg = self.getCmapConfig( cmapName )
      self.ocmap = self.getCmapSubConfig( cmapName )
      self.ocmapVers = self.ocmap.cmapVersion if self.ocmap else None
      self.oAcl2Seq = {}

      # new class map subconfig that may be modified in this session
      self.ncmap = None

   def modeIs( self, mode ):
      self.mode = weakref.ref( mode ) if mode else None

   def copyEditCmap( self, entry ):
      t0( 'Copy existing class map for %s' % self.cmapName_ )
      cmaps = self.config().cmapType.cmap
      cmapCfg = cmaps.get( self.cmapName_, None )
      if not cmapCfg:
         cmapCfg = self.config().cmapType.newCmapInput( self.cmapName_ )
         cmaps.addMember( cmapCfg )
      self.ncmap = cmapCfg.subconfig.newMember( self.cmapName_,
                                                UniqueId() )
      if self.ocmap:
         self.copyClassMap( self.ocmap, self.ncmap, self.oAcl2Seq )

   # Copy contents of one map entry to another
   def copyClassMap( self, srcEntry, dstEntry, acl2seq ):
      dstEntry.matchCondition = srcEntry.matchCondition
      # Copy Match Rules
      for op in srcEntry.match:
         srcMatch = srcEntry.match[ op ]
         if op in dstEntry.match:
            dstMatch = dstEntry.match[ op ]
            acl2seqOp = acl2seq[ op ]
         else:
            dstMatch = dstEntry.match.newMember( op )
            acl2seq[ op ] = {}
            acl2seqOp = acl2seq[ op ]
   
         # copy acl rules
         for prio, aclName in srcMatch.acl.iteritems():
            dstMatch.acl[ prio ] = aclName
            acl2seqOp[ aclName ] = prio
         for prio, aclName in dstMatch.acl.iteritems():
            if prio not in srcMatch.acl:
               del dstMatch.acl[ prio ]
   
      for op in dstEntry.match:
         dstMatch = dstEntry.match[ op ]
         if op in [ matchIpAccessGroup, matchIpv6AccessGroup, 
                    matchMacAccessGroup ] \
               and op not in srcEntry.match:
            del dstEntry.match[ op ]


   def newEditCmap( self ):
      # Add an edit by creating a new entry
      t0( 'Create new class map for %s' % self.cmapName_ )
      cmaps = self.config().cmapType.cmap
      cmapCfg = cmaps.get( self.cmapName_, None )
      if not cmapCfg:
         cmapCfg = self.config().cmapType.newCmapInput( self.cmapName_ )
         cmaps.addMember( cmapCfg )
      self.ncmap = cmapCfg.subconfig.newMember( self.cmapName_,
                                                UniqueId() )

   def delClassResources( self ):
      self.delCmapSubConfig( self.ncmap )
      cmapCfg = self.getCmapConfig( self.cmapName_ )
      if cmapCfg:
         if len( cmapCfg.subconfig.keys() ) == 0:
            cmapCfg.currCfg = None
            self.delCmapCfg( self.mode(), self.cmapName_ )

   def config( self ):
      return self.config_

   def statusReqDir( self ):
      return self.statusReqDir_

   def status( self ):
      return self.status_

   def cmapName( self ):
      return self.cmapName_

   def mapType( self ):
      return self.mapType_

   def matchType( self ):
      return self.matchType_

   def currentCmap( self ):
      return self.ncmap

   def rollback( self, ncmap=None, ocmap=None ):
      ''' Roll back from ncmap to ocmap'''
      if not ncmap:
         ncmap = self.ncmap
      if not ocmap:
         ocmap = self.ocmap
      if ncmap:    # delete new class subCfg
         self.delCmapSubConfig( ncmap )
      if ocmap:    # restore old class subCfg
         self.ocmapCfg.currCfg = ocmap
      else:        # delete class cfg if none existed.
         self.delCmapCfg( self.mode(), self.cmapName_ )

   def commit( self ):
      cmaps = self.config().cmapType.cmap
      cmapCfg = cmaps.get( self.cmapName_, None )

      if not cmapCfg:
         # cmap has been deleted by another session
         errmsg = 'Aborting this change. Class map deleted by another CLI session.'
         self.mode().addError( errmsg )
         return

      # Class map changed by another session?
      if cmapCfg and cmapCfg.currCfg and self.ocmapCfg and self.ocmapCfg.currCfg \
             and self.identicalClassMap( cmapCfg.currCfg,
                                    self.ocmapCfg.currCfg ) != IDENTICAL:
         errmsg = 'Aborting this change. Class map changed by another CLI session.'
         self.delClassResources()
         self.mode().addError( errmsg )
         return

      # Class map changed by "this" session?
      cfgChanged = CHANGED
      if self.ocmap:
         cfgChanged = self.identicalClassMap( self.ocmap, self.ncmap )

      # Update current class map config
      if cfgChanged == CHANGED:
         cmapCfg.currCfg = self.ncmap

         commitStatus = True
         result = None
         if self.mode() and not ( self.mode().session_.inConfigSession() or
                                  self.mode().session_.startupConfig() or
                                  self.mode().session_.isStandalone() ):
            chkr = PolicyOpChkr( self.statusReqDir(), self.status() )
            commitStatus, result = chkr.verify( 'class', self.cmapName_ )

         # Rollback config
         if commitStatus is False:
            reason = result.error if result and result.error else 'unknown'
            self.mode().addError( 'Failed to commit class map : %s' % reason )
            self.rollback()
         else:
            if self.ocmap:
               self.delCmapSubConfig( self.ocmap )
               
            if result and result.warning:
               self.mode().addWarning( "Warning: class map %s %s" % 
                                       ( self.cmapName_, result.warning ) )

      elif cfgChanged == RESEQUENCED:
         # Just copy over the sequence numbers to the current config.
         for match in self.ocmap.match:
            omatch = self.ocmap.match[ match ]
            nmatch = self.ncmap.match[ match ]
            
            omatch.acl.clear()
            for key, val in nmatch.acl.iteritems():
               omatch.acl[ key ] = val

            omatch.ipRule.clear()
            for key, val in nmatch.ipRule.iteritems():
               omatch.ipRule[ key ] = val

            omatch.ip6Rule.clear()
            for key, val in nmatch.ip6Rule.iteritems():
               omatch.ip6Rule[ key ] = val

            omatch.macRule.clear()
            for key, val in nmatch.macRule.iteritems():
               omatch.macRule[ key ] = val

         self.delClassResources()
      else:
         self.delClassResources()

   def getCmaps( self ):
      return self.config().cmapType.cmap

   def getCmapConfig( self, cmapName ):
      return self.config().cmapType.cmap.get( cmapName, None )

   def getCmapSubConfig( self, cmapName, version=None ):
      cmap = self.getCmapConfig( cmapName )
      subconfig = None
      if cmap:
         if version:
            subconfig = cmap.subconfig.get( version, None )
         else:
            subconfig = cmap.currCfg
      return subconfig

   def delCmapSubConfig( self, cmap ):
      cmapCfg = self.getCmapConfig( cmap.className )
      if cmapCfg and cmap.cmapVersion in cmapCfg.subconfig:
         del cmapCfg.subconfig[ cmap.cmapVersion ]

   def delCmapCfg( self, mode, cmapname ):
      if cmapname in self.config().cmapType.cmap:
         del self.config().cmapType.cmap[ cmapname ]
         del self.config().cmapType.cmapInput[ cmapname ]
      mode.removeComment()

   def identicalClassMap( self, c1, c2 ):
      if c1 is None or c2 is None:
         return CHANGED
   
      if c1 == c2:
         return IDENTICAL
   
      if c1.matchCondition != c2.matchCondition:
         return CHANGED
   
      c1MatchOps = set( c1.match.keys() )
      c2MatchOps = set( c2.match.keys() )
   
      if c1MatchOps != c2MatchOps:
         return CHANGED
   
      identicalResult = IDENTICAL
      for op in c1MatchOps:
         c1Match = c1.match[ op ]
         c2Match = c2.match[ op ]
         if c1Match.option != c2Match.option:
            return CHANGED
         # compare Acls
         c1Acls = c1Match.acl.keys()
         c2Acls = c2Match.acl.keys()
         if c1Acls != c2Acls:
            if c1Match.acl.values() == c2Match.acl.values():
               # Values are the same, but the keys have changed.
               identicalResult = RESEQUENCED
            else:
               return CHANGED
         if c1Match.acl.values() != c2Match.acl.values():
            return CHANGED
   
         # compare IpRuleConfig
         c1IpRuleKeys = c1Match.ipRule.keys()
         c2IpRuleKeys = c2Match.ipRule.keys()
         if c1IpRuleKeys != c2IpRuleKeys:
            if c1Match.ipRule.values() == c2Match.ipRule.values():
               # Values are the same, but the keys have changed.
               identicalResult = RESEQUENCED
            else:
               return CHANGED
         if c1Match.ipRule.values() != c2Match.ipRule.values():
            return CHANGED
   
         # compare Ip6RuleConfig
         c1Ip6RuleKeys = c1Match.ip6Rule.keys()
         c2Ip6RuleKeys = c2Match.ip6Rule.keys()
         if c1Ip6RuleKeys != c2Ip6RuleKeys:
            if c1Match.ip6Rule.values() == c2Match.ip6Rule.values():
               # Values are the same, but the keys have changed.
               identicalResult = RESEQUENCED
            else:
               return CHANGED
         if c1Match.ip6Rule.values() != c2Match.ip6Rule.values():
            return CHANGED
   
         # compare MacRuleConfig
         c1MacRuleKeys = c1Match.macRule.keys()
         c2MacRuleKeys = c2Match.macRule.keys()
         if c1MacRuleKeys != c2MacRuleKeys:
            if c1Match.macRule.values() == c2Match.macRule.values():
               # Values are the same, but the keys have changed.
               identicalResult = RESEQUENCED
            else:
               return CHANGED
         if c1Match.macRule.values() != c2Match.macRule.values():
            return CHANGED
   
      return identicalResult



class PolicyMapMode( PolicyMapModeBase, BasicCli.ConfigModeBase ):
   name = 'Policy Map Configuration'
   modeParseTree = CliParser.ModeParseTree()
   autoConfigSessionAllowed = False

   def __init__( self, parent, session, context, mapStr ):
      self.pmapContext = context
      context.modeIs( self )
      self.mapType_ = context.mapType_
      self.mapStr   = mapStr
      self.pmapName = context.pmapName()
      self.pmapClassContext = None
      param = ( self.mapType_, self.mapStr, self.pmapName )
      PolicyMapModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'PolicyMapMode onExit...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.pmapContext.delPolicyResources()
      self.pmapContext = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.pmapContext is None:
         t0( 'commitContext has no context' )
         return

      context = self.pmapContext
      self.pmapContext = None
      context.commit()

   def switchToPmapClassMode( self, context ):
      raise NotImplementedError

   def gotoPolicyMapClassMode( self, pmapName, cmapName, seqnum=None ):
      def pmapClassPresent( cmapName, seqnum ):
         existingRule = self.pmapContext.ruleToSeq[ 'classMap' ].get(
            cmapName, None )
         return cmapName in currentPmap.classAction and \
            ( seqnum is None or existingRule in ( None, seqnum ) )

      context = None
      if context == None:
         context = PolicyMapClassContext( self.pmapContext.currentPmap(),
                                          cmapName, self.mapType_,
                                          self.pmapContext )
      currentPmap = self.pmapContext.currentPmap()
      if pmapClassPresent( cmapName, seqnum ):
         context.copyEditPCmap( cmapName, seqnum )
      else:
         context.newEditPCmap( cmapName, seqnum )
      self.pmapClassContext = context
      self.switchToPmapClassMode( context )

   def actionDataToClassAction( self, cmapName, actType, actData ):
      raise NotImplementedError

   def actionArgsToActionData( self, rawRuleActType=None, **kwargs ):
      if rawRuleActType == None:
         return self.defaultActionTypeAndData()
      else:
         return rawRuleActType

   @staticmethod
   def _rawCmapName( seqnum, pmapName ):
      uid = UniqueId()
      return '%s_r_%s_%s_%s' % ( pmapName, uid.timestamp, uid.pid, uid.count )

   def defaultActionTypeAndData( self ):
      raise NotImplementedError

   def handleRawMatchStmt( self, no, seqnum, action, vlanSpec, ruleFilter, log,
                           remark, **kwargs ):
      ipRuleConfig = None
      ip6RuleConfig = None
      macRuleConfig = None
      policyRuleAction = None
      rawCmapName = self._rawCmapName( seqnum, self.pmapName )
      if vlanSpec:
         ( ruleFilter.vlan, ruleFilter.vlanMask,
           ruleFilter.innerVlan, ruleFilter.innerVlanMask ) = vlanSpec
      if isinstance( ruleFilter, tacAclIpFilterType ):
         ipRuleConfig = tacAclIpRuleConfigType( filter=ruleFilter,
                                                action=action,
                                                log=log )
         policyRuleAction = PolicyIpRuleAction( self.pmapContext, rawCmapName, 
                                                ipRuleConfig )
      elif isinstance( ruleFilter, tacAclIp6FilterType ):
         ip6RuleConfig = tacAclIp6RuleConfigType( filter=ruleFilter,
                                                  action=action,
                                                  log=log )
         policyRuleAction = PolicyIp6RuleAction( self.pmapContext, rawCmapName, 
                                                 ip6RuleConfig )
      elif isinstance( ruleFilter, tacAclMacFilterType ):
         macRuleConfig = tacAclMacRuleConfigType( filter=ruleFilter,
                                                  action = action,
                                                  log=log )
         policyRuleAction = PolicyMacRuleAction( self.pmapContext, rawCmapName,
                                                 macRuleConfig )
      else:
         assert ruleFilter == PolicyMplsRuleAction.keyTag()
         policyRuleAction = PolicyMplsRuleAction( self.pmapContext )

      if no:
         self.pmapContext.delRuleCommon( 0, policyRuleAction )
      else:
         actType, actData = self.actionArgsToActionData(
               rawRuleActType=kwargs.pop( 'rawRuleActType' ), **kwargs )

         if isinstance( actType, str ):
            classAction = self.actionDataToClassAction( rawCmapName, 
                                                        actType, actData )
            policyRuleAction.addAction( actType, classAction )
         else:
            for index in range( len( actType ) ):
               classAction = self.actionDataToClassAction( rawCmapName,
                                                           actType[ index ],
                                                           actData[ index ] )
               policyRuleAction.addAction( actType[ index ], classAction )
               
         self.pmapContext.addRuleCommon( seqnum, policyRuleAction )

   def removeRulesBySequence( self, seqStart, seqEnd ):
      context = self.pmapContext
      seqnum = seqStart
      while seqnum < ( seqEnd + 1 ):
         existingRule = context.getRuleAtSeqnum( seqnum )
         if existingRule:
            context.delRuleCommon( seqnum, existingRule )
         seqnum = seqnum + 1

   def resequence( self, start, inc ):
      context = self.pmapContext
      return context.resequence( start, inc )

   def setClass( self, no, seqnum, cmapName ):
      if no:
         lastMode = self.session_.modeOfLastPrompt()
         if isinstance( lastMode, PolicyMapClassMode ):
            if cmapName == lastMode.cmapName:
               lastMode.pmapClassContext = None
         policyRuleAction = PolicyClassRuleAction( self.pmapContext, 
                                                   cmapName, seqnum )
         self.pmapContext.delRuleCommon( seqnum, policyRuleAction )
         commentKey = "pmap-c-%s-%s-%s" % ( self.mapStr,
                                            self.pmapName,
                                            cmapName )
         self.removeCommentWithKey( commentKey )
      else:
         pmapName = self.pmapContext.pmapName()
         self.gotoPolicyMapClassMode( pmapName, cmapName, seqnum )

class ClassMapMode( ClassMapModeBase, BasicCli.ConfigModeBase ):
   name = 'Class Map Configuration'
   modeParseTree = CliParser.ModeParseTree()
   autoConfigSessionAllowed = False

   # Each mode object has a session object. We associate the classmap
   # context with the mode object.
   def __init__( self, parent, session, context, mapStr ):
      self.cmapContext = context
      context.modeIs( self )
      self.cmapName_ = context.cmapName()
      self.mapType_ = context.mapType()
      self.mapStr = mapStr
      self.matchType_ = context.matchType()
      param = ( self.mapType_, self.mapStr, self.cmapName_ )
      ClassMapModeBase.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def onExit( self ):
      t0( 'ClassMapMode onExit ...' )
      self.commitContext()
      BasicCli.ConfigModeBase.onExit( self )

   def abort( self ):
      self.cmapContext  = None
      self.session_.gotoParentMode()

   def commitContext( self ):
      if self.cmapContext is None:
         t0( 'commitContext has no context' )
         return
      context = self.cmapContext
      self.cmapContext = None
      context.commit()

   def setMatchValue( self, no, seqnum, accessGroupType, value ):
      context = self.cmapContext
      entry = context.currentCmap()

      matchOption = matchOptionToEnum( accessGroupType )

      def matchOptionExists( matchOption ):
         return matchOption in entry.match

      if no:
         if matchOption:
            options = [ matchOption ]
         else:
            options = tacMatchOption.attributes

         for option in options:
            if matchOptionExists( option ):
               entryMatch = entry.match[ option ]
               acl2seq = context.oAcl2Seq[ option ]
               oseqnum = acl2seq.get( value, None )
               if seqnum is None and value:
                  seqnum = oseqnum

               # Is there an ACL with this seqnum. Bogus seqnum?
               if seqnum:
                  aclName = entryMatch.acl.get( seqnum, None )
                  if aclName and ( not value or ( value == aclName ) ):
                     del entryMatch.acl[ seqnum ]
                     del acl2seq[ aclName ]
      else:
         if not matchOptionExists( matchOption ):
            entry.match.newMember( matchOption )
            context.oAcl2Seq[ matchOption ] = {}

         entryMatch = entry.match[ matchOption ]
         acl2seq = context.oAcl2Seq[ matchOption ]
         oseqnum = acl2seq.get( value, None )
         if oseqnum:       # acl exists?
            if not seqnum:
               seqnum = oseqnum
            else:
               if oseqnum != seqnum:
                  del entryMatch.acl[ oseqnum ]
         else:
            if seqnum is None:
               maxprio = max( entryMatch.acl.keys() + [ 0 ] )
               seqnum = ( ( maxprio / 10 ) + 1 ) * 10

         if entryMatch.acl.get( seqnum, None ): # diff acl at input seq?
            del acl2seq[ entryMatch.acl[ seqnum ] ]
         entryMatch.acl[ seqnum ] = value  # add/replace acl at seqnum
         acl2seq[ value ] = seqnum

         # Delete other matchOption if any
         #if matchOptionExists( PbrLib.matchOptionOtherToEnum( matchOption ) ):
         #   del entry.match[ PbrLib.matchOptionOtherToEnum( matchOption )  ]
         #   del context.oAcl2Seq[ PbrLib.matchOptionOtherToEnum( matchOption ) ]
         otherOptions = matchOptionOtherToEnum( matchOption )
         for option in otherOptions:
            if option in entry.match:
               del entry.match[ option ]
            if option in context.oAcl2Seq:
               del context.oAcl2Seq[ option ]

   def handleNoSeq( self, args ):
      seqRanges = args[ 'SEQUENCES' ].ranges()
      for seqRange in seqRanges:
         seq = seqRange[ 0 ]
         while seq < ( seqRange[1] + 1 ):
            self.setMatchValue( True, seq, None, None )
            seq = seq + 1

   def resequence( self, start, inc ):
      context = self.cmapContext
      entry = context.currentCmap()
      if not entry or len( entry.match.keys() ) != 1:
         return
      entryMatch = entry.match.values()[ 0 ]

      # copy the match.acl collection and clear it out
      oAclPrio = {}
      for seq, aclName in entryMatch.acl.iteritems():
         oAclPrio[ seq ] = aclName
      for seq in oAclPrio:
         del entryMatch.acl[ seq ]

      currseq = start
      for oseqnum in sorted( oAclPrio.keys() ):
         currseq = start
         entryMatch.acl[ currseq ] = oAclPrio[ oseqnum ]
         start += inc

class PolicyCfgCmdBinder( AclCli.AclCfgCmdChains ):

   def __init__( self, status ):
      AclCli.AclCfgCmdChains.__init__( self, status )

      self.actionMatcher = CliMatcher.EnumMatcher( {
         'match' : 'Specify packets to accept',
      } )

      class IpFilterGeneric( CliCommand.CliExpression ):
         expression = ( '( PROTOCOL_NUM | GENERIC_PROTOCOL ) '
                        'IP_SRC IP_DST [ IP_OPT ]' )
         data = {
            'PROTOCOL_NUM' : CliCommand.Node( matcher=self.ipProtoNumMatcher,
               alias='protocol' ),
            'GENERIC_PROTOCOL' : AclCliRules.GenericProtocolsMatcher(
               AclCliLib.genericIpProtocols ),
            'IP_SRC' : self.ipSourceMatcher,
            'IP_DST' : self.ipDestinationMatcher,
            'IP_OPT' : self.ipOptGenericExpression # no 'tracked'
         }

      class ProtoIpExpression( CliCommand.CliExpression ):
         expression = 'IP_SRC IP_DST [ NEXTHOP_GROUP ] [ IP_OPT ]'
         data = {
            'IP_SRC' : self.ipSourceMatcher,
            'IP_DST' : self.ipDestinationMatcher,
            'NEXTHOP_GROUP' : self.nexthopGroupNameExpression,
            'IP_OPT' : self.ipOptExpression,
         }

      class ProtoIcmpExpression( CliCommand.CliExpression ):
         expression = 'ICMP_PROTOCOL IP_SRC IP_DST [ ICMP_OPT ] [ IP_OPT ]'
         data = {
            'ICMP_PROTOCOL' : self.protoIcmpMatcher,
            'IP_SRC' : self.ipSourceMatcher,
            'IP_DST' : self.ipDestinationMatcher,
            'ICMP_OPT' : self.icmpMatcher,
            'IP_OPT' : self.ipOptExpression,
         }

      class ProtoTcpExpression( CliCommand.CliExpression ):
         expression = ( 'TCP_PROTOCOL IP_SRC [ TCP_SRC_PORT ] '
                        'IP_DST [ TCP_DST_PORT ] [ TCP_OPT ] [ IP_OPT ]' )
         data = {
            'TCP_PROTOCOL' : self.protoTcpMatcher,
            'IP_SRC' : self.ipSourceMatcher,
            'TCP_SRC_PORT' : self.tcpSrcPortSpecMatcher,
            'IP_DST' : self.ipDestinationMatcher,
            'TCP_DST_PORT' : self.tcpDstPortSpecMatcher,
            'TCP_OPT' : self.tcpOptMatcher,
            'IP_OPT' : self.ipOptExpression,
         }

      class ProtoUdpExpression( CliCommand.CliExpression ):
         expression = ( 'UDP_PROTOCOL IP_SRC [ UDP_SRC_PORT ] '
                        'IP_DST [ UDP_DST_PORT ] [ IP_OPT ]' )
         data = {
            'UDP_PROTOCOL' : self.protoUdpMatcher,
            'IP_SRC' : self.ipSourceMatcher,
            'UDP_SRC_PORT' : self.udpSrcPortSpecMatcher,
            'IP_DST' : self.ipDestinationMatcher,
            'UDP_DST_PORT' : self.udpDstPortSpecMatcher,
            'IP_OPT' : self.ipOptExpression,
         }

      class ProtoSctpExpression( CliCommand.CliExpression ):
         expression = ( 'SCTP_PROTOCOL IP_SRC [ SCTP_SRC_PORT ] '
                        'IP_DST [ SCTP_DST_PORT ] [ IP_OPT ]' )
         data = {
            'SCTP_PROTOCOL' : self.protoSctpMatcher,
            'IP_SRC' : self.ipSourceMatcher,
            'SCTP_SRC_PORT' : self.sctpSrcPortSpecMatcher,
            'IP_DST' : self.ipDestinationMatcher,
            'SCTP_DST_PORT' : self.sctpDstPortSpecMatcher,
            'IP_OPT' : self.ipOptExpression,
         }

      class IpFilterExpression( CliCommand.CliExpression ):
         expression = ( 'IP_FILTER_GENERIC | PROTO_IP | PROTO_ICMP | '
                        'PROTO_UDP | PROTO_TCP | PROTO_SCTP' )
         data = {
            'IP_FILTER_GENERIC' : IpFilterGeneric,
            'PROTO_IP' : ProtoIpExpression,
            'PROTO_ICMP' : ProtoIcmpExpression,
            'PROTO_UDP' : ProtoUdpExpression,
            'PROTO_TCP' : ProtoTcpExpression,
            'PROTO_SCTP' : ProtoSctpExpression
         }

      # override IP rules defined in Acl cmd chains
      self.ipFilterExpression = IpFilterExpression

      class Ip6FilterGeneric( CliCommand.CliExpression ):
         expression = ( '( PROTOCOL_NUM | GENERIC_PROTOCOL ) '
                        'IP6_SRC IP6_DST [ IP6_OPT ]' )
         data = {
            'PROTOCOL_NUM' : CliCommand.Node( matcher=self.ipProtoNumMatcher,
               alias='protocol' ),
            'GENERIC_PROTOCOL' : AclCliRules.GenericProtocolsMatcher(
               AclCliLib.genericIpProtocols ),
            'IP6_SRC' : self.ip6SourceMatcher,
            'IP6_DST' : self.ip6DestinationMatcher,
            'IP6_OPT' : self.ip6OptGenericExpression
         }

      class ProtoIp6Expression( CliCommand.CliExpression ):
         expression = 'IP6_SRC IP6_DST [ NEXTHOP_GROUP ] [ IP6_OPT ]'
         data = {
            'IP6_SRC' : self.ip6SourceMatcher,
            'IP6_DST' : self.ip6DestinationMatcher,
            'NEXTHOP_GROUP' : self.nexthopGroupNameExpression,
            'IP6_OPT' : self.ip6OptExpression,
         }

      class ProtoIcmp6Expression( CliCommand.CliExpression ):
         expression = 'ICMP6_PROTOCOL IP6_SRC IP6_DST [ ICMP6_OPT ] [ IP6_OPT ]'
         data = {
            'ICMP6_PROTOCOL' : self.protoIcmp6Matcher,
            'IP6_SRC' : self.ip6SourceMatcher,
            'IP6_DST' : self.ip6DestinationMatcher,
            'ICMP6_OPT' : self.icmp6Matcher,
            'IP6_OPT' : self.ip6OptExpression,
         }

      class ProtoTcp6Expression( CliCommand.CliExpression ):
         expression = ( 'TCP_PROTOCOL IP6_SRC [ TCP_SRC_PORT ] '
                        'IP6_DST [ TCP_DST_PORT ] [ TCP_OPT ] [ IP6_OPT ]' )
         data = {
            'TCP_PROTOCOL' : self.protoTcpMatcher,
            'IP6_SRC' : self.ip6SourceMatcher,
            'TCP_SRC_PORT' : self.tcpSrcPortSpecMatcher,
            'IP6_DST' : self.ip6DestinationMatcher,
            'TCP_DST_PORT' : self.tcpDstPortSpecMatcher,
            'TCP_OPT' : self.tcpOptMatcher,
            'IP6_OPT' : self.ip6OptExpression,
         }

      class ProtoUdp6Expression( CliCommand.CliExpression ):
         expression = ( 'UDP_PROTOCOL IP6_SRC [ UDP_SRC_PORT ] '
                        'IP6_DST [ UDP_DST_PORT ] [ IP6_OPT ]' )
         data = {
            'UDP_PROTOCOL' : self.protoUdpMatcher,
            'IP6_SRC' : self.ip6SourceMatcher,
            'UDP_SRC_PORT' : self.udpSrcPortSpecMatcher,
            'IP6_DST' : self.ip6DestinationMatcher,
            'UDP_DST_PORT' : self.udpDstPortSpecMatcher,
            'IP6_OPT' : self.ip6OptExpression,
         }

      class ProtoSctp6Expression( CliCommand.CliExpression ):
         expression = ( 'SCTP_PROTOCOL IP6_SRC [ SCTP_SRC_PORT ] '
                        'IP6_DST [ SCTP_DST_PORT ] [ IP6_OPT ]' )
         data = {
            'SCTP_PROTOCOL' : self.protoSctpMatcher,
            'IP6_SRC' : self.ip6SourceMatcher,
            'SCTP_SRC_PORT' : self.sctpSrcPortSpecMatcher,
            'IP6_DST' : self.ip6DestinationMatcher,
            'SCTP_DST_PORT' : self.sctpDstPortSpecMatcher,
            'IP6_OPT' : self.ip6OptExpression,
         }

      class Ip6FilterExpression( CliCommand.CliExpression ):
         expression = ( 'IP6_FILTER_GENERIC | PROTO_IP6 | PROTO_ICMP6 | '
                        'PROTO_UDP | PROTO_TCP | PROTO_SCTP' )
         data = {
            'IP6_FILTER_GENERIC' : Ip6FilterGeneric,
            'PROTO_IP6' : ProtoIp6Expression,
            'PROTO_ICMP6' : ProtoIcmp6Expression,
            'PROTO_UDP' : ProtoUdp6Expression,
            'PROTO_TCP' : ProtoTcp6Expression,
            'PROTO_SCTP' : ProtoSctp6Expression
         }

      # override IP6 rules defined in Acl cmd chains
      self.ip6FilterExpression = Ip6FilterExpression

   def nexthopGroupMatchSupportedGuard( self, mode, token ):
      if 'Pbr' not in mode.name:
         return CliParser.guardNotThisPlatform
      return IraIpCli.nexthopGroupSupportedGuard( mode, token )

   #-----------------------------------------------------------------
   #              no sequence-number
   #-----------------------------------------------------------------
   # Similar to the interface command, we allow comma separated ranges,
   # such as no 10 , 30 - 60 , 100 - $
   @staticmethod
   def handleNoSeq( mode, args ):
      seqRanges = args[ 'SEQUENCES' ].ranges()
      for seqRange in seqRanges:
         ret = mode.removeRulesBySequence( seqRange[0], seqRange[1] )
         if ret == 'errReadOnly':
            mode.addError(
               AclCliLib.aclModifyError( mode.aclName, mode.aclType,
                                         'Readonly list' ) )

   @staticmethod
   def handleResequence( mode, start, inc ):
      ret = mode.resequence( start, inc )
      if ret == 'errSequenceOutOfRange':
         mode.addError( "Error: Sequence number out of range" )

   @classmethod
   def _handleRuleCommon( cls, mode, noOrSeqnum, action, vlanSpec, ruleFilter,
                          mirrorSession, log, remark, copyToCpuDst, **kwargs ):
      seqnum = 0
      no = False

      if action != 'match':
         print 'action not supported'
         return

      if action == 'match':
         action = 'permit'

      log = log is True

      if noOrSeqnum is None:
         seqnum = 0
      elif noOrSeqnum is True or noOrSeqnum == 'default':
         no = True
      else:
         seqnum = noOrSeqnum

      mode.handleRawMatchStmt( no, seqnum, action, vlanSpec, ruleFilter,
                               log=log, remark=None, **kwargs )

def _checkStatus( mode, no, statusReqDir, status, intfName, pmapName, opmapName,
                  pmapIntfs ):
   # Verify status of service policy apply/delete operation
   chkr = PolicyOpChkr( statusReqDir, status )
   success, result = chkr.verify( 'intf', intfName )

   # Rollback in case application of service policy failed.
   # In case of delete, just print a warning; no rollback.
   if not success:
      reason = result.error if result and result.error else 'unknown'
      errmsg = 'Failed to apply policy on %s: %s' % ( intfName, reason )
      if no:
         mode.addWarning( errmsg )
      else:
         mode.addError( errmsg )
         if mode.session_.inConfigSession():
            # Let CliPlugin config-session logic handle rollback of checkpoint
            return reason
         if opmapName:
            pmapIntfs[ intfName ] = opmapName
         else:
            if intfName in pmapIntfs:
               del pmapIntfs[ intfName ]
      return reason
   else:
      if result and result.warning:
         mode.addWarning( "Warning: policy map %s %s" % ( pmapName,
                                                          result.warning ) )
      return None

def handleServicePolicy( mode, no, pmapName, config, statusReqDir, status,
                         fallback=False, intfConfig=None,
                         configSessionRollbackSupported=False ):
   intfName = mode.intf.name
   cfgChange = False

   assert intfConfig

   if fallback:
      pmapIntfs = intfConfig.intfFallback
   else:
      pmapIntfs = intfConfig.intf

   opmapName = pmapIntfs.get( intfName, None )

   # Apply or delete service policy to intf
   if no:
      if pmapName is None:
         pmapName = opmapName

      if pmapName == opmapName:
         del pmapIntfs[ intfName ]
         cfgChange = True
   else:
      if pmapName != opmapName:
         invalidConfig = False
         if fallback:
            intfs = intfConfig.intf
            if pmapName == intfs.get( intfName, None ):
               invalidConfig = True
         else:
            intfs = intfConfig.intfFallback
            if pmapName == intfs.get( intfName, None ):
               invalidConfig = True
         # Make sure the same policy is not attached as primary and fallback policy.
         if invalidConfig:
            mode.addError( 'Same policy cannot be attached as '
                           'both primary and fallback policy to an interface.' )
            return
         pmapIntfs[ intfName ] = pmapName
         cfgChange = True

   if not cfgChange:
      return

   if mode.session_.inConfigSession() and configSessionRollbackSupported:
      handler = lambda mode, onSessionCommit=True: \
            _checkStatus( mode, no, statusReqDir, status, intfName, pmapName,
                          opmapName, pmapIntfs )
      CliSession.registerSessionOnCommitHandler(
            mode.session_.entityManager, "policy-map-%s" % intfName, handler )
      return

   if not ( mode.session_.startupConfig() or
            mode.session_.isStandalone() or
            mode.session_.inConfigSession() ):
      _checkStatus( mode, no, statusReqDir, status, intfName, pmapName,
                    opmapName, pmapIntfs )

def deleteClassMap( mode, cmapName, config, statusRequestDir, status):
   if not cmapName in config.cmapType.cmap:
      return

   del config.cmapType.cmap[ cmapName ]
   del config.cmapType.cmapInput[ cmapName ]
   mode.removeComment()
   
   chkr = PolicyOpChkr( statusRequestDir, status )
   delStatus, result = chkr.verify( 'class', cmapName )
   if delStatus is False:
      reason = result.error if result and result.error else 'unknown'
      mode.addWarning( 'Failed to delete class map: %s' % reason )
