#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

from CliMode.Qos import PolicingModeBase
import CliPlugin.QosCli as QosCli
import CliPlugin.EthIntfCli as EthIntfCli
import CliPlugin.IntfCli as IntfCli
from CliPlugin.QosCliModel import ModePolicingModel, InterfacePolicerAllModel
from CliPlugin.QosCliLib import PolicingModelContainer, IntfPolicerModelContainer
import CliPlugin.QosShowCommands as QosShowCommands
import CliPlugin.SubIntfCli as SubIntfCli
from QosLib import isLagPort
from QosTypes import ( tacPolicerType, tacBurstUnit, tacRateUnit )
import Toggles.QosToggleLib

import BasicCli
import CliCommand
import CliMatcher
import CliParser
import ConfigMount
import LazyMount
import Plugins
import ShowCommand
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'PolicingCli' )
t0 = Tracing.trace0

#------------------------------------------------------------------------------------
# Name Rules for Interface Policing
#------------------------------------------------------------------------------------
def getProfileNameRule( mode ):
   suggestedCompletions = qosInputConfig.policingConfig.profile
   return suggestedCompletions

def getPolicerInstanceNameRule( mode ):
   suggestedCompletions = qosInputConfig.policingConfig.policerInstance
   return suggestedCompletions

#------------------------------------------------------------------------------------
# Guards for Interface Policing
#------------------------------------------------------------------------------------
def guardPolicingMode( mode, token ):
   if qosAclHwStatus.interfacePolicingSupported and \
      Toggles.QosToggleLib.toggleQosInterfacePolicingEnabled():
      return None
   return CliParser.guardNotThisPlatform

def guardPolicingSupported( mode, token ):
   return guardPolicingMode( mode, token )

def guardPolicingTrTcmConfig( mode, token ):
   if qosAclHwStatus.interfacePolicingTrTcmSupported:
      return None
   return CliParser.guardNotThisPlatform

def guardIntfDedicatedPolicer( mode, token ):
   if not ( mode.intf.isSubIntf() and \
            qosAclHwStatus.interfacePolicingDedicatedPolicerSupported and \
            qosAclHwStatus.interfacePolicingSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardIntfGroupPolicer( mode, token ):
   if not ( mode.intf.isSubIntf() and \
            qosAclHwStatus.interfacePolicingGroupPolicerSupported and \
            qosAclHwStatus.interfacePolicingSupported ):
      return CliParser.guardNotThisPlatform
   return None

def guardIntfPolicing( mode, token ):
   if guardIntfDedicatedPolicer( mode, token ) and \
      guardIntfGroupPolicer( mode, token ):
      return CliParser.guardNotThisPlatform
   return None

#-------------------------------------------------------------------------------
# Policing Mode
#-------------------------------------------------------------------------------
def gotoPolicingMode( mode, args ):
   if qosInputConfig.policingConfig is None:
      t0( 'gotoPolicingMode: Creating policingConfig in qos/input/config/cli' )
      policingConfigName = "QosPolicing"
      qosInputConfig.policingConfig = ( policingConfigName, )
   childMode = mode.childMode( PolicingMode )
   mode.session_.gotoChildMode( childMode )

#-------------------------------------------------------------------------------
# Tokens for Policing Mode, Interface-Policers in SubIntf Mode and Show Policing
#-------------------------------------------------------------------------------
nodePolicingMode = CliCommand.guardedKeyword( 'policing',
      helpdesc='Policing Configuration', guard=guardPolicingMode )
nodeProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Policer rate configuration", guard=guardPolicingSupported )
nodeProfileName = CliMatcher.DynamicNameMatcher( getProfileNameRule,
      helpdesc="Policer profile name",
   pattern=r'(?!counters$|interface$|summary$|group$|input$)[A-Za-z0-9_:{}\[\]-]+' )
nodePoliceHRate = CliCommand.guardedKeyword( 'rate',
      helpdesc="Set higher rate", guard=guardPolicingTrTcmConfig )
nodePoliceHBs = CliMatcher.KeywordMatcher( 'burst-size',
      helpdesc="Set burst-size for higher rate" )
nodePolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Associate policer name with a policer profile",
      guard=guardPolicingSupported )
nodePolicerName = CliMatcher.DynamicNameMatcher( getPolicerInstanceNameRule,
      helpdesc="Policer name",
   pattern=r'(?!counters$|interface$|summary$|group$|input$)[A-Za-z0-9_:{}\[\]-]+' )
nodePolicerProfile = CliMatcher.KeywordMatcher( 'profile',
      helpdesc="Associate policer name with a policer profile" )
nodeIntfPolicer = CliCommand.guardedKeyword( 'policer',
      helpdesc="Policer configuration on the interface", guard=guardIntfPolicing )
nodeIntfProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Associate interface with a policer profile",
      guard=guardIntfDedicatedPolicer )
nodeIntfGroupPolicer = CliCommand.guardedKeyword( 'group',
      helpdesc="Associate interface with a group policer",
      guard=guardIntfGroupPolicer )
nodeInput = CliMatcher.KeywordMatcher( 'input',
      helpdesc="Apply policing to inbound packets" )
nodeShowPolicing = CliCommand.guardedKeyword( 'policing',
      helpdesc='Show policing configuration', guard=guardPolicingMode )
nodeShowProfile = CliCommand.guardedKeyword( 'profile',
      helpdesc="Show policer profile", guard=guardPolicingSupported )
nodeShowPolicer = CliCommand.guardedKeyword( 'group', helpdesc="Show group policer",
                                             guard=guardPolicingSupported )

def checkExistingInterfacePolicer( intfPairKey ):
   t0( 'Checking if', intfPairKey, 'exists' )
   return intfPairKey in qosInputConfig.policingConfig.intfPolicer

def isInterfacePolicerSame( intfPairKey, profile, policerInstance ):
   t0( 'IsInterfacePolicerSame for profile: ', profile, ' and policer: ',
       policerInstance )
   intfPolicer = qosInputConfig.policingConfig.intfPolicer[ intfPairKey ]
   profileCheck = False
   policerCheck = False
   profileToCheck = intfPolicer.policerName.get( tacPolicerType.dedicatedPolicer )
   profileCheck = profile in ( None, profileToCheck )
   policerToCheck = intfPolicer.policerName.get( tacPolicerType.policerInstance )
   policerCheck = policerInstance in ( None, policerToCheck )
   return profileCheck and policerCheck

def addIntfPolicerToConfig( intfPairKey, profile, policerInstance ):
   t0( 'Adding ', intfPairKey, 'with profile: ', profile, ' and policerInstance: ',
       policerInstance, ' to the intf-Policer collection' )
   intfPolicer = \
         qosInputConfig.policingConfig.intfPolicer.newMember( intfPairKey )
   if profile:
      intfPolicer.policerName[ tacPolicerType.dedicatedPolicer ] = profile
   if policerInstance:
      intfPolicer.policerName[ tacPolicerType.policerInstance ] = policerInstance
   intfPolicer.uniqueId = Tac.Value( 'Qos::UniqueId' )

def deleteIntfPolicerFromConfig( intfPairKey ):
   t0( 'Deleting ', intfPairKey, ' from intf-Policer collection' )
   del qosInputConfig.policingConfig.intfPolicer[ intfPairKey ]

def configureInterfacePolicer( mode, profile=None, policerInstance=None,
                               inputKeyword=True, no=False ):
   t0( 'Handler to configure Interface-Policers on ', mode.intf.name )
   if not qosInputConfig.policingConfig:
      t0( 'Creating policingConfig in qos/input/config/cli' )
      qosInputConfig.policingConfig = ( 'QosPolicing', )

   intfName = mode.intf.name
   if isLagPort( intfName ):
      intfPair = Tac.Value( "Qos::IntfPairKey", "", intfName )
   else:
      intfPair = Tac.Value( "Qos::IntfPairKey", intfName, "" )
   if not no:
      if inputKeyword:
         exists = checkExistingInterfacePolicer( intfPair )
         if not exists:
            addIntfPolicerToConfig( intfPair, profile, policerInstance )
         else:
            if not isInterfacePolicerSame( intfPair, profile, policerInstance ):
               deleteIntfPolicerFromConfig( intfPair )
               addIntfPolicerToConfig( intfPair, profile, policerInstance )
   else:
      deleteIntfPolicerFromConfig( intfPair )

#------------------------------------------------------------------------------------
# Show CLI handlers
#------------------------------------------------------------------------------------
def showPolicingHandler( mode, args ):
   modePolicingModel = ModePolicingModel()
   policingModelContainer = PolicingModelContainer( qosInputConfig, qosConfig,
                                                    qosStatus, intfPolicingHwStatus,
                                                    modePolicingModel )
   profileName = args.get( 'PROFILE' )
   policerName = args.get( 'POLICER' )
   if profileName:
      policingModelContainer.populateProfile( profileName )
   elif policerName:
      policingModelContainer.populatePolicerInstance( policerName )
   else:
      profileOutput = 'profile' in args
      policerOutput = 'group' in args
      if not profileOutput and not policerOutput:
         profileOutput = True
         policerOutput = True
      policingModelContainer.populateAllPolicerProfiles( profileOutput,
                                                         policerOutput )
   return modePolicingModel

def showPolicingIntfHandler( mode, args ):
   intfPolicerAllModel = InterfacePolicerAllModel()
   intfPolicerModelContainer = IntfPolicerModelContainer( qosInputConfig, qosConfig,
                                                          qosStatus,
                                                          intfPolicingHwStatus,
                                                          intfPolicerAllModel )
   intf = args.get( 'INTF' )
   intfs = IntfCli.Intf.getAll( mode, intf, None, [ EthIntfCli.EthIntf,
                                                    SubIntfCli.SubIntf ] )
   detail_ = 'detail' in args
   if intf is None or not intfs:
      intfPolicerModelContainer.populateAllIntfPolicers( detail_ )
   else:
      intfPolicerModelContainer.populateIntfRangePolicers( intfs, detail_ )
   return intfPolicerAllModel

#------------------------------------------------------------------------------------
# The Policing Mode
#------------------------------------------------------------------------------------
class PolicingMode( PolicingModeBase, BasicCli.ConfigModeBase ):
   name = "Policing Mode Configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      PolicingModeBase.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def identicalProfile( self, profile, cir, cirUnit, bc, bcUnit, pir, pirUnit, be,
                         beUnit ):
      t0( 'Checking if profile:', profile.profileName, 'is identical' )
      pol = profile.profileConfig
      if ( ( pol.cir != cir ) or ( pol.cirUnit != cirUnit ) or ( pol.bc != bc ) or
           ( pol.bcUnit != bcUnit ) or ( pol.pir != pir ) or
           ( pol.pirUnit != pirUnit ) or ( pol.be != be ) or
           ( pol.beUnit != beUnit ) ):
         return False
      return True

   def adjustRateAndBurstUnits( self, cirUnit, bcUnit, pirUnit, beUnit ):
      if cirUnit is None:
         if bcUnit == tacBurstUnit.burstUnitPackets:
            cirUnit = tacRateUnit.rateUnitPps
         else:
            cirUnit = tacRateUnit.rateUnitbps
      if bcUnit is None:
         if cirUnit == tacRateUnit.rateUnitPps:
            bcUnit = tacBurstUnit.burstUnitPackets
         else:
            bcUnit = tacBurstUnit.burstUnitBytes
      if pirUnit is None:
         if cirUnit != tacRateUnit.rateUnitPps:
            pirUnit = tacRateUnit.rateUnitbps
         else:
            pirUnit = tacRateUnit.rateUnitPps
      if beUnit is None:
         if pirUnit != tacRateUnit.rateUnitPps:
            beUnit = tacBurstUnit.burstUnitBytes
         else:
            beUnit = tacBurstUnit.burstUnitPackets
      return cirUnit, bcUnit, pirUnit, beUnit

   def checkExistingProfile( self, profileName ):
      t0( 'Checking if profile:', profileName, 'exists' )
      return profileName in qosInputConfig.policingConfig.profile

   def checkExistingPolicer( self, policerName ):
      t0( 'Checking if policerInstance:', policerName, 'exists' )
      return policerName in qosInputConfig.policingConfig.policerInstance

   def configureProfileParameters( self, profile, cir, cirUnit, bc, bcUnit, pir,
                                   pirUnit, be, beUnit ):
      profile.profileConfig.cir = cir
      profile.profileConfig.cirUnit = cirUnit
      profile.profileConfig.bc = bc
      profile.profileConfig.bcUnit = bcUnit
      profile.profileConfig.pir = pir
      profile.profileConfig.pirUnit = pirUnit
      profile.profileConfig.be = be
      profile.profileConfig.beUnit = beUnit

   def changeProfileConfig( self, profileName, cir, cirUnit, bc, bcUnit, pir,
                            pirUnit, be, beUnit ):
      t0( 'Change profile config for profile: ', profileName )
      assert profileName in qosInputConfig.policingConfig.profile
      profile = qosInputConfig.policingConfig.profile[ profileName ]
      if not self.identicalProfile( profile, cir, cirUnit, bc, bcUnit, pir,
                                    pirUnit, be, beUnit ):
         self.configureProfileParameters( profile, cir, cirUnit, bc, bcUnit, pir,
                                          pirUnit, be, beUnit )
         profile.uniqueId = Tac.Value( 'Qos::UniqueId' )

   def changePolicerConfig( self, policerName, profileName ):
      t0( 'Change profile association config for policerInstance: ', policerName,
          'with new profile: ', profileName )
      assert policerName in qosInputConfig.policingConfig.policerInstance
      policer = qosInputConfig.policingConfig.policerInstance[ policerName ]
      if policer.profileName != profileName:
         policer.profileName = profileName
         policer.uniqueId = Tac.Value( 'Qos::UniqueId' )

   def configureProfile( self, profileName, policerMode='committed',
                         cir=None, cirUnit=None, bc=None, bcUnit=None, pir=None,
                         pirUnit=None, be=None, beUnit=None ):
      t0( 'Handler to configure profile: ', profileName, ' with associated rates' )
      cirUnit, bcUnit, pirUnit, beUnit = self.adjustRateAndBurstUnits( cirUnit,
                                             bcUnit, pirUnit, beUnit )
      if policerMode == 'committed':
         pir = be = 0
      if QosCli.policerOutOfRange( self, cir, cirUnit, bc, bcUnit, pir, pirUnit,
                                   be, beUnit, policerMode ):
         return
      exists = self.checkExistingProfile( profileName )
      if not exists:
         # Add new profile to qosInputConfig.policingConfig.profile
         newProfile = qosInputConfig.policingConfig.profile.newMember( profileName )
         newProfile.profileConfig = ( "", cir, bc, pir, be )
         self.configureProfileParameters( newProfile, cir, cirUnit, bc, bcUnit,
                                          pir, pirUnit, be, beUnit )
         newProfile.uniqueId = Tac.Value( 'Qos::UniqueId' )
      else:
         self.changeProfileConfig( profileName, cir, cirUnit, bc, bcUnit, pir,
                                   pirUnit, be, beUnit )

   def configurePolicer( self, policerName, profileName ):
      t0( 'Handler to configure policerInstance: ', policerName, ' with profile: ',
          profileName )
      exists = self.checkExistingPolicer( policerName )
      if not exists:
         # Add new policerInstance to qosInputConfig.policingConfig.policerInstance
         newPolicer = qosInputConfig.policingConfig.policerInstance.newMember(
            policerName )
         newPolicer.profileName = profileName
         newPolicer.uniqueId = Tac.Value( 'Qos::UniqueId' )
      else:
         self.changePolicerConfig( policerName, profileName )

#------------------------------------------------------------------------------------
# Cli commands for Interface Policing
#------------------------------------------------------------------------------------

#------------------------------------------------------------------------------------
# [ no | default ] profile PROFILE rate <value> {bps, kbps, mbps, gbps} burst-size
# <value> {bytes, kbytes, mbytes} rate <value> {bps, kbps, mbps, gbps} burst-size
# <value> {bytes, kbytes, mbytes}
#------------------------------------------------------------------------------------
class ProfileConfigHigherRateCmd( CliCommand.CliCommandClass ):
   syntax = ( 'profile PROFILE rate CIR [ CIRUNIT ] burst-size BURST [ BURSTUNIT ] '
              '[highrate PIR [ PIRUNIT ] highbsize HBURST [ HBUNIT ]]' )
   noOrDefaultSyntax = 'profile PROFILE ...'
   data = {
      'profile': nodeProfile,
      'PROFILE': nodeProfileName,
      'rate': QosCli.nodePoliceLRate,
      'CIR': QosCli.matcherPoliceCirValue,
      'CIRUNIT': QosCli.CirPirUnit(),
      'burst-size': QosCli.nodePoliceLBs,
      'BURST': QosCli.matcherPoliceBurstValue,
      'BURSTUNIT': QosCli.BurstHigherUnit(),
      'highrate': nodePoliceHRate,
      'PIR': QosCli.matcherPolicePirValue,
      'PIRUNIT': QosCli.CirPirUnit(),
      'highbsize': nodePoliceHBs,
      'HBURST': QosCli.matcherPoliceExcessBurstValue,
      'HBUNIT': QosCli.BurstHigherUnit(),
   }

   @staticmethod
   def handler( mode, args ):
      if args.get( 'PIR' ):
         mode.configureProfile( profileName=args[ 'PROFILE' ],
               policerMode='trTCM', cir=int( args[ 'CIR' ] ),
               cirUnit=args[ 'CIR_SPEED_UNIT' ], bc=int( args[ 'BURST' ] ),
               bcUnit=args[ 'BURST_RATE_UNIT' ], pir=int( args[ 'PIR' ] ),
               pirUnit=args[ 'PIR_SPEED_UNIT' ], be=int( args[ 'HBURST' ] ),
               beUnit=args[ 'HBURST_RATE_UNIT' ] )
      else:
         mode.configureProfile( profileName=args[ 'PROFILE' ],
               cir=int( args[ 'CIR' ] ), cirUnit=args[ 'CIR_SPEED_UNIT' ],
               bc=int( args[ 'BURST' ] ), bcUnit=args[ 'BURST_RATE_UNIT' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      profileName = args[ 'PROFILE' ]
      del qosInputConfig.policingConfig.profile[ profileName ]

PolicingMode.addCommandClass( ProfileConfigHigherRateCmd )

#------------------------------------------------------------------------------------
# [ no | default ] policer POLICER profile PROFILE
#------------------------------------------------------------------------------------
class PolicerConfigCmd( CliCommand.CliCommandClass ):
   syntax = 'policer POLICER profile PROFILE'
   noOrDefaultSyntax = 'policer POLICER ...'
   data = {
      'policer': nodePolicer,
      'POLICER': nodePolicerName,
      'profile': nodePolicerProfile,
      'PROFILE': nodeProfileName,
   }

   @staticmethod
   def handler( mode, args ):
      mode.configurePolicer( policerName=args[ 'POLICER' ],
                             profileName=args[ 'PROFILE' ] )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      policerName = args[ 'POLICER' ]
      del qosInputConfig.policingConfig.policerInstance[ policerName ]

PolicingMode.addCommandClass( PolicerConfigCmd )

#------------------------------------------------------------------------------------
# [ no | default ] policer profile PROFILE group POLICER input
#------------------------------------------------------------------------------------
class InterfacePolicerConfig( CliCommand.CliCommandClass ):
   syntax = 'policer (([profile PROFILE] group POLICER) | (profile PROFILE)) input'
   noOrDefaultSyntax = 'policer ...'
   data = {
      'policer': nodeIntfPolicer,
      'profile': nodeIntfProfile,
      'PROFILE': nodeProfileName,
      'group': nodeIntfGroupPolicer,
      'POLICER': nodePolicerName,
      'input': nodeInput,
   }

   @staticmethod
   def handler( mode, args ):
      inputKeyword = 'input' in args
      configureInterfacePolicer( mode, profile=args.get( 'PROFILE' ),
                                 policerInstance=args.get( 'POLICER' ),
                                 inputKeyword=inputKeyword )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      configureInterfacePolicer( mode, no=True )

QosCli.QosSubIntfModelet.addCommandClass( InterfacePolicerConfig )

#------------------------------------------------------------------------------------
# Show CLI commands
#------------------------------------------------------------------------------------

#------------------------------------------------------------------------------------
# show policing [ ( profile [ profileName ] ) | ( group [ groupName ] ) ]
#------------------------------------------------------------------------------------
class ShowPolicingPolicerProfile( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policing [ ( profile [ PROFILE ] ) | ( group [ POLICER ] ) ]'
   data = {
      'policing': nodeShowPolicing,
      'profile': nodeShowProfile,
      'PROFILE': nodeProfileName,
      'group': nodeShowPolicer,
      'POLICER': nodePolicerName,
   }

   handler = showPolicingHandler
   cliModel = ModePolicingModel

BasicCli.addShowCommandClass( ShowPolicingPolicerProfile )

#------------------------------------------------------------------------------------
# show policing [ ( interfaces [ INTFNAME ] [ detail ] )
#------------------------------------------------------------------------------------
class ShowPolicingIntfPolicer( ShowCommand.ShowCliCommandClass ):
   syntax = 'show policing [ ( interfaces [ INTF ] [ detail ] ) ]'
   data = {
      'policing': nodeShowPolicing,
      'interfaces': QosShowCommands.matcherInterfaces,
      'INTF': QosShowCommands.matcherIntf,
      'detail': 'More comprehensive output'
   }

   handler = showPolicingIntfHandler
   cliModel = InterfacePolicerAllModel

BasicCli.addShowCommandClass( ShowPolicingIntfPolicer )

#------------------------------------------------------------------------------------
# Mount path holders ( Define all mount path holders here )
#------------------------------------------------------------------------------------
qosInputConfig = None
qosConfig = None
qosStatus = None
qosAclHwStatus = None
intfPolicingHwStatus = None

#------------------------------------------------------------------------------------
# Have the Cli Agent mount all needed state from sysdb
#------------------------------------------------------------------------------------
@Plugins.plugin( provides=( "QosPolicingCli", ) )
def Plugin( entityManager ):
   global qosInputConfig, qosConfig, qosStatus, qosAclHwStatus, intfPolicingHwStatus

   qosInputConfig = ConfigMount.mount( entityManager, "qos/input/config/cli",
                                       "Qos::Input::Config", "w" )
   qosConfig = LazyMount.mount( entityManager, "qos/config",
                                "Qos::Config", "r" )
   qosStatus = LazyMount.mount( entityManager, "qos/status", "Qos::Status", "r" )
   qosAclHwStatus = LazyMount.mount( entityManager,
                        "qos/hardware/acl/status/global", "Qos::AclHwStatus", "r" )
   intfPolicingHwStatusDir = "qos/hardware/policing/status"
   intfPolicingHwStatus = LazyMount.mount( entityManager, intfPolicingHwStatusDir,
                                           "Tac::Dir", "ri" )
