#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import LazyMount
import Cell
import TableOutput
import Tac
import os
from CliModel import Dict, Enum, Model
import FruPlugin.PLSlot # pylint: disable=unused-import

class SlotStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Hardware::ModularSystem::FruStatus'

   def __init__( self, slotStatus ):
      Tac.Notifiee.__init__( self, slotStatus )
      self.handleState()

   @Tac.handler( 'state' )
   def handleState( self ):
      slot = Slot( self.notifier_.name )
      if not slot.inserted:
         slot.config.powerCycle = False

# Pylint can't handle property decorators
# pylint: disable-msg=E1101,E0102,E0202
class Slot( object ):
   def __init__( self, name ):
      self.name = name
      self.config = None
      self.status = None
      self.powerSupply = None
      if name.startswith( 'PowerSupply' ):
         shortName = name[ len( 'PowerSupply' ) : ]
         if shortName:
            slotId = int( shortName )
            self.status = powerSlotStatusMount.slotStatus.get( shortName )
            if self.status:
               self.config = powerSlotCliConfigMount.newCliSlotConfig( shortName )
               self.powerSupply = (
                  powerSupplyCliConfigMount.newPmbusCliConfig( slotId ) )
      elif name.startswith( 'Fan' ):
         self.config = fanSlotConfigMount.newSlotConfig( name )
         self.status = fanSlotStatusMount.slotStatus.get( name )
      elif name.startswith( ( 'Linecard', 'Fabric', 'Switchcard' ) ):
         # System may not support these modules
         if cardSlotStatusMount is None:
            return
         self.status = cardSlotStatusMount.slotStatus.get( name )
         self.config = cardCliConfigMount.slotConfig.get( name )
         if not self.config:
            self.config = cardCliConfigMount.newSlotConfig( name )
      if not self.status and platformSlotStatusMount is not None:
         self.status = platformSlotStatusMount.slotStatus.get( name )
         if not self.status:
            if name.startswith( 'Fan' ):
               name = name.replace( 'Fan', 'FanTray' )
               self.status = platformSlotStatusMount.slotStatus.get( name )
         if self.status:
            self.config = platformCliSlotConfigMount.newCliSlotConfig( name )

   @property
   def inserted( self ):
      return self.status and self.status.state not in [ 'unknown', 'notPresent',
                                                        'notOwned' ]

   @inserted.setter
   def inserted( self, inserted ):
      if inserted:
         try:
            self.config.enterDiagsModeOnFruRemoval = False
         except AttributeError:
            pass
         self.config.diagsMode = False
      else:
         originalValue = False
         try:
            originalValue = self.config.enterDiagsModeOnFruRemoval
            self.config.enterDiagsModeOnFruRemoval = True
         except AttributeError:
            raise NotImplementedError

         try:
            self.powerCycle()
         except NotImplementedError:
            self.config.enterDiagsModeOnFruRemoval = originalValue
            raise
      if 'SIMULATION_NORCALCARD' not in os.environ:
         Tac.waitFor( lambda: self.inserted == inserted, sleep=True, warnAfter=None,
                      timeout=60 )

   @property
   def diagsMode( self ):
      enterDiagsModeOnFruRemoval = False
      try:
         enterDiagsModeOnFruRemoval = self.config.enterDiagsModeOnFruRemoval
      except AttributeError:
         pass
      return ( not self.inserted and ( self.config.diagsMode or
                                       enterDiagsModeOnFruRemoval ) )

   @diagsMode.setter
   def diagsMode( self, diagsMode ):
      self.config.diagsMode = diagsMode

   @property
   def empty( self ):
      return not self.inserted and not self.diagsMode

   def powerCycle( self ):
      try:
         if self.inserted:
            try:
               self.config.powerCycle = True
               if 'SIMULATION_NORCALCARD' not in os.environ:
                  _slotStatusReactor = SlotStatusReactor( self.status )
                  Tac.waitFor( lambda: not self.config.powerCycle,
                               sleep=True,
                               warnAfter=None,
                               maxDelay=1,
                               timeout=30 )
            finally:
               if 'SIMULATION_NORCALCARD' not in os.environ:
                  self.config.powerCycle = False
         else:
            raise NotImplementedError
      except AttributeError:
         raise NotImplementedError

   @property
   def statusString( self ):
      if self.inserted:
         return 'inserted'
      if self.diagsMode:
         return 'diags'
      if self.empty:
         return 'empty'
      return 'unknown'

   def validNameNotEmpty( self, mode ):
      if self.config is None:
         mode.addError( '"%s" is not a valid module name' % self.name )
         return False
      if self.empty:
         mode.addError( '"%s" is empty' % self.name )
         return False
      return True
# pylint: enable-msg=E1101,E0102,E0202

#-------------------------------------------------------------------------------
# command to power cycle a module
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> power-cycle
#-------------------------------------------------------------------------------
def doPowercycleModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validNameNotEmpty( mode ):
      if slot.diagsMode:
         mode.addError( '"%s" cannot be power cycled, already in diags mode' %
                        slot.name )
         return
      try:
         slot.powerCycle()
      except NotImplementedError:
         mode.addError( '"%s" not available for power cycle' % slot.name )

#-------------------------------------------------------------------------------
# command to remove a module
#    simulates physical removal by power cycling and
#        immediately placing in to diags mode ( to prevent rediscovery )
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> remove
#-------------------------------------------------------------------------------
def doRemoveModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validNameNotEmpty( mode ):
      if slot.diagsMode:
         mode.addError( '"%s" cannot be removed, already in diags mode' %
                        slot.name )
         return
      try:
         slot.inserted = False
      except NotImplementedError:
         mode.addError( '"%s" not available for removal' % slot.name )
      except Tac.Timeout:
         mode.addError( 'Timed out waiting for "%s" to be removed' % slot.name )

#-------------------------------------------------------------------------------
# command to place module in to diags mode
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> diags
#-------------------------------------------------------------------------------
def doDiagModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validNameNotEmpty( mode ):
      slot.diagsMode = True

#-------------------------------------------------------------------------------
# command to insert a module, take out of diags
#    does not matter if the module reached diags mode
#        being removed or diagsed ( #stillaword )
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> insert
#-------------------------------------------------------------------------------
def doInsertModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   if slot.validNameNotEmpty( mode ):
      try:
         slot.inserted = True
      except Tac.Timeout:
         mode.addError( 'Timed out waiting for "%s" to be inserted' % slot.name )

#-------------------------------------------------------------------------------
# command to power on, soft power off, or immediately power off a power supply
#    requires "enable" mode, hidden
#
# full syntax:
#    platform module <slot> ( on | off | immediate-off )
#-------------------------------------------------------------------------------
def doPowerToggleModule( mode, args ):
   slot = Slot( args.get( "SLOTNAME" ) )
   action = args.get( "VALUE" )
   if slot.validNameNotEmpty( mode ) and slot.powerSupply:
      slot.powerSupply.powerOn = action == 'on'
      if not slot.powerSupply.powerOn:
         slot.powerSupply.immediateOff = action == 'immediate-off'

#-------------------------------------------------------------------------------
# command to show module(s)
#    requires "enable" mode, hidden
#    displays status of module or that the module's slot is empty
#
# full syntax:
#    show platform module [ <slot> | all ]
#-------------------------------------------------------------------------------
class PlatformModuleStatus( Model ):
   status = Enum( values=( "inserted", "diags", "empty", "unknown" ),
         help='Status of module' )

class PlatformModules( Model ):
   modules = Dict( keyType=str, valueType=PlatformModuleStatus,
         help='Mapping between module name and module status' )

   def render( self ):
      if self.modules:
         tableHeadings = ( "Name", "Status" )
         t = TableOutput.createTable( tableHeadings )
         f = TableOutput.Format( justify="left" )
         f.padLimitIs( True )
         t.formatColumns( *( [ f ] * len( tableHeadings ) ) )
         for label, module in sorted( self.modules.iteritems() ):
            t.newRow( label, module.status )
         print t.output()

def doShowPlatformModule( mode, args ):
   modules = {}
   slotName = args.get( "SLOTNAME" )
   if slotName is None or slotName == "all":
      mounts = [ cardSlotStatusMount, fanSlotStatusMount, powerSlotStatusMount,
                 platformSlotStatusMount ]
      slotGroups = [ m.slotStatus.keys() for m in mounts if m is not None ]
      slotKeys = [ key for group in slotGroups for key in group ]
      for slotKey in slotKeys:
         if slotKey in powerSlotStatusMount.slotStatus.keys():
            slotKey = "PowerSupply%s" % slotKey
         slot = Slot( slotKey )
         modules[ slot.name ] = PlatformModuleStatus( status=slot.statusString )
   else:
      slot = Slot( slotName )
      if slot.config is None:
         mode.addError( '"%s" is not a valid module name' % slot.name )
      else:
         modules[ slot.name ] = PlatformModuleStatus( status=slot.statusString )

   return PlatformModules( modules=modules )

cardCliConfigMount = None
cardSlotStatusMount = None
powerSlotCliConfigMount = None
powerSlotStatusMount = None
fanSlotConfigMount = None
fanSlotStatusMount = None
platformCliSlotConfigMount = None
platformSlotStatusMount = None
powerSupplyCliConfigMount = None

def Plugin( entityManager ):
   global cardCliConfigMount
   global cardSlotStatusMount
   global powerSlotCliConfigMount
   global powerSlotStatusMount
   global fanSlotConfigMount
   global fanSlotStatusMount
   global platformCliSlotConfigMount
   global platformSlotStatusMount
   global powerSupplyCliConfigMount
   powerSlotCliConfigMount = LazyMount.mount(
      entityManager,
      "hardware/powerSupply/slot/cli/config",
      "Hardware::PowerSupplySlot::CliConfig", "w" )
   powerSlotStatusMount = LazyMount.mount(
      entityManager,
      "hardware/powerSupply/slot/status",
      "Hardware::PowerSupplySlot::Status", "r" )
   powerSupplyCliConfigMount = LazyMount.mount(
      entityManager,
      "hardware/powerSupply/cli/pmbus11",
      "Hardware::PowerSupply::Pmbus::CliConfig", "w" )
   fanSlotConfigMount = LazyMount.mount(
      entityManager,
      "hardware/fan/cliconfig",
      "Hardware::Fan::CliConfig", "w" )
   fanSlotStatusMount = LazyMount.mount(
      entityManager,
      "hardware/fan/status",
      "Hardware::Fan::Status", "r" )
   if Cell.cellType() == "supervisor":
      cardCliConfigMount = LazyMount.mount(
         entityManager,
         "hardware/modularSystem/config/cli",
         "Hardware::ModularSystem::CliConfig", "w" )
      cardSlotStatusMount = LazyMount.mount(
         entityManager,
         "hardware/modularSystem/status",
         "Hardware::ModularSystem::Status", "r" )
   elif Cell.cellType() == "fixed":
      platformCliSlotConfigMount = LazyMount.mount(
         entityManager,
         "hardware/platform/slot/cli/config",
         "Hardware::PLSlot::CliConfig", "w" )
      platformSlotStatusMount = LazyMount.mount(
         entityManager,
         "hardware/platform/slot/status",
         "Hardware::PLSlot::Status", "r" )
