#!/usr/bin/env/python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import print_function

import re
import cStringIO

import AgentCommandRequest
import AgentDirectory
import ArPyUtils
import BasicCli
import CliCommand
import CliMatcher
import CliMode.Poe
import CliModel
import CliParser
import CliPlugin.TechSupportCli
import CliPlugin.FruCli as FruCli
import CliPlugin.ModuleCli as ModuleCli
import ConfigMount
import EthIntfCli
import IntfCli
import LazyMount
import PLDeviceAgent
import PLSystemAgent
import ReloadCli
import ShowCommand
import Tac
import Toggles.PlutoToggleLib
from CliToken.Platform import platformMatcherForShow
from TableOutput import Format, createTable

DEFAULT_REBOOT_ACTION = 'powerOff'
DEFAULT_INTF_POE_REBOOT_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_ACTION = 'unset'
DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME = 5
DEFAULT_INTERFACE_SHUTDOWN_ACTION = 'maintain'
DEFAULT_INTF_POE_SHUTDOWN_ACTION = 'unset'
DEFAULT_POE_PRIORITY = 'defaultPriority'
DEFAULT_POE_POWER_LIMIT = Tac.Value( "Units::Watts" ).value
MAX_POE_POWER_LIMIT = 60
MIN_POE_POWER_LIMIT = 1

sliceDir = None
poeTecConfig = None
globalPoeStatus = None
poeStatus = None
ethIntfStatus = None
intfCliConfig = None
poeCliConfig = None

classMapping = {
   'class0': 15.4, # Watts
   'class1': 4.0,
   'class1D': 8.0,
   'class2': 7.0,
   'class2D': 14.0,
   'class3': 15.4,
   'class3D' : 31.0,
   'class4': 30.0,
   'class4D': 60.0,
   'class5': 45.0,
   'class6': 60.0,
   'class7': 60.0, # support up to 60W
   'class8': 60.0,
   'invalid': 0.0,
   'unknown': 0.0,
}

def printTable( table ):
   # Strip whitespace from the end of the lines, because that's how it was
   # before and it's what some parsers expect. Also remove the last newline.
   print( '\n'.join( map( str.rstrip, table.output().split( '\n' )[ : -1 ] ) ) )

#------------------------------------------------------------------------------------
# Adds PoE related CLI commands to the "config-if" mode for PoE ports.
#------------------------------------------------------------------------------------
class PoePortModelet( CliParser.Modelet ):

   @staticmethod
   def shouldAddModeletRule( mode ):
      return ( isinstance( mode.intf, EthIntfCli.EthPhyIntf ) and
               mode.intf.name.startswith( "Ethernet" ) )

   modeletParseTree = CliParser.ModeletParseTree()

#------------------------------------------------------------------------------------
# Associate the PoePortModelet with the "config-if" mode.
#------------------------------------------------------------------------------------
IntfCli.IntfConfigMode.addModelet( PoePortModelet )

def poeGuard( mode, token ):
   if not poeStatus or not poeStatus.entryState:
      return CliParser.guardNotThisPlatform
   if isinstance( mode, IntfCli.IntfConfigMode ):
      if mode.intf.name not in poeStatus:
         return "not supported on this interface"
   return None

matcherPoe = CliMatcher.KeywordMatcher(
   'poe',
   helpdesc='PoE configuration' )
nodePoe = CliCommand.Node(
   matcher=matcherPoe,
   guard=poeGuard )

matcherClassNumber = CliMatcher.PatternMatcher(
   pattern='class[0-6]',
   helpname='class<0-6>',
   helpdesc='Class for setting the power limit' )
matcherPowerLimit = CliMatcher.FloatMatcher(
   MIN_POE_POWER_LIMIT,
   MAX_POE_POWER_LIMIT,
   helpdesc='Number in watts',
   precisionString='%.25g' )

#------------------------------------------------------------------------------------
# [no|default] poe [disabled]
#------------------------------------------------------------------------------------
class PoePortDisabled( CliCommand.CliCommandClass ):
   syntax = 'poe [disabled]'
   noOrDefaultSyntax = 'poe disabled ...'
   data = {
      'poe': nodePoe,
      'disabled': 'Disable PoE related functionality',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.poeEnabled = 'disabled' not in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.poeEnabled = True

PoePortModelet.addCommandClass( PoePortDisabled )

#------------------------------------------------------------------------------------
# [no|default] poe priority (low|medium|high|critical) command, in "config-if" mode
#------------------------------------------------------------------------------------
matcherPriorityEnum = CliMatcher.EnumMatcher( {
   'low': 'Low priority (default)',
   'medium': 'Medium priority',
   'high': 'High priority',
   'critical': 'Critical priority',
} )

class PoePortPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'poe priority PRIORITY'
   noOrDefaultSyntax = 'poe priority ...'
   data = {
      'poe': nodePoe,
      'priority': 'Port priority',
      'PRIORITY': matcherPriorityEnum,
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.portPriority = args[ 'PRIORITY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.portPriority = DEFAULT_POE_PRIORITY

PoePortModelet.addCommandClass( PoePortPriorityCmd )

#------------------------------------------------------------------------------------
# [no|default] poe limit ((class CLASS_NUM)|(NUM_WATTS watts)) [fixed]
#------------------------------------------------------------------------------------
class PoeLimitCmd( CliCommand.CliCommandClass ):
   syntax = 'poe limit ( ( class CLASS_NUM ) | ( NUM_WATTS watts ) ) [ fixed ]'
   noOrDefaultSyntax = 'poe limit ...'
   data = {
      'poe': nodePoe,
      'limit': 'Set power limit',
      'class': 'Powered device classification',
      'CLASS_NUM': matcherClassNumber,
      'NUM_WATTS': matcherPowerLimit,
      'watts': 'Unit watts',
      'fixed': 'Ignore hardware classification',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      capacity = MAX_POE_POWER_LIMIT
      if mode.intf.name in poeStatus:
         capacity = poeStatus[ mode.intf.name ].capacity
      if 'CLASS_NUM' in args:
         powerLimit = classMapping[ args[ 'CLASS_NUM' ] ]
      elif 'NUM_WATTS' in args:
         powerLimit = args[ 'NUM_WATTS' ]
      if 'fixed' in args:
         message = ( "The power limit may exceed what is supported by the device "
                     "and could cause damage to the device." )
         mode.addWarning( message )
         config.fixedLimit = True
      else:
         config.fixedLimit = False
      if powerLimit > capacity:
         message = "Power limit exceeds capacity of port %s" % mode.intf.name
         mode.addWarning( message )
      else:
         config.powerLimit = powerLimit

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.powerLimit = DEFAULT_POE_POWER_LIMIT

PoePortModelet.addCommandClass( PoeLimitCmd )

#------------------------------------------------------------------------------------
# [no|default] poe negotiation lldp [disabled]
#------------------------------------------------------------------------------------
class PoePortLldpDisabled( CliCommand.CliCommandClass ):
   syntax = 'poe negotiation lldp [disabled]'
   noOrDefaultSyntax = 'poe negotiation lldp disabled ...'
   data = {
      'poe': nodePoe,
      'negotiation': 'Power limit negotiation',
      'lldp': 'LLDP',
      'disabled': 'Disable PoE negotiation over LLDP',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.lldpEnabled = 'disabled' not in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.lldpEnabled = True

PoePortModelet.addCommandClass( PoePortLldpDisabled )

#------------------------------------------------------------------------------------
# [no|default] poe legacy detect
#------------------------------------------------------------------------------------
def guardLegacy( mode, token ):
   if ( mode.intf.name in poeStatus and
        poeStatus[ mode.intf.name ].legacyDetectSupported ):
      return None
   return "not supported on this interface"

nodeLegacy = CliCommand.guardedKeyword( 'legacy', helpdesc='Legacy detection',
                                        guard=guardLegacy )
class PoeLegacyDetectCmd( CliCommand.CliCommandClass ):
   syntax = 'poe legacy detect'
   noOrDefaultSyntax = 'poe legacy detect ...'
   data = {
      'poe': nodePoe,
      'legacy': nodeLegacy,
      'detect': 'Detection type',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.legacyDetect = True

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.legacyDetect = False

PoePortModelet.addCommandClass( PoeLegacyDetectCmd )

#------------------------------------------------------------------------------------
# [no|default] poe pairset ( ( alternative ( a | b ) ) | 4-pair )
#------------------------------------------------------------------------------------
def guardPairset( mode, token ):
   poeConfig = sliceDir.get( 'PoeDevice', poeTecConfig )
   if ( poeConfig is poeTecConfig and
        mode.intf.name in poeConfig.poePortCapacity and
        poeConfig.poePortCapacity[ mode.intf.name ] > 30.0 ):
      return None
   return "not supported on this interface"

nodePairset = CliCommand.guardedKeyword( 'pairset',
                                         helpdesc='Configure pairset(s) to enable',
                                         guard=guardPairset )
class PoePairsetCmd( CliCommand.CliCommandClass ):
   syntax = 'poe pairset ( ( alternative ( a | b ) ) | 4-pair )'
   noOrDefaultSyntax = 'poe pairset ...'
   data = {
      'poe': nodePoe,
      'pairset': nodePairset,
      'alternative': 'Enable one of the pairsets',
      'a': 'Enable alternative A',
      'b': 'Enable alternative B',
      '4-pair': 'Enable both pairsets',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      if '4-pair' in args:
         config.portPairset = 'fourPair'
      elif 'a' in args:
         config.portPairset = 'altA'
      else:
         config.portPairset = 'altB'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.portPairset = 'fourPair'

PoePortModelet.addCommandClass( PoePairsetCmd )

#------------------------------------------------------------------------------------
# show poe [interface <intf-range>]
#------------------------------------------------------------------------------------
class PoePort( CliModel.Model ):
   portName = CliModel.Str( help='PoE port name' )
   portPresent = CliModel.Bool( help='PoE port present' )
   portCapacity = CliModel.Float( help='PoE port capacity (watts)', optional=True )
   pseEnabled = CliModel.Bool( help='PSE functionalities enabled', optional=True )
   portPriority = CliModel.Enum( values=( 'low', 'medium', 'high', 'critical' ),
                                 help='PoE port priority. Available power is given '
                                      'to higher priority ports. A power supply '
                                      'failure can reduce the available power.',
                                 optional=True )
   portPairset = CliModel.Enum( values=( 'fourPair', 'altA', 'altB' ),
                                help='Enabled pairset(s)', optional=True )
   lldpEnabled = CliModel.Bool( help='LLDP enabled', optional=True )
   legacyDetectEnabled = CliModel.Bool( help='Legacy detect enabled', optional=True )
   powerLimit = CliModel.Float( help='PoE port power limit (watts)', optional=True )
   grantedPower = CliModel.Float( help='PoE port granted power (watts)',
                                  optional=True )
   portState = CliModel.Enum( values=( 'unknown', 'disabled', 'detecting',
                                       'classified', 'powered', 'failed',
                                       'overloaded', 'linkDown', 'adminShutdown' ),
                              help='PoE port state', optional=True )
   pdClass = CliModel.Str( help='Powered device class', optional=True )
   power = CliModel.Float( help='Output power (watts)', optional=True )
   current = CliModel.Float( help='Output current (milliamps)', optional=True )
   voltage = CliModel.Float( help='Output voltage (volts)', optional=True )
   temperature = CliModel.Float( help='Temperature (celsius)', optional=True )
   misconfigured = CliModel.Bool( help='Legacy detect and port mode misconfiguraton',
                                  optional=True )

   def getRow( self ):
      def valueStr( value, deci, unit ):
         if value is None:
            return 'N/A'
         if round( value, deci ) == round( value, 0 ):
            return ( "%.0f" % value ) + unit
         else:
            return ( "%.*f" % ( deci, value ) ) + unit
      def trueFalseStr( value ):
         if value:
            return "true"
         else:
            return "false"
      def portPairsetStr( value ):
         if value == 'fourPair':
            value = '4-pair'
         elif value == 'altA':
            value = 'Alt A'
         else:
            value = 'Alt B'
         if self.misconfigured:
            value = value + '*'
         return value
      def stateStr( state ):
         camelCaseMap = { 'linkDown': 'link down', 'adminShutdown': 'admin down' }
         return camelCaseMap.get( state, state )
      return [ "Et" + re.split( '([0-9/]+)', self.portName )[ 1 ],
               valueStr( self.portCapacity, 0, 'W' ),
               trueFalseStr( self.pseEnabled ),
               portPairsetStr( self.portPairset ),
               self.portPriority,
               trueFalseStr( self.lldpEnabled ),
               trueFalseStr( self.legacyDetectEnabled ),
               valueStr( self.powerLimit, 0, 'W' ),
               valueStr( self.grantedPower, 2, 'W' ),
               stateStr( self.portState ),
               self.pdClass,
               valueStr( self.power, 1, 'W' ),
               valueStr( self.current, 0, 'mA' ),
               valueStr( self.voltage, 1, 'V' ),
               valueStr( self.temperature, 0, 'C' )
             ]

class PoePorts( CliModel.Model ):
   poePorts = CliModel.Dict( keyType=str, valueType=PoePort, help='PoE ports' )

   def render( self ):
      if not self.poePorts:
         return

      poeTableHeader = [
         [ "", "hl", [ "Interface" ], ],
         [ "", "hr", [ "Capacity" ], ],
         [ "PSE", "hr", [ "Enabled" ], ],
         [ "", "hr", [ "Pairset" ] ],
         [ "", "hr", [ "Priority" ], ],
         [ "LLDP", "hr", [ "Enabled" ], ],
         [ "Legacy", "hr", [ "Detect" ], ],
         [ "Power", "hr", [ "Max" ], ],
         [ "Power", "hr", [ "Granted" ], ],
         [ "Port", "hr", [ "State" ], ],
         [ "", "hr", [ "Class" ], ],
         [ "", "hr", [ "Power" ], ],
         [ "", "hr", [ "Current" ], ],
         [ "", "hr", [ "Voltage" ], ],
         [ "", "hr", [ "Temp" ], ]
      ]

      def makePoeTable():
         t = createTable( poeTableHeader )
         fl = Format( justify="left" )
         fl.noPadLeftIs( True )
         fl.padLimitIs( True )
         fr = Format( justify="right" )
         fr.noPadLeftIs( True )
         fr.padLimitIs( True )
         t.formatColumns( fl, fr, fr, fr, fr, fr, fr, fr,
                          fr, fr, fr, fr, fr, fr, fr )
         return t

      outputTable = makePoeTable()
      misconfigured = False
      for port in ArPyUtils.naturalsorted( self.poePorts ):
         if self.poePorts[ port ].misconfigured:
            misconfigured = True
         outputTable.newRow( *self.poePorts[ port ].getRow() )
      printTable( outputTable )
      if misconfigured:
         print( "* The pairset configuration is overridden "
                "from 4-pair to Alternative A due to the "
                "incompatible legacy detect configuration." )

def populateEachPoePort( intf ):
   portName = intf
   portPresent = False
   portCapacity = None
   pseEnabled = None
   portPairset = None
   portPriority = None
   lldpEnabled = None
   legacyDetectEnabled = None
   powerLimit = None
   grantedPower = None
   portState = None
   pdClass = None
   power = None
   current = None
   voltage = None
   temperature = None
   misconfigured = False

   if intf in poeStatus and poeStatus[ intf ]:
      status = poeStatus[ intf ]
      config = poeCliConfig.newPoeCliConfig( intf )
      portPresent = True
      portCapacity = status.capacity
      pdClass = status.portStatus.pdClass
      if pdClass == "classUnknown":
         pdClass = "unknown"
      if ( status.portStatus.psePoweringStatus == "FourPairPoweringDualSignaturePse"
           and pdClass.startswith( 'class' ) ):
         pdClass = pdClass + 'D'
      pseEnabled = config.poeEnabled
      if status.capacity <= 30.0:
         portPairset = "altA"
      else:
         portPairset = status.portPairset
         if portPairset != config.portPairset:
            misconfigured = True
      portPriority = status.portPriority
      if portPriority == "defaultPriority":
         portPriority = "low"
      lldpEnabled = config.lldpEnabled
      legacyDetectEnabled = config.legacyDetect
      if config.powerLimit == DEFAULT_POE_POWER_LIMIT:
         if pdClass in classMapping:
            powerLimit = classMapping[ pdClass ]
         else:
            pdClass = "invalid"
            powerLimit = 0.0
      else:
         powerLimit = config.powerLimit
      grantedPower = status.pseApprovedPower
      portState = status.portStatus.poePortState
      if config.poeEnabled and portState == "disabled":
         if config.linkDownAction == "powerOff":
            intfStatus = ethIntfStatus.intfStatus.get( intf )
            if intfStatus and intfStatus.linkStatus == "linkDown":
               portState = "linkDown"
         # adminShutdown status will take priority over linkDown status
         if ( config.intfShutdownAction == 'powerOff' or
              ( config.intfShutdownAction == 'unset' and
                poeCliConfig.intfShutdownAction == 'powerOff' ) ):
            intfConfig = intfCliConfig.intfConfig.get( intf )
            if intfConfig and intfConfig.enabledState == "shutdown":
               portState = "adminShutdown"
      power = status.outputPower
      current = status.outputCurrent * 1000
      voltage = status.outputVoltage
      temperature = status.temperature

   return PoePort( portName=portName, portPresent=portPresent,
                   portCapacity=portCapacity, pseEnabled=pseEnabled,
                   portPairset=portPairset, portPriority=portPriority,
                   lldpEnabled=lldpEnabled, legacyDetectEnabled=legacyDetectEnabled,
                   powerLimit=powerLimit, grantedPower=grantedPower,
                   portState=portState, pdClass=pdClass, power=power,
                   current=current, voltage=voltage, temperature=temperature,
                   misconfigured=misconfigured )

#------------------------------------------------------------------------------------
# show poe [ interface INTF ]
#------------------------------------------------------------------------------------
class ShowPoe( ShowCommand.ShowCliCommandClass ):
   syntax = 'show poe [ interface INTF ]'
   data = {
      'poe' : nodePoe,
      'interface' : 'Interface',
      'INTF' : IntfCli.Intf.rangeMatcher,
   }
   privileged = True
   cliModel = PoePorts

   @staticmethod
   def handler( mode, args ):
      if 'INTF' in args:
         intfs = IntfCli.Intf.getAll( mode, args[ 'INTF' ],
                                      intfType=EthIntfCli.EthIntf )
      else:
         intfs = poeStatus.values()
      poePorts = { i.name : populateEachPoePort( i.name ) for i in intfs }
      return PoePorts( poePorts=poePorts )

BasicCli.addShowCommandClass( ShowPoe )

def _showTechPoeCmds():
   cmds = []
   if not poeGuard( None, None ):
      cmds += [ 'show poe' ]
   return cmds

CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback( '2019-09-19 16:10:00',
                                                             _showTechPoeCmds )

#------------------------------------------------------------------------------------
# Common
#------------------------------------------------------------------------------------
class PlutoDevice( CliModel.Model ):
   __public__ = False
   debugInfo = CliModel.Str( help='Debug information from device' )

def queryDevice( mode, command ):
   def f( name, dirName ):
      outputBuff = cStringIO.StringIO()
      AgentCommandRequest.runSocketCommand( mode.entityManager,
                                            dirName, name,
                                            command, stringBuff=outputBuff )
      return PlutoDevice( debugInfo=outputBuff.getvalue() )
   return f

class PlutoDeviceContainer( CliModel.Model ):
   __public__ = False
   devices = CliModel.Dict( valueType=PlutoDevice,
                            help='Maps a device name to debug information about'
                            ' its components' )

   def render( self ):
      for deviceName in sorted( self.devices ):
         device = self.devices[ deviceName ]
         if device.debugInfo:
            print( deviceName )
            print( device.debugInfo, )

def dictMap( f, names ):
   container = PlutoDeviceContainer()
   devices = container.devices
   sysname = names.entityManager_.sysname()
   for name in names:
      if AgentDirectory.agentIsRunning( sysname, 'PLDevice-' + name ):
         devices[ name ] = f( name, PLDeviceAgent.name + '-' + name )
   if AgentDirectory.agentIsRunning( sysname, 'PLSystem' ):
      devices[ 'system' ] = f( 'system', PLSystemAgent.name )

   return container

#------------------------------------------------------------------------------------
# show platform pluto COMPONENT
#------------------------------------------------------------------------------------
class ShowPlatformPluto( ShowCommand.ShowCliCommandClass ):
   syntax = 'show platform pluto COMPONENT'
   data = {
      'platform' : platformMatcherForShow,
      'pluto' : 'Components managed with Pluto agents',
      'COMPONENT' : CliMatcher.EnumMatcher( {
         'all' : 'Information about all Pluto components',
         'fans' : 'Fan debug information',
         'poe-ports' : 'PoE debug information',
         'power-supplies' : 'Power supply debug information',
         'system' : 'System debug information',
         'temperature-sensors' : 'Temperature sensor debug information',
      } )
   }
   privileged = True
   cliModel = PlutoDeviceContainer
   tokenMap_ = {
      'all' : 'all',
      'fans' : 'fan',
      'poe-ports' : 'poe',
      'power-supplies' : 'power',
      'system' : 'system',
      'temperature-sensors' : 'tempSensor',
   }

   @staticmethod
   def handler( mode, args ):
      command = ShowPlatformPluto.tokenMap_[ args[ 'COMPONENT' ] ]
      return dictMap( queryDevice( mode, command ), sliceDir )

BasicCli.addShowCommandClass( ShowPlatformPluto )

#------------------------------------------------------------------------------------
# poe config mode
#------------------------------------------------------------------------------------
class PoeMode( CliMode.Poe.PoeMode, BasicCli.ConfigModeBase ):
   name = 'PoE configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      CliMode.Poe.PoeMode.__init__( self )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#------------------------------------------------------------------------------------
# poe linecard config mode
#------------------------------------------------------------------------------------
class PoeLinecardMode( CliMode.Poe.PoeLinecardMode, BasicCli.ConfigModeBase ):
   name = 'PoE linecard configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, param ):
      CliMode.Poe.PoeLinecardMode.__init__( self, param )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

#-----------------------------------------------------------------------------------
# [ no | default ] poe
# in global config mode
#-----------------------------------------------------------------------------------
class Poe( CliCommand.CliCommandClass ):
   syntax = 'poe'
   noOrDefaultSyntax = 'poe'
   data = {
      'poe': nodePoe,
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PoeMode )
      mode.session_.gotoChildMode( childMode )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      poeCliConfig.rebootAction = DEFAULT_REBOOT_ACTION

BasicCli.GlobalConfigMode.addCommandClass( Poe )

#------------------------------------------------------------------------------------
# [ no | default ] reboot action ( maintain | power-off )
# in poe config mode
#------------------------------------------------------------------------------------
class RebootActionCmd( CliCommand.CliCommandClass ):
   syntax = 'reboot action ( maintain | power-off )'
   noOrDefaultSyntax = 'reboot action ...'
   data = {
      'reboot': 'Configure behavior on reboot',
      'action': 'Configure the action to take',
      'maintain': 'Maintain PoE power during reboot',
      'power-off': 'Power off PoE power during reboot (default)',
   }

   @staticmethod
   def handler( mode, args ):
      if 'maintain' in args:
         poeCliConfig.rebootAction = 'maintain'
      elif 'power-off' in args:
         poeCliConfig.rebootAction = 'powerOff'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      poeCliConfig.rebootAction = DEFAULT_REBOOT_ACTION

PoeMode.addCommandClass( RebootActionCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] poe reboot action ( maintain | power-off )
# in config-if mode
#-----------------------------------------------------------------------------------
class PoeRebootActionConfigIfCmd( CliCommand.CliCommandClass ):
   syntax = 'poe reboot action ( maintain | power-off )'
   noOrDefaultSyntax = 'poe reboot action ...'
   data = {
         'poe': nodePoe,
         'reboot': 'Configure behavior on reboot',
         'action': 'Configure the action to take',
         'maintain': 'Maintain PoE power during reboot',
         'power-off': 'Power off PoE power during reboot',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      if 'maintain' in args:
         config.rebootAction = 'maintain'
      elif 'power-off' in args:
         config.rebootAction = 'powerOff'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.rebootAction = DEFAULT_INTF_POE_REBOOT_ACTION

if Toggles.PlutoToggleLib.togglePoeRebootActionCliCommandEnabled():
   PoePortModelet.addCommandClass( PoeRebootActionConfigIfCmd )

#------------------------------------------------------------------------------------
# [ no | default ] interface shutdown action ( maintain | power-off )
# in poe config mode
#------------------------------------------------------------------------------------
class InterfaceShutdownActionCmd( CliCommand.CliCommandClass ):
   syntax = 'interface shutdown action ( maintain | power-off )'
   noOrDefaultSyntax = 'interface shutdown action ...'
   data = {
      'interface': 'Interface behavior',
      'shutdown': 'Configure behavior on admin shutdown',
      'action': 'Configure the action to take',
      'maintain': 'Maintain PoE power on interface shutdown',
      'power-off': 'Power off PoE power on interface shutdown',
      }

   @staticmethod
   def handler( mode, args ):
      if 'maintain' in args:
         poeCliConfig.intfShutdownAction = 'maintain'
      elif 'power-off' in args:
         poeCliConfig.intfShutdownAction = 'powerOff'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      poeCliConfig.intfShutdownAction = DEFAULT_INTERFACE_SHUTDOWN_ACTION

PoeMode.addCommandClass( InterfaceShutdownActionCmd )

#-----------------------------------------------------------------------------------
# [ no | default ] poe shutdown action ( maintain | power-off )
# in config-if mode
#-----------------------------------------------------------------------------------
class PoeShutdownActionConfigIfCmd( CliCommand.CliCommandClass ):
   syntax = 'poe shutdown action ( maintain | power-off )'
   noOrDefaultSyntax = 'poe shutdown action ...'
   data = {
         'poe': nodePoe,
         'shutdown': 'Configure behavior on admin shutdown',
         'action': 'Configure the action to take',
         'maintain': 'Maintain PoE power on interface shutdown',
         'power-off': 'Power off PoE power on interface shutdown',
         }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      if 'maintain' in args:
         config.intfShutdownAction = 'maintain'
      elif 'power-off' in args:
         config.intfShutdownAction = 'powerOff'

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.intfShutdownAction = DEFAULT_INTF_POE_SHUTDOWN_ACTION

PoePortModelet.addCommandClass( PoeShutdownActionConfigIfCmd )

#------------------------------------------------------------------------------------
# [ no | default ] poe link down action ( maintain |
#                                         ( power-off [ NUM_SECONDS seconds ] ) )
# in config-if mode
#------------------------------------------------------------------------------------
class PoeLinkDownActionConfigIfCmd( CliCommand.CliCommandClass ):
   syntax = ( 'poe link down action ( maintain | '
              '( power-off [ NUM_SECONDS seconds ] ) )' )
   noOrDefaultSyntax = 'poe link down action ...'
   data = {
         'poe': nodePoe,
         'link': 'Configure link behavior',
         'down': 'Configure behavior on link down',
         'action': 'Configure the action to take',
         'maintain': 'Maintain PoE power during link down',
         'power-off': 'Power off PoE power during link down',
         'NUM_SECONDS': CliMatcher.IntegerMatcher(
                           1, 86400,
                           helpdesc=( 'Number in seconds (default: %d)' %
                                      DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME ) ),
         'seconds': 'Unit seconds',
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      if 'maintain' in args:
         config.linkDownAction = 'maintain'
      elif 'power-off' in args:
         config.linkDownAction = 'powerOff'
         config.linkDownPoeOffTime = args.get(
                                        'NUM_SECONDS',
                                        DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newPoeCliConfig( mode.intf.name )
      config.linkDownAction = DEFAULT_INTF_POE_LINK_DOWN_ACTION
      config.linkDownPoeOffTime = DEFAULT_INTF_POE_LINK_DOWN_POWER_OFF_TIME

if Toggles.PlutoToggleLib.togglePoeLinkDownActionCliCommandEnabled():
   PoePortModelet.addCommandClass( PoeLinkDownActionConfigIfCmd )

#------------------------------------------------------------------------------------
# [ no | default ] power budget allocation ( class-based | usage-based )
#------------------------------------------------------------------------------------
def guardPoeTec( mode, token ):
   poeConfig = sliceDir.get( 'PoeDevice', poeTecConfig )
   if poeConfig is poeTecConfig:
      return None
   return CliParser.guardNotThisPlatform

nodePower = CliCommand.guardedKeyword(
                  'power',
                  helpdesc='Configure PoE power behavior',
                  guard=guardPoeTec )

class PowerBudgetAllocationCmd( CliCommand.CliCommandClass ):
   syntax = 'power budget allocation ( class-based | usage-based )'
   noOrDefaultSyntax = 'power budget allocation ...'
   data = {
         'power': nodePower,
         'budget': 'Configure PoE power budget',
         'allocation': 'Configure how PoE power is allocated',
         'class-based': 'Use classified power limit to determine available power',
         'usage-based': 'Use real-time power usage to determine available power',
   }

   @staticmethod
   def handler( mode, args ):
      poeCliConfig.dynamicPoeEnabled = 'usage-based' in args

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      poeCliConfig.dynamicPoeEnabled = False

if Toggles.PlutoToggleLib.toggleDynamicPoeConfigEnabled():
   PoeMode.addCommandClass( PowerBudgetAllocationCmd )

#------------------------------------------------------------------------------------
# [ no | default ] module MODNAME
# in poe config mode
#------------------------------------------------------------------------------------
class LinecardExpressionFactory( FruCli.SlotExpressionFactory ):

   def generate( self, name ):
      expression_ = name
      data_ = { name : FruCli.SlotExpressionFactory.slotMatcher_ }
      names = []
      k = 'Linecard'
      kwName = name + '_' + k.lower()
      slotName = name + '_' + k.upper()
      expression_ += '| ( %s %s )' % ( kwName, slotName )
      data_[ kwName ] = CliCommand.guardedKeyword(
         k, "%s modules" % k, FruCli.cardTypeGuard )
      data_[ slotName ] = FruCli.SlotMatcher( forceTags=[ k ] )
      names.append( slotName )

      class _SlotExpression( CliCommand.CliExpression ):
         expression = expression_
         data = data_
         slotNames_ = names

         @staticmethod
         def adapter( mode, args, argsList ):
            for n in _SlotExpression.slotNames_:
               val = args.pop( n, None )
               if val is not None:
                  if isinstance( val, list ):
                     # the expression is used in a list
                     args.setdefault( name, [] )
                     args[ name ].extend( val )
                  else:
                     args[ name ] = val
                     # only one arg is possible, just stop
                     break
      return _SlotExpression

class EnterModuleLinecardCmd( CliCommand.CliCommandClass ):
   syntax = 'module MODNAME'
   noOrDefaultSyntax = 'module MODNAME'
   data = {
      'module': ModuleCli.moduleNode,
      'MODNAME': LinecardExpressionFactory(),
   }

   @staticmethod
   def handler( mode, args ):
      childMode = mode.childMode( PoeLinecardMode, param=args[ 'MODNAME' ].label )
      mode.session_.gotoChildMode( childMode )
      poeCliConfig.newLinecardPortPriority( str( args[ 'MODNAME' ].label ) )

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      del poeCliConfig.linecardPortPriority[ str( args[ 'MODNAME' ].label ) ]

if Toggles.PlutoToggleLib.togglePoeLinecardConfigModeEnabled():
   PoeMode.addCommandClass( EnterModuleLinecardCmd )

#------------------------------------------------------------------------------------
# [no|default] priority (low|medium|high|critical) command,
# in "config-poelinecard" mode
#------------------------------------------------------------------------------------
class PoeLinecardPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'priority PRIORITY'
   noOrDefaultSyntax = 'priority ...'
   data = {
      'priority': 'Port priority',
      'PRIORITY': matcherPriorityEnum,
   }

   @staticmethod
   def handler( mode, args ):
      config = poeCliConfig.newLinecardPortPriority( str( mode.number ) )
      config.linecardPortPriority = args[ 'PRIORITY' ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config = poeCliConfig.newLinecardPortPriority( str( mode.number ) )
      config.linecardPortPriority = DEFAULT_POE_PRIORITY

if Toggles.PlutoToggleLib.togglePoeLinecardConfigModeEnabled():
   PoeLinecardMode.addCommandClass( PoeLinecardPriorityCmd )

#------------------------------------------------------------------------------------
# Support for default interface command
#------------------------------------------------------------------------------------
class PoeIntfCleaner( IntfCli.IntfDependentBase ):
   def setDefault( self ):
      if not poeCliConfig or not poeCliConfig.poeCliConfig:
         return
      if self.intf_.name in poeCliConfig.poeCliConfig:
         del poeCliConfig.poeCliConfig[ self.intf_.name ]

IntfCli.Intf.registerDependentClass( PoeIntfCleaner )

#------------------------------------------------------------------------------------
# Reload Cli hook for poe reboot action processing before reboot
#------------------------------------------------------------------------------------
def _verifyPoePortStateStatus( intfs, setting ):
   def _checkPoePortState( intfs, setting ):
      for intf in intfs:
         if intf in poeStatus and poeStatus[ intf ]:
            portState = poeStatus[ intf ].portStatus.poePortState
            if portState != setting:
               return False
      return True

   try:
      Tac.waitFor( lambda: _checkPoePortState( intfs, setting ),
                   timeout=30, warnAfter=5.0, maxDelay=5,
                   sleep=not Tac.activityManager.inExecTime.isZero,
                   description='Processing poe reboot action configurations' )
   except Tac.Timeout:
      return False
   return True

def _verifyGlobalPoeRebootActionStatus( setting ):
   if "PoeStatus" in globalPoeStatus and globalPoeStatus[ 'PoeStatus' ]:
      try:
         Tac.waitFor( lambda: globalPoeStatus[ 'PoeStatus' ].rebootAction == setting,
                      timeout=30, warnAfter=5.0, maxDelay=5,
                      sleep=not Tac.activityManager.inExecTime.isZero,
                      description='Processing poe reboot action configurations' )
      except Tac.Timeout:
         return False
   return True

def doRebootActionBeforeReload( mode, power, now ):
   # If there are no poe ports, don't need to do anything
   if not poeStatus or not poeStatus.entryState:
      return True

   if not _verifyGlobalPoeRebootActionStatus( poeCliConfig.rebootAction ):
      return False

   intfsDisabled = []
   # Make a copy of the global config because it may be modified in
   # the loop
   globalRebootActionConfig = poeCliConfig.rebootAction
   for intf in poeStatus:
      config = poeCliConfig.newPoeCliConfig( intf )
      if globalRebootActionConfig == 'maintain':
         if config.rebootAction == 'powerOff':
            config.poeEnabled = False
            intfsDisabled.append( intf )
      elif globalRebootActionConfig == 'powerOff':
         if config.rebootAction == 'unset' or config.rebootAction == 'powerOff':
            config.poeEnabled = False
            intfsDisabled.append( intf )
         elif config.rebootAction == 'maintain':
            # Set the global register to maintain power, so that ports set to
            # "maintain" stay powered on, this is temporary and will be set to
            # "powerOff" after reload
            if poeCliConfig.rebootAction != 'maintain':
               poeCliConfig.rebootAction = 'maintain'
               if not _verifyGlobalPoeRebootActionStatus( 'maintain' ):
                  return False

   # Only need to check on maintain, since if global config is powerOff then the
   # hardware register is set to power off, and will power off regardless of software
   # settings
   if poeCliConfig.rebootAction == 'maintain':
      if not _verifyPoePortStateStatus( intfsDisabled, 'disabled' ):
         return False
   return True

if Toggles.PlutoToggleLib.togglePoeRebootActionCliCommandEnabled():
   ReloadCli.registerReloadHook( doRebootActionBeforeReload,
                                 'ReloadPoeRebootActionCli',
                                 'RUN_AFTER_CONFIRMATION' )

#------------------------------------------------------------------------------------
# Register debug cmd into 'show tech-support'
#------------------------------------------------------------------------------------
CliPlugin.TechSupportCli.registerShowTechSupportCmdCallback(
                                '2014-10-31 14:14:00',
                                lambda: [ 'show platform pluto all' ] )

#------------------------------------------------------------------------------------
# Register Cli Plugin
#------------------------------------------------------------------------------------
def Plugin( em ):
   global sliceDir
   global poeTecConfig
   global globalPoeStatus
   global poeStatus
   global ethIntfStatus
   global intfCliConfig
   global poeCliConfig
   sliceDir = LazyMount.mount( em,
         'hardware/platform/device/config/slice', 'Tac::Dir', 'ri' )
   poeTecConfig = LazyMount.mount( em,
         'hardware/poe/config/poetec', 'Hardware::PoeTec::PoeConfig', 'r' )
   globalPoeStatus = LazyMount.mount( em,
         'environment/power/status/poe', 'Tac::Dir', 'ri' )
   poeStatus = LazyMount.mount( em,
         'environment/power/status/poePort', 'Tac::Dir', 'ri' )
   ethIntfStatus = LazyMount.mount( em,
         'interface/status/eth/intf', 'Interface::EthIntfStatusDir', 'r' )
   intfCliConfig = LazyMount.mount( em,
         'interface/config/all', 'Interface::AllIntfConfigDir', 'r' )
   poeCliConfig = ConfigMount.mount( em,
         'hardware/poe/config/cli', 'Hardware::PLPoePort::CliConfig', 'w' )
