#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import CliCommand
import CliMatcher
import CliPlugin.PimCliLib as PimCliLib
import CliPlugin.PimsmCliConfig as PimsmCliConfig
from CliPlugin.AclCli import standardIpAclNameMatcher
from CliPlugin.IpAddrMatcher import IpAddrMatcher
import Toggles.PimToggleLib
from CliToken.Ip import ipMatcherForConfig
from CliToken.Pim import (
      fastRerouteMatcher,
      pimNode,
      sparseModeMatcher,
      ssmMatcher,
      ssmRangeMatcher,
      thresholdMatcher,
)

matcherSgExpiryTimer = CliMatcher.KeywordMatcher( 'sg-expiry-timer',
      helpdesc='Protocol keepalive timer' )

#--------------------------------------------------------------------------------
# Begin RouterPimSparseAfSharedModelet.
#--------------------------------------------------------------------------------
# ( no | default ) fast-reroute ...
#--------------------------------------------------------------------------------
class FastRerouteCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'fast-reroute ...'
   data = {
      'fast-reroute': fastRerouteMatcher,
   }

   noOrDefaultHandler = PimsmCliConfig.noIpPimsmFrrFilter

PimCliLib.RouterPimSparseAfSharedModelet.addCommandClass( FastRerouteCmd )

#--------------------------------------------------------------------------------
# ( no | default ) ssm range ...
#--------------------------------------------------------------------------------
class SsmRangeCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ssm range ...'
   data = {
      'ssm': ssmMatcher,
      'range': ssmRangeMatcher,
   }

   noOrDefaultHandler = PimsmCliConfig.noIpPimsmSsmFilter

PimCliLib.RouterPimSparseAfSharedModelet.addCommandClass( SsmRangeCmd )
#--------------------------------------------------------------------------------
# End RouterPimSparseAfSharedModelet.
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# Begin RouterPimSparseSharedModelet.
#--------------------------------------------------------------------------------
# [ no | default ] ip pim sparse-mode sg-expiry-timer EXPIRY
#--------------------------------------------------------------------------------
class IpPimSparseModeSgExpiryTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim sparse-mode sg-expiry-timer EXPIRY'
   noOrDefaultSyntax = 'ip pim sparse-mode sg-expiry-timer ...'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'sparse-mode': sparseModeMatcher,
      'sg-expiry-timer': matcherSgExpiryTimer,
      'EXPIRY': CliMatcher.IntegerMatcher( 120, 259200,
                                           helpdesc='Keepalive timer expiry' ),
   }

   handler = PimsmCliConfig.setPimsmSgExpiry
   noOrDefaultHandler = handler

PimCliLib.RouterPimSparseSharedModelet.addCommandClass(
      IpPimSparseModeSgExpiryTimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip pim ssm range ( standard | ACL )
#--------------------------------------------------------------------------------
class IpPimSsmRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim ssm range ( standard | ACL )'
   noOrDefaultSyntax = 'ip pim ssm range ...'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'ssm': CliCommand.Node(
                  matcher=ssmMatcher,
                  deprecatedByCmd='ssm in ipv4 or ipv6 submode under '
                                  'router pim sparsemode' ),
      'range': ssmRangeMatcher,
      'standard': 'Sets the SSM range to 232/8',
      'ACL': standardIpAclNameMatcher,
   }

   handler = PimsmCliConfig.setIpPimsmSsmFilter
   noOrDefaultHandler = PimsmCliConfig.noIpPimsmSsmFilter

PimCliLib.RouterPimSparseSharedModelet.addCommandClass( IpPimSsmRangeCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip pim spt-threshold THRESHOLD [ group-list ACL ]
#--------------------------------------------------------------------------------
class IpPimSptThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim spt-threshold THRESHOLD [ group-list ACL ]'
   noOrDefaultSyntax = 'ip pim spt-threshold [ THRESHOLD [ group-list ACL ] ]'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'spt-threshold': CliCommand.Node(
                           matcher=CliMatcher.KeywordMatcher( 'spt-threshold',
                                helpdesc='Shortest path tree switchover threshold' ),
                           deprecatedByCmd='spt threshold in router pim '
                                           'sparse-mode ipv4 submode' ),
      'THRESHOLD': thresholdMatcher,
      'group-list': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'group-list',
                             helpdesc='ACL specifying groups threshold applies to' ),
                        deprecatedByCmd='spt threshold match list in router pim '
                                        'sparse-mode ipv4 submode' ),
      'ACL': standardIpAclNameMatcher,
   }

   handler = PimsmCliConfig.setPimsmSptThresh
   noOrDefaultHandler = PimsmCliConfig.noPimsmSptThresh

PimCliLib.RouterPimSparseSharedModelet.addCommandClass( IpPimSptThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip pim ssm default source GROUP SOURCE
#--------------------------------------------------------------------------------
class IpPimSsmDefaultSourceGroupSourceCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim ssm default source GROUP SOURCE'
   noOrDefaultSyntax = 'ip pim ssm default source [ GROUP [ SOURCE ] ]'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'ssm': CliCommand.Node(
                 matcher=ssmMatcher,
                 deprecatedByCmd='ssm in ipv4 or ipv6 submode under '
                                 'router pim sparsemode' ),
      'default': 'Source-Specific Multicast default source',
      'source': 'Source-Specific Multicast default source',
      'GROUP': IpAddrMatcher( helpdesc='Multicast group address' ),
      'SOURCE': IpAddrMatcher( helpdesc='IPv4 address used as a source' ),
   }

   handler = PimsmCliConfig.setIpPimSsmConvert
   noOrDefaultHandler = PimsmCliConfig.noIpPimSsmConvert

PimCliLib.RouterPimSparseSharedModelet.addCommandClass(
      IpPimSsmDefaultSourceGroupSourceCmd )

#--------------------------------------------------------------------------------
# ip pim sparse-mode fast-reroute ACL
#--------------------------------------------------------------------------------
class IpPimSparseModeFastRerouteAclCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim sparse-mode fast-reroute ACL'
   noOrDefaultSyntax = 'ip pim sparse-mode fast-reroute ...'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'sparse-mode': sparseModeMatcher,
      'fast-reroute': fastRerouteMatcher,
      'ACL': standardIpAclNameMatcher,
   }

   handler = PimsmCliConfig.setIpPimsmFrrFilter
   noOrDefaultHandler = PimsmCliConfig.noIpPimsmFrrFilter

PimCliLib.RouterPimSparseSharedModelet.addCommandClass(
      IpPimSparseModeFastRerouteAclCmd )

#--------------------------------------------------------------------------------
# End RouterPimSparseSharedModelet.
#--------------------------------------------------------------------------------

#--------------------------------------------------------------------------------
# Begin RouterPimSparseAfCommonSharedModelet.
#--------------------------------------------------------------------------------
# [ no | default ] sg-expiry-timer EXPIRY
#--------------------------------------------------------------------------------
class SgExpiryTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'sg-expiry-timer EXPIRY'
   noOrDefaultSyntax = 'sg-expiry-timer ...'
   data = {
      'sg-expiry-timer': matcherSgExpiryTimer,
      'EXPIRY': CliMatcher.IntegerMatcher( 120, 259200,
                                           helpdesc='Keepalive timer expiry' ),
   }

   handler = PimsmCliConfig.setPimsmSgExpiry
   noOrDefaultHandler = handler

PimCliLib.RouterPimSparseAfCommonSharedModelet.addCommandClass( SgExpiryTimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] spt threshold
#--------------------------------------------------------------------------------
class SptThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'spt threshold THRESHOLD'
   noOrDefaultSyntax = 'spt threshold'
   data = {
      'spt': 'Shortest path tree',
      'threshold': 'Switchover threshold',
      'THRESHOLD': thresholdMatcher,
   }

   handler = PimsmCliConfig.setPimsmSptThresh
   noOrDefaultHandler = PimsmCliConfig.noPimsmSptThresh

PimCliLib.RouterPimSparseAfCommonSharedModelet.addCommandClass( SptThresholdCmd )

#--------------------------------------------------------------------------------
# make-before-break [ disabled ]
#--------------------------------------------------------------------------------
class MakeBeforeBreakCmd( CliCommand.CliCommandClass ):
   syntax = 'make-before-break [ disabled ]'
   data = {
      'make-before-break': 'Make-before-break option for IIF switching',
      'disabled': 'Disabled token',
   }

   handler = PimsmCliConfig.setMbb

PimCliLib.RouterPimSparseAfCommonSharedModelet.addCommandClass( MakeBeforeBreakCmd )

#------------------------------------------------------------------------------
# [ no | default ] route sg installation threshold [ join | traffic ]
#------------------------------------------------------------------------------
class RouteSgInstallationThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'route sg installation threshold CRITERIA'
   noOrDefaultSyntax = 'route sg installation threshold ...'
   data = {
      'route': 'Sparse mode PIM route',
      'sg': '(S,G) route',
      'installation': 'Install in the hardware',
      'threshold': 'Installation threshold',
      'CRITERIA': CliMatcher.EnumMatcher( {
         'join': 'On receiving join',
         'traffic': 'On receiving traffic',
       } ),
   }

   handler = PimsmCliConfig.setRouteSgInstallThresh
   noOrDefaultHandler = handler

if Toggles.PimToggleLib.togglePimSgInstallOnTrafficEnabled():
   PimCliLib.RouterPimSparseIpv4Mode.addCommandClass(
      RouteSgInstallationThresholdCmd )

#--------------------------------------------------------------------------------
# End RouterPimSparseAfCommonSharedModelet.
#--------------------------------------------------------------------------------
