#!/usr/bin/env python
# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from __future__ import absolute_import, division, print_function

import BasicCliModes
import CliCommand
import CliMatcher
import CliPlugin.PimsmCliConfig as PimsmCliConfig
from CliPlugin.AclCli import standardIpAclNameMatcher
from CliToken.Ip import ipMatcherForConfig
from CliToken.Pim import (
      fastRerouteMatcher,
      pimNode,
      sparseModeMatcher,
      ssmMatcher,
      ssmRangeMatcher,
      thresholdMatcher,
)

#--------------------------------------------------------------------------------
# [ no | default ] ip pim sparse-mode fast-reroute ACLNAME
#--------------------------------------------------------------------------------
class IpPimSparseModeFastRerouteAclCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim sparse-mode fast-reroute ACL'
   noOrDefaultSyntax = 'ip pim sparse-mode fast-reroute ...'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'sparse-mode': sparseModeMatcher,
      'fast-reroute': fastRerouteMatcher,
      'ACL': standardIpAclNameMatcher,
   }

   handler = PimsmCliConfig.setIpPimsmFrrFilter
   noOrDefaultHandler = PimsmCliConfig.noIpPimsmFrrFilter

BasicCliModes.GlobalConfigMode.addCommandClass( IpPimSparseModeFastRerouteAclCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip pim sparse-mode sg-expiry-timer EXPIRY
#--------------------------------------------------------------------------------
class IpPimSparseModeSgExpiryTimerCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim sparse-mode sg-expiry-timer EXPIRY'
   noOrDefaultSyntax = 'ip pim sparse-mode sg-expiry-timer ...'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'sparse-mode': sparseModeMatcher,
      'sg-expiry-timer': 'Protocol keepalive timer',
      'EXPIRY': CliMatcher.IntegerMatcher( 120, 259200,
                                           helpdesc='Keepalive timer expiry' ),
   }

   handler = PimsmCliConfig.setPimsmSgExpiry
   noOrDefaultHandler = handler

BasicCliModes.GlobalConfigMode.addCommandClass( IpPimSparseModeSgExpiryTimerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip pim spt-threshold THRESHOLD [ group-list ACL ]
#--------------------------------------------------------------------------------
class IpPimSptThresholdCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim spt-threshold THRESHOLD [ group-list ACL ]'
   noOrDefaultSyntax = 'ip pim spt-threshold [ THRESHOLD [ group-list ACL ] ]'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'spt-threshold': CliCommand.Node(
                           matcher=CliMatcher.KeywordMatcher( 'spt-threshold',
                                helpdesc='Shortest path tree switchover threshold' ),
                           deprecatedByCmd='spt threshold in router pim '
                                           'sparse-mode ipv4 submode' ),
      'THRESHOLD': thresholdMatcher,
      'group-list': CliCommand.Node(
                        matcher=CliMatcher.KeywordMatcher( 'group-list',
                             helpdesc='ACL specifying groups threshold applies to' ),
                        deprecatedByCmd='spt threshold match list in router pim '
                                        'sparse-mode ipv4 submode' ),
      'ACL': standardIpAclNameMatcher,
   }

   handler = PimsmCliConfig.setPimsmSptThresh
   noOrDefaultHandler = PimsmCliConfig.noPimsmSptThresh

BasicCliModes.GlobalConfigMode.addCommandClass( IpPimSptThresholdCmd )

#--------------------------------------------------------------------------------
# [ no | default ] ip pim ssm range ( standard | ACL )
#--------------------------------------------------------------------------------
class IpPimSsmRangeCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim ssm range ( standard | ACL )'
   noOrDefaultSyntax = 'ip pim ssm range ...'
   data = {
      'ip': ipMatcherForConfig,
      'pim': pimNode,
      'ssm': CliCommand.Node(
                 matcher=ssmMatcher,
                 deprecatedByCmd='ssm in ipv4 or ipv6 submode under '
                                 'router pim sparsemode' ),
      'range': ssmRangeMatcher,
      'standard': 'Sets the SSM range to 232/8',
      'ACL': standardIpAclNameMatcher,
   }

   handler = PimsmCliConfig.setIpPimsmSsmFilter
   noOrDefaultHandler = PimsmCliConfig.noIpPimsmSsmFilter

BasicCliModes.GlobalConfigMode.addCommandClass( IpPimSsmRangeCmd )
