#!/usr/bin/env python
# Copyright (c) 2010, 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import LazyMount
import PimCliLib
from CliCommon import AlreadyHandledError
from IpLibConsts import DEFAULT_VRF as vrfDefault
from McastCommonCliLib import validateMulticastRouting
from McastCommonCliLib import AddressFamily
from TypeFuture import TacLazyType

allVrfConfig = None
mfibVrfConfig = None
pimsmClearConfigColl = None
igmpClearConfigColl = None
pimsm6ClearConfigColl = None
mfib6VrfConfig = None

def getVrfNames( mode=None ):
   vrfNames = allVrfConfig.vrf.members()
   return vrfNames

GmpMrouteClearFlags = TacLazyType( 'Routing::Gmp::MrouteClearFlags' )
GmpRoute = TacLazyType( 'Routing::Gmp::ClearMrouteConfig::Route' )
PimMrouteClearFlags = TacLazyType( 'Routing::Pim::MrouteClearFlags' )
PimRoute = TacLazyType( 'Routing::Pim::ClearMrouteConfig::Route' )

IPV4 = AddressFamily.ipv4
IPV6 = AddressFamily.ipv6

def _clearConfigCreation( vrfName ):
   pimsmClearConfigColl.vrfClearMrouteConfig.newMember( vrfName )
   pimsm6ClearConfigColl.vrfClearMrouteConfig.newMember( vrfName )
   igmpClearConfigColl.vrfClearMrouteConfig.newMember( vrfName )

def _clearConfigDeletion( vrfName ):
   if vrfName != vrfDefault:
      del pimsmClearConfigColl.vrfClearMrouteConfig[ vrfName ]
      del pimsm6ClearConfigColl.vrfClearMrouteConfig[ vrfName ]
      del igmpClearConfigColl.vrfClearMrouteConfig[ vrfName ]

def clearPimMessageCounters( vrfName, af=IPV4 ):
   configColl = pimsmClearConfigColl if af == IPV4 else pimsm6ClearConfigColl
   if vrfName in configColl.vrfClearMrouteConfig:
      getPimsmClearConfig( vrfName, af=af ).countersCount += 1

def getPimsmClearConfig( vrfName, af=IPV4 ):
   if af == IPV4 :
      return pimsmClearConfigColl.vrfClearMrouteConfig.get( vrfName )
   elif af == IPV6:
      return pimsm6ClearConfigColl.vrfClearMrouteConfig.get( vrfName )

#------------------------------------------------------------------------------------
# Legacy:
# clear ip mroute:
#   clear ip mroute *
#   clear ip mroute S
#   clear ip mroute G
#   clear ip mroute G S
#------------------------------------------------------------------------------------
# clear pim [ipv4|ipv6] [<vrf>] sparse-mode [*|S|G|G S]
#------------------------------------------------------------------------------------
def getPimClearConfigOrStop( mode, args ):
   '''Get the PIM clear config based on VRF and address family.
   Fail gracefully if we have no PIM clear config or VRF/AF combo isn't valid.
   '''
   vrfName = args.get( 'VRF', vrfDefault )
   if 'ipv6' in args:
      af = IPV6
      clearConfigColl = pimsm6ClearConfigColl
   else:
      af = IPV4
      clearConfigColl = pimsmClearConfigColl

   if vrfName != vrfDefault and not validateMulticastRouting( mode, vrfName, af ):
      raise AlreadyHandledError

   pimsmClearConfig = clearConfigColl.vrfClearMrouteConfig.get( vrfName )
   if not pimsmClearConfig:
      raise AlreadyHandledError

   return pimsmClearConfig

def doClearPimRoute( mode, args ):
   '''Handler for CLI commands:
   `clear ( ( ip mroute [ vrf VRF ] sparse-mode )
          | ( pim ipv4 [ vrf VRF ] sparse-mode route ) ) ( * | { ADDRS } )`
   `clear pim ipv6 [ vrf VRF ] sparse-mode route ( * | { ADDRS } )`
   '''
   pimsmClearConfig = getPimClearConfigOrStop( mode, args )
   info = PimRoute()
   flags = PimMrouteClearFlags()

   if '*' in args:
      flags.clearAll = True
   else:
      addrs = args[ 'ADDRS' ]
      groupOrSource = str( addrs[ 0 ] )
      secondAddr = str( addrs[ 1 ] ) if len( addrs ) == 2 else None
      try:
         source, group = PimCliLib.ipPimParseSg( groupOrSource, secondAddr )
      except ValueError:
         mode.addErrorAndStop( "Must enter a multicast group or unicast source" )

      if source:
         info.sAddr = source
      if group:
         info.gAddr = group

      if source and group:
         flags.clearOne = True
      elif group:
         flags.clearGroup = True
      elif source:
         flags.clearSource = True

   info.flags = flags
   pimsmClearConfig.route = info
   pimsmClearConfig.routesCount = ( pimsmClearConfig.routesCount + 1 ) & 0xff
   pimsmClearConfig.route = PimRoute()

def doClearIpMrouteCommon( mode, vrfName=vrfDefault, clearAll=False,
                           groupOrSource=None, secondAddr=None ):
   if not vrfName :
      vrfName = vrfDefault
   if vrfName != vrfDefault and \
         not PimCliLib.validVrfName( mode, vrfName, mfibVrfConfig ):
      return

   if clearAll:
      doClearIpMrouteCommonAll( vrfName )
   else:
      doClearIpMrouteCommonVrf( mode, vrfName, groupOrSource, secondAddr )

def getPimAndIgmpClearConfigOrStop( vrfName ):
   '''Get the PIM and IGMP clear configs based on VRF.
   Fail gracefully if we have no PIM clear config or VRF/AF combo isn't valid.
   '''
   pimsmClearConfig = pimsmClearConfigColl.vrfClearMrouteConfig.get( vrfName )
   igmpClearConfig = igmpClearConfigColl.vrfClearMrouteConfig.get( vrfName )

   if pimsmClearConfig is None or igmpClearConfig is None:
      raise AlreadyHandledError

   return pimsmClearConfig, igmpClearConfig

def createPimAndIgmpClearConfigFlags():
   '''Get PIM and IGMP clear flags.
   '''
   return ( PimRoute(), PimMrouteClearFlags(), GmpRoute(), GmpMrouteClearFlags() )

def setFlagsToPimAndIgmpConfig( pimsmClearConfig, igmpClearConfig,
                                info, flags, igmpinfo, igmpflags ):
   '''Set PIM and IGMP clear config flags and increment routes counter (U8).
   '''
   info.flags = flags
   igmpinfo.flags = igmpflags
   pimsmClearConfig.route = info
   igmpClearConfig.route = igmpinfo
   newCount = ( pimsmClearConfig.routesCount + 1 ) & 0xff
   pimsmClearConfig.routesCount = newCount
   igmpClearConfig.routesCount = newCount
   pimsmClearConfig.route = PimRoute()
   igmpClearConfig.route = GmpRoute()

def doClearIpMrouteCommonAll( vrfName=vrfDefault ):
   pimsmClearConfig, igmpClearConfig = getPimAndIgmpClearConfigOrStop( vrfName )
   info, flags, igmpinfo, igmpflags = createPimAndIgmpClearConfigFlags()

   flags.clearAll = True
   igmpflags.clearAll = True

   setFlagsToPimAndIgmpConfig( pimsmClearConfig, igmpClearConfig,
                               info, flags, igmpinfo, igmpflags )

def doClearIpMrouteCommonVrf( mode, vrfName, groupOrSource, secondAddr=None ):
   pimsmClearConfig, igmpClearConfig = getPimAndIgmpClearConfigOrStop( vrfName )
   info, flags, igmpinfo, igmpflags = createPimAndIgmpClearConfigFlags()

   try:
      source, group = PimCliLib.ipPimParseSg( groupOrSource, secondAddr )
   except ValueError:
      mode.addErrorAndStop( "Must enter a multicast group or unicast source" )

   if source:
      info.sAddr = source
      igmpinfo.sAddr = source
   if group:
      info.gAddr = group
      igmpinfo.gAddr = group

   if secondAddr:
      flags.clearOne = True
      igmpflags.clearOne = True
   elif group:
      flags.clearGroup = True
      igmpflags.clearGroup = True
   elif source:
      flags.clearSource = True
      igmpflags.clearSource = True

   setFlagsToPimAndIgmpConfig( pimsmClearConfig, igmpClearConfig,
                               info, flags, igmpinfo, igmpflags )

def Plugin( entityManager ):
   global pimsmClearConfigColl
   global pimsm6ClearConfigColl
   global igmpClearConfigColl
   global allVrfConfig
   global mfibVrfConfig
   global mfib6VrfConfig

   pimsmClearConfigColl = LazyMount.mount(
         entityManager, "routing/pim/sparsemode/clear/config",
         "Routing::Pim::ClearMrouteConfigColl", "w" )

   pimsm6ClearConfigColl = LazyMount.mount(
         entityManager, "routing6/pim/sparsemode/clear/config",
         "Routing::Pim::ClearMrouteConfigColl", "w" )

   igmpClearConfigColl = LazyMount.mount(
         entityManager, "routing/gmp/clearMrouteConfig",
         "Routing::Gmp::ClearMrouteConfigColl", "w" )

   allVrfConfig = LazyMount.mount( entityManager,
                                    "ip/vrf/config", "Ip::AllVrfConfig", "w" )

   mfibVrfConfig = LazyMount.mount( entityManager,
         'routing/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   mfib6VrfConfig = LazyMount.mount( entityManager,
         'routing6/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   PimCliLib.pimClearMessageCountersHook.addExtension( clearPimMessageCounters )
   PimCliLib.clearIpMrouteHook.addExtension(
         ( 'sparseMode', doClearIpMrouteCommon ) )
   PimCliLib.pimSparseModeVrfConfiguredHook.addExtension( _clearConfigCreation )
   PimCliLib.pimSparseModeVrfDeletedHook.addExtension( _clearConfigDeletion )
