#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli
import CliMatcher, ShowCommand
from CliToken.Ip import ipMatcherForShow
from CliToken.Pim import pimNodeAfterShow, pimNodeAfterShowIp
import LazyMount
import SharedMem
import PimCliLib
from PimCliLib import getAllIpAddrs, getAllPhysIntfs, RouteTrie, checkForRoute
from McastCommonCliLib import validateRouting, ShowAddressFamilyExpr
from IntfModels import Interface
from ArnetModel import IpGenericAddress
from CliModel import Model, List, Dict
from PimConfigCheckModel import RouteCheck, GenRouteCheck
from PimRegConfigCli import anycastRpKwMatcher
from PimRegShowCli import vrfExprFactory
import CliPlugin.VrfCli as VrfCli
import FibUtils
from RouterMulticastCliLib import doLazyMounts, lazyGetters
from IraCommonCli import AddressFamily
import Tac

_ipConfig = None
_ip6Config = None
_shmemEm = None

PimRegConfigColl = "Routing::PimReg::ConfigColl"

( pimRegConfigColl,
  pimRegConfigCollFromMode,
  pimRegConfigCollVrf,
  pimRegConfigCollVrfFromMode ) = lazyGetters( PimRegConfigColl )

RpConfigColl = "Routing::Pim::Static::RpConfigColl"

( rpConfigColl,
  rpConfigCollFromMode,
  rpConfigCollVrf,
  rpConfigCollVrfFromMode ) = lazyGetters( RpConfigColl )

PimGlobalStatus = "Routing::Pim::GlobalStatus"

( pimGlobalStatus,
  pimGlobalStatusFromMode,
  pimGlobalStatusVrf,
  pimGlobalStatusVrfFromMode ) = lazyGetters( PimGlobalStatus,
                                          collectionName='pimEnabledSparseModeVrf' )

MfibVrfConfig = "Routing::Multicast::Fib::VrfConfig"

( mfibVrfConfig,
  mfibVrfConfigFromMode,
  mfibVrfConfigVrf,
  mfibVrfConfigVrfFromMode ) = lazyGetters( MfibVrfConfig,
                                            collectionName='config' )

class AnycastRPConfigCheckModelBase( Model ):
   notIntfCfgRPs = List( valueType = IpGenericAddress, 
                         help="Anycast RP not configured on any interface" )
   notLoopback = List( valueType = Interface,
                       help="Anycast RP interfaces which are not loopback" )
   downIntfs = List( valueType = Interface, 
                     help = "Anycast RP interfaces which are down" )
   notAnycastRPs = List ( valueType = IpGenericAddress,
                          help = "This RP address is not configured as anycast RP" )
   notCfgRPs = List( valueType = IpGenericAddress,
                     help="Anycast RPs not configured on the router" )
   notRPs = List( valueType = IpGenericAddress,
                     help="Anycast RPs not configured as RP addresses" )

   def render( self ):
      for k in self.notIntfCfgRPs:
         print "Anycast RP address %s is not configured on any interface" % k
      for k in self.notLoopback:
         print "Anycast RP interface %s is not a loopback" % k
      for k in self.downIntfs:
         print "Anycast RP interface %s is down" % k
      for k in self.notAnycastRPs:
         print "This router is not configured as anycast RP for %s " % k
      for k in self.routeCheck:
         k.render()
      for k in self.notCfgRPs:
         print "This router is not configured as anycast RP for %s" % k
      for k in self.notRPs:
         print "Anycast RP address %s is not configured as RP address" % k

class AnycastRPConfigCheckModel( AnycastRPConfigCheckModelBase ):
   routeCheck = List( valueType = RouteCheck,
                      help='route check hints' )

class AnycastRPGenConfigCheckModel( AnycastRPConfigCheckModelBase ):
   routeCheck = List( valueType = GenRouteCheck,
                      help='route check hints' )

class AllVrfAnycastRPConfigCheckModel( Model ):
   vrfModel = Dict( help="A map of VRF name to AnycastRpConfigCheckModel",
         valueType=AnycastRPConfigCheckModel )
   def render( self ):
      for model in self.vrfModel.itervalues():
         model.render()

# Check Anycast-RP config
def anycastRpConfigCheck( mode, ipConfig, pimRegConfig, routingStatus, 
                          fwdStatus, rpConfig, legacy=False ):
   allIpAddrs = getAllIpAddrs( ipConfig )
   allPhysIntfs = getAllPhysIntfs( mode  )
   routeTrie = RouteTrie( routingStatus, fwdStatus )
   if legacy:
      anycastModel = AnycastRPConfigCheckModel()
   else:
      anycastModel = AnycastRPGenConfigCheckModel()
   anycastModel.notIntfCfgRPs = [ ]
   anycastModel.notLoopback = [ ]
   anycastModel.downIntfs = [ ]
   anycastModel.notAnycastRPs = [ ]
   anycastModel.notCfgRPs = [ ]
   anycastModel.routeCheck = [ ]
   anycastModel.notRPs = [ ]


   if pimRegConfig.anycastRpConfig:
      for k, v in pimRegConfig.anycastRpConfig.iteritems( ):
         if k not in allIpAddrs:
            anycastModel.notIntfCfgRPs.append( k )
         else:
            name = allIpAddrs[ k ].intfId
            if not name.startswith( "Loopback" ):
               anycastModel.notLoopback.append( name )
            intf = allPhysIntfs.get( name )
            if intf and not \
               ( intf.config( ).enabled and intf.config( ).adminEnabled
                 and intf.status( ).operStatus == 'intfOperUp' ):
               anycastModel.downIntfs.append( name )
         found = False
         for p in v.peer:
            if p in allIpAddrs:
               found = True
            rcModel = checkForRoute( p, "anycast RP", routeTrie,
                                     {}, allIpAddrs, [ ], legacy=legacy )
            anycastModel.routeCheck.append( rcModel )
         if not found:
            anycastModel.notCfgRPs.append( k )
         if not k in rpConfig.rpTable:
            anycastModel.notRPs.append( k )

   return anycastModel


def getUribStatus( af, vrfName ):
   routeInfo = FibUtils.routeStatusInfo( 'reader' )
   fwdInfo = FibUtils.forwardingStatusInfo( 'reader' )
   if af == AddressFamily.ipv4:
      routeStatusTypeName = "Smash::Fib::RouteStatus"
      forwardingStatusTypeName = "Smash::Fib::ForwardingStatus"
   else:
      routeStatusTypeName = "Smash::Fib6::RouteStatus"
      forwardingStatusTypeName = "Smash::Fib6::ForwardingStatus"

   fibPath = Tac.Type( "Smash::Fib::TableInfo" ).fibMountPath( af, vrfName, "rib" )
   routingStatus = _shmemEm.doMount( fibPath, routeStatusTypeName, routeInfo )

   fecPath = Tac.Type( "Smash::Fib::TableInfo" ).fecMountPath( af, vrfName, "rib" )
   forwardingStatus = _shmemEm.doMount( fecPath, forwardingStatusTypeName, fwdInfo )

   return ( routingStatus, forwardingStatus )

#------------------------------------------------------------------------------------
# show ip pim config-sanity anycast-rp
#------------------------------------------------------------------------------------

configSanityKwMatcher = CliMatcher.KeywordMatcher(
   'config-sanity',
   helpdesc='Show hints of potential PIM problems' )

def anycastConfigSanity( mode, args, legacy=False ):
   af = args.get( 'addressFamily', AddressFamily.ipv4 )
   vrfName = args.get( 'VRF' )
   if not vrfName:
      vrfName = VrfCli.vrfMap.getCliSessVrf( mode.session )

   if not validateRouting( mode, vrfName, af ):
      return None
   if vrfName not in pimGlobalStatus( af ).pimEnabledSparseModeVrf:
      if legacy:
         return AnycastRPConfigCheckModel()
      else:
         return AnycastRPGenConfigCheckModel()

   pimRegConfig = pimRegConfigColl( af ).vrfConfig[ vrfName ]
   rpConfig = rpConfigColl( af, kwargs='sparsemode' ).vrfConfig[ vrfName ]
   routingStatus, forwardingStatus = getUribStatus( af, vrfName )

   ipConfig = _ipConfig if af == AddressFamily.ipv4 else _ip6Config

   return  anycastRpConfigCheck( mode, ipConfig, pimRegConfig, routingStatus,
                                 forwardingStatus, rpConfig, legacy=legacy )

def anycastConfigSanityLegacy( mode, args ):
   return anycastConfigSanity( mode, args, legacy=True )

class ShowAnycastConfigSanityCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show pim AF [ VRF ] config-sanity anycast-rp'
   data = {
      'pim': pimNodeAfterShow,
      'AF': ShowAddressFamilyExpr,
      'VRF': vrfExprFactory,
      'config-sanity': configSanityKwMatcher,
      'anycast-rp': anycastRpKwMatcher
   }
   handler = anycastConfigSanity
   cliModel = AnycastRPGenConfigCheckModel

BasicCli.addShowCommandClass( ShowAnycastConfigSanityCmd )

class ShowAnycastConfigSanityLegacyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] config-sanity anycast-rp'
   data = {
      'ip': ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfExprFactory,
      'config-sanity': configSanityKwMatcher,
      'anycast-rp': anycastRpKwMatcher
   }
   handler = anycastConfigSanityLegacy
   cliModel = AnycastRPConfigCheckModel

BasicCli.addShowCommandClass( ShowAnycastConfigSanityLegacyCmd )

PimCliLib.registerShowPimConfigSanityCmdCallback(
   [ 'show ip pim config-sanity anycast-rp' ] )

def Plugin( entityManager ):
   global _shmemEm, _ipConfig, _ip6Config

   lazyMountTypes = [ PimRegConfigColl, PimGlobalStatus, MfibVrfConfig ]
   doLazyMounts( entityManager, lazyMountTypes )
   doLazyMounts( entityManager, [ RpConfigColl ], kwargs='sparsemode' )

   _ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )
   _ip6Config = LazyMount.mount( entityManager, 'ip6/config', 'Ip6::Config', 'r' )

   _shmemEm = SharedMem.entityManager( sysdbEm=entityManager )
