# Copyright (c) 2010, 2011, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import BasicCli, ConfigMount, IraIpIntfCli, LazyMount
import CliParser, Tac, GmpCli
import CliToken.Ip, CliToken.Pim
import PimCliLib
import PimBidirCliLib
import IraIpCli, IraIp6Cli, IraIp6RouteCliLib
from IraCommonCli import AddressFamily
import IntfCli
from IraIpRouteCliLib import isValidPrefixWithError
import AclCli, AclCliLib, AclLib
from AclCli import getAclConfig
from AclCli import standardIpAclNameMatcher
from Arnet import Ip6AddrWithMask, IpAddr
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
from McastCommonCliLib import getAfFromIpFamilyRule
import McastCommonCliLib
import CliPlugin.ConfigConvert
from PimBidirCliLib import RouterPimBidirSharedModelet
from PimBidirCliLib import RouterPimBidirAfCommonSharedModelet
from PimCliLib import RouterPimSparseSharedModelet, RouterPimSharedModelet
from PimCliLib import RouterPimSparseIpv4Mode, RouterPimSparseIpv6Mode
from PimCliLib import RouterPimSparseAfCommonSharedModelet
from PimBidirCliLib import RouterPimBidirBaseMode, RouterPimBidirVrfMode
from PimBidirCliLib import RouterPimBidirIpv4Mode, RouterPimBidirIpv6Mode
from PimBidirCliLib import isBidirMode
import MrouteCli
import Tracing
import RouterMulticastCliLib
from McastCommonCliLib import mcastRoutingSupported, mcast6RoutingSupported
from RouterMulticastCliLib import configGetters
from RouterMulticastCliLib import getAddressFamilyFromMode
import Toggles.PimToggleLib
import CliMatcher
import CliCommand

traceHandle = Tracing.Handle( "PimConfigCli" )
t0 = traceHandle.trace0

# Globals
_pimConfigRoot = None
_mfibVrfConfig = None
_mfibVrfConfig6 = None
_mrouteLegacyConfig = None
_routingInfoVrfStatus = None
_routing6InfoVrfStatus = None
_routingHwStatus = None
_routing6HwStatus = None
_ipConfig = None
_ipStatus = None
ipGenStatus = None
_allVrfConfig = None
_pimXLegacyConfig = None
_pimLegacyConfig = None
_pimBidirLegacyConfig = None
aclConfig = None
aclCpConfig = None

HELLO_HOLD_FOREVER = 2**16 - 1
JOINPRUNE_HOLD_FOREVER = 2**16 - 1

# Af Independent Config Types
PimConfigColl = "Routing::Pim::ConfigColl"

( pimConfigColl,
  pimConfigCollFromMode,
  pimConfigVrf,
  pimConfigVrfFromMode ) = configGetters( PimConfigColl, collectionName='vrfConfig' )

( bidirConfigColl,
  bidirConfigCollFromMode,
  bidirConfigVrf,
  bidirConfigVrfFromMode ) = configGetters( PimConfigColl,
                                             collectionName='bidirVrfConfig' )

PimRpConfigColl = "Routing::Pim::RpConfigColl"

( pimRpConfigColl,
  pimRpConfigCollFromMode,
  pimRpConfigVrf,
  pimRpConfigVrfFromMode ) = configGetters( PimRpConfigColl,
                                             collectionName='vrfConfig' )

StaticRpConfigColl = "Routing::Pim::Static::RpConfigColl"

( staticRpConfigColl,
  staticRpConfigCollFromMode,
  staticRpConfigVrf,
  staticRpConfigVrfFromMode ) = configGetters( StaticRpConfigColl,
                                             collectionName='vrfConfig' )

# Install a hook for pim interface collection. This way cli keeps track of all
# multicast routing interfaces based on VRF and AF.
def _pimIntfConfColl( vrfName, af ):
   # Return set of all intf on which Pim is configured based on AF.
   allPimIntfs = { }

   _pim6ConfigRoot = pimConfigColl( af=AddressFamily.ipv6 )
   configColl = _pimConfigRoot if af == AddressFamily.ipv4 else _pim6ConfigRoot

   for pimIntf in configColl.intfConfig.keys():
      v = McastCommonCliLib.getVrfNameFromIntf( ipGenStatus, pimIntf,
                                                af=AddressFamily.ipv4 )
      if vrfName == v:
         allPimIntfs[ pimIntf ] = configColl.intfConfig[ pimIntf ]

   return allPimIntfs

def _pimLegacyConfigHook():
   if not ( _mrouteLegacyConfig.version == _mrouteLegacyConfig.globalMode ):
      _pimLegacyConfig.legacy = False
      if _pimLegacyConfig.version != _pimLegacyConfig.ipMode:
         _pimLegacyConfig.version = _pimLegacyConfig.routerMode
   else:
      pimConfig6 = pimConfigColl( af=AddressFamily.ipv6 )
      if len( _pimConfigRoot.vrfConfig.keys() ) == 1 and \
            len( pimConfig6.vrfConfig.keys() ) == 1:
         _pimLegacyConfig.legacy = True
         _pimLegacyConfig.version = _pimLegacyConfig.globalMode
      else:
         _pimLegacyConfig.legacy = False
         if _pimLegacyConfig.version != _pimLegacyConfig.ipMode:
            _pimLegacyConfig.version = _pimLegacyConfig.routerMode

def _cleanupPimBidirConfig( vrfName ):
   t0( "_cleanupPimBidirConfig : %s" % vrfName )
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      _pimConfig = bidirConfigColl( af=af )
      if vrfName in _pimConfig.bidirVrfConfig:
         _pimConfig.bidirVrfConfig[ vrfName ].logNeighborChanges = True

      bidirStatic = staticRpConfigColl( af=af, protocol='bidir' )
      if vrfName in bidirStatic.vrfConfig:
         bidirStatic.vrfConfig[ vrfName ].rpTable.clear()

   for hook in PimCliLib.pimBidirVrfConfigCleanupHook.extensions():
      hook( vrfName )

   for hook in PimCliLib.pimRpCandidateCleanupHook.extensions():
      hook( vrfName, mode='modePimBidir' )

def _cleanupPimSparseModeConfig( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      _pimConfig = pimConfigColl( af=af )
      if vrfName in _pimConfig.vrfConfig:
         _pimConfig.vrfConfig[ vrfName ].logNeighborChanges = True
         _pimConfig.vrfConfig[ vrfName ].allowRp = False
         _pimConfig.vrfConfig[ vrfName ].bfdIntfDefault = False

      sparseStatic = staticRpConfigColl( af=af, protocol='sparsemode' )
      if vrfName in sparseStatic.vrfConfig:
         sparseStatic.vrfConfig[ vrfName ].rpTable.clear()

   for hook in PimCliLib.pimSparseModeCleanupHook.extensions():
      hook( vrfName )

   for hook in PimCliLib.pimRpCandidateCleanupHook.extensions():
      hook( vrfName, mode='modePimSm' )

def _sparseModeVrfDefinitionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigVrf( af=af, vrfName=vrfName )
      afRoot = pimConfigColl( af=af )
      assert vrfName in afRoot.vrfConfig
      staticRpConfigVrf( af=af, vrfName=vrfName, protocol='sparsemode' )
      afStatic = staticRpConfigColl( af=af, protocol='sparsemode' )
      assert vrfName in afStatic.vrfConfig
      pimRpConfigVrf( af=af, vrfName=vrfName, protocol='sparsemode' )
      afRp = pimRpConfigColl( af=af, protocol='sparsemode' )
      assert vrfName in afRp.vrfConfig
   for hook in PimCliLib.pimSparseModeVrfConfiguredHook.extensions():
      hook( vrfName )

def _bidirModeVrfDefinitionHook( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      bidirConfigVrf( af=af, vrfName=vrfName )
      afRoot = bidirConfigColl( af=af )
      assert vrfName in afRoot.bidirVrfConfig
      staticRpConfigVrf( af=af, vrfName=vrfName, protocol='bidir' )
      afStatic = staticRpConfigColl( af=af, protocol='bidir' )
      assert vrfName in afStatic.vrfConfig
      pimRpConfigVrf( af=af, vrfName=vrfName, protocol='bidir' )
      afRp = pimRpConfigColl( af=af, protocol='bidir' )
      assert vrfName in afRp.vrfConfig
   for hook in PimCliLib.pimBidirVrfConfiguredHook.extensions():
      hook( vrfName )

def _sparseModeVrfDeletionHook( vrfName ):
   if vrfName == DEFAULT_VRF:
      return
   canDeleteSparseModeVrfConfig = True

   for hook in PimCliLib.canDeletePimSparseModeVrfHook.extensions():
      output = hook( vrfName )
      t0( "Hook: %s, output : %s" %( hook, output ) )
      if not output:
         canDeleteSparseModeVrfConfig = False

   if not canDeleteSparseModeVrfConfig or vrfName in _mfibVrfConfig.config or \
                                          vrfName in _mfibVrfConfig6.config:
      return

   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = pimConfigColl( af=af )
      staticRpColl = staticRpConfigColl( af=af, protocol='sparsemode' )
      rpConfigColl = pimRpConfigColl( af=af, protocol='sparsemode' )
      if vrfName in pimConfigRoot.vrfConfig:
         pvc = pimConfigVrf( af=af, vrfName=vrfName )
         if not pvc.isDefault() or \
            ( vrfName in staticRpColl.vrfConfig and \
            staticRpColl.vrfConfig[ vrfName ].rpTable ) or \
            ( vrfName in rpConfigColl.vrfConfig and \
            rpConfigColl.vrfConfig[ vrfName ].rpHashAlgorithm != \
            Tac.Type( "Routing::Pim::RpHashAlgorithm" ).rpHashAlgorithmDefault ):
            canDeleteSparseModeVrfConfig = False

   t0( "canDeleteSparseModeVrfConfig: %s" % canDeleteSparseModeVrfConfig )
   if not canDeleteSparseModeVrfConfig or ( vrfName in _mfibVrfConfig.config and \
                                             vrfName in _mfibVrfConfig6.config ):
      return

   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = pimConfigColl( af=af )
      staticRpColl = staticRpConfigColl( af=af, protocol='sparsemode' )
      rpConfigColl = pimRpConfigColl( af=af, protocol='sparsemode' )
      if vrfName in pimConfigRoot.vrfConfig:
         del pimConfigRoot.vrfConfig[ vrfName ]
      if vrfName in staticRpColl.vrfConfig:
         del staticRpColl.vrfConfig[ vrfName ]
      if vrfName in rpConfigColl.vrfConfig:
         del rpConfigColl.vrfConfig[ vrfName ]

   for hook in PimCliLib.pimSparseModeVrfDeletedHook.extensions():
      t0( "Calling hook: %s" % hook )
      hook( vrfName )

def _bidirModeVrfDeletionHook( vrfName ):
   if vrfName == DEFAULT_VRF:
      return

   canDeleteBidirModeVrfConfig = True

   for hook in PimCliLib.canDeletePimBidirModeVrfHook.extensions():
      output = hook( vrfName )
      if not output:
         canDeleteBidirModeVrfConfig = False

   if not canDeleteBidirModeVrfConfig or vrfName in _mfibVrfConfig.config or \
                                         vrfName in _mfibVrfConfig6.config:
      return
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = bidirConfigColl( af=af )
      staticRpColl = staticRpConfigColl( af=af, protocol='bidir' )
      rpConfigColl = pimRpConfigColl( af=af, protocol='bidir' )
      if vrfName in pimConfigRoot.bidirVrfConfig:
         pvc = bidirConfigVrf( af=af, vrfName=vrfName )
         if not pvc.isDefault() or \
               ( vrfName in staticRpColl.vrfConfig and \
               staticRpColl.vrfConfig[ vrfName ].rpTable ) or \
               ( vrfName in rpConfigColl.vrfConfig and \
               rpConfigColl.vrfConfig[ vrfName ].rpHashAlgorithm != \
               Tac.Type( "Routing::Pim::RpHashAlgorithm" ).rpHashAlgorithmDefault ):
            canDeleteBidirModeVrfConfig = False

   t0( "canDeleteBidirModeVrfConfig: %s" % canDeleteBidirModeVrfConfig )
   if not canDeleteBidirModeVrfConfig:
      return

   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = bidirConfigColl( af=af )
      staticRpColl = staticRpConfigColl( af=af, protocol='bidir' )
      rpConfigColl = pimRpConfigColl( af=af, protocol='bidir' )
      if vrfName in pimConfigRoot.bidirVrfConfig:
         del pimConfigRoot.bidirVrfConfig[ vrfName ]
      if vrfName in staticRpColl.vrfConfig:
         del staticRpColl.vrfConfig[ vrfName ]
      if vrfName in rpConfigColl.vrfConfig:
         del rpConfigColl.vrfConfig[ vrfName ]

   for hook in PimCliLib.pimBidirVrfDeletedHook.extensions():
      hook( vrfName )

def getOrCreatePimSparseModeVrfConfig( pimConfig, vrfName, af ):
   pimConfigRoot = pimConfigColl( af=af )
   staticRpColl = staticRpConfigColl( af=af, protocol='sparsemode' )
   rpConfigColl = pimRpConfigColl( af=af, protocol='sparsemode' )
   pimVrfConfig = None
   if vrfName in pimConfigRoot.vrfConfig:
      pimVrfConfig = pimConfigRoot.vrfConfig[ vrfName ]
   if vrfName not in pimConfigRoot.vrfConfig or \
         vrfName not in staticRpColl.vrfConfig or \
         vrfName not in rpConfigColl.vrfConfig:
      _sparseModeVrfDefinitionHook( vrfName )
      for hook in PimCliLib.pimSparseModeVrfConfiguredHook.extensions():
         hook( vrfName )
      pimVrfConfig = pimConfigRoot.vrfConfig[ vrfName ]
   return pimVrfConfig

def getOrCreatePimBidirVrfConfig( pimConfig, vrfName, af ):
   pimConfigRoot = bidirConfigColl( af=af )
   staticRpColl = staticRpConfigColl( af=af, protocol='bidir' )
   rpConfigColl = pimRpConfigColl( af=af, protocol='bidir' )
   pimVrfConfig = None
   if vrfName in pimConfigRoot.bidirVrfConfig:
      pimVrfConfig = pimConfigRoot.bidirVrfConfig[ vrfName ]
   if vrfName not in pimConfigRoot.bidirVrfConfig or \
         vrfName not in staticRpColl.vrfConfig or \
         vrfName not in rpConfigColl.vrfConfig:
      _bidirModeVrfDefinitionHook( vrfName )
      for hook in PimCliLib.pimBidirVrfConfiguredHook.extensions():
         hook( vrfName )
      pimVrfConfig = pimConfigRoot.bidirVrfConfig[ vrfName ]
   return pimVrfConfig

def _allowed( mode, af, vrfName ):
   mfibVrfConfig = McastCommonCliLib.getMount(
                     af, mType="Routing::Multicast::Fib::VrfConfig" )
   routingInfoVrfStatus = _routingInfoVrfStatus if af == AddressFamily.ipv4 \
                                                else _routing6InfoVrfStatus
   if vrfName not in mfibVrfConfig.config or \
         vrfName not in routingInfoVrfStatus.entityPtr:
      return False
   else:
      return mfibVrfConfig.config[ vrfName ].routing and \
          routingInfoVrfStatus[ vrfName ].routing

def getPimIntfConfigFromAf( mode, af, legacy=False, printError=True ):
   _pimConfigColl = pimConfigColl( af )
   if mode.intf.name not in _pimConfigColl.intfConfig:
      def createFcn():
         _pimConfigColl.intfConfig.newMember( mode.intf.name )
      vrfName = McastCommonCliLib.getVrfNameFromIntf( ipGenStatus, mode.intf.name,
                                                      af=af )
      McastCommonCliLib.createMcastIntfConfig( mode, vrfName, af, mode.intf.name,
                                               createFcn, printError )
   return _pimConfigColl.intfConfig.get( mode.intf.name )

def _maybeDelPimIntfConfig( pic, af=AddressFamily.ipv4 ):
   ''' Check if all configs are back to default, before deleting the interface
       config object itself. '''

   if pic.isDefault() :
      _pimConfigColl = pimConfigColl( af )
      del _pimConfigColl.intfConfig[ pic.intfId ]

def _igmpImplicitlyEnabledIs( mode, pic, enabled ):
   if enabled:
      for hook in GmpCli.pimEnableHook:
         hook( mode, True )
   else:
      # Igmp will remain implicitly enabled if any of 'ip pim sparse-mode',
      # 'ip pim bidir' or 'ip pim border-router' is configured
      if pic.mode == 'modePimNone' and pic.borderRouter == False:
         for hook in GmpCli.pimEnableHook:
            hook( mode, False )

def _igmpLocalInterfaceIs( mode, intfName ):
   for hook in GmpCli.pimLocalInterfaceHook:
      hook( mode, intfName )

def pimCliConfigCommand( func ):
   ''' Decorator for config commands.  If either routing or multicast-routing
       is disabled, print warning. '''
   def newFunc( mode, *args, **kwargs ):
      ipFamily = kwargs.get( 'ipFamily' )
      legacy = kwargs.get( 'legacy' )
      if not legacy:
         return func( mode, *args, **kwargs )

      if isinstance( mode, IntfCli.IntfConfigMode ):
         af = getAfFromIpFamilyRule( ipFamily, legacy )
         vrfName = McastCommonCliLib.getVrfNameFromIntf( ipGenStatus, mode.intf.name,
                                                      af=af )
      else:
         af = RouterMulticastCliLib.getAddressFamilyFromMode( mode, legacy )
         vrfName = McastCommonCliLib.vrfFromMode( mode )
      # Temporary while ipv4 is current base mode
      if af == 'ipunknown':
         af = 'ipv4'

      if not _allowed( mode, af, vrfName ):
         mode.addWarning( "'ip routing' or 'ip multicast-routing' "
                          "is not configured  for vrf %s, so the "
                          "command will have no effect." %
                          vrfName )
      return func( mode, *args, **kwargs )
   return newFunc

def _getVrfNameFromMode( mode ):
   if hasattr( mode, 'vrfName' ):
      return mode.vrfName
   return DEFAULT_VRF

def _pimBidirectionalGuard( mode, token ):
   if _routingHwStatus.pimBidirectionalSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def pimGenBidirectionalGuard( mode, token ):
   if _routingHwStatus.pimBidirectionalSupported or \
         _routing6HwStatus.pimBidirectionalSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def pimSparseModeIpv4SupportedGuard( mode, token ):
   if mcastRoutingSupported( mode.sysdbRoot, _routingHwStatus ):
      return None
   return CliParser.guardNotThisPlatform

def pimSparseModeIpv6SupportedGuard( mode, token ):
   if mcast6RoutingSupported( mode.sysdbRoot, _routing6HwStatus ):
      return None
   return CliParser.guardNotThisPlatform

def pimBidirModeIpv4SupportedGuard( mode, token ):
   if mcastRoutingSupported( mode.sysdbRoot, _routingHwStatus ) and \
         _routingHwStatus.pimBidirectionalSupported:
      return None
   return CliParser.guardNotThisPlatform

def pimBidirModeIpv6SupportedGuard( mode, token ):
   if mcast6RoutingSupported( mode.sysdbRoot, _routing6HwStatus ) and \
         _routing6HwStatus.pimBidirectionalSupported:
      return None
   return CliParser.guardNotThisPlatform

def logNeighborChanges( mode, args ):
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   if isBidirMode( mode ):
      _pimBidirVrfConfig = getOrCreatePimBidirVrfConfig( _pimConfigRoot,
           _getVrfNameFromMode( mode ), af )
      if _pimBidirVrfConfig:      
         if CliCommand.isNoCmd( args ):
            _pimBidirVrfConfig.logNeighborChanges = False
         else:
            _pimBidirVrfConfig.logNeighborChanges = True
   else:
      _pimVrfConfig = getOrCreatePimSparseModeVrfConfig( _pimConfigRoot,
           _getVrfNameFromMode( mode ), af )
      if _pimVrfConfig:
         if CliCommand.isNoCmd( args ):  
            _pimVrfConfig.logNeighborChanges = False
         else:
            _pimVrfConfig.logNeighborChanges = True

def setPimBfdDefault( mode, args ):
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   _pimVrfConfig = getOrCreatePimSparseModeVrfConfig( _pimConfigRoot,
               _getVrfNameFromMode( mode ), af )
   if _pimVrfConfig:
      _pimVrfConfig.bfdIntfDefault = not CliCommand.isNoOrDefaultCmd( args )

def setAllowRp( mode, args ):
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   _pimVrfConfig = getOrCreatePimSparseModeVrfConfig( _pimConfigRoot,
               _getVrfNameFromMode( mode ), af )
   if _pimVrfConfig:
      _pimVrfConfig.allowRp = not CliCommand.isNoOrDefaultCmd( args )

def getVrfNames( mode=None ):
   return sorted( _allVrfConfig.vrf.members() )

def getMulticastVrfNames( mode=None ):
   return sorted( _mfibVrfConfig.config.members() )

def getMulticast6VrfNames( mode=None ):
   return sorted( _mfibVrfConfig6.config.members() )

#------------------------------------------------------------------------------------
# Config tokens for new parser
#------------------------------------------------------------------------------------
vrfMatcher = CliMatcher.KeywordMatcher( 'vrf', helpdesc='Configure PIM in a VRF' )
vrfNameMatcher = CliMatcher.DynamicNameMatcher( getVrfNames, 'VRF name' )
transportKw = CliMatcher.KeywordMatcher( 'transport',
      helpdesc='Transport protocol to be used to send Join-Prune messages' )
helloKw = CliMatcher.KeywordMatcher( 'hello',
      helpdesc='PIM hello parameters' )
joinPruneKw = CliMatcher.KeywordMatcher( 'join-prune',
      helpdesc='PIM join/prune message attributes' )
sctpKw = CliMatcher.KeywordMatcher( 'sctp', 
      helpdesc='SCTP' )
srcInterfaceKw = CliMatcher.KeywordMatcher( 'source-interface',
      helpdesc="Interface's address to use to for SCTP connection" )

#----------------------------------------------------------------------
# (config-router-pim)# [no|default] vrf <vrf>
#----------------------------------------------------------------------
def gotoRouterPimVrfMode( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return
   if vrfName not in getVrfNames():
      mode.addWarning( "vrf name %s is not defined." % vrfName )
   childMode = mode.childMode( PimCliLib.RouterPimVrfMode, vrfName=vrfName )
   if _getServiceAcl( vrfName ) is None:
      # create new service acl config, but set acl name to ''
      # this way we see it in "show run all", but it has no effect
      _setServiceAcl( mode, '', vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimVrfMode( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   assert vrfName not in VRFNAMES_RESERVED
   _noServiceAcl( mode, vrfName=vrfName )

class RouterPimVrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher
   }
   handler = gotoRouterPimVrfMode
   noOrDefaultHandler = deleteRouterPimVrfMode

PimCliLib.RouterPimMode.addCommandClass( RouterPimVrfModeCmd )

#----------------------------------------------------------------------
# (config-router-pim-sparse)# [no|default] vrf <vrf>
#----------------------------------------------------------------------
def gotoRouterPimSparseVrfMode( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return

   _pim6ConfigRoot = pimConfigColl( af=AddressFamily.ipv6 )
   if ( vrfName not in getMulticastVrfNames() and \
         vrfName not in getMulticast6VrfNames() ) or \
         ( vrfName not in _pimConfigRoot.vrfConfig and \
           vrfName not in _pim6ConfigRoot.vrfConfig ):
      _sparseModeVrfDefinitionHook( vrfName )
      for hook in PimCliLib.pimSparseModeVrfConfiguredHook.extensions():
         hook( vrfName )

   if vrfName not in getVrfNames():
      mode.addWarning( "vrf name %s is not defined." % vrfName )

   childMode = mode.childMode( PimCliLib.RouterPimSparseVrfMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimSparseVrfMode( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   assert vrfName not in VRFNAMES_RESERVED
   _pimVrfConfig = PimCliLib.getPimVrfConfig( _pimConfigRoot, vrfName )
   _pim6VrfConfig = pimConfigVrf( af=AddressFamily.ipv6,
                                  vrfName=vrfName )

   if _pimVrfConfig and _pim6VrfConfig:
      _cleanupPimSparseModeConfig( vrfName )

   # If vrf is not defined then delete the vrfConfig when pim configuration for
   # that vrf goes away
   _pim6ConfigRoot = pimConfigColl( af=AddressFamily.ipv6 )
   if ( vrfName not in getMulticastVrfNames() and \
         vrfName not in getMulticast6VrfNames() ) or \
         ( vrfName in _pimConfigRoot.vrfConfig or \
           vrfName in _pim6ConfigRoot.vrfConfig ):
      _sparseModeVrfDeletionHook( vrfName )

class RouterPimSparseVrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher
   }
   handler = gotoRouterPimSparseVrfMode
   noOrDefaultHandler = deleteRouterPimSparseVrfMode

PimCliLib.RouterPimSparseMode.addCommandClass( RouterPimSparseVrfModeCmd )

#----------------------------------------------------------------------
# (config-router-pim-bidir)# [no|default] vrf <vrf>
#----------------------------------------------------------------------
def gotoRouterPimBidirVrfMode( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return
   _pim6ConfigRoot = bidirConfigColl( af=AddressFamily.ipv6 )
   if ( vrfName not in getMulticastVrfNames() and \
         vrfName not in getMulticast6VrfNames() ) or \
         ( vrfName not in _pimConfigRoot.bidirVrfConfig and \
           vrfName not in _pim6ConfigRoot.bidirVrfConfig ):
      _bidirModeVrfDefinitionHook( vrfName )
      for hook in PimCliLib.pimBidirVrfConfiguredHook.extensions():
         hook( vrfName )
   if vrfName not in getVrfNames():
      mode.addWarning( "vrf name %s is not defined." % vrfName )
   childMode = mode.childMode( RouterPimBidirVrfMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimBidirVrfMode( mode, args ):
   vrfName = args[ 'VRF_NAME' ]
   t0( "deleteRouterPimBidirVrfMode : %s" % vrfName )
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return
   _pimBidirVrfConfig = PimCliLib.getPimBidirVrfConfig( _pimConfigRoot, vrfName )
   _pim6BidirVrfConfig = bidirConfigVrf( af=AddressFamily.ipv6,
                                  vrfName=vrfName )

   if _pimBidirVrfConfig and _pim6BidirVrfConfig:
      _cleanupPimBidirConfig( vrfName )
   #If vrf is not defined then delete the vrfConfig when PIM Bidir configuration
   #for that vrf goes away.
   _pim6ConfigRoot = bidirConfigColl( af=AddressFamily.ipv6 )
   if ( vrfName not in getMulticastVrfNames() and \
         vrfName not in getMulticast6VrfNames() ) or \
         ( vrfName in _pimConfigRoot.bidirVrfConfig and \
           vrfName in _pim6ConfigRoot.bidirVrfConfig ):
      t0( "Calling _bidirModeVrfDeletionHook : % s" % vrfName )
      _bidirModeVrfDeletionHook( vrfName )

class RouterPimBidirVrfModeCmd( CliCommand.CliCommandClass ):
   syntax = 'vrf VRF_NAME'
   noOrDefaultSyntax = syntax
   data = {
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher
   }
   handler = gotoRouterPimBidirVrfMode
   noOrDefaultHandler = deleteRouterPimBidirVrfMode

RouterPimBidirBaseMode.addCommandClass( RouterPimBidirVrfModeCmd )

#-------------------------------------------------------------------------------
modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

def _egressAclGuard( mode, token ):
   if _routingHwStatus.multicastRoutingSupportsIpEgressAcl:
      return None
   else:
      return CliParser.guardNotThisPlatform

#------------------------------------------------------------------------------
# Legacy:
# switch(config)# [no | default ] ip pim rp-hash algorithm modulo ( legacy )
#------------------------------------------------------------------------------
# switch(config-router-pim-bidir-*) [no | default ] rp hash algorithm modulo
# switch(config-router-pim-sparse-*) [no | default ] rp hash algorithm modulo
#------------------------------------------------------------------------------
hashKw = CliMatcher.KeywordMatcher( 'hash',
      helpdesc='Rendezvous Point hashing' )

def setIpPimRpHashAlgorithm( mode, args ):
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   rpConfigColl = None
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   if isBidirMode( mode ):
      rpConfigColl = pimRpConfigColl( af, protocol='bidir' )
      if vrfName not in rpConfigColl.vrfConfig:
         getOrCreatePimBidirVrfConfig( _pimConfigRoot, vrfName, af )
   else:
      rpConfigColl = pimRpConfigColl( af, protocol='sparsemode' )
      if vrfName not in rpConfigColl.vrfConfig:
         getOrCreatePimSparseModeVrfConfig( _pimConfigRoot, vrfName, af )

   if rpConfigColl is None:
      return
   PimCliLib.configureIpPimHashAlgorithm( mode, rpConfigColl )

def noSetIpPimRpHashAlgorithm( mode, args ):
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   rpConfigColl = None
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   if isBidirMode( mode ):
      rpConfigColl = pimRpConfigColl( af, protocol='bidir' )
      if vrfName not in rpConfigColl.vrfConfig:
         getOrCreatePimBidirVrfConfig( _pimConfigRoot, vrfName, af )
   else:
      rpConfigColl = pimRpConfigColl( af, protocol='sparsemode' )
      if vrfName not in rpConfigColl.vrfConfig:
         getOrCreatePimSparseModeVrfConfig( _pimConfigRoot, vrfName, af )

   if rpConfigColl is None:
      return
   PimCliLib.resetIpPimHashAlgorithm( mode, rpConfigColl )

class RouterIpPimSparseRpHashAlgorithmLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim rp-hash algorithm modulo'
   noOrDefaultSyntax = 'ip pim rp-hash algorithm ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'rp-hash' : CliToken.Pim.rpHashMatcher,
      'algorithm' : CliToken.Pim.algorithmMatcher,
      'modulo' : CliToken.Pim.moduloMatcher
   }
   handler = setIpPimRpHashAlgorithm
   noOrDefaultHandler = noSetIpPimRpHashAlgorithm

RouterPimSparseSharedModelet.addCommandClass( 
      RouterIpPimSparseRpHashAlgorithmLegacyCmd )

class RouterIpPimBidirRpHashAlgorithmLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim rp-hash algorithm modulo'
   noOrDefaultSyntax = 'ip pim rp-hash algorithm ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'rp-hash' : CliToken.Pim.rpHashMatcher,
      'algorithm' : CliToken.Pim.algorithmMatcher,
      'modulo' : CliToken.Pim.moduloMatcher
   }
   handler = setIpPimRpHashAlgorithm
   noOrDefaultHandler = noSetIpPimRpHashAlgorithm

RouterPimBidirSharedModelet.addCommandClass( 
      RouterIpPimBidirRpHashAlgorithmLegacyCmd )

class RouterIpPimSparseRpHashAlgorithmCmd( CliCommand.CliCommandClass ):
   syntax = 'rp hash algorithm modulo'
   noOrDefaultSyntax = 'rp hash algorithm ...'
   data = {
      'rp' : CliToken.Pim.rpMatcher,
      'hash' : hashKw,
      'algorithm' : CliToken.Pim.algorithmMatcher,
      'modulo' : CliToken.Pim.moduloMatcher
   }
   handler = setIpPimRpHashAlgorithm
   noOrDefaultHandler = noSetIpPimRpHashAlgorithm

RouterPimSparseAfCommonSharedModelet.addCommandClass( 
      RouterIpPimSparseRpHashAlgorithmCmd )

class RouterIpPimBidirRpHashAlgorithmCmd( CliCommand.CliCommandClass ):
   syntax = 'rp hash algorithm modulo'
   noOrDefaultSyntax = 'rp hash algorithm ...'
   data = {
      'rp' : CliToken.Pim.rpMatcher,
      'hash' : hashKw,
      'algorithm' : CliToken.Pim.algorithmMatcher,
      'modulo' : CliToken.Pim.moduloMatcher
   }
   handler = setIpPimRpHashAlgorithm
   noOrDefaultHandler = noSetIpPimRpHashAlgorithm

RouterPimBidirAfCommonSharedModelet.addCommandClass( 
      RouterIpPimBidirRpHashAlgorithmCmd )

#------------------------------------------------------------------------------
# Legacy:
# switch(config)# ip pim rp-address ip-address
#               [ group/prefix | access-list acl-name ]
#               [ priority priority-value ] [ hashmask hash-mask-len ]
#               [ override ]
#------------------------------------------------------------------------------
# switch(config-router-pim-bidir-*)# rp address <rp-address>
# switch(config-router-pim-sparse-*)# rp address <rp-address>
#------------------------------------------------------------------------------
class NotAccessListMatcher( CliMatcher.Matcher ):
   '''Takes a matcher, but filter out input that is access-list'''
   def __init__( self, matcher ):
      CliMatcher.Matcher.__init__( self )
      self.matcher_ = matcher

   def match( self, mode, context, token ):
      if context.state is None and token == 'access-list':
         # first token
         return CliMatcher.noMatch
      return self.matcher_.match( mode, context, token )

   def completions( self, mode, context, token ):
      return self.matcher_.completions( mode, context, token )

addressKw = CliMatcher.KeywordMatcher( 'address',
   helpdesc='Address' )

def validateArgsForPimRpAddress( mode, af, rp, groupRangeOrAcl, hashmask=None ):
   if af == 'ipv4':
      rpAddr = IpAddr( rp )
      if not rpAddr.isUnicast:
         mode.addError( 'RP Address is not a valid IPv4 unicast address' )
         return False
      if not groupRangeOrAcl:
         return True
      if '.' in groupRangeOrAcl:
         if not isValidPrefixWithError( mode, groupRangeOrAcl ):
            return False
         mcastGroup = IpAddr( groupRangeOrAcl.split('/')[0] )
         if groupRangeOrAcl != '224.0.0.0/4' and \
               ( not mcastGroup.isMulticast ):
            mode.addError( "Group address is not a valid IPv4 multicast address" )
            return False
         if hashmask and hashmask > 32:
            mode.addError( 'Invalid mask %s for group %s ' % \
                           ( hashmask, groupRangeOrAcl ) )
            return False
      else:
         acl = getAclConfig( 'ip' ).get( groupRangeOrAcl )
         if acl and not acl.standard:
            mode.addError( '%s is not a standard acl' % groupRangeOrAcl )
            return False
   else:
      if not rp.isUnicast:
         mode.addError( 'RP Address is not a valid IPv6 unicast address' )
         return False
      if not groupRangeOrAcl:
         return True
      isGroup = True
      if isinstance( groupRangeOrAcl, str ):
         if ':' not in groupRangeOrAcl:
            # Assume an IPv6 ACL
            isGroup = False
            acl = getAclConfig( af ).get( groupRangeOrAcl )
            if acl:
               if not acl.standard:
                  mode.addError( '%s is not a standard acl' % groupRangeOrAcl )
                  return False
      else:
         groupRangeOrAcl = groupRangeOrAcl.stringValue

      if isGroup:
         addr = Ip6AddrWithMask( groupRangeOrAcl )
         if not IraIp6RouteCliLib.isValidIpv6PrefixWithError( mode, addr ) or \
               not addr.address.isMulticast:
            mode.addError( 'Group address is not a valid IPv6 multicast address' )
            return False
      if hashmask and hashmask > 128:
         mode.addError( 'Invalid IPv6 mask %s for group %s.' %
                        ( hashmask, groupRangeOrAcl ) )
         return False
   return True

def setIpPimRpAddress( mode, args ):
   rp = args[ 'RP_ADDR' ]
   priority = args.get( 'PRIORITY_VALUE', 0 )
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   staticRpColl = None
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   if af == AddressFamily.ipv4:
      groupRangeOrAcl = args.get( 'GROUP_OR_RANGE' ) or args.get( 'ACL_NAME1' ) or \
                        args.get( 'ACL_NAME2', '224.0.0.0/4' )
      hashmask = args.get( 'HASH_VALUE', 30 )
   else:
      groupRangeOrAcl = args.get( 'GROUP_OR_RANGE' ) or args.get( 'ACL_NAME1' ) or \
                        args.get( 'ACL_NAME2', 'ff00::/8' )
      hashmask = args.get( 'HASH_VALUE', 126 )

   if not validateArgsForPimRpAddress( mode, af, rp, groupRangeOrAcl, hashmask ):
      return

   if af == AddressFamily.ipv6:
      rp = rp.stringValue
      if not isinstance( groupRangeOrAcl, str ):
         groupRangeOrAcl = groupRangeOrAcl.stringValue

   if isBidirMode( mode ):
      staticRpColl = staticRpConfigColl( af=af, protocol='bidir' )
      if vrfName not in staticRpColl.vrfConfig:
         getOrCreatePimBidirVrfConfig( _pimConfigRoot, vrfName, af )
   else:
      staticRpColl = staticRpConfigColl( af=af,
                                         protocol='sparsemode' )
      if vrfName not in staticRpColl.vrfConfig:
         getOrCreatePimSparseModeVrfConfig( _pimConfigRoot, vrfName, af )

   PimCliLib.configureIpPimRpAddress( mode, staticRpColl,
         rp, groupRangeOrAcl, priority, hashmask, 'override' in args )

def noIpPimRpAddress( mode, args ):
   rp = args[ 'RP_ADDR' ]
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   staticRpColl = None
   af = getAddressFamilyFromMode( mode, legacy='ip' in args )
   if af == AddressFamily.ipv4:
      groupRangeOrAcl = args.get( 'GROUP_OR_RANGE' ) or args.get( 'ACL_NAME1' ) or \
                        args.get( 'ACL_NAME2', '224.0.0.0/4' )
   else:
      groupRangeOrAcl = args.get( 'GROUP_OR_RANGE' ) or args.get( 'ACL_NAME1' ) or \
                        args.get( 'ACL_NAME2', 'ff00::/8' )

   if not validateArgsForPimRpAddress( mode, af, rp, groupRangeOrAcl ):
      return

   if af == "ipv6" and not isinstance( groupRangeOrAcl, str ):
      groupRangeOrAcl = groupRangeOrAcl.stringValue

   if isBidirMode( mode ):
      staticRpColl = staticRpConfigColl( af=af, protocol='bidir' )
      if vrfName not in staticRpColl.vrfConfig:
         getOrCreatePimBidirVrfConfig( _pimConfigRoot, vrfName, af )
   else:
      staticRpColl = staticRpConfigColl( af=af,
                                         protocol='sparsemode' )
      if vrfName not in staticRpColl.vrfConfig:
         getOrCreatePimSparseModeVrfConfig( _pimConfigRoot, vrfName, af )

   PimCliLib.resetIpPimRpAddress( mode, staticRpColl, rp, groupRangeOrAcl )

class RouterPimRpAddressLegacyCmd( CliCommand.CliCommandClass ):
   syntax = '''ip pim rp-address RP_ADDR
               [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ]
               [ priority PRIORITY_VALUE ]
               [ hashmask HASH_VALUE ] [ override ]'''
   noOrDefaultSyntax = '''ip pim rp-address RP_ADDR
                   [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ] ...'''
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'rp-address' : CliToken.Pim.rpAddressMatcher,
      'RP_ADDR' : CliToken.Pim.rpAddrValueMatcher,
      'GROUP_OR_RANGE' : CliToken.Pim.groupRangeMatcher,
      'access-list' : CliToken.Pim.accessListMatcher,
      'ACL_NAME1' : CliToken.Pim.aclNameMatcher,
      'ACL_NAME2' : NotAccessListMatcher( CliToken.Pim.aclNameMatcher ),
      'priority' : CliToken.Pim.priorityMatcher,
      'PRIORITY_VALUE' : CliToken.Pim.priorityValueMatcher,
      'hashmask' : CliToken.Pim.hashmaskMatcher,
      'HASH_VALUE' : CliToken.Pim.hashValueMatcher,
      'override' : CliToken.Pim.overrideMatcher
   }
   handler = setIpPimRpAddress
   noOrDefaultHandler = noIpPimRpAddress

RouterPimSparseSharedModelet.addCommandClass( RouterPimRpAddressLegacyCmd )

RouterPimBidirSharedModelet.addCommandClass( RouterPimRpAddressLegacyCmd )

class RouterPimGlobalRpAddressLegacyCmd( CliCommand.CliCommandClass ):
   syntax = '''ip pim rp-address RP_ADDR
               [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ]
               [ priority PRIORITY_VALUE ]
               [ hashmask HASH_VALUE ] [ override ]'''
   noOrDefaultSyntax = '''ip pim rp-address RP_ADDR
                   [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ] ...'''
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'rp-address' : CliToken.Pim.rpAddressMatcher,
      'RP_ADDR' : CliToken.Pim.rpAddrValueMatcher,
      'GROUP_OR_RANGE' : CliToken.Pim.groupRangeMatcher,
      'access-list' : CliToken.Pim.accessListMatcher,
      'ACL_NAME1' : CliToken.Pim.aclNameMatcher,
      'ACL_NAME2' : NotAccessListMatcher( CliToken.Pim.aclNameMatcher ),
      'priority' : CliToken.Pim.priorityMatcher,
      'PRIORITY_VALUE' : CliToken.Pim.priorityValueMatcher,
      'hashmask' : CliToken.Pim.hashmaskMatcher,
      'HASH_VALUE' : CliToken.Pim.hashValueMatcher,
      'override' : CliToken.Pim.overrideMatcher
   }
   handler = setIpPimRpAddress
   noOrDefaultHandler = noIpPimRpAddress

BasicCli.GlobalConfigMode.addCommandClass( RouterPimGlobalRpAddressLegacyCmd )

class RouterPimIpv4RpAddressCmd( CliCommand.CliCommandClass ):
   syntax = '''rp address RP_ADDR
               [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ]
               [ priority PRIORITY_VALUE ]
               [ hashmask HASH_VALUE ] [ override ]'''
   noOrDefaultSyntax = '''rp address RP_ADDR
                   [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ] ...'''
   data = {
      'rp' : CliToken.Pim.rpMatcher,
      'address' : addressKw,
      'RP_ADDR' : CliToken.Pim.rpAddrValueMatcher,
      'GROUP_OR_RANGE' : CliToken.Pim.groupRangeMatcher,
      'access-list' : CliToken.Pim.accessListMatcher,
      'ACL_NAME1' : CliToken.Pim.aclNameMatcher,
      'ACL_NAME2' : NotAccessListMatcher( CliToken.Pim.aclNameMatcher ),
      'priority' : CliToken.Pim.priorityMatcher,
      'PRIORITY_VALUE' : CliToken.Pim.priorityValueMatcher,
      'hashmask' : CliToken.Pim.hashmaskMatcher,
      'HASH_VALUE' : CliToken.Pim.hashValueMatcher,
      'override' : CliToken.Pim.overrideMatcher
   }
   handler = setIpPimRpAddress
   noOrDefaultHandler = noIpPimRpAddress

RouterPimSparseIpv4Mode.addCommandClass( RouterPimIpv4RpAddressCmd )

RouterPimBidirIpv4Mode.addCommandClass( RouterPimIpv4RpAddressCmd )

class RouterPimIpv6RpAddressCmd( CliCommand.CliCommandClass ):
   syntax = '''rp address RP_ADDR
               [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ]
               [ priority PRIORITY_VALUE ]
               [ hashmask HASH_VALUE ] [ override ]'''
   noOrDefaultSyntax = '''rp address RP_ADDR
                   [ GROUP_OR_RANGE | ( access-list ACL_NAME1 ) | ACL_NAME2 ] ...'''
   data = {
      'rp' : CliToken.Pim.rpMatcher,
      'address' : addressKw,
      'RP_ADDR' : CliToken.Pim.rpAddr6ValueMatcher,
      'GROUP_OR_RANGE' : CliToken.Pim.groupRange6Matcher,
      'access-list' : CliToken.Pim.accessListMatcher,
      'ACL_NAME1' : CliToken.Pim.aclNameMatcher,
      'ACL_NAME2' : NotAccessListMatcher( CliToken.Pim.aclNameMatcher ),
      'priority' : CliToken.Pim.priorityMatcher,
      'PRIORITY_VALUE' : CliToken.Pim.priorityValueMatcher,
      'hashmask' : CliToken.Pim.hashmaskMatcher,
      'HASH_VALUE' : CliToken.Pim.hashValueMatcher6,
      'override' : CliToken.Pim.overrideMatcher
   }
   handler = setIpPimRpAddress
   noOrDefaultHandler = noIpPimRpAddress

RouterPimSparseIpv6Mode.addCommandClass( RouterPimIpv6RpAddressCmd )

RouterPimBidirIpv6Mode.addCommandClass( RouterPimIpv6RpAddressCmd )

#-------------------------------------------------------------------------------

# service acls
def _noServiceAcl( mode, vrfName=DEFAULT_VRF ):
   AclCliLib.noServiceAcl( mode, 'pim', aclConfig, aclCpConfig,
                           None, 'ip', vrfName )

def _setServiceAcl( mode, aclName, vrfName=DEFAULT_VRF ):
   AclCliLib.setServiceAcl( mode, 'pim', AclLib.IPPROTO_PIM,
                            aclConfig, aclCpConfig, aclName, 'ip', vrfName )

def _getServiceAcl( vrfName ):
   serviceAclVrfConfig = aclCpConfig.cpConfig[ 'ip' ].serviceAcl.get( vrfName )
   if serviceAclVrfConfig:
      serviceConfig = serviceAclVrfConfig.service.get( 'pim' )
      if serviceConfig:
         return serviceConfig.aclName
   return None

#-------------------------------------------------------------------------------
# "[no|default] router pim" command, in "config" mode.
#-------------------------------------------------------------------------------
def gotoRouterPimMode( mode, args ):
   vrfName = DEFAULT_VRF
   _pimXLegacyConfig.legacy = False
   if not _pimXLegacyConfig.version == _pimXLegacyConfig.ipMode:
      _pimXLegacyConfig.version = _pimXLegacyConfig.routerMode
   if _getServiceAcl( vrfName ) is None:
      # create new service acl config, but set acl name to ''
      # this way we see it in "show run all", but it has no effect
      _setServiceAcl( mode, '', vrfName )
   childMode = mode.childMode( PimCliLib.RouterPimMode )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimMode( mode, args ):
   # delete non-default config all for all VRFs
   for vrf in aclCpConfig.cpConfig[ 'ip' ].serviceAcl.keys():
      _noServiceAcl( mode, vrf )
   
   _pimXLegacyConfig.version = _pimXLegacyConfig.globalMode
   _pimXLegacyConfig.legacy = True

#-------------------------------------------------------------------------------
# "[no|default] router pim sparse-mode" command, in "config" mode.
#-------------------------------------------------------------------------------

def gotoRouterPimSparseMode( mode, args ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      _pimVrfConfig = getOrCreatePimSparseModeVrfConfig( _pimConfigRoot,
            DEFAULT_VRF, af )
   _pimLegacyConfig.legacy = False
   if not _pimLegacyConfig.version == _pimLegacyConfig.ipMode:
      _pimLegacyConfig.version = _pimLegacyConfig.routerMode
   childMode = mode.childMode( PimCliLib.RouterPimSparseMode )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimSparseMode( mode, args ):
   # delete non-default config all for all VRFs
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = pimConfigColl( af=af )
      for vrfName in pimConfigRoot.vrfConfig:
         if vrfName == DEFAULT_VRF:
            _cleanupPimSparseModeConfig( vrfName )
         else:
            args = { 'VRF_NAME': vrfName }
            deleteRouterPimSparseVrfMode( mode, args )
   _pim6ConfigRoot = pimConfigColl( af=AddressFamily.ipv6 )
   if len( _pimConfigRoot.vrfConfig.keys() ) == 1 and \
         len( _pim6ConfigRoot.vrfConfig.keys() ) == 1 and \
         _mrouteLegacyConfig.version == _mrouteLegacyConfig.globalMode:
      _pimLegacyConfig.version = _pimLegacyConfig.globalMode
      _pimLegacyConfig.legacy = True

#--------------------------------------------------------------------
# "[no|default] router pim bidirectional" command, in "config" mode.
#--------------------------------------------------------------------
def deleteRouterPimBidirBaseMode( mode, args ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = bidirConfigColl( af=af )
      for vrfName in pimConfigRoot.bidirVrfConfig:
         if vrfName == DEFAULT_VRF:
            _cleanupPimBidirConfig( vrfName )
         else:
            args = { 'VRF_NAME': vrfName }
            deleteRouterPimBidirVrfMode( mode, args )
   _pim6ConfigRoot = bidirConfigColl( af=AddressFamily.ipv6 )
   if len( _pimConfigRoot.bidirVrfConfig.keys() ) == 1 and \
         len( _pim6ConfigRoot.bidirVrfConfig.keys() ) == 1 and \
         _mrouteLegacyConfig.version == _mrouteLegacyConfig.globalMode:
      _pimBidirLegacyConfig.legacy = True
      _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.globalMode

def gotoRouterPimBidirBaseMode( mode, args ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      _pimBidirVrfConfig = getOrCreatePimBidirVrfConfig(
            _pimConfigRoot, DEFAULT_VRF, af )
   _pimBidirLegacyConfig.legacy = False
   if not _pimBidirLegacyConfig.version == _pimBidirLegacyConfig.ipMode:
      _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.routerMode
   childMode = mode.childMode( RouterPimBidirBaseMode )
   mode.session_.gotoChildMode( childMode )

#------------------------------------------------------------------------------------
# switch(config-if)# pim ipv4 local-interface <intf-name>
#------------------------------------------------------------------------------------
def noPimLocalInterface( mode, args ):
   af = args[ 'AF' ]
   pic = getPimIntfConfigFromAf( mode, af, legacy=False, printError=False )
   if pic is None:
      return
   pic.localIntfId = ''
   _igmpLocalInterfaceIs( mode, '' )
   _maybeDelPimIntfConfig( pic, 'ipv4' )

def setPimLocalInterface( mode, args ):
   af = args[ 'AF' ]
   intf = args[ 'INTF' ]
   # Explicitly move to new format
   convertLegacyConfigPim( mode )
   pic = getPimIntfConfigFromAf( mode, af, legacy=False )
   if pic is None:
      return
   pic.localIntfId = intf.name
   _igmpLocalInterfaceIs( mode, intf.name )

class RouterPimLocalInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'pim ipv4 local-interface INTF'
   noOrDefaultSyntax = 'pim ipv4 local-interface ...'
   data = {
      'pim' : CliToken.Pim.pimNode,
      'ipv4' : McastCommonCliLib.ipv4Node,
      'local-interface' : 'Specify the local interface for PIM ' \
                          'to get interface address',
      'INTF' : IntfCli.Intf.matcher
   }
   handler = setPimLocalInterface
   noOrDefaultHandler = noPimLocalInterface

modelet.addCommandClass( RouterPimLocalInterfaceCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim neighbor-filter <acl-name> ( legacy )
# switch(config-if)# pim [ipv4|ipv6] neighbor-filter <acl-name>
#------------------------------------------------------------------------------------
def noPimNeighborFilter( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )
   if pic is None:
      return
   pic.neighborFilter = ''
   _maybeDelPimIntfConfig( pic, af )

def setPimNeighborFilter( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )
   aclName = args[ 'ACL_NAME' ]
   if pic is None:
      return
   if af == AddressFamily.ipv6:
      acl = getAclConfig( 'ipv6' ).get( aclName )
   else:
      acl = getAclConfig( 'ip' ).get( aclName )
   if acl and not acl.standard:
      mode.addError( '%s is not a standard acl' % aclName )
      return
   pic.neighborFilter = aclName

class RouterPimNeighborFilterLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim neighbor-filter ACL_NAME'
   noOrDefaultSyntax = 'ip pim neighbor-filter ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'neighbor-filter' : CliCommand.Node( 
                             CliMatcher.KeywordMatcher( 'neighbor-filter',
                             helpdesc='Limit to specified PIM neighbors' ),
                             deprecatedByCmd='pim ipv4 neighbor filter' ),
      'ACL_NAME' : standardIpAclNameMatcher
   }
   handler = setPimNeighborFilter
   noOrDefaultHandler = noPimNeighborFilter

modelet.addCommandClass( RouterPimNeighborFilterLegacyCmd )

class RouterPimNeighborFilterCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF neighbor filter ACL_NAME'
   noOrDefaultSyntax = 'pim AF neighbor filter ...'
   data = {
      'pim' : CliToken.Pim.pimNode,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'neighbor' : 'PIM neighbor',
      'filter' : 'Limit to specified PIM neighbors',
      'ACL_NAME' : standardIpAclNameMatcher
   }
   handler = setPimNeighborFilter
   noOrDefaultHandler = noPimNeighborFilter

modelet.addCommandClass( RouterPimNeighborFilterCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim join-prune-interval <interval> ( legacy )
# switch(config-if)# pim [ipv4|ipv6] join-prune interval <interval>
#------------------------------------------------------------------------------------
def pimJoinPruneOverflowCheck( mode, pic ):
   maxJoinPruneInterval = JOINPRUNE_HOLD_FOREVER / pic.joinPruneCount
   if pic.joinPruneInterval >= maxJoinPruneInterval:
      mode.addWarning( "The advertized hold-time will be set to %d as "
                       "the product of the join/prune interval and join/prune count "
                       "exceeds %d" % ( JOINPRUNE_HOLD_FOREVER,
                          JOINPRUNE_HOLD_FOREVER ) )

def noPimJoinPruneInterval( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )

   if pic is None:
      return
   pic.joinPruneInterval = pic.joinPruneIntervalDefault
   pimJoinPruneOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

def setPimJoinPruneInterval( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   joinPruneInterval = args[ 'INTERVAL_VALUE' ]
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )

   if pic is None:
      return
   pic.joinPruneInterval = joinPruneInterval
   pimJoinPruneOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

joinPruneIntervalRangeMatcher = CliMatcher.IntegerMatcher( 1, 18724,
      helpdesc='PIM join/prune interval in seconds' )

class RouterPimJoinPruneIntervalLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim join-prune-interval INTERVAL_VALUE'
   noOrDefaultSyntax = 'ip pim join-prune-interval [ INTERVAL_VALUE ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf, 
      'pim' : CliToken.Pim.pimNode,
      'join-prune-interval' : 'Join/prune interval', 
      'INTERVAL_VALUE' : joinPruneIntervalRangeMatcher
   }
   handler = setPimJoinPruneInterval
   noOrDefaultHandler = noPimJoinPruneInterval

modelet.addCommandClass( RouterPimJoinPruneIntervalLegacyCmd )

class RouterPimJoinPruneIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF join-prune interval INTERVAL_VALUE'
   noOrDefaultSyntax = 'pim AF join-prune interval ...'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'join-prune' : joinPruneKw,
      'interval' : 'PIM join/prune interval',
      'INTERVAL_VALUE' : joinPruneIntervalRangeMatcher
   }
   handler = setPimJoinPruneInterval
   noOrDefaultHandler = noPimJoinPruneInterval

modelet.addCommandClass( RouterPimJoinPruneIntervalCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim join-prune-count <count>( legacy )
# switch(config-if)# pim [ipv4|ipv6]  join-prune count <count>
#------------------------------------------------------------------------------------
def noPimJoinPruneCount( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )
   if pic is None:
      return
   pic.joinPruneCount = pic.joinPruneCountDefault
   pimJoinPruneOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

def setPimJoinPruneCount( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )
   joinPruneCount = args[ 'COUNT_VALUE' ]

   if pic is None:
      return
   pic.joinPruneCount = joinPruneCount
   pimJoinPruneOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

joinPruneCountRangeMatcher = CliMatcher.FloatMatcher( 1.5, HELLO_HOLD_FOREVER,
      helpdesc='Number of missed join/prune after which the route is expired',
      precisionString='%.25g' )

class RouterPimJoinPruneCountLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim join-prune-count COUNT_VALUE'
   noOrDefaultSyntax = 'ip pim join-prune-count [ COUNT_VALUE ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'join-prune-count' : 'Used to calculate holdtime that is included in the ' 
                           'join/prune message',
      'COUNT_VALUE' : joinPruneCountRangeMatcher
   }
   handler = setPimJoinPruneCount
   noOrDefaultHandler = noPimJoinPruneCount

modelet.addCommandClass( RouterPimJoinPruneCountLegacyCmd )

class RouterPimJoinPruneCountCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF join-prune count COUNT_VALUE'
   noOrDefaultSyntax = 'pim AF join-prune count ...'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'join-prune' : joinPruneKw,
      'count' : 'Used to calculate holdtime that is included in the ' 
                'join/prune message',
      'COUNT_VALUE' : joinPruneCountRangeMatcher
   }
   handler = setPimJoinPruneCount
   noOrDefaultHandler = noPimJoinPruneCount

modelet.addCommandClass( RouterPimJoinPruneCountCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim query-interval <interval> ( legacy )
# switch(config-if)# pim [ipv4|ipv6] hello interval <interval>
#------------------------------------------------------------------------------------
def pimQueryOverflowCheck( mode, pic ):
   maxHelloInterval = HELLO_HOLD_FOREVER / pic.helloCount
   if pic.helloInterval > maxHelloInterval:
      mode.addWarning( "The advertized hold-time will be set to %d as "
                       "the product of the hello interval and hello count exceeds "
                       "%d" % ( HELLO_HOLD_FOREVER, HELLO_HOLD_FOREVER ) )

def noPimHelloInterval( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )

   if pic is None:
      return
   pic.helloInterval = pic.helloIntervalDefault
   pimQueryOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

def setPimHelloInterval( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )
   helloInterval = args[ 'HELLO_INTERVAL' ]

   if pic is None:
      return
   pic.helloInterval = helloInterval
   pimQueryOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

helloIntervalRangeMatcher = CliMatcher.IntegerMatcher( 1, HELLO_HOLD_FOREVER,
      helpdesc='PIM hello interval in seconds' )

class RouterPimHelloIntervalLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim query-interval HELLO_INTERVAL'
   noOrDefaultSyntax = 'ip pim query-interval ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'query-interval' : CliCommand.Node( 
                            CliMatcher.KeywordMatcher( 'query-interval',
                            helpdesc='Time between hello packets' ),
                            deprecatedByCmd='pim ipv4 hello interval' ),
      'HELLO_INTERVAL' : helloIntervalRangeMatcher
   }
   handler = setPimHelloInterval
   noOrDefaultHandler = noPimHelloInterval

modelet.addCommandClass( RouterPimHelloIntervalLegacyCmd )

class RouterPimHelloIntervalCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF hello interval HELLO_INTERVAL'
   noOrDefaultSyntax = 'pim AF hello interval ...'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'hello' : helloKw,
      'interval' : 'Time between hello packets',
      'HELLO_INTERVAL' : helloIntervalRangeMatcher
   }
   handler = setPimHelloInterval
   noOrDefaultHandler = noPimHelloInterval

modelet.addCommandClass( RouterPimHelloIntervalCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim query-count <count> ( legacy )
# switch(config-if)# pim [ipv4|ipv6] hello count <count>
#------------------------------------------------------------------------------------
def noPimHelloCount( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )

   if pic is None:
      return
   pic.helloCount = pic.helloCountDefault
   pimQueryOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

def setPimHelloCount( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )
   helloCount = args[ 'COUNT_VALUE' ]

   if pic is None:
      return
   pic.helloCount = helloCount
   pimQueryOverflowCheck( mode, pic )
   _maybeDelPimIntfConfig( pic, af )

helloCountRangeMatcher = CliMatcher.FloatMatcher( 1.5, HELLO_HOLD_FOREVER,
      helpdesc='Number of missed hellos after which the neighbor expires',
      precisionString='%.25g' )

class RouterPimHelloCountLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim query-count COUNT_VALUE'
   noOrDefaultSyntax = 'ip pim query-count ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'query-count' : 'Number of missed hellos after which the neighbor expires',
      'COUNT_VALUE' : helloCountRangeMatcher 
   }
   handler = setPimHelloCount
   noOrDefaultHandler = noPimHelloCount

modelet.addCommandClass( RouterPimHelloCountLegacyCmd )

class RouterPimHelloCountCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF hello count COUNT_VALUE'
   noOrDefaultSyntax = 'pim AF hello count ...'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'hello' : helloKw,
      'count' : 'Number of missed hellos after which the neighbor expires',
      'COUNT_VALUE' : helloCountRangeMatcher
   }
   handler = setPimHelloCount
   noOrDefaultHandler = noPimHelloCount

modelet.addCommandClass( RouterPimHelloCountCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim dr-priority <priority>( legacy )
# switch(config-if)# pim [ipv4|ipv6] dr-priority <priority>
#------------------------------------------------------------------------------------
def setPimDrPriority( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   if CliCommand.isNoOrDefaultCmd( args ):
      pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )
   else:
      pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )
   
   if pic is None:
      return
   drPriority = args.get( 'PRIORITY_VALUE', pic.helloPriorityDefault )
   pic.helloPriority = drPriority
   _maybeDelPimIntfConfig( pic, af )

drPriorityRangeMatcher = CliMatcher.IntegerMatcher( 0, 4294967295,
      helpdesc='PIM designated router priority' )

class RouterPimDrPriorityLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim dr-priority PRIORITY_VALUE'
   noOrDefaultSyntax = 'ip pim dr-priority ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'dr-priority' : 'DR priority',
      'PRIORITY_VALUE' : drPriorityRangeMatcher 
   }
   handler = setPimDrPriority
   noOrDefaultHandler = handler

modelet.addCommandClass( RouterPimDrPriorityLegacyCmd )

class RouterPimDrPriorityCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF dr-priority PRIORITY_VALUE'
   noOrDefaultSyntax = 'pim AF dr-priority ...'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'dr-priority' : 'DR priority',
      'PRIORITY_VALUE' : drPriorityRangeMatcher
   }
   handler = setPimDrPriority
   noOrDefaultHandler = handler

modelet.addCommandClass( RouterPimDrPriorityCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# pim ipv4 non-dr install-oifs
#------------------------------------------------------------------------------------
def noPimNonDrInstallOifs( mode, args ):
   af = args[ 'AF' ]
   pic = getPimIntfConfigFromAf( mode, af, legacy=False, printError=False )
   if pic is None:
      return
   pic.fastFailover = False
   _maybeDelPimIntfConfig( pic, af )

def pimNonDrInstallOifs( mode, args ):
   # Explicitly move to new format
   af = args[ 'AF' ]
   convertLegacyConfigPim( mode )
   pic = getPimIntfConfigFromAf( mode, af, legacy=False )
   if pic is None:
      return
   pic.fastFailover = True

class RouterPimNonDrInstallOifsCmd( CliCommand.CliCommandClass ):
   syntax = 'pim ipv4 non-DR install-oifs'
   noOrDefaultSyntax = 'pim ipv4 non-DR ...'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'ipv4' : McastCommonCliLib.ipv4Node,
      'non-DR' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'non-DR',
                    helpdesc='Non DR' ), guard=_egressAclGuard ),
      'install-oifs' : 'Install interface in route outgoing interface list'
   }
   handler = pimNonDrInstallOifs
   noOrDefaultHandler = noPimNonDrInstallOifs

modelet.addCommandClass( RouterPimNonDrInstallOifsCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim border-router ( legacy )
# switch(config-if)# pim [ipv4|ipv6] border-router
#------------------------------------------------------------------------------------
def noPimBorderRouter( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args, printError=False )

   if pic is None:
      return
   if pic.borderRouter:
      pic.borderRouter = False
      if af == AddressFamily.ipv4:
         _igmpImplicitlyEnabledIs( mode, pic, False )
   _maybeDelPimIntfConfig( pic, af )

def setPimBorderRouter( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )

   if pic is None:
      return
   if not pic.borderRouter:
      pic.borderRouter = True
      if af == AddressFamily.ipv4:
         _igmpImplicitlyEnabledIs( mode, pic, True )

class RouterPimBorderRouterLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim border-router'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'border-router' : 'Multicast border router'
   }
   handler = setPimBorderRouter
   noOrDefaultHandler = noPimBorderRouter

modelet.addCommandClass( RouterPimBorderRouterLegacyCmd )

class RouterPimBorderRouterCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF border-router'
   noOrDefaultSyntax = syntax
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'border-router' : 'Multicast Border Router'
   }
   handler = setPimBorderRouter
   noOrDefaultHandler = noPimBorderRouter

modelet.addCommandClass( RouterPimBorderRouterCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim sparse-mode ( legacy )
# switch(config-if)# pim [ipv4|ipv6] sparse-mode
#------------------------------------------------------------------------------------
def unsetMode( mode, af, legacy=False, sparseMode=None, bidir=None ):
   af = getAfFromIpFamilyRule( af, legacy )
   pic = getPimIntfConfigFromAf( mode, af, legacy=legacy, printError=False )
   if pic is None:
      return
   if pic.mode == 'modePimNone':
      _maybeDelPimIntfConfig( pic, af )
      return

   if sparseMode:
      if pic.mode == 'modePimSm':
         pic.mode = 'modePimNone'
      elif pic.mode == 'modePimSmAndBidir':
         pic.mode = 'modePimBidir'

   if bidir:
      if pic.mode == 'modePimBidir':
         pic.mode = 'modePimNone'
      elif pic.mode == 'modePimSmAndBidir':
         pic.mode = 'modePimSm'

   if af == AddressFamily.ipv4:
      _igmpImplicitlyEnabledIs( mode, pic, False )

   _maybeDelPimIntfConfig( pic, af )

def unsetSparseMode( mode, args ):
   ipFamily = args.get( 'AF', 'ipv4' )
   unsetMode( mode, ipFamily, sparseMode=True, legacy='ip' in args )

def setSparseMode( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   legacy = 'ip' in args
   pic = getPimIntfConfigFromAf( mode, af, legacy )
   if pic is None:
      return
   if pic.mode == 'modePimBidir':
      pic.mode = 'modePimSmAndBidir'
   elif pic.mode == 'modePimNone':
      pic.mode = 'modePimSm'
   if not legacy:
      convertLegacyConfigPim( mode )
   if legacy == True or af == AddressFamily.ipv4:
      _igmpImplicitlyEnabledIs( mode, pic, True )

class RouterPimSparseModeLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim sparse-mode'
   noOrDefaultSyntax = syntax
   data = { 
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'sparse-mode' : 'PIM sparse-mode information'
   }
   handler = setSparseMode
   noOrDefaultHandler = unsetSparseMode

modelet.addCommandClass( RouterPimSparseModeLegacyCmd )

class RouterPimSparseModeCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF sparse-mode'
   noOrDefaultSyntax = syntax
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'sparse-mode' : CliToken.Pim.sparseModeMatcher
   }
   handler = setSparseMode
   noOrDefaultHandler = unsetSparseMode

modelet.addCommandClass( RouterPimSparseModeCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# ip pim bidirectional
# switch(config-if)# pim ipv4 bidirectional
#------------------------------------------------------------------------------------
def unsetBidirMode( mode, args ):
   ipFamily = args.get( 'AF', 'ipv4' )
   unsetMode( mode, ipFamily, bidir=True, legacy='ip' in args )

def setBidirMode( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   legacy = 'ip' in args
   pic = getPimIntfConfigFromAf( mode, af, legacy )

   if pic is None:
      return
   if pic.mode == 'modePimSm':
      pic.mode = 'modePimSmAndBidir'
   elif pic.mode == 'modePimNone':
      pic.mode = 'modePimBidir'
   if not legacy:
      convertLegacyConfigPim( mode )
   _igmpImplicitlyEnabledIs( mode, pic, True )

bidirNode = CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'bidirectional', 
   helpdesc='Bidirectional mode PIM' ), guard=_pimBidirectionalGuard )

class RouterPimBidirModeLegacyCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim bidirectional'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'bidirectional' : bidirNode
   }
   handler = setBidirMode
   noOrDefaultHandler = unsetBidirMode

modelet.addCommandClass( RouterPimBidirModeLegacyCmd )

class RouterPimBidirModeCmd( CliCommand.CliCommandClass ):
   syntax = 'pim ipv4 bidirectional'
   noOrDefaultSyntax = syntax
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'ipv4' : McastCommonCliLib.ipv4Node,
      'bidirectional' : bidirNode
   }
   handler = setBidirMode
   noOrDefaultHandler = unsetBidirMode

modelet.addCommandClass( RouterPimBidirModeCmd )

#-----------------------------------------------------------------------------------
# switch(config-if)# no ip pim
# switch(config-if)# no pim [ipv4|ipv6]
#-----------------------------------------------------------------------------------
def unsetAllMode( mode, args ):
   legacy = 'ip' in args
   if not legacy:
      af = args.get( 'AF', AddressFamily.ipunknown )
   else:
      af = AddressFamily.ipv4

   if af in [ AddressFamily.ipunknown, AddressFamily.ipv4 ]:
      unsetMode( mode, AddressFamily.ipv4, sparseMode=True, bidir=True,
                 legacy=legacy )
   if af in [ AddressFamily.ipunknown, AddressFamily.ipv6 ]:
      unsetMode( mode, AddressFamily.ipv6, sparseMode=True, bidir=True,
                 legacy=legacy )

class RouterPimUnsetAllModeLegacyCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'ip pim'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode
   }
   noOrDefaultHandler = unsetAllMode

modelet.addCommandClass( RouterPimUnsetAllModeLegacyCmd )

class RouterPimUnsetAllModeCmd( CliCommand.CliCommandClass ):
   noOrDefaultSyntax = 'pim [ AF ]'
   data = {
      'pim' : CliToken.Pim.pimMatcherForIntfConfig,
      'AF' : McastCommonCliLib.IpFamilyExpr
   }
   noOrDefaultHandler = unsetAllMode

modelet.addCommandClass( RouterPimUnsetAllModeCmd )

#------------------------------------------------------------------------------------
# legacy: switch(config-router-pim-*)# [ no ] ip pim bfd
# switch(config-router-pim-sparse-*)# [ no ] bfd
#------------------------------------------------------------------------------------

matcherBfd = CliMatcher.KeywordMatcher( 'bfd',
   helpdesc='Set the default for whether Bidirectional Forwarding Detection is '
            'enabled for PIM' )
bfdDeprecatedNode = CliCommand.Node( CliMatcher.KeywordMatcher( 'bfd',
   helpdesc='Set the default for whether BiDirectional Forwarding Detection is '
            'enabled for PIM' ),
   deprecatedByCmd='bfd in router-pim-sparse-ipv4 mode' )

class IpPimBfdCmdLegacy( CliCommand.CliCommandClass ):
   syntax = 'ip pim bfd'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'bfd' : bfdDeprecatedNode
   }

   handler = setPimBfdDefault
   noOrDefaultHandler = setPimBfdDefault

RouterPimSparseSharedModelet.addCommandClass( IpPimBfdCmdLegacy )

class IpPimBfdGlobalCmdLegacy( CliCommand.CliCommandClass ):
   syntax = 'ip pim bfd'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'bfd' : bfdDeprecatedNode
   }

   handler = setPimBfdDefault
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( IpPimBfdGlobalCmdLegacy )

class IpPimBfdCmd( CliCommand.CliCommandClass ):
   syntax = 'bfd'
   noOrDefaultSyntax = syntax
   data = {
      'bfd' : matcherBfd
   }

   handler = setPimBfdDefault
   noOrDefaultHandler = handler

RouterPimSparseAfCommonSharedModelet.addCommandClass( IpPimBfdCmd )

#------------------------------------------------------------------------------------
# switch(config-if)# [ no ] ip pim bfd-instance ( legacy )
# switch(config-if)# [ no ] pim [ ipv4|ipv6 ] bfd
#------------------------------------------------------------------------------------

matcherBfdInstanceDeprecated = CliCommand.Node( CliMatcher.KeywordMatcher( 
      'bfd-instance',
      helpdesc='Enable Bidirectional Forwarding Detection' ),
      deprecatedByCmd='pim ipv4 bfd in interface mode')
matcherBfdIntf = CliMatcher.KeywordMatcher( 'bfd',
      helpdesc='Enable Bidirectional Forwarding Detection' )

def setPimBfdForIntf( mode, args ):
   af = args.get( 'AF', 'ipv4' )
   pic = getPimIntfConfigFromAf( mode, af, legacy='ip' in args )
   if pic is None:
      return
   # Override the global default to enable/disable BFD.
   pic.bfd = 'bfdEnabled' if not CliCommand.isNoCmd( args ) else 'bfdDisabled'
   if CliCommand.isDefaultCmd( args ):
      # Use the global default.
      pic.bfd = 'bfdDefault'
   _maybeDelPimIntfConfig( pic, af )

class IpPimBfdIntCmdLegacy( CliCommand.CliCommandClass ):
   syntax = 'ip pim bfd-instance'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfigIf,
      'pim' : CliToken.Pim.pimNode,
      'bfd-instance' : matcherBfdInstanceDeprecated
   }
   handler = setPimBfdForIntf
   noOrDefaultHandler = handler

# to be deprecated
modelet.addCommandClass( IpPimBfdIntCmdLegacy )

class IpPimBfdIntCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF bfd'
   noOrDefaultSyntax = syntax
   data = {
      'pim' : CliToken.Pim.pimNode,
      'AF' : McastCommonCliLib.IpFamilyExpr,
      'bfd' : matcherBfd
   }
   handler = setPimBfdForIntf
   noOrDefaultHandler = handler

# new command
modelet.addCommandClass( IpPimBfdIntCmd )

#------------------------------------------------------------------------------------
# legacy: switch(config)# [no|default] ip pim allow-rp
# (config-router-pim-sparse-*) [no|default] rp allow
#------------------------------------------------------------------------------------

allowRpMatcher = CliMatcher.KeywordMatcher( 'allow-rp',
               helpdesc='allows (*,G )s from RPs other than those defined.' )

class IpPimAllowRpLegacy( CliCommand.CliCommandClass ):
   syntax = 'ip pim allow-rp'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'allow-rp' : allowRpMatcher
   }

   handler = setAllowRp
   noOrDefaultHandler = handler

class IpPimRpAllow( CliCommand.CliCommandClass ):
   syntax = 'rp allow'
   noOrDefaultSyntax = syntax
   data = {
      'rp' : CliToken.Pim.rpMatcher,
      'allow' : CliToken.Pim.allowMatcher
   }

   handler = setAllowRp
   noOrDefaultHandler = handler

RouterPimSparseSharedModelet.addCommandClass( IpPimAllowRpLegacy )

RouterPimSparseIpv4Mode.addCommandClass( IpPimRpAllow )

RouterPimSparseAfCommonSharedModelet.addCommandClass( IpPimRpAllow )

#------------------------------------------------------------------------------------
# legacy: switch(config)# [ no ] ip pim log-neighbor-changes
# switch(config-router-pim-*)# [ no ] log neighbors
#------------------------------------------------------------------------------------

logKw = CliMatcher.KeywordMatcher( 'log', helpdesc='Configure logging' )
neighborsKw = CliMatcher.KeywordMatcher( 'neighbors',
      helpdesc='Syslogs for neighbor UP/DOWN events' )
logNeighborChangesNode = CliCommand.Node( CliMatcher.KeywordMatcher( 
      'log-neighbor-changes', helpdesc='Syslogs for neighbor UP/DOWN events' ),
      deprecatedByCmd='log neighbors in router-pim-sparse-ipv4 mode' )

class IpPimLogNeighborChangesCmdGlobalLegacy( CliCommand.CliCommandClass ):
   syntax = 'ip pim log-neighbor-changes'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'log-neighbor-changes' : logNeighborChangesNode
   }
   handler = logNeighborChanges
   noOrDefaultHandler = logNeighborChanges

BasicCli.GlobalConfigMode.addCommandClass( 
      IpPimLogNeighborChangesCmdGlobalLegacy )

class IpPimLogNeighborChangesCmdLegacy( CliCommand.CliCommandClass ):
   syntax = 'ip pim log-neighbor-changes'
   noOrDefaultSyntax = syntax
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'log-neighbor-changes' : logNeighborChangesNode
   }
   handler = logNeighborChanges
   noOrDefaultHandler = logNeighborChanges

RouterPimBidirSharedModelet.addCommandClass( 
      IpPimLogNeighborChangesCmdLegacy )
RouterPimSparseSharedModelet.addCommandClass(
      IpPimLogNeighborChangesCmdLegacy )

class IpPimLogNeighborChangesCmd( CliCommand.CliCommandClass ):
   syntax = 'log neighbors'
   noOrDefaultSyntax = syntax
   data = {
      'log' : logKw,
      'neighbors' : neighborsKw
   }
   handler = logNeighborChanges
   noOrDefaultHandler = logNeighborChanges

PimCliLib.RouterPimSparseAfCommonSharedModelet.addCommandClass( 
      IpPimLogNeighborChangesCmd )
PimBidirCliLib.RouterPimBidirAfCommonSharedModelet.addCommandClass( 
      IpPimLogNeighborChangesCmd )

#------------------------------------------------------------------------------------
# legacy: switch(config)# ip pim dr-notify-delay <period>
# switch(config-router-pim-sparse)# [no|default] dr-notify-delay <period>
#------------------------------------------------------------------------------------
def setPimDrNotifyDelay( mode, args ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      pimConfigRoot = pimConfigColl( af=af )
      delay = args.get( 'DR_NOTIFY_DELAY_RANGE', 
            pimConfigRoot.globalConfig.drNotifyDelayDefault )
      if delay < 0:
         pimConfigRoot.globalConfig.drNotifyDelay = False
         pimConfigRoot.globalConfig.drNotifyDelayTime = -delay
      else:
         pimConfigRoot.globalConfig.drNotifyDelay = True
         pimConfigRoot.globalConfig.drNotifyDelayTime = delay

drNotifyDelayRangeMatcher = CliMatcher.IntegerMatcher( -32767, 32768,
      helpdesc='PIM designated router notify delay' )

class RouterPimDrNotifyDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim dr-notify-delay DR_NOTIFY_DELAY_RANGE'
   noOrDefaultSyntax = 'ip pim dr-notify-delay ...'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'dr-notify-delay' : 'DR notify delay',
      'DR_NOTIFY_DELAY_RANGE' : drNotifyDelayRangeMatcher
   }
   handler = setPimDrNotifyDelay
   noOrDefaultHandler = handler

BasicCli.GlobalConfigMode.addCommandClass( RouterPimDrNotifyDelayCmd )

class RouterPimSparseDrNotifyDelayCmd( CliCommand.CliCommandClass ):
   syntax = 'dr-notify-delay DR_NOTIFY_DELAY_RANGE'
   noOrDefaultSyntax = 'dr-notify-delay ...'
   data = {
      'dr-notify-delay' : 'DR notify delay',
      'DR_NOTIFY_DELAY_RANGE' : drNotifyDelayRangeMatcher
   }
   handler = setPimDrNotifyDelay
   noOrDefaultHandler = handler

PimCliLib.RouterPimSparseSharedModelet.addCommandClass( 
      RouterPimSparseDrNotifyDelayCmd )

#------------------------------------------------------------------------------------
# switch(config-router-pim)# ip pim access-group <acl> [in]
#------------------------------------------------------------------------------------
def setPimServiceAcl( mode, args ):
   aclName = args[ 'ACL_GROUP_NAME' ]
   _setServiceAcl( mode, aclName, mode.vrfName )

def noPimServiceAcl( mode, args ):
   _setServiceAcl( mode, '', mode.vrfName )

class RouterPimServiceAclCmd( CliCommand.CliCommandClass ):
   syntax = 'ip pim access-group ACL_GROUP_NAME [ in ]'
   noOrDefaultSyntax = 'ip pim access-group [ ACL_GROUP_NAME ] [ in ]'
   data = {
      'ip' : CliToken.Ip.ipMatcherForConfig,
      'pim' : CliToken.Pim.pimNode,
      'access-group' : AclCli.accessGroupKwMatcher,
      'ACL_GROUP_NAME' : AclCli.ipAclNameMatcher,
      'in' : AclCli.inKwMatcher
   }
   handler = setPimServiceAcl
   noOrDefaultHandler = noPimServiceAcl

RouterPimSharedModelet.addCommandClass( RouterPimServiceAclCmd )

#-------------------------------------------------------------------------------
def convertLegacyConfigPim( mode ):
   _pimXLegacyConfig.legacy = False
   _pimXLegacyConfig.version = _pimXLegacyConfig.ipMode

   _pimLegacyConfig.legacy = False
   _pimLegacyConfig.version = _pimLegacyConfig.ipMode

   _pimBidirLegacyConfig.legacy = False
   _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.ipMode

#-------------------------------------------------------------------------------
#Register convertLegacyConfigPim via "config convert new-syntax"
#-------------------------------------------------------------------------------
CliPlugin.ConfigConvert.registerConfigConvertCallback( convertLegacyConfigPim )

#-------------------------------------------------------------------------------
# (config-router-pim-sparse-*) [no|default] ipv6
#-------------------------------------------------------------------------------
def gotoRouterPimSparseIpv6Mode( mode, args ):
   _pimLegacyConfig.version = _pimLegacyConfig.ipMode
   vrfName = _getVrfNameFromMode( mode )
   IraIp6Cli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterPimSparseIpv6Mode,
                               vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimSparseIpv6Mode( mode, args ):
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = pimConfigVrf( af=AddressFamily.ipv6, vrfName=vrfName )
   vrfConfig.reset()
   _pim6ConfigRoot = pimConfigColl( af=AddressFamily.ipv6 )
   if all( vrfConfig.isDefault() for vrfConfig in
            _pim6ConfigRoot.vrfConfig.itervalues()
            if vrfConfig.vrfName != DEFAULT_VRF ):
      _pimLegacyConfig.version = _pimLegacyConfig.routerMode
      _pimLegacyConfig.legacy = False

class RouterPimSparseIpv6ModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ipv6',
                  helpdesc='Configure Pim sparse-mode for address family IPv6' ),
                  guard=pimSparseModeIpv6SupportedGuard )
   }
   handler = gotoRouterPimSparseIpv6Mode
   noOrDefaultHandler = deleteRouterPimSparseIpv6Mode

RouterPimSparseSharedModelet.addCommandClass( RouterPimSparseIpv6ModeCmd )

#-------------------------------------------------------------------------------
# (config-router-pim-sparse-*) [no|default] ipv4
#-------------------------------------------------------------------------------
def gotoRouterPimSparseIpv4Mode( mode, args ):
   _pimLegacyConfig.version = _pimLegacyConfig.ipMode
   vrfName = _getVrfNameFromMode( mode )
   IraIpCli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterPimSparseIpv4Mode,
                               vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimSparseIpv4Mode( mode, args ):
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = pimConfigVrf( af=AddressFamily.ipv4, vrfName=vrfName )
   vrfConfig.reset()
   pimConfigRoot = pimConfigColl( af=AddressFamily.ipv4 )
   if all( vrfConfig.isDefault() for vrfConfig in
            pimConfigRoot.vrfConfig.itervalues()
            if vrfConfig.vrfName != DEFAULT_VRF ):
      _pimLegacyConfig.version = _pimLegacyConfig.routerMode
      _pimLegacyConfig.legacy = False

class RouterPimSparseIpv4ModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'ipv4' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ipv4',
                  helpdesc='Configure Pim sparse-mode for address family IPv4' ),
                  guard=pimSparseModeIpv4SupportedGuard )
   }
   handler = gotoRouterPimSparseIpv4Mode
   noOrDefaultHandler = deleteRouterPimSparseIpv4Mode

RouterPimSparseSharedModelet.addCommandClass( RouterPimSparseIpv4ModeCmd )

#-------------------------------------------------------------------------------
# (config-router-pim-bidir-*) [no|default] ipv4
#-------------------------------------------------------------------------------
def gotoRouterPimBidirIpv4Mode( mode, args ):
   _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.ipMode
   vrfName = _getVrfNameFromMode( mode )
   IraIpCli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterPimBidirIpv4Mode,
                               vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimBidirIpv4Mode( mode, args ):
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = bidirConfigVrf( af=AddressFamily.ipv4, vrfName=vrfName )
   vrfConfig.reset()
   pimConfigRoot = bidirConfigColl( af=AddressFamily.ipv4 )
   if all( vrfConfig.isDefault() for vrfConfig in
            pimConfigRoot.vrfConfig.itervalues()
            if vrfConfig.vrfName != DEFAULT_VRF ):
      _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.routerMode
      _pimBidirLegacyConfig.legacy = False

class RouterPimBidirIpv4ModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv4'
   noOrDefaultSyntax = syntax
   data = {
      'ipv4' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ipv4',
                  helpdesc='Configure Pim bidir-mode for address family IPv4' ),
                  guard=pimBidirModeIpv4SupportedGuard )
   }
   handler = gotoRouterPimBidirIpv4Mode
   noOrDefaultHandler = deleteRouterPimBidirIpv4Mode

RouterPimBidirSharedModelet.addCommandClass( RouterPimBidirIpv4ModeCmd )

#-------------------------------------------------------------------------------
# (config-router-pim-bidir-*) [no|default] ipv6
#-------------------------------------------------------------------------------
def gotoRouterPimBidirIpv6Mode( mode, args ):
   _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.ipMode
   vrfName = _getVrfNameFromMode( mode )
   IraIp6Cli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterPimBidirIpv6Mode,
                               vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimBidirIpv6Mode( mode, args ):
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = bidirConfigVrf( af=AddressFamily.ipv6, vrfName=vrfName )
   vrfConfig.reset()
   _pim6ConfigRoot = bidirConfigColl( af=AddressFamily.ipv6 )
   if all( vrfConfig.isDefault() for vrfConfig in
            _pim6ConfigRoot.vrfConfig.itervalues()
            if vrfConfig.vrfName != DEFAULT_VRF ):
      _pimBidirLegacyConfig.version = _pimBidirLegacyConfig.routerMode
      _pimBidirLegacyConfig.legacy = False

class RouterPimBidirIpv6ModeCmd( CliCommand.CliCommandClass ):
   syntax = 'ipv6'
   noOrDefaultSyntax = syntax
   data = {
      'ipv6' : CliCommand.Node( matcher=CliMatcher.KeywordMatcher( 'ipv6',
                  helpdesc='Configure Pim bidir-mode for address family IPv6' ),
                  guard=pimBidirModeIpv6SupportedGuard )
   }
   handler = gotoRouterPimBidirIpv6Mode
   noOrDefaultHandler = deleteRouterPimBidirIpv6Mode

RouterPimBidirSharedModelet.addCommandClass( RouterPimBidirIpv6ModeCmd )

#-------------------------------------------------------------------------------
# (config-if-*) [no|default] pim [ipv4|ipv6] join-prune transport sctp
#-------------------------------------------------------------------------------
def setSctpTransport( mode, args ):
   no = CliCommand.isNoOrDefaultCmd( args )
   protocol = 'sctp'
   if not no:
      # Explicitly move to new format
      convertLegacyConfigPim( mode )
   af = args[ 'AF' ]
   pic = getPimIntfConfigFromAf( mode, af, legacy=False, printError=not no )
   if pic is None:
      return

   if no or protocol != 'sctp':
      portFlags = Tac.Value( "Routing::Pim::Reliable::CapabilityFlags" )
   else:
      portFlags = Tac.Value( "Routing::Pim::Reliable::CapabilityFlags", 1 )

   pic.portCapabilityFlags = portFlags
   _maybeDelPimIntfConfig( pic, af )

if Toggles.PimToggleLib.togglePimIpv6PortEnabled():
   ipMatcher = McastCommonCliLib.IpFamilyExpr
else:
   ipMatcher = McastCommonCliLib.ipv4Node

class RouterPimSctpTransportCmd( CliCommand.CliCommandClass ):
   syntax = 'pim AF join-prune transport sctp'
   noOrDefaultSyntax = 'pim AF join-prune transport ...'
   data = {
      'pim' : CliToken.Pim.pimNode,
      'AF' : ipMatcher,
      'join-prune' : joinPruneKw,
      'transport' : transportKw,
      'sctp' : sctpKw 
   }
   handler = setSctpTransport
   noOrDefaultHandler = handler 

modelet.addCommandClass( RouterPimSctpTransportCmd )

#-------------------------------------------------------------------------------
# (config-router-pim-sparse-*ipv?)
# [no|default] transport sctp source-interface <interface>
#-------------------------------------------------------------------------------
def setSctpSrcInterface( mode, args ):
   intf = args.get( 'INTF' )
   config = pimConfigVrfFromMode( mode )
   if CliCommand.isNoOrDefaultCmd( args ):
      config.sctpSrcIntf = Tac.Value( "Arnet::IntfId" )
   else:
      config.sctpSrcIntf = intf.name

class RouterPimSparseSctpSrcInterfaceCmd( CliCommand.CliCommandClass ):
   syntax = 'transport sctp source-interface INTF'
   noOrDefaultSyntax = 'transport sctp source-interface ...'
   data = {
      'transport' : transportKw,
      'sctp' : sctpKw,
      'source-interface' : srcInterfaceKw,
      'INTF' : IntfCli.Intf.matcherWithIpSupport
   }
   handler = setSctpSrcInterface
   noOrDefaultHandler = handler

RouterPimSparseIpv4Mode.addCommandClass( RouterPimSparseSctpSrcInterfaceCmd )

if Toggles.PimToggleLib.togglePimIpv6PortEnabled():
   RouterPimSparseIpv6Mode.addCommandClass( RouterPimSparseSctpSrcInterfaceCmd )

#-------------------------------------------------------------------------------
def Plugin( entityManager ):
   global _pimConfigRoot, _ipConfig, _ipStatus
   global _allVrfConfig, _mfibVrfConfig, _routingInfoVrfStatus
   global _pimXLegacyConfig
   global _pimLegacyConfig
   global _pimBidirLegacyConfig
   global _routingHwStatus, _routing6HwStatus
   global _mrouteLegacyConfig
   global ipGenStatus, _mfibVrfConfig6
   global _routing6InfoVrfStatus
   global aclConfig, aclCpConfig

   configTypes = [ PimConfigColl ]
   configTypesBoth = [ PimRpConfigColl, StaticRpConfigColl ]
   RouterMulticastCliLib.doConfigMounts( entityManager, configTypes )
   RouterMulticastCliLib.doConfigMounts( entityManager, configTypesBoth,
                                                protocol='sparsemode' )
   RouterMulticastCliLib.doConfigMounts( entityManager, configTypesBoth,
                                                   protocol='bidir' )
   _pimConfigRoot = ConfigMount.mount( entityManager, 'routing/pim/config',
                                   'Routing::Pim::ConfigColl', 'wi' )
   _pimXLegacyConfig = ConfigMount.mount( entityManager,
                                 'routing/pim/pimlegacyconfig',
                                 'McastCommon::LegacyConfig', 'w' )
   _pimLegacyConfig = ConfigMount.mount( entityManager, 'routing/pim/legacyconfig',
                                 'McastCommon::LegacyConfig', 'w' )
   _pimBidirLegacyConfig = ConfigMount.mount( entityManager,
                                 'routing/pim/bidir/legacyconfig',
                                 'McastCommon::LegacyConfig', 'w' )
   _mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
                                     'Routing::Multicast::Fib::VrfConfig', 'r' )

   _mfibVrfConfig6 = LazyMount.mount( entityManager, 'routing6/multicast/vrf/config',
                                       'Routing::Multicast::Fib::VrfConfig', 'r' )

   _routingInfoVrfStatus = LazyMount.mount( entityManager,
                                            'routing/vrf/routingInfo/status',
                                            'Tac::Dir', 'r' )
   _routing6InfoVrfStatus = LazyMount.mount( entityManager,
                                            'routing6/vrf/routingInfo/status',
                                            'Tac::Dir', 'r' )

   _ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )
   _ipStatus = LazyMount.mount( entityManager, 'ip/status', 'Ip::Status', 'r' )
   _ip6Config = LazyMount.mount( entityManager, 'ip6/config', 'Ip6::Config', 'r' )
   _ip6Status = LazyMount.mount( entityManager, 'ip6/status', 'Ip6::Status', 'r' )

   ipGenStatus = McastCommonCliLib.ipGenStatusInit( _ipConfig, _ipStatus, _ip6Config,
                                                    _ip6Status )

   McastCommonCliLib.doReadMounts( entityManager,
                                   [ "Routing::Multicast::Fib::VrfConfig" ] )


   _allVrfConfig = LazyMount.mount( entityManager, 'ip/vrf/config',
                                   'Ip::AllVrfConfig', 'r' )

   _routingHwStatus = LazyMount.mount( entityManager,
         'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )

   _routing6HwStatus = LazyMount.mount( entityManager,
         'routing6/hardware/status',
         'Routing6::Hardware::Status', 'r' )

   _mrouteLegacyConfig = LazyMount.mount( entityManager,
         'routing/multicast/legacyconfig',
         'Routing::Multicast::MulticastLegacyConfig', 'r' )

   aclConfig = ConfigMount.mount( entityManager, "acl/config/cli",
                                  "Acl::Input::Config", "w" )

   aclCpConfig = ConfigMount.mount( entityManager, "acl/cpconfig/cli",
                                    "Acl::Input::CpConfig", "w" )

   McastCommonCliLib.mcastIfCollHook.addExtension( _pimIntfConfColl )

   MrouteCli.routerMcastVrfDefinitionHook.addExtension(
         _sparseModeVrfDefinitionHook )
   MrouteCli.routerMcastVrfDeletionHook.addExtension(
         _sparseModeVrfDeletionHook )
   MrouteCli.legacyConfigHook.addExtension( _pimLegacyConfigHook )
   MrouteCli.routerMcastVrfDefinitionHook.addExtension(
         _bidirModeVrfDefinitionHook )
   MrouteCli.routerMcastVrfDeletionHook.addExtension(
         _bidirModeVrfDeletionHook )
