#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
import Arnet
from CliModel import Model, List, Int, Str, Bool, Submodel
from ArnetModel import Ip4Address, IpGenericAddress, IpGenericPrefix, IpAddrAndMask
from IntfModels import Interface

class StaticRp( Model ):
   mGroup = Str( help='group address' )
   mRpList = List( valueType=str, help="List of RPs configured" )

   def render( self ):
      if self.mGroup:
         print "Multiple RPs are configured for %s: " % self.mGroup
         for rp in sorted( self.mRpList ):
            print "%s " % rp

class RouteCheckBase( Model ):
   nodeType = Str( help="Node type " )
   noRoute = Bool( "No route to the prefix" )
   nonPimIntf = List( valueType=Interface, help="Route via non-Pim interface" )
   hintsFound = Int( help='number of hints found' )

   def render( self ):
      if not self.hintsFound:
         return
      if self.noRoute:
         print "No route to %s %s" % ( self.nodeType, self.prefix )
      if len( self.nonPimIntf ):
         print "Route to %s %s goes via non-PIM interfaces:" \
             % ( self.nodeType, self.prefix )
         for k in sorted( self.nonPimIntf ):
            print "%s " % k
      if len( self.nonPimHop ):
         print "Route to %s %s goes via non-PIM hops:" \
             % ( self.nodeType, self.prefix )
         for k in sorted( self.nonPimHop ):
            print "%s " % k

class RouteCheck( RouteCheckBase ):
   prefix = Ip4Address( help="Route check for this prefix" )
   nonPimHop = List( valueType=Ip4Address, help="Route via non-Pim hops" )

class GenRouteCheck( RouteCheckBase ):
   prefix = IpGenericAddress( help="Route check for this prefix" )
   nonPimHop = List( valueType=IpGenericAddress, help="Route via non-Pim hops" )

def _strGr( g ):
   if g == Arnet.IpGenPrefix( '224.0.0.0/4' ):
      return g.stringValue + "(default)"
   return g.stringValue

class OverlapGroup( Model ):
   group = IpGenericPrefix( help='Group address prefix' )
   overlaps = List( valueType=IpGenericPrefix, 
                    help='List of overlaping group ranges' )
   def render( self ):
      if self.overlaps:
         print "Group %s overlaps with groups:" % _strGr( self.group )
         istr = " " * 8
         lineStart = True
         for k in sorted( self.overlaps ):
            if lineStart:
               istr += _strGr( k )
               lineStart = False
            else:
               istr += ", %s" % _strGr( k )
         print istr
         
class StaticRpSet( Model ):
   noRp = Bool( help='No RP configured' )
   noDefaultRp = Bool( help='No default RP configured' )
   rpSet = List( valueType=StaticRp, help='List of group RP mappings' )
   overlapSet = List( valueType=OverlapGroup,
                      help='List of overlaping group ranges' )
   rcModel = List( valueType=RouteCheck,  help='Route config check' )
   hintsFound = Int( help='number of hints found' )
   def render( self ):
      if not self.hintsFound:
         return
      if self.noRp:
         print "No RP configured"
      if self.noDefaultRp:
         print "No default RP configured (no RP for '224.0.0.0/4')"
      for k in self.rcModel:
         if k.hintsFound:
            k.render()
      for rp in self.rpSet:
         rp.render()
      if self.overlapSet:
         print "The overlapping group ranges with distinct RPs:"
         for k in self.overlapSet:
            k.render()

class NotDefinedBAcl( Model ):
   acl = Str( help='Acl name' )
   intf = Interface( help='Interface Name' )
   def render( self ):
      print "%s boundary acl %s is not defined yet" % \
          ( self.intf, self.acl )

class NotStandardBAcl( Model ):
   acl = Str( help='Acl name' )
   intf = Interface( help='Interface Name' )
   def render( self ):
      print "%s boundary acl %s is not a standard Acl" % \
          ( self.intf, self.acl )

class InvalidBoundaryGroup( Model ):
   group = Submodel( valueType=IpAddrAndMask, help="Invalid group prefix" )
   reason = Str( help='Reason for the group being invalid' )
   def render( self ):
      print self.reason

class IgmpConfigCheck( Model ):
   noRpGroups = List( valueType=IpGenericAddress, help='Static group without RP' )
   rcModel = List( valueType=RouteCheck,  help='Route config check' )
   disabledIntfs = List( valueType=Interface, help='Disabled interfaces' )
   nonDrSgIntfs = List( valueType=Interface, help='Interfaces which are not DR' )
   nonDrIntfs = List( valueType=Interface, help='Interfaces which are not DR' )
   boundaryIntfs = List( valueType=Interface, 
                         help='Interfaces with multicast boundaries' )
   notDefinedBAcl = List( valueType=NotDefinedBAcl, help='Not defined Acl' )
   notStandardBAcl = List( valueType=NotStandardBAcl, help='Not standard Acl' )
   invalidBoundaryGroups = List( valueType=InvalidBoundaryGroup, 
         help='Invalid groups configured for boundary' )
   noRpGroupc = List( valueType=IpGenericAddress, help='Connected group without RP' )
   nonDrIntfs = List( valueType=Interface, help='Non DR interfaces' )
   hintsFound = Int( help='number of hints found' )
   def render( self ):
      if not self.hintsFound:
         return
      if len( self.noRpGroups ):
         print "No RP configured for static groups"
         for k in sorted( self.noRpGroups ):
            print "%s " % k
      for k in self.rcModel:
         if k.hintsFound:
            k.render()
      if len( self.disabledIntfs ):
         print "The interfaces with static groups configured but PIM disabled:"
         for k in sorted( self.disabledIntfs ):
            print "%-8s " % k
      if len( self.nonDrSgIntfs ):
         print "The interfaces with static groups configured, " \
             "where we are not DR:"
         for k in sorted( self.nonDrSgIntfs ):
            print "%-8s " % k
      if len( self.boundaryIntfs ):
         print "The interfaces with multicast boundaries:"
         for k in sorted( self.boundaryIntfs ):
            print "%-8s " % k
      for k in self.notDefinedBAcl:
         k.render()
      for k in self.notStandardBAcl:
         k.render()        
      for grp in self.invalidBoundaryGroups:
         grp.render()
      if len( self.noRpGroupc ):
         print "No RP configured for directly connected groups"
         for k in sorted( self.noRpGroupc ):
            print "%s " % k
      if len( self.nonDrIntfs ):
         print "The interfaces with directly connected groups, " \
             "where we are not DR:"
         for k in sorted( self.nonDrIntfs ):
            print "%-8s " % k  

class MrtConfigCheck( Model ):
   noRpGroups = List( valueType=IpGenericAddress, help='Group without RP' )
   rcModel = List( valueType=RouteCheck,  help='Route config check' )
   uroutes = Int( help='number of unresolved routes' )
   hintsFound = Int( help='number of hints found' )
   def render( self ):
      if not self.hintsFound:
         return
      if len( self.noRpGroups ):
         print "No RP configured for groups"
         for k in sorted( self.noRpGroups ):
            print "%s " % k
      for k in self.rcModel:
         if k.hintsFound:
            k.render()
      if self.uroutes > 0:
         print "There are %d unresolved multicast routes" % self.uroutes


class PimBidirConfigCheck( Model ):
   pimBidirSupported = Bool( help='PIM Bidirectional supported platform', 
                             optional=True )
   pimBidirEnabled = Bool( help='PIM Bidirectional enabled', optional=True )
   staticRpSet = Submodel( 
         help='staticRp config check for PIM Bidirectional mode',
         valueType = StaticRpSet, optional=True )
   mroute = Submodel( help='mroute config check for PIM Bidirectional mode',
         valueType = MrtConfigCheck, optional=True )
   def render( self ):
      if not self.pimBidirSupported:
         print "PIM Bidirectional mode not supported on this platform."
         return
      if not self.pimBidirEnabled:
         print "PIM Bidirectional is not configured"
         return
      if self.staticRpSet:
         self.staticRpSet.render()
      if self.mroute:
         self.mroute.render()

class PimConfigCheck( Model ):
   staticRpSet = Submodel( 
         help='staticRp config check for PIM sparse-mode',
         valueType = StaticRpSet, optional=True )
   mroute = Submodel( help='mroute config check for PIM sparse-mode',
         valueType = MrtConfigCheck, optional=True )
   def render( self ):
      if self.staticRpSet:
         self.staticRpSet.render()
      if self.mroute:
         self.mroute.render()

class IntfConfigCheck( Model ):
   disabled = Bool( help=' PIM is disabled ' )
   intfsNoIp = List( valueType=Interface, help='Interfaces without IP address' )
   intfsDown = List( valueType=Interface, help='Interfaces which are down' )
   intfsNotRouted = List( valueType=Interface, 
                          help='Interfaces where routing not enabled' )
   intfsSparseModeBorderRouter = List( valueType=Interface,
      help='Interfaces with PIM sparse-mode and multicast border router enabled' )
   hintsFound = Int( help='number of hints found' )
   def render( self ):
      if not self.hintsFound:
         return
      if self.disabled:
         print "\nPIM is not enabled on any interface"
      if len( self.intfsNoIp ):
         print "The interfaces with PIM but without IP address:"
         for k in sorted( self.intfsNoIp ):
            print "%s " % k 
      if len( self.intfsDown ):
         print "The interfaces with PIM which are down:"
         for k in sorted( self.intfsDown ):
            print "%s " % k 
      if len( self.intfsNotRouted ):
         print "The interfaces with PIM where routing is not enabled:"
         for k in sorted( self.intfsNotRouted ):
            print "%s " % k
      if len( self.intfsSparseModeBorderRouter ):
         print "The interfaces with PIM sparse-mode" \
               " and multicast border router enabled."
         for k in sorted( self.intfsSparseModeBorderRouter ):
            print "%s " % k

class InvalidHello( Model ):
   delta = Int( help='Time difference in second between invalid HELLOS' )
   intfs = List( valueType=Interface, help='interfaces with secondary IP address' )
   def render( self ):
      print "%d invalid HELLOs received since last check. " \
          "Could it be due to secondary IP addresses?\n" \
          "The interfaces with PIM and secondary IP address:" % self.delta
      for k in sorted( self.intfs ):
         print "%s " % k

class NbrConfigCheck( Model ):
   invalidHello = List( valueType=InvalidHello, help='List of invalid HELLOs' )
   delta = Int( help='time difference' )
   nfIntf = List( valueType=Interface, help='Neighbor filter enabled interfaces' )
   hintsFound = Int( help='number of hints found' )
   def render( self ):
      if not self.hintsFound:
         return
      for k in self.invalidHello:
         k.render()
      if not self.invalidHello and self.delta > 0:
         print "%d invalid HELLOs received since last check. "\
                   "Misconfigured IP addresses?" % self.delta
      if len( self.nfIntf ):
         print "Neighbor filters are enabled on the following PIM interfaces:"
         for k in sorted( self.nfIntf ):
            print "%s " % k
