# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from AclCli import  getAclConfig
from Arnet import IpGenAddr, IpGenPrefix
import Arnet, IpUtils, IntfCli
import CliParser, GmpCli
import PimConfigCheckModel
import Tac
import CliExtensions
import LazyMount
import ArnetModel
import McastBoundaryCli
import PimCountersLib
from CliMode.Pim import RoutingPimSparseMode, RoutingPimSparseVrfMode
from CliMode.Pim import RoutingPimSparseAfMode
from CliMode.Pim import RoutingPimMode, RoutingPimVrfMode, RoutingPimAfMode
import CliPlugin.VrfCli as VrfCli
import BasicCli
from IpLibConsts import DEFAULT_VRF, VRFNAMES_RESERVED
import McastCommonCliLib
from IraCommonCli import AddressFamily
from PimBidirCliLib import isBidirMode

# pylint: disable-msg=W0612
# pylint: disable-msg=W0105
# pylint: disable-msg=E1120

pimCliError = "PimCli::Error"
zeroAddr = Tac.newInstance( "Arnet::IpGenPrefix" )

configSanityCallbacks = []
def getPimVrfConfig( pimConfig, vrfName ):
   pimVrfConfig = None
   if vrfName in pimConfig.vrfConfig:
      pimVrfConfig = pimConfig.vrfConfig[ vrfName ]
   return pimVrfConfig

def getPimBidirVrfConfig( pimConfig, vrfName ):
   pimBidirVrfConfig = None
   if vrfName in pimConfig.bidirVrfConfig:
      pimBidirVrfConfig = pimConfig.bidirVrfConfig[ vrfName ]
   return pimBidirVrfConfig

def validVrfName ( mode, vrfName, mfibVrfConfig, af=AddressFamily.ipv4 ):
   versionStr = "ipv4" if af == AddressFamily.ipv4 else "ipv6"
   afStr = "" if af == AddressFamily.ipv4 else "6"
   vrfErrorMsg = "%s multicast routing is not configured on VRF %s. " \
         % ( versionStr, vrfName )
   pimErrorMsg = "PIM%s is not running" % afStr

   if vrfName is None or vrfName == '':
      return False
   if vrfName == 'all':
      flag = False

      for vrf in mfibVrfConfig.config:
         mcastEnabledVrf = isMulticastRoutingEnabled( vrf, mfibVrfConfig )
         if not mcastEnabledVrf:
            mode.addError( vrfErrorMsg )
         flag |= mcastEnabledVrf
      if not flag:
         mode.addError( pimErrorMsg )
         return False
      else:
         return True
   try:
      if isMulticastRoutingEnabled( vrfName, mfibVrfConfig ):
         return True
      else:
         mode.addError( vrfErrorMsg )
         return False

   except IndexError:
      return False

def makeVrfNoneVrfDefault ( func ):
   ''' Decorator for show commands. If vrfName argument is None replace
       it with DEFAULT_VRF. '''
   def newFunc ( mode, *args, **kwargs ):
      if 'vrfName' in kwargs and kwargs[ 'vrfName' ] is None:
         kwargs[ 'vrfName' ] = VrfCli.vrfMap.getCliSessVrf( mode.session )
      return func( mode, *args, **kwargs )
   return newFunc

def isRoutingEnabled( vrfName, routingVrfInfoDir ):
   routingInfo = False
   if vrfName in routingVrfInfoDir.entityPtr:
      if routingVrfInfoDir.entityPtr[ vrfName ]:
         routingInfo = routingVrfInfoDir[ vrfName ].routing
   return routingInfo

def isMulticastRoutingEnabled( vrfName, mfibVrfConfig ):
   if vrfName in mfibVrfConfig.config:
      return mfibVrfConfig.config[ vrfName ].routing
   return False

def allowed( vrfName, routingVrfInfoDir, mfibVrfConfig ):
   return isRoutingEnabled( vrfName, routingVrfInfoDir ) and \
       isMulticastRoutingEnabled( vrfName, mfibVrfConfig )

def registerShowPimConfigSanityCmdCallback( cmdCallback ):
   """
   API is used by other Pim agents CliPlugins to register commands
   to be called when "show ip pim config-sanity" is called
   """
   configSanityCallbacks.extend( cmdCallback )

# Add a function that will return Routing::Pim::IO::Smash::MessageCounters object.
pimMessageCountersHook = CliExtensions.CliHook()

# Add a function that will increment the clear count in the config, which a
# state machine will react to and clear the
# Routing::Pim::IO::Smash::PimMessageCounters object.
pimClearMessageCountersHook = CliExtensions.CliHook()

# Add a hook to call clear ip mroute for sparsemode and bidirmode
clearIpMrouteHook = CliExtensions.CliHook()

# Add a hook to notify all interested agents when "vrf <vrfName>" is configured
# under router pim submode
pimSparseModeVrfConfiguredHook = CliExtensions.CliHook()
canDeletePimSparseModeVrfHook = CliExtensions.CliHook()
pimSparseModeVrfDeletedHook = CliExtensions.CliHook()

pimBidirVrfConfiguredHook = CliExtensions.CliHook()
canDeletePimBidirModeVrfHook = CliExtensions.CliHook()
pimBidirVrfDeletedHook = CliExtensions.CliHook()
pimBidirVrfConfigCleanupHook = CliExtensions.CliHook()

pimUpstreamJoinsHook = CliExtensions.CliHook()

pimShowIpPimRpHook = CliExtensions.CliHook()
pimShowIpPimRpHashHook = CliExtensions.CliHook()

pimRpCandidateCleanupHook = CliExtensions.CliHook()
pimSparseModeCleanupHook = CliExtensions.CliHook()

# register new hook
pimShowIpMrouteHook = CliExtensions.CliHook()
pimShowIpMrouteIntfHook = CliExtensions.CliHook()
pimShowIpMrouteCountHook = CliExtensions.CliHook()

class RouterPimMode( RoutingPimMode, BasicCli.ConfigModeBase ):
   name = 'PIM configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      RoutingPimMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimVrfMode( RoutingPimVrfMode, BasicCli.ConfigModeBase ):
   name = 'PIM VRF configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      RoutingPimVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimIpv4Mode( RoutingPimAfMode, BasicCli.ConfigModeBase ):
   name = 'IPv4 Pim Mode configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      RoutingPimAfMode.__init__( self, vrfName, AddressFamily.ipv4 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimSharedModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName

RouterPimMode.addModelet( RouterPimSharedModelet )
RouterPimVrfMode.addModelet( RouterPimSharedModelet )

class RouterPimSparseMode( RoutingPimSparseMode, BasicCli.ConfigModeBase ):
   name = 'PIM Sparse Mode configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session ):
      self.vrfName = DEFAULT_VRF
      self.af = AddressFamily.ipunknown
      self.protocol = 'sparse-mode'
      RoutingPimSparseMode.__init__( self, None )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimSparseVrfMode( RoutingPimSparseVrfMode, BasicCli.ConfigModeBase ):
   name = 'PIM Sparse Mode VRF configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName ):
      assert vrfName not in VRFNAMES_RESERVED
      self.vrfName = vrfName
      self.af = AddressFamily.ipunknown
      self.protocol = 'sparse-mode'
      RoutingPimSparseVrfMode.__init__( self, vrfName )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimSparseIpv4Mode( RoutingPimSparseAfMode, BasicCli.ConfigModeBase ):
   name = 'IPv4 Pim Sparse Mode configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      self.protocol = 'sparse-mode'
      RoutingPimSparseAfMode.__init__( self, vrfName, AddressFamily.ipv4 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimSparseIpv6Mode( RoutingPimSparseAfMode, BasicCli.ConfigModeBase ):
   name = 'IPv6 Pim Sparse Mode configuration'
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, vrfName=DEFAULT_VRF ):
      self.protocol = 'sparse-mode'
      RoutingPimSparseAfMode.__init__( self, vrfName, AddressFamily.ipv6 )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

class RouterPimSparseSharedModelet( CliParser.Modelet ):
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName

class RouterPimSparseAfSharedModelet( CliParser.Modelet ):
   '''Modelet that has commands available only under ipv* submodes
      Add commands that have ipAddresses or ACLs as arguments here'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

class RouterPimSparseAfCommonSharedModelet( CliParser.Modelet ):
   '''Modelet that has commands that is common for ipv* sub modes and
   router pim sparse submode
   Add commands that dont have arguments that are ipAddresses or ACL'''
   modeletParseTree = CliParser.ModeletParseTree()

   def __init__( self, mode ):
      CliParser.Modelet.__init__( self )
      self.mode_ = mode
      self.vrfName = self.mode_.vrfName
      self.af = self.mode_.af

#TODO this
RouterPimSparseMode.addModelet( RouterPimSparseSharedModelet )
RouterPimSparseVrfMode.addModelet( RouterPimSparseSharedModelet )
RouterPimSparseIpv4Mode.addModelet( RouterPimSparseAfSharedModelet )
RouterPimSparseIpv6Mode.addModelet( RouterPimSparseAfSharedModelet )

#RouterPimSparseMode.addModelet( RouterPimSparseAfCommonSharedModelet )
#RouterPimSparseVrfMode.addModelet( RouterPimSparseAfCommonSharedModelet )
RouterPimSparseIpv4Mode.addModelet( RouterPimSparseAfCommonSharedModelet )
RouterPimSparseIpv6Mode.addModelet( RouterPimSparseAfCommonSharedModelet )

def configureIpPimHashAlgorithm( mode, rpConfigColl ):
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   if vrfName not in rpConfigColl.vrfConfig:
      return
   rpConfig = rpConfigColl.vrfConfig[ vrfName ]
   rpConfig.rpHashAlgorithm = Tac.Type(
      "Routing::Pim::RpHashAlgorithm" ).rpHashAlgorithmModulo

def resetIpPimHashAlgorithm( mode, rpConfigColl ):
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   if vrfName not in rpConfigColl.vrfConfig:
      return
   rpConfig = rpConfigColl.vrfConfig[ vrfName ]
   rpConfig.rpHashAlgorithm = Tac.Type(
      "Routing::Pim::RpHashAlgorithm" ).rpHashAlgorithmDefault


def configureIpPimRpAddress( mode, pimConfigColl, rp, groupRangeOrAcl='224.0.0.0/4',
                             priority=0, hashmask=30, override=False ):
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   pimConfig = pimConfigColl.vrfConfig[ vrfName ]

   rpAddr = IpGenAddr( rp )
   if not pimConfig.rpTable.get( rpAddr ):
      rpc = pimConfig.rpTable.newMember( rpAddr )
      if isBidirMode( mode ):
         rpc.mode = 'modePimBidir'
   else:
      rpc = pimConfig.rpTable.get( rpAddr )
   rpPriority = Tac.Value( "Routing::Pim::RpPriority", priority )

   if '.' in groupRangeOrAcl or ':' in groupRangeOrAcl: #a prefix
      if override:
         rpc.overrideDynGrp[ IpGenPrefix( groupRangeOrAcl ) ] = True
      else:
         rpc.overrideDynGrp[ IpGenPrefix( groupRangeOrAcl ) ] = False
      prefix = IpGenPrefix( groupRangeOrAcl )
      rpc.group[ prefix ] = True
      rpc.priorityGrp[prefix] = rpPriority
      rpc.hashMaskGrp[ prefix ] = Tac.Value( "Routing::Pim::RpHashMaskLen",
                                              hashmask )
   else:
      if override:
         rpc.overrideDynAcl[ groupRangeOrAcl ] = True
      else:
         rpc.overrideDynAcl[ groupRangeOrAcl ] = False
      rpc.acl[ groupRangeOrAcl ] = True
      rpc.priorityAcl[ groupRangeOrAcl ] = rpPriority
      rpc.hashMaskAcl[ groupRangeOrAcl ] = Tac.Value( "Routing::Pim::RpHashMaskLen",
                                                      hashmask )

def resetIpPimRpAddress( mode, pimConfigColl, rp, groupRangeOrAcl='224.0.0.0/4' ):
   vrfName = McastCommonCliLib.vrfFromMode( mode )
   if vrfName not in pimConfigColl.vrfConfig:
      return
   pimConfig = pimConfigColl.vrfConfig[ vrfName ]

   rpAddr = rp if isinstance( rp, str ) else rp.stringValue
   rpc = pimConfig.rpTable.get( IpGenAddr( rpAddr ) )
   if rpc:
      if '.' in groupRangeOrAcl or ':' in groupRangeOrAcl: #a prefix
         prefix = IpGenPrefix( groupRangeOrAcl )
         del rpc.group[ prefix ]
         del rpc.priorityGrp[ prefix ]
         del rpc.hashMaskGrp[ prefix ]
         del rpc.overrideDynGrp[ prefix ]
      else:
         del rpc.acl[ groupRangeOrAcl ]
         del rpc.priorityAcl[ groupRangeOrAcl ]
         del rpc.hashMaskAcl[ groupRangeOrAcl ]
         del rpc.overrideDynAcl[ groupRangeOrAcl ]

      if not rpc.group and not rpc.acl:
         del pimConfig.rpTable[ IpGenAddr( rpAddr ) ]

def subnet( prefix ):
   return Arnet.Subnet( IpUtils.Prefix( prefix ) )

def containsSubnetUsingPrefix( outer, inner ):
   return outer.contains( inner.address ) and \
         outer.len <= inner.len

def containsSubnet( outer, inner ):
   return outer.containsAddr( inner.toNum() ) and \
          outer.mask_.masklen_ <= inner.mask_.masklen_

class RouteTrie:
   def __init__( self, routingStatus, forwardingStatus ):
      if routingStatus.tacType == Tac.Type( "Smash::Fib::RouteStatus" ).tacType:
         self.af = AddressFamily.ipv4
      else:
         self.af = AddressFamily.ipv6
      self.routingStatus = routingStatus
      self.forwardingStatus = forwardingStatus
      # pylint: disable-msg=W0212
      if type( routingStatus ) == LazyMount._Proxy:
         # force a mount, Tac.newInstance does not like unmounted proxies
         LazyMount.force( routingStatus )
      self.trie = Tac.newInstance( "Routing::TrieGen", "trie", self.af )
      routing4Status = routingStatus if self.af == AddressFamily.ipv4 else None
      routing6Status = routingStatus if self.af == AddressFamily.ipv6 else None
      self.trieBuilder = Tac.newInstance(
            "Routing::TrieGenBuilder", routing4Status, routing6Status,
            None, None, None, None, self.trie )

   def getRoute( self, prefix ):
      ''' Find longest prefix match for the route. '''
      while True:
         genPrefixMatch = self.trie.longestMatch( prefix )
         if self.af == AddressFamily.ipv4:
            prefixMatch = genPrefixMatch.v4Prefix
            maxLen = 32
         elif self.af == AddressFamily.ipv6:
            prefixMatch = genPrefixMatch.v6Prefix
            maxLen = 128

         if prefixMatch.isNullPrefix:
            break

         if self.routingStatus.route.has_key( prefixMatch ):
            route = self.routingStatus.route[ prefixMatch ]
            fec = self.forwardingStatus.fec.get( route.fecId )
            if not fec:
               return None
            else:
               for i in range( len( fec.via ) ):
                  if fec.via[ i ].intfId:
                     return route

         if prefixMatch.len == 0:
            break

         # Look for the next best prefix
         ip = IpUtils.IpAddress( prefixMatch.address )
         ip = IpUtils.IpAddress(
               ip.toNum() & ( 0xFFFFFFFF << ( maxLen - prefixMatch.len + 1 ) ) )
         prefix = Arnet.IpGenPrefix( str( ip ) + "/" + str( prefixMatch.len - 1 ) )

      return None

def findRps( g, groupRanges ):
   # first, try the fast way
   if g in groupRanges:
      return groupRanges[ g ]
   # now try the slow one
   for gr in groupRanges:
      if gr.contains( g ):
         return groupRanges[ gr ]
   return [ ]

def checkForRoute( addr, nodeType, routeTrie, allPimIntfs, allIpAddrs,
                   neighbors, pimRoute=False, legacy=False ):
   assert routeTrie
   if legacy:
      rcModel = PimConfigCheckModel.RouteCheck()
      rcModel.prefix = addr.v4Addr
   else:
      rcModel = PimConfigCheckModel.GenRouteCheck()
      rcModel.prefix = addr
   rcModel.hintsFound = 0
   rcModel.nodeType = nodeType
   rcModel.noRoute = False
   # is this addrress local?
   if addr in allIpAddrs:
      return rcModel

   # Look for the best match with a valid via interface
   route = routeTrie.getRoute( Arnet.IpGenPrefix( str( addr ) ) )
   if not route:
      rcModel.hintsFound += 1
      rcModel.noRoute = True
      return rcModel
   if not pimRoute:
      return rcModel
   # now check if the route goes via pim-enabled interface
   nonPimIntfs = [ ]
   nonPimHops = [ ]
   fec = routeTrie.forwardingStatus.fec.get( route.fecId )
   if not fec:
      return None
   else:
      for i in range( len( fec.via ) ):
         via = fec.via[ i ]

         hop = Arnet.IpGenAddr( str( via.hop ) )
         if not via.intfId in allPimIntfs:
            nonPimIntfs.append( via.intfId )
            rcModel.hintsFound += 1
         elif not hop.isAddrZero and hop != addr and hop not in neighbors:
            if legacy:
               nonPimHops.append( hop.v4Addr )
            else:
               nonPimHops.append( hop )
            rcModel.hintsFound += 1

   if nonPimIntfs:
      rcModel.nonPimIntf = nonPimIntfs
   if nonPimHops:
      rcModel.nonPimHop = nonPimHops
   return rcModel

def checkForPimRoute( prefix, nodeType, routeTrie, allPimIntfs, allIpAddrs,
                      neighbors, legacy=False ):
   return checkForRoute( prefix, nodeType, routeTrie, allPimIntfs, allIpAddrs,
                         neighbors, pimRoute=True, legacy=legacy )

def _pfxFromRule( rule ):
   mask = bin( rule.filter.source.mask )
   firstZeroPos = mask.find( '0', 1 )
   if firstZeroPos == -1:
      pfxLen = 32
   else:
      pfxLen = firstZeroPos - 2
   prefixAddr = rule.filter.source.allZerosAddr
   return Arnet.Prefix( prefixAddr + '/' + str( pfxLen ) )

def _appendAclGroups( rp, aclName, groupRanges ):
   acl = getAclConfig( 'ip' ).get( aclName )
   if acl:
      for seqNum1, ruleId1 in acl.currCfg.ruleBySequence.iteritems( ):
         ipRule1 = acl.currCfg.ipRuleById[ ruleId1 ]
         ruleGrp1 = _pfxFromRule( ipRule1 )
         if ipRule1.action == 'permit':
            covered = False
            for seqNum2, ruleId2 in acl.currCfg.ruleBySequence.iteritems( ):
               if seqNum2 < seqNum1:
                  # don't append covered groups
                  ipRule2 = acl.currCfg.ipRuleById[ ruleId2 ]
                  ruleGrp2 = _pfxFromRule( ipRule2 )
                  if containsSubnetUsingPrefix( ruleGrp2, ruleGrp1 ):
                     covered = True
                     break
            if not covered:
               if ruleGrp1 not in groupRanges:
                  groupRanges[ ruleGrp1 ] = [ ]
               groupRanges[ ruleGrp1 ].append( ( rp, aclName ) )
   return groupRanges

def getAllPhysIntfs( mode ):
   allPhysIntfs = { }
   intfs = IntfCli.Intf.getAll( mode, None )
   if intfs is not None:
      for i in intfs:
         allPhysIntfs[ i.name ] = i
   return allPhysIntfs

def getAllPimIntfs( pimConfigRoot ):
   # get all PIM-enabled intterfaces
   if not pimConfigRoot:
      return
   allPimIntfs =  { }
   for k, pic in pimConfigRoot.intfConfig.iteritems( ):
      if pic.mode != 'modePimNone':
         allPimIntfs[ k ] = pic
   return allPimIntfs

def getPimVrfIntfs( pimStatus ):
   # get all PIM-enabled interface with a specific vrf
   allPimVrfIntfs = {}
   if pimStatus:
      for k, pic in pimStatus.pimIntf.iteritems():
         if pic.mode != 'modePimNone':
            allPimVrfIntfs[ k ] = pic
   return allPimVrfIntfs

def getAllIpAddrs( ipConfig ):
   # get all local IP addresses
   if not ipConfig:
      return
   allIpAddrs = { }
   if ipConfig.tacType == Tac.Type( "Ip::Config" ).tacType:
      for ipc in ipConfig.ipIntfConfig.values():
         allIpAddrs[ Arnet.IpGenAddr( ipc.addrWithMask.address ) ] = ipc
         for addr in ipc.secondaryWithMask:
            allIpAddrs[ Arnet.IpGenAddr( addr.address ) ] = ipc
   else:
      for ipc in ipConfig.intf.values():
         for addr in ipc.addr:
            allIpAddrs[ Arnet.IpGenAddr( str( addr.address ) ) ] = ipc
   return allIpAddrs

def getAllPimNeighborsFromSysdb( pimStatus ):
   if not pimStatus:
      return
   neighbors = [ ]
   if pimStatus:
      for intf in pimStatus.pimIntf:
         for addr in pimStatus.pimIntf[intf].neighbor:
            neighbors.append( addr )
   return neighbors

def checkMroute( rp, nodeType ):
   # check static mroute -- TODO
   return

def getDrIntfs( pimStatus ):
   # get all interfaces where we are DR
   drIntfs = [ ]
   for intf, pintf in pimStatus.pimIntf.iteritems( ):
      if pintf.iAmDr:
         drIntfs.append( intf )
   if pimStatus is None:
      return drIntfs
   for intf, pintf in pimStatus.pimIntf.iteritems( ):
      if pintf.iAmDr:
         drIntfs.append( intf )
   return drIntfs

def getStaticGroups( vrfName ):
   staticGroups = set( )
   staticGroupSources = set( )
   for hook in McastCommonCliLib.mcastIfCollHook.extensions():
      ifConfigColl = hook( vrfName, AddressFamily.ipv4 )
      for ic in ifConfigColl.values():
         if hasattr( ic, 'staticJoinSourceGroup' ):
            for sg in ic.staticJoinSourceGroup:
               staticGroups.add( sg.groupAddr )
               staticGroupSources.add( sg.sourceAddr )
   return ( staticGroups, staticGroupSources )

def getConnectedGroups( ):
   connectedGroups = set( )
   for hook in GmpCli.igmpStatusCollHook.extensions():
      for intfId, intfStatus in hook().iteritems():
         if intfStatus.gmpQuerierStatus:
            for grpAddr, qst in intfStatus.gmpQuerierStatus.group.iteritems():
               connectedGroups.add( grpAddr )
   return connectedGroups

def pimsmConfigCheck( mode, vrfName, ipConfig, pimConfigRoot, aclConfig,
                     rpConfig, pimStatus, pimsmStatus,
                     routingStatus, forwardingStatus ):

   pimsmConfigCheckModel = PimConfigCheckModel.PimConfigCheck()

   pimsmConfigCheckModel.staticRpSet = staticRpConfigCheck( mode, ipConfig,
                                       pimConfigRoot, rpConfig, pimStatus,
                                       routingStatus, forwardingStatus, aclConfig )
   pimsmConfigCheckModel.mroute = mrtConfigCheck( mode, vrfName, pimsmStatus,
                                 pimConfigRoot, pimStatus, rpConfig,
                                 ipConfig, routingStatus, forwardingStatus,
                                 aclConfig )

   return pimsmConfigCheckModel

def pimBidirConfigCheck( mode, vrfName, ipConfig, pimConfigRoot, aclConfig,
                        pimBidirStaticRpConfig, pimBidirStatus,
                        pimStatus, routingStatus, routingHwStatus,
                        forwardingStatus, pimGlobalStatus ):
   pimBidirConfigCheckModel = PimConfigCheckModel.PimBidirConfigCheck()

   pimBidirConfigCheckModel.pimBidirSupported = \
         routingHwStatus.pimBidirectionalSupported
   pimBidirConfigCheckModel.pimBidirEnabled = \
         vrfName in pimGlobalStatus.pimEnabledBidirVrf

   pimBidirConfigCheckModel.staticRpSet = staticRpConfigCheck( mode, ipConfig,
                                          pimConfigRoot, pimBidirStaticRpConfig,
                                          pimStatus, routingStatus, forwardingStatus,
                                          aclConfig )
   pimBidirConfigCheckModel.mroute = mrtBidirConfigCheck( mode, vrfName,
                                          pimBidirStatus, pimConfigRoot, pimStatus,
                                          pimBidirStaticRpConfig,
                                          ipConfig, routingStatus, forwardingStatus,
                                          aclConfig )
   return pimBidirConfigCheckModel



# Check basic RP configuration
def staticRpConfigCheck( mode, ipConfig, pimConfigRoot, rpConfig, pimStatus,
                         routingStatus, fwdStatus, aclConfig ):
   allIpAddrs = getAllIpAddrs( ipConfig )
   allPimIntfs = getAllPimIntfs( pimConfigRoot )
   neighbors = getAllPimNeighborsFromSysdb( pimStatus )
   routeTrie = RouteTrie( routingStatus, fwdStatus )
   rpSetModel = PimConfigCheckModel.StaticRpSet()
   rpSetModel.hintsFound = 0
   rpSetModel.noRp = False
   rpSetModel.noDefaultRp = False
   rpSetModel.rcModel = []
   # is any RP configured?

   if not rpConfig or not rpConfig.rpTable:
      rpSetModel.hintsFound += 1
      rpSetModel.noRp = True
      return rpSetModel

   for rpt, rpc in rpConfig.rpTable.iteritems( ):
      # is there a pim-enabled route to RP?
      rcModel = checkForPimRoute( rpt, "RP", routeTrie, allPimIntfs,
                                  allIpAddrs, neighbors, legacy=True )
      rpSetModel.rcModel.append( rcModel )
      rpSetModel.hintsFound += rcModel.hintsFound

   rpConfigTrieSm = Tac.newInstance( "Routing::Pim::Static::RpConfigTrieSm",
                                     rpConfig, aclConfig.config[ 'ip' ] )
   trieEntry = rpConfigTrieSm.trieEntry
   rpConfigCheckSm = Tac.newInstance( "Routing::Pim::Static::RpConfigCheckSm",
                                      rpConfigTrieSm )
   dupRp = rpConfigCheckSm.dupRp
   overlapGroup = rpConfigCheckSm.overlapGroup

   # is default RP configured?
   if Arnet.IpGenPrefix( '224.0.0.0/4' ) not in trieEntry:
      rpSetModel.hintsFound += 1
      rpSetModel.noDefaultRp = True

   # are there redundant RPs?
   rpSetModel.rpSet = []

   for k, dr in dupRp.iteritems():
      rpModel = PimConfigCheckModel.StaticRp()
      rpModel.mGroup = k.stringValue
      for rp in dr.rp.keys():
         rpModel.mRpList.append( rp.stringValue )
      rpSetModel.hintsFound += 1
      # BUG25925
      rpSetModel.rpSet.append( rpModel )

   # are there overlapping group ranges with distinct RPs?
   rpSetModel.overlapSet = []
   for k, ent in overlapGroup.iteritems():
      overlap = PimConfigCheckModel.OverlapGroup()
      overlap.group = k
      rpSetModel.hintsFound += 1
      for group in ent.overlap.keys():
         overlap.overlaps.append( group )
      rpSetModel.overlapSet.append( overlap )

   return rpSetModel

# Check interfaces
# are there any pim-enabled interfaces?
# are all pim-enabled interfaces configured with ip address?
# are there pim interfaces with secondary addresses?
# are there pim enabled intefaces down?
def interfaceConfigCheck( mode, vrfName, ipConfig, pimConfigColl ):
   allPhysIntfs = getAllPhysIntfs( mode )
   pimIntfsNoIp = [ ]
   pimIntfsDown = [ ]
   pimIntfsNotRouted = [ ]
   pimIntfsSparseModeBorderRouter = [ ]

   intfModel = PimConfigCheckModel.IntfConfigCheck()
   intfModel.hintsFound = 0
   intfModel.disabled = False

   pimConfiguredIntfInVrf = 0
   for i, pic in pimConfigColl.intfConfig.iteritems():
      ic = ipConfig.ipIntfConfig.get( i )
      if ic and ic.vrf != vrfName:
         continue
      elif not ic and vrfName != DEFAULT_VRF:
         continue

      pimConfiguredIntfInVrf += 1
      if not ic or ic.addrWithMask.stringValue == '0.0.0.0/0':
         intfModel.hintsFound += 1
         pimIntfsNoIp.append( i )
      intf = allPhysIntfs.get( i )
      if intf:
         if not ( intf.config( ).enabled and intf.config( ).adminEnabled
                  and intf.status( ).operStatus == 'intfOperUp' ):
            intfModel.hintsFound += 1
            pimIntfsDown.append( i )
         elif not ( intf.routingSupported() and intf.routingCurrentlySupported() ):
            intfModel.hintsFound += 1
            pimIntfsNotRouted.append( i )
      sparseModeEnabled = pic.mode == "modePimSm" or pic.mode == "modePimSmAndBidir"
      if sparseModeEnabled and pic.borderRouter:
         intfModel.hintsFound += 1
         pimIntfsSparseModeBorderRouter.append( i )

   if pimConfiguredIntfInVrf == 0:
      intfModel.hintsFound += 1
      intfModel.disabled = True
      return intfModel

   intfModel.intfsNoIp = pimIntfsNoIp
   intfModel.intfsDown =  pimIntfsDown
   intfModel.intfsNotRouted = pimIntfsNotRouted
   intfModel.intfsSparseModeBorderRouter = pimIntfsSparseModeBorderRouter
   return intfModel

# Check static groups and boundaries (set in igmp config)
# check if there is valid iif for each (s,g) or (*.g) entry
def igmpConfigCheck( mode, vrfName, ipConfig, aclConfig, mcastBoundaryConfig,
                     pimConfigRoot, pimStatus, rpConfig, routingStatus,
                     fwdStatus ):
   allIpAddrs = getAllIpAddrs( ipConfig )
   allPimIntfs = getAllPimIntfs( pimConfigRoot )
   neighbors = getAllPimNeighborsFromSysdb( pimStatus )
   routeTrie = RouteTrie( routingStatus, fwdStatus )
   drIntfs = getDrIntfs( pimStatus )
   pimIntfsDown = []
   igmpModel = PimConfigCheckModel.IgmpConfigCheck()
   igmpModel.hintsFound = 0
   igmpModel.rcModel = []
   igmpModel.disabledIntfs = []
   igmpModel.nonDrIntfs = []
   igmpModel.boundaryIntfs = []
   igmpModel.notDefinedBAcl = []
   igmpModel.notStandardBAcl = []
   igmpModel.noRpGroups = []
   igmpModel.noRpGroupc = []
   igmpModel.invalidBoundaryGroups = []

   disabledSgIntfs = set( )
   enabledSgIntfs = set( )
   boundaryIntfs = set()

   rpTrie = Tac.newInstance( "Routing::Pim::Static::RpConfigTrieSm",
                             rpConfig, aclConfig.config[ 'ip' ] )
   ( staticGroups, staticGroupSources ) = getStaticGroups( vrfName )

   for hook in McastCommonCliLib.mcastIfCollHook.extensions():
      ifConfigColl = hook( vrfName, AddressFamily.ipv4 )
      for ic in ifConfigColl.values():
         if hasattr( ic, 'staticJoinSourceGroup' ):
            if ic.intfId in allPimIntfs:
               enabledSgIntfs.add( \
                  ( ic.intfId, len( ic.staticJoinSourceGroup.members() ) ) )
            else:
               igmpModel.hintsFound += 1
               disabledSgIntfs.add( ic.intfId )

   boundaryConfigVrf = {}
   for intfId, intfConfig in mcastBoundaryConfig.intfConfig.iteritems():
      ic = ipConfig.ipIntfConfig.get( intfId )
      if ic and ic.vrf == vrfName:
         boundaryConfigVrf[ intfId ] = intfConfig

   for intfId, intfConfig in sorted( boundaryConfigVrf.items() ):
      if intfConfig.boundary.keys():
         igmpModel.hintsFound += 1
         boundaryIntfs.add( intfId )
      if intfConfig.boundaryAcl.acl != '':
         igmpModel.hintsFound += 1
         boundaryIntfs.add( intfId )

   if rpTrie.trie.routes() > 0:
      for g in staticGroups:
         if rpTrie.trie.longestMatch( IpGenPrefix( g ) ) == zeroAddr:
            igmpModel.hintsFound += 1
            igmpModel.noRpGroups.append( IpGenAddr( g ) )

   for s in staticGroupSources:
      if s == '0.0.0.0':
         continue
      rcModel = checkForPimRoute( Arnet.IpGenAddr( s ), "static group source",
                                 routeTrie,
                                 allPimIntfs, allIpAddrs, neighbors, legacy=True )
      igmpModel.rcModel.append( rcModel )
      igmpModel.hintsFound += rcModel.hintsFound

   # are there static groups configured on pim-disabled interfaces?
   if disabledSgIntfs:
      igmpModel.disabledIntfs = list( disabledSgIntfs )

   # are we DR for all static groups?
   nonDrSgIntfs = [ ]
   for ( i, num ) in enabledSgIntfs:
      if i not in drIntfs and i not in pimIntfsDown and num:
         igmpModel.hintsFound += 1
         nonDrSgIntfs.append( i )
   igmpModel.nonDrSgIntfs = nonDrSgIntfs

   # are there multicast boundaries?
   igmpModel.boundaryIntfs = list ( boundaryIntfs )

   # check if multicast boundary Acl rules are valid
   for intfId, intfConfig in sorted( boundaryConfigVrf.items() ):
      if intfConfig.boundaryAcl.acl != '':
         aclName = intfConfig.boundaryAcl.acl
         acl = getAclConfig( 'ip' ).get( aclName )
         if not acl or not acl.currCfg:
            igmpModel.hintsFound += 1
            aclModel = PimConfigCheckModel.NotDefinedBAcl(
               acl = aclName, intf = intfId )
            igmpModel.notDefinedBAcl.append( aclModel )
         elif not acl.standard:
            igmpModel.hintsFound += 1
            aclModel = PimConfigCheckModel.NotStandardBAcl(
               acl = aclName, intf = intfId )
            igmpModel.notStandardBAcl.append( aclModel )
         else:
            for _, uid in acl.currCfg.ruleBySequence.iteritems():
               # the following func is deleted, add new func once available
               # maybeRet = IgmpCli.checkBoundaryAclRule( acl.ipRuleById[ uid ] )
               # return value is tuple. If first element is False, it indicates
               # an error
               rule = acl.currCfg.ipRuleById[ uid]
               ( valid, reason ) = McastBoundaryCli.checkBoundaryAclRule( rule )
               if not valid :
                  igmpModel.hintsFound += 1
                  g = rule.filter.source
                  grp = PimConfigCheckModel.InvalidBoundaryGroup()
                  grp.group = ArnetModel.IpAddrAndMask( ip = g.address,
                        mask = g.mask )
                  grp.reason = reason
                  igmpModel.invalidBoundaryGroups.append( grp )

   # Check connnected groups
   connectedGroups = set( )
   connectedGroupIntfs = set( )
   for hook in GmpCli.igmpStatusCollHook.extensions():
      for intfId, intfStatus in hook().iteritems():
         ic = ipConfig.ipIntfConfig.get( intfId )
         if ic and ic.vrf == vrfName:
            if intfStatus.gmpQuerierStatus:
               for grpAddr, qst in intfStatus.gmpQuerierStatus.group.iteritems():
                  connectedGroups.add( grpAddr )
                  connectedGroupIntfs.add( intfId )

   # are there RPs for all connected groups?
   # (if there are no RPs configured at all, don't bother)
   if rpTrie.trie.routes() > 0:
      for g in connectedGroups:
         if rpTrie.trie.longestMatch( IpGenPrefix( g ) ) == zeroAddr:
            igmpModel.hintsFound += 1
            igmpModel.noRpGroupc.append( IpGenAddr ( g ) )

   # are we DR for all connected groups?
   nonDrCgIntfs = [ ]
   for i in connectedGroupIntfs:
      if i not in drIntfs and i not in pimIntfsDown:
         igmpModel.hintsFound += 1
         nonDrCgIntfs.append( i )
   igmpModel.nonDrIntfs = nonDrCgIntfs
   return igmpModel

# Check neighbors, any invalid hellos?
def pimNeighborConfigCheck(
      mode, ipConfig, pimConfigRoot, pimVrfStatus, pimCounters ):
   pimCountersWrapper = PimCountersLib.PimCountersWrapper( pimCounters )
   allVrfPimIntfs = getPimVrfIntfs( pimVrfStatus )
   allPimIntfs = getAllPimIntfs( pimConfigRoot )

   pimIntfsSecIp = []
   for i in allPimIntfs:
      ic = ipConfig.ipIntfConfig.get( i )
      if ic and ic.secondaryWithMask:
         pimIntfsSecIp.append( i )

   nbrModel = PimConfigCheckModel.NbrConfigCheck()
   nbrModel.hintsFound = 0
   nbrModel.delta = 0

   # get invalid hello counter
   session = mode.session
   invalidHellos = session.sessionData( 'pimInvalidHellosCounter', defaultValue=0 )
   crntCounter = pimCountersWrapper.getCounter( 'rx', 0 ).invalid
   session.sessionDataIs( 'pimInvalidHellosCounter', crntCounter )
   delta = max( crntCounter - invalidHellos, 0 )
   if delta > 0:
      nbrModel.hintsFound += 1
      if pimIntfsSecIp:
         invalidModel = PimConfigCheckModel.InvalidHello(
            delta = delta, intfs = pimIntfsSecIp )
         nbrModel.invalidHello.append( invalidModel )
      nbrModel.delta = delta

   # are there neighbor filters?
   nfIntfs = [ ]
   for k in allVrfPimIntfs:
      if k in allPimIntfs:
         pic = allPimIntfs[k]
         if pic and pic.neighborFilter:
            nbrModel.hintsFound += 1
            nfIntfs.append( pic.intfId )
   nbrModel.nfIntf = nfIntfs
   return nbrModel

def mrtConfigCheck( mode, vrfName, pimsmStatus,
                    pimConfigRoot, pimStatus,
                    rpConfig, ipConfig, routingStatus,
                    fwdStatus, aclConfig ):
   if not rpConfig or not pimsmStatus or not pimStatus:
      return
   allPimIntfs = getAllPimIntfs( pimConfigRoot )
   allIpAddrs = getAllIpAddrs( ipConfig )
   neighbors = getAllPimNeighborsFromSysdb( pimStatus )
   rpTrie = Tac.newInstance( "Routing::Pim::Static::RpConfigTrieSm",
                             rpConfig, aclConfig.config[ 'ip' ] )
   routeTrie = RouteTrie( routingStatus, fwdStatus )

   # get groups
   groups = set()
   sources = set()
   uroutes = 0

   for k, route in pimsmStatus.route.iteritems( ):
      if k.s.v4Addr == '0.0.0.0':
         continue
      if k.g not in groups:
         groups.add( k.g.v4Addr )
      if k.s not in sources:
         sources.add( k.s.v4Addr )

   def _ntoh( s ):
      import struct, socket
      addr = int( s, 16 )
      return socket.inet_ntoa( struct.pack( "I", addr ) )
   lines = file( "/proc/net/ip_mr_cache" ).readlines()
   for line in lines[ 1: ]:
      vals = line.split()
      ( g, s, i ) = vals[ 0:3 ]
      groups.add( _ntoh( g ) )
      sources.add( _ntoh( s ) )
      if int( i ) == -1:
         uroutes += 1

   ( staticGroups, staticGroupSources ) = getStaticGroups( vrfName )
   groups -= staticGroups
   groups -= getConnectedGroups()
   sources -= staticGroupSources

   mrtModel = PimConfigCheckModel.MrtConfigCheck()
   mrtModel.hintsFound = 0
   mrtModel.rcModel = []
   # are there RPs for all remaining groups?
   if rpTrie.trie.routes() > 0:
      for g in groups:
         if rpTrie.trie.longestMatch( IpGenPrefix( g ) ) == zeroAddr:
            mrtModel.hintsFound += 1
            mrtModel.noRpGroups.append( IpGenAddr( g ) )

   # are routes to all remaining sources pim-enabled?
   for s in sources:
      if s == '0.0.0.0':
         continue
      rcModel = checkForPimRoute( Arnet.IpGenAddr( s ), "source", routeTrie,
                                  allPimIntfs, allIpAddrs, neighbors, legacy=True )
      if rcModel.hintsFound > 0:
         mrtModel.hintsFound += rcModel.hintsFound
         mrtModel.rcModel.append( rcModel )
   # any unsolved routes?
   mrtModel.uroutes = uroutes
   if uroutes > 0:
      mrtModel.hintsFound += 1
   return mrtModel


def mrtBidirConfigCheck( mode, vrfName, pimBidirStatus,
                         pimConfigRoot, pimStatus,
                         rpConfig, ipConfig, routingStatus,
                         fwdStatus, aclConfig ):
   if not rpConfig or not pimBidirStatus or not pimStatus:
      return

   allPimIntfs = getAllPimIntfs( pimConfigRoot )
   neighbors = getAllPimNeighborsFromSysdb( pimStatus )
   rpTrie = Tac.newInstance( "Routing::Pim::Static::RpConfigTrieSm",
                             rpConfig, aclConfig.config[ 'ip' ] )
   routeTrie = RouteTrie( routingStatus, fwdStatus )

   #get groups
   groups = set()
   uroutes = 0

   for g, group in pimBidirStatus.group.iteritems():
      if g not in groups:
         groups.add( g.v4Addr )

   ( staticGroups, staticGroupSources ) = getStaticGroups( vrfName )
   groups -= staticGroups
   groups -= getConnectedGroups()
   mrtModel = PimConfigCheckModel.MrtConfigCheck()
   mrtModel.hintsFound = 0
   mrtModel.rcModel = []

   if rpTrie.trie.routes() > 0:
      for g in groups:
         if rpTrie.trie.longestMatch( IpGenPrefix( g ) ) == zeroAddr:
            mrtModel.hintsFound += 1
            mrtModel.noRpGroups.append( IpGenAddr( g ) )

   mrtModel.uroutes = uroutes
   if uroutes > 0:
      mrtModel.hintsFound += 1
   return mrtModel

def defaultAddr( af ):
   assert af in [ "ipv4", "ipv6" ]
   if af == "ipv4":
      return Arnet.IpGenAddr( "0.0.0.0" )
   else:
      return Arnet.IpGenAddr( "::" )

def validateMulticastAddr( addr ):
   if addr.af == "ipv4":
      return Arnet.IpGenPrefix( "224.0.0.0/4" ).contains( addr )
   else:
      return Arnet.IpGenPrefix( "ff00::/8" ).contains( addr )

def ipPimParseSg ( groupOrSource1, groupOrSource2 ):
   """Given two IPv4/v6 addresses, where one is unicast and the other is multicast,
   returns an (S,G) pair. Where S is the unicast address and G is the multicast
   address"""

   if ( groupOrSource1 is None ) and ( groupOrSource2 is None ):
      return ( None, None )

   addrType1 = ""
   addrType2 = ""

   if groupOrSource1:
      groupOrSource1 = Arnet.IpGenAddr( groupOrSource1 )
      addrType1 = "M" if validateMulticastAddr( groupOrSource1 ) else "U"

   if groupOrSource2:
      groupOrSource2 = Arnet.IpGenAddr( groupOrSource2 )
      addrType2 = "M" if validateMulticastAddr( groupOrSource2 ) else "U"

   if addrType1 == addrType2:
      raise ValueError

   if addrType1 == "M":
      # The first arg is a multicast address
      return ( groupOrSource2, groupOrSource1 )
   else:
      # The second arg is a multicast address
      return ( groupOrSource1, groupOrSource2 )

def ipPimParseGroup( group ):
   """Given an IPv4/Ipv6 address, validates that the address is multicast"""

   if ( group is None ):
      return None

   group = Arnet.IpGenAddr( group )
   if not validateMulticastAddr( group ):
      raise ValueError

   return group

def getPath( typeName, *args ):
   return Tac.Type( typeName ).mountPath( *args )
