#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements PimBsr show commands
# show ip pim bsr [ groupPrefix ]
# show ip pim bsr rp [ groupPrefix ] [ detail ]
# show ip pim bsr rp-hash <group>
# show ip pim rp-candidate
# show ip pim bsr counters
# show ip pim bsr debug
#-------------------------------------------------------------------------------
'''Show commands supported for  Pim BSR '''

import LazyMount
import SmashLazyMount
import BasicCli, Tac
import CliToken.Clear, CliToken.Ip
import CliToken.Pim
from CliToken.Pim import pimNodeAfterShow, pimNodeAfterShowIp, \
                         rpMatcher, detailMatcher, rpHashMatcher, \
                         countersMatcher
from CliMatcher import KeywordMatcher
from IpGenAddrMatcher import IpGenPrefixMatcher, IpGenAddrMatcher
import PimCliLib
import PimBsrModel, PimModelLib
from IraIpRouteCliLib import isValidPrefixWithError
from IraIp6RouteCliLib import isValidIpv6PrefixWithError
from IpLibConsts import DEFAULT_VRF as vrfDefault
import TechSupportCli
from CliPlugin.VrfCli import VrfExprFactory
import Tracing
from McastCommonCliLib import AddressFamily, ipv4NodeForShow, ipv6NodeForShow
from McastCommonCliLib import mcastRoutingSupported
from ShowCommand import ShowCliCommandClass

__defaultTraceHandle__ = Tracing.Handle( 'PimBsr::Cli' )
t8 = Tracing.trace8
# pylint: disable-msg=W0621

# Globals
pimCountersColl = {}
pimBsrStatusColl = None
pimBsr6StatusColl = None
pimGlobalStatus = None
pim6GlobalStatus = None
pimConfigColl = None
pimGlobalStatusReaderSm = None
pimBsrRpCandidateStatusColl = None
pimBsr6RpCandidateStatusColl = None
pimRpConfigColl = None
pim6RpConfigColl = None
pimBsrRpHashColl = {}
pimBsr6RpHashColl = {}
bsrRpComputeObj = None
bsr6RpComputeObj = None
bsrTrieBuilder = None
bsr6TrieBuilder = None
_mfibVrfConfig = None
_mfib6VrfConfig = None
_entityManager = None

# Add to message counters extension
def getPimMessageCounters( vrfName, af, **kwargs ):
   if vrfName in pimGlobalStatus.pimEnabledSparseModeVrf or \
         pimGlobalStatus.pimEnabledBidirVrf:
      path = PimCliLib.getPath( 'Routing::Pim::Smash::MessageCounters', 
                                af, vrfName, "bsr" )
      return SmashLazyMount.mount(
            _entityManager,
            path,
            'Routing::Pim::Smash::MessageCounters',
            SmashLazyMount.mountInfo( 'reader' ) )
   return None

def allowed( vrfName, af=AddressFamily.ipv4 ):
   status = pimGlobalStatus if af==AddressFamily.ipv4 else pim6GlobalStatus
   return vrfName in status.pimEnabledSparseModeVrf or \
         vrfName in status.pimEnabledBidirVrf

def validVrfName( mode, vrfName, af=AddressFamily.ipv4 ):
   vrfCfg = _mfibVrfConfig if af==AddressFamily.ipv4 else _mfib6VrfConfig
   if vrfName in vrfCfg.config or vrfName == vrfDefault:
      if not allowed( vrfName, af=af ):
         mode.addError( 
            "%s multicast-routing is not configured on VRF %s. PIM is not running." \
                  % ( af, vrfName ) )
         return False
      return True
   else:
      mode.addError( 
            "%s multicast-routing is not configured on VRF %s. PIM is not running." \
                  % ( af, vrfName ) )
      return False

vrfNameExpr = VrfExprFactory(
      helpdesc='VRF name' )

bsrMatcher = KeywordMatcher( 'bsr', helpdesc='PIM-BSR' )

ipGenPrefixMatcher = IpGenPrefixMatcher(
      'group address range with prefix', addrWithMask=False )

ipGenAddrMatcher = IpGenAddrMatcher( helpdesc="Multicast group Address" )

rpCandidateMatcher = KeywordMatcher( 'rp-candidate',
                                     helpdesc='Candidate RP information' )

debugMatcher = KeywordMatcher( 'debug', helpdesc='PIM BSR Debug Information' )

#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrfName> ] bsr [ zonePrefix ]
#------------------------------------------------------------------------------------

def cmdShowIpPimBsr( mode, args ):
   groupRange = args.get( 'PREFIX' )
   vrfName = args.get( 'VRF', vrfDefault )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   if not validVrfName( mode, vrfName, af=af ):
      return None

   if af == AddressFamily.ipv4:
      pimBsrStatus = pimBsrStatusColl.vrfStatus.get( vrfName )
      if groupRange and not isValidPrefixWithError( mode, str( groupRange ) ):
         return None
   else:
      pimBsrStatus = pimBsr6StatusColl.vrfStatus.get( vrfName )
      if groupRange and not isValidIpv6PrefixWithError(
            mode, groupRange.v6Prefix ):
         return None
   model = PimBsrModel.BootstrapRouter()
   if pimBsrStatus:
      model.generate( pimBsrStatus, zonePrefix=groupRange )
   return model


class ShowIpPimBsr( ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] bsr [ PREFIX ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'PREFIX': ipGenPrefixMatcher
   }
   handler = cmdShowIpPimBsr
   cliModel = PimBsrModel.BootstrapRouter

BasicCli.addShowCommandClass( ShowIpPimBsr )

class ShowPimAfBsr( ShowCliCommandClass ):
   syntax = 'show pim ( ipv4 | ipv6 ) [ VRF ] bsr [ PREFIX ]'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'PREFIX': ipGenPrefixMatcher
   }
   handler = cmdShowIpPimBsr
   cliModel = PimBsrModel.BootstrapRouter

BasicCli.addShowCommandClass( ShowPimAfBsr )

#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrfName> ] bsr rp [ groupPrefix ] [ detail ]
#------------------------------------------------------------------------------------
def doShowIpPimBsrRp( mode, args ):
   groupRange = args.get( 'PREFIX' )
   detail = 'detail' in args
   vrfName = args.get( 'VRF', vrfDefault )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   if not validVrfName( mode, vrfName, af=af ):
      return None
   if af == AddressFamily.ipv4:
      pimBsrRpCandidateStatus = pimBsrRpCandidateStatusColl.vrfStatus[ vrfName ]
      if groupRange and not isValidPrefixWithError( mode, str( groupRange ) ):
         return None
   else:
      pimBsrRpCandidateStatus = pimBsr6RpCandidateStatusColl.vrfStatus[ vrfName ]
      if groupRange and not isValidIpv6PrefixWithError(
            mode, groupRange.v6Prefix ):
         return None
   model = PimModelLib.PimRpCandidates()

   model.generate( None, pimBsrRpCandidateStatus,
                   groupPrefix=groupRange,
                   detail=detail )
   return model

class ShowIpPimBsrRp( ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] bsr rp [ PREFIX ] [ detail ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'rp': rpMatcher,
      'PREFIX': ipGenPrefixMatcher,
      'detail': detailMatcher
   }
   handler = doShowIpPimBsrRp
   cliModel = PimModelLib.PimRpCandidates

BasicCli.addShowCommandClass( ShowIpPimBsrRp )

class ShowPimAfBsrRp( ShowCliCommandClass ):
   syntax = 'show pim ( ipv4 | ipv6 ) [ VRF ] bsr rp [ PREFIX ] [ detail ]'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'rp': rpMatcher,
      'PREFIX': ipGenPrefixMatcher,
      'detail': detailMatcher
   }
   handler = doShowIpPimBsrRp
   cliModel = PimModelLib.PimRpCandidates

BasicCli.addShowCommandClass( ShowPimAfBsrRp )

#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrfName> ] bsr rp-hash <group>
#------------------------------------------------------------------------------------
def doShowIpPimBsrRpHash( mode, args ):
   global bsrRpComputeObj
   global bsr6RpComputeObj
   global bsrTrieBuilder
   global bsr6TrieBuilder

   group = str( args.get( 'GROUP' ) )
   vrfName = args.get( 'VRF', vrfDefault )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   model = PimModelLib.PimRpHash()

   if not validVrfName( mode, vrfName, af=af ):
      return None

   if af == AddressFamily.ipv4:
      pimBsrRpCandidateStatus = pimBsrRpCandidateStatusColl.vrfStatus[ vrfName ]
      rpConfig = pimRpConfigColl.vrfConfig[ vrfName ]
      rpHashColl = pimBsrRpHashColl
      afStr = ''
      if not bsrRpComputeObj:
         bsrRpComputeObj = Tac.newInstance( "Routing::Pim::RpCompute",
                                            pimBsrRpCandidateStatus,
                                            rpConfig  )
         bsrTrieBuilder = Tac.newInstance( "Routing::Pim::GrpPrefixTrieBuilderSm",
                                           pimBsrRpCandidateStatus,
                                           bsrRpComputeObj )
      computeObj = bsrRpComputeObj
   else:
      pimBsrRpCandidateStatus = pimBsr6RpCandidateStatusColl.vrfStatus[ vrfName ]
      rpConfig = pim6RpConfigColl.vrfConfig[ vrfName ]
      rpHashColl = pimBsr6RpHashColl
      afStr = '6'
      if not bsr6RpComputeObj:
         bsr6RpComputeObj = Tac.newInstance( "Routing::Pim::RpCompute",
                                             pimBsrRpCandidateStatus,
                                             rpConfig  )
         bsr6TrieBuilder = Tac.newInstance( "Routing::Pim::GrpPrefixTrieBuilderSm",
                                            pimBsrRpCandidateStatus,
                                            bsr6RpComputeObj )
      computeObj = bsr6RpComputeObj

   if vrfName not in rpHashColl:
      rpHashColl[ vrfName ] = SmashLazyMount.mount( _entityManager, 
            "routing%s/pim/bsr/rphash/%s" % ( afStr, vrfName ),
            "Routing::Pim::RpHashStatus",
            SmashLazyMount.mountInfo( 'reader' ) )
   pimBsrRpHash =  rpHashColl[ vrfName ]
   model.generate( pimBsrRpHash, None, computeObj, group, 
                   rpConfig.rpHashAlgorithm )
   return model

class ShowIpPimBsrRpHash( ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] bsr rp-hash GROUP [ detail ]'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'rp-hash': rpHashMatcher,
      'GROUP': ipGenAddrMatcher,
      'detail': detailMatcher
   }
   handler = doShowIpPimBsrRpHash
   cliModel = PimModelLib.PimRpHash

BasicCli.addShowCommandClass( ShowIpPimBsrRpHash )

class ShowPimAfBsrRpHash( ShowCliCommandClass ):
   syntax = 'show pim ( ipv4 | ipv6 ) [ VRF ] bsr rp-hash GROUP [ detail ]'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'rp-hash': rpHashMatcher,
      'GROUP': ipGenAddrMatcher,
      'detail': detailMatcher
   }
   handler = doShowIpPimBsrRpHash
   cliModel = PimModelLib.PimRpHash

BasicCli.addShowCommandClass( ShowPimAfBsrRpHash )

#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrfName> ] rp-candidate
#------------------------------------------------------------------------------------
def doShowIpPimRpCandidate( mode, args ):
   vrfName = args.get( 'VRF', vrfDefault )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   if not validVrfName( mode, vrfName, af=af ):
      return None

   if af == AddressFamily.ipv4:
      pimBsrStatus = pimBsrStatusColl.vrfStatus[ vrfName ]
   else:
      pimBsrStatus = pimBsr6StatusColl.vrfStatus[ vrfName ]

   model = PimBsrModel.PimRpCandidates()
   model.generate( pimBsrStatus.crp )
   return model

class ShowIpPimRpCandidate( ShowCliCommandClass ):
   syntax = 'show ip pim [ VRF ] rp-candidate'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfNameExpr,
      'rp-candidate': rpCandidateMatcher
   }
   handler = doShowIpPimRpCandidate
   cliModel = PimBsrModel.PimRpCandidates

BasicCli.addShowCommandClass( ShowIpPimRpCandidate )

class ShowPimAfRpCandidate( ShowCliCommandClass ):
   syntax = 'show pim ( ipv4 | ipv6 ) [ VRF ] rp-candidate'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfNameExpr,
      'rp-candidate': rpCandidateMatcher
   }
   handler = doShowIpPimRpCandidate
   cliModel = PimBsrModel.PimRpCandidates

BasicCli.addShowCommandClass( ShowPimAfRpCandidate )

#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrfName> bsr counters
#------------------------------------------------------------------------------------
def doShowIpPimBsrCounters( mode, args ):
   vrfName = args.get( 'VRF', vrfDefault )
   af = AddressFamily.ipv6 if 'ipv6' in args else AddressFamily.ipv4

   if not validVrfName( mode, vrfName, af=af ):
      return None

   if af == AddressFamily.ipv4:
      pimBsrStatus = pimBsrStatusColl.vrfStatus[ vrfName ]
   else:
      pimBsrStatus = pimBsr6StatusColl.vrfStatus[ vrfName ]

   model = PimBsrModel.PimBsrCounters()
   if pimBsrStatus.counters:
      model.generate( pimBsrStatus.counters )
   return model

class ShowIpPimBsrCounters( ShowCliCommandClass ):
   hidden = True
   syntax = 'show ip pim [ VRF ] bsr counters'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'counters': countersMatcher
   }
   handler = doShowIpPimBsrCounters
   cliModel = PimBsrModel.PimBsrCounters

BasicCli.addShowCommandClass( ShowIpPimBsrCounters )

class ShowPimAfBsrCounters( ShowCliCommandClass ):
   hidden = True
   syntax = 'show pim ( ipv4 | ipv6 ) [ VRF ] bsr counters'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'VRF': vrfNameExpr,
      'bsr': bsrMatcher,
      'counters': countersMatcher
   }
   handler = doShowIpPimBsrCounters
   cliModel = PimBsrModel.PimBsrCounters

BasicCli.addShowCommandClass( ShowPimAfBsrCounters )

#------------------------------------------------------------------------------------
# show ip pim bsr debug
#------------------------------------------------------------------------------------
def doShowIpPimBsrDebug( mode, args ):
   statusColl = pimBsr6StatusColl if 'ipv6' in args else pimBsrStatusColl
   pimBsrStatus = statusColl.vrfStatus[ vrfDefault ]
   model = PimBsrModel.PimBsrDebug()
   model.generate( pimBsrStatus )
   return model

class ShowIpPimBsrDebug( ShowCliCommandClass ):
   hidden = True
   syntax = 'show ip pim bsr debug'
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': pimNodeAfterShowIp,
      'bsr': bsrMatcher,
      'debug': debugMatcher
   }
   handler = doShowIpPimBsrDebug
   cliModel = PimBsrModel.PimBsrDebug

BasicCli.addShowCommandClass( ShowIpPimBsrDebug )

class ShowPimAfBsrDebug( ShowCliCommandClass ):
   hidden = True
   syntax = 'show pim ( ipv4 | ipv6 ) bsr debug'
   data = {
      'pim': pimNodeAfterShow,
      'ipv4': ipv4NodeForShow,
      'ipv6': ipv6NodeForShow,
      'bsr': bsrMatcher,
      'debug': debugMatcher
   }
   handler = doShowIpPimBsrDebug
   cliModel = PimBsrModel.PimBsrDebug

BasicCli.addShowCommandClass( ShowPimAfBsrDebug )

#-------------------------------------------------------------------------------
# Pim Bsr commands in "show tech-support"
#-------------------------------------------------------------------------------
def _showTechSupportCmds():
   cmds = []
   if mcastRoutingSupported( _entityManager.root(), routingHardwareStatus=None ):
      cmds += [ "show ip pim bsr",
                "show ip pim bsr rp",
                "show ip pim bsr debug" ]
   return cmds

TechSupportCli.registerShowTechSupportCmdCallback(
   '2010-06-11 02:00:00',
   _showTechSupportCmds )

def Plugin( entityManager ):
   global pimBsrStatusColl
   global pimBsr6StatusColl
   global pimGlobalStatus
   global pim6GlobalStatus
   global pimConfigColl
   global pimBsrRpCandidateStatusColl
   global pimBsr6RpCandidateStatusColl
   global pimRpConfigColl
   global pim6RpConfigColl
   global _mfibVrfConfig
   global _mfib6VrfConfig
   global _entityManager

   _entityManager = entityManager

   pimGlobalStatus = LazyMount.mount( entityManager, 
         Tac.Type( 'Routing::Pim::GlobalStatus' ).mountPath( 'ipv4' ),
         'Routing::Pim::GlobalStatus', 'r' )

   pim6GlobalStatus = LazyMount.mount( entityManager,
         Tac.Type( 'Routing::Pim::GlobalStatus' ).mountPath( 'ipv6' ),
         'Routing::Pim::GlobalStatus', 'r' )

   pimConfigColl = LazyMount.mount( entityManager, 'routing/pim/config',
                                    'Routing::Pim::ConfigColl', 'r' )

   pimBsrStatusColl = LazyMount.mount( entityManager, 'routing/pim/bsr/statuscoll',
                                'Routing::Pim::Bsr::StatusColl', 'r' )

   pimBsr6StatusColl = LazyMount.mount( entityManager, 'routing6/pim/bsr/statuscoll',
                                'Routing::Pim::Bsr::StatusColl', 'r' )

   pimBsrRpCandidateStatusColl = \
       LazyMount.mount( entityManager,
                        "routing/pim/bsr/rpcandidate",
                        "Routing::Pim::RpCandidateStatusColl", "r" )

   pimBsr6RpCandidateStatusColl = \
       LazyMount.mount( entityManager,
                        "routing6/pim/bsr/rpcandidate",
                        "Routing::Pim::RpCandidateStatusColl", "r" )

   pimRpConfigColl = LazyMount.mount( entityManager,
                                 'routing/pim/rpConfig/sparsemode/',
                                 'Routing::Pim::RpConfigColl', 'r')

   pim6RpConfigColl = LazyMount.mount( entityManager,
                                 'routing6/pim/rpConfig/sparsemode/',
                                 'Routing::Pim::RpConfigColl', 'r')


   pimBsrRpHashColl[ vrfDefault ]  = SmashLazyMount.mount( entityManager,
         "routing/pim/bsr/rphash/default", "Routing::Pim::RpHashStatus",
         SmashLazyMount.mountInfo( 'reader' ) )

   pimBsr6RpHashColl[ vrfDefault ]  = SmashLazyMount.mount( entityManager,
         "routing6/pim/bsr/rphash/default", "Routing::Pim::RpHashStatus",
         SmashLazyMount.mountInfo( 'reader' ) )

   _mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   _mfib6VrfConfig = LazyMount.mount( entityManager, 'routing6/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   PimCliLib.pimMessageCountersHook.addExtension( getPimMessageCounters )

