#!/usr/bin/env python
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tracing
import LazyMount
import IraIpIntfCli, ConfigMount
from CliParser import guardNotThisPlatform
import IntfCli, Tac
import IraIpCli, IraIp6Cli
from McastCommonCliLib import mcastRoutingSupported
from AclCli import getAclConfig
import PimCliLib
from IpLibConsts import DEFAULT_VRF as vrfDefault
from IpLibConsts import VRFNAMES_RESERVED
from PimBidirCliLib import isBidirMode
import CliPlugin.ConfigConvert
from CliPlugin.VrfCli import vrfExists
import MrouteCli
import RouterMulticastCliLib
from RouterMulticastCliLib import configGetters, lazyGetters, AddressFamily
from PimBsrCliLib import RouterPimBsrMode, RouterPimBsrVrfMode
from PimBsrCliLib import RouterPimBsrIpv4Mode
from PimBsrCliLib import RouterPimBsrIpv6Mode

__defaultTraceHandle__ = Tracing.Handle( 'PimBsr::Cli' )
t0 = Tracing.trace0

# pylint: disable-msg=W0105,C0321,E1103,W0621,W0612,C0123

# Globals
pimBsrConfigColl = None
pimBsr6ConfigColl = None
pimBsrStatusColl = None
_pimBsrLegacyConfig = None
_mfibVrfConfig = None
_mfibVrfConfig6 = None
_routingHwStatus = None
_routing6HwStatus = None

PimBsrConfigColl = 'Routing::Pim::Bsr::ConfigColl'

( configColl,
  configCollFromMode,
  config,
  configFromMode ) = configGetters( PimBsrConfigColl )

PimBsrClearConfig = 'Routing::Pim::Bsr::ClearConfigColl'
( clearConfigColl,
  clearConfigCollFromMode,
  clearConfig,
  clearConfigFromMode ) = lazyGetters( PimBsrClearConfig,
                                       collectionName='vrfClearConfig',
                                       writeMount=True )

PimBsrStatusColl = 'Routing::Pim::Bsr::StatusColl'
( statusColl,
  statusCollFromMode,
  status,
  statusFromMode ) = lazyGetters( PimBsrStatusColl,
                                  collectionName='vrfStatus',
                                  writeMount=False )

PimGlobalStatusSparse = 'Routing::Pim::GlobalStatus'
( pimGlobalStatusSparse,
  pimGlobalStatusSparseFromMode,
  pimGlobalStatusSparseVrf,
  pimGlobalStatusSparseVrfFromMode
) = lazyGetters(
      PimGlobalStatusSparse,
      collectionName='pimEnabledSparseModeVrf',
      writeMount=False )
                                      
PimGlobalStatusBidir = 'Routing::Pim::GlobalStatus'
( pimGlobalStatusBidir,
  pimGlobalStatusBidir,
  pimGlobalStatusBidirVrf,
  pimGlobalStatusBidirVrfFromMode
) = lazyGetters(
      PimGlobalStatusBidir,
      collectionName='pimEnabledBidirVrf',
      writeMount=False )
                                      
def pimBsrConfigCollFromFamily( ipFamily ):
   if ipFamily is None:
      af = AddressFamily.ipv4
   #elif ipFamily is None: TODO: Uncomment when common config is supported
      # Applies to common config
   #   af = AddressFamily.ipunknown
   else:
      af = ipFamily
   return configColl( af )

def allowed( ipFamily, vrfName ):
   # Possibly ipFamily == ipunknown, could be a common mode
   afSet = ( AddressFamily.ipv4, AddressFamily.ipv6 )
   if ipFamily in afSet:
      afSet = ( ipFamily, )
   return any ( ( vrfName in pimGlobalStatusSparse( af ).pimEnabledSparseModeVrf or
                  vrfName in pimGlobalStatusBidir( af ).pimEnabledBidirVrf
                ) for af in afSet )

def getMulticastVrfNames( mode=None ):
   return  _mfibVrfConfig.config.members()

def getMulticast6VrfNames( mode=None ):
   return _mfibVrfConfig6.config.members()

def minSupportVersion( minVersion ):
   if _pimBsrLegacyConfig.version < minVersion:
      _pimBsrLegacyConfig.version = minVersion

   if _pimBsrLegacyConfig.version > _pimBsrLegacyConfig.globalMode:
      _pimBsrLegacyConfig.legacy = False
   else:
      _pimBsrLegacyConfig.legacy = True

def _getPimBsrAddressFamilyFromMode( mode, args ):
   # All PimBsr commands are either ipv4 or ipv6; ipunknown indicates legacy ipv4
   af = args.get( AddressFamily.ipv4) or args.get( AddressFamily.ipv6 ) or (
         mode.af if hasattr( mode, 'af' ) else AddressFamily.ipunknown )
   if af == AddressFamily.ipunknown:
      af = AddressFamily.ipv4
   return af

def _getVrfNameFromMode( mode ):
   if hasattr( mode, 'vrfName' ):
      return mode.vrfName
   return vrfDefault

def _getPimBsrVrfConfig( vrfName ):
   pimBsrVrfConfig = None
   if vrfName in pimBsrConfigColl.vrfConfig:
      pimBsrVrfConfig = pimBsrConfigColl.vrfConfig[ vrfName ]
   return pimBsrVrfConfig

def _getOrCreatePimBsrIntfConfig( pimBsrConfigColl, intfName ):
   intfConfig = pimBsrConfigColl.intfConfig
   if intfName not in intfConfig:
      intfConfig.newMember( intfName )

   return intfConfig[ intfName ]

def _maybeDeletePimBsrIntfConfig( pimBsrConfigColl, intfName ):
   intfConfig = pimBsrConfigColl.intfConfig
   if intfName in intfConfig:
      if not len( intfConfig[ intfName ].bsrBorderGrp ) and \
            not len( intfConfig[ intfName ].bsrBorderAcl ):
         del intfConfig[ intfName ]

def _pimBsrVrfDefinitionHook( vrfName ) :
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      config( af, vrfName )
      clearConfig( af, vrfName )
   # "router pim bsr" will now show up
   # Set legacy flag to minSupported version to ensure we continue
   # to stay in ip mode, once we have entered that mode.
   minSupportVersion( _pimBsrLegacyConfig.routerMode )

def _pimBsrVrfDeletionHook( vrfName ):
   if vrfName == vrfDefault or \
         not _canDeletePimBsrVrfConfig( vrfName ):
      return
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in configColl( af ).vrfConfig:
         _config = config( af, vrfName )
         if _config.isDefault():
            # only delete if there is no non-default config and
            # the VRF is not defined
            del configColl( af ).vrfConfig[ vrfName ]

      if vrfName in clearConfigColl( af ).vrfClearConfig and \
            vrfName != vrfDefault:
         del clearConfigColl( af ).vrfClearConfig[ vrfName ]

   if len( pimBsrConfigColl.vrfConfig.keys() ) == 1 and \
         len( pimBsr6ConfigColl.vrfConfig.keys() ) == 1:
      if _canDeletePimBsrVrfConfig( vrfDefault ):
         _pimBsrLegacyConfig.legacy = True
         _pimBsrLegacyConfig.version = _pimBsrLegacyConfig.globalMode

def _canDeletePimBsrVrfConfig( vrfName ):
   canDelete = True
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in configColl( af ).vrfConfig:
         _config = config( af, vrfName )
         if not _config.isDefault():
            canDelete = False
   return canDelete

def _cleanupPimRpCandidateConfig( vrfName, mode ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in configColl( af ).vrfConfig:
         pimBsrVrfConfig = config( af, vrfName )
         for intfId, rpCandidateConfig in \
               pimBsrVrfConfig.rpCandidateConfig.items():
            if rpCandidateConfig.mode == mode:
               del pimBsrVrfConfig.rpCandidateConfig[ intfId ]

def cleanupPimBsrConfig( vrfName ):
   for af in [ AddressFamily.ipv4, AddressFamily.ipv6 ]:
      if vrfName in configColl( af ).vrfConfig:
         pimBsrConfig = config( af, vrfName )
         pimBsrConfig.reset()

def clearPimMessageCounters( vrfName, af=AddressFamily.ipv4 ):
   vrfClearConfig = clearConfig( af, vrfName )
   vrfClearConfig.countersCount += 1

def pimBsrModeIpv4SupportedGuard( mode, token ):
   if mcastRoutingSupported( mode.sysdbRoot, _routingHwStatus ):
      return None
   return guardNotThisPlatform

def pimBsrModeIpv6SupportedGuard( mode, token ):
   if mcastRoutingSupported( mode.sysdbRoot, _routing6HwStatus ):
      return None
   return guardNotThisPlatform

#----------------------------------------------------------
#( config )# [no|default] router pim bsr
#----------------------------------------------------------
def deleteRouterPimBsrBaseMode( mode, args ):
   # delete non-default config for all VRFs
   for configRoot in [ pimBsrConfigColl, pimBsr6ConfigColl ]:
      for vrfName in configRoot.vrfConfig.keys():
         if vrfName == vrfDefault:
            configRoot.vrfConfig[ vrfName ].reset()
         else:
            deleteRouterPimBsrVrfModeInternal( mode, vrfName )

   _pimBsrLegacyConfig.legacy = True
   _pimBsrLegacyConfig.version = _pimBsrLegacyConfig.globalMode

def gotoRouterPimBsrBaseMode( mode, args ):
   pimBsrVrfConfig = _getPimBsrVrfConfig( vrfDefault )
   if pimBsrVrfConfig is None:
      return
   minSupportVersion( _pimBsrLegacyConfig.routerMode )
   childMode = mode.childMode( RouterPimBsrMode )
   mode.session_.gotoChildMode( childMode )

#----------------------------------------------------------------------
# (config-router-pim-bsr)#[no|default] vrf <vrf>
#----------------------------------------------------------------------
def gotoRouterPimBsrVrfMode( mode, args ):
   vrfName = args.get( 'VRF' )

   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return

   _pimBsrVrfDefinitionHook( vrfName )
   if not vrfExists( vrfName ):
      mode.addWarning( "vrf name %s is not defined." % vrfName )
   
   childMode = mode.childMode( RouterPimBsrVrfMode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimBsrVrfModeInternal( mode, vrfName):
   if vrfName in VRFNAMES_RESERVED:
      mode.addError( "vrf name %s is reserved." % vrfName )
      return

   cleanupPimBsrConfig( vrfName )

   # If vrf is not under router multicast then delete the
   # vrfConfig when PIM BSR configuration
   # for that vrf goes away.
   if ( vrfName not in getMulticastVrfNames() and
         vrfName not in getMulticast6VrfNames() ):
      _pimBsrVrfDeletionHook( vrfName )

def deleteRouterPimBsrVrfMode( mode, args ):
   vrfName = args.get( 'VRF' )
   deleteRouterPimBsrVrfModeInternal( mode, vrfName )

#-------------------------------------------------------------------------------
# (config-router-pim-bsr-*) [no|default] ipv4
#-------------------------------------------------------------------------------
def gotoRouterPimBsrIpv4Mode( mode, args ):
   minSupportVersion( _pimBsrLegacyConfig.ipMode )
   vrfName = _getVrfNameFromMode( mode )
   IraIpCli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterPimBsrIpv4Mode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimBsrIpv4Mode( mode, args ):
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = config( af=AddressFamily.ipv4, vrfName=vrfName )
   vrfConfig.reset()
   pimBsrConfigRoot = configColl( af=AddressFamily.ipv4 )
   if all( vrfConfig.isDefault() for vrfConfig in
         pimBsrConfigRoot.vrfConfig.itervalues()
         if vrfConfig.vrfName != vrfDefault ):
      minSupportVersion( _pimBsrLegacyConfig.routerMode )

#-------------------------------------------------------------------------------
# (config-router-pim-bsr-*) [no|default] ipv6
#-------------------------------------------------------------------------------
def gotoRouterPimBsrIpv6Mode( mode, args ):
   minSupportVersion( _pimBsrLegacyConfig.ipMode )
   vrfName = _getVrfNameFromMode( mode )
   IraIp6Cli.warnIfRoutingDisabled( mode, vrfName )
   childMode = mode.childMode( RouterPimBsrIpv6Mode, vrfName=vrfName )
   mode.session_.gotoChildMode( childMode )

def deleteRouterPimBsrIpv6Mode( mode, args ):
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = config( af=AddressFamily.ipv6, vrfName=vrfName )
   vrfConfig.reset()
   pimBsrConfigRoot = configColl( af=AddressFamily.ipv6 )
   if all( vrfConfig.isDefault() for vrfConfig in
         pimBsrConfigRoot.vrfConfig.itervalues()
         if vrfConfig.vrfName != vrfDefault ):
      minSupportVersion( _pimBsrLegacyConfig.routerMode )

# decorator for config commands,
# if either routing or multicast routing is disabled,
# print warning.
def pimBsrCliConfigCommand( func ):
   def newFunc( *args, **kwargs ):
      mode = args[ 0 ]
      af = _getPimBsrAddressFamilyFromMode( mode, kwargs[ 'args' ] )
      vrfName = _getVrfNameFromMode( mode )
      if not allowed( af, vrfName ):
         mode.addWarning(
           "WARNING: 'PIM' is not enabled, "
           " the command will have no effect." )
      return func( *args, **kwargs )
   return newFunc


modelet = IraIpIntfCli.RoutingProtocolIntfConfigModelet

#-----------------------------------------------------------------------------------
# legacy: switch( config-router-pim-bsr-*)#ip pim bsr-candidate interface
#                                             [ groupOrAcl ] [ priority ]
#                                             [ hashMask ] [ interval ]
# switch( config-router-pim-bsr-*)# [ no|default ] candidate interface
#                                      [ groupOrAcl ] [ priority ]
#                                      [ hashMask ] [ interval ]
#-----------------------------------------------------------------------------------

def _validPimBsrCandidate( mode, vrfConfig, intfName, groupPrefix ):

   for intf, bsrCandidateConfig in vrfConfig.bsrCandidateConfig.iteritems():
      if intf == intfName:
         continue
      if groupPrefix in bsrCandidateConfig.group:
         mode.addError(
            "ERROR: groupRange %s is already served by %s\n"
               "           fix the error to apply this configuration." %
               ( groupPrefix, intf ) )
         return False
   return True

def _validPimBsrCandidateAcl( mode, vrfConfig, intfName, aclName ):

   for intf, bsrCandidateConfig in vrfConfig.bsrCandidateConfig.iteritems():
      if intf == intfName:
         continue
      if aclName in bsrCandidateConfig.acl:
         mode.addError(
            "ERROR: Acl %s is already served by %s\n"
               "           fix the error to apply this configuration." %
               ( aclName, intf ) )
         return False
   return True

@pimBsrCliConfigCommand
def setPimBsrCandidate( mode, args ):
   # First check for any errors caught by our adapter functions.
   if mode.session.hasError():
      return
   intf = args.get( 'INTF' )
   priority = args.get( 'PRIORITY' )
   hashMask = args.get( 'HASHMASK' )
   intvl = args.get( 'INTERVAL' )

   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = config( af, vrfName )
   vrfStatus = status( af, vrfName )

   gAcl = None

   gPrefix = args.get( 'ADDRPREFIX' )
   if gPrefix is None:
      gPrefix = args[ 'DEFAULTPREFIX' ]
      gAcl = args.get( 'ACL' )

   intfId = Tac.Value( "Arnet::IntfId", str( intf ) )

   if gAcl is None:
      assert gPrefix

      if not _validPimBsrCandidate( mode, vrfConfig, intfId.stringValue, gPrefix ):
         return

      if not vrfConfig.bsrCandidateConfig.get( intfId ):
         bsrc = vrfConfig.bsrCandidateConfig.newMember( intfId )
      else:
         bsrc = vrfConfig.bsrCandidateConfig.get( intfId )

      bsrGrpConf = bsrc.group.get( gPrefix ) or bsrc.group.newMember( gPrefix )

      if priority is not None:
         bsrGrpConf.priority = priority
      else:
         bsrGrpConf.priority = bsrGrpConf.priorityDefault

      if hashMask is not None:
         bsrGrpConf.hashMaskLen = hashMask
      else:
         bsrGrpConf.hashMaskLen = \
            Tac.Type( "Routing::Pim::RpHashMaskLen" ).hashMaskLenDefault( af )

      if intvl is not None:
         bsrGrpConf.interval = intvl
      else:
         bsrGrpConf.interval = bsrGrpConf.intervalDefault

   else:
      if not _validPimBsrCandidateAcl( mode, vrfConfig, intfId.stringValue, gAcl ):
         return

      if not vrfConfig.bsrCandidateConfig.get( intfId ):
         bsrc = vrfConfig.bsrCandidateConfig.newMember( intfId )
      else:
         bsrc = vrfConfig.bsrCandidateConfig.get( intfId )

      bsrAclConf = bsrc.acl.get( gAcl) or bsrc.acl.newMember( gAcl )

      if priority is not None:
         bsrAclConf.priority = priority
      else:
         bsrAclConf.priority = bsrAclConf.priorityDefault

      if hashMask is not None:
         bsrAclConf.hashMaskLen = hashMask
      else:
         bsrAclConf.hashMaskLen = \
            Tac.Type( "Routing::Pim::RpHashMaskLen" ).hashMaskLenDefault( af )

      if intvl is not None:
         bsrAclConf.interval = intvl
      else:
         bsrAclConf.interval = bsrAclConf.intervalDefault


def noPimBsrCandidate( mode, args ):
   # First check for any errors caught by our adapter functions.
   if mode.session.hasError():
      return
   bsrc = None
   gPrefix = None
   gAcl = None

   intf = args.get( 'INTF' )
   priorityOpt = 'priority' in args
   hashMaskOpt = 'hashmask' in args
   intvlOpt = 'interval' in args
   # There should be at most one of these three keywords
   assert sum( 1 for opt in [ priorityOpt, hashMaskOpt, intvlOpt ] if opt ) <= 1
  
   af = _getPimBsrAddressFamilyFromMode( mode, args )

   vrfName = _getVrfNameFromMode( mode )
   vrfConfig = config( af, vrfName )

   gPrefix = args.get( 'ADDRPREFIX' )
   if gPrefix is None:
      gAcl = args.get( 'ACL' )

   def delBsrCandidate( intf ):
      if intf not in vrfConfig.bsrCandidateConfig:
         return
      vrfConfig.bsrCandidateConfig[ intf ].group.clear()
      vrfConfig.bsrCandidateConfig[ intf ].acl.clear()
      del vrfConfig.bsrCandidateConfig[ intf ]

   if intf:
      intfId = str( intf )

      if intfId not in vrfConfig.bsrCandidateConfig:
         return

      bsrc = vrfConfig.bsrCandidateConfig[ intfId ]
      # If prefix groups exists, treat this as a request to delete the  default
      # prefix.  Otherwise, the entire interface config will be deleted below.
      if len( bsrc.group ) and not gAcl and not gPrefix:
         gPrefix = args[ 'DEFAULTPREFIX' ]

      vrfStatus = status( af, vrfName )

      if gPrefix is not None:
         if gPrefix in vrfStatus.zoneGroup and \
                vrfStatus.zoneGroup[ gPrefix ] != intfId:
            return
         if priorityOpt:
            bsrc.group[ gPrefix ].priority = \
                bsrc.group[ gPrefix ].priorityDefault
         elif hashMaskOpt:
            bsrc.group[ gPrefix ].hashMaskLen = \
                Tac.Type( "Routing::Pim::RpHashMaskLen" ).hashMaskLenDefault( af )
         elif intvlOpt:
            bsrc.group[ gPrefix ].interval = \
                bsrc.group[ gPrefix ].intervalDefault
         else:
            del bsrc.group[ gPrefix ]
      elif gAcl:
         if vrfStatus and gAcl in vrfStatus.zoneAcl and \
                vrfStatus.zoneAcl[ gAcl ] != intfId:
            return
         elif gAcl not in vrfStatus.zoneAcl:
            return
         if priorityOpt:
            bsrc.acl[ gAcl ].priority = \
                bsrc.acl[ gAcl ].priorityDefault
         elif hashMaskOpt:
            bsrc.acl[ gAcl ].hashMaskLen = \
                Tac.Type( "Routing::Pim::RpHashMaskLen" ).hashMaskLenDefault( af )
         elif intvlOpt:
            bsrc.acl[ gAcl ].interval = \
                bsrc.acl[ gAcl ].intervalDefault
         else:
            del bsrc.acl[ gAcl ]
      else:
         delBsrCandidate( intfId )
         return

      if not len( bsrc.group ) \
             and not len( bsrc.acl ):
         delBsrCandidate( intfId )
   else:
      for i in vrfConfig.bsrCandidateConfig:
         delBsrCandidate( i )

#----------------------------------------------------------------------
# legacy: switch(config-router-pim-bsr-*)# [ no ] ip pim bsr-holdtime
# switch( config-router-pim-bsr-*)# [ no ] holdtime
#----------------------------------------------------------------------

@pimBsrCliConfigCommand
def setPimBsrHoldtime( mode, args ):
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )
   holdtime = args.get( 'HOLDTIME' )

   vrfConfig = config( af, vrfName )
   assert vrfConfig
   vrfConfig.holdTime = holdtime

def noPimBsrHoldtime( mode, args=None ):
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )

   vrfConfig = config( af, vrfName )
   assert vrfConfig

   vrfConfig.holdTime = vrfConfig.holdTimeDefault

#----------------------------------------------------------------------
# legacy: switch(config-router-pim-bsr-*)# [ no ] ip pim bsr-szTimeout
# switch( config-router-pim-bsr-*)# [ no ] szTimeout
#----------------------------------------------------------------------

@pimBsrCliConfigCommand
def setPimBsrSzTimeout( mode, args ):
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )
   bsrSzTimeout = args.get( 'SZTIMEOUT' )

   vrfConfig = config( af, vrfName )
   assert vrfConfig

   vrfConfig.szTimeout = bsrSzTimeout
   if vrfConfig.szTimeout < ( 10 * vrfConfig.holdTime ):
      mode.addWarning(
           "WARNING: Scoped Zone timeout too low, "
           "system will use 10 x holdtime. Consider "
           "changing to a value greater than or equal to %s." %
           ( 10 * vrfConfig.holdTime ) )

def noPimBsrSzTimeout( mode, args ):
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )

   vrfConfig = config( af, vrfName )
   assert vrfConfig

   vrfConfig.szTimeout = vrfConfig.szTimeoutDefault

#----------------------------------------------------------------------
# Legacy: switch(config-if)# [ no ] ip pim bsr-border <aclName>
# switch(config-if)# [ no ] pim [ ipv4|ipv6] bsr border <aclName>
#----------------------------------------------------------------------

def noPimBsrBorder( mode, args ):
   # First check for any errors caught by our adapter functions.
   if mode.session.hasError():
      return
   ipFamily = args.get( AddressFamily.ipv4 ) or args.get( AddressFamily.ipv6 )
   # Allow legacy command syntax
   ipFamily = AddressFamily.ipv4 if args.get( 'ip' ) else ipFamily
   assert ipFamily

   bsrConfigColl = pimBsrConfigCollFromFamily( ipFamily )
   pic = _getOrCreatePimBsrIntfConfig( bsrConfigColl, mode.intf.name )

   if not pic:
      return

   gAcl = None
   gPrefix = args.get( 'ADDRPREFIX' )

   if not gPrefix:
      gAcl = args.get( 'ACL' )

   if gAcl:
      del pic.bsrBorderAcl[ gAcl ]

   elif gPrefix:
      del pic.bsrBorderGrp[ gPrefix ]

   else:
      pic.bsrBorderGrp.clear()
      pic.bsrBorderAcl.clear()

   _maybeDeletePimBsrIntfConfig( bsrConfigColl, mode.intf.name )

@pimBsrCliConfigCommand
def setPimBsrBorder( mode, args ):
   # First check for any errors caught by our adapter functions.
   if mode.session.hasError():
      return
   ipFamily = args.get( AddressFamily.ipv4 ) or args.get( AddressFamily.ipv6 )
   # Allow legacy command syntax
   ipFamily = AddressFamily.ipv4 if args.get( 'ip' ) else ipFamily
   assert ipFamily

   bsrConfigColl = pimBsrConfigCollFromFamily( ipFamily )
   pic = _getOrCreatePimBsrIntfConfig( bsrConfigColl, mode.intf.name )

   if pic is None:
      return

   gAcl = None
   gPrefix = args.get( 'ADDRPREFIX' )

   if not gPrefix: 
      gPrefix = args[ 'DEFAULTPREFIX' ]
      gAcl = args.get( 'ACL' )

   if not gAcl:
      pic.bsrBorderGrp[ gPrefix ] = True

   else:
      pic.bsrBorderAcl[ gAcl ] = True

#-------------------------------------------------------------------------------
# The PimBsrIntf class is used to remove the PimBsr IntfConfig object when
# an interface is deleted. The Intf class will create a new instance
# of PimBsrIntf and call destroy when the interface is deleted
# -------------------------------------------------------------------------------
class PimBsrIntf( IntfCli.IntfDependentBase ):
   #----------------------------------------------------------------------------
   # Destroys the IntfConfig object for this interface if it exists.
   #----------------------------------------------------------------------------
   def setDefault( self ):
      name = self.intf_.name
      del pimBsrConfigColl.intfConfig[ name ]
      del pimBsr6ConfigColl.intfConfig[ name ] 

rpIntvl =  Tac.Value( "Routing::Pim::Bsr::CrpAdvInterval" )
rpPriority = Tac.Value( "Routing::Pim::RpPriority" )

#------------------------------------------------------------------------------
# Legacy:
# ip pim rp-candidate interface-type interface-number [ groupOrAcl ] [ priority ]
#                      [ interval ]
#------------------------------------------------------------------------------
# rp candidate interface-type interface-number [ groupOrAcl ] [ priority ]
#               [ interval ]
#------------------------------------------------------------------------------

@pimBsrCliConfigCommand
def setIpPimRpCandidate( mode, args ):
   # First check for any errors caught by our adapter functions.
   if mode.session.hasError():
      return
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   intf = args.get( 'INTF' )
   priority = args.get( 'PRIORITY' )
   intvl = args.get( 'INTERVAL' )
   gAcl = None

   gPrefix = args.get( 'ADDRPREFIX' )

   if gPrefix is None:
      gPrefix = args[ 'DEFAULTPREFIX' ]
      gAcl = args.get( 'ACL' )

   _config = configFromMode( mode, legacy=False )
   intfId = Tac.Value( "Arnet::IntfId", str( intf ) )

   if not _config.rpCandidateConfig.get( intfId ):
      rpc = _config.rpCandidateConfig.newMember( intfId )
      if intvl:
         rpc.advInterval = intvl
      else:
         rpc.advInterval = rpIntvl.intervalDefault

   else:
      rpc = _config.rpCandidateConfig.get( intfId )
      if intvl:
         rpc.advInterval = intvl

   if priority is None:
      priority = rpPriority.priorityBsrDefault

   if gAcl:
      rpc.acl[ gAcl ] = True
      rpc.aclOptions.newMember( gAcl )
      rpc.aclOptions[ gAcl ].priority = priority
   else:
      rpc.groupRange[ gPrefix ] = True
      rpc.groupRangeOptions.newMember( gPrefix )
      rpc.groupRangeOptions[ gPrefix ].priority = priority

   # We don't currently support an overlap between Bidir and Sparse-mode so make
   # the most recently configured value take effect.
   if isBidirMode( mode ):
      rpc.mode = 'modePimBidir'
   else:
      rpc.mode = 'modePimSm'

def noIpPimRpCandidate( mode, args ):
   # First check for any errors caught by our adapter functions.
   if mode.session.hasError():
      return
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )
   _config = config( af, vrfName )
   intf = args.get( 'INTF' )
   priorityOpt = 'priority' in args
   intervalOpt = 'interval' in args
   gAcl = None
   gPrefix = args.get( 'ADDRPREFIX' )

   if gPrefix is None:
      gAcl= args.get( 'ACL' )

   if not intf:
      for i in _config.rpCandidateConfig:
         if isBidirMode( mode ) and \
               _config.rpCandidateConfig[ i ].mode == 'modePimBidir':
            del _config.rpCandidateConfig[ i ]
         if not isBidirMode( mode ) and \
               _config.rpCandidateConfig[ i ].mode == 'modePimSm':
            del _config.rpCandidateConfig[ i ]
      return


   intfId = Tac.Value( "Arnet::IntfId", str( intf ) )
   rpc = _config.rpCandidateConfig.get( intfId )

   if not rpc:
      return

   if ( isBidirMode( mode ) and rpc.mode != 'modePimBidir' ) or \
         ( not isBidirMode( mode ) and rpc.mode != 'modePimSm' ):
      return

   if not gPrefix and not gAcl:
      if priorityOpt:
         defaultPrefix = args[ 'DEFAULTPREFIX' ]
         if defaultPrefix in rpc.groupRange:
            rpc.groupRangeOptions[ defaultPrefix ].priority = \
                  rpPriority.priorityBsrDefault
         else:
            return
      elif intervalOpt:
         rpc.advInterval = rpIntvl.intervalDefault
      else:
         del _config.rpCandidateConfig[ intfId ]
         return
   elif gPrefix:
      if gPrefix in rpc.groupRange and priorityOpt:
         rpc.groupRangeOptions[ gPrefix ].priority = \
               rpPriority.priorityBsrDefault
      else:
         del rpc.groupRange[ gPrefix ]
         del rpc.groupRangeOptions[ gPrefix ]
   else:
      if gAcl in rpc.acl and priorityOpt:
         rpc.aclOptions[ gAcl ].priority = \
               rpPriority.priorityBsrDefault
      else:
         del rpc.acl[ gAcl ]
         del rpc.aclOptions[ gAcl ]

   if not rpc.groupRange and not rpc.acl:
      del _config.rpCandidateConfig[ intfId ]

#------------------------------------------------------------------------------------
# legacy: switch( config-router-pim-bsr-*)#[ no ] ip pim bsr rp-candidate \
#                                                   advertisement-filter access-list
#switch( config-router-pim-bsr-*)#[ no ] rp-candidate \
#                                                   advertisement-filter access-list
#------------------------------------------------------------------------------------

def setIpPimBsrRpCandidateAdsFilter ( mode, args ):
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )

   pimBsrConfig = config( af, vrfName )
   assert pimBsrConfig

   aclAf = af if af == AddressFamily.ipv6 else 'ip'
   acl = args.get( 'ACL' )
   aclCheck = getAclConfig( aclAf, cliConf=True ).get( acl )
   if aclCheck and not aclCheck.standard:
      mode.addError(
            '%s is not a standard acl' % acl )
      return

   if acl not in pimBsrConfig.crpFilterAcl:
      pimBsrConfig.crpFilterAcl[ acl ] = True

def noIpPimBsrRpCandidateAdsFilter ( mode, args ):
   af = _getPimBsrAddressFamilyFromMode( mode, args )
   vrfName = _getVrfNameFromMode( mode )

   pimBsrConfig = config( af, vrfName )
   assert pimBsrConfig

   acl = args.get( 'ACL' )
   if acl:
      aclAf = af if af == AddressFamily.ipv6 else 'ip'
      aclCheck = getAclConfig( aclAf, cliConf=True ).get( acl )
      if aclCheck:
         if not aclCheck.standard:
            mode.addError(
                  '%s is not a standard acl' % acl )
            return

      if acl in pimBsrConfig.crpFilterAcl:
         del pimBsrConfig.crpFilterAcl[acl]
   else:
      for acl in pimBsrConfig.crpFilterAcl:
         del pimBsrConfig.crpFilterAcl[acl]

#-------------------------------------------------------------------------------
#Register convertLegacyConfigPimBsr via "config convert new-syntax"
#-------------------------------------------------------------------------------
def convertLegacyConfigPimBsr( mode ):
   minSupportVersion( _pimBsrLegacyConfig.ipMode )

CliPlugin.ConfigConvert.registerConfigConvertCallback( convertLegacyConfigPimBsr )

#-------------------------------------------------------------------------------
# Plugin
#-------------------------------------------------------------------------------

def Plugin( entityManager ):
   global pimBsrConfigColl, pimBsr6ConfigColl
   global pimBsrStatusColl
   global _mfibVrfConfig
   global _mfibVrfConfig6
   global _routingHwStatus
   global _routing6HwStatus
   global _pimBsrLegacyConfig

   configTypes = [ PimBsrConfigColl ]
   RouterMulticastCliLib.doConfigMounts( entityManager, configTypes )

   pimBsrConfigColl = configColl( af=AddressFamily.ipv4 )
   pimBsr6ConfigColl = configColl( af=AddressFamily.ipv6 )

   lazyMountTypes = [ PimBsrStatusColl,
                      PimGlobalStatusSparse,
                      PimGlobalStatusBidir ]
   RouterMulticastCliLib.doLazyMounts( entityManager, lazyMountTypes,
                                       useWriteMount=False )
   # Keeping this for legacy only
   pimBsrStatusColl = statusColl( af=AddressFamily.ipv4 )

   _mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   _mfibVrfConfig6 = LazyMount.mount( entityManager, 'routing6/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )

   _routingHwStatus = LazyMount.mount( entityManager,
         'routing/hardware/status',
         'Routing::Hardware::Status', 'r' )

   _routing6HwStatus = LazyMount.mount( entityManager,
         'routing6/hardware/status',
         'Routing6::Hardware::Status', 'r' )

   _pimBsrLegacyConfig = ConfigMount.mount( entityManager,
         'routing/pim/bsr/legacyconfig',
         'McastCommon::LegacyConfig', 'w' )

   RouterMulticastCliLib.doLazyMounts( entityManager, [ PimBsrClearConfig ],
                                       useWriteMount=True )

   PimCliLib.pimClearMessageCountersHook.addExtension(
         clearPimMessageCounters )

   IntfCli.Intf.registerDependentClass( PimBsrIntf, priority=21 )

   MrouteCli.routerMcastVrfDefinitionHook.addExtension( _pimBsrVrfDefinitionHook )
   MrouteCli.routerMcastVrfDeletionHook.addExtension( _pimBsrVrfDeletionHook )

   PimCliLib.pimSparseModeVrfConfiguredHook.addExtension( _pimBsrVrfDefinitionHook )
   PimCliLib.canDeletePimSparseModeVrfHook.addExtension( _canDeletePimBsrVrfConfig )
   PimCliLib.pimSparseModeVrfDeletedHook.addExtension( _pimBsrVrfDeletionHook )

   PimCliLib.pimBidirVrfConfiguredHook.addExtension( _pimBsrVrfDefinitionHook )
   PimCliLib.canDeletePimBidirModeVrfHook.addExtension( _canDeletePimBsrVrfConfig )
   PimCliLib.pimBidirVrfDeletedHook.addExtension( _pimBsrVrfDeletionHook )

   PimCliLib.pimRpCandidateCleanupHook.addExtension( _cleanupPimRpCandidateConfig )
