#!/usr/bin/env python
# Copyright (c) 2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#-------------------------------------------------------------------------------
# This module implements Bidir DF show commands
#
# show ip pim [ vrf <vrfName> ] bidirectional df
#                    [ interface <interface> ] [ rpa <rpa> ]
#-------------------------------------------------------------------------------

import sys

import Arnet
import BasicCli
import CliCommand
import CliParser
import CliToken.Ip
import CliToken.Pim
import IntfCli
import IpAddrMatcher
import LazyMount
import MrouteCli
import PimBidirDfSmashHelper
import PimBidirDfModel
import PimCliLib
import ShowCommand
import SmashLazyMount
import Tac
from CliPlugin.VrfCli import VrfExprFactory, DEFAULT_VRF

_dfStatusColl = {}
_dfOrderedStatusColl = {}
_pimGlobalStatus = None
_dfStatusReaderSmColl = {}
_dfStatusRestartSmColl = {}
_routingHwStatus = None
_mfibVrfConfig = None
_entityManager = None

def _pimBidirDfVrfDeleted( vrfName ):
   if vrfName in _dfStatusColl and vrfName != DEFAULT_VRF:
      del _dfStatusColl[ vrfName ]
      del _dfOrderedStatusColl[ vrfName ]
      del _dfStatusReaderSmColl[ vrfName ]
      del _dfStatusRestartSmColl[ vrfName ]

def _pimBidirectionalGuard( mode, token ):
   if _routingHwStatus.pimBidirectionalSupported:
      return None
   else:
      return CliParser.guardNotThisPlatform

def allowed( vrfName ):
   return vrfName in _pimGlobalStatus.pimEnabledBidirVrf

# Add to message counters extension
def getPimMessageCounters( vrfName, af, **kwargs ):
   if vrfName in _pimGlobalStatus.pimEnabledBidirVrf:
      path = PimCliLib.getPath( 'Routing::Pim::Smash::MessageCounters', 
                                af, vrfName, "bidirDf" )
      return SmashLazyMount.mount(
            _entityManager,
            path,
            'Routing::Pim::Smash::MessageCounters',
            SmashLazyMount.mountInfo( 'reader' ) )
   return None

def doMaybeInitDfStatusSms( vrfName ):
   # pylint: disable-msg=W0602
   global _dfStatusColl
   global _dfOrderedStatusColl
   global _dfStatusReaderSmColl
   global _dfStatusRestartSmColl

   if vrfName in _dfStatusColl:
      return _dfStatusColl[ vrfName ].initialized

   _dfStatusColl[ vrfName ] = Tac.newInstance(
         "Routing::Pim::Bidir::Df::Status" )
   _dfOrderedStatusColl[ vrfName ] = Tac.newInstance(
         "Routing::Pim::Bidir::Df::OrderedStatus" )

   dfStatus = _dfStatusColl[ vrfName ]
   dfOrderedStatus = _dfOrderedStatusColl[ vrfName ]

   dfSmashStatus = PimBidirDfSmashHelper.mountInDependencyOrder(
      _entityManager, "routing/pim/bidir/df/status/" + vrfName, 'keyshadow' )

   _dfStatusRestartSmColl[ vrfName ] = Tac.newInstance(
      "Routing::Pim::Bidir::Df::Smash::BidirIntfSmashRestartSm",
      dfStatus, dfOrderedStatus, dfSmashStatus, 2, Tac.activityManager.clock )

   _dfStatusReaderSmColl[ vrfName ] = Tac.newInstance(
      "Routing::Pim::Bidir::Df::Smash::BidirIntfSmashReaderSm",
      dfStatus, dfOrderedStatus, dfSmashStatus )

   return dfStatus.initialized

def waitForDfStatusSms( mode, vrfName ):
   try:
      Tac.waitFor( lambda: doMaybeInitDfStatusSms( vrfName ),
                   timeout=60, warnAfter=None,
                   sleep=not Tac.activityManager.inExecTime.isZero, maxDelay=0.1,
                   description=' mroute table to be created' )
   except Tac.Timeout:
      mode.addError( "Command timed out" )
      return False
   return True

def showBidirDfInfo( mode, args ):
   vrfName = args.get( 'VRF', DEFAULT_VRF )
   intf = args.get( 'INTF' )
   rpAddr = args.get( 'RPADDR' )

   if not PimCliLib.validVrfName( mode, vrfName, _mfibVrfConfig ):
      return None

   if not waitForDfStatusSms( mode, vrfName ):
      return None

   dfStatus = _dfStatusColl[ vrfName ]
   dfOrderedStatus = _dfOrderedStatusColl[ vrfName ]
   showDfInfo = Tac.newInstance( "BidirDfCliCapi::ShowDfInfo", dfStatus,
                                 dfOrderedStatus )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()

   if not intf:
      intf = ""
   else:
      intf = str( intf )

   if not rpAddr:
      rpAddr = Tac.newInstance( "Arnet::IpGenAddr" )
   else:
      rpAddr = Arnet.IpGenAddr( rpAddr )

   showDfInfo.render( fd, fmt, intf, rpAddr )

   return PimBidirDfModel.Intfs

#--------------------------------------------------------------------------------
# show ip pim [ vrf VRF ] bidirectional df [ interface INTF ] [ rpa RPADDR ]
#--------------------------------------------------------------------------------
class IpPimBidirectionalDfCmd( ShowCommand.ShowCliCommandClass ):
   syntax = ( 'show ip pim [ VRF ] bidirectional df [ interface INTF ] '
                                                        '[ rpa RPADDR ]' )
   data = {
      'ip': CliToken.Ip.ipMatcherForShow,
      'pim': CliToken.Pim.pimNodeAfterShowIp,
      'VRF': VrfExprFactory( helpdesc='VRF name' ),
      'bidirectional': CliCommand.guardedKeyword( 'bidirectional',
         helpdesc='PIM Bidirectional information', guard=_pimBidirectionalGuard ),
      'df': 'PIM Bidirectional DF information',
      'interface': 'PIM Bidirectional interface specific state DF information',
      'INTF': IntfCli.Intf.matcher,
      'rpa': 'PIM Bidirectional RPA specific state DF information',
      'RPADDR': IpAddrMatcher.IpAddrMatcher(
         helpdesc='IP address of PIM Bidirectional RP' ),
   }
   handler = showBidirDfInfo
   cliModel = PimBidirDfModel.Intfs

BasicCli.addShowCommandClass( IpPimBidirectionalDfCmd )

def Plugin( entityManager ):
   global _entityManager
   global _pimGlobalStatus
   global _routingHwStatus
   global _mfibVrfConfig

   _entityManager = entityManager
   _pimGlobalStatus = LazyMount.mount( entityManager,
         Tac.Type( 'Routing::Pim::GlobalStatus' ).mountPath( 'ipv4' ),
         "Routing::Pim::GlobalStatus", "r" )

   _routingHwStatus = LazyMount.mount( entityManager, 'routing/hardware/status', 
         'Routing::Hardware::Status', 'r' )
   
   _mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
         'Routing::Multicast::Fib::VrfConfig', 'r' )
   
   PimCliLib.pimMessageCountersHook.addExtension( getPimMessageCounters )

   MrouteCli.routerMcastVrfDeletionHook.addExtension( _pimBidirDfVrfDeleted )

