#!/usr/bin/env python
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

#---------------------------------------------------------------------
# This module implements PimBidir show commands
#
#show ip pim upstream joins bidirectional [ <sourceOrGroup> ]
#                                         [ <sourceOrGroup> ]
#                                         [ neighbor <address> ]
#
# show ip mroute bidirectional [ <group> ] [ detail ]
#
# show ip mroute bidirectional <group> interface [ <interface> ][ <detail> ]
#
# show ip mroute bidirectional count
#
# show ip pim config-sanity bidirectional
#
# show ip pim [ vrf <vrf-name> ] bidirectional rp [ groupPrefix ] [ detail ]
#
# show ip pim [ vrf <vrf-name> ] bidirectional rp-hash <group>
#
#---------------------------------------------------------------------

'''Show commands supported for bidirectional PIM'''

import Tac
import Tracing
import LazyMount
import BasicCli
import ShowCommand
from CliToken.Ip import ipMatcherForShow
import Arnet
import CliPlugin.VrfCli as VrfCli
from CliToken.Pim import ( pimNodeAfterShowIp, vrfMatcher, vrfNameMatcher,
                           upstreamMatcher, neighborsMatcher, joinsMatcher,
                           joinsNeighborAddrMatcher, mrouteMatcher,
                           mrouteInterfaceMatcherForShow, rpMatcher,
                           groupRangeMatcher, detailMatcher, configSanityMatcher,
                           bidirMatcher, mrouteCountMatcher, groupMatcher,
                           rpHashMatcher )
from PimBidirCliToken import detailBidirMatcher
import PimLib
from PimLib import AddressFamily
import PimModelLib
import PimCliLib
import PimModel
import SharedMem, SmashLazyMount
import PimBidirSmashHelper
from PimCliLib import validVrfName
from PimCliLib import pimBidirConfigCheck
from PimCliLib import PimConfigCheckModel
from IpLibConsts import DEFAULT_VRF
import FibUtils
import sys
import IntfCli
from IraIpRouteCliLib import isValidPrefixWithError

__defaultTraceHandle__ = Tracing.Handle( 'PimBidirCli' )
t0 = Tracing.trace0

_pimStatusColl = None
_pimGlobalStatus = None
_joinPruneStatusColl = None
_entityManager = None
_pimBidirStatusColl = {}
_pimBidirGroupSmashReaderSmColl = {}
_pimBidirGroupSmashRestartSmColl = {}
_staticRpColl = None
_pimBsrCrpStatusColl = None
_staticRpComputeObjColl = {}
_staticTrieBuilderColl = {}
_bsrRpComputeObjColl = {}
_bsrTrieBuilderColl = {}


_ipConfig = None
_pimConfigRoot = None
_pimBidirStaticRpConfigColl = None
_pimBidirRpConfigColl = None
_aclConfigCli = None
_mcastBoundaryConfig = None
_routingHwStatus = None
_routingVrfInfoDir = None
_mfibVrfConfig = None

def getPimUpstreamJoinsBidir( vrfName ):
   status = _joinPruneStatusColl.vrfStatus.get( vrfName )
   return status

# Add to message counters extension
def getPimMessageCounters( vrfName, af, **kwargs ):
   # pylint: disable-msg=W0212
   if vrfName in _pimGlobalStatus.pimEnabledBidirVrf:
      path = PimCliLib.getPath( 'Routing::Pim::Smash::MessageCounters',
                                af, vrfName, "bidir" )
      return SmashLazyMount.mount(
            _entityManager,
            path,
            'Routing::Pim::Smash::MessageCounters',
            SmashLazyMount.mountInfo( 'reader' ) )
   return None

def getPimStatus( vrfName ):
   # pylint: disable-msg=W0212
   if ( vrfName in _pimGlobalStatus.pimEnabledBidirVrf ) and \
         vrfName in _pimStatusColl.vrfStatus:
      return _pimStatusColl.vrfStatus[ vrfName ]
   return None

def displayIntfMode( intf ):
   if intf.mode in [ 'modePimBidir', 'modePimsmAndBidir' ] :
      return True

def waitForPimStatusSms( mode ):
   try:
      description = ' Pim table to be populated'
      _pimGlobalStatusReaderSm = PimLib.getPimGlobalStatusReaderSm(
            _entityManager, AddressFamily.ipv4 )
      Tac.waitFor( lambda: _pimGlobalStatusReaderSm.initialized, timeout=10,
                   warnAfter=None, sleep=not Tac.activityManager.inExecTime.isZero,
                   maxDelay=0.1, description=description )
   except Tac.Timeout:
      mode.addError( "Command timed out" )
      return False

   return True


def getUribStatus( vrfName ):
   shmemEm = SharedMem.entityManager( sysdbEm=_entityManager )
   routeInfo = FibUtils.routeStatusInfo( 'reader' )
   fwdInfo = FibUtils.forwardingStatusInfo( 'reader' )

   if vrfName == DEFAULT_VRF:
      routingStatus = shmemEm.doMount( "routing/status",
                                       "Smash::Fib::RouteStatus", routeInfo )
      forwardingStatus = shmemEm.doMount( 'forwarding/status',
                                          "Smash::Fib::ForwardingStatus", fwdInfo )
   else:
      routingStatus = shmemEm.doMount( "routing/vrf/status/" + vrfName,
                                       "Smash::Fib::RouteStatus", routeInfo )
      forwardingStatus = shmemEm.doMount( 'forwarding/vrf/status/' + vrfName,
                                          "Smash::Fib::ForwardingStatus", fwdInfo )

   return ( routingStatus, forwardingStatus )

def doMaybeInitPimBidirStatusSms( vrfName ):
    # pylint: disable-msg=W0602
   global _pimBidirStatusColl

   if vrfName not in _pimBidirStatusColl:
      _pimBidirStatusColl[ vrfName ] = Tac.newInstance(
             "Routing::Pim::Bidir::Status",
             "status-%s" % vrfName )
   pimBidirStatus = _pimBidirStatusColl[ vrfName ]

   pimBidirSmashStatus = PimBidirSmashHelper.mountInDependencyOrder(
      _entityManager, "routing/pim/bidir/status/%s" % vrfName, 'shadow' )

   if not _pimBidirGroupSmashReaderSmColl.get( vrfName ):
      _pimBidirGroupSmashReaderSmColl[ vrfName ] = Tac.newInstance(
            "Routing::Pim::Bidir::Smash::GroupSmashReaderSm",
            pimBidirStatus, pimBidirSmashStatus )
   if not _pimBidirGroupSmashRestartSmColl.get( vrfName ):
      _pimBidirGroupSmashRestartSmColl[ vrfName ] = Tac.newInstance(
            "Routing::Pim::Bidir::Smash::GroupSmashRestartSm",
            pimBidirStatus, pimBidirSmashStatus, 2, Tac.activityManager.clock )

   return pimBidirStatus.initialized

def waitForPimBidirStatusSms( mode, vrfName ):
   try:
      Tac.waitFor( lambda: doMaybeInitPimBidirStatusSms( vrfName ),
            timeout=60, warnAfter=None,
            sleep=not Tac.activityManager.inExecTime.isZero, maxDelay=0.1,
            description=' mroute table to be created' )
   except Tac.Timeout:
      mode.addError( "Command timed out" )
      return False
   return True


#-----------------------------------------------------------------------------------
# show ip pim bidirectional upstream joins [ <groupAddress> ]
#                                          [ neighbor <address> ]
#-----------------------------------------------------------------------------------
def doShowIpPimBidirUpstreamJoins( mode, args ):
   model = PimModelLib.PimUpstreamJoins()
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   af = 'ipv4'
   groupAddr = args.get( 'GROUP_ADDR' )
   neighborAddr = args.get( 'NEIGHBOR_ADDR' )

   if not validVrfName( mode, vrfName, _mfibVrfConfig, af ):
      return

   try:
      group = PimCliLib.ipPimParseGroup( groupAddr )

   except ValueError:
      mode.addError( "Must enter a multicast group" )
      return

   if not waitForPimStatusSms( mode ):
      return

   if neighborAddr is not None:
      neighborAddr = Arnet.IpGenAddr( neighborAddr )

   pimStatus = getPimStatus( vrfName )
   status = getPimUpstreamJoinsBidir( vrfName )

   joinPruneStatus = [ status ]
   model.initialize( pimStatus, joinPruneStatus, None, group, neighborAddr )

   return model

class IpPimBidirUpstreamJoinsCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip pim [ vrf VRF_NAME ] bidirectional upstream joins
               [ GROUP_ADDR ] [ neighbor NEIGHBOR_ADDR ]'''
   data = {
      'ip' : ipMatcherForShow,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'bidirectional' : bidirMatcher,
      'upstream' : upstreamMatcher,
      'joins' : joinsMatcher,
      'GROUP_ADDR' : groupMatcher,
      'neighbor' : neighborsMatcher,
      'NEIGHBOR_ADDR' : joinsNeighborAddrMatcher
   }
   handler = doShowIpPimBidirUpstreamJoins
   cliModel = PimModelLib.PimUpstreamJoins

BasicCli.addShowCommandClass( IpPimBidirUpstreamJoinsCmd )

#---------------------------------------------------------------------------
#show ip mroute bidirectional [ <group> ] [ detail ]
#---------------------------------------------------------------------------
def getIpMrouteBidir( mode, vrfName=DEFAULT_VRF, source=None, group=None,
                        detail=False, submodel=False ):
   if not waitForPimStatusSms( mode ):
      return
   if not waitForPimBidirStatusSms( mode, vrfName ):
      return

   pimBidirStatus = _pimBidirStatusColl[ vrfName ]

   showDetail = bool( detail )

   rpaRpfStatus = SmashLazyMount.mount( _entityManager,
         "routing/pim/bidir/df/rpaRpf/" + vrfName,
         "Routing::Pim::Bidir::Df::Smash::RpaRpfStatus",
         SmashLazyMount.mountInfo( "reader" ) )

   pimBidirRpHash = SmashLazyMount.mount( _entityManager,
                                   "routing/pim/rphash/bidir/" + vrfName,
                                   "Routing::Pim::RpHashStatus",
                                    SmashLazyMount.mountInfo('reader' ) )

   if not group:
      group = Arnet.IpGenAddr("0.0.0.0")

   fd = sys.stdout.fileno()

   fmt = mode.session_.outputFormat()

   showMroute = Tac.newInstance( "PimBidirCliCapi::ShowMroute",
         pimBidirStatus, pimBidirRpHash, rpaRpfStatus )

   showMroute.render( fd, fmt, group, showDetail, submodel )

   return 'modePimBidir', PimModel.Groups

def doShowIpMrouteBidir( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupAddr = args.get( 'GROUP_ADDR' )
   detail = args.get( 'detail' )
   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      return

   try:
      group = PimCliLib.ipPimParseGroup( groupAddr )
   except ValueError:
      mode.addError( "Must enter a multicast group" )
      return

   _, model = getIpMrouteBidir( mode, vrfName, None, group, detail )
   return model

class IpMrouteBidirCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip mroute bidirectional [ vrf VRF_NAME ]
               [ GROUP_ADDR ] [ detail ]'''
   data = {
      'ip' : ipMatcherForShow,
      'mroute' : mrouteMatcher,
      'bidirectional' : bidirMatcher,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'GROUP_ADDR' : groupMatcher,
      'detail' : detailBidirMatcher
   }
   handler = doShowIpMrouteBidir
   cliModel = PimModel.Groups

BasicCli.addShowCommandClass( IpMrouteBidirCmd )

#------------------------------------------------------------------------------
#show ip mroute bidirectional <group> interface [ <interface> ][ <detail> ]
#------------------------------------------------------------------------------
def getIpMrouteIntfBidir( mode, vrfName=DEFAULT_VRF, source=None, group=None,
                           intf=None, detail=False, submodel=False ):

   if not waitForPimStatusSms( mode ):
      return

   if not waitForPimBidirStatusSms( mode, vrfName ):
      return

   pimBidirStatus = _pimBidirStatusColl[ vrfName ]

   showDetail = bool( detail )

   fd = sys.stdout.fileno()

   fmt = mode.session_.outputFormat()

   if not group:
      group = Arnet.IpGenAddr("0.0.0.0")

   if intf:
      interface = str( intf )
   else:
      interface = ""
   showMroute = Tac.newInstance( "PimBidirCliCapi::ShowMrouteInterface",
         pimBidirStatus )
   showMroute.render( fd, fmt, group, interface, showDetail, submodel )

   return 'modePimBidir', PimModel.MrouteInterfaceDetails


def doShowIpMrouteBidirInterface( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupAddr = args.get( 'GROUP_ADDR' )
   intf = args.get( 'INTF' )
   detail = args.get( 'detail' )

   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      return

   try:
      group = PimCliLib.ipPimParseGroup( groupAddr )
   except ValueError:
      mode.addError( "Must enter a multicast group" )
      return

   _, model = getIpMrouteIntfBidir( mode, vrfName, None,
         group, intf, detail )
   return model


class IpMrouteBidirInterfaceCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip mroute bidirectional [ vrf VRF_NAME ] GROUP_ADDR interface
               [ INTF ] [ detail ]'''
   data = {
      'ip' : ipMatcherForShow,
      'mroute' : mrouteMatcher,
      'bidirectional' : bidirMatcher,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'GROUP_ADDR' : groupMatcher,
      'interface' : mrouteInterfaceMatcherForShow,
      'INTF' : IntfCli.Intf.matcher,
      'detail' : detailBidirMatcher
   }
   handler = doShowIpMrouteBidirInterface
   cliModel = PimModel.Groups

BasicCli.addShowCommandClass( IpMrouteBidirInterfaceCmd )

def getIpMrouteCountBidir( mode, vrfName, submodel=False):
   if not waitForPimStatusSms( mode ):
      return
   if not waitForPimBidirStatusSms ( mode, vrfName ):
      return

   pimBidirStatus = _pimBidirStatusColl[ vrfName ]

   pimBidirRpHash = SmashLazyMount.mount( _entityManager,
         "routing/pim/rphash/bidir/" + vrfName,
         "Routing::Pim::RpHashStatus",
         SmashLazyMount.mountInfo('reader') )

   count = Tac.newInstance("PimBidirCliCapi::ShowMrouteCount", pimBidirStatus,
                                    pimBidirRpHash )

   fd = sys.stdout.fileno()
   fmt = mode.session_.outputFormat()
   count.render( fd, fmt, submodel )
   return 'modePimBidir', PimModel.MrouteBidirCount
#-----------------------------------------------------------------------------------
# show ip mroute bidirectional count
#-----------------------------------------------------------------------------------
def doShowIpMrouteBidirCount( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      return
   if not waitForPimStatusSms( mode ):
      return
   if not waitForPimBidirStatusSms ( mode, vrfName ):
      return

   _, model = getIpMrouteCountBidir( mode, vrfName )
   return model

class IpMrouteBidirCountCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip mroute bidirectional [ vrf VRF_NAME ] count'
   data = {
      'ip' : ipMatcherForShow,
      'mroute' : mrouteMatcher,
      'bidirectional' : bidirMatcher,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'count' : mrouteCountMatcher
   }
   handler = doShowIpMrouteBidirCount
   cliModel = PimModel.MrouteBidirCount

BasicCli.addShowCommandClass( IpMrouteBidirCountCmd )

def getPimRpCandidatesBidir( mode, vrfName=DEFAULT_VRF, groupRange=None,
                             detail=None, af=AddressFamily.ipv4 ):
   model = PimModelLib.PimRpCandidates()
   pimBsrRpCandidateStatus = _pimBsrCrpStatusColl.vrfStatus.get( vrfName )
   staticRp = _staticRpColl.vrfStatus.get( vrfName )

   model.generate( staticRp, pimBsrRpCandidateStatus, pimMode='Bidir',
                   groupPrefix=Arnet.IpGenPrefix( groupRange ) \
                      if groupRange else groupRange,
                   detail=True if detail else False )

   return 'modePimBidir', model

#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrf-name> ] bidirectional rp [ groupPrefix ] [ detail ]
#------------------------------------------------------------------------------------
def doShowIpPimBidirRp( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   groupRange = args.get( 'PREFIX' )
   detail = args.get( 'detail' )
   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      print "not valid vrf"
      return
   if groupRange and not isValidPrefixWithError( mode, groupRange ):
      print "not valid prefix"
      return

   _, model = getPimRpCandidatesBidir( mode, vrfName, groupRange, detail )
   return model

class IpPimBidirRpCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] bidirectional rp [ PREFIX ] [ detail ]'
   data = {
      'ip' : ipMatcherForShow,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'bidirectional' : bidirMatcher,
      'rp' : rpMatcher,
      'PREFIX' : groupRangeMatcher,
      'detail' : detailMatcher
   }
   handler = doShowIpPimBidirRp
   cliModel = PimModelLib.PimRpCandidates

BasicCli.addShowCommandClass( IpPimBidirRpCmd )

def getIpPimRpHash( mode, vrfName, rpHash, group, detail=None ):
   model = PimModelLib.PimRpHash()

   rpConfig = _pimBidirRpConfigColl.vrfConfig[ vrfName ]

   if vrfName not in _staticRpComputeObjColl:
      staticRp = _staticRpColl.vrfStatus.get( vrfName )
      if staticRp:
         _staticRpComputeObjColl[ vrfName ] = Tac.newInstance(
                                                "Routing::Pim::RpCompute",
                                                staticRp,
                                                rpConfig )
         _staticTrieBuilderColl[ vrfName ] = \
             Tac.newInstance( "Routing::Pim::GrpPrefixTrieBuilderSm",
                              staticRp,
                              _staticRpComputeObjColl[ vrfName ] )

   if vrfName not in _bsrRpComputeObjColl:
      if not _pimBsrCrpStatusColl.mounted():
         # pylint: disable-msg=W0212
         _pimBsrCrpStatusColl._mount()

      pimBsrRpCandidateStatus = _pimBsrCrpStatusColl.vrfStatus.get( vrfName )

      if pimBsrRpCandidateStatus:
         _bsrRpComputeObjColl[ vrfName ] = \
               Tac.newInstance( "Routing::Pim::RpCompute",
                                pimBsrRpCandidateStatus,
                                rpConfig )
         _bsrTrieBuilderColl[ vrfName ] = \
               Tac.newInstance( "Routing::Pim::GrpPrefixTrieBuilderSm",
                                pimBsrRpCandidateStatus,
                                _bsrRpComputeObjColl[ vrfName ] )

   staticCompute = _staticRpComputeObjColl.get( vrfName )
   bsrCompute = _bsrRpComputeObjColl.get( vrfName )

   rpHashAlgorithm = rpConfig.rpHashAlgorithm

   model.generate( rpHash, staticCompute, bsrCompute, group,
                   rpHashAlgorithm, 'modePimBidir' )

   return model

def getIpPimRpHashBidir( mode, vrfName, group, detail=None ):
   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      return None, None

   pimBidirRpHash = SmashLazyMount.mount( _entityManager,
                                   "routing/pim/rphash/bidir/" + vrfName,
                                   "Routing::Pim::RpHashStatus",
                                    SmashLazyMount.mountInfo( 'reader' ) )
   model = getIpPimRpHash( mode, vrfName, pimBidirRpHash, group, detail )

   return 'modePimBidir', model


#------------------------------------------------------------------------------------
# show ip pim [ vrf <vrf-name> ] bidirectional rp-hash <group> [ detail ]
#------------------------------------------------------------------------------------
def doShowIpPimBidirRpHash( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   group = args.get( 'GROUP_ADDR' )
   detail = args.get( 'detail' )
   _, model = getIpPimRpHashBidir( mode, vrfName, group, detail )
   return model


class IpPimBidirRpHashCmd( ShowCommand.ShowCliCommandClass ):
   syntax = '''show ip pim [ vrf VRF_NAME ] bidirectional rp-hash
            GROUP_ADDR [ detail ]'''
   data = {
      'ip' : ipMatcherForShow,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'bidirectional' : bidirMatcher,
      'rp-hash' : rpHashMatcher,
      'GROUP_ADDR' : groupMatcher,
      'detail' : detailMatcher
   }
   handler = doShowIpPimBidirRpHash
   cliModel = PimModelLib.PimRpHash

BasicCli.addShowCommandClass( IpPimBidirRpHashCmd )

def allowed( vrfName ):
   return PimCliLib.allowed( vrfName, _routingVrfInfoDir, _mfibVrfConfig )

#-----------------------------------------------------------------------------------
# show ip pim config-sanity bidirectional
#----------------------------------------------------------------------------------
def doShowIpPimConfigSanityBidir( mode, args ):
   vrfName = args.get( 'VRF_NAME', VrfCli.vrfMap.getCliSessVrf( mode.session ) )
   if not validVrfName( mode, vrfName, _mfibVrfConfig ):
      print "not valid vrf"
      return

   model = PimConfigCheckModel.PimBidirConfigCheck()

   if not allowed( vrfName ):
      mode.addWarning( "PIM bidirectional is not running for vrf %s." % vrfName )
      return model

   if not waitForPimStatusSms( mode ):
      return

   if not waitForPimBidirStatusSms( mode, vrfName ):
      return

   print "\nBelow are hints of potential PIM Bidirectional mode misconfiguration:\n"

   pimBidirStatus = _pimBidirStatusColl[ vrfName ]

   pimStatus = _pimStatusColl.vrfStatus.get( vrfName )

   pimBidirStaticRpConfig = _pimBidirStaticRpConfigColl.vrfConfig.get( vrfName )

   routingStatus, forwardingStatus = getUribStatus( vrfName )

   return pimBidirConfigCheck( mode, vrfName, _ipConfig, _pimConfigRoot,
                               _aclConfigCli, pimBidirStaticRpConfig,
                               pimBidirStatus, pimStatus, routingStatus,
                               _routingHwStatus, forwardingStatus,
                               _pimGlobalStatus )


class IpPimConfigSanityBidirCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show ip pim [ vrf VRF_NAME ] config-sanity bidirectional'
   data = {
      'ip' : ipMatcherForShow,
      'pim' : pimNodeAfterShowIp,
      'vrf' : vrfMatcher,
      'VRF_NAME' : vrfNameMatcher,
      'config-sanity' : configSanityMatcher,
      'bidirectional' : bidirMatcher
   }
   handler = doShowIpPimConfigSanityBidir
   cliModel = PimConfigCheckModel.PimBidirConfigCheck

BasicCli.addShowCommandClass( IpPimConfigSanityBidirCmd )

PimCliLib.registerShowPimConfigSanityCmdCallback(
         [ 'show ip pim config-sanity bidirectional' ])

def Plugin ( entityManager ):
   global _entityManager
   global _pimGlobalStatus
   global _pimStatusColl
   global _joinPruneStatusColl
   global _staticRpColl
   global _pimBsrCrpStatusColl
   global _aclConfigCli
   global _pimConfigRoot
   global _ipConfig
   global _pimBidirStaticRpConfigColl
   global _pimBidirRpConfigColl
   global _mcastBoundaryConfig
   global _routingHwStatus
   global _routingVrfInfoDir
   global _mfibVrfConfig

   _entityManager = entityManager

   _pimStatusColl = PimLib.getPimStatusColl( AddressFamily.ipv4, "modePimBidir")

   _joinPruneStatusColl = LazyMount.mount( entityManager,
         'routing/pim/bidir/joinprune',
         'Routing::Pim::Packet::JoinPruneStatusColl',
         'r' )

   _staticRpColl = entityManager.mount( 'routing/pim/staticRpStatus/bidir',
                                        'Routing::Pim::RpCandidateStatusColl', 'r' )

   _pimBsrCrpStatusColl = LazyMount.mount( entityManager,
                              'routing/pim/bsr/rpcandidate',
                              'Routing::Pim::RpCandidateStatusColl', 'r' )

   _pimBidirStaticRpConfigColl = LazyMount.mount( entityManager,
                                 'routing/pim/staticRpConfig/bidir/',
                                 'Routing::Pim::Static::RpConfigColl', 'r')
   _pimBidirRpConfigColl = LazyMount.mount( entityManager,
                                 'routing/pim/rpConfig/bidir/',
                                 'Routing::Pim::RpConfigColl', 'r')

   _ipConfig = LazyMount.mount( entityManager, 'ip/config', 'Ip::Config', 'r' )

   _pimConfigRoot = LazyMount.mount( entityManager, 'routing/pim/config',
                                 'Routing::Pim::ConfigColl', 'r' )
   _mcastBoundaryConfig = LazyMount.mount( entityManager,
                                 'routing/mcastboundary/config',
                                 'Routing::McastBoundary::Config', 'r')

   _routingHwStatus = LazyMount.mount( entityManager,
               'routing/hardware/status', 'Routing::Hardware::Status', 'r' )
   _routingVrfInfoDir =  LazyMount.mount( entityManager,
                                          "routing/vrf/routingInfo/status",
                                          "Tac::Dir", "r" )
   _mfibVrfConfig = LazyMount.mount( entityManager, 'routing/multicast/vrf/config',
                                     'Routing::Multicast::Fib::VrfConfig', 'r' )
   _aclConfigCli = LazyMount.mount( entityManager, 'acl/config/cli',
                                   'Acl::Input::Config', 'r' )

   _pimGlobalStatus = PimLib.getPimGlobalStatus( entityManager,
                                                   AddressFamily.ipv4)

   _ = PimLib.getPimStatusModeFilterSm( AddressFamily.ipv4, "modePimBidir" )

   PimCliLib.pimUpstreamJoinsHook.addExtension ( getPimUpstreamJoinsBidir )
   PimCliLib.pimMessageCountersHook.addExtension( getPimMessageCounters )
   PimCliLib.pimShowIpPimRpHook.addExtension(
         ( 'bidirectional', getPimRpCandidatesBidir ) )
   PimCliLib.pimShowIpPimRpHashHook.addExtension( getIpPimRpHashBidir )
   PimCliLib.pimShowIpMrouteHook.addExtension(
         ('bidirectional', getIpMrouteBidir ) )
   PimCliLib.pimShowIpMrouteIntfHook.addExtension(
         ('bidirectional', getIpMrouteIntfBidir ) )
   PimCliLib.pimShowIpMrouteCountHook.addExtension(
         ('bidirectional', getIpMrouteCountBidir ) )
