# Copyright (c) 2006-2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re

import CliMatcher
import CliParser
from CliPlugin.VirtualIntfRule import _VirtualIntfMatcher # old matcher
from CliPlugin.VirtualIntfRule import VirtualIntfMatcherBase # new matcher
import Tac
import Toggles.IntfToggleLib
import Tracing

t0 = Tracing.trace0

entityMibHelper = Tac.newInstance( "EntityMib::Helper" )

dynamicIfIndexEnabled = Toggles.IntfToggleLib.toggleDynamicIfIndexEnabled()
if dynamicIfIndexEnabled:
   SubIntfRangeDetails = Tac.Type( "Interface::SubIntfIdConstants" )
   subLowerBound = SubIntfRangeDetails.subIntfExtendedIdMin
   subUpperBound = SubIntfRangeDetails.subIntfExtendedIdMax
else:
   subLowerBound = CliParser.subLowerBound
   subUpperBound = CliParser.subUpperBound

t0( 'enabled', dynamicIfIndexEnabled, 'bounds', subLowerBound, subUpperBound )

class _IntfMatcherBase( object ):
   """ Base class for interface matching classes """

   def __init__( self ):
      super( _IntfMatcherBase, self ).__init__()
      peer_ = None
      tag_ = None

   def _findSlot( self, chassis, slotNum, tag=None ):
      # find the slot whose label matches slotNum.
      # the slot collection is indexed by relPos, which could be
      # different from slotNum, but is always the same today,
      # so we try to get the slot quickly by directly using slotNum
      # as the key. 
      #
      # When searching for valid slots, it is important to ensure
      # that a slot contains ports of the desired type. Thus if a tag
      # is specified only slots containing ports with that tag are
      # returned. 
      #
      # ie. if the chassis is configured as follows:
      # Slot 1 - 2: Supervisor
      # Slot 3: Empty
      # Slot 4-6: Ethernet Linecards
      #
      # Then, the completions/valid slots will be as follows:
      # >show (...) ethernet ?
      # <3-6>
      # >show (...) management ?
      # <1-3>
      if self.peer_:
         class dummy: 
            pass
         slot = dummy()
         slot.card = None
         return slot
      defaultSlotTags = chassis.defaultChildTags.get( 'CardSlot', {} )
      slot = chassis.cardSlot.get( int( slotNum ) )
      if slot and slot.label == slotNum and \
             slot.tag in defaultSlotTags and \
             self._findPortOfType( slot.card, tag ):
         return slot
      # try to search for it in a for loop
      for slot in chassis.cardSlot.itervalues():
         if slot.tag in defaultSlotTags and slot.label == slotNum and \
                self._findPortOfType( slot.card, tag ):
            return slot
      return None

   def _findPort( self, card, tag, portNum ):
      if self.peer_:
         return portNum
      # Similar to slot, try to find it quickly by indexing PortNum
      if type( portNum ) is int:
         port = card.port.get( portNum )
         if port and port.tag == tag and port.label == portNum:
            return port
      # If none found, try to iterate over slots
      return entityMibHelper.getPortFromCard( card, tag, portNum )

   def _findAllPortLabels( self, card, tag ):
      # Return the port labels as a list of ints, if the port label contains a '/'
      # return the first part of the name
      if self.peer_:
         # We don't have information about peer interfaces, so return dummy values
         # Callers only use max/min of these values so no need to return the range
         return [ 1, 256 ]
      ret = []
      for port in card.port.itervalues():
         if port.tag == tag:
            label = int( port.label.split( '/' )[ 0 ] )
            ret.append( label )
      return ret

   def _moduleStart( self, card, tag, module ):
      modStarts = [ port.label.split('/')[0] for port in card.port.values() 
                    if port.tag == tag and '/' in port.label ]
      return module in modStarts

   def _findModuleLabels( self, card, tag, module ):
      if module.endswith( '/' ):
         return [ int( port.label.split('/')[1] ) for port in card.port.values() 
               if port.tag == tag and port.label.startswith( module ) ]
      return None

class _FixedSystemIntfMatcher( _IntfMatcherBase ):
   """ Helper class to match a path to a port in a fixed system, return
   the string '<interface number>' for that port. 
   If the system has interfaces with lanes (e.g. QSFPs) then this class will
   match on '<interface number>/<lane number>' for interfaces with lanes."""

   # Regular expression for parsing interface path with sub interface.
   portPathRe = re.compile( '([0-9]+\.[0-9]*)?$|[0-9]+/([0-9]+\.[0-9]*)?$' )
   # Regular expression for interface path with no sub interface.
   portPathReNoSub = re.compile( '[0-9]*$|[0-9]+/[0-9]*$' )
   # Regular expression for QSFP interface path matcher with sub
   # interface. This is used by IntfPathMatcher.
   portPathMatchRe = re.compile( '([0-9]+)/([0-9]+\.[0-9]+)$' )
   # Regular expression for QSFP interface path matcher with no sub
   # interface. This is used by IntfPathMatcher.
   portPathMatchReNoSub = re.compile( '([0-9]+)/([0-9]+)$' )

   def __init__( self, tag, peer=False, subIntf=False ):
      super( _FixedSystemIntfMatcher, self ).__init__()
      self.peer_ = peer
      self.tag_ = tag
      self.subIntf_ = subIntf

   def match( self, mode, token, fixedSystem ):
      thePortPathRe = self.portPathRe if self.subIntf_ else self.portPathReNoSub
      if not token or not thePortPathRe.match( token ):
         return None

      if fixedSystem.initStatus != "ok":
         # We don't have all the fixed system information yet
         return None

      port = None
      if self.subIntf_:
         if '.' not in token:
            return None
         ( portToken, subToken ) = token.split( '.' )
         if not subToken or subToken == '':
            return None
         if not subLowerBound <= int( subToken ) <= subUpperBound:
            return None
         port = self._findPort( fixedSystem, self.tag_, portToken )
      else:
         port = self._findPort( fixedSystem, self.tag_, token )

      if port:
         return token
      else:
         return None

   def completions( self, mode, token, fixedSystem ):
      thePortPathRe = self.portPathRe if self.subIntf_ else self.portPathReNoSub
      if not thePortPathRe.match( token ):
         return []

      if fixedSystem.initStatus != "ok":
         # We don't have all the fixed system information yet
         return []

      if not token:
         # Return all port completions
         portLabels = self._findAllPortLabels( fixedSystem, self.tag_ )
         if portLabels and not self.subIntf_:
            return [ CliParser.Completion( '<%d-%d>' %( min( portLabels ),
                                                        max( portLabels ) ),
                                           '%s Port number' % self.tag_,
                                           False ) ]
         else:
            return []
      else:
         ct = []
         ( portToken, subToken ) = token.split( '.' ) if '.' in token else \
             ( token, None )
         # Sub interface number out of range case:
         if subToken and subToken != '':
            assert self.subIntf_
            if not subLowerBound <= int( subToken ) <= subUpperBound:
               return []
         port = self._findPort( fixedSystem, self.tag_, portToken )
         if port:
            subRange = '.<%d-%d>' % ( subLowerBound, subUpperBound ) \
                if self.subIntf_ else ''
            helpName = '%s%s' % ( portToken, subRange )
            helpDesc = '%s %s Number' % ( self.tag_,
                                          'Subinterface' if self.subIntf_ \
                                             else 'Interface' )
            ct += [ CliParser.Completion( helpName, helpDesc, not self.subIntf_ ) ]
         if ( self._moduleStart( fixedSystem, self.tag_, portToken ) and
              not self.subIntf_ ):
            ct += [ CliParser.Completion( '/', '', False ) ]
         moduleLabels = self._findModuleLabels( fixedSystem, self.tag_, portToken )
         if moduleLabels and not self.subIntf_:
            ct += [ CliParser.Completion( '<%d-%d>' %( min( moduleLabels ),
                                                       max( moduleLabels ) ),
                                          '%s Lane number' % self.tag_,
                                          False ) ]
         return ct

class _ModularSystemIntfMatcher( _IntfMatcherBase ):
   """ Helper class to match a path to a port in a modular system, returning the
   string '<module number>/<interface number>' for that port. If no module
   is inserted into a given slot, then any interface from 1-300 is accepted.
   If a module is inserted, then the interface numbers are limited to the ports
   on that specific linecard.
   If the system has interfaces with lanes (e.g. CXPs) then this class will
   match on '<module number>/<interface number>/<lane number>' for interfaces
   with lanes."""

   # Regular expression for interface path matcher with sub interface.
   portPathMatchRe = re.compile( '([0-9]+)(/([0-9]+))(/([0-9]+))?(\.([0-9]+))$' )
   # Regular expression for interface path matcher with no sub interface.
   portPathMatchReNoSub = re.compile( '([0-9]+)(/([0-9]+))(/([0-9]+))?$' )
   # Regular expression for interface path completions with sub interface.
   portPathCompletionRe = re.compile( '([0-9]*)(/([0-9]*))?(/([0-9]*))?' +
                                      '(\.([0-9]*))$' )
   # Regular expression for interface path completions with no sub interface.
   portPathCompletionReNoSub = re.compile( '([0-9]*)(/([0-9]*))?(/([0-9]*))?$' )

   def __init__( self, tag, peer=False, subIntf=False ):
      super( _ModularSystemIntfMatcher, self ).__init__()
      self.peer_ = peer
      self.tag_ = tag
      self.subIntf_ = subIntf

      # Some intf types (or tags) should not match certain slots tags.
      # Here we manually filter them out.
      self.excludedSlotTags = []
      if self.tag_ == "Management":
         self.excludedSlotTags.append( "Linecard" )

      self.noModuleIntfNumMatcher_ = CliMatcher.IntegerMatcher(
         1, 300, helpdesc="%s Port Number" %( self.tag_ ) )
      # max value for an interface lane is taken from the max number of
      # lanes on a CXP xcvr
      self.noModuleIntfLaneMatcher_ = CliMatcher.IntegerMatcher(
         1, 12, helpdesc="%s Number" %( self.tag_ or "Port" ) )

      self.subIntfNumMatcher_ = CliMatcher.IntegerMatcher(
         subLowerBound, subUpperBound, helpdesc="Sub Interface Number" )

   def _validSlotsLabels( self, chassis, tag ):
      if self.peer_:
         return [ 3, 10 ]

      defaultSlotTags = chassis.defaultChildTags.get( 'CardSlot', {} )
      ret = []
      for slot in chassis.cardSlot.itervalues():
         card = slot.card
         # Filter out any card that does not contain matching ports
         portFound = self._portWithTagExists( card, tag )
         if not portFound:
            continue
         # portFound might be true even if the card is not valid.
         # Since we want to only return valid slots, filter them out.
         if not card or card.initStatus != "ok":
            continue
         if card.hasUplink and card.uplinkInitStatus != 'ok':
            continue
         # Finally, filter out undesired tags
         if ( slot.tag not in defaultSlotTags
              or slot.tag in self.excludedSlotTags ):
            continue
         ret.append( int( slot.label ) )
      return ret

   def _portWithTagExists( self, card, tag ):
      if not tag:
         # If tag is not provided, assume port found
         return True

      if ( not card 
           or card.initStatus != "ok"
           or card.powerStatus != 'poweredOn'
           or ( card.hasUplink and ( card.uplinkInitStatus != 'ok'
                                     or card.uplinkPowerStatus != 'poweredOn' ) ) ):
         # For configuration purposes, also accept cards that do not
         # exist or that are not yet inited.
         return True

      for port in card.port.itervalues():
         if port.tag == tag:
            # Port with tag found
            return True

      return False

   def _findSlot( self, chassis, slotNum, tag=None ):
      # Find the slot whose label matches slotNum.
      # When searching for valid slots, it is important to ensure
      # that a slot contains ports of the desired type. Thus if a tag
      # is specified, only slots containing ports with that tag are
      # returned. 
      #
      # e.g. if the chassis is configured as follows:
      # Slot 1 - 2: Supervisor
      # Slot 3: Empty
      # Slot 4-6: Ethernet Linecards
      #
      # Then, the completions/valid slots will be as follows:
      # >show (...) ethernet ?
      # <3-6>
      # >show (...) management ?
      # <1-2>

      if self.peer_:
         class dummy:
            card = None
         return dummy()
      defaultSlotTags = chassis.defaultChildTags.get( 'CardSlot', {} )
      slot = chassis.cardSlot.get( int( slotNum ) )
      # The slot collection is indexed by relPos, which could be
      # different from slotNum, but is always the same today,
      # so we try to get the slot quickly by directly using slotNum
      # as the key.
      if ( slot
           and self._portWithTagExists( slot.card, tag )
           and slot.label == slotNum
           and slot.tag in defaultSlotTags
           and slot.tag not in self.excludedSlotTags ):
         return slot
      # If none found, try to iterate over slots
      for slot in chassis.cardSlot.itervalues():
         portFound = self._portWithTagExists( slot.card, tag )
         if ( portFound
              and slot.label == slotNum
              and slot.tag not in self.excludedSlotTags
              and slot.tag in defaultSlotTags ):
            return slot
      return None

   def _getCurrentSupeSlotId( self ):
      # Case 1: SlotId is in the environment.
      slotId = os.environ.get( 'SLOTID' )
      if slotId:
         return slotId

      # Case 2: SlotId is in /etc/slotid.
      try:
         with open( '/etc/slotid' ) as f:
            return int( f.readline() )
      except IOError as e:
         t0( "Failed to open /etc/slotid. Error: %s" % e.strerror )
      except ValueError as e:
         t0( "Failed to read slotid from /etc/slotid. Error: %s" % e.strerror )

      # Case 3: Default slotId.
      return 1

   def match( self, mode, token, chassis ):
      thePortPathMatchRe = self.portPathMatchRe if self.subIntf_ else \
          self.portPathMatchReNoSub
      m = thePortPathMatchRe.match( token )
      if not m:
         return None

      slotNum = m.group( 1 )
      portNum = m.group( 3 )
      laneToken = m.group( 4 ) or '' # includes leading '/'
      laneNum = m.group( 5 )
      subNum = m.group( 7 ) if self.subIntf_ else None

      cardNum = slotNum
      # Match the slot
      slot = self._findSlot( chassis, slotNum )
      if not slot:
         return None
      card = slot.card

      # Match the port
      if ( card is None 
           or card.initStatus != "ok" 
           or card.powerStatus != 'poweredOn'
           or ( card.hasUplink and ( card.uplinkInitStatus != 'ok'
                                     or card.uplinkPowerStatus != 'poweredOn' ) ) ):
         # No module is inserted, or the module isn't fully
         # initialized. Accept default range
         if( self.noModuleIntfNumMatcher_.match( mode, None, portNum )
               is CliMatcher.noMatch ):
            return None
         if ( laneNum is not None
              and self.noModuleIntfLaneMatcher_.match( mode, None, laneNum )
              is CliMatcher.noMatch ):
            return None
      else:
         portNumStr = '%s%s' % ( portNum, laneToken )
         # A module is inserted and initialized
         port = self._findPort( card, self.tag_, portNumStr )

         if not port:
            # If this is the management port of the standby supervisor
            # which wasn't found, return token instead of None to display
            # "Interface does not exist." instead of "Invalid Input.".
            # This handles the case for platforms where the card entityMib
            # of the standby supervisor is created but the management
            # ports may not have been created if the redundancy protocol
            # is set to simplex.
            if ( slot.tag == 'Supervisor' and
                 slot.label != self._getCurrentSupeSlotId() and
                 self.tag_ == 'Management' ):
               return token
            return None

      if( self.subIntf_ and self.subIntfNumMatcher_.match( mode, None, subNum ) is
            CliMatcher.noMatch ):
         return None
            
      # Match succeeded
      return token

   def completions( self, mode, token, chassis ):
      thePortPathCompletionRe = self.portPathCompletionRe if self.subIntf_ else \
          self.portPathCompletionReNoSub
      m = thePortPathCompletionRe.match( token )
      if not m:
         return []

      slotStr = m.group( 1 )
      portToken = m.group( 2 ) or '' # includes leading '/'
      portStr = m.group( 3 )
      laneToken = m.group( 4 ) or '' # includes leading '/'
      laneStr = m.group( 5 )
      subToken = m.group( 6 ) or '' if self.subIntf_ else ''
      subStr = m.group( 7 ) if self.subIntf_ else None

      # Make sure match does not have any missing intermediate strings before
      # slashes ('/'); for example, 3//2 is not a valid match
      if ( ( slotStr == '' and
             ( portToken.startswith( '/' ) or laneToken.startswith( '/' ) ) ) or
           ( portStr == '' and laneToken.startswith( '/' ) ) ):
         return []

      # Make sure that sub interface without valid slot/port/lane number is
      # not a valid match.
      if ( subToken.startswith( '.' ) and
           ( ( slotStr == '' or portStr == '' ) or
             ( laneToken.startswith( '/' ) and laneStr == '' ) ) ):
         return []
      
      # Sub interface number out of range case:
      if subStr and subStr != '':
         assert self.subIntf_
         if not subLowerBound <= int( subStr ) <= subUpperBound:
            return []
      subRange = '.<%d-%d>' % ( subLowerBound, subUpperBound ) \
          if self.subIntf_ else ''

      if m.group( 2 ) is None:
         # We're returning slot completions
         if not slotStr:
            # Return completions only for valid slots
            slotLabels = self._validSlotsLabels( chassis, self.tag_ )
            if slotLabels and not self.subIntf_:
               # We set partial=True in order to prevent the basename from being 
               # added to the completion (we get "<1-48>" instead of "Ethernet<1-48>"
               return [ CliParser.Completion( '<%d-%d>' %( min( slotLabels ),
                                                           max( slotLabels ) ),
                                              'Slot number', literal=False, 
                                              partial=True ) ]
            else:
               return []
         else:
            slot = self._findSlot( chassis, slotStr, self.tag_ )
            if slot and not self.subIntf_:
               # We've matched a slot exactly. Return a completion for a '/'
               # We set partial=True in order to prevent the basename from being
               # added to the completion (ie. we get "/" instead of "Ethernet/"
               return [ CliParser.Completion( '/', '', literal=False,
                                              partial=True ) ]
            else:
               return []
      else:
         # We're returning a port or lane completion
         # Of note is the fact that if the interface type and slot were typed
         # as a single word without a seperating space (ie. Ethernet3), then
         # the completion returned will be appended to the basename, in this
         # case, Ethernet. The following completions are confusing:
         # > show (...) Ethernet3 ?
         #   Ethernet/
         # > show (...) Ethernet3/ ?
         #   Ethernet<1-48>
         # So we set partial to true to ensure that the completions are not 
         # appended to the basenames in these cases, so we see:
         # > show (...) Ethernet3 ?
         #   /
         # > show (...) Ethernet3/ ?
         #   <1-48> 
         slot = slotStr and self._findSlot( chassis, slotStr ) or None
         if not slot:
            return []

         card = slot.card
         if ( card is None
              or card.initStatus != "ok"
              or card.powerStatus != 'poweredOn'
              or ( card.hasUplink and ( card.uplinkInitStatus != 'ok'
                                        or card.uplinkPowerStatus != 'poweredOn' ) )
            ):
            # no module is inserted or the module is not fully initialized.
            # Return the default.
            if self.subIntf_:
               # Subinterface completions.
               if not subToken.startswith( '.' ):
                  return []
               if( self.noModuleIntfNumMatcher_.match( mode, None, portStr )
                     is CliMatcher.noMatch ):
                  return []
               if ( laneStr and laneStr != ''
                    and self.noModuleIntfLaneMatcher_.match( mode, None, laneStr )
                     is CliMatcher.noMatch ):
                  return []
               helpName = '%s%s' % ( token.split( '.' )[0], subRange )
               helpDesc = '%s Subinterface number' % self.tag_
               return [ CliParser.Completion( helpName, helpDesc, False ) ]
            else:
               if laneToken:
                  # Lane completion
                  return self.noModuleIntfLaneMatcher_.completions( mode, None,
                        laneStr )
               else:
                  # Port completion
                  completions = self.noModuleIntfNumMatcher_.completions( mode, None,
                                                                          portStr )
                  for c in completions:
                     if portStr:
                        # No need to set partial to True since this is
                        # a valid port In this case there will only be
                        # one element in completions, so there is no
                        # issue with setting all completions to the
                        # same name
                        c.name = '%s/%s' % ( slotStr, portStr )
                     else:
                        # Set partial=True to get "<1-48>" instead of
                        # "Ethernet<1-48>"
                        c.partial = True
                  return completions

         if not portStr:
            # Return completions for all ports
            portLabels = self._findAllPortLabels( card, self.tag_ )
            if portLabels and not self.subIntf_:
               # We set partial=True in order to prevent the basename from being 
               # added to the completion (we get "<1-48>" instead of "Ethernet<1-48>"
               return [ CliParser.Completion( '<%d-%d>' % ( min( portLabels ),
                                                            max( portLabels ) ),
                                              '%s Port number' % self.tag_,
                                              literal=False, partial=True ) ]
            else:
               return []
         else:
            ct = []
            portLaneToken = '%s%s' % ( portStr, laneToken )
            port = self._findPort( card, self.tag_, portLaneToken )
            if port:
               helpName = '%s%s' % ( token.split( '.' )[0], subRange )
               helpDesc = '%s %s number' % ( self.tag_,
                                             'Subinterface' if self.subIntf_ \
                                                else 'Port' )
               ct += [ CliParser.Completion( helpName, helpDesc,
                                             not self.subIntf_ ) ]
            if ( self._moduleStart( card, self.tag_, portLaneToken ) and
                 not self.subIntf_ ):
               ct += [ CliParser.Completion( '/', '', False ) ]
            moduleLabels = self._findModuleLabels( card, self.tag_, portLaneToken )
            if moduleLabels and not self.subIntf_:
               ct += [ CliParser.Completion( '<%d-%d>' % ( min( moduleLabels ),
                                                           max( moduleLabels ) ),
                                             '%s Lane number' % self.tag_,
                                             False ) ]
            return ct

class IntfPathMatcher( object ):

   def __init__( self, tag, peer=False, subIntf=False ):
      self.peer_ = peer
      self.tag_ = tag
      self.subIntf_ = subIntf

      self.modularSystemIntfMatcher_ = _ModularSystemIntfMatcher( tag, peer=peer,
                                                                  subIntf=subIntf )
      self.fixedSystemIntfMatcher_ = _FixedSystemIntfMatcher( tag, peer=peer,
                                                              subIntf=subIntf )

      self.noEntityMibRootSlotNumMatcher_ = CliMatcher.IntegerMatcher(
         1, 32, helpdesc="Module Number" )
      self.noModuleIntfNumMatcherNoSub_ = CliMatcher.IntegerMatcher(
         1, 300, helpdesc="%s Number" %( self.tag_ or "Port" ) )
      self.noModuleIntfNumMatcher_ = CliMatcher.DottedIntegerMatcher(
         1, 300, helpname=None, helpdesc="%s Number" %( self.tag_ or "Port" ),
         subLbound=subLowerBound, subUbound=subUpperBound ) \
         if subIntf else self.noModuleIntfNumMatcherNoSub_
      # max value for an interface lane is taken from the max number of
      # lanes on a MBO xcvr
      self.noModuleIntfLaneMatcher_ = CliMatcher.DottedIntegerMatcher(
         1, 12, helpname=None, helpdesc="%s Number" %( self.tag_ or "Port" ),
         subLbound=subLowerBound, subUbound=subUpperBound ) \
         if subIntf else CliMatcher.IntegerMatcher(
            1, 12, helpdesc="%s Number" %( self.tag_ or "Port" ) )

   def _entityMibRoot( self, mode ):
      return mode.entityManager.lookup( 'hardware/entmib' ).root

   def match( self, mode, context, token ):
      # To match an interface path, we first match the slot, assuming that
      # the first component of the path refers to the slot
      t0( "path matcher", token, self.subIntf_, self.peer_ )
      entityMibRoot = self._entityMibRoot( mode )

      if ( entityMibRoot is None or
           entityMibRoot.initStatus != "ok" or
           self.peer_ ):
         # Handle the case that Fru has not started yet or that
         # the entity mib root has not yet fully been created.
         # The matches for fixed system module with lanes, and for modular
         # system without lanes can overlap (for example with a token of "2/4"),
         # but this is the best we can do prior to the mib root being created.
         # Attempt to match a fixed system path
         t0( "match", self.noModuleIntfNumMatcher_, self.subIntf_ )
         if( self.noModuleIntfNumMatcher_.match( mode, context, token )
               is not CliMatcher.noMatch ):
            return CliMatcher.MatchResult( token, token )

         m = None
         # Attempt to match to a fixed system module (e.g. QSFP), that is, with
         # lanes
         if self.subIntf_:
            m = self.fixedSystemIntfMatcher_.portPathMatchRe.match( token )
         else:
            m = self.fixedSystemIntfMatcher_.portPathMatchReNoSub.match( token )
         if ( ( m is not None ) and \
              ( self.noModuleIntfNumMatcherNoSub_.match( mode, context,
                 m.group( 1 ) ) is not CliMatcher.noMatch ) and \
              ( self.noModuleIntfLaneMatcher_.match( mode, context, m.group( 2 ) )
                is not CliMatcher.noMatch ) ):
            return CliMatcher.MatchResult( token, token )

         # Attempt to match a modular system path, either with a plain interface
         # or with an interface module (e.g. CXP)
         if self.subIntf_:
            m = self.modularSystemIntfMatcher_.portPathMatchRe.match( token )
         else:
            m = self.modularSystemIntfMatcher_.portPathMatchReNoSub.match( token )
         if ( ( m is not None ) and \
              ( self.noEntityMibRootSlotNumMatcher_.match( mode, context,
                 m.group( 1 ) ) is not CliMatcher.noMatch ) ):
            if self.subIntf_:
               # With lane specification
               if ( ( m.group( 5 ) ) and \
                    ( self.noModuleIntfNumMatcherNoSub_.match( mode, context,
                       m.group( 3 ) ) is not CliMatcher.noMatch ) and \
                    ( self.noModuleIntfLaneMatcher_.match( mode, context,
                         ''.join( g for g in m.group( 5, 6 ) if g ) )
                      is not CliMatcher.noMatch ) ):
                  return CliMatcher.MatchResult( token, token )
               # Without lane specification
               if ( ( m.group( 5 ) is None ) and \
                    ( self.noModuleIntfNumMatcher_.match( mode, context,
                         ''.join( g for g in m.group( 3, 6 ) if g ) )
                      is not CliMatcher.noMatch ) ):
                  return CliMatcher.MatchResult( token, token )
            else:
               if ( ( self.noModuleIntfNumMatcher_.match( mode, context,
                  m.group( 3 ) )
                      is not CliMatcher.noMatch ) and \
                    ( m.group( 5 ) is None or \
                         self.noModuleIntfLaneMatcher_.match( mode, context,
                            m.group( 5 ) ) is not CliMatcher.noMatch ) ):
                  return CliMatcher.MatchResult( token, token )
         return CliMatcher.noMatch

      # The entity mib root has been created

      if entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
         result = self.modularSystemIntfMatcher_.match( mode, token, entityMibRoot )
         if result is None:
            return CliMatcher.noMatch
         return CliMatcher.MatchResult( result, result )
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
         result = self.fixedSystemIntfMatcher_.match( mode, token, entityMibRoot )
         if result is None:
            return CliMatcher.noMatch
         return CliMatcher.MatchResult( result, result )
      else:
         raise Tac.InternalException( "Unknown entity mib root class %s"
                                      % entityMibRoot.tacType.fullTypeName )
         
   def completions( self, mode, context, token ):
      entityMibRoot = self._entityMibRoot( mode )

      if ( not entityMibRoot or
           not entityMibRoot.tacType or
           not entityMibRoot.tacType.fullTypeName ):
         # No completions are returned if the entity mib is not populated
         # with chassis and slot information.
         return []
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::Chassis":
         return self.modularSystemIntfMatcher_.completions( mode, token,
                                                            entityMibRoot )
      elif entityMibRoot.tacType.fullTypeName == "EntityMib::FixedSystem":
         return self.fixedSystemIntfMatcher_.completions( mode, token,
                                                          entityMibRoot )
      else:
         raise Tac.InternalException( "Unknown entity mib root class %s"
                                      % entityMibRoot.tacType.fullTypeName )
         
#-------------------------------------------------------------------------------
# Type of matcher that matches an interface name without whitespace, and
# evaluates to the canonical interface name.
#-------------------------------------------------------------------------------
class _PhysicalIntfMatcher( _VirtualIntfMatcher ):
   def __init__( self, basename, matcherIntfTag=None, helpdesc=None, peer=False,
                 subIntf=False, subIntfGuard=None, pathMatcher=None,
                 alternates=None ):
      if matcherIntfTag is None:
         matcherIntfTag = basename

      if pathMatcher is None:
         pathMatcher = IntfPathMatcher( tag=matcherIntfTag, peer=peer,
                                        subIntf=subIntf )
      _VirtualIntfMatcher.__init__( self, basename, pathMatcher,
                                    subIntf=subIntf, subIntfGuard=subIntfGuard,
                                    alternates=alternates )

   def __str__( self ):
      # This is just for 'show cli command' to show the right name as it cannot
      # use the __str__() function of the superclass automatically.
      return _VirtualIntfMatcher.__str__( self )

# New parser
class PhysicalIntfMatcher( VirtualIntfMatcherBase ):
   def __init__( self, basename, matcherIntfTag=None, helpdesc=None, alternates=None,
                 peer=False, subIntf=False, subIntfGuard=None, guard=None,
                 **kwargs ):
      if matcherIntfTag is None:
         matcherIntfTag = basename

      t0( "PhysicalIntfMatcher", basename, "tag", matcherIntfTag,
          "subIntf", subIntf )
      intfNameMatcher = _PhysicalIntfMatcher( basename, matcherIntfTag,
                                              peer=peer,
                                              subIntf=subIntf,
                                              subIntfGuard=subIntfGuard,
                                              alternates=alternates )
      intfPathMatcher = IntfPathMatcher( tag=matcherIntfTag, peer=peer,
                                         subIntf=subIntf )
      VirtualIntfMatcherBase.__init__( self, basename, intfNameMatcher,
                                       intfPathMatcher,
                                       helpdesc=helpdesc,
                                       alternates=alternates,
                                       guard=guard, subIntf=subIntf,
                                       **kwargs )

