# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Arnet
import BasicCli
import CliCommand
import CliMatcher
import CliParser
# pylint: disable-msg=ungrouped-imports
import CliPlugin.ControllerdbLib
import CliPlugin.IpAddrMatcher as IpAddrMatcher
import CliPlugin.IpGenAddrMatcher as IpGenAddrMatcher
import ConfigMount
import Plugins
import ShowCommand
import Tac
import Toggles.ControllerAclToggleLib
from CliMode.Pcs import PcsServiceMode
from CliPlugin.ControllerCli import ( CvxConfigMode, addCvxShutdownCallback,
                                     addNoCvxCallback,
                                     serviceAfterShowKwMatcher,
                                     serviceKwMatcher )
from CliPlugin.ControllerdbLib import switchIdCache
from PcsModel import PolicySwitchErrorMsg
from PcsModel import RuleCounters
from PcsModel import PolicyCounters
from PcsModel import PolicyCountersList
from ReversibleSecretCli import reversibleSecretCliExpression

import LazyMount
from PcsModel import ( ControllerHost, ControllerNsGroup, ControllerNsGroupList,
                      ControllerPolicy, ControllerPolicyList, ControllerRule,
                      IntfInfo, PcsApiState, PcsStatus, PcsTagInfo,
                      PolicySectionStatus, PolicySectionSwitchStatus,
                      PolicyStatus, SwitchAclStatus, SwitchInfo, TagInfo )

config = None
status = None
apiStatus = None
controllerConfig = None
httpServiceConfig = None
serviceInputConfigDir = None
endpointDb = None
sectionDir = None
policyData = None
switchAclConfig = None
switchAclStatus = None
groupDir = None
policyStatusDir = None

SwitchAclStatusType = Tac.Type( "ControllerAcl::AclStatusV1" )

def addToShowCommandClass( cmdClass ):
   if Toggles.ControllerAclToggleLib.toggleControllerAclEnabled():
      BasicCli.addShowCommandClass( cmdClass )

def cleanName( name ):
   # Strip the prefix, if present. Currently, for section name it'll be
   # in the form 'default.<sectionUuid>' and for a rule it'll be as
   # 'default.<sectionUuid>.<ruleUuid>'. This cleanup can be dropped
   # after Vmware fixes their display_name issue.
   return name.split( '.' )[ -1 ]

def policyCompletor( mode ):
   return [ ( cleanName( s.displayName ) or s.id )
            for s in sectionDir.section.itervalues() ]

matcherController = CliMatcher.KeywordMatcher( 'controller',
      helpdesc='Show policy received from NSX controller' )
matcherPcs = CliMatcher.KeywordMatcher( 'pcs', helpdesc='Policy Control Service' )
matcherPolicy = CliMatcher.KeywordMatcher( 'policy',
      helpdesc='Display policy information' )
# XXX : This doesn't work for a policy display name with spaces in it
matcherPolicyName = CliMatcher.DynamicNameMatcher( policyCompletor, 'Policy name',
      pattern=r'[A-Za-z0-9_\.-]+' )

switchAclStatusType = Tac.Type( "ControllerAcl::AclStatusV1" )

CVX_SERVICE_NAME = "Pcs"
HTTP_SERVICE_NAME = "pcs"
GROUP_PREFIX = '/infra/domains/default/groups/'

class PcsServiceConfigMode( PcsServiceMode, BasicCli.ConfigModeBase ):
   name = "PCS service configuration"
   modeParseTree = CliParser.ModeParseTree()

   def __init__( self, parent, session, name=None ):
      PcsServiceMode.__init__( self, name )
      BasicCli.ConfigModeBase.__init__( self, parent, session )

   def enableService( self, enable ):
      config.enabled = enable
      httpServiceConfig.service[ HTTP_SERVICE_NAME ].enabled = enable
      serviceInputConfigDir.service[ CVX_SERVICE_NAME ].enabled = enable
      if enable and not controllerConfig.enabled:
         self.addWarning( "PCS service will not start when CVX is shutdown" )

   def setController( self, args ):
      ipAddr = args[ 'IPADDR' ]
      port = args.get( 'PORT' )
      if ipAddr == '0.0.0.0' or IpAddrMatcher.isReservedIpAddr( ipAddr ):
         self.addWarning( "Invalid IP Address" )
         return
      # No further validation is required as Cli tokens take care of it
      config.controllerIp = Arnet.IpAddr( ipAddr )
      config.port = int( port ) if port else config.defaultPort

   def deleteController( self, args ):
      config.controllerIp = Arnet.IpAddr( '0.0.0.0' )
      config.port = config.defaultPort

#-----------------------------------------
# (config-cvx)# service pcs
# (config-cvx-acl)#
#-----------------------------------------
def enterPcsServiceCfgMode( mode, args ):
   childMode = mode.childMode( PcsServiceConfigMode )
   mode.session_.gotoChildMode( childMode )

class ServicePcsCmd( CliCommand.CliCommandClass ):
   syntax = 'service pcs'
   data = {
      'service': serviceKwMatcher,
      'pcs': 'Configure Policy Control Service',
   }
   handler = enterPcsServiceCfgMode

if Toggles.ControllerAclToggleLib.toggleControllerAclEnabled():
   CvxConfigMode.addCommandClass( ServicePcsCmd )

#--------------------------------------------------------------------------------
# [ no | default ] shutdown
#--------------------------------------------------------------------------------
class ShutdownCmd( CliCommand.CliCommandClass ):
   syntax = 'shutdown'
   noOrDefaultSyntax = syntax
   data = {
      'shutdown': 'Shutdown Policy Control service',
   }

   handler = lambda mode, args: mode.enableService( False )
   noHandler = lambda mode, args: mode.enableService( True )
   defaultHandler = handler

PcsServiceConfigMode.addCommandClass( ShutdownCmd )

#--------------------------------------------------------------------------------
# [ no | default ] controller IPADDR [ PORT ]
#--------------------------------------------------------------------------------
class ControllerCmd( CliCommand.CliCommandClass ):
   syntax = 'controller IPADDR [ PORT ]'
   noOrDefaultSyntax = 'controller ...'
   data = {
      'controller': 'Configure policy-service controller',
      'IPADDR': IpAddrMatcher.IpAddrMatcher( helpdesc='IP address' ),
      'PORT': CliMatcher.IntegerMatcher( 1, 65535,
         helpdesc='Port (default is 443)' ),
   }
   handler = PcsServiceConfigMode.setController
   noOrDefaultHandler = PcsServiceConfigMode.deleteController

PcsServiceConfigMode.addCommandClass( ControllerCmd )

#--------------------------------------------------------------------------------
# [ no | default ] username UNAME
#--------------------------------------------------------------------------------
def username( mode, args ):
   config.username = args[ 'UNAME' ]

def noUsername( mode, args ):
   config.username = ""

class UsernameCmd( CliCommand.CliCommandClass ):
   syntax = 'username UNAME'
   noOrDefaultSyntax = 'username ...'
   data = {
      'username': 'Configure username for controller',
      'UNAME': CliMatcher.QuotedStringMatcher( helpdesc='Service Account Username',
         helpname='name' ),
   }
   handler = username
   noOrDefaultHandler = noUsername

PcsServiceConfigMode.addCommandClass( UsernameCmd )

#-------------------------------------------------------------------------------
# [no] password <secret>
#-------------------------------------------------------------------------------
class PasswordCommand( CliCommand.CliCommandClass ):
   syntax = "password SECRET"
   noOrDefaultSyntax = "password ..."
   data = {
      "password" : "Configure password for controller",
      "SECRET": reversibleSecretCliExpression( "SECRET" )
      }
   @staticmethod
   def handler( mode, args ):
      config.password = args[ "SECRET" ]

   @staticmethod
   def noOrDefaultHandler( mode, args ):
      config.password = ""

PcsServiceConfigMode.addCommandClass( PasswordCommand )

#--------------------------------------------------------------------------------
# [ no | default ] enforcement-point EPNAME
#--------------------------------------------------------------------------------
def endpointName( mode, args ):
   config.endpointName = args[ 'EPNAME' ]

def noEndpointName( mode, args ):
   config.endpointName = ""

class EndpointNameCmd( CliCommand.CliCommandClass ):
   syntax = 'enforcement-point EPNAME'
   noOrDefaultSyntax = 'enforcement-point ...'
   data = {
      'enforcement-point': 'Name identifying enforcement point',
      'EPNAME': CliMatcher.QuotedStringMatcher(
         helpdesc='Name of the enforcement point', helpname='name' ),
   }
   handler = endpointName
   noOrDefaultHandler = noEndpointName

PcsServiceConfigMode.addCommandClass( EndpointNameCmd )

#--------------------------------------------------------------------------------
# [ no | default ] pinned-public-key TP
#--------------------------------------------------------------------------------
def thumbprint( mode, args ):
   config.thumbprint = args[ 'TP' ]

def noThumbprint( mode, args ):
   config.thumbprint = ''

class ThumbprintCmd( CliCommand.CliCommandClass ):
   syntax = 'pinned-public-key TP'
   noOrDefaultSyntax = 'pinned-public-key ...'
   data = {
      'pinned-public-key': ( 'BASE64 PEM/DER encoded SHA256 of controller\'s '
                             'public key' ),
      'TP': CliMatcher.QuotedStringMatcher( helpdesc='Controller public key',
         helpname='IDENTITY' ),
   }
   handler = thumbprint
   noOrDefaultHandler = noThumbprint

PcsServiceConfigMode.addCommandClass( ThumbprintCmd )

#--------------------------------------------------------------------------------
# [ no | default ] notification-id NOTIF_ID
#--------------------------------------------------------------------------------
def setNotificationId( mode, args ):
   config.notificationUuid = args[ 'NOTIF_ID' ]

def noNotificationId( mode, args ):
   config.notificationUuid = ""

class NotificationIdCmd( CliCommand.CliCommandClass ):
   syntax = 'notification-id NOTIF_ID'
   noOrDefaultSyntax = 'notification-id ...'
   data = {
      'notification-id': ( 'UUID for notification watcher registration with '
                           'controller' ),
      'NOTIF_ID': CliMatcher.QuotedStringMatcher( helpdesc='Notification ID',
         helpname='UUID' ),
   }
   handler = setNotificationId
   noOrDefaultHandler = noNotificationId

PcsServiceConfigMode.addCommandClass( NotificationIdCmd )

#------------------------------------------------------------------------------------
# show service pcs status
#------------------------------------------------------------------------------------
def showStatus( mode, args ):
   pcsStatus = PcsStatus()
   enabled = ( status.enabled and apiStatus.enabled )
   pcsStatus.status = ( "running" if enabled else "notRunning" )
   pcsStatus.policyCount = len( sectionDir.section )
   pcsStatus.tagCount = len( endpointDb.tag )

   for apiType, apiState in apiStatus.perApiState.items():
      if not apiState.method:
         continue
      pcsStatus.apiState[ apiType ] = PcsApiState()
      pcsStatus.apiState[ apiType ].method = apiState.method.lower()
      pcsStatus.apiState[ apiType ].uri = apiState.uri
      pcsStatus.apiState[ apiType ].success = apiState.success
      pcsStatus.apiState[ apiType ].timestamp = apiState.utcTimestamp
      if apiState.detailString:
         pcsStatus.apiState[ apiType ].errorDetail = apiState.detailString

   return pcsStatus

class ServicePcsStatusCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs status'
   data = {
      'service': serviceAfterShowKwMatcher,
      'pcs': matcherPcs,
      'status': 'Policy Control Service status',
   }
   handler = showStatus
   cliModel = PcsStatus

addToShowCommandClass( ServicePcsStatusCmd )

#--------------------------------------------------------------------------------
# show service pcs policy [ POLICY ]
#--------------------------------------------------------------------------------
def policyStateToModel( state ):
   state = state.split( 'prs' )[ 1 ].lower()
   if state == 'inprogress':
      return 'pending'
   elif state == 'failure':
      return 'fail'
   else:
      return state

def getSection( name ):
   if name in sectionDir.section:
      return [ sectionDir.section[ name ] ]

   # its a display name and not policy ID.
   sections = []
   for s in sectionDir.section.itervalues():
      if s.displayName == name or cleanName( s.displayName ) == name:
         sections.append( s )
   return sections

def showPolicySection( policySection, displayName ):
   sectionModel = PolicySectionStatus()
   sectionModel.sectionId = policySection.sectionId
   sectionModel.sectionName = cleanName( displayName )
   sectionModel.status = 'unknown'
   policyStatus = policyStatusDir.pcs.get( policySection.sectionId )
   if policyStatus:
      # PolicyStatus should exist because both sectionDir and policyData
      # is accessed for it. Can't add assert here because it's CLI
      sectionModel.status = policyStateToModel( policyStatus.state )
   switchData = {}
   for ruleMap in policySection.ruleIdMap.itervalues():
      for switchMap in ruleMap.ruleToSwitchMap.itervalues():
         for aclCfg in switchMap.aclBySwitchId.itervalues():
            switch = switchIdCache.getHost( aclCfg.switchId ) or aclCfg.switchId
            if switch not in switchData:
               switchData[ switch ] = []
            switchData[ switch ].extend( aclCfg.acl.keys() )
   # Switch status codes that don't directly map to CLI codes
   statusMap = {
      'waitingForHw' : 'pending',
      'errUnsupportedRules' : 'fail',
      'errHw' : 'fail',
   }
   for switch in switchData:
      switchModel = PolicySectionSwitchStatus( switchId=switch )
      switchId = switchIdCache.getSwitch( switch )
      switchStatus = None
      if switchId and switchAclStatus:
         switchStatus = switchAclStatus.get( switchId )
      for acl in set( switchData[ switch ] ):
         aclStatus = 'unknown'
         if isinstance( switchStatus, SwitchAclStatusType ):
            aclStatus = 'pending'
            for intfSts in switchStatus.intfStatus.itervalues():
               for aclApplyStatus in intfSts.applyStatus.itervalues():
                  if aclApplyStatus.acl == acl:
                     aclStatus = statusMap.get( aclApplyStatus.status,
                                                aclApplyStatus.status )
         switchModel.acl.append(
               SwitchAclStatus( aclName=acl, status=aclStatus ) )
      sectionModel.switch.append( switchModel )
   return sectionModel

def showPolicy( mode, args ):
   policyName = args.get( 'POLICY' )
   policy = PolicyStatus()
   if policyName:
      sections = getSection( policyName )
      for sec in sorted( sections, key=lambda x: x.priority ):
         policySection = policyData.section.get( sec.id )
         if policySection:
            policy.section.append(
                  showPolicySection( policySection, sec.displayName ) )
   else:
      # Sort based on priority to make the order of sections displayed to be
      # consistent with NSX UI
      sectionByPriority = [ ( s.id, s.priority )
                            for s in sectionDir.section.itervalues() ]
      for sid, _ in sorted( sectionByPriority, key=lambda x: x[ 1 ] ):
         policySection = policyData.section.get( sid )
         if policySection:
            sections = getSection( policySection.sectionId )
            for sec in sections:
               policy.section.append(
                     showPolicySection( policySection, sec.displayName ) )
   return policy

class ServicePcsPolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs policy [ POLICY ]'
   data = {
      'service': serviceAfterShowKwMatcher,
      'pcs': matcherPcs,
      'policy': matcherPolicy,
      'POLICY': matcherPolicyName,
   }
   handler = showPolicy
   cliModel = PolicyStatus

addToShowCommandClass( ServicePcsPolicyCmd )

#--------------------------------------------------------------------------------
# show service pcs policy [ POLICY ] counters
#--------------------------------------------------------------------------------

def buildPolicyCounters( section, policy ):
   policyCounters = PolicyCounters()
   policyCounters.policyId = section.id
   policyCounters.policyName = cleanName( section.displayName )
   policyCounters.status = policyStateToModel( policy.state )
   for r in sorted( policy.prs, key=lambda x: section.rule[ x ].priority if
                                              x in section.rule else x ):
      ruleStatus = policy.prs[ r ]
      rule = section.rule.get( r )
      if not rule:
         continue
      ruleCounters = RuleCounters()
      ruleCounters.ruleId = r
      ruleCounters.ruleName = cleanName( rule.displayName )
      ruleCounters.packetCount = ruleStatus.hitCount
      for sw, msg in ruleStatus.info.items():
         errMsg = PolicySwitchErrorMsg()
         errMsg.switchId = sw
         errMsg.error = msg
         ruleCounters.switchError.append( errMsg )
      policyCounters.rules.append( ruleCounters )
   return policyCounters

def showPolicyCounters( mode, args ):
   policyCountersList = PolicyCountersList()
   policyName = args.get( 'POLICY' )
   if policyName:
      sections = getSection( policyName )
      for sec in sorted( sections, key=lambda x: x.priority ):
         policy = policyStatusDir.pcs.get( sec.id )
         policyCounters = buildPolicyCounters( sec, policy )
         policyCountersList.policies.append( policyCounters )
   else:
      sectionByPriority = [ ( s.id, s.priority )
                           for s in sectionDir.section.itervalues() ]
      # Append policies to policyCountersList in order of section priority
      for sid, _ in sorted( sectionByPriority, key=lambda x: x[ 1 ] ):
         policy = policyStatusDir.pcs[ sid ]
         sections = getSection( sid )
         for sec in sections:
            policyCounters = buildPolicyCounters( sec, policy )
            policyCountersList.policies.append( policyCounters )

   return policyCountersList

class ServicePcsPolicyCountersCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs policy [ POLICY ] counters'
   data = {
      'service' : serviceAfterShowKwMatcher,
      'pcs' : matcherPcs,
      'policy' : matcherPolicy,
      'POLICY' : matcherPolicyName,
      'counters' : 'Display counters for rules of the policy',
   }
   handler = showPolicyCounters
   cliModel = PolicyCountersList

BasicCli.addShowCommandClass( ServicePcsPolicyCountersCmd )

#--------------------------------------------------------------------------------
# show service pcs controller policy [ POLICY ]
#--------------------------------------------------------------------------------
def createControllerRuleModel( r ):
   rule = ControllerRule()
   rule.ruleId = r.id
   rule.ruleName = cleanName( r.displayName )
   rule.priority = r.priority
   targetNames = []
   groupPrefixLen = len( GROUP_PREFIX )
   for src in r.source:
      # targetDisplayName won't have target if it's not a group or
      # if display_name wasn't sent by NSX
      name = r.targetDisplayName.get( src ) or src
      if name.startswith( GROUP_PREFIX ):
         name = name[ groupPrefixLen : ]
      targetNames.append( name )
   rule.sources = targetNames
   targetNames = []
   for dst in r.destination:
      name = r.targetDisplayName.get( dst ) or dst
      if name.startswith( GROUP_PREFIX ):
         name = name[ groupPrefixLen : ]
      targetNames.append( name )
   rule.destinations = targetNames
   rule.sourcePort = ",".join( r.srcPort.range )
   rule.destinationPort = ",".join( r.dstPort.range )
   rule.protocol = "unknown" if r.ipProto == "pipUnknown" else r.ipProto
   rule.action = "unknown" if r.action == "paUnknown" else r.action
   return rule

def createControllerPolicyModel( section ):
   policy = ControllerPolicy()
   policy.sectionId = section.id
   policy.sectionName = cleanName( section.displayName )
   policy.priority = section.priority
   ruleByPriority = [ ( r.id, r.priority ) for r in section.rule.itervalues() ]
   for rid, _ in sorted( ruleByPriority, key=lambda x: x[ 1 ] ):
      policy.rules.append( createControllerRuleModel( section.rule[ rid ] ) )
   return policy

def showControllerPolicy( mode, args ):
   policyName = args.get( 'POLICY' )
   policies = []
   if policyName:
      sections = getSection( policyName )
      for section in sorted( sections, key=lambda x: x.priority ):
         policies.append( createControllerPolicyModel( section ) )
   else:
      sectionByPriority = [ ( s.id, s.priority )
                            for s in sectionDir.section.itervalues() ]
      for sid, _ in sorted( sectionByPriority, key=lambda x: x[ 1 ] ):
         policies.append( createControllerPolicyModel( sectionDir.section[ sid ] ) )
   return ControllerPolicyList( policies=policies )

class ServicePcsControllerPolicyCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs controller policy [ POLICY ]'
   data = {
      'service': serviceAfterShowKwMatcher,
      'pcs': matcherPcs,
      'controller': matcherController,
      'policy': matcherPolicy,
      'POLICY': matcherPolicyName,
   }
   handler = showControllerPolicy
   cliModel = ControllerPolicyList

addToShowCommandClass( ServicePcsControllerPolicyCmd )

#--------------------------------------------------------------------------------
# show service pcs controller group [ GROUP ]
#--------------------------------------------------------------------------------
def createControllerNsGroupModel( group ):
   nsGroup = ControllerNsGroup()
   nsGroup.name = group.name
   addrList = []
   for addr in group.ipAddr:
      if addr.isHost:
         addrList.append( str( addr.ipGenAddr ) )
      else:
         addrList.append( str( addr ) )
   nsGroup.addrList = addrList
   return nsGroup

def showNsGroup( mode, args ):
   groupName = args.get( 'GROUP' )
   groups = []
   if groupName:
      group = groupDir.group.get( groupName )
      if group:
         groups.append( createControllerNsGroupModel( group ) )
   else:
      for group in groupDir.group.itervalues():
         groups.append( createControllerNsGroupModel( group ) )
   return ControllerNsGroupList( nsGroups=groups )

class ServicePcsControllerGroupCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs controller group [ GROUP ]'
   data = {
      'service': serviceAfterShowKwMatcher,
      'pcs': matcherPcs,
      'controller': matcherController,
      'group': 'Display NSX security group information',
      'GROUP': CliMatcher.DynamicNameMatcher( lambda mode: groupDir.group,
         'NSX security group name' ),
   }
   handler = showNsGroup
   cliModel = ControllerNsGroupList

addToShowCommandClass( ServicePcsControllerGroupCmd )

#------------------------------------------------------------------------------------
# show service pcs controller host IP_GEN_ADDR
#------------------------------------------------------------------------------------
def showControllerHost( mode, args ):
   # Gather host -> group mapping
   groups = []
   hostAddr = args[ 'IP_GEN_ADDR' ]
   hostIpGenAddr = Arnet.IpGenAddrWithMask( str( hostAddr ) )
   for group in groupDir.group.values():
      if group.resolved and hostIpGenAddr in group.ipAddr:
         groups.append( group.name )

   tags = []
   host = endpointDb.host.get( hostIpGenAddr, None )
   if host:
      tags = host.tag.keys()

   return ControllerHost( groups=groups, tags=tags )

class ServicePcsControllerHostCmd( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs controller host IP_GEN_ADDR'
   data = {
      'service': serviceAfterShowKwMatcher,
      'pcs': matcherPcs,
      'controller': matcherController,
      'host': 'Show host information',
      'IP_GEN_ADDR': IpGenAddrMatcher.ipGenAddrMatcher
   }
   handler = showControllerHost
   cliModel = ControllerHost

addToShowCommandClass( ServicePcsControllerHostCmd )

#------------------------------------------------------------------------------------
# show service pcs tag [TAG_NAME]
#------------------------------------------------------------------------------------
class ShowPcsTagInfo( ShowCommand.ShowCliCommandClass ):
   syntax = 'show service pcs tag [ TAG_NAME ]'
   data = {
      'service' : 'Show CVX services',
      'pcs' : 'Policy Control Service',
      'tag' : 'Show tag to host mapping',
      'TAG_NAME' : CliMatcher.DynamicNameMatcher( lambda mode : endpointDb.tag,
                                                  'String representing the tag',
                                                  helpname='Tagname' )
   }

   cliModel = PcsTagInfo

   @staticmethod
   def _buildTagInfo( tag ):
      tagInfo = TagInfo()
      tagInfo.tag = tag
      swIntfs = {}

      for sw in endpointDb.tag[ tag ].switchTag.values():
         switchId = switchIdCache.getHost( sw.switchId ) or sw.switchId
         switch = SwitchInfo( switchId=switchId )
         swIntfs[ sw.switchId ] = {}
         for intf in sw.intf:
            intfInfo = IntfInfo( name=intf )
            switch.intfs[ intf ] = intfInfo
            swIntfs[ sw.switchId ][ intf ] = intfInfo
         tagInfo.switches[ switchId ] = switch

      for host in endpointDb.tag[ tag ].host.values():
         for sw in set( swIntfs ).intersection( set( host.switchInfo ) ):
            switchInfo = host.switchInfo[ sw ]
            switchIntfs = swIntfs[ sw ]
            hostIntfs = [ str( i ) for i in switchInfo.intfId.values() ]
            for i in set( hostIntfs ).intersection( set( switchIntfs ) ):
               intfInfo = swIntfs[ sw ][ i ]
               intfInfo.hosts.append( str( host.ipAddr ).split( '/' )[ 0 ] )

      return tagInfo

   @staticmethod
   def handler( mode, args ):
      tag = args.get( 'TAG_NAME', None )
      tags = {}
      if tag:
         if tag in endpointDb.tag:
            tags[ tag ] = ShowPcsTagInfo._buildTagInfo( tag )
      else:
         for tag in endpointDb.tag:
            tags[ tag ] = ShowPcsTagInfo._buildTagInfo( tag )
      return PcsTagInfo( tags=tags )

addToShowCommandClass( ShowPcsTagInfo )

# Register cleanup callbacks
def pcsNoCvx( mode ):
   config.enabled = False
   config.controllerIp = Arnet.IpAddr( '0.0.0.0' )
   config.port = config.defaultPort
   config.username = ""
   config.password = ""
   config.endpointName = ""
   config.thumbprint = ""

addNoCvxCallback( pcsNoCvx )

def pcsCvxShutdown( mode, no=False ):
   global switchAclConfig
   global switchAclStatus
   if not no:
      switchAclConfig = None
      switchAclStatus = None

addCvxShutdownCallback( pcsCvxShutdown )

def doPcsControllerdbMounts( controllerdbEm ):
   global switchAclConfig
   global switchAclStatus
   # Used only for 'detail' show commands that need printing of the
   # generated ACL rules
   switchAclConfig = LazyMount.mount( controllerdbEm,
                        "acl/v1/config/switch", "Tac::Dir", "ri" )
   switchAclStatus = LazyMount.mount( controllerdbEm,
                        "acl/v1/status/switch", "Tac::Dir", "ri" )

@Plugins.plugin( requires=( "ControllerdbMgr", ) )
def Plugin( entMgr ):
   global config
   global status
   global apiStatus
   global controllerConfig
   global httpServiceConfig
   global serviceInputConfigDir
   global endpointDb
   global sectionDir
   global policyData
   global groupDir
   global policyStatusDir

   config = ConfigMount.mount( entMgr,
               "pcs/config", "Pcs::Config", "w" )
   status = LazyMount.mount( entMgr,
               "pcs/status", "Pcs::Status", "r" )
   apiStatus = LazyMount.mount( entMgr,
               "pcs/apiStatus", "Pcs::ApiStatus", "r" )
   controllerConfig = LazyMount.mount( entMgr,
                        "controller/config", "Controllerdb::Config", "r" )
   httpServiceConfig = ConfigMount.mount( entMgr,
                           "mgmt/capi/config", "HttpService::Config", "w" )
   serviceInputConfigDir = ConfigMount.mount( entMgr,
                              "controller/service/config",
                              "Controller::ServiceConfigDir", "w" )
   endpointDb = LazyMount.mount( entMgr,
                  "pcs/controller/endpoint", "Pcs::PolicyEndpointDb", "r" )
   sectionDir = LazyMount.mount( entMgr,
                  "pcs/input/http/section", "Pcs::PolicySectionDir", "r" )
   groupDir = LazyMount.mount( entMgr,
                  "pcs/input/http/group", "Pcs::PolicyGroupDir", "r" )
   policyData = LazyMount.mount( entMgr,
                  "pcs/agent/policy", "PcsAgent::PolicyData", "r" )
   policyStatusDir = LazyMount.mount( entMgr,
                  "pcs/controller/status", "Pcs::PolicyStatusDir", "r" )

   CliPlugin.ControllerdbLib.registerNotifiee( doPcsControllerdbMounts )
