# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
from CliModel import (
      Bool,
      Enum,
      Int,
      List,
      Str,
      Submodel,
)
from SegmentRoutingCliShowHelper import (
   npHelpStr,
   reachabilityAlgorithmEnumValues,
   renderSRGB,
   renderCommonSummary,
   renderPrefixSegmentSummary,
   SrCommonHeader,
   SelfOriginatedSegmentStatisticsModel,
   SrPrefixSegmentFlagsModel,
   SrPrefixSegmentModel,
   SrGlobalBlockModel,
)
from TableOutput import terminalWidth
import TableOutput

class OspfSrCommonModel( SrCommonHeader ):
   instanceId = Int( help='OSPF instance ID' )

   def render( self ):
      print "OSPF Instance ID: %s" % self.instanceId
      SrCommonHeader.render( self )

class OspfShowSrModel( OspfSrCommonModel ):
   """Model for show ip ospf segment-routing
   """
   srgbBase = Int( help="Base of SRGB" )
   srgbSize = Int( help="Size of SRGB" )
   srPeerCount = Int( help="Number of OSPF SR capable peers" )
   reachabilityAlgorithm = Enum( values=reachabilityAlgorithmEnumValues,
                                 help="OSPF Reachability Algorithm" )
   mappingServer = Bool( help="This is a Segment Routing Mapping Server (SRMS)" )
   selfOriginatedSegmentStatistics = Submodel( 
         valueType=SelfOriginatedSegmentStatisticsModel,
         help="Self-originated segment statistics" )

   def render( self ):
      OspfSrCommonModel.render( self )
      renderSRGB( self )
      renderCommonSummary( self, "OSPF" )      
      self.selfOriginatedSegmentStatistics.render()

class OspfPrefixSegmentFlagsModel( SrPrefixSegmentFlagsModel ):
   """Model for flags used in prefix segment
   """
   np = Bool( help=npHelpStr, default=False )
   m = Bool( help="Set if the Prefix-SID was advertised by a "
                  "Segment Routing Mapping Server",
             default=False )

   def toStr( self ):
      flags = "NP:%d M:%d E:%d V:%d L:%d N:%d" % ( self.np, self.m, self.e,
                                                   self.v, self.l, self.n )
      return flags

class OspfPrefixSegmentModel( SrPrefixSegmentModel ):
   """Model to store individual OSPF prefix segment record
   """
   flags = Submodel( valueType=OspfPrefixSegmentFlagsModel, help="Flags" )
   routerId = Str( help="Router ID" )

class OspfShowSrPrefixSegmentSummaryModel( OspfSrCommonModel ):
   """Model for show ip ospf segment-routing prefix-segments
   """
   nodeSidCount = Int( help="Number of SR node segments" )
   proxyNodeSidCount = Int(help="Number of SR proxy node segments" )
   prefixSidCount = Int( help="Number of SR prefix segments" )
   prefixSegments = List( valueType=OspfPrefixSegmentModel,
                          help="Prefix segments" )

   def render( self ):
      OspfSrCommonModel.render( self )

      legend = \
         "Flag Descriptions: NP: No-PHP, M: Mapping Server, E: Explicit-NULL,\n" \
         "                   V: Value, L: Local, N: Node Segment\n" \
         "Segment status codes: * - Self originated Prefix"

      headings = ( "", "Prefix", "SID", "Type", "Flags", "Router ID" )
      table = TableOutput.createTable( headings, tableWidth=terminalWidth() )
      for item in self.prefixSegments:
         table.newRow( item.statusCode(), item.prefix, item.segmentId,
                       item.segmentType, item.flags.toStr(), item.routerId )

      renderPrefixSegmentSummary( legend, table, self.nodeSidCount,
                                  self.prefixSidCount, self.proxyNodeSidCount )

class OspfGlobalBlockModel( SrGlobalBlockModel ):
   """Model to store individual global block record.
   """
   routerId = Str( help="Router ID" )

class OspfShowSrGlobalBlocksSummaryModel( OspfSrCommonModel ):
   """Model for show isis segment-routing global-blocks
   """
   srPeerCount = Int( help="Number of OSPF SR peers capable segments" )
   globalBlocks = List( valueType=OspfGlobalBlockModel, help="Global blocks" )

   def render(self):
      OspfSrCommonModel.render( self )
      
      headings = ( "Router ID", "Base", "Size" )
      table = TableOutput.createTable( headings )
      for item in self.globalBlocks:
         table.newRow( item.routerId, item.base, item.size )

      print "Number of OSPF segment routing capable peers: %d" % self.srPeerCount
      print
      print table.output()
